% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rmrf.R
\name{rmrf2d}
\alias{rmrf2d}
\title{Sampling of Markov Random Fields on 2d lattices}
\usage{
rmrf2d(
  init_Z,
  mrfi,
  theta,
  cycles = 60,
  sub_region = NULL,
  fixed_region = NULL
)
}
\arguments{
\item{init_Z}{One of two options:
\itemize{
\item A \code{matrix} object with the initial field configuration. Its
valuesmust be integers in \verb{\{0,...,C\}}.
\item A length 2 \code{numeric} vector with the lattice dimensions.
}}

\item{mrfi}{A \code{\link[=mrfi-class]{mrfi}} object representing the
interaction structure.}

\item{theta}{A 3-dimensional array describing potentials. Slices represent
interacting positions, rows represent pixel values and columns represent
neighbor values. As an example: \code{theta[1,3,2]} has the potential for the
pair of values 0,2 observed in the second relative position of \code{mrfi}.}

\item{cycles}{The number of updates to be done (for each each pixel).}

\item{sub_region}{\code{NULL} if the whole lattice is considered or a \code{logical}
\code{matrix} with \code{TRUE} for pixels in the considered region.}

\item{fixed_region}{\code{NULL} if the whole lattice is to be sampled or a
\code{logical} \code{matrix} with \code{TRUE} for pixels to be considered fixed. Fixed
pixels are not updated in the Gibbs Sampler.}
}
\value{
A \code{matrix} with the sampled field.
}
\description{
Performs pixelwise updates based on conditional distributions
to sample from a Markov random field.
}
\details{
This function implements a Gibbs Sampling scheme to sample from
a Markov random field by iteratively sampling pixel values from the
conditional distribution
\deqn{P(Z_i | Z_{{N}_i}, \theta).}

A cycle means exactly one update to each pixel. The order pixels are
sampled is randomized within each cycle.

If \code{init_Z} is passed as a length 2 vector with lattice dimensions, the
initial field is sampled from independent discrete uniform distributions in
\verb{\{0,...,C\}}. The value of C is obtained from the number of rows/columns of
\code{theta}.

A MRF can be sampled in a non-rectangular region of the lattice with the use of
the \code{sub_region} argument or by setting pixels to \code{NA} in the initial
configuration \code{init_Z}. Pixels with \code{NA} values in \code{init_Z} are completely
disconsidered from the conditional probabilities and have the same effect as
setting \code{sub_region = is.na(init_Z)}. If \code{init_Z} has \code{NA} values,
\code{sub_region} is ignored and a warning is produced.

A specific region can be kept constant during the Gibbs Sampler by using the
\code{fixed_region} argument. Keeping a subset of pixels constant is useful when
you want to sample in a specific region of the image conditional to the
rest, for example, in texture synthesis problems.
}
\note{
As in any Gibbs Sampling scheme, a large number of cycles may be
required to achieve the target distribution, specially for strong
interaction systems.
}
\examples{
# Sample using specified lattice dimension
Z <- rmrf2d(c(150,150), mrfi(1), theta_potts)

#Sample using itial configuration
\donttest{
Z2 <- rmrf2d(Z, mrfi(1), theta_potts)

# View results
dplot(Z)
dplot(Z2)

# Using sub-regions
subreg <- matrix(TRUE, 150, 150)
subreg <- abs(row(subreg) - 75) + abs(col(subreg) - 75) <= 80
# view the sub-region
dplot(subreg)

Z3 <- rmrf2d(c(150,150), mrfi(1), theta_potts, sub_region = subreg)
dplot(Z3)

# Using fixed regions
fixreg <- matrix(as.logical(diag(150)), 150, 150)
# Set initial configuration: diagonal values are 0.
init_Z4 <- Z
init_Z4[fixreg] <- 0

Z4 <- rmrf2d(init_Z4, mrfi(1), theta_potts, fixed_region = fixreg)
dplot(Z4)

# Combine fixed regions and sub-regions
Z5 <- rmrf2d(init_Z4, mrfi(1), theta_potts,
fixed_region = fixreg, sub_region = subreg)
dplot(Z5)
}

}
\seealso{
A paper with detailed description of the package can be found at
\url{https://arxiv.org/abs/2006.00383}.

\code{\link[=rmrf2d_mc]{rmrf2d_mc}} for generating multiple points of a
Markov Chain to be used in Monte-Carlo methods.
}
\author{
Victor Freguglia
}
