% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/adp.rdi.R
\name{read.adp.rdi}
\alias{read.adp.rdi}
\title{Read a Teledyne/RDI ADP File}
\usage{
read.adp.rdi(file, from, to, by, tz = getOption("oceTz"),
  longitude = NA, latitude = NA, type = c("workhorse"),
  monitor = FALSE, despike = FALSE, processingLog, testing = FALSE,
  debug = getOption("oceDebug"), ...)
}
\arguments{
\item{file}{a connection or a character string giving the name of the file
to load.  (For \code{read.adp.sontek.serial}, this is generally a list of
files, which will be concatenated.)}

\item{from}{indication of the first profile to read.  This can be an
integer, the sequence number of the first profile to read, or a POSIXt time
before which profiles should be skipped, or a character string that converts
to a POSIXt time (assuming UTC timezone).  See \dQuote{Examples}, and make
careful note of the use of the \code{tz} argument. If \code{from} is not
supplied, it defaults to 1.}

\item{to}{an optional indication of the last profile to read, in a
format as described for \code{from}.  As a special case, \code{to=0} means
to read the file to the end.  If \code{to} is not supplied, then it defaults
to 0.}

\item{by}{an optional indication of the stride length to use while walking through
the file.  If this is an integer, then \code{by-1} profiles are skipped
between each pair of profiles that is read, e.g. the default \code{by=1}
means to read all the data.  (For RDI files \emph{only}, there are some
extra features to avoid running out of memory; see \dQuote{Memory considerations}.)}

\item{tz}{character string indicating time zone to be assumed in the data.}

\item{longitude}{optional signed number indicating the longitude in degrees
East.}

\item{latitude}{optional signed number indicating the latitude in degrees
North.}

\item{type}{character string indicating the type of instrument.}

\item{monitor}{boolean, set to \code{TRUE} to provide an indication of progress
in reading the file, either by printing a dot for each profile or by writing
a textual progress bar with \code{\link{txtProgressBar}}.}

\item{despike}{if \code{TRUE}, \code{\link{despike}} will be used to clean
anomalous spikes in heading, etc.}

\item{processingLog}{if provided, the action item to be stored in the log.
(Typically only provided for internal calls; the default that it provides is
better for normal calls by a user.)}

\item{testing}{logical value (IGNORED).}

\item{debug}{a flag that turns on debugging.  Set to 1 to get a moderate
amount of debugging information, or to 2 to get more.}

\item{\dots}{additional arguments, passed to called routines.}
}
\value{
An \code{adp} object, i.e. one inheriting from \code{\link{adp-class}}.
The form of this object varies with instrument type. In some cases
navigational data may be included, e.g. \code{read.adp.rdi} can
read NMEA strings (which get stored in an item called \code{nmea}
in the \code{data} slot).
}
\description{
Read a Teledyne/RDI ADCP file (called 'adp' in oce).
}
\details{
As of 2016-09-25, this function has provisional functionality to
read data from the new "SentinelV" series ADCP -- essentially a
combination of a 4 beam workhorse with an additional vertical
centre beam.

If a heading bias had been set with the \code{EB} command during the setup
for the deployment, then a heading bias will have been stored in the file's
header.  This value is stored in the object's metadata as
\code{metadata$heading.bias}.  \strong{Importantly}, this value is
subtracted from the headings stored in the file, and the result of this
subtraction is stored in the objects heading value (in \code{data$heading}).
It should be noted that \code{read.adp.rdi()} was tested for firmware
version 16.30.  For other versions, there may be problems.  For example, the
serial number is not recognized properly for version 16.28.

In Teledyne/RDI ADP data files, velocities are coded to signed 2-byte integers, with a
scale factor being used to convert to velocity in metres per second.  These
two facts control the maximum recordable velocity and the velocity
resolution, values that may be retrieved for an ADP object name \code{d}
with \code{d[["velocityMaximum"]]} and \code{d[["velocityResolution"]]}.
}
\section{Names of items in data slot}{


The names of items in the \code{data} slot are below. Not all items are present
for ll file varieties; use \code{\link{names}(d[["data"]])} to determine the
names used in an object named \code{d}. In this list, items are either
a vector (with one sample per time of measurement), a 
``matrix'' with first index for time and second for bin number,
or an ``array'' with first index for time, second for bin number,
and third for beam number. (Items are of vector type, unless
otherwise indicated.)

\strong{\code{a}}=signal amplitude array [units?];
\strong{\code{ambientTemp}}=ambient temperature [degC];
\strong{\code{attitude}}=attitude [deg];
\strong{\code{attitudeTemp}}=XXX;
\strong{\code{avgMagnitudeVelocityEast}}=XXX;
\strong{\code{avgMagnitudeVelocityNorth}}=XXX;
\strong{\code{avgSpeed}}=XXX;
\strong{\code{avgTrackMagnetic}}=XXX;
\strong{\code{avgTrackTrue}}=XXX;
\strong{\code{avgTrueVelocityEast}}=XXX;
\strong{\code{avgTrueVelocityNorth}}=XXX;
\strong{\code{br}}=bottom range matrix [m];
\strong{\code{bv}}=bottom velocity matrix [m/s];
\strong{\code{contaminationSensor}}=XXX;
\strong{\code{depth}}=depth [m];
\strong{\code{directionMadeGood}}=XXX;
\strong{\code{distance}}=XXX;
\strong{\code{firstLatitude}}=latitude at start of profile [deg];
\strong{\code{firstLongitude}}=longitude at start of profile [deg];
\strong{\code{firstTime}}=XXX;
\strong{\code{g}}=data goodness matrix [units?];
\strong{\code{heading}}=instrument heading [degrees];
\strong{\code{headingStd}}=instrument heading std-dev [deg];
\strong{\code{lastLatitude}}=latitude at end of profile [deg];
\strong{\code{lastLongitude}}=longitude at end of profile [deg];
\strong{\code{lastTime}}=XXX;
\strong{\code{numberOfHeadingSamplesAveraged}}=XXX;
\strong{\code{numberOfMagneticTrackSamplesAveraged}}=XXX;
\strong{\code{numberOfPitchRollSamplesAveraged}}=XXX;
\strong{\code{numberOfSpeedSamplesAveraged}}=XXX;
\strong{\code{numberOfTrueTrackSamplesAveraged}}=XXX;
\strong{\code{pitch}}=instrument pitch [deg];
\strong{\code{pitchStd}}=instrument pitch std-dev [deg];
\strong{\code{pressure}}=pressure [dbar];
\strong{\code{pressureMinus}}=XXX;
\strong{\code{pressurePlus}}=XXX;
\strong{\code{pressureStd}}=pressure std-dev [dbar];
\strong{\code{primaryFlags}}=XXX;
\strong{\code{q}}=data quality array;
\strong{\code{roll}}=instrument roll [deg];
\strong{\code{rollStd}}=instrument roll std-dev [deg];
\strong{\code{salinity}}=salinity;
\strong{\code{shipHeading}}=ship heading [deg];
\strong{\code{shipPitch}}=ship pitch [deg];
\strong{\code{shipRoll}}=ship roll [deg];
\strong{\code{soundSpeed}}=sound speed [m/s];
\strong{\code{speedMadeGood}}=speed over ground (?) [m/s];
\strong{\code{speedMadeGoodEast}}=XXX;
\strong{\code{speedMadeGoodNorth}}=XXX;
\strong{\code{temperature}}=temperature [degC];
\strong{\code{time}}=profile time [POSIXct];
\strong{\code{v}}=velocity array [m/s];
\strong{\code{xmitCurrent}}=transmit current [unit?];
and
\strong{\code{xmitVoltage}}=transmit voltage.
}

\section{Memory considerations}{


For \code{RDI} files only, and only in the case where \code{by} is not specified,
an attempt is made to avoid running out of memory by skipping some profiles
in large input files. This only applies if \code{from} and \code{to} are both
integers; if they are times, none of the rest of this section applies.

A key issue is that RDI files store velocities in 2-byte values, which is
not a format that R supports. These velocities become 8-byte (numeric) values
in R. Thus, the R object created by \code{read.adp.rdi} will require more memory
than that of the data file. A scale factor can be estimated by ignoring
vector quantities (e.g. time, which has just one value per profile) and concentrating on matrix properties
such as velocity, backscatter, and correlation. These three elements have equal dimensions.
Thus, each 4-byte slide in the data file (2 bytes + 1 byte + 1 byte)
corresponds to 10 bytes in the object (8 bytes + 1 byte + 1 byte).
Rounding up the resultant 10/4 to 3 for safety, we conclude that any limit on the
size of the R object corresponds to a 3X smaller limit on file size.

Various things can limit the size of objects in R, but a strong upper limit
is set by the space the operating system provides to R. The least-performant machines
in typical use appear to be Microsoft-Windows systems, which limit R objects to
about 2e6 bytes (see \code{?Memory-limits}).  Since R routinely duplicates objects for certain tasks
(e.g. for call-by-value in function evaluation), \code{read.adp.rdi} uses a safety
factor in its calculation of when to auto-decimate a file. This factor is set to 3,
based partly on the developers' experience with datasets in their possession.
Multiplied by the previously stated safety factor of 3,
this suggests that the 2 GB limit on R objects corresponds to approximately a
222 MB limit on file size. In the present version of \code{read.adp.rdi}, this
value is lowered to 200 MB for simplicity. Larger files are considered to be "big",
and are decimated unless the user supplies a value for the \code{by} argument.

The decimation procedure has two cases.
\enumerate{
\item \emph{Case 1.} If \code{from=1} and
\code{to=0} (or if neither \code{from} or \code{to} is given), then the
intention is to process the full span of the data.  If the input file is
under 200 MB, then \code{by} defaults to 1, so that all profiles are read.
For larger files, \code{by} is set to the \code{\link{ceiling}} of the
ratio of input file size to 200 MB.

\item \emph{Case 2.} If \code{from} exceeds 1, and/or \code{to} is nonzero, then
the intention is to process only an interior subset of the file. In this
case, \code{by} is calculated as the \code{\link{ceiling}} of
the ratio of \code{bbp*(1+to-from)} to 200 MB, where \code{bbp} is the number
of file bytes per profile. Of course, \code{by} is set to 1, if this
ratio is less than 1.
}

If the result of these calculations is that \code{by} exceeds 1, then
messages are printed to alert the user that the file will be decimated,
and also \code{monitor} is set to \code{TRUE}, so that a textual progress bar
is shown.
}

\section{Development Notes}{

An important part of the work of this function is to recognize what
will be called "data chunks" by two-byte ID sequences. This function is
developed in a practical way, with emphasis being focussed on
data files in the possession of the developers. Since Teledyne-RDI tends
to introduce new ID codes with new instruments, that means that
\code{read.adp.rdi} may not work on recently-developed instruments.

The following two-byte ID codes are recognized by \code{read.adp.rdi}
at this time (with bytes listed in natural order, LSB byte before
MSB). Items preceeded by an asterisk are recognized, but not handled,
and so produce a warning.
\preformatted{
  0x00 0x01 velocity
  0x00 0x02 correlation
  0x00 0x03 echo intensity
  0x00 0x04 percent good
  0x00 0x06 bottom track
  0x00 0x0a Sentinel vertical beam velocity
  0x00 0x0b Sentinel vertical beam correlation
  0x00 0x0c Sentinel vertical beam amplitude
  0x00 0x0d Sentinel vertical beam percent good
  0x00 0x20 VMDASS
* 0x00 0x30 binary fixed attitude (developer: see p169 of [3] for format)
  0x00 0x32 Sentinel transformation matrix
  0x00 0x0a Sentinel data
  0x00 0x0b Sentinel correlation
  0x00 0x0c Sentinel amplitude
  0x00 0x0d Sentinel percent good
  0x01 0x0f ?? something to do with V series and 4-beam
}

Lacking a comprehensive Teledyne-RDI listing of ID codes,
the authors have cobbled together a listing from documents to which they
have access, viz.
\itemize{

\item
Table 33 of [3] lists codes as follows:
\preformatted{
Standard ID    Standard plus 1D    DESCRIPTION
===========    ================    ===========================
 MSB    LSB          MSB    LSB
 ---    ---          ---    ---
  7F     7F           7F     7F    Header
  00     00           00     01    Fixed Leader
  00     80           00     81    Variable Leader
  01     00           01     01    Velocity Profile Data
  02     00           02     01    Correlation Profile Data
  03     00           03     01    Echo Intensity Profile Data
  04     00           04     01    Percent Good Profile Data
  05     00           05     01    Status Profile Data
  06     00           06     01    Bottom Track Data
  20     00           20     00    Navigation
  30     00           30     00    Binary Fixed Attitude
  30  40-F0           30  40-F0    Binary Variable Attitude
}
\item
Table 6 on p90 of [4] lists "Fixed Leader Navigation" ID
codes (none of which are handled by \code{read.adp.rdi} yet)
as follows (the format is reproduced literally; note that
e.g. 0x2100 is 0x00,0x21 in the oce notation):
\preformatted{
ID       Description
======   ===========
0x2100   $xxDBT
0x2101   $xxGGA
0x2102   $xxVTG
0x2103   $xxGSA
0x2104   $xxHDT, $xxHGD or $PRDID
}
and following pages in that manual reveal that
DBT refers to depth below transducer;
GGA refers to global positioning system;
VTA refers to track made good and ground speed;
GSA refers to GPS DOP and active satellites;
HDT refers to heading, true;
HDG refers to heading, deviation, and variation;
and
PRDID refers to heading, pitch and roll.
}
}

\section{Error recovery}{


Files can sometimes be corrupted, and \code{read.adp.rdi} has ways to handle two types
of error that have been noticed in files supplied by users.
\enumerate{

\item There are two bytes within each ensemble that indicate the number of bytes to check within
that ensemble, to get the checksum. Sometimes, those two bytes can be erroneous, so that
the wrong number of bytes are checked, leading to a failed checksum. As a preventative
measure, \code{read.adp.rdi} checks the stated ensemble length, whenever it detects a
failed checksum. If that length agrees with the length of the most recent ensemble that
had a good checksum, then the ensemble is declared as faulty and is ignored. However,
if the length differs from that of the most recent accepted ensemble, then \code{read.adp.rdi}
goes back to just after the start of the ensemble, and searches forward for the next two-byte
pair, namely \code{0x7f 0x7f}, that designates the start of an ensemble.  Distinct notifications
are given about these two cases, and they give the byte numbers in the original file, as a way
to help analysts who want to look at the data stream with other tools.

\item At the end of an ensemble, the next two characters ought to be \code{0x7f 0x7f}, and if they
are not, then the next ensemble is faulty. If this error occurs, \code{read.adp.rdi} attempts
to recover by searching forward to the next instance of this two-byte pair, discarding any
information that is present in the mangled ensemble.
}

In each of these cases, warnings are printed about ensembles that seem problematic.
Advanced users who want to diagnose the problem further might find it helpful to
examine the original data file using other tools. To this end, \code{read.adp.rdi}
inserts an element named \code{ensembleInFile} into the \code{metadata} slot.
This gives the starting byte number of each inferred ensemble within the original data
file.  For example, if \code{d} is an object read with \code{read.adp.rdi}, then using
\preformatted{
plot(d[["time"]][-1], diff(d[["ensembleInFile"]]))
}
can be a good way to narrow in on problems.
}

\examples{
adp <- read.adp.rdi(system.file("extdata", "adp_rdi.000", package="oce"))
summary(adp)

}
\references{
1. Teledyne-RDI, 2007. \emph{WorkHorse commands and output data
format.} P/N 957-6156-00 (November 2007).  (Section 5.3 h details the binary
format, e.g. the file should start with the byte \code{0x7f} repeated twice,
and each profile starts with the bytes \code{0x80}, followed by \code{0x00},
followed by the sequence number of the profile, represented as a
little-endian two-byte short integer.  \code{read.adp.rdi} uses these
sequences to interpret data files.)

2. Teledyne RD Instruments, 2015. \emph{V Series monitor, sentinel Output Data Format.}
P/N 95D-6022-00 (May 2015). \code{SV_ODF_May15.pdf}

3. Teledyne RD Instruments, 2014. \emph{Ocean Surveyor / Ocean Observer Technical Manual.}
P/N 95A-6012-00 (April 2014). \code{OS_TM_Apr14.pdf}

4. Teledyne RD Instruments, 201?. \emph{WinRiver User's Guide International Version}
P/N 957-6171-00 (June 2001) \code{WinRiver User Guide International Version.pdf.pdf}
}
\seealso{
Other things related to adp data: \code{\link{[[,adp-method}},
  \code{\link{[[<-,adp-method}},
  \code{\link{ad2cpHeaderValue}}, \code{\link{adp-class}},
  \code{\link{adpEnsembleAverage}}, \code{\link{adp}},
  \code{\link{as.adp}}, \code{\link{beamName}},
  \code{\link{beamToXyzAdpAD2CP}},
  \code{\link{beamToXyzAdp}}, \code{\link{beamToXyzAdv}},
  \code{\link{beamToXyz}}, \code{\link{beamUnspreadAdp}},
  \code{\link{binmapAdp}}, \code{\link{enuToOtherAdp}},
  \code{\link{enuToOther}},
  \code{\link{handleFlags,adp-method}},
  \code{\link{is.ad2cp}}, \code{\link{plot,adp-method}},
  \code{\link{read.adp.ad2cp}},
  \code{\link{read.adp.nortek}},
  \code{\link{read.adp.sontek.serial}},
  \code{\link{read.adp.sontek}}, \code{\link{read.adp}},
  \code{\link{read.aquadoppHR}},
  \code{\link{read.aquadoppProfiler}},
  \code{\link{read.aquadopp}}, \code{\link{rotateAboutZ}},
  \code{\link{setFlags,adp-method}},
  \code{\link{subset,adp-method}},
  \code{\link{summary,adp-method}}, \code{\link{toEnuAdp}},
  \code{\link{toEnu}}, \code{\link{velocityStatistics}},
  \code{\link{xyzToEnuAdpAD2CP}},
  \code{\link{xyzToEnuAdp}}, \code{\link{xyzToEnu}}
}
\author{
Dan Kelley and Clark Richards
}
\concept{things related to adp data}
