% Generated by roxygen2 (4.1.1): do not edit by hand
% Please edit documentation in R/batch_opt.R
\name{batch.estimation}
\alias{batch.estimation}
\title{Two stage estimation, a pilot estimate of mixing alpha and a following importance sampling, with or without control variates}
\usage{
batch.estimation(seed, batch.size, mixture.param,
  eps = rep(0.1/mixture.param$J, mixture.param$J), fname = "f",
  rpname = "rp", rqname = "rq", dpname = "dp", dqname = "dq",
  cv = TRUE, opt.info = FALSE, opt.param = list(reltol = 0.001, relerr =
  0.001, rho0 = 1, maxin = 20, maxout = 30))
}
\arguments{
\item{seed}{seed for sampling}

\item{batch.size}{length two vector of batch sizes}

\item{mixture.param}{mixture.param = list(p, J, ...), where \eqn{p} is the dimension of the sample, and \eqn{J} is the number of mixture components, including the defensive one. mixture.param should be compatible with user defined functions \emph{f(n, j, mixture.param), rp(n, mixture.param), rq(n, j, mixture.param), dp(xmat, mixture.param), dq(xmat, j, mixture.param)}}

\item{eps}{the lower bound for optimizing \emph{alpha}. if \emph{eps} is of length 1, it is expanded to \emph{rep(eps, J)}, default to be \emph{rep(0.1/J, J)}}

\item{fname}{name of user defined function \emph{fname(xmat, j, mixture.param)}. \emph{xmat} is an \eqn{n \times p} matrix of \eqn{n} samples with \eqn{p} dimensions. \emph{fname} returns a vector of function values for each row in \emph{xmat}. \emph{fname} is defined for \eqn{j = 1, \cdots, J}. \eqn{j = 1, \cdots, J - 1} corresponds to different proposal mixture components, and \eqn{j = J} corresponds to the defensive mixture component.}

\item{rpname}{name of user definded function \emph{rpname(n, mixture.param)}. It generates \eqn{n} random samples from target distribution \emph{pname}. Parameters can be specified in \emph{mixture.param}. \emph{rpname} returns an \eqn{n \times p} matrix.}

\item{rqname}{name of user defined function \emph{rqname(n, j, mixture.param)}. It generate \eqn{n} random samples from the \eqn{j^{th}} mixture component of proposal mixture distribution. \emph{rqname} returns an \eqn{n \times p} matrix. \emph{rqname} is defined for \eqn{j = 1, \cdots, J - 1}.}

\item{dpname}{name of user defined function \emph{dpname(xmat, mixture.param)}. It returns the density of \emph{xmat} from the target distribution \eqn{pname} as a vector of length \emph{nrow(xmat)}. Note that only the ratio between \emph{dpname} and \emph{dqname} matters. So \emph{dpname} and \emph{dqname} can return values of \eqn{C \times}\emph{dpname} and \eqn{C \times}\emph{dqname} respectively.}

\item{dqname}{name of user defined function \emph{dqname(xmat, j, mixture.param)}. It returns the density of \emph{xmat} from the proposal distribution \eqn{q_j} as a vector of length \emph{nrow(xmat)}. \emph{dqname} is defined for \eqn{j = 1, \cdots, J - 1}. Note that only the ratio between \emph{dpname} and \emph{dqname} matters. So \emph{dpname} and \emph{dqname} can  return values of \eqn{C \times}\emph{dpname} and \eqn{C \times}\emph{dqname} respectively.}

\item{cv}{TRUE indicates optimizing \emph{alpha} and \emph{beta} at the same time,
and estimate with the formula \deqn{\hat{\mu}_{\alpha_{**},\beta} = \frac{1}{n_2}\sum\limits_{i = 1}^{n_2}\frac{f(x_{i})p(x_{i}) - {\beta}^{\mathsf{T}}\bigl(\boldsymbol{q}(x_{i}) - p(x_{i})\boldsymbol{1}\bigr)}{q_{\alpha_{**}}(x_{i})}.}
FALSE indicates optimizing \emph{alpha} only, and estimate with the formula \deqn{\hat{\mu}_{\alpha_{*}}= \frac{1}{n_2}\sum\limits_{i = 1}^{n_2}\frac{f(x_{i})p(x_{i})}{q_{\alpha_{*}}(x_{i})}.}}

\item{opt.info}{logical value indicating whether to save the returned value of the optimization procedure. See \code{\link{penoptpersp}} and \code{\link{penoptpersp.alpha.only}} for the returned value.}

\item{opt.param}{a list of paramters for the  damped Newton method with backtracking line search \describe{
  \item{reltol}{relative tolerence in dampednewton step, default to be \eqn{10^{-2}}}
  \item{relerr}{Newton step stop when within (1+\emph{relerr}) of minimum variance, default to be \eqn{10^{-3}}}
  \item{rho0}{initial value for \emph{rho}, default to be 1}
} Only need to supply part of the list to change the default value.}
}
\value{
a list of \describe{
  \item{mu.hat}{the estimate for mu}
  \item{sd.hat}{estimated sd of mu.hat}
  \item{alpha.opt}{the estimated optimal alpha}
  \item{beta.opt}{if cv = TRUE, the estimated optimal beta}
  \item{opt.info}{if opt.info = TRUE, also return the list(x=x, y=y, z=z, alpha=alpha,beta=beta,rho=rho,f=f}
  \item{}{rhopen=rhopen, outer=log(rho0/rho,mu), relerr = relerr, alphasum = sum(alpha))}
  \item{}{from the optimization after batch 1.}
}
}
\description{
Two stage estimation, a pilot estimate of mixing alpha and a following importance sampling, with or without control variates
}
\details{
Estimate \eqn{E_p f} with a two step importance sample procedure. See He & Owen(2014) for details.
}
\examples{
library(optismixture)
seed <- 1
p <- 5
rho <- 1/2
gamma <- 2.4
sigma.dvar <- function(rho, p){
   sigma <- matrix(0, p, p)
   for(i in 1:(p-1)){
       for(j in (i+1):p){
           sigma[i,j] <- rho^(abs(i-j))
       }
   }
   sigma <- sigma + t(sigma)
   diag(sigma) <- 1
   return(sigma)
}
sigma <- sigma.dvar(rho, p)
batch.size <- c(10^4, 1002)
j.vec <- 2^-(seq(1,5,1))
eps.status <- 1
eps.safe <- 0.1
## initialization and construct derived parameters
mu <- rep(0, p)
x0 <- matrix(1, 1, p)
x0.mat <- rbind(rep(1,p), rep(-1, p))
j.mat <- data.frame(centerid = rep(1:dim(x0.mat)[1], each = length(j.vec)),
                    variance = rep(j.vec, 2))
J <- dim(j.mat)[1] + 1
eps <- rep(0.1/J, J)
mixture.param <- list(x0 = x0, x0.mat = x0.mat, p = p,
sigma = sigma, gamma = gamma, j.mat = j.mat, J = J)
f <- function(x, j, mixture.param){
    f1 <- function(x, mixture.param){
        x0 <- mixture.param$x0
        gamma <- mixture.param$gamma
        return(sum((x - x0)^2)^(-gamma/2))
    }
    return(apply(x, 1, f1, mixture.param))
}
dq <- function(x, j, mixture.param){
    centerid <- mixture.param$j.mat[j, 1]
    j.param <- mixture.param$j.mat[j, 2]
    return(mvtnorm::dmvnorm(x, mixture.param$x0.mat[centerid,], j.param*diag(mixture.param$p)))
}
dp <- function(x, mixture.param){
    return(mvtnorm::dmvnorm(x, rep(0, mixture.param$p), mixture.param$sigma))
}
rq <- function(n, j, mixture.param){
    centerid <- mixture.param$j.mat[j, 1]
    j.param <- mixture.param$j.mat[j,2]
    return(mvtnorm::rmvnorm(n, mixture.param$x0.mat[centerid, ], j.param*diag(mixture.param$p)))
}
rp <- function(n, mixture.param){
    mu <- rep(0, mixture.param$p)
    sigma <- mixture.param$sigma
    return(mvtnorm::rmvnorm(n, mu, sigma))
}
a <- batch.estimation(seed, batch.size, mixture.param, eps, cv = FALSE,
fname = "f", rpname = "rp", rqname = "rq", dpname = "dp", dqname = "dq")
}
\references{
Boyd, S. and Vandeberghe, L. (2004). \emph{Convex optimization.} Cambridge University Press, Cambridge
}

