\name{labels}

\alias{labels.default}
\alias{labels.data.frame}
\alias{labels<-}

\alias{as.labeled.data.frame}
\alias{as.labelled.data.frame}
\alias{is.labeled.data.frame}
\alias{is.labelled.data.frame}
\alias{[.labeled.data.frame}
\alias{subset.labeled.data.frame}
\alias{cbind.labeled.data.frame}
\alias{rbind.labeled.data.frame}
\alias{plot.labeled.data.frame}

\title{
  Extract and Set Labels from / to Data Frames
}
\description{
  Labels can be stored to an attribute \code{"variable.labels"} using
  the assignment function. With the extractor function one can assess
  these labels. Usually, these labels are generated by
  \code{\link{read.spss}} in package \pkg{foreign}.
}
\usage{

\method{labels}{data.frame}(object, which = NULL, abbreviate = FALSE, ...)

## assign labels
labels(data, which = NULL) <- value

## check if data.frame is a special labeled.data.frame
is.labeled.data.frame(object)
## set as.labeled.data.frame
as.labeled.data.frame(object, ...)

## special functions for labeled.data.frame objects that keep the labels
\method{[}{labeled.data.frame}(x, ..., drop = TRUE)
\method{subset}{labeled.data.frame}(x, ...)
\method{cbind}{labeled.data.frame}(..., deparse.level = 1)
\method{rbind}{labeled.data.frame}(..., deparse.level = 1)

## special plotting function for labeled.data.frame objects
\method{plot}{labeled.data.frame}(data, variables = names(data),
     labels = TRUE, group = NULL, ...)
}

\arguments{
  \item{object}{
    a \code{data.frame} or \code{labeled.data.frame}. The former is
    usually a result from \code{\link{read.spss}} in package
    \pkg{foreign}, the latter results from adding labels in R or from a
    call to \code{as.labeled.data.frame}.
  }
  \item{data}{
    a \code{data.frame} or a \code{labeled.data.frame}, where labels
    should be added or altered.
  }
  \item{which}{
    either a number indicating the label to extract or a character
    string with the \emph{variable name} for which the label should be
    extracted. One can also use a vector of numerics or character
    strings to extract mutiple labels. If \code{which} is \code{NULL}
    (default), all labels are returned.
  }
  \item{value}{
    a vector containing the labels (in the order of the variables). If
    which is given, only the corresponding subset is labeled. Note that
    all other labels contain the variable name as label afterwards.
  }
  \item{abbreviate}{
    logical (default: \code{FALSE}). If \code{TRUE} variable labels are
    abbreviated such that they remain unique. See
    \code{\link{abbreviate}} for details. Further arguments to
    \code{\link{abbreviate}} can be specified (see below).
  }
  \item{\dots}{
    further options passed to function \code{\link{abbreviate}} if argument
    \code{abbreviate = TRUE}.

    In \code{x[...]}, \dots can be used to specify indices for
    extraction. See \code{\link{[}} for details.

    In \code{plot}, \code{\dots} can be used to specify further graphial
    parameters.
  }
  \item{x}{
    a \code{labeled.data.frame}.
  }
  \item{drop}{
    logical (default: TRUE). If \code{TRUE} the result is coerced to
    the lowest possible dimension (i.e. a vector in case of a single
    column) and labels might be dropped in this case.
  }
  \item{deparse.level}{
    see \code{\link{cbind}}.
  }
  \item{variables}{
    character vector or numeric vector defining (continuous) variables
    that should be included in the table. Per default, all numeric and
    factor variables of \code{data} are used.
  }
  \item{labels}{
    labels for the variables. If \code{labels = TRUE} (the default),
    \code{labels(data, which = variables)} is used as labels. If
    \code{labels = NULL} \code{variables} is used as label.
    \code{labels} can also be specified as character vector.
  }
  \item{group}{
    character or numeric specifying a grouping factor. Per default no
    grouping is applied.
  }
}
\details{
  One can set or extract labels from \code{\link{data.frame}} objects.
  If no labels are specified \code{labels(data)} returns the column
  names of the data frame. If labels are set (attached to a
  \code{data.frame}) the \code{data.frame} gets a special class
  \code{labeled.data.frame} with specific subset and combination
  functions.

  Using \code{abbreviate = TRUE}, all labels are abbreviated to (at
  least) 4 characters such that they are unique. Other minimal lengths
  can specified by setting \code{minlength} (see examples below).
}
\value{
  \code{labels(data)} returns a named vector of variable labels, where
  the names match the variable names and the values represent the labels.
}

\author{
  Benjamin Hofner
}
\note{
  If you import data using \code{\link{read.spss}}, \code{labels} are
  set but the \code{data.frame} is not coerced to a
  \code{labeled.data.frame}. Use \code{as.labeled.data.frame} in this
  case or do any manipulation of labels using the asignment function
  (see examples below).
}
\seealso{
  \code{\link{read.spss}} in package \pkg{foreign}
}
\examples{
############################################################
### Basic labels manipulations

data <- data.frame(a = 1:10, b = 10:1, c = rep(1:2, 5))
labels(data)  ## only the variable names
is.labeled.data.frame(data) ## not yet

## now set labels
labels(data) <- c("my_a", "my_b", "my_c")
## one gets a named character vector of labels
labels(data)
## data is now a labeled.data.frame:
is.labeled.data.frame(data)

## Altervatively one could use as.labeled.data.frame(data);
## This would keep the default labels but set the class
## correctly.

## set labels for a and b only
## Note that which represents the variable names!
labels(data, which = c("a", "b")) <- c("x", "y")
labels(data)

## reset labels (to variable names):
labels(data) <- NULL
labels(data)

## set label for a only and use default for other labels:
labels(data, which = "a") <- "x"
labels(data)

## attach label for new variable:
data2 <- data
data2$z <- as.factor(rep(2:3, each = 5))
labels(data2)  ## no real label for z, only variable name
labels(data2, which = "z") <- "new_label"
labels(data2)


############################################################
### Abbreviate labels

## attach long labels to data
labels(data) <- c("This is a long label", "This is another long label",
                  "This also")
labels(data)
labels(data, abbreviate = TRUE, minlength = 10)


############################################################
### Data manipulations

## reorder dataset:
tmp <- data2[, c(1, 4, 3, 2)]
labels(tmp)
## labels are kept and order is updated
## (but only if data.set has class "labeled.data.frame")

## subsetting to single variables:
labels(tmp[, 2])  ## label got lost as tmp drops to vector
labels(tmp[, 2, drop = FALSE]) ## prevent dropping labels

## one can also cbind labeled.data.set objects:
labels(cbind(data, tmp[, 2, drop = TRUE]))
## or better:
labels(cbind(data, tmp[, 2, drop = FALSE]))
## or rbind labeled.data.set objects:
labels(rbind(data, tmp[, -2]))


############################################################
### Plotting labeled.data.set objects

## plot the data auto"magically"; numerics as boxplot, factors as barplots
par(mfrow = c(2,2))
plot(data2)

## a single plot
plot(data2, variables = "a")
## grouped plot
plot(data2, variables = "a", group = "z")
## make "c" a factor and plot "c" vs. "z"
data2$c <- as.factor(data2$c)
plot(data2, variables = "c", group = "z")
## the same
plot(data2, variables = 3, group = 4)
}

\keyword{methods}
