% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/exports.R
\name{start_backend}
\alias{start_backend}
\title{Start a Backend}
\usage{
start_backend(cores, cluster_type = "psock", backend_type = "async")
}
\arguments{
\item{cores}{A positive integer representing the number of cores to use
(i.e., the number of processes to start). This value must be between \code{2} and
\code{parallel::detectCores() - 1}.}

\item{cluster_type}{A character string representing the type of cluster to
create. Possible values are \code{"fork"} and \code{"psock"}. Defaults to \code{"psock"}.
See the section \strong{Cluster Type} for more information.}

\item{backend_type}{A character string representing the type of backend to
create. Possible values are \code{"sync"} and \code{"async"}. Defaults to \code{"async"}.
See the section \strong{Backend Type} for more information.}
}
\value{
A \code{\link{Backend}} instance that can be used to parallelize computations.
The methods available on the \code{\link{Backend}} instance are defined by the
\code{\link{Service}} interface.
}
\description{
This function can be used to start a backend. Check out the \strong{Details}
section for more information.
}
\details{
This function is a convenience wrapper around the lower-lever API of
\code{\link{parabar}} aimed at developers. More specifically, this function
uses the \code{\link{Specification}} class to create a specification object,
and the \code{\link{BackendFactory}} class to create a \code{\link{Backend}}
instance based on the specification object.
}
\section{Cluster Type}{

The cluster type determines the type of cluster to create. The requested
value is validated and passed to the \code{type} argument of the
\code{\link[parallel:makeCluster]{parallel::makeCluster()}} function. The following table lists the possible
values and their corresponding description.\tabular{ll}{
   \strong{Cluster} \tab \strong{Description} \cr
   \code{"fork"} \tab For Unix-based systems. \cr
   \code{"psock"} \tab For Windows-based systems. \cr
}
}

\section{Backend Type}{

The backend type determines the type of backend to create. The requested
value is passed to the \code{\link{BackendFactory}} class, which returns a
\code{\link{Backend}} instance of the desired type. The following table lists
the possible backend types and their corresponding description.\tabular{lllc}{
   \strong{Backend} \tab \strong{Description} \tab \strong{Implementation} \tab \strong{Progress} \cr
   \code{"sync"} \tab A synchronous backend. \tab \code{\link{SyncBackend}} \tab no \cr
   \code{"async"} \tab An asynchronous backend. \tab \code{\link{AsyncBackend}} \tab yes \cr
}


In a nutshell, the difference between the two backend types is that for the
synchronous backend the cluster is created in the main process, while for the
asynchronous backend the cluster is created in a backend \code{R} process using
\code{\link[callr:r_session]{callr::r_session}}. Therefore, the synchronous backend is blocking the main
process during task execution, while the asynchronous backend is
non-blocking. Check out the implementations listed in the table above for
more information. All concrete implementations extend the
\code{\link{Backend}} abstract class and implement the \code{\link{Service}}
interface.
}

\examples{
# Create an asynchronous backend.
backend <- start_backend(cores = 2, cluster_type = "psock", backend_type = "async")

# Check that the backend is active.
backend$active

# Check if there is anything on the backend.
peek(backend)

# Create a dummy variable.
name <- "parabar"

# Export the `name` variable in the current environment to the backend.
export(backend, "name", environment())

# Remove the dummy variable from the current environment.
rm(name)

# Check the backend to see that the variable has been exported.
peek(backend)

# Run an expression on the backend.
# Note that the symbols in the expression are resolved on the backend.
evaluate(backend, {
    # Print the name.
    print(paste0("Hello, ", name, "!"))
})

# Clear the backend.
clear(backend)

# Check that there is nothing on the backend.
peek(backend)

# Use a basic progress bar (i.e., see `parabar::Bar`).
configure_bar(type = "basic", style = 3)

# Run a task in parallel (i.e., approx. 1.25 seconds).
output <- par_sapply(backend, x = 1:10, fun = function(x) {
    # Sleep a bit.
    Sys.sleep(0.25)

    # Compute and return.
    return(x + 1)
})

# Print the output.
print(output)

# Stop the backend.
stop_backend(backend)

# Check that the backend is not active.
backend$active

}
\seealso{
\code{\link[=peek]{peek()}}, \code{\link[=export]{export()}}, \code{\link[=evaluate]{evaluate()}},
\code{\link[=clear]{clear()}}, \code{\link[=configure_bar]{configure_bar()}}, \code{\link[=par_sapply]{par_sapply()}},
\code{\link[=par_lapply]{par_lapply()}}, \code{\link[=par_apply]{par_apply()}}, \code{\link[=stop_backend]{stop_backend()}},
and \code{\link{Service}}.
}
