% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/n_clusters.R, R/n_clusters_easystats.R
\name{n_clusters}
\alias{n_clusters}
\alias{n_clusters_elbow}
\alias{n_clusters_gap}
\alias{n_clusters_silhouette}
\alias{n_clusters_dbscan}
\alias{n_clusters_hclust}
\title{Find number of clusters in your data}
\usage{
n_clusters(
  x,
  standardize = TRUE,
  include_factors = FALSE,
  package = c("easystats", "NbClust", "mclust"),
  fast = TRUE,
  nbclust_method = "kmeans",
  n_max = 10,
  ...
)

n_clusters_elbow(
  x,
  standardize = TRUE,
  include_factors = FALSE,
  clustering_function = stats::kmeans,
  n_max = 10,
  ...
)

n_clusters_gap(
  x,
  standardize = TRUE,
  include_factors = FALSE,
  clustering_function = stats::kmeans,
  n_max = 10,
  gap_method = "firstSEmax",
  ...
)

n_clusters_silhouette(
  x,
  standardize = TRUE,
  include_factors = FALSE,
  clustering_function = stats::kmeans,
  n_max = 10,
  ...
)

n_clusters_dbscan(
  x,
  standardize = TRUE,
  include_factors = FALSE,
  method = c("kNN", "SS"),
  min_size = 0.1,
  eps_n = 50,
  eps_range = c(0.1, 3),
  ...
)

n_clusters_hclust(
  x,
  standardize = TRUE,
  include_factors = FALSE,
  distance_method = "correlation",
  hclust_method = "average",
  ci = 0.95,
  iterations = 100,
  ...
)
}
\arguments{
\item{x}{A data frame.}

\item{standardize}{Standardize the dataframe before clustering (default).}

\item{include_factors}{Logical, if \code{TRUE}, factors are converted to numerical
values in order to be included in the data for determining the number of
clusters. By default, factors are removed, because most methods that
determine the number of clusters need numeric input only.}

\item{package}{Package from which methods are to be called to determine the
number of clusters. Can be \code{"all"} or a vector containing
\code{"easystats"}, \code{"NbClust"}, \code{"mclust"}, and \code{"M3C"}.}

\item{fast}{If \code{FALSE}, will compute 4 more indices (sets \code{index = "allong"}
in \code{NbClust}). This has been deactivated by default as it is
computationally heavy.}

\item{nbclust_method}{The clustering method (passed to \code{NbClust::NbClust()}
as \code{method}).}

\item{n_max}{Maximal number of clusters to test.}

\item{...}{Arguments passed to or from other methods.}

\item{clustering_function, gap_method}{Other arguments passed to other
functions. \code{clustering_function} is used by \code{fviz_nbclust} and
can be \code{kmeans}, code{cluster::pam}, code{cluster::clara},
code{cluster::fanny}, and more. \code{gap_method} is used by
\code{cluster::maxSE} to extract the optimal numbers of clusters (see its
\code{method} argument).}

\item{method, min_size, eps_n, eps_range}{Arguments for DBSCAN algorithm.}

\item{distance_method}{The distance method (passed to \code{\link[=dist]{dist()}}). Used by
algorithms relying on the distance matrix, such as \code{hclust} or \code{dbscan}.}

\item{hclust_method}{The hierarchical clustering method (passed to \code{\link[=hclust]{hclust()}}).}

\item{ci}{Confidence Interval (CI) level. Default to \code{0.95} (\verb{95\%}).}

\item{iterations}{The number of bootstrap replicates. This only apply in the
case of bootstrapped frequentist models.}
}
\description{
Similarly to \code{\link[=n_factors]{n_factors()}} for factor / principal component analysis,
\code{n_clusters} is the main function to find out the optimal numbers of clusters
present in the data based on the maximum consensus of a large number of
methods.
\cr
Essentially, there exist many methods to determine the optimal number of
clusters, each with pros and cons, benefits and limitations. The main
\code{n_clusters} function proposes to run all of them, and find out the number of
clusters that is suggested by the majority of methods (in case of ties, it
will select the most parsimonious solution with fewer clusters).
\cr
Note that we also implement some specific, commonly used methods, like the
Elbow or the Gap method, with their own visualization functionalities. See
the examples below for more details.
}
\note{
There is also a \href{https://easystats.github.io/see/articles/parameters.html}{\code{plot()}-method} implemented in the \href{https://easystats.github.io/see/}{\pkg{see}-package}.
}
\examples{
\dontrun{
library(parameters)

# The main 'n_clusters' function ===============================
if (require("mclust", quietly = TRUE) && require("NbClust", quietly = TRUE) &&
  require("cluster", quietly = TRUE) && require("see", quietly = TRUE)) {
  n <- n_clusters(iris[, 1:4], package = c("NbClust", "mclust")) # package can be "all"
  n
  summary(n)
  as.data.frame(n)  # Duration is the time elapsed for each method in seconds
  plot(n)

  # The following runs all the method but it significantly slower
  # n_clusters(iris[1:4], standardize = FALSE, package = "all", fast = FALSE)
}
}
\donttest{
#
# Specific Methods =========================
# Elbow method --------------------
if (require("openxlsx", quietly = TRUE) &&
  require("see", quietly = TRUE) &&
  require("factoextra", quietly = TRUE)) {
  x <- n_clusters_elbow(iris[1:4])
  x
  as.data.frame(x)
  plot(x)
}
}
\donttest{
#
# Gap method --------------------
if (require("see", quietly = TRUE) &&
  require("cluster", quietly = TRUE) &&
  require("factoextra", quietly = TRUE)) {
  x <- n_clusters_gap(iris[1:4])
  x
  as.data.frame(x)
  plot(x)
}
}
\donttest{
#
# Silhouette method --------------------------
if (require("factoextra", quietly = TRUE)) {
  x <- n_clusters_silhouette(iris[1:4])
  x
  as.data.frame(x)
  plot(x)
}
}
\donttest{
#
if (require("dbscan", quietly = TRUE)) {
  # DBSCAN method -------------------------
  # NOTE: This actually primarily estimates the 'eps' parameter, the number of
  # clusters is a side effect (it's the number of clusters corresponding to
  # this 'optimal' EPS parameter).
  x <- n_clusters_dbscan(iris[1:4], method = "kNN", min_size = 0.05) # 5 percent
  x
  head(as.data.frame(x))
  plot(x)

  x <- n_clusters_dbscan(iris[1:4], method = "SS", eps_n = 100, eps_range = c(0.1, 2))
  x
  head(as.data.frame(x))
  plot(x)
}
}
\donttest{
#
# hclust method -------------------------------
if (require("pvclust", quietly = TRUE) &&
  getRversion() >= "3.6.0") {
  # iterations should be higher for real analyses
  x <- n_clusters_hclust(iris[1:4], iterations = 50, ci = 0.90)
  x
  head(as.data.frame(x), n = 10) # Print 10 first rows
  plot(x)
}
}
}
