% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/IBDestimate.R
\name{IBDestimate}
\alias{IBDestimate}
\title{Relatedness estimation}
\usage{
IBDestimate(x, ids, markers = NULL, start = c(0.99, 0.001), tol = 1e-07)
}
\arguments{
\item{x}{A single \code{linkdat} object or a list of \code{linkdat} and/or
\code{singleton} objects.}

\item{ids}{Either a vector of length 2, or a matrix with two columns, indicating the 
the pair(s) of individuals for which IBD estimates should be computed. If a matrix, 
each row corresponds to a pair. The entries can be either characters (matching the 
\code{plot.labels} of the linkdat object(s)) or integers (matching the \code{orig.ids}
identifiers of the linkdat object(s)).}

\item{markers}{A numeric indicating which marker(s) to include. If NULL (default),
all markers are used.}

\item{start}{Numeric of length 2, indicating the initial value of \eqn{(\kappa_0, \kappa_2)}
in the optimisation (passed on to \code{maxLik}).}

\item{tol}{A single numeric: the optimising tolerance value; passed on to \code{maxLik}).}
}
\value{
A data.frame with 8 columns: ID1, ID2 (numeric IDs), Name1, Name2 (plot labels, if present), N
(#markers with no missing alleles), \eqn{\kappa_0}, \eqn{\kappa_1}, \eqn{\kappa_2}.
}
\description{
Estimate the pairwise IBD coefficients \eqn{(\kappa_0, \kappa_1, \kappa_2)} for
specified pairs of pedigree members, using maximum likelihood methods.
The optimization machinery is imported from the \code{maxLik} package.
}
\details{
This function optimises the log-likelihood function first described in (Thompson, 1975).
Optimisation is done in the \eqn{(\kappa_0, \kappa_2)}-plane and restricted to the 
probability triangle defined by \eqn{\kappa_0 \ge 0, \kappa_2 \ge 0, \kappa_0 + \kappa_2 \le 1}.
}
\examples{

# Simulate marker data for two siblings
x = nuclearPed(2)
x = setPlotLabels(x, c("Sib1", "Sib2"), c(3,4))
x = simpleSim(x, 200, 1:2) # 200 equifrequent SNPs

# Estimate the IBD coefficients for the siblings
est1 = IBDestimate(x, ids=c(3,4))

# Estimate should be the same if pedigree structure is unknown
xlist = list(branch(x, 3), branch(x, 4))
est2 = IBDestimate(xlist, ids=c(3,4)) 
stopifnot(identical(est1, est2))

# If the pedigree has plot.labels, they can be used as IDs
est3 = IBDestimate(x, ids=c("Sib1", "Sib2"))
stopifnot(identical(est1, est3))

}
\references{
E. A. Thompson (2000). \emph{Statistical Inferences from Genetic Data
 on Pedigrees.} NSF-CBMS Regional Conference Series in Probability and
 Statistics. Volume 6.
}
\seealso{
\code{\link{examineKinships}}, 
 \code{\link{IBDtriangle}}, \code{\link[maxLik]{maxLik}}
}
\author{
Magnus Dehli Vigeland
}
