% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/create_informant.R
\name{create_informant}
\alias{create_informant}
\title{Create a \strong{pointblank} \emph{informant} object}
\usage{
create_informant(
  tbl = NULL,
  read_fn = NULL,
  agent = NULL,
  tbl_name = NULL,
  label = NULL,
  lang = NULL,
  locale = NULL
)
}
\arguments{
\item{tbl}{The input table. This can be a data frame, a tibble, a \code{tbl_dbi}
object, or a \code{tbl_spark} object. Alternatively, a function can be used to
read in the input data table with the \code{read_fn} argument (in which case,
\code{tbl} can be \code{NULL}).}

\item{read_fn}{A function that's used for reading in the data. Even if a
\code{tbl} is provided, this function will be invoked to obtain the data (i.e.,
the \code{read_fn} takes priority). There are two ways to specify a \code{read_fn}:
(1) using a function (e.g., \verb{function() \{ <table reading code> \}}) or, (2)
with an R formula expression.}

\item{agent}{A pointblank \emph{agent} object. This object can be used instead of
supplying a table in \code{tbl} or a table-prep formula in \code{read_fn}.}

\item{tbl_name}{A optional name to assign to the input table object. If no
value is provided, a name will be generated based on whatever information
is available.}

\item{label}{An optional label for the information report. If no value is
provided, a label will be generated based on the current system time.
Markdown can be used here to make the label more visually appealing (it
will appear in the header area of the information report).}

\item{lang}{The language to use for the information report (a summary table
that provides all of the available information for the table. By default,
\code{NULL} will create English (\code{"en"}) text. Other options include French
(\code{"fr"}), German (\code{"de"}), Italian (\code{"it"}), Spanish (\code{"es"}), Portuguese,
(\code{"pt"}), Chinese (\code{"zh"}), and Russian (\code{"ru"}).}

\item{locale}{An optional locale ID to use for formatting values in the
information report according the locale's rules. Examples include \code{"en_US"}
for English (United States) and \code{"fr_FR"} for French (France); more simply,
this can be a language identifier without a country designation, like "es"
for Spanish (Spain, same as \code{"es_ES"}).}
}
\value{
A \code{ptblank_informant} object.
}
\description{
The \code{create_informant()} function creates an \emph{informant} object, which is
used in an \emph{information management} workflow. The overall aim of this
workflow is to record, collect, and generate useful information on data
tables. We can supply as information that is useful for describing a
particular data table. The \emph{informant} object created by the
\code{create_informant()} function takes information-focused functions:
\code{\link[=info_columns]{info_columns()}}, \code{\link[=info_tabular]{info_tabular()}}, \code{\link[=info_section]{info_section()}}, and \code{\link[=info_snippet]{info_snippet()}}.

The \verb{info_*()} series of functions allows for a progressive build up of
information about the target table. The \code{\link[=info_columns]{info_columns()}} and \code{\link[=info_tabular]{info_tabular()}}
functions facilitate the entry of \emph{info text} that concerns the table columns
and the table proper; the \code{\link[=info_section]{info_section()}} function allows for the creation
of arbitrary sections that can have multiple subsections full of additional
\emph{info text}. The system allows for dynamic values culled from the target
table by way of \code{\link[=info_snippet]{info_snippet()}}, for getting named text extracts from
queries, and the use of \verb{\{<snippet_name>\}} in the \emph{info text}. To make the
use of \code{\link[=info_snippet]{info_snippet()}} more convenient for common queries, a set of
\verb{snip_*()} functions are provided in the package (\code{\link[=snip_list]{snip_list()}},
\code{\link[=snip_stats]{snip_stats()}}, \code{\link[=snip_lowest]{snip_lowest()}}, and \code{\link[=snip_highest]{snip_highest()}}) though you are free to
use your own expressions.

Because snippets need to query the target table to return fragments of \emph{info
text}, the \code{\link[=incorporate]{incorporate()}} function needs to be used to initiate this action.
This is also necessary for the \emph{informant} to update other metadata elements
such as row and column counts. Once the incorporation process is complete,
snippets and other metadata will be updated. Calling the \emph{informant} itself
will result in a reporting table. This reporting can also be accessed with
the \code{\link[=get_informant_report]{get_informant_report()}} function, where there are more reporting
options.
}
\section{YAML}{

A \strong{pointblank} informant can be written to YAML with \code{\link[=yaml_write]{yaml_write()}} and the
resulting YAML can be used to regenerate an informant (with
\code{\link[=yaml_read_informant]{yaml_read_informant()}}) or perform the 'incorporate' action using the target
table (via \code{\link[=yaml_informant_incorporate]{yaml_informant_incorporate()}}). Here is an example of how a
complex call of \code{create_informant()} is expressed in R code and in the
corresponding YAML representation.\preformatted{# R statement
create_informant(
  read_fn = ~ small_table,
  tbl_name = "small_table",
  label = "An example.",
  lang = "fr", 
  locale = "fr_CA"
)

# YAML representation
type: informant
read_fn: ~small_table
tbl_name: small_table
info_label: An example.
lang: fr
locale: fr_CA
table:
  name: small_table
  _columns: 8
  _rows: 13.0
  _type: tbl_df
columns:
  date_time:
    _type: POSIXct, POSIXt
  date:
    _type: Date
  a:
    _type: integer
  b:
    _type: character
  c:
    _type: numeric
  d:
    _type: numeric
  e:
    _type: logical
  f:
    _type: character
}

The generated YAML includes some top-level keys where \code{type} and \code{read_fn}
are mandatory, and, two metadata sections: \code{table} and \code{columns}. Keys that
begin with an underscore character are those that are updated whenever
\code{\link[=incorporate]{incorporate()}} is called on an \emph{informant}. The \code{table} metadata section can
have multiple subsections with \emph{info text}. The \code{columns} metadata section
can similarly have have multiple subsections, so long as they are children to
each of the column keys (in the above YAML example, \code{date_time} and \code{date}
are column keys and they match the table's column names). Additional sections
can be added but they must have key names on the top level that don't
duplicate the default set (i.e., \code{type}, \code{table}, \code{columns}, etc. are treated
as reserved keys).
}

\section{Writing an Informant to Disk}{

An \emph{informant} object can be written to disk with the \code{\link[=x_write_disk]{x_write_disk()}}
function. Informants are stored in the serialized RDS format and can be
easily retrieved with the \code{\link[=x_read_disk]{x_read_disk()}} function.

It's recommended that table-prep formulas are supplied to the \code{read_fn}
argument of \code{create_informant()}. In this way, when an \emph{informant} is read
from disk through \code{\link[=x_read_disk]{x_read_disk()}}, it can be reused to access the target
table (which may changed, hence the need to use an expression for this).
}

\section{Figures}{

\if{html}{\figure{man_create_informant_1.png}{options: width=100\%}}
}

\section{Function ID}{

1-3
}

\examples{
# Let's walk through how we can
# generate some useful information for a
# really small table; it's actually
# called `small_table` and we can find
# it as a dataset in this package
small_table

# Create a pointblank `informant`
# object with `create_informant()`
# and the `small_table` dataset
informant <- 
  create_informant(
    read_fn = ~small_table,
    tbl_name = "small_table",
    label = "An example."
  )

# This function creates some information
# without any extra help by profiling
# the supplied table object; it adds
# the sections: (1) 'table' and
# (2) 'columns' and we can print the
# object to see the information report

# Alternatively, we can get the same report
# by using `get_informant_report()`
report <- get_informant_report(informant)
class(report)

}
\seealso{
Other Planning and Prep: 
\code{\link{action_levels}()},
\code{\link{create_agent}()},
\code{\link{db_tbl}()},
\code{\link{file_tbl}()},
\code{\link{scan_data}()},
\code{\link{tbl_get}()},
\code{\link{tbl_source}()},
\code{\link{tbl_store}()},
\code{\link{validate_rmd}()}
}
\concept{Planning and Prep}
