% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kernels.R
\name{kernel}
\alias{kernel}
\alias{L}
\alias{c_kern}
\alias{grad_L}
\alias{hess_L}
\title{Kernels on the hypersphere and their derivatives}
\usage{
L(t, kernel = "1", squared = FALSE, deriv = 0, k = 10,
  inc_sfp = TRUE)

c_kern(h, d, kernel = "1", kernel_type = "1", k = 10, log = FALSE,
  inc_sfp = TRUE, intrinsic = FALSE)

grad_L(x, y, h, kernel = 1, k = 10)

hess_L(x, y, h, kernel = 1, k = 10)
}
\arguments{
\item{t}{vector with the evaluation points.}

\item{kernel}{kernel employed: \code{1} for von Mises--Fisher (default);
\code{2} for Epanechnikov; \code{3} for softplus.}

\item{squared}{square the kernel? Only for \code{deriv = 0}. Defaults to
\code{FALSE}.}

\item{deriv}{kernel derivative. Must be \code{0}, \code{1}, or \code{2}.
Defaults to \code{0}.}

\item{k}{softplus kernel parameter. Defaults to \code{10.0}.}

\item{inc_sfp}{include \code{softplus(k)} in the constant? Defaults to
\code{TRUE}.}

\item{h}{vector of size \code{r} with bandwidths.}

\item{d}{vector of size \code{r} with dimensions.}

\item{kernel_type}{type of kernel employed: \code{1} for product kernel
(default); \code{2} for spherically symmetric kernel.}

\item{log}{compute the logarithm of the constant? Defaults to \code{FALSE}.}

\item{intrinsic}{consider the intrinsic distance? Defaults to \code{FALSE}.}

\item{x}{a matrix of size \code{c(nx, sum(d) + r)} with the evaluation
points.}

\item{y}{center of the kernel, a vector of size \code{sum(d) + r}.}
}
\value{
\itemize{
\item{\code{L}: a vector with the kernel evaluated at \code{t}.}
\item{\code{grad_L}: a vector with the gradient evaluated at \code{x}.}
\item{\code{hess_L}: a matrix with the Hessian evaluated at \code{x}.}
}
}
\description{
An isotropic kernel \eqn{L} on \eqn{\mathcal{S}^d} and its
normalizing constant are such that \eqn{\int_{\mathcal{S}^d} c(h, d, L)
L\left(\frac{1 - \boldsymbol{x}'\boldsymbol{y}}{h^2}\right)
\,\mathrm{d}\boldsymbol{x} = 1} (extrinsic-chordal distance) or
\eqn{\int_{\mathcal{S}^d} c(h, d, L)
L\left(\frac{\cos^{-1}(\boldsymbol{x}'\boldsymbol{y})^2}{2h^2}\right)
\,\mathrm{d}\boldsymbol{x} = 1} (intrinsic distance).
}
\details{
The gradient and Hessian are computed for the functions
\eqn{\boldsymbol{x} \mapsto
L\left(\frac{1 - \boldsymbol{x}'\boldsymbol{y}}{h^2}\right)}.
}
\examples{
# Constants in terms of h
h_grid <- seq(0.01, 4, l = 100)
r <- 2
d <- 2
dr <- rep(d, r)
c_vmf <- sapply(h_grid, function(hi)
  log(c_kern(h = rep(hi, r), d = dr, kernel = 1, kernel_type = 2)))
c_epa <- sapply(h_grid, function(hi)
  log(c_kern(h = rep(hi, r), d = dr, kernel = 2, kernel_type = 2)))
c_sfp <- sapply(h_grid, function(hi)
  log(c_kern(h = rep(hi, r), d = dr, kernel = 3, k = 1, kernel_type = 2)))
plot(h_grid, c_epa, type = "l", ylab = "Constant", xlab = "h", col = 2)
lines(h_grid, c_sfp, col = 3)
lines(h_grid, c_vmf, col = 1)
abline(v = sqrt(2), lty = 2, col = 2)

# Kernel and its derivatives
h <- 0.5
x <- c(sqrt(2), -sqrt(2), 0) / 2
y <- c(-sqrt(2), sqrt(3), sqrt(3)) / 3
L(t = (1 - sum(x * y)) / h^2)
grad_L(x = x, y = y, h = h)
hess_L(x = x, y = y, h = h)
}
