############################################################################################
## Tetraploid likelihood, LOD and recombination frequency functions
## Assumes preferential bivalent pairing. 
## Peter Bourke, Wageningen UR Plant Breeding. December 2016
############################################################################################

#' Calculate recombination frequency, LOD and log-likelihood from frequency tables in a preferential pairing tetraploid
#' @description This group of functions is called by \code{\link{linkage}}.
#' @param x A frequency table of the different classes of dosages in the progeny. The column names start with \code{"n_"}. Followed by the dosage of the first marker and then of the second.
#' @param p1 Preferential pairing parameter for parent 1, numeric value in range 0 <= p1 < 2/3
#' @param p2 Preferential pairing parameter for parent 2, numeric value in range 0 <= p2 < 2/3
#' @param ncores Number of cores to use for parallel processing (deprecated).
#' @return
#' A list with the following items:
#' \item{r_mat}{A matrix with recombination frequencies for the different phases}
#' \item{LOD_mat}{A matrix with LOD scores for the different phases}
#' \item{logL_mat}{A matrix with log likelihood ratios for the different phases}
#' \item{phasing_strategy}{A character string specifying the phasing strategy. \code{"MLL"} for maximum likelihood en \code{"MINR"} for minimum recombination frequency.}
#' \item{possible_phases}{The phases between markers that are possible. Same order and length as column names of output matrices.}
#' @name p4_functions
NULL

#' @rdname p4_functions
p4_1.0_1.0 <- function(x,p1,p2,ncores=1){


r_c <- (x[,"n_01"] + x[,"n_10"])/(x[,"n_00"] + x[,"n_01"] + x[,"n_10"] + x[,"n_11"])
logL_c <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_10"] - x[,"n_11"])*log(2) + (x[,"n_00"] + x[,"n_11"])*log(pmax(1e-6,1 - r_c)) + (x[,"n_01"] + x[,"n_10"])*log(pmax(1e-6,r_c))
LOD_c <- (x[,"n_00"] + x[,"n_01"] + x[,"n_10"] + x[,"n_11"])*log10(2) + (x[,"n_00"] + x[,"n_11"])*log10(pmax(1e-6,1 - r_c)) + (x[,"n_01"] + x[,"n_10"])*log10(pmax(1e-6,r_c))


r_r1 <- (4*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_10"] + 4*x[,"n_11"] + 3*x[,"n_00"]*p1 + 3*x[,"n_01"]*p1 + 3*x[,"n_10"]*p1 + 3*x[,"n_11"]*p1)/(2*(x[,"n_00"] + x[,"n_01"] + x[,"n_10"] + x[,"n_11"])*(1 + 3*p1))
logL_r1 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_10"] - x[,"n_11"])*(2*log(2) + log(3)) + (x[,"n_01"] + x[,"n_10"])*log(4 + p1*(3 - 6*r_r1) - 2*r_r1) + (x[,"n_00"] + x[,"n_11"])*log(2 - 3*p1 + 2*r_r1 + 6*p1*r_r1)
LOD_r1 <- -((x[,"n_00"] + x[,"n_01"] + x[,"n_10"] + x[,"n_11"])*log10(3)) + (x[,"n_01"] + x[,"n_10"])*log10(4 + p1*(3 - 6*r_r1) - 2*r_r1) + (x[,"n_00"] + x[,"n_11"])*log10(2 - 3*p1 + 2*r_r1 + 6*p1*r_r1)


r_r2 <- (-8*x[,"n_00"] + 4*x[,"n_01"] + 4*x[,"n_10"] - 8*x[,"n_11"] + 3*x[,"n_00"]*p1 + 3*x[,"n_01"]*p1 + 3*x[,"n_10"]*p1 + 3*x[,"n_11"]*p1)/(2*(x[,"n_00"] + x[,"n_01"] + x[,"n_10"] + x[,"n_11"])*(-2 + 3*p1))
logL_r2 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_10"] - x[,"n_11"])*(3*log(2) + log(3)) + (x[,"n_00"] + x[,"n_11"])*log(4 + 3*p1 + 4*r_r2 - 6*p1*r_r2) + (x[,"n_01"] + x[,"n_10"])*log(8 - 4*r_r2 + p1*(-3 + 6*r_r2))
LOD_r2 <- -((x[,"n_00"] + x[,"n_01"] + x[,"n_10"] + x[,"n_11"])*log10(6)) + (x[,"n_00"] + x[,"n_11"])*log10(4 + 3*p1 + 4*r_r2 - 6*p1*r_r2) + (x[,"n_01"] + x[,"n_10"])*log10(8 - 3*p1 - 4*r_r2 + 6*p1*r_r2)


return(list(
r_mat = cbind(r_c,r_r1,r_r2,0.499),
LOD_mat = cbind(LOD_c,LOD_r1,LOD_r2,0),
logL_mat = cbind(logL_c,logL_r1,logL_r2,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","repulsion","repulsion","unknown")
)
)
}

#' @rdname p4_functions
p4_1.0_1.1 <- function(x,p1,p2,ncores=1){


r_c <- (x[,"n_02"] + x[,"n_10"])/(x[,"n_00"] + x[,"n_02"] + x[,"n_10"] + x[,"n_12"])
logL_c <- 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"])*log(2) + (x[,"n_00"] + x[,"n_12"])*log(pmax(1e-6,1 - r_c)) + (x[,"n_02"] + x[,"n_10"])*log(pmax(1e-6,r_c))
LOD_c <- (x[,"n_00"] + x[,"n_02"] + x[,"n_10"] + x[,"n_12"])*log10(2) + (x[,"n_00"] + x[,"n_12"])*log10(pmax(1e-6,1 - r_c)) + (x[,"n_02"] + x[,"n_10"])*log10(pmax(1e-6,r_c))


r_r1 <- (4*x[,"n_00"] - 2*x[,"n_02"] - 2*x[,"n_10"] + 4*x[,"n_12"] + 3*x[,"n_00"]*p1 + 3*x[,"n_02"]*p1 + 3*x[,"n_10"]*p1 + 3*x[,"n_12"]*p1)/(2*(x[,"n_00"] + x[,"n_02"] + x[,"n_10"] + x[,"n_12"])*(1 + 3*p1))
logL_r1 <- (-3*x[,"n_00"] - 2*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_10"] - 2*x[,"n_11"] - 3*x[,"n_12"])*log(2) + (-x[,"n_00"] - x[,"n_02"] - x[,"n_10"] - x[,"n_12"])*log(3) + (x[,"n_02"] + x[,"n_10"])*log(4 + p1*(3 - 6*r_r1) - 2*r_r1) + (x[,"n_00"] + x[,"n_12"])*log(2 - 3*p1 + 2*r_r1 + 6*p1*r_r1)
LOD_r1 <- -((x[,"n_00"] + x[,"n_02"] + x[,"n_10"] + x[,"n_12"])*log10(3)) + (x[,"n_02"] + x[,"n_10"])*log10(4 + p1*(3 - 6*r_r1) - 2*r_r1) + (x[,"n_00"] + x[,"n_12"])*log10(2 - 3*p1 + 2*r_r1 + 6*p1*r_r1)


r_r2 <- (-8*x[,"n_00"] + 4*x[,"n_02"] + 4*x[,"n_10"] - 8*x[,"n_12"] + 3*x[,"n_00"]*p1 + 3*x[,"n_02"]*p1 + 3*x[,"n_10"]*p1 + 3*x[,"n_12"]*p1)/(2*(x[,"n_00"] + x[,"n_02"] + x[,"n_10"] + x[,"n_12"])*(-2 + 3*p1))
logL_r2 <- 2*(-2*x[,"n_00"] - x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_10"] - x[,"n_11"] - 2*x[,"n_12"])*log(2) + (-x[,"n_00"] - x[,"n_02"] - x[,"n_10"] - x[,"n_12"])*log(3) + (x[,"n_00"] + x[,"n_12"])*log(4 + 3*p1 + 4*r_r2 - 6*p1*r_r2) + (x[,"n_02"] + x[,"n_10"])*log(8 - 4*r_r2 + p1*(-3 + 6*r_r2))
LOD_r2 <- -((x[,"n_00"] + x[,"n_02"] + x[,"n_10"] + x[,"n_12"])*log10(6)) + (x[,"n_00"] + x[,"n_12"])*log10(4 + 3*p1 + 4*r_r2 - 6*p1*r_r2) + (x[,"n_02"] + x[,"n_10"])*log10(8 - 3*p1 - 4*r_r2 + 6*p1*r_r2)


return(list(
r_mat = cbind(r_c,r_r1,r_r2,0.499),
LOD_mat = cbind(LOD_c,LOD_r1,LOD_r2,0),
logL_mat = cbind(logL_c,logL_r1,logL_r2,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","repulsion","repulsion","unknown")
)
)
}

#' @rdname p4_functions
p4_1.0_1.2 <- function(x,p1,p2,ncores=1){


logL_c1 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13)*(3*log(2) + log(3)) + (n01 + n12)*log(8 + p2*(6 - 9*r) - 6*r) + (n00 + n13)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r))) + (n03 + n10)*(log(2 - 3*p2) + log(pmax(1e-6,r))) + (n02 + n11)*log(2 + 6*r + p2*(-3 + 9*r))
return(L)}
interlogL_c1 <- function(n00,n01,n02,n03,n10,n11,n12,n13,p1,p2) {
optimize(logL_c1,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_c1 <- parallel::mcmapply(interlogL_c1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"], p1, p2, mc.cores = ncores)


LOD_c1 <- -((x[,"n_03"] + x[,"n_10"])*log10(1 - (3*p2)/2)) - (x[,"n_00"] + x[,"n_13"])*log10(1 - (3*p2)/2) - (x[,"n_02"] + x[,"n_11"])*log10(5 + (3*p2)/2) - (x[,"n_01"] + x[,"n_12"])*log10(5 + (3*p2)/2) + (x[,"n_01"] + x[,"n_12"])*log10(8 + p2*(6 - 9*r_c1) - 6*r_c1) + (x[,"n_00"] + x[,"n_13"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_c1))) + (x[,"n_03"] + x[,"n_10"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_c1))) + (x[,"n_02"] + x[,"n_11"])*log10(2 + 6*r_c1 + p2*(-3 + 9*r_c1))


logL_c2 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13)*(4*log(2) + log(3)) + (n00 + n13)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r))) + (n03 + n10)*(log(4 + 3*p2) + log(pmax(1e-6,r))) + (n02 + n11)*log(4 + p2*(3 - 9*r) + 12*r) + (n01 + n12)*log(16 - 12*r + p2*(-6 + 9*r))
return(L)}
interlogL_c2 <- function(n00,n01,n02,n03,n10,n11,n12,n13,p1,p2) {
optimize(logL_c2,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_c2 <- parallel::mcmapply(interlogL_c2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"], p1, p2, mc.cores = ncores)


LOD_c2 <- -((x[,"n_02"] + x[,"n_11"])*log10(10 - (3*p2)/2)) - (x[,"n_01"] + x[,"n_12"])*log10(10 - (3*p2)/2) - (x[,"n_03"] + x[,"n_10"])*(-log10(2) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_13"])*(-log10(2) + log10(4 + 3*p2)) + (x[,"n_00"] + x[,"n_13"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_c2))) + (x[,"n_03"] + x[,"n_10"])*(log10(4 + 3*p2) + log10(pmax(1e-6,r_c2))) + (x[,"n_02"] + x[,"n_11"])*log10(4 + p2*(3 - 9*r_c2) + 12*r_c2) + (x[,"n_01"] + x[,"n_12"])*log10(16 - 12*r_c2 + p2*(-6 + 9*r_c2))


logL_r1 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,p1,p2) {
L <- 4*(-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13)*log(2) + (-2*n00 - n01 - n02 - 2*n03 - 2*n10 - n11 - n12 - 2*n13)*log(3) + (n03 + n10)*(log(2 - 3*p2) + log(4 + 3*p1 - 2*r - 6*p1*r)) + (n00 + n13)*(log(2 - 3*p2) + log(2 - 3*p1 + 2*r + 6*p1*r)) + (n02 + n11)*log(12 + 6*p2 - 4*r - 6*p2*r - 3*p1*(2 + 3*p2)*(-1 + 2*r)) + (n01 + n12)*log(8 + (4 + 6*p2)*r + 3*p1*(2 + 3*p2)*(-1 + 2*r))
return(L)}
interlogL_r1 <- function(n00,n01,n02,n03,n10,n11,n12,n13,p1,p2) {
optimize(logL_r1,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_r1 <- parallel::mcmapply(interlogL_r1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"], p1, p2, mc.cores = ncores)


LOD_r1 <- -((x[,"n_03"] + x[,"n_10"])*(log10(3) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_13"])*(log10(3) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_11"])*log10(10 + 3*p2) - (x[,"n_01"] + x[,"n_12"])*log10(8 + (4 + 6*p2)/2) + (x[,"n_03"] + x[,"n_10"])*(log10(2 - 3*p2) + log10(4 + 3*p1 - 2*r_r1 - 6*p1*r_r1)) + (x[,"n_00"] + x[,"n_13"])*(log10(2 - 3*p2) + log10(2 - 3*p1 + 2*r_r1 + 6*p1*r_r1)) + (x[,"n_02"] + x[,"n_11"])*log10(12 + 6*p2 - 4*r_r1 - 6*p2*r_r1 - 3*p1*(2 + 3*p2)*(-1 + 2*r_r1)) + (x[,"n_01"] + x[,"n_12"])*log10(8 + (4 + 6*p2)*r_r1 + 3*p1*(2 + 3*p2)*(-1 + 2*r_r1))


logL_r2 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,p1,p2) {
L <- 5*(-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13)*log(2) + (-2*n00 - n01 - n02 - 2*n03 - 2*n10 - n11 - n12 - 2*n13)*log(3) + (n02 + n11)*log(4*(6 - 3*p2*(-1 + r) - 2*r) + 3*p1*(2 + 3*p2)*(-1 + 2*r)) + (n00 + n13)*(log(2 - 3*p2) + log(4*(1 + r) - p1*(-3 + 6*r))) + (n03 + n10)*(log(2 - 3*p2) + log(8 - 4*r + p1*(-3 + 6*r))) + (n01 + n12)*log(-3*p1*(2 + 3*p2)*(-1 + 2*r) + 4*(4 + (2 + 3*p2)*r))
return(L)}
interlogL_r2 <- function(n00,n01,n02,n03,n10,n11,n12,n13,p1,p2) {
optimize(logL_r2,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_r2 <- parallel::mcmapply(interlogL_r2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"], p1, p2, mc.cores = ncores)


LOD_r2 <- -((x[,"n_03"] + x[,"n_10"])*(log10(2) + log10(3) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_13"])*(log10(2) + log10(3) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_11"])*(2*log10(2) + log10(5 + (3*p2)/2)) - (x[,"n_01"] + x[,"n_12"])*(2*log10(2) + log10(4 + (2 + 3*p2)/2)) + (x[,"n_02"] + x[,"n_11"])*log10(4*(6 - 3*p2*(-1 + r_r2) - 2*r_r2) + 3*p1*(2 + 3*p2)*(-1 + 2*r_r2)) + (x[,"n_00"] + x[,"n_13"])*(log10(2 - 3*p2) + log10(4*(1 + r_r2) - p1*(-3 + 6*r_r2))) + (x[,"n_03"] + x[,"n_10"])*(log10(2 - 3*p2) + log10(8 - 4*r_r2 + p1*(-3 + 6*r_r2))) + (x[,"n_01"] + x[,"n_12"])*log10(-3*p1*(2 + 3*p2)*(-1 + 2*r_r2) + 4*(4 + (2 + 3*p2)*r_r2))


logL_r3 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,p1,p2) {
L <- 5*(-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13)*log(2) + (-2*n00 - n01 - n02 - 2*n03 - 2*n10 - n11 - n12 - 2*n13)*log(3) + (n03 + n10)*(log(4 + 3*p2) + log(4 + 3*p1 - 2*r - 6*p1*r)) + (n00 + n13)*(log(4 + 3*p2) + log(2 - 3*p1 + 2*r + 6*p1*r)) + (n01 + n12)*log(16 + (8 - 6*p2)*r - 3*p1*(-4 + 3*p2)*(-1 + 2*r)) + (n02 + n11)*log(-8*(-3 + r) + 6*p2*(-1 + r) + 3*p1*(-4 + 3*p2)*(-1 + 2*r))
return(L)}
interlogL_r3 <- function(n00,n01,n02,n03,n10,n11,n12,n13,p1,p2) {
optimize(logL_r3,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_r3 <- parallel::mcmapply(interlogL_r3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"], p1, p2, mc.cores = ncores)


LOD_r3 <- -((x[,"n_01"] + x[,"n_12"])*log10(16 + (8 - 6*p2)/2)) - (x[,"n_02"] + x[,"n_11"])*log10(20 - 3*p2) - (x[,"n_03"] + x[,"n_10"])*(log10(3) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_13"])*(log10(3) + log10(4 + 3*p2)) + (x[,"n_03"] + x[,"n_10"])*(log10(4 + 3*p2) + log10(4 + 3*p1 - 2*r_r3 - 6*p1*r_r3)) + (x[,"n_00"] + x[,"n_13"])*(log10(4 + 3*p2) + log10(2 - 3*p1 + 2*r_r3 + 6*p1*r_r3)) + (x[,"n_01"] + x[,"n_12"])*log10(16 + (8 - 6*p2)*r_r3 - 3*p1*(-4 + 3*p2)*(-1 + 2*r_r3)) + (x[,"n_02"] + x[,"n_11"])*log10(-8*(-3 + r_r3) + 6*p2*(-1 + r_r3) + 3*p1*(-4 + 3*p2)*(-1 + 2*r_r3))


logL_r4 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,p1,p2) {
L <- 6*(-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13)*log(2) + (-2*n00 - n01 - n02 - 2*n03 - 2*n10 - n11 - n12 - 2*n13)*log(3) + (n02 + n11)*log(4*(-4*(-3 + r) + 3*p2*(-1 + r)) - 3*p1*(-4 + 3*p2)*(-1 + 2*r)) + (n00 + n13)*(log(4 + 3*p2) + log(4*(1 + r) - p1*(-3 + 6*r))) + (n03 + n10)*(log(4 + 3*p2) + log(8 - 4*r + p1*(-3 + 6*r))) + (n01 + n12)*log(3*p1*(-4 + 3*p2)*(-1 + 2*r) + 4*(8 + (4 - 3*p2)*r))
return(L)}
interlogL_r4 <- function(n00,n01,n02,n03,n10,n11,n12,n13,p1,p2) {
optimize(logL_r4,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_r4 <- parallel::mcmapply(interlogL_r4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"], p1, p2, mc.cores = ncores)


LOD_r4 <- -((x[,"n_01"] + x[,"n_12"])*(2*log10(2) + log10(8 + (4 - 3*p2)/2))) - (x[,"n_02"] + x[,"n_11"])*(2*log10(2) + log10(10 - (3*p2)/2)) - (x[,"n_03"] + x[,"n_10"])*(log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_13"])*(log10(2) + log10(3) + log10(4 + 3*p2)) + (x[,"n_02"] + x[,"n_11"])*log10(4*(-4*(-3 + r_r4) + 3*p2*(-1 + r_r4)) - 3*p1*(-4 + 3*p2)*(-1 + 2*r_r4)) + (x[,"n_00"] + x[,"n_13"])*(log10(4 + 3*p2) + log10(4*(1 + r_r4) - p1*(-3 + 6*r_r4))) + (x[,"n_03"] + x[,"n_10"])*(log10(4 + 3*p2) + log10(8 - 4*r_r4 + p1*(-3 + 6*r_r4))) + (x[,"n_01"] + x[,"n_12"])*log10(3*p1*(-4 + 3*p2)*(-1 + 2*r_r4) + 4*(8 + (4 - 3*p2)*r_r4))


return(list(
r_mat = cbind(r_c1,r_c2,r_r1,r_r2,r_r3,r_r4,0.499),
LOD_mat = cbind(LOD_c1,LOD_c2,LOD_r1,LOD_r2,LOD_r3,LOD_r4,0),
logL_mat = NULL,
phasing_strategy = "MINR",
possible_phases = c("coupling","coupling","repulsion","repulsion","repulsion","repulsion","unknown")
)
)
}

#' @rdname p4_functions
p4_1.0_1.3 <- function(x,p1,p2,ncores=1){


r_c <- (x[,"n_03"] + x[,"n_11"])/(x[,"n_01"] + x[,"n_03"] + x[,"n_11"] + x[,"n_13"])
logL_c <- 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"])*log(2) + (x[,"n_01"] + x[,"n_13"])*log(pmax(1e-6,1 - r_c)) + (x[,"n_03"] + x[,"n_11"])*log(pmax(1e-6,r_c))
LOD_c <- (x[,"n_01"] + x[,"n_03"] + x[,"n_11"] + x[,"n_13"])*log10(2) + (x[,"n_01"] + x[,"n_13"])*log10(pmax(1e-6,1 - r_c)) + (x[,"n_03"] + x[,"n_11"])*log10(pmax(1e-6,r_c))


r_r1 <- (4*x[,"n_01"] - 2*x[,"n_03"] - 2*x[,"n_11"] + 4*x[,"n_13"] + 3*x[,"n_01"]*p1 + 3*x[,"n_03"]*p1 + 3*x[,"n_11"]*p1 + 3*x[,"n_13"]*p1)/(2*(x[,"n_01"] + x[,"n_03"] + x[,"n_11"] + x[,"n_13"])*(1 + 3*p1))
logL_r1 <- (-3*x[,"n_01"] - 2*x[,"n_02"] - 3*x[,"n_03"] - 3*x[,"n_11"] - 2*x[,"n_12"] - 3*x[,"n_13"])*log(2) + (-x[,"n_01"] - x[,"n_03"] - x[,"n_11"] - x[,"n_13"])*log(3) + (x[,"n_03"] + x[,"n_11"])*log(4 + p1*(3 - 6*r_r1) - 2*r_r1) + (x[,"n_01"] + x[,"n_13"])*log(2 - 3*p1 + 2*r_r1 + 6*p1*r_r1)
LOD_r1 <- -((x[,"n_01"] + x[,"n_03"] + x[,"n_11"] + x[,"n_13"])*log10(3)) + (x[,"n_03"] + x[,"n_11"])*log10(4 + p1*(3 - 6*r_r1) - 2*r_r1) + (x[,"n_01"] + x[,"n_13"])*log10(2 - 3*p1 + 2*r_r1 + 6*p1*r_r1)


r_r2 <- (-8*x[,"n_01"] + 4*x[,"n_03"] + 4*x[,"n_11"] - 8*x[,"n_13"] + 3*x[,"n_01"]*p1 + 3*x[,"n_03"]*p1 + 3*x[,"n_11"]*p1 + 3*x[,"n_13"]*p1)/(2*(x[,"n_01"] + x[,"n_03"] + x[,"n_11"] + x[,"n_13"])*(-2 + 3*p1))
logL_r2 <- 2*(-2*x[,"n_01"] - x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_11"] - x[,"n_12"] - 2*x[,"n_13"])*log(2) + (-x[,"n_01"] - x[,"n_03"] - x[,"n_11"] - x[,"n_13"])*log(3) + (x[,"n_01"] + x[,"n_13"])*log(4 + 3*p1 + 4*r_r2 - 6*p1*r_r2) + (x[,"n_03"] + x[,"n_11"])*log(8 - 4*r_r2 + p1*(-3 + 6*r_r2))
LOD_r2 <- -((x[,"n_01"] + x[,"n_03"] + x[,"n_11"] + x[,"n_13"])*log10(6)) + (x[,"n_01"] + x[,"n_13"])*log10(4 + 3*p1 + 4*r_r2 - 6*p1*r_r2) + (x[,"n_03"] + x[,"n_11"])*log10(8 - 3*p1 - 4*r_r2 + 6*p1*r_r2)


return(list(
r_mat = cbind(r_c,r_r1,r_r2,0.499),
LOD_mat = cbind(LOD_c,LOD_r1,LOD_r2,0),
logL_mat = cbind(logL_c,logL_r1,logL_r2,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","repulsion","repulsion","unknown")
)
)
}

#' @rdname p4_functions
p4_1.0_2.0 <- function(x,p1,p2,ncores=1){


r_c1 <- (x[,"n_02"] + x[,"n_10"])/(x[,"n_00"] + x[,"n_02"] + x[,"n_10"] + x[,"n_12"])
logL_c1 <- (-x[,"n_00"] - x[,"n_02"] - x[,"n_10"] - x[,"n_12"])*(2*log(2) + log(3)) + (x[,"n_01"] + x[,"n_11"])*log(1/3 + p1/4) + (x[,"n_00"] + x[,"n_12"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_c1))) + (x[,"n_02"] + x[,"n_10"])*(log(2 - 3*p1) + log(pmax(1e-6,r_c1)))
LOD_c1 <- (x[,"n_00"] + x[,"n_02"] + x[,"n_10"] + x[,"n_12"])*log10(2) + (x[,"n_00"] + x[,"n_12"])*log10(pmax(1e-6,1 - r_c1)) + (x[,"n_02"] + x[,"n_10"])*log10(pmax(1e-6,r_c1))


r_c2 <- (x[,"n_02"] + x[,"n_10"])/(x[,"n_00"] + x[,"n_02"] + x[,"n_10"] + x[,"n_12"])
logL_c2 <- (-x[,"n_00"] - x[,"n_02"] - x[,"n_10"] - x[,"n_12"])*(3*log(2) + log(3)) + (x[,"n_01"] + x[,"n_11"])*log(1/3 - p1/8) + (x[,"n_00"] + x[,"n_12"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_c2))) + (x[,"n_02"] + x[,"n_10"])*(log(4 + 3*p1) + log(pmax(1e-6,r_c2)))
LOD_c2 <- -((x[,"n_02"] + x[,"n_10"])*(-log10(2) + log10(4 + 3*p1))) - (x[,"n_00"] + x[,"n_12"])*(-log10(2) + log10(4 + 3*p1)) + (x[,"n_00"] + x[,"n_12"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_c2))) + (x[,"n_02"] + x[,"n_10"])*(log10(4 + 3*p1) + log10(pmax(1e-6,r_c2)))


r_r1 <- (x[,"n_00"] + x[,"n_12"])/(x[,"n_00"] + x[,"n_02"] + x[,"n_10"] + x[,"n_12"])
logL_r1 <- (-x[,"n_00"] - x[,"n_02"] - x[,"n_10"] - x[,"n_12"])*(2*log(2) + log(3)) + (x[,"n_01"] + x[,"n_11"])*log(1/3 + p1/4) + (x[,"n_02"] + x[,"n_10"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_r1))) + (x[,"n_00"] + x[,"n_12"])*(log(2 - 3*p1) + log(pmax(1e-6,r_r1)))
LOD_r1 <- -((x[,"n_02"] + x[,"n_10"])*(-log10(2) + log10(2 - 3*p1))) - (x[,"n_00"] + x[,"n_12"])*(-log10(2) + log10(2 - 3*p1)) + (x[,"n_02"] + x[,"n_10"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_r1))) + (x[,"n_00"] + x[,"n_12"])*(log10(2 - 3*p1) + log10(pmax(1e-6,r_r1)))


r_r2 <- (x[,"n_00"] + x[,"n_12"])/(x[,"n_00"] + x[,"n_02"] + x[,"n_10"] + x[,"n_12"])
logL_r2 <- (-x[,"n_00"] - x[,"n_02"] - x[,"n_10"] - x[,"n_12"])*(3*log(2) + log(3)) + (x[,"n_01"] + x[,"n_11"])*log(1/3 - p1/8) + (x[,"n_02"] + x[,"n_10"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_r2))) + (x[,"n_00"] + x[,"n_12"])*(log(4 + 3*p1) + log(pmax(1e-6,r_r2)))
LOD_r2 <- -((x[,"n_02"] + x[,"n_10"])*(-log10(2) + log10(4 + 3*p1))) - (x[,"n_00"] + x[,"n_12"])*(-log10(2) + log10(4 + 3*p1)) + (x[,"n_02"] + x[,"n_10"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_r2))) + (x[,"n_00"] + x[,"n_12"])*(log10(4 + 3*p1) + log10(pmax(1e-6,r_r2)))


return(list(
r_mat = cbind(r_c1,r_c2,r_r1,r_r2,0.499),
LOD_mat = cbind(LOD_c1,LOD_c2,LOD_r1,LOD_r2,0),
logL_mat = cbind(logL_c1,logL_c2,logL_r1,logL_r2,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","coupling","repulsion","repulsion","unknown")
)
)
}

#' @rdname p4_functions
p4_1.0_2.1 <- function(x,p1,p2,ncores=1){


logL_c1 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13)*(3*log(2) + log(3)) + (n00 + n13)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r))) + (n03 + n10)*(log(2 - 3*p1) + log(pmax(1e-6,r))) + (n02 + n11)*log(4 + 3*p1 + 2*r - 3*p1*r) + (n01 + n12)*log(6 + (-2 + 3*p1)*r)
return(L)}
interlogL_c1 <- function(n00,n01,n02,n03,n10,n11,n12,n13,p1,p2) {
optimize(logL_c1,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_c1 <- parallel::mcmapply(interlogL_c1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"], p1, p2, mc.cores = ncores)


LOD_c1 <- -((x[,"n_03"] + x[,"n_10"])*log10(1 - (3*p1)/2)) - (x[,"n_00"] + x[,"n_13"])*log10(1 - (3*p1)/2) - (x[,"n_02"] + x[,"n_11"])*log10(5 + (3*p1)/2) - (x[,"n_01"] + x[,"n_12"])*log10(5 + (3*p1)/2) + (x[,"n_00"] + x[,"n_13"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_c1))) + (x[,"n_03"] + x[,"n_10"])*(log10(2 - 3*p1) + log10(pmax(1e-6,r_c1))) + (x[,"n_02"] + x[,"n_11"])*log10(4 + 3*p1 + 2*r_c1 - 3*p1*r_c1) + (x[,"n_01"] + x[,"n_12"])*log10(6 + (-2 + 3*p1)*r_c1)


logL_c2 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13)*(4*log(2) + log(3)) + (n00 + n13)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r))) + (n03 + n10)*(log(4 + 3*p1) + log(pmax(1e-6,r))) + (n01 + n12)*log(12 - (4 + 3*p1)*r) + (n02 + n11)*log(3*p1*(-1 + r) + 4*(2 + r))
return(L)}
interlogL_c2 <- function(n00,n01,n02,n03,n10,n11,n12,n13,p1,p2) {
optimize(logL_c2,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_c2 <- parallel::mcmapply(interlogL_c2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"], p1, p2, mc.cores = ncores)


LOD_c2 <- -((x[,"n_01"] + x[,"n_12"])*log10(12 + (-4 - 3*p1)/2)) - (x[,"n_02"] + x[,"n_11"])*log10(10 - (3*p1)/2) - (x[,"n_03"] + x[,"n_10"])*(-log10(2) + log10(4 + 3*p1)) - (x[,"n_00"] + x[,"n_13"])*(-log10(2) + log10(4 + 3*p1)) + (x[,"n_00"] + x[,"n_13"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_c2))) + (x[,"n_03"] + x[,"n_10"])*(log10(4 + 3*p1) + log10(pmax(1e-6,r_c2))) + (x[,"n_01"] + x[,"n_12"])*log10(12 - (4 + 3*p1)*r_c2) + (x[,"n_02"] + x[,"n_11"])*log10(3*p1*(-1 + r_c2) + 4*(2 + r_c2))


logL_r1 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13)*(4*log(2) + log(3)) + (n03 + n10)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r))) + (n00 + n13)*(log(4 + 3*p1) + log(pmax(1e-6,r))) + (n02 + n11)*log(12 - (4 + 3*p1)*r) + (n01 + n12)*log(3*p1*(-1 + r) + 4*(2 + r))
return(L)}
interlogL_r1 <- function(n00,n01,n02,n03,n10,n11,n12,n13,p1,p2) {
optimize(logL_r1,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_r1 <- parallel::mcmapply(interlogL_r1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"], p1, p2, mc.cores = ncores)


LOD_r1 <- -((x[,"n_02"] + x[,"n_11"])*log10(12 + (-4 - 3*p1)/2)) - (x[,"n_01"] + x[,"n_12"])*log10(10 - (3*p1)/2) - (x[,"n_03"] + x[,"n_10"])*(-log10(2) + log10(4 + 3*p1)) - (x[,"n_00"] + x[,"n_13"])*(-log10(2) + log10(4 + 3*p1)) + (x[,"n_03"] + x[,"n_10"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_r1))) + (x[,"n_00"] + x[,"n_13"])*(log10(4 + 3*p1) + log10(pmax(1e-6,r_r1))) + (x[,"n_02"] + x[,"n_11"])*log10(12 - (4 + 3*p1)*r_r1) + (x[,"n_01"] + x[,"n_12"])*log10(3*p1*(-1 + r_r1) + 4*(2 + r_r1))


logL_r2 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13)*(3*log(2) + log(3)) + (n03 + n10)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r))) + (n00 + n13)*(log(2 - 3*p1) + log(pmax(1e-6,r))) + (n01 + n12)*log(4 + 3*p1 + 2*r - 3*p1*r) + (n02 + n11)*log(6 + (-2 + 3*p1)*r)
return(L)}
interlogL_r2 <- function(n00,n01,n02,n03,n10,n11,n12,n13,p1,p2) {
optimize(logL_r2,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_r2 <- parallel::mcmapply(interlogL_r2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"], p1, p2, mc.cores = ncores)


LOD_r2 <- -((x[,"n_03"] + x[,"n_10"])*log10(1 - (3*p1)/2)) - (x[,"n_00"] + x[,"n_13"])*log10(1 - (3*p1)/2) - (x[,"n_02"] + x[,"n_11"])*log10(5 + (3*p1)/2) - (x[,"n_01"] + x[,"n_12"])*log10(5 + (3*p1)/2) + (x[,"n_03"] + x[,"n_10"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_r2))) + (x[,"n_00"] + x[,"n_13"])*(log10(2 - 3*p1) + log10(pmax(1e-6,r_r2))) + (x[,"n_01"] + x[,"n_12"])*log10(4 + 3*p1 + 2*r_r2 - 3*p1*r_r2) + (x[,"n_02"] + x[,"n_11"])*log10(6 + (-2 + 3*p1)*r_r2)


return(list(
r_mat = cbind(r_c1,r_c2,r_r1,r_r2,0.499),
LOD_mat = cbind(LOD_c1,LOD_c2,LOD_r1,LOD_r2,0),
logL_mat = NULL,
phasing_strategy = "MINR",
possible_phases = c("coupling","coupling","repulsion","repulsion","unknown")
)
)
}

#' @rdname p4_functions
p4_1.0_2.2 <- function(x,p1,p2,ncores=1){


logL_c1 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,p1,p2) {
L <- 4*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14)*log(2) + 2*(-n00 - n01 - n03 - n04 - n10 - n11 - n13 - n14)*log(3) + (n02 + n12)*log(4 + 2*p1 + 2*p2 + 3*p1*p2) + (n00 + n14)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r))) + (n04 + n10)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r))) + (n03 + n11)*log((4 + 3*p1)*(2 - 3*p2) + 2*(2 - 3*p1)*(4 + 3*p2)*r) + (n01 + n13)*log(-4*(-6 + (4 + 3*p2)*r) + 3*p1*(-6 + 8*r + p2*(-9 + 6*r)))
return(L)}
interlogL_c1 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,p1,p2) {
optimize(logL_c1,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_c1 <- parallel::mcmapply(interlogL_c1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"], p1, p2, mc.cores = ncores)


LOD_c1 <- -((x[,"n_04"] + x[,"n_10"])*(-log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_14"])*(-log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_03"] + x[,"n_11"])*log10((4 + 3*p1)*(2 - 3*p2) + (2 - 3*p1)*(4 + 3*p2)) - (x[,"n_01"] + x[,"n_13"])*log10(3*p1*(-2 - 6*p2) - 4*(-6 + (4 + 3*p2)/2)) + (x[,"n_00"] + x[,"n_14"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_c1))) + (x[,"n_04"] + x[,"n_10"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_c1))) + (x[,"n_03"] + x[,"n_11"])*log10((4 + 3*p1)*(2 - 3*p2) + 2*(2 - 3*p1)*(4 + 3*p2)*r_c1) + (x[,"n_01"] + x[,"n_13"])*log10(-4*(-6 + (4 + 3*p2)*r_c1) + 3*p1*(-6 + 8*r_c1 + p2*(-9 + 6*r_c1)))


logL_c2 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,p1,p2) {
L <- 5*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14)*log(2) + 2*(-n00 - n01 - n03 - n04 - n10 - n11 - n13 - n14)*log(3) + (n02 + n12)*log(8 + p1*(4 - 3*p2) - 2*p2) + (n00 + n14)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r))) + (n04 + n10)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r))) + (n03 + n11)*log((4 + 3*p1)*(4 + 3*p2) + 2*(-2 + 3*p1)*(-8 + 3*p2)*r) + (n01 + n13)*log(4*(12 + (-8 + 3*p2)*r) - 3*p1*(12 - 16*r + p2*(-9 + 6*r)))
return(L)}
interlogL_c2 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,p1,p2) {
optimize(logL_c2,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_c2 <- parallel::mcmapply(interlogL_c2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"], p1, p2, mc.cores = ncores)


LOD_c2 <- -((x[,"n_04"] + x[,"n_10"])*(-log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2))) - (x[,"n_00"] + x[,"n_14"])*(-log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_03"] + x[,"n_11"])*log10((-2 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)) - (x[,"n_01"] + x[,"n_13"])*log10(-3*p1*(4 - 6*p2) + 4*(12 + (-8 + 3*p2)/2)) + (x[,"n_00"] + x[,"n_14"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_c2))) + (x[,"n_04"] + x[,"n_10"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_c2))) + (x[,"n_03"] + x[,"n_11"])*log10((4 + 3*p1)*(4 + 3*p2) + 2*(-2 + 3*p1)*(-8 + 3*p2)*r_c2) + (x[,"n_01"] + x[,"n_13"])*log10(4*(12 + (-8 + 3*p2)*r_c2) - 3*p1*(12 - 16*r_c2 + p2*(-9 + 6*r_c2)))


logL_c3 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,p1,p2) {
L <- 5*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14)*log(2) + 2*(-n00 - n01 - n03 - n04 - n10 - n11 - n13 - n14)*log(3) + (n02 + n12)*log(8 - 2*p1 + 4*p2 - 3*p1*p2) + (n00 + n14)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r))) + (n04 + n10)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r))) + (n03 + n11)*log((-8 + 3*p1)*(-2 + 3*p2) + 2*(4 + 3*p1)*(4 + 3*p2)*r) + (n01 + n13)*log(-8*(-6 + (4 + 3*p2)*r) - 3*p1*(-6 + 8*r + p2*(-9 + 6*r)))
return(L)}
interlogL_c3 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,p1,p2) {
optimize(logL_c3,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_c3 <- parallel::mcmapply(interlogL_c3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"], p1, p2, mc.cores = ncores)


LOD_c3 <- -((x[,"n_04"] + x[,"n_10"])*(-log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_14"])*(-log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_03"] + x[,"n_11"])*log10((-8 + 3*p1)*(-2 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)) - (x[,"n_01"] + x[,"n_13"])*log10(-3*p1*(-2 - 6*p2) - 8*(-6 + (4 + 3*p2)/2)) + (x[,"n_00"] + x[,"n_14"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_c3))) + (x[,"n_04"] + x[,"n_10"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_c3))) + (x[,"n_03"] + x[,"n_11"])*log10((-8 + 3*p1)*(-2 + 3*p2) + 2*(4 + 3*p1)*(4 + 3*p2)*r_c3) + (x[,"n_01"] + x[,"n_13"])*log10(-8*(-6 + (4 + 3*p2)*r_c3) - 3*p1*(-6 + 8*r_c3 + p2*(-9 + 6*r_c3)))


logL_c4 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,p1,p2) {
L <- 6*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14)*log(2) + 2*(-n00 - n01 - n03 - n04 - n10 - n11 - n13 - n14)*log(3) + (n02 + n12)*log(16 - 4*p1 - 4*p2 + 3*p1*p2) + (n00 + n14)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r))) + (n04 + n10)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r))) + (n03 + n11)*log((8 - 3*p1)*(4 + 3*p2) - 2*(4 + 3*p1)*(-8 + 3*p2)*r) + (n01 + n13)*log(8*(12 + (-8 + 3*p2)*r) + 3*p1*(12 - 16*r + p2*(-9 + 6*r)))
return(L)}
interlogL_c4 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,p1,p2) {
optimize(logL_c4,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_c4 <- parallel::mcmapply(interlogL_c4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"], p1, p2, mc.cores = ncores)


LOD_c4 <- -((x[,"n_04"] + x[,"n_10"])*(-log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2))) - (x[,"n_00"] + x[,"n_14"])*(-log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_03"] + x[,"n_11"])*log10(-((4 + 3*p1)*(-8 + 3*p2)) + (8 - 3*p1)*(4 + 3*p2)) - (x[,"n_01"] + x[,"n_13"])*log10(3*p1*(4 - 6*p2) + 8*(12 + (-8 + 3*p2)/2)) + (x[,"n_00"] + x[,"n_14"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_c4))) + (x[,"n_04"] + x[,"n_10"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_c4))) + (x[,"n_03"] + x[,"n_11"])*log10((8 - 3*p1)*(4 + 3*p2) - 2*(4 + 3*p1)*(-8 + 3*p2)*r_c4) + (x[,"n_01"] + x[,"n_13"])*log10(8*(12 + (-8 + 3*p2)*r_c4) + 3*p1*(12 - 16*r_c4 + p2*(-9 + 6*r_c4)))


logL_r1 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,p1,p2) {
L <- 5*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14)*log(2) + 2*(-n00 - n01 - n03 - n04 - n10 - n11 - n13 - n14)*log(3) + (n02 + n12)*log(8 - 2*p1 + 4*p2 - 3*p1*p2) + (n04 + n10)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r))) + (n00 + n14)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r))) + (n01 + n13)*log((-8 + 3*p1)*(-2 + 3*p2) + 2*(4 + 3*p1)*(4 + 3*p2)*r) + (n03 + n11)*log(-8*(-6 + (4 + 3*p2)*r) - 3*p1*(-6 + 8*r + p2*(-9 + 6*r)))
return(L)}
interlogL_r1 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,p1,p2) {
optimize(logL_r1,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_r1 <- parallel::mcmapply(interlogL_r1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"], p1, p2, mc.cores = ncores)


LOD_r1 <- -((x[,"n_04"] + x[,"n_10"])*(-log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_14"])*(-log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_13"])*log10((-8 + 3*p1)*(-2 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)) - (x[,"n_03"] + x[,"n_11"])*log10(-3*p1*(-2 - 6*p2) - 8*(-6 + (4 + 3*p2)/2)) + (x[,"n_04"] + x[,"n_10"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_r1))) + (x[,"n_00"] + x[,"n_14"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_r1))) + (x[,"n_01"] + x[,"n_13"])*log10((-8 + 3*p1)*(-2 + 3*p2) + 2*(4 + 3*p1)*(4 + 3*p2)*r_r1) + (x[,"n_03"] + x[,"n_11"])*log10(-8*(-6 + (4 + 3*p2)*r_r1) - 3*p1*(-6 + 8*r_r1 + p2*(-9 + 6*r_r1)))


logL_r2 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,p1,p2) {
L <- 6*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14)*log(2) + 2*(-n00 - n01 - n03 - n04 - n10 - n11 - n13 - n14)*log(3) + (n02 + n12)*log(16 - 4*p1 - 4*p2 + 3*p1*p2) + (n04 + n10)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r))) + (n00 + n14)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r))) + (n01 + n13)*log((8 - 3*p1)*(4 + 3*p2) - 2*(4 + 3*p1)*(-8 + 3*p2)*r) + (n03 + n11)*log(8*(12 + (-8 + 3*p2)*r) + 3*p1*(12 - 16*r + p2*(-9 + 6*r)))
return(L)}
interlogL_r2 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,p1,p2) {
optimize(logL_r2,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_r2 <- parallel::mcmapply(interlogL_r2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"], p1, p2, mc.cores = ncores)


LOD_r2 <- -((x[,"n_04"] + x[,"n_10"])*(-log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2))) - (x[,"n_00"] + x[,"n_14"])*(-log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_13"])*log10(-((4 + 3*p1)*(-8 + 3*p2)) + (8 - 3*p1)*(4 + 3*p2)) - (x[,"n_03"] + x[,"n_11"])*log10(3*p1*(4 - 6*p2) + 8*(12 + (-8 + 3*p2)/2)) + (x[,"n_04"] + x[,"n_10"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_r2))) + (x[,"n_00"] + x[,"n_14"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_r2))) + (x[,"n_01"] + x[,"n_13"])*log10((8 - 3*p1)*(4 + 3*p2) - 2*(4 + 3*p1)*(-8 + 3*p2)*r_r2) + (x[,"n_03"] + x[,"n_11"])*log10(8*(12 + (-8 + 3*p2)*r_r2) + 3*p1*(12 - 16*r_r2 + p2*(-9 + 6*r_r2)))


logL_r3 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,p1,p2) {
L <- 4*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14)*log(2) + 2*(-n00 - n01 - n03 - n04 - n10 - n11 - n13 - n14)*log(3) + (n02 + n12)*log(4 + 2*p1 + 2*p2 + 3*p1*p2) + (n04 + n10)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r))) + (n00 + n14)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r))) + (n01 + n13)*log((-4 - 3*p1)*(-2 + 3*p2) + 2*(2 - 3*p1)*(4 + 3*p2)*r) + (n03 + n11)*log(-4*(-6 + (4 + 3*p2)*r) + 3*p1*(-6 + 8*r + p2*(-9 + 6*r)))
return(L)}
interlogL_r3 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,p1,p2) {
optimize(logL_r3,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_r3 <- parallel::mcmapply(interlogL_r3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"], p1, p2, mc.cores = ncores)


LOD_r3 <- -((x[,"n_04"] + x[,"n_10"])*(-log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_14"])*(-log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_13"])*log10((-4 - 3*p1)*(-2 + 3*p2) + (2 - 3*p1)*(4 + 3*p2)) - (x[,"n_03"] + x[,"n_11"])*log10(3*p1*(-2 - 6*p2) - 4*(-6 + (4 + 3*p2)/2)) + (x[,"n_04"] + x[,"n_10"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_r3))) + (x[,"n_00"] + x[,"n_14"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_r3))) + (x[,"n_01"] + x[,"n_13"])*log10((-4 - 3*p1)*(-2 + 3*p2) + 2*(2 - 3*p1)*(4 + 3*p2)*r_r3) + (x[,"n_03"] + x[,"n_11"])*log10(-4*(-6 + (4 + 3*p2)*r_r3) + 3*p1*(-6 + 8*r_r3 + p2*(-9 + 6*r_r3)))


logL_r4 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,p1,p2) {
L <- 5*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14)*log(2) + 2*(-n00 - n01 - n03 - n04 - n10 - n11 - n13 - n14)*log(3) + (n02 + n12)*log(8 + p1*(4 - 3*p2) - 2*p2) + (n04 + n10)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r))) + (n00 + n14)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r))) + (n01 + n13)*log((4 + 3*p1)*(4 + 3*p2) + 2*(-2 + 3*p1)*(-8 + 3*p2)*r) + (n03 + n11)*log(4*(12 + (-8 + 3*p2)*r) - 3*p1*(12 - 16*r + p2*(-9 + 6*r)))
return(L)}
interlogL_r4 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,p1,p2) {
optimize(logL_r4,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_r4 <- parallel::mcmapply(interlogL_r4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"], p1, p2, mc.cores = ncores)


LOD_r4 <- -((x[,"n_04"] + x[,"n_10"])*(-log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2))) - (x[,"n_00"] + x[,"n_14"])*(-log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_13"])*log10((-2 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)) - (x[,"n_03"] + x[,"n_11"])*log10(-3*p1*(4 - 6*p2) + 4*(12 + (-8 + 3*p2)/2)) + (x[,"n_04"] + x[,"n_10"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_r4))) + (x[,"n_00"] + x[,"n_14"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_r4))) + (x[,"n_01"] + x[,"n_13"])*log10((4 + 3*p1)*(4 + 3*p2) + 2*(-2 + 3*p1)*(-8 + 3*p2)*r_r4) + (x[,"n_03"] + x[,"n_11"])*log10(4*(12 + (-8 + 3*p2)*r_r4) - 3*p1*(12 - 16*r_r4 + p2*(-9 + 6*r_r4)))


return(list(
r_mat = cbind(r_c1,r_c2,r_c3,r_c4,r_r1,r_r2,r_r3,r_r4,0.499),
LOD_mat = cbind(LOD_c1,LOD_c2,LOD_c3,LOD_c4,LOD_r1,LOD_r2,LOD_r3,LOD_r4,0),
logL_mat = NULL,
phasing_strategy = "MINR",
possible_phases = c("coupling","coupling","coupling","coupling",
                    "repulsion","repulsion","repulsion","repulsion","unknown")
)
)
}

#' @rdname p4_functions
p4_1.1_1.1 <- function(x,p1,p2,ncores=1){


logL_cc <- function(r,n00,n01,n02,n10,n12,n20,n21,n22,n11,p1,p2) {
L <- (-2*n00 - n01 - 2*n02 - n10 - n12 - 2*n20 - n21 - 2*n22)*log(2) + 2*(n02 + n20)*log(pmax(1e-6,r)) + (n01 + n10 + n12 + n21)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n22)*log(1 - 2*r + r^2) + n11*log(1/2 - r + r^2)
return(L)}
interlogL_cc <- function(n00,n01,n02,n10,n12,n20,n21,n22,n11,p1,p2) {
optimize(logL_cc,c(0,0.5), n00,n01,n02,n10,n12,n20,n21,n22,n11,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc <- parallel::mcmapply(interlogL_cc,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_10"],x[,"n_12"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_11"], p1, p2, mc.cores = ncores)


LOD_cc <- 2*x[,"n_11"]*log10(2) + 2*(x[,"n_02"] + x[,"n_20"])*log10(2) + 2*(x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log10(2) + 2*(x[,"n_00"] + x[,"n_22"])*log10(2) + 2*(x[,"n_02"] + x[,"n_20"])*log10(pmax(1e-6,r_cc)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_00"] + x[,"n_22"])*log10(1 - 2*r_cc + r_cc^2) + x[,"n_11"]*log10(1/2 - r_cc + r_cc^2)


logL_cc <- (-2*x[,"n_00"] - x[,"n_01"] - 2*x[,"n_02"] - x[,"n_10"] - x[,"n_12"] - 2*x[,"n_20"] - x[,"n_21"] - 2*x[,"n_22"])*log(2) + 2*(x[,"n_02"] + x[,"n_20"])*log(pmax(1e-6,r_cc)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_00"] + x[,"n_22"])*log(1 - 2*r_cc + r_cc^2) + x[,"n_11"]*log(1/2 - r_cc + r_cc^2)


logL_cr1 <- function(r,n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
L <- (-3*n00 - 3*n01 - 3*n02 - 3*n10 - 2*n11 - 3*n12 - 3*n20 - 3*n21 - 3*n22)*log(2) + (-n00 - n01 - n02 - n10 - n11 - n12 - n20 - n21 - n22)*log(3) + (n02 + n20)*(log(4 + p2*(3 - 6*r) - 2*r) + log(pmax(1e-6,r))) + (n00 + n22)*(log(pmax(1e-6,1 - r)) + log(2 - 3*p2 + 2*r + 6*p2*r)) + n11*log(2 - 3*p2*(1 - 2*r)^2 + 4*r - 4*r^2) + (n01 + n10 + n12 + n21)*log(3*p2*(1 - 2*r)^2 + 4*(1 - r + r^2))
return(L)}
interlogL_cr1 <- function(n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
optimize(logL_cr1,c(0,0.5), n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr1 <- parallel::mcmapply(interlogL_cr1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_20"],x[,"n_21"],x[,"n_22"], p1, p2, mc.cores = ncores)


LOD_cr1 <- -(x[,"n_11"]*log10(3)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log10(3) - (x[,"n_02"] + x[,"n_20"])*(-log10(2) + log10(3)) - (x[,"n_00"] + x[,"n_22"])*(-log10(2) + log10(3)) + (x[,"n_02"] + x[,"n_20"])*(log10(4 + p2*(3 - 6*r_cr1) - 2*r_cr1) + log10(pmax(1e-6,r_cr1))) + (x[,"n_00"] + x[,"n_22"])*(log10(pmax(1e-6,1 - r_cr1)) + log10(2 - 3*p2 + 2*r_cr1 + 6*p2*r_cr1)) + x[,"n_11"]*log10(2 - 3*p2*(1 - 2*r_cr1)^2 + 4*r_cr1 - 4*r_cr1^2) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log10(3*p2*(1 - 2*r_cr1)^2 + 4*(1 - r_cr1 + r_cr1^2))


logL_cr1 <- (-3*x[,"n_00"] - 3*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_10"] - 2*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"])*log(3) + (x[,"n_02"] + x[,"n_20"])*(log(4 + p2*(3 - 6*r_cr1) - 2*r_cr1) + log(pmax(1e-6,r_cr1))) + (x[,"n_00"] + x[,"n_22"])*(log(pmax(1e-6,1 - r_cr1)) + log(2 - 3*p2 + 2*r_cr1 + 6*p2*r_cr1)) + x[,"n_11"]*log(2 - 3*p2*(1 - 2*r_cr1)^2 + 4*r_cr1 - 4*r_cr1^2) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log(3*p2*(1 - 2*r_cr1)^2 + 4*(1 - r_cr1 + r_cr1^2))


logL_cr2 <- function(r,n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
L <- (-4*n00 - 4*n01 - 4*n02 - 4*n10 - 3*n11 - 4*n12 - 4*n20 - 4*n21 - 4*n22)*log(2) + (-n00 - n01 - n02 - n10 - n11 - n12 - n20 - n21 - n22)*log(3) + n11*log(4 + 3*p2*(1 - 2*r)^2 + 8*r - 8*r^2) + (n00 + n22)*(log(pmax(1e-6,1 - r)) + log(4*(1 + r) - p2*(-3 + 6*r))) + (n02 + n20)*(log(pmax(1e-6,r)) + log(8 - 4*r + p2*(-3 + 6*r))) + (n01 + n10 + n12 + n21)*log(-3*p2*(1 - 2*r)^2 + 8*(1 - r + r^2))
return(L)}
interlogL_cr2 <- function(n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
optimize(logL_cr2,c(0,0.5), n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr2 <- parallel::mcmapply(interlogL_cr2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_20"],x[,"n_21"],x[,"n_22"], p1, p2, mc.cores = ncores)


LOD_cr2 <- -((x[,"n_02"] + x[,"n_20"])*log10(3)) - (x[,"n_00"] + x[,"n_22"])*log10(3) - x[,"n_11"]*(log10(2) + log10(3)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(log10(2) + log10(3)) + x[,"n_11"]*log10(4 + 3*p2*(1 - 2*r_cr2)^2 + 8*r_cr2 - 8*r_cr2^2) + (x[,"n_00"] + x[,"n_22"])*(log10(pmax(1e-6,1 - r_cr2)) + log10(4*(1 + r_cr2) - p2*(-3 + 6*r_cr2))) + (x[,"n_02"] + x[,"n_20"])*(log10(pmax(1e-6,r_cr2)) + log10(8 - 4*r_cr2 + p2*(-3 + 6*r_cr2))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log10(-3*p2*(1 - 2*r_cr2)^2 + 8*(1 - r_cr2 + r_cr2^2))


logL_cr2 <- (-4*x[,"n_00"] - 4*x[,"n_01"] - 4*x[,"n_02"] - 4*x[,"n_10"] - 3*x[,"n_11"] - 4*x[,"n_12"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"])*log(3) + x[,"n_11"]*log(4 + 3*p2*(1 - 2*r_cr2)^2 + 8*r_cr2 - 8*r_cr2^2) + (x[,"n_00"] + x[,"n_22"])*(log(pmax(1e-6,1 - r_cr2)) + log(4*(1 + r_cr2) - p2*(-3 + 6*r_cr2))) + (x[,"n_02"] + x[,"n_20"])*(log(pmax(1e-6,r_cr2)) + log(8 - 4*r_cr2 + p2*(-3 + 6*r_cr2))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log(-3*p2*(1 - 2*r_cr2)^2 + 8*(1 - r_cr2 + r_cr2^2))


logL_rc1 <- function(r,n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
L <- (-3*n00 - 3*n01 - 3*n02 - 3*n10 - 2*n11 - 3*n12 - 3*n20 - 3*n21 - 3*n22)*log(2) + (-n00 - n01 - n02 - n10 - n11 - n12 - n20 - n21 - n22)*log(3) + (n02 + n20)*(log(4 + p1*(3 - 6*r) - 2*r) + log(pmax(1e-6,r))) + (n00 + n22)*(log(pmax(1e-6,1 - r)) + log(2 - 3*p1 + 2*r + 6*p1*r)) + n11*log(2 - 3*p1*(1 - 2*r)^2 + 4*r - 4*r^2) + (n01 + n10 + n12 + n21)*log(3*p1*(1 - 2*r)^2 + 4*(1 - r + r^2))
return(L)}
interlogL_rc1 <- function(n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
optimize(logL_rc1,c(0,0.5), n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc1 <- parallel::mcmapply(interlogL_rc1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_20"],x[,"n_21"],x[,"n_22"], p1, p2, mc.cores = ncores)


LOD_rc1 <- -(x[,"n_11"]*log10(3)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log10(3) - (x[,"n_02"] + x[,"n_20"])*(-log10(2) + log10(3)) - (x[,"n_00"] + x[,"n_22"])*(-log10(2) + log10(3)) + (x[,"n_02"] + x[,"n_20"])*(log10(4 + p1*(3 - 6*r_rc1) - 2*r_rc1) + log10(pmax(1e-6,r_rc1))) + (x[,"n_00"] + x[,"n_22"])*(log10(pmax(1e-6,1 - r_rc1)) + log10(2 - 3*p1 + 2*r_rc1 + 6*p1*r_rc1)) + x[,"n_11"]*log10(2 - 3*p1*(1 - 2*r_rc1)^2 + 4*r_rc1 - 4*r_rc1^2) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log10(3*p1*(1 - 2*r_rc1)^2 + 4*(1 - r_rc1 + r_rc1^2))


logL_rc1 <- (-3*x[,"n_00"] - 3*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_10"] - 2*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"])*log(3) + (x[,"n_02"] + x[,"n_20"])*(log(4 + p1*(3 - 6*r_rc1) - 2*r_rc1) + log(pmax(1e-6,r_rc1))) + (x[,"n_00"] + x[,"n_22"])*(log(pmax(1e-6,1 - r_rc1)) + log(2 - 3*p1 + 2*r_rc1 + 6*p1*r_rc1)) + x[,"n_11"]*log(2 - 3*p1*(1 - 2*r_rc1)^2 + 4*r_rc1 - 4*r_rc1^2) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log(3*p1*(1 - 2*r_rc1)^2 + 4*(1 - r_rc1 + r_rc1^2))


logL_rc2 <- function(r,n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
L <- (-4*n00 - 4*n01 - 4*n02 - 4*n10 - 3*n11 - 4*n12 - 4*n20 - 4*n21 - 4*n22)*log(2) + (-n00 - n01 - n02 - n10 - n11 - n12 - n20 - n21 - n22)*log(3) + n11*log(4 + 3*p1*(1 - 2*r)^2 + 8*r - 8*r^2) + (n00 + n22)*(log(pmax(1e-6,1 - r)) + log(4*(1 + r) - p1*(-3 + 6*r))) + (n02 + n20)*(log(pmax(1e-6,r)) + log(8 - 4*r + p1*(-3 + 6*r))) + (n01 + n10 + n12 + n21)*log(-3*p1*(1 - 2*r)^2 + 8*(1 - r + r^2))
return(L)}
interlogL_rc2 <- function(n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
optimize(logL_rc2,c(0,0.5), n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc2 <- parallel::mcmapply(interlogL_rc2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_20"],x[,"n_21"],x[,"n_22"], p1, p2, mc.cores = ncores)


LOD_rc2 <- -((x[,"n_02"] + x[,"n_20"])*log10(3)) - (x[,"n_00"] + x[,"n_22"])*log10(3) - x[,"n_11"]*(log10(2) + log10(3)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(log10(2) + log10(3)) + x[,"n_11"]*log10(4 + 3*p1*(1 - 2*r_rc2)^2 + 8*r_rc2 - 8*r_rc2^2) + (x[,"n_00"] + x[,"n_22"])*(log10(pmax(1e-6,1 - r_rc2)) + log10(4*(1 + r_rc2) - p1*(-3 + 6*r_rc2))) + (x[,"n_02"] + x[,"n_20"])*(log10(pmax(1e-6,r_rc2)) + log10(8 - 4*r_rc2 + p1*(-3 + 6*r_rc2))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log10(-3*p1*(1 - 2*r_rc2)^2 + 8*(1 - r_rc2 + r_rc2^2))


logL_rc2 <- (-4*x[,"n_00"] - 4*x[,"n_01"] - 4*x[,"n_02"] - 4*x[,"n_10"] - 3*x[,"n_11"] - 4*x[,"n_12"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"])*log(3) + x[,"n_11"]*log(4 + 3*p1*(1 - 2*r_rc2)^2 + 8*r_rc2 - 8*r_rc2^2) + (x[,"n_00"] + x[,"n_22"])*(log(pmax(1e-6,1 - r_rc2)) + log(4*(1 + r_rc2) - p1*(-3 + 6*r_rc2))) + (x[,"n_02"] + x[,"n_20"])*(log(pmax(1e-6,r_rc2)) + log(8 - 4*r_rc2 + p1*(-3 + 6*r_rc2))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log(-3*p1*(1 - 2*r_rc2)^2 + 8*(1 - r_rc2 + r_rc2^2))


logL_rr1 <- function(r,n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
L <- (-4*n00 - 3*n01 - 4*n02 - 3*n10 - 2*n11 - 3*n12 - 4*n20 - 3*n21 - 4*n22)*log(2) + 2*(-n00 - n01 - n02 - n10 - n11 - n12 - n20 - n21 - n22)*log(3) + (n02 + n20)*(log(4 + 3*p1 - 2*r - 6*p1*r) + log(4 + 3*p2 - 2*r - 6*p2*r)) + (n00 + n22)*(log(2 - 3*p1 + 2*r + 6*p1*r) + log(2 - 3*p2 + 2*r + 6*p2*r)) + (n01 + n10 + n12 + n21)*log(8 - 3*p2*(1 - 2*r)^2 - 3*p1*(1 + 3*p2)*(1 - 2*r)^2 + 4*r - 4*r^2) + n11*log(10 + 3*p2*(1 - 2*r)^2 + 3*p1*(1 + 3*p2)*(1 - 2*r)^2 - 4*r + 4*r^2)
return(L)}
interlogL_rr1 <- function(n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
optimize(logL_rr1,c(0,0.5), n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr1 <- parallel::mcmapply(interlogL_rr1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_20"],x[,"n_21"],x[,"n_22"], p1, p2, mc.cores = ncores)


LOD_rr1 <- -2*x[,"n_11"]*log10(3) - 2*(x[,"n_02"] + x[,"n_20"])*log10(3) - 2*(x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log10(3) - 2*(x[,"n_00"] + x[,"n_22"])*log10(3) + (x[,"n_02"] + x[,"n_20"])*(log10(4 + 3*p1 - 2*r_rr1 - 6*p1*r_rr1) + log10(4 + 3*p2 - 2*r_rr1 - 6*p2*r_rr1)) + (x[,"n_00"] + x[,"n_22"])*(log10(2 - 3*p1 + 2*r_rr1 + 6*p1*r_rr1) + log10(2 - 3*p2 + 2*r_rr1 + 6*p2*r_rr1)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log10(8 - 3*p2*(1 - 2*r_rr1)^2 - 3*p1*(1 + 3*p2)*(1 - 2*r_rr1)^2 + 4*r_rr1 - 4*r_rr1^2) + x[,"n_11"]*log10(10 + 3*p2*(1 - 2*r_rr1)^2 + 3*p1*(1 + 3*p2)*(1 - 2*r_rr1)^2 - 4*r_rr1 + 4*r_rr1^2)


logL_rr1 <- (-4*x[,"n_00"] - 3*x[,"n_01"] - 4*x[,"n_02"] - 3*x[,"n_10"] - 2*x[,"n_11"] - 3*x[,"n_12"] - 4*x[,"n_20"] - 3*x[,"n_21"] - 4*x[,"n_22"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"])*log(3) + (x[,"n_02"] + x[,"n_20"])*(log(4 + 3*p1 - 2*r_rr1 - 6*p1*r_rr1) + log(4 + 3*p2 - 2*r_rr1 - 6*p2*r_rr1)) + (x[,"n_00"] + x[,"n_22"])*(log(2 - 3*p1 + 2*r_rr1 + 6*p1*r_rr1) + log(2 - 3*p2 + 2*r_rr1 + 6*p2*r_rr1)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log(8 - 3*p2*(1 - 2*r_rr1)^2 - 3*p1*(1 + 3*p2)*(1 - 2*r_rr1)^2 + 4*r_rr1 - 4*r_rr1^2) + x[,"n_11"]*log(10 + 3*p2*(1 - 2*r_rr1)^2 + 3*p1*(1 + 3*p2)*(1 - 2*r_rr1)^2 - 4*r_rr1 + 4*r_rr1^2)


logL_rr2 <- function(r,n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
L <- (-5*n00 - 4*n01 - 5*n02 - 4*n10 - 3*n11 - 4*n12 - 5*n20 - 4*n21 - 5*n22)*log(2) + 2*(-n00 - n01 - n02 - n10 - n11 - n12 - n20 - n21 - n22)*log(3) + (n00 + n22)*(log(2 - 3*p1 + 2*r + 6*p1*r) + log(4 + 3*p2 + 4*r - 6*p2*r)) + n11*log(20 - 3*p2*(1 - 2*r)^2 - 3*p1*(-2 + 3*p2)*(1 - 2*r)^2 - 8*r + 8*r^2) + (n02 + n20)*(log(4 + p1*(3 - 6*r) - 2*r) + log(8 - 4*r + p2*(-3 + 6*r))) + (n01 + n10 + n12 + n21)*log(3*p2*(1 - 2*r)^2 + 3*p1*(-2 + 3*p2)*(1 - 2*r)^2 + 8*(2 + r - r^2))
return(L)}
interlogL_rr2 <- function(n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
optimize(logL_rr2,c(0,0.5), n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr2 <- parallel::mcmapply(interlogL_rr2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_20"],x[,"n_21"],x[,"n_22"], p1, p2, mc.cores = ncores)


LOD_rr2 <- -(x[,"n_11"]*(log10(2) + 2*log10(3))) - (x[,"n_02"] + x[,"n_20"])*(log10(2) + 2*log10(3)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(log10(2) + 2*log10(3)) - (x[,"n_00"] + x[,"n_22"])*(log10(2) + 2*log10(3)) + (x[,"n_00"] + x[,"n_22"])*(log10(2 - 3*p1 + 2*r_rr2 + 6*p1*r_rr2) + log10(4 + 3*p2 + 4*r_rr2 - 6*p2*r_rr2)) + x[,"n_11"]*log10(20 - 3*p2*(1 - 2*r_rr2)^2 - 3*p1*(-2 + 3*p2)*(1 - 2*r_rr2)^2 - 8*r_rr2 + 8*r_rr2^2) + (x[,"n_02"] + x[,"n_20"])*(log10(4 + p1*(3 - 6*r_rr2) - 2*r_rr2) + log10(8 - 4*r_rr2 + p2*(-3 + 6*r_rr2))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log10(3*p2*(1 - 2*r_rr2)^2 + 3*p1*(-2 + 3*p2)*(1 - 2*r_rr2)^2 + 8*(2 + r_rr2 - r_rr2^2))


logL_rr2 <- (-5*x[,"n_00"] - 4*x[,"n_01"] - 5*x[,"n_02"] - 4*x[,"n_10"] - 3*x[,"n_11"] - 4*x[,"n_12"] - 5*x[,"n_20"] - 4*x[,"n_21"] - 5*x[,"n_22"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"])*log(3) + (x[,"n_00"] + x[,"n_22"])*(log(2 - 3*p1 + 2*r_rr2 + 6*p1*r_rr2) + log(4 + 3*p2 + 4*r_rr2 - 6*p2*r_rr2)) + x[,"n_11"]*log(20 - 3*p2*(1 - 2*r_rr2)^2 - 3*p1*(-2 + 3*p2)*(1 - 2*r_rr2)^2 - 8*r_rr2 + 8*r_rr2^2) + (x[,"n_02"] + x[,"n_20"])*(log(4 + p1*(3 - 6*r_rr2) - 2*r_rr2) + log(8 - 4*r_rr2 + p2*(-3 + 6*r_rr2))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log(3*p2*(1 - 2*r_rr2)^2 + 3*p1*(-2 + 3*p2)*(1 - 2*r_rr2)^2 + 8*(2 + r_rr2 - r_rr2^2))


logL_rr3 <- function(r,n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
L <- (-5*n00 - 4*n01 - 5*n02 - 4*n10 - 3*n11 - 4*n12 - 5*n20 - 4*n21 - 5*n22)*log(2) + 2*(-n00 - n01 - n02 - n10 - n11 - n12 - n20 - n21 - n22)*log(3) + (n00 + n22)*(log(4 + 3*p1 + 4*r - 6*p1*r) + log(2 - 3*p2 + 2*r + 6*p2*r)) + n11*log(20 + 6*p2*(1 - 2*r)^2 - 3*p1*(1 + 3*p2)*(1 - 2*r)^2 - 8*r + 8*r^2) + (n02 + n20)*(log(4 + p2*(3 - 6*r) - 2*r) + log(8 - 4*r + p1*(-3 + 6*r))) + (n01 + n10 + n12 + n21)*log(-6*p2*(1 - 2*r)^2 + 3*p1*(1 + 3*p2)*(1 - 2*r)^2 + 8*(2 + r - r^2))
return(L)}
interlogL_rr3 <- function(n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
optimize(logL_rr3,c(0,0.5), n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr3 <- parallel::mcmapply(interlogL_rr3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_20"],x[,"n_21"],x[,"n_22"], p1, p2, mc.cores = ncores)


LOD_rr3 <- -(x[,"n_11"]*(log10(2) + 2*log10(3))) - (x[,"n_02"] + x[,"n_20"])*(log10(2) + 2*log10(3)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(log10(2) + 2*log10(3)) - (x[,"n_00"] + x[,"n_22"])*(log10(2) + 2*log10(3)) + (x[,"n_00"] + x[,"n_22"])*(log10(4 + 3*p1 + 4*r_rr3 - 6*p1*r_rr3) + log10(2 - 3*p2 + 2*r_rr3 + 6*p2*r_rr3)) + x[,"n_11"]*log10(20 + 6*p2*(1 - 2*r_rr3)^2 - 3*p1*(1 + 3*p2)*(1 - 2*r_rr3)^2 - 8*r_rr3 + 8*r_rr3^2) + (x[,"n_02"] + x[,"n_20"])*(log10(4 + p2*(3 - 6*r_rr3) - 2*r_rr3) + log10(8 - 4*r_rr3 + p1*(-3 + 6*r_rr3))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log10(-6*p2*(1 - 2*r_rr3)^2 + 3*p1*(1 + 3*p2)*(1 - 2*r_rr3)^2 + 8*(2 + r_rr3 - r_rr3^2))


logL_rr3 <- (-5*x[,"n_00"] - 4*x[,"n_01"] - 5*x[,"n_02"] - 4*x[,"n_10"] - 3*x[,"n_11"] - 4*x[,"n_12"] - 5*x[,"n_20"] - 4*x[,"n_21"] - 5*x[,"n_22"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"])*log(3) + (x[,"n_00"] + x[,"n_22"])*(log(4 + 3*p1 + 4*r_rr3 - 6*p1*r_rr3) + log(2 - 3*p2 + 2*r_rr3 + 6*p2*r_rr3)) + x[,"n_11"]*log(20 + 6*p2*(1 - 2*r_rr3)^2 - 3*p1*(1 + 3*p2)*(1 - 2*r_rr3)^2 - 8*r_rr3 + 8*r_rr3^2) + (x[,"n_02"] + x[,"n_20"])*(log(4 + p2*(3 - 6*r_rr3) - 2*r_rr3) + log(8 - 4*r_rr3 + p1*(-3 + 6*r_rr3))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log(-6*p2*(1 - 2*r_rr3)^2 + 3*p1*(1 + 3*p2)*(1 - 2*r_rr3)^2 + 8*(2 + r_rr3 - r_rr3^2))


logL_rr4 <- function(r,n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
L <- (-6*n00 - 5*n01 - 6*n02 - 5*n10 - 4*n11 - 5*n12 - 6*n20 - 5*n21 - 6*n22)*log(2) + 2*(-n00 - n01 - n02 - n10 - n11 - n12 - n20 - n21 - n22)*log(3) + (n00 + n22)*(log(4 + 3*p1 + 4*r - 6*p1*r) + log(4 + 3*p2 + 4*r - 6*p2*r)) + (n02 + n20)*(log(8 - 4*r + p1*(-3 + 6*r)) + log(8 - 4*r + p2*(-3 + 6*r))) + n11*log(-6*p2*(1 - 2*r)^2 + 3*p1*(-2 + 3*p2)*(1 - 2*r)^2 + 8*(5 - 2*r + 2*r^2)) + (n01 + n10 + n12 + n21)*log(-3*p1*(-2 + 3*p2)*(1 - 2*r)^2 + 2*(3*p2*(1 - 2*r)^2 + 8*(2 + r - r^2)))
return(L)}
interlogL_rr4 <- function(n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
optimize(logL_rr4,c(0,0.5), n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr4 <- parallel::mcmapply(interlogL_rr4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_20"],x[,"n_21"],x[,"n_22"], p1, p2, mc.cores = ncores)


LOD_rr4 <- -2*(x[,"n_02"] + x[,"n_20"])*(log10(2) + log10(3)) - 2*(x[,"n_00"] + x[,"n_22"])*(log10(2) + log10(3)) - x[,"n_11"]*(2*log10(2) + 2*log10(3)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(2*log10(2) + 2*log10(3)) + (x[,"n_00"] + x[,"n_22"])*(log10(4 + 3*p1 + 4*r_rr4 - 6*p1*r_rr4) + log10(4 + 3*p2 + 4*r_rr4 - 6*p2*r_rr4)) + (x[,"n_02"] + x[,"n_20"])*(log10(8 - 4*r_rr4 + p1*(-3 + 6*r_rr4)) + log10(8 - 4*r_rr4 + p2*(-3 + 6*r_rr4))) + x[,"n_11"]*log10(-6*p2*(1 - 2*r_rr4)^2 + 3*p1*(-2 + 3*p2)*(1 - 2*r_rr4)^2 + 8*(5 - 2*r_rr4 + 2*r_rr4^2)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log10(-3*p1*(-2 + 3*p2)*(1 - 2*r_rr4)^2 + 2*(3*p2*(1 - 2*r_rr4)^2 + 8*(2 + r_rr4 - r_rr4^2)))


logL_rr4 <- (-6*x[,"n_00"] - 5*x[,"n_01"] - 6*x[,"n_02"] - 5*x[,"n_10"] - 4*x[,"n_11"] - 5*x[,"n_12"] - 6*x[,"n_20"] - 5*x[,"n_21"] - 6*x[,"n_22"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"])*log(3) + (x[,"n_00"] + x[,"n_22"])*(log(4 + 3*p1 + 4*r_rr4 - 6*p1*r_rr4) + log(4 + 3*p2 + 4*r_rr4 - 6*p2*r_rr4)) + (x[,"n_02"] + x[,"n_20"])*(log(8 - 4*r_rr4 + p1*(-3 + 6*r_rr4)) + log(8 - 4*r_rr4 + p2*(-3 + 6*r_rr4))) + x[,"n_11"]*log(-6*p2*(1 - 2*r_rr4)^2 + 3*p1*(-2 + 3*p2)*(1 - 2*r_rr4)^2 + 8*(5 - 2*r_rr4 + 2*r_rr4^2)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*log(-3*p1*(-2 + 3*p2)*(1 - 2*r_rr4)^2 + 2*(3*p2*(1 - 2*r_rr4)^2 + 8*(2 + r_rr4 - r_rr4^2)))


return(list(
r_mat = cbind(r_cc,r_cr1,r_cr2,r_rc1,r_rc2,r_rr1,r_rr2,r_rr3,r_rr4,0.499),
LOD_mat = cbind(LOD_cc,LOD_cr1,LOD_cr2,LOD_rc1,LOD_rc2,LOD_rr1,LOD_rr2,LOD_rr3,LOD_rr4,0),
logL_mat = cbind(logL_cc,logL_cr1,logL_cr2,logL_rc1,logL_rc2,logL_rr1,logL_rr2,logL_rr3,logL_rr4,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling repulsion","coupling repulsion","repulsion coupling","repulsion coupling","repulsion repulsion","repulsion repulsion","repulsion repulsion","repulsion repulsion","unknown")
)
)
}

#' @rdname p4_functions
p4_1.1_1.2 <- function(x,p1,p2,ncores=1){


logL_cc1 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
L <- (-3*n00 - 3*n01 - 3*n02 - 3*n03 - 2*n10 - 2*n11 - 2*n12 - 2*n13 - 3*n20 - 3*n21 - 3*n22 - 3*n23)*log(2) + (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*log(3) + (n10 + n13)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n03 + n20)*(log(2 - 3*p2) + 2*log(pmax(1e-6,r))) + (n02 + n21)*(log(pmax(1e-6,r)) + log(6 + (-2 + 3*p2)*r)) + (n00 + n23)*(log(2 - 3*p2) + log(1 - 2*r + r^2)) + (n11 + n12)*log(3 + (-2 + 3*p2)*r + (2 - 3*p2)*r^2) + (n01 + n22)*(log(pmax(1e-6,1 - r)) + log(3*p2*(1 - r) + 2*(2 + r)))
return(L)}
interlogL_cc1 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
optimize(logL_cc1,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc1 <- parallel::mcmapply(interlogL_cc1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_cc1 <- -((x[,"n_10"] + x[,"n_13"])*(-2*log10(2) + log10(2 - 3*p2))) - (x[,"n_03"] + x[,"n_20"])*(-2*log10(2) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_23"])*(-2*log10(2) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_22"])*(-log10(2) + log10(5 + (3*p2)/2)) - (x[,"n_02"] + x[,"n_21"])*(-log10(2) + log10(6 + (-2 + 3*p2)/2)) - (x[,"n_11"] + x[,"n_12"])*log10(3 + (2 - 3*p2)/4 + (-2 + 3*p2)/2) + (x[,"n_10"] + x[,"n_13"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cc1)) + log10(pmax(1e-6,r_cc1))) + (x[,"n_03"] + x[,"n_20"])*(log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_cc1))) + (x[,"n_02"] + x[,"n_21"])*(log10(pmax(1e-6,r_cc1)) + log10(6 + (-2 + 3*p2)*r_cc1)) + (x[,"n_00"] + x[,"n_23"])*(log10(2 - 3*p2) + log10(1 - 2*r_cc1 + r_cc1^2)) + (x[,"n_11"] + x[,"n_12"])*log10(3 + (-2 + 3*p2)*r_cc1 + (2 - 3*p2)*r_cc1^2) + (x[,"n_01"] + x[,"n_22"])*(log10(pmax(1e-6,1 - r_cc1)) + log10(3*p2*(1 - r_cc1) + 2*(2 + r_cc1)))


logL_cc1 <- (-3*x[,"n_00"] - 3*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 2*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 3*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*log(3) + (x[,"n_10"] + x[,"n_13"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_cc1)) + log(pmax(1e-6,r_cc1))) + (x[,"n_03"] + x[,"n_20"])*(log(2 - 3*p2) + 2*log(pmax(1e-6,r_cc1))) + (x[,"n_02"] + x[,"n_21"])*(log(pmax(1e-6,r_cc1)) + log(6 + (-2 + 3*p2)*r_cc1)) + (x[,"n_00"] + x[,"n_23"])*(log(2 - 3*p2) + log(1 - 2*r_cc1 + r_cc1^2)) + (x[,"n_11"] + x[,"n_12"])*log(3 + (-2 + 3*p2)*r_cc1 + (2 - 3*p2)*r_cc1^2) + (x[,"n_01"] + x[,"n_22"])*(log(pmax(1e-6,1 - r_cc1)) + log(3*p2*(1 - r_cc1) + 2*(2 + r_cc1)))


logL_cc2 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
L <- (-4*n00 - 4*n01 - 4*n02 - 4*n03 - 3*n10 - 3*n11 - 3*n12 - 3*n13 - 4*n20 - 4*n21 - 4*n22 - 4*n23)*log(2) + (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*log(3) + (n10 + n13)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n03 + n20)*(log(4 + 3*p2) + 2*log(pmax(1e-6,r))) + (n02 + n21)*(log(pmax(1e-6,r)) + log(12 - (4 + 3*p2)*r)) + (n00 + n23)*(log(4 + 3*p2) + log(1 - 2*r + r^2)) + (n11 + n12)*log(6 - (4 + 3*p2)*r + (4 + 3*p2)*r^2) + (n01 + n22)*(log(pmax(1e-6,1 - r)) + log(3*p2*(-1 + r) + 4*(2 + r)))
return(L)}
interlogL_cc2 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
optimize(logL_cc2,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc2 <- parallel::mcmapply(interlogL_cc2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_cc2 <- -((x[,"n_02"] + x[,"n_21"])*(-log10(2) + log10(12 + (-4 - 3*p2)/2))) - (x[,"n_01"] + x[,"n_22"])*(-log10(2) + log10(10 - (3*p2)/2)) - (x[,"n_10"] + x[,"n_13"])*(-2*log10(2) + log10(4 + 3*p2)) - (x[,"n_03"] + x[,"n_20"])*(-2*log10(2) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_23"])*(-2*log10(2) + log10(4 + 3*p2)) - (x[,"n_11"] + x[,"n_12"])*log10(6 + (-4 - 3*p2)/2 + (4 + 3*p2)/4) + (x[,"n_10"] + x[,"n_13"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cc2)) + log10(pmax(1e-6,r_cc2))) + (x[,"n_03"] + x[,"n_20"])*(log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_cc2))) + (x[,"n_02"] + x[,"n_21"])*(log10(pmax(1e-6,r_cc2)) + log10(12 - (4 + 3*p2)*r_cc2)) + (x[,"n_00"] + x[,"n_23"])*(log10(4 + 3*p2) + log10(1 - 2*r_cc2 + r_cc2^2)) + (x[,"n_11"] + x[,"n_12"])*log10(6 - (4 + 3*p2)*r_cc2 + (4 + 3*p2)*r_cc2^2) + (x[,"n_01"] + x[,"n_22"])*(log10(pmax(1e-6,1 - r_cc2)) + log10(3*p2*(-1 + r_cc2) + 4*(2 + r_cc2)))


logL_cc2 <- (-4*x[,"n_00"] - 4*x[,"n_01"] - 4*x[,"n_02"] - 4*x[,"n_03"] - 3*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 4*x[,"n_23"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*log(3) + (x[,"n_10"] + x[,"n_13"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_cc2)) + log(pmax(1e-6,r_cc2))) + (x[,"n_03"] + x[,"n_20"])*(log(4 + 3*p2) + 2*log(pmax(1e-6,r_cc2))) + (x[,"n_02"] + x[,"n_21"])*(log(pmax(1e-6,r_cc2)) + log(12 - (4 + 3*p2)*r_cc2)) + (x[,"n_00"] + x[,"n_23"])*(log(4 + 3*p2) + log(1 - 2*r_cc2 + r_cc2^2)) + (x[,"n_11"] + x[,"n_12"])*log(6 - (4 + 3*p2)*r_cc2 + (4 + 3*p2)*r_cc2^2) + (x[,"n_01"] + x[,"n_22"])*(log(pmax(1e-6,1 - r_cc2)) + log(3*p2*(-1 + r_cc2) + 4*(2 + r_cc2)))


logL_cr1 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*(4*log(2) + log(3)) + (n00 + n03 + n20 + n23)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n13)*(log(4 + 3*p2) + log(1 - 2*r + 2*r^2)) + (n11 + n12)*log(8 + 8*r - 8*r^2 + p2*(-3 + 6*r - 6*r^2)) + (n02 + n21)*log(4*(1 + r^2) + 3*p2*(1 - 3*r + r^2)) + (n01 + n22)*log(4*(2 - 2*r + r^2) + 3*p2*(-1 + r + r^2))
return(L)}
interlogL_cr1 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
optimize(logL_cr1,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr1 <- parallel::mcmapply(interlogL_cr1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_cr1 <- -((x[,"n_11"] + x[,"n_12"])*log10(10 - (3*p2)/2)) - (x[,"n_02"] + x[,"n_21"])*log10(5 - (3*p2)/4) - (x[,"n_01"] + x[,"n_22"])*log10(5 - (3*p2)/4) - (x[,"n_00"] + x[,"n_03"] + x[,"n_20"] + x[,"n_23"])*(-2*log10(2) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_13"])*(-log10(2) + log10(4 + 3*p2)) + (x[,"n_00"] + x[,"n_03"] + x[,"n_20"] + x[,"n_23"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cr1)) + log10(pmax(1e-6,r_cr1))) + (x[,"n_10"] + x[,"n_13"])*(log10(4 + 3*p2) + log10(1 - 2*r_cr1 + 2*r_cr1^2)) + (x[,"n_11"] + x[,"n_12"])*log10(8 + 8*r_cr1 - 8*r_cr1^2 + p2*(-3 + 6*r_cr1 - 6*r_cr1^2)) + (x[,"n_02"] + x[,"n_21"])*log10(4*(1 + r_cr1^2) + 3*p2*(1 - 3*r_cr1 + r_cr1^2)) + (x[,"n_01"] + x[,"n_22"])*log10(4*(2 - 2*r_cr1 + r_cr1^2) + 3*p2*(-1 + r_cr1 + r_cr1^2))


logL_cr1 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*(4*log(2) + log(3)) + (x[,"n_00"] + x[,"n_03"] + x[,"n_20"] + x[,"n_23"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_cr1)) + log(pmax(1e-6,r_cr1))) + (x[,"n_10"] + x[,"n_13"])*(log(4 + 3*p2) + log(1 - 2*r_cr1 + 2*r_cr1^2)) + (x[,"n_11"] + x[,"n_12"])*log(8 + 8*r_cr1 - 8*r_cr1^2 + p2*(-3 + 6*r_cr1 - 6*r_cr1^2)) + (x[,"n_02"] + x[,"n_21"])*log(4*(1 + r_cr1^2) + 3*p2*(1 - 3*r_cr1 + r_cr1^2)) + (x[,"n_01"] + x[,"n_22"])*log(4*(2 - 2*r_cr1 + r_cr1^2) + 3*p2*(-1 + r_cr1 + r_cr1^2))


logL_cr2 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*(3*log(2) + log(3)) + (n00 + n03 + n20 + n23)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n13)*(log(2 - 3*p2) + log(1 - 2*r + 2*r^2)) + (n02 + n21)*log(2*(1 + r^2) - 3*p2*(1 - 3*r + r^2)) + (n01 + n22)*log(2*(2 - 2*r + r^2) - 3*p2*(-1 + r + r^2)) + (n11 + n12)*log(4 + 4*r - 4*r^2 + p2*(3 - 6*r + 6*r^2))
return(L)}
interlogL_cr2 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
optimize(logL_cr2,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr2 <- parallel::mcmapply(interlogL_cr2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_cr2 <- -((x[,"n_00"] + x[,"n_03"] + x[,"n_20"] + x[,"n_23"])*(-2*log10(2) + log10(2 - 3*p2))) - (x[,"n_10"] + x[,"n_13"])*(-log10(2) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_21"])*log10(5/2 + (3*p2)/4) - (x[,"n_01"] + x[,"n_22"])*log10(5/2 + (3*p2)/4) - (x[,"n_11"] + x[,"n_12"])*log10(5 + (3*p2)/2) + (x[,"n_00"] + x[,"n_03"] + x[,"n_20"] + x[,"n_23"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cr2)) + log10(pmax(1e-6,r_cr2))) + (x[,"n_10"] + x[,"n_13"])*(log10(2 - 3*p2) + log10(1 - 2*r_cr2 + 2*r_cr2^2)) + (x[,"n_02"] + x[,"n_21"])*log10(2*(1 + r_cr2^2) - 3*p2*(1 - 3*r_cr2 + r_cr2^2)) + (x[,"n_01"] + x[,"n_22"])*log10(2*(2 - 2*r_cr2 + r_cr2^2) - 3*p2*(-1 + r_cr2 + r_cr2^2)) + (x[,"n_11"] + x[,"n_12"])*log10(4 + 4*r_cr2 - 4*r_cr2^2 + p2*(3 - 6*r_cr2 + 6*r_cr2^2))


logL_cr2 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*(3*log(2) + log(3)) + (x[,"n_00"] + x[,"n_03"] + x[,"n_20"] + x[,"n_23"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_cr2)) + log(pmax(1e-6,r_cr2))) + (x[,"n_10"] + x[,"n_13"])*(log(2 - 3*p2) + log(1 - 2*r_cr2 + 2*r_cr2^2)) + (x[,"n_02"] + x[,"n_21"])*log(2*(1 + r_cr2^2) - 3*p2*(1 - 3*r_cr2 + r_cr2^2)) + (x[,"n_01"] + x[,"n_22"])*log(2*(2 - 2*r_cr2 + r_cr2^2) - 3*p2*(-1 + r_cr2 + r_cr2^2)) + (x[,"n_11"] + x[,"n_12"])*log(4 + 4*r_cr2 - 4*r_cr2^2 + p2*(3 - 6*r_cr2 + 6*r_cr2^2))


logL_rc1 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*(4*log(2) + 2*log(3)) + (n03 + n20)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(4 + 3*p1 - 2*r - 6*p1*r)) + (n00 + n23)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(2 - 3*p1 + 2*r + 6*p1*r)) + (n02 + n21)*log((-4 - 3*p2)*(-4 - 3*p1 + 2*r + 6*p1*r) + (2 - 3*p2)*r*(2 - 3*p1 + 2*r + 6*p1*r)) + (n10 + n13)*(log(2 - 3*p2) + log(3*p1*(1 - 2*r)^2 + 4*(1 - r + r^2))) + (n01 + n22)*log(-6*p2*(1 - 4*r + r^2) + 4*(4 - r + r^2) - 3*p1*(-1 + 2*r)*(3*p2*(-2 + r) - 2*(1 + r))) + (n11 + n12)*log(3*p1*(-2 + 3*p2)*(1 - 2*r)^2 + 4*(7 + 2*r - 2*r^2 + 3*p2*(1 - r + r^2)))
return(L)}
interlogL_rc1 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
optimize(logL_rc1,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc1 <- parallel::mcmapply(interlogL_rc1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_rc1 <- -((x[,"n_02"] + x[,"n_21"])*log10(-3*(-4 - 3*p2) + (3*(2 - 3*p2))/2)) - (x[,"n_10"] + x[,"n_13"])*(log10(3) + log10(2 - 3*p2)) - (x[,"n_03"] + x[,"n_20"])*(-log10(2) + log10(3) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_23"])*(-log10(2) + log10(3) + log10(2 - 3*p2)) - (x[,"n_11"] + x[,"n_12"])*(2*log10(2) + log10(15/2 + (9*p2)/4)) - (x[,"n_01"] + x[,"n_22"])*log10(15 + (9*p2)/2) + (x[,"n_03"] + x[,"n_20"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_rc1)) + log10(4 + 3*p1 - 2*r_rc1 - 6*p1*r_rc1)) + (x[,"n_00"] + x[,"n_23"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rc1)) + log10(2 - 3*p1 + 2*r_rc1 + 6*p1*r_rc1)) + (x[,"n_02"] + x[,"n_21"])*log10((-4 - 3*p2)*(-4 - 3*p1 + 2*r_rc1 + 6*p1*r_rc1) + (2 - 3*p2)*r_rc1*(2 - 3*p1 + 2*r_rc1 + 6*p1*r_rc1)) + (x[,"n_10"] + x[,"n_13"])*(log10(2 - 3*p2) + log10(3*p1*(1 - 2*r_rc1)^2 + 4*(1 - r_rc1 + r_rc1^2))) + (x[,"n_01"] + x[,"n_22"])*log10(-6*p2*(1 - 4*r_rc1 + r_rc1^2) + 4*(4 - r_rc1 + r_rc1^2) - 3*p1*(-1 + 2*r_rc1)*(3*p2*(-2 + r_rc1) - 2*(1 + r_rc1))) + (x[,"n_11"] + x[,"n_12"])*log10(3*p1*(-2 + 3*p2)*(1 - 2*r_rc1)^2 + 4*(7 + 2*r_rc1 - 2*r_rc1^2 + 3*p2*(1 - r_rc1 + r_rc1^2)))


logL_rc1 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*(4*log(2) + 2*log(3)) + (x[,"n_03"] + x[,"n_20"])*(log(2 - 3*p2) + log(pmax(1e-6,r_rc1)) + log(4 + 3*p1 - 2*r_rc1 - 6*p1*r_rc1)) + (x[,"n_00"] + x[,"n_23"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_rc1)) + log(2 - 3*p1 + 2*r_rc1 + 6*p1*r_rc1)) + (x[,"n_02"] + x[,"n_21"])*log((-4 - 3*p2)*(-4 - 3*p1 + 2*r_rc1 + 6*p1*r_rc1) + (2 - 3*p2)*r_rc1*(2 - 3*p1 + 2*r_rc1 + 6*p1*r_rc1)) + (x[,"n_10"] + x[,"n_13"])*(log(2 - 3*p2) + log(3*p1*(1 - 2*r_rc1)^2 + 4*(1 - r_rc1 + r_rc1^2))) + (x[,"n_01"] + x[,"n_22"])*log(-6*p2*(1 - 4*r_rc1 + r_rc1^2) + 4*(4 - r_rc1 + r_rc1^2) - 3*p1*(-1 + 2*r_rc1)*(3*p2*(-2 + r_rc1) - 2*(1 + r_rc1))) + (x[,"n_11"] + x[,"n_12"])*log(3*p1*(-2 + 3*p2)*(1 - 2*r_rc1)^2 + 4*(7 + 2*r_rc1 - 2*r_rc1^2 + 3*p2*(1 - r_rc1 + r_rc1^2)))


logL_rc2 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*(5*log(2) + 2*log(3)) + (n03 + n20)*(log(4 + 3*p2) + log(4 + p1*(3 - 6*r) - 2*r) + log(pmax(1e-6,r))) + (n00 + n23)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(2 - 3*p1 + 2*r + 6*p1*r)) + (n02 + n21)*log((-8 + 3*p2)*(-4 - 3*p1 + 2*r + 6*p1*r) + (4 + 3*p2)*r*(2 - 3*p1 + 2*r + 6*p1*r)) + (n10 + n13)*(log(4 + 3*p2) + log(3*p1*(1 - 2*r)^2 + 4*(1 - r + r^2))) + (n01 + n22)*log(6*p2*(1 - 4*r + r^2) + 8*(4 - r + r^2) + 3*p1*(-1 + 2*r)*(3*p2*(-2 + r) + 4*(1 + r))) + (n11 + n12)*log(-3*p1*(4 + 3*p2)*(1 - 2*r)^2 - 4*(-14 - 4*r + 4*r^2 + 3*p2*(1 - r + r^2)))
return(L)}
interlogL_rc2 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
optimize(logL_rc2,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc2 <- parallel::mcmapply(interlogL_rc2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_rc2 <- -((x[,"n_01"] + x[,"n_22"])*log10(30 - (9*p2)/2)) - (x[,"n_11"] + x[,"n_12"])*(2*log10(2) + log10(15 - (9*p2)/4)) - (x[,"n_10"] + x[,"n_13"])*(log10(3) + log10(4 + 3*p2)) - (x[,"n_03"] + x[,"n_20"])*(-log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_23"])*(-log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_21"])*log10(-3*(-8 + 3*p2) + (3*(4 + 3*p2))/2) + (x[,"n_03"] + x[,"n_20"])*(log10(4 + 3*p2) + log10(4 + p1*(3 - 6*r_rc2) - 2*r_rc2) + log10(pmax(1e-6,r_rc2))) + (x[,"n_00"] + x[,"n_23"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rc2)) + log10(2 - 3*p1 + 2*r_rc2 + 6*p1*r_rc2)) + (x[,"n_02"] + x[,"n_21"])*log10((-8 + 3*p2)*(-4 - 3*p1 + 2*r_rc2 + 6*p1*r_rc2) + (4 + 3*p2)*r_rc2*(2 - 3*p1 + 2*r_rc2 + 6*p1*r_rc2)) + (x[,"n_10"] + x[,"n_13"])*(log10(4 + 3*p2) + log10(3*p1*(1 - 2*r_rc2)^2 + 4*(1 - r_rc2 + r_rc2^2))) + (x[,"n_01"] + x[,"n_22"])*log10(6*p2*(1 - 4*r_rc2 + r_rc2^2) + 8*(4 - r_rc2 + r_rc2^2) + 3*p1*(-1 + 2*r_rc2)*(3*p2*(-2 + r_rc2) + 4*(1 + r_rc2))) + (x[,"n_11"] + x[,"n_12"])*log10(-3*p1*(4 + 3*p2)*(1 - 2*r_rc2)^2 - 4*(-14 - 4*r_rc2 + 4*r_rc2^2 + 3*p2*(1 - r_rc2 + r_rc2^2)))


logL_rc2 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*(5*log(2) + 2*log(3)) + (x[,"n_03"] + x[,"n_20"])*(log(4 + 3*p2) + log(4 + p1*(3 - 6*r_rc2) - 2*r_rc2) + log(pmax(1e-6,r_rc2))) + (x[,"n_00"] + x[,"n_23"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_rc2)) + log(2 - 3*p1 + 2*r_rc2 + 6*p1*r_rc2)) + (x[,"n_02"] + x[,"n_21"])*log((-8 + 3*p2)*(-4 - 3*p1 + 2*r_rc2 + 6*p1*r_rc2) + (4 + 3*p2)*r_rc2*(2 - 3*p1 + 2*r_rc2 + 6*p1*r_rc2)) + (x[,"n_10"] + x[,"n_13"])*(log(4 + 3*p2) + log(3*p1*(1 - 2*r_rc2)^2 + 4*(1 - r_rc2 + r_rc2^2))) + (x[,"n_01"] + x[,"n_22"])*log(6*p2*(1 - 4*r_rc2 + r_rc2^2) + 8*(4 - r_rc2 + r_rc2^2) + 3*p1*(-1 + 2*r_rc2)*(3*p2*(-2 + r_rc2) + 4*(1 + r_rc2))) + (x[,"n_11"] + x[,"n_12"])*log(-3*p1*(4 + 3*p2)*(1 - 2*r_rc2)^2 - 4*(-14 - 4*r_rc2 + 4*r_rc2^2 + 3*p2*(1 - r_rc2 + r_rc2^2)))


logL_rc3 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*(5*log(2) + 2*log(3)) + (n00 + n23)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(4 + 3*p1 + 4*r - 6*p1*r)) + (n03 + n20)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(8 - 4*r + p1*(-3 + 6*r))) + (n10 + n13)*(log(2 - 3*p2) + log(-3*p1*(1 - 2*r)^2 + 8*(1 - r + r^2))) + (n02 + n21)*log((2 - 3*p2)*r*(4 + 3*p1 + 4*r - 6*p1*r) + (4 + 3*p2)*(8 - 4*r + p1*(-3 + 6*r))) + (n01 + n22)*log((-2 + 3*p2)*(-1 + r)*(8 - 4*r + p1*(-3 + 6*r)) - (4 + 3*p2)*(-4*(1 + r) + p1*(-3 + 6*r))) + (n11 + n12)*log(-3*p1*(-2 + 3*p2)*(1 - 2*r)^2 + 8*(7 + 2*r - 2*r^2 + 3*p2*(1 - r + r^2)))
return(L)}
interlogL_rc3 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
optimize(logL_rc3,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc3 <- parallel::mcmapply(interlogL_rc3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_rc3 <- -((x[,"n_03"] + x[,"n_20"])*(log10(3) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_23"])*(log10(3) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_13"])*(log10(2) + log10(3) + log10(2 - 3*p2)) - (x[,"n_11"] + x[,"n_12"])*(3*log10(2) + log10(15/2 + (9*p2)/4)) - (x[,"n_02"] + x[,"n_21"])*log10(3*(2 - 3*p2) + 6*(4 + 3*p2)) - (x[,"n_01"] + x[,"n_22"])*log10(-3*(-2 + 3*p2) + 6*(4 + 3*p2)) + (x[,"n_00"] + x[,"n_23"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rc3)) + log10(4 + 3*p1 + 4*r_rc3 - 6*p1*r_rc3)) + (x[,"n_03"] + x[,"n_20"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_rc3)) + log10(8 - 4*r_rc3 + p1*(-3 + 6*r_rc3))) + (x[,"n_10"] + x[,"n_13"])*(log10(2 - 3*p2) + log10(-3*p1*(1 - 2*r_rc3)^2 + 8*(1 - r_rc3 + r_rc3^2))) + (x[,"n_02"] + x[,"n_21"])*log10((2 - 3*p2)*r_rc3*(4 + 3*p1 + 4*r_rc3 - 6*p1*r_rc3) + (4 + 3*p2)*(8 - 4*r_rc3 + p1*(-3 + 6*r_rc3))) + (x[,"n_01"] + x[,"n_22"])*log10((-2 + 3*p2)*(-1 + r_rc3)*(8 - 4*r_rc3 + p1*(-3 + 6*r_rc3)) - (4 + 3*p2)*(-4*(1 + r_rc3) + p1*(-3 + 6*r_rc3))) + (x[,"n_11"] + x[,"n_12"])*log10(-3*p1*(-2 + 3*p2)*(1 - 2*r_rc3)^2 + 8*(7 + 2*r_rc3 - 2*r_rc3^2 + 3*p2*(1 - r_rc3 + r_rc3^2)))


logL_rc3 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*(5*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_23"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_rc3)) + log(4 + 3*p1 + 4*r_rc3 - 6*p1*r_rc3)) + (x[,"n_03"] + x[,"n_20"])*(log(2 - 3*p2) + log(pmax(1e-6,r_rc3)) + log(8 - 4*r_rc3 + p1*(-3 + 6*r_rc3))) + (x[,"n_10"] + x[,"n_13"])*(log(2 - 3*p2) + log(-3*p1*(1 - 2*r_rc3)^2 + 8*(1 - r_rc3 + r_rc3^2))) + (x[,"n_02"] + x[,"n_21"])*log((2 - 3*p2)*r_rc3*(4 + 3*p1 + 4*r_rc3 - 6*p1*r_rc3) + (4 + 3*p2)*(8 - 4*r_rc3 + p1*(-3 + 6*r_rc3))) + (x[,"n_01"] + x[,"n_22"])*log((-2 + 3*p2)*(-1 + r_rc3)*(8 - 4*r_rc3 + p1*(-3 + 6*r_rc3)) - (4 + 3*p2)*(-4*(1 + r_rc3) + p1*(-3 + 6*r_rc3))) + (x[,"n_11"] + x[,"n_12"])*log(-3*p1*(-2 + 3*p2)*(1 - 2*r_rc3)^2 + 8*(7 + 2*r_rc3 - 2*r_rc3^2 + 3*p2*(1 - r_rc3 + r_rc3^2)))


logL_rc4 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*(6*log(2) + 2*log(3)) + (n00 + n23)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(4 + 3*p1 + 4*r - 6*p1*r)) + (n03 + n20)*(log(4 + 3*p2) + log(pmax(1e-6,r)) + log(8 - 4*r + p1*(-3 + 6*r))) + (n10 + n13)*(log(4 + 3*p2) + log(-3*p1*(1 - 2*r)^2 + 8*(1 - r + r^2))) + (n02 + n21)*log((4 + 3*p2)*r*(4 + 3*p1 + 4*r - 6*p1*r) - (-8 + 3*p2)*(8 - 4*r + p1*(-3 + 6*r))) + (n01 + n22)*log((-4 - 3*p2)*(-1 + r)*(8 - 4*r + p1*(-3 + 6*r)) + (-8 + 3*p2)*(-4*(1 + r) + p1*(-3 + 6*r))) + (n11 + n12)*log(3*p1*(4 + 3*p2)*(1 - 2*r)^2 - 8*(-14 - 4*r + 4*r^2 + 3*p2*(1 - r + r^2)))
return(L)}
interlogL_rc4 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
optimize(logL_rc4,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc4 <- parallel::mcmapply(interlogL_rc4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_rc4 <- -((x[,"n_11"] + x[,"n_12"])*(3*log10(2) + log10(15 - (9*p2)/4))) - (x[,"n_03"] + x[,"n_20"])*(log10(3) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_23"])*(log10(3) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_13"])*(log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_22"])*log10(-3*(-4 - 3*p2) - 6*(-8 + 3*p2)) - (x[,"n_02"] + x[,"n_21"])*log10(-6*(-8 + 3*p2) + 3*(4 + 3*p2)) + (x[,"n_00"] + x[,"n_23"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rc4)) + log10(4 + 3*p1 + 4*r_rc4 - 6*p1*r_rc4)) + (x[,"n_03"] + x[,"n_20"])*(log10(4 + 3*p2) + log10(pmax(1e-6,r_rc4)) + log10(8 - 4*r_rc4 + p1*(-3 + 6*r_rc4))) + (x[,"n_10"] + x[,"n_13"])*(log10(4 + 3*p2) + log10(-3*p1*(1 - 2*r_rc4)^2 + 8*(1 - r_rc4 + r_rc4^2))) + (x[,"n_02"] + x[,"n_21"])*log10((4 + 3*p2)*r_rc4*(4 + 3*p1 + 4*r_rc4 - 6*p1*r_rc4) - (-8 + 3*p2)*(8 - 4*r_rc4 + p1*(-3 + 6*r_rc4))) + (x[,"n_01"] + x[,"n_22"])*log10((-4 - 3*p2)*(-1 + r_rc4)*(8 - 4*r_rc4 + p1*(-3 + 6*r_rc4)) + (-8 + 3*p2)*(-4*(1 + r_rc4) + p1*(-3 + 6*r_rc4))) + (x[,"n_11"] + x[,"n_12"])*log10(3*p1*(4 + 3*p2)*(1 - 2*r_rc4)^2 - 8*(-14 - 4*r_rc4 + 4*r_rc4^2 + 3*p2*(1 - r_rc4 + r_rc4^2)))


logL_rc4 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*(6*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_23"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_rc4)) + log(4 + 3*p1 + 4*r_rc4 - 6*p1*r_rc4)) + (x[,"n_03"] + x[,"n_20"])*(log(4 + 3*p2) + log(pmax(1e-6,r_rc4)) + log(8 - 4*r_rc4 + p1*(-3 + 6*r_rc4))) + (x[,"n_10"] + x[,"n_13"])*(log(4 + 3*p2) + log(-3*p1*(1 - 2*r_rc4)^2 + 8*(1 - r_rc4 + r_rc4^2))) + (x[,"n_02"] + x[,"n_21"])*log((4 + 3*p2)*r_rc4*(4 + 3*p1 + 4*r_rc4 - 6*p1*r_rc4) - (-8 + 3*p2)*(8 - 4*r_rc4 + p1*(-3 + 6*r_rc4))) + (x[,"n_01"] + x[,"n_22"])*log((-4 - 3*p2)*(-1 + r_rc4)*(8 - 4*r_rc4 + p1*(-3 + 6*r_rc4)) + (-8 + 3*p2)*(-4*(1 + r_rc4) + p1*(-3 + 6*r_rc4))) + (x[,"n_11"] + x[,"n_12"])*log(3*p1*(4 + 3*p2)*(1 - 2*r_rc4)^2 - 8*(-14 - 4*r_rc4 + 4*r_rc4^2 + 3*p2*(1 - r_rc4 + r_rc4^2)))


logL_rr1 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*(5*log(2) + 2*log(3)) + (n03 + n20)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(4 + 3*p1 - 2*r - 6*p1*r)) + (n00 + n23)*(log(4 + 3*p2) + log(pmax(1e-6,r)) + log(2 - 3*p1 + 2*r + 6*p1*r)) + (n10 + n13)*(log(4 + 3*p2) + log(2 - 3*p1*(1 - 2*r)^2 + 4*r - 4*r^2)) + (n01 + n22)*log((4 + 3*p2)*(4 + p1*(3 - 6*r) - 2*r)*r - (-8 + 3*p2)*(2 - 3*p1 + 2*r + 6*p1*r)) + (n02 + n21)*log((-8 + 3*p2)*(-4 - 3*p1 + 2*r + 6*p1*r) - (4 + 3*p2)*(-1 + r)*(2 - 3*p1 + 2*r + 6*p1*r)) + (n11 + n12)*log(3*p1*(4 + 3*p2)*(1 - 2*r)^2 + 2*(8*(4 - r + r^2) + p2*(-3 - 6*r + 6*r^2)))
return(L)}
interlogL_rr1 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
optimize(logL_rr1,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr1 <- parallel::mcmapply(interlogL_rr1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_rr1 <- -((x[,"n_11"] + x[,"n_12"])*(log10(2) + log10(30 - (9*p2)/2))) - (x[,"n_10"] + x[,"n_13"])*(log10(3) + log10(4 + 3*p2)) - (x[,"n_03"] + x[,"n_20"])*(-log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_23"])*(-log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_21"])*log10(-3*(-8 + 3*p2) + (3*(4 + 3*p2))/2) - (x[,"n_01"] + x[,"n_22"])*log10(-3*(-8 + 3*p2) + (3*(4 + 3*p2))/2) + (x[,"n_03"] + x[,"n_20"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rr1)) + log10(4 + 3*p1 - 2*r_rr1 - 6*p1*r_rr1)) + (x[,"n_00"] + x[,"n_23"])*(log10(4 + 3*p2) + log10(pmax(1e-6,r_rr1)) + log10(2 - 3*p1 + 2*r_rr1 + 6*p1*r_rr1)) + (x[,"n_10"] + x[,"n_13"])*(log10(4 + 3*p2) + log10(2 - 3*p1*(1 - 2*r_rr1)^2 + 4*r_rr1 - 4*r_rr1^2)) + (x[,"n_01"] + x[,"n_22"])*log10((4 + 3*p2)*(4 + p1*(3 - 6*r_rr1) - 2*r_rr1)*r_rr1 - (-8 + 3*p2)*(2 - 3*p1 + 2*r_rr1 + 6*p1*r_rr1)) + (x[,"n_02"] + x[,"n_21"])*log10((-8 + 3*p2)*(-4 - 3*p1 + 2*r_rr1 + 6*p1*r_rr1) - (4 + 3*p2)*(-1 + r_rr1)*(2 - 3*p1 + 2*r_rr1 + 6*p1*r_rr1)) + (x[,"n_11"] + x[,"n_12"])*log10(3*p1*(4 + 3*p2)*(1 - 2*r_rr1)^2 + 2*(8*(4 - r_rr1 + r_rr1^2) + p2*(-3 - 6*r_rr1 + 6*r_rr1^2)))


logL_rr1 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*(5*log(2) + 2*log(3)) + (x[,"n_03"] + x[,"n_20"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_rr1)) + log(4 + 3*p1 - 2*r_rr1 - 6*p1*r_rr1)) + (x[,"n_00"] + x[,"n_23"])*(log(4 + 3*p2) + log(pmax(1e-6,r_rr1)) + log(2 - 3*p1 + 2*r_rr1 + 6*p1*r_rr1)) + (x[,"n_10"] + x[,"n_13"])*(log(4 + 3*p2) + log(2 - 3*p1*(1 - 2*r_rr1)^2 + 4*r_rr1 - 4*r_rr1^2)) + (x[,"n_01"] + x[,"n_22"])*log((4 + 3*p2)*(4 + p1*(3 - 6*r_rr1) - 2*r_rr1)*r_rr1 - (-8 + 3*p2)*(2 - 3*p1 + 2*r_rr1 + 6*p1*r_rr1)) + (x[,"n_02"] + x[,"n_21"])*log((-8 + 3*p2)*(-4 - 3*p1 + 2*r_rr1 + 6*p1*r_rr1) - (4 + 3*p2)*(-1 + r_rr1)*(2 - 3*p1 + 2*r_rr1 + 6*p1*r_rr1)) + (x[,"n_11"] + x[,"n_12"])*log(3*p1*(4 + 3*p2)*(1 - 2*r_rr1)^2 + 2*(8*(4 - r_rr1 + r_rr1^2) + p2*(-3 - 6*r_rr1 + 6*r_rr1^2)))


logL_rr2 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*(4*log(2) + 2*log(3)) + (n03 + n20)*(log(2 - 3*p2) + log(4 + p1*(3 - 6*r) - 2*r) + log(pmax(1e-6,1 - r))) + (n00 + n23)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(2 - 3*p1 + 2*r + 6*p1*r)) + (n10 + n13)*(log(2 - 3*p2) + log(2 - 3*p1*(1 - 2*r)^2 + 4*r - 4*r^2)) + (n11 + n12)*log(-3*p1*(-2 + 3*p2)*(1 - 2*r)^2 + p2*(6 + 12*r - 12*r^2) + 8*(4 - r + r^2)) + (n02 + n21)*log(6*p2*(1 - r + r^2) - 4*(-5 + 2*r + r^2) + 3*p1*(-1 + 2*r)*(3*p2*(-2 + r) - 2*(1 + r))) + (n01 + n22)*log(8 + 16*r - 4*r^2 + 6*p2*(1 - r + r^2) + 3*p1*(-1 + 2*r)*(4 - 2*r + 3*p2*(1 + r)))
return(L)}
interlogL_rr2 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
optimize(logL_rr2,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr2 <- parallel::mcmapply(interlogL_rr2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_rr2 <- -((x[,"n_10"] + x[,"n_13"])*(log10(3) + log10(2 - 3*p2))) - (x[,"n_03"] + x[,"n_20"])*(-log10(2) + log10(3) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_23"])*(-log10(2) + log10(3) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_21"])*log10(15 + (9*p2)/2) - (x[,"n_01"] + x[,"n_22"])*log10(15 + (9*p2)/2) - (x[,"n_11"] + x[,"n_12"])*log10(30 + 9*p2) + (x[,"n_03"] + x[,"n_20"])*(log10(2 - 3*p2) + log10(4 + p1*(3 - 6*r_rr2) - 2*r_rr2) + log10(pmax(1e-6,1 - r_rr2))) + (x[,"n_00"] + x[,"n_23"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_rr2)) + log10(2 - 3*p1 + 2*r_rr2 + 6*p1*r_rr2)) + (x[,"n_10"] + x[,"n_13"])*(log10(2 - 3*p2) + log10(2 - 3*p1*(1 - 2*r_rr2)^2 + 4*r_rr2 - 4*r_rr2^2)) + (x[,"n_11"] + x[,"n_12"])*log10(-3*p1*(-2 + 3*p2)*(1 - 2*r_rr2)^2 + p2*(6 + 12*r_rr2 - 12*r_rr2^2) + 8*(4 - r_rr2 + r_rr2^2)) + (x[,"n_02"] + x[,"n_21"])*log10(6*p2*(1 - r_rr2 + r_rr2^2) - 4*(-5 + 2*r_rr2 + r_rr2^2) + 3*p1*(-1 + 2*r_rr2)*(3*p2*(-2 + r_rr2) - 2*(1 + r_rr2))) + (x[,"n_01"] + x[,"n_22"])*log10(8 + 16*r_rr2 - 4*r_rr2^2 + 6*p2*(1 - r_rr2 + r_rr2^2) + 3*p1*(-1 + 2*r_rr2)*(4 - 2*r_rr2 + 3*p2*(1 + r_rr2)))


logL_rr2 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*(4*log(2) + 2*log(3)) + (x[,"n_03"] + x[,"n_20"])*(log(2 - 3*p2) + log(4 + p1*(3 - 6*r_rr2) - 2*r_rr2) + log(pmax(1e-6,1 - r_rr2))) + (x[,"n_00"] + x[,"n_23"])*(log(2 - 3*p2) + log(pmax(1e-6,r_rr2)) + log(2 - 3*p1 + 2*r_rr2 + 6*p1*r_rr2)) + (x[,"n_10"] + x[,"n_13"])*(log(2 - 3*p2) + log(2 - 3*p1*(1 - 2*r_rr2)^2 + 4*r_rr2 - 4*r_rr2^2)) + (x[,"n_11"] + x[,"n_12"])*log(-3*p1*(-2 + 3*p2)*(1 - 2*r_rr2)^2 + p2*(6 + 12*r_rr2 - 12*r_rr2^2) + 8*(4 - r_rr2 + r_rr2^2)) + (x[,"n_02"] + x[,"n_21"])*log(6*p2*(1 - r_rr2 + r_rr2^2) - 4*(-5 + 2*r_rr2 + r_rr2^2) + 3*p1*(-1 + 2*r_rr2)*(3*p2*(-2 + r_rr2) - 2*(1 + r_rr2))) + (x[,"n_01"] + x[,"n_22"])*log(8 + 16*r_rr2 - 4*r_rr2^2 + 6*p2*(1 - r_rr2 + r_rr2^2) + 3*p1*(-1 + 2*r_rr2)*(4 - 2*r_rr2 + 3*p2*(1 + r_rr2)))


logL_rr3 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*(6*log(2) + 2*log(3)) + (n00 + n23)*(log(4 + 3*p2) + log(pmax(1e-6,r)) + log(4 + 3*p1 + 4*r - 6*p1*r)) + (n10 + n13)*(log(4 + 3*p2) + log(4 + 3*p1*(1 - 2*r)^2 + 8*r - 8*r^2)) + (n03 + n20)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(8 - 4*r + p1*(-3 + 6*r))) + (n01 + n22)*log((4 + 3*p2)*r*(8 - 4*r + p1*(-3 + 6*r)) + (-8 + 3*p2)*(-4*(1 + r) + p1*(-3 + 6*r))) + (n02 + n21)*log((8 - 3*p2)*(8 - 4*r + p1*(-3 + 6*r)) + (4 + 3*p2)*(-1 + r)*(-4*(1 + r) + p1*(-3 + 6*r))) + (n11 + n12)*log(-3*p1*(4 + 3*p2)*(1 - 2*r)^2 + 4*(8*(4 - r + r^2) + p2*(-3 - 6*r + 6*r^2)))
return(L)}
interlogL_rr3 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
optimize(logL_rr3,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr3 <- parallel::mcmapply(interlogL_rr3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_rr3 <- -((x[,"n_11"] + x[,"n_12"])*(2*log10(2) + log10(30 - (9*p2)/2))) - (x[,"n_03"] + x[,"n_20"])*(log10(3) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_23"])*(log10(3) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_13"])*(log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_21"])*log10(6*(8 - 3*p2) + 3*(4 + 3*p2)) - (x[,"n_01"] + x[,"n_22"])*log10(-6*(-8 + 3*p2) + 3*(4 + 3*p2)) + (x[,"n_00"] + x[,"n_23"])*(log10(4 + 3*p2) + log10(pmax(1e-6,r_rr3)) + log10(4 + 3*p1 + 4*r_rr3 - 6*p1*r_rr3)) + (x[,"n_10"] + x[,"n_13"])*(log10(4 + 3*p2) + log10(4 + 3*p1*(1 - 2*r_rr3)^2 + 8*r_rr3 - 8*r_rr3^2)) + (x[,"n_03"] + x[,"n_20"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rr3)) + log10(8 - 4*r_rr3 + p1*(-3 + 6*r_rr3))) + (x[,"n_01"] + x[,"n_22"])*log10((4 + 3*p2)*r_rr3*(8 - 4*r_rr3 + p1*(-3 + 6*r_rr3)) + (-8 + 3*p2)*(-4*(1 + r_rr3) + p1*(-3 + 6*r_rr3))) + (x[,"n_02"] + x[,"n_21"])*log10((8 - 3*p2)*(8 - 4*r_rr3 + p1*(-3 + 6*r_rr3)) + (4 + 3*p2)*(-1 + r_rr3)*(-4*(1 + r_rr3) + p1*(-3 + 6*r_rr3))) + (x[,"n_11"] + x[,"n_12"])*log10(-3*p1*(4 + 3*p2)*(1 - 2*r_rr3)^2 + 4*(8*(4 - r_rr3 + r_rr3^2) + p2*(-3 - 6*r_rr3 + 6*r_rr3^2)))


logL_rr3 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*(6*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_23"])*(log(4 + 3*p2) + log(pmax(1e-6,r_rr3)) + log(4 + 3*p1 + 4*r_rr3 - 6*p1*r_rr3)) + (x[,"n_10"] + x[,"n_13"])*(log(4 + 3*p2) + log(4 + 3*p1*(1 - 2*r_rr3)^2 + 8*r_rr3 - 8*r_rr3^2)) + (x[,"n_03"] + x[,"n_20"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_rr3)) + log(8 - 4*r_rr3 + p1*(-3 + 6*r_rr3))) + (x[,"n_01"] + x[,"n_22"])*log((4 + 3*p2)*r_rr3*(8 - 4*r_rr3 + p1*(-3 + 6*r_rr3)) + (-8 + 3*p2)*(-4*(1 + r_rr3) + p1*(-3 + 6*r_rr3))) + (x[,"n_02"] + x[,"n_21"])*log((8 - 3*p2)*(8 - 4*r_rr3 + p1*(-3 + 6*r_rr3)) + (4 + 3*p2)*(-1 + r_rr3)*(-4*(1 + r_rr3) + p1*(-3 + 6*r_rr3))) + (x[,"n_11"] + x[,"n_12"])*log(-3*p1*(4 + 3*p2)*(1 - 2*r_rr3)^2 + 4*(8*(4 - r_rr3 + r_rr3^2) + p2*(-3 - 6*r_rr3 + 6*r_rr3^2)))


logL_rr4 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*(5*log(2) + 2*log(3)) + (n00 + n23)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(4 + 3*p1 + 4*r - 6*p1*r)) + (n10 + n13)*(log(2 - 3*p2) + log(4 + 3*p1*(1 - 2*r)^2 + 8*r - 8*r^2)) + (n03 + n20)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(8 - 4*r + p1*(-3 + 6*r))) + (n02 + n21)*log((-2 + 3*p2)*(-1 + r)*(4 + 3*p1 + 4*r - 6*p1*r) + (4 + 3*p2)*(8 - 4*r + p1*(-3 + 6*r))) + (n01 + n22)*log((2 - 3*p2)*r*(8 - 4*r + p1*(-3 + 6*r)) - (4 + 3*p2)*(-4*(1 + r) + p1*(-3 + 6*r))) + (n11 + n12)*log(3*p1*(-2 + 3*p2)*(1 - 2*r)^2 + 4*(p2*(3 + 6*r - 6*r^2) + 4*(4 - r + r^2)))
return(L)}
interlogL_rr4 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
optimize(logL_rr4,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr4 <- parallel::mcmapply(interlogL_rr4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_rr4 <- -((x[,"n_03"] + x[,"n_20"])*(log10(3) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_23"])*(log10(3) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_13"])*(log10(2) + log10(3) + log10(2 - 3*p2)) - (x[,"n_11"] + x[,"n_12"])*(2*log10(2) + log10(15 + (9*p2)/2)) - (x[,"n_01"] + x[,"n_22"])*log10(3*(2 - 3*p2) + 6*(4 + 3*p2)) - (x[,"n_02"] + x[,"n_21"])*log10(-3*(-2 + 3*p2) + 6*(4 + 3*p2)) + (x[,"n_00"] + x[,"n_23"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_rr4)) + log10(4 + 3*p1 + 4*r_rr4 - 6*p1*r_rr4)) + (x[,"n_10"] + x[,"n_13"])*(log10(2 - 3*p2) + log10(4 + 3*p1*(1 - 2*r_rr4)^2 + 8*r_rr4 - 8*r_rr4^2)) + (x[,"n_03"] + x[,"n_20"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rr4)) + log10(8 - 4*r_rr4 + p1*(-3 + 6*r_rr4))) + (x[,"n_02"] + x[,"n_21"])*log10((-2 + 3*p2)*(-1 + r_rr4)*(4 + 3*p1 + 4*r_rr4 - 6*p1*r_rr4) + (4 + 3*p2)*(8 - 4*r_rr4 + p1*(-3 + 6*r_rr4))) + (x[,"n_01"] + x[,"n_22"])*log10((2 - 3*p2)*r_rr4*(8 - 4*r_rr4 + p1*(-3 + 6*r_rr4)) - (4 + 3*p2)*(-4*(1 + r_rr4) + p1*(-3 + 6*r_rr4))) + (x[,"n_11"] + x[,"n_12"])*log10(3*p1*(-2 + 3*p2)*(1 - 2*r_rr4)^2 + 4*(p2*(3 + 6*r_rr4 - 6*r_rr4^2) + 4*(4 - r_rr4 + r_rr4^2)))


logL_rr4 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*(5*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_23"])*(log(2 - 3*p2) + log(pmax(1e-6,r_rr4)) + log(4 + 3*p1 + 4*r_rr4 - 6*p1*r_rr4)) + (x[,"n_10"] + x[,"n_13"])*(log(2 - 3*p2) + log(4 + 3*p1*(1 - 2*r_rr4)^2 + 8*r_rr4 - 8*r_rr4^2)) + (x[,"n_03"] + x[,"n_20"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_rr4)) + log(8 - 4*r_rr4 + p1*(-3 + 6*r_rr4))) + (x[,"n_02"] + x[,"n_21"])*log((-2 + 3*p2)*(-1 + r_rr4)*(4 + 3*p1 + 4*r_rr4 - 6*p1*r_rr4) + (4 + 3*p2)*(8 - 4*r_rr4 + p1*(-3 + 6*r_rr4))) + (x[,"n_01"] + x[,"n_22"])*log((2 - 3*p2)*r_rr4*(8 - 4*r_rr4 + p1*(-3 + 6*r_rr4)) - (4 + 3*p2)*(-4*(1 + r_rr4) + p1*(-3 + 6*r_rr4))) + (x[,"n_11"] + x[,"n_12"])*log(3*p1*(-2 + 3*p2)*(1 - 2*r_rr4)^2 + 4*(p2*(3 + 6*r_rr4 - 6*r_rr4^2) + 4*(4 - r_rr4 + r_rr4^2)))


return(list(
r_mat = cbind(r_cc1,r_cc2,r_cr1,r_cr2,r_rc1,r_rc2,r_rc3,r_rc4,r_rr1,r_rr2,r_rr3,r_rr4,0.499),
LOD_mat = cbind(LOD_cc1,LOD_cc2,LOD_cr1,LOD_cr2,LOD_rc1,LOD_rc2,LOD_rc3,LOD_rc4,LOD_rr1,LOD_rr2,LOD_rr3,LOD_rr4,0),
logL_mat = cbind(logL_cc1,logL_cc2,logL_cr1,logL_cr2,logL_rc1,logL_rc2,logL_rc3,logL_rc4,logL_rr1,logL_rr2,logL_rr3,logL_rr4,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling coupling","coupling repulsion","coupling repulsion","repulsion coupling","repulsion coupling","repulsion coupling","repulsion coupling","repulsion repulsion","repulsion repulsion","repulsion repulsion","repulsion repulsion","unknown")
)
)
}

#' @rdname p4_functions
p4_1.1_1.3 <- function(x,p1,p2,ncores=1){


logL_cc1 <- function(r,n01,n02,n03,n11,n12,n13,n21,n22,n23,p1,p2) {
L <- (-4*n01 - 4*n02 - 4*n03 - 4*n11 - 3*n12 - 4*n13 - 4*n21 - 4*n22 - 4*n23)*log(2) + (-n01 - n02 - n03 - n11 - n12 - n13 - n21 - n22 - n23)*log(3) + (n03 + n21)*(log(pmax(1e-6,r)) + log(4 + 3*p2 + 4*r - 6*p2*r)) + (n02 + n11 + n13 + n22)*log(4 + 3*p2*(1 - 2*r)^2 + 8*r - 8*r^2) + (n01 + n23)*(log(pmax(1e-6,1 - r)) + log(8 - 4*r + p2*(-3 + 6*r))) + n12*log(-3*p2*(1 - 2*r)^2 + 8*(1 - r + r^2))
return(L)}
interlogL_cc1 <- function(n01,n02,n03,n11,n12,n13,n21,n22,n23,p1,p2) {
optimize(logL_cc1,c(0,0.5), n01,n02,n03,n11,n12,n13,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc1 <- parallel::mcmapply(interlogL_cc1,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_cc1 <- -((x[,"n_03"] + x[,"n_21"])*log10(3)) - (x[,"n_01"] + x[,"n_23"])*log10(3) - x[,"n_12"]*(log10(2) + log10(3)) - (x[,"n_02"] + x[,"n_11"] + x[,"n_13"] + x[,"n_22"])*(log10(2) + log10(3)) + (x[,"n_03"] + x[,"n_21"])*(log10(pmax(1e-6,r_cc1)) + log10(4 + 3*p2 + 4*r_cc1 - 6*p2*r_cc1)) + (x[,"n_02"] + x[,"n_11"] + x[,"n_13"] + x[,"n_22"])*log10(4 + 3*p2*(1 - 2*r_cc1)^2 + 8*r_cc1 - 8*r_cc1^2) + (x[,"n_01"] + x[,"n_23"])*(log10(pmax(1e-6,1 - r_cc1)) + log10(8 - 4*r_cc1 + p2*(-3 + 6*r_cc1))) + x[,"n_12"]*log10(-3*p2*(1 - 2*r_cc1)^2 + 8*(1 - r_cc1 + r_cc1^2))


logL_cc1 <- (-4*x[,"n_01"] - 4*x[,"n_02"] - 4*x[,"n_03"] - 4*x[,"n_11"] - 3*x[,"n_12"] - 4*x[,"n_13"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 4*x[,"n_23"])*log(2) + (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*log(3) + (x[,"n_03"] + x[,"n_21"])*(log(pmax(1e-6,r_cc1)) + log(4 + 3*p2 + 4*r_cc1 - 6*p2*r_cc1)) + (x[,"n_02"] + x[,"n_11"] + x[,"n_13"] + x[,"n_22"])*log(4 + 3*p2*(1 - 2*r_cc1)^2 + 8*r_cc1 - 8*r_cc1^2) + (x[,"n_01"] + x[,"n_23"])*(log(pmax(1e-6,1 - r_cc1)) + log(8 - 4*r_cc1 + p2*(-3 + 6*r_cc1))) + x[,"n_12"]*log(-3*p2*(1 - 2*r_cc1)^2 + 8*(1 - r_cc1 + r_cc1^2))


logL_cc2 <- function(r,n01,n02,n03,n11,n12,n13,n21,n22,n23,p1,p2) {
L <- (-3*n01 - 3*n02 - 3*n03 - 3*n11 - 2*n12 - 3*n13 - 3*n21 - 3*n22 - 3*n23)*log(2) + (-n01 - n02 - n03 - n11 - n12 - n13 - n21 - n22 - n23)*log(3) + (n01 + n23)*(log(pmax(1e-6,1 - r)) + log(4 + 3*p2 - 2*r - 6*p2*r)) + (n03 + n21)*(log(pmax(1e-6,r)) + log(2 - 3*p2 + 2*r + 6*p2*r)) + (n02 + n11 + n13 + n22)*log(2 - 3*p2*(1 - 2*r)^2 + 4*r - 4*r^2) + n12*log(3*p2*(1 - 2*r)^2 + 4*(1 - r + r^2))
return(L)}
interlogL_cc2 <- function(n01,n02,n03,n11,n12,n13,n21,n22,n23,p1,p2) {
optimize(logL_cc2,c(0,0.5), n01,n02,n03,n11,n12,n13,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc2 <- parallel::mcmapply(interlogL_cc2,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_cc2 <- -(x[,"n_12"]*log10(3)) - (x[,"n_02"] + x[,"n_11"] + x[,"n_13"] + x[,"n_22"])*log10(3) - (x[,"n_03"] + x[,"n_21"])*(-log10(2) + log10(3)) - (x[,"n_01"] + x[,"n_23"])*(-log10(2) + log10(3)) + (x[,"n_01"] + x[,"n_23"])*(log10(pmax(1e-6,1 - r_cc2)) + log10(4 + 3*p2 - 2*r_cc2 - 6*p2*r_cc2)) + (x[,"n_03"] + x[,"n_21"])*(log10(pmax(1e-6,r_cc2)) + log10(2 - 3*p2 + 2*r_cc2 + 6*p2*r_cc2)) + (x[,"n_02"] + x[,"n_11"] + x[,"n_13"] + x[,"n_22"])*log10(2 - 3*p2*(1 - 2*r_cc2)^2 + 4*r_cc2 - 4*r_cc2^2) + x[,"n_12"]*log10(3*p2*(1 - 2*r_cc2)^2 + 4*(1 - r_cc2 + r_cc2^2))


logL_cc2 <- (-3*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 3*x[,"n_11"] - 2*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"])*log(2) + (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*log(3) + (x[,"n_01"] + x[,"n_23"])*(log(pmax(1e-6,1 - r_cc2)) + log(4 + 3*p2 - 2*r_cc2 - 6*p2*r_cc2)) + (x[,"n_03"] + x[,"n_21"])*(log(pmax(1e-6,r_cc2)) + log(2 - 3*p2 + 2*r_cc2 + 6*p2*r_cc2)) + (x[,"n_02"] + x[,"n_11"] + x[,"n_13"] + x[,"n_22"])*log(2 - 3*p2*(1 - 2*r_cc2)^2 + 4*r_cc2 - 4*r_cc2^2) + x[,"n_12"]*log(3*p2*(1 - 2*r_cc2)^2 + 4*(1 - r_cc2 + r_cc2^2))


logL_cr <- function(r,n01,n02,n03,n11,n13,n21,n22,n23,n12,p1,p2) {
L <- 2*(-n01 - n02 - n03 - n11 - n13 - n21 - n22 - n23)*log(2) + (n01 + n03 + n12 + n21 + n23)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n02 + n11 + n13 + n22)*log(1 - 2*r + 2*r^2)
return(L)}
interlogL_cr <- function(n01,n02,n03,n11,n13,n21,n22,n23,n12,p1,p2) {
optimize(logL_cr,c(0,0.5), n01,n02,n03,n11,n13,n21,n22,n23,n12,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr <- parallel::mcmapply(interlogL_cr,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_11"],x[,"n_13"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_12"], p1, p2, mc.cores = ncores)


LOD_cr <- (x[,"n_02"] + x[,"n_11"] + x[,"n_13"] + x[,"n_22"])*log10(2) + 2*(x[,"n_01"] + x[,"n_03"] + x[,"n_12"] + x[,"n_21"] + x[,"n_23"])*log10(2) + (x[,"n_01"] + x[,"n_03"] + x[,"n_12"] + x[,"n_21"] + x[,"n_23"])*(log10(pmax(1e-6,1 - r_cr)) + log10(pmax(1e-6,r_cr))) + (x[,"n_02"] + x[,"n_11"] + x[,"n_13"] + x[,"n_22"])*log10(1 - 2*r_cr + 2*r_cr^2)


logL_cr <- 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_11"] - x[,"n_13"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*log(2) + (x[,"n_01"] + x[,"n_03"] + x[,"n_12"] + x[,"n_21"] + x[,"n_23"])*(log(pmax(1e-6,1 - r_cr)) + log(pmax(1e-6,r_cr))) + (x[,"n_02"] + x[,"n_11"] + x[,"n_13"] + x[,"n_22"])*log(1 - 2*r_cr + 2*r_cr^2)


logL_rc1 <- function(r,n01,n02,n03,n11,n12,n13,n21,n22,n23,p1,p2) {
L <- (-5*n01 - 4*n02 - 5*n03 - 4*n11 - 3*n12 - 4*n13 - 5*n21 - 4*n22 - 5*n23)*log(2) + 2*(-n01 - n02 - n03 - n11 - n12 - n13 - n21 - n22 - n23)*log(3) + (n03 + n21)*(log(4 + p1*(3 - 6*r) - 2*r) + log(4 + 3*p2 + 4*r - 6*p2*r)) + (n02 + n11 + n13 + n22)*log(20 - 3*p2*(1 - 2*r)^2 - 3*p1*(-2 + 3*p2)*(1 - 2*r)^2 - 8*r + 8*r^2) + (n01 + n23)*(log(2 - 3*p1 + 2*r + 6*p1*r) + log(8 - 4*r + p2*(-3 + 6*r))) + n12*log(3*p2*(1 - 2*r)^2 + 3*p1*(-2 + 3*p2)*(1 - 2*r)^2 + 8*(2 + r - r^2))
return(L)}
interlogL_rc1 <- function(n01,n02,n03,n11,n12,n13,n21,n22,n23,p1,p2) {
optimize(logL_rc1,c(0,0.5), n01,n02,n03,n11,n12,n13,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc1 <- parallel::mcmapply(interlogL_rc1,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


# LOD_rc1 <- -(x[,"n_12"]*(log10(2) + 2*log10(3))) - (x[,"n_03"] + x[,"n_21"])*(log10(2) + 2*log10(3)) - (x[,"n_02"] + x[,"n_11"] + x[,"n_13"] + x[,"n_22"])*(log10(2) + 2*log10(3)) - (x[,"n_01"] + x[,"n_23"])*(log10(2) + 2*log10(3)) + (x[,"n_03"] + x[,"n_21"])*(log10(4 + p1*(3 - 6*r_rc1) - 2*r_rc1) + log10(4 + 3*p2 + 4*r_rc1 - 6*p2*r_rc1)) + (x[,"n_02"] + x[,"n_11"] + x[,"n_13"] + x[,"n_22"])*log10(20 - 3*p2*(1 - 2*r_rc1)^2 - 3*p1*(-2 + 3*p2)*(1 - 2*r_rc1)^2 - 8*r_rc1 + 8*r_rc1^2) + (x[,"n_01"] + x[,"n_23"])*(log10(2 - 3*p1 + 2*r_rc1 + 6*p1*r_rc1) + log10(8 - 4*r_rc1 + p2*(-3 + 6*r_rc1))) + x[,"n_12"]*log10(3*p2*(1 - 2*r_rc1)^2 + 3*p1*(-2 + 3*p2)*(1 - 2*r_rc1)^2 + 8*(2 + r_rc1 - r_rc1^2))


logL_rc1 <- (-5*x[,"n_01"] - 4*x[,"n_02"] - 5*x[,"n_03"] - 4*x[,"n_11"] - 3*x[,"n_12"] - 4*x[,"n_13"] - 5*x[,"n_21"] - 4*x[,"n_22"] - 5*x[,"n_23"])*log(2) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*log(3) + (x[,"n_03"] + x[,"n_21"])*(log(4 + p1*(3 - 6*r_rc1) - 2*r_rc1) + log(4 + 3*p2 + 4*r_rc1 - 6*p2*r_rc1)) + (x[,"n_02"] + x[,"n_11"] + x[,"n_13"] + x[,"n_22"])*log(20 - 3*p2*(1 - 2*r_rc1)^2 - 3*p1*(-2 + 3*p2)*(1 - 2*r_rc1)^2 - 8*r_rc1 + 8*r_rc1^2) + (x[,"n_01"] + x[,"n_23"])*(log(2 - 3*p1 + 2*r_rc1 + 6*p1*r_rc1) + log(8 - 4*r_rc1 + p2*(-3 + 6*r_rc1))) + x[,"n_12"]*log(3*p2*(1 - 2*r_rc1)^2 + 3*p1*(-2 + 3*p2)*(1 - 2*r_rc1)^2 + 8*(2 + r_rc1 - r_rc1^2))


logL_rc2 <- function(r,n01,n02,n03,n11,n12,n13,n21,n22,n23,p1,p2) {
L <- (-4*n01 - 3*n02 - 4*n03 - 3*n11 - 2*n12 - 3*n13 - 4*n21 - 3*n22 - 4*n23)*log(2) + 2*(-n01 - n02 - n03 - n11 - n12 - n13 - n21 - n22 - n23)*log(3) + (n01 + n23)*(log(4 + p2*(3 - 6*r) - 2*r) + log(2 - 3*p1 + 2*r + 6*p1*r)) + (n03 + n21)*(log(4 + p1*(3 - 6*r) - 2*r) + log(2 - 3*p2 + 2*r + 6*p2*r)) + n12*log(8 - 3*p2*(1 - 2*r)^2 - 3*p1*(1 + 3*p2)*(1 - 2*r)^2 + 4*r - 4*r^2) + (n02 + n11 + n13 + n22)*log(10 + 3*p2*(1 - 2*r)^2 + 3*p1*(1 + 3*p2)*(1 - 2*r)^2 - 4*r + 4*r^2)
return(L)}
interlogL_rc2 <- function(n01,n02,n03,n11,n12,n13,n21,n22,n23,p1,p2) {
optimize(logL_rc2,c(0,0.5), n01,n02,n03,n11,n12,n13,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc2 <- parallel::mcmapply(interlogL_rc2,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


# LOD_rc2 <- -2*x[,"n_12"]*log10(3) - 2*(x[,"n_03"] + x[,"n_21"])*log10(3) - 2*(x[,"n_02"] + x[,"n_11"] + x[,"n_13"] + x[,"n_22"])*log10(3) - 2*(x[,"n_01"] + x[,"n_23"])*log10(3) + (x[,"n_01"] + x[,"n_23"])*(log10(4 + p2*(3 - 6*r_rc2) - 2*r_rc2) + log10(2 - 3*p1 + 2*r_rc2 + 6*p1*r_rc2)) + (x[,"n_03"] + x[,"n_21"])*(log10(4 + p1*(3 - 6*r_rc2) - 2*r_rc2) + log10(2 - 3*p2 + 2*r_rc2 + 6*p2*r_rc2)) + x[,"n_12"]*log10(8 - 3*p2*(1 - 2*r_rc2)^2 - 3*p1*(1 + 3*p2)*(1 - 2*r_rc2)^2 + 4*r_rc2 - 4*r_rc2^2) + (x[,"n_02"] + x[,"n_11"] + x[,"n_13"] + x[,"n_22"])*log10(10 + 3*p2*(1 - 2*r_rc2)^2 + 3*p1*(1 + 3*p2)*(1 - 2*r_rc2)^2 - 4*r_rc2 + 4*r_rc2^2)


logL_rc2 <- (-4*x[,"n_01"] - 3*x[,"n_02"] - 4*x[,"n_03"] - 3*x[,"n_11"] - 2*x[,"n_12"] - 3*x[,"n_13"] - 4*x[,"n_21"] - 3*x[,"n_22"] - 4*x[,"n_23"])*log(2) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*log(3) + (x[,"n_01"] + x[,"n_23"])*(log(4 + p2*(3 - 6*r_rc2) - 2*r_rc2) + log(2 - 3*p1 + 2*r_rc2 + 6*p1*r_rc2)) + (x[,"n_03"] + x[,"n_21"])*(log(4 + p1*(3 - 6*r_rc2) - 2*r_rc2) + log(2 - 3*p2 + 2*r_rc2 + 6*p2*r_rc2)) + x[,"n_12"]*log(8 - 3*p2*(1 - 2*r_rc2)^2 - 3*p1*(1 + 3*p2)*(1 - 2*r_rc2)^2 + 4*r_rc2 - 4*r_rc2^2) + (x[,"n_02"] + x[,"n_11"] + x[,"n_13"] + x[,"n_22"])*log(10 + 3*p2*(1 - 2*r_rc2)^2 + 3*p1*(1 + 3*p2)*(1 - 2*r_rc2)^2 - 4*r_rc2 + 4*r_rc2^2)


logL_rc3 <- function(r,n01,n02,n03,n11,n12,n13,n21,n22,n23,p1,p2) {
L <- (-6*n01 - 5*n02 - 6*n03 - 5*n11 - 4*n12 - 5*n13 - 6*n21 - 5*n22 - 6*n23)*log(2) + 2*(-n01 - n02 - n03 - n11 - n12 - n13 - n21 - n22 - n23)*log(3) + (n03 + n21)*(log(4 + 3*p2 + 4*r - 6*p2*r) + log(8 - 4*r + p1*(-3 + 6*r))) + (n01 + n23)*(log(4 + 3*p1 + 4*r - 6*p1*r) + log(8 - 4*r + p2*(-3 + 6*r))) + (n02 + n11 + n13 + n22)*log(-6*p2*(1 - 2*r)^2 + 3*p1*(-2 + 3*p2)*(1 - 2*r)^2 + 8*(5 - 2*r + 2*r^2)) + n12*log(-3*p1*(-2 + 3*p2)*(1 - 2*r)^2 + 2*(3*p2*(1 - 2*r)^2 + 8*(2 + r - r^2)))
return(L)}
interlogL_rc3 <- function(n01,n02,n03,n11,n12,n13,n21,n22,n23,p1,p2) {
optimize(logL_rc3,c(0,0.5), n01,n02,n03,n11,n12,n13,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc3 <- parallel::mcmapply(interlogL_rc3,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


# LOD_rc3 <- -2*(x[,"n_03"] + x[,"n_21"])*(log10(2) + log10(3)) - 2*(x[,"n_01"] + x[,"n_23"])*(log10(2) + log10(3)) - x[,"n_12"]*(2*log10(2) + 2*log10(3)) - (x[,"n_02"] + x[,"n_11"] + x[,"n_13"] + x[,"n_22"])*(2*log10(2) + 2*log10(3)) + (x[,"n_03"] + x[,"n_21"])*(log10(4 + 3*p2 + 4*r_rc3 - 6*p2*r_rc3) + log10(8 - 4*r_rc3 + p1*(-3 + 6*r_rc3))) + (x[,"n_01"] + x[,"n_23"])*(log10(4 + 3*p1 + 4*r_rc3 - 6*p1*r_rc3) + log10(8 - 4*r_rc3 + p2*(-3 + 6*r_rc3))) + (x[,"n_02"] + x[,"n_11"] + x[,"n_13"] + x[,"n_22"])*log10(-6*p2*(1 - 2*r_rc3)^2 + 3*p1*(-2 + 3*p2)*(1 - 2*r_rc3)^2 + 8*(5 - 2*r_rc3 + 2*r_rc3^2)) + x[,"n_12"]*log10(-3*p1*(-2 + 3*p2)*(1 - 2*r_rc3)^2 + 2*(3*p2*(1 - 2*r_rc3)^2 + 8*(2 + r_rc3 - r_rc3^2)))


logL_rc3 <- (-6*x[,"n_01"] - 5*x[,"n_02"] - 6*x[,"n_03"] - 5*x[,"n_11"] - 4*x[,"n_12"] - 5*x[,"n_13"] - 6*x[,"n_21"] - 5*x[,"n_22"] - 6*x[,"n_23"])*log(2) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*log(3) + (x[,"n_03"] + x[,"n_21"])*(log(4 + 3*p2 + 4*r_rc3 - 6*p2*r_rc3) + log(8 - 4*r_rc3 + p1*(-3 + 6*r_rc3))) + (x[,"n_01"] + x[,"n_23"])*(log(4 + 3*p1 + 4*r_rc3 - 6*p1*r_rc3) + log(8 - 4*r_rc3 + p2*(-3 + 6*r_rc3))) + (x[,"n_02"] + x[,"n_11"] + x[,"n_13"] + x[,"n_22"])*log(-6*p2*(1 - 2*r_rc3)^2 + 3*p1*(-2 + 3*p2)*(1 - 2*r_rc3)^2 + 8*(5 - 2*r_rc3 + 2*r_rc3^2)) + x[,"n_12"]*log(-3*p1*(-2 + 3*p2)*(1 - 2*r_rc3)^2 + 2*(3*p2*(1 - 2*r_rc3)^2 + 8*(2 + r_rc3 - r_rc3^2)))


logL_rc4 <- function(r,n01,n02,n03,n11,n12,n13,n21,n22,n23,p1,p2) {
L <- (-5*n01 - 4*n02 - 5*n03 - 4*n11 - 3*n12 - 4*n13 - 5*n21 - 4*n22 - 5*n23)*log(2) + 2*(-n01 - n02 - n03 - n11 - n12 - n13 - n21 - n22 - n23)*log(3) + (n01 + n23)*(log(4 + p2*(3 - 6*r) - 2*r) + log(4 + 3*p1 + 4*r - 6*p1*r)) + (n02 + n11 + n13 + n22)*log(20 + 6*p2*(1 - 2*r)^2 - 3*p1*(1 + 3*p2)*(1 - 2*r)^2 - 8*r + 8*r^2) + (n03 + n21)*(log(2 - 3*p2 + 2*r + 6*p2*r) + log(8 - 4*r + p1*(-3 + 6*r))) + n12*log(-6*p2*(1 - 2*r)^2 + 3*p1*(1 + 3*p2)*(1 - 2*r)^2 + 8*(2 + r - r^2))
return(L)}
interlogL_rc4 <- function(n01,n02,n03,n11,n12,n13,n21,n22,n23,p1,p2) {
optimize(logL_rc4,c(0,0.5), n01,n02,n03,n11,n12,n13,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc4 <- parallel::mcmapply(interlogL_rc4,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


# LOD_rc4 <- -(x[,"n_12"]*(log10(2) + 2*log10(3))) - (x[,"n_03"] + x[,"n_21"])*(log10(2) + 2*log10(3)) - (x[,"n_02"] + x[,"n_11"] + x[,"n_13"] + x[,"n_22"])*(log10(2) + 2*log10(3)) - (x[,"n_01"] + x[,"n_23"])*(log10(2) + 2*log10(3)) + (x[,"n_01"] + x[,"n_23"])*(log10(4 + p2*(3 - 6*r_rc4) - 2*r_rc4) + log10(4 + 3*p1 + 4*r_rc4 - 6*p1*r_rc4)) + (x[,"n_02"] + x[,"n_11"] + x[,"n_13"] + x[,"n_22"])*log10(20 + 6*p2*(1 - 2*r_rc4)^2 - 3*p1*(1 + 3*p2)*(1 - 2*r_rc4)^2 - 8*r_rc4 + 8*r_rc4^2) + (x[,"n_03"] + x[,"n_21"])*(log10(2 - 3*p2 + 2*r_rc4 + 6*p2*r_rc4) + log10(8 - 4*r_rc4 + p1*(-3 + 6*r_rc4))) + x[,"n_12"]*log10(-6*p2*(1 - 2*r_rc4)^2 + 3*p1*(1 + 3*p2)*(1 - 2*r_rc4)^2 + 8*(2 + r_rc4 - r_rc4^2))


logL_rc4 <- (-5*x[,"n_01"] - 4*x[,"n_02"] - 5*x[,"n_03"] - 4*x[,"n_11"] - 3*x[,"n_12"] - 4*x[,"n_13"] - 5*x[,"n_21"] - 4*x[,"n_22"] - 5*x[,"n_23"])*log(2) + 2*(-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*log(3) + (x[,"n_01"] + x[,"n_23"])*(log(4 + p2*(3 - 6*r_rc4) - 2*r_rc4) + log(4 + 3*p1 + 4*r_rc4 - 6*p1*r_rc4)) + (x[,"n_02"] + x[,"n_11"] + x[,"n_13"] + x[,"n_22"])*log(20 + 6*p2*(1 - 2*r_rc4)^2 - 3*p1*(1 + 3*p2)*(1 - 2*r_rc4)^2 - 8*r_rc4 + 8*r_rc4^2) + (x[,"n_03"] + x[,"n_21"])*(log(2 - 3*p2 + 2*r_rc4 + 6*p2*r_rc4) + log(8 - 4*r_rc4 + p1*(-3 + 6*r_rc4))) + x[,"n_12"]*log(-6*p2*(1 - 2*r_rc4)^2 + 3*p1*(1 + 3*p2)*(1 - 2*r_rc4)^2 + 8*(2 + r_rc4 - r_rc4^2))


logL_rr1 <- function(r,n01,n02,n03,n11,n12,n13,n21,n22,n23,p1,p2) {
L <- (-3*n01 - 3*n02 - 3*n03 - 3*n11 - 2*n12 - 3*n13 - 3*n21 - 3*n22 - 3*n23)*log(2) + (-n01 - n02 - n03 - n11 - n12 - n13 - n21 - n22 - n23)*log(3) + (n03 + n21)*(log(pmax(1e-6,1 - r)) + log(4 + 3*p1 - 2*r - 6*p1*r)) + (n01 + n23)*(log(pmax(1e-6,r)) + log(2 - 3*p1 + 2*r + 6*p1*r)) + (n02 + n11 + n13 + n22)*log(2 - 3*p1*(1 - 2*r)^2 + 4*r - 4*r^2) + n12*log(3*p1*(1 - 2*r)^2 + 4*(1 - r + r^2))
return(L)}
interlogL_rr1 <- function(n01,n02,n03,n11,n12,n13,n21,n22,n23,p1,p2) {
optimize(logL_rr1,c(0,0.5), n01,n02,n03,n11,n12,n13,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr1 <- parallel::mcmapply(interlogL_rr1,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_rr1 <- -(x[,"n_12"]*log10(3)) - (x[,"n_02"] + x[,"n_11"] + x[,"n_13"] + x[,"n_22"])*log10(3) - (x[,"n_03"] + x[,"n_21"])*(-log10(2) + log10(3)) - (x[,"n_01"] + x[,"n_23"])*(-log10(2) + log10(3)) + (x[,"n_03"] + x[,"n_21"])*(log10(pmax(1e-6,1 - r_rr1)) + log10(4 + 3*p1 - 2*r_rr1 - 6*p1*r_rr1)) + (x[,"n_01"] + x[,"n_23"])*(log10(pmax(1e-6,r_rr1)) + log10(2 - 3*p1 + 2*r_rr1 + 6*p1*r_rr1)) + (x[,"n_02"] + x[,"n_11"] + x[,"n_13"] + x[,"n_22"])*log10(2 - 3*p1*(1 - 2*r_rr1)^2 + 4*r_rr1 - 4*r_rr1^2) + x[,"n_12"]*log10(3*p1*(1 - 2*r_rr1)^2 + 4*(1 - r_rr1 + r_rr1^2))


logL_rr1 <- (-3*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 3*x[,"n_11"] - 2*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"])*log(2) + (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*log(3) + (x[,"n_03"] + x[,"n_21"])*(log(pmax(1e-6,1 - r_rr1)) + log(4 + 3*p1 - 2*r_rr1 - 6*p1*r_rr1)) + (x[,"n_01"] + x[,"n_23"])*(log(pmax(1e-6,r_rr1)) + log(2 - 3*p1 + 2*r_rr1 + 6*p1*r_rr1)) + (x[,"n_02"] + x[,"n_11"] + x[,"n_13"] + x[,"n_22"])*log(2 - 3*p1*(1 - 2*r_rr1)^2 + 4*r_rr1 - 4*r_rr1^2) + x[,"n_12"]*log(3*p1*(1 - 2*r_rr1)^2 + 4*(1 - r_rr1 + r_rr1^2))


logL_rr2 <- function(r,n01,n02,n03,n11,n12,n13,n21,n22,n23,p1,p2) {
L <- (-4*n01 - 4*n02 - 4*n03 - 4*n11 - 3*n12 - 4*n13 - 4*n21 - 4*n22 - 4*n23)*log(2) + (-n01 - n02 - n03 - n11 - n12 - n13 - n21 - n22 - n23)*log(3) + (n01 + n23)*(log(pmax(1e-6,r)) + log(4 + 3*p1 + 4*r - 6*p1*r)) + (n02 + n11 + n13 + n22)*log(4 + 3*p1*(1 - 2*r)^2 + 8*r - 8*r^2) + (n03 + n21)*(log(pmax(1e-6,1 - r)) + log(8 - 4*r + p1*(-3 + 6*r))) + n12*log(-3*p1*(1 - 2*r)^2 + 8*(1 - r + r^2))
return(L)}
interlogL_rr2 <- function(n01,n02,n03,n11,n12,n13,n21,n22,n23,p1,p2) {
optimize(logL_rr2,c(0,0.5), n01,n02,n03,n11,n12,n13,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr2 <- parallel::mcmapply(interlogL_rr2,x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_rr2 <- -((x[,"n_03"] + x[,"n_21"])*log10(3)) - (x[,"n_01"] + x[,"n_23"])*log10(3) - x[,"n_12"]*(log10(2) + log10(3)) - (x[,"n_02"] + x[,"n_11"] + x[,"n_13"] + x[,"n_22"])*(log10(2) + log10(3)) + (x[,"n_01"] + x[,"n_23"])*(log10(pmax(1e-6,r_rr2)) + log10(4 + 3*p1 + 4*r_rr2 - 6*p1*r_rr2)) + (x[,"n_02"] + x[,"n_11"] + x[,"n_13"] + x[,"n_22"])*log10(4 + 3*p1*(1 - 2*r_rr2)^2 + 8*r_rr2 - 8*r_rr2^2) + (x[,"n_03"] + x[,"n_21"])*(log10(pmax(1e-6,1 - r_rr2)) + log10(8 - 4*r_rr2 + p1*(-3 + 6*r_rr2))) + x[,"n_12"]*log10(-3*p1*(1 - 2*r_rr2)^2 + 8*(1 - r_rr2 + r_rr2^2))


logL_rr2 <- (-4*x[,"n_01"] - 4*x[,"n_02"] - 4*x[,"n_03"] - 4*x[,"n_11"] - 3*x[,"n_12"] - 4*x[,"n_13"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 4*x[,"n_23"])*log(2) + (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*log(3) + (x[,"n_01"] + x[,"n_23"])*(log(pmax(1e-6,r_rr2)) + log(4 + 3*p1 + 4*r_rr2 - 6*p1*r_rr2)) + (x[,"n_02"] + x[,"n_11"] + x[,"n_13"] + x[,"n_22"])*log(4 + 3*p1*(1 - 2*r_rr2)^2 + 8*r_rr2 - 8*r_rr2^2) + (x[,"n_03"] + x[,"n_21"])*(log(pmax(1e-6,1 - r_rr2)) + log(8 - 4*r_rr2 + p1*(-3 + 6*r_rr2))) + x[,"n_12"]*log(-3*p1*(1 - 2*r_rr2)^2 + 8*(1 - r_rr2 + r_rr2^2))

r_rc1<-r_rc2<-r_rc3<-r_rc4<-0.499 #This is because the r estimate in this phase is very suspect
LOD_rc1<-LOD_rc2<-LOD_rc3<-LOD_rc4<-0

return(list(
r_mat = cbind(r_cc1,r_cc2,r_cr,r_rc1,r_rc2,r_rc3,r_rc4,r_rr1,r_rr2,0.499),
LOD_mat = cbind(LOD_cc1,LOD_cc2,LOD_cr,LOD_rc1,LOD_rc2,LOD_rc3,LOD_rc4,LOD_rr1,LOD_rr2,0),
logL_mat = cbind(logL_cc1,logL_cc2,logL_cr,logL_rc1,logL_rc2,logL_rc3,logL_rc4,logL_rr1,logL_rr2,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling coupling","coupling repulsion","repulsion coupling","repulsion coupling","repulsion coupling","repulsion coupling","repulsion repulsion","repulsion repulsion","unknown")
)
)
}

#' @rdname p4_functions
p4_1.1_2.2 <- function(x,p1,p2,ncores=1){


logL_cc1 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-4*n00 - 3*n01 - 4*n02 - 3*n03 - 4*n04 - 3*n10 - 3*n11 - 2*n12 - 3*n13 - 3*n14 - 4*n20 - 3*n21 - 4*n22 - 3*n23 - 4*n24)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(3) + (n11 + n13)*log(8 - 3*p2 - 3*p1*(1 + 3*p2)) + (n01 + n23)*(log(8 - 3*p2 - p1*(3 + 9*p2)) + log(pmax(1e-6,1 - r))) + (n03 + n21)*(log(8 - 3*p2 - p1*(3 + 9*p2)) + log(pmax(1e-6,r))) + (n10 + n14)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n04 + n20)*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r))) + (n00 + n24)*(log(2 - 3*p1) + log(2 - 3*p2) + log(1 - 2*r + r^2)) + n12*log(10 - 4*r + 4*r^2 + p2*(3 + 6*r - 6*r^2) + p1*(3 + 6*r - 6*r^2 + 9*p2*(1 - r + r^2))) + (n02 + n22)*log(4*(4 + 2*r - 2*r^2 + 3*p2*(1 - r + r^2)) - 3*p1*(-4*(1 - r + r^2) + p2*(-3 - 6*r + 6*r^2)))
return(L)}
interlogL_cc1 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_cc1,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc1 <- parallel::mcmapply(interlogL_cc1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_cc1 <- -((x[,"n_10"] + x[,"n_14"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2))) - (x[,"n_04"] + x[,"n_20"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_24"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_22"])*log10(-3*p1*(-3 - (9*p2)/2) + 4*(9/2 + (9*p2)/4)) - x[,"n_12"]*log10(9 + (9*p2)/2 + p1*(9/2 + (27*p2)/4)) - (x[,"n_03"] + x[,"n_21"])*(-log10(2) + log10(8 - 3*p2 - p1*(3 + 9*p2))) - (x[,"n_01"] + x[,"n_23"])*(-log10(2) + log10(8 - 3*p2 - p1*(3 + 9*p2))) + (x[,"n_01"] + x[,"n_23"])*(log10(8 - 3*p2 - p1*(3 + 9*p2)) + log10(pmax(1e-6,1 - r_cc1))) + (x[,"n_03"] + x[,"n_21"])*(log10(8 - 3*p2 - p1*(3 + 9*p2)) + log10(pmax(1e-6,r_cc1))) + (x[,"n_10"] + x[,"n_14"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cc1)) + log10(pmax(1e-6,r_cc1))) + (x[,"n_04"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_cc1))) + (x[,"n_00"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(1 - 2*r_cc1 + r_cc1^2)) + x[,"n_12"]*log10(10 - 4*r_cc1 + 4*r_cc1^2 + p2*(3 + 6*r_cc1 - 6*r_cc1^2) + p1*(3 + 6*r_cc1 - 6*r_cc1^2 + 9*p2*(1 - r_cc1 + r_cc1^2))) + (x[,"n_02"] + x[,"n_22"])*log10(4*(4 + 2*r_cc1 - 2*r_cc1^2 + 3*p2*(1 - r_cc1 + r_cc1^2)) - 3*p1*(-4*(1 - r_cc1 + r_cc1^2) + p2*(-3 - 6*r_cc1 + 6*r_cc1^2)))


logL_cc1 <- (-4*x[,"n_00"] - 3*x[,"n_01"] - 4*x[,"n_02"] - 3*x[,"n_03"] - 4*x[,"n_04"] - 3*x[,"n_10"] - 3*x[,"n_11"] - 2*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 4*x[,"n_20"] - 3*x[,"n_21"] - 4*x[,"n_22"] - 3*x[,"n_23"] - 4*x[,"n_24"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(3) + (x[,"n_11"] + x[,"n_13"])*log(8 - 3*p2 - 3*p1*(1 + 3*p2)) + (x[,"n_01"] + x[,"n_23"])*(log(8 - 3*p2 - p1*(3 + 9*p2)) + log(pmax(1e-6,1 - r_cc1))) + (x[,"n_03"] + x[,"n_21"])*(log(8 - 3*p2 - p1*(3 + 9*p2)) + log(pmax(1e-6,r_cc1))) + (x[,"n_10"] + x[,"n_14"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cc1)) + log(pmax(1e-6,r_cc1))) + (x[,"n_04"] + x[,"n_20"])*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r_cc1))) + (x[,"n_00"] + x[,"n_24"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(1 - 2*r_cc1 + r_cc1^2)) + x[,"n_12"]*log(10 - 4*r_cc1 + 4*r_cc1^2 + p2*(3 + 6*r_cc1 - 6*r_cc1^2) + p1*(3 + 6*r_cc1 - 6*r_cc1^2 + 9*p2*(1 - r_cc1 + r_cc1^2))) + (x[,"n_02"] + x[,"n_22"])*log(4*(4 + 2*r_cc1 - 2*r_cc1^2 + 3*p2*(1 - r_cc1 + r_cc1^2)) - 3*p1*(-4*(1 - r_cc1 + r_cc1^2) + p2*(-3 - 6*r_cc1 + 6*r_cc1^2)))


logL_cc2 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-5*n00 - 4*n01 - 5*n02 - 4*n03 - 5*n04 - 4*n10 - 4*n11 - 3*n12 - 4*n13 - 4*n14 - 5*n20 - 4*n21 - 5*n22 - 4*n23 - 5*n24)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(3) + (n11 + n13)*log(16 + 3*p2 + p1*(-6 + 9*p2)) + (n01 + n23)*(log(16 + 3*p2 + p1*(-6 + 9*p2)) + log(pmax(1e-6,1 - r))) + (n03 + n21)*(log(16 + 3*p2 + p1*(-6 + 9*p2)) + log(pmax(1e-6,r))) + (n10 + n14)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n04 + n20)*(log(2 - 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r))) + (n02 + n22)*log((-4 - 3*p1)*(-8 + 3*p2) + 2*(-2 + 3*p1)*(4 + 3*p2)*(-1 + r)*r) + (n00 + n24)*(log(2 - 3*p1) + log(4 + 3*p2) + log(1 - 2*r + r^2)) + n12*log(20 - 8*r + 8*r^2 + p2*(-3 - 6*r + 6*r^2) - 3*p1*(-2 - 4*r + 4*r^2 + 3*p2*(1 - r + r^2)))
return(L)}
interlogL_cc2 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_cc2,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc2 <- parallel::mcmapply(interlogL_cc2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_cc2 <- -((x[,"n_10"] + x[,"n_14"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2))) - (x[,"n_04"] + x[,"n_20"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_24"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - x[,"n_12"]*log10(18 - (9*p2)/2 - 3*p1*(-3 + (9*p2)/4)) - (x[,"n_02"] + x[,"n_22"])*log10((-4 - 3*p1)*(-8 + 3*p2) - ((-2 + 3*p1)*(4 + 3*p2))/2) - (x[,"n_03"] + x[,"n_21"])*(-log10(2) + log10(16 + 3*p2 + p1*(-6 + 9*p2))) - (x[,"n_01"] + x[,"n_23"])*(-log10(2) + log10(16 + 3*p2 + p1*(-6 + 9*p2))) + (x[,"n_01"] + x[,"n_23"])*(log10(16 + 3*p2 + p1*(-6 + 9*p2)) + log10(pmax(1e-6,1 - r_cc2))) + (x[,"n_03"] + x[,"n_21"])*(log10(16 + 3*p2 + p1*(-6 + 9*p2)) + log10(pmax(1e-6,r_cc2))) + (x[,"n_10"] + x[,"n_14"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cc2)) + log10(pmax(1e-6,r_cc2))) + (x[,"n_04"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_cc2))) + (x[,"n_02"] + x[,"n_22"])*log10((-4 - 3*p1)*(-8 + 3*p2) + 2*(-2 + 3*p1)*(4 + 3*p2)*(-1 + r_cc2)*r_cc2) + (x[,"n_00"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(1 - 2*r_cc2 + r_cc2^2)) + x[,"n_12"]*log10(20 - 8*r_cc2 + 8*r_cc2^2 + p2*(-3 - 6*r_cc2 + 6*r_cc2^2) - 3*p1*(-2 - 4*r_cc2 + 4*r_cc2^2 + 3*p2*(1 - r_cc2 + r_cc2^2)))


logL_cc2 <- (-5*x[,"n_00"] - 4*x[,"n_01"] - 5*x[,"n_02"] - 4*x[,"n_03"] - 5*x[,"n_04"] - 4*x[,"n_10"] - 4*x[,"n_11"] - 3*x[,"n_12"] - 4*x[,"n_13"] - 4*x[,"n_14"] - 5*x[,"n_20"] - 4*x[,"n_21"] - 5*x[,"n_22"] - 4*x[,"n_23"] - 5*x[,"n_24"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(3) + (x[,"n_11"] + x[,"n_13"])*log(16 + 3*p2 + p1*(-6 + 9*p2)) + (x[,"n_01"] + x[,"n_23"])*(log(16 + 3*p2 + p1*(-6 + 9*p2)) + log(pmax(1e-6,1 - r_cc2))) + (x[,"n_03"] + x[,"n_21"])*(log(16 + 3*p2 + p1*(-6 + 9*p2)) + log(pmax(1e-6,r_cc2))) + (x[,"n_10"] + x[,"n_14"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cc2)) + log(pmax(1e-6,r_cc2))) + (x[,"n_04"] + x[,"n_20"])*(log(2 - 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r_cc2))) + (x[,"n_02"] + x[,"n_22"])*log((-4 - 3*p1)*(-8 + 3*p2) + 2*(-2 + 3*p1)*(4 + 3*p2)*(-1 + r_cc2)*r_cc2) + (x[,"n_00"] + x[,"n_24"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(1 - 2*r_cc2 + r_cc2^2)) + x[,"n_12"]*log(20 - 8*r_cc2 + 8*r_cc2^2 + p2*(-3 - 6*r_cc2 + 6*r_cc2^2) - 3*p1*(-2 - 4*r_cc2 + 4*r_cc2^2 + 3*p2*(1 - r_cc2 + r_cc2^2)))


logL_cc3 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-5*n00 - 4*n01 - 5*n02 - 4*n03 - 5*n04 - 4*n10 - 4*n11 - 3*n12 - 4*n13 - 4*n14 - 5*n20 - 4*n21 - 5*n22 - 4*n23 - 5*n24)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(3) + (n11 + n13)*log(16 - 6*p2 + p1*(3 + 9*p2)) + (n01 + n23)*(log(16 - 6*p2 + p1*(3 + 9*p2)) + log(pmax(1e-6,1 - r))) + (n03 + n21)*(log(16 - 6*p2 + p1*(3 + 9*p2)) + log(pmax(1e-6,r))) + (n10 + n14)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n04 + n20)*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r))) + (n02 + n22)*log((8 - 3*p1)*(4 + 3*p2) + 2*(4 + 3*p1)*(-2 + 3*p2)*(-1 + r)*r) + (n00 + n24)*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r + r^2)) + n12*log(20 - 8*r + 8*r^2 + p2*(6 + 12*r - 12*r^2) - 3*p1*(1 + 2*r - 2*r^2 + 3*p2*(1 - r + r^2)))
return(L)}
interlogL_cc3 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_cc3,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc3 <- parallel::mcmapply(interlogL_cc3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_cc3 <- -((x[,"n_10"] + x[,"n_14"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2))) - (x[,"n_04"] + x[,"n_20"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_24"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - x[,"n_12"]*log10(18 + 9*p2 - 3*p1*(3/2 + (9*p2)/4)) - (x[,"n_02"] + x[,"n_22"])*log10(-((4 + 3*p1)*(-2 + 3*p2))/2 + (8 - 3*p1)*(4 + 3*p2)) - (x[,"n_03"] + x[,"n_21"])*(-log10(2) + log10(16 - 6*p2 + p1*(3 + 9*p2))) - (x[,"n_01"] + x[,"n_23"])*(-log10(2) + log10(16 - 6*p2 + p1*(3 + 9*p2))) + (x[,"n_01"] + x[,"n_23"])*(log10(16 - 6*p2 + p1*(3 + 9*p2)) + log10(pmax(1e-6,1 - r_cc3))) + (x[,"n_03"] + x[,"n_21"])*(log10(16 - 6*p2 + p1*(3 + 9*p2)) + log10(pmax(1e-6,r_cc3))) + (x[,"n_10"] + x[,"n_14"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cc3)) + log10(pmax(1e-6,r_cc3))) + (x[,"n_04"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_cc3))) + (x[,"n_02"] + x[,"n_22"])*log10((8 - 3*p1)*(4 + 3*p2) + 2*(4 + 3*p1)*(-2 + 3*p2)*(-1 + r_cc3)*r_cc3) + (x[,"n_00"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(1 - 2*r_cc3 + r_cc3^2)) + x[,"n_12"]*log10(20 - 8*r_cc3 + 8*r_cc3^2 + p2*(6 + 12*r_cc3 - 12*r_cc3^2) - 3*p1*(1 + 2*r_cc3 - 2*r_cc3^2 + 3*p2*(1 - r_cc3 + r_cc3^2)))


logL_cc3 <- (-5*x[,"n_00"] - 4*x[,"n_01"] - 5*x[,"n_02"] - 4*x[,"n_03"] - 5*x[,"n_04"] - 4*x[,"n_10"] - 4*x[,"n_11"] - 3*x[,"n_12"] - 4*x[,"n_13"] - 4*x[,"n_14"] - 5*x[,"n_20"] - 4*x[,"n_21"] - 5*x[,"n_22"] - 4*x[,"n_23"] - 5*x[,"n_24"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(3) + (x[,"n_11"] + x[,"n_13"])*log(16 - 6*p2 + p1*(3 + 9*p2)) + (x[,"n_01"] + x[,"n_23"])*(log(16 - 6*p2 + p1*(3 + 9*p2)) + log(pmax(1e-6,1 - r_cc3))) + (x[,"n_03"] + x[,"n_21"])*(log(16 - 6*p2 + p1*(3 + 9*p2)) + log(pmax(1e-6,r_cc3))) + (x[,"n_10"] + x[,"n_14"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cc3)) + log(pmax(1e-6,r_cc3))) + (x[,"n_04"] + x[,"n_20"])*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r_cc3))) + (x[,"n_02"] + x[,"n_22"])*log((8 - 3*p1)*(4 + 3*p2) + 2*(4 + 3*p1)*(-2 + 3*p2)*(-1 + r_cc3)*r_cc3) + (x[,"n_00"] + x[,"n_24"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r_cc3 + r_cc3^2)) + x[,"n_12"]*log(20 - 8*r_cc3 + 8*r_cc3^2 + p2*(6 + 12*r_cc3 - 12*r_cc3^2) - 3*p1*(1 + 2*r_cc3 - 2*r_cc3^2 + 3*p2*(1 - r_cc3 + r_cc3^2)))


logL_cc4 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-6*n00 - 5*n01 - 6*n02 - 5*n03 - 6*n04 - 5*n10 - 5*n11 - 5*n12 - 5*n13 - 5*n14 - 6*n20 - 5*n21 - 6*n22 - 5*n23 - 6*n24)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(3) + (n11 + n13)*log(32 + p1*(6 - 9*p2) + 6*p2) + (n01 + n23)*(log(32 + 6*p2 - p1*(-6 + 9*p2)) + log(pmax(1e-6,1 - r))) + (n03 + n21)*(log(32 + p1*(6 - 9*p2) + 6*p2) + log(pmax(1e-6,r))) + (n10 + n14)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n04 + n20)*(log(4 + 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r))) + (n02 + n22)*log((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r)*r) + (n00 + n24)*(log(4 + 3*p1) + log(4 + 3*p2) + log(1 - 2*r + r^2)) + n12*log((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r)^2 + (4 + 3*p1)*(4 + 3*p2)*r^2)
return(L)}
interlogL_cc4 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_cc4,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc4 <- parallel::mcmapply(interlogL_cc4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_cc4 <- -((x[,"n_10"] + x[,"n_14"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2))) - (x[,"n_04"] + x[,"n_20"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_24"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_03"] + x[,"n_21"])*(-log10(2) + log10(32 + p1*(6 - 9*p2) + 6*p2)) - x[,"n_12"]*log10((-8 + 3*p1)*(-8 + 3*p2) + ((4 + 3*p1)*(4 + 3*p2))/2) - (x[,"n_02"] + x[,"n_22"])*log10((-8 + 3*p1)*(-8 + 3*p2) + ((4 + 3*p1)*(4 + 3*p2))/2) - (x[,"n_01"] + x[,"n_23"])*(-log10(2) + log10(32 + 6*p2 - p1*(-6 + 9*p2))) + (x[,"n_01"] + x[,"n_23"])*(log10(32 + 6*p2 - p1*(-6 + 9*p2)) + log10(pmax(1e-6,1 - r_cc4))) + (x[,"n_03"] + x[,"n_21"])*(log10(32 + p1*(6 - 9*p2) + 6*p2) + log10(pmax(1e-6,r_cc4))) + (x[,"n_10"] + x[,"n_14"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cc4)) + log10(pmax(1e-6,r_cc4))) + (x[,"n_04"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_cc4))) + (x[,"n_02"] + x[,"n_22"])*log10((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_cc4)*r_cc4) + (x[,"n_00"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(1 - 2*r_cc4 + r_cc4^2)) + x[,"n_12"]*log10((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r_cc4)^2 + (4 + 3*p1)*(4 + 3*p2)*r_cc4^2)


logL_cc4 <- (-6*x[,"n_00"] - 5*x[,"n_01"] - 6*x[,"n_02"] - 5*x[,"n_03"] - 6*x[,"n_04"] - 5*x[,"n_10"] - 5*x[,"n_11"] - 5*x[,"n_12"] - 5*x[,"n_13"] - 5*x[,"n_14"] - 6*x[,"n_20"] - 5*x[,"n_21"] - 6*x[,"n_22"] - 5*x[,"n_23"] - 6*x[,"n_24"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(3) + (x[,"n_11"] + x[,"n_13"])*log(32 + p1*(6 - 9*p2) + 6*p2) + (x[,"n_01"] + x[,"n_23"])*(log(32 + 6*p2 - p1*(-6 + 9*p2)) + log(pmax(1e-6,1 - r_cc4))) + (x[,"n_03"] + x[,"n_21"])*(log(32 + p1*(6 - 9*p2) + 6*p2) + log(pmax(1e-6,r_cc4))) + (x[,"n_10"] + x[,"n_14"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cc4)) + log(pmax(1e-6,r_cc4))) + (x[,"n_04"] + x[,"n_20"])*(log(4 + 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r_cc4))) + (x[,"n_02"] + x[,"n_22"])*log((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_cc4)*r_cc4) + (x[,"n_00"] + x[,"n_24"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(1 - 2*r_cc4 + r_cc4^2)) + x[,"n_12"]*log((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r_cc4)^2 + (4 + 3*p1)*(4 + 3*p2)*r_cc4^2)


logL_cr1 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-5*n00 - 5*n01 - 5*n02 - 5*n03 - 5*n04 - 5*n10 - 4*n11 - 4*n12 - 4*n13 - 5*n14 - 5*n20 - 5*n21 - 5*n22 - 5*n23 - 5*n24)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(3) + (n11 + n13)*log(16 + 3*p2 + p1*(-6 + 9*p2)) + (n03 + n21)*log(2*(8 + p2*(6 - 9*r)) + 3*p1*(4 + 3*p2 - 12*r)) + (n00 + n04 + n20 + n24)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n14)*(log(2 - 3*p1) + log(4 + 3*p2) + log(1 - 2*r + 2*r^2)) + (n02 + n22)*log((-4 - 3*p1)*(-8 + 3*p2) - (-2 + 3*p1)*(4 + 3*p2)*(-1 + r)^2 + (2 - 3*p1)*(4 + 3*p2)*r^2) + (n01 + n23)*log(3*p1*(-8 + 3*p2 + 12*r) + 2*(8 + p2*(-3 + 9*r))) + n12*log(-4*(4*(-2 - r + r^2) + 3*p2*(1 - r + r^2)) + 3*p1*(8*(1 - r + r^2) + p2*(-3 - 6*r + 6*r^2)))
return(L)}
interlogL_cr1 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_cr1,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr1 <- parallel::mcmapply(interlogL_cr1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_cr1 <- -((x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2))) - (x[,"n_10"] + x[,"n_14"])*(-log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - x[,"n_12"]*log10(3*p1*(6 - (9*p2)/2) - 4*(-9 + (9*p2)/4)) - (x[,"n_03"] + x[,"n_21"])*log10(2*(8 + (3*p2)/2) + 3*p1*(-2 + 3*p2)) - (x[,"n_01"] + x[,"n_23"])*log10(2*(8 + (3*p2)/2) + 3*p1*(-2 + 3*p2)) - (x[,"n_02"] + x[,"n_22"])*log10((-4 - 3*p1)*(-8 + 3*p2) + ((2 - 3*p1)*(4 + 3*p2))/4 - ((-2 + 3*p1)*(4 + 3*p2))/4) + (x[,"n_03"] + x[,"n_21"])*log10(2*(8 + p2*(6 - 9*r_cr1)) + 3*p1*(4 + 3*p2 - 12*r_cr1)) + (x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cr1)) + log10(pmax(1e-6,r_cr1))) + (x[,"n_10"] + x[,"n_14"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(1 - 2*r_cr1 + 2*r_cr1^2)) + (x[,"n_02"] + x[,"n_22"])*log10((-4 - 3*p1)*(-8 + 3*p2) - (-2 + 3*p1)*(4 + 3*p2)*(-1 + r_cr1)^2 + (2 - 3*p1)*(4 + 3*p2)*r_cr1^2) + (x[,"n_01"] + x[,"n_23"])*log10(3*p1*(-8 + 3*p2 + 12*r_cr1) + 2*(8 + p2*(-3 + 9*r_cr1))) + x[,"n_12"]*log10(-4*(4*(-2 - r_cr1 + r_cr1^2) + 3*p2*(1 - r_cr1 + r_cr1^2)) + 3*p1*(8*(1 - r_cr1 + r_cr1^2) + p2*(-3 - 6*r_cr1 + 6*r_cr1^2)))


logL_cr1 <- (-5*x[,"n_00"] - 5*x[,"n_01"] - 5*x[,"n_02"] - 5*x[,"n_03"] - 5*x[,"n_04"] - 5*x[,"n_10"] - 4*x[,"n_11"] - 4*x[,"n_12"] - 4*x[,"n_13"] - 5*x[,"n_14"] - 5*x[,"n_20"] - 5*x[,"n_21"] - 5*x[,"n_22"] - 5*x[,"n_23"] - 5*x[,"n_24"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(3) + (x[,"n_11"] + x[,"n_13"])*log(16 + 3*p2 + p1*(-6 + 9*p2)) + (x[,"n_03"] + x[,"n_21"])*log(2*(8 + p2*(6 - 9*r_cr1)) + 3*p1*(4 + 3*p2 - 12*r_cr1)) + (x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cr1)) + log(pmax(1e-6,r_cr1))) + (x[,"n_10"] + x[,"n_14"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(1 - 2*r_cr1 + 2*r_cr1^2)) + (x[,"n_02"] + x[,"n_22"])*log((-4 - 3*p1)*(-8 + 3*p2) - (-2 + 3*p1)*(4 + 3*p2)*(-1 + r_cr1)^2 + (2 - 3*p1)*(4 + 3*p2)*r_cr1^2) + (x[,"n_01"] + x[,"n_23"])*log(3*p1*(-8 + 3*p2 + 12*r_cr1) + 2*(8 + p2*(-3 + 9*r_cr1))) + x[,"n_12"]*log(-4*(4*(-2 - r_cr1 + r_cr1^2) + 3*p2*(1 - r_cr1 + r_cr1^2)) + 3*p1*(8*(1 - r_cr1 + r_cr1^2) + p2*(-3 - 6*r_cr1 + 6*r_cr1^2)))


logL_cr2 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-4*n00 - 4*n01 - 3*n02 - 4*n03 - 4*n04 - 4*n10 - 3*n11 - 3*n12 - 3*n13 - 4*n14 - 4*n20 - 4*n21 - 3*n22 - 4*n23 - 4*n24)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(3) + (n11 + n13)*log(8 - 3*p2 - 3*p1*(1 + 3*p2)) + (n01 + n23)*log(8 + p2*(6 - 18*r) - 3*p1*(4 + 3*p2 - 6*r)) + (n04 + n20)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n24)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n14)*(log(2 - 3*p1) + log(2 - 3*p2) + log(1 - 2*r + 2*r^2)) + (n03 + n21)*log(-3*p1*(-2 + 3*p2 + 6*r) + 2*(4 + p2*(-6 + 9*r))) + (n02 + n22)*log(10 - 4*r + 4*r^2 + p2*(3 + 6*r - 6*r^2) + p1*(3 + 6*r - 6*r^2 + 9*p2*(1 - r + r^2))) + n12*log(4*(4 + 2*r - 2*r^2 + 3*p2*(1 - r + r^2)) - 3*p1*(-4*(1 - r + r^2) + p2*(-3 - 6*r + 6*r^2)))
return(L)}
interlogL_cr2 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_cr2,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr2 <- parallel::mcmapply(interlogL_cr2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_cr2 <- -((x[,"n_04"] + x[,"n_20"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_24"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_14"])*(-log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - x[,"n_12"]*log10(-3*p1*(-3 - (9*p2)/2) + 4*(9/2 + (9*p2)/4)) - (x[,"n_03"] + x[,"n_21"])*log10(2*(4 - (3*p2)/2) - 3*p1*(1 + 3*p2)) - (x[,"n_01"] + x[,"n_23"])*log10(8 - 3*p2 - 3*p1*(1 + 3*p2)) - (x[,"n_02"] + x[,"n_22"])*log10(9 + (9*p2)/2 + p1*(9/2 + (27*p2)/4)) + (x[,"n_01"] + x[,"n_23"])*log10(8 + p2*(6 - 18*r_cr2) - 3*p1*(4 + 3*p2 - 6*r_cr2)) + (x[,"n_04"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cr2)) + log10(pmax(1e-6,r_cr2))) + (x[,"n_00"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cr2)) + log10(pmax(1e-6,r_cr2))) + (x[,"n_10"] + x[,"n_14"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(1 - 2*r_cr2 + 2*r_cr2^2)) + (x[,"n_03"] + x[,"n_21"])*log10(-3*p1*(-2 + 3*p2 + 6*r_cr2) + 2*(4 + p2*(-6 + 9*r_cr2))) + (x[,"n_02"] + x[,"n_22"])*log10(10 - 4*r_cr2 + 4*r_cr2^2 + p2*(3 + 6*r_cr2 - 6*r_cr2^2) + p1*(3 + 6*r_cr2 - 6*r_cr2^2 + 9*p2*(1 - r_cr2 + r_cr2^2))) + x[,"n_12"]*log10(4*(4 + 2*r_cr2 - 2*r_cr2^2 + 3*p2*(1 - r_cr2 + r_cr2^2)) - 3*p1*(-4*(1 - r_cr2 + r_cr2^2) + p2*(-3 - 6*r_cr2 + 6*r_cr2^2)))


logL_cr2 <- (-4*x[,"n_00"] - 4*x[,"n_01"] - 3*x[,"n_02"] - 4*x[,"n_03"] - 4*x[,"n_04"] - 4*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 4*x[,"n_14"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 3*x[,"n_22"] - 4*x[,"n_23"] - 4*x[,"n_24"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(3) + (x[,"n_11"] + x[,"n_13"])*log(8 - 3*p2 - 3*p1*(1 + 3*p2)) + (x[,"n_01"] + x[,"n_23"])*log(8 + p2*(6 - 18*r_cr2) - 3*p1*(4 + 3*p2 - 6*r_cr2)) + (x[,"n_04"] + x[,"n_20"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cr2)) + log(pmax(1e-6,r_cr2))) + (x[,"n_00"] + x[,"n_24"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cr2)) + log(pmax(1e-6,r_cr2))) + (x[,"n_10"] + x[,"n_14"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(1 - 2*r_cr2 + 2*r_cr2^2)) + (x[,"n_03"] + x[,"n_21"])*log(-3*p1*(-2 + 3*p2 + 6*r_cr2) + 2*(4 + p2*(-6 + 9*r_cr2))) + (x[,"n_02"] + x[,"n_22"])*log(10 - 4*r_cr2 + 4*r_cr2^2 + p2*(3 + 6*r_cr2 - 6*r_cr2^2) + p1*(3 + 6*r_cr2 - 6*r_cr2^2 + 9*p2*(1 - r_cr2 + r_cr2^2))) + x[,"n_12"]*log(4*(4 + 2*r_cr2 - 2*r_cr2^2 + 3*p2*(1 - r_cr2 + r_cr2^2)) - 3*p1*(-4*(1 - r_cr2 + r_cr2^2) + p2*(-3 - 6*r_cr2 + 6*r_cr2^2)))


logL_cr3 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-6*n00 - 6*n01 - 6*n02 - 6*n03 - 6*n04 - 6*n10 - 5*n11 - 5*n12 - 5*n13 - 6*n14 - 6*n20 - 6*n21 - 6*n22 - 6*n23 - 6*n24)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(3) + (n11 + n13)*log(32 + p1*(6 - 9*p2) + 6*p2) + (n03 + n21)*log(4*(8 + p2*(6 - 9*r)) - 3*p1*(4 + 3*p2 - 12*r)) + (n00 + n04 + n20 + n24)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + n12*log((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r)*r) + (n10 + n14)*(log(4 + 3*p1) + log(4 + 3*p2) + log((-1 + r)^2 + r^2)) + (n02 + n22)*log((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r)^2 + (4 + 3*p1)*(4 + 3*p2)*r^2) + (n01 + n23)*log(-3*p1*(-8 + 3*p2 + 12*r) + 4*(8 + p2*(-3 + 9*r)))
return(L)}
interlogL_cr3 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_cr3,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr3 <- parallel::mcmapply(interlogL_cr3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_cr3 <- -((x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2))) - (x[,"n_10"] + x[,"n_14"])*(-log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_03"] + x[,"n_21"])*log10(4*(8 + (3*p2)/2) - 3*p1*(-2 + 3*p2)) - (x[,"n_01"] + x[,"n_23"])*log10(4*(8 + (3*p2)/2) - 3*p1*(-2 + 3*p2)) - x[,"n_12"]*log10((-8 + 3*p1)*(-8 + 3*p2) + ((4 + 3*p1)*(4 + 3*p2))/2) - (x[,"n_02"] + x[,"n_22"])*log10((-8 + 3*p1)*(-8 + 3*p2) + ((4 + 3*p1)*(4 + 3*p2))/2) + (x[,"n_03"] + x[,"n_21"])*log10(4*(8 + p2*(6 - 9*r_cr3)) - 3*p1*(4 + 3*p2 - 12*r_cr3)) + (x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cr3)) + log10(pmax(1e-6,r_cr3))) + x[,"n_12"]*log10((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_cr3)*r_cr3) + (x[,"n_10"] + x[,"n_14"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10((-1 + r_cr3)^2 + r_cr3^2)) + (x[,"n_02"] + x[,"n_22"])*log10((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r_cr3)^2 + (4 + 3*p1)*(4 + 3*p2)*r_cr3^2) + (x[,"n_01"] + x[,"n_23"])*log10(-3*p1*(-8 + 3*p2 + 12*r_cr3) + 4*(8 + p2*(-3 + 9*r_cr3)))


logL_cr3 <- (-6*x[,"n_00"] - 6*x[,"n_01"] - 6*x[,"n_02"] - 6*x[,"n_03"] - 6*x[,"n_04"] - 6*x[,"n_10"] - 5*x[,"n_11"] - 5*x[,"n_12"] - 5*x[,"n_13"] - 6*x[,"n_14"] - 6*x[,"n_20"] - 6*x[,"n_21"] - 6*x[,"n_22"] - 6*x[,"n_23"] - 6*x[,"n_24"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(3) + (x[,"n_11"] + x[,"n_13"])*log(32 + p1*(6 - 9*p2) + 6*p2) + (x[,"n_03"] + x[,"n_21"])*log(4*(8 + p2*(6 - 9*r_cr3)) - 3*p1*(4 + 3*p2 - 12*r_cr3)) + (x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cr3)) + log(pmax(1e-6,r_cr3))) + x[,"n_12"]*log((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_cr3)*r_cr3) + (x[,"n_10"] + x[,"n_14"])*(log(4 + 3*p1) + log(4 + 3*p2) + log((-1 + r_cr3)^2 + r_cr3^2)) + (x[,"n_02"] + x[,"n_22"])*log((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r_cr3)^2 + (4 + 3*p1)*(4 + 3*p2)*r_cr3^2) + (x[,"n_01"] + x[,"n_23"])*log(-3*p1*(-8 + 3*p2 + 12*r_cr3) + 4*(8 + p2*(-3 + 9*r_cr3)))


logL_cr4 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-5*n00 - 5*n01 - 5*n02 - 5*n03 - 5*n04 - 5*n10 - 4*n11 - 4*n12 - 4*n13 - 5*n14 - 5*n20 - 5*n21 - 5*n22 - 5*n23 - 5*n24)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(3) + (n11 + n13)*log(16 - 6*p2 + p1*(3 + 9*p2)) + (n01 + n23)*log(4*(4 + p2*(3 - 9*r)) + 3*p1*(4 + 3*p2 - 6*r)) + (n00 + n04 + n20 + n24)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n14)*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r + 2*r^2)) + (n02 + n22)*log((8 - 3*p1)*(4 + 3*p2) - (4 + 3*p1)*(-2 + 3*p2)*(-1 + r)^2 + (4 + 3*p1)*(2 - 3*p2)*r^2) + (n03 + n21)*log(3*p1*(-2 + 3*p2 + 6*r) + 4*(4 + p2*(-6 + 9*r))) + n12*log(8*(4 + 2*r - 2*r^2 + 3*p2*(1 - r + r^2)) + 3*p1*(-4*(1 - r + r^2) + p2*(-3 - 6*r + 6*r^2)))
return(L)}
interlogL_cr4 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_cr4,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr4 <- parallel::mcmapply(interlogL_cr4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_cr4 <- -((x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2))) - (x[,"n_10"] + x[,"n_14"])*(-log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - x[,"n_12"]*log10(3*p1*(-3 - (9*p2)/2) + 8*(9/2 + (9*p2)/4)) - (x[,"n_03"] + x[,"n_21"])*log10(4*(4 - (3*p2)/2) + 3*p1*(1 + 3*p2)) - (x[,"n_01"] + x[,"n_23"])*log10(4*(4 - (3*p2)/2) + 3*p1*(1 + 3*p2)) - (x[,"n_02"] + x[,"n_22"])*log10(((4 + 3*p1)*(2 - 3*p2))/4 - ((4 + 3*p1)*(-2 + 3*p2))/4 + (8 - 3*p1)*(4 + 3*p2)) + (x[,"n_01"] + x[,"n_23"])*log10(4*(4 + p2*(3 - 9*r_cr4)) + 3*p1*(4 + 3*p2 - 6*r_cr4)) + (x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cr4)) + log10(pmax(1e-6,r_cr4))) + (x[,"n_10"] + x[,"n_14"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(1 - 2*r_cr4 + 2*r_cr4^2)) + (x[,"n_02"] + x[,"n_22"])*log10((8 - 3*p1)*(4 + 3*p2) - (4 + 3*p1)*(-2 + 3*p2)*(-1 + r_cr4)^2 + (4 + 3*p1)*(2 - 3*p2)*r_cr4^2) + (x[,"n_03"] + x[,"n_21"])*log10(3*p1*(-2 + 3*p2 + 6*r_cr4) + 4*(4 + p2*(-6 + 9*r_cr4))) + x[,"n_12"]*log10(8*(4 + 2*r_cr4 - 2*r_cr4^2 + 3*p2*(1 - r_cr4 + r_cr4^2)) + 3*p1*(-4*(1 - r_cr4 + r_cr4^2) + p2*(-3 - 6*r_cr4 + 6*r_cr4^2)))


logL_cr4 <- (-5*x[,"n_00"] - 5*x[,"n_01"] - 5*x[,"n_02"] - 5*x[,"n_03"] - 5*x[,"n_04"] - 5*x[,"n_10"] - 4*x[,"n_11"] - 4*x[,"n_12"] - 4*x[,"n_13"] - 5*x[,"n_14"] - 5*x[,"n_20"] - 5*x[,"n_21"] - 5*x[,"n_22"] - 5*x[,"n_23"] - 5*x[,"n_24"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(3) + (x[,"n_11"] + x[,"n_13"])*log(16 - 6*p2 + p1*(3 + 9*p2)) + (x[,"n_01"] + x[,"n_23"])*log(4*(4 + p2*(3 - 9*r_cr4)) + 3*p1*(4 + 3*p2 - 6*r_cr4)) + (x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cr4)) + log(pmax(1e-6,r_cr4))) + (x[,"n_10"] + x[,"n_14"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r_cr4 + 2*r_cr4^2)) + (x[,"n_02"] + x[,"n_22"])*log((8 - 3*p1)*(4 + 3*p2) - (4 + 3*p1)*(-2 + 3*p2)*(-1 + r_cr4)^2 + (4 + 3*p1)*(2 - 3*p2)*r_cr4^2) + (x[,"n_03"] + x[,"n_21"])*log(3*p1*(-2 + 3*p2 + 6*r_cr4) + 4*(4 + p2*(-6 + 9*r_cr4))) + x[,"n_12"]*log(8*(4 + 2*r_cr4 - 2*r_cr4^2 + 3*p2*(1 - r_cr4 + r_cr4^2)) + 3*p1*(-4*(1 - r_cr4 + r_cr4^2) + p2*(-3 - 6*r_cr4 + 6*r_cr4^2)))


logL_rc1 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-5*n00 - 5*n01 - 5*n02 - 5*n03 - 5*n04 - 5*n10 - 4*n11 - 4*n12 - 4*n13 - 5*n14 - 5*n20 - 5*n21 - 5*n22 - 5*n23 - 5*n24)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(3) + (n11 + n13)*log(16 - 6*p2 + p1*(3 + 9*p2)) + (n03 + n21)*log(4*(4 + p2*(3 - 9*r)) + 3*p1*(4 + 3*p2 - 6*r)) + (n00 + n04 + n20 + n24)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n14)*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r + 2*r^2)) + (n02 + n22)*log((8 - 3*p1)*(4 + 3*p2) - (4 + 3*p1)*(-2 + 3*p2)*(-1 + r)^2 + (4 + 3*p1)*(2 - 3*p2)*r^2) + (n01 + n23)*log(3*p1*(-2 + 3*p2 + 6*r) + 4*(4 + p2*(-6 + 9*r))) + n12*log(8*(4 + 2*r - 2*r^2 + 3*p2*(1 - r + r^2)) + 3*p1*(-4*(1 - r + r^2) + p2*(-3 - 6*r + 6*r^2)))
return(L)}
interlogL_rc1 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_rc1,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc1 <- parallel::mcmapply(interlogL_rc1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_rc1 <- -((x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2))) - (x[,"n_10"] + x[,"n_14"])*(-log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - x[,"n_12"]*log10(3*p1*(-3 - (9*p2)/2) + 8*(9/2 + (9*p2)/4)) - (x[,"n_03"] + x[,"n_21"])*log10(4*(4 - (3*p2)/2) + 3*p1*(1 + 3*p2)) - (x[,"n_01"] + x[,"n_23"])*log10(4*(4 - (3*p2)/2) + 3*p1*(1 + 3*p2)) - (x[,"n_02"] + x[,"n_22"])*log10(((4 + 3*p1)*(2 - 3*p2))/4 - ((4 + 3*p1)*(-2 + 3*p2))/4 + (8 - 3*p1)*(4 + 3*p2)) + (x[,"n_03"] + x[,"n_21"])*log10(4*(4 + p2*(3 - 9*r_rc1)) + 3*p1*(4 + 3*p2 - 6*r_rc1)) + (x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rc1)) + log10(pmax(1e-6,r_rc1))) + (x[,"n_10"] + x[,"n_14"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(1 - 2*r_rc1 + 2*r_rc1^2)) + (x[,"n_02"] + x[,"n_22"])*log10((8 - 3*p1)*(4 + 3*p2) - (4 + 3*p1)*(-2 + 3*p2)*(-1 + r_rc1)^2 + (4 + 3*p1)*(2 - 3*p2)*r_rc1^2) + (x[,"n_01"] + x[,"n_23"])*log10(3*p1*(-2 + 3*p2 + 6*r_rc1) + 4*(4 + p2*(-6 + 9*r_rc1))) + x[,"n_12"]*log10(8*(4 + 2*r_rc1 - 2*r_rc1^2 + 3*p2*(1 - r_rc1 + r_rc1^2)) + 3*p1*(-4*(1 - r_rc1 + r_rc1^2) + p2*(-3 - 6*r_rc1 + 6*r_rc1^2)))


logL_rc1 <- (-5*x[,"n_00"] - 5*x[,"n_01"] - 5*x[,"n_02"] - 5*x[,"n_03"] - 5*x[,"n_04"] - 5*x[,"n_10"] - 4*x[,"n_11"] - 4*x[,"n_12"] - 4*x[,"n_13"] - 5*x[,"n_14"] - 5*x[,"n_20"] - 5*x[,"n_21"] - 5*x[,"n_22"] - 5*x[,"n_23"] - 5*x[,"n_24"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(3) + (x[,"n_11"] + x[,"n_13"])*log(16 - 6*p2 + p1*(3 + 9*p2)) + (x[,"n_03"] + x[,"n_21"])*log(4*(4 + p2*(3 - 9*r_rc1)) + 3*p1*(4 + 3*p2 - 6*r_rc1)) + (x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rc1)) + log(pmax(1e-6,r_rc1))) + (x[,"n_10"] + x[,"n_14"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r_rc1 + 2*r_rc1^2)) + (x[,"n_02"] + x[,"n_22"])*log((8 - 3*p1)*(4 + 3*p2) - (4 + 3*p1)*(-2 + 3*p2)*(-1 + r_rc1)^2 + (4 + 3*p1)*(2 - 3*p2)*r_rc1^2) + (x[,"n_01"] + x[,"n_23"])*log(3*p1*(-2 + 3*p2 + 6*r_rc1) + 4*(4 + p2*(-6 + 9*r_rc1))) + x[,"n_12"]*log(8*(4 + 2*r_rc1 - 2*r_rc1^2 + 3*p2*(1 - r_rc1 + r_rc1^2)) + 3*p1*(-4*(1 - r_rc1 + r_rc1^2) + p2*(-3 - 6*r_rc1 + 6*r_rc1^2)))


logL_rc2 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-6*n00 - 6*n01 - 6*n02 - 6*n03 - 6*n04 - 6*n10 - 5*n11 - 5*n12 - 5*n13 - 6*n14 - 6*n20 - 6*n21 - 6*n22 - 6*n23 - 6*n24)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(3) + (n11 + n13)*log(32 + p1*(6 - 9*p2) + 6*p2) + (n01 + n23)*log(4*(8 + p2*(6 - 9*r)) - 3*p1*(4 + 3*p2 - 12*r)) + (n00 + n04 + n20 + n24)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + n12*log((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r)*r) + (n10 + n14)*(log(4 + 3*p1) + log(4 + 3*p2) + log((-1 + r)^2 + r^2)) + (n02 + n22)*log((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r)^2 + (4 + 3*p1)*(4 + 3*p2)*r^2) + (n03 + n21)*log(-3*p1*(-8 + 3*p2 + 12*r) + 4*(8 + p2*(-3 + 9*r)))
return(L)}
interlogL_rc2 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_rc2,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc2 <- parallel::mcmapply(interlogL_rc2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_rc2 <- -((x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2))) - (x[,"n_10"] + x[,"n_14"])*(-log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_03"] + x[,"n_21"])*log10(4*(8 + (3*p2)/2) - 3*p1*(-2 + 3*p2)) - (x[,"n_01"] + x[,"n_23"])*log10(4*(8 + (3*p2)/2) - 3*p1*(-2 + 3*p2)) - x[,"n_12"]*log10((-8 + 3*p1)*(-8 + 3*p2) + ((4 + 3*p1)*(4 + 3*p2))/2) - (x[,"n_02"] + x[,"n_22"])*log10((-8 + 3*p1)*(-8 + 3*p2) + ((4 + 3*p1)*(4 + 3*p2))/2) + (x[,"n_01"] + x[,"n_23"])*log10(4*(8 + p2*(6 - 9*r_rc2)) - 3*p1*(4 + 3*p2 - 12*r_rc2)) + (x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rc2)) + log10(pmax(1e-6,r_rc2))) + x[,"n_12"]*log10((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_rc2)*r_rc2) + (x[,"n_10"] + x[,"n_14"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10((-1 + r_rc2)^2 + r_rc2^2)) + (x[,"n_02"] + x[,"n_22"])*log10((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r_rc2)^2 + (4 + 3*p1)*(4 + 3*p2)*r_rc2^2) + (x[,"n_03"] + x[,"n_21"])*log10(-3*p1*(-8 + 3*p2 + 12*r_rc2) + 4*(8 + p2*(-3 + 9*r_rc2)))


logL_rc2 <- (-6*x[,"n_00"] - 6*x[,"n_01"] - 6*x[,"n_02"] - 6*x[,"n_03"] - 6*x[,"n_04"] - 6*x[,"n_10"] - 5*x[,"n_11"] - 5*x[,"n_12"] - 5*x[,"n_13"] - 6*x[,"n_14"] - 6*x[,"n_20"] - 6*x[,"n_21"] - 6*x[,"n_22"] - 6*x[,"n_23"] - 6*x[,"n_24"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(3) + (x[,"n_11"] + x[,"n_13"])*log(32 + p1*(6 - 9*p2) + 6*p2) + (x[,"n_01"] + x[,"n_23"])*log(4*(8 + p2*(6 - 9*r_rc2)) - 3*p1*(4 + 3*p2 - 12*r_rc2)) + (x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rc2)) + log(pmax(1e-6,r_rc2))) + x[,"n_12"]*log((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_rc2)*r_rc2) + (x[,"n_10"] + x[,"n_14"])*(log(4 + 3*p1) + log(4 + 3*p2) + log((-1 + r_rc2)^2 + r_rc2^2)) + (x[,"n_02"] + x[,"n_22"])*log((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r_rc2)^2 + (4 + 3*p1)*(4 + 3*p2)*r_rc2^2) + (x[,"n_03"] + x[,"n_21"])*log(-3*p1*(-8 + 3*p2 + 12*r_rc2) + 4*(8 + p2*(-3 + 9*r_rc2)))


logL_rc3 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-4*n00 - 4*n01 - 3*n02 - 4*n03 - 4*n04 - 4*n10 - 3*n11 - 3*n12 - 3*n13 - 4*n14 - 4*n20 - 4*n21 - 3*n22 - 4*n23 - 4*n24)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(3) + (n11 + n13)*log(8 - 3*p2 - 3*p1*(1 + 3*p2)) + (n03 + n21)*log(8 + p2*(6 - 18*r) - 3*p1*(4 + 3*p2 - 6*r)) + (n04 + n20)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n24)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n14)*(log(2 - 3*p1) + log(2 - 3*p2) + log(1 - 2*r + 2*r^2)) + (n01 + n23)*log(-3*p1*(-2 + 3*p2 + 6*r) + 2*(4 + p2*(-6 + 9*r))) + (n02 + n22)*log(10 - 4*r + 4*r^2 + p2*(3 + 6*r - 6*r^2) + p1*(3 + 6*r - 6*r^2 + 9*p2*(1 - r + r^2))) + n12*log(4*(4 + 2*r - 2*r^2 + 3*p2*(1 - r + r^2)) - 3*p1*(-4*(1 - r + r^2) + p2*(-3 - 6*r + 6*r^2)))
return(L)}
interlogL_rc3 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_rc3,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc3 <- parallel::mcmapply(interlogL_rc3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_rc3 <- -((x[,"n_04"] + x[,"n_20"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_24"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_14"])*(-log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - x[,"n_12"]*log10(-3*p1*(-3 - (9*p2)/2) + 4*(9/2 + (9*p2)/4)) - (x[,"n_01"] + x[,"n_23"])*log10(2*(4 - (3*p2)/2) - 3*p1*(1 + 3*p2)) - (x[,"n_03"] + x[,"n_21"])*log10(8 - 3*p2 - 3*p1*(1 + 3*p2)) - (x[,"n_02"] + x[,"n_22"])*log10(9 + (9*p2)/2 + p1*(9/2 + (27*p2)/4)) + (x[,"n_03"] + x[,"n_21"])*log10(8 + p2*(6 - 18*r_rc3) - 3*p1*(4 + 3*p2 - 6*r_rc3)) + (x[,"n_04"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rc3)) + log10(pmax(1e-6,r_rc3))) + (x[,"n_00"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rc3)) + log10(pmax(1e-6,r_rc3))) + (x[,"n_10"] + x[,"n_14"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(1 - 2*r_rc3 + 2*r_rc3^2)) + (x[,"n_01"] + x[,"n_23"])*log10(-3*p1*(-2 + 3*p2 + 6*r_rc3) + 2*(4 + p2*(-6 + 9*r_rc3))) + (x[,"n_02"] + x[,"n_22"])*log10(10 - 4*r_rc3 + 4*r_rc3^2 + p2*(3 + 6*r_rc3 - 6*r_rc3^2) + p1*(3 + 6*r_rc3 - 6*r_rc3^2 + 9*p2*(1 - r_rc3 + r_rc3^2))) + x[,"n_12"]*log10(4*(4 + 2*r_rc3 - 2*r_rc3^2 + 3*p2*(1 - r_rc3 + r_rc3^2)) - 3*p1*(-4*(1 - r_rc3 + r_rc3^2) + p2*(-3 - 6*r_rc3 + 6*r_rc3^2)))


logL_rc3 <- (-4*x[,"n_00"] - 4*x[,"n_01"] - 3*x[,"n_02"] - 4*x[,"n_03"] - 4*x[,"n_04"] - 4*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 4*x[,"n_14"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 3*x[,"n_22"] - 4*x[,"n_23"] - 4*x[,"n_24"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(3) + (x[,"n_11"] + x[,"n_13"])*log(8 - 3*p2 - 3*p1*(1 + 3*p2)) + (x[,"n_03"] + x[,"n_21"])*log(8 + p2*(6 - 18*r_rc3) - 3*p1*(4 + 3*p2 - 6*r_rc3)) + (x[,"n_04"] + x[,"n_20"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rc3)) + log(pmax(1e-6,r_rc3))) + (x[,"n_00"] + x[,"n_24"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rc3)) + log(pmax(1e-6,r_rc3))) + (x[,"n_10"] + x[,"n_14"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(1 - 2*r_rc3 + 2*r_rc3^2)) + (x[,"n_01"] + x[,"n_23"])*log(-3*p1*(-2 + 3*p2 + 6*r_rc3) + 2*(4 + p2*(-6 + 9*r_rc3))) + (x[,"n_02"] + x[,"n_22"])*log(10 - 4*r_rc3 + 4*r_rc3^2 + p2*(3 + 6*r_rc3 - 6*r_rc3^2) + p1*(3 + 6*r_rc3 - 6*r_rc3^2 + 9*p2*(1 - r_rc3 + r_rc3^2))) + x[,"n_12"]*log(4*(4 + 2*r_rc3 - 2*r_rc3^2 + 3*p2*(1 - r_rc3 + r_rc3^2)) - 3*p1*(-4*(1 - r_rc3 + r_rc3^2) + p2*(-3 - 6*r_rc3 + 6*r_rc3^2)))


logL_rc4 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-5*n00 - 5*n01 - 5*n02 - 5*n03 - 5*n04 - 5*n10 - 4*n11 - 4*n12 - 4*n13 - 5*n14 - 5*n20 - 5*n21 - 5*n22 - 5*n23 - 5*n24)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(3) + (n11 + n13)*log(16 + 3*p2 + p1*(-6 + 9*p2)) + (n01 + n23)*log(2*(8 + p2*(6 - 9*r)) + 3*p1*(4 + 3*p2 - 12*r)) + (n00 + n04 + n20 + n24)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n14)*(log(2 - 3*p1) + log(4 + 3*p2) + log(1 - 2*r + 2*r^2)) + (n02 + n22)*log((-4 - 3*p1)*(-8 + 3*p2) - (-2 + 3*p1)*(4 + 3*p2)*(-1 + r)^2 + (2 - 3*p1)*(4 + 3*p2)*r^2) + (n03 + n21)*log(3*p1*(-8 + 3*p2 + 12*r) + 2*(8 + p2*(-3 + 9*r))) + n12*log(-4*(4*(-2 - r + r^2) + 3*p2*(1 - r + r^2)) + 3*p1*(8*(1 - r + r^2) + p2*(-3 - 6*r + 6*r^2)))
return(L)}
interlogL_rc4 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_rc4,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc4 <- parallel::mcmapply(interlogL_rc4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_rc4 <- -((x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2))) - (x[,"n_10"] + x[,"n_14"])*(-log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - x[,"n_12"]*log10(3*p1*(6 - (9*p2)/2) - 4*(-9 + (9*p2)/4)) - (x[,"n_03"] + x[,"n_21"])*log10(2*(8 + (3*p2)/2) + 3*p1*(-2 + 3*p2)) - (x[,"n_01"] + x[,"n_23"])*log10(2*(8 + (3*p2)/2) + 3*p1*(-2 + 3*p2)) - (x[,"n_02"] + x[,"n_22"])*log10((-4 - 3*p1)*(-8 + 3*p2) + ((2 - 3*p1)*(4 + 3*p2))/4 - ((-2 + 3*p1)*(4 + 3*p2))/4) + (x[,"n_01"] + x[,"n_23"])*log10(2*(8 + p2*(6 - 9*r_rc4)) + 3*p1*(4 + 3*p2 - 12*r_rc4)) + (x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rc4)) + log10(pmax(1e-6,r_rc4))) + (x[,"n_10"] + x[,"n_14"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(1 - 2*r_rc4 + 2*r_rc4^2)) + (x[,"n_02"] + x[,"n_22"])*log10((-4 - 3*p1)*(-8 + 3*p2) - (-2 + 3*p1)*(4 + 3*p2)*(-1 + r_rc4)^2 + (2 - 3*p1)*(4 + 3*p2)*r_rc4^2) + (x[,"n_03"] + x[,"n_21"])*log10(3*p1*(-8 + 3*p2 + 12*r_rc4) + 2*(8 + p2*(-3 + 9*r_rc4))) + x[,"n_12"]*log10(-4*(4*(-2 - r_rc4 + r_rc4^2) + 3*p2*(1 - r_rc4 + r_rc4^2)) + 3*p1*(8*(1 - r_rc4 + r_rc4^2) + p2*(-3 - 6*r_rc4 + 6*r_rc4^2)))


logL_rc4 <- (-5*x[,"n_00"] - 5*x[,"n_01"] - 5*x[,"n_02"] - 5*x[,"n_03"] - 5*x[,"n_04"] - 5*x[,"n_10"] - 4*x[,"n_11"] - 4*x[,"n_12"] - 4*x[,"n_13"] - 5*x[,"n_14"] - 5*x[,"n_20"] - 5*x[,"n_21"] - 5*x[,"n_22"] - 5*x[,"n_23"] - 5*x[,"n_24"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(3) + (x[,"n_11"] + x[,"n_13"])*log(16 + 3*p2 + p1*(-6 + 9*p2)) + (x[,"n_01"] + x[,"n_23"])*log(2*(8 + p2*(6 - 9*r_rc4)) + 3*p1*(4 + 3*p2 - 12*r_rc4)) + (x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rc4)) + log(pmax(1e-6,r_rc4))) + (x[,"n_10"] + x[,"n_14"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(1 - 2*r_rc4 + 2*r_rc4^2)) + (x[,"n_02"] + x[,"n_22"])*log((-4 - 3*p1)*(-8 + 3*p2) - (-2 + 3*p1)*(4 + 3*p2)*(-1 + r_rc4)^2 + (2 - 3*p1)*(4 + 3*p2)*r_rc4^2) + (x[,"n_03"] + x[,"n_21"])*log(3*p1*(-8 + 3*p2 + 12*r_rc4) + 2*(8 + p2*(-3 + 9*r_rc4))) + x[,"n_12"]*log(-4*(4*(-2 - r_rc4 + r_rc4^2) + 3*p2*(1 - r_rc4 + r_rc4^2)) + 3*p1*(8*(1 - r_rc4 + r_rc4^2) + p2*(-3 - 6*r_rc4 + 6*r_rc4^2)))


logL_rr1 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-6*n00 - 5*n01 - 6*n02 - 5*n03 - 6*n04 - 5*n10 - 5*n11 - 5*n12 - 5*n13 - 5*n14 - 6*n20 - 5*n21 - 6*n22 - 5*n23 - 6*n24)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(3) + (n11 + n13)*log(32 + p1*(6 - 9*p2) + 6*p2) + (n03 + n21)*(log(32 + 6*p2 - p1*(-6 + 9*p2)) + log(pmax(1e-6,1 - r))) + (n01 + n23)*(log(32 + p1*(6 - 9*p2) + 6*p2) + log(pmax(1e-6,r))) + (n10 + n14)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n24)*(log(4 + 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r))) + (n02 + n22)*log((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r)*r) + (n04 + n20)*(log(4 + 3*p1) + log(4 + 3*p2) + log(1 - 2*r + r^2)) + n12*log((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r)^2 + (4 + 3*p1)*(4 + 3*p2)*r^2)
return(L)}
interlogL_rr1 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_rr1,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr1 <- parallel::mcmapply(interlogL_rr1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_rr1 <- -((x[,"n_10"] + x[,"n_14"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2))) - (x[,"n_04"] + x[,"n_20"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_24"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_23"])*(-log10(2) + log10(32 + p1*(6 - 9*p2) + 6*p2)) - x[,"n_12"]*log10((-8 + 3*p1)*(-8 + 3*p2) + ((4 + 3*p1)*(4 + 3*p2))/2) - (x[,"n_02"] + x[,"n_22"])*log10((-8 + 3*p1)*(-8 + 3*p2) + ((4 + 3*p1)*(4 + 3*p2))/2) - (x[,"n_03"] + x[,"n_21"])*(-log10(2) + log10(32 + 6*p2 - p1*(-6 + 9*p2))) + (x[,"n_03"] + x[,"n_21"])*(log10(32 + 6*p2 - p1*(-6 + 9*p2)) + log10(pmax(1e-6,1 - r_rr1))) + (x[,"n_01"] + x[,"n_23"])*(log10(32 + p1*(6 - 9*p2) + 6*p2) + log10(pmax(1e-6,r_rr1))) + (x[,"n_10"] + x[,"n_14"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rr1)) + log10(pmax(1e-6,r_rr1))) + (x[,"n_00"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_rr1))) + (x[,"n_02"] + x[,"n_22"])*log10((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_rr1)*r_rr1) + (x[,"n_04"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(1 - 2*r_rr1 + r_rr1^2)) + x[,"n_12"]*log10((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r_rr1)^2 + (4 + 3*p1)*(4 + 3*p2)*r_rr1^2)


logL_rr1 <- (-6*x[,"n_00"] - 5*x[,"n_01"] - 6*x[,"n_02"] - 5*x[,"n_03"] - 6*x[,"n_04"] - 5*x[,"n_10"] - 5*x[,"n_11"] - 5*x[,"n_12"] - 5*x[,"n_13"] - 5*x[,"n_14"] - 6*x[,"n_20"] - 5*x[,"n_21"] - 6*x[,"n_22"] - 5*x[,"n_23"] - 6*x[,"n_24"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(3) + (x[,"n_11"] + x[,"n_13"])*log(32 + p1*(6 - 9*p2) + 6*p2) + (x[,"n_03"] + x[,"n_21"])*(log(32 + 6*p2 - p1*(-6 + 9*p2)) + log(pmax(1e-6,1 - r_rr1))) + (x[,"n_01"] + x[,"n_23"])*(log(32 + p1*(6 - 9*p2) + 6*p2) + log(pmax(1e-6,r_rr1))) + (x[,"n_10"] + x[,"n_14"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rr1)) + log(pmax(1e-6,r_rr1))) + (x[,"n_00"] + x[,"n_24"])*(log(4 + 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r_rr1))) + (x[,"n_02"] + x[,"n_22"])*log((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_rr1)*r_rr1) + (x[,"n_04"] + x[,"n_20"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(1 - 2*r_rr1 + r_rr1^2)) + x[,"n_12"]*log((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r_rr1)^2 + (4 + 3*p1)*(4 + 3*p2)*r_rr1^2)


logL_rr2 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-5*n00 - 4*n01 - 5*n02 - 4*n03 - 5*n04 - 4*n10 - 4*n11 - 3*n12 - 4*n13 - 4*n14 - 5*n20 - 4*n21 - 5*n22 - 4*n23 - 5*n24)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(3) + (n11 + n13)*log(16 - 6*p2 + p1*(3 + 9*p2)) + (n03 + n21)*(log(16 - 6*p2 + p1*(3 + 9*p2)) + log(pmax(1e-6,1 - r))) + (n01 + n23)*(log(16 - 6*p2 + p1*(3 + 9*p2)) + log(pmax(1e-6,r))) + (n10 + n14)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n24)*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r))) + (n02 + n22)*log((8 - 3*p1)*(4 + 3*p2) + 2*(4 + 3*p1)*(-2 + 3*p2)*(-1 + r)*r) + (n04 + n20)*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r + r^2)) + n12*log(20 - 8*r + 8*r^2 + p2*(6 + 12*r - 12*r^2) - 3*p1*(1 + 2*r - 2*r^2 + 3*p2*(1 - r + r^2)))
return(L)}
interlogL_rr2 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_rr2,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr2 <- parallel::mcmapply(interlogL_rr2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_rr2 <- -((x[,"n_10"] + x[,"n_14"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2))) - (x[,"n_04"] + x[,"n_20"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_24"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - x[,"n_12"]*log10(18 + 9*p2 - 3*p1*(3/2 + (9*p2)/4)) - (x[,"n_02"] + x[,"n_22"])*log10(-((4 + 3*p1)*(-2 + 3*p2))/2 + (8 - 3*p1)*(4 + 3*p2)) - (x[,"n_03"] + x[,"n_21"])*(-log10(2) + log10(16 - 6*p2 + p1*(3 + 9*p2))) - (x[,"n_01"] + x[,"n_23"])*(-log10(2) + log10(16 - 6*p2 + p1*(3 + 9*p2))) + (x[,"n_03"] + x[,"n_21"])*(log10(16 - 6*p2 + p1*(3 + 9*p2)) + log10(pmax(1e-6,1 - r_rr2))) + (x[,"n_01"] + x[,"n_23"])*(log10(16 - 6*p2 + p1*(3 + 9*p2)) + log10(pmax(1e-6,r_rr2))) + (x[,"n_10"] + x[,"n_14"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rr2)) + log10(pmax(1e-6,r_rr2))) + (x[,"n_00"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_rr2))) + (x[,"n_02"] + x[,"n_22"])*log10((8 - 3*p1)*(4 + 3*p2) + 2*(4 + 3*p1)*(-2 + 3*p2)*(-1 + r_rr2)*r_rr2) + (x[,"n_04"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(1 - 2*r_rr2 + r_rr2^2)) + x[,"n_12"]*log10(20 - 8*r_rr2 + 8*r_rr2^2 + p2*(6 + 12*r_rr2 - 12*r_rr2^2) - 3*p1*(1 + 2*r_rr2 - 2*r_rr2^2 + 3*p2*(1 - r_rr2 + r_rr2^2)))


logL_rr2 <- (-5*x[,"n_00"] - 4*x[,"n_01"] - 5*x[,"n_02"] - 4*x[,"n_03"] - 5*x[,"n_04"] - 4*x[,"n_10"] - 4*x[,"n_11"] - 3*x[,"n_12"] - 4*x[,"n_13"] - 4*x[,"n_14"] - 5*x[,"n_20"] - 4*x[,"n_21"] - 5*x[,"n_22"] - 4*x[,"n_23"] - 5*x[,"n_24"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(3) + (x[,"n_11"] + x[,"n_13"])*log(16 - 6*p2 + p1*(3 + 9*p2)) + (x[,"n_03"] + x[,"n_21"])*(log(16 - 6*p2 + p1*(3 + 9*p2)) + log(pmax(1e-6,1 - r_rr2))) + (x[,"n_01"] + x[,"n_23"])*(log(16 - 6*p2 + p1*(3 + 9*p2)) + log(pmax(1e-6,r_rr2))) + (x[,"n_10"] + x[,"n_14"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rr2)) + log(pmax(1e-6,r_rr2))) + (x[,"n_00"] + x[,"n_24"])*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r_rr2))) + (x[,"n_02"] + x[,"n_22"])*log((8 - 3*p1)*(4 + 3*p2) + 2*(4 + 3*p1)*(-2 + 3*p2)*(-1 + r_rr2)*r_rr2) + (x[,"n_04"] + x[,"n_20"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r_rr2 + r_rr2^2)) + x[,"n_12"]*log(20 - 8*r_rr2 + 8*r_rr2^2 + p2*(6 + 12*r_rr2 - 12*r_rr2^2) - 3*p1*(1 + 2*r_rr2 - 2*r_rr2^2 + 3*p2*(1 - r_rr2 + r_rr2^2)))


logL_rr3 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-5*n00 - 4*n01 - 5*n02 - 4*n03 - 5*n04 - 4*n10 - 4*n11 - 3*n12 - 4*n13 - 4*n14 - 5*n20 - 4*n21 - 5*n22 - 4*n23 - 5*n24)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(3) + (n11 + n13)*log(16 + 3*p2 + p1*(-6 + 9*p2)) + (n03 + n21)*(log(16 + 3*p2 + p1*(-6 + 9*p2)) + log(pmax(1e-6,1 - r))) + (n01 + n23)*(log(16 + 3*p2 + p1*(-6 + 9*p2)) + log(pmax(1e-6,r))) + (n10 + n14)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n24)*(log(2 - 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r))) + (n02 + n22)*log((-4 - 3*p1)*(-8 + 3*p2) + 2*(-2 + 3*p1)*(4 + 3*p2)*(-1 + r)*r) + (n04 + n20)*(log(2 - 3*p1) + log(4 + 3*p2) + log(1 - 2*r + r^2)) + n12*log(20 - 8*r + 8*r^2 + p2*(-3 - 6*r + 6*r^2) - 3*p1*(-2 - 4*r + 4*r^2 + 3*p2*(1 - r + r^2)))
return(L)}
interlogL_rr3 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_rr3,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr3 <- parallel::mcmapply(interlogL_rr3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_rr3 <- -((x[,"n_10"] + x[,"n_14"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2))) - (x[,"n_04"] + x[,"n_20"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_24"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - x[,"n_12"]*log10(18 - (9*p2)/2 - 3*p1*(-3 + (9*p2)/4)) - (x[,"n_02"] + x[,"n_22"])*log10((-4 - 3*p1)*(-8 + 3*p2) - ((-2 + 3*p1)*(4 + 3*p2))/2) - (x[,"n_03"] + x[,"n_21"])*(-log10(2) + log10(16 + 3*p2 + p1*(-6 + 9*p2))) - (x[,"n_01"] + x[,"n_23"])*(-log10(2) + log10(16 + 3*p2 + p1*(-6 + 9*p2))) + (x[,"n_03"] + x[,"n_21"])*(log10(16 + 3*p2 + p1*(-6 + 9*p2)) + log10(pmax(1e-6,1 - r_rr3))) + (x[,"n_01"] + x[,"n_23"])*(log10(16 + 3*p2 + p1*(-6 + 9*p2)) + log10(pmax(1e-6,r_rr3))) + (x[,"n_10"] + x[,"n_14"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rr3)) + log10(pmax(1e-6,r_rr3))) + (x[,"n_00"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_rr3))) + (x[,"n_02"] + x[,"n_22"])*log10((-4 - 3*p1)*(-8 + 3*p2) + 2*(-2 + 3*p1)*(4 + 3*p2)*(-1 + r_rr3)*r_rr3) + (x[,"n_04"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(1 - 2*r_rr3 + r_rr3^2)) + x[,"n_12"]*log10(20 - 8*r_rr3 + 8*r_rr3^2 + p2*(-3 - 6*r_rr3 + 6*r_rr3^2) - 3*p1*(-2 - 4*r_rr3 + 4*r_rr3^2 + 3*p2*(1 - r_rr3 + r_rr3^2)))


logL_rr3 <- (-5*x[,"n_00"] - 4*x[,"n_01"] - 5*x[,"n_02"] - 4*x[,"n_03"] - 5*x[,"n_04"] - 4*x[,"n_10"] - 4*x[,"n_11"] - 3*x[,"n_12"] - 4*x[,"n_13"] - 4*x[,"n_14"] - 5*x[,"n_20"] - 4*x[,"n_21"] - 5*x[,"n_22"] - 4*x[,"n_23"] - 5*x[,"n_24"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(3) + (x[,"n_11"] + x[,"n_13"])*log(16 + 3*p2 + p1*(-6 + 9*p2)) + (x[,"n_03"] + x[,"n_21"])*(log(16 + 3*p2 + p1*(-6 + 9*p2)) + log(pmax(1e-6,1 - r_rr3))) + (x[,"n_01"] + x[,"n_23"])*(log(16 + 3*p2 + p1*(-6 + 9*p2)) + log(pmax(1e-6,r_rr3))) + (x[,"n_10"] + x[,"n_14"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rr3)) + log(pmax(1e-6,r_rr3))) + (x[,"n_00"] + x[,"n_24"])*(log(2 - 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r_rr3))) + (x[,"n_02"] + x[,"n_22"])*log((-4 - 3*p1)*(-8 + 3*p2) + 2*(-2 + 3*p1)*(4 + 3*p2)*(-1 + r_rr3)*r_rr3) + (x[,"n_04"] + x[,"n_20"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(1 - 2*r_rr3 + r_rr3^2)) + x[,"n_12"]*log(20 - 8*r_rr3 + 8*r_rr3^2 + p2*(-3 - 6*r_rr3 + 6*r_rr3^2) - 3*p1*(-2 - 4*r_rr3 + 4*r_rr3^2 + 3*p2*(1 - r_rr3 + r_rr3^2)))


logL_rr4 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-4*n00 - 3*n01 - 4*n02 - 3*n03 - 4*n04 - 3*n10 - 3*n11 - 2*n12 - 3*n13 - 3*n14 - 4*n20 - 3*n21 - 4*n22 - 3*n23 - 4*n24)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(3) + (n11 + n13)*log(8 - 3*p2 - 3*p1*(1 + 3*p2)) + (n03 + n21)*(log(8 - 3*p2 - p1*(3 + 9*p2)) + log(pmax(1e-6,1 - r))) + (n01 + n23)*(log(8 - 3*p2 - p1*(3 + 9*p2)) + log(pmax(1e-6,r))) + (n10 + n14)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n24)*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r))) + (n04 + n20)*(log(2 - 3*p1) + log(2 - 3*p2) + log(1 - 2*r + r^2)) + n12*log(10 - 4*r + 4*r^2 + p2*(3 + 6*r - 6*r^2) + p1*(3 + 6*r - 6*r^2 + 9*p2*(1 - r + r^2))) + (n02 + n22)*log(4*(4 + 2*r - 2*r^2 + 3*p2*(1 - r + r^2)) - 3*p1*(-4*(1 - r + r^2) + p2*(-3 - 6*r + 6*r^2)))
return(L)}
interlogL_rr4 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_rr4,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr4 <- parallel::mcmapply(interlogL_rr4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_rr4 <- -((x[,"n_10"] + x[,"n_14"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2))) - (x[,"n_04"] + x[,"n_20"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_24"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_22"])*log10(-3*p1*(-3 - (9*p2)/2) + 4*(9/2 + (9*p2)/4)) - x[,"n_12"]*log10(9 + (9*p2)/2 + p1*(9/2 + (27*p2)/4)) - (x[,"n_03"] + x[,"n_21"])*(-log10(2) + log10(8 - 3*p2 - p1*(3 + 9*p2))) - (x[,"n_01"] + x[,"n_23"])*(-log10(2) + log10(8 - 3*p2 - p1*(3 + 9*p2))) + (x[,"n_03"] + x[,"n_21"])*(log10(8 - 3*p2 - p1*(3 + 9*p2)) + log10(pmax(1e-6,1 - r_rr4))) + (x[,"n_01"] + x[,"n_23"])*(log10(8 - 3*p2 - p1*(3 + 9*p2)) + log10(pmax(1e-6,r_rr4))) + (x[,"n_10"] + x[,"n_14"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rr4)) + log10(pmax(1e-6,r_rr4))) + (x[,"n_00"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_rr4))) + (x[,"n_04"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(1 - 2*r_rr4 + r_rr4^2)) + x[,"n_12"]*log10(10 - 4*r_rr4 + 4*r_rr4^2 + p2*(3 + 6*r_rr4 - 6*r_rr4^2) + p1*(3 + 6*r_rr4 - 6*r_rr4^2 + 9*p2*(1 - r_rr4 + r_rr4^2))) + (x[,"n_02"] + x[,"n_22"])*log10(4*(4 + 2*r_rr4 - 2*r_rr4^2 + 3*p2*(1 - r_rr4 + r_rr4^2)) - 3*p1*(-4*(1 - r_rr4 + r_rr4^2) + p2*(-3 - 6*r_rr4 + 6*r_rr4^2)))


logL_rr4 <- (-4*x[,"n_00"] - 3*x[,"n_01"] - 4*x[,"n_02"] - 3*x[,"n_03"] - 4*x[,"n_04"] - 3*x[,"n_10"] - 3*x[,"n_11"] - 2*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 4*x[,"n_20"] - 3*x[,"n_21"] - 4*x[,"n_22"] - 3*x[,"n_23"] - 4*x[,"n_24"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(3) + (x[,"n_11"] + x[,"n_13"])*log(8 - 3*p2 - 3*p1*(1 + 3*p2)) + (x[,"n_03"] + x[,"n_21"])*(log(8 - 3*p2 - p1*(3 + 9*p2)) + log(pmax(1e-6,1 - r_rr4))) + (x[,"n_01"] + x[,"n_23"])*(log(8 - 3*p2 - p1*(3 + 9*p2)) + log(pmax(1e-6,r_rr4))) + (x[,"n_10"] + x[,"n_14"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rr4)) + log(pmax(1e-6,r_rr4))) + (x[,"n_00"] + x[,"n_24"])*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r_rr4))) + (x[,"n_04"] + x[,"n_20"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(1 - 2*r_rr4 + r_rr4^2)) + x[,"n_12"]*log(10 - 4*r_rr4 + 4*r_rr4^2 + p2*(3 + 6*r_rr4 - 6*r_rr4^2) + p1*(3 + 6*r_rr4 - 6*r_rr4^2 + 9*p2*(1 - r_rr4 + r_rr4^2))) + (x[,"n_02"] + x[,"n_22"])*log(4*(4 + 2*r_rr4 - 2*r_rr4^2 + 3*p2*(1 - r_rr4 + r_rr4^2)) - 3*p1*(-4*(1 - r_rr4 + r_rr4^2) + p2*(-3 - 6*r_rr4 + 6*r_rr4^2)))


return(list(
r_mat = cbind(r_cc1,r_cc2,r_cc3,r_cc4,r_cr1,r_cr2,r_cr3,r_cr4,r_rc1,r_rc2,r_rc3,r_rc4,r_rr1,r_rr2,r_rr3,r_rr4,0.499),
LOD_mat = cbind(LOD_cc1,LOD_cc2,LOD_cc3,LOD_cc4,LOD_cr1,LOD_cr2,LOD_cr3,LOD_cr4,LOD_rc1,LOD_rc2,LOD_rc3,LOD_rc4,LOD_rr1,LOD_rr2,LOD_rr3,LOD_rr4,0),
logL_mat = cbind(logL_cc1,logL_cc2,logL_cc3,logL_cc4,logL_cr1,logL_cr2,logL_cr3,logL_cr4,logL_rc1,logL_rc2,logL_rc3,logL_rc4,logL_rr1,logL_rr2,logL_rr3,logL_rr4,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling coupling","coupling coupling","coupling coupling","coupling repulsion","coupling repulsion","coupling repulsion","coupling repulsion","repulsion coupling","repulsion coupling","repulsion coupling","repulsion coupling","repulsion repulsion","repulsion repulsion","repulsion repulsion","repulsion repulsion","unknown")
)
)
}

#' @rdname p4_functions
p4_1.2_1.2 <- function(x,p1,p2,ncores=1){


logL_cc1 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- 3*(-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*log(2) + (-n00 - n03 - n11 - n12 - n21 - n22 - n30 - n33)*log(3) + (n02 + n13 + n20 + n31)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n03 + n30)*(log(2 - 3*p2) + 3*log(pmax(1e-6,r))) + (n00 + n33)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + r^2)) + (n01 + n10 + n23 + n32)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n12 + n21)*(log(pmax(1e-6,r)) + log(-3*p2*(2 - 12*r + 9*r^2) + 2*(8 - 12*r + 9*r^2))) + (n11 + n22)*(log(pmax(1e-6,1 - r)) + log(3*p2*(1 + 6*r - 9*r^2) + 2*(5 - 6*r + 9*r^2)))
return(L)}
interlogL_cc1 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_cc1,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc1 <- parallel::mcmapply(interlogL_cc1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_cc1 <- -((x[,"n_03"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p2))) - (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(-3*log10(2) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(-3*log10(2) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_33"])*(-3*log10(2) + log10(2 - 3*p2)) - (x[,"n_12"] + x[,"n_21"])*(-log10(2) + log10(17/2 + (21*p2)/4)) - (x[,"n_11"] + x[,"n_22"])*(-log10(2) + log10(17/2 + (21*p2)/4)) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cc1)) + 2*log10(pmax(1e-6,r_cc1))) + (x[,"n_03"] + x[,"n_30"])*(log10(2 - 3*p2) + 3*log10(pmax(1e-6,r_cc1))) + (x[,"n_00"] + x[,"n_33"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cc1)) + log10(1 - 2*r_cc1 + r_cc1^2)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_cc1)) + log10(1 - 2*r_cc1 + r_cc1^2)) + (x[,"n_12"] + x[,"n_21"])*(log10(pmax(1e-6,r_cc1)) + log10(-3*p2*(2 - 12*r_cc1 + 9*r_cc1^2) + 2*(8 - 12*r_cc1 + 9*r_cc1^2))) + (x[,"n_11"] + x[,"n_22"])*(log10(pmax(1e-6,1 - r_cc1)) + log10(3*p2*(1 + 6*r_cc1 - 9*r_cc1^2) + 2*(5 - 6*r_cc1 + 9*r_cc1^2)))


logL_cc1 <- 3*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*log(2) + (-x[,"n_00"] - x[,"n_03"] - x[,"n_11"] - x[,"n_12"] - x[,"n_21"] - x[,"n_22"] - x[,"n_30"] - x[,"n_33"])*log(3) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_cc1)) + 2*log(pmax(1e-6,r_cc1))) + (x[,"n_03"] + x[,"n_30"])*(log(2 - 3*p2) + 3*log(pmax(1e-6,r_cc1))) + (x[,"n_00"] + x[,"n_33"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_cc1)) + log(1 - 2*r_cc1 + r_cc1^2)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(log(2 - 3*p2) + log(pmax(1e-6,r_cc1)) + log(1 - 2*r_cc1 + r_cc1^2)) + (x[,"n_12"] + x[,"n_21"])*(log(pmax(1e-6,r_cc1)) + log(-3*p2*(2 - 12*r_cc1 + 9*r_cc1^2) + 2*(8 - 12*r_cc1 + 9*r_cc1^2))) + (x[,"n_11"] + x[,"n_22"])*(log(pmax(1e-6,1 - r_cc1)) + log(3*p2*(1 + 6*r_cc1 - 9*r_cc1^2) + 2*(5 - 6*r_cc1 + 9*r_cc1^2)))


logL_cc2 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- 4*(-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*log(2) + (-n00 - n03 - n11 - n12 - n21 - n22 - n30 - n33)*log(3) + (n02 + n13 + n20 + n31)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n03 + n30)*(log(4 + 3*p2) + 3*log(pmax(1e-6,r))) + (n00 + n33)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + r^2)) + (n01 + n10 + n23 + n32)*(log(4 + 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n12 + n21)*(log(pmax(1e-6,r)) + log(3*p2*(2 - 12*r + 9*r^2) + 4*(8 - 12*r + 9*r^2))) + (n11 + n22)*(log(pmax(1e-6,1 - r)) + log(3*p2*(-1 - 6*r + 9*r^2) + 4*(5 - 6*r + 9*r^2)))
return(L)}
interlogL_cc2 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_cc2,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc2 <- parallel::mcmapply(interlogL_cc2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_cc2 <- -((x[,"n_12"] + x[,"n_21"])*(-log10(2) + log10(17 - (21*p2)/4))) - (x[,"n_11"] + x[,"n_22"])*(-log10(2) + log10(17 - (21*p2)/4)) - (x[,"n_03"] + x[,"n_30"])*(-3*log10(2) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(-3*log10(2) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(-3*log10(2) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_33"])*(-3*log10(2) + log10(4 + 3*p2)) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cc2)) + 2*log10(pmax(1e-6,r_cc2))) + (x[,"n_03"] + x[,"n_30"])*(log10(4 + 3*p2) + 3*log10(pmax(1e-6,r_cc2))) + (x[,"n_00"] + x[,"n_33"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cc2)) + log10(1 - 2*r_cc2 + r_cc2^2)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(log10(4 + 3*p2) + log10(pmax(1e-6,r_cc2)) + log10(1 - 2*r_cc2 + r_cc2^2)) + (x[,"n_12"] + x[,"n_21"])*(log10(pmax(1e-6,r_cc2)) + log10(3*p2*(2 - 12*r_cc2 + 9*r_cc2^2) + 4*(8 - 12*r_cc2 + 9*r_cc2^2))) + (x[,"n_11"] + x[,"n_22"])*(log10(pmax(1e-6,1 - r_cc2)) + log10(3*p2*(-1 - 6*r_cc2 + 9*r_cc2^2) + 4*(5 - 6*r_cc2 + 9*r_cc2^2)))


logL_cc2 <- 4*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*log(2) + (-x[,"n_00"] - x[,"n_03"] - x[,"n_11"] - x[,"n_12"] - x[,"n_21"] - x[,"n_22"] - x[,"n_30"] - x[,"n_33"])*log(3) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_cc2)) + 2*log(pmax(1e-6,r_cc2))) + (x[,"n_03"] + x[,"n_30"])*(log(4 + 3*p2) + 3*log(pmax(1e-6,r_cc2))) + (x[,"n_00"] + x[,"n_33"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_cc2)) + log(1 - 2*r_cc2 + r_cc2^2)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(log(4 + 3*p2) + log(pmax(1e-6,r_cc2)) + log(1 - 2*r_cc2 + r_cc2^2)) + (x[,"n_12"] + x[,"n_21"])*(log(pmax(1e-6,r_cc2)) + log(3*p2*(2 - 12*r_cc2 + 9*r_cc2^2) + 4*(8 - 12*r_cc2 + 9*r_cc2^2))) + (x[,"n_11"] + x[,"n_22"])*(log(pmax(1e-6,1 - r_cc2)) + log(3*p2*(-1 - 6*r_cc2 + 9*r_cc2^2) + 4*(5 - 6*r_cc2 + 9*r_cc2^2)))


logL_cm1 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(4*log(2) + log(3)) + (n00 + n33)*(log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n03 + n30)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n02 + n31)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(3 - 4*r + 3*r^2)) + (n13 + n20)*(log(pmax(1e-6,r)) + log(6 + 4*(-2 + 3*p2)*r + (6 - 9*p2)*r^2)) + (n01 + n32)*(log(2 - 3*p2) + log(2 - 4*r + 5*r^2 - 3*r^3)) + (n10 + n23)*(log(pmax(1e-6,1 - r)) + log(4 - 4*r + 6*r^2 - p2*(-3 - 6*r + 9*r^2))) + (n11 + n22)*log(2*(4 + 5*r - 14*r^2 + 9*r^3) - 3*p2*(-2 + 8*r - 14*r^2 + 9*r^3)) + (n12 + n21)*log(8 - 8*r + 26*r^2 - 18*r^3 + 3*p2*(-1 + 7*r - 13*r^2 + 9*r^3))
return(L)}
interlogL_cm1 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_cm1,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cm1 <- parallel::mcmapply(interlogL_cm1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_cm1 <- -((x[,"n_03"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_33"])*(-3*log10(2) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_31"])*(-3*log10(2) + log10(7) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_32"])*(-3*log10(2) + log10(7) + log10(2 - 3*p2)) - (x[,"n_12"] + x[,"n_21"])*log10(33/4 + (9*p2)/8) - (x[,"n_11"] + x[,"n_22"])*log10(33/4 + (9*p2)/8) - (x[,"n_10"] + x[,"n_23"])*(-log10(2) + log10(7/2 + (15*p2)/4)) - (x[,"n_13"] + x[,"n_20"])*(-log10(2) + log10(6 + (6 - 9*p2)/4 + 2*(-2 + 3*p2))) + (x[,"n_00"] + x[,"n_33"])*(log10(2 - 3*p2) + 2*log10(pmax(1e-6,1 - r_cm1)) + log10(pmax(1e-6,r_cm1))) + (x[,"n_03"] + x[,"n_30"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cm1)) + 2*log10(pmax(1e-6,r_cm1))) + (x[,"n_02"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_cm1)) + log10(3 - 4*r_cm1 + 3*r_cm1^2)) + (x[,"n_13"] + x[,"n_20"])*(log10(pmax(1e-6,r_cm1)) + log10(6 + 4*(-2 + 3*p2)*r_cm1 + (6 - 9*p2)*r_cm1^2)) + (x[,"n_01"] + x[,"n_32"])*(log10(2 - 3*p2) + log10(2 - 4*r_cm1 + 5*r_cm1^2 - 3*r_cm1^3)) + (x[,"n_10"] + x[,"n_23"])*(log10(pmax(1e-6,1 - r_cm1)) + log10(4 - 4*r_cm1 + 6*r_cm1^2 - p2*(-3 - 6*r_cm1 + 9*r_cm1^2))) + (x[,"n_11"] + x[,"n_22"])*log10(2*(4 + 5*r_cm1 - 14*r_cm1^2 + 9*r_cm1^3) - 3*p2*(-2 + 8*r_cm1 - 14*r_cm1^2 + 9*r_cm1^3)) + (x[,"n_12"] + x[,"n_21"])*log10(8 - 8*r_cm1 + 26*r_cm1^2 - 18*r_cm1^3 + 3*p2*(-1 + 7*r_cm1 - 13*r_cm1^2 + 9*r_cm1^3))


logL_cm1 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(4*log(2) + log(3)) + (x[,"n_00"] + x[,"n_33"])*(log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r_cm1)) + log(pmax(1e-6,r_cm1))) + (x[,"n_03"] + x[,"n_30"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_cm1)) + 2*log(pmax(1e-6,r_cm1))) + (x[,"n_02"] + x[,"n_31"])*(log(2 - 3*p2) + log(pmax(1e-6,r_cm1)) + log(3 - 4*r_cm1 + 3*r_cm1^2)) + (x[,"n_13"] + x[,"n_20"])*(log(pmax(1e-6,r_cm1)) + log(6 + 4*(-2 + 3*p2)*r_cm1 + (6 - 9*p2)*r_cm1^2)) + (x[,"n_01"] + x[,"n_32"])*(log(2 - 3*p2) + log(2 - 4*r_cm1 + 5*r_cm1^2 - 3*r_cm1^3)) + (x[,"n_10"] + x[,"n_23"])*(log(pmax(1e-6,1 - r_cm1)) + log(4 - 4*r_cm1 + 6*r_cm1^2 - p2*(-3 - 6*r_cm1 + 9*r_cm1^2))) + (x[,"n_11"] + x[,"n_22"])*log(2*(4 + 5*r_cm1 - 14*r_cm1^2 + 9*r_cm1^3) - 3*p2*(-2 + 8*r_cm1 - 14*r_cm1^2 + 9*r_cm1^3)) + (x[,"n_12"] + x[,"n_21"])*log(8 - 8*r_cm1 + 26*r_cm1^2 - 18*r_cm1^3 + 3*p2*(-1 + 7*r_cm1 - 13*r_cm1^2 + 9*r_cm1^3))


logL_cm2 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(4*log(2) + log(3)) + (n00 + n33)*(log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n03 + n30)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n13 + n20)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(3 - 4*r + 3*r^2)) + (n02 + n31)*(log(pmax(1e-6,r)) + log(6 + 4*(-2 + 3*p2)*r + (6 - 9*p2)*r^2)) + (n10 + n23)*(log(2 - 3*p2) + log(2 - 4*r + 5*r^2 - 3*r^3)) + (n01 + n32)*(log(pmax(1e-6,1 - r)) + log(4 - 4*r + 6*r^2 - p2*(-3 - 6*r + 9*r^2))) + (n11 + n22)*log(2*(4 + 5*r - 14*r^2 + 9*r^3) - 3*p2*(-2 + 8*r - 14*r^2 + 9*r^3)) + (n12 + n21)*log(8 - 8*r + 26*r^2 - 18*r^3 + 3*p2*(-1 + 7*r - 13*r^2 + 9*r^3))
return(L)}
interlogL_cm2 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_cm2,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cm2 <- parallel::mcmapply(interlogL_cm2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_cm2 <- -((x[,"n_03"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_33"])*(-3*log10(2) + log10(2 - 3*p2)) - (x[,"n_13"] + x[,"n_20"])*(-3*log10(2) + log10(7) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_23"])*(-3*log10(2) + log10(7) + log10(2 - 3*p2)) - (x[,"n_12"] + x[,"n_21"])*log10(33/4 + (9*p2)/8) - (x[,"n_11"] + x[,"n_22"])*log10(33/4 + (9*p2)/8) - (x[,"n_01"] + x[,"n_32"])*(-log10(2) + log10(7/2 + (15*p2)/4)) - (x[,"n_02"] + x[,"n_31"])*(-log10(2) + log10(6 + (6 - 9*p2)/4 + 2*(-2 + 3*p2))) + (x[,"n_00"] + x[,"n_33"])*(log10(2 - 3*p2) + 2*log10(pmax(1e-6,1 - r_cm2)) + log10(pmax(1e-6,r_cm2))) + (x[,"n_03"] + x[,"n_30"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cm2)) + 2*log10(pmax(1e-6,r_cm2))) + (x[,"n_13"] + x[,"n_20"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_cm2)) + log10(3 - 4*r_cm2 + 3*r_cm2^2)) + (x[,"n_02"] + x[,"n_31"])*(log10(pmax(1e-6,r_cm2)) + log10(6 + 4*(-2 + 3*p2)*r_cm2 + (6 - 9*p2)*r_cm2^2)) + (x[,"n_10"] + x[,"n_23"])*(log10(2 - 3*p2) + log10(2 - 4*r_cm2 + 5*r_cm2^2 - 3*r_cm2^3)) + (x[,"n_01"] + x[,"n_32"])*(log10(pmax(1e-6,1 - r_cm2)) + log10(4 - 4*r_cm2 + 6*r_cm2^2 - p2*(-3 - 6*r_cm2 + 9*r_cm2^2))) + (x[,"n_11"] + x[,"n_22"])*log10(2*(4 + 5*r_cm2 - 14*r_cm2^2 + 9*r_cm2^3) - 3*p2*(-2 + 8*r_cm2 - 14*r_cm2^2 + 9*r_cm2^3)) + (x[,"n_12"] + x[,"n_21"])*log10(8 - 8*r_cm2 + 26*r_cm2^2 - 18*r_cm2^3 + 3*p2*(-1 + 7*r_cm2 - 13*r_cm2^2 + 9*r_cm2^3))


logL_cm2 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(4*log(2) + log(3)) + (x[,"n_00"] + x[,"n_33"])*(log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r_cm2)) + log(pmax(1e-6,r_cm2))) + (x[,"n_03"] + x[,"n_30"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_cm2)) + 2*log(pmax(1e-6,r_cm2))) + (x[,"n_13"] + x[,"n_20"])*(log(2 - 3*p2) + log(pmax(1e-6,r_cm2)) + log(3 - 4*r_cm2 + 3*r_cm2^2)) + (x[,"n_02"] + x[,"n_31"])*(log(pmax(1e-6,r_cm2)) + log(6 + 4*(-2 + 3*p2)*r_cm2 + (6 - 9*p2)*r_cm2^2)) + (x[,"n_10"] + x[,"n_23"])*(log(2 - 3*p2) + log(2 - 4*r_cm2 + 5*r_cm2^2 - 3*r_cm2^3)) + (x[,"n_01"] + x[,"n_32"])*(log(pmax(1e-6,1 - r_cm2)) + log(4 - 4*r_cm2 + 6*r_cm2^2 - p2*(-3 - 6*r_cm2 + 9*r_cm2^2))) + (x[,"n_11"] + x[,"n_22"])*log(2*(4 + 5*r_cm2 - 14*r_cm2^2 + 9*r_cm2^3) - 3*p2*(-2 + 8*r_cm2 - 14*r_cm2^2 + 9*r_cm2^3)) + (x[,"n_12"] + x[,"n_21"])*log(8 - 8*r_cm2 + 26*r_cm2^2 - 18*r_cm2^3 + 3*p2*(-1 + 7*r_cm2 - 13*r_cm2^2 + 9*r_cm2^3))


logL_cr1 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(3*log(2) + log(3)) + (n00 + n33)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n03 + n30)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n01 + n10 + n23 + n32)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n02 + n13 + n20 + n31)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + 3*r^2)) + (n12 + n21)*(log(pmax(1e-6,r)) + log(-3*p2*(3 - 14*r + 9*r^2) + 2*(9 - 14*r + 9*r^2))) + (n11 + n22)*(log(pmax(1e-6,1 - r)) + log(8 - 8*r + 18*r^2 - 3*p2*(-2 - 4*r + 9*r^2)))
return(L)}
interlogL_cr1 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_cr1,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr1 <- parallel::mcmapply(interlogL_cr1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_cr1 <- -((x[,"n_03"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_33"])*(-3*log10(2) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(-3*log10(2) + log10(3) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(-3*log10(2) + log10(3) + log10(2 - 3*p2)) - (x[,"n_12"] + x[,"n_21"])*(-log10(2) + log10(17/2 + (21*p2)/4)) - (x[,"n_11"] + x[,"n_22"])*(-log10(2) + log10(17/2 + (21*p2)/4)) + (x[,"n_00"] + x[,"n_33"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cr1)) + 2*log10(pmax(1e-6,r_cr1))) + (x[,"n_03"] + x[,"n_30"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_cr1)) + log10(1 - 2*r_cr1 + r_cr1^2)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_cr1)) + log10(2 - 4*r_cr1 + 3*r_cr1^2)) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cr1)) + log10(1 - 2*r_cr1 + 3*r_cr1^2)) + (x[,"n_12"] + x[,"n_21"])*(log10(pmax(1e-6,r_cr1)) + log10(-3*p2*(3 - 14*r_cr1 + 9*r_cr1^2) + 2*(9 - 14*r_cr1 + 9*r_cr1^2))) + (x[,"n_11"] + x[,"n_22"])*(log10(pmax(1e-6,1 - r_cr1)) + log10(8 - 8*r_cr1 + 18*r_cr1^2 - 3*p2*(-2 - 4*r_cr1 + 9*r_cr1^2)))


logL_cr1 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(3*log(2) + log(3)) + (x[,"n_00"] + x[,"n_33"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_cr1)) + 2*log(pmax(1e-6,r_cr1))) + (x[,"n_03"] + x[,"n_30"])*(log(2 - 3*p2) + log(pmax(1e-6,r_cr1)) + log(1 - 2*r_cr1 + r_cr1^2)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(log(2 - 3*p2) + log(pmax(1e-6,r_cr1)) + log(2 - 4*r_cr1 + 3*r_cr1^2)) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_cr1)) + log(1 - 2*r_cr1 + 3*r_cr1^2)) + (x[,"n_12"] + x[,"n_21"])*(log(pmax(1e-6,r_cr1)) + log(-3*p2*(3 - 14*r_cr1 + 9*r_cr1^2) + 2*(9 - 14*r_cr1 + 9*r_cr1^2))) + (x[,"n_11"] + x[,"n_22"])*(log(pmax(1e-6,1 - r_cr1)) + log(8 - 8*r_cr1 + 18*r_cr1^2 - 3*p2*(-2 - 4*r_cr1 + 9*r_cr1^2)))


logL_cr2 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(4*log(2) + log(3)) + (n00 + n33)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n03 + n30)*(log(4 + 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n01 + n10 + n23 + n32)*(log(4 + 3*p2) + log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n02 + n13 + n20 + n31)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + 3*r^2)) + (n12 + n21)*(log(pmax(1e-6,r)) + log(3*p2*(3 - 14*r + 9*r^2) + 4*(9 - 14*r + 9*r^2))) + (n11 + n22)*(log(pmax(1e-6,1 - r)) + log(3*p2*(-2 - 4*r + 9*r^2) + 4*(4 - 4*r + 9*r^2)))
return(L)}
interlogL_cr2 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_cr2,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr2 <- parallel::mcmapply(interlogL_cr2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_cr2 <- -((x[,"n_12"] + x[,"n_21"])*(-log10(2) + log10(17 - (21*p2)/4))) - (x[,"n_11"] + x[,"n_22"])*(-log10(2) + log10(17 - (21*p2)/4)) - (x[,"n_03"] + x[,"n_30"])*(-3*log10(2) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_33"])*(-3*log10(2) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(-3*log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(-3*log10(2) + log10(3) + log10(4 + 3*p2)) + (x[,"n_00"] + x[,"n_33"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cr2)) + 2*log10(pmax(1e-6,r_cr2))) + (x[,"n_03"] + x[,"n_30"])*(log10(4 + 3*p2) + log10(pmax(1e-6,r_cr2)) + log10(1 - 2*r_cr2 + r_cr2^2)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(log10(4 + 3*p2) + log10(pmax(1e-6,r_cr2)) + log10(2 - 4*r_cr2 + 3*r_cr2^2)) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cr2)) + log10(1 - 2*r_cr2 + 3*r_cr2^2)) + (x[,"n_12"] + x[,"n_21"])*(log10(pmax(1e-6,r_cr2)) + log10(3*p2*(3 - 14*r_cr2 + 9*r_cr2^2) + 4*(9 - 14*r_cr2 + 9*r_cr2^2))) + (x[,"n_11"] + x[,"n_22"])*(log10(pmax(1e-6,1 - r_cr2)) + log10(3*p2*(-2 - 4*r_cr2 + 9*r_cr2^2) + 4*(4 - 4*r_cr2 + 9*r_cr2^2)))


logL_cr2 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(4*log(2) + log(3)) + (x[,"n_00"] + x[,"n_33"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_cr2)) + 2*log(pmax(1e-6,r_cr2))) + (x[,"n_03"] + x[,"n_30"])*(log(4 + 3*p2) + log(pmax(1e-6,r_cr2)) + log(1 - 2*r_cr2 + r_cr2^2)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(log(4 + 3*p2) + log(pmax(1e-6,r_cr2)) + log(2 - 4*r_cr2 + 3*r_cr2^2)) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_cr2)) + log(1 - 2*r_cr2 + 3*r_cr2^2)) + (x[,"n_12"] + x[,"n_21"])*(log(pmax(1e-6,r_cr2)) + log(3*p2*(3 - 14*r_cr2 + 9*r_cr2^2) + 4*(9 - 14*r_cr2 + 9*r_cr2^2))) + (x[,"n_11"] + x[,"n_22"])*(log(pmax(1e-6,1 - r_cr2)) + log(3*p2*(-2 - 4*r_cr2 + 9*r_cr2^2) + 4*(4 - 4*r_cr2 + 9*r_cr2^2)))


logL_rc1 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(4*log(2) + 2*log(3)) + (n03 + n30)*(log(2 - 3*p2) + 2*log(pmax(1e-6,r)) + log(4 + 3*p1 - 2*r - 6*p1*r)) + (n00 + n33)*(log(2 - 3*p2) + log(2 - 3*p1 + 2*r + 6*p1*r) + log(1 - 2*r + r^2)) + (n02 + n13 + n20 + n31)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(8 - 10*r + 6*r^2 + 3*p1*(2 - 7*r + 6*r^2))) + (n01 + n10 + n23 + n32)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(4 - 2*r + 6*r^2 + 3*p1*(1 - 5*r + 6*r^2))) + (n11 + n22)*log(-6*p2*(-1 + r - 13*r^2 + 9*r^3) + 4*(5 + 7*r - 13*r^2 + 9*r^3) - 3*p1*(-1 + 2*r)*(3*p2*(-1 - 10*r + 9*r^2) - 2*(5 - 10*r + 9*r^2))) + (n12 + n21)*log(4*(8 - 8*r + 14*r^2 - 9*r^3) + 6*p2*(4 + 2*r - 14*r^2 + 9*r^3) + 3*p1*(-1 + 2*r)*(3*p2*(-2 - 8*r + 9*r^2) - 2*(4 - 8*r + 9*r^2)))
return(L)}
interlogL_rc1 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rc1,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc1 <- parallel::mcmapply(interlogL_rc1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rc1 <- -((x[,"n_03"] + x[,"n_30"])*(-2*log10(2) + log10(3) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_33"])*(-2*log10(2) + log10(3) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(-2*log10(2) + 2*log10(3) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(-2*log10(2) + 2*log10(3) + log10(2 - 3*p2)) - (x[,"n_12"] + x[,"n_21"])*log10(51/2 + (63*p2)/4) - (x[,"n_11"] + x[,"n_22"])*log10(51/2 + (63*p2)/4) + (x[,"n_03"] + x[,"n_30"])*(log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_rc1)) + log10(4 + 3*p1 - 2*r_rc1 - 6*p1*r_rc1)) + (x[,"n_00"] + x[,"n_33"])*(log10(2 - 3*p2) + log10(2 - 3*p1 + 2*r_rc1 + 6*p1*r_rc1) + log10(1 - 2*r_rc1 + r_rc1^2)) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_rc1)) + log10(8 - 10*r_rc1 + 6*r_rc1^2 + 3*p1*(2 - 7*r_rc1 + 6*r_rc1^2))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rc1)) + log10(4 - 2*r_rc1 + 6*r_rc1^2 + 3*p1*(1 - 5*r_rc1 + 6*r_rc1^2))) + (x[,"n_11"] + x[,"n_22"])*log10(-6*p2*(-1 + r_rc1 - 13*r_rc1^2 + 9*r_rc1^3) + 4*(5 + 7*r_rc1 - 13*r_rc1^2 + 9*r_rc1^3) - 3*p1*(-1 + 2*r_rc1)*(3*p2*(-1 - 10*r_rc1 + 9*r_rc1^2) - 2*(5 - 10*r_rc1 + 9*r_rc1^2))) + (x[,"n_12"] + x[,"n_21"])*log10(4*(8 - 8*r_rc1 + 14*r_rc1^2 - 9*r_rc1^3) + 6*p2*(4 + 2*r_rc1 - 14*r_rc1^2 + 9*r_rc1^3) + 3*p1*(-1 + 2*r_rc1)*(3*p2*(-2 - 8*r_rc1 + 9*r_rc1^2) - 2*(4 - 8*r_rc1 + 9*r_rc1^2)))


logL_rc1 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(4*log(2) + 2*log(3)) + (x[,"n_03"] + x[,"n_30"])*(log(2 - 3*p2) + 2*log(pmax(1e-6,r_rc1)) + log(4 + 3*p1 - 2*r_rc1 - 6*p1*r_rc1)) + (x[,"n_00"] + x[,"n_33"])*(log(2 - 3*p2) + log(2 - 3*p1 + 2*r_rc1 + 6*p1*r_rc1) + log(1 - 2*r_rc1 + r_rc1^2)) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log(2 - 3*p2) + log(pmax(1e-6,r_rc1)) + log(8 - 10*r_rc1 + 6*r_rc1^2 + 3*p1*(2 - 7*r_rc1 + 6*r_rc1^2))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_rc1)) + log(4 - 2*r_rc1 + 6*r_rc1^2 + 3*p1*(1 - 5*r_rc1 + 6*r_rc1^2))) + (x[,"n_11"] + x[,"n_22"])*log(-6*p2*(-1 + r_rc1 - 13*r_rc1^2 + 9*r_rc1^3) + 4*(5 + 7*r_rc1 - 13*r_rc1^2 + 9*r_rc1^3) - 3*p1*(-1 + 2*r_rc1)*(3*p2*(-1 - 10*r_rc1 + 9*r_rc1^2) - 2*(5 - 10*r_rc1 + 9*r_rc1^2))) + (x[,"n_12"] + x[,"n_21"])*log(4*(8 - 8*r_rc1 + 14*r_rc1^2 - 9*r_rc1^3) + 6*p2*(4 + 2*r_rc1 - 14*r_rc1^2 + 9*r_rc1^3) + 3*p1*(-1 + 2*r_rc1)*(3*p2*(-2 - 8*r_rc1 + 9*r_rc1^2) - 2*(4 - 8*r_rc1 + 9*r_rc1^2)))


logL_rc2 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(5*log(2) + 2*log(3)) + (n03 + n30)*(log(4 + 3*p2) + log(4 + p1*(3 - 6*r) - 2*r) + 2*log(pmax(1e-6,r))) + (n00 + n33)*(log(4 + 3*p2) + log(2 - 3*p1 + 2*r + 6*p1*r) + log(1 - 2*r + r^2)) + (n02 + n13 + n20 + n31)*(log(4 + 3*p2) + log(pmax(1e-6,r)) + log(8 - 10*r + 6*r^2 + 3*p1*(2 - 7*r + 6*r^2))) + (n01 + n10 + n23 + n32)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(4 - 2*r + 6*r^2 + 3*p1*(1 - 5*r + 6*r^2))) + (n11 + n22)*log(6*p2*(-1 + r - 13*r^2 + 9*r^3) + 8*(5 + 7*r - 13*r^2 + 9*r^3) + 3*p1*(-1 + 2*r)*(3*p2*(-1 - 10*r + 9*r^2) + 4*(5 - 10*r + 9*r^2))) + (n12 + n21)*log(-3*p1*(-1 + 2*r)*(3*p2*(-2 - 8*r + 9*r^2) + 4*(4 - 8*r + 9*r^2)) - 2*(3*p2*(4 + 2*r - 14*r^2 + 9*r^3) + 4*(-8 + 8*r - 14*r^2 + 9*r^3)))
return(L)}
interlogL_rc2 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rc2,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc2 <- parallel::mcmapply(interlogL_rc2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rc2 <- -((x[,"n_11"] + x[,"n_22"])*log10(51 - (63*p2)/4)) - (x[,"n_03"] + x[,"n_30"])*(-2*log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_33"])*(-2*log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(-2*log10(2) + 2*log10(3) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(-2*log10(2) + 2*log10(3) + log10(4 + 3*p2)) - (x[,"n_12"] + x[,"n_21"])*(log10(2) + log10(51/2 - (63*p2)/8)) + (x[,"n_03"] + x[,"n_30"])*(log10(4 + 3*p2) + log10(4 + p1*(3 - 6*r_rc2) - 2*r_rc2) + 2*log10(pmax(1e-6,r_rc2))) + (x[,"n_00"] + x[,"n_33"])*(log10(4 + 3*p2) + log10(2 - 3*p1 + 2*r_rc2 + 6*p1*r_rc2) + log10(1 - 2*r_rc2 + r_rc2^2)) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log10(4 + 3*p2) + log10(pmax(1e-6,r_rc2)) + log10(8 - 10*r_rc2 + 6*r_rc2^2 + 3*p1*(2 - 7*r_rc2 + 6*r_rc2^2))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rc2)) + log10(4 - 2*r_rc2 + 6*r_rc2^2 + 3*p1*(1 - 5*r_rc2 + 6*r_rc2^2))) + (x[,"n_11"] + x[,"n_22"])*log10(6*p2*(-1 + r_rc2 - 13*r_rc2^2 + 9*r_rc2^3) + 8*(5 + 7*r_rc2 - 13*r_rc2^2 + 9*r_rc2^3) + 3*p1*(-1 + 2*r_rc2)*(3*p2*(-1 - 10*r_rc2 + 9*r_rc2^2) + 4*(5 - 10*r_rc2 + 9*r_rc2^2))) + (x[,"n_12"] + x[,"n_21"])*log10(-3*p1*(-1 + 2*r_rc2)*(3*p2*(-2 - 8*r_rc2 + 9*r_rc2^2) + 4*(4 - 8*r_rc2 + 9*r_rc2^2)) - 2*(3*p2*(4 + 2*r_rc2 - 14*r_rc2^2 + 9*r_rc2^3) + 4*(-8 + 8*r_rc2 - 14*r_rc2^2 + 9*r_rc2^3)))


logL_rc2 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(5*log(2) + 2*log(3)) + (x[,"n_03"] + x[,"n_30"])*(log(4 + 3*p2) + log(4 + p1*(3 - 6*r_rc2) - 2*r_rc2) + 2*log(pmax(1e-6,r_rc2))) + (x[,"n_00"] + x[,"n_33"])*(log(4 + 3*p2) + log(2 - 3*p1 + 2*r_rc2 + 6*p1*r_rc2) + log(1 - 2*r_rc2 + r_rc2^2)) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log(4 + 3*p2) + log(pmax(1e-6,r_rc2)) + log(8 - 10*r_rc2 + 6*r_rc2^2 + 3*p1*(2 - 7*r_rc2 + 6*r_rc2^2))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_rc2)) + log(4 - 2*r_rc2 + 6*r_rc2^2 + 3*p1*(1 - 5*r_rc2 + 6*r_rc2^2))) + (x[,"n_11"] + x[,"n_22"])*log(6*p2*(-1 + r_rc2 - 13*r_rc2^2 + 9*r_rc2^3) + 8*(5 + 7*r_rc2 - 13*r_rc2^2 + 9*r_rc2^3) + 3*p1*(-1 + 2*r_rc2)*(3*p2*(-1 - 10*r_rc2 + 9*r_rc2^2) + 4*(5 - 10*r_rc2 + 9*r_rc2^2))) + (x[,"n_12"] + x[,"n_21"])*log(-3*p1*(-1 + 2*r_rc2)*(3*p2*(-2 - 8*r_rc2 + 9*r_rc2^2) + 4*(4 - 8*r_rc2 + 9*r_rc2^2)) - 2*(3*p2*(4 + 2*r_rc2 - 14*r_rc2^2 + 9*r_rc2^3) + 4*(-8 + 8*r_rc2 - 14*r_rc2^2 + 9*r_rc2^3)))


logL_rc3 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(5*log(2) + 2*log(3)) + (n00 + n33)*(log(2 - 3*p2) + log(4 + 3*p1 + 4*r - 6*p1*r) + log(1 - 2*r + r^2)) + (n03 + n30)*(log(2 - 3*p2) + 2*log(pmax(1e-6,r)) + log(8 - 4*r + p1*(-3 + 6*r))) + (n02 + n13 + n20 + n31)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(4*(4 - 5*r + 3*r^2) - 3*p1*(2 - 7*r + 6*r^2))) + (n01 + n10 + n23 + n32)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(4*(2 - r + 3*r^2) - 3*p1*(1 - 5*r + 6*r^2))) + (n12 + n21)*log(-3*p1*(-1 + 2*r)*(3*p2*(-2 - 8*r + 9*r^2) - 2*(4 - 8*r + 9*r^2)) + 4*(2*(8 - 8*r + 14*r^2 - 9*r^3) + 3*p2*(4 + 2*r - 14*r^2 + 9*r^3))) + (n11 + n22)*log(3*p1*(-1 + 2*r)*(3*p2*(-1 - 10*r + 9*r^2) - 2*(5 - 10*r + 9*r^2)) + 4*(-3*p2*(-1 + r - 13*r^2 + 9*r^3) + 2*(5 + 7*r - 13*r^2 + 9*r^3)))
return(L)}
interlogL_rc3 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rc3,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc3 <- parallel::mcmapply(interlogL_rc3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rc3 <- -((x[,"n_03"] + x[,"n_30"])*(-log10(2) + log10(3) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(-log10(2) + 2*log10(3) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(-log10(2) + 2*log10(3) + log10(2 - 3*p2)) - (x[,"n_12"] + x[,"n_21"])*(2*log10(2) + log10(51/4 + (63*p2)/8)) - (x[,"n_11"] + x[,"n_22"])*(2*log10(2) + log10(51/4 + (63*p2)/8)) + (x[,"n_00"] + x[,"n_33"])*(log10(2 - 3*p2) + log10(4 + 3*p1 + 4*r_rc3 - 6*p1*r_rc3) + log10(1 - 2*r_rc3 + r_rc3^2)) + (x[,"n_03"] + x[,"n_30"])*(log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_rc3)) + log10(8 - 4*r_rc3 + p1*(-3 + 6*r_rc3))) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_rc3)) + log10(4*(4 - 5*r_rc3 + 3*r_rc3^2) - 3*p1*(2 - 7*r_rc3 + 6*r_rc3^2))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rc3)) + log10(4*(2 - r_rc3 + 3*r_rc3^2) - 3*p1*(1 - 5*r_rc3 + 6*r_rc3^2))) + (x[,"n_12"] + x[,"n_21"])*log10(-3*p1*(-1 + 2*r_rc3)*(3*p2*(-2 - 8*r_rc3 + 9*r_rc3^2) - 2*(4 - 8*r_rc3 + 9*r_rc3^2)) + 4*(2*(8 - 8*r_rc3 + 14*r_rc3^2 - 9*r_rc3^3) + 3*p2*(4 + 2*r_rc3 - 14*r_rc3^2 + 9*r_rc3^3))) + (x[,"n_11"] + x[,"n_22"])*log10(3*p1*(-1 + 2*r_rc3)*(3*p2*(-1 - 10*r_rc3 + 9*r_rc3^2) - 2*(5 - 10*r_rc3 + 9*r_rc3^2)) + 4*(-3*p2*(-1 + r_rc3 - 13*r_rc3^2 + 9*r_rc3^3) + 2*(5 + 7*r_rc3 - 13*r_rc3^2 + 9*r_rc3^3)))


logL_rc3 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(5*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_33"])*(log(2 - 3*p2) + log(4 + 3*p1 + 4*r_rc3 - 6*p1*r_rc3) + log(1 - 2*r_rc3 + r_rc3^2)) + (x[,"n_03"] + x[,"n_30"])*(log(2 - 3*p2) + 2*log(pmax(1e-6,r_rc3)) + log(8 - 4*r_rc3 + p1*(-3 + 6*r_rc3))) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log(2 - 3*p2) + log(pmax(1e-6,r_rc3)) + log(4*(4 - 5*r_rc3 + 3*r_rc3^2) - 3*p1*(2 - 7*r_rc3 + 6*r_rc3^2))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_rc3)) + log(4*(2 - r_rc3 + 3*r_rc3^2) - 3*p1*(1 - 5*r_rc3 + 6*r_rc3^2))) + (x[,"n_12"] + x[,"n_21"])*log(-3*p1*(-1 + 2*r_rc3)*(3*p2*(-2 - 8*r_rc3 + 9*r_rc3^2) - 2*(4 - 8*r_rc3 + 9*r_rc3^2)) + 4*(2*(8 - 8*r_rc3 + 14*r_rc3^2 - 9*r_rc3^3) + 3*p2*(4 + 2*r_rc3 - 14*r_rc3^2 + 9*r_rc3^3))) + (x[,"n_11"] + x[,"n_22"])*log(3*p1*(-1 + 2*r_rc3)*(3*p2*(-1 - 10*r_rc3 + 9*r_rc3^2) - 2*(5 - 10*r_rc3 + 9*r_rc3^2)) + 4*(-3*p2*(-1 + r_rc3 - 13*r_rc3^2 + 9*r_rc3^3) + 2*(5 + 7*r_rc3 - 13*r_rc3^2 + 9*r_rc3^3)))


logL_rc4 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(6*log(2) + 2*log(3)) + (n00 + n33)*(log(4 + 3*p2) + log(4 + 3*p1 + 4*r - 6*p1*r) + log(1 - 2*r + r^2)) + (n03 + n30)*(log(4 + 3*p2) + 2*log(pmax(1e-6,r)) + log(8 - 4*r + p1*(-3 + 6*r))) + (n02 + n13 + n20 + n31)*(log(4 + 3*p2) + log(pmax(1e-6,r)) + log(4*(4 - 5*r + 3*r^2) - 3*p1*(2 - 7*r + 6*r^2))) + (n01 + n10 + n23 + n32)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(4*(2 - r + 3*r^2) - 3*p1*(1 - 5*r + 6*r^2))) + (n12 + n21)*log(3*p1*(-1 + 2*r)*(3*p2*(-2 - 8*r + 9*r^2) + 4*(4 - 8*r + 9*r^2)) - 4*(3*p2*(4 + 2*r - 14*r^2 + 9*r^3) + 4*(-8 + 8*r - 14*r^2 + 9*r^3))) + (n11 + n22)*log(-3*p1*(-1 + 2*r)*(3*p2*(-1 - 10*r + 9*r^2) + 4*(5 - 10*r + 9*r^2)) + 4*(3*p2*(-1 + r - 13*r^2 + 9*r^3) + 4*(5 + 7*r - 13*r^2 + 9*r^3)))
return(L)}
interlogL_rc4 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rc4,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc4 <- parallel::mcmapply(interlogL_rc4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rc4 <- -((x[,"n_11"] + x[,"n_22"])*(2*log10(2) + log10(51/2 - (63*p2)/8))) - (x[,"n_03"] + x[,"n_30"])*(-log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(-log10(2) + 2*log10(3) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(-log10(2) + 2*log10(3) + log10(4 + 3*p2)) - (x[,"n_12"] + x[,"n_21"])*(2*log10(2) + log10(51/2 - (63*p2)/8)) + (x[,"n_00"] + x[,"n_33"])*(log10(4 + 3*p2) + log10(4 + 3*p1 + 4*r_rc4 - 6*p1*r_rc4) + log10(1 - 2*r_rc4 + r_rc4^2)) + (x[,"n_03"] + x[,"n_30"])*(log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_rc4)) + log10(8 - 4*r_rc4 + p1*(-3 + 6*r_rc4))) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log10(4 + 3*p2) + log10(pmax(1e-6,r_rc4)) + log10(4*(4 - 5*r_rc4 + 3*r_rc4^2) - 3*p1*(2 - 7*r_rc4 + 6*r_rc4^2))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rc4)) + log10(4*(2 - r_rc4 + 3*r_rc4^2) - 3*p1*(1 - 5*r_rc4 + 6*r_rc4^2))) + (x[,"n_12"] + x[,"n_21"])*log10(3*p1*(-1 + 2*r_rc4)*(3*p2*(-2 - 8*r_rc4 + 9*r_rc4^2) + 4*(4 - 8*r_rc4 + 9*r_rc4^2)) - 4*(3*p2*(4 + 2*r_rc4 - 14*r_rc4^2 + 9*r_rc4^3) + 4*(-8 + 8*r_rc4 - 14*r_rc4^2 + 9*r_rc4^3))) + (x[,"n_11"] + x[,"n_22"])*log10(-3*p1*(-1 + 2*r_rc4)*(3*p2*(-1 - 10*r_rc4 + 9*r_rc4^2) + 4*(5 - 10*r_rc4 + 9*r_rc4^2)) + 4*(3*p2*(-1 + r_rc4 - 13*r_rc4^2 + 9*r_rc4^3) + 4*(5 + 7*r_rc4 - 13*r_rc4^2 + 9*r_rc4^3)))


logL_rc4 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(6*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_33"])*(log(4 + 3*p2) + log(4 + 3*p1 + 4*r_rc4 - 6*p1*r_rc4) + log(1 - 2*r_rc4 + r_rc4^2)) + (x[,"n_03"] + x[,"n_30"])*(log(4 + 3*p2) + 2*log(pmax(1e-6,r_rc4)) + log(8 - 4*r_rc4 + p1*(-3 + 6*r_rc4))) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log(4 + 3*p2) + log(pmax(1e-6,r_rc4)) + log(4*(4 - 5*r_rc4 + 3*r_rc4^2) - 3*p1*(2 - 7*r_rc4 + 6*r_rc4^2))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_rc4)) + log(4*(2 - r_rc4 + 3*r_rc4^2) - 3*p1*(1 - 5*r_rc4 + 6*r_rc4^2))) + (x[,"n_12"] + x[,"n_21"])*log(3*p1*(-1 + 2*r_rc4)*(3*p2*(-2 - 8*r_rc4 + 9*r_rc4^2) + 4*(4 - 8*r_rc4 + 9*r_rc4^2)) - 4*(3*p2*(4 + 2*r_rc4 - 14*r_rc4^2 + 9*r_rc4^3) + 4*(-8 + 8*r_rc4 - 14*r_rc4^2 + 9*r_rc4^3))) + (x[,"n_11"] + x[,"n_22"])*log(-3*p1*(-1 + 2*r_rc4)*(3*p2*(-1 - 10*r_rc4 + 9*r_rc4^2) + 4*(5 - 10*r_rc4 + 9*r_rc4^2)) + 4*(3*p2*(-1 + r_rc4 - 13*r_rc4^2 + 9*r_rc4^3) + 4*(5 + 7*r_rc4 - 13*r_rc4^2 + 9*r_rc4^3)))


logL_rm1 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- 5*(-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*log(2) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n10 - n11 - n12 - 2*n13 - 2*n20 - n21 - n22 - 2*n23 - 2*n30 - 2*n31 - 2*n32 - 2*n33)*log(3) + (n03 + n30)*(log(2 - 3*p2) + log(4 + p1*(3 - 6*r) - 2*r) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n33)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(2 - 3*p1 + 2*r + 6*p1*r)) + (n02 + n31)*(log(2 - 3*p2) + log(-2*(-4 + 5*r - 6*r^2 + 3*r^3) - 3*p1*(-2 + 7*r - 9*r^2 + 6*r^3))) + (n01 + n32)*(log(2 - 3*p2) + log(4 + 4*r - 6*r^2 + 6*r^3 + 3*p1*(-2 + 7*r - 9*r^2 + 6*r^3))) + (n13 + n20)*log((-2 + 3*p2)*(-1 + r)*r*(2 - 3*p1 + 2*r + 6*p1*r) + (4 + p1*(3 - 6*r) - 2*r)*(p2*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2))) + (n10 + n23)*log((-2 + 3*p2)*(4 + p1*(3 - 6*r) - 2*r)*(-1 + r)*r + (2 - 3*p1 + 2*r + 6*p1*r)*(p2*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2))) + (n11 + n22)*log(2*(8 + (-2 + 6*p2)*r + (8 - 12*p2)*r^2 + (-6 + 9*p2)*r^3) + 9*p1*(-1 + 2*r)*(-2*(-1 + r)*r + p2*(1 - 3*r + 3*r^2))) + (n12 + n21)*log(2*(8 + (4 + 9*p1)*r - (10 + 27*p1)*r^2 + 6*(1 + 3*p1)*r^3) - 3*p2*(-2 + 6*r - 10*r^2 + 6*r^3 + 3*p1*(-1 + 5*r - 9*r^2 + 6*r^3)))
return(L)}
interlogL_rm1 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rm1,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rm1 <- parallel::mcmapply(interlogL_rm1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rm1 <- -((x[,"n_03"] + x[,"n_30"])*(-2*log10(2) + log10(3) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_33"])*(-2*log10(2) + log10(3) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_31"])*(-2*log10(2) + log10(3) + log10(7) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_32"])*(-2*log10(2) + log10(3) + log10(7) + log10(2 - 3*p2)) - (x[,"n_12"] + x[,"n_21"])*log10(2*(8 + (-10 - 27*p1)/4 + (3*(1 + 3*p1))/4 + (4 + 9*p1)/2) + (9*p2)/4) - (x[,"n_13"] + x[,"n_20"])*log10((-3*(-2 + 3*p2))/4 + 3*(3 + (9*p2)/2)) - (x[,"n_10"] + x[,"n_23"])*log10((-3*(-2 + 3*p2))/4 + 3*(3 + (9*p2)/2)) - (x[,"n_11"] + x[,"n_22"])*(log10(2) + log10(8 + (8 - 12*p2)/4 + (-2 + 6*p2)/2 + (-6 + 9*p2)/8)) + (x[,"n_03"] + x[,"n_30"])*(log10(2 - 3*p2) + log10(4 + p1*(3 - 6*r_rm1) - 2*r_rm1) + log10(pmax(1e-6,1 - r_rm1)) + log10(pmax(1e-6,r_rm1))) + (x[,"n_00"] + x[,"n_33"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rm1)) + log10(pmax(1e-6,r_rm1)) + log10(2 - 3*p1 + 2*r_rm1 + 6*p1*r_rm1)) + (x[,"n_02"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(-2*(-4 + 5*r_rm1 - 6*r_rm1^2 + 3*r_rm1^3) - 3*p1*(-2 + 7*r_rm1 - 9*r_rm1^2 + 6*r_rm1^3))) + (x[,"n_01"] + x[,"n_32"])*(log10(2 - 3*p2) + log10(4 + 4*r_rm1 - 6*r_rm1^2 + 6*r_rm1^3 + 3*p1*(-2 + 7*r_rm1 - 9*r_rm1^2 + 6*r_rm1^3))) + (x[,"n_13"] + x[,"n_20"])*log10((-2 + 3*p2)*(-1 + r_rm1)*r_rm1*(2 - 3*p1 + 2*r_rm1 + 6*p1*r_rm1) + (4 + p1*(3 - 6*r_rm1) - 2*r_rm1)*(p2*(3 + 6*r_rm1 - 6*r_rm1^2) + 4*(1 - r_rm1 + r_rm1^2))) + (x[,"n_10"] + x[,"n_23"])*log10((-2 + 3*p2)*(4 + p1*(3 - 6*r_rm1) - 2*r_rm1)*(-1 + r_rm1)*r_rm1 + (2 - 3*p1 + 2*r_rm1 + 6*p1*r_rm1)*(p2*(3 + 6*r_rm1 - 6*r_rm1^2) + 4*(1 - r_rm1 + r_rm1^2))) + (x[,"n_11"] + x[,"n_22"])*log10(2*(8 + (-2 + 6*p2)*r_rm1 + (8 - 12*p2)*r_rm1^2 + (-6 + 9*p2)*r_rm1^3) + 9*p1*(-1 + 2*r_rm1)*(-2*(-1 + r_rm1)*r_rm1 + p2*(1 - 3*r_rm1 + 3*r_rm1^2))) + (x[,"n_12"] + x[,"n_21"])*log10(2*(8 + (4 + 9*p1)*r_rm1 - (10 + 27*p1)*r_rm1^2 + 6*(1 + 3*p1)*r_rm1^3) - 3*p2*(-2 + 6*r_rm1 - 10*r_rm1^2 + 6*r_rm1^3 + 3*p1*(-1 + 5*r_rm1 - 9*r_rm1^2 + 6*r_rm1^3)))


logL_rm1 <- 5*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"])*log(3) + (x[,"n_03"] + x[,"n_30"])*(log(2 - 3*p2) + log(4 + p1*(3 - 6*r_rm1) - 2*r_rm1) + log(pmax(1e-6,1 - r_rm1)) + log(pmax(1e-6,r_rm1))) + (x[,"n_00"] + x[,"n_33"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_rm1)) + log(pmax(1e-6,r_rm1)) + log(2 - 3*p1 + 2*r_rm1 + 6*p1*r_rm1)) + (x[,"n_02"] + x[,"n_31"])*(log(2 - 3*p2) + log(-2*(-4 + 5*r_rm1 - 6*r_rm1^2 + 3*r_rm1^3) - 3*p1*(-2 + 7*r_rm1 - 9*r_rm1^2 + 6*r_rm1^3))) + (x[,"n_01"] + x[,"n_32"])*(log(2 - 3*p2) + log(4 + 4*r_rm1 - 6*r_rm1^2 + 6*r_rm1^3 + 3*p1*(-2 + 7*r_rm1 - 9*r_rm1^2 + 6*r_rm1^3))) + (x[,"n_13"] + x[,"n_20"])*log((-2 + 3*p2)*(-1 + r_rm1)*r_rm1*(2 - 3*p1 + 2*r_rm1 + 6*p1*r_rm1) + (4 + p1*(3 - 6*r_rm1) - 2*r_rm1)*(p2*(3 + 6*r_rm1 - 6*r_rm1^2) + 4*(1 - r_rm1 + r_rm1^2))) + (x[,"n_10"] + x[,"n_23"])*log((-2 + 3*p2)*(4 + p1*(3 - 6*r_rm1) - 2*r_rm1)*(-1 + r_rm1)*r_rm1 + (2 - 3*p1 + 2*r_rm1 + 6*p1*r_rm1)*(p2*(3 + 6*r_rm1 - 6*r_rm1^2) + 4*(1 - r_rm1 + r_rm1^2))) + (x[,"n_11"] + x[,"n_22"])*log(2*(8 + (-2 + 6*p2)*r_rm1 + (8 - 12*p2)*r_rm1^2 + (-6 + 9*p2)*r_rm1^3) + 9*p1*(-1 + 2*r_rm1)*(-2*(-1 + r_rm1)*r_rm1 + p2*(1 - 3*r_rm1 + 3*r_rm1^2))) + (x[,"n_12"] + x[,"n_21"])*log(2*(8 + (4 + 9*p1)*r_rm1 - (10 + 27*p1)*r_rm1^2 + 6*(1 + 3*p1)*r_rm1^3) - 3*p2*(-2 + 6*r_rm1 - 10*r_rm1^2 + 6*r_rm1^3 + 3*p1*(-1 + 5*r_rm1 - 9*r_rm1^2 + 6*r_rm1^3)))


logL_rm2 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- 5*(-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*log(2) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n10 - n11 - n12 - 2*n13 - 2*n20 - n21 - n22 - 2*n23 - 2*n30 - 2*n31 - 2*n32 - 2*n33)*log(3) + (n03 + n30)*(log(2 - 3*p2) + log(4 + p1*(3 - 6*r) - 2*r) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n33)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(2 - 3*p1 + 2*r + 6*p1*r)) + (n02 + n31)*(log(2 - 3*p2) + log(-2*(-4 + 5*r - 6*r^2 + 3*r^3) - 3*p1*(-2 + 7*r - 9*r^2 + 6*r^3))) + (n01 + n32)*(log(2 - 3*p2) + log(4 + 4*r - 6*r^2 + 6*r^3 + 3*p1*(-2 + 7*r - 9*r^2 + 6*r^3))) + (n13 + n20)*log((-2 + 3*p2)*(-1 + r)*r*(2 - 3*p1 + 2*r + 6*p1*r) + (4 + p1*(3 - 6*r) - 2*r)*(p2*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2))) + (n10 + n23)*log((-2 + 3*p2)*(4 + p1*(3 - 6*r) - 2*r)*(-1 + r)*r + (2 - 3*p1 + 2*r + 6*p1*r)*(p2*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2))) + (n11 + n22)*log(2*(8 + (-2 + 6*p2)*r + (8 - 12*p2)*r^2 + (-6 + 9*p2)*r^3) + 9*p1*(-1 + 2*r)*(-2*(-1 + r)*r + p2*(1 - 3*r + 3*r^2))) + (n12 + n21)*log(2*(8 + (4 + 9*p1)*r - (10 + 27*p1)*r^2 + 6*(1 + 3*p1)*r^3) - 3*p2*(-2 + 6*r - 10*r^2 + 6*r^3 + 3*p1*(-1 + 5*r - 9*r^2 + 6*r^3)))
return(L)}
interlogL_rm2 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rm2,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rm2 <- parallel::mcmapply(interlogL_rm2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rm2 <- -((x[,"n_03"] + x[,"n_30"])*(-2*log10(2) + log10(3) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_33"])*(-2*log10(2) + log10(3) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_31"])*(-2*log10(2) + log10(3) + log10(7) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_32"])*(-2*log10(2) + log10(3) + log10(7) + log10(2 - 3*p2)) - (x[,"n_12"] + x[,"n_21"])*log10(2*(8 + (-10 - 27*p1)/4 + (3*(1 + 3*p1))/4 + (4 + 9*p1)/2) + (9*p2)/4) - (x[,"n_13"] + x[,"n_20"])*log10((-3*(-2 + 3*p2))/4 + 3*(3 + (9*p2)/2)) - (x[,"n_10"] + x[,"n_23"])*log10((-3*(-2 + 3*p2))/4 + 3*(3 + (9*p2)/2)) - (x[,"n_11"] + x[,"n_22"])*(log10(2) + log10(8 + (8 - 12*p2)/4 + (-2 + 6*p2)/2 + (-6 + 9*p2)/8)) + (x[,"n_03"] + x[,"n_30"])*(log10(2 - 3*p2) + log10(4 + p1*(3 - 6*r_rm2) - 2*r_rm2) + log10(pmax(1e-6,1 - r_rm2)) + log10(pmax(1e-6,r_rm2))) + (x[,"n_00"] + x[,"n_33"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rm2)) + log10(pmax(1e-6,r_rm2)) + log10(2 - 3*p1 + 2*r_rm2 + 6*p1*r_rm2)) + (x[,"n_02"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(-2*(-4 + 5*r_rm2 - 6*r_rm2^2 + 3*r_rm2^3) - 3*p1*(-2 + 7*r_rm2 - 9*r_rm2^2 + 6*r_rm2^3))) + (x[,"n_01"] + x[,"n_32"])*(log10(2 - 3*p2) + log10(4 + 4*r_rm2 - 6*r_rm2^2 + 6*r_rm2^3 + 3*p1*(-2 + 7*r_rm2 - 9*r_rm2^2 + 6*r_rm2^3))) + (x[,"n_13"] + x[,"n_20"])*log10((-2 + 3*p2)*(-1 + r_rm2)*r_rm2*(2 - 3*p1 + 2*r_rm2 + 6*p1*r_rm2) + (4 + p1*(3 - 6*r_rm2) - 2*r_rm2)*(p2*(3 + 6*r_rm2 - 6*r_rm2^2) + 4*(1 - r_rm2 + r_rm2^2))) + (x[,"n_10"] + x[,"n_23"])*log10((-2 + 3*p2)*(4 + p1*(3 - 6*r_rm2) - 2*r_rm2)*(-1 + r_rm2)*r_rm2 + (2 - 3*p1 + 2*r_rm2 + 6*p1*r_rm2)*(p2*(3 + 6*r_rm2 - 6*r_rm2^2) + 4*(1 - r_rm2 + r_rm2^2))) + (x[,"n_11"] + x[,"n_22"])*log10(2*(8 + (-2 + 6*p2)*r_rm2 + (8 - 12*p2)*r_rm2^2 + (-6 + 9*p2)*r_rm2^3) + 9*p1*(-1 + 2*r_rm2)*(-2*(-1 + r_rm2)*r_rm2 + p2*(1 - 3*r_rm2 + 3*r_rm2^2))) + (x[,"n_12"] + x[,"n_21"])*log10(2*(8 + (4 + 9*p1)*r_rm2 - (10 + 27*p1)*r_rm2^2 + 6*(1 + 3*p1)*r_rm2^3) - 3*p2*(-2 + 6*r_rm2 - 10*r_rm2^2 + 6*r_rm2^3 + 3*p1*(-1 + 5*r_rm2 - 9*r_rm2^2 + 6*r_rm2^3)))


logL_rm2 <- 5*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"])*log(3) + (x[,"n_03"] + x[,"n_30"])*(log(2 - 3*p2) + log(4 + p1*(3 - 6*r_rm2) - 2*r_rm2) + log(pmax(1e-6,1 - r_rm2)) + log(pmax(1e-6,r_rm2))) + (x[,"n_00"] + x[,"n_33"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_rm2)) + log(pmax(1e-6,r_rm2)) + log(2 - 3*p1 + 2*r_rm2 + 6*p1*r_rm2)) + (x[,"n_02"] + x[,"n_31"])*(log(2 - 3*p2) + log(-2*(-4 + 5*r_rm2 - 6*r_rm2^2 + 3*r_rm2^3) - 3*p1*(-2 + 7*r_rm2 - 9*r_rm2^2 + 6*r_rm2^3))) + (x[,"n_01"] + x[,"n_32"])*(log(2 - 3*p2) + log(4 + 4*r_rm2 - 6*r_rm2^2 + 6*r_rm2^3 + 3*p1*(-2 + 7*r_rm2 - 9*r_rm2^2 + 6*r_rm2^3))) + (x[,"n_13"] + x[,"n_20"])*log((-2 + 3*p2)*(-1 + r_rm2)*r_rm2*(2 - 3*p1 + 2*r_rm2 + 6*p1*r_rm2) + (4 + p1*(3 - 6*r_rm2) - 2*r_rm2)*(p2*(3 + 6*r_rm2 - 6*r_rm2^2) + 4*(1 - r_rm2 + r_rm2^2))) + (x[,"n_10"] + x[,"n_23"])*log((-2 + 3*p2)*(4 + p1*(3 - 6*r_rm2) - 2*r_rm2)*(-1 + r_rm2)*r_rm2 + (2 - 3*p1 + 2*r_rm2 + 6*p1*r_rm2)*(p2*(3 + 6*r_rm2 - 6*r_rm2^2) + 4*(1 - r_rm2 + r_rm2^2))) + (x[,"n_11"] + x[,"n_22"])*log(2*(8 + (-2 + 6*p2)*r_rm2 + (8 - 12*p2)*r_rm2^2 + (-6 + 9*p2)*r_rm2^3) + 9*p1*(-1 + 2*r_rm2)*(-2*(-1 + r_rm2)*r_rm2 + p2*(1 - 3*r_rm2 + 3*r_rm2^2))) + (x[,"n_12"] + x[,"n_21"])*log(2*(8 + (4 + 9*p1)*r_rm2 - (10 + 27*p1)*r_rm2^2 + 6*(1 + 3*p1)*r_rm2^3) - 3*p2*(-2 + 6*r_rm2 - 10*r_rm2^2 + 6*r_rm2^3 + 3*p1*(-1 + 5*r_rm2 - 9*r_rm2^2 + 6*r_rm2^3)))


logL_rm3 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- 6*(-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*log(2) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n10 - n11 - n12 - 2*n13 - 2*n20 - n21 - n22 - 2*n23 - 2*n30 - 2*n31 - 2*n32 - 2*n33)*log(3) + (n00 + n33)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(4 + 3*p1 + 4*r - 6*p1*r)) + (n03 + n30)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(8 - 4*r + p1*(-3 + 6*r))) + (n01 + n32)*(log(2 - 3*p2) + log(4*(2 + 2*r - 3*r^2 + 3*r^3) - 3*p1*(-2 + 7*r - 9*r^2 + 6*r^3))) + (n02 + n31)*(log(2 - 3*p2) + log(-4*(-4 + 5*r - 6*r^2 + 3*r^3) + 3*p1*(-2 + 7*r - 9*r^2 + 6*r^3))) + (n10 + n23)*log((-2 + 3*p2)*(-1 + r)*r*(8 - 4*r + p1*(-3 + 6*r)) + (4 + 3*p1 + 4*r - 6*p1*r)*(p2*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2))) + (n13 + n20)*log((-2 + 3*p2)*(-1 + r)*r*(4 + 3*p1 + 4*r - 6*p1*r) + (8 - 4*r + p1*(-3 + 6*r))*(p2*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2))) + (n11 + n22)*log(4*(8 + (-2 + 6*p2)*r + (8 - 12*p2)*r^2 + (-6 + 9*p2)*r^3) - 9*p1*(-1 + 2*r)*(-2*(-1 + r)*r + p2*(1 - 3*r + 3*r^2))) + (n12 + n21)*log(32 - 2*(-8 + 9*p1)*r + (-40 + 54*p1)*r^2 + (24 - 36*p1)*r^3 + 3*p2*(4*(1 - 3*r + 5*r^2 - 3*r^3) + 3*p1*(-1 + 5*r - 9*r^2 + 6*r^3)))
return(L)}
interlogL_rm3 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rm3,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rm3 <- parallel::mcmapply(interlogL_rm3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rm3 <- -((x[,"n_03"] + x[,"n_30"])*(-log10(2) + log10(3) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_31"])*(-log10(2) + log10(3) + log10(7) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_32"])*(-log10(2) + log10(3) + log10(7) + log10(2 - 3*p2)) - (x[,"n_12"] + x[,"n_21"])*log10(40 + (24 - 36*p1)/8 - 9*p1 + (-40 + 54*p1)/4 + (9*p2)/2) - (x[,"n_13"] + x[,"n_20"])*log10((-3*(-2 + 3*p2))/2 + 6*(3 + (9*p2)/2)) - (x[,"n_10"] + x[,"n_23"])*log10((-3*(-2 + 3*p2))/2 + 6*(3 + (9*p2)/2)) - (x[,"n_11"] + x[,"n_22"])*(2*log10(2) + log10(8 + (8 - 12*p2)/4 + (-2 + 6*p2)/2 + (-6 + 9*p2)/8)) + (x[,"n_00"] + x[,"n_33"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rm3)) + log10(pmax(1e-6,r_rm3)) + log10(4 + 3*p1 + 4*r_rm3 - 6*p1*r_rm3)) + (x[,"n_03"] + x[,"n_30"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rm3)) + log10(pmax(1e-6,r_rm3)) + log10(8 - 4*r_rm3 + p1*(-3 + 6*r_rm3))) + (x[,"n_01"] + x[,"n_32"])*(log10(2 - 3*p2) + log10(4*(2 + 2*r_rm3 - 3*r_rm3^2 + 3*r_rm3^3) - 3*p1*(-2 + 7*r_rm3 - 9*r_rm3^2 + 6*r_rm3^3))) + (x[,"n_02"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(-4*(-4 + 5*r_rm3 - 6*r_rm3^2 + 3*r_rm3^3) + 3*p1*(-2 + 7*r_rm3 - 9*r_rm3^2 + 6*r_rm3^3))) + (x[,"n_10"] + x[,"n_23"])*log10((-2 + 3*p2)*(-1 + r_rm3)*r_rm3*(8 - 4*r_rm3 + p1*(-3 + 6*r_rm3)) + (4 + 3*p1 + 4*r_rm3 - 6*p1*r_rm3)*(p2*(3 + 6*r_rm3 - 6*r_rm3^2) + 4*(1 - r_rm3 + r_rm3^2))) + (x[,"n_13"] + x[,"n_20"])*log10((-2 + 3*p2)*(-1 + r_rm3)*r_rm3*(4 + 3*p1 + 4*r_rm3 - 6*p1*r_rm3) + (8 - 4*r_rm3 + p1*(-3 + 6*r_rm3))*(p2*(3 + 6*r_rm3 - 6*r_rm3^2) + 4*(1 - r_rm3 + r_rm3^2))) + (x[,"n_11"] + x[,"n_22"])*log10(4*(8 + (-2 + 6*p2)*r_rm3 + (8 - 12*p2)*r_rm3^2 + (-6 + 9*p2)*r_rm3^3) - 9*p1*(-1 + 2*r_rm3)*(-2*(-1 + r_rm3)*r_rm3 + p2*(1 - 3*r_rm3 + 3*r_rm3^2))) + (x[,"n_12"] + x[,"n_21"])*log10(32 - 2*(-8 + 9*p1)*r_rm3 + (-40 + 54*p1)*r_rm3^2 + (24 - 36*p1)*r_rm3^3 + 3*p2*(4*(1 - 3*r_rm3 + 5*r_rm3^2 - 3*r_rm3^3) + 3*p1*(-1 + 5*r_rm3 - 9*r_rm3^2 + 6*r_rm3^3)))


logL_rm3 <- 6*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"])*log(3) + (x[,"n_00"] + x[,"n_33"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_rm3)) + log(pmax(1e-6,r_rm3)) + log(4 + 3*p1 + 4*r_rm3 - 6*p1*r_rm3)) + (x[,"n_03"] + x[,"n_30"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_rm3)) + log(pmax(1e-6,r_rm3)) + log(8 - 4*r_rm3 + p1*(-3 + 6*r_rm3))) + (x[,"n_01"] + x[,"n_32"])*(log(2 - 3*p2) + log(4*(2 + 2*r_rm3 - 3*r_rm3^2 + 3*r_rm3^3) - 3*p1*(-2 + 7*r_rm3 - 9*r_rm3^2 + 6*r_rm3^3))) + (x[,"n_02"] + x[,"n_31"])*(log(2 - 3*p2) + log(-4*(-4 + 5*r_rm3 - 6*r_rm3^2 + 3*r_rm3^3) + 3*p1*(-2 + 7*r_rm3 - 9*r_rm3^2 + 6*r_rm3^3))) + (x[,"n_10"] + x[,"n_23"])*log((-2 + 3*p2)*(-1 + r_rm3)*r_rm3*(8 - 4*r_rm3 + p1*(-3 + 6*r_rm3)) + (4 + 3*p1 + 4*r_rm3 - 6*p1*r_rm3)*(p2*(3 + 6*r_rm3 - 6*r_rm3^2) + 4*(1 - r_rm3 + r_rm3^2))) + (x[,"n_13"] + x[,"n_20"])*log((-2 + 3*p2)*(-1 + r_rm3)*r_rm3*(4 + 3*p1 + 4*r_rm3 - 6*p1*r_rm3) + (8 - 4*r_rm3 + p1*(-3 + 6*r_rm3))*(p2*(3 + 6*r_rm3 - 6*r_rm3^2) + 4*(1 - r_rm3 + r_rm3^2))) + (x[,"n_11"] + x[,"n_22"])*log(4*(8 + (-2 + 6*p2)*r_rm3 + (8 - 12*p2)*r_rm3^2 + (-6 + 9*p2)*r_rm3^3) - 9*p1*(-1 + 2*r_rm3)*(-2*(-1 + r_rm3)*r_rm3 + p2*(1 - 3*r_rm3 + 3*r_rm3^2))) + (x[,"n_12"] + x[,"n_21"])*log(32 - 2*(-8 + 9*p1)*r_rm3 + (-40 + 54*p1)*r_rm3^2 + (24 - 36*p1)*r_rm3^3 + 3*p2*(4*(1 - 3*r_rm3 + 5*r_rm3^2 - 3*r_rm3^3) + 3*p1*(-1 + 5*r_rm3 - 9*r_rm3^2 + 6*r_rm3^3)))


logL_rm4 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- 6*(-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*log(2) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n10 - n11 - n12 - 2*n13 - 2*n20 - n21 - n22 - 2*n23 - 2*n30 - 2*n31 - 2*n32 - 2*n33)*log(3) + (n00 + n33)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(4 + 3*p1 + 4*r - 6*p1*r)) + (n03 + n30)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(8 - 4*r + p1*(-3 + 6*r))) + (n10 + n23)*(log(2 - 3*p2) + log(4*(2 + 2*r - 3*r^2 + 3*r^3) - 3*p1*(-2 + 7*r - 9*r^2 + 6*r^3))) + (n13 + n20)*(log(2 - 3*p2) + log(-4*(-4 + 5*r - 6*r^2 + 3*r^3) + 3*p1*(-2 + 7*r - 9*r^2 + 6*r^3))) + (n01 + n32)*log((-2 + 3*p2)*(-1 + r)*r*(8 - 4*r + p1*(-3 + 6*r)) + (4 + 3*p1 + 4*r - 6*p1*r)*(p2*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2))) + (n02 + n31)*log((-2 + 3*p2)*(-1 + r)*r*(4 + 3*p1 + 4*r - 6*p1*r) + (8 - 4*r + p1*(-3 + 6*r))*(p2*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2))) + (n11 + n22)*log(4*(8 + (-2 + 6*p2)*r + (8 - 12*p2)*r^2 + (-6 + 9*p2)*r^3) - 9*p1*(-1 + 2*r)*(-2*(-1 + r)*r + p2*(1 - 3*r + 3*r^2))) + (n12 + n21)*log(32 - 2*(-8 + 9*p1)*r + (-40 + 54*p1)*r^2 + (24 - 36*p1)*r^3 + 3*p2*(4*(1 - 3*r + 5*r^2 - 3*r^3) + 3*p1*(-1 + 5*r - 9*r^2 + 6*r^3)))
return(L)}
interlogL_rm4 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rm4,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rm4 <- parallel::mcmapply(interlogL_rm4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rm4 <- -((x[,"n_03"] + x[,"n_30"])*(-log10(2) + log10(3) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(2 - 3*p2)) - (x[,"n_13"] + x[,"n_20"])*(-log10(2) + log10(3) + log10(7) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_23"])*(-log10(2) + log10(3) + log10(7) + log10(2 - 3*p2)) - (x[,"n_12"] + x[,"n_21"])*log10(40 + (24 - 36*p1)/8 - 9*p1 + (-40 + 54*p1)/4 + (9*p2)/2) - (x[,"n_02"] + x[,"n_31"])*log10((-3*(-2 + 3*p2))/2 + 6*(3 + (9*p2)/2)) - (x[,"n_01"] + x[,"n_32"])*log10((-3*(-2 + 3*p2))/2 + 6*(3 + (9*p2)/2)) - (x[,"n_11"] + x[,"n_22"])*(2*log10(2) + log10(8 + (8 - 12*p2)/4 + (-2 + 6*p2)/2 + (-6 + 9*p2)/8)) + (x[,"n_00"] + x[,"n_33"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rm4)) + log10(pmax(1e-6,r_rm4)) + log10(4 + 3*p1 + 4*r_rm4 - 6*p1*r_rm4)) + (x[,"n_03"] + x[,"n_30"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rm4)) + log10(pmax(1e-6,r_rm4)) + log10(8 - 4*r_rm4 + p1*(-3 + 6*r_rm4))) + (x[,"n_10"] + x[,"n_23"])*(log10(2 - 3*p2) + log10(4*(2 + 2*r_rm4 - 3*r_rm4^2 + 3*r_rm4^3) - 3*p1*(-2 + 7*r_rm4 - 9*r_rm4^2 + 6*r_rm4^3))) + (x[,"n_13"] + x[,"n_20"])*(log10(2 - 3*p2) + log10(-4*(-4 + 5*r_rm4 - 6*r_rm4^2 + 3*r_rm4^3) + 3*p1*(-2 + 7*r_rm4 - 9*r_rm4^2 + 6*r_rm4^3))) + (x[,"n_01"] + x[,"n_32"])*log10((-2 + 3*p2)*(-1 + r_rm4)*r_rm4*(8 - 4*r_rm4 + p1*(-3 + 6*r_rm4)) + (4 + 3*p1 + 4*r_rm4 - 6*p1*r_rm4)*(p2*(3 + 6*r_rm4 - 6*r_rm4^2) + 4*(1 - r_rm4 + r_rm4^2))) + (x[,"n_02"] + x[,"n_31"])*log10((-2 + 3*p2)*(-1 + r_rm4)*r_rm4*(4 + 3*p1 + 4*r_rm4 - 6*p1*r_rm4) + (8 - 4*r_rm4 + p1*(-3 + 6*r_rm4))*(p2*(3 + 6*r_rm4 - 6*r_rm4^2) + 4*(1 - r_rm4 + r_rm4^2))) + (x[,"n_11"] + x[,"n_22"])*log10(4*(8 + (-2 + 6*p2)*r_rm4 + (8 - 12*p2)*r_rm4^2 + (-6 + 9*p2)*r_rm4^3) - 9*p1*(-1 + 2*r_rm4)*(-2*(-1 + r_rm4)*r_rm4 + p2*(1 - 3*r_rm4 + 3*r_rm4^2))) + (x[,"n_12"] + x[,"n_21"])*log10(32 - 2*(-8 + 9*p1)*r_rm4 + (-40 + 54*p1)*r_rm4^2 + (24 - 36*p1)*r_rm4^3 + 3*p2*(4*(1 - 3*r_rm4 + 5*r_rm4^2 - 3*r_rm4^3) + 3*p1*(-1 + 5*r_rm4 - 9*r_rm4^2 + 6*r_rm4^3)))


logL_rm4 <- 6*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"])*log(3) + (x[,"n_00"] + x[,"n_33"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_rm4)) + log(pmax(1e-6,r_rm4)) + log(4 + 3*p1 + 4*r_rm4 - 6*p1*r_rm4)) + (x[,"n_03"] + x[,"n_30"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_rm4)) + log(pmax(1e-6,r_rm4)) + log(8 - 4*r_rm4 + p1*(-3 + 6*r_rm4))) + (x[,"n_10"] + x[,"n_23"])*(log(2 - 3*p2) + log(4*(2 + 2*r_rm4 - 3*r_rm4^2 + 3*r_rm4^3) - 3*p1*(-2 + 7*r_rm4 - 9*r_rm4^2 + 6*r_rm4^3))) + (x[,"n_13"] + x[,"n_20"])*(log(2 - 3*p2) + log(-4*(-4 + 5*r_rm4 - 6*r_rm4^2 + 3*r_rm4^3) + 3*p1*(-2 + 7*r_rm4 - 9*r_rm4^2 + 6*r_rm4^3))) + (x[,"n_01"] + x[,"n_32"])*log((-2 + 3*p2)*(-1 + r_rm4)*r_rm4*(8 - 4*r_rm4 + p1*(-3 + 6*r_rm4)) + (4 + 3*p1 + 4*r_rm4 - 6*p1*r_rm4)*(p2*(3 + 6*r_rm4 - 6*r_rm4^2) + 4*(1 - r_rm4 + r_rm4^2))) + (x[,"n_02"] + x[,"n_31"])*log((-2 + 3*p2)*(-1 + r_rm4)*r_rm4*(4 + 3*p1 + 4*r_rm4 - 6*p1*r_rm4) + (8 - 4*r_rm4 + p1*(-3 + 6*r_rm4))*(p2*(3 + 6*r_rm4 - 6*r_rm4^2) + 4*(1 - r_rm4 + r_rm4^2))) + (x[,"n_11"] + x[,"n_22"])*log(4*(8 + (-2 + 6*p2)*r_rm4 + (8 - 12*p2)*r_rm4^2 + (-6 + 9*p2)*r_rm4^3) - 9*p1*(-1 + 2*r_rm4)*(-2*(-1 + r_rm4)*r_rm4 + p2*(1 - 3*r_rm4 + 3*r_rm4^2))) + (x[,"n_12"] + x[,"n_21"])*log(32 - 2*(-8 + 9*p1)*r_rm4 + (-40 + 54*p1)*r_rm4^2 + (24 - 36*p1)*r_rm4^3 + 3*p2*(4*(1 - 3*r_rm4 + 5*r_rm4^2 - 3*r_rm4^3) + 3*p1*(-1 + 5*r_rm4 - 9*r_rm4^2 + 6*r_rm4^3)))


logL_rr1 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(4*log(2) + 2*log(3)) + (n03 + n30)*(log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r)) + log(4 + 3*p1 - 2*r - 6*p1*r)) + (n00 + n33)*(log(2 - 3*p2) + 2*log(pmax(1e-6,r)) + log(2 - 3*p1 + 2*r + 6*p1*r)) + (n01 + n10 + n23 + n32)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(4 + 4*r - 6*r^2 - 3*p1*(2 - 7*r + 6*r^2))) + (n02 + n13 + n20 + n31)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(2 + 8*r - 6*r^2 - 3*p1*(1 - 5*r + 6*r^2))) + (n12 + n21)*log(40 - 52*r + 64*r^2 - 36*r^3 + 6*p2*(2 + 7*r - 16*r^2 + 9*r^3) + 3*p1*(-1 + 2*r)*(3*p2*(-1 - 10*r + 9*r^2) - 2*(5 - 10*r + 9*r^2))) + (n11 + n22)*log(-3*p1*(-1 + 2*r)*(3*p2*(-2 - 8*r + 9*r^2) - 2*(4 - 8*r + 9*r^2)) + 2*(p2*(6 - 6*r + 33*r^2 - 27*r^3) + 2*(4 + 8*r - 11*r^2 + 9*r^3)))
return(L)}
interlogL_rr1 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rr1,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr1 <- parallel::mcmapply(interlogL_rr1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rr1 <- -((x[,"n_03"] + x[,"n_30"])*(-2*log10(2) + log10(3) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_33"])*(-2*log10(2) + log10(3) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(-2*log10(2) + 2*log10(3) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(-2*log10(2) + 2*log10(3) + log10(2 - 3*p2)) - (x[,"n_11"] + x[,"n_22"])*(log10(2) + log10(51/4 + (63*p2)/8)) - (x[,"n_12"] + x[,"n_21"])*log10(51/2 + (63*p2)/4) + (x[,"n_03"] + x[,"n_30"])*(log10(2 - 3*p2) + 2*log10(pmax(1e-6,1 - r_rr1)) + log10(4 + 3*p1 - 2*r_rr1 - 6*p1*r_rr1)) + (x[,"n_00"] + x[,"n_33"])*(log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_rr1)) + log10(2 - 3*p1 + 2*r_rr1 + 6*p1*r_rr1)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_rr1)) + log10(4 + 4*r_rr1 - 6*r_rr1^2 - 3*p1*(2 - 7*r_rr1 + 6*r_rr1^2))) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rr1)) + log10(2 + 8*r_rr1 - 6*r_rr1^2 - 3*p1*(1 - 5*r_rr1 + 6*r_rr1^2))) + (x[,"n_12"] + x[,"n_21"])*log10(40 - 52*r_rr1 + 64*r_rr1^2 - 36*r_rr1^3 + 6*p2*(2 + 7*r_rr1 - 16*r_rr1^2 + 9*r_rr1^3) + 3*p1*(-1 + 2*r_rr1)*(3*p2*(-1 - 10*r_rr1 + 9*r_rr1^2) - 2*(5 - 10*r_rr1 + 9*r_rr1^2))) + (x[,"n_11"] + x[,"n_22"])*log10(-3*p1*(-1 + 2*r_rr1)*(3*p2*(-2 - 8*r_rr1 + 9*r_rr1^2) - 2*(4 - 8*r_rr1 + 9*r_rr1^2)) + 2*(p2*(6 - 6*r_rr1 + 33*r_rr1^2 - 27*r_rr1^3) + 2*(4 + 8*r_rr1 - 11*r_rr1^2 + 9*r_rr1^3)))


logL_rr1 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(4*log(2) + 2*log(3)) + (x[,"n_03"] + x[,"n_30"])*(log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r_rr1)) + log(4 + 3*p1 - 2*r_rr1 - 6*p1*r_rr1)) + (x[,"n_00"] + x[,"n_33"])*(log(2 - 3*p2) + 2*log(pmax(1e-6,r_rr1)) + log(2 - 3*p1 + 2*r_rr1 + 6*p1*r_rr1)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(log(2 - 3*p2) + log(pmax(1e-6,r_rr1)) + log(4 + 4*r_rr1 - 6*r_rr1^2 - 3*p1*(2 - 7*r_rr1 + 6*r_rr1^2))) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_rr1)) + log(2 + 8*r_rr1 - 6*r_rr1^2 - 3*p1*(1 - 5*r_rr1 + 6*r_rr1^2))) + (x[,"n_12"] + x[,"n_21"])*log(40 - 52*r_rr1 + 64*r_rr1^2 - 36*r_rr1^3 + 6*p2*(2 + 7*r_rr1 - 16*r_rr1^2 + 9*r_rr1^3) + 3*p1*(-1 + 2*r_rr1)*(3*p2*(-1 - 10*r_rr1 + 9*r_rr1^2) - 2*(5 - 10*r_rr1 + 9*r_rr1^2))) + (x[,"n_11"] + x[,"n_22"])*log(-3*p1*(-1 + 2*r_rr1)*(3*p2*(-2 - 8*r_rr1 + 9*r_rr1^2) - 2*(4 - 8*r_rr1 + 9*r_rr1^2)) + 2*(p2*(6 - 6*r_rr1 + 33*r_rr1^2 - 27*r_rr1^3) + 2*(4 + 8*r_rr1 - 11*r_rr1^2 + 9*r_rr1^3)))


logL_rr2 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(5*log(2) + 2*log(3)) + (n00 + n33)*(log(4 + 3*p2) + 2*log(pmax(1e-6,r)) + log(2 - 3*p1 + 2*r + 6*p1*r)) + (n03 + n30)*(log(4 + 3*p2) + log(4 + p1*(3 - 6*r) - 2*r) + log(1 - 2*r + r^2)) + (n01 + n10 + n23 + n32)*(log(4 + 3*p2) + log(pmax(1e-6,r)) + log(4 + 4*r - 6*r^2 - 3*p1*(2 - 7*r + 6*r^2))) + (n02 + n13 + n20 + n31)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(2 + 8*r - 6*r^2 - 3*p1*(1 - 5*r + 6*r^2))) + (n12 + n21)*log(-3*p1*(-1 + 2*r)*(3*p2*(-1 - 10*r + 9*r^2) + 4*(5 - 10*r + 9*r^2)) - 2*(-40 + 52*r - 64*r^2 + 36*r^3 + 3*p2*(2 + 7*r - 16*r^2 + 9*r^3))) + (n11 + n22)*log(3*p1*(-1 + 2*r)*(3*p2*(-2 - 8*r + 9*r^2) + 4*(4 - 8*r + 9*r^2)) + 2*(3*p2*(-2 + 2*r - 11*r^2 + 9*r^3) + 4*(4 + 8*r - 11*r^2 + 9*r^3)))
return(L)}
interlogL_rr2 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rr2,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr2 <- parallel::mcmapply(interlogL_rr2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rr2 <- -((x[,"n_11"] + x[,"n_22"])*(log10(2) + log10(51/2 - (63*p2)/8))) - (x[,"n_03"] + x[,"n_30"])*(-2*log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_33"])*(-2*log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(-2*log10(2) + 2*log10(3) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(-2*log10(2) + 2*log10(3) + log10(4 + 3*p2)) - (x[,"n_12"] + x[,"n_21"])*(log10(2) + log10(51/2 - (63*p2)/8)) + (x[,"n_00"] + x[,"n_33"])*(log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_rr2)) + log10(2 - 3*p1 + 2*r_rr2 + 6*p1*r_rr2)) + (x[,"n_03"] + x[,"n_30"])*(log10(4 + 3*p2) + log10(4 + p1*(3 - 6*r_rr2) - 2*r_rr2) + log10(1 - 2*r_rr2 + r_rr2^2)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(log10(4 + 3*p2) + log10(pmax(1e-6,r_rr2)) + log10(4 + 4*r_rr2 - 6*r_rr2^2 - 3*p1*(2 - 7*r_rr2 + 6*r_rr2^2))) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rr2)) + log10(2 + 8*r_rr2 - 6*r_rr2^2 - 3*p1*(1 - 5*r_rr2 + 6*r_rr2^2))) + (x[,"n_12"] + x[,"n_21"])*log10(-3*p1*(-1 + 2*r_rr2)*(3*p2*(-1 - 10*r_rr2 + 9*r_rr2^2) + 4*(5 - 10*r_rr2 + 9*r_rr2^2)) - 2*(-40 + 52*r_rr2 - 64*r_rr2^2 + 36*r_rr2^3 + 3*p2*(2 + 7*r_rr2 - 16*r_rr2^2 + 9*r_rr2^3))) + (x[,"n_11"] + x[,"n_22"])*log10(3*p1*(-1 + 2*r_rr2)*(3*p2*(-2 - 8*r_rr2 + 9*r_rr2^2) + 4*(4 - 8*r_rr2 + 9*r_rr2^2)) + 2*(3*p2*(-2 + 2*r_rr2 - 11*r_rr2^2 + 9*r_rr2^3) + 4*(4 + 8*r_rr2 - 11*r_rr2^2 + 9*r_rr2^3)))


logL_rr2 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(5*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_33"])*(log(4 + 3*p2) + 2*log(pmax(1e-6,r_rr2)) + log(2 - 3*p1 + 2*r_rr2 + 6*p1*r_rr2)) + (x[,"n_03"] + x[,"n_30"])*(log(4 + 3*p2) + log(4 + p1*(3 - 6*r_rr2) - 2*r_rr2) + log(1 - 2*r_rr2 + r_rr2^2)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(log(4 + 3*p2) + log(pmax(1e-6,r_rr2)) + log(4 + 4*r_rr2 - 6*r_rr2^2 - 3*p1*(2 - 7*r_rr2 + 6*r_rr2^2))) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_rr2)) + log(2 + 8*r_rr2 - 6*r_rr2^2 - 3*p1*(1 - 5*r_rr2 + 6*r_rr2^2))) + (x[,"n_12"] + x[,"n_21"])*log(-3*p1*(-1 + 2*r_rr2)*(3*p2*(-1 - 10*r_rr2 + 9*r_rr2^2) + 4*(5 - 10*r_rr2 + 9*r_rr2^2)) - 2*(-40 + 52*r_rr2 - 64*r_rr2^2 + 36*r_rr2^3 + 3*p2*(2 + 7*r_rr2 - 16*r_rr2^2 + 9*r_rr2^3))) + (x[,"n_11"] + x[,"n_22"])*log(3*p1*(-1 + 2*r_rr2)*(3*p2*(-2 - 8*r_rr2 + 9*r_rr2^2) + 4*(4 - 8*r_rr2 + 9*r_rr2^2)) + 2*(3*p2*(-2 + 2*r_rr2 - 11*r_rr2^2 + 9*r_rr2^3) + 4*(4 + 8*r_rr2 - 11*r_rr2^2 + 9*r_rr2^3)))


logL_rr3 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(5*log(2) + 2*log(3)) + (n00 + n33)*(log(2 - 3*p2) + 2*log(pmax(1e-6,r)) + log(4 + 3*p1 + 4*r - 6*p1*r)) + (n03 + n30)*(log(2 - 3*p2) + log(1 - 2*r + r^2) + log(8 - 4*r + p1*(-3 + 6*r))) + (n01 + n10 + n23 + n32)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(8 + 8*r - 12*r^2 + 3*p1*(2 - 7*r + 6*r^2))) + (n02 + n13 + n20 + n31)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(4 + 16*r - 12*r^2 + 3*p1*(1 - 5*r + 6*r^2))) + (n12 + n21)*log(-3*p1*(-1 + 2*r)*(3*p2*(-1 - 10*r + 9*r^2) - 2*(5 - 10*r + 9*r^2)) + 4*(20 - 26*r + 32*r^2 - 18*r^3 + 3*p2*(2 + 7*r - 16*r^2 + 9*r^3))) + (n11 + n22)*log(3*p1*(-1 + 2*r)*(3*p2*(-2 - 8*r + 9*r^2) - 2*(4 - 8*r + 9*r^2)) + 4*(p2*(6 - 6*r + 33*r^2 - 27*r^3) + 2*(4 + 8*r - 11*r^2 + 9*r^3)))
return(L)}
interlogL_rr3 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rr3,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr3 <- parallel::mcmapply(interlogL_rr3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rr3 <- -((x[,"n_03"] + x[,"n_30"])*(-log10(2) + log10(3) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(-log10(2) + 2*log10(3) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(-log10(2) + 2*log10(3) + log10(2 - 3*p2)) - (x[,"n_12"] + x[,"n_21"])*(2*log10(2) + log10(51/4 + (63*p2)/8)) - (x[,"n_11"] + x[,"n_22"])*(2*log10(2) + log10(51/4 + (63*p2)/8)) + (x[,"n_00"] + x[,"n_33"])*(log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_rr3)) + log10(4 + 3*p1 + 4*r_rr3 - 6*p1*r_rr3)) + (x[,"n_03"] + x[,"n_30"])*(log10(2 - 3*p2) + log10(1 - 2*r_rr3 + r_rr3^2) + log10(8 - 4*r_rr3 + p1*(-3 + 6*r_rr3))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_rr3)) + log10(8 + 8*r_rr3 - 12*r_rr3^2 + 3*p1*(2 - 7*r_rr3 + 6*r_rr3^2))) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rr3)) + log10(4 + 16*r_rr3 - 12*r_rr3^2 + 3*p1*(1 - 5*r_rr3 + 6*r_rr3^2))) + (x[,"n_12"] + x[,"n_21"])*log10(-3*p1*(-1 + 2*r_rr3)*(3*p2*(-1 - 10*r_rr3 + 9*r_rr3^2) - 2*(5 - 10*r_rr3 + 9*r_rr3^2)) + 4*(20 - 26*r_rr3 + 32*r_rr3^2 - 18*r_rr3^3 + 3*p2*(2 + 7*r_rr3 - 16*r_rr3^2 + 9*r_rr3^3))) + (x[,"n_11"] + x[,"n_22"])*log10(3*p1*(-1 + 2*r_rr3)*(3*p2*(-2 - 8*r_rr3 + 9*r_rr3^2) - 2*(4 - 8*r_rr3 + 9*r_rr3^2)) + 4*(p2*(6 - 6*r_rr3 + 33*r_rr3^2 - 27*r_rr3^3) + 2*(4 + 8*r_rr3 - 11*r_rr3^2 + 9*r_rr3^3)))


logL_rr3 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(5*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_33"])*(log(2 - 3*p2) + 2*log(pmax(1e-6,r_rr3)) + log(4 + 3*p1 + 4*r_rr3 - 6*p1*r_rr3)) + (x[,"n_03"] + x[,"n_30"])*(log(2 - 3*p2) + log(1 - 2*r_rr3 + r_rr3^2) + log(8 - 4*r_rr3 + p1*(-3 + 6*r_rr3))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(log(2 - 3*p2) + log(pmax(1e-6,r_rr3)) + log(8 + 8*r_rr3 - 12*r_rr3^2 + 3*p1*(2 - 7*r_rr3 + 6*r_rr3^2))) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_rr3)) + log(4 + 16*r_rr3 - 12*r_rr3^2 + 3*p1*(1 - 5*r_rr3 + 6*r_rr3^2))) + (x[,"n_12"] + x[,"n_21"])*log(-3*p1*(-1 + 2*r_rr3)*(3*p2*(-1 - 10*r_rr3 + 9*r_rr3^2) - 2*(5 - 10*r_rr3 + 9*r_rr3^2)) + 4*(20 - 26*r_rr3 + 32*r_rr3^2 - 18*r_rr3^3 + 3*p2*(2 + 7*r_rr3 - 16*r_rr3^2 + 9*r_rr3^3))) + (x[,"n_11"] + x[,"n_22"])*log(3*p1*(-1 + 2*r_rr3)*(3*p2*(-2 - 8*r_rr3 + 9*r_rr3^2) - 2*(4 - 8*r_rr3 + 9*r_rr3^2)) + 4*(p2*(6 - 6*r_rr3 + 33*r_rr3^2 - 27*r_rr3^3) + 2*(4 + 8*r_rr3 - 11*r_rr3^2 + 9*r_rr3^3)))


logL_rr4 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(6*log(2) + 2*log(3)) + (n00 + n33)*(log(4 + 3*p2) + 2*log(pmax(1e-6,r)) + log(4 + 3*p1 + 4*r - 6*p1*r)) + (n03 + n30)*(log(4 + 3*p2) + log(1 - 2*r + r^2) + log(8 - 4*r + p1*(-3 + 6*r))) + (n01 + n10 + n23 + n32)*(log(4 + 3*p2) + log(pmax(1e-6,r)) + log(8 + 8*r - 12*r^2 + 3*p1*(2 - 7*r + 6*r^2))) + (n02 + n13 + n20 + n31)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(4 + 16*r - 12*r^2 + 3*p1*(1 - 5*r + 6*r^2))) + (n12 + n21)*log(3*p1*(-1 + 2*r)*(3*p2*(-1 - 10*r + 9*r^2) + 4*(5 - 10*r + 9*r^2)) - 4*(-40 + 52*r - 64*r^2 + 36*r^3 + 3*p2*(2 + 7*r - 16*r^2 + 9*r^3))) + (n11 + n22)*log(-3*p1*(-1 + 2*r)*(3*p2*(-2 - 8*r + 9*r^2) + 4*(4 - 8*r + 9*r^2)) + 4*(3*p2*(-2 + 2*r - 11*r^2 + 9*r^3) + 4*(4 + 8*r - 11*r^2 + 9*r^3)))
return(L)}
interlogL_rr4 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rr4,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr4 <- parallel::mcmapply(interlogL_rr4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rr4 <- -((x[,"n_11"] + x[,"n_22"])*(2*log10(2) + log10(51/2 - (63*p2)/8))) - (x[,"n_03"] + x[,"n_30"])*(-log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(-log10(2) + 2*log10(3) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(-log10(2) + 2*log10(3) + log10(4 + 3*p2)) - (x[,"n_12"] + x[,"n_21"])*(2*log10(2) + log10(51/2 - (63*p2)/8)) + (x[,"n_00"] + x[,"n_33"])*(log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_rr4)) + log10(4 + 3*p1 + 4*r_rr4 - 6*p1*r_rr4)) + (x[,"n_03"] + x[,"n_30"])*(log10(4 + 3*p2) + log10(1 - 2*r_rr4 + r_rr4^2) + log10(8 - 4*r_rr4 + p1*(-3 + 6*r_rr4))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(log10(4 + 3*p2) + log10(pmax(1e-6,r_rr4)) + log10(8 + 8*r_rr4 - 12*r_rr4^2 + 3*p1*(2 - 7*r_rr4 + 6*r_rr4^2))) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rr4)) + log10(4 + 16*r_rr4 - 12*r_rr4^2 + 3*p1*(1 - 5*r_rr4 + 6*r_rr4^2))) + (x[,"n_12"] + x[,"n_21"])*log10(3*p1*(-1 + 2*r_rr4)*(3*p2*(-1 - 10*r_rr4 + 9*r_rr4^2) + 4*(5 - 10*r_rr4 + 9*r_rr4^2)) - 4*(-40 + 52*r_rr4 - 64*r_rr4^2 + 36*r_rr4^3 + 3*p2*(2 + 7*r_rr4 - 16*r_rr4^2 + 9*r_rr4^3))) + (x[,"n_11"] + x[,"n_22"])*log10(-3*p1*(-1 + 2*r_rr4)*(3*p2*(-2 - 8*r_rr4 + 9*r_rr4^2) + 4*(4 - 8*r_rr4 + 9*r_rr4^2)) + 4*(3*p2*(-2 + 2*r_rr4 - 11*r_rr4^2 + 9*r_rr4^3) + 4*(4 + 8*r_rr4 - 11*r_rr4^2 + 9*r_rr4^3)))


logL_rr4 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(6*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_33"])*(log(4 + 3*p2) + 2*log(pmax(1e-6,r_rr4)) + log(4 + 3*p1 + 4*r_rr4 - 6*p1*r_rr4)) + (x[,"n_03"] + x[,"n_30"])*(log(4 + 3*p2) + log(1 - 2*r_rr4 + r_rr4^2) + log(8 - 4*r_rr4 + p1*(-3 + 6*r_rr4))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_23"] + x[,"n_32"])*(log(4 + 3*p2) + log(pmax(1e-6,r_rr4)) + log(8 + 8*r_rr4 - 12*r_rr4^2 + 3*p1*(2 - 7*r_rr4 + 6*r_rr4^2))) + (x[,"n_02"] + x[,"n_13"] + x[,"n_20"] + x[,"n_31"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_rr4)) + log(4 + 16*r_rr4 - 12*r_rr4^2 + 3*p1*(1 - 5*r_rr4 + 6*r_rr4^2))) + (x[,"n_12"] + x[,"n_21"])*log(3*p1*(-1 + 2*r_rr4)*(3*p2*(-1 - 10*r_rr4 + 9*r_rr4^2) + 4*(5 - 10*r_rr4 + 9*r_rr4^2)) - 4*(-40 + 52*r_rr4 - 64*r_rr4^2 + 36*r_rr4^3 + 3*p2*(2 + 7*r_rr4 - 16*r_rr4^2 + 9*r_rr4^3))) + (x[,"n_11"] + x[,"n_22"])*log(-3*p1*(-1 + 2*r_rr4)*(3*p2*(-2 - 8*r_rr4 + 9*r_rr4^2) + 4*(4 - 8*r_rr4 + 9*r_rr4^2)) + 4*(3*p2*(-2 + 2*r_rr4 - 11*r_rr4^2 + 9*r_rr4^3) + 4*(4 + 8*r_rr4 - 11*r_rr4^2 + 9*r_rr4^3)))


return(list(
r_mat = cbind(r_cc1,r_cc2,r_cm1,r_cm2,r_cr1,r_cr2,r_rc1,r_rc2,r_rc3,r_rc4,r_rm1,r_rm2,r_rm3,r_rm4,r_rr1,r_rr2,r_rr3,r_rr4,0.499),
LOD_mat = cbind(LOD_cc1,LOD_cc2,LOD_cm1,LOD_cm2,LOD_cr1,LOD_cr2,LOD_rc1,LOD_rc2,LOD_rc3,LOD_rc4,LOD_rm1,LOD_rm2,LOD_rm3,LOD_rm4,LOD_rr1,LOD_rr2,LOD_rr3,LOD_rr4,0),
logL_mat = cbind(logL_cc1,logL_cc2,logL_cm1,logL_cm2,logL_cr1,logL_cr2,logL_rc1,logL_rc2,logL_rc3,logL_rc4,logL_rm1,logL_rm2,logL_rm3,logL_rm4,logL_rr1,logL_rr2,logL_rr3,logL_rr4,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling coupling","coupling mixed","coupling mixed","coupling repulsion","coupling repulsion","repulsion coupling","repulsion coupling","repulsion coupling","repulsion coupling","repulsion mixed","repulsion mixed","repulsion mixed","repulsion mixed","repulsion repulsion","repulsion repulsion","repulsion repulsion","repulsion repulsion","unknown")
)
)
}

#' @rdname p4_functions
p4_1.2_2.1 <- function(x,p1,p2,ncores=1){


logL_cc1 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(4*log(2) + 2*log(3)) + (n03 + n30)*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r))) + (n02 + n31)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(6 + (-2 + 3*p1)*r)) + (n13 + n20)*(log(2 - 3*p1) + log(pmax(1e-6,r)) + log(6 + (-2 + 3*p2)*r)) + (n00 + n33)*(log(2 - 3*p1) + log(2 - 3*p2) + log(1 - 2*r + r^2)) + (n01 + n32)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(3*p1*(1 - r) + 2*(2 + r))) + (n10 + n23)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(3*p2*(1 - r) + 2*(2 + r))) + (n12 + n21)*log(p2*(6 + 6*r - 6*r^2) + 4*(5 + 2*r + r^2) + 3*p1*(2 + 2*r - 2*r^2 + 3*p2*(2 - 4*r + r^2))) + (n11 + n22)*log(p2*(6 + 6*r - 6*r^2) + 4*(8 - 4*r + r^2) + 3*p1*(2 + 2*r - 2*r^2 + 3*p2*(-1 + 2*r + r^2)))
return(L)}
interlogL_cc1 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_cc1,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc1 <- parallel::mcmapply(interlogL_cc1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_cc1 <- -((x[,"n_03"] + x[,"n_30"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_33"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_32"])*(-log10(2) + log10(5 + (3*p1)/2) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_31"])*(-log10(2) + log10(6 + (-2 + 3*p1)/2) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_23"])*(-log10(2) + log10(2 - 3*p1) + log10(5 + (3*p2)/2)) - (x[,"n_12"] + x[,"n_21"])*log10(25 + 3*p1*(5/2 + (3*p2)/4) + (15*p2)/2) - (x[,"n_11"] + x[,"n_22"])*log10(25 + 3*p1*(5/2 + (3*p2)/4) + (15*p2)/2) - (x[,"n_13"] + x[,"n_20"])*(-log10(2) + log10(2 - 3*p1) + log10(6 + (-2 + 3*p2)/2)) + (x[,"n_03"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_cc1))) + (x[,"n_02"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_cc1)) + log10(6 + (-2 + 3*p1)*r_cc1)) + (x[,"n_13"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(pmax(1e-6,r_cc1)) + log10(6 + (-2 + 3*p2)*r_cc1)) + (x[,"n_00"] + x[,"n_33"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(1 - 2*r_cc1 + r_cc1^2)) + (x[,"n_01"] + x[,"n_32"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cc1)) + log10(3*p1*(1 - r_cc1) + 2*(2 + r_cc1))) + (x[,"n_10"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_cc1)) + log10(3*p2*(1 - r_cc1) + 2*(2 + r_cc1))) + (x[,"n_12"] + x[,"n_21"])*log10(p2*(6 + 6*r_cc1 - 6*r_cc1^2) + 4*(5 + 2*r_cc1 + r_cc1^2) + 3*p1*(2 + 2*r_cc1 - 2*r_cc1^2 + 3*p2*(2 - 4*r_cc1 + r_cc1^2))) + (x[,"n_11"] + x[,"n_22"])*log10(p2*(6 + 6*r_cc1 - 6*r_cc1^2) + 4*(8 - 4*r_cc1 + r_cc1^2) + 3*p1*(2 + 2*r_cc1 - 2*r_cc1^2 + 3*p2*(-1 + 2*r_cc1 + r_cc1^2)))


logL_cc1 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(4*log(2) + 2*log(3)) + (x[,"n_03"] + x[,"n_30"])*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r_cc1))) + (x[,"n_02"] + x[,"n_31"])*(log(2 - 3*p2) + log(pmax(1e-6,r_cc1)) + log(6 + (-2 + 3*p1)*r_cc1)) + (x[,"n_13"] + x[,"n_20"])*(log(2 - 3*p1) + log(pmax(1e-6,r_cc1)) + log(6 + (-2 + 3*p2)*r_cc1)) + (x[,"n_00"] + x[,"n_33"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(1 - 2*r_cc1 + r_cc1^2)) + (x[,"n_01"] + x[,"n_32"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_cc1)) + log(3*p1*(1 - r_cc1) + 2*(2 + r_cc1))) + (x[,"n_10"] + x[,"n_23"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_cc1)) + log(3*p2*(1 - r_cc1) + 2*(2 + r_cc1))) + (x[,"n_12"] + x[,"n_21"])*log(p2*(6 + 6*r_cc1 - 6*r_cc1^2) + 4*(5 + 2*r_cc1 + r_cc1^2) + 3*p1*(2 + 2*r_cc1 - 2*r_cc1^2 + 3*p2*(2 - 4*r_cc1 + r_cc1^2))) + (x[,"n_11"] + x[,"n_22"])*log(p2*(6 + 6*r_cc1 - 6*r_cc1^2) + 4*(8 - 4*r_cc1 + r_cc1^2) + 3*p1*(2 + 2*r_cc1 - 2*r_cc1^2 + 3*p2*(-1 + 2*r_cc1 + r_cc1^2)))


logL_cc2 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(5*log(2) + 2*log(3)) + (n03 + n30)*(log(2 - 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r))) + (n02 + n31)*(log(4 + 3*p2) + log(pmax(1e-6,r)) + log(6 + (-2 + 3*p1)*r)) + (n13 + n20)*(log(2 - 3*p1) + log(pmax(1e-6,r)) + log(12 - (4 + 3*p2)*r)) + (n00 + n33)*(log(2 - 3*p1) + log(4 + 3*p2) + log(1 - 2*r + r^2)) + (n01 + n32)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(3*p1*(1 - r) + 2*(2 + r))) + (n10 + n23)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(3*p2*(-1 + r) + 4*(2 + r))) + (n12 + n21)*log(6*p2*(-1 - r + r^2) + 8*(5 + 2*r + r^2) - 3*p1*(3*p2*(2 - 4*r + r^2) + 4*(-1 - r + r^2))) + (n11 + n22)*log(8*(8 - 4*r + r^2) + 6*p2*(-1 - r + r^2) - 3*p1*(4*(-1 - r + r^2) + 3*p2*(-1 + 2*r + r^2)))
return(L)}
interlogL_cc2 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_cc2,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc2 <- parallel::mcmapply(interlogL_cc2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_cc2 <- -((x[,"n_13"] + x[,"n_20"])*(-log10(2) + log10(2 - 3*p1) + log10(12 + (-4 - 3*p2)/2))) - (x[,"n_12"] + x[,"n_21"])*log10(50 - 3*p1*(-5 + (3*p2)/4) - (15*p2)/2) - (x[,"n_11"] + x[,"n_22"])*log10(50 - 3*p1*(-5 + (3*p2)/4) - (15*p2)/2) - (x[,"n_10"] + x[,"n_23"])*(-log10(2) + log10(2 - 3*p1) + log10(10 - (3*p2)/2)) - (x[,"n_03"] + x[,"n_30"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_33"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_32"])*(-log10(2) + log10(5 + (3*p1)/2) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_31"])*(-log10(2) + log10(6 + (-2 + 3*p1)/2) + log10(4 + 3*p2)) + (x[,"n_03"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_cc2))) + (x[,"n_02"] + x[,"n_31"])*(log10(4 + 3*p2) + log10(pmax(1e-6,r_cc2)) + log10(6 + (-2 + 3*p1)*r_cc2)) + (x[,"n_13"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(pmax(1e-6,r_cc2)) + log10(12 - (4 + 3*p2)*r_cc2)) + (x[,"n_00"] + x[,"n_33"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(1 - 2*r_cc2 + r_cc2^2)) + (x[,"n_01"] + x[,"n_32"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cc2)) + log10(3*p1*(1 - r_cc2) + 2*(2 + r_cc2))) + (x[,"n_10"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_cc2)) + log10(3*p2*(-1 + r_cc2) + 4*(2 + r_cc2))) + (x[,"n_12"] + x[,"n_21"])*log10(6*p2*(-1 - r_cc2 + r_cc2^2) + 8*(5 + 2*r_cc2 + r_cc2^2) - 3*p1*(3*p2*(2 - 4*r_cc2 + r_cc2^2) + 4*(-1 - r_cc2 + r_cc2^2))) + (x[,"n_11"] + x[,"n_22"])*log10(8*(8 - 4*r_cc2 + r_cc2^2) + 6*p2*(-1 - r_cc2 + r_cc2^2) - 3*p1*(4*(-1 - r_cc2 + r_cc2^2) + 3*p2*(-1 + 2*r_cc2 + r_cc2^2)))


logL_cc2 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(5*log(2) + 2*log(3)) + (x[,"n_03"] + x[,"n_30"])*(log(2 - 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r_cc2))) + (x[,"n_02"] + x[,"n_31"])*(log(4 + 3*p2) + log(pmax(1e-6,r_cc2)) + log(6 + (-2 + 3*p1)*r_cc2)) + (x[,"n_13"] + x[,"n_20"])*(log(2 - 3*p1) + log(pmax(1e-6,r_cc2)) + log(12 - (4 + 3*p2)*r_cc2)) + (x[,"n_00"] + x[,"n_33"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(1 - 2*r_cc2 + r_cc2^2)) + (x[,"n_01"] + x[,"n_32"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_cc2)) + log(3*p1*(1 - r_cc2) + 2*(2 + r_cc2))) + (x[,"n_10"] + x[,"n_23"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_cc2)) + log(3*p2*(-1 + r_cc2) + 4*(2 + r_cc2))) + (x[,"n_12"] + x[,"n_21"])*log(6*p2*(-1 - r_cc2 + r_cc2^2) + 8*(5 + 2*r_cc2 + r_cc2^2) - 3*p1*(3*p2*(2 - 4*r_cc2 + r_cc2^2) + 4*(-1 - r_cc2 + r_cc2^2))) + (x[,"n_11"] + x[,"n_22"])*log(8*(8 - 4*r_cc2 + r_cc2^2) + 6*p2*(-1 - r_cc2 + r_cc2^2) - 3*p1*(4*(-1 - r_cc2 + r_cc2^2) + 3*p2*(-1 + 2*r_cc2 + r_cc2^2)))


logL_cc3 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(5*log(2) + 2*log(3)) + (n03 + n30)*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r))) + (n02 + n31)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(12 - (4 + 3*p1)*r)) + (n13 + n20)*(log(4 + 3*p1) + log(pmax(1e-6,r)) + log(6 + (-2 + 3*p2)*r)) + (n00 + n33)*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r + r^2)) + (n10 + n23)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(3*p2*(1 - r) + 2*(2 + r))) + (n01 + n32)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(3*p1*(-1 + r) + 4*(2 + r))) + (n11 + n22)*log(4*(p2*(3 + 3*r - 3*r^2) + 2*(8 - 4*r + r^2)) - 3*p1*(2 + 2*r - 2*r^2 + 3*p2*(-1 + 2*r + r^2))) + (n12 + n21)*log(-3*p1*(2 + 2*r - 2*r^2 + 3*p2*(2 - 4*r + r^2)) + 4*(p2*(3 + 3*r - 3*r^2) + 2*(5 + 2*r + r^2)))
return(L)}
interlogL_cc3 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_cc3,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc3 <- parallel::mcmapply(interlogL_cc3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_cc3 <- -((x[,"n_02"] + x[,"n_31"])*(-log10(2) + log10(12 + (-4 - 3*p1)/2) + log10(2 - 3*p2))) - (x[,"n_01"] + x[,"n_32"])*(-log10(2) + log10(10 - (3*p1)/2) + log10(2 - 3*p2)) - (x[,"n_03"] + x[,"n_30"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_33"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_23"])*(-log10(2) + log10(4 + 3*p1) + log10(5 + (3*p2)/2)) - (x[,"n_13"] + x[,"n_20"])*(-log10(2) + log10(4 + 3*p1) + log10(6 + (-2 + 3*p2)/2)) - (x[,"n_12"] + x[,"n_21"])*log10(-3*p1*(5/2 + (3*p2)/4) + 4*(25/2 + (15*p2)/4)) - (x[,"n_11"] + x[,"n_22"])*log10(-3*p1*(5/2 + (3*p2)/4) + 4*(25/2 + (15*p2)/4)) + (x[,"n_03"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_cc3))) + (x[,"n_02"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_cc3)) + log10(12 - (4 + 3*p1)*r_cc3)) + (x[,"n_13"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(pmax(1e-6,r_cc3)) + log10(6 + (-2 + 3*p2)*r_cc3)) + (x[,"n_00"] + x[,"n_33"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(1 - 2*r_cc3 + r_cc3^2)) + (x[,"n_10"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_cc3)) + log10(3*p2*(1 - r_cc3) + 2*(2 + r_cc3))) + (x[,"n_01"] + x[,"n_32"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cc3)) + log10(3*p1*(-1 + r_cc3) + 4*(2 + r_cc3))) + (x[,"n_11"] + x[,"n_22"])*log10(4*(p2*(3 + 3*r_cc3 - 3*r_cc3^2) + 2*(8 - 4*r_cc3 + r_cc3^2)) - 3*p1*(2 + 2*r_cc3 - 2*r_cc3^2 + 3*p2*(-1 + 2*r_cc3 + r_cc3^2))) + (x[,"n_12"] + x[,"n_21"])*log10(-3*p1*(2 + 2*r_cc3 - 2*r_cc3^2 + 3*p2*(2 - 4*r_cc3 + r_cc3^2)) + 4*(p2*(3 + 3*r_cc3 - 3*r_cc3^2) + 2*(5 + 2*r_cc3 + r_cc3^2)))


logL_cc3 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(5*log(2) + 2*log(3)) + (x[,"n_03"] + x[,"n_30"])*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r_cc3))) + (x[,"n_02"] + x[,"n_31"])*(log(2 - 3*p2) + log(pmax(1e-6,r_cc3)) + log(12 - (4 + 3*p1)*r_cc3)) + (x[,"n_13"] + x[,"n_20"])*(log(4 + 3*p1) + log(pmax(1e-6,r_cc3)) + log(6 + (-2 + 3*p2)*r_cc3)) + (x[,"n_00"] + x[,"n_33"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r_cc3 + r_cc3^2)) + (x[,"n_10"] + x[,"n_23"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_cc3)) + log(3*p2*(1 - r_cc3) + 2*(2 + r_cc3))) + (x[,"n_01"] + x[,"n_32"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_cc3)) + log(3*p1*(-1 + r_cc3) + 4*(2 + r_cc3))) + (x[,"n_11"] + x[,"n_22"])*log(4*(p2*(3 + 3*r_cc3 - 3*r_cc3^2) + 2*(8 - 4*r_cc3 + r_cc3^2)) - 3*p1*(2 + 2*r_cc3 - 2*r_cc3^2 + 3*p2*(-1 + 2*r_cc3 + r_cc3^2))) + (x[,"n_12"] + x[,"n_21"])*log(-3*p1*(2 + 2*r_cc3 - 2*r_cc3^2 + 3*p2*(2 - 4*r_cc3 + r_cc3^2)) + 4*(p2*(3 + 3*r_cc3 - 3*r_cc3^2) + 2*(5 + 2*r_cc3 + r_cc3^2)))


logL_cc4 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(6*log(2) + 2*log(3)) + (n03 + n30)*(log(4 + 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r))) + (n02 + n31)*(log(4 + 3*p2) + log(pmax(1e-6,r)) + log(12 - (4 + 3*p1)*r)) + (n13 + n20)*(log(4 + 3*p1) + log(pmax(1e-6,r)) + log(12 - (4 + 3*p2)*r)) + (n00 + n33)*(log(4 + 3*p1) + log(4 + 3*p2) + log(1 - 2*r + r^2)) + (n01 + n32)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(3*p1*(-1 + r) + 4*(2 + r))) + (n10 + n23)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(3*p2*(-1 + r) + 4*(2 + r))) + (n11 + n22)*log(4*(4*(8 - 4*r + r^2) + 3*p2*(-1 - r + r^2)) + 3*p1*(4*(-1 - r + r^2) + 3*p2*(-1 + 2*r + r^2))) + (n12 + n21)*log(3*p1*(3*p2*(2 - 4*r + r^2) + 4*(-1 - r + r^2)) + 4*(3*p2*(-1 - r + r^2) + 4*(5 + 2*r + r^2)))
return(L)}
interlogL_cc4 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_cc4,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc4 <- parallel::mcmapply(interlogL_cc4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_cc4 <- -((x[,"n_13"] + x[,"n_20"])*(-log10(2) + log10(4 + 3*p1) + log10(12 + (-4 - 3*p2)/2))) - (x[,"n_12"] + x[,"n_21"])*log10(4*(25 - (15*p2)/4) + 3*p1*(-5 + (3*p2)/4)) - (x[,"n_11"] + x[,"n_22"])*log10(4*(25 - (15*p2)/4) + 3*p1*(-5 + (3*p2)/4)) - (x[,"n_10"] + x[,"n_23"])*(-log10(2) + log10(4 + 3*p1) + log10(10 - (3*p2)/2)) - (x[,"n_02"] + x[,"n_31"])*(-log10(2) + log10(12 + (-4 - 3*p1)/2) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_32"])*(-log10(2) + log10(10 - (3*p1)/2) + log10(4 + 3*p2)) - (x[,"n_03"] + x[,"n_30"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_33"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) + (x[,"n_03"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_cc4))) + (x[,"n_02"] + x[,"n_31"])*(log10(4 + 3*p2) + log10(pmax(1e-6,r_cc4)) + log10(12 - (4 + 3*p1)*r_cc4)) + (x[,"n_13"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(pmax(1e-6,r_cc4)) + log10(12 - (4 + 3*p2)*r_cc4)) + (x[,"n_00"] + x[,"n_33"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(1 - 2*r_cc4 + r_cc4^2)) + (x[,"n_01"] + x[,"n_32"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cc4)) + log10(3*p1*(-1 + r_cc4) + 4*(2 + r_cc4))) + (x[,"n_10"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_cc4)) + log10(3*p2*(-1 + r_cc4) + 4*(2 + r_cc4))) + (x[,"n_11"] + x[,"n_22"])*log10(4*(4*(8 - 4*r_cc4 + r_cc4^2) + 3*p2*(-1 - r_cc4 + r_cc4^2)) + 3*p1*(4*(-1 - r_cc4 + r_cc4^2) + 3*p2*(-1 + 2*r_cc4 + r_cc4^2))) + (x[,"n_12"] + x[,"n_21"])*log10(3*p1*(3*p2*(2 - 4*r_cc4 + r_cc4^2) + 4*(-1 - r_cc4 + r_cc4^2)) + 4*(3*p2*(-1 - r_cc4 + r_cc4^2) + 4*(5 + 2*r_cc4 + r_cc4^2)))


logL_cc4 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(6*log(2) + 2*log(3)) + (x[,"n_03"] + x[,"n_30"])*(log(4 + 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r_cc4))) + (x[,"n_02"] + x[,"n_31"])*(log(4 + 3*p2) + log(pmax(1e-6,r_cc4)) + log(12 - (4 + 3*p1)*r_cc4)) + (x[,"n_13"] + x[,"n_20"])*(log(4 + 3*p1) + log(pmax(1e-6,r_cc4)) + log(12 - (4 + 3*p2)*r_cc4)) + (x[,"n_00"] + x[,"n_33"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(1 - 2*r_cc4 + r_cc4^2)) + (x[,"n_01"] + x[,"n_32"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_cc4)) + log(3*p1*(-1 + r_cc4) + 4*(2 + r_cc4))) + (x[,"n_10"] + x[,"n_23"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_cc4)) + log(3*p2*(-1 + r_cc4) + 4*(2 + r_cc4))) + (x[,"n_11"] + x[,"n_22"])*log(4*(4*(8 - 4*r_cc4 + r_cc4^2) + 3*p2*(-1 - r_cc4 + r_cc4^2)) + 3*p1*(4*(-1 - r_cc4 + r_cc4^2) + 3*p2*(-1 + 2*r_cc4 + r_cc4^2))) + (x[,"n_12"] + x[,"n_21"])*log(3*p1*(3*p2*(2 - 4*r_cc4 + r_cc4^2) + 4*(-1 - r_cc4 + r_cc4^2)) + 4*(3*p2*(-1 - r_cc4 + r_cc4^2) + 4*(5 + 2*r_cc4 + r_cc4^2)))


logL_cr1 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(4*log(2) + 2*log(3)) + (n03 + n30)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n33)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n12 + n21)*(log(6 + (-2 + 3*p2)*r) + log(3*p1*(1 - r) + 2*(2 + r))) + (n11 + n22)*(log(6 + (-2 + 3*p1)*r) + log(3*p2*(1 - r) + 2*(2 + r))) + (n01 + n32)*(log(2 - 3*p2) + log(2*(1 + r^2) - 3*p1*(1 - 3*r + r^2))) + (n13 + n20)*(log(2 - 3*p1) + log(2*(1 + r^2) - 3*p2*(1 - 3*r + r^2))) + (n02 + n31)*(log(2 - 3*p2) + log(2*(2 - 2*r + r^2) - 3*p1*(-1 + r + r^2))) + (n10 + n23)*(log(2 - 3*p1) + log(2*(2 - 2*r + r^2) - 3*p2*(-1 + r + r^2)))
return(L)}
interlogL_cr1 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_cr1,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr1 <- parallel::mcmapply(interlogL_cr1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_cr1 <- -((x[,"n_03"] + x[,"n_30"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_33"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_31"])*(log10(5/2 + (3*p1)/4) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_32"])*(log10(5/2 + (3*p1)/4) + log10(2 - 3*p2)) - (x[,"n_13"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(5/2 + (3*p2)/4)) - (x[,"n_10"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(5/2 + (3*p2)/4)) - (x[,"n_11"] + x[,"n_22"])*(log10(6 + (-2 + 3*p1)/2) + log10(5 + (3*p2)/2)) - (x[,"n_12"] + x[,"n_21"])*(log10(5 + (3*p1)/2) + log10(6 + (-2 + 3*p2)/2)) + (x[,"n_03"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cr1)) + log10(pmax(1e-6,r_cr1))) + (x[,"n_00"] + x[,"n_33"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cr1)) + log10(pmax(1e-6,r_cr1))) + (x[,"n_12"] + x[,"n_21"])*(log10(6 + (-2 + 3*p2)*r_cr1) + log10(3*p1*(1 - r_cr1) + 2*(2 + r_cr1))) + (x[,"n_11"] + x[,"n_22"])*(log10(6 + (-2 + 3*p1)*r_cr1) + log10(3*p2*(1 - r_cr1) + 2*(2 + r_cr1))) + (x[,"n_01"] + x[,"n_32"])*(log10(2 - 3*p2) + log10(2*(1 + r_cr1^2) - 3*p1*(1 - 3*r_cr1 + r_cr1^2))) + (x[,"n_13"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(2*(1 + r_cr1^2) - 3*p2*(1 - 3*r_cr1 + r_cr1^2))) + (x[,"n_02"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(2*(2 - 2*r_cr1 + r_cr1^2) - 3*p1*(-1 + r_cr1 + r_cr1^2))) + (x[,"n_10"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(2*(2 - 2*r_cr1 + r_cr1^2) - 3*p2*(-1 + r_cr1 + r_cr1^2)))


logL_cr1 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(4*log(2) + 2*log(3)) + (x[,"n_03"] + x[,"n_30"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cr1)) + log(pmax(1e-6,r_cr1))) + (x[,"n_00"] + x[,"n_33"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cr1)) + log(pmax(1e-6,r_cr1))) + (x[,"n_12"] + x[,"n_21"])*(log(6 + (-2 + 3*p2)*r_cr1) + log(3*p1*(1 - r_cr1) + 2*(2 + r_cr1))) + (x[,"n_11"] + x[,"n_22"])*(log(6 + (-2 + 3*p1)*r_cr1) + log(3*p2*(1 - r_cr1) + 2*(2 + r_cr1))) + (x[,"n_01"] + x[,"n_32"])*(log(2 - 3*p2) + log(2*(1 + r_cr1^2) - 3*p1*(1 - 3*r_cr1 + r_cr1^2))) + (x[,"n_13"] + x[,"n_20"])*(log(2 - 3*p1) + log(2*(1 + r_cr1^2) - 3*p2*(1 - 3*r_cr1 + r_cr1^2))) + (x[,"n_02"] + x[,"n_31"])*(log(2 - 3*p2) + log(2*(2 - 2*r_cr1 + r_cr1^2) - 3*p1*(-1 + r_cr1 + r_cr1^2))) + (x[,"n_10"] + x[,"n_23"])*(log(2 - 3*p1) + log(2*(2 - 2*r_cr1 + r_cr1^2) - 3*p2*(-1 + r_cr1 + r_cr1^2)))


logL_cr2 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(5*log(2) + 2*log(3)) + (n00 + n03 + n30 + n33)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n12 + n21)*(log(12 - (4 + 3*p2)*r) + log(3*p1*(1 - r) + 2*(2 + r))) + (n11 + n22)*(log(6 + (-2 + 3*p1)*r) + log(3*p2*(-1 + r) + 4*(2 + r))) + (n01 + n32)*(log(4 + 3*p2) + log(2*(1 + r^2) - 3*p1*(1 - 3*r + r^2))) + (n13 + n20)*(log(2 - 3*p1) + log(4*(1 + r^2) + 3*p2*(1 - 3*r + r^2))) + (n02 + n31)*(log(4 + 3*p2) + log(2*(2 - 2*r + r^2) - 3*p1*(-1 + r + r^2))) + (n10 + n23)*(log(2 - 3*p1) + log(4*(2 - 2*r + r^2) + 3*p2*(-1 + r + r^2)))
return(L)}
interlogL_cr2 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_cr2,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr2 <- parallel::mcmapply(interlogL_cr2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_cr2 <- -((x[,"n_12"] + x[,"n_21"])*(log10(5 + (3*p1)/2) + log10(12 + (-4 - 3*p2)/2))) - (x[,"n_11"] + x[,"n_22"])*(log10(6 + (-2 + 3*p1)/2) + log10(10 - (3*p2)/2)) - (x[,"n_13"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(5 - (3*p2)/4)) - (x[,"n_10"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(5 - (3*p2)/4)) - (x[,"n_00"] + x[,"n_03"] + x[,"n_30"] + x[,"n_33"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_31"])*(log10(5/2 + (3*p1)/4) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_32"])*(log10(5/2 + (3*p1)/4) + log10(4 + 3*p2)) + (x[,"n_00"] + x[,"n_03"] + x[,"n_30"] + x[,"n_33"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cr2)) + log10(pmax(1e-6,r_cr2))) + (x[,"n_12"] + x[,"n_21"])*(log10(12 - (4 + 3*p2)*r_cr2) + log10(3*p1*(1 - r_cr2) + 2*(2 + r_cr2))) + (x[,"n_11"] + x[,"n_22"])*(log10(6 + (-2 + 3*p1)*r_cr2) + log10(3*p2*(-1 + r_cr2) + 4*(2 + r_cr2))) + (x[,"n_01"] + x[,"n_32"])*(log10(4 + 3*p2) + log10(2*(1 + r_cr2^2) - 3*p1*(1 - 3*r_cr2 + r_cr2^2))) + (x[,"n_13"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(4*(1 + r_cr2^2) + 3*p2*(1 - 3*r_cr2 + r_cr2^2))) + (x[,"n_02"] + x[,"n_31"])*(log10(4 + 3*p2) + log10(2*(2 - 2*r_cr2 + r_cr2^2) - 3*p1*(-1 + r_cr2 + r_cr2^2))) + (x[,"n_10"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(4*(2 - 2*r_cr2 + r_cr2^2) + 3*p2*(-1 + r_cr2 + r_cr2^2)))


logL_cr2 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(5*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_03"] + x[,"n_30"] + x[,"n_33"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cr2)) + log(pmax(1e-6,r_cr2))) + (x[,"n_12"] + x[,"n_21"])*(log(12 - (4 + 3*p2)*r_cr2) + log(3*p1*(1 - r_cr2) + 2*(2 + r_cr2))) + (x[,"n_11"] + x[,"n_22"])*(log(6 + (-2 + 3*p1)*r_cr2) + log(3*p2*(-1 + r_cr2) + 4*(2 + r_cr2))) + (x[,"n_01"] + x[,"n_32"])*(log(4 + 3*p2) + log(2*(1 + r_cr2^2) - 3*p1*(1 - 3*r_cr2 + r_cr2^2))) + (x[,"n_13"] + x[,"n_20"])*(log(2 - 3*p1) + log(4*(1 + r_cr2^2) + 3*p2*(1 - 3*r_cr2 + r_cr2^2))) + (x[,"n_02"] + x[,"n_31"])*(log(4 + 3*p2) + log(2*(2 - 2*r_cr2 + r_cr2^2) - 3*p1*(-1 + r_cr2 + r_cr2^2))) + (x[,"n_10"] + x[,"n_23"])*(log(2 - 3*p1) + log(4*(2 - 2*r_cr2 + r_cr2^2) + 3*p2*(-1 + r_cr2 + r_cr2^2)))


logL_cr3 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(5*log(2) + 2*log(3)) + (n00 + n03 + n30 + n33)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n11 + n22)*(log(12 - (4 + 3*p1)*r) + log(3*p2*(1 - r) + 2*(2 + r))) + (n12 + n21)*(log(6 + (-2 + 3*p2)*r) + log(3*p1*(-1 + r) + 4*(2 + r))) + (n01 + n32)*(log(2 - 3*p2) + log(4*(1 + r^2) + 3*p1*(1 - 3*r + r^2))) + (n13 + n20)*(log(4 + 3*p1) + log(2*(1 + r^2) - 3*p2*(1 - 3*r + r^2))) + (n02 + n31)*(log(2 - 3*p2) + log(4*(2 - 2*r + r^2) + 3*p1*(-1 + r + r^2))) + (n10 + n23)*(log(4 + 3*p1) + log(2*(2 - 2*r + r^2) - 3*p2*(-1 + r + r^2)))
return(L)}
interlogL_cr3 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_cr3,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr3 <- parallel::mcmapply(interlogL_cr3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_cr3 <- -((x[,"n_02"] + x[,"n_31"])*(log10(5 - (3*p1)/4) + log10(2 - 3*p2))) - (x[,"n_01"] + x[,"n_32"])*(log10(5 - (3*p1)/4) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_03"] + x[,"n_30"] + x[,"n_33"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_13"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(5/2 + (3*p2)/4)) - (x[,"n_10"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(5/2 + (3*p2)/4)) - (x[,"n_11"] + x[,"n_22"])*(log10(12 + (-4 - 3*p1)/2) + log10(5 + (3*p2)/2)) - (x[,"n_12"] + x[,"n_21"])*(log10(10 - (3*p1)/2) + log10(6 + (-2 + 3*p2)/2)) + (x[,"n_00"] + x[,"n_03"] + x[,"n_30"] + x[,"n_33"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cr3)) + log10(pmax(1e-6,r_cr3))) + (x[,"n_11"] + x[,"n_22"])*(log10(12 - (4 + 3*p1)*r_cr3) + log10(3*p2*(1 - r_cr3) + 2*(2 + r_cr3))) + (x[,"n_12"] + x[,"n_21"])*(log10(6 + (-2 + 3*p2)*r_cr3) + log10(3*p1*(-1 + r_cr3) + 4*(2 + r_cr3))) + (x[,"n_01"] + x[,"n_32"])*(log10(2 - 3*p2) + log10(4*(1 + r_cr3^2) + 3*p1*(1 - 3*r_cr3 + r_cr3^2))) + (x[,"n_13"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(2*(1 + r_cr3^2) - 3*p2*(1 - 3*r_cr3 + r_cr3^2))) + (x[,"n_02"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(4*(2 - 2*r_cr3 + r_cr3^2) + 3*p1*(-1 + r_cr3 + r_cr3^2))) + (x[,"n_10"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(2*(2 - 2*r_cr3 + r_cr3^2) - 3*p2*(-1 + r_cr3 + r_cr3^2)))


logL_cr3 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(5*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_03"] + x[,"n_30"] + x[,"n_33"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cr3)) + log(pmax(1e-6,r_cr3))) + (x[,"n_11"] + x[,"n_22"])*(log(12 - (4 + 3*p1)*r_cr3) + log(3*p2*(1 - r_cr3) + 2*(2 + r_cr3))) + (x[,"n_12"] + x[,"n_21"])*(log(6 + (-2 + 3*p2)*r_cr3) + log(3*p1*(-1 + r_cr3) + 4*(2 + r_cr3))) + (x[,"n_01"] + x[,"n_32"])*(log(2 - 3*p2) + log(4*(1 + r_cr3^2) + 3*p1*(1 - 3*r_cr3 + r_cr3^2))) + (x[,"n_13"] + x[,"n_20"])*(log(4 + 3*p1) + log(2*(1 + r_cr3^2) - 3*p2*(1 - 3*r_cr3 + r_cr3^2))) + (x[,"n_02"] + x[,"n_31"])*(log(2 - 3*p2) + log(4*(2 - 2*r_cr3 + r_cr3^2) + 3*p1*(-1 + r_cr3 + r_cr3^2))) + (x[,"n_10"] + x[,"n_23"])*(log(4 + 3*p1) + log(2*(2 - 2*r_cr3 + r_cr3^2) - 3*p2*(-1 + r_cr3 + r_cr3^2)))


logL_cr4 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(6*log(2) + 2*log(3)) + (n00 + n03 + n30 + n33)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n12 + n21)*(log(12 - (4 + 3*p2)*r) + log(3*p1*(-1 + r) + 4*(2 + r))) + (n11 + n22)*(log(12 - (4 + 3*p1)*r) + log(3*p2*(-1 + r) + 4*(2 + r))) + (n01 + n32)*(log(4 + 3*p2) + log(4*(1 + r^2) + 3*p1*(1 - 3*r + r^2))) + (n13 + n20)*(log(4 + 3*p1) + log(4*(1 + r^2) + 3*p2*(1 - 3*r + r^2))) + (n02 + n31)*(log(4 + 3*p2) + log(4*(2 - 2*r + r^2) + 3*p1*(-1 + r + r^2))) + (n10 + n23)*(log(4 + 3*p1) + log(4*(2 - 2*r + r^2) + 3*p2*(-1 + r + r^2)))
return(L)}
interlogL_cr4 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_cr4,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr4 <- parallel::mcmapply(interlogL_cr4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_cr4 <- -((x[,"n_12"] + x[,"n_21"])*(log10(10 - (3*p1)/2) + log10(12 + (-4 - 3*p2)/2))) - (x[,"n_11"] + x[,"n_22"])*(log10(12 + (-4 - 3*p1)/2) + log10(10 - (3*p2)/2)) - (x[,"n_13"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(5 - (3*p2)/4)) - (x[,"n_10"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(5 - (3*p2)/4)) - (x[,"n_02"] + x[,"n_31"])*(log10(5 - (3*p1)/4) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_32"])*(log10(5 - (3*p1)/4) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_03"] + x[,"n_30"] + x[,"n_33"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) + (x[,"n_00"] + x[,"n_03"] + x[,"n_30"] + x[,"n_33"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cr4)) + log10(pmax(1e-6,r_cr4))) + (x[,"n_12"] + x[,"n_21"])*(log10(12 - (4 + 3*p2)*r_cr4) + log10(3*p1*(-1 + r_cr4) + 4*(2 + r_cr4))) + (x[,"n_11"] + x[,"n_22"])*(log10(12 - (4 + 3*p1)*r_cr4) + log10(3*p2*(-1 + r_cr4) + 4*(2 + r_cr4))) + (x[,"n_01"] + x[,"n_32"])*(log10(4 + 3*p2) + log10(4*(1 + r_cr4^2) + 3*p1*(1 - 3*r_cr4 + r_cr4^2))) + (x[,"n_13"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(4*(1 + r_cr4^2) + 3*p2*(1 - 3*r_cr4 + r_cr4^2))) + (x[,"n_02"] + x[,"n_31"])*(log10(4 + 3*p2) + log10(4*(2 - 2*r_cr4 + r_cr4^2) + 3*p1*(-1 + r_cr4 + r_cr4^2))) + (x[,"n_10"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(4*(2 - 2*r_cr4 + r_cr4^2) + 3*p2*(-1 + r_cr4 + r_cr4^2)))


logL_cr4 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(6*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_03"] + x[,"n_30"] + x[,"n_33"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cr4)) + log(pmax(1e-6,r_cr4))) + (x[,"n_12"] + x[,"n_21"])*(log(12 - (4 + 3*p2)*r_cr4) + log(3*p1*(-1 + r_cr4) + 4*(2 + r_cr4))) + (x[,"n_11"] + x[,"n_22"])*(log(12 - (4 + 3*p1)*r_cr4) + log(3*p2*(-1 + r_cr4) + 4*(2 + r_cr4))) + (x[,"n_01"] + x[,"n_32"])*(log(4 + 3*p2) + log(4*(1 + r_cr4^2) + 3*p1*(1 - 3*r_cr4 + r_cr4^2))) + (x[,"n_13"] + x[,"n_20"])*(log(4 + 3*p1) + log(4*(1 + r_cr4^2) + 3*p2*(1 - 3*r_cr4 + r_cr4^2))) + (x[,"n_02"] + x[,"n_31"])*(log(4 + 3*p2) + log(4*(2 - 2*r_cr4 + r_cr4^2) + 3*p1*(-1 + r_cr4 + r_cr4^2))) + (x[,"n_10"] + x[,"n_23"])*(log(4 + 3*p1) + log(4*(2 - 2*r_cr4 + r_cr4^2) + 3*p2*(-1 + r_cr4 + r_cr4^2)))


logL_rc1 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(5*log(2) + 2*log(3)) + (n00 + n03 + n30 + n33)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n12 + n21)*(log(12 - (4 + 3*p1)*r) + log(3*p2*(1 - r) + 2*(2 + r))) + (n11 + n22)*(log(6 + (-2 + 3*p2)*r) + log(3*p1*(-1 + r) + 4*(2 + r))) + (n02 + n31)*(log(2 - 3*p2) + log(4*(1 + r^2) + 3*p1*(1 - 3*r + r^2))) + (n10 + n23)*(log(4 + 3*p1) + log(2*(1 + r^2) - 3*p2*(1 - 3*r + r^2))) + (n01 + n32)*(log(2 - 3*p2) + log(4*(2 - 2*r + r^2) + 3*p1*(-1 + r + r^2))) + (n13 + n20)*(log(4 + 3*p1) + log(2*(2 - 2*r + r^2) - 3*p2*(-1 + r + r^2)))
return(L)}
interlogL_rc1 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rc1,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc1 <- parallel::mcmapply(interlogL_rc1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rc1 <- -((x[,"n_02"] + x[,"n_31"])*(log10(5 - (3*p1)/4) + log10(2 - 3*p2))) - (x[,"n_01"] + x[,"n_32"])*(log10(5 - (3*p1)/4) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_03"] + x[,"n_30"] + x[,"n_33"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_13"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(5/2 + (3*p2)/4)) - (x[,"n_10"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(5/2 + (3*p2)/4)) - (x[,"n_12"] + x[,"n_21"])*(log10(12 + (-4 - 3*p1)/2) + log10(5 + (3*p2)/2)) - (x[,"n_11"] + x[,"n_22"])*(log10(10 - (3*p1)/2) + log10(6 + (-2 + 3*p2)/2)) + (x[,"n_00"] + x[,"n_03"] + x[,"n_30"] + x[,"n_33"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rc1)) + log10(pmax(1e-6,r_rc1))) + (x[,"n_12"] + x[,"n_21"])*(log10(12 - (4 + 3*p1)*r_rc1) + log10(3*p2*(1 - r_rc1) + 2*(2 + r_rc1))) + (x[,"n_11"] + x[,"n_22"])*(log10(6 + (-2 + 3*p2)*r_rc1) + log10(3*p1*(-1 + r_rc1) + 4*(2 + r_rc1))) + (x[,"n_02"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(4*(1 + r_rc1^2) + 3*p1*(1 - 3*r_rc1 + r_rc1^2))) + (x[,"n_10"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(2*(1 + r_rc1^2) - 3*p2*(1 - 3*r_rc1 + r_rc1^2))) + (x[,"n_01"] + x[,"n_32"])*(log10(2 - 3*p2) + log10(4*(2 - 2*r_rc1 + r_rc1^2) + 3*p1*(-1 + r_rc1 + r_rc1^2))) + (x[,"n_13"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(2*(2 - 2*r_rc1 + r_rc1^2) - 3*p2*(-1 + r_rc1 + r_rc1^2)))


logL_rc1 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(5*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_03"] + x[,"n_30"] + x[,"n_33"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rc1)) + log(pmax(1e-6,r_rc1))) + (x[,"n_12"] + x[,"n_21"])*(log(12 - (4 + 3*p1)*r_rc1) + log(3*p2*(1 - r_rc1) + 2*(2 + r_rc1))) + (x[,"n_11"] + x[,"n_22"])*(log(6 + (-2 + 3*p2)*r_rc1) + log(3*p1*(-1 + r_rc1) + 4*(2 + r_rc1))) + (x[,"n_02"] + x[,"n_31"])*(log(2 - 3*p2) + log(4*(1 + r_rc1^2) + 3*p1*(1 - 3*r_rc1 + r_rc1^2))) + (x[,"n_10"] + x[,"n_23"])*(log(4 + 3*p1) + log(2*(1 + r_rc1^2) - 3*p2*(1 - 3*r_rc1 + r_rc1^2))) + (x[,"n_01"] + x[,"n_32"])*(log(2 - 3*p2) + log(4*(2 - 2*r_rc1 + r_rc1^2) + 3*p1*(-1 + r_rc1 + r_rc1^2))) + (x[,"n_13"] + x[,"n_20"])*(log(4 + 3*p1) + log(2*(2 - 2*r_rc1 + r_rc1^2) - 3*p2*(-1 + r_rc1 + r_rc1^2)))


logL_rc2 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(6*log(2) + 2*log(3)) + (n00 + n03 + n30 + n33)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n11 + n22)*(log(12 - (4 + 3*p2)*r) + log(3*p1*(-1 + r) + 4*(2 + r))) + (n12 + n21)*(log(12 - (4 + 3*p1)*r) + log(3*p2*(-1 + r) + 4*(2 + r))) + (n02 + n31)*(log(4 + 3*p2) + log(4*(1 + r^2) + 3*p1*(1 - 3*r + r^2))) + (n10 + n23)*(log(4 + 3*p1) + log(4*(1 + r^2) + 3*p2*(1 - 3*r + r^2))) + (n01 + n32)*(log(4 + 3*p2) + log(4*(2 - 2*r + r^2) + 3*p1*(-1 + r + r^2))) + (n13 + n20)*(log(4 + 3*p1) + log(4*(2 - 2*r + r^2) + 3*p2*(-1 + r + r^2)))
return(L)}
interlogL_rc2 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rc2,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc2 <- parallel::mcmapply(interlogL_rc2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rc2 <- -((x[,"n_11"] + x[,"n_22"])*(log10(10 - (3*p1)/2) + log10(12 + (-4 - 3*p2)/2))) - (x[,"n_12"] + x[,"n_21"])*(log10(12 + (-4 - 3*p1)/2) + log10(10 - (3*p2)/2)) - (x[,"n_13"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(5 - (3*p2)/4)) - (x[,"n_10"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(5 - (3*p2)/4)) - (x[,"n_02"] + x[,"n_31"])*(log10(5 - (3*p1)/4) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_32"])*(log10(5 - (3*p1)/4) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_03"] + x[,"n_30"] + x[,"n_33"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) + (x[,"n_00"] + x[,"n_03"] + x[,"n_30"] + x[,"n_33"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rc2)) + log10(pmax(1e-6,r_rc2))) + (x[,"n_11"] + x[,"n_22"])*(log10(12 - (4 + 3*p2)*r_rc2) + log10(3*p1*(-1 + r_rc2) + 4*(2 + r_rc2))) + (x[,"n_12"] + x[,"n_21"])*(log10(12 - (4 + 3*p1)*r_rc2) + log10(3*p2*(-1 + r_rc2) + 4*(2 + r_rc2))) + (x[,"n_02"] + x[,"n_31"])*(log10(4 + 3*p2) + log10(4*(1 + r_rc2^2) + 3*p1*(1 - 3*r_rc2 + r_rc2^2))) + (x[,"n_10"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(4*(1 + r_rc2^2) + 3*p2*(1 - 3*r_rc2 + r_rc2^2))) + (x[,"n_01"] + x[,"n_32"])*(log10(4 + 3*p2) + log10(4*(2 - 2*r_rc2 + r_rc2^2) + 3*p1*(-1 + r_rc2 + r_rc2^2))) + (x[,"n_13"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(4*(2 - 2*r_rc2 + r_rc2^2) + 3*p2*(-1 + r_rc2 + r_rc2^2)))


logL_rc2 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(6*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_03"] + x[,"n_30"] + x[,"n_33"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rc2)) + log(pmax(1e-6,r_rc2))) + (x[,"n_11"] + x[,"n_22"])*(log(12 - (4 + 3*p2)*r_rc2) + log(3*p1*(-1 + r_rc2) + 4*(2 + r_rc2))) + (x[,"n_12"] + x[,"n_21"])*(log(12 - (4 + 3*p1)*r_rc2) + log(3*p2*(-1 + r_rc2) + 4*(2 + r_rc2))) + (x[,"n_02"] + x[,"n_31"])*(log(4 + 3*p2) + log(4*(1 + r_rc2^2) + 3*p1*(1 - 3*r_rc2 + r_rc2^2))) + (x[,"n_10"] + x[,"n_23"])*(log(4 + 3*p1) + log(4*(1 + r_rc2^2) + 3*p2*(1 - 3*r_rc2 + r_rc2^2))) + (x[,"n_01"] + x[,"n_32"])*(log(4 + 3*p2) + log(4*(2 - 2*r_rc2 + r_rc2^2) + 3*p1*(-1 + r_rc2 + r_rc2^2))) + (x[,"n_13"] + x[,"n_20"])*(log(4 + 3*p1) + log(4*(2 - 2*r_rc2 + r_rc2^2) + 3*p2*(-1 + r_rc2 + r_rc2^2)))


logL_rc3 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(4*log(2) + 2*log(3)) + (n03 + n30)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n33)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n11 + n22)*(log(6 + (-2 + 3*p2)*r) + log(3*p1*(1 - r) + 2*(2 + r))) + (n12 + n21)*(log(6 + (-2 + 3*p1)*r) + log(3*p2*(1 - r) + 2*(2 + r))) + (n02 + n31)*(log(2 - 3*p2) + log(2*(1 + r^2) - 3*p1*(1 - 3*r + r^2))) + (n10 + n23)*(log(2 - 3*p1) + log(2*(1 + r^2) - 3*p2*(1 - 3*r + r^2))) + (n01 + n32)*(log(2 - 3*p2) + log(2*(2 - 2*r + r^2) - 3*p1*(-1 + r + r^2))) + (n13 + n20)*(log(2 - 3*p1) + log(2*(2 - 2*r + r^2) - 3*p2*(-1 + r + r^2)))
return(L)}
interlogL_rc3 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rc3,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc3 <- parallel::mcmapply(interlogL_rc3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rc3 <- -((x[,"n_03"] + x[,"n_30"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_33"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_31"])*(log10(5/2 + (3*p1)/4) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_32"])*(log10(5/2 + (3*p1)/4) + log10(2 - 3*p2)) - (x[,"n_13"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(5/2 + (3*p2)/4)) - (x[,"n_10"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(5/2 + (3*p2)/4)) - (x[,"n_12"] + x[,"n_21"])*(log10(6 + (-2 + 3*p1)/2) + log10(5 + (3*p2)/2)) - (x[,"n_11"] + x[,"n_22"])*(log10(5 + (3*p1)/2) + log10(6 + (-2 + 3*p2)/2)) + (x[,"n_03"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rc3)) + log10(pmax(1e-6,r_rc3))) + (x[,"n_00"] + x[,"n_33"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rc3)) + log10(pmax(1e-6,r_rc3))) + (x[,"n_11"] + x[,"n_22"])*(log10(6 + (-2 + 3*p2)*r_rc3) + log10(3*p1*(1 - r_rc3) + 2*(2 + r_rc3))) + (x[,"n_12"] + x[,"n_21"])*(log10(6 + (-2 + 3*p1)*r_rc3) + log10(3*p2*(1 - r_rc3) + 2*(2 + r_rc3))) + (x[,"n_02"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(2*(1 + r_rc3^2) - 3*p1*(1 - 3*r_rc3 + r_rc3^2))) + (x[,"n_10"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(2*(1 + r_rc3^2) - 3*p2*(1 - 3*r_rc3 + r_rc3^2))) + (x[,"n_01"] + x[,"n_32"])*(log10(2 - 3*p2) + log10(2*(2 - 2*r_rc3 + r_rc3^2) - 3*p1*(-1 + r_rc3 + r_rc3^2))) + (x[,"n_13"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(2*(2 - 2*r_rc3 + r_rc3^2) - 3*p2*(-1 + r_rc3 + r_rc3^2)))


logL_rc3 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(4*log(2) + 2*log(3)) + (x[,"n_03"] + x[,"n_30"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rc3)) + log(pmax(1e-6,r_rc3))) + (x[,"n_00"] + x[,"n_33"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rc3)) + log(pmax(1e-6,r_rc3))) + (x[,"n_11"] + x[,"n_22"])*(log(6 + (-2 + 3*p2)*r_rc3) + log(3*p1*(1 - r_rc3) + 2*(2 + r_rc3))) + (x[,"n_12"] + x[,"n_21"])*(log(6 + (-2 + 3*p1)*r_rc3) + log(3*p2*(1 - r_rc3) + 2*(2 + r_rc3))) + (x[,"n_02"] + x[,"n_31"])*(log(2 - 3*p2) + log(2*(1 + r_rc3^2) - 3*p1*(1 - 3*r_rc3 + r_rc3^2))) + (x[,"n_10"] + x[,"n_23"])*(log(2 - 3*p1) + log(2*(1 + r_rc3^2) - 3*p2*(1 - 3*r_rc3 + r_rc3^2))) + (x[,"n_01"] + x[,"n_32"])*(log(2 - 3*p2) + log(2*(2 - 2*r_rc3 + r_rc3^2) - 3*p1*(-1 + r_rc3 + r_rc3^2))) + (x[,"n_13"] + x[,"n_20"])*(log(2 - 3*p1) + log(2*(2 - 2*r_rc3 + r_rc3^2) - 3*p2*(-1 + r_rc3 + r_rc3^2)))


logL_rc4 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(5*log(2) + 2*log(3)) + (n00 + n03 + n30 + n33)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n11 + n22)*(log(12 - (4 + 3*p2)*r) + log(3*p1*(1 - r) + 2*(2 + r))) + (n12 + n21)*(log(6 + (-2 + 3*p1)*r) + log(3*p2*(-1 + r) + 4*(2 + r))) + (n02 + n31)*(log(4 + 3*p2) + log(2*(1 + r^2) - 3*p1*(1 - 3*r + r^2))) + (n10 + n23)*(log(2 - 3*p1) + log(4*(1 + r^2) + 3*p2*(1 - 3*r + r^2))) + (n01 + n32)*(log(4 + 3*p2) + log(2*(2 - 2*r + r^2) - 3*p1*(-1 + r + r^2))) + (n13 + n20)*(log(2 - 3*p1) + log(4*(2 - 2*r + r^2) + 3*p2*(-1 + r + r^2)))
return(L)}
interlogL_rc4 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rc4,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc4 <- parallel::mcmapply(interlogL_rc4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rc4 <- -((x[,"n_11"] + x[,"n_22"])*(log10(5 + (3*p1)/2) + log10(12 + (-4 - 3*p2)/2))) - (x[,"n_12"] + x[,"n_21"])*(log10(6 + (-2 + 3*p1)/2) + log10(10 - (3*p2)/2)) - (x[,"n_13"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(5 - (3*p2)/4)) - (x[,"n_10"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(5 - (3*p2)/4)) - (x[,"n_00"] + x[,"n_03"] + x[,"n_30"] + x[,"n_33"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_31"])*(log10(5/2 + (3*p1)/4) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_32"])*(log10(5/2 + (3*p1)/4) + log10(4 + 3*p2)) + (x[,"n_00"] + x[,"n_03"] + x[,"n_30"] + x[,"n_33"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rc4)) + log10(pmax(1e-6,r_rc4))) + (x[,"n_11"] + x[,"n_22"])*(log10(12 - (4 + 3*p2)*r_rc4) + log10(3*p1*(1 - r_rc4) + 2*(2 + r_rc4))) + (x[,"n_12"] + x[,"n_21"])*(log10(6 + (-2 + 3*p1)*r_rc4) + log10(3*p2*(-1 + r_rc4) + 4*(2 + r_rc4))) + (x[,"n_02"] + x[,"n_31"])*(log10(4 + 3*p2) + log10(2*(1 + r_rc4^2) - 3*p1*(1 - 3*r_rc4 + r_rc4^2))) + (x[,"n_10"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(4*(1 + r_rc4^2) + 3*p2*(1 - 3*r_rc4 + r_rc4^2))) + (x[,"n_01"] + x[,"n_32"])*(log10(4 + 3*p2) + log10(2*(2 - 2*r_rc4 + r_rc4^2) - 3*p1*(-1 + r_rc4 + r_rc4^2))) + (x[,"n_13"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(4*(2 - 2*r_rc4 + r_rc4^2) + 3*p2*(-1 + r_rc4 + r_rc4^2)))


logL_rc4 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(5*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_03"] + x[,"n_30"] + x[,"n_33"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rc4)) + log(pmax(1e-6,r_rc4))) + (x[,"n_11"] + x[,"n_22"])*(log(12 - (4 + 3*p2)*r_rc4) + log(3*p1*(1 - r_rc4) + 2*(2 + r_rc4))) + (x[,"n_12"] + x[,"n_21"])*(log(6 + (-2 + 3*p1)*r_rc4) + log(3*p2*(-1 + r_rc4) + 4*(2 + r_rc4))) + (x[,"n_02"] + x[,"n_31"])*(log(4 + 3*p2) + log(2*(1 + r_rc4^2) - 3*p1*(1 - 3*r_rc4 + r_rc4^2))) + (x[,"n_10"] + x[,"n_23"])*(log(2 - 3*p1) + log(4*(1 + r_rc4^2) + 3*p2*(1 - 3*r_rc4 + r_rc4^2))) + (x[,"n_01"] + x[,"n_32"])*(log(4 + 3*p2) + log(2*(2 - 2*r_rc4 + r_rc4^2) - 3*p1*(-1 + r_rc4 + r_rc4^2))) + (x[,"n_13"] + x[,"n_20"])*(log(2 - 3*p1) + log(4*(2 - 2*r_rc4 + r_rc4^2) + 3*p2*(-1 + r_rc4 + r_rc4^2)))


logL_rr1 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(5*log(2) + 2*log(3)) + (n00 + n33)*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r))) + (n01 + n32)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(12 - (4 + 3*p1)*r)) + (n10 + n23)*(log(4 + 3*p1) + log(pmax(1e-6,r)) + log(6 + (-2 + 3*p2)*r)) + (n03 + n30)*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r + r^2)) + (n13 + n20)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(3*p2*(1 - r) + 2*(2 + r))) + (n02 + n31)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(3*p1*(-1 + r) + 4*(2 + r))) + (n12 + n21)*log(4*(p2*(3 + 3*r - 3*r^2) + 2*(8 - 4*r + r^2)) - 3*p1*(2 + 2*r - 2*r^2 + 3*p2*(-1 + 2*r + r^2))) + (n11 + n22)*log(-3*p1*(2 + 2*r - 2*r^2 + 3*p2*(2 - 4*r + r^2)) + 4*(p2*(3 + 3*r - 3*r^2) + 2*(5 + 2*r + r^2)))
return(L)}
interlogL_rr1 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rr1,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr1 <- parallel::mcmapply(interlogL_rr1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rr1 <- -((x[,"n_01"] + x[,"n_32"])*(-log10(2) + log10(12 + (-4 - 3*p1)/2) + log10(2 - 3*p2))) - (x[,"n_02"] + x[,"n_31"])*(-log10(2) + log10(10 - (3*p1)/2) + log10(2 - 3*p2)) - (x[,"n_03"] + x[,"n_30"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_33"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_13"] + x[,"n_20"])*(-log10(2) + log10(4 + 3*p1) + log10(5 + (3*p2)/2)) - (x[,"n_10"] + x[,"n_23"])*(-log10(2) + log10(4 + 3*p1) + log10(6 + (-2 + 3*p2)/2)) - (x[,"n_12"] + x[,"n_21"])*log10(-3*p1*(5/2 + (3*p2)/4) + 4*(25/2 + (15*p2)/4)) - (x[,"n_11"] + x[,"n_22"])*log10(-3*p1*(5/2 + (3*p2)/4) + 4*(25/2 + (15*p2)/4)) + (x[,"n_00"] + x[,"n_33"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_rr1))) + (x[,"n_01"] + x[,"n_32"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_rr1)) + log10(12 - (4 + 3*p1)*r_rr1)) + (x[,"n_10"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(pmax(1e-6,r_rr1)) + log10(6 + (-2 + 3*p2)*r_rr1)) + (x[,"n_03"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(1 - 2*r_rr1 + r_rr1^2)) + (x[,"n_13"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_rr1)) + log10(3*p2*(1 - r_rr1) + 2*(2 + r_rr1))) + (x[,"n_02"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rr1)) + log10(3*p1*(-1 + r_rr1) + 4*(2 + r_rr1))) + (x[,"n_12"] + x[,"n_21"])*log10(4*(p2*(3 + 3*r_rr1 - 3*r_rr1^2) + 2*(8 - 4*r_rr1 + r_rr1^2)) - 3*p1*(2 + 2*r_rr1 - 2*r_rr1^2 + 3*p2*(-1 + 2*r_rr1 + r_rr1^2))) + (x[,"n_11"] + x[,"n_22"])*log10(-3*p1*(2 + 2*r_rr1 - 2*r_rr1^2 + 3*p2*(2 - 4*r_rr1 + r_rr1^2)) + 4*(p2*(3 + 3*r_rr1 - 3*r_rr1^2) + 2*(5 + 2*r_rr1 + r_rr1^2)))


logL_rr1 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(5*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_33"])*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r_rr1))) + (x[,"n_01"] + x[,"n_32"])*(log(2 - 3*p2) + log(pmax(1e-6,r_rr1)) + log(12 - (4 + 3*p1)*r_rr1)) + (x[,"n_10"] + x[,"n_23"])*(log(4 + 3*p1) + log(pmax(1e-6,r_rr1)) + log(6 + (-2 + 3*p2)*r_rr1)) + (x[,"n_03"] + x[,"n_30"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r_rr1 + r_rr1^2)) + (x[,"n_13"] + x[,"n_20"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_rr1)) + log(3*p2*(1 - r_rr1) + 2*(2 + r_rr1))) + (x[,"n_02"] + x[,"n_31"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_rr1)) + log(3*p1*(-1 + r_rr1) + 4*(2 + r_rr1))) + (x[,"n_12"] + x[,"n_21"])*log(4*(p2*(3 + 3*r_rr1 - 3*r_rr1^2) + 2*(8 - 4*r_rr1 + r_rr1^2)) - 3*p1*(2 + 2*r_rr1 - 2*r_rr1^2 + 3*p2*(-1 + 2*r_rr1 + r_rr1^2))) + (x[,"n_11"] + x[,"n_22"])*log(-3*p1*(2 + 2*r_rr1 - 2*r_rr1^2 + 3*p2*(2 - 4*r_rr1 + r_rr1^2)) + 4*(p2*(3 + 3*r_rr1 - 3*r_rr1^2) + 2*(5 + 2*r_rr1 + r_rr1^2)))


logL_rr2 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(6*log(2) + 2*log(3)) + (n00 + n33)*(log(4 + 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r))) + (n01 + n32)*(log(4 + 3*p2) + log(pmax(1e-6,r)) + log(12 - (4 + 3*p1)*r)) + (n10 + n23)*(log(4 + 3*p1) + log(pmax(1e-6,r)) + log(12 - (4 + 3*p2)*r)) + (n03 + n30)*(log(4 + 3*p1) + log(4 + 3*p2) + log(1 - 2*r + r^2)) + (n02 + n31)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(3*p1*(-1 + r) + 4*(2 + r))) + (n13 + n20)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(3*p2*(-1 + r) + 4*(2 + r))) + (n12 + n21)*log(4*(4*(8 - 4*r + r^2) + 3*p2*(-1 - r + r^2)) + 3*p1*(4*(-1 - r + r^2) + 3*p2*(-1 + 2*r + r^2))) + (n11 + n22)*log(3*p1*(3*p2*(2 - 4*r + r^2) + 4*(-1 - r + r^2)) + 4*(3*p2*(-1 - r + r^2) + 4*(5 + 2*r + r^2)))
return(L)}
interlogL_rr2 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rr2,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr2 <- parallel::mcmapply(interlogL_rr2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rr2 <- -((x[,"n_10"] + x[,"n_23"])*(-log10(2) + log10(4 + 3*p1) + log10(12 + (-4 - 3*p2)/2))) - (x[,"n_12"] + x[,"n_21"])*log10(4*(25 - (15*p2)/4) + 3*p1*(-5 + (3*p2)/4)) - (x[,"n_11"] + x[,"n_22"])*log10(4*(25 - (15*p2)/4) + 3*p1*(-5 + (3*p2)/4)) - (x[,"n_13"] + x[,"n_20"])*(-log10(2) + log10(4 + 3*p1) + log10(10 - (3*p2)/2)) - (x[,"n_01"] + x[,"n_32"])*(-log10(2) + log10(12 + (-4 - 3*p1)/2) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_31"])*(-log10(2) + log10(10 - (3*p1)/2) + log10(4 + 3*p2)) - (x[,"n_03"] + x[,"n_30"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_33"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) + (x[,"n_00"] + x[,"n_33"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_rr2))) + (x[,"n_01"] + x[,"n_32"])*(log10(4 + 3*p2) + log10(pmax(1e-6,r_rr2)) + log10(12 - (4 + 3*p1)*r_rr2)) + (x[,"n_10"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(pmax(1e-6,r_rr2)) + log10(12 - (4 + 3*p2)*r_rr2)) + (x[,"n_03"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(1 - 2*r_rr2 + r_rr2^2)) + (x[,"n_02"] + x[,"n_31"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rr2)) + log10(3*p1*(-1 + r_rr2) + 4*(2 + r_rr2))) + (x[,"n_13"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_rr2)) + log10(3*p2*(-1 + r_rr2) + 4*(2 + r_rr2))) + (x[,"n_12"] + x[,"n_21"])*log10(4*(4*(8 - 4*r_rr2 + r_rr2^2) + 3*p2*(-1 - r_rr2 + r_rr2^2)) + 3*p1*(4*(-1 - r_rr2 + r_rr2^2) + 3*p2*(-1 + 2*r_rr2 + r_rr2^2))) + (x[,"n_11"] + x[,"n_22"])*log10(3*p1*(3*p2*(2 - 4*r_rr2 + r_rr2^2) + 4*(-1 - r_rr2 + r_rr2^2)) + 4*(3*p2*(-1 - r_rr2 + r_rr2^2) + 4*(5 + 2*r_rr2 + r_rr2^2)))


logL_rr2 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(6*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_33"])*(log(4 + 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r_rr2))) + (x[,"n_01"] + x[,"n_32"])*(log(4 + 3*p2) + log(pmax(1e-6,r_rr2)) + log(12 - (4 + 3*p1)*r_rr2)) + (x[,"n_10"] + x[,"n_23"])*(log(4 + 3*p1) + log(pmax(1e-6,r_rr2)) + log(12 - (4 + 3*p2)*r_rr2)) + (x[,"n_03"] + x[,"n_30"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(1 - 2*r_rr2 + r_rr2^2)) + (x[,"n_02"] + x[,"n_31"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_rr2)) + log(3*p1*(-1 + r_rr2) + 4*(2 + r_rr2))) + (x[,"n_13"] + x[,"n_20"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_rr2)) + log(3*p2*(-1 + r_rr2) + 4*(2 + r_rr2))) + (x[,"n_12"] + x[,"n_21"])*log(4*(4*(8 - 4*r_rr2 + r_rr2^2) + 3*p2*(-1 - r_rr2 + r_rr2^2)) + 3*p1*(4*(-1 - r_rr2 + r_rr2^2) + 3*p2*(-1 + 2*r_rr2 + r_rr2^2))) + (x[,"n_11"] + x[,"n_22"])*log(3*p1*(3*p2*(2 - 4*r_rr2 + r_rr2^2) + 4*(-1 - r_rr2 + r_rr2^2)) + 4*(3*p2*(-1 - r_rr2 + r_rr2^2) + 4*(5 + 2*r_rr2 + r_rr2^2)))


logL_rr3 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(4*log(2) + 2*log(3)) + (n00 + n33)*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r))) + (n01 + n32)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(6 + (-2 + 3*p1)*r)) + (n10 + n23)*(log(2 - 3*p1) + log(pmax(1e-6,r)) + log(6 + (-2 + 3*p2)*r)) + (n03 + n30)*(log(2 - 3*p1) + log(2 - 3*p2) + log(1 - 2*r + r^2)) + (n02 + n31)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(3*p1*(1 - r) + 2*(2 + r))) + (n13 + n20)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(3*p2*(1 - r) + 2*(2 + r))) + (n11 + n22)*log(p2*(6 + 6*r - 6*r^2) + 4*(5 + 2*r + r^2) + 3*p1*(2 + 2*r - 2*r^2 + 3*p2*(2 - 4*r + r^2))) + (n12 + n21)*log(p2*(6 + 6*r - 6*r^2) + 4*(8 - 4*r + r^2) + 3*p1*(2 + 2*r - 2*r^2 + 3*p2*(-1 + 2*r + r^2)))
return(L)}
interlogL_rr3 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rr3,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr3 <- parallel::mcmapply(interlogL_rr3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rr3 <- -((x[,"n_03"] + x[,"n_30"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_33"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_31"])*(-log10(2) + log10(5 + (3*p1)/2) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_32"])*(-log10(2) + log10(6 + (-2 + 3*p1)/2) + log10(2 - 3*p2)) - (x[,"n_13"] + x[,"n_20"])*(-log10(2) + log10(2 - 3*p1) + log10(5 + (3*p2)/2)) - (x[,"n_12"] + x[,"n_21"])*log10(25 + 3*p1*(5/2 + (3*p2)/4) + (15*p2)/2) - (x[,"n_11"] + x[,"n_22"])*log10(25 + 3*p1*(5/2 + (3*p2)/4) + (15*p2)/2) - (x[,"n_10"] + x[,"n_23"])*(-log10(2) + log10(2 - 3*p1) + log10(6 + (-2 + 3*p2)/2)) + (x[,"n_00"] + x[,"n_33"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_rr3))) + (x[,"n_01"] + x[,"n_32"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_rr3)) + log10(6 + (-2 + 3*p1)*r_rr3)) + (x[,"n_10"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(pmax(1e-6,r_rr3)) + log10(6 + (-2 + 3*p2)*r_rr3)) + (x[,"n_03"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(1 - 2*r_rr3 + r_rr3^2)) + (x[,"n_02"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rr3)) + log10(3*p1*(1 - r_rr3) + 2*(2 + r_rr3))) + (x[,"n_13"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_rr3)) + log10(3*p2*(1 - r_rr3) + 2*(2 + r_rr3))) + (x[,"n_11"] + x[,"n_22"])*log10(p2*(6 + 6*r_rr3 - 6*r_rr3^2) + 4*(5 + 2*r_rr3 + r_rr3^2) + 3*p1*(2 + 2*r_rr3 - 2*r_rr3^2 + 3*p2*(2 - 4*r_rr3 + r_rr3^2))) + (x[,"n_12"] + x[,"n_21"])*log10(p2*(6 + 6*r_rr3 - 6*r_rr3^2) + 4*(8 - 4*r_rr3 + r_rr3^2) + 3*p1*(2 + 2*r_rr3 - 2*r_rr3^2 + 3*p2*(-1 + 2*r_rr3 + r_rr3^2)))


logL_rr3 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(4*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_33"])*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r_rr3))) + (x[,"n_01"] + x[,"n_32"])*(log(2 - 3*p2) + log(pmax(1e-6,r_rr3)) + log(6 + (-2 + 3*p1)*r_rr3)) + (x[,"n_10"] + x[,"n_23"])*(log(2 - 3*p1) + log(pmax(1e-6,r_rr3)) + log(6 + (-2 + 3*p2)*r_rr3)) + (x[,"n_03"] + x[,"n_30"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(1 - 2*r_rr3 + r_rr3^2)) + (x[,"n_02"] + x[,"n_31"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_rr3)) + log(3*p1*(1 - r_rr3) + 2*(2 + r_rr3))) + (x[,"n_13"] + x[,"n_20"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_rr3)) + log(3*p2*(1 - r_rr3) + 2*(2 + r_rr3))) + (x[,"n_11"] + x[,"n_22"])*log(p2*(6 + 6*r_rr3 - 6*r_rr3^2) + 4*(5 + 2*r_rr3 + r_rr3^2) + 3*p1*(2 + 2*r_rr3 - 2*r_rr3^2 + 3*p2*(2 - 4*r_rr3 + r_rr3^2))) + (x[,"n_12"] + x[,"n_21"])*log(p2*(6 + 6*r_rr3 - 6*r_rr3^2) + 4*(8 - 4*r_rr3 + r_rr3^2) + 3*p1*(2 + 2*r_rr3 - 2*r_rr3^2 + 3*p2*(-1 + 2*r_rr3 + r_rr3^2)))


logL_rr4 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(5*log(2) + 2*log(3)) + (n00 + n33)*(log(2 - 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r))) + (n01 + n32)*(log(4 + 3*p2) + log(pmax(1e-6,r)) + log(6 + (-2 + 3*p1)*r)) + (n10 + n23)*(log(2 - 3*p1) + log(pmax(1e-6,r)) + log(12 - (4 + 3*p2)*r)) + (n03 + n30)*(log(2 - 3*p1) + log(4 + 3*p2) + log(1 - 2*r + r^2)) + (n02 + n31)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(3*p1*(1 - r) + 2*(2 + r))) + (n13 + n20)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(3*p2*(-1 + r) + 4*(2 + r))) + (n11 + n22)*log(6*p2*(-1 - r + r^2) + 8*(5 + 2*r + r^2) - 3*p1*(3*p2*(2 - 4*r + r^2) + 4*(-1 - r + r^2))) + (n12 + n21)*log(8*(8 - 4*r + r^2) + 6*p2*(-1 - r + r^2) - 3*p1*(4*(-1 - r + r^2) + 3*p2*(-1 + 2*r + r^2)))
return(L)}
interlogL_rr4 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rr4,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr4 <- parallel::mcmapply(interlogL_rr4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rr4 <- -((x[,"n_10"] + x[,"n_23"])*(-log10(2) + log10(2 - 3*p1) + log10(12 + (-4 - 3*p2)/2))) - (x[,"n_12"] + x[,"n_21"])*log10(50 - 3*p1*(-5 + (3*p2)/4) - (15*p2)/2) - (x[,"n_11"] + x[,"n_22"])*log10(50 - 3*p1*(-5 + (3*p2)/4) - (15*p2)/2) - (x[,"n_13"] + x[,"n_20"])*(-log10(2) + log10(2 - 3*p1) + log10(10 - (3*p2)/2)) - (x[,"n_03"] + x[,"n_30"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_33"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_31"])*(-log10(2) + log10(5 + (3*p1)/2) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_32"])*(-log10(2) + log10(6 + (-2 + 3*p1)/2) + log10(4 + 3*p2)) + (x[,"n_00"] + x[,"n_33"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_rr4))) + (x[,"n_01"] + x[,"n_32"])*(log10(4 + 3*p2) + log10(pmax(1e-6,r_rr4)) + log10(6 + (-2 + 3*p1)*r_rr4)) + (x[,"n_10"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(pmax(1e-6,r_rr4)) + log10(12 - (4 + 3*p2)*r_rr4)) + (x[,"n_03"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(1 - 2*r_rr4 + r_rr4^2)) + (x[,"n_02"] + x[,"n_31"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rr4)) + log10(3*p1*(1 - r_rr4) + 2*(2 + r_rr4))) + (x[,"n_13"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_rr4)) + log10(3*p2*(-1 + r_rr4) + 4*(2 + r_rr4))) + (x[,"n_11"] + x[,"n_22"])*log10(6*p2*(-1 - r_rr4 + r_rr4^2) + 8*(5 + 2*r_rr4 + r_rr4^2) - 3*p1*(3*p2*(2 - 4*r_rr4 + r_rr4^2) + 4*(-1 - r_rr4 + r_rr4^2))) + (x[,"n_12"] + x[,"n_21"])*log10(8*(8 - 4*r_rr4 + r_rr4^2) + 6*p2*(-1 - r_rr4 + r_rr4^2) - 3*p1*(4*(-1 - r_rr4 + r_rr4^2) + 3*p2*(-1 + 2*r_rr4 + r_rr4^2)))


logL_rr4 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(5*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_33"])*(log(2 - 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r_rr4))) + (x[,"n_01"] + x[,"n_32"])*(log(4 + 3*p2) + log(pmax(1e-6,r_rr4)) + log(6 + (-2 + 3*p1)*r_rr4)) + (x[,"n_10"] + x[,"n_23"])*(log(2 - 3*p1) + log(pmax(1e-6,r_rr4)) + log(12 - (4 + 3*p2)*r_rr4)) + (x[,"n_03"] + x[,"n_30"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(1 - 2*r_rr4 + r_rr4^2)) + (x[,"n_02"] + x[,"n_31"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_rr4)) + log(3*p1*(1 - r_rr4) + 2*(2 + r_rr4))) + (x[,"n_13"] + x[,"n_20"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_rr4)) + log(3*p2*(-1 + r_rr4) + 4*(2 + r_rr4))) + (x[,"n_11"] + x[,"n_22"])*log(6*p2*(-1 - r_rr4 + r_rr4^2) + 8*(5 + 2*r_rr4 + r_rr4^2) - 3*p1*(3*p2*(2 - 4*r_rr4 + r_rr4^2) + 4*(-1 - r_rr4 + r_rr4^2))) + (x[,"n_12"] + x[,"n_21"])*log(8*(8 - 4*r_rr4 + r_rr4^2) + 6*p2*(-1 - r_rr4 + r_rr4^2) - 3*p1*(4*(-1 - r_rr4 + r_rr4^2) + 3*p2*(-1 + 2*r_rr4 + r_rr4^2)))


return(list(
r_mat = cbind(r_cc1,r_cc2,r_cc3,r_cc4,r_cr1,r_cr2,r_cr3,r_cr4,r_rc1,r_rc2,r_rc3,r_rc4,r_rr1,r_rr2,r_rr3,r_rr4,0.499),
LOD_mat = cbind(LOD_cc1,LOD_cc2,LOD_cc3,LOD_cc4,LOD_cr1,LOD_cr2,LOD_cr3,LOD_cr4,LOD_rc1,LOD_rc2,LOD_rc3,LOD_rc4,LOD_rr1,LOD_rr2,LOD_rr3,LOD_rr4,0),
logL_mat = cbind(logL_cc1,logL_cc2,logL_cc3,logL_cc4,logL_cr1,logL_cr2,logL_cr3,logL_cr4,logL_rc1,logL_rc2,logL_rc3,logL_rc4,logL_rr1,logL_rr2,logL_rr3,logL_rr4,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling coupling","coupling coupling","coupling coupling","coupling repulsion","coupling repulsion","coupling repulsion","coupling repulsion","repulsion coupling","repulsion coupling","repulsion coupling","repulsion coupling","repulsion repulsion","repulsion repulsion","repulsion repulsion","repulsion repulsion","unknown")
)
)
}

#' @rdname p4_functions
p4_1.2_2.2 <- function(x,p1,p2,ncores=1){


logL_cc1 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- 4*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(2) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - n10 - 2*n11 - 2*n12 - 2*n13 - n14 - n20 - 2*n21 - 2*n22 - 2*n23 - n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34)*log(3) + (n02 + n32)*(log(10 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n14 + n20)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n04 + n30)*(log(2 - 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,r))) + (n00 + n34)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + r^2)) + (n10 + n24)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n01 + n33)*(log(2 - 3*p2) + log(1 - 2*r + r^2) + log(4*(1 + r) - p1*(-3 + 6*r))) + (n03 + n31)*(log(2 - 3*p2) + 2*log(pmax(1e-6,r)) + log(8 - 4*r + p1*(-3 + 6*r))) + (n12 + n22)*log(-6*p2*(-3 - 5*r + 5*r^2) + 4*(9 - 5*r + 5*r^2) + p1*(-9*p2*(-3 - r + r^2) + 6*(3 - r + r^2))) + (n13 + n21)*(log(pmax(1e-6,r)) + log(4*(-3*p2*(2 - 5*r + 3*r^2) + 2*(5 - 5*r + 3*r^2)) + 3*p1*(3*p2*(-2 - 7*r + 6*r^2) - 2*(4 - 7*r + 6*r^2)))) + (n11 + n23)*(log(pmax(1e-6,1 - r)) + log(4*(6 + (-2 + 3*p2)*r + (6 - 9*p2)*r^2) + 3*p1*(3*p2*(-3 - 5*r + 6*r^2) - 2*(3 - 5*r + 6*r^2))))
return(L)}
interlogL_cc1 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cc1,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc1 <- parallel::mcmapply(interlogL_cc1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cc1 <- -((x[,"n_13"] + x[,"n_21"])*(-log10(2) + log10(3*p1*(-4 - 12*p2) + 4*(13/2 - (3*p2)/4)))) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(3) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(2 - 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(-2*log10(2) + log10(10 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_12"] + x[,"n_22"])*log10(31 + (51*p2)/2 + p1*(33/2 + (117*p2)/4)) - (x[,"n_11"] + x[,"n_23"])*(-log10(2) + log10(3*p1*(-4 - 12*p2) + 4*(6 + (6 - 9*p2)/4 + (-2 + 3*p2)/2))) + (x[,"n_02"] + x[,"n_32"])*(log10(10 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cc1)) + log10(pmax(1e-6,r_cc1))) + (x[,"n_14"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cc1)) + 2*log10(pmax(1e-6,r_cc1))) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 3*log10(pmax(1e-6,r_cc1))) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cc1)) + log10(1 - 2*r_cc1 + r_cc1^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_cc1)) + log10(1 - 2*r_cc1 + r_cc1^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(2 - 3*p2) + log10(1 - 2*r_cc1 + r_cc1^2) + log10(4*(1 + r_cc1) - p1*(-3 + 6*r_cc1))) + (x[,"n_03"] + x[,"n_31"])*(log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_cc1)) + log10(8 - 4*r_cc1 + p1*(-3 + 6*r_cc1))) + (x[,"n_12"] + x[,"n_22"])*log10(-6*p2*(-3 - 5*r_cc1 + 5*r_cc1^2) + 4*(9 - 5*r_cc1 + 5*r_cc1^2) + p1*(-9*p2*(-3 - r_cc1 + r_cc1^2) + 6*(3 - r_cc1 + r_cc1^2))) + (x[,"n_13"] + x[,"n_21"])*(log10(pmax(1e-6,r_cc1)) + log10(4*(-3*p2*(2 - 5*r_cc1 + 3*r_cc1^2) + 2*(5 - 5*r_cc1 + 3*r_cc1^2)) + 3*p1*(3*p2*(-2 - 7*r_cc1 + 6*r_cc1^2) - 2*(4 - 7*r_cc1 + 6*r_cc1^2)))) + (x[,"n_11"] + x[,"n_23"])*(log10(pmax(1e-6,1 - r_cc1)) + log10(4*(6 + (-2 + 3*p2)*r_cc1 + (6 - 9*p2)*r_cc1^2) + 3*p1*(3*p2*(-3 - 5*r_cc1 + 6*r_cc1^2) - 2*(3 - 5*r_cc1 + 6*r_cc1^2))))


logL_cc1 <- 4*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"])*log(3) + (x[,"n_02"] + x[,"n_32"])*(log(10 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cc1)) + log(pmax(1e-6,r_cc1))) + (x[,"n_14"] + x[,"n_20"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cc1)) + 2*log(pmax(1e-6,r_cc1))) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,r_cc1))) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cc1)) + log(1 - 2*r_cc1 + r_cc1^2)) + (x[,"n_10"] + x[,"n_24"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_cc1)) + log(1 - 2*r_cc1 + r_cc1^2)) + (x[,"n_01"] + x[,"n_33"])*(log(2 - 3*p2) + log(1 - 2*r_cc1 + r_cc1^2) + log(4*(1 + r_cc1) - p1*(-3 + 6*r_cc1))) + (x[,"n_03"] + x[,"n_31"])*(log(2 - 3*p2) + 2*log(pmax(1e-6,r_cc1)) + log(8 - 4*r_cc1 + p1*(-3 + 6*r_cc1))) + (x[,"n_12"] + x[,"n_22"])*log(-6*p2*(-3 - 5*r_cc1 + 5*r_cc1^2) + 4*(9 - 5*r_cc1 + 5*r_cc1^2) + p1*(-9*p2*(-3 - r_cc1 + r_cc1^2) + 6*(3 - r_cc1 + r_cc1^2))) + (x[,"n_13"] + x[,"n_21"])*(log(pmax(1e-6,r_cc1)) + log(4*(-3*p2*(2 - 5*r_cc1 + 3*r_cc1^2) + 2*(5 - 5*r_cc1 + 3*r_cc1^2)) + 3*p1*(3*p2*(-2 - 7*r_cc1 + 6*r_cc1^2) - 2*(4 - 7*r_cc1 + 6*r_cc1^2)))) + (x[,"n_11"] + x[,"n_23"])*(log(pmax(1e-6,1 - r_cc1)) + log(4*(6 + (-2 + 3*p2)*r_cc1 + (6 - 9*p2)*r_cc1^2) + 3*p1*(3*p2*(-3 - 5*r_cc1 + 6*r_cc1^2) - 2*(3 - 5*r_cc1 + 6*r_cc1^2))))


logL_cc2 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- 5*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(2) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - n10 - 2*n11 - 2*n12 - 2*n13 - n14 - n20 - 2*n21 - 2*n22 - 2*n23 - n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34)*log(3) + (n02 + n32)*(log(10 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n14 + n20)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n04 + n30)*(log(2 - 3*p1) + log(4 + 3*p2) + 3*log(pmax(1e-6,r))) + (n00 + n34)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + r^2)) + (n10 + n24)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n01 + n33)*(log(4 + 3*p2) + log(1 - 2*r + r^2) + log(4*(1 + r) - p1*(-3 + 6*r))) + (n03 + n31)*(log(4 + 3*p2) + 2*log(pmax(1e-6,r)) + log(8 - 4*r + p1*(-3 + 6*r))) + (n12 + n22)*log(6*p2*(-3 - 5*r + 5*r^2) + 8*(9 - 5*r + 5*r^2) + 3*p1*(3*p2*(-3 - r + r^2) + 4*(3 - r + r^2))) + (n13 + n21)*(log(pmax(1e-6,r)) + log(4*(3*p2*(2 - 5*r + 3*r^2) + 4*(5 - 5*r + 3*r^2)) + p1*(-9*p2*(-2 - 7*r + 6*r^2) - 12*(4 - 7*r + 6*r^2)))) + (n11 + n23)*(log(pmax(1e-6,1 - r)) + log(4*(12 - (4 + 3*p2)*r + 3*(4 + 3*p2)*r^2) - 3*p1*(3*p2*(-3 - 5*r + 6*r^2) + 4*(3 - 5*r + 6*r^2))))
return(L)}
interlogL_cc2 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cc2,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc2 <- parallel::mcmapply(interlogL_cc2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cc2 <- -((x[,"n_12"] + x[,"n_22"])*log10(62 + 3*p1*(11 - (39*p2)/4) - (51*p2)/2)) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(-2*log10(2) + log10(10 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_13"] + x[,"n_21"])*(-log10(2) + log10(4*(13 + (3*p2)/4) + p1*(-24 + 36*p2))) - (x[,"n_11"] + x[,"n_23"])*(-log10(2) + log10(-3*p1*(8 - 12*p2) + 4*(12 + (-4 - 3*p2)/2 + (3*(4 + 3*p2))/4))) + (x[,"n_02"] + x[,"n_32"])*(log10(10 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cc2)) + log10(pmax(1e-6,r_cc2))) + (x[,"n_14"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cc2)) + 2*log10(pmax(1e-6,r_cc2))) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + 3*log10(pmax(1e-6,r_cc2))) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cc2)) + log10(1 - 2*r_cc2 + r_cc2^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_cc2)) + log10(1 - 2*r_cc2 + r_cc2^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(4 + 3*p2) + log10(1 - 2*r_cc2 + r_cc2^2) + log10(4*(1 + r_cc2) - p1*(-3 + 6*r_cc2))) + (x[,"n_03"] + x[,"n_31"])*(log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_cc2)) + log10(8 - 4*r_cc2 + p1*(-3 + 6*r_cc2))) + (x[,"n_12"] + x[,"n_22"])*log10(6*p2*(-3 - 5*r_cc2 + 5*r_cc2^2) + 8*(9 - 5*r_cc2 + 5*r_cc2^2) + 3*p1*(3*p2*(-3 - r_cc2 + r_cc2^2) + 4*(3 - r_cc2 + r_cc2^2))) + (x[,"n_13"] + x[,"n_21"])*(log10(pmax(1e-6,r_cc2)) + log10(4*(3*p2*(2 - 5*r_cc2 + 3*r_cc2^2) + 4*(5 - 5*r_cc2 + 3*r_cc2^2)) + p1*(-9*p2*(-2 - 7*r_cc2 + 6*r_cc2^2) - 12*(4 - 7*r_cc2 + 6*r_cc2^2)))) + (x[,"n_11"] + x[,"n_23"])*(log10(pmax(1e-6,1 - r_cc2)) + log10(4*(12 - (4 + 3*p2)*r_cc2 + 3*(4 + 3*p2)*r_cc2^2) - 3*p1*(3*p2*(-3 - 5*r_cc2 + 6*r_cc2^2) + 4*(3 - 5*r_cc2 + 6*r_cc2^2))))


logL_cc2 <- 5*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"])*log(3) + (x[,"n_02"] + x[,"n_32"])*(log(10 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cc2)) + log(pmax(1e-6,r_cc2))) + (x[,"n_14"] + x[,"n_20"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cc2)) + 2*log(pmax(1e-6,r_cc2))) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(4 + 3*p2) + 3*log(pmax(1e-6,r_cc2))) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cc2)) + log(1 - 2*r_cc2 + r_cc2^2)) + (x[,"n_10"] + x[,"n_24"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_cc2)) + log(1 - 2*r_cc2 + r_cc2^2)) + (x[,"n_01"] + x[,"n_33"])*(log(4 + 3*p2) + log(1 - 2*r_cc2 + r_cc2^2) + log(4*(1 + r_cc2) - p1*(-3 + 6*r_cc2))) + (x[,"n_03"] + x[,"n_31"])*(log(4 + 3*p2) + 2*log(pmax(1e-6,r_cc2)) + log(8 - 4*r_cc2 + p1*(-3 + 6*r_cc2))) + (x[,"n_12"] + x[,"n_22"])*log(6*p2*(-3 - 5*r_cc2 + 5*r_cc2^2) + 8*(9 - 5*r_cc2 + 5*r_cc2^2) + 3*p1*(3*p2*(-3 - r_cc2 + r_cc2^2) + 4*(3 - r_cc2 + r_cc2^2))) + (x[,"n_13"] + x[,"n_21"])*(log(pmax(1e-6,r_cc2)) + log(4*(3*p2*(2 - 5*r_cc2 + 3*r_cc2^2) + 4*(5 - 5*r_cc2 + 3*r_cc2^2)) + p1*(-9*p2*(-2 - 7*r_cc2 + 6*r_cc2^2) - 12*(4 - 7*r_cc2 + 6*r_cc2^2)))) + (x[,"n_11"] + x[,"n_23"])*(log(pmax(1e-6,1 - r_cc2)) + log(4*(12 - (4 + 3*p2)*r_cc2 + 3*(4 + 3*p2)*r_cc2^2) - 3*p1*(3*p2*(-3 - 5*r_cc2 + 6*r_cc2^2) + 4*(3 - 5*r_cc2 + 6*r_cc2^2))))


logL_cc3 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- 5*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(2) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - n10 - 2*n11 - 2*n12 - 2*n13 - n14 - n20 - 2*n21 - 2*n22 - 2*n23 - n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34)*log(3) + (n02 + n32)*(log(20 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n14 + n20)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n04 + n30)*(log(4 + 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,r))) + (n00 + n34)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + r^2)) + (n10 + n24)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n01 + n33)*(log(2 - 3*p2) + log(8 - 3*p1 + 8*r + 6*p1*r) + log(1 - 2*r + r^2)) + (n03 + n31)*(log(2 - 3*p2) + 2*log(pmax(1e-6,r)) + log(8*(2 - r) - p1*(-3 + 6*r))) + (n12 + n22)*log(p2*(36 + 60*r - 60*r^2) + 8*(9 - 5*r + 5*r^2) + 3*p1*(3*p2*(-3 - r + r^2) - 2*(3 - r + r^2))) + (n13 + n21)*(log(pmax(1e-6,r)) + log(-8*(3*p2*(2 - 5*r + 3*r^2) - 2*(5 - 5*r + 3*r^2)) + p1*(-9*p2*(-2 - 7*r + 6*r^2) + 6*(4 - 7*r + 6*r^2)))) + (n11 + n23)*(log(pmax(1e-6,1 - r)) + log(-8*(-6 + (2 - 3*p2)*r + (-6 + 9*p2)*r^2) - 3*p1*(3*p2*(-3 - 5*r + 6*r^2) - 2*(3 - 5*r + 6*r^2))))
return(L)}
interlogL_cc3 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cc3,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc3 <- parallel::mcmapply(interlogL_cc3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cc3 <- -((x[,"n_03"] + x[,"n_31"])*(log10(3) + log10(2 - 3*p2))) - (x[,"n_01"] + x[,"n_33"])*(log10(3) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(-2*log10(2) + log10(20 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_12"] + x[,"n_22"])*log10(62 + 3*p1*(-11/2 - (39*p2)/4) + 51*p2) - (x[,"n_13"] + x[,"n_21"])*(-log10(2) + log10(-8*(-13/2 + (3*p2)/4) + p1*(12 + 36*p2))) - (x[,"n_11"] + x[,"n_23"])*(-log10(2) + log10(-3*p1*(-4 - 12*p2) - 8*(-6 + (2 - 3*p2)/2 + (-6 + 9*p2)/4))) + (x[,"n_02"] + x[,"n_32"])*(log10(20 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cc3)) + log10(pmax(1e-6,r_cc3))) + (x[,"n_14"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cc3)) + 2*log10(pmax(1e-6,r_cc3))) + (x[,"n_04"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + 3*log10(pmax(1e-6,r_cc3))) + (x[,"n_00"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cc3)) + log10(1 - 2*r_cc3 + r_cc3^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_cc3)) + log10(1 - 2*r_cc3 + r_cc3^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(2 - 3*p2) + log10(8 - 3*p1 + 8*r_cc3 + 6*p1*r_cc3) + log10(1 - 2*r_cc3 + r_cc3^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_cc3)) + log10(8*(2 - r_cc3) - p1*(-3 + 6*r_cc3))) + (x[,"n_12"] + x[,"n_22"])*log10(p2*(36 + 60*r_cc3 - 60*r_cc3^2) + 8*(9 - 5*r_cc3 + 5*r_cc3^2) + 3*p1*(3*p2*(-3 - r_cc3 + r_cc3^2) - 2*(3 - r_cc3 + r_cc3^2))) + (x[,"n_13"] + x[,"n_21"])*(log10(pmax(1e-6,r_cc3)) + log10(-8*(3*p2*(2 - 5*r_cc3 + 3*r_cc3^2) - 2*(5 - 5*r_cc3 + 3*r_cc3^2)) + p1*(-9*p2*(-2 - 7*r_cc3 + 6*r_cc3^2) + 6*(4 - 7*r_cc3 + 6*r_cc3^2)))) + (x[,"n_11"] + x[,"n_23"])*(log10(pmax(1e-6,1 - r_cc3)) + log10(-8*(-6 + (2 - 3*p2)*r_cc3 + (-6 + 9*p2)*r_cc3^2) - 3*p1*(3*p2*(-3 - 5*r_cc3 + 6*r_cc3^2) - 2*(3 - 5*r_cc3 + 6*r_cc3^2))))


logL_cc3 <- 5*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"])*log(3) + (x[,"n_02"] + x[,"n_32"])*(log(20 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cc3)) + log(pmax(1e-6,r_cc3))) + (x[,"n_14"] + x[,"n_20"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cc3)) + 2*log(pmax(1e-6,r_cc3))) + (x[,"n_04"] + x[,"n_30"])*(log(4 + 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,r_cc3))) + (x[,"n_00"] + x[,"n_34"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cc3)) + log(1 - 2*r_cc3 + r_cc3^2)) + (x[,"n_10"] + x[,"n_24"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_cc3)) + log(1 - 2*r_cc3 + r_cc3^2)) + (x[,"n_01"] + x[,"n_33"])*(log(2 - 3*p2) + log(8 - 3*p1 + 8*r_cc3 + 6*p1*r_cc3) + log(1 - 2*r_cc3 + r_cc3^2)) + (x[,"n_03"] + x[,"n_31"])*(log(2 - 3*p2) + 2*log(pmax(1e-6,r_cc3)) + log(8*(2 - r_cc3) - p1*(-3 + 6*r_cc3))) + (x[,"n_12"] + x[,"n_22"])*log(p2*(36 + 60*r_cc3 - 60*r_cc3^2) + 8*(9 - 5*r_cc3 + 5*r_cc3^2) + 3*p1*(3*p2*(-3 - r_cc3 + r_cc3^2) - 2*(3 - r_cc3 + r_cc3^2))) + (x[,"n_13"] + x[,"n_21"])*(log(pmax(1e-6,r_cc3)) + log(-8*(3*p2*(2 - 5*r_cc3 + 3*r_cc3^2) - 2*(5 - 5*r_cc3 + 3*r_cc3^2)) + p1*(-9*p2*(-2 - 7*r_cc3 + 6*r_cc3^2) + 6*(4 - 7*r_cc3 + 6*r_cc3^2)))) + (x[,"n_11"] + x[,"n_23"])*(log(pmax(1e-6,1 - r_cc3)) + log(-8*(-6 + (2 - 3*p2)*r_cc3 + (-6 + 9*p2)*r_cc3^2) - 3*p1*(3*p2*(-3 - 5*r_cc3 + 6*r_cc3^2) - 2*(3 - 5*r_cc3 + 6*r_cc3^2))))


logL_cc4 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- 6*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(2) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - n10 - 2*n11 - 2*n12 - 2*n13 - n14 - n20 - 2*n21 - 2*n22 - 2*n23 - n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34)*log(3) + (n02 + n32)*(log(20 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n14 + n20)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n04 + n30)*(log(4 + 3*p1) + log(4 + 3*p2) + 3*log(pmax(1e-6,r))) + (n00 + n34)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + r^2)) + (n10 + n24)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n01 + n33)*(log(4 + 3*p2) + log(8 - 3*p1 + 8*r + 6*p1*r) + log(1 - 2*r + r^2)) + (n03 + n31)*(log(4 + 3*p2) + 2*log(pmax(1e-6,r)) + log(8*(2 - r) - p1*(-3 + 6*r))) + (n12 + n22)*log(-3*p1*(3*p2*(-3 - r + r^2) + 4*(3 - r + r^2)) + 4*(3*p2*(-3 - 5*r + 5*r^2) + 4*(9 - 5*r + 5*r^2))) + (n13 + n21)*(log(pmax(1e-6,r)) + log(8*(3*p2*(2 - 5*r + 3*r^2) + 4*(5 - 5*r + 3*r^2)) + 3*p1*(3*p2*(-2 - 7*r + 6*r^2) + 4*(4 - 7*r + 6*r^2)))) + (n11 + n23)*(log(pmax(1e-6,1 - r)) + log(8*(12 - (4 + 3*p2)*r + 3*(4 + 3*p2)*r^2) + 3*p1*(3*p2*(-3 - 5*r + 6*r^2) + 4*(3 - 5*r + 6*r^2))))
return(L)}
interlogL_cc4 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cc4,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc4 <- parallel::mcmapply(interlogL_cc4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cc4 <- -((x[,"n_12"] + x[,"n_22"])*log10(4*(31 - (51*p2)/4) - 3*p1*(11 - (39*p2)/4))) - (x[,"n_13"] + x[,"n_21"])*(-log10(2) + log10(3*p1*(8 - 12*p2) + 8*(13 + (3*p2)/4))) - (x[,"n_03"] + x[,"n_31"])*(log10(3) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_33"])*(log10(3) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(-2*log10(2) + log10(20 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_11"] + x[,"n_23"])*(-log10(2) + log10(3*p1*(8 - 12*p2) + 8*(12 + (-4 - 3*p2)/2 + (3*(4 + 3*p2))/4))) + (x[,"n_02"] + x[,"n_32"])*(log10(20 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cc4)) + log10(pmax(1e-6,r_cc4))) + (x[,"n_14"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cc4)) + 2*log10(pmax(1e-6,r_cc4))) + (x[,"n_04"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + 3*log10(pmax(1e-6,r_cc4))) + (x[,"n_00"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cc4)) + log10(1 - 2*r_cc4 + r_cc4^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_cc4)) + log10(1 - 2*r_cc4 + r_cc4^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(4 + 3*p2) + log10(8 - 3*p1 + 8*r_cc4 + 6*p1*r_cc4) + log10(1 - 2*r_cc4 + r_cc4^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_cc4)) + log10(8*(2 - r_cc4) - p1*(-3 + 6*r_cc4))) + (x[,"n_12"] + x[,"n_22"])*log10(-3*p1*(3*p2*(-3 - r_cc4 + r_cc4^2) + 4*(3 - r_cc4 + r_cc4^2)) + 4*(3*p2*(-3 - 5*r_cc4 + 5*r_cc4^2) + 4*(9 - 5*r_cc4 + 5*r_cc4^2))) + (x[,"n_13"] + x[,"n_21"])*(log10(pmax(1e-6,r_cc4)) + log10(8*(3*p2*(2 - 5*r_cc4 + 3*r_cc4^2) + 4*(5 - 5*r_cc4 + 3*r_cc4^2)) + 3*p1*(3*p2*(-2 - 7*r_cc4 + 6*r_cc4^2) + 4*(4 - 7*r_cc4 + 6*r_cc4^2)))) + (x[,"n_11"] + x[,"n_23"])*(log10(pmax(1e-6,1 - r_cc4)) + log10(8*(12 - (4 + 3*p2)*r_cc4 + 3*(4 + 3*p2)*r_cc4^2) + 3*p1*(3*p2*(-3 - 5*r_cc4 + 6*r_cc4^2) + 4*(3 - 5*r_cc4 + 6*r_cc4^2))))


logL_cc4 <- 6*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"])*log(3) + (x[,"n_02"] + x[,"n_32"])*(log(20 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cc4)) + log(pmax(1e-6,r_cc4))) + (x[,"n_14"] + x[,"n_20"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cc4)) + 2*log(pmax(1e-6,r_cc4))) + (x[,"n_04"] + x[,"n_30"])*(log(4 + 3*p1) + log(4 + 3*p2) + 3*log(pmax(1e-6,r_cc4))) + (x[,"n_00"] + x[,"n_34"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cc4)) + log(1 - 2*r_cc4 + r_cc4^2)) + (x[,"n_10"] + x[,"n_24"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_cc4)) + log(1 - 2*r_cc4 + r_cc4^2)) + (x[,"n_01"] + x[,"n_33"])*(log(4 + 3*p2) + log(8 - 3*p1 + 8*r_cc4 + 6*p1*r_cc4) + log(1 - 2*r_cc4 + r_cc4^2)) + (x[,"n_03"] + x[,"n_31"])*(log(4 + 3*p2) + 2*log(pmax(1e-6,r_cc4)) + log(8*(2 - r_cc4) - p1*(-3 + 6*r_cc4))) + (x[,"n_12"] + x[,"n_22"])*log(-3*p1*(3*p2*(-3 - r_cc4 + r_cc4^2) + 4*(3 - r_cc4 + r_cc4^2)) + 4*(3*p2*(-3 - 5*r_cc4 + 5*r_cc4^2) + 4*(9 - 5*r_cc4 + 5*r_cc4^2))) + (x[,"n_13"] + x[,"n_21"])*(log(pmax(1e-6,r_cc4)) + log(8*(3*p2*(2 - 5*r_cc4 + 3*r_cc4^2) + 4*(5 - 5*r_cc4 + 3*r_cc4^2)) + 3*p1*(3*p2*(-2 - 7*r_cc4 + 6*r_cc4^2) + 4*(4 - 7*r_cc4 + 6*r_cc4^2)))) + (x[,"n_11"] + x[,"n_23"])*(log(pmax(1e-6,1 - r_cc4)) + log(8*(12 - (4 + 3*p2)*r_cc4 + 3*(4 + 3*p2)*r_cc4^2) + 3*p1*(3*p2*(-3 - 5*r_cc4 + 6*r_cc4^2) + 4*(3 - 5*r_cc4 + 6*r_cc4^2))))


logL_cm1 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(5*log(2) + 2*log(3)) + (n04 + n30)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n00 + n34)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n14 + n20)*(log(2 - 3*p1) + log(pmax(1e-6,r)) + log(6 - (8 - 12*p2)*r - (-6 + 9*p2)*r^2)) + (n01 + n33)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(4*(1 + r^2) - p1*(6 - 9*r + 6*r^2))) + (n03 + n31)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(4*(2 - 2*r + r^2) - p1*(3 - 3*r + 6*r^2))) + (n02 + n32)*(log(2 - 3*p2) + log(8 - 6*r + 6*r^2 + p1*(6 - 9*r + 9*r^2))) + (n10 + n24)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(4 - 4*r + 6*r^2 - p2*(-3 - 6*r + 9*r^2))) + (n12 + n22)*log(3*p1*(8 + 6*r - 6*r^2 + p2*(-3 - 9*r + 9*r^2)) + 2*(28 + 6*r - 6*r^2 + p2*(3 - 9*r + 9*r^2))) + (n11 + n23)*log(-4*(-8 + 8*r^2 - 6*r^3 + 3*p2*(-2 + 3*r - 4*r^2 + 3*r^3)) + 3*p1*(-2*(2 + 3*r - 11*r^2 + 6*r^3) + 3*p2*(-1 + 9*r - 11*r^2 + 6*r^3))) + (n13 + n21)*log(4*(6 + 2*(-1 + 6*p2)*r - 5*(-2 + 3*p2)*r^2 + (-6 + 9*p2)*r^3) - 3*p1*(2*r*(1 + 7*r - 6*r^2) + 3*p2*(-3 + 5*r - 7*r^2 + 6*r^3)))
return(L)}
interlogL_cm1 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cm1,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cm1 <- parallel::mcmapply(interlogL_cm1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cm1 <- -((x[,"n_12"] + x[,"n_22"])*log10(3*p1*(19/2 - (21*p2)/4) + 2*(59/2 + (3*p2)/4))) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(5 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(5 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(log10(13/2 + (15*p1)/4) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-log10(2) + log10(2 - 3*p1) + log10(7/2 + (15*p2)/4)) - (x[,"n_11"] + x[,"n_23"])*log10(-4*(-27/4 - (27*p2)/8) + 3*p1*(-3 + (9*p2)/2)) - (x[,"n_14"] + x[,"n_20"])*(-log10(2) + log10(2 - 3*p1) + log10(6 + (6 - 9*p2)/4 + (-8 + 12*p2)/2)) - (x[,"n_13"] + x[,"n_21"])*log10(-3*p1*(3 - (9*p2)/2) + 4*(5 + 6*p2 - (5*(-2 + 3*p2))/4 + (-6 + 9*p2)/8)) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cm1)) + 2*log10(pmax(1e-6,r_cm1))) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_cm1)) + log10(1 - 2*r_cm1 + r_cm1^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(pmax(1e-6,r_cm1)) + log10(6 - (8 - 12*p2)*r_cm1 - (-6 + 9*p2)*r_cm1^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cm1)) + log10(4*(1 + r_cm1^2) - p1*(6 - 9*r_cm1 + 6*r_cm1^2))) + (x[,"n_03"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_cm1)) + log10(4*(2 - 2*r_cm1 + r_cm1^2) - p1*(3 - 3*r_cm1 + 6*r_cm1^2))) + (x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p2) + log10(8 - 6*r_cm1 + 6*r_cm1^2 + p1*(6 - 9*r_cm1 + 9*r_cm1^2))) + (x[,"n_10"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_cm1)) + log10(4 - 4*r_cm1 + 6*r_cm1^2 - p2*(-3 - 6*r_cm1 + 9*r_cm1^2))) + (x[,"n_12"] + x[,"n_22"])*log10(3*p1*(8 + 6*r_cm1 - 6*r_cm1^2 + p2*(-3 - 9*r_cm1 + 9*r_cm1^2)) + 2*(28 + 6*r_cm1 - 6*r_cm1^2 + p2*(3 - 9*r_cm1 + 9*r_cm1^2))) + (x[,"n_11"] + x[,"n_23"])*log10(-4*(-8 + 8*r_cm1^2 - 6*r_cm1^3 + 3*p2*(-2 + 3*r_cm1 - 4*r_cm1^2 + 3*r_cm1^3)) + 3*p1*(-2*(2 + 3*r_cm1 - 11*r_cm1^2 + 6*r_cm1^3) + 3*p2*(-1 + 9*r_cm1 - 11*r_cm1^2 + 6*r_cm1^3))) + (x[,"n_13"] + x[,"n_21"])*log10(4*(6 + 2*(-1 + 6*p2)*r_cm1 - 5*(-2 + 3*p2)*r_cm1^2 + (-6 + 9*p2)*r_cm1^3) - 3*p1*(2*r_cm1*(1 + 7*r_cm1 - 6*r_cm1^2) + 3*p2*(-3 + 5*r_cm1 - 7*r_cm1^2 + 6*r_cm1^3)))


logL_cm1 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(5*log(2) + 2*log(3)) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cm1)) + 2*log(pmax(1e-6,r_cm1))) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_cm1)) + log(1 - 2*r_cm1 + r_cm1^2)) + (x[,"n_14"] + x[,"n_20"])*(log(2 - 3*p1) + log(pmax(1e-6,r_cm1)) + log(6 - (8 - 12*p2)*r_cm1 - (-6 + 9*p2)*r_cm1^2)) + (x[,"n_01"] + x[,"n_33"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_cm1)) + log(4*(1 + r_cm1^2) - p1*(6 - 9*r_cm1 + 6*r_cm1^2))) + (x[,"n_03"] + x[,"n_31"])*(log(2 - 3*p2) + log(pmax(1e-6,r_cm1)) + log(4*(2 - 2*r_cm1 + r_cm1^2) - p1*(3 - 3*r_cm1 + 6*r_cm1^2))) + (x[,"n_02"] + x[,"n_32"])*(log(2 - 3*p2) + log(8 - 6*r_cm1 + 6*r_cm1^2 + p1*(6 - 9*r_cm1 + 9*r_cm1^2))) + (x[,"n_10"] + x[,"n_24"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_cm1)) + log(4 - 4*r_cm1 + 6*r_cm1^2 - p2*(-3 - 6*r_cm1 + 9*r_cm1^2))) + (x[,"n_12"] + x[,"n_22"])*log(3*p1*(8 + 6*r_cm1 - 6*r_cm1^2 + p2*(-3 - 9*r_cm1 + 9*r_cm1^2)) + 2*(28 + 6*r_cm1 - 6*r_cm1^2 + p2*(3 - 9*r_cm1 + 9*r_cm1^2))) + (x[,"n_11"] + x[,"n_23"])*log(-4*(-8 + 8*r_cm1^2 - 6*r_cm1^3 + 3*p2*(-2 + 3*r_cm1 - 4*r_cm1^2 + 3*r_cm1^3)) + 3*p1*(-2*(2 + 3*r_cm1 - 11*r_cm1^2 + 6*r_cm1^3) + 3*p2*(-1 + 9*r_cm1 - 11*r_cm1^2 + 6*r_cm1^3))) + (x[,"n_13"] + x[,"n_21"])*log(4*(6 + 2*(-1 + 6*p2)*r_cm1 - 5*(-2 + 3*p2)*r_cm1^2 + (-6 + 9*p2)*r_cm1^3) - 3*p1*(2*r_cm1*(1 + 7*r_cm1 - 6*r_cm1^2) + 3*p2*(-3 + 5*r_cm1 - 7*r_cm1^2 + 6*r_cm1^3)))


logL_cm2 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(5*log(2) + 2*log(3)) + (n04 + n30)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n00 + n34)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n14 + n20)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(3 - 4*r + 3*r^2)) + (n10 + n24)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(2 - 2*r + 3*r^2)) + (n03 + n31)*(log(pmax(1e-6,r)) + log((4 + 3*p1)*(-2 + 3*p2)*(-1 + r) + (2 - 3*p1)*(p2*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2)))) + (n12 + n22)*log(2*(28 + 6*r - 6*r^2 + 3*p2*(4 - 3*r + 3*r^2)) + 3*p1*(8 + 6*r - 6*r^2 + 3*p2*(5 - 3*r + 3*r^2))) + (n01 + n33)*(log(pmax(1e-6,1 - r)) + log(8 + 6*p2 + 8*r^2 - 12*p2*r^2 + 3*p1*(-4 + 6*r - 4*r^2 + p2*(-3 - 9*r + 6*r^2)))) + (n02 + n32)*log(-2*(-8 + 6*r - 6*r^2 + p2*(-6 - 9*r + 9*r^2)) - 3*p1*(-4 + 6*r - 6*r^2 + p2*(-3 - 9*r + 9*r^2))) + (n11 + n23)*log(3*p1*(-2*(2 + 3*r - 11*r^2 + 6*r^3) + 3*p2*(-4 + 6*r - 11*r^2 + 6*r^3)) - 2*(-4*(4 - 4*r^2 + 3*r^3) + 3*p2*(2 + 3*r - 8*r^2 + 6*r^3))) + (n13 + n21)*log(24 - 2*(4 + 3*p1)*r + (40 - 42*p1)*r^2 + 12*(-2 + 3*p1)*r^3 - 3*p2*(6 - 10*r + 20*r^2 - 12*r^3 + 3*p1*(3 + 2*r - 7*r^2 + 6*r^3)))
return(L)}
interlogL_cm2 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cm2,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cm2 <- parallel::mcmapply(interlogL_cm2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cm2 <- -((x[,"n_02"] + x[,"n_32"])*log10(-2*(-13/2 - (33*p2)/4) - 3*p1*(-5/2 - (21*p2)/4))) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(7) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(7) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(10 + 3*p1*(-2 - 6*p2) + 3*p2)) - (x[,"n_13"] + x[,"n_21"])*log10(20 + (40 - 42*p1)/4 - 3*p1 + (3*(-2 + 3*p1))/2 - 3*(9/2 + 9*p1)*p2) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(-((4 + 3*p1)*(-2 + 3*p2))/2 + (2 - 3*p1)*(3 + (9*p2)/2))) - (x[,"n_11"] + x[,"n_23"])*log10(3*p1*(-3 - 9*p2) - 2*(-27/2 + (27*p2)/4)) - (x[,"n_12"] + x[,"n_22"])*log10(2*(59/2 + (39*p2)/4) + 3*p1*(19/2 + (51*p2)/4)) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cm2)) + 2*log10(pmax(1e-6,r_cm2))) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_cm2)) + log10(1 - 2*r_cm2 + r_cm2^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_cm2)) + log10(3 - 4*r_cm2 + 3*r_cm2^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cm2)) + log10(2 - 2*r_cm2 + 3*r_cm2^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(pmax(1e-6,r_cm2)) + log10((4 + 3*p1)*(-2 + 3*p2)*(-1 + r_cm2) + (2 - 3*p1)*(p2*(3 + 6*r_cm2 - 6*r_cm2^2) + 4*(1 - r_cm2 + r_cm2^2)))) + (x[,"n_12"] + x[,"n_22"])*log10(2*(28 + 6*r_cm2 - 6*r_cm2^2 + 3*p2*(4 - 3*r_cm2 + 3*r_cm2^2)) + 3*p1*(8 + 6*r_cm2 - 6*r_cm2^2 + 3*p2*(5 - 3*r_cm2 + 3*r_cm2^2))) + (x[,"n_01"] + x[,"n_33"])*(log10(pmax(1e-6,1 - r_cm2)) + log10(8 + 6*p2 + 8*r_cm2^2 - 12*p2*r_cm2^2 + 3*p1*(-4 + 6*r_cm2 - 4*r_cm2^2 + p2*(-3 - 9*r_cm2 + 6*r_cm2^2)))) + (x[,"n_02"] + x[,"n_32"])*log10(-2*(-8 + 6*r_cm2 - 6*r_cm2^2 + p2*(-6 - 9*r_cm2 + 9*r_cm2^2)) - 3*p1*(-4 + 6*r_cm2 - 6*r_cm2^2 + p2*(-3 - 9*r_cm2 + 9*r_cm2^2))) + (x[,"n_11"] + x[,"n_23"])*log10(3*p1*(-2*(2 + 3*r_cm2 - 11*r_cm2^2 + 6*r_cm2^3) + 3*p2*(-4 + 6*r_cm2 - 11*r_cm2^2 + 6*r_cm2^3)) - 2*(-4*(4 - 4*r_cm2^2 + 3*r_cm2^3) + 3*p2*(2 + 3*r_cm2 - 8*r_cm2^2 + 6*r_cm2^3))) + (x[,"n_13"] + x[,"n_21"])*log10(24 - 2*(4 + 3*p1)*r_cm2 + (40 - 42*p1)*r_cm2^2 + 12*(-2 + 3*p1)*r_cm2^3 - 3*p2*(6 - 10*r_cm2 + 20*r_cm2^2 - 12*r_cm2^3 + 3*p1*(3 + 2*r_cm2 - 7*r_cm2^2 + 6*r_cm2^3)))


logL_cm2 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(5*log(2) + 2*log(3)) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cm2)) + 2*log(pmax(1e-6,r_cm2))) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_cm2)) + log(1 - 2*r_cm2 + r_cm2^2)) + (x[,"n_14"] + x[,"n_20"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_cm2)) + log(3 - 4*r_cm2 + 3*r_cm2^2)) + (x[,"n_10"] + x[,"n_24"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cm2)) + log(2 - 2*r_cm2 + 3*r_cm2^2)) + (x[,"n_03"] + x[,"n_31"])*(log(pmax(1e-6,r_cm2)) + log((4 + 3*p1)*(-2 + 3*p2)*(-1 + r_cm2) + (2 - 3*p1)*(p2*(3 + 6*r_cm2 - 6*r_cm2^2) + 4*(1 - r_cm2 + r_cm2^2)))) + (x[,"n_12"] + x[,"n_22"])*log(2*(28 + 6*r_cm2 - 6*r_cm2^2 + 3*p2*(4 - 3*r_cm2 + 3*r_cm2^2)) + 3*p1*(8 + 6*r_cm2 - 6*r_cm2^2 + 3*p2*(5 - 3*r_cm2 + 3*r_cm2^2))) + (x[,"n_01"] + x[,"n_33"])*(log(pmax(1e-6,1 - r_cm2)) + log(8 + 6*p2 + 8*r_cm2^2 - 12*p2*r_cm2^2 + 3*p1*(-4 + 6*r_cm2 - 4*r_cm2^2 + p2*(-3 - 9*r_cm2 + 6*r_cm2^2)))) + (x[,"n_02"] + x[,"n_32"])*log(-2*(-8 + 6*r_cm2 - 6*r_cm2^2 + p2*(-6 - 9*r_cm2 + 9*r_cm2^2)) - 3*p1*(-4 + 6*r_cm2 - 6*r_cm2^2 + p2*(-3 - 9*r_cm2 + 9*r_cm2^2))) + (x[,"n_11"] + x[,"n_23"])*log(3*p1*(-2*(2 + 3*r_cm2 - 11*r_cm2^2 + 6*r_cm2^3) + 3*p2*(-4 + 6*r_cm2 - 11*r_cm2^2 + 6*r_cm2^3)) - 2*(-4*(4 - 4*r_cm2^2 + 3*r_cm2^3) + 3*p2*(2 + 3*r_cm2 - 8*r_cm2^2 + 6*r_cm2^3))) + (x[,"n_13"] + x[,"n_21"])*log(24 - 2*(4 + 3*p1)*r_cm2 + (40 - 42*p1)*r_cm2^2 + 12*(-2 + 3*p1)*r_cm2^3 - 3*p2*(6 - 10*r_cm2 + 20*r_cm2^2 - 12*r_cm2^3 + 3*p1*(3 + 2*r_cm2 - 7*r_cm2^2 + 6*r_cm2^3)))


logL_cm3 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(6*log(2) + 2*log(3)) + (n04 + n30)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n00 + n34)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n14 + n20)*(log(4 + 3*p1) + log(pmax(1e-6,r)) + log(6 - (8 - 12*p2)*r - (-6 + 9*p2)*r^2)) + (n01 + n33)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(8*(1 + r^2) + p1*(6 - 9*r + 6*r^2))) + (n03 + n31)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(8*(2 - 2*r + r^2) + p1*(3 - 3*r + 6*r^2))) + (n02 + n32)*(log(2 - 3*p2) + log(4*(4 - 3*r + 3*r^2) - p1*(6 - 9*r + 9*r^2))) + (n10 + n24)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(4 - 4*r + 6*r^2 - p2*(-3 - 6*r + 9*r^2))) + (n12 + n22)*log(-3*p1*(8 + 6*r - 6*r^2 + p2*(-3 - 9*r + 9*r^2)) + 4*(28 + 6*r - 6*r^2 + p2*(3 - 9*r + 9*r^2))) + (n11 + n23)*log(-8*(-8 + 8*r^2 - 6*r^3 + 3*p2*(-2 + 3*r - 4*r^2 + 3*r^3)) + p1*(6*(2 + 3*r - 11*r^2 + 6*r^3) - 9*p2*(-1 + 9*r - 11*r^2 + 6*r^3))) + (n13 + n21)*log(8*(6 + 2*(-1 + 6*p2)*r - 5*(-2 + 3*p2)*r^2 + (-6 + 9*p2)*r^3) + 3*p1*(2*r*(1 + 7*r - 6*r^2) + 3*p2*(-3 + 5*r - 7*r^2 + 6*r^3)))
return(L)}
interlogL_cm3 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cm3,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cm3 <- parallel::mcmapply(interlogL_cm3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cm3 <- -((x[,"n_11"] + x[,"n_23"])*log10(p1*(9 - (27*p2)/2) - 8*(-27/4 - (27*p2)/8))) - (x[,"n_12"] + x[,"n_22"])*log10(-3*p1*(19/2 - (21*p2)/4) + 4*(59/2 + (3*p2)/4)) - (x[,"n_02"] + x[,"n_32"])*(log10(13 - (15*p1)/4) + log10(2 - 3*p2)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(10 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(10 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-log10(2) + log10(4 + 3*p1) + log10(7/2 + (15*p2)/4)) - (x[,"n_14"] + x[,"n_20"])*(-log10(2) + log10(4 + 3*p1) + log10(6 + (6 - 9*p2)/4 + (-8 + 12*p2)/2)) - (x[,"n_13"] + x[,"n_21"])*log10(3*p1*(3 - (9*p2)/2) + 8*(5 + 6*p2 - (5*(-2 + 3*p2))/4 + (-6 + 9*p2)/8)) + (x[,"n_04"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cm3)) + 2*log10(pmax(1e-6,r_cm3))) + (x[,"n_00"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_cm3)) + log10(1 - 2*r_cm3 + r_cm3^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(pmax(1e-6,r_cm3)) + log10(6 - (8 - 12*p2)*r_cm3 - (-6 + 9*p2)*r_cm3^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cm3)) + log10(8*(1 + r_cm3^2) + p1*(6 - 9*r_cm3 + 6*r_cm3^2))) + (x[,"n_03"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_cm3)) + log10(8*(2 - 2*r_cm3 + r_cm3^2) + p1*(3 - 3*r_cm3 + 6*r_cm3^2))) + (x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p2) + log10(4*(4 - 3*r_cm3 + 3*r_cm3^2) - p1*(6 - 9*r_cm3 + 9*r_cm3^2))) + (x[,"n_10"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_cm3)) + log10(4 - 4*r_cm3 + 6*r_cm3^2 - p2*(-3 - 6*r_cm3 + 9*r_cm3^2))) + (x[,"n_12"] + x[,"n_22"])*log10(-3*p1*(8 + 6*r_cm3 - 6*r_cm3^2 + p2*(-3 - 9*r_cm3 + 9*r_cm3^2)) + 4*(28 + 6*r_cm3 - 6*r_cm3^2 + p2*(3 - 9*r_cm3 + 9*r_cm3^2))) + (x[,"n_11"] + x[,"n_23"])*log10(-8*(-8 + 8*r_cm3^2 - 6*r_cm3^3 + 3*p2*(-2 + 3*r_cm3 - 4*r_cm3^2 + 3*r_cm3^3)) + p1*(6*(2 + 3*r_cm3 - 11*r_cm3^2 + 6*r_cm3^3) - 9*p2*(-1 + 9*r_cm3 - 11*r_cm3^2 + 6*r_cm3^3))) + (x[,"n_13"] + x[,"n_21"])*log10(8*(6 + 2*(-1 + 6*p2)*r_cm3 - 5*(-2 + 3*p2)*r_cm3^2 + (-6 + 9*p2)*r_cm3^3) + 3*p1*(2*r_cm3*(1 + 7*r_cm3 - 6*r_cm3^2) + 3*p2*(-3 + 5*r_cm3 - 7*r_cm3^2 + 6*r_cm3^3)))


logL_cm3 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(6*log(2) + 2*log(3)) + (x[,"n_04"] + x[,"n_30"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cm3)) + 2*log(pmax(1e-6,r_cm3))) + (x[,"n_00"] + x[,"n_34"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_cm3)) + log(1 - 2*r_cm3 + r_cm3^2)) + (x[,"n_14"] + x[,"n_20"])*(log(4 + 3*p1) + log(pmax(1e-6,r_cm3)) + log(6 - (8 - 12*p2)*r_cm3 - (-6 + 9*p2)*r_cm3^2)) + (x[,"n_01"] + x[,"n_33"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_cm3)) + log(8*(1 + r_cm3^2) + p1*(6 - 9*r_cm3 + 6*r_cm3^2))) + (x[,"n_03"] + x[,"n_31"])*(log(2 - 3*p2) + log(pmax(1e-6,r_cm3)) + log(8*(2 - 2*r_cm3 + r_cm3^2) + p1*(3 - 3*r_cm3 + 6*r_cm3^2))) + (x[,"n_02"] + x[,"n_32"])*(log(2 - 3*p2) + log(4*(4 - 3*r_cm3 + 3*r_cm3^2) - p1*(6 - 9*r_cm3 + 9*r_cm3^2))) + (x[,"n_10"] + x[,"n_24"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_cm3)) + log(4 - 4*r_cm3 + 6*r_cm3^2 - p2*(-3 - 6*r_cm3 + 9*r_cm3^2))) + (x[,"n_12"] + x[,"n_22"])*log(-3*p1*(8 + 6*r_cm3 - 6*r_cm3^2 + p2*(-3 - 9*r_cm3 + 9*r_cm3^2)) + 4*(28 + 6*r_cm3 - 6*r_cm3^2 + p2*(3 - 9*r_cm3 + 9*r_cm3^2))) + (x[,"n_11"] + x[,"n_23"])*log(-8*(-8 + 8*r_cm3^2 - 6*r_cm3^3 + 3*p2*(-2 + 3*r_cm3 - 4*r_cm3^2 + 3*r_cm3^3)) + p1*(6*(2 + 3*r_cm3 - 11*r_cm3^2 + 6*r_cm3^3) - 9*p2*(-1 + 9*r_cm3 - 11*r_cm3^2 + 6*r_cm3^3))) + (x[,"n_13"] + x[,"n_21"])*log(8*(6 + 2*(-1 + 6*p2)*r_cm3 - 5*(-2 + 3*p2)*r_cm3^2 + (-6 + 9*p2)*r_cm3^3) + 3*p1*(2*r_cm3*(1 + 7*r_cm3 - 6*r_cm3^2) + 3*p2*(-3 + 5*r_cm3 - 7*r_cm3^2 + 6*r_cm3^3)))


logL_cm4 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(6*log(2) + 2*log(3)) + (n04 + n30)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n00 + n34)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n14 + n20)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(3 - 4*r + 3*r^2)) + (n10 + n24)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(2 - 2*r + 3*r^2)) + (n03 + n31)*(log(pmax(1e-6,r)) + log((8 - 3*p1)*(-2 + 3*p2)*(-1 + r) + (4 + 3*p1)*(p2*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2)))) + (n12 + n22)*log(4*(28 + 6*r - 6*r^2 + 3*p2*(4 - 3*r + 3*r^2)) - 3*p1*(8 + 6*r - 6*r^2 + 3*p2*(5 - 3*r + 3*r^2))) + (n01 + n33)*(log(pmax(1e-6,1 - r)) + log(-4*(-4*(1 + r^2) + p2*(-3 + 6*r^2)) - 3*p1*(-4 + 6*r - 4*r^2 + p2*(-3 - 9*r + 6*r^2)))) + (n02 + n32)*log(-4*(-8 + 6*r - 6*r^2 + p2*(-6 - 9*r + 9*r^2)) + 3*p1*(-4 + 6*r - 6*r^2 + p2*(-3 - 9*r + 9*r^2))) + (n11 + n23)*log(p1*(6*(2 + 3*r - 11*r^2 + 6*r^3) - 9*p2*(-4 + 6*r - 11*r^2 + 6*r^3)) - 4*(-4*(4 - 4*r^2 + 3*r^3) + 3*p2*(2 + 3*r - 8*r^2 + 6*r^3))) + (n13 + n21)*log(48 + 2*(-8 + 3*p1)*r + (80 + 42*p1)*r^2 - 12*(4 + 3*p1)*r^3 + 3*p2*(4*(-3 + 5*r - 10*r^2 + 6*r^3) + 3*p1*(3 + 2*r - 7*r^2 + 6*r^3)))
return(L)}
interlogL_cm4 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cm4,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cm4 <- parallel::mcmapply(interlogL_cm4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cm4 <- -((x[,"n_02"] + x[,"n_32"])*log10(-4*(-13/2 - (33*p2)/4) + 3*p1*(-5/2 - (21*p2)/4))) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(-3*p1*(-2 - 6*p2) - 4*(-5 - (3*p2)/2))) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(7) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(7) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_13"] + x[,"n_21"])*log10(40 + 3*p1 - (3*(4 + 3*p1))/2 + (80 + 42*p1)/4 + 3*(-9 + 9*p1)*p2) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(-((8 - 3*p1)*(-2 + 3*p2))/2 + (4 + 3*p1)*(3 + (9*p2)/2))) - (x[,"n_12"] + x[,"n_22"])*log10(4*(59/2 + (39*p2)/4) - 3*p1*(19/2 + (51*p2)/4)) - (x[,"n_11"] + x[,"n_23"])*log10(-4*(-27/2 + (27*p2)/4) + p1*(9 + 27*p2)) + (x[,"n_04"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cm4)) + 2*log10(pmax(1e-6,r_cm4))) + (x[,"n_00"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_cm4)) + log10(1 - 2*r_cm4 + r_cm4^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_cm4)) + log10(3 - 4*r_cm4 + 3*r_cm4^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cm4)) + log10(2 - 2*r_cm4 + 3*r_cm4^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(pmax(1e-6,r_cm4)) + log10((8 - 3*p1)*(-2 + 3*p2)*(-1 + r_cm4) + (4 + 3*p1)*(p2*(3 + 6*r_cm4 - 6*r_cm4^2) + 4*(1 - r_cm4 + r_cm4^2)))) + (x[,"n_12"] + x[,"n_22"])*log10(4*(28 + 6*r_cm4 - 6*r_cm4^2 + 3*p2*(4 - 3*r_cm4 + 3*r_cm4^2)) - 3*p1*(8 + 6*r_cm4 - 6*r_cm4^2 + 3*p2*(5 - 3*r_cm4 + 3*r_cm4^2))) + (x[,"n_01"] + x[,"n_33"])*(log10(pmax(1e-6,1 - r_cm4)) + log10(-4*(-4*(1 + r_cm4^2) + p2*(-3 + 6*r_cm4^2)) - 3*p1*(-4 + 6*r_cm4 - 4*r_cm4^2 + p2*(-3 - 9*r_cm4 + 6*r_cm4^2)))) + (x[,"n_02"] + x[,"n_32"])*log10(-4*(-8 + 6*r_cm4 - 6*r_cm4^2 + p2*(-6 - 9*r_cm4 + 9*r_cm4^2)) + 3*p1*(-4 + 6*r_cm4 - 6*r_cm4^2 + p2*(-3 - 9*r_cm4 + 9*r_cm4^2))) + (x[,"n_11"] + x[,"n_23"])*log10(p1*(6*(2 + 3*r_cm4 - 11*r_cm4^2 + 6*r_cm4^3) - 9*p2*(-4 + 6*r_cm4 - 11*r_cm4^2 + 6*r_cm4^3)) - 4*(-4*(4 - 4*r_cm4^2 + 3*r_cm4^3) + 3*p2*(2 + 3*r_cm4 - 8*r_cm4^2 + 6*r_cm4^3))) + (x[,"n_13"] + x[,"n_21"])*log10(48 + 2*(-8 + 3*p1)*r_cm4 + (80 + 42*p1)*r_cm4^2 - 12*(4 + 3*p1)*r_cm4^3 + 3*p2*(4*(-3 + 5*r_cm4 - 10*r_cm4^2 + 6*r_cm4^3) + 3*p1*(3 + 2*r_cm4 - 7*r_cm4^2 + 6*r_cm4^3)))


logL_cm4 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(6*log(2) + 2*log(3)) + (x[,"n_04"] + x[,"n_30"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cm4)) + 2*log(pmax(1e-6,r_cm4))) + (x[,"n_00"] + x[,"n_34"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_cm4)) + log(1 - 2*r_cm4 + r_cm4^2)) + (x[,"n_14"] + x[,"n_20"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_cm4)) + log(3 - 4*r_cm4 + 3*r_cm4^2)) + (x[,"n_10"] + x[,"n_24"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cm4)) + log(2 - 2*r_cm4 + 3*r_cm4^2)) + (x[,"n_03"] + x[,"n_31"])*(log(pmax(1e-6,r_cm4)) + log((8 - 3*p1)*(-2 + 3*p2)*(-1 + r_cm4) + (4 + 3*p1)*(p2*(3 + 6*r_cm4 - 6*r_cm4^2) + 4*(1 - r_cm4 + r_cm4^2)))) + (x[,"n_12"] + x[,"n_22"])*log(4*(28 + 6*r_cm4 - 6*r_cm4^2 + 3*p2*(4 - 3*r_cm4 + 3*r_cm4^2)) - 3*p1*(8 + 6*r_cm4 - 6*r_cm4^2 + 3*p2*(5 - 3*r_cm4 + 3*r_cm4^2))) + (x[,"n_01"] + x[,"n_33"])*(log(pmax(1e-6,1 - r_cm4)) + log(-4*(-4*(1 + r_cm4^2) + p2*(-3 + 6*r_cm4^2)) - 3*p1*(-4 + 6*r_cm4 - 4*r_cm4^2 + p2*(-3 - 9*r_cm4 + 6*r_cm4^2)))) + (x[,"n_02"] + x[,"n_32"])*log(-4*(-8 + 6*r_cm4 - 6*r_cm4^2 + p2*(-6 - 9*r_cm4 + 9*r_cm4^2)) + 3*p1*(-4 + 6*r_cm4 - 6*r_cm4^2 + p2*(-3 - 9*r_cm4 + 9*r_cm4^2))) + (x[,"n_11"] + x[,"n_23"])*log(p1*(6*(2 + 3*r_cm4 - 11*r_cm4^2 + 6*r_cm4^3) - 9*p2*(-4 + 6*r_cm4 - 11*r_cm4^2 + 6*r_cm4^3)) - 4*(-4*(4 - 4*r_cm4^2 + 3*r_cm4^3) + 3*p2*(2 + 3*r_cm4 - 8*r_cm4^2 + 6*r_cm4^3))) + (x[,"n_13"] + x[,"n_21"])*log(48 + 2*(-8 + 3*p1)*r_cm4 + (80 + 42*p1)*r_cm4^2 - 12*(4 + 3*p1)*r_cm4^3 + 3*p2*(4*(-3 + 5*r_cm4 - 10*r_cm4^2 + 6*r_cm4^3) + 3*p1*(3 + 2*r_cm4 - 7*r_cm4^2 + 6*r_cm4^3)))


logL_cr1 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-4*n00 - 3*n01 - 4*n02 - 3*n03 - 4*n04 - 4*n10 - 4*n11 - 4*n12 - 4*n13 - 4*n14 - 4*n20 - 4*n21 - 4*n22 - 4*n23 - 4*n24 - 4*n30 - 3*n31 - 4*n32 - 3*n33 - 4*n34)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(3) + (n00 + n34)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n01 + n33)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log((2 - 3*p1)*(-1 + r)^2 + ((4 + 3*p1)*r)/2)) + (n04 + n30)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n10 + n24)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n14 + n20)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + 3*r^2)) + (n03 + n31)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(-((4 + 3*p1)*(-1 + r))/2 + (2 - 3*p1)*r^2)) + (n02 + n32)*(log(2 - 3*p2) + log(-2*(-1 - r + r^2) - 3*p1*(1 - 5*r + 5*r^2))) + (n12 + n22)*log(-6*p2*(-4 - r + r^2) + 4*(8 - r + r^2) + p1*(-9*p2*(-2 - 5*r + 5*r^2) + 6*(4 - 5*r + 5*r^2))) + (n11 + n23)*log(4*(4 - 4*r + 8*r^2 - 6*r^3 + 3*p2*(1 - r - 4*r^2 + 3*r^3)) - 3*p1*(8 - 20*r + 22*r^2 - 12*r^3 + 3*p2*(2 + 4*r - 11*r^2 + 6*r^3))) + (n13 + n21)*log(4*(2 + 6*r - 10*r^2 + 6*r^3 - 3*p2*(1 - 5*r^2 + 3*r^3)) + p1*(6*(1 - 6*r + 7*r^2 - 6*r^3) + 9*p2*(-1 - 7*r^2 + 6*r^3)))
return(L)}
interlogL_cr1 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cr1,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr1 <- parallel::mcmapply(interlogL_cr1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cr1 <- -((x[,"n_13"] + x[,"n_21"])*log10(p1*(-6 - 18*p2) + 4*(13/4 - (3*p2)/8))) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(3) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(3) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(log10(5/2 + (3*p1)/4) + log10(2 - 3*p2)) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10((2 - 3*p1)/4 + (4 + 3*p1)/4) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10((2 - 3*p1)/4 + (4 + 3*p1)/4) + log10(2 - 3*p2)) - (x[,"n_11"] + x[,"n_23"])*log10(4*(13/4 - (3*p2)/8) - 3*p1*(2 + 6*p2)) - (x[,"n_12"] + x[,"n_22"])*log10(31 + (51*p2)/2 + p1*(33/2 + (117*p2)/4)) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cr1)) + 2*log10(pmax(1e-6,r_cr1))) + (x[,"n_01"] + x[,"n_33"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_cr1)) + log10((2 - 3*p1)*(-1 + r_cr1)^2 + ((4 + 3*p1)*r_cr1)/2)) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_cr1)) + log10(1 - 2*r_cr1 + r_cr1^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_cr1)) + log10(2 - 4*r_cr1 + 3*r_cr1^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cr1)) + log10(1 - 2*r_cr1 + 3*r_cr1^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cr1)) + log10(-((4 + 3*p1)*(-1 + r_cr1))/2 + (2 - 3*p1)*r_cr1^2)) + (x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p2) + log10(-2*(-1 - r_cr1 + r_cr1^2) - 3*p1*(1 - 5*r_cr1 + 5*r_cr1^2))) + (x[,"n_12"] + x[,"n_22"])*log10(-6*p2*(-4 - r_cr1 + r_cr1^2) + 4*(8 - r_cr1 + r_cr1^2) + p1*(-9*p2*(-2 - 5*r_cr1 + 5*r_cr1^2) + 6*(4 - 5*r_cr1 + 5*r_cr1^2))) + (x[,"n_11"] + x[,"n_23"])*log10(4*(4 - 4*r_cr1 + 8*r_cr1^2 - 6*r_cr1^3 + 3*p2*(1 - r_cr1 - 4*r_cr1^2 + 3*r_cr1^3)) - 3*p1*(8 - 20*r_cr1 + 22*r_cr1^2 - 12*r_cr1^3 + 3*p2*(2 + 4*r_cr1 - 11*r_cr1^2 + 6*r_cr1^3))) + (x[,"n_13"] + x[,"n_21"])*log10(4*(2 + 6*r_cr1 - 10*r_cr1^2 + 6*r_cr1^3 - 3*p2*(1 - 5*r_cr1^2 + 3*r_cr1^3)) + p1*(6*(1 - 6*r_cr1 + 7*r_cr1^2 - 6*r_cr1^3) + 9*p2*(-1 - 7*r_cr1^2 + 6*r_cr1^3)))


logL_cr1 <- (-4*x[,"n_00"] - 3*x[,"n_01"] - 4*x[,"n_02"] - 3*x[,"n_03"] - 4*x[,"n_04"] - 4*x[,"n_10"] - 4*x[,"n_11"] - 4*x[,"n_12"] - 4*x[,"n_13"] - 4*x[,"n_14"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 4*x[,"n_23"] - 4*x[,"n_24"] - 4*x[,"n_30"] - 3*x[,"n_31"] - 4*x[,"n_32"] - 3*x[,"n_33"] - 4*x[,"n_34"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(3) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cr1)) + 2*log(pmax(1e-6,r_cr1))) + (x[,"n_01"] + x[,"n_33"])*(log(2 - 3*p2) + log(pmax(1e-6,r_cr1)) + log((2 - 3*p1)*(-1 + r_cr1)^2 + ((4 + 3*p1)*r_cr1)/2)) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_cr1)) + log(1 - 2*r_cr1 + r_cr1^2)) + (x[,"n_10"] + x[,"n_24"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_cr1)) + log(2 - 4*r_cr1 + 3*r_cr1^2)) + (x[,"n_14"] + x[,"n_20"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cr1)) + log(1 - 2*r_cr1 + 3*r_cr1^2)) + (x[,"n_03"] + x[,"n_31"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_cr1)) + log(-((4 + 3*p1)*(-1 + r_cr1))/2 + (2 - 3*p1)*r_cr1^2)) + (x[,"n_02"] + x[,"n_32"])*(log(2 - 3*p2) + log(-2*(-1 - r_cr1 + r_cr1^2) - 3*p1*(1 - 5*r_cr1 + 5*r_cr1^2))) + (x[,"n_12"] + x[,"n_22"])*log(-6*p2*(-4 - r_cr1 + r_cr1^2) + 4*(8 - r_cr1 + r_cr1^2) + p1*(-9*p2*(-2 - 5*r_cr1 + 5*r_cr1^2) + 6*(4 - 5*r_cr1 + 5*r_cr1^2))) + (x[,"n_11"] + x[,"n_23"])*log(4*(4 - 4*r_cr1 + 8*r_cr1^2 - 6*r_cr1^3 + 3*p2*(1 - r_cr1 - 4*r_cr1^2 + 3*r_cr1^3)) - 3*p1*(8 - 20*r_cr1 + 22*r_cr1^2 - 12*r_cr1^3 + 3*p2*(2 + 4*r_cr1 - 11*r_cr1^2 + 6*r_cr1^3))) + (x[,"n_13"] + x[,"n_21"])*log(4*(2 + 6*r_cr1 - 10*r_cr1^2 + 6*r_cr1^3 - 3*p2*(1 - 5*r_cr1^2 + 3*r_cr1^3)) + p1*(6*(1 - 6*r_cr1 + 7*r_cr1^2 - 6*r_cr1^3) + 9*p2*(-1 - 7*r_cr1^2 + 6*r_cr1^3)))


logL_cr2 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(5*log(2) + 2*log(3)) + (n00 + n34)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n04 + n30)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n10 + n24)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n14 + n20)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + 3*r^2)) + (n01 + n33)*(log(4 + 3*p2) + log(pmax(1e-6,r)) + log(4*(1 - r + r^2) - 3*p1*(2 - 5*r + 2*r^2))) + (n03 + n31)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(4*(1 - r + r^2) - 3*p1*(-1 + r + 2*r^2))) + (n02 + n32)*(log(4 + 3*p2) + log(-2*(-1 - r + r^2) - 3*p1*(1 - 5*r + 5*r^2))) + (n12 + n22)*log(6*p2*(-4 - r + r^2) + 8*(8 - r + r^2) + 3*p1*(3*p2*(-2 - 5*r + 5*r^2) + 4*(4 - 5*r + 5*r^2))) + (n13 + n21)*log(p1*(p2*(9 + 63*r^2 - 54*r^3) + 12*(1 - 6*r + 7*r^2 - 6*r^3)) + 4*(3*p2*(1 - 5*r^2 + 3*r^3) + 4*(1 + 3*r - 5*r^2 + 3*r^3))) + (n11 + n23)*log(-4*(3*p2*(1 - r - 4*r^2 + 3*r^3) + 4*(-2 + 2*r - 4*r^2 + 3*r^3)) + 3*p1*(3*p2*(2 + 4*r - 11*r^2 + 6*r^3) + 4*(-4 + 10*r - 11*r^2 + 6*r^3)))
return(L)}
interlogL_cr2 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cr2,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr2 <- parallel::mcmapply(interlogL_cr2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cr2 <- -((x[,"n_12"] + x[,"n_22"])*log10(62 + 3*p1*(11 - (39*p2)/4) - (51*p2)/2)) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(3) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(3) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(log10(5/2 + (3*p1)/4) + log10(4 + 3*p2)) - (x[,"n_11"] + x[,"n_23"])*log10(-4*(-13/2 - (3*p2)/8) + 3*p1*(-4 + 6*p2)) - (x[,"n_13"] + x[,"n_21"])*log10(4*(13/2 + (3*p2)/8) + p1*(-12 + 18*p2)) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cr2)) + 2*log10(pmax(1e-6,r_cr2))) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_cr2)) + log10(1 - 2*r_cr2 + r_cr2^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_cr2)) + log10(2 - 4*r_cr2 + 3*r_cr2^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cr2)) + log10(1 - 2*r_cr2 + 3*r_cr2^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(4 + 3*p2) + log10(pmax(1e-6,r_cr2)) + log10(4*(1 - r_cr2 + r_cr2^2) - 3*p1*(2 - 5*r_cr2 + 2*r_cr2^2))) + (x[,"n_03"] + x[,"n_31"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cr2)) + log10(4*(1 - r_cr2 + r_cr2^2) - 3*p1*(-1 + r_cr2 + 2*r_cr2^2))) + (x[,"n_02"] + x[,"n_32"])*(log10(4 + 3*p2) + log10(-2*(-1 - r_cr2 + r_cr2^2) - 3*p1*(1 - 5*r_cr2 + 5*r_cr2^2))) + (x[,"n_12"] + x[,"n_22"])*log10(6*p2*(-4 - r_cr2 + r_cr2^2) + 8*(8 - r_cr2 + r_cr2^2) + 3*p1*(3*p2*(-2 - 5*r_cr2 + 5*r_cr2^2) + 4*(4 - 5*r_cr2 + 5*r_cr2^2))) + (x[,"n_13"] + x[,"n_21"])*log10(p1*(p2*(9 + 63*r_cr2^2 - 54*r_cr2^3) + 12*(1 - 6*r_cr2 + 7*r_cr2^2 - 6*r_cr2^3)) + 4*(3*p2*(1 - 5*r_cr2^2 + 3*r_cr2^3) + 4*(1 + 3*r_cr2 - 5*r_cr2^2 + 3*r_cr2^3))) + (x[,"n_11"] + x[,"n_23"])*log10(-4*(3*p2*(1 - r_cr2 - 4*r_cr2^2 + 3*r_cr2^3) + 4*(-2 + 2*r_cr2 - 4*r_cr2^2 + 3*r_cr2^3)) + 3*p1*(3*p2*(2 + 4*r_cr2 - 11*r_cr2^2 + 6*r_cr2^3) + 4*(-4 + 10*r_cr2 - 11*r_cr2^2 + 6*r_cr2^3)))


logL_cr2 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(5*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cr2)) + 2*log(pmax(1e-6,r_cr2))) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_cr2)) + log(1 - 2*r_cr2 + r_cr2^2)) + (x[,"n_10"] + x[,"n_24"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_cr2)) + log(2 - 4*r_cr2 + 3*r_cr2^2)) + (x[,"n_14"] + x[,"n_20"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cr2)) + log(1 - 2*r_cr2 + 3*r_cr2^2)) + (x[,"n_01"] + x[,"n_33"])*(log(4 + 3*p2) + log(pmax(1e-6,r_cr2)) + log(4*(1 - r_cr2 + r_cr2^2) - 3*p1*(2 - 5*r_cr2 + 2*r_cr2^2))) + (x[,"n_03"] + x[,"n_31"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_cr2)) + log(4*(1 - r_cr2 + r_cr2^2) - 3*p1*(-1 + r_cr2 + 2*r_cr2^2))) + (x[,"n_02"] + x[,"n_32"])*(log(4 + 3*p2) + log(-2*(-1 - r_cr2 + r_cr2^2) - 3*p1*(1 - 5*r_cr2 + 5*r_cr2^2))) + (x[,"n_12"] + x[,"n_22"])*log(6*p2*(-4 - r_cr2 + r_cr2^2) + 8*(8 - r_cr2 + r_cr2^2) + 3*p1*(3*p2*(-2 - 5*r_cr2 + 5*r_cr2^2) + 4*(4 - 5*r_cr2 + 5*r_cr2^2))) + (x[,"n_13"] + x[,"n_21"])*log(p1*(p2*(9 + 63*r_cr2^2 - 54*r_cr2^3) + 12*(1 - 6*r_cr2 + 7*r_cr2^2 - 6*r_cr2^3)) + 4*(3*p2*(1 - 5*r_cr2^2 + 3*r_cr2^3) + 4*(1 + 3*r_cr2 - 5*r_cr2^2 + 3*r_cr2^3))) + (x[,"n_11"] + x[,"n_23"])*log(-4*(3*p2*(1 - r_cr2 - 4*r_cr2^2 + 3*r_cr2^3) + 4*(-2 + 2*r_cr2 - 4*r_cr2^2 + 3*r_cr2^3)) + 3*p1*(3*p2*(2 + 4*r_cr2 - 11*r_cr2^2 + 6*r_cr2^3) + 4*(-4 + 10*r_cr2 - 11*r_cr2^2 + 6*r_cr2^3)))


logL_cr3 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(5*log(2) + 2*log(3)) + (n00 + n34)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n04 + n30)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n10 + n24)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n14 + n20)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + 3*r^2)) + (n01 + n33)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(8*(1 - r + r^2) + 3*p1*(2 - 5*r + 2*r^2))) + (n03 + n31)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(8*(1 - r + r^2) + 3*p1*(-1 + r + 2*r^2))) + (n02 + n32)*(log(2 - 3*p2) + log(4 + 4*r - 4*r^2 + 3*p1*(1 - 5*r + 5*r^2))) + (n12 + n22)*log(-12*p2*(-4 - r + r^2) + 8*(8 - r + r^2) + 3*p1*(3*p2*(-2 - 5*r + 5*r^2) - 2*(4 - 5*r + 5*r^2))) + (n11 + n23)*log(8*(4 - 4*r + 8*r^2 - 6*r^3 + 3*p2*(1 - r - 4*r^2 + 3*r^3)) + 3*p1*(8 - 20*r + 22*r^2 - 12*r^3 + 3*p2*(2 + 4*r - 11*r^2 + 6*r^3))) + (n13 + n21)*log(-8*(3*p2*(1 - 5*r^2 + 3*r^3) - 2*(1 + 3*r - 5*r^2 + 3*r^3)) + p1*(p2*(9 + 63*r^2 - 54*r^3) + 6*(-1 + 6*r - 7*r^2 + 6*r^3)))
return(L)}
interlogL_cr3 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cr3,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr3 <- parallel::mcmapply(interlogL_cr3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cr3 <- -((x[,"n_03"] + x[,"n_31"])*(log10(3) + log10(2 - 3*p2))) - (x[,"n_01"] + x[,"n_33"])*(log10(3) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(log10(5 - (3*p1)/4) + log10(2 - 3*p2)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(3) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(3) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_12"] + x[,"n_22"])*log10(62 + 3*p1*(-11/2 - (39*p2)/4) + 51*p2) - (x[,"n_11"] + x[,"n_23"])*log10(8*(13/4 - (3*p2)/8) + 3*p1*(2 + 6*p2)) - (x[,"n_13"] + x[,"n_21"])*log10(-8*(-13/4 + (3*p2)/8) + p1*(6 + 18*p2)) + (x[,"n_00"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cr3)) + 2*log10(pmax(1e-6,r_cr3))) + (x[,"n_04"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_cr3)) + log10(1 - 2*r_cr3 + r_cr3^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_cr3)) + log10(2 - 4*r_cr3 + 3*r_cr3^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cr3)) + log10(1 - 2*r_cr3 + 3*r_cr3^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_cr3)) + log10(8*(1 - r_cr3 + r_cr3^2) + 3*p1*(2 - 5*r_cr3 + 2*r_cr3^2))) + (x[,"n_03"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cr3)) + log10(8*(1 - r_cr3 + r_cr3^2) + 3*p1*(-1 + r_cr3 + 2*r_cr3^2))) + (x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p2) + log10(4 + 4*r_cr3 - 4*r_cr3^2 + 3*p1*(1 - 5*r_cr3 + 5*r_cr3^2))) + (x[,"n_12"] + x[,"n_22"])*log10(-12*p2*(-4 - r_cr3 + r_cr3^2) + 8*(8 - r_cr3 + r_cr3^2) + 3*p1*(3*p2*(-2 - 5*r_cr3 + 5*r_cr3^2) - 2*(4 - 5*r_cr3 + 5*r_cr3^2))) + (x[,"n_11"] + x[,"n_23"])*log10(8*(4 - 4*r_cr3 + 8*r_cr3^2 - 6*r_cr3^3 + 3*p2*(1 - r_cr3 - 4*r_cr3^2 + 3*r_cr3^3)) + 3*p1*(8 - 20*r_cr3 + 22*r_cr3^2 - 12*r_cr3^3 + 3*p2*(2 + 4*r_cr3 - 11*r_cr3^2 + 6*r_cr3^3))) + (x[,"n_13"] + x[,"n_21"])*log10(-8*(3*p2*(1 - 5*r_cr3^2 + 3*r_cr3^3) - 2*(1 + 3*r_cr3 - 5*r_cr3^2 + 3*r_cr3^3)) + p1*(p2*(9 + 63*r_cr3^2 - 54*r_cr3^3) + 6*(-1 + 6*r_cr3 - 7*r_cr3^2 + 6*r_cr3^3)))


logL_cr3 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(5*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_34"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cr3)) + 2*log(pmax(1e-6,r_cr3))) + (x[,"n_04"] + x[,"n_30"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_cr3)) + log(1 - 2*r_cr3 + r_cr3^2)) + (x[,"n_10"] + x[,"n_24"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_cr3)) + log(2 - 4*r_cr3 + 3*r_cr3^2)) + (x[,"n_14"] + x[,"n_20"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cr3)) + log(1 - 2*r_cr3 + 3*r_cr3^2)) + (x[,"n_01"] + x[,"n_33"])*(log(2 - 3*p2) + log(pmax(1e-6,r_cr3)) + log(8*(1 - r_cr3 + r_cr3^2) + 3*p1*(2 - 5*r_cr3 + 2*r_cr3^2))) + (x[,"n_03"] + x[,"n_31"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_cr3)) + log(8*(1 - r_cr3 + r_cr3^2) + 3*p1*(-1 + r_cr3 + 2*r_cr3^2))) + (x[,"n_02"] + x[,"n_32"])*(log(2 - 3*p2) + log(4 + 4*r_cr3 - 4*r_cr3^2 + 3*p1*(1 - 5*r_cr3 + 5*r_cr3^2))) + (x[,"n_12"] + x[,"n_22"])*log(-12*p2*(-4 - r_cr3 + r_cr3^2) + 8*(8 - r_cr3 + r_cr3^2) + 3*p1*(3*p2*(-2 - 5*r_cr3 + 5*r_cr3^2) - 2*(4 - 5*r_cr3 + 5*r_cr3^2))) + (x[,"n_11"] + x[,"n_23"])*log(8*(4 - 4*r_cr3 + 8*r_cr3^2 - 6*r_cr3^3 + 3*p2*(1 - r_cr3 - 4*r_cr3^2 + 3*r_cr3^3)) + 3*p1*(8 - 20*r_cr3 + 22*r_cr3^2 - 12*r_cr3^3 + 3*p2*(2 + 4*r_cr3 - 11*r_cr3^2 + 6*r_cr3^3))) + (x[,"n_13"] + x[,"n_21"])*log(-8*(3*p2*(1 - 5*r_cr3^2 + 3*r_cr3^3) - 2*(1 + 3*r_cr3 - 5*r_cr3^2 + 3*r_cr3^3)) + p1*(p2*(9 + 63*r_cr3^2 - 54*r_cr3^3) + 6*(-1 + 6*r_cr3 - 7*r_cr3^2 + 6*r_cr3^3)))


logL_cr4 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(6*log(2) + 2*log(3)) + (n00 + n34)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n04 + n30)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n10 + n24)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n14 + n20)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + 3*r^2)) + (n01 + n33)*(log(4 + 3*p2) + log(pmax(1e-6,r)) + log(8*(1 - r + r^2) + 3*p1*(2 - 5*r + 2*r^2))) + (n03 + n31)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(8*(1 - r + r^2) + 3*p1*(-1 + r + 2*r^2))) + (n02 + n32)*(log(4 + 3*p2) + log(4 + 4*r - 4*r^2 + 3*p1*(1 - 5*r + 5*r^2))) + (n12 + n22)*log(4*(3*p2*(-4 - r + r^2) + 4*(8 - r + r^2)) + p1*(-9*p2*(-2 - 5*r + 5*r^2) - 12*(4 - 5*r + 5*r^2))) + (n11 + n23)*log(-8*(3*p2*(1 - r - 4*r^2 + 3*r^3) + 4*(-2 + 2*r - 4*r^2 + 3*r^3)) - 3*p1*(3*p2*(2 + 4*r - 11*r^2 + 6*r^3) + 4*(-4 + 10*r - 11*r^2 + 6*r^3))) + (n13 + n21)*log(8*(3*p2*(1 - 5*r^2 + 3*r^3) + 4*(1 + 3*r - 5*r^2 + 3*r^3)) + 3*p1*(3*p2*(-1 - 7*r^2 + 6*r^3) + 4*(-1 + 6*r - 7*r^2 + 6*r^3)))
return(L)}
interlogL_cr4 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cr4,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr4 <- parallel::mcmapply(interlogL_cr4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cr4 <- -((x[,"n_13"] + x[,"n_21"])*log10(3*p1*(4 - 6*p2) + 8*(13/2 + (3*p2)/8))) - (x[,"n_03"] + x[,"n_31"])*(log10(3) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_33"])*(log10(3) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(log10(5 - (3*p1)/4) + log10(4 + 3*p2)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(3) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(3) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_11"] + x[,"n_23"])*log10(-8*(-13/2 - (3*p2)/8) - 3*p1*(-4 + 6*p2)) - (x[,"n_12"] + x[,"n_22"])*log10(4*(31 - (51*p2)/4) + p1*(-33 + (117*p2)/4)) + (x[,"n_00"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cr4)) + 2*log10(pmax(1e-6,r_cr4))) + (x[,"n_04"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_cr4)) + log10(1 - 2*r_cr4 + r_cr4^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_cr4)) + log10(2 - 4*r_cr4 + 3*r_cr4^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cr4)) + log10(1 - 2*r_cr4 + 3*r_cr4^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(4 + 3*p2) + log10(pmax(1e-6,r_cr4)) + log10(8*(1 - r_cr4 + r_cr4^2) + 3*p1*(2 - 5*r_cr4 + 2*r_cr4^2))) + (x[,"n_03"] + x[,"n_31"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cr4)) + log10(8*(1 - r_cr4 + r_cr4^2) + 3*p1*(-1 + r_cr4 + 2*r_cr4^2))) + (x[,"n_02"] + x[,"n_32"])*(log10(4 + 3*p2) + log10(4 + 4*r_cr4 - 4*r_cr4^2 + 3*p1*(1 - 5*r_cr4 + 5*r_cr4^2))) + (x[,"n_12"] + x[,"n_22"])*log10(4*(3*p2*(-4 - r_cr4 + r_cr4^2) + 4*(8 - r_cr4 + r_cr4^2)) + p1*(-9*p2*(-2 - 5*r_cr4 + 5*r_cr4^2) - 12*(4 - 5*r_cr4 + 5*r_cr4^2))) + (x[,"n_11"] + x[,"n_23"])*log10(-8*(3*p2*(1 - r_cr4 - 4*r_cr4^2 + 3*r_cr4^3) + 4*(-2 + 2*r_cr4 - 4*r_cr4^2 + 3*r_cr4^3)) - 3*p1*(3*p2*(2 + 4*r_cr4 - 11*r_cr4^2 + 6*r_cr4^3) + 4*(-4 + 10*r_cr4 - 11*r_cr4^2 + 6*r_cr4^3))) + (x[,"n_13"] + x[,"n_21"])*log10(8*(3*p2*(1 - 5*r_cr4^2 + 3*r_cr4^3) + 4*(1 + 3*r_cr4 - 5*r_cr4^2 + 3*r_cr4^3)) + 3*p1*(3*p2*(-1 - 7*r_cr4^2 + 6*r_cr4^3) + 4*(-1 + 6*r_cr4 - 7*r_cr4^2 + 6*r_cr4^3)))


logL_cr4 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(6*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_34"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cr4)) + 2*log(pmax(1e-6,r_cr4))) + (x[,"n_04"] + x[,"n_30"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_cr4)) + log(1 - 2*r_cr4 + r_cr4^2)) + (x[,"n_10"] + x[,"n_24"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_cr4)) + log(2 - 4*r_cr4 + 3*r_cr4^2)) + (x[,"n_14"] + x[,"n_20"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cr4)) + log(1 - 2*r_cr4 + 3*r_cr4^2)) + (x[,"n_01"] + x[,"n_33"])*(log(4 + 3*p2) + log(pmax(1e-6,r_cr4)) + log(8*(1 - r_cr4 + r_cr4^2) + 3*p1*(2 - 5*r_cr4 + 2*r_cr4^2))) + (x[,"n_03"] + x[,"n_31"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_cr4)) + log(8*(1 - r_cr4 + r_cr4^2) + 3*p1*(-1 + r_cr4 + 2*r_cr4^2))) + (x[,"n_02"] + x[,"n_32"])*(log(4 + 3*p2) + log(4 + 4*r_cr4 - 4*r_cr4^2 + 3*p1*(1 - 5*r_cr4 + 5*r_cr4^2))) + (x[,"n_12"] + x[,"n_22"])*log(4*(3*p2*(-4 - r_cr4 + r_cr4^2) + 4*(8 - r_cr4 + r_cr4^2)) + p1*(-9*p2*(-2 - 5*r_cr4 + 5*r_cr4^2) - 12*(4 - 5*r_cr4 + 5*r_cr4^2))) + (x[,"n_11"] + x[,"n_23"])*log(-8*(3*p2*(1 - r_cr4 - 4*r_cr4^2 + 3*r_cr4^3) + 4*(-2 + 2*r_cr4 - 4*r_cr4^2 + 3*r_cr4^3)) - 3*p1*(3*p2*(2 + 4*r_cr4 - 11*r_cr4^2 + 6*r_cr4^3) + 4*(-4 + 10*r_cr4 - 11*r_cr4^2 + 6*r_cr4^3))) + (x[,"n_13"] + x[,"n_21"])*log(8*(3*p2*(1 - 5*r_cr4^2 + 3*r_cr4^3) + 4*(1 + 3*r_cr4 - 5*r_cr4^2 + 3*r_cr4^3)) + 3*p1*(3*p2*(-1 - 7*r_cr4^2 + 6*r_cr4^3) + 4*(-1 + 6*r_cr4 - 7*r_cr4^2 + 6*r_cr4^3)))


logL_rc1 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(5*log(2) + 2*log(3)) + (n04 + n30)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n00 + n34)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n14 + n20)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n10 + n24)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + 3*r^2)) + (n03 + n31)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(8*(1 - r + r^2) + 3*p1*(2 - 5*r + 2*r^2))) + (n01 + n33)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(8*(1 - r + r^2) + 3*p1*(-1 + r + 2*r^2))) + (n02 + n32)*(log(2 - 3*p2) + log(4 + 4*r - 4*r^2 + 3*p1*(1 - 5*r + 5*r^2))) + (n12 + n22)*log(-12*p2*(-4 - r + r^2) + 8*(8 - r + r^2) + 3*p1*(3*p2*(-2 - 5*r + 5*r^2) - 2*(4 - 5*r + 5*r^2))) + (n13 + n21)*log(8*(4 - 4*r + 8*r^2 - 6*r^3 + 3*p2*(1 - r - 4*r^2 + 3*r^3)) + 3*p1*(8 - 20*r + 22*r^2 - 12*r^3 + 3*p2*(2 + 4*r - 11*r^2 + 6*r^3))) + (n11 + n23)*log(-8*(3*p2*(1 - 5*r^2 + 3*r^3) - 2*(1 + 3*r - 5*r^2 + 3*r^3)) + p1*(p2*(9 + 63*r^2 - 54*r^3) + 6*(-1 + 6*r - 7*r^2 + 6*r^3)))
return(L)}
interlogL_rc1 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rc1,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc1 <- parallel::mcmapply(interlogL_rc1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rc1 <- -((x[,"n_03"] + x[,"n_31"])*(log10(3) + log10(2 - 3*p2))) - (x[,"n_01"] + x[,"n_33"])*(log10(3) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(log10(5 - (3*p1)/4) + log10(2 - 3*p2)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(3) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(3) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_12"] + x[,"n_22"])*log10(62 + 3*p1*(-11/2 - (39*p2)/4) + 51*p2) - (x[,"n_13"] + x[,"n_21"])*log10(8*(13/4 - (3*p2)/8) + 3*p1*(2 + 6*p2)) - (x[,"n_11"] + x[,"n_23"])*log10(-8*(-13/4 + (3*p2)/8) + p1*(6 + 18*p2)) + (x[,"n_04"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rc1)) + 2*log10(pmax(1e-6,r_rc1))) + (x[,"n_00"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_rc1)) + log10(1 - 2*r_rc1 + r_rc1^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_rc1)) + log10(2 - 4*r_rc1 + 3*r_rc1^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rc1)) + log10(1 - 2*r_rc1 + 3*r_rc1^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_rc1)) + log10(8*(1 - r_rc1 + r_rc1^2) + 3*p1*(2 - 5*r_rc1 + 2*r_rc1^2))) + (x[,"n_01"] + x[,"n_33"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rc1)) + log10(8*(1 - r_rc1 + r_rc1^2) + 3*p1*(-1 + r_rc1 + 2*r_rc1^2))) + (x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p2) + log10(4 + 4*r_rc1 - 4*r_rc1^2 + 3*p1*(1 - 5*r_rc1 + 5*r_rc1^2))) + (x[,"n_12"] + x[,"n_22"])*log10(-12*p2*(-4 - r_rc1 + r_rc1^2) + 8*(8 - r_rc1 + r_rc1^2) + 3*p1*(3*p2*(-2 - 5*r_rc1 + 5*r_rc1^2) - 2*(4 - 5*r_rc1 + 5*r_rc1^2))) + (x[,"n_13"] + x[,"n_21"])*log10(8*(4 - 4*r_rc1 + 8*r_rc1^2 - 6*r_rc1^3 + 3*p2*(1 - r_rc1 - 4*r_rc1^2 + 3*r_rc1^3)) + 3*p1*(8 - 20*r_rc1 + 22*r_rc1^2 - 12*r_rc1^3 + 3*p2*(2 + 4*r_rc1 - 11*r_rc1^2 + 6*r_rc1^3))) + (x[,"n_11"] + x[,"n_23"])*log10(-8*(3*p2*(1 - 5*r_rc1^2 + 3*r_rc1^3) - 2*(1 + 3*r_rc1 - 5*r_rc1^2 + 3*r_rc1^3)) + p1*(p2*(9 + 63*r_rc1^2 - 54*r_rc1^3) + 6*(-1 + 6*r_rc1 - 7*r_rc1^2 + 6*r_rc1^3)))


logL_rc1 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(5*log(2) + 2*log(3)) + (x[,"n_04"] + x[,"n_30"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rc1)) + 2*log(pmax(1e-6,r_rc1))) + (x[,"n_00"] + x[,"n_34"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_rc1)) + log(1 - 2*r_rc1 + r_rc1^2)) + (x[,"n_14"] + x[,"n_20"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_rc1)) + log(2 - 4*r_rc1 + 3*r_rc1^2)) + (x[,"n_10"] + x[,"n_24"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rc1)) + log(1 - 2*r_rc1 + 3*r_rc1^2)) + (x[,"n_03"] + x[,"n_31"])*(log(2 - 3*p2) + log(pmax(1e-6,r_rc1)) + log(8*(1 - r_rc1 + r_rc1^2) + 3*p1*(2 - 5*r_rc1 + 2*r_rc1^2))) + (x[,"n_01"] + x[,"n_33"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_rc1)) + log(8*(1 - r_rc1 + r_rc1^2) + 3*p1*(-1 + r_rc1 + 2*r_rc1^2))) + (x[,"n_02"] + x[,"n_32"])*(log(2 - 3*p2) + log(4 + 4*r_rc1 - 4*r_rc1^2 + 3*p1*(1 - 5*r_rc1 + 5*r_rc1^2))) + (x[,"n_12"] + x[,"n_22"])*log(-12*p2*(-4 - r_rc1 + r_rc1^2) + 8*(8 - r_rc1 + r_rc1^2) + 3*p1*(3*p2*(-2 - 5*r_rc1 + 5*r_rc1^2) - 2*(4 - 5*r_rc1 + 5*r_rc1^2))) + (x[,"n_13"] + x[,"n_21"])*log(8*(4 - 4*r_rc1 + 8*r_rc1^2 - 6*r_rc1^3 + 3*p2*(1 - r_rc1 - 4*r_rc1^2 + 3*r_rc1^3)) + 3*p1*(8 - 20*r_rc1 + 22*r_rc1^2 - 12*r_rc1^3 + 3*p2*(2 + 4*r_rc1 - 11*r_rc1^2 + 6*r_rc1^3))) + (x[,"n_11"] + x[,"n_23"])*log(-8*(3*p2*(1 - 5*r_rc1^2 + 3*r_rc1^3) - 2*(1 + 3*r_rc1 - 5*r_rc1^2 + 3*r_rc1^3)) + p1*(p2*(9 + 63*r_rc1^2 - 54*r_rc1^3) + 6*(-1 + 6*r_rc1 - 7*r_rc1^2 + 6*r_rc1^3)))


logL_rc2 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(6*log(2) + 2*log(3)) + (n04 + n30)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n00 + n34)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n14 + n20)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n10 + n24)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + 3*r^2)) + (n03 + n31)*(log(4 + 3*p2) + log(pmax(1e-6,r)) + log(8*(1 - r + r^2) + 3*p1*(2 - 5*r + 2*r^2))) + (n01 + n33)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(8*(1 - r + r^2) + 3*p1*(-1 + r + 2*r^2))) + (n02 + n32)*(log(4 + 3*p2) + log(4 + 4*r - 4*r^2 + 3*p1*(1 - 5*r + 5*r^2))) + (n12 + n22)*log(4*(3*p2*(-4 - r + r^2) + 4*(8 - r + r^2)) + p1*(-9*p2*(-2 - 5*r + 5*r^2) - 12*(4 - 5*r + 5*r^2))) + (n13 + n21)*log(-8*(3*p2*(1 - r - 4*r^2 + 3*r^3) + 4*(-2 + 2*r - 4*r^2 + 3*r^3)) - 3*p1*(3*p2*(2 + 4*r - 11*r^2 + 6*r^3) + 4*(-4 + 10*r - 11*r^2 + 6*r^3))) + (n11 + n23)*log(8*(3*p2*(1 - 5*r^2 + 3*r^3) + 4*(1 + 3*r - 5*r^2 + 3*r^3)) + 3*p1*(3*p2*(-1 - 7*r^2 + 6*r^3) + 4*(-1 + 6*r - 7*r^2 + 6*r^3)))
return(L)}
interlogL_rc2 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rc2,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc2 <- parallel::mcmapply(interlogL_rc2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rc2 <- -((x[,"n_11"] + x[,"n_23"])*log10(3*p1*(4 - 6*p2) + 8*(13/2 + (3*p2)/8))) - (x[,"n_03"] + x[,"n_31"])*(log10(3) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_33"])*(log10(3) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(log10(5 - (3*p1)/4) + log10(4 + 3*p2)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(3) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(3) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_13"] + x[,"n_21"])*log10(-8*(-13/2 - (3*p2)/8) - 3*p1*(-4 + 6*p2)) - (x[,"n_12"] + x[,"n_22"])*log10(4*(31 - (51*p2)/4) + p1*(-33 + (117*p2)/4)) + (x[,"n_04"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rc2)) + 2*log10(pmax(1e-6,r_rc2))) + (x[,"n_00"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_rc2)) + log10(1 - 2*r_rc2 + r_rc2^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_rc2)) + log10(2 - 4*r_rc2 + 3*r_rc2^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rc2)) + log10(1 - 2*r_rc2 + 3*r_rc2^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(4 + 3*p2) + log10(pmax(1e-6,r_rc2)) + log10(8*(1 - r_rc2 + r_rc2^2) + 3*p1*(2 - 5*r_rc2 + 2*r_rc2^2))) + (x[,"n_01"] + x[,"n_33"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rc2)) + log10(8*(1 - r_rc2 + r_rc2^2) + 3*p1*(-1 + r_rc2 + 2*r_rc2^2))) + (x[,"n_02"] + x[,"n_32"])*(log10(4 + 3*p2) + log10(4 + 4*r_rc2 - 4*r_rc2^2 + 3*p1*(1 - 5*r_rc2 + 5*r_rc2^2))) + (x[,"n_12"] + x[,"n_22"])*log10(4*(3*p2*(-4 - r_rc2 + r_rc2^2) + 4*(8 - r_rc2 + r_rc2^2)) + p1*(-9*p2*(-2 - 5*r_rc2 + 5*r_rc2^2) - 12*(4 - 5*r_rc2 + 5*r_rc2^2))) + (x[,"n_13"] + x[,"n_21"])*log10(-8*(3*p2*(1 - r_rc2 - 4*r_rc2^2 + 3*r_rc2^3) + 4*(-2 + 2*r_rc2 - 4*r_rc2^2 + 3*r_rc2^3)) - 3*p1*(3*p2*(2 + 4*r_rc2 - 11*r_rc2^2 + 6*r_rc2^3) + 4*(-4 + 10*r_rc2 - 11*r_rc2^2 + 6*r_rc2^3))) + (x[,"n_11"] + x[,"n_23"])*log10(8*(3*p2*(1 - 5*r_rc2^2 + 3*r_rc2^3) + 4*(1 + 3*r_rc2 - 5*r_rc2^2 + 3*r_rc2^3)) + 3*p1*(3*p2*(-1 - 7*r_rc2^2 + 6*r_rc2^3) + 4*(-1 + 6*r_rc2 - 7*r_rc2^2 + 6*r_rc2^3)))


logL_rc2 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(6*log(2) + 2*log(3)) + (x[,"n_04"] + x[,"n_30"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rc2)) + 2*log(pmax(1e-6,r_rc2))) + (x[,"n_00"] + x[,"n_34"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_rc2)) + log(1 - 2*r_rc2 + r_rc2^2)) + (x[,"n_14"] + x[,"n_20"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_rc2)) + log(2 - 4*r_rc2 + 3*r_rc2^2)) + (x[,"n_10"] + x[,"n_24"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rc2)) + log(1 - 2*r_rc2 + 3*r_rc2^2)) + (x[,"n_03"] + x[,"n_31"])*(log(4 + 3*p2) + log(pmax(1e-6,r_rc2)) + log(8*(1 - r_rc2 + r_rc2^2) + 3*p1*(2 - 5*r_rc2 + 2*r_rc2^2))) + (x[,"n_01"] + x[,"n_33"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_rc2)) + log(8*(1 - r_rc2 + r_rc2^2) + 3*p1*(-1 + r_rc2 + 2*r_rc2^2))) + (x[,"n_02"] + x[,"n_32"])*(log(4 + 3*p2) + log(4 + 4*r_rc2 - 4*r_rc2^2 + 3*p1*(1 - 5*r_rc2 + 5*r_rc2^2))) + (x[,"n_12"] + x[,"n_22"])*log(4*(3*p2*(-4 - r_rc2 + r_rc2^2) + 4*(8 - r_rc2 + r_rc2^2)) + p1*(-9*p2*(-2 - 5*r_rc2 + 5*r_rc2^2) - 12*(4 - 5*r_rc2 + 5*r_rc2^2))) + (x[,"n_13"] + x[,"n_21"])*log(-8*(3*p2*(1 - r_rc2 - 4*r_rc2^2 + 3*r_rc2^3) + 4*(-2 + 2*r_rc2 - 4*r_rc2^2 + 3*r_rc2^3)) - 3*p1*(3*p2*(2 + 4*r_rc2 - 11*r_rc2^2 + 6*r_rc2^3) + 4*(-4 + 10*r_rc2 - 11*r_rc2^2 + 6*r_rc2^3))) + (x[,"n_11"] + x[,"n_23"])*log(8*(3*p2*(1 - 5*r_rc2^2 + 3*r_rc2^3) + 4*(1 + 3*r_rc2 - 5*r_rc2^2 + 3*r_rc2^3)) + 3*p1*(3*p2*(-1 - 7*r_rc2^2 + 6*r_rc2^3) + 4*(-1 + 6*r_rc2 - 7*r_rc2^2 + 6*r_rc2^3)))


logL_rc3 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-4*n00 - 3*n01 - 4*n02 - 3*n03 - 4*n04 - 4*n10 - 4*n11 - 4*n12 - 4*n13 - 4*n14 - 4*n20 - 4*n21 - 4*n22 - 4*n23 - 4*n24 - 4*n30 - 3*n31 - 4*n32 - 3*n33 - 4*n34)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(3) + (n04 + n30)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n03 + n31)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log((2 - 3*p1)*(-1 + r)^2 + ((4 + 3*p1)*r)/2)) + (n00 + n34)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n14 + n20)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n10 + n24)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + 3*r^2)) + (n01 + n33)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(-((4 + 3*p1)*(-1 + r))/2 + (2 - 3*p1)*r^2)) + (n02 + n32)*(log(2 - 3*p2) + log(2*(1 + r - r^2) - 3*p1*(1 - 5*r + 5*r^2))) + (n12 + n22)*log(-6*p2*(-4 - r + r^2) + 4*(8 - r + r^2) + p1*(-9*p2*(-2 - 5*r + 5*r^2) + 6*(4 - 5*r + 5*r^2))) + (n13 + n21)*log(4*(4 - 4*r + 8*r^2 - 6*r^3 + 3*p2*(1 - r - 4*r^2 + 3*r^3)) - 3*p1*(8 - 20*r + 22*r^2 - 12*r^3 + 3*p2*(2 + 4*r - 11*r^2 + 6*r^3))) + (n11 + n23)*log(4*(2 + 6*r - 10*r^2 + 6*r^3 - 3*p2*(1 - 5*r^2 + 3*r^3)) + p1*(6*(1 - 6*r + 7*r^2 - 6*r^3) + 9*p2*(-1 - 7*r^2 + 6*r^3)))
return(L)}
interlogL_rc3 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rc3,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc3 <- parallel::mcmapply(interlogL_rc3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rc3 <- -((x[,"n_11"] + x[,"n_23"])*log10(p1*(-6 - 18*p2) + 4*(13/4 - (3*p2)/8))) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(3) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(3) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(log10(5/2 + (3*p1)/4) + log10(2 - 3*p2)) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10((2 - 3*p1)/4 + (4 + 3*p1)/4) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10((2 - 3*p1)/4 + (4 + 3*p1)/4) + log10(2 - 3*p2)) - (x[,"n_13"] + x[,"n_21"])*log10(4*(13/4 - (3*p2)/8) - 3*p1*(2 + 6*p2)) - (x[,"n_12"] + x[,"n_22"])*log10(31 + (51*p2)/2 + p1*(33/2 + (117*p2)/4)) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rc3)) + 2*log10(pmax(1e-6,r_rc3))) + (x[,"n_03"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_rc3)) + log10((2 - 3*p1)*(-1 + r_rc3)^2 + ((4 + 3*p1)*r_rc3)/2)) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_rc3)) + log10(1 - 2*r_rc3 + r_rc3^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_rc3)) + log10(2 - 4*r_rc3 + 3*r_rc3^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rc3)) + log10(1 - 2*r_rc3 + 3*r_rc3^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rc3)) + log10(-((4 + 3*p1)*(-1 + r_rc3))/2 + (2 - 3*p1)*r_rc3^2)) + (x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p2) + log10(2*(1 + r_rc3 - r_rc3^2) - 3*p1*(1 - 5*r_rc3 + 5*r_rc3^2))) + (x[,"n_12"] + x[,"n_22"])*log10(-6*p2*(-4 - r_rc3 + r_rc3^2) + 4*(8 - r_rc3 + r_rc3^2) + p1*(-9*p2*(-2 - 5*r_rc3 + 5*r_rc3^2) + 6*(4 - 5*r_rc3 + 5*r_rc3^2))) + (x[,"n_13"] + x[,"n_21"])*log10(4*(4 - 4*r_rc3 + 8*r_rc3^2 - 6*r_rc3^3 + 3*p2*(1 - r_rc3 - 4*r_rc3^2 + 3*r_rc3^3)) - 3*p1*(8 - 20*r_rc3 + 22*r_rc3^2 - 12*r_rc3^3 + 3*p2*(2 + 4*r_rc3 - 11*r_rc3^2 + 6*r_rc3^3))) + (x[,"n_11"] + x[,"n_23"])*log10(4*(2 + 6*r_rc3 - 10*r_rc3^2 + 6*r_rc3^3 - 3*p2*(1 - 5*r_rc3^2 + 3*r_rc3^3)) + p1*(6*(1 - 6*r_rc3 + 7*r_rc3^2 - 6*r_rc3^3) + 9*p2*(-1 - 7*r_rc3^2 + 6*r_rc3^3)))


logL_rc3 <- (-4*x[,"n_00"] - 3*x[,"n_01"] - 4*x[,"n_02"] - 3*x[,"n_03"] - 4*x[,"n_04"] - 4*x[,"n_10"] - 4*x[,"n_11"] - 4*x[,"n_12"] - 4*x[,"n_13"] - 4*x[,"n_14"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 4*x[,"n_23"] - 4*x[,"n_24"] - 4*x[,"n_30"] - 3*x[,"n_31"] - 4*x[,"n_32"] - 3*x[,"n_33"] - 4*x[,"n_34"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(3) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rc3)) + 2*log(pmax(1e-6,r_rc3))) + (x[,"n_03"] + x[,"n_31"])*(log(2 - 3*p2) + log(pmax(1e-6,r_rc3)) + log((2 - 3*p1)*(-1 + r_rc3)^2 + ((4 + 3*p1)*r_rc3)/2)) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_rc3)) + log(1 - 2*r_rc3 + r_rc3^2)) + (x[,"n_14"] + x[,"n_20"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_rc3)) + log(2 - 4*r_rc3 + 3*r_rc3^2)) + (x[,"n_10"] + x[,"n_24"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rc3)) + log(1 - 2*r_rc3 + 3*r_rc3^2)) + (x[,"n_01"] + x[,"n_33"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_rc3)) + log(-((4 + 3*p1)*(-1 + r_rc3))/2 + (2 - 3*p1)*r_rc3^2)) + (x[,"n_02"] + x[,"n_32"])*(log(2 - 3*p2) + log(2*(1 + r_rc3 - r_rc3^2) - 3*p1*(1 - 5*r_rc3 + 5*r_rc3^2))) + (x[,"n_12"] + x[,"n_22"])*log(-6*p2*(-4 - r_rc3 + r_rc3^2) + 4*(8 - r_rc3 + r_rc3^2) + p1*(-9*p2*(-2 - 5*r_rc3 + 5*r_rc3^2) + 6*(4 - 5*r_rc3 + 5*r_rc3^2))) + (x[,"n_13"] + x[,"n_21"])*log(4*(4 - 4*r_rc3 + 8*r_rc3^2 - 6*r_rc3^3 + 3*p2*(1 - r_rc3 - 4*r_rc3^2 + 3*r_rc3^3)) - 3*p1*(8 - 20*r_rc3 + 22*r_rc3^2 - 12*r_rc3^3 + 3*p2*(2 + 4*r_rc3 - 11*r_rc3^2 + 6*r_rc3^3))) + (x[,"n_11"] + x[,"n_23"])*log(4*(2 + 6*r_rc3 - 10*r_rc3^2 + 6*r_rc3^3 - 3*p2*(1 - 5*r_rc3^2 + 3*r_rc3^3)) + p1*(6*(1 - 6*r_rc3 + 7*r_rc3^2 - 6*r_rc3^3) + 9*p2*(-1 - 7*r_rc3^2 + 6*r_rc3^3)))


logL_rc4 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(5*log(2) + 2*log(3)) + (n00 + n34)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n04 + n30)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n14 + n20)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n10 + n24)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + 3*r^2)) + (n03 + n31)*(log(4 + 3*p2) + log(pmax(1e-6,r)) + log(4*(1 - r + r^2) - 3*p1*(2 - 5*r + 2*r^2))) + (n01 + n33)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(4*(1 - r + r^2) - 3*p1*(-1 + r + 2*r^2))) + (n02 + n32)*(log(4 + 3*p2) + log(2*(1 + r - r^2) - 3*p1*(1 - 5*r + 5*r^2))) + (n12 + n22)*log(6*p2*(-4 - r + r^2) + 8*(8 - r + r^2) + 3*p1*(3*p2*(-2 - 5*r + 5*r^2) + 4*(4 - 5*r + 5*r^2))) + (n11 + n23)*log(p1*(p2*(9 + 63*r^2 - 54*r^3) + 12*(1 - 6*r + 7*r^2 - 6*r^3)) + 4*(3*p2*(1 - 5*r^2 + 3*r^3) + 4*(1 + 3*r - 5*r^2 + 3*r^3))) + (n13 + n21)*log(-4*(3*p2*(1 - r - 4*r^2 + 3*r^3) + 4*(-2 + 2*r - 4*r^2 + 3*r^3)) + 3*p1*(3*p2*(2 + 4*r - 11*r^2 + 6*r^3) + 4*(-4 + 10*r - 11*r^2 + 6*r^3)))
return(L)}
interlogL_rc4 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rc4,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc4 <- parallel::mcmapply(interlogL_rc4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rc4 <- -((x[,"n_12"] + x[,"n_22"])*log10(62 + 3*p1*(11 - (39*p2)/4) - (51*p2)/2)) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(3) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(3) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(log10(5/2 + (3*p1)/4) + log10(4 + 3*p2)) - (x[,"n_13"] + x[,"n_21"])*log10(-4*(-13/2 - (3*p2)/8) + 3*p1*(-4 + 6*p2)) - (x[,"n_11"] + x[,"n_23"])*log10(4*(13/2 + (3*p2)/8) + p1*(-12 + 18*p2)) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rc4)) + log10(pmax(1e-6,r_rc4))) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rc4)) + 2*log10(pmax(1e-6,r_rc4))) + (x[,"n_14"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_rc4)) + log10(2 - 4*r_rc4 + 3*r_rc4^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rc4)) + log10(1 - 2*r_rc4 + 3*r_rc4^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(4 + 3*p2) + log10(pmax(1e-6,r_rc4)) + log10(4*(1 - r_rc4 + r_rc4^2) - 3*p1*(2 - 5*r_rc4 + 2*r_rc4^2))) + (x[,"n_01"] + x[,"n_33"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rc4)) + log10(4*(1 - r_rc4 + r_rc4^2) - 3*p1*(-1 + r_rc4 + 2*r_rc4^2))) + (x[,"n_02"] + x[,"n_32"])*(log10(4 + 3*p2) + log10(2*(1 + r_rc4 - r_rc4^2) - 3*p1*(1 - 5*r_rc4 + 5*r_rc4^2))) + (x[,"n_12"] + x[,"n_22"])*log10(6*p2*(-4 - r_rc4 + r_rc4^2) + 8*(8 - r_rc4 + r_rc4^2) + 3*p1*(3*p2*(-2 - 5*r_rc4 + 5*r_rc4^2) + 4*(4 - 5*r_rc4 + 5*r_rc4^2))) + (x[,"n_11"] + x[,"n_23"])*log10(p1*(p2*(9 + 63*r_rc4^2 - 54*r_rc4^3) + 12*(1 - 6*r_rc4 + 7*r_rc4^2 - 6*r_rc4^3)) + 4*(3*p2*(1 - 5*r_rc4^2 + 3*r_rc4^3) + 4*(1 + 3*r_rc4 - 5*r_rc4^2 + 3*r_rc4^3))) + (x[,"n_13"] + x[,"n_21"])*log10(-4*(3*p2*(1 - r_rc4 - 4*r_rc4^2 + 3*r_rc4^3) + 4*(-2 + 2*r_rc4 - 4*r_rc4^2 + 3*r_rc4^3)) + 3*p1*(3*p2*(2 + 4*r_rc4 - 11*r_rc4^2 + 6*r_rc4^3) + 4*(-4 + 10*r_rc4 - 11*r_rc4^2 + 6*r_rc4^3)))


logL_rc4 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(5*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rc4)) + log(pmax(1e-6,r_rc4))) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rc4)) + 2*log(pmax(1e-6,r_rc4))) + (x[,"n_14"] + x[,"n_20"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_rc4)) + log(2 - 4*r_rc4 + 3*r_rc4^2)) + (x[,"n_10"] + x[,"n_24"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rc4)) + log(1 - 2*r_rc4 + 3*r_rc4^2)) + (x[,"n_03"] + x[,"n_31"])*(log(4 + 3*p2) + log(pmax(1e-6,r_rc4)) + log(4*(1 - r_rc4 + r_rc4^2) - 3*p1*(2 - 5*r_rc4 + 2*r_rc4^2))) + (x[,"n_01"] + x[,"n_33"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_rc4)) + log(4*(1 - r_rc4 + r_rc4^2) - 3*p1*(-1 + r_rc4 + 2*r_rc4^2))) + (x[,"n_02"] + x[,"n_32"])*(log(4 + 3*p2) + log(2*(1 + r_rc4 - r_rc4^2) - 3*p1*(1 - 5*r_rc4 + 5*r_rc4^2))) + (x[,"n_12"] + x[,"n_22"])*log(6*p2*(-4 - r_rc4 + r_rc4^2) + 8*(8 - r_rc4 + r_rc4^2) + 3*p1*(3*p2*(-2 - 5*r_rc4 + 5*r_rc4^2) + 4*(4 - 5*r_rc4 + 5*r_rc4^2))) + (x[,"n_11"] + x[,"n_23"])*log(p1*(p2*(9 + 63*r_rc4^2 - 54*r_rc4^3) + 12*(1 - 6*r_rc4 + 7*r_rc4^2 - 6*r_rc4^3)) + 4*(3*p2*(1 - 5*r_rc4^2 + 3*r_rc4^3) + 4*(1 + 3*r_rc4 - 5*r_rc4^2 + 3*r_rc4^3))) + (x[,"n_13"] + x[,"n_21"])*log(-4*(3*p2*(1 - r_rc4 - 4*r_rc4^2 + 3*r_rc4^3) + 4*(-2 + 2*r_rc4 - 4*r_rc4^2 + 3*r_rc4^3)) + 3*p1*(3*p2*(2 + 4*r_rc4 - 11*r_rc4^2 + 6*r_rc4^3) + 4*(-4 + 10*r_rc4 - 11*r_rc4^2 + 6*r_rc4^3)))


logL_rm1 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(6*log(2) + 2*log(3)) + (n00 + n34)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n04 + n30)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n10 + n24)*(log(4 + 3*p1) + log(pmax(1e-6,r)) + log(6 - (8 - 12*p2)*r - (-6 + 9*p2)*r^2)) + (n03 + n31)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(8*(1 + r^2) + p1*(6 - 9*r + 6*r^2))) + (n01 + n33)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(8*(2 - 2*r + r^2) + p1*(3 - 3*r + 6*r^2))) + (n02 + n32)*(log(2 - 3*p2) + log(4*(4 - 3*r + 3*r^2) - p1*(6 - 9*r + 9*r^2))) + (n14 + n20)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(4 - 4*r + 6*r^2 - p2*(-3 - 6*r + 9*r^2))) + (n12 + n22)*log(-3*p1*(8 + 6*r - 6*r^2 + p2*(-3 - 9*r + 9*r^2)) + 4*(28 + 6*r - 6*r^2 + p2*(3 - 9*r + 9*r^2))) + (n13 + n21)*log(-8*(-8 + 8*r^2 - 6*r^3 + 3*p2*(-2 + 3*r - 4*r^2 + 3*r^3)) + p1*(6*(2 + 3*r - 11*r^2 + 6*r^3) - 9*p2*(-1 + 9*r - 11*r^2 + 6*r^3))) + (n11 + n23)*log(8*(6 + 2*(-1 + 6*p2)*r - 5*(-2 + 3*p2)*r^2 + (-6 + 9*p2)*r^3) + 3*p1*(2*r*(1 + 7*r - 6*r^2) + 3*p2*(-3 + 5*r - 7*r^2 + 6*r^3)))
return(L)}
interlogL_rm1 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rm1,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rm1 <- parallel::mcmapply(interlogL_rm1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rm1 <- -((x[,"n_13"] + x[,"n_21"])*log10(p1*(9 - (27*p2)/2) - 8*(-27/4 - (27*p2)/8))) - (x[,"n_12"] + x[,"n_22"])*log10(-3*p1*(19/2 - (21*p2)/4) + 4*(59/2 + (3*p2)/4)) - (x[,"n_02"] + x[,"n_32"])*(log10(13 - (15*p1)/4) + log10(2 - 3*p2)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(10 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(10 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-log10(2) + log10(4 + 3*p1) + log10(7/2 + (15*p2)/4)) - (x[,"n_10"] + x[,"n_24"])*(-log10(2) + log10(4 + 3*p1) + log10(6 + (6 - 9*p2)/4 + (-8 + 12*p2)/2)) - (x[,"n_11"] + x[,"n_23"])*log10(3*p1*(3 - (9*p2)/2) + 8*(5 + 6*p2 - (5*(-2 + 3*p2))/4 + (-6 + 9*p2)/8)) + (x[,"n_00"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rm1)) + 2*log10(pmax(1e-6,r_rm1))) + (x[,"n_04"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_rm1)) + log10(1 - 2*r_rm1 + r_rm1^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(pmax(1e-6,r_rm1)) + log10(6 - (8 - 12*p2)*r_rm1 - (-6 + 9*p2)*r_rm1^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rm1)) + log10(8*(1 + r_rm1^2) + p1*(6 - 9*r_rm1 + 6*r_rm1^2))) + (x[,"n_01"] + x[,"n_33"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_rm1)) + log10(8*(2 - 2*r_rm1 + r_rm1^2) + p1*(3 - 3*r_rm1 + 6*r_rm1^2))) + (x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p2) + log10(4*(4 - 3*r_rm1 + 3*r_rm1^2) - p1*(6 - 9*r_rm1 + 9*r_rm1^2))) + (x[,"n_14"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_rm1)) + log10(4 - 4*r_rm1 + 6*r_rm1^2 - p2*(-3 - 6*r_rm1 + 9*r_rm1^2))) + (x[,"n_12"] + x[,"n_22"])*log10(-3*p1*(8 + 6*r_rm1 - 6*r_rm1^2 + p2*(-3 - 9*r_rm1 + 9*r_rm1^2)) + 4*(28 + 6*r_rm1 - 6*r_rm1^2 + p2*(3 - 9*r_rm1 + 9*r_rm1^2))) + (x[,"n_13"] + x[,"n_21"])*log10(-8*(-8 + 8*r_rm1^2 - 6*r_rm1^3 + 3*p2*(-2 + 3*r_rm1 - 4*r_rm1^2 + 3*r_rm1^3)) + p1*(6*(2 + 3*r_rm1 - 11*r_rm1^2 + 6*r_rm1^3) - 9*p2*(-1 + 9*r_rm1 - 11*r_rm1^2 + 6*r_rm1^3))) + (x[,"n_11"] + x[,"n_23"])*log10(8*(6 + 2*(-1 + 6*p2)*r_rm1 - 5*(-2 + 3*p2)*r_rm1^2 + (-6 + 9*p2)*r_rm1^3) + 3*p1*(2*r_rm1*(1 + 7*r_rm1 - 6*r_rm1^2) + 3*p2*(-3 + 5*r_rm1 - 7*r_rm1^2 + 6*r_rm1^3)))


logL_rm1 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(6*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_34"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rm1)) + 2*log(pmax(1e-6,r_rm1))) + (x[,"n_04"] + x[,"n_30"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_rm1)) + log(1 - 2*r_rm1 + r_rm1^2)) + (x[,"n_10"] + x[,"n_24"])*(log(4 + 3*p1) + log(pmax(1e-6,r_rm1)) + log(6 - (8 - 12*p2)*r_rm1 - (-6 + 9*p2)*r_rm1^2)) + (x[,"n_03"] + x[,"n_31"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_rm1)) + log(8*(1 + r_rm1^2) + p1*(6 - 9*r_rm1 + 6*r_rm1^2))) + (x[,"n_01"] + x[,"n_33"])*(log(2 - 3*p2) + log(pmax(1e-6,r_rm1)) + log(8*(2 - 2*r_rm1 + r_rm1^2) + p1*(3 - 3*r_rm1 + 6*r_rm1^2))) + (x[,"n_02"] + x[,"n_32"])*(log(2 - 3*p2) + log(4*(4 - 3*r_rm1 + 3*r_rm1^2) - p1*(6 - 9*r_rm1 + 9*r_rm1^2))) + (x[,"n_14"] + x[,"n_20"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_rm1)) + log(4 - 4*r_rm1 + 6*r_rm1^2 - p2*(-3 - 6*r_rm1 + 9*r_rm1^2))) + (x[,"n_12"] + x[,"n_22"])*log(-3*p1*(8 + 6*r_rm1 - 6*r_rm1^2 + p2*(-3 - 9*r_rm1 + 9*r_rm1^2)) + 4*(28 + 6*r_rm1 - 6*r_rm1^2 + p2*(3 - 9*r_rm1 + 9*r_rm1^2))) + (x[,"n_13"] + x[,"n_21"])*log(-8*(-8 + 8*r_rm1^2 - 6*r_rm1^3 + 3*p2*(-2 + 3*r_rm1 - 4*r_rm1^2 + 3*r_rm1^3)) + p1*(6*(2 + 3*r_rm1 - 11*r_rm1^2 + 6*r_rm1^3) - 9*p2*(-1 + 9*r_rm1 - 11*r_rm1^2 + 6*r_rm1^3))) + (x[,"n_11"] + x[,"n_23"])*log(8*(6 + 2*(-1 + 6*p2)*r_rm1 - 5*(-2 + 3*p2)*r_rm1^2 + (-6 + 9*p2)*r_rm1^3) + 3*p1*(2*r_rm1*(1 + 7*r_rm1 - 6*r_rm1^2) + 3*p2*(-3 + 5*r_rm1 - 7*r_rm1^2 + 6*r_rm1^3)))


logL_rm2 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(6*log(2) + 2*log(3)) + (n00 + n34)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n04 + n30)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n10 + n24)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(3 - 4*r + 3*r^2)) + (n14 + n20)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(2 - 2*r + 3*r^2)) + (n01 + n33)*(log(pmax(1e-6,r)) + log((8 - 3*p1)*(-2 + 3*p2)*(-1 + r) + (4 + 3*p1)*(p2*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2)))) + (n12 + n22)*log(4*(28 + 6*r - 6*r^2 + 3*p2*(4 - 3*r + 3*r^2)) - 3*p1*(8 + 6*r - 6*r^2 + 3*p2*(5 - 3*r + 3*r^2))) + (n03 + n31)*(log(pmax(1e-6,1 - r)) + log(4*(4*(1 + r^2) - p2*(-3 + 6*r^2)) - 3*p1*(-4 + 6*r - 4*r^2 + p2*(-3 - 9*r + 6*r^2)))) + (n02 + n32)*log(-4*(-8 + 6*r - 6*r^2 + p2*(-6 - 9*r + 9*r^2)) + 3*p1*(-4 + 6*r - 6*r^2 + p2*(-3 - 9*r + 9*r^2))) + (n13 + n21)*log(p1*(6*(2 + 3*r - 11*r^2 + 6*r^3) - 9*p2*(-4 + 6*r - 11*r^2 + 6*r^3)) - 4*(-4*(4 - 4*r^2 + 3*r^3) + 3*p2*(2 + 3*r - 8*r^2 + 6*r^3))) + (n11 + n23)*log(48 + 2*(-8 + 3*p1)*r + (80 + 42*p1)*r^2 - 12*(4 + 3*p1)*r^3 + 3*p2*(4*(-3 + 5*r - 10*r^2 + 6*r^3) + 3*p1*(3 + 2*r - 7*r^2 + 6*r^3)))
return(L)}
interlogL_rm2 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rm2,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rm2 <- parallel::mcmapply(interlogL_rm2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rm2 <- -((x[,"n_02"] + x[,"n_32"])*log10(-4*(-13/2 - (33*p2)/4) + 3*p1*(-5/2 - (21*p2)/4))) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(7) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(7) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_11"] + x[,"n_23"])*log10(40 + 3*p1 - (3*(4 + 3*p1))/2 + (80 + 42*p1)/4 + 3*(-9 + 9*p1)*p2) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(-3*p1*(-2 - 6*p2) + 4*(5 + (3*p2)/2))) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(-((8 - 3*p1)*(-2 + 3*p2))/2 + (4 + 3*p1)*(3 + (9*p2)/2))) - (x[,"n_12"] + x[,"n_22"])*log10(4*(59/2 + (39*p2)/4) - 3*p1*(19/2 + (51*p2)/4)) - (x[,"n_13"] + x[,"n_21"])*log10(-4*(-27/2 + (27*p2)/4) + p1*(9 + 27*p2)) + (x[,"n_00"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rm2)) + 2*log10(pmax(1e-6,r_rm2))) + (x[,"n_04"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_rm2)) + log10(1 - 2*r_rm2 + r_rm2^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_rm2)) + log10(3 - 4*r_rm2 + 3*r_rm2^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rm2)) + log10(2 - 2*r_rm2 + 3*r_rm2^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(pmax(1e-6,r_rm2)) + log10((8 - 3*p1)*(-2 + 3*p2)*(-1 + r_rm2) + (4 + 3*p1)*(p2*(3 + 6*r_rm2 - 6*r_rm2^2) + 4*(1 - r_rm2 + r_rm2^2)))) + (x[,"n_12"] + x[,"n_22"])*log10(4*(28 + 6*r_rm2 - 6*r_rm2^2 + 3*p2*(4 - 3*r_rm2 + 3*r_rm2^2)) - 3*p1*(8 + 6*r_rm2 - 6*r_rm2^2 + 3*p2*(5 - 3*r_rm2 + 3*r_rm2^2))) + (x[,"n_03"] + x[,"n_31"])*(log10(pmax(1e-6,1 - r_rm2)) + log10(4*(4*(1 + r_rm2^2) - p2*(-3 + 6*r_rm2^2)) - 3*p1*(-4 + 6*r_rm2 - 4*r_rm2^2 + p2*(-3 - 9*r_rm2 + 6*r_rm2^2)))) + (x[,"n_02"] + x[,"n_32"])*log10(-4*(-8 + 6*r_rm2 - 6*r_rm2^2 + p2*(-6 - 9*r_rm2 + 9*r_rm2^2)) + 3*p1*(-4 + 6*r_rm2 - 6*r_rm2^2 + p2*(-3 - 9*r_rm2 + 9*r_rm2^2))) + (x[,"n_13"] + x[,"n_21"])*log10(p1*(6*(2 + 3*r_rm2 - 11*r_rm2^2 + 6*r_rm2^3) - 9*p2*(-4 + 6*r_rm2 - 11*r_rm2^2 + 6*r_rm2^3)) - 4*(-4*(4 - 4*r_rm2^2 + 3*r_rm2^3) + 3*p2*(2 + 3*r_rm2 - 8*r_rm2^2 + 6*r_rm2^3))) + (x[,"n_11"] + x[,"n_23"])*log10(48 + 2*(-8 + 3*p1)*r_rm2 + (80 + 42*p1)*r_rm2^2 - 12*(4 + 3*p1)*r_rm2^3 + 3*p2*(4*(-3 + 5*r_rm2 - 10*r_rm2^2 + 6*r_rm2^3) + 3*p1*(3 + 2*r_rm2 - 7*r_rm2^2 + 6*r_rm2^3)))


logL_rm2 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(6*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_34"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rm2)) + 2*log(pmax(1e-6,r_rm2))) + (x[,"n_04"] + x[,"n_30"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_rm2)) + log(1 - 2*r_rm2 + r_rm2^2)) + (x[,"n_10"] + x[,"n_24"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_rm2)) + log(3 - 4*r_rm2 + 3*r_rm2^2)) + (x[,"n_14"] + x[,"n_20"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rm2)) + log(2 - 2*r_rm2 + 3*r_rm2^2)) + (x[,"n_01"] + x[,"n_33"])*(log(pmax(1e-6,r_rm2)) + log((8 - 3*p1)*(-2 + 3*p2)*(-1 + r_rm2) + (4 + 3*p1)*(p2*(3 + 6*r_rm2 - 6*r_rm2^2) + 4*(1 - r_rm2 + r_rm2^2)))) + (x[,"n_12"] + x[,"n_22"])*log(4*(28 + 6*r_rm2 - 6*r_rm2^2 + 3*p2*(4 - 3*r_rm2 + 3*r_rm2^2)) - 3*p1*(8 + 6*r_rm2 - 6*r_rm2^2 + 3*p2*(5 - 3*r_rm2 + 3*r_rm2^2))) + (x[,"n_03"] + x[,"n_31"])*(log(pmax(1e-6,1 - r_rm2)) + log(4*(4*(1 + r_rm2^2) - p2*(-3 + 6*r_rm2^2)) - 3*p1*(-4 + 6*r_rm2 - 4*r_rm2^2 + p2*(-3 - 9*r_rm2 + 6*r_rm2^2)))) + (x[,"n_02"] + x[,"n_32"])*log(-4*(-8 + 6*r_rm2 - 6*r_rm2^2 + p2*(-6 - 9*r_rm2 + 9*r_rm2^2)) + 3*p1*(-4 + 6*r_rm2 - 6*r_rm2^2 + p2*(-3 - 9*r_rm2 + 9*r_rm2^2))) + (x[,"n_13"] + x[,"n_21"])*log(p1*(6*(2 + 3*r_rm2 - 11*r_rm2^2 + 6*r_rm2^3) - 9*p2*(-4 + 6*r_rm2 - 11*r_rm2^2 + 6*r_rm2^3)) - 4*(-4*(4 - 4*r_rm2^2 + 3*r_rm2^3) + 3*p2*(2 + 3*r_rm2 - 8*r_rm2^2 + 6*r_rm2^3))) + (x[,"n_11"] + x[,"n_23"])*log(48 + 2*(-8 + 3*p1)*r_rm2 + (80 + 42*p1)*r_rm2^2 - 12*(4 + 3*p1)*r_rm2^3 + 3*p2*(4*(-3 + 5*r_rm2 - 10*r_rm2^2 + 6*r_rm2^3) + 3*p1*(3 + 2*r_rm2 - 7*r_rm2^2 + 6*r_rm2^3)))


logL_rm3 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(5*log(2) + 2*log(3)) + (n00 + n34)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n04 + n30)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n10 + n24)*(log(2 - 3*p1) + log(pmax(1e-6,r)) + log(6 - (8 - 12*p2)*r - (-6 + 9*p2)*r^2)) + (n03 + n31)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(4*(1 + r^2) - p1*(6 - 9*r + 6*r^2))) + (n01 + n33)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(4*(2 - 2*r + r^2) - p1*(3 - 3*r + 6*r^2))) + (n02 + n32)*(log(2 - 3*p2) + log(8 - 6*r + 6*r^2 + p1*(6 - 9*r + 9*r^2))) + (n14 + n20)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(4 - 4*r + 6*r^2 - p2*(-3 - 6*r + 9*r^2))) + (n12 + n22)*log(3*p1*(8 + 6*r - 6*r^2 + p2*(-3 - 9*r + 9*r^2)) + 2*(28 + 6*r - 6*r^2 + p2*(3 - 9*r + 9*r^2))) + (n13 + n21)*log(-4*(-8 + 8*r^2 - 6*r^3 + 3*p2*(-2 + 3*r - 4*r^2 + 3*r^3)) + 3*p1*(-2*(2 + 3*r - 11*r^2 + 6*r^3) + 3*p2*(-1 + 9*r - 11*r^2 + 6*r^3))) + (n11 + n23)*log(4*(6 + 2*(-1 + 6*p2)*r - 5*(-2 + 3*p2)*r^2 + (-6 + 9*p2)*r^3) - 3*p1*(2*r*(1 + 7*r - 6*r^2) + 3*p2*(-3 + 5*r - 7*r^2 + 6*r^3)))
return(L)}
interlogL_rm3 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rm3,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rm3 <- parallel::mcmapply(interlogL_rm3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rm3 <- -((x[,"n_12"] + x[,"n_22"])*log10(3*p1*(19/2 - (21*p2)/4) + 2*(59/2 + (3*p2)/4))) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(5 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(5 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(log10(13/2 + (15*p1)/4) + log10(2 - 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-log10(2) + log10(2 - 3*p1) + log10(7/2 + (15*p2)/4)) - (x[,"n_13"] + x[,"n_21"])*log10(-4*(-27/4 - (27*p2)/8) + 3*p1*(-3 + (9*p2)/2)) - (x[,"n_10"] + x[,"n_24"])*(-log10(2) + log10(2 - 3*p1) + log10(6 + (6 - 9*p2)/4 + (-8 + 12*p2)/2)) - (x[,"n_11"] + x[,"n_23"])*log10(-3*p1*(3 - (9*p2)/2) + 4*(5 + 6*p2 - (5*(-2 + 3*p2))/4 + (-6 + 9*p2)/8)) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rm3)) + 2*log10(pmax(1e-6,r_rm3))) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_rm3)) + log10(1 - 2*r_rm3 + r_rm3^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(pmax(1e-6,r_rm3)) + log10(6 - (8 - 12*p2)*r_rm3 - (-6 + 9*p2)*r_rm3^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rm3)) + log10(4*(1 + r_rm3^2) - p1*(6 - 9*r_rm3 + 6*r_rm3^2))) + (x[,"n_01"] + x[,"n_33"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_rm3)) + log10(4*(2 - 2*r_rm3 + r_rm3^2) - p1*(3 - 3*r_rm3 + 6*r_rm3^2))) + (x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p2) + log10(8 - 6*r_rm3 + 6*r_rm3^2 + p1*(6 - 9*r_rm3 + 9*r_rm3^2))) + (x[,"n_14"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_rm3)) + log10(4 - 4*r_rm3 + 6*r_rm3^2 - p2*(-3 - 6*r_rm3 + 9*r_rm3^2))) + (x[,"n_12"] + x[,"n_22"])*log10(3*p1*(8 + 6*r_rm3 - 6*r_rm3^2 + p2*(-3 - 9*r_rm3 + 9*r_rm3^2)) + 2*(28 + 6*r_rm3 - 6*r_rm3^2 + p2*(3 - 9*r_rm3 + 9*r_rm3^2))) + (x[,"n_13"] + x[,"n_21"])*log10(-4*(-8 + 8*r_rm3^2 - 6*r_rm3^3 + 3*p2*(-2 + 3*r_rm3 - 4*r_rm3^2 + 3*r_rm3^3)) + 3*p1*(-2*(2 + 3*r_rm3 - 11*r_rm3^2 + 6*r_rm3^3) + 3*p2*(-1 + 9*r_rm3 - 11*r_rm3^2 + 6*r_rm3^3))) + (x[,"n_11"] + x[,"n_23"])*log10(4*(6 + 2*(-1 + 6*p2)*r_rm3 - 5*(-2 + 3*p2)*r_rm3^2 + (-6 + 9*p2)*r_rm3^3) - 3*p1*(2*r_rm3*(1 + 7*r_rm3 - 6*r_rm3^2) + 3*p2*(-3 + 5*r_rm3 - 7*r_rm3^2 + 6*r_rm3^3)))


logL_rm3 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(5*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rm3)) + 2*log(pmax(1e-6,r_rm3))) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_rm3)) + log(1 - 2*r_rm3 + r_rm3^2)) + (x[,"n_10"] + x[,"n_24"])*(log(2 - 3*p1) + log(pmax(1e-6,r_rm3)) + log(6 - (8 - 12*p2)*r_rm3 - (-6 + 9*p2)*r_rm3^2)) + (x[,"n_03"] + x[,"n_31"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_rm3)) + log(4*(1 + r_rm3^2) - p1*(6 - 9*r_rm3 + 6*r_rm3^2))) + (x[,"n_01"] + x[,"n_33"])*(log(2 - 3*p2) + log(pmax(1e-6,r_rm3)) + log(4*(2 - 2*r_rm3 + r_rm3^2) - p1*(3 - 3*r_rm3 + 6*r_rm3^2))) + (x[,"n_02"] + x[,"n_32"])*(log(2 - 3*p2) + log(8 - 6*r_rm3 + 6*r_rm3^2 + p1*(6 - 9*r_rm3 + 9*r_rm3^2))) + (x[,"n_14"] + x[,"n_20"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_rm3)) + log(4 - 4*r_rm3 + 6*r_rm3^2 - p2*(-3 - 6*r_rm3 + 9*r_rm3^2))) + (x[,"n_12"] + x[,"n_22"])*log(3*p1*(8 + 6*r_rm3 - 6*r_rm3^2 + p2*(-3 - 9*r_rm3 + 9*r_rm3^2)) + 2*(28 + 6*r_rm3 - 6*r_rm3^2 + p2*(3 - 9*r_rm3 + 9*r_rm3^2))) + (x[,"n_13"] + x[,"n_21"])*log(-4*(-8 + 8*r_rm3^2 - 6*r_rm3^3 + 3*p2*(-2 + 3*r_rm3 - 4*r_rm3^2 + 3*r_rm3^3)) + 3*p1*(-2*(2 + 3*r_rm3 - 11*r_rm3^2 + 6*r_rm3^3) + 3*p2*(-1 + 9*r_rm3 - 11*r_rm3^2 + 6*r_rm3^3))) + (x[,"n_11"] + x[,"n_23"])*log(4*(6 + 2*(-1 + 6*p2)*r_rm3 - 5*(-2 + 3*p2)*r_rm3^2 + (-6 + 9*p2)*r_rm3^3) - 3*p1*(2*r_rm3*(1 + 7*r_rm3 - 6*r_rm3^2) + 3*p2*(-3 + 5*r_rm3 - 7*r_rm3^2 + 6*r_rm3^3)))


logL_rm4 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(5*log(2) + 2*log(3)) + (n00 + n34)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n04 + n30)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n10 + n24)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(3 - 4*r + 3*r^2)) + (n14 + n20)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(2 - 2*r + 3*r^2)) + (n01 + n33)*(log(pmax(1e-6,r)) + log((4 + 3*p1)*(-2 + 3*p2)*(-1 + r) + (2 - 3*p1)*(p2*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2)))) + (n12 + n22)*log(2*(28 + 6*r - 6*r^2 + 3*p2*(4 - 3*r + 3*r^2)) + 3*p1*(8 + 6*r - 6*r^2 + 3*p2*(5 - 3*r + 3*r^2))) + (n03 + n31)*(log(pmax(1e-6,1 - r)) + log(8 + 6*p2 + 8*r^2 - 12*p2*r^2 + 3*p1*(-4 + 6*r - 4*r^2 + p2*(-3 - 9*r + 6*r^2)))) + (n02 + n32)*log(-2*(-8 + 6*r - 6*r^2 + p2*(-6 - 9*r + 9*r^2)) - 3*p1*(-4 + 6*r - 6*r^2 + p2*(-3 - 9*r + 9*r^2))) + (n13 + n21)*log(3*p1*(-2*(2 + 3*r - 11*r^2 + 6*r^3) + 3*p2*(-4 + 6*r - 11*r^2 + 6*r^3)) - 2*(-4*(4 - 4*r^2 + 3*r^3) + 3*p2*(2 + 3*r - 8*r^2 + 6*r^3))) + (n11 + n23)*log(24 - 2*(4 + 3*p1)*r + (40 - 42*p1)*r^2 + 12*(-2 + 3*p1)*r^3 - 3*p2*(6 - 10*r + 20*r^2 - 12*r^3 + 3*p1*(3 + 2*r - 7*r^2 + 6*r^3)))
return(L)}
interlogL_rm4 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rm4,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rm4 <- parallel::mcmapply(interlogL_rm4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rm4 <- -((x[,"n_02"] + x[,"n_32"])*log10(-2*(-13/2 - (33*p2)/4) - 3*p1*(-5/2 - (21*p2)/4))) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(7) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(7) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(10 + 3*p1*(-2 - 6*p2) + 3*p2)) - (x[,"n_11"] + x[,"n_23"])*log10(20 + (40 - 42*p1)/4 - 3*p1 + (3*(-2 + 3*p1))/2 - 3*(9/2 + 9*p1)*p2) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(-((4 + 3*p1)*(-2 + 3*p2))/2 + (2 - 3*p1)*(3 + (9*p2)/2))) - (x[,"n_13"] + x[,"n_21"])*log10(3*p1*(-3 - 9*p2) - 2*(-27/2 + (27*p2)/4)) - (x[,"n_12"] + x[,"n_22"])*log10(2*(59/2 + (39*p2)/4) + 3*p1*(19/2 + (51*p2)/4)) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rm4)) + 2*log10(pmax(1e-6,r_rm4))) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_rm4)) + log10(1 - 2*r_rm4 + r_rm4^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_rm4)) + log10(3 - 4*r_rm4 + 3*r_rm4^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rm4)) + log10(2 - 2*r_rm4 + 3*r_rm4^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(pmax(1e-6,r_rm4)) + log10((4 + 3*p1)*(-2 + 3*p2)*(-1 + r_rm4) + (2 - 3*p1)*(p2*(3 + 6*r_rm4 - 6*r_rm4^2) + 4*(1 - r_rm4 + r_rm4^2)))) + (x[,"n_12"] + x[,"n_22"])*log10(2*(28 + 6*r_rm4 - 6*r_rm4^2 + 3*p2*(4 - 3*r_rm4 + 3*r_rm4^2)) + 3*p1*(8 + 6*r_rm4 - 6*r_rm4^2 + 3*p2*(5 - 3*r_rm4 + 3*r_rm4^2))) + (x[,"n_03"] + x[,"n_31"])*(log10(pmax(1e-6,1 - r_rm4)) + log10(8 + 6*p2 + 8*r_rm4^2 - 12*p2*r_rm4^2 + 3*p1*(-4 + 6*r_rm4 - 4*r_rm4^2 + p2*(-3 - 9*r_rm4 + 6*r_rm4^2)))) + (x[,"n_02"] + x[,"n_32"])*log10(-2*(-8 + 6*r_rm4 - 6*r_rm4^2 + p2*(-6 - 9*r_rm4 + 9*r_rm4^2)) - 3*p1*(-4 + 6*r_rm4 - 6*r_rm4^2 + p2*(-3 - 9*r_rm4 + 9*r_rm4^2))) + (x[,"n_13"] + x[,"n_21"])*log10(3*p1*(-2*(2 + 3*r_rm4 - 11*r_rm4^2 + 6*r_rm4^3) + 3*p2*(-4 + 6*r_rm4 - 11*r_rm4^2 + 6*r_rm4^3)) - 2*(-4*(4 - 4*r_rm4^2 + 3*r_rm4^3) + 3*p2*(2 + 3*r_rm4 - 8*r_rm4^2 + 6*r_rm4^3))) + (x[,"n_11"] + x[,"n_23"])*log10(24 - 2*(4 + 3*p1)*r_rm4 + (40 - 42*p1)*r_rm4^2 + 12*(-2 + 3*p1)*r_rm4^3 - 3*p2*(6 - 10*r_rm4 + 20*r_rm4^2 - 12*r_rm4^3 + 3*p1*(3 + 2*r_rm4 - 7*r_rm4^2 + 6*r_rm4^3)))


logL_rm4 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(5*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rm4)) + 2*log(pmax(1e-6,r_rm4))) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_rm4)) + log(1 - 2*r_rm4 + r_rm4^2)) + (x[,"n_10"] + x[,"n_24"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_rm4)) + log(3 - 4*r_rm4 + 3*r_rm4^2)) + (x[,"n_14"] + x[,"n_20"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rm4)) + log(2 - 2*r_rm4 + 3*r_rm4^2)) + (x[,"n_01"] + x[,"n_33"])*(log(pmax(1e-6,r_rm4)) + log((4 + 3*p1)*(-2 + 3*p2)*(-1 + r_rm4) + (2 - 3*p1)*(p2*(3 + 6*r_rm4 - 6*r_rm4^2) + 4*(1 - r_rm4 + r_rm4^2)))) + (x[,"n_12"] + x[,"n_22"])*log(2*(28 + 6*r_rm4 - 6*r_rm4^2 + 3*p2*(4 - 3*r_rm4 + 3*r_rm4^2)) + 3*p1*(8 + 6*r_rm4 - 6*r_rm4^2 + 3*p2*(5 - 3*r_rm4 + 3*r_rm4^2))) + (x[,"n_03"] + x[,"n_31"])*(log(pmax(1e-6,1 - r_rm4)) + log(8 + 6*p2 + 8*r_rm4^2 - 12*p2*r_rm4^2 + 3*p1*(-4 + 6*r_rm4 - 4*r_rm4^2 + p2*(-3 - 9*r_rm4 + 6*r_rm4^2)))) + (x[,"n_02"] + x[,"n_32"])*log(-2*(-8 + 6*r_rm4 - 6*r_rm4^2 + p2*(-6 - 9*r_rm4 + 9*r_rm4^2)) - 3*p1*(-4 + 6*r_rm4 - 6*r_rm4^2 + p2*(-3 - 9*r_rm4 + 9*r_rm4^2))) + (x[,"n_13"] + x[,"n_21"])*log(3*p1*(-2*(2 + 3*r_rm4 - 11*r_rm4^2 + 6*r_rm4^3) + 3*p2*(-4 + 6*r_rm4 - 11*r_rm4^2 + 6*r_rm4^3)) - 2*(-4*(4 - 4*r_rm4^2 + 3*r_rm4^3) + 3*p2*(2 + 3*r_rm4 - 8*r_rm4^2 + 6*r_rm4^3))) + (x[,"n_11"] + x[,"n_23"])*log(24 - 2*(4 + 3*p1)*r_rm4 + (40 - 42*p1)*r_rm4^2 + 12*(-2 + 3*p1)*r_rm4^3 - 3*p2*(6 - 10*r_rm4 + 20*r_rm4^2 - 12*r_rm4^3 + 3*p1*(3 + 2*r_rm4 - 7*r_rm4^2 + 6*r_rm4^3)))


logL_rr1 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- 5*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(2) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - n10 - 2*n11 - 2*n12 - 2*n13 - n14 - n20 - 2*n21 - 2*n22 - 2*n23 - n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34)*log(3) + (n02 + n32)*(log(20 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n24)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n00 + n34)*(log(4 + 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,r))) + (n04 + n30)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + r^2)) + (n14 + n20)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n03 + n31)*(log(2 - 3*p2) + log(8 - 3*p1 + 8*r + 6*p1*r) + log(1 - 2*r + r^2)) + (n01 + n33)*(log(2 - 3*p2) + 2*log(pmax(1e-6,r)) + log(8*(2 - r) - p1*(-3 + 6*r))) + (n12 + n22)*log(p2*(36 + 60*r - 60*r^2) + 8*(9 - 5*r + 5*r^2) + 3*p1*(3*p2*(-3 - r + r^2) - 2*(3 - r + r^2))) + (n11 + n23)*(log(pmax(1e-6,r)) + log(-8*(3*p2*(2 - 5*r + 3*r^2) - 2*(5 - 5*r + 3*r^2)) + p1*(-9*p2*(-2 - 7*r + 6*r^2) + 6*(4 - 7*r + 6*r^2)))) + (n13 + n21)*(log(pmax(1e-6,1 - r)) + log(-8*(-6 + (2 - 3*p2)*r + (-6 + 9*p2)*r^2) - 3*p1*(3*p2*(-3 - 5*r + 6*r^2) - 2*(3 - 5*r + 6*r^2))))
return(L)}
interlogL_rr1 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rr1,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr1 <- parallel::mcmapply(interlogL_rr1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rr1 <- -((x[,"n_03"] + x[,"n_31"])*(log10(3) + log10(2 - 3*p2))) - (x[,"n_01"] + x[,"n_33"])*(log10(3) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(-2*log10(2) + log10(20 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_12"] + x[,"n_22"])*log10(62 + 3*p1*(-11/2 - (39*p2)/4) + 51*p2) - (x[,"n_11"] + x[,"n_23"])*(-log10(2) + log10(-8*(-13/2 + (3*p2)/4) + p1*(12 + 36*p2))) - (x[,"n_13"] + x[,"n_21"])*(-log10(2) + log10(-3*p1*(-4 - 12*p2) - 8*(-6 + (2 - 3*p2)/2 + (-6 + 9*p2)/4))) + (x[,"n_02"] + x[,"n_32"])*(log10(20 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rr1)) + log10(pmax(1e-6,r_rr1))) + (x[,"n_10"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rr1)) + 2*log10(pmax(1e-6,r_rr1))) + (x[,"n_00"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + 3*log10(pmax(1e-6,r_rr1))) + (x[,"n_04"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rr1)) + log10(1 - 2*r_rr1 + r_rr1^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_rr1)) + log10(1 - 2*r_rr1 + r_rr1^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(8 - 3*p1 + 8*r_rr1 + 6*p1*r_rr1) + log10(1 - 2*r_rr1 + r_rr1^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_rr1)) + log10(8*(2 - r_rr1) - p1*(-3 + 6*r_rr1))) + (x[,"n_12"] + x[,"n_22"])*log10(p2*(36 + 60*r_rr1 - 60*r_rr1^2) + 8*(9 - 5*r_rr1 + 5*r_rr1^2) + 3*p1*(3*p2*(-3 - r_rr1 + r_rr1^2) - 2*(3 - r_rr1 + r_rr1^2))) + (x[,"n_11"] + x[,"n_23"])*(log10(pmax(1e-6,r_rr1)) + log10(-8*(3*p2*(2 - 5*r_rr1 + 3*r_rr1^2) - 2*(5 - 5*r_rr1 + 3*r_rr1^2)) + p1*(-9*p2*(-2 - 7*r_rr1 + 6*r_rr1^2) + 6*(4 - 7*r_rr1 + 6*r_rr1^2)))) + (x[,"n_13"] + x[,"n_21"])*(log10(pmax(1e-6,1 - r_rr1)) + log10(-8*(-6 + (2 - 3*p2)*r_rr1 + (-6 + 9*p2)*r_rr1^2) - 3*p1*(3*p2*(-3 - 5*r_rr1 + 6*r_rr1^2) - 2*(3 - 5*r_rr1 + 6*r_rr1^2))))


logL_rr1 <- 5*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"])*log(3) + (x[,"n_02"] + x[,"n_32"])*(log(20 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rr1)) + log(pmax(1e-6,r_rr1))) + (x[,"n_10"] + x[,"n_24"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rr1)) + 2*log(pmax(1e-6,r_rr1))) + (x[,"n_00"] + x[,"n_34"])*(log(4 + 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,r_rr1))) + (x[,"n_04"] + x[,"n_30"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rr1)) + log(1 - 2*r_rr1 + r_rr1^2)) + (x[,"n_14"] + x[,"n_20"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_rr1)) + log(1 - 2*r_rr1 + r_rr1^2)) + (x[,"n_03"] + x[,"n_31"])*(log(2 - 3*p2) + log(8 - 3*p1 + 8*r_rr1 + 6*p1*r_rr1) + log(1 - 2*r_rr1 + r_rr1^2)) + (x[,"n_01"] + x[,"n_33"])*(log(2 - 3*p2) + 2*log(pmax(1e-6,r_rr1)) + log(8*(2 - r_rr1) - p1*(-3 + 6*r_rr1))) + (x[,"n_12"] + x[,"n_22"])*log(p2*(36 + 60*r_rr1 - 60*r_rr1^2) + 8*(9 - 5*r_rr1 + 5*r_rr1^2) + 3*p1*(3*p2*(-3 - r_rr1 + r_rr1^2) - 2*(3 - r_rr1 + r_rr1^2))) + (x[,"n_11"] + x[,"n_23"])*(log(pmax(1e-6,r_rr1)) + log(-8*(3*p2*(2 - 5*r_rr1 + 3*r_rr1^2) - 2*(5 - 5*r_rr1 + 3*r_rr1^2)) + p1*(-9*p2*(-2 - 7*r_rr1 + 6*r_rr1^2) + 6*(4 - 7*r_rr1 + 6*r_rr1^2)))) + (x[,"n_13"] + x[,"n_21"])*(log(pmax(1e-6,1 - r_rr1)) + log(-8*(-6 + (2 - 3*p2)*r_rr1 + (-6 + 9*p2)*r_rr1^2) - 3*p1*(3*p2*(-3 - 5*r_rr1 + 6*r_rr1^2) - 2*(3 - 5*r_rr1 + 6*r_rr1^2))))


logL_rr2 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- 6*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(2) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - n10 - 2*n11 - 2*n12 - 2*n13 - n14 - n20 - 2*n21 - 2*n22 - 2*n23 - n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34)*log(3) + (n02 + n32)*(log(20 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n24)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n00 + n34)*(log(4 + 3*p1) + log(4 + 3*p2) + 3*log(pmax(1e-6,r))) + (n04 + n30)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + r^2)) + (n14 + n20)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n03 + n31)*(log(4 + 3*p2) + log(8 - 3*p1 + 8*r + 6*p1*r) + log(1 - 2*r + r^2)) + (n01 + n33)*(log(4 + 3*p2) + 2*log(pmax(1e-6,r)) + log(8*(2 - r) - p1*(-3 + 6*r))) + (n12 + n22)*log(-3*p1*(3*p2*(-3 - r + r^2) + 4*(3 - r + r^2)) + 4*(3*p2*(-3 - 5*r + 5*r^2) + 4*(9 - 5*r + 5*r^2))) + (n11 + n23)*(log(pmax(1e-6,r)) + log(8*(3*p2*(2 - 5*r + 3*r^2) + 4*(5 - 5*r + 3*r^2)) + 3*p1*(3*p2*(-2 - 7*r + 6*r^2) + 4*(4 - 7*r + 6*r^2)))) + (n13 + n21)*(log(pmax(1e-6,1 - r)) + log(8*(12 - (4 + 3*p2)*r + 3*(4 + 3*p2)*r^2) + 3*p1*(3*p2*(-3 - 5*r + 6*r^2) + 4*(3 - 5*r + 6*r^2))))
return(L)}
interlogL_rr2 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rr2,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr2 <- parallel::mcmapply(interlogL_rr2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rr2 <- -((x[,"n_12"] + x[,"n_22"])*log10(4*(31 - (51*p2)/4) - 3*p1*(11 - (39*p2)/4))) - (x[,"n_11"] + x[,"n_23"])*(-log10(2) + log10(3*p1*(8 - 12*p2) + 8*(13 + (3*p2)/4))) - (x[,"n_03"] + x[,"n_31"])*(log10(3) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_33"])*(log10(3) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(-2*log10(2) + log10(20 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_13"] + x[,"n_21"])*(-log10(2) + log10(3*p1*(8 - 12*p2) + 8*(12 + (-4 - 3*p2)/2 + (3*(4 + 3*p2))/4))) + (x[,"n_02"] + x[,"n_32"])*(log10(20 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rr2)) + log10(pmax(1e-6,r_rr2))) + (x[,"n_10"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rr2)) + 2*log10(pmax(1e-6,r_rr2))) + (x[,"n_00"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + 3*log10(pmax(1e-6,r_rr2))) + (x[,"n_04"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rr2)) + log10(1 - 2*r_rr2 + r_rr2^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_rr2)) + log10(1 - 2*r_rr2 + r_rr2^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(4 + 3*p2) + log10(8 - 3*p1 + 8*r_rr2 + 6*p1*r_rr2) + log10(1 - 2*r_rr2 + r_rr2^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_rr2)) + log10(8*(2 - r_rr2) - p1*(-3 + 6*r_rr2))) + (x[,"n_12"] + x[,"n_22"])*log10(-3*p1*(3*p2*(-3 - r_rr2 + r_rr2^2) + 4*(3 - r_rr2 + r_rr2^2)) + 4*(3*p2*(-3 - 5*r_rr2 + 5*r_rr2^2) + 4*(9 - 5*r_rr2 + 5*r_rr2^2))) + (x[,"n_11"] + x[,"n_23"])*(log10(pmax(1e-6,r_rr2)) + log10(8*(3*p2*(2 - 5*r_rr2 + 3*r_rr2^2) + 4*(5 - 5*r_rr2 + 3*r_rr2^2)) + 3*p1*(3*p2*(-2 - 7*r_rr2 + 6*r_rr2^2) + 4*(4 - 7*r_rr2 + 6*r_rr2^2)))) + (x[,"n_13"] + x[,"n_21"])*(log10(pmax(1e-6,1 - r_rr2)) + log10(8*(12 - (4 + 3*p2)*r_rr2 + 3*(4 + 3*p2)*r_rr2^2) + 3*p1*(3*p2*(-3 - 5*r_rr2 + 6*r_rr2^2) + 4*(3 - 5*r_rr2 + 6*r_rr2^2))))


logL_rr2 <- 6*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"])*log(3) + (x[,"n_02"] + x[,"n_32"])*(log(20 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rr2)) + log(pmax(1e-6,r_rr2))) + (x[,"n_10"] + x[,"n_24"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rr2)) + 2*log(pmax(1e-6,r_rr2))) + (x[,"n_00"] + x[,"n_34"])*(log(4 + 3*p1) + log(4 + 3*p2) + 3*log(pmax(1e-6,r_rr2))) + (x[,"n_04"] + x[,"n_30"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rr2)) + log(1 - 2*r_rr2 + r_rr2^2)) + (x[,"n_14"] + x[,"n_20"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_rr2)) + log(1 - 2*r_rr2 + r_rr2^2)) + (x[,"n_03"] + x[,"n_31"])*(log(4 + 3*p2) + log(8 - 3*p1 + 8*r_rr2 + 6*p1*r_rr2) + log(1 - 2*r_rr2 + r_rr2^2)) + (x[,"n_01"] + x[,"n_33"])*(log(4 + 3*p2) + 2*log(pmax(1e-6,r_rr2)) + log(8*(2 - r_rr2) - p1*(-3 + 6*r_rr2))) + (x[,"n_12"] + x[,"n_22"])*log(-3*p1*(3*p2*(-3 - r_rr2 + r_rr2^2) + 4*(3 - r_rr2 + r_rr2^2)) + 4*(3*p2*(-3 - 5*r_rr2 + 5*r_rr2^2) + 4*(9 - 5*r_rr2 + 5*r_rr2^2))) + (x[,"n_11"] + x[,"n_23"])*(log(pmax(1e-6,r_rr2)) + log(8*(3*p2*(2 - 5*r_rr2 + 3*r_rr2^2) + 4*(5 - 5*r_rr2 + 3*r_rr2^2)) + 3*p1*(3*p2*(-2 - 7*r_rr2 + 6*r_rr2^2) + 4*(4 - 7*r_rr2 + 6*r_rr2^2)))) + (x[,"n_13"] + x[,"n_21"])*(log(pmax(1e-6,1 - r_rr2)) + log(8*(12 - (4 + 3*p2)*r_rr2 + 3*(4 + 3*p2)*r_rr2^2) + 3*p1*(3*p2*(-3 - 5*r_rr2 + 6*r_rr2^2) + 4*(3 - 5*r_rr2 + 6*r_rr2^2))))


logL_rr3 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- 4*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(2) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - n10 - 2*n11 - 2*n12 - 2*n13 - n14 - n20 - 2*n21 - 2*n22 - 2*n23 - n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34)*log(3) + (n02 + n32)*(log(10 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n24)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n00 + n34)*(log(2 - 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,r))) + (n04 + n30)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + r^2)) + (n14 + n20)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n03 + n31)*(log(2 - 3*p2) + log(1 - 2*r + r^2) + log(4*(1 + r) - p1*(-3 + 6*r))) + (n01 + n33)*(log(2 - 3*p2) + 2*log(pmax(1e-6,r)) + log(8 - 4*r + p1*(-3 + 6*r))) + (n12 + n22)*log(-6*p2*(-3 - 5*r + 5*r^2) + 4*(9 - 5*r + 5*r^2) + p1*(-9*p2*(-3 - r + r^2) + 6*(3 - r + r^2))) + (n11 + n23)*(log(pmax(1e-6,r)) + log(4*(-3*p2*(2 - 5*r + 3*r^2) + 2*(5 - 5*r + 3*r^2)) + 3*p1*(3*p2*(-2 - 7*r + 6*r^2) - 2*(4 - 7*r + 6*r^2)))) + (n13 + n21)*(log(pmax(1e-6,1 - r)) + log(4*(6 + (-2 + 3*p2)*r + (6 - 9*p2)*r^2) + 3*p1*(3*p2*(-3 - 5*r + 6*r^2) - 2*(3 - 5*r + 6*r^2))))
return(L)}
interlogL_rr3 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rr3,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr3 <- parallel::mcmapply(interlogL_rr3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rr3 <- -((x[,"n_11"] + x[,"n_23"])*(-log10(2) + log10(3*p1*(-4 - 12*p2) + 4*(13/2 - (3*p2)/4)))) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(3) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(2 - 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(-2*log10(2) + log10(10 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_12"] + x[,"n_22"])*log10(31 + (51*p2)/2 + p1*(33/2 + (117*p2)/4)) - (x[,"n_13"] + x[,"n_21"])*(-log10(2) + log10(3*p1*(-4 - 12*p2) + 4*(6 + (6 - 9*p2)/4 + (-2 + 3*p2)/2))) + (x[,"n_02"] + x[,"n_32"])*(log10(10 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rr3)) + log10(pmax(1e-6,r_rr3))) + (x[,"n_10"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rr3)) + 2*log10(pmax(1e-6,r_rr3))) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 3*log10(pmax(1e-6,r_rr3))) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rr3)) + log10(1 - 2*r_rr3 + r_rr3^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_rr3)) + log10(1 - 2*r_rr3 + r_rr3^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(2 - 3*p2) + log10(1 - 2*r_rr3 + r_rr3^2) + log10(4*(1 + r_rr3) - p1*(-3 + 6*r_rr3))) + (x[,"n_01"] + x[,"n_33"])*(log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_rr3)) + log10(8 - 4*r_rr3 + p1*(-3 + 6*r_rr3))) + (x[,"n_12"] + x[,"n_22"])*log10(-6*p2*(-3 - 5*r_rr3 + 5*r_rr3^2) + 4*(9 - 5*r_rr3 + 5*r_rr3^2) + p1*(-9*p2*(-3 - r_rr3 + r_rr3^2) + 6*(3 - r_rr3 + r_rr3^2))) + (x[,"n_11"] + x[,"n_23"])*(log10(pmax(1e-6,r_rr3)) + log10(4*(-3*p2*(2 - 5*r_rr3 + 3*r_rr3^2) + 2*(5 - 5*r_rr3 + 3*r_rr3^2)) + 3*p1*(3*p2*(-2 - 7*r_rr3 + 6*r_rr3^2) - 2*(4 - 7*r_rr3 + 6*r_rr3^2)))) + (x[,"n_13"] + x[,"n_21"])*(log10(pmax(1e-6,1 - r_rr3)) + log10(4*(6 + (-2 + 3*p2)*r_rr3 + (6 - 9*p2)*r_rr3^2) + 3*p1*(3*p2*(-3 - 5*r_rr3 + 6*r_rr3^2) - 2*(3 - 5*r_rr3 + 6*r_rr3^2))))


logL_rr3 <- 4*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"])*log(3) + (x[,"n_02"] + x[,"n_32"])*(log(10 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rr3)) + log(pmax(1e-6,r_rr3))) + (x[,"n_10"] + x[,"n_24"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rr3)) + 2*log(pmax(1e-6,r_rr3))) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,r_rr3))) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rr3)) + log(1 - 2*r_rr3 + r_rr3^2)) + (x[,"n_14"] + x[,"n_20"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_rr3)) + log(1 - 2*r_rr3 + r_rr3^2)) + (x[,"n_03"] + x[,"n_31"])*(log(2 - 3*p2) + log(1 - 2*r_rr3 + r_rr3^2) + log(4*(1 + r_rr3) - p1*(-3 + 6*r_rr3))) + (x[,"n_01"] + x[,"n_33"])*(log(2 - 3*p2) + 2*log(pmax(1e-6,r_rr3)) + log(8 - 4*r_rr3 + p1*(-3 + 6*r_rr3))) + (x[,"n_12"] + x[,"n_22"])*log(-6*p2*(-3 - 5*r_rr3 + 5*r_rr3^2) + 4*(9 - 5*r_rr3 + 5*r_rr3^2) + p1*(-9*p2*(-3 - r_rr3 + r_rr3^2) + 6*(3 - r_rr3 + r_rr3^2))) + (x[,"n_11"] + x[,"n_23"])*(log(pmax(1e-6,r_rr3)) + log(4*(-3*p2*(2 - 5*r_rr3 + 3*r_rr3^2) + 2*(5 - 5*r_rr3 + 3*r_rr3^2)) + 3*p1*(3*p2*(-2 - 7*r_rr3 + 6*r_rr3^2) - 2*(4 - 7*r_rr3 + 6*r_rr3^2)))) + (x[,"n_13"] + x[,"n_21"])*(log(pmax(1e-6,1 - r_rr3)) + log(4*(6 + (-2 + 3*p2)*r_rr3 + (6 - 9*p2)*r_rr3^2) + 3*p1*(3*p2*(-3 - 5*r_rr3 + 6*r_rr3^2) - 2*(3 - 5*r_rr3 + 6*r_rr3^2))))


logL_rr4 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- 5*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(2) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - n10 - 2*n11 - 2*n12 - 2*n13 - n14 - n20 - 2*n21 - 2*n22 - 2*n23 - n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34)*log(3) + (n02 + n32)*(log(10 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n24)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n00 + n34)*(log(2 - 3*p1) + log(4 + 3*p2) + 3*log(pmax(1e-6,r))) + (n04 + n30)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + r^2)) + (n14 + n20)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n03 + n31)*(log(4 + 3*p2) + log(1 - 2*r + r^2) + log(4*(1 + r) - p1*(-3 + 6*r))) + (n01 + n33)*(log(4 + 3*p2) + 2*log(pmax(1e-6,r)) + log(8 - 4*r + p1*(-3 + 6*r))) + (n12 + n22)*log(6*p2*(-3 - 5*r + 5*r^2) + 8*(9 - 5*r + 5*r^2) + 3*p1*(3*p2*(-3 - r + r^2) + 4*(3 - r + r^2))) + (n11 + n23)*(log(pmax(1e-6,r)) + log(4*(3*p2*(2 - 5*r + 3*r^2) + 4*(5 - 5*r + 3*r^2)) + p1*(-9*p2*(-2 - 7*r + 6*r^2) - 12*(4 - 7*r + 6*r^2)))) + (n13 + n21)*(log(pmax(1e-6,1 - r)) + log(4*(12 - (4 + 3*p2)*r + 3*(4 + 3*p2)*r^2) - 3*p1*(3*p2*(-3 - 5*r + 6*r^2) + 4*(3 - 5*r + 6*r^2))))
return(L)}
interlogL_rr4 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rr4,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr4 <- parallel::mcmapply(interlogL_rr4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rr4 <- -((x[,"n_12"] + x[,"n_22"])*log10(62 + 3*p1*(11 - (39*p2)/4) - (51*p2)/2)) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(-2*log10(2) + log10(10 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_11"] + x[,"n_23"])*(-log10(2) + log10(4*(13 + (3*p2)/4) + p1*(-24 + 36*p2))) - (x[,"n_13"] + x[,"n_21"])*(-log10(2) + log10(-3*p1*(8 - 12*p2) + 4*(12 + (-4 - 3*p2)/2 + (3*(4 + 3*p2))/4))) + (x[,"n_02"] + x[,"n_32"])*(log10(10 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rr4)) + log10(pmax(1e-6,r_rr4))) + (x[,"n_10"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rr4)) + 2*log10(pmax(1e-6,r_rr4))) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + 3*log10(pmax(1e-6,r_rr4))) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rr4)) + log10(1 - 2*r_rr4 + r_rr4^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_rr4)) + log10(1 - 2*r_rr4 + r_rr4^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(4 + 3*p2) + log10(1 - 2*r_rr4 + r_rr4^2) + log10(4*(1 + r_rr4) - p1*(-3 + 6*r_rr4))) + (x[,"n_01"] + x[,"n_33"])*(log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_rr4)) + log10(8 - 4*r_rr4 + p1*(-3 + 6*r_rr4))) + (x[,"n_12"] + x[,"n_22"])*log10(6*p2*(-3 - 5*r_rr4 + 5*r_rr4^2) + 8*(9 - 5*r_rr4 + 5*r_rr4^2) + 3*p1*(3*p2*(-3 - r_rr4 + r_rr4^2) + 4*(3 - r_rr4 + r_rr4^2))) + (x[,"n_11"] + x[,"n_23"])*(log10(pmax(1e-6,r_rr4)) + log10(4*(3*p2*(2 - 5*r_rr4 + 3*r_rr4^2) + 4*(5 - 5*r_rr4 + 3*r_rr4^2)) + p1*(-9*p2*(-2 - 7*r_rr4 + 6*r_rr4^2) - 12*(4 - 7*r_rr4 + 6*r_rr4^2)))) + (x[,"n_13"] + x[,"n_21"])*(log10(pmax(1e-6,1 - r_rr4)) + log10(4*(12 - (4 + 3*p2)*r_rr4 + 3*(4 + 3*p2)*r_rr4^2) - 3*p1*(3*p2*(-3 - 5*r_rr4 + 6*r_rr4^2) + 4*(3 - 5*r_rr4 + 6*r_rr4^2))))


logL_rr4 <- 5*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"])*log(3) + (x[,"n_02"] + x[,"n_32"])*(log(10 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rr4)) + log(pmax(1e-6,r_rr4))) + (x[,"n_10"] + x[,"n_24"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rr4)) + 2*log(pmax(1e-6,r_rr4))) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(4 + 3*p2) + 3*log(pmax(1e-6,r_rr4))) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rr4)) + log(1 - 2*r_rr4 + r_rr4^2)) + (x[,"n_14"] + x[,"n_20"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_rr4)) + log(1 - 2*r_rr4 + r_rr4^2)) + (x[,"n_03"] + x[,"n_31"])*(log(4 + 3*p2) + log(1 - 2*r_rr4 + r_rr4^2) + log(4*(1 + r_rr4) - p1*(-3 + 6*r_rr4))) + (x[,"n_01"] + x[,"n_33"])*(log(4 + 3*p2) + 2*log(pmax(1e-6,r_rr4)) + log(8 - 4*r_rr4 + p1*(-3 + 6*r_rr4))) + (x[,"n_12"] + x[,"n_22"])*log(6*p2*(-3 - 5*r_rr4 + 5*r_rr4^2) + 8*(9 - 5*r_rr4 + 5*r_rr4^2) + 3*p1*(3*p2*(-3 - r_rr4 + r_rr4^2) + 4*(3 - r_rr4 + r_rr4^2))) + (x[,"n_11"] + x[,"n_23"])*(log(pmax(1e-6,r_rr4)) + log(4*(3*p2*(2 - 5*r_rr4 + 3*r_rr4^2) + 4*(5 - 5*r_rr4 + 3*r_rr4^2)) + p1*(-9*p2*(-2 - 7*r_rr4 + 6*r_rr4^2) - 12*(4 - 7*r_rr4 + 6*r_rr4^2)))) + (x[,"n_13"] + x[,"n_21"])*(log(pmax(1e-6,1 - r_rr4)) + log(4*(12 - (4 + 3*p2)*r_rr4 + 3*(4 + 3*p2)*r_rr4^2) - 3*p1*(3*p2*(-3 - 5*r_rr4 + 6*r_rr4^2) + 4*(3 - 5*r_rr4 + 6*r_rr4^2))))


return(list(
r_mat = cbind(r_cc1,r_cc2,r_cc3,r_cc4,r_cm1,r_cm2,r_cm3,r_cm4,r_cr1,r_cr2,r_cr3,r_cr4,r_rc1,r_rc2,r_rc3,r_rc4,r_rm1,r_rm2,r_rm3,r_rm4,r_rr1,r_rr2,r_rr3,r_rr4,0.499),
LOD_mat = cbind(LOD_cc1,LOD_cc2,LOD_cc3,LOD_cc4,LOD_cm1,LOD_cm2,LOD_cm3,LOD_cm4,LOD_cr1,LOD_cr2,LOD_cr3,LOD_cr4,LOD_rc1,LOD_rc2,LOD_rc3,LOD_rc4,LOD_rm1,LOD_rm2,LOD_rm3,LOD_rm4,LOD_rr1,LOD_rr2,LOD_rr3,LOD_rr4,0),
logL_mat = cbind(logL_cc1,logL_cc2,logL_cc3,logL_cc4,logL_cm1,logL_cm2,logL_cm3,logL_cm4,logL_cr1,logL_cr2,logL_cr3,logL_cr4,logL_rc1,logL_rc2,logL_rc3,logL_rc4,logL_rm1,logL_rm2,logL_rm3,logL_rm4,logL_rr1,logL_rr2,logL_rr3,logL_rr4,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling coupling","coupling coupling","coupling coupling","coupling mixed","coupling mixed","coupling mixed","coupling mixed","coupling repulsion","coupling repulsion","coupling repulsion","coupling repulsion","repulsion coupling","repulsion coupling","repulsion coupling","repulsion coupling","repulsion mixed","repulsion mixed","repulsion mixed","repulsion mixed","repulsion repulsion","repulsion repulsion","repulsion repulsion","repulsion repulsion","unknown")
)
)
}

#' @rdname p4_functions
p4_1.3_1.2 <- function(x,p1,p2,ncores=1){


logL_cc1 <- function(r,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-3*n10 - 3*n11 - 3*n12 - 3*n13 - 2*n20 - 2*n21 - 2*n22 - 2*n23 - 3*n30 - 3*n31 - 3*n32 - 3*n33)*log(2) + (-n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*log(3) + (n20 + n23)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n13 + n30)*(log(2 - 3*p2) + 2*log(pmax(1e-6,r))) + (n12 + n31)*(log(pmax(1e-6,r)) + log(6 + (-2 + 3*p2)*r)) + (n10 + n33)*(log(2 - 3*p2) + log(1 - 2*r + r^2)) + (n21 + n22)*log(3 + (-2 + 3*p2)*r + (2 - 3*p2)*r^2) + (n11 + n32)*(log(-1 + r) + log(3*p2*(-1 + r) - 2*(2 + r)))
return(L)}
interlogL_cc1 <- function(n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_cc1,c(0,0.5), n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc1 <- parallel::mcmapply(interlogL_cc1,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_cc1 <- -((x[,"n_20"] + x[,"n_23"])*(-2*log10(2) + log10(2 - 3*p2))) - (x[,"n_13"] + x[,"n_30"])*(-2*log10(2) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_33"])*(-2*log10(2) + log10(2 - 3*p2)) - (x[,"n_11"] + x[,"n_32"])*(- log10(2) + log10(5 + (3*p2)/2)) - (x[,"n_12"] + x[,"n_31"])*(-log10(2) + log10(6 + (-2 + 3*p2)/2)) - (x[,"n_21"] + x[,"n_22"])*log10(3 + (2 - 3*p2)/4 + (-2 + 3*p2)/2) + (x[,"n_20"] + x[,"n_23"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cc1)) + log10(pmax(1e-6,r_cc1))) + (x[,"n_13"] + x[,"n_30"])*(log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_cc1))) + (x[,"n_12"] + x[,"n_31"])*(log10(pmax(1e-6,r_cc1)) + log10(6 + (-2 + 3*p2)*r_cc1)) + (x[,"n_10"] + x[,"n_33"])*(log10(2 - 3*p2) + log10(1 - 2*r_cc1 + r_cc1^2)) + (x[,"n_21"] + x[,"n_22"])*log10(3 + (-2 + 3*p2)*r_cc1 + (2 - 3*p2)*r_cc1^2) + (x[,"n_11"] + x[,"n_32"])*(log10(-1 + r_cc1) + log10(3*p2*(-1 + r_cc1) - 2*(2 + r_cc1)))


logL_cc1 <- (-3*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 2*x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 3*x[,"n_30"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"])*log(2) + (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*log(3) + (x[,"n_20"] + x[,"n_23"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_cc1)) + log(pmax(1e-6,r_cc1))) + (x[,"n_13"] + x[,"n_30"])*(log(2 - 3*p2) + 2*log(pmax(1e-6,r_cc1))) + (x[,"n_12"] + x[,"n_31"])*(log(pmax(1e-6,r_cc1)) + log(6 + (-2 + 3*p2)*r_cc1)) + (x[,"n_10"] + x[,"n_33"])*(log(2 - 3*p2) + log(1 - 2*r_cc1 + r_cc1^2)) + (x[,"n_21"] + x[,"n_22"])*log(3 + (-2 + 3*p2)*r_cc1 + (2 - 3*p2)*r_cc1^2) + (x[,"n_11"] + x[,"n_32"])*(log(-1 + r_cc1) + log(3*p2*(-1 + r_cc1) - 2*(2 + r_cc1)))


logL_cc2 <- function(r,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-4*n10 - 4*n11 - 4*n12 - 4*n13 - 3*n20 - 3*n21 - 3*n22 - 3*n23 - 4*n30 - 4*n31 - 4*n32 - 4*n33)*log(2) + (-n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*log(3) + (n20 + n23)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n13 + n30)*(log(4 + 3*p2) + 2*log(pmax(1e-6,r))) + (n12 + n31)*(log(pmax(1e-6,r)) + log(12 - (4 + 3*p2)*r)) + (n10 + n33)*(log(4 + 3*p2) + log(1 - 2*r + r^2)) + (n21 + n22)*log(6 - (4 + 3*p2)*r + (4 + 3*p2)*r^2) + (n11 + n32)*(log(pmax(1e-6,1 - r)) + log(3*p2*(-1 + r) + 4*(2 + r)))
return(L)}
interlogL_cc2 <- function(n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_cc2,c(0,0.5), n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc2 <- parallel::mcmapply(interlogL_cc2,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_cc2 <- -((x[,"n_12"] + x[,"n_31"])*(-log10(2) + log10(12 + (-4 - 3*p2)/2))) - (x[,"n_11"] + x[,"n_32"])*(-log10(2) + log10(10 - (3*p2)/2)) - (x[,"n_20"] + x[,"n_23"])*(-2*log10(2) + log10(4 + 3*p2)) - (x[,"n_13"] + x[,"n_30"])*(-2*log10(2) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_33"])*(-2*log10(2) + log10(4 + 3*p2)) - (x[,"n_21"] + x[,"n_22"])*log10(6 + (-4 - 3*p2)/2 + (4 + 3*p2)/4) + (x[,"n_20"] + x[,"n_23"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cc2)) + log10(pmax(1e-6,r_cc2))) + (x[,"n_13"] + x[,"n_30"])*(log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_cc2))) + (x[,"n_12"] + x[,"n_31"])*(log10(pmax(1e-6,r_cc2)) + log10(12 - (4 + 3*p2)*r_cc2)) + (x[,"n_10"] + x[,"n_33"])*(log10(4 + 3*p2) + log10(1 - 2*r_cc2 + r_cc2^2)) + (x[,"n_21"] + x[,"n_22"])*log10(6 - (4 + 3*p2)*r_cc2 + (4 + 3*p2)*r_cc2^2) + (x[,"n_11"] + x[,"n_32"])*(log10(pmax(1e-6,1 - r_cc2)) + log10(3*p2*(-1 + r_cc2) + 4*(2 + r_cc2)))


logL_cc2 <- (-4*x[,"n_10"] - 4*x[,"n_11"] - 4*x[,"n_12"] - 4*x[,"n_13"] - 3*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 4*x[,"n_30"] - 4*x[,"n_31"] - 4*x[,"n_32"] - 4*x[,"n_33"])*log(2) + (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*log(3) + (x[,"n_20"] + x[,"n_23"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_cc2)) + log(pmax(1e-6,r_cc2))) + (x[,"n_13"] + x[,"n_30"])*(log(4 + 3*p2) + 2*log(pmax(1e-6,r_cc2))) + (x[,"n_12"] + x[,"n_31"])*(log(pmax(1e-6,r_cc2)) + log(12 - (4 + 3*p2)*r_cc2)) + (x[,"n_10"] + x[,"n_33"])*(log(4 + 3*p2) + log(1 - 2*r_cc2 + r_cc2^2)) + (x[,"n_21"] + x[,"n_22"])*log(6 - (4 + 3*p2)*r_cc2 + (4 + 3*p2)*r_cc2^2) + (x[,"n_11"] + x[,"n_32"])*(log(pmax(1e-6,1 - r_cc2)) + log(3*p2*(-1 + r_cc2) + 4*(2 + r_cc2)))


logL_cm1 <- function(r,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(4*log(2) + log(3)) + (n10 + n13 + n30 + n33)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n20 + n23)*(log(4 + 3*p2) + log(1 - 2*r + 2*r^2)) + (n21 + n22)*log(8 + 8*r - 8*r^2 + p2*(-3 + 6*r - 6*r^2)) + (n12 + n31)*log(4*(1 + r^2) + 3*p2*(1 - 3*r + r^2)) + (n11 + n32)*log(4*(2 - 2*r + r^2) + 3*p2*(-1 + r + r^2))
return(L)}
interlogL_cm1 <- function(n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_cm1,c(0,0.5), n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cm1 <- parallel::mcmapply(interlogL_cm1,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_cm1 <- -((x[,"n_21"] + x[,"n_22"])*log10(10 - (3*p2)/2)) - (x[,"n_12"] + x[,"n_31"])*log10(5 - (3*p2)/4) - (x[,"n_11"] + x[,"n_32"])*log10(5 - (3*p2)/4) - (x[,"n_10"] + x[,"n_13"] + x[,"n_30"] + x[,"n_33"])*(-2*log10(2) + log10(4 + 3*p2)) - (x[,"n_20"] + x[,"n_23"])*(-log10(2) + log10(4 + 3*p2)) + (x[,"n_10"] + x[,"n_13"] + x[,"n_30"] + x[,"n_33"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cm1)) + log10(pmax(1e-6,r_cm1))) + (x[,"n_20"] + x[,"n_23"])*(log10(4 + 3*p2) + log10(1 - 2*r_cm1 + 2*r_cm1^2)) + (x[,"n_21"] + x[,"n_22"])*log10(8 + 8*r_cm1 - 8*r_cm1^2 + p2*(-3 + 6*r_cm1 - 6*r_cm1^2)) + (x[,"n_12"] + x[,"n_31"])*log10(4*(1 + r_cm1^2) + 3*p2*(1 - 3*r_cm1 + r_cm1^2)) + (x[,"n_11"] + x[,"n_32"])*log10(4*(2 - 2*r_cm1 + r_cm1^2) + 3*p2*(-1 + r_cm1 + r_cm1^2))


logL_cm1 <- (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(4*log(2) + log(3)) + (x[,"n_10"] + x[,"n_13"] + x[,"n_30"] + x[,"n_33"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_cm1)) + log(pmax(1e-6,r_cm1))) + (x[,"n_20"] + x[,"n_23"])*(log(4 + 3*p2) + log(1 - 2*r_cm1 + 2*r_cm1^2)) + (x[,"n_21"] + x[,"n_22"])*log(8 + 8*r_cm1 - 8*r_cm1^2 + p2*(-3 + 6*r_cm1 - 6*r_cm1^2)) + (x[,"n_12"] + x[,"n_31"])*log(4*(1 + r_cm1^2) + 3*p2*(1 - 3*r_cm1 + r_cm1^2)) + (x[,"n_11"] + x[,"n_32"])*log(4*(2 - 2*r_cm1 + r_cm1^2) + 3*p2*(-1 + r_cm1 + r_cm1^2))


logL_cm2 <- function(r,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(3*log(2) + log(3)) + (n10 + n13 + n30 + n33)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n20 + n23)*(log(2 - 3*p2) + log(1 - 2*r + 2*r^2)) + (n12 + n31)*log(2*(1 + r^2) - 3*p2*(1 - 3*r + r^2)) + (n11 + n32)*log(2*(2 - 2*r + r^2) - 3*p2*(-1 + r + r^2)) + (n21 + n22)*log(4 + 4*r - 4*r^2 + p2*(3 - 6*r + 6*r^2))
return(L)}
interlogL_cm2 <- function(n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_cm2,c(0,0.5), n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cm2 <- parallel::mcmapply(interlogL_cm2,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_cm2 <- -((x[,"n_10"] + x[,"n_13"] + x[,"n_30"] + x[,"n_33"])*(-2*log10(2) + log10(2 - 3*p2))) - (x[,"n_20"] + x[,"n_23"])*(-log10(2) + log10(2 - 3*p2)) - (x[,"n_12"] + x[,"n_31"])*log10(5/2 + (3*p2)/4) - (x[,"n_11"] + x[,"n_32"])*log10(5/2 + (3*p2)/4) - (x[,"n_21"] + x[,"n_22"])*log10(5 + (3*p2)/2) + (x[,"n_10"] + x[,"n_13"] + x[,"n_30"] + x[,"n_33"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cm2)) + log10(pmax(1e-6,r_cm2))) + (x[,"n_20"] + x[,"n_23"])*(log10(2 - 3*p2) + log10(1 - 2*r_cm2 + 2*r_cm2^2)) + (x[,"n_12"] + x[,"n_31"])*log10(2*(1 + r_cm2^2) - 3*p2*(1 - 3*r_cm2 + r_cm2^2)) + (x[,"n_11"] + x[,"n_32"])*log10(2*(2 - 2*r_cm2 + r_cm2^2) - 3*p2*(-1 + r_cm2 + r_cm2^2)) + (x[,"n_21"] + x[,"n_22"])*log10(4 + 4*r_cm2 - 4*r_cm2^2 + p2*(3 - 6*r_cm2 + 6*r_cm2^2))


logL_cm2 <- (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(3*log(2) + log(3)) + (x[,"n_10"] + x[,"n_13"] + x[,"n_30"] + x[,"n_33"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_cm2)) + log(pmax(1e-6,r_cm2))) + (x[,"n_20"] + x[,"n_23"])*(log(2 - 3*p2) + log(1 - 2*r_cm2 + 2*r_cm2^2)) + (x[,"n_12"] + x[,"n_31"])*log(2*(1 + r_cm2^2) - 3*p2*(1 - 3*r_cm2 + r_cm2^2)) + (x[,"n_11"] + x[,"n_32"])*log(2*(2 - 2*r_cm2 + r_cm2^2) - 3*p2*(-1 + r_cm2 + r_cm2^2)) + (x[,"n_21"] + x[,"n_22"])*log(4 + 4*r_cm2 - 4*r_cm2^2 + p2*(3 - 6*r_cm2 + 6*r_cm2^2))


logL_rc1 <- function(r,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(4*log(2) + 2*log(3)) + (n13 + n30)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(4 + 3*p1 - 2*r - 6*p1*r)) + (n10 + n33)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(2 - 3*p1 + 2*r + 6*p1*r)) + (n12 + n31)*log((-4 - 3*p2)*(-4 - 3*p1 + 2*r + 6*p1*r) + (2 - 3*p2)*r*(2 - 3*p1 + 2*r + 6*p1*r)) + (n20 + n23)*(log(2 - 3*p2) + log(3*p1*(1 - 2*r)^2 + 4*(1 - r + r^2))) + (n11 + n32)*log(-6*p2*(1 - 4*r + r^2) + 4*(4 - r + r^2) - 3*p1*(-1 + 2*r)*(3*p2*(-2 + r) - 2*(1 + r))) + (n21 + n22)*log(3*p1*(-2 + 3*p2)*(1 - 2*r)^2 + 4*(7 + 2*r - 2*r^2 + 3*p2*(1 - r + r^2)))
return(L)}
interlogL_rc1 <- function(n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rc1,c(0,0.5), n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc1 <- parallel::mcmapply(interlogL_rc1,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rc1 <- -((x[,"n_12"] + x[,"n_31"])*log10(-3*(-4 - 3*p2) + (3*(2 - 3*p2))/2)) - (x[,"n_20"] + x[,"n_23"])*(log10(3) + log10(2 - 3*p2)) - (x[,"n_13"] + x[,"n_30"])*(-log10(2) + log10(3) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(2 - 3*p2)) - (x[,"n_21"] + x[,"n_22"])*(2*log10(2) + log10(15/2 + (9*p2)/4)) - (x[,"n_11"] + x[,"n_32"])*log10(15 + (9*p2)/2) + (x[,"n_13"] + x[,"n_30"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_rc1)) + log10(4 + 3*p1 - 2*r_rc1 - 6*p1*r_rc1)) + (x[,"n_10"] + x[,"n_33"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rc1)) + log10(2 - 3*p1 + 2*r_rc1 + 6*p1*r_rc1)) + (x[,"n_12"] + x[,"n_31"])*log10((-4 - 3*p2)*(-4 - 3*p1 + 2*r_rc1 + 6*p1*r_rc1) + (2 - 3*p2)*r_rc1*(2 - 3*p1 + 2*r_rc1 + 6*p1*r_rc1)) + (x[,"n_20"] + x[,"n_23"])*(log10(2 - 3*p2) + log10(3*p1*(1 - 2*r_rc1)^2 + 4*(1 - r_rc1 + r_rc1^2))) + (x[,"n_11"] + x[,"n_32"])*log10(-6*p2*(1 - 4*r_rc1 + r_rc1^2) + 4*(4 - r_rc1 + r_rc1^2) - 3*p1*(-1 + 2*r_rc1)*(3*p2*(-2 + r_rc1) - 2*(1 + r_rc1))) + (x[,"n_21"] + x[,"n_22"])*log10(3*p1*(-2 + 3*p2)*(1 - 2*r_rc1)^2 + 4*(7 + 2*r_rc1 - 2*r_rc1^2 + 3*p2*(1 - r_rc1 + r_rc1^2)))


logL_rc1 <- (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(4*log(2) + 2*log(3)) + (x[,"n_13"] + x[,"n_30"])*(log(2 - 3*p2) + log(pmax(1e-6,r_rc1)) + log(4 + 3*p1 - 2*r_rc1 - 6*p1*r_rc1)) + (x[,"n_10"] + x[,"n_33"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_rc1)) + log(2 - 3*p1 + 2*r_rc1 + 6*p1*r_rc1)) + (x[,"n_12"] + x[,"n_31"])*log((-4 - 3*p2)*(-4 - 3*p1 + 2*r_rc1 + 6*p1*r_rc1) + (2 - 3*p2)*r_rc1*(2 - 3*p1 + 2*r_rc1 + 6*p1*r_rc1)) + (x[,"n_20"] + x[,"n_23"])*(log(2 - 3*p2) + log(3*p1*(1 - 2*r_rc1)^2 + 4*(1 - r_rc1 + r_rc1^2))) + (x[,"n_11"] + x[,"n_32"])*log(-6*p2*(1 - 4*r_rc1 + r_rc1^2) + 4*(4 - r_rc1 + r_rc1^2) - 3*p1*(-1 + 2*r_rc1)*(3*p2*(-2 + r_rc1) - 2*(1 + r_rc1))) + (x[,"n_21"] + x[,"n_22"])*log(3*p1*(-2 + 3*p2)*(1 - 2*r_rc1)^2 + 4*(7 + 2*r_rc1 - 2*r_rc1^2 + 3*p2*(1 - r_rc1 + r_rc1^2)))


logL_rc2 <- function(r,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(5*log(2) + 2*log(3)) + (n13 + n30)*(log(4 + 3*p2) + log(4 + p1*(3 - 6*r) - 2*r) + log(pmax(1e-6,r))) + (n10 + n33)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(2 - 3*p1 + 2*r + 6*p1*r)) + (n12 + n31)*log((-8 + 3*p2)*(-4 - 3*p1 + 2*r + 6*p1*r) + (4 + 3*p2)*r*(2 - 3*p1 + 2*r + 6*p1*r)) + (n20 + n23)*(log(4 + 3*p2) + log(3*p1*(1 - 2*r)^2 + 4*(1 - r + r^2))) + (n11 + n32)*log(6*p2*(1 - 4*r + r^2) + 8*(4 - r + r^2) + 3*p1*(-1 + 2*r)*(3*p2*(-2 + r) + 4*(1 + r))) + (n21 + n22)*log(-3*p1*(4 + 3*p2)*(1 - 2*r)^2 - 4*(-14 - 4*r + 4*r^2 + 3*p2*(1 - r + r^2)))
return(L)}
interlogL_rc2 <- function(n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rc2,c(0,0.5), n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc2 <- parallel::mcmapply(interlogL_rc2,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rc2 <- -((x[,"n_11"] + x[,"n_32"])*log10(30 - (9*p2)/2)) - (x[,"n_21"] + x[,"n_22"])*(2*log10(2) + log10(15 - (9*p2)/4)) - (x[,"n_20"] + x[,"n_23"])*(log10(3) + log10(4 + 3*p2)) - (x[,"n_13"] + x[,"n_30"])*(-log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_12"] + x[,"n_31"])*log10(-3*(-8 + 3*p2) + (3*(4 + 3*p2))/2) + (x[,"n_13"] + x[,"n_30"])*(log10(4 + 3*p2) + log10(4 + p1*(3 - 6*r_rc2) - 2*r_rc2) + log10(pmax(1e-6,r_rc2))) + (x[,"n_10"] + x[,"n_33"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rc2)) + log10(2 - 3*p1 + 2*r_rc2 + 6*p1*r_rc2)) + (x[,"n_12"] + x[,"n_31"])*log10((-8 + 3*p2)*(-4 - 3*p1 + 2*r_rc2 + 6*p1*r_rc2) + (4 + 3*p2)*r_rc2*(2 - 3*p1 + 2*r_rc2 + 6*p1*r_rc2)) + (x[,"n_20"] + x[,"n_23"])*(log10(4 + 3*p2) + log10(3*p1*(1 - 2*r_rc2)^2 + 4*(1 - r_rc2 + r_rc2^2))) + (x[,"n_11"] + x[,"n_32"])*log10(6*p2*(1 - 4*r_rc2 + r_rc2^2) + 8*(4 - r_rc2 + r_rc2^2) + 3*p1*(-1 + 2*r_rc2)*(3*p2*(-2 + r_rc2) + 4*(1 + r_rc2))) + (x[,"n_21"] + x[,"n_22"])*log10(-3*p1*(4 + 3*p2)*(1 - 2*r_rc2)^2 - 4*(-14 - 4*r_rc2 + 4*r_rc2^2 + 3*p2*(1 - r_rc2 + r_rc2^2)))


logL_rc2 <- (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(5*log(2) + 2*log(3)) + (x[,"n_13"] + x[,"n_30"])*(log(4 + 3*p2) + log(4 + p1*(3 - 6*r_rc2) - 2*r_rc2) + log(pmax(1e-6,r_rc2))) + (x[,"n_10"] + x[,"n_33"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_rc2)) + log(2 - 3*p1 + 2*r_rc2 + 6*p1*r_rc2)) + (x[,"n_12"] + x[,"n_31"])*log((-8 + 3*p2)*(-4 - 3*p1 + 2*r_rc2 + 6*p1*r_rc2) + (4 + 3*p2)*r_rc2*(2 - 3*p1 + 2*r_rc2 + 6*p1*r_rc2)) + (x[,"n_20"] + x[,"n_23"])*(log(4 + 3*p2) + log(3*p1*(1 - 2*r_rc2)^2 + 4*(1 - r_rc2 + r_rc2^2))) + (x[,"n_11"] + x[,"n_32"])*log(6*p2*(1 - 4*r_rc2 + r_rc2^2) + 8*(4 - r_rc2 + r_rc2^2) + 3*p1*(-1 + 2*r_rc2)*(3*p2*(-2 + r_rc2) + 4*(1 + r_rc2))) + (x[,"n_21"] + x[,"n_22"])*log(-3*p1*(4 + 3*p2)*(1 - 2*r_rc2)^2 - 4*(-14 - 4*r_rc2 + 4*r_rc2^2 + 3*p2*(1 - r_rc2 + r_rc2^2)))


logL_rc3 <- function(r,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(5*log(2) + 2*log(3)) + (n10 + n33)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(4 + 3*p1 + 4*r - 6*p1*r)) + (n13 + n30)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(8 - 4*r + p1*(-3 + 6*r))) + (n20 + n23)*(log(2 - 3*p2) + log(-3*p1*(1 - 2*r)^2 + 8*(1 - r + r^2))) + (n12 + n31)*log((2 - 3*p2)*r*(4 + 3*p1 + 4*r - 6*p1*r) + (4 + 3*p2)*(8 - 4*r + p1*(-3 + 6*r))) + (n11 + n32)*log((-2 + 3*p2)*(-1 + r)*(8 - 4*r + p1*(-3 + 6*r)) - (4 + 3*p2)*(-4*(1 + r) + p1*(-3 + 6*r))) + (n21 + n22)*log(-3*p1*(-2 + 3*p2)*(1 - 2*r)^2 + 8*(7 + 2*r - 2*r^2 + 3*p2*(1 - r + r^2)))
return(L)}
interlogL_rc3 <- function(n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rc3,c(0,0.5), n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc3 <- parallel::mcmapply(interlogL_rc3,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rc3 <- -((x[,"n_13"] + x[,"n_30"])*(log10(3) + log10(2 - 3*p2))) - (x[,"n_10"] + x[,"n_33"])*(log10(3) + log10(2 - 3*p2)) - (x[,"n_20"] + x[,"n_23"])*(log10(2) + log10(3) + log10(2 - 3*p2)) - (x[,"n_21"] + x[,"n_22"])*(3*log10(2) + log10(15/2 + (9*p2)/4)) - (x[,"n_12"] + x[,"n_31"])*log10(3*(2 - 3*p2) + 6*(4 + 3*p2)) - (x[,"n_11"] + x[,"n_32"])*log10(-3*(-2 + 3*p2) + 6*(4 + 3*p2)) + (x[,"n_10"] + x[,"n_33"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rc3)) + log10(4 + 3*p1 + 4*r_rc3 - 6*p1*r_rc3)) + (x[,"n_13"] + x[,"n_30"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_rc3)) + log10(8 - 4*r_rc3 + p1*(-3 + 6*r_rc3))) + (x[,"n_20"] + x[,"n_23"])*(log10(2 - 3*p2) + log10(-3*p1*(1 - 2*r_rc3)^2 + 8*(1 - r_rc3 + r_rc3^2))) + (x[,"n_12"] + x[,"n_31"])*log10((2 - 3*p2)*r_rc3*(4 + 3*p1 + 4*r_rc3 - 6*p1*r_rc3) + (4 + 3*p2)*(8 - 4*r_rc3 + p1*(-3 + 6*r_rc3))) + (x[,"n_11"] + x[,"n_32"])*log10((-2 + 3*p2)*(-1 + r_rc3)*(8 - 4*r_rc3 + p1*(-3 + 6*r_rc3)) - (4 + 3*p2)*(-4*(1 + r_rc3) + p1*(-3 + 6*r_rc3))) + (x[,"n_21"] + x[,"n_22"])*log10(-3*p1*(-2 + 3*p2)*(1 - 2*r_rc3)^2 + 8*(7 + 2*r_rc3 - 2*r_rc3^2 + 3*p2*(1 - r_rc3 + r_rc3^2)))


logL_rc3 <- (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(5*log(2) + 2*log(3)) + (x[,"n_10"] + x[,"n_33"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_rc3)) + log(4 + 3*p1 + 4*r_rc3 - 6*p1*r_rc3)) + (x[,"n_13"] + x[,"n_30"])*(log(2 - 3*p2) + log(pmax(1e-6,r_rc3)) + log(8 - 4*r_rc3 + p1*(-3 + 6*r_rc3))) + (x[,"n_20"] + x[,"n_23"])*(log(2 - 3*p2) + log(-3*p1*(1 - 2*r_rc3)^2 + 8*(1 - r_rc3 + r_rc3^2))) + (x[,"n_12"] + x[,"n_31"])*log((2 - 3*p2)*r_rc3*(4 + 3*p1 + 4*r_rc3 - 6*p1*r_rc3) + (4 + 3*p2)*(8 - 4*r_rc3 + p1*(-3 + 6*r_rc3))) + (x[,"n_11"] + x[,"n_32"])*log((-2 + 3*p2)*(-1 + r_rc3)*(8 - 4*r_rc3 + p1*(-3 + 6*r_rc3)) - (4 + 3*p2)*(-4*(1 + r_rc3) + p1*(-3 + 6*r_rc3))) + (x[,"n_21"] + x[,"n_22"])*log(-3*p1*(-2 + 3*p2)*(1 - 2*r_rc3)^2 + 8*(7 + 2*r_rc3 - 2*r_rc3^2 + 3*p2*(1 - r_rc3 + r_rc3^2)))


logL_rc4 <- function(r,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(6*log(2) + 2*log(3)) + (n10 + n33)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(4 + 3*p1 + 4*r - 6*p1*r)) + (n13 + n30)*(log(4 + 3*p2) + log(pmax(1e-6,r)) + log(8 - 4*r + p1*(-3 + 6*r))) + (n20 + n23)*(log(4 + 3*p2) + log(-3*p1*(1 - 2*r)^2 + 8*(1 - r + r^2))) + (n12 + n31)*log((4 + 3*p2)*r*(4 + 3*p1 + 4*r - 6*p1*r) - (-8 + 3*p2)*(8 - 4*r + p1*(-3 + 6*r))) + (n11 + n32)*log((-4 - 3*p2)*(-1 + r)*(8 - 4*r + p1*(-3 + 6*r)) + (-8 + 3*p2)*(-4*(1 + r) + p1*(-3 + 6*r))) + (n21 + n22)*log(3*p1*(4 + 3*p2)*(1 - 2*r)^2 - 8*(-14 - 4*r + 4*r^2 + 3*p2*(1 - r + r^2)))
return(L)}
interlogL_rc4 <- function(n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rc4,c(0,0.5), n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc4 <- parallel::mcmapply(interlogL_rc4,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rc4 <- -((x[,"n_21"] + x[,"n_22"])*(3*log10(2) + log10(15 - (9*p2)/4))) - (x[,"n_13"] + x[,"n_30"])*(log10(3) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_33"])*(log10(3) + log10(4 + 3*p2)) - (x[,"n_20"] + x[,"n_23"])*(log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_11"] + x[,"n_32"])*log10(-3*(-4 - 3*p2) - 6*(-8 + 3*p2)) - (x[,"n_12"] + x[,"n_31"])*log10(-6*(-8 + 3*p2) + 3*(4 + 3*p2)) + (x[,"n_10"] + x[,"n_33"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rc4)) + log10(4 + 3*p1 + 4*r_rc4 - 6*p1*r_rc4)) + (x[,"n_13"] + x[,"n_30"])*(log10(4 + 3*p2) + log10(pmax(1e-6,r_rc4)) + log10(8 - 4*r_rc4 + p1*(-3 + 6*r_rc4))) + (x[,"n_20"] + x[,"n_23"])*(log10(4 + 3*p2) + log10(-3*p1*(1 - 2*r_rc4)^2 + 8*(1 - r_rc4 + r_rc4^2))) + (x[,"n_12"] + x[,"n_31"])*log10((4 + 3*p2)*r_rc4*(4 + 3*p1 + 4*r_rc4 - 6*p1*r_rc4) - (-8 + 3*p2)*(8 - 4*r_rc4 + p1*(-3 + 6*r_rc4))) + (x[,"n_11"] + x[,"n_32"])*log10((-4 - 3*p2)*(-1 + r_rc4)*(8 - 4*r_rc4 + p1*(-3 + 6*r_rc4)) + (-8 + 3*p2)*(-4*(1 + r_rc4) + p1*(-3 + 6*r_rc4))) + (x[,"n_21"] + x[,"n_22"])*log10(3*p1*(4 + 3*p2)*(1 - 2*r_rc4)^2 - 8*(-14 - 4*r_rc4 + 4*r_rc4^2 + 3*p2*(1 - r_rc4 + r_rc4^2)))


logL_rc4 <- (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(6*log(2) + 2*log(3)) + (x[,"n_10"] + x[,"n_33"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_rc4)) + log(4 + 3*p1 + 4*r_rc4 - 6*p1*r_rc4)) + (x[,"n_13"] + x[,"n_30"])*(log(4 + 3*p2) + log(pmax(1e-6,r_rc4)) + log(8 - 4*r_rc4 + p1*(-3 + 6*r_rc4))) + (x[,"n_20"] + x[,"n_23"])*(log(4 + 3*p2) + log(-3*p1*(1 - 2*r_rc4)^2 + 8*(1 - r_rc4 + r_rc4^2))) + (x[,"n_12"] + x[,"n_31"])*log((4 + 3*p2)*r_rc4*(4 + 3*p1 + 4*r_rc4 - 6*p1*r_rc4) - (-8 + 3*p2)*(8 - 4*r_rc4 + p1*(-3 + 6*r_rc4))) + (x[,"n_11"] + x[,"n_32"])*log((-4 - 3*p2)*(-1 + r_rc4)*(8 - 4*r_rc4 + p1*(-3 + 6*r_rc4)) + (-8 + 3*p2)*(-4*(1 + r_rc4) + p1*(-3 + 6*r_rc4))) + (x[,"n_21"] + x[,"n_22"])*log(3*p1*(4 + 3*p2)*(1 - 2*r_rc4)^2 - 8*(-14 - 4*r_rc4 + 4*r_rc4^2 + 3*p2*(1 - r_rc4 + r_rc4^2)))


logL_rm1 <- function(r,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(5*log(2) + 2*log(3)) + (n13 + n30)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(4 + 3*p1 - 2*r - 6*p1*r)) + (n10 + n33)*(log(4 + 3*p2) + log(pmax(1e-6,r)) + log(2 - 3*p1 + 2*r + 6*p1*r)) + (n20 + n23)*(log(4 + 3*p2) + log(2 - 3*p1*(1 - 2*r)^2 + 4*r - 4*r^2)) + (n11 + n32)*log((4 + 3*p2)*(4 + p1*(3 - 6*r) - 2*r)*r - (-8 + 3*p2)*(2 - 3*p1 + 2*r + 6*p1*r)) + (n12 + n31)*log((-8 + 3*p2)*(-4 - 3*p1 + 2*r + 6*p1*r) - (4 + 3*p2)*(-1 + r)*(2 - 3*p1 + 2*r + 6*p1*r)) + (n21 + n22)*log(3*p1*(4 + 3*p2)*(1 - 2*r)^2 + 2*(8*(4 - r + r^2) + p2*(-3 - 6*r + 6*r^2)))
return(L)}
interlogL_rm1 <- function(n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rm1,c(0,0.5), n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rm1 <- parallel::mcmapply(interlogL_rm1,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rm1 <- -((x[,"n_21"] + x[,"n_22"])*(log10(2) + log10(30 - (9*p2)/2))) - (x[,"n_20"] + x[,"n_23"])*(log10(3) + log10(4 + 3*p2)) - (x[,"n_13"] + x[,"n_30"])*(-log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_12"] + x[,"n_31"])*log10(-3*(-8 + 3*p2) + (3*(4 + 3*p2))/2) - (x[,"n_11"] + x[,"n_32"])*log10(-3*(-8 + 3*p2) + (3*(4 + 3*p2))/2) + (x[,"n_13"] + x[,"n_30"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rm1)) + log10(4 + 3*p1 - 2*r_rm1 - 6*p1*r_rm1)) + (x[,"n_10"] + x[,"n_33"])*(log10(4 + 3*p2) + log10(pmax(1e-6,r_rm1)) + log10(2 - 3*p1 + 2*r_rm1 + 6*p1*r_rm1)) + (x[,"n_20"] + x[,"n_23"])*(log10(4 + 3*p2) + log10(2 - 3*p1*(1 - 2*r_rm1)^2 + 4*r_rm1 - 4*r_rm1^2)) + (x[,"n_11"] + x[,"n_32"])*log10((4 + 3*p2)*(4 + p1*(3 - 6*r_rm1) - 2*r_rm1)*r_rm1 - (-8 + 3*p2)*(2 - 3*p1 + 2*r_rm1 + 6*p1*r_rm1)) + (x[,"n_12"] + x[,"n_31"])*log10((-8 + 3*p2)*(-4 - 3*p1 + 2*r_rm1 + 6*p1*r_rm1) - (4 + 3*p2)*(-1 + r_rm1)*(2 - 3*p1 + 2*r_rm1 + 6*p1*r_rm1)) + (x[,"n_21"] + x[,"n_22"])*log10(3*p1*(4 + 3*p2)*(1 - 2*r_rm1)^2 + 2*(8*(4 - r_rm1 + r_rm1^2) + p2*(-3 - 6*r_rm1 + 6*r_rm1^2)))


logL_rm1 <- (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(5*log(2) + 2*log(3)) + (x[,"n_13"] + x[,"n_30"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_rm1)) + log(4 + 3*p1 - 2*r_rm1 - 6*p1*r_rm1)) + (x[,"n_10"] + x[,"n_33"])*(log(4 + 3*p2) + log(pmax(1e-6,r_rm1)) + log(2 - 3*p1 + 2*r_rm1 + 6*p1*r_rm1)) + (x[,"n_20"] + x[,"n_23"])*(log(4 + 3*p2) + log(2 - 3*p1*(1 - 2*r_rm1)^2 + 4*r_rm1 - 4*r_rm1^2)) + (x[,"n_11"] + x[,"n_32"])*log((4 + 3*p2)*(4 + p1*(3 - 6*r_rm1) - 2*r_rm1)*r_rm1 - (-8 + 3*p2)*(2 - 3*p1 + 2*r_rm1 + 6*p1*r_rm1)) + (x[,"n_12"] + x[,"n_31"])*log((-8 + 3*p2)*(-4 - 3*p1 + 2*r_rm1 + 6*p1*r_rm1) - (4 + 3*p2)*(-1 + r_rm1)*(2 - 3*p1 + 2*r_rm1 + 6*p1*r_rm1)) + (x[,"n_21"] + x[,"n_22"])*log(3*p1*(4 + 3*p2)*(1 - 2*r_rm1)^2 + 2*(8*(4 - r_rm1 + r_rm1^2) + p2*(-3 - 6*r_rm1 + 6*r_rm1^2)))


logL_rm2 <- function(r,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(4*log(2) + 2*log(3)) + (n13 + n30)*(log(2 - 3*p2) + log(4 + p1*(3 - 6*r) - 2*r) + log(pmax(1e-6,1 - r))) + (n10 + n33)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(2 - 3*p1 + 2*r + 6*p1*r)) + (n20 + n23)*(log(2 - 3*p2) + log(2 - 3*p1*(1 - 2*r)^2 + 4*r - 4*r^2)) + (n21 + n22)*log(-3*p1*(-2 + 3*p2)*(1 - 2*r)^2 + p2*(6 + 12*r - 12*r^2) + 8*(4 - r + r^2)) + (n12 + n31)*log(6*p2*(1 - r + r^2) - 4*(-5 + 2*r + r^2) + 3*p1*(-1 + 2*r)*(3*p2*(-2 + r) - 2*(1 + r))) + (n11 + n32)*log(8 + 16*r - 4*r^2 + 6*p2*(1 - r + r^2) + 3*p1*(-1 + 2*r)*(4 - 2*r + 3*p2*(1 + r)))
return(L)}
interlogL_rm2 <- function(n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rm2,c(0,0.5), n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rm2 <- parallel::mcmapply(interlogL_rm2,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rm2 <- -((x[,"n_20"] + x[,"n_23"])*(log10(3) + log10(2 - 3*p2))) - (x[,"n_13"] + x[,"n_30"])*(-log10(2) + log10(3) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(2 - 3*p2)) - (x[,"n_12"] + x[,"n_31"])*log10(15 + (9*p2)/2) - (x[,"n_11"] + x[,"n_32"])*log10(15 + (9*p2)/2) - (x[,"n_21"] + x[,"n_22"])*log10(30 + 9*p2) + (x[,"n_13"] + x[,"n_30"])*(log10(2 - 3*p2) + log10(4 + p1*(3 - 6*r_rm2) - 2*r_rm2) + log10(pmax(1e-6,1 - r_rm2))) + (x[,"n_10"] + x[,"n_33"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_rm2)) + log10(2 - 3*p1 + 2*r_rm2 + 6*p1*r_rm2)) + (x[,"n_20"] + x[,"n_23"])*(log10(2 - 3*p2) + log10(2 - 3*p1*(1 - 2*r_rm2)^2 + 4*r_rm2 - 4*r_rm2^2)) + (x[,"n_21"] + x[,"n_22"])*log10(-3*p1*(-2 + 3*p2)*(1 - 2*r_rm2)^2 + p2*(6 + 12*r_rm2 - 12*r_rm2^2) + 8*(4 - r_rm2 + r_rm2^2)) + (x[,"n_12"] + x[,"n_31"])*log10(6*p2*(1 - r_rm2 + r_rm2^2) - 4*(-5 + 2*r_rm2 + r_rm2^2) + 3*p1*(-1 + 2*r_rm2)*(3*p2*(-2 + r_rm2) - 2*(1 + r_rm2))) + (x[,"n_11"] + x[,"n_32"])*log10(8 + 16*r_rm2 - 4*r_rm2^2 + 6*p2*(1 - r_rm2 + r_rm2^2) + 3*p1*(-1 + 2*r_rm2)*(4 - 2*r_rm2 + 3*p2*(1 + r_rm2)))


logL_rm2 <- (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(4*log(2) + 2*log(3)) + (x[,"n_13"] + x[,"n_30"])*(log(2 - 3*p2) + log(4 + p1*(3 - 6*r_rm2) - 2*r_rm2) + log(pmax(1e-6,1 - r_rm2))) + (x[,"n_10"] + x[,"n_33"])*(log(2 - 3*p2) + log(pmax(1e-6,r_rm2)) + log(2 - 3*p1 + 2*r_rm2 + 6*p1*r_rm2)) + (x[,"n_20"] + x[,"n_23"])*(log(2 - 3*p2) + log(2 - 3*p1*(1 - 2*r_rm2)^2 + 4*r_rm2 - 4*r_rm2^2)) + (x[,"n_21"] + x[,"n_22"])*log(-3*p1*(-2 + 3*p2)*(1 - 2*r_rm2)^2 + p2*(6 + 12*r_rm2 - 12*r_rm2^2) + 8*(4 - r_rm2 + r_rm2^2)) + (x[,"n_12"] + x[,"n_31"])*log(6*p2*(1 - r_rm2 + r_rm2^2) - 4*(-5 + 2*r_rm2 + r_rm2^2) + 3*p1*(-1 + 2*r_rm2)*(3*p2*(-2 + r_rm2) - 2*(1 + r_rm2))) + (x[,"n_11"] + x[,"n_32"])*log(8 + 16*r_rm2 - 4*r_rm2^2 + 6*p2*(1 - r_rm2 + r_rm2^2) + 3*p1*(-1 + 2*r_rm2)*(4 - 2*r_rm2 + 3*p2*(1 + r_rm2)))


logL_rm3 <- function(r,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(6*log(2) + 2*log(3)) + (n10 + n33)*(log(4 + 3*p2) + log(pmax(1e-6,r)) + log(4 + 3*p1 + 4*r - 6*p1*r)) + (n20 + n23)*(log(4 + 3*p2) + log(4 + 3*p1*(1 - 2*r)^2 + 8*r - 8*r^2)) + (n13 + n30)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(8 - 4*r + p1*(-3 + 6*r))) + (n11 + n32)*log((4 + 3*p2)*r*(8 - 4*r + p1*(-3 + 6*r)) + (-8 + 3*p2)*(-4*(1 + r) + p1*(-3 + 6*r))) + (n12 + n31)*log((8 - 3*p2)*(8 - 4*r + p1*(-3 + 6*r)) + (4 + 3*p2)*(-1 + r)*(-4*(1 + r) + p1*(-3 + 6*r))) + (n21 + n22)*log(-3*p1*(4 + 3*p2)*(1 - 2*r)^2 + 4*(8*(4 - r + r^2) + p2*(-3 - 6*r + 6*r^2)))
return(L)}
interlogL_rm3 <- function(n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rm3,c(0,0.5), n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rm3 <- parallel::mcmapply(interlogL_rm3,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rm3 <- -((x[,"n_21"] + x[,"n_22"])*(2*log10(2) + log10(30 - (9*p2)/2))) - (x[,"n_13"] + x[,"n_30"])*(log10(3) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_33"])*(log10(3) + log10(4 + 3*p2)) - (x[,"n_20"] + x[,"n_23"])*(log10(2) + log10(3) + log10(4 + 3*p2)) - (x[,"n_12"] + x[,"n_31"])*log10(6*(8 - 3*p2) + 3*(4 + 3*p2)) - (x[,"n_11"] + x[,"n_32"])*log10(-6*(-8 + 3*p2) + 3*(4 + 3*p2)) + (x[,"n_10"] + x[,"n_33"])*(log10(4 + 3*p2) + log10(pmax(1e-6,r_rm3)) + log10(4 + 3*p1 + 4*r_rm3 - 6*p1*r_rm3)) + (x[,"n_20"] + x[,"n_23"])*(log10(4 + 3*p2) + log10(4 + 3*p1*(1 - 2*r_rm3)^2 + 8*r_rm3 - 8*r_rm3^2)) + (x[,"n_13"] + x[,"n_30"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rm3)) + log10(8 - 4*r_rm3 + p1*(-3 + 6*r_rm3))) + (x[,"n_11"] + x[,"n_32"])*log10((4 + 3*p2)*r_rm3*(8 - 4*r_rm3 + p1*(-3 + 6*r_rm3)) + (-8 + 3*p2)*(-4*(1 + r_rm3) + p1*(-3 + 6*r_rm3))) + (x[,"n_12"] + x[,"n_31"])*log10((8 - 3*p2)*(8 - 4*r_rm3 + p1*(-3 + 6*r_rm3)) + (4 + 3*p2)*(-1 + r_rm3)*(-4*(1 + r_rm3) + p1*(-3 + 6*r_rm3))) + (x[,"n_21"] + x[,"n_22"])*log10(-3*p1*(4 + 3*p2)*(1 - 2*r_rm3)^2 + 4*(8*(4 - r_rm3 + r_rm3^2) + p2*(-3 - 6*r_rm3 + 6*r_rm3^2)))


logL_rm3 <- (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(6*log(2) + 2*log(3)) + (x[,"n_10"] + x[,"n_33"])*(log(4 + 3*p2) + log(pmax(1e-6,r_rm3)) + log(4 + 3*p1 + 4*r_rm3 - 6*p1*r_rm3)) + (x[,"n_20"] + x[,"n_23"])*(log(4 + 3*p2) + log(4 + 3*p1*(1 - 2*r_rm3)^2 + 8*r_rm3 - 8*r_rm3^2)) + (x[,"n_13"] + x[,"n_30"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_rm3)) + log(8 - 4*r_rm3 + p1*(-3 + 6*r_rm3))) + (x[,"n_11"] + x[,"n_32"])*log((4 + 3*p2)*r_rm3*(8 - 4*r_rm3 + p1*(-3 + 6*r_rm3)) + (-8 + 3*p2)*(-4*(1 + r_rm3) + p1*(-3 + 6*r_rm3))) + (x[,"n_12"] + x[,"n_31"])*log((8 - 3*p2)*(8 - 4*r_rm3 + p1*(-3 + 6*r_rm3)) + (4 + 3*p2)*(-1 + r_rm3)*(-4*(1 + r_rm3) + p1*(-3 + 6*r_rm3))) + (x[,"n_21"] + x[,"n_22"])*log(-3*p1*(4 + 3*p2)*(1 - 2*r_rm3)^2 + 4*(8*(4 - r_rm3 + r_rm3^2) + p2*(-3 - 6*r_rm3 + 6*r_rm3^2)))


logL_rm4 <- function(r,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(5*log(2) + 2*log(3)) + (n10 + n33)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(4 + 3*p1 + 4*r - 6*p1*r)) + (n20 + n23)*(log(2 - 3*p2) + log(4 + 3*p1*(1 - 2*r)^2 + 8*r - 8*r^2)) + (n13 + n30)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(8 - 4*r + p1*(-3 + 6*r))) + (n12 + n31)*log((-2 + 3*p2)*(-1 + r)*(4 + 3*p1 + 4*r - 6*p1*r) + (4 + 3*p2)*(8 - 4*r + p1*(-3 + 6*r))) + (n11 + n32)*log((2 - 3*p2)*r*(8 - 4*r + p1*(-3 + 6*r)) - (4 + 3*p2)*(-4*(1 + r) + p1*(-3 + 6*r))) + (n21 + n22)*log(3*p1*(-2 + 3*p2)*(1 - 2*r)^2 + 4*(p2*(3 + 6*r - 6*r^2) + 4*(4 - r + r^2)))
return(L)}
interlogL_rm4 <- function(n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rm4,c(0,0.5), n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rm4 <- parallel::mcmapply(interlogL_rm4,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rm4 <- -((x[,"n_13"] + x[,"n_30"])*(log10(3) + log10(2 - 3*p2))) - (x[,"n_10"] + x[,"n_33"])*(log10(3) + log10(2 - 3*p2)) - (x[,"n_20"] + x[,"n_23"])*(log10(2) + log10(3) + log10(2 - 3*p2)) - (x[,"n_21"] + x[,"n_22"])*(2*log10(2) + log10(15 + (9*p2)/2)) - (x[,"n_11"] + x[,"n_32"])*log10(3*(2 - 3*p2) + 6*(4 + 3*p2)) - (x[,"n_12"] + x[,"n_31"])*log10(-3*(-2 + 3*p2) + 6*(4 + 3*p2)) + (x[,"n_10"] + x[,"n_33"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_rm4)) + log10(4 + 3*p1 + 4*r_rm4 - 6*p1*r_rm4)) + (x[,"n_20"] + x[,"n_23"])*(log10(2 - 3*p2) + log10(4 + 3*p1*(1 - 2*r_rm4)^2 + 8*r_rm4 - 8*r_rm4^2)) + (x[,"n_13"] + x[,"n_30"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rm4)) + log10(8 - 4*r_rm4 + p1*(-3 + 6*r_rm4))) + (x[,"n_12"] + x[,"n_31"])*log10((-2 + 3*p2)*(-1 + r_rm4)*(4 + 3*p1 + 4*r_rm4 - 6*p1*r_rm4) + (4 + 3*p2)*(8 - 4*r_rm4 + p1*(-3 + 6*r_rm4))) + (x[,"n_11"] + x[,"n_32"])*log10((2 - 3*p2)*r_rm4*(8 - 4*r_rm4 + p1*(-3 + 6*r_rm4)) - (4 + 3*p2)*(-4*(1 + r_rm4) + p1*(-3 + 6*r_rm4))) + (x[,"n_21"] + x[,"n_22"])*log10(3*p1*(-2 + 3*p2)*(1 - 2*r_rm4)^2 + 4*(p2*(3 + 6*r_rm4 - 6*r_rm4^2) + 4*(4 - r_rm4 + r_rm4^2)))


logL_rm4 <- (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(5*log(2) + 2*log(3)) + (x[,"n_10"] + x[,"n_33"])*(log(2 - 3*p2) + log(pmax(1e-6,r_rm4)) + log(4 + 3*p1 + 4*r_rm4 - 6*p1*r_rm4)) + (x[,"n_20"] + x[,"n_23"])*(log(2 - 3*p2) + log(4 + 3*p1*(1 - 2*r_rm4)^2 + 8*r_rm4 - 8*r_rm4^2)) + (x[,"n_13"] + x[,"n_30"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_rm4)) + log(8 - 4*r_rm4 + p1*(-3 + 6*r_rm4))) + (x[,"n_12"] + x[,"n_31"])*log((-2 + 3*p2)*(-1 + r_rm4)*(4 + 3*p1 + 4*r_rm4 - 6*p1*r_rm4) + (4 + 3*p2)*(8 - 4*r_rm4 + p1*(-3 + 6*r_rm4))) + (x[,"n_11"] + x[,"n_32"])*log((2 - 3*p2)*r_rm4*(8 - 4*r_rm4 + p1*(-3 + 6*r_rm4)) - (4 + 3*p2)*(-4*(1 + r_rm4) + p1*(-3 + 6*r_rm4))) + (x[,"n_21"] + x[,"n_22"])*log(3*p1*(-2 + 3*p2)*(1 - 2*r_rm4)^2 + 4*(p2*(3 + 6*r_rm4 - 6*r_rm4^2) + 4*(4 - r_rm4 + r_rm4^2)))


return(list(
r_mat = cbind(r_cc1,r_cc2,r_cm1,r_cm2,r_rc1,r_rc2,r_rc3,r_rc4,r_rm1,r_rm2,r_rm3,r_rm4,0.499),
LOD_mat = cbind(LOD_cc1,LOD_cc2,LOD_cm1,LOD_cm2,LOD_rc1,LOD_rc2,LOD_rc3,LOD_rc4,LOD_rm1,LOD_rm2,LOD_rm3,LOD_rm4,0),
logL_mat = cbind(logL_cc1,logL_cc2,logL_cm1,logL_cm2,logL_rc1,logL_rc2,logL_rc3,logL_rc4,logL_rm1,logL_rm2,logL_rm3,logL_rm4,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling coupling","coupling mixed","coupling mixed","repulsion coupling","repulsion coupling","repulsion coupling","repulsion coupling","repulsion mixed","repulsion mixed","repulsion mixed","repulsion mixed","unknown")
)
)
}

#' @rdname p4_functions
p4_1.3_1.3 <- function(x,p1,p2,ncores=1){


logL_cc <- function(r,n11,n12,n13,n21,n23,n31,n32,n33,n22,p1,p2) {
L <- (-2*n11 - n12 - 2*n13 - n21 - n23 - 2*n31 - n32 - 2*n33)*log(2) + 2*(n13 + n31)*log(pmax(1e-6,r)) + (n12 + n21 + n23 + n32)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n11 + n33)*log(1 - 2*r + r^2) + n22*log(1/2 - r + r^2)
return(L)}
interlogL_cc <- function(n11,n12,n13,n21,n23,n31,n32,n33,n22,p1,p2) {
optimize(logL_cc,c(0,0.5), n11,n12,n13,n21,n23,n31,n32,n33,n22,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc <- parallel::mcmapply(interlogL_cc,x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_21"],x[,"n_23"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_22"], p1, p2, mc.cores = ncores)


LOD_cc <- 2*x[,"n_22"]*log10(2) + 2*(x[,"n_13"] + x[,"n_31"])*log10(2) + 2*(x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log10(2) + 2*(x[,"n_11"] + x[,"n_33"])*log10(2) + 2*(x[,"n_13"] + x[,"n_31"])*log10(pmax(1e-6,r_cc)) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(log10(pmax(1e-6,1 - r_cc)) + log10(pmax(1e-6,r_cc))) + (x[,"n_11"] + x[,"n_33"])*log10(1 - 2*r_cc + r_cc^2) + x[,"n_22"]*log10(1/2 - r_cc + r_cc^2)


logL_cc <- (-2*x[,"n_11"] - x[,"n_12"] - 2*x[,"n_13"] - x[,"n_21"] - x[,"n_23"] - 2*x[,"n_31"] - x[,"n_32"] - 2*x[,"n_33"])*log(2) + 2*(x[,"n_13"] + x[,"n_31"])*log(pmax(1e-6,r_cc)) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(log(pmax(1e-6,1 - r_cc)) + log(pmax(1e-6,r_cc))) + (x[,"n_11"] + x[,"n_33"])*log(1 - 2*r_cc + r_cc^2) + x[,"n_22"]*log(1/2 - r_cc + r_cc^2)


logL_cm1 <- function(r,n11,n12,n13,n21,n22,n23,n31,n32,n33,p1,p2) {
L <- (-4*n11 - 4*n12 - 4*n13 - 4*n21 - 3*n22 - 4*n23 - 4*n31 - 4*n32 - 4*n33)*log(2) + (-n11 - n12 - n13 - n21 - n22 - n23 - n31 - n32 - n33)*log(3) + n22*log(4 + 3*p2*(1 - 2*r)^2 + 8*r - 8*r^2) + (n11 + n33)*(log(pmax(1e-6,1 - r)) + log(4*(1 + r) - p2*(-3 + 6*r))) + (n13 + n31)*(log(pmax(1e-6,r)) + log(8 - 4*r + p2*(-3 + 6*r))) + (n12 + n21 + n23 + n32)*log(-3*p2*(1 - 2*r)^2 + 8*(1 - r + r^2))
return(L)}
interlogL_cm1 <- function(n11,n12,n13,n21,n22,n23,n31,n32,n33,p1,p2) {
optimize(logL_cm1,c(0,0.5), n11,n12,n13,n21,n22,n23,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cm1 <- parallel::mcmapply(interlogL_cm1,x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_cm1 <- -((x[,"n_13"] + x[,"n_31"])*log10(3)) - (x[,"n_11"] + x[,"n_33"])*log10(3) - x[,"n_22"]*(log10(2) + log10(3)) - (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(log10(2) + log10(3)) + x[,"n_22"]*log10(4 + 3*p2*(1 - 2*r_cm1)^2 + 8*r_cm1 - 8*r_cm1^2) + (x[,"n_11"] + x[,"n_33"])*(log10(pmax(1e-6,1 - r_cm1)) + log10(4*(1 + r_cm1) - p2*(-3 + 6*r_cm1))) + (x[,"n_13"] + x[,"n_31"])*(log10(pmax(1e-6,r_cm1)) + log10(8 - 4*r_cm1 + p2*(-3 + 6*r_cm1))) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log10(-3*p2*(1 - 2*r_cm1)^2 + 8*(1 - r_cm1 + r_cm1^2))


logL_cm1 <- (-4*x[,"n_11"] - 4*x[,"n_12"] - 4*x[,"n_13"] - 4*x[,"n_21"] - 3*x[,"n_22"] - 4*x[,"n_23"] - 4*x[,"n_31"] - 4*x[,"n_32"] - 4*x[,"n_33"])*log(2) + (-x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*log(3) + x[,"n_22"]*log(4 + 3*p2*(1 - 2*r_cm1)^2 + 8*r_cm1 - 8*r_cm1^2) + (x[,"n_11"] + x[,"n_33"])*(log(pmax(1e-6,1 - r_cm1)) + log(4*(1 + r_cm1) - p2*(-3 + 6*r_cm1))) + (x[,"n_13"] + x[,"n_31"])*(log(pmax(1e-6,r_cm1)) + log(8 - 4*r_cm1 + p2*(-3 + 6*r_cm1))) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log(-3*p2*(1 - 2*r_cm1)^2 + 8*(1 - r_cm1 + r_cm1^2))


logL_cm2 <- function(r,n11,n12,n13,n21,n22,n23,n31,n32,n33,p1,p2) {
L <- (-3*n11 - 3*n12 - 3*n13 - 3*n21 - 2*n22 - 3*n23 - 3*n31 - 3*n32 - 3*n33)*log(2) + (-n11 - n12 - n13 - n21 - n22 - n23 - n31 - n32 - n33)*log(3) + (n13 + n31)*(log(4 + p2*(3 - 6*r) - 2*r) + log(pmax(1e-6,r))) + (n11 + n33)*(log(pmax(1e-6,1 - r)) + log(2 - 3*p2 + 2*r + 6*p2*r)) + n22*log(2 - 3*p2*(1 - 2*r)^2 + 4*r - 4*r^2) + (n12 + n21 + n23 + n32)*log(3*p2*(1 - 2*r)^2 + 4*(1 - r + r^2))
return(L)}
interlogL_cm2 <- function(n11,n12,n13,n21,n22,n23,n31,n32,n33,p1,p2) {
optimize(logL_cm2,c(0,0.5), n11,n12,n13,n21,n22,n23,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cm2 <- parallel::mcmapply(interlogL_cm2,x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_cm2 <- -(x[,"n_22"]*log10(3)) - (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log10(3) - (x[,"n_13"] + x[,"n_31"])*(-log10(2) + log10(3)) - (x[,"n_11"] + x[,"n_33"])*(-log10(2) + log10(3)) + (x[,"n_13"] + x[,"n_31"])*(log10(4 + p2*(3 - 6*r_cm2) - 2*r_cm2) + log10(pmax(1e-6,r_cm2))) + (x[,"n_11"] + x[,"n_33"])*(log10(pmax(1e-6,1 - r_cm2)) + log10(2 - 3*p2 + 2*r_cm2 + 6*p2*r_cm2)) + x[,"n_22"]*log10(2 - 3*p2*(1 - 2*r_cm2)^2 + 4*r_cm2 - 4*r_cm2^2) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log10(3*p2*(1 - 2*r_cm2)^2 + 4*(1 - r_cm2 + r_cm2^2))


logL_cm2 <- (-3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_21"] - 2*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"])*log(2) + (-x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*log(3) + (x[,"n_13"] + x[,"n_31"])*(log(4 + p2*(3 - 6*r_cm2) - 2*r_cm2) + log(pmax(1e-6,r_cm2))) + (x[,"n_11"] + x[,"n_33"])*(log(pmax(1e-6,1 - r_cm2)) + log(2 - 3*p2 + 2*r_cm2 + 6*p2*r_cm2)) + x[,"n_22"]*log(2 - 3*p2*(1 - 2*r_cm2)^2 + 4*r_cm2 - 4*r_cm2^2) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log(3*p2*(1 - 2*r_cm2)^2 + 4*(1 - r_cm2 + r_cm2^2))


logL_rc1 <- function(r,n11,n12,n13,n21,n22,n23,n31,n32,n33,p1,p2) {
L <- (-3*n11 - 3*n12 - 3*n13 - 3*n21 - 2*n22 - 3*n23 - 3*n31 - 3*n32 - 3*n33)*log(2) + (-n11 - n12 - n13 - n21 - n22 - n23 - n31 - n32 - n33)*log(3) + (n13 + n31)*(log(4 + p1*(3 - 6*r) - 2*r) + log(pmax(1e-6,r))) + (n11 + n33)*(log(pmax(1e-6,1 - r)) + log(2 - 3*p1 + 2*r + 6*p1*r)) + n22*log(2 - 3*p1*(1 - 2*r)^2 + 4*r - 4*r^2) + (n12 + n21 + n23 + n32)*log(3*p1*(1 - 2*r)^2 + 4*(1 - r + r^2))
return(L)}
interlogL_rc1 <- function(n11,n12,n13,n21,n22,n23,n31,n32,n33,p1,p2) {
optimize(logL_rc1,c(0,0.5), n11,n12,n13,n21,n22,n23,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc1 <- parallel::mcmapply(interlogL_rc1,x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rc1 <- -(x[,"n_22"]*log10(3)) - (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log10(3) - (x[,"n_13"] + x[,"n_31"])*(-log10(2) + log10(3)) - (x[,"n_11"] + x[,"n_33"])*(-log10(2) + log10(3)) + (x[,"n_13"] + x[,"n_31"])*(log10(4 + p1*(3 - 6*r_rc1) - 2*r_rc1) + log10(pmax(1e-6,r_rc1))) + (x[,"n_11"] + x[,"n_33"])*(log10(pmax(1e-6,1 - r_rc1)) + log10(2 - 3*p1 + 2*r_rc1 + 6*p1*r_rc1)) + x[,"n_22"]*log10(2 - 3*p1*(1 - 2*r_rc1)^2 + 4*r_rc1 - 4*r_rc1^2) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log10(3*p1*(1 - 2*r_rc1)^2 + 4*(1 - r_rc1 + r_rc1^2))


logL_rc1 <- (-3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_21"] - 2*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"])*log(2) + (-x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*log(3) + (x[,"n_13"] + x[,"n_31"])*(log(4 + p1*(3 - 6*r_rc1) - 2*r_rc1) + log(pmax(1e-6,r_rc1))) + (x[,"n_11"] + x[,"n_33"])*(log(pmax(1e-6,1 - r_rc1)) + log(2 - 3*p1 + 2*r_rc1 + 6*p1*r_rc1)) + x[,"n_22"]*log(2 - 3*p1*(1 - 2*r_rc1)^2 + 4*r_rc1 - 4*r_rc1^2) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log(3*p1*(1 - 2*r_rc1)^2 + 4*(1 - r_rc1 + r_rc1^2))


logL_rc2 <- function(r,n11,n12,n13,n21,n22,n23,n31,n32,n33,p1,p2) {
L <- (-4*n11 - 4*n12 - 4*n13 - 4*n21 - 3*n22 - 4*n23 - 4*n31 - 4*n32 - 4*n33)*log(2) + (-n11 - n12 - n13 - n21 - n22 - n23 - n31 - n32 - n33)*log(3) + n22*log(4 + 3*p1*(1 - 2*r)^2 + 8*r - 8*r^2) + (n11 + n33)*(log(pmax(1e-6,1 - r)) + log(4*(1 + r) - p1*(-3 + 6*r))) + (n13 + n31)*(log(pmax(1e-6,r)) + log(8 - 4*r + p1*(-3 + 6*r))) + (n12 + n21 + n23 + n32)*log(-3*p1*(1 - 2*r)^2 + 8*(1 - r + r^2))
return(L)}
interlogL_rc2 <- function(n11,n12,n13,n21,n22,n23,n31,n32,n33,p1,p2) {
optimize(logL_rc2,c(0,0.5), n11,n12,n13,n21,n22,n23,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc2 <- parallel::mcmapply(interlogL_rc2,x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rc2 <- -((x[,"n_13"] + x[,"n_31"])*log10(3)) - (x[,"n_11"] + x[,"n_33"])*log10(3) - x[,"n_22"]*(log10(2) + log10(3)) - (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(log10(2) + log10(3)) + x[,"n_22"]*log10(4 + 3*p1*(1 - 2*r_rc2)^2 + 8*r_rc2 - 8*r_rc2^2) + (x[,"n_11"] + x[,"n_33"])*(log10(pmax(1e-6,1 - r_rc2)) + log10(4*(1 + r_rc2) - p1*(-3 + 6*r_rc2))) + (x[,"n_13"] + x[,"n_31"])*(log10(pmax(1e-6,r_rc2)) + log10(8 - 4*r_rc2 + p1*(-3 + 6*r_rc2))) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log10(-3*p1*(1 - 2*r_rc2)^2 + 8*(1 - r_rc2 + r_rc2^2))


logL_rc2 <- (-4*x[,"n_11"] - 4*x[,"n_12"] - 4*x[,"n_13"] - 4*x[,"n_21"] - 3*x[,"n_22"] - 4*x[,"n_23"] - 4*x[,"n_31"] - 4*x[,"n_32"] - 4*x[,"n_33"])*log(2) + (-x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*log(3) + x[,"n_22"]*log(4 + 3*p1*(1 - 2*r_rc2)^2 + 8*r_rc2 - 8*r_rc2^2) + (x[,"n_11"] + x[,"n_33"])*(log(pmax(1e-6,1 - r_rc2)) + log(4*(1 + r_rc2) - p1*(-3 + 6*r_rc2))) + (x[,"n_13"] + x[,"n_31"])*(log(pmax(1e-6,r_rc2)) + log(8 - 4*r_rc2 + p1*(-3 + 6*r_rc2))) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log(-3*p1*(1 - 2*r_rc2)^2 + 8*(1 - r_rc2 + r_rc2^2))


logL_rm1 <- function(r,n11,n12,n13,n21,n22,n23,n31,n32,n33,p1,p2) {
L <- (-5*n11 - 4*n12 - 5*n13 - 4*n21 - 3*n22 - 4*n23 - 5*n31 - 4*n32 - 5*n33)*log(2) + 2*(-n11 - n12 - n13 - n21 - n22 - n23 - n31 - n32 - n33)*log(3) + (n11 + n33)*(log(2 - 3*p1 + 2*r + 6*p1*r) + log(4 + 3*p2 + 4*r - 6*p2*r)) + n22*log(20 - 3*p2*(1 - 2*r)^2 - 3*p1*(-2 + 3*p2)*(1 - 2*r)^2 - 8*r + 8*r^2) + (n13 + n31)*(log(4 + p1*(3 - 6*r) - 2*r) + log(8 - 4*r + p2*(-3 + 6*r))) + (n12 + n21 + n23 + n32)*log(3*p2*(1 - 2*r)^2 + 3*p1*(-2 + 3*p2)*(1 - 2*r)^2 + 8*(2 + r - r^2))
return(L)}
interlogL_rm1 <- function(n11,n12,n13,n21,n22,n23,n31,n32,n33,p1,p2) {
optimize(logL_rm1,c(0,0.5), n11,n12,n13,n21,n22,n23,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rm1 <- parallel::mcmapply(interlogL_rm1,x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rm1 <- -(x[,"n_22"]*(log10(2) + 2*log10(3))) - (x[,"n_13"] + x[,"n_31"])*(log10(2) + 2*log10(3)) - (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(log10(2) + 2*log10(3)) - (x[,"n_11"] + x[,"n_33"])*(log10(2) + 2*log10(3)) + (x[,"n_11"] + x[,"n_33"])*(log10(2 - 3*p1 + 2*r_rm1 + 6*p1*r_rm1) + log10(4 + 3*p2 + 4*r_rm1 - 6*p2*r_rm1)) + x[,"n_22"]*log10(20 - 3*p2*(1 - 2*r_rm1)^2 - 3*p1*(-2 + 3*p2)*(1 - 2*r_rm1)^2 - 8*r_rm1 + 8*r_rm1^2) + (x[,"n_13"] + x[,"n_31"])*(log10(4 + p1*(3 - 6*r_rm1) - 2*r_rm1) + log10(8 - 4*r_rm1 + p2*(-3 + 6*r_rm1))) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log10(3*p2*(1 - 2*r_rm1)^2 + 3*p1*(-2 + 3*p2)*(1 - 2*r_rm1)^2 + 8*(2 + r_rm1 - r_rm1^2))


logL_rm1 <- (-5*x[,"n_11"] - 4*x[,"n_12"] - 5*x[,"n_13"] - 4*x[,"n_21"] - 3*x[,"n_22"] - 4*x[,"n_23"] - 5*x[,"n_31"] - 4*x[,"n_32"] - 5*x[,"n_33"])*log(2) + 2*(-x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*log(3) + (x[,"n_11"] + x[,"n_33"])*(log(2 - 3*p1 + 2*r_rm1 + 6*p1*r_rm1) + log(4 + 3*p2 + 4*r_rm1 - 6*p2*r_rm1)) + x[,"n_22"]*log(20 - 3*p2*(1 - 2*r_rm1)^2 - 3*p1*(-2 + 3*p2)*(1 - 2*r_rm1)^2 - 8*r_rm1 + 8*r_rm1^2) + (x[,"n_13"] + x[,"n_31"])*(log(4 + p1*(3 - 6*r_rm1) - 2*r_rm1) + log(8 - 4*r_rm1 + p2*(-3 + 6*r_rm1))) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log(3*p2*(1 - 2*r_rm1)^2 + 3*p1*(-2 + 3*p2)*(1 - 2*r_rm1)^2 + 8*(2 + r_rm1 - r_rm1^2))


logL_rm2 <- function(r,n11,n12,n13,n21,n22,n23,n31,n32,n33,p1,p2) {
L <- (-4*n11 - 3*n12 - 4*n13 - 3*n21 - 2*n22 - 3*n23 - 4*n31 - 3*n32 - 4*n33)*log(2) + 2*(-n11 - n12 - n13 - n21 - n22 - n23 - n31 - n32 - n33)*log(3) + (n13 + n31)*(log(4 + 3*p1 - 2*r - 6*p1*r) + log(4 + 3*p2 - 2*r - 6*p2*r)) + (n11 + n33)*(log(2 - 3*p1 + 2*r + 6*p1*r) + log(2 - 3*p2 + 2*r + 6*p2*r)) + (n12 + n21 + n23 + n32)*log(8 - 3*p2*(1 - 2*r)^2 - 3*p1*(1 + 3*p2)*(1 - 2*r)^2 + 4*r - 4*r^2) + n22*log(10 + 3*p2*(1 - 2*r)^2 + 3*p1*(1 + 3*p2)*(1 - 2*r)^2 - 4*r + 4*r^2)
return(L)}
interlogL_rm2 <- function(n11,n12,n13,n21,n22,n23,n31,n32,n33,p1,p2) {
optimize(logL_rm2,c(0,0.5), n11,n12,n13,n21,n22,n23,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rm2 <- parallel::mcmapply(interlogL_rm2,x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rm2 <- -2*x[,"n_22"]*log10(3) - 2*(x[,"n_13"] + x[,"n_31"])*log10(3) - 2*(x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log10(3) - 2*(x[,"n_11"] + x[,"n_33"])*log10(3) + (x[,"n_13"] + x[,"n_31"])*(log10(4 + 3*p1 - 2*r_rm2 - 6*p1*r_rm2) + log10(4 + 3*p2 - 2*r_rm2 - 6*p2*r_rm2)) + (x[,"n_11"] + x[,"n_33"])*(log10(2 - 3*p1 + 2*r_rm2 + 6*p1*r_rm2) + log10(2 - 3*p2 + 2*r_rm2 + 6*p2*r_rm2)) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log10(8 - 3*p2*(1 - 2*r_rm2)^2 - 3*p1*(1 + 3*p2)*(1 - 2*r_rm2)^2 + 4*r_rm2 - 4*r_rm2^2) + x[,"n_22"]*log10(10 + 3*p2*(1 - 2*r_rm2)^2 + 3*p1*(1 + 3*p2)*(1 - 2*r_rm2)^2 - 4*r_rm2 + 4*r_rm2^2)


logL_rm2 <- (-4*x[,"n_11"] - 3*x[,"n_12"] - 4*x[,"n_13"] - 3*x[,"n_21"] - 2*x[,"n_22"] - 3*x[,"n_23"] - 4*x[,"n_31"] - 3*x[,"n_32"] - 4*x[,"n_33"])*log(2) + 2*(-x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*log(3) + (x[,"n_13"] + x[,"n_31"])*(log(4 + 3*p1 - 2*r_rm2 - 6*p1*r_rm2) + log(4 + 3*p2 - 2*r_rm2 - 6*p2*r_rm2)) + (x[,"n_11"] + x[,"n_33"])*(log(2 - 3*p1 + 2*r_rm2 + 6*p1*r_rm2) + log(2 - 3*p2 + 2*r_rm2 + 6*p2*r_rm2)) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log(8 - 3*p2*(1 - 2*r_rm2)^2 - 3*p1*(1 + 3*p2)*(1 - 2*r_rm2)^2 + 4*r_rm2 - 4*r_rm2^2) + x[,"n_22"]*log(10 + 3*p2*(1 - 2*r_rm2)^2 + 3*p1*(1 + 3*p2)*(1 - 2*r_rm2)^2 - 4*r_rm2 + 4*r_rm2^2)


logL_rm3 <- function(r,n11,n12,n13,n21,n22,n23,n31,n32,n33,p1,p2) {
L <- (-6*n11 - 5*n12 - 6*n13 - 5*n21 - 4*n22 - 5*n23 - 6*n31 - 5*n32 - 6*n33)*log(2) + 2*(-n11 - n12 - n13 - n21 - n22 - n23 - n31 - n32 - n33)*log(3) + (n11 + n33)*(log(4 + 3*p1 + 4*r - 6*p1*r) + log(4 + 3*p2 + 4*r - 6*p2*r)) + (n13 + n31)*(log(8 - 4*r + p1*(-3 + 6*r)) + log(8 - 4*r + p2*(-3 + 6*r))) + n22*log(-6*p2*(1 - 2*r)^2 + 3*p1*(-2 + 3*p2)*(1 - 2*r)^2 + 8*(5 - 2*r + 2*r^2)) + (n12 + n21 + n23 + n32)*log(-3*p1*(-2 + 3*p2)*(1 - 2*r)^2 + 2*(3*p2*(1 - 2*r)^2 + 8*(2 + r - r^2)))
return(L)}
interlogL_rm3 <- function(n11,n12,n13,n21,n22,n23,n31,n32,n33,p1,p2) {
optimize(logL_rm3,c(0,0.5), n11,n12,n13,n21,n22,n23,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rm3 <- parallel::mcmapply(interlogL_rm3,x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rm3 <- -2*(x[,"n_13"] + x[,"n_31"])*(log10(2) + log10(3)) - 2*(x[,"n_11"] + x[,"n_33"])*(log10(2) + log10(3)) - x[,"n_22"]*(2*log10(2) + 2*log10(3)) - (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(2*log10(2) + 2*log10(3)) + (x[,"n_11"] + x[,"n_33"])*(log10(4 + 3*p1 + 4*r_rm3 - 6*p1*r_rm3) + log10(4 + 3*p2 + 4*r_rm3 - 6*p2*r_rm3)) + (x[,"n_13"] + x[,"n_31"])*(log10(8 - 4*r_rm3 + p1*(-3 + 6*r_rm3)) + log10(8 - 4*r_rm3 + p2*(-3 + 6*r_rm3))) + x[,"n_22"]*log10(-6*p2*(1 - 2*r_rm3)^2 + 3*p1*(-2 + 3*p2)*(1 - 2*r_rm3)^2 + 8*(5 - 2*r_rm3 + 2*r_rm3^2)) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log10(-3*p1*(-2 + 3*p2)*(1 - 2*r_rm3)^2 + 2*(3*p2*(1 - 2*r_rm3)^2 + 8*(2 + r_rm3 - r_rm3^2)))


logL_rm3 <- (-6*x[,"n_11"] - 5*x[,"n_12"] - 6*x[,"n_13"] - 5*x[,"n_21"] - 4*x[,"n_22"] - 5*x[,"n_23"] - 6*x[,"n_31"] - 5*x[,"n_32"] - 6*x[,"n_33"])*log(2) + 2*(-x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*log(3) + (x[,"n_11"] + x[,"n_33"])*(log(4 + 3*p1 + 4*r_rm3 - 6*p1*r_rm3) + log(4 + 3*p2 + 4*r_rm3 - 6*p2*r_rm3)) + (x[,"n_13"] + x[,"n_31"])*(log(8 - 4*r_rm3 + p1*(-3 + 6*r_rm3)) + log(8 - 4*r_rm3 + p2*(-3 + 6*r_rm3))) + x[,"n_22"]*log(-6*p2*(1 - 2*r_rm3)^2 + 3*p1*(-2 + 3*p2)*(1 - 2*r_rm3)^2 + 8*(5 - 2*r_rm3 + 2*r_rm3^2)) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log(-3*p1*(-2 + 3*p2)*(1 - 2*r_rm3)^2 + 2*(3*p2*(1 - 2*r_rm3)^2 + 8*(2 + r_rm3 - r_rm3^2)))


logL_rm4 <- function(r,n11,n12,n13,n21,n22,n23,n31,n32,n33,p1,p2) {
L <- (-5*n11 - 4*n12 - 5*n13 - 4*n21 - 3*n22 - 4*n23 - 5*n31 - 4*n32 - 5*n33)*log(2) + 2*(-n11 - n12 - n13 - n21 - n22 - n23 - n31 - n32 - n33)*log(3) + (n11 + n33)*(log(4 + 3*p1 + 4*r - 6*p1*r) + log(2 - 3*p2 + 2*r + 6*p2*r)) + n22*log(20 + 6*p2*(1 - 2*r)^2 - 3*p1*(1 + 3*p2)*(1 - 2*r)^2 - 8*r + 8*r^2) + (n13 + n31)*(log(4 + p2*(3 - 6*r) - 2*r) + log(8 - 4*r + p1*(-3 + 6*r))) + (n12 + n21 + n23 + n32)*log(-6*p2*(1 - 2*r)^2 + 3*p1*(1 + 3*p2)*(1 - 2*r)^2 + 8*(2 + r - r^2))
return(L)}
interlogL_rm4 <- function(n11,n12,n13,n21,n22,n23,n31,n32,n33,p1,p2) {
optimize(logL_rm4,c(0,0.5), n11,n12,n13,n21,n22,n23,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rm4 <- parallel::mcmapply(interlogL_rm4,x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rm4 <- -(x[,"n_22"]*(log10(2) + 2*log10(3))) - (x[,"n_13"] + x[,"n_31"])*(log10(2) + 2*log10(3)) - (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(log10(2) + 2*log10(3)) - (x[,"n_11"] + x[,"n_33"])*(log10(2) + 2*log10(3)) + (x[,"n_11"] + x[,"n_33"])*(log10(4 + 3*p1 + 4*r_rm4 - 6*p1*r_rm4) + log10(2 - 3*p2 + 2*r_rm4 + 6*p2*r_rm4)) + x[,"n_22"]*log10(20 + 6*p2*(1 - 2*r_rm4)^2 - 3*p1*(1 + 3*p2)*(1 - 2*r_rm4)^2 - 8*r_rm4 + 8*r_rm4^2) + (x[,"n_13"] + x[,"n_31"])*(log10(4 + p2*(3 - 6*r_rm4) - 2*r_rm4) + log10(8 - 4*r_rm4 + p1*(-3 + 6*r_rm4))) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log10(-6*p2*(1 - 2*r_rm4)^2 + 3*p1*(1 + 3*p2)*(1 - 2*r_rm4)^2 + 8*(2 + r_rm4 - r_rm4^2))


logL_rm4 <- (-5*x[,"n_11"] - 4*x[,"n_12"] - 5*x[,"n_13"] - 4*x[,"n_21"] - 3*x[,"n_22"] - 4*x[,"n_23"] - 5*x[,"n_31"] - 4*x[,"n_32"] - 5*x[,"n_33"])*log(2) + 2*(-x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*log(3) + (x[,"n_11"] + x[,"n_33"])*(log(4 + 3*p1 + 4*r_rm4 - 6*p1*r_rm4) + log(2 - 3*p2 + 2*r_rm4 + 6*p2*r_rm4)) + x[,"n_22"]*log(20 + 6*p2*(1 - 2*r_rm4)^2 - 3*p1*(1 + 3*p2)*(1 - 2*r_rm4)^2 - 8*r_rm4 + 8*r_rm4^2) + (x[,"n_13"] + x[,"n_31"])*(log(4 + p2*(3 - 6*r_rm4) - 2*r_rm4) + log(8 - 4*r_rm4 + p1*(-3 + 6*r_rm4))) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*log(-6*p2*(1 - 2*r_rm4)^2 + 3*p1*(1 + 3*p2)*(1 - 2*r_rm4)^2 + 8*(2 + r_rm4 - r_rm4^2))


return(list(
r_mat = cbind(r_cc,r_cm1,r_cm2,r_rc1,r_rc2,r_rm1,r_rm2,r_rm3,r_rm4,0.499),
LOD_mat = cbind(LOD_cc,LOD_cm1,LOD_cm2,LOD_rc1,LOD_rc2,LOD_rm1,LOD_rm2,LOD_rm3,LOD_rm4,0),
logL_mat = cbind(logL_cc,logL_cm1,logL_cm2,logL_rc1,logL_rc2,logL_rm1,logL_rm2,logL_rm3,logL_rm4,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling mixed","coupling mixed","coupling repulsion","coupling repulsion","repulsion coupling","repulsion coupling","repulsion mixed","repulsion mixed","repulsion mixed","repulsion mixed","unknown")
)
)
}

#' @rdname p4_functions
p4_1.3_2.1 <- function(x,p1,p2,ncores=1){


logL_cc1 <- function(r,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(5*log(2) + 2*log(3)) + (n13 + n30)*(log(2 - 3*p1) + log(pmax(1e-6,r)) + log(4 + 3*p2 + 4*r - 6*p2*r)) + (n20 + n23)*(log(2 - 3*p1) + log(4 + 3*p2*(1 - 2*r)^2 + 8*r - 8*r^2)) + (n10 + n33)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(8 - 4*r + p2*(-3 + 6*r))) + (n21 + n22)*log(-6*p2*(1 - 2*r)^2 + 3*p1*(4 + 3*p2*(1 - 2*r)^2 + 8*r - 8*r^2) + 16*(4 - r + r^2)) + (n11 + n32)*log((-2 + 3*p1)*(-1 + r)*(4 + 3*p2 + 4*r - 6*p2*r) + (4 + 3*p1)*(8 - 4*r + p2*(-3 + 6*r))) + (n12 + n31)*log((2 - 3*p1)*r*(8 - 4*r + p2*(-3 + 6*r)) - (4 + 3*p1)*(-4*(1 + r) + p2*(-3 + 6*r)))
return(L)}
interlogL_cc1 <- function(n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_cc1,c(0,0.5), n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc1 <- parallel::mcmapply(interlogL_cc1,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_cc1 <- -((x[,"n_13"] + x[,"n_30"])*(log10(3) + log10(2 - 3*p1))) - (x[,"n_10"] + x[,"n_33"])*(log10(3) + log10(2 - 3*p1)) - (x[,"n_20"] + x[,"n_23"])*(log10(2) + log10(3) + log10(2 - 3*p1)) - (x[,"n_21"] + x[,"n_22"])*log10(60 + 18*p1) - (x[,"n_12"] + x[,"n_31"])*log10(3*(2 - 3*p1) + 6*(4 + 3*p1)) - (x[,"n_11"] + x[,"n_32"])*log10(-3*(-2 + 3*p1) + 6*(4 + 3*p1)) + (x[,"n_13"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(pmax(1e-6,r_cc1)) + log10(4 + 3*p2 + 4*r_cc1 - 6*p2*r_cc1)) + (x[,"n_20"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(4 + 3*p2*(1 - 2*r_cc1)^2 + 8*r_cc1 - 8*r_cc1^2)) + (x[,"n_10"] + x[,"n_33"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_cc1)) + log10(8 - 4*r_cc1 + p2*(-3 + 6*r_cc1))) + (x[,"n_21"] + x[,"n_22"])*log10(-6*p2*(1 - 2*r_cc1)^2 + 3*p1*(4 + 3*p2*(1 - 2*r_cc1)^2 + 8*r_cc1 - 8*r_cc1^2) + 16*(4 - r_cc1 + r_cc1^2)) + (x[,"n_11"] + x[,"n_32"])*log10((-2 + 3*p1)*(-1 + r_cc1)*(4 + 3*p2 + 4*r_cc1 - 6*p2*r_cc1) + (4 + 3*p1)*(8 - 4*r_cc1 + p2*(-3 + 6*r_cc1))) + (x[,"n_12"] + x[,"n_31"])*log10((2 - 3*p1)*r_cc1*(8 - 4*r_cc1 + p2*(-3 + 6*r_cc1)) - (4 + 3*p1)*(-4*(1 + r_cc1) + p2*(-3 + 6*r_cc1)))


logL_cc1 <- (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(5*log(2) + 2*log(3)) + (x[,"n_13"] + x[,"n_30"])*(log(2 - 3*p1) + log(pmax(1e-6,r_cc1)) + log(4 + 3*p2 + 4*r_cc1 - 6*p2*r_cc1)) + (x[,"n_20"] + x[,"n_23"])*(log(2 - 3*p1) + log(4 + 3*p2*(1 - 2*r_cc1)^2 + 8*r_cc1 - 8*r_cc1^2)) + (x[,"n_10"] + x[,"n_33"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_cc1)) + log(8 - 4*r_cc1 + p2*(-3 + 6*r_cc1))) + (x[,"n_21"] + x[,"n_22"])*log(-6*p2*(1 - 2*r_cc1)^2 + 3*p1*(4 + 3*p2*(1 - 2*r_cc1)^2 + 8*r_cc1 - 8*r_cc1^2) + 16*(4 - r_cc1 + r_cc1^2)) + (x[,"n_11"] + x[,"n_32"])*log((-2 + 3*p1)*(-1 + r_cc1)*(4 + 3*p2 + 4*r_cc1 - 6*p2*r_cc1) + (4 + 3*p1)*(8 - 4*r_cc1 + p2*(-3 + 6*r_cc1))) + (x[,"n_12"] + x[,"n_31"])*log((2 - 3*p1)*r_cc1*(8 - 4*r_cc1 + p2*(-3 + 6*r_cc1)) - (4 + 3*p1)*(-4*(1 + r_cc1) + p2*(-3 + 6*r_cc1)))


logL_cc2 <- function(r,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(4*log(2) + 2*log(3)) + (n10 + n33)*(log(2 - 3*p1) + log(4 + p2*(3 - 6*r) - 2*r) + log(pmax(1e-6,1 - r))) + (n13 + n30)*(log(2 - 3*p1) + log(pmax(1e-6,r)) + log(2 - 3*p2 + 2*r + 6*p2*r)) + (n20 + n23)*(log(2 - 3*p1) + log(2 - 3*p2*(1 - 2*r)^2 + 4*r - 4*r^2)) + (n12 + n31)*log((-2 + 3*p1)*r*(-4 - 3*p2 + 2*r + 6*p2*r) + (4 + 3*p1)*(2 - 3*p2 + 2*r + 6*p2*r)) + (n11 + n32)*log((-4 - 3*p1)*(-4 - 3*p2 + 2*r + 6*p2*r) + (-2 + 3*p1)*(-1 + r)*(2 - 3*p2 + 2*r + 6*p2*r)) + (n21 + n22)*log(6*p2*(1 - 2*r)^2 + 8*(4 - r + r^2) - 3*p1*(-2 + 3*p2*(1 - 2*r)^2 - 4*r + 4*r^2))
return(L)}
interlogL_cc2 <- function(n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_cc2,c(0,0.5), n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc2 <- parallel::mcmapply(interlogL_cc2,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_cc2 <- -((x[,"n_20"] + x[,"n_23"])*(log10(3) + log10(2 - 3*p1))) - (x[,"n_13"] + x[,"n_30"])*(-log10(2) + log10(3) + log10(2 - 3*p1)) - (x[,"n_10"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(2 - 3*p1)) - (x[,"n_21"] + x[,"n_22"])*log10(30 + 9*p1) - (x[,"n_11"] + x[,"n_32"])*log10(-3*(-4 - 3*p1) - (3*(-2 + 3*p1))/2) - (x[,"n_12"] + x[,"n_31"])*log10((-3*(-2 + 3*p1))/2 + 3*(4 + 3*p1)) + (x[,"n_10"] + x[,"n_33"])*(log10(2 - 3*p1) + log10(4 + p2*(3 - 6*r_cc2) - 2*r_cc2) + log10(pmax(1e-6,1 - r_cc2))) + (x[,"n_13"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(pmax(1e-6,r_cc2)) + log10(2 - 3*p2 + 2*r_cc2 + 6*p2*r_cc2)) + (x[,"n_20"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(2 - 3*p2*(1 - 2*r_cc2)^2 + 4*r_cc2 - 4*r_cc2^2)) + (x[,"n_12"] + x[,"n_31"])*log10((-2 + 3*p1)*r_cc2*(-4 - 3*p2 + 2*r_cc2 + 6*p2*r_cc2) + (4 + 3*p1)*(2 - 3*p2 + 2*r_cc2 + 6*p2*r_cc2)) + (x[,"n_11"] + x[,"n_32"])*log10((-4 - 3*p1)*(-4 - 3*p2 + 2*r_cc2 + 6*p2*r_cc2) + (-2 + 3*p1)*(-1 + r_cc2)*(2 - 3*p2 + 2*r_cc2 + 6*p2*r_cc2)) + (x[,"n_21"] + x[,"n_22"])*log10(6*p2*(1 - 2*r_cc2)^2 + 8*(4 - r_cc2 + r_cc2^2) - 3*p1*(-2 + 3*p2*(1 - 2*r_cc2)^2 - 4*r_cc2 + 4*r_cc2^2))


logL_cc2 <- (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(4*log(2) + 2*log(3)) + (x[,"n_10"] + x[,"n_33"])*(log(2 - 3*p1) + log(4 + p2*(3 - 6*r_cc2) - 2*r_cc2) + log(pmax(1e-6,1 - r_cc2))) + (x[,"n_13"] + x[,"n_30"])*(log(2 - 3*p1) + log(pmax(1e-6,r_cc2)) + log(2 - 3*p2 + 2*r_cc2 + 6*p2*r_cc2)) + (x[,"n_20"] + x[,"n_23"])*(log(2 - 3*p1) + log(2 - 3*p2*(1 - 2*r_cc2)^2 + 4*r_cc2 - 4*r_cc2^2)) + (x[,"n_12"] + x[,"n_31"])*log((-2 + 3*p1)*r_cc2*(-4 - 3*p2 + 2*r_cc2 + 6*p2*r_cc2) + (4 + 3*p1)*(2 - 3*p2 + 2*r_cc2 + 6*p2*r_cc2)) + (x[,"n_11"] + x[,"n_32"])*log((-4 - 3*p1)*(-4 - 3*p2 + 2*r_cc2 + 6*p2*r_cc2) + (-2 + 3*p1)*(-1 + r_cc2)*(2 - 3*p2 + 2*r_cc2 + 6*p2*r_cc2)) + (x[,"n_21"] + x[,"n_22"])*log(6*p2*(1 - 2*r_cc2)^2 + 8*(4 - r_cc2 + r_cc2^2) - 3*p1*(-2 + 3*p2*(1 - 2*r_cc2)^2 - 4*r_cc2 + 4*r_cc2^2))


logL_cc3 <- function(r,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(6*log(2) + 2*log(3)) + (n13 + n30)*(log(4 + 3*p1) + log(pmax(1e-6,r)) + log(4 + 3*p2 + 4*r - 6*p2*r)) + (n20 + n23)*(log(4 + 3*p1) + log(4 + 3*p2*(1 - 2*r)^2 + 8*r - 8*r^2)) + (n10 + n33)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(8 - 4*r + p2*(-3 + 6*r))) + (n12 + n31)*log((4 + 3*p1)*r*(8 - 4*r + p2*(-3 + 6*r)) + (-8 + 3*p1)*(-4*(1 + r) + p2*(-3 + 6*r))) + (n11 + n32)*log((8 - 3*p1)*(8 - 4*r + p2*(-3 + 6*r)) + (4 + 3*p1)*(-1 + r)*(-4*(1 + r) + p2*(-3 + 6*r))) + (n21 + n22)*log(-3*p1*(4 + 3*p2*(1 - 2*r)^2 + 8*r - 8*r^2) - 4*(3*p2*(1 - 2*r)^2 - 8*(4 - r + r^2)))
return(L)}
interlogL_cc3 <- function(n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_cc3,c(0,0.5), n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc3 <- parallel::mcmapply(interlogL_cc3,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_cc3 <- -((x[,"n_21"] + x[,"n_22"])*log10(120 - 18*p1)) - (x[,"n_13"] + x[,"n_30"])*(log10(3) + log10(4 + 3*p1)) - (x[,"n_10"] + x[,"n_33"])*(log10(3) + log10(4 + 3*p1)) - (x[,"n_20"] + x[,"n_23"])*(log10(2) + log10(3) + log10(4 + 3*p1)) - (x[,"n_11"] + x[,"n_32"])*log10(6*(8 - 3*p1) + 3*(4 + 3*p1)) - (x[,"n_12"] + x[,"n_31"])*log10(-6*(-8 + 3*p1) + 3*(4 + 3*p1)) + (x[,"n_13"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(pmax(1e-6,r_cc3)) + log10(4 + 3*p2 + 4*r_cc3 - 6*p2*r_cc3)) + (x[,"n_20"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(4 + 3*p2*(1 - 2*r_cc3)^2 + 8*r_cc3 - 8*r_cc3^2)) + (x[,"n_10"] + x[,"n_33"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_cc3)) + log10(8 - 4*r_cc3 + p2*(-3 + 6*r_cc3))) + (x[,"n_12"] + x[,"n_31"])*log10((4 + 3*p1)*r_cc3*(8 - 4*r_cc3 + p2*(-3 + 6*r_cc3)) + (-8 + 3*p1)*(-4*(1 + r_cc3) + p2*(-3 + 6*r_cc3))) + (x[,"n_11"] + x[,"n_32"])*log10((8 - 3*p1)*(8 - 4*r_cc3 + p2*(-3 + 6*r_cc3)) + (4 + 3*p1)*(-1 + r_cc3)*(-4*(1 + r_cc3) + p2*(-3 + 6*r_cc3))) + (x[,"n_21"] + x[,"n_22"])*log10(-3*p1*(4 + 3*p2*(1 - 2*r_cc3)^2 + 8*r_cc3 - 8*r_cc3^2) - 4*(3*p2*(1 - 2*r_cc3)^2 - 8*(4 - r_cc3 + r_cc3^2)))


logL_cc3 <- (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(6*log(2) + 2*log(3)) + (x[,"n_13"] + x[,"n_30"])*(log(4 + 3*p1) + log(pmax(1e-6,r_cc3)) + log(4 + 3*p2 + 4*r_cc3 - 6*p2*r_cc3)) + (x[,"n_20"] + x[,"n_23"])*(log(4 + 3*p1) + log(4 + 3*p2*(1 - 2*r_cc3)^2 + 8*r_cc3 - 8*r_cc3^2)) + (x[,"n_10"] + x[,"n_33"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_cc3)) + log(8 - 4*r_cc3 + p2*(-3 + 6*r_cc3))) + (x[,"n_12"] + x[,"n_31"])*log((4 + 3*p1)*r_cc3*(8 - 4*r_cc3 + p2*(-3 + 6*r_cc3)) + (-8 + 3*p1)*(-4*(1 + r_cc3) + p2*(-3 + 6*r_cc3))) + (x[,"n_11"] + x[,"n_32"])*log((8 - 3*p1)*(8 - 4*r_cc3 + p2*(-3 + 6*r_cc3)) + (4 + 3*p1)*(-1 + r_cc3)*(-4*(1 + r_cc3) + p2*(-3 + 6*r_cc3))) + (x[,"n_21"] + x[,"n_22"])*log(-3*p1*(4 + 3*p2*(1 - 2*r_cc3)^2 + 8*r_cc3 - 8*r_cc3^2) - 4*(3*p2*(1 - 2*r_cc3)^2 - 8*(4 - r_cc3 + r_cc3^2)))


logL_cc4 <- function(r,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(5*log(2) + 2*log(3)) + (n10 + n33)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(4 + 3*p2 - 2*r - 6*p2*r)) + (n13 + n30)*(log(4 + 3*p1) + log(pmax(1e-6,r)) + log(2 - 3*p2 + 2*r + 6*p2*r)) + (n20 + n23)*(log(4 + 3*p1) + log(2 - 3*p2*(1 - 2*r)^2 + 4*r - 4*r^2)) + (n12 + n31)*log((4 + 3*p1)*(4 + p2*(3 - 6*r) - 2*r)*r - (-8 + 3*p1)*(2 - 3*p2 + 2*r + 6*p2*r)) + (n11 + n32)*log((-8 + 3*p1)*(-4 - 3*p2 + 2*r + 6*p2*r) - (4 + 3*p1)*(-1 + r)*(2 - 3*p2 + 2*r + 6*p2*r)) + (n21 + n22)*log(3*p1*(-2 + 3*p2*(1 - 2*r)^2 - 4*r + 4*r^2) + 4*(3*p2*(1 - 2*r)^2 + 4*(4 - r + r^2)))
return(L)}
interlogL_cc4 <- function(n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_cc4,c(0,0.5), n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc4 <- parallel::mcmapply(interlogL_cc4,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_cc4 <- -((x[,"n_21"] + x[,"n_22"])*log10(60 - 9*p1)) - (x[,"n_20"] + x[,"n_23"])*(log10(3) + log10(4 + 3*p1)) - (x[,"n_13"] + x[,"n_30"])*(-log10(2) + log10(3) + log10(4 + 3*p1)) - (x[,"n_10"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(4 + 3*p1)) - (x[,"n_12"] + x[,"n_31"])*log10(-3*(-8 + 3*p1) + (3*(4 + 3*p1))/2) - (x[,"n_11"] + x[,"n_32"])*log10(-3*(-8 + 3*p1) + (3*(4 + 3*p1))/2) + (x[,"n_10"] + x[,"n_33"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_cc4)) + log10(4 + 3*p2 - 2*r_cc4 - 6*p2*r_cc4)) + (x[,"n_13"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(pmax(1e-6,r_cc4)) + log10(2 - 3*p2 + 2*r_cc4 + 6*p2*r_cc4)) + (x[,"n_20"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(2 - 3*p2*(1 - 2*r_cc4)^2 + 4*r_cc4 - 4*r_cc4^2)) + (x[,"n_12"] + x[,"n_31"])*log10((4 + 3*p1)*(4 + p2*(3 - 6*r_cc4) - 2*r_cc4)*r_cc4 - (-8 + 3*p1)*(2 - 3*p2 + 2*r_cc4 + 6*p2*r_cc4)) + (x[,"n_11"] + x[,"n_32"])*log10((-8 + 3*p1)*(-4 - 3*p2 + 2*r_cc4 + 6*p2*r_cc4) - (4 + 3*p1)*(-1 + r_cc4)*(2 - 3*p2 + 2*r_cc4 + 6*p2*r_cc4)) + (x[,"n_21"] + x[,"n_22"])*log10(3*p1*(-2 + 3*p2*(1 - 2*r_cc4)^2 - 4*r_cc4 + 4*r_cc4^2) + 4*(3*p2*(1 - 2*r_cc4)^2 + 4*(4 - r_cc4 + r_cc4^2)))


logL_cc4 <- (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(5*log(2) + 2*log(3)) + (x[,"n_10"] + x[,"n_33"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_cc4)) + log(4 + 3*p2 - 2*r_cc4 - 6*p2*r_cc4)) + (x[,"n_13"] + x[,"n_30"])*(log(4 + 3*p1) + log(pmax(1e-6,r_cc4)) + log(2 - 3*p2 + 2*r_cc4 + 6*p2*r_cc4)) + (x[,"n_20"] + x[,"n_23"])*(log(4 + 3*p1) + log(2 - 3*p2*(1 - 2*r_cc4)^2 + 4*r_cc4 - 4*r_cc4^2)) + (x[,"n_12"] + x[,"n_31"])*log((4 + 3*p1)*(4 + p2*(3 - 6*r_cc4) - 2*r_cc4)*r_cc4 - (-8 + 3*p1)*(2 - 3*p2 + 2*r_cc4 + 6*p2*r_cc4)) + (x[,"n_11"] + x[,"n_32"])*log((-8 + 3*p1)*(-4 - 3*p2 + 2*r_cc4 + 6*p2*r_cc4) - (4 + 3*p1)*(-1 + r_cc4)*(2 - 3*p2 + 2*r_cc4 + 6*p2*r_cc4)) + (x[,"n_21"] + x[,"n_22"])*log(3*p1*(-2 + 3*p2*(1 - 2*r_cc4)^2 - 4*r_cc4 + 4*r_cc4^2) + 4*(3*p2*(1 - 2*r_cc4)^2 + 4*(4 - r_cc4 + r_cc4^2)))


logL_cr1 <- function(r,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(3*log(2) + log(3)) + (n10 + n13 + n30 + n33)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n20 + n23)*(log(2 - 3*p1) + log(1 - 2*r + 2*r^2)) + (n11 + n32)*log(2*(1 + r^2) - 3*p1*(1 - 3*r + r^2)) + (n12 + n31)*log(2*(2 - 2*r + r^2) - 3*p1*(-1 + r + r^2)) + (n21 + n22)*log(4 + 4*r - 4*r^2 + p1*(3 - 6*r + 6*r^2))
return(L)}
interlogL_cr1 <- function(n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_cr1,c(0,0.5), n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr1 <- parallel::mcmapply(interlogL_cr1,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_cr1 <- -((x[,"n_10"] + x[,"n_13"] + x[,"n_30"] + x[,"n_33"])*(-2*log10(2) + log10(2 - 3*p1))) - (x[,"n_20"] + x[,"n_23"])*(-log10(2) + log10(2 - 3*p1)) - (x[,"n_12"] + x[,"n_31"])*log10(5/2 + (3*p1)/4) - (x[,"n_11"] + x[,"n_32"])*log10(5/2 + (3*p1)/4) - (x[,"n_21"] + x[,"n_22"])*log10(5 + (3*p1)/2) + (x[,"n_10"] + x[,"n_13"] + x[,"n_30"] + x[,"n_33"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_cr1)) + log10(pmax(1e-6,r_cr1))) + (x[,"n_20"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(1 - 2*r_cr1 + 2*r_cr1^2)) + (x[,"n_11"] + x[,"n_32"])*log10(2*(1 + r_cr1^2) - 3*p1*(1 - 3*r_cr1 + r_cr1^2)) + (x[,"n_12"] + x[,"n_31"])*log10(2*(2 - 2*r_cr1 + r_cr1^2) - 3*p1*(-1 + r_cr1 + r_cr1^2)) + (x[,"n_21"] + x[,"n_22"])*log10(4 + 4*r_cr1 - 4*r_cr1^2 + p1*(3 - 6*r_cr1 + 6*r_cr1^2))


logL_cr1 <- (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(3*log(2) + log(3)) + (x[,"n_10"] + x[,"n_13"] + x[,"n_30"] + x[,"n_33"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_cr1)) + log(pmax(1e-6,r_cr1))) + (x[,"n_20"] + x[,"n_23"])*(log(2 - 3*p1) + log(1 - 2*r_cr1 + 2*r_cr1^2)) + (x[,"n_11"] + x[,"n_32"])*log(2*(1 + r_cr1^2) - 3*p1*(1 - 3*r_cr1 + r_cr1^2)) + (x[,"n_12"] + x[,"n_31"])*log(2*(2 - 2*r_cr1 + r_cr1^2) - 3*p1*(-1 + r_cr1 + r_cr1^2)) + (x[,"n_21"] + x[,"n_22"])*log(4 + 4*r_cr1 - 4*r_cr1^2 + p1*(3 - 6*r_cr1 + 6*r_cr1^2))


logL_cr2 <- function(r,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(4*log(2) + log(3)) + (n10 + n13 + n30 + n33)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n20 + n23)*(log(4 + 3*p1) + log(1 - 2*r + 2*r^2)) + (n21 + n22)*log(8 + 8*r - 8*r^2 + p1*(-3 + 6*r - 6*r^2)) + (n11 + n32)*log(4*(1 + r^2) + 3*p1*(1 - 3*r + r^2)) + (n12 + n31)*log(4*(2 - 2*r + r^2) + 3*p1*(-1 + r + r^2))
return(L)}
interlogL_cr2 <- function(n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_cr2,c(0,0.5), n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr2 <- parallel::mcmapply(interlogL_cr2,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_cr2 <- -((x[,"n_21"] + x[,"n_22"])*log10(10 - (3*p1)/2)) - (x[,"n_12"] + x[,"n_31"])*log10(5 - (3*p1)/4) - (x[,"n_11"] + x[,"n_32"])*log10(5 - (3*p1)/4) - (x[,"n_10"] + x[,"n_13"] + x[,"n_30"] + x[,"n_33"])*(-2*log10(2) + log10(4 + 3*p1)) - (x[,"n_20"] + x[,"n_23"])*(-log10(2) + log10(4 + 3*p1)) + (x[,"n_10"] + x[,"n_13"] + x[,"n_30"] + x[,"n_33"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_cr2)) + log10(pmax(1e-6,r_cr2))) + (x[,"n_20"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(1 - 2*r_cr2 + 2*r_cr2^2)) + (x[,"n_21"] + x[,"n_22"])*log10(8 + 8*r_cr2 - 8*r_cr2^2 + p1*(-3 + 6*r_cr2 - 6*r_cr2^2)) + (x[,"n_11"] + x[,"n_32"])*log10(4*(1 + r_cr2^2) + 3*p1*(1 - 3*r_cr2 + r_cr2^2)) + (x[,"n_12"] + x[,"n_31"])*log10(4*(2 - 2*r_cr2 + r_cr2^2) + 3*p1*(-1 + r_cr2 + r_cr2^2))


logL_cr2 <- (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(4*log(2) + log(3)) + (x[,"n_10"] + x[,"n_13"] + x[,"n_30"] + x[,"n_33"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_cr2)) + log(pmax(1e-6,r_cr2))) + (x[,"n_20"] + x[,"n_23"])*(log(4 + 3*p1) + log(1 - 2*r_cr2 + 2*r_cr2^2)) + (x[,"n_21"] + x[,"n_22"])*log(8 + 8*r_cr2 - 8*r_cr2^2 + p1*(-3 + 6*r_cr2 - 6*r_cr2^2)) + (x[,"n_11"] + x[,"n_32"])*log(4*(1 + r_cr2^2) + 3*p1*(1 - 3*r_cr2 + r_cr2^2)) + (x[,"n_12"] + x[,"n_31"])*log(4*(2 - 2*r_cr2 + r_cr2^2) + 3*p1*(-1 + r_cr2 + r_cr2^2))


logL_rc1 <- function(r,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(6*log(2) + 2*log(3)) + (n13 + n30)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(4 + 3*p2 + 4*r - 6*p2*r)) + (n10 + n33)*(log(4 + 3*p1) + log(pmax(1e-6,r)) + log(8 - 4*r + p2*(-3 + 6*r))) + (n20 + n23)*(log(4 + 3*p1) + log(-3*p2*(1 - 2*r)^2 + 8*(1 - r + r^2))) + (n11 + n32)*log((4 + 3*p1)*r*(4 + 3*p2 + 4*r - 6*p2*r) - (-8 + 3*p1)*(8 - 4*r + p2*(-3 + 6*r))) + (n12 + n31)*log((-4 - 3*p1)*(-1 + r)*(8 - 4*r + p2*(-3 + 6*r)) + (-8 + 3*p1)*(-4*(1 + r) + p2*(-3 + 6*r))) + (n21 + n22)*log(4*(28 + 3*p2*(1 - 2*r)^2 + 8*r - 8*r^2) + 3*p1*(3*p2*(1 - 2*r)^2 - 8*(1 - r + r^2)))
return(L)}
interlogL_rc1 <- function(n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rc1,c(0,0.5), n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc1 <- parallel::mcmapply(interlogL_rc1,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rc1 <- -((x[,"n_21"] + x[,"n_22"])*log10(120 - 18*p1)) - (x[,"n_13"] + x[,"n_30"])*(log10(3) + log10(4 + 3*p1)) - (x[,"n_10"] + x[,"n_33"])*(log10(3) + log10(4 + 3*p1)) - (x[,"n_20"] + x[,"n_23"])*(log10(2) + log10(3) + log10(4 + 3*p1)) - (x[,"n_12"] + x[,"n_31"])*log10(-3*(-4 - 3*p1) - 6*(-8 + 3*p1)) - (x[,"n_11"] + x[,"n_32"])*log10(-6*(-8 + 3*p1) + 3*(4 + 3*p1)) + (x[,"n_13"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_rc1)) + log10(4 + 3*p2 + 4*r_rc1 - 6*p2*r_rc1)) + (x[,"n_10"] + x[,"n_33"])*(log10(4 + 3*p1) + log10(pmax(1e-6,r_rc1)) + log10(8 - 4*r_rc1 + p2*(-3 + 6*r_rc1))) + (x[,"n_20"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(-3*p2*(1 - 2*r_rc1)^2 + 8*(1 - r_rc1 + r_rc1^2))) + (x[,"n_11"] + x[,"n_32"])*log10((4 + 3*p1)*r_rc1*(4 + 3*p2 + 4*r_rc1 - 6*p2*r_rc1) - (-8 + 3*p1)*(8 - 4*r_rc1 + p2*(-3 + 6*r_rc1))) + (x[,"n_12"] + x[,"n_31"])*log10((-4 - 3*p1)*(-1 + r_rc1)*(8 - 4*r_rc1 + p2*(-3 + 6*r_rc1)) + (-8 + 3*p1)*(-4*(1 + r_rc1) + p2*(-3 + 6*r_rc1))) + (x[,"n_21"] + x[,"n_22"])*log10(4*(28 + 3*p2*(1 - 2*r_rc1)^2 + 8*r_rc1 - 8*r_rc1^2) + 3*p1*(3*p2*(1 - 2*r_rc1)^2 - 8*(1 - r_rc1 + r_rc1^2)))


logL_rc1 <- (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(6*log(2) + 2*log(3)) + (x[,"n_13"] + x[,"n_30"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_rc1)) + log(4 + 3*p2 + 4*r_rc1 - 6*p2*r_rc1)) + (x[,"n_10"] + x[,"n_33"])*(log(4 + 3*p1) + log(pmax(1e-6,r_rc1)) + log(8 - 4*r_rc1 + p2*(-3 + 6*r_rc1))) + (x[,"n_20"] + x[,"n_23"])*(log(4 + 3*p1) + log(-3*p2*(1 - 2*r_rc1)^2 + 8*(1 - r_rc1 + r_rc1^2))) + (x[,"n_11"] + x[,"n_32"])*log((4 + 3*p1)*r_rc1*(4 + 3*p2 + 4*r_rc1 - 6*p2*r_rc1) - (-8 + 3*p1)*(8 - 4*r_rc1 + p2*(-3 + 6*r_rc1))) + (x[,"n_12"] + x[,"n_31"])*log((-4 - 3*p1)*(-1 + r_rc1)*(8 - 4*r_rc1 + p2*(-3 + 6*r_rc1)) + (-8 + 3*p1)*(-4*(1 + r_rc1) + p2*(-3 + 6*r_rc1))) + (x[,"n_21"] + x[,"n_22"])*log(4*(28 + 3*p2*(1 - 2*r_rc1)^2 + 8*r_rc1 - 8*r_rc1^2) + 3*p1*(3*p2*(1 - 2*r_rc1)^2 - 8*(1 - r_rc1 + r_rc1^2)))


logL_rc2 <- function(r,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(5*log(2) + 2*log(3)) + (n10 + n33)*(log(4 + 3*p1) + log(4 + p2*(3 - 6*r) - 2*r) + log(pmax(1e-6,r))) + (n13 + n30)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(2 - 3*p2 + 2*r + 6*p2*r)) + (n12 + n31)*log((4 + 3*p1)*(-1 + r)*(-4 - 3*p2 + 2*r + 6*p2*r) - (-8 + 3*p1)*(2 - 3*p2 + 2*r + 6*p2*r)) + (n11 + n32)*log((-8 + 3*p1)*(-4 - 3*p2 + 2*r + 6*p2*r) + (4 + 3*p1)*r*(2 - 3*p2 + 2*r + 6*p2*r)) + (n20 + n23)*(log(4 + 3*p1) + log(3*p2*(1 - 2*r)^2 + 4*(1 - r + r^2))) + (n21 + n22)*log(4*(14 - 3*p2*(1 - 2*r)^2 + 4*r - 4*r^2) - 3*p1*(3*p2*(1 - 2*r)^2 + 4*(1 - r + r^2)))
return(L)}
interlogL_rc2 <- function(n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rc2,c(0,0.5), n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc2 <- parallel::mcmapply(interlogL_rc2,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rc2 <- -((x[,"n_21"] + x[,"n_22"])*log10(60 - 9*p1)) - (x[,"n_20"] + x[,"n_23"])*(log10(3) + log10(4 + 3*p1)) - (x[,"n_13"] + x[,"n_30"])*(-log10(2) + log10(3) + log10(4 + 3*p1)) - (x[,"n_10"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(4 + 3*p1)) - (x[,"n_12"] + x[,"n_31"])*log10(-3*(-8 + 3*p1) + (3*(4 + 3*p1))/2) - (x[,"n_11"] + x[,"n_32"])*log10(-3*(-8 + 3*p1) + (3*(4 + 3*p1))/2) + (x[,"n_10"] + x[,"n_33"])*(log10(4 + 3*p1) + log10(4 + p2*(3 - 6*r_rc2) - 2*r_rc2) + log10(pmax(1e-6,r_rc2))) + (x[,"n_13"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_rc2)) + log10(2 - 3*p2 + 2*r_rc2 + 6*p2*r_rc2)) + (x[,"n_12"] + x[,"n_31"])*log10((4 + 3*p1)*(-1 + r_rc2)*(-4 - 3*p2 + 2*r_rc2 + 6*p2*r_rc2) - (-8 + 3*p1)*(2 - 3*p2 + 2*r_rc2 + 6*p2*r_rc2)) + (x[,"n_11"] + x[,"n_32"])*log10((-8 + 3*p1)*(-4 - 3*p2 + 2*r_rc2 + 6*p2*r_rc2) + (4 + 3*p1)*r_rc2*(2 - 3*p2 + 2*r_rc2 + 6*p2*r_rc2)) + (x[,"n_20"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(3*p2*(1 - 2*r_rc2)^2 + 4*(1 - r_rc2 + r_rc2^2))) + (x[,"n_21"] + x[,"n_22"])*log10(4*(14 - 3*p2*(1 - 2*r_rc2)^2 + 4*r_rc2 - 4*r_rc2^2) - 3*p1*(3*p2*(1 - 2*r_rc2)^2 + 4*(1 - r_rc2 + r_rc2^2)))


logL_rc2 <- (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(5*log(2) + 2*log(3)) + (x[,"n_10"] + x[,"n_33"])*(log(4 + 3*p1) + log(4 + p2*(3 - 6*r_rc2) - 2*r_rc2) + log(pmax(1e-6,r_rc2))) + (x[,"n_13"] + x[,"n_30"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_rc2)) + log(2 - 3*p2 + 2*r_rc2 + 6*p2*r_rc2)) + (x[,"n_12"] + x[,"n_31"])*log((4 + 3*p1)*(-1 + r_rc2)*(-4 - 3*p2 + 2*r_rc2 + 6*p2*r_rc2) - (-8 + 3*p1)*(2 - 3*p2 + 2*r_rc2 + 6*p2*r_rc2)) + (x[,"n_11"] + x[,"n_32"])*log((-8 + 3*p1)*(-4 - 3*p2 + 2*r_rc2 + 6*p2*r_rc2) + (4 + 3*p1)*r_rc2*(2 - 3*p2 + 2*r_rc2 + 6*p2*r_rc2)) + (x[,"n_20"] + x[,"n_23"])*(log(4 + 3*p1) + log(3*p2*(1 - 2*r_rc2)^2 + 4*(1 - r_rc2 + r_rc2^2))) + (x[,"n_21"] + x[,"n_22"])*log(4*(14 - 3*p2*(1 - 2*r_rc2)^2 + 4*r_rc2 - 4*r_rc2^2) - 3*p1*(3*p2*(1 - 2*r_rc2)^2 + 4*(1 - r_rc2 + r_rc2^2)))


logL_rc3 <- function(r,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(6*log(2) + 2*log(3)) + (n13 + n30)*(log(4 + 3*p1) + log(pmax(1e-6,r)) + log(4 + 3*p2 + 4*r - 6*p2*r)) + (n20 + n23)*(log(4 + 3*p1) + log(4 + 3*p2*(1 - 2*r)^2 + 8*r - 8*r^2)) + (n10 + n33)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(8 - 4*r + p2*(-3 + 6*r))) + (n12 + n31)*log((4 + 3*p1)*r*(8 - 4*r + p2*(-3 + 6*r)) + (-8 + 3*p1)*(-4*(1 + r) + p2*(-3 + 6*r))) + (n11 + n32)*log((8 - 3*p1)*(8 - 4*r + p2*(-3 + 6*r)) + (4 + 3*p1)*(-1 + r)*(-4*(1 + r) + p2*(-3 + 6*r))) + (n21 + n22)*log(-3*p1*(4 + 3*p2*(1 - 2*r)^2 + 8*r - 8*r^2) - 4*(3*p2*(1 - 2*r)^2 - 8*(4 - r + r^2)))
return(L)}
interlogL_rc3 <- function(n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rc3,c(0,0.5), n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc3 <- parallel::mcmapply(interlogL_rc3,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rc3 <- -((x[,"n_21"] + x[,"n_22"])*log10(120 - 18*p1)) - (x[,"n_13"] + x[,"n_30"])*(log10(3) + log10(4 + 3*p1)) - (x[,"n_10"] + x[,"n_33"])*(log10(3) + log10(4 + 3*p1)) - (x[,"n_20"] + x[,"n_23"])*(log10(2) + log10(3) + log10(4 + 3*p1)) - (x[,"n_11"] + x[,"n_32"])*log10(6*(8 - 3*p1) + 3*(4 + 3*p1)) - (x[,"n_12"] + x[,"n_31"])*log10(-6*(-8 + 3*p1) + 3*(4 + 3*p1)) + (x[,"n_13"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(pmax(1e-6,r_rc3)) + log10(4 + 3*p2 + 4*r_rc3 - 6*p2*r_rc3)) + (x[,"n_20"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(4 + 3*p2*(1 - 2*r_rc3)^2 + 8*r_rc3 - 8*r_rc3^2)) + (x[,"n_10"] + x[,"n_33"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_rc3)) + log10(8 - 4*r_rc3 + p2*(-3 + 6*r_rc3))) + (x[,"n_12"] + x[,"n_31"])*log10((4 + 3*p1)*r_rc3*(8 - 4*r_rc3 + p2*(-3 + 6*r_rc3)) + (-8 + 3*p1)*(-4*(1 + r_rc3) + p2*(-3 + 6*r_rc3))) + (x[,"n_11"] + x[,"n_32"])*log10((8 - 3*p1)*(8 - 4*r_rc3 + p2*(-3 + 6*r_rc3)) + (4 + 3*p1)*(-1 + r_rc3)*(-4*(1 + r_rc3) + p2*(-3 + 6*r_rc3))) + (x[,"n_21"] + x[,"n_22"])*log10(-3*p1*(4 + 3*p2*(1 - 2*r_rc3)^2 + 8*r_rc3 - 8*r_rc3^2) - 4*(3*p2*(1 - 2*r_rc3)^2 - 8*(4 - r_rc3 + r_rc3^2)))


logL_rc3 <- (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(6*log(2) + 2*log(3)) + (x[,"n_13"] + x[,"n_30"])*(log(4 + 3*p1) + log(pmax(1e-6,r_rc3)) + log(4 + 3*p2 + 4*r_rc3 - 6*p2*r_rc3)) + (x[,"n_20"] + x[,"n_23"])*(log(4 + 3*p1) + log(4 + 3*p2*(1 - 2*r_rc3)^2 + 8*r_rc3 - 8*r_rc3^2)) + (x[,"n_10"] + x[,"n_33"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_rc3)) + log(8 - 4*r_rc3 + p2*(-3 + 6*r_rc3))) + (x[,"n_12"] + x[,"n_31"])*log((4 + 3*p1)*r_rc3*(8 - 4*r_rc3 + p2*(-3 + 6*r_rc3)) + (-8 + 3*p1)*(-4*(1 + r_rc3) + p2*(-3 + 6*r_rc3))) + (x[,"n_11"] + x[,"n_32"])*log((8 - 3*p1)*(8 - 4*r_rc3 + p2*(-3 + 6*r_rc3)) + (4 + 3*p1)*(-1 + r_rc3)*(-4*(1 + r_rc3) + p2*(-3 + 6*r_rc3))) + (x[,"n_21"] + x[,"n_22"])*log(-3*p1*(4 + 3*p2*(1 - 2*r_rc3)^2 + 8*r_rc3 - 8*r_rc3^2) - 4*(3*p2*(1 - 2*r_rc3)^2 - 8*(4 - r_rc3 + r_rc3^2)))


logL_rc4 <- function(r,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*(5*log(2) + 2*log(3)) + (n10 + n33)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(4 + 3*p2 - 2*r - 6*p2*r)) + (n13 + n30)*(log(4 + 3*p1) + log(pmax(1e-6,r)) + log(2 - 3*p2 + 2*r + 6*p2*r)) + (n20 + n23)*(log(4 + 3*p1) + log(2 - 3*p2*(1 - 2*r)^2 + 4*r - 4*r^2)) + (n12 + n31)*log((4 + 3*p1)*(4 + p2*(3 - 6*r) - 2*r)*r - (-8 + 3*p1)*(2 - 3*p2 + 2*r + 6*p2*r)) + (n11 + n32)*log((-8 + 3*p1)*(-4 - 3*p2 + 2*r + 6*p2*r) - (4 + 3*p1)*(-1 + r)*(2 - 3*p2 + 2*r + 6*p2*r)) + (n21 + n22)*log(3*p1*(-2 + 3*p2*(1 - 2*r)^2 - 4*r + 4*r^2) + 4*(3*p2*(1 - 2*r)^2 + 4*(4 - r + r^2)))
return(L)}
interlogL_rc4 <- function(n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rc4,c(0,0.5), n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc4 <- parallel::mcmapply(interlogL_rc4,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rc4 <- -((x[,"n_21"] + x[,"n_22"])*log10(60 - 9*p1)) - (x[,"n_20"] + x[,"n_23"])*(log10(3) + log10(4 + 3*p1)) - (x[,"n_13"] + x[,"n_30"])*(-log10(2) + log10(3) + log10(4 + 3*p1)) - (x[,"n_10"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(4 + 3*p1)) - (x[,"n_12"] + x[,"n_31"])*log10(-3*(-8 + 3*p1) + (3*(4 + 3*p1))/2) - (x[,"n_11"] + x[,"n_32"])*log10(-3*(-8 + 3*p1) + (3*(4 + 3*p1))/2) + (x[,"n_10"] + x[,"n_33"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_rc4)) + log10(4 + 3*p2 - 2*r_rc4 - 6*p2*r_rc4)) + (x[,"n_13"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(pmax(1e-6,r_rc4)) + log10(2 - 3*p2 + 2*r_rc4 + 6*p2*r_rc4)) + (x[,"n_20"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(2 - 3*p2*(1 - 2*r_rc4)^2 + 4*r_rc4 - 4*r_rc4^2)) + (x[,"n_12"] + x[,"n_31"])*log10((4 + 3*p1)*(4 + p2*(3 - 6*r_rc4) - 2*r_rc4)*r_rc4 - (-8 + 3*p1)*(2 - 3*p2 + 2*r_rc4 + 6*p2*r_rc4)) + (x[,"n_11"] + x[,"n_32"])*log10((-8 + 3*p1)*(-4 - 3*p2 + 2*r_rc4 + 6*p2*r_rc4) - (4 + 3*p1)*(-1 + r_rc4)*(2 - 3*p2 + 2*r_rc4 + 6*p2*r_rc4)) + (x[,"n_21"] + x[,"n_22"])*log10(3*p1*(-2 + 3*p2*(1 - 2*r_rc4)^2 - 4*r_rc4 + 4*r_rc4^2) + 4*(3*p2*(1 - 2*r_rc4)^2 + 4*(4 - r_rc4 + r_rc4^2)))


logL_rc4 <- (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*(5*log(2) + 2*log(3)) + (x[,"n_10"] + x[,"n_33"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_rc4)) + log(4 + 3*p2 - 2*r_rc4 - 6*p2*r_rc4)) + (x[,"n_13"] + x[,"n_30"])*(log(4 + 3*p1) + log(pmax(1e-6,r_rc4)) + log(2 - 3*p2 + 2*r_rc4 + 6*p2*r_rc4)) + (x[,"n_20"] + x[,"n_23"])*(log(4 + 3*p1) + log(2 - 3*p2*(1 - 2*r_rc4)^2 + 4*r_rc4 - 4*r_rc4^2)) + (x[,"n_12"] + x[,"n_31"])*log((4 + 3*p1)*(4 + p2*(3 - 6*r_rc4) - 2*r_rc4)*r_rc4 - (-8 + 3*p1)*(2 - 3*p2 + 2*r_rc4 + 6*p2*r_rc4)) + (x[,"n_11"] + x[,"n_32"])*log((-8 + 3*p1)*(-4 - 3*p2 + 2*r_rc4 + 6*p2*r_rc4) - (4 + 3*p1)*(-1 + r_rc4)*(2 - 3*p2 + 2*r_rc4 + 6*p2*r_rc4)) + (x[,"n_21"] + x[,"n_22"])*log(3*p1*(-2 + 3*p2*(1 - 2*r_rc4)^2 - 4*r_rc4 + 4*r_rc4^2) + 4*(3*p2*(1 - 2*r_rc4)^2 + 4*(4 - r_rc4 + r_rc4^2)))


logL_rr1 <- function(r,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-4*n10 - 4*n11 - 4*n12 - 4*n13 - 3*n20 - 3*n21 - 3*n22 - 3*n23 - 4*n30 - 4*n31 - 4*n32 - 4*n33)*log(2) + (-n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*log(3) + (n20 + n23)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n33)*(log(4 + 3*p1) + 2*log(pmax(1e-6,r))) + (n11 + n32)*(log(pmax(1e-6,r)) + log(12 - (4 + 3*p1)*r)) + (n13 + n30)*(log(4 + 3*p1) + log(1 - 2*r + r^2)) + (n21 + n22)*log(6 - (4 + 3*p1)*r + (4 + 3*p1)*r^2) + (n12 + n31)*(log(pmax(1e-6,1 - r)) + log(3*p1*(-1 + r) + 4*(2 + r)))
return(L)}
interlogL_rr1 <- function(n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rr1,c(0,0.5), n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr1 <- parallel::mcmapply(interlogL_rr1,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rr1 <- -((x[,"n_11"] + x[,"n_32"])*(-log10(2) + log10(12 + (-4 - 3*p1)/2))) - (x[,"n_12"] + x[,"n_31"])*(-log10(2) + log10(10 - (3*p1)/2)) - (x[,"n_20"] + x[,"n_23"])*(-2*log10(2) + log10(4 + 3*p1)) - (x[,"n_13"] + x[,"n_30"])*(-2*log10(2) + log10(4 + 3*p1)) - (x[,"n_10"] + x[,"n_33"])*(-2*log10(2) + log10(4 + 3*p1)) - (x[,"n_21"] + x[,"n_22"])*log10(6 + (-4 - 3*p1)/2 + (4 + 3*p1)/4) + (x[,"n_20"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_rr1)) + log10(pmax(1e-6,r_rr1))) + (x[,"n_10"] + x[,"n_33"])*(log10(4 + 3*p1) + 2*log10(pmax(1e-6,r_rr1))) + (x[,"n_11"] + x[,"n_32"])*(log10(pmax(1e-6,r_rr1)) + log10(12 - (4 + 3*p1)*r_rr1)) + (x[,"n_13"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(1 - 2*r_rr1 + r_rr1^2)) + (x[,"n_21"] + x[,"n_22"])*log10(6 - (4 + 3*p1)*r_rr1 + (4 + 3*p1)*r_rr1^2) + (x[,"n_12"] + x[,"n_31"])*(log10(pmax(1e-6,1 - r_rr1)) + log10(3*p1*(-1 + r_rr1) + 4*(2 + r_rr1)))


logL_rr1 <- (-4*x[,"n_10"] - 4*x[,"n_11"] - 4*x[,"n_12"] - 4*x[,"n_13"] - 3*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 4*x[,"n_30"] - 4*x[,"n_31"] - 4*x[,"n_32"] - 4*x[,"n_33"])*log(2) + (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*log(3) + (x[,"n_20"] + x[,"n_23"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_rr1)) + log(pmax(1e-6,r_rr1))) + (x[,"n_10"] + x[,"n_33"])*(log(4 + 3*p1) + 2*log(pmax(1e-6,r_rr1))) + (x[,"n_11"] + x[,"n_32"])*(log(pmax(1e-6,r_rr1)) + log(12 - (4 + 3*p1)*r_rr1)) + (x[,"n_13"] + x[,"n_30"])*(log(4 + 3*p1) + log(1 - 2*r_rr1 + r_rr1^2)) + (x[,"n_21"] + x[,"n_22"])*log(6 - (4 + 3*p1)*r_rr1 + (4 + 3*p1)*r_rr1^2) + (x[,"n_12"] + x[,"n_31"])*(log(pmax(1e-6,1 - r_rr1)) + log(3*p1*(-1 + r_rr1) + 4*(2 + r_rr1)))


logL_rr2 <- function(r,n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
L <- (-3*n10 - 3*n11 - 3*n12 - 3*n13 - 2*n20 - 2*n21 - 2*n22 - 2*n23 - 3*n30 - 3*n31 - 3*n32 - 3*n33)*log(2) + (-n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23 - n30 - n31 - n32 - n33)*log(3) + (n13 + n30)*(log(2 - 3*p1) + 2*log(pmax(1e-6,1 - r))) + (n20 + n23)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n33)*(log(2 - 3*p1) + 2*log(pmax(1e-6,r))) + (n11 + n32)*(log(pmax(1e-6,r)) + log(6 + (-2 + 3*p1)*r)) + (n21 + n22)*log(3 + (-2 + 3*p1)*r + (2 - 3*p1)*r^2) + (n12 + n31)*(log(pmax(1e-6,1 - r)) + log(3*p1*(1 - r) + 2*(2 + r)))
return(L)}
interlogL_rr2 <- function(n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2) {
optimize(logL_rr2,c(0,0.5), n10,n11,n12,n13,n20,n21,n22,n23,n30,n31,n32,n33,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr2 <- parallel::mcmapply(interlogL_rr2,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"], p1, p2, mc.cores = ncores)


LOD_rr2 <- -((x[,"n_20"] + x[,"n_23"])*(-2*log10(2) + log10(2 - 3*p1))) - (x[,"n_13"] + x[,"n_30"])*(-2*log10(2) + log10(2 - 3*p1)) - (x[,"n_10"] + x[,"n_33"])*(-2*log10(2) + log10(2 - 3*p1)) - (x[,"n_12"] + x[,"n_31"])*(-log10(2) + log10(5 + (3*p1)/2)) - (x[,"n_11"] + x[,"n_32"])*(-log10(2) + log10(6 + (-2 + 3*p1)/2)) - (x[,"n_21"] + x[,"n_22"])*log10(3 + (2 - 3*p1)/4 + (-2 + 3*p1)/2) + (x[,"n_13"] + x[,"n_30"])*(log10(2 - 3*p1) + 2*log10(pmax(1e-6,1 - r_rr2))) + (x[,"n_20"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_rr2)) + log10(pmax(1e-6,r_rr2))) + (x[,"n_10"] + x[,"n_33"])*(log10(2 - 3*p1) + 2*log10(pmax(1e-6,r_rr2))) + (x[,"n_11"] + x[,"n_32"])*(log10(pmax(1e-6,r_rr2)) + log10(6 + (-2 + 3*p1)*r_rr2)) + (x[,"n_21"] + x[,"n_22"])*log10(3 + (-2 + 3*p1)*r_rr2 + (2 - 3*p1)*r_rr2^2) + (x[,"n_12"] + x[,"n_31"])*(log10(pmax(1e-6,1 - r_rr2)) + log10(3*p1*(1 - r_rr2) + 2*(2 + r_rr2)))


logL_rr2 <- (-3*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 2*x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 3*x[,"n_30"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"])*log(2) + (-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"])*log(3) + (x[,"n_13"] + x[,"n_30"])*(log(2 - 3*p1) + 2*log(pmax(1e-6,1 - r_rr2))) + (x[,"n_20"] + x[,"n_23"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_rr2)) + log(pmax(1e-6,r_rr2))) + (x[,"n_10"] + x[,"n_33"])*(log(2 - 3*p1) + 2*log(pmax(1e-6,r_rr2))) + (x[,"n_11"] + x[,"n_32"])*(log(pmax(1e-6,r_rr2)) + log(6 + (-2 + 3*p1)*r_rr2)) + (x[,"n_21"] + x[,"n_22"])*log(3 + (-2 + 3*p1)*r_rr2 + (2 - 3*p1)*r_rr2^2) + (x[,"n_12"] + x[,"n_31"])*(log(pmax(1e-6,1 - r_rr2)) + log(3*p1*(1 - r_rr2) + 2*(2 + r_rr2)))


return(list(
r_mat = cbind(r_cc1,r_cc2,r_cc3,r_cc4,r_cr1,r_cr2,r_rc1,r_rc2,r_rc3,r_rc4,r_rr1,r_rr2,0.499),
LOD_mat = cbind(LOD_cc1,LOD_cc2,LOD_cc3,LOD_cc4,LOD_cr1,LOD_cr2,LOD_rc1,LOD_rc2,LOD_rc3,LOD_rc4,LOD_rr1,LOD_rr2,0),
logL_mat = cbind(logL_cc1,logL_cc2,logL_cc3,logL_cc4,logL_cr1,logL_cr2,logL_rc1,logL_rc2,logL_rc3,logL_rc4,logL_rr1,logL_rr2,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling coupling","coupling coupling","coupling coupling","coupling repulsion","coupling repulsion","repulsion coupling","repulsion coupling","repulsion coupling","repulsion coupling","repulsion repulsion","repulsion repulsion","unknown")
)
)
}

#' @rdname p4_functions
p4_1.3_2.2 <- function(x,p1,p2,ncores=1){


logL_cc1 <- function(r,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-4*n10 - 3*n11 - 4*n12 - 3*n13 - 4*n14 - 3*n20 - 3*n21 - 2*n22 - 3*n23 - 3*n24 - 4*n30 - 3*n31 - 4*n32 - 3*n33 - 4*n34)*log(2) + 2*(-n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(3) + (n21 + n23)*log(8 - 3*p2 - 3*p1*(1 + 3*p2)) + (n11 + n33)*(log(8 - 3*p2 - p1*(3 + 9*p2)) + log(pmax(1e-6,1 - r))) + (n13 + n31)*(log(8 - 3*p2 - p1*(3 + 9*p2)) + log(pmax(1e-6,r))) + (n20 + n24)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n14 + n30)*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r))) + (n10 + n34)*(log(2 - 3*p1) + log(2 - 3*p2) + log(1 - 2*r + r^2)) + n22*log(10 - 4*r + 4*r^2 + p2*(3 + 6*r - 6*r^2) + p1*(3 + 6*r - 6*r^2 + 9*p2*(1 - r + r^2))) + (n12 + n32)*log(4*(4 + 2*r - 2*r^2 + 3*p2*(1 - r + r^2)) - 3*p1*(-4*(1 - r + r^2) + p2*(-3 - 6*r + 6*r^2)))
return(L)}
interlogL_cc1 <- function(n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cc1,c(0,0.5), n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc1 <- parallel::mcmapply(interlogL_cc1,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cc1 <- -((x[,"n_20"] + x[,"n_24"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2))) - (x[,"n_14"] + x[,"n_30"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_34"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_12"] + x[,"n_32"])*log10(-3*p1*(-3 - (9*p2)/2) + 4*(9/2 + (9*p2)/4)) - x[,"n_22"]*log10(9 + (9*p2)/2 + p1*(9/2 + (27*p2)/4)) - (x[,"n_13"] + x[,"n_31"])*(-log10(2) + log10(8 - 3*p2 - p1*(3 + 9*p2))) - (x[,"n_11"] + x[,"n_33"])*(-log10(2) + log10(8 - 3*p2 - p1*(3 + 9*p2))) + (x[,"n_11"] + x[,"n_33"])*(log10(8 - 3*p2 - p1*(3 + 9*p2)) + log10(pmax(1e-6,1 - r_cc1))) + (x[,"n_13"] + x[,"n_31"])*(log10(8 - 3*p2 - p1*(3 + 9*p2)) + log10(pmax(1e-6,r_cc1))) + (x[,"n_20"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cc1)) + log10(pmax(1e-6,r_cc1))) + (x[,"n_14"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_cc1))) + (x[,"n_10"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(1 - 2*r_cc1 + r_cc1^2)) + x[,"n_22"]*log10(10 - 4*r_cc1 + 4*r_cc1^2 + p2*(3 + 6*r_cc1 - 6*r_cc1^2) + p1*(3 + 6*r_cc1 - 6*r_cc1^2 + 9*p2*(1 - r_cc1 + r_cc1^2))) + (x[,"n_12"] + x[,"n_32"])*log10(4*(4 + 2*r_cc1 - 2*r_cc1^2 + 3*p2*(1 - r_cc1 + r_cc1^2)) - 3*p1*(-4*(1 - r_cc1 + r_cc1^2) + p2*(-3 - 6*r_cc1 + 6*r_cc1^2)))


logL_cc1 <- (-4*x[,"n_10"] - 3*x[,"n_11"] - 4*x[,"n_12"] - 3*x[,"n_13"] - 4*x[,"n_14"] - 3*x[,"n_20"] - 3*x[,"n_21"] - 2*x[,"n_22"] - 3*x[,"n_23"] - 3*x[,"n_24"] - 4*x[,"n_30"] - 3*x[,"n_31"] - 4*x[,"n_32"] - 3*x[,"n_33"] - 4*x[,"n_34"])*log(2) + 2*(-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(3) + (x[,"n_21"] + x[,"n_23"])*log(8 - 3*p2 - 3*p1*(1 + 3*p2)) + (x[,"n_11"] + x[,"n_33"])*(log(8 - 3*p2 - p1*(3 + 9*p2)) + log(pmax(1e-6,1 - r_cc1))) + (x[,"n_13"] + x[,"n_31"])*(log(8 - 3*p2 - p1*(3 + 9*p2)) + log(pmax(1e-6,r_cc1))) + (x[,"n_20"] + x[,"n_24"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cc1)) + log(pmax(1e-6,r_cc1))) + (x[,"n_14"] + x[,"n_30"])*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r_cc1))) + (x[,"n_10"] + x[,"n_34"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(1 - 2*r_cc1 + r_cc1^2)) + x[,"n_22"]*log(10 - 4*r_cc1 + 4*r_cc1^2 + p2*(3 + 6*r_cc1 - 6*r_cc1^2) + p1*(3 + 6*r_cc1 - 6*r_cc1^2 + 9*p2*(1 - r_cc1 + r_cc1^2))) + (x[,"n_12"] + x[,"n_32"])*log(4*(4 + 2*r_cc1 - 2*r_cc1^2 + 3*p2*(1 - r_cc1 + r_cc1^2)) - 3*p1*(-4*(1 - r_cc1 + r_cc1^2) + p2*(-3 - 6*r_cc1 + 6*r_cc1^2)))


logL_cc2 <- function(r,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-5*n10 - 4*n11 - 5*n12 - 4*n13 - 5*n14 - 4*n20 - 4*n21 - 3*n22 - 4*n23 - 4*n24 - 5*n30 - 4*n31 - 5*n32 - 4*n33 - 5*n34)*log(2) + 2*(-n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(3) + (n21 + n23)*log(16 + 3*p2 + p1*(-6 + 9*p2)) + (n11 + n33)*(log(16 + 3*p2 + p1*(-6 + 9*p2)) + log(pmax(1e-6,1 - r))) + (n13 + n31)*(log(16 + 3*p2 + p1*(-6 + 9*p2)) + log(pmax(1e-6,r))) + (n20 + n24)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n14 + n30)*(log(2 - 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r))) + (n12 + n32)*log((-4 - 3*p1)*(-8 + 3*p2) + 2*(-2 + 3*p1)*(4 + 3*p2)*(-1 + r)*r) + (n10 + n34)*(log(2 - 3*p1) + log(4 + 3*p2) + log(1 - 2*r + r^2)) + n22*log(20 - 8*r + 8*r^2 + p2*(-3 - 6*r + 6*r^2) - 3*p1*(-2 - 4*r + 4*r^2 + 3*p2*(1 - r + r^2)))
return(L)}
interlogL_cc2 <- function(n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cc2,c(0,0.5), n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc2 <- parallel::mcmapply(interlogL_cc2,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cc2 <- -((x[,"n_20"] + x[,"n_24"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2))) - (x[,"n_14"] + x[,"n_30"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_34"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - x[,"n_22"]*log10(18 - (9*p2)/2 - 3*p1*(-3 + (9*p2)/4)) - (x[,"n_12"] + x[,"n_32"])*log10((-4 - 3*p1)*(-8 + 3*p2) - ((-2 + 3*p1)*(4 + 3*p2))/2) - (x[,"n_13"] + x[,"n_31"])*(-log10(2) + log10(16 + 3*p2 + p1*(-6 + 9*p2))) - (x[,"n_11"] + x[,"n_33"])*(-log10(2) + log10(16 + 3*p2 + p1*(-6 + 9*p2))) + (x[,"n_11"] + x[,"n_33"])*(log10(16 + 3*p2 + p1*(-6 + 9*p2)) + log10(pmax(1e-6,1 - r_cc2))) + (x[,"n_13"] + x[,"n_31"])*(log10(16 + 3*p2 + p1*(-6 + 9*p2)) + log10(pmax(1e-6,r_cc2))) + (x[,"n_20"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cc2)) + log10(pmax(1e-6,r_cc2))) + (x[,"n_14"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_cc2))) + (x[,"n_12"] + x[,"n_32"])*log10((-4 - 3*p1)*(-8 + 3*p2) + 2*(-2 + 3*p1)*(4 + 3*p2)*(-1 + r_cc2)*r_cc2) + (x[,"n_10"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(1 - 2*r_cc2 + r_cc2^2)) + x[,"n_22"]*log10(20 - 8*r_cc2 + 8*r_cc2^2 + p2*(-3 - 6*r_cc2 + 6*r_cc2^2) - 3*p1*(-2 - 4*r_cc2 + 4*r_cc2^2 + 3*p2*(1 - r_cc2 + r_cc2^2)))


logL_cc2 <- (-5*x[,"n_10"] - 4*x[,"n_11"] - 5*x[,"n_12"] - 4*x[,"n_13"] - 5*x[,"n_14"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 3*x[,"n_22"] - 4*x[,"n_23"] - 4*x[,"n_24"] - 5*x[,"n_30"] - 4*x[,"n_31"] - 5*x[,"n_32"] - 4*x[,"n_33"] - 5*x[,"n_34"])*log(2) + 2*(-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(3) + (x[,"n_21"] + x[,"n_23"])*log(16 + 3*p2 + p1*(-6 + 9*p2)) + (x[,"n_11"] + x[,"n_33"])*(log(16 + 3*p2 + p1*(-6 + 9*p2)) + log(pmax(1e-6,1 - r_cc2))) + (x[,"n_13"] + x[,"n_31"])*(log(16 + 3*p2 + p1*(-6 + 9*p2)) + log(pmax(1e-6,r_cc2))) + (x[,"n_20"] + x[,"n_24"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cc2)) + log(pmax(1e-6,r_cc2))) + (x[,"n_14"] + x[,"n_30"])*(log(2 - 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r_cc2))) + (x[,"n_12"] + x[,"n_32"])*log((-4 - 3*p1)*(-8 + 3*p2) + 2*(-2 + 3*p1)*(4 + 3*p2)*(-1 + r_cc2)*r_cc2) + (x[,"n_10"] + x[,"n_34"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(1 - 2*r_cc2 + r_cc2^2)) + x[,"n_22"]*log(20 - 8*r_cc2 + 8*r_cc2^2 + p2*(-3 - 6*r_cc2 + 6*r_cc2^2) - 3*p1*(-2 - 4*r_cc2 + 4*r_cc2^2 + 3*p2*(1 - r_cc2 + r_cc2^2)))


logL_cc3 <- function(r,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-5*n10 - 4*n11 - 5*n12 - 4*n13 - 5*n14 - 4*n20 - 4*n21 - 3*n22 - 4*n23 - 4*n24 - 5*n30 - 4*n31 - 5*n32 - 4*n33 - 5*n34)*log(2) + 2*(-n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(3) + (n21 + n23)*log(16 - 6*p2 + p1*(3 + 9*p2)) + (n11 + n33)*(log(16 - 6*p2 + p1*(3 + 9*p2)) + log(pmax(1e-6,1 - r))) + (n13 + n31)*(log(16 - 6*p2 + p1*(3 + 9*p2)) + log(pmax(1e-6,r))) + (n20 + n24)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n14 + n30)*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r))) + (n12 + n32)*log((8 - 3*p1)*(4 + 3*p2) + 2*(4 + 3*p1)*(-2 + 3*p2)*(-1 + r)*r) + (n10 + n34)*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r + r^2)) + n22*log(20 - 8*r + 8*r^2 + p2*(6 + 12*r - 12*r^2) - 3*p1*(1 + 2*r - 2*r^2 + 3*p2*(1 - r + r^2)))
return(L)}
interlogL_cc3 <- function(n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cc3,c(0,0.5), n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc3 <- parallel::mcmapply(interlogL_cc3,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cc3 <- -((x[,"n_20"] + x[,"n_24"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2))) - (x[,"n_14"] + x[,"n_30"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_34"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - x[,"n_22"]*log10(18 + 9*p2 - 3*p1*(3/2 + (9*p2)/4)) - (x[,"n_12"] + x[,"n_32"])*log10(-((4 + 3*p1)*(-2 + 3*p2))/2 + (8 - 3*p1)*(4 + 3*p2)) - (x[,"n_13"] + x[,"n_31"])*(-log10(2) + log10(16 - 6*p2 + p1*(3 + 9*p2))) - (x[,"n_11"] + x[,"n_33"])*(-log10(2) + log10(16 - 6*p2 + p1*(3 + 9*p2))) + (x[,"n_11"] + x[,"n_33"])*(log10(16 - 6*p2 + p1*(3 + 9*p2)) + log10(pmax(1e-6,1 - r_cc3))) + (x[,"n_13"] + x[,"n_31"])*(log10(16 - 6*p2 + p1*(3 + 9*p2)) + log10(pmax(1e-6,r_cc3))) + (x[,"n_20"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cc3)) + log10(pmax(1e-6,r_cc3))) + (x[,"n_14"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_cc3))) + (x[,"n_12"] + x[,"n_32"])*log10((8 - 3*p1)*(4 + 3*p2) + 2*(4 + 3*p1)*(-2 + 3*p2)*(-1 + r_cc3)*r_cc3) + (x[,"n_10"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(1 - 2*r_cc3 + r_cc3^2)) + x[,"n_22"]*log10(20 - 8*r_cc3 + 8*r_cc3^2 + p2*(6 + 12*r_cc3 - 12*r_cc3^2) - 3*p1*(1 + 2*r_cc3 - 2*r_cc3^2 + 3*p2*(1 - r_cc3 + r_cc3^2)))


logL_cc3 <- (-5*x[,"n_10"] - 4*x[,"n_11"] - 5*x[,"n_12"] - 4*x[,"n_13"] - 5*x[,"n_14"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 3*x[,"n_22"] - 4*x[,"n_23"] - 4*x[,"n_24"] - 5*x[,"n_30"] - 4*x[,"n_31"] - 5*x[,"n_32"] - 4*x[,"n_33"] - 5*x[,"n_34"])*log(2) + 2*(-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(3) + (x[,"n_21"] + x[,"n_23"])*log(16 - 6*p2 + p1*(3 + 9*p2)) + (x[,"n_11"] + x[,"n_33"])*(log(16 - 6*p2 + p1*(3 + 9*p2)) + log(pmax(1e-6,1 - r_cc3))) + (x[,"n_13"] + x[,"n_31"])*(log(16 - 6*p2 + p1*(3 + 9*p2)) + log(pmax(1e-6,r_cc3))) + (x[,"n_20"] + x[,"n_24"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cc3)) + log(pmax(1e-6,r_cc3))) + (x[,"n_14"] + x[,"n_30"])*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r_cc3))) + (x[,"n_12"] + x[,"n_32"])*log((8 - 3*p1)*(4 + 3*p2) + 2*(4 + 3*p1)*(-2 + 3*p2)*(-1 + r_cc3)*r_cc3) + (x[,"n_10"] + x[,"n_34"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r_cc3 + r_cc3^2)) + x[,"n_22"]*log(20 - 8*r_cc3 + 8*r_cc3^2 + p2*(6 + 12*r_cc3 - 12*r_cc3^2) - 3*p1*(1 + 2*r_cc3 - 2*r_cc3^2 + 3*p2*(1 - r_cc3 + r_cc3^2)))


logL_cc4 <- function(r,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-6*n10 - 5*n11 - 6*n12 - 5*n13 - 6*n14 - 5*n20 - 5*n21 - 5*n22 - 5*n23 - 5*n24 - 6*n30 - 5*n31 - 6*n32 - 5*n33 - 6*n34)*log(2) + 2*(-n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(3) + (n21 + n23)*log(32 + p1*(6 - 9*p2) + 6*p2) + (n11 + n33)*(log(32 + 6*p2 - p1*(-6 + 9*p2)) + log(pmax(1e-6,1 - r))) + (n13 + n31)*(log(32 + p1*(6 - 9*p2) + 6*p2) + log(pmax(1e-6,r))) + (n20 + n24)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n14 + n30)*(log(4 + 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r))) + (n12 + n32)*log((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r)*r) + (n10 + n34)*(log(4 + 3*p1) + log(4 + 3*p2) + log(1 - 2*r + r^2)) + n22*log((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r)^2 + (4 + 3*p1)*(4 + 3*p2)*r^2)
return(L)}
interlogL_cc4 <- function(n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cc4,c(0,0.5), n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc4 <- parallel::mcmapply(interlogL_cc4,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cc4 <- -((x[,"n_20"] + x[,"n_24"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2))) - (x[,"n_14"] + x[,"n_30"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_34"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_13"] + x[,"n_31"])*(-log10(2) + log10(32 + p1*(6 - 9*p2) + 6*p2)) - x[,"n_22"]*log10((-8 + 3*p1)*(-8 + 3*p2) + ((4 + 3*p1)*(4 + 3*p2))/2) - (x[,"n_12"] + x[,"n_32"])*log10((-8 + 3*p1)*(-8 + 3*p2) + ((4 + 3*p1)*(4 + 3*p2))/2) - (x[,"n_11"] + x[,"n_33"])*(-log10(2) + log10(32 + 6*p2 - p1*(-6 + 9*p2))) + (x[,"n_11"] + x[,"n_33"])*(log10(32 + 6*p2 - p1*(-6 + 9*p2)) + log10(pmax(1e-6,1 - r_cc4))) + (x[,"n_13"] + x[,"n_31"])*(log10(32 + p1*(6 - 9*p2) + 6*p2) + log10(pmax(1e-6,r_cc4))) + (x[,"n_20"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cc4)) + log10(pmax(1e-6,r_cc4))) + (x[,"n_14"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_cc4))) + (x[,"n_12"] + x[,"n_32"])*log10((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_cc4)*r_cc4) + (x[,"n_10"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(1 - 2*r_cc4 + r_cc4^2)) + x[,"n_22"]*log10((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r_cc4)^2 + (4 + 3*p1)*(4 + 3*p2)*r_cc4^2)


logL_cc4 <- (-6*x[,"n_10"] - 5*x[,"n_11"] - 6*x[,"n_12"] - 5*x[,"n_13"] - 6*x[,"n_14"] - 5*x[,"n_20"] - 5*x[,"n_21"] - 5*x[,"n_22"] - 5*x[,"n_23"] - 5*x[,"n_24"] - 6*x[,"n_30"] - 5*x[,"n_31"] - 6*x[,"n_32"] - 5*x[,"n_33"] - 6*x[,"n_34"])*log(2) + 2*(-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(3) + (x[,"n_21"] + x[,"n_23"])*log(32 + p1*(6 - 9*p2) + 6*p2) + (x[,"n_11"] + x[,"n_33"])*(log(32 + 6*p2 - p1*(-6 + 9*p2)) + log(pmax(1e-6,1 - r_cc4))) + (x[,"n_13"] + x[,"n_31"])*(log(32 + p1*(6 - 9*p2) + 6*p2) + log(pmax(1e-6,r_cc4))) + (x[,"n_20"] + x[,"n_24"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cc4)) + log(pmax(1e-6,r_cc4))) + (x[,"n_14"] + x[,"n_30"])*(log(4 + 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r_cc4))) + (x[,"n_12"] + x[,"n_32"])*log((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_cc4)*r_cc4) + (x[,"n_10"] + x[,"n_34"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(1 - 2*r_cc4 + r_cc4^2)) + x[,"n_22"]*log((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r_cc4)^2 + (4 + 3*p1)*(4 + 3*p2)*r_cc4^2)


logL_cm1 <- function(r,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-5*n10 - 5*n11 - 5*n12 - 5*n13 - 5*n14 - 5*n20 - 4*n21 - 4*n22 - 4*n23 - 5*n24 - 5*n30 - 5*n31 - 5*n32 - 5*n33 - 5*n34)*log(2) + 2*(-n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(3) + (n21 + n23)*log(16 + 3*p2 + p1*(-6 + 9*p2)) + (n13 + n31)*log(2*(8 + p2*(6 - 9*r)) + 3*p1*(4 + 3*p2 - 12*r)) + (n10 + n14 + n30 + n34)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n20 + n24)*(log(2 - 3*p1) + log(4 + 3*p2) + log(1 - 2*r + 2*r^2)) + (n12 + n32)*log((-4 - 3*p1)*(-8 + 3*p2) - (-2 + 3*p1)*(4 + 3*p2)*(-1 + r)^2 + (2 - 3*p1)*(4 + 3*p2)*r^2) + (n11 + n33)*log(3*p1*(-8 + 3*p2 + 12*r) + 2*(8 + p2*(-3 + 9*r))) + n22*log(-4*(4*(-2 - r + r^2) + 3*p2*(1 - r + r^2)) + 3*p1*(8*(1 - r + r^2) + p2*(-3 - 6*r + 6*r^2)))
return(L)}
interlogL_cm1 <- function(n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cm1,c(0,0.5), n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cm1 <- parallel::mcmapply(interlogL_cm1,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cm1 <- -((x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2))) - (x[,"n_20"] + x[,"n_24"])*(-log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - x[,"n_22"]*log10(3*p1*(6 - (9*p2)/2) - 4*(-9 + (9*p2)/4)) - (x[,"n_13"] + x[,"n_31"])*log10(2*(8 + (3*p2)/2) + 3*p1*(-2 + 3*p2)) - (x[,"n_11"] + x[,"n_33"])*log10(2*(8 + (3*p2)/2) + 3*p1*(-2 + 3*p2)) - (x[,"n_12"] + x[,"n_32"])*log10((-4 - 3*p1)*(-8 + 3*p2) + ((2 - 3*p1)*(4 + 3*p2))/4 - ((-2 + 3*p1)*(4 + 3*p2))/4) + (x[,"n_13"] + x[,"n_31"])*log10(2*(8 + p2*(6 - 9*r_cm1)) + 3*p1*(4 + 3*p2 - 12*r_cm1)) + (x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cm1)) + log10(pmax(1e-6,r_cm1))) + (x[,"n_20"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(1 - 2*r_cm1 + 2*r_cm1^2)) + (x[,"n_12"] + x[,"n_32"])*log10((-4 - 3*p1)*(-8 + 3*p2) - (-2 + 3*p1)*(4 + 3*p2)*(-1 + r_cm1)^2 + (2 - 3*p1)*(4 + 3*p2)*r_cm1^2) + (x[,"n_11"] + x[,"n_33"])*log10(3*p1*(-8 + 3*p2 + 12*r_cm1) + 2*(8 + p2*(-3 + 9*r_cm1))) + x[,"n_22"]*log10(-4*(4*(-2 - r_cm1 + r_cm1^2) + 3*p2*(1 - r_cm1 + r_cm1^2)) + 3*p1*(8*(1 - r_cm1 + r_cm1^2) + p2*(-3 - 6*r_cm1 + 6*r_cm1^2)))


logL_cm1 <- (-5*x[,"n_10"] - 5*x[,"n_11"] - 5*x[,"n_12"] - 5*x[,"n_13"] - 5*x[,"n_14"] - 5*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 4*x[,"n_23"] - 5*x[,"n_24"] - 5*x[,"n_30"] - 5*x[,"n_31"] - 5*x[,"n_32"] - 5*x[,"n_33"] - 5*x[,"n_34"])*log(2) + 2*(-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(3) + (x[,"n_21"] + x[,"n_23"])*log(16 + 3*p2 + p1*(-6 + 9*p2)) + (x[,"n_13"] + x[,"n_31"])*log(2*(8 + p2*(6 - 9*r_cm1)) + 3*p1*(4 + 3*p2 - 12*r_cm1)) + (x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cm1)) + log(pmax(1e-6,r_cm1))) + (x[,"n_20"] + x[,"n_24"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(1 - 2*r_cm1 + 2*r_cm1^2)) + (x[,"n_12"] + x[,"n_32"])*log((-4 - 3*p1)*(-8 + 3*p2) - (-2 + 3*p1)*(4 + 3*p2)*(-1 + r_cm1)^2 + (2 - 3*p1)*(4 + 3*p2)*r_cm1^2) + (x[,"n_11"] + x[,"n_33"])*log(3*p1*(-8 + 3*p2 + 12*r_cm1) + 2*(8 + p2*(-3 + 9*r_cm1))) + x[,"n_22"]*log(-4*(4*(-2 - r_cm1 + r_cm1^2) + 3*p2*(1 - r_cm1 + r_cm1^2)) + 3*p1*(8*(1 - r_cm1 + r_cm1^2) + p2*(-3 - 6*r_cm1 + 6*r_cm1^2)))


logL_cm2 <- function(r,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-4*n10 - 4*n11 - 3*n12 - 4*n13 - 4*n14 - 4*n20 - 3*n21 - 3*n22 - 3*n23 - 4*n24 - 4*n30 - 4*n31 - 3*n32 - 4*n33 - 4*n34)*log(2) + 2*(-n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(3) + (n21 + n23)*log(8 - 3*p2 - 3*p1*(1 + 3*p2)) + (n11 + n33)*log(8 + p2*(6 - 18*r) - 3*p1*(4 + 3*p2 - 6*r)) + (n14 + n30)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n34)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n20 + n24)*(log(2 - 3*p1) + log(2 - 3*p2) + log(1 - 2*r + 2*r^2)) + (n13 + n31)*log(-3*p1*(-2 + 3*p2 + 6*r) + 2*(4 + p2*(-6 + 9*r))) + (n12 + n32)*log(10 - 4*r + 4*r^2 + p2*(3 + 6*r - 6*r^2) + p1*(3 + 6*r - 6*r^2 + 9*p2*(1 - r + r^2))) + n22*log(4*(4 + 2*r - 2*r^2 + 3*p2*(1 - r + r^2)) - 3*p1*(-4*(1 - r + r^2) + p2*(-3 - 6*r + 6*r^2)))
return(L)}
interlogL_cm2 <- function(n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cm2,c(0,0.5), n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cm2 <- parallel::mcmapply(interlogL_cm2,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cm2 <- -((x[,"n_14"] + x[,"n_30"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2))) - (x[,"n_10"] + x[,"n_34"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_20"] + x[,"n_24"])*(-log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - x[,"n_22"]*log10(-3*p1*(-3 - (9*p2)/2) + 4*(9/2 + (9*p2)/4)) - (x[,"n_13"] + x[,"n_31"])*log10(2*(4 - (3*p2)/2) - 3*p1*(1 + 3*p2)) - (x[,"n_11"] + x[,"n_33"])*log10(8 - 3*p2 - 3*p1*(1 + 3*p2)) - (x[,"n_12"] + x[,"n_32"])*log10(9 + (9*p2)/2 + p1*(9/2 + (27*p2)/4)) + (x[,"n_11"] + x[,"n_33"])*log10(8 + p2*(6 - 18*r_cm2) - 3*p1*(4 + 3*p2 - 6*r_cm2)) + (x[,"n_14"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cm2)) + log10(pmax(1e-6,r_cm2))) + (x[,"n_10"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cm2)) + log10(pmax(1e-6,r_cm2))) + (x[,"n_20"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(1 - 2*r_cm2 + 2*r_cm2^2)) + (x[,"n_13"] + x[,"n_31"])*log10(-3*p1*(-2 + 3*p2 + 6*r_cm2) + 2*(4 + p2*(-6 + 9*r_cm2))) + (x[,"n_12"] + x[,"n_32"])*log10(10 - 4*r_cm2 + 4*r_cm2^2 + p2*(3 + 6*r_cm2 - 6*r_cm2^2) + p1*(3 + 6*r_cm2 - 6*r_cm2^2 + 9*p2*(1 - r_cm2 + r_cm2^2))) + x[,"n_22"]*log10(4*(4 + 2*r_cm2 - 2*r_cm2^2 + 3*p2*(1 - r_cm2 + r_cm2^2)) - 3*p1*(-4*(1 - r_cm2 + r_cm2^2) + p2*(-3 - 6*r_cm2 + 6*r_cm2^2)))


logL_cm2 <- (-4*x[,"n_10"] - 4*x[,"n_11"] - 3*x[,"n_12"] - 4*x[,"n_13"] - 4*x[,"n_14"] - 4*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 4*x[,"n_24"] - 4*x[,"n_30"] - 4*x[,"n_31"] - 3*x[,"n_32"] - 4*x[,"n_33"] - 4*x[,"n_34"])*log(2) + 2*(-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(3) + (x[,"n_21"] + x[,"n_23"])*log(8 - 3*p2 - 3*p1*(1 + 3*p2)) + (x[,"n_11"] + x[,"n_33"])*log(8 + p2*(6 - 18*r_cm2) - 3*p1*(4 + 3*p2 - 6*r_cm2)) + (x[,"n_14"] + x[,"n_30"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cm2)) + log(pmax(1e-6,r_cm2))) + (x[,"n_10"] + x[,"n_34"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cm2)) + log(pmax(1e-6,r_cm2))) + (x[,"n_20"] + x[,"n_24"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(1 - 2*r_cm2 + 2*r_cm2^2)) + (x[,"n_13"] + x[,"n_31"])*log(-3*p1*(-2 + 3*p2 + 6*r_cm2) + 2*(4 + p2*(-6 + 9*r_cm2))) + (x[,"n_12"] + x[,"n_32"])*log(10 - 4*r_cm2 + 4*r_cm2^2 + p2*(3 + 6*r_cm2 - 6*r_cm2^2) + p1*(3 + 6*r_cm2 - 6*r_cm2^2 + 9*p2*(1 - r_cm2 + r_cm2^2))) + x[,"n_22"]*log(4*(4 + 2*r_cm2 - 2*r_cm2^2 + 3*p2*(1 - r_cm2 + r_cm2^2)) - 3*p1*(-4*(1 - r_cm2 + r_cm2^2) + p2*(-3 - 6*r_cm2 + 6*r_cm2^2)))


logL_cm3 <- function(r,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-6*n10 - 6*n11 - 6*n12 - 6*n13 - 6*n14 - 6*n20 - 5*n21 - 5*n22 - 5*n23 - 6*n24 - 6*n30 - 6*n31 - 6*n32 - 6*n33 - 6*n34)*log(2) + 2*(-n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(3) + (n21 + n23)*log(32 + p1*(6 - 9*p2) + 6*p2) + (n13 + n31)*log(4*(8 + p2*(6 - 9*r)) - 3*p1*(4 + 3*p2 - 12*r)) + (n10 + n14 + n30 + n34)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + n22*log((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r)*r) + (n20 + n24)*(log(4 + 3*p1) + log(4 + 3*p2) + log((-1 + r)^2 + r^2)) + (n12 + n32)*log((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r)^2 + (4 + 3*p1)*(4 + 3*p2)*r^2) + (n11 + n33)*log(-3*p1*(-8 + 3*p2 + 12*r) + 4*(8 + p2*(-3 + 9*r)))
return(L)}
interlogL_cm3 <- function(n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cm3,c(0,0.5), n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cm3 <- parallel::mcmapply(interlogL_cm3,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cm3 <- -((x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2))) - (x[,"n_20"] + x[,"n_24"])*(-log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_13"] + x[,"n_31"])*log10(4*(8 + (3*p2)/2) - 3*p1*(-2 + 3*p2)) - (x[,"n_11"] + x[,"n_33"])*log10(4*(8 + (3*p2)/2) - 3*p1*(-2 + 3*p2)) - x[,"n_22"]*log10((-8 + 3*p1)*(-8 + 3*p2) + ((4 + 3*p1)*(4 + 3*p2))/2) - (x[,"n_12"] + x[,"n_32"])*log10((-8 + 3*p1)*(-8 + 3*p2) + ((4 + 3*p1)*(4 + 3*p2))/2) + (x[,"n_13"] + x[,"n_31"])*log10(4*(8 + p2*(6 - 9*r_cm3)) - 3*p1*(4 + 3*p2 - 12*r_cm3)) + (x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cm3)) + log10(pmax(1e-6,r_cm3))) + x[,"n_22"]*log10((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_cm3)*r_cm3) + (x[,"n_20"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10((-1 + r_cm3)^2 + r_cm3^2)) + (x[,"n_12"] + x[,"n_32"])*log10((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r_cm3)^2 + (4 + 3*p1)*(4 + 3*p2)*r_cm3^2) + (x[,"n_11"] + x[,"n_33"])*log10(-3*p1*(-8 + 3*p2 + 12*r_cm3) + 4*(8 + p2*(-3 + 9*r_cm3)))


logL_cm3 <- (-6*x[,"n_10"] - 6*x[,"n_11"] - 6*x[,"n_12"] - 6*x[,"n_13"] - 6*x[,"n_14"] - 6*x[,"n_20"] - 5*x[,"n_21"] - 5*x[,"n_22"] - 5*x[,"n_23"] - 6*x[,"n_24"] - 6*x[,"n_30"] - 6*x[,"n_31"] - 6*x[,"n_32"] - 6*x[,"n_33"] - 6*x[,"n_34"])*log(2) + 2*(-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(3) + (x[,"n_21"] + x[,"n_23"])*log(32 + p1*(6 - 9*p2) + 6*p2) + (x[,"n_13"] + x[,"n_31"])*log(4*(8 + p2*(6 - 9*r_cm3)) - 3*p1*(4 + 3*p2 - 12*r_cm3)) + (x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cm3)) + log(pmax(1e-6,r_cm3))) + x[,"n_22"]*log((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_cm3)*r_cm3) + (x[,"n_20"] + x[,"n_24"])*(log(4 + 3*p1) + log(4 + 3*p2) + log((-1 + r_cm3)^2 + r_cm3^2)) + (x[,"n_12"] + x[,"n_32"])*log((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r_cm3)^2 + (4 + 3*p1)*(4 + 3*p2)*r_cm3^2) + (x[,"n_11"] + x[,"n_33"])*log(-3*p1*(-8 + 3*p2 + 12*r_cm3) + 4*(8 + p2*(-3 + 9*r_cm3)))


logL_cm4 <- function(r,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-5*n10 - 5*n11 - 5*n12 - 5*n13 - 5*n14 - 5*n20 - 4*n21 - 4*n22 - 4*n23 - 5*n24 - 5*n30 - 5*n31 - 5*n32 - 5*n33 - 5*n34)*log(2) + 2*(-n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(3) + (n21 + n23)*log(16 - 6*p2 + p1*(3 + 9*p2)) + (n11 + n33)*log(4*(4 + p2*(3 - 9*r)) + 3*p1*(4 + 3*p2 - 6*r)) + (n10 + n14 + n30 + n34)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n20 + n24)*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r + 2*r^2)) + (n12 + n32)*log((8 - 3*p1)*(4 + 3*p2) - (4 + 3*p1)*(-2 + 3*p2)*(-1 + r)^2 + (4 + 3*p1)*(2 - 3*p2)*r^2) + (n13 + n31)*log(3*p1*(-2 + 3*p2 + 6*r) + 4*(4 + p2*(-6 + 9*r))) + n22*log(8*(4 + 2*r - 2*r^2 + 3*p2*(1 - r + r^2)) + 3*p1*(-4*(1 - r + r^2) + p2*(-3 - 6*r + 6*r^2)))
return(L)}
interlogL_cm4 <- function(n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cm4,c(0,0.5), n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cm4 <- parallel::mcmapply(interlogL_cm4,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cm4 <- -((x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2))) - (x[,"n_20"] + x[,"n_24"])*(-log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - x[,"n_22"]*log10(3*p1*(-3 - (9*p2)/2) + 8*(9/2 + (9*p2)/4)) - (x[,"n_13"] + x[,"n_31"])*log10(4*(4 - (3*p2)/2) + 3*p1*(1 + 3*p2)) - (x[,"n_11"] + x[,"n_33"])*log10(4*(4 - (3*p2)/2) + 3*p1*(1 + 3*p2)) - (x[,"n_12"] + x[,"n_32"])*log10(((4 + 3*p1)*(2 - 3*p2))/4 - ((4 + 3*p1)*(-2 + 3*p2))/4 + (8 - 3*p1)*(4 + 3*p2)) + (x[,"n_11"] + x[,"n_33"])*log10(4*(4 + p2*(3 - 9*r_cm4)) + 3*p1*(4 + 3*p2 - 6*r_cm4)) + (x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cm4)) + log10(pmax(1e-6,r_cm4))) + (x[,"n_20"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(1 - 2*r_cm4 + 2*r_cm4^2)) + (x[,"n_12"] + x[,"n_32"])*log10((8 - 3*p1)*(4 + 3*p2) - (4 + 3*p1)*(-2 + 3*p2)*(-1 + r_cm4)^2 + (4 + 3*p1)*(2 - 3*p2)*r_cm4^2) + (x[,"n_13"] + x[,"n_31"])*log10(3*p1*(-2 + 3*p2 + 6*r_cm4) + 4*(4 + p2*(-6 + 9*r_cm4))) + x[,"n_22"]*log10(8*(4 + 2*r_cm4 - 2*r_cm4^2 + 3*p2*(1 - r_cm4 + r_cm4^2)) + 3*p1*(-4*(1 - r_cm4 + r_cm4^2) + p2*(-3 - 6*r_cm4 + 6*r_cm4^2)))


logL_cm4 <- (-5*x[,"n_10"] - 5*x[,"n_11"] - 5*x[,"n_12"] - 5*x[,"n_13"] - 5*x[,"n_14"] - 5*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 4*x[,"n_23"] - 5*x[,"n_24"] - 5*x[,"n_30"] - 5*x[,"n_31"] - 5*x[,"n_32"] - 5*x[,"n_33"] - 5*x[,"n_34"])*log(2) + 2*(-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(3) + (x[,"n_21"] + x[,"n_23"])*log(16 - 6*p2 + p1*(3 + 9*p2)) + (x[,"n_11"] + x[,"n_33"])*log(4*(4 + p2*(3 - 9*r_cm4)) + 3*p1*(4 + 3*p2 - 6*r_cm4)) + (x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cm4)) + log(pmax(1e-6,r_cm4))) + (x[,"n_20"] + x[,"n_24"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r_cm4 + 2*r_cm4^2)) + (x[,"n_12"] + x[,"n_32"])*log((8 - 3*p1)*(4 + 3*p2) - (4 + 3*p1)*(-2 + 3*p2)*(-1 + r_cm4)^2 + (4 + 3*p1)*(2 - 3*p2)*r_cm4^2) + (x[,"n_13"] + x[,"n_31"])*log(3*p1*(-2 + 3*p2 + 6*r_cm4) + 4*(4 + p2*(-6 + 9*r_cm4))) + x[,"n_22"]*log(8*(4 + 2*r_cm4 - 2*r_cm4^2 + 3*p2*(1 - r_cm4 + r_cm4^2)) + 3*p1*(-4*(1 - r_cm4 + r_cm4^2) + p2*(-3 - 6*r_cm4 + 6*r_cm4^2)))


logL_rc1 <- function(r,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-5*n10 - 5*n11 - 5*n12 - 5*n13 - 5*n14 - 5*n20 - 4*n21 - 4*n22 - 4*n23 - 5*n24 - 5*n30 - 5*n31 - 5*n32 - 5*n33 - 5*n34)*log(2) + 2*(-n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(3) + (n21 + n23)*log(16 - 6*p2 + p1*(3 + 9*p2)) + (n13 + n31)*log(4*(4 + p2*(3 - 9*r)) + 3*p1*(4 + 3*p2 - 6*r)) + (n10 + n14 + n30 + n34)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n20 + n24)*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r + 2*r^2)) + (n12 + n32)*log((8 - 3*p1)*(4 + 3*p2) - (4 + 3*p1)*(-2 + 3*p2)*(-1 + r)^2 + (4 + 3*p1)*(2 - 3*p2)*r^2) + (n11 + n33)*log(3*p1*(-2 + 3*p2 + 6*r) + 4*(4 + p2*(-6 + 9*r))) + n22*log(8*(4 + 2*r - 2*r^2 + 3*p2*(1 - r + r^2)) + 3*p1*(-4*(1 - r + r^2) + p2*(-3 - 6*r + 6*r^2)))
return(L)}
interlogL_rc1 <- function(n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rc1,c(0,0.5), n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc1 <- parallel::mcmapply(interlogL_rc1,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rc1 <- -((x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2))) - (x[,"n_20"] + x[,"n_24"])*(-log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - x[,"n_22"]*log10(3*p1*(-3 - (9*p2)/2) + 8*(9/2 + (9*p2)/4)) - (x[,"n_13"] + x[,"n_31"])*log10(4*(4 - (3*p2)/2) + 3*p1*(1 + 3*p2)) - (x[,"n_11"] + x[,"n_33"])*log10(4*(4 - (3*p2)/2) + 3*p1*(1 + 3*p2)) - (x[,"n_12"] + x[,"n_32"])*log10(((4 + 3*p1)*(2 - 3*p2))/4 - ((4 + 3*p1)*(-2 + 3*p2))/4 + (8 - 3*p1)*(4 + 3*p2)) + (x[,"n_13"] + x[,"n_31"])*log10(4*(4 + p2*(3 - 9*r_rc1)) + 3*p1*(4 + 3*p2 - 6*r_rc1)) + (x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rc1)) + log10(pmax(1e-6,r_rc1))) + (x[,"n_20"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(1 - 2*r_rc1 + 2*r_rc1^2)) + (x[,"n_12"] + x[,"n_32"])*log10((8 - 3*p1)*(4 + 3*p2) - (4 + 3*p1)*(-2 + 3*p2)*(-1 + r_rc1)^2 + (4 + 3*p1)*(2 - 3*p2)*r_rc1^2) + (x[,"n_11"] + x[,"n_33"])*log10(3*p1*(-2 + 3*p2 + 6*r_rc1) + 4*(4 + p2*(-6 + 9*r_rc1))) + x[,"n_22"]*log10(8*(4 + 2*r_rc1 - 2*r_rc1^2 + 3*p2*(1 - r_rc1 + r_rc1^2)) + 3*p1*(-4*(1 - r_rc1 + r_rc1^2) + p2*(-3 - 6*r_rc1 + 6*r_rc1^2)))


logL_rc1 <- (-5*x[,"n_10"] - 5*x[,"n_11"] - 5*x[,"n_12"] - 5*x[,"n_13"] - 5*x[,"n_14"] - 5*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 4*x[,"n_23"] - 5*x[,"n_24"] - 5*x[,"n_30"] - 5*x[,"n_31"] - 5*x[,"n_32"] - 5*x[,"n_33"] - 5*x[,"n_34"])*log(2) + 2*(-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(3) + (x[,"n_21"] + x[,"n_23"])*log(16 - 6*p2 + p1*(3 + 9*p2)) + (x[,"n_13"] + x[,"n_31"])*log(4*(4 + p2*(3 - 9*r_rc1)) + 3*p1*(4 + 3*p2 - 6*r_rc1)) + (x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rc1)) + log(pmax(1e-6,r_rc1))) + (x[,"n_20"] + x[,"n_24"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r_rc1 + 2*r_rc1^2)) + (x[,"n_12"] + x[,"n_32"])*log((8 - 3*p1)*(4 + 3*p2) - (4 + 3*p1)*(-2 + 3*p2)*(-1 + r_rc1)^2 + (4 + 3*p1)*(2 - 3*p2)*r_rc1^2) + (x[,"n_11"] + x[,"n_33"])*log(3*p1*(-2 + 3*p2 + 6*r_rc1) + 4*(4 + p2*(-6 + 9*r_rc1))) + x[,"n_22"]*log(8*(4 + 2*r_rc1 - 2*r_rc1^2 + 3*p2*(1 - r_rc1 + r_rc1^2)) + 3*p1*(-4*(1 - r_rc1 + r_rc1^2) + p2*(-3 - 6*r_rc1 + 6*r_rc1^2)))


logL_rc2 <- function(r,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-6*n10 - 6*n11 - 6*n12 - 6*n13 - 6*n14 - 6*n20 - 5*n21 - 5*n22 - 5*n23 - 6*n24 - 6*n30 - 6*n31 - 6*n32 - 6*n33 - 6*n34)*log(2) + 2*(-n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(3) + (n21 + n23)*log(32 + p1*(6 - 9*p2) + 6*p2) + (n11 + n33)*log(4*(8 + p2*(6 - 9*r)) - 3*p1*(4 + 3*p2 - 12*r)) + (n10 + n14 + n30 + n34)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + n22*log((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r)*r) + (n20 + n24)*(log(4 + 3*p1) + log(4 + 3*p2) + log((-1 + r)^2 + r^2)) + (n12 + n32)*log((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r)^2 + (4 + 3*p1)*(4 + 3*p2)*r^2) + (n13 + n31)*log(-3*p1*(-8 + 3*p2 + 12*r) + 4*(8 + p2*(-3 + 9*r)))
return(L)}
interlogL_rc2 <- function(n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rc2,c(0,0.5), n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc2 <- parallel::mcmapply(interlogL_rc2,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rc2 <- -((x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2))) - (x[,"n_20"] + x[,"n_24"])*(-log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_13"] + x[,"n_31"])*log10(4*(8 + (3*p2)/2) - 3*p1*(-2 + 3*p2)) - (x[,"n_11"] + x[,"n_33"])*log10(4*(8 + (3*p2)/2) - 3*p1*(-2 + 3*p2)) - x[,"n_22"]*log10((-8 + 3*p1)*(-8 + 3*p2) + ((4 + 3*p1)*(4 + 3*p2))/2) - (x[,"n_12"] + x[,"n_32"])*log10((-8 + 3*p1)*(-8 + 3*p2) + ((4 + 3*p1)*(4 + 3*p2))/2) + (x[,"n_11"] + x[,"n_33"])*log10(4*(8 + p2*(6 - 9*r_rc2)) - 3*p1*(4 + 3*p2 - 12*r_rc2)) + (x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rc2)) + log10(pmax(1e-6,r_rc2))) + x[,"n_22"]*log10((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_rc2)*r_rc2) + (x[,"n_20"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10((-1 + r_rc2)^2 + r_rc2^2)) + (x[,"n_12"] + x[,"n_32"])*log10((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r_rc2)^2 + (4 + 3*p1)*(4 + 3*p2)*r_rc2^2) + (x[,"n_13"] + x[,"n_31"])*log10(-3*p1*(-8 + 3*p2 + 12*r_rc2) + 4*(8 + p2*(-3 + 9*r_rc2)))


logL_rc2 <- (-6*x[,"n_10"] - 6*x[,"n_11"] - 6*x[,"n_12"] - 6*x[,"n_13"] - 6*x[,"n_14"] - 6*x[,"n_20"] - 5*x[,"n_21"] - 5*x[,"n_22"] - 5*x[,"n_23"] - 6*x[,"n_24"] - 6*x[,"n_30"] - 6*x[,"n_31"] - 6*x[,"n_32"] - 6*x[,"n_33"] - 6*x[,"n_34"])*log(2) + 2*(-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(3) + (x[,"n_21"] + x[,"n_23"])*log(32 + p1*(6 - 9*p2) + 6*p2) + (x[,"n_11"] + x[,"n_33"])*log(4*(8 + p2*(6 - 9*r_rc2)) - 3*p1*(4 + 3*p2 - 12*r_rc2)) + (x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rc2)) + log(pmax(1e-6,r_rc2))) + x[,"n_22"]*log((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_rc2)*r_rc2) + (x[,"n_20"] + x[,"n_24"])*(log(4 + 3*p1) + log(4 + 3*p2) + log((-1 + r_rc2)^2 + r_rc2^2)) + (x[,"n_12"] + x[,"n_32"])*log((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r_rc2)^2 + (4 + 3*p1)*(4 + 3*p2)*r_rc2^2) + (x[,"n_13"] + x[,"n_31"])*log(-3*p1*(-8 + 3*p2 + 12*r_rc2) + 4*(8 + p2*(-3 + 9*r_rc2)))


logL_rc3 <- function(r,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-5*n10 - 5*n11 - 5*n12 - 5*n13 - 5*n14 - 5*n20 - 4*n21 - 4*n22 - 4*n23 - 5*n24 - 5*n30 - 5*n31 - 5*n32 - 5*n33 - 5*n34)*log(2) + 2*(-n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(3) + (n21 + n23)*log(16 - 6*p2 + p1*(3 + 9*p2)) + (n13 + n31)*log(4*(4 + p2*(3 - 9*r)) + 3*p1*(4 + 3*p2 - 6*r)) + (n10 + n14 + n30 + n34)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n20 + n24)*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r + 2*r^2)) + (n12 + n32)*log((8 - 3*p1)*(4 + 3*p2) - (4 + 3*p1)*(-2 + 3*p2)*(-1 + r)^2 + (4 + 3*p1)*(2 - 3*p2)*r^2) + (n11 + n33)*log(3*p1*(-2 + 3*p2 + 6*r) + 4*(4 + p2*(-6 + 9*r))) + n22*log(8*(4 + 2*r - 2*r^2 + 3*p2*(1 - r + r^2)) + 3*p1*(-4*(1 - r + r^2) + p2*(-3 - 6*r + 6*r^2)))
return(L)}
interlogL_rc3 <- function(n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rc3,c(0,0.5), n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc3 <- parallel::mcmapply(interlogL_rc3,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rc3 <- -((x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2))) - (x[,"n_20"] + x[,"n_24"])*(-log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - x[,"n_22"]*log10(3*p1*(-3 - (9*p2)/2) + 8*(9/2 + (9*p2)/4)) - (x[,"n_13"] + x[,"n_31"])*log10(4*(4 - (3*p2)/2) + 3*p1*(1 + 3*p2)) - (x[,"n_11"] + x[,"n_33"])*log10(4*(4 - (3*p2)/2) + 3*p1*(1 + 3*p2)) - (x[,"n_12"] + x[,"n_32"])*log10(((4 + 3*p1)*(2 - 3*p2))/4 - ((4 + 3*p1)*(-2 + 3*p2))/4 + (8 - 3*p1)*(4 + 3*p2)) + (x[,"n_13"] + x[,"n_31"])*log10(4*(4 + p2*(3 - 9*r_rc3)) + 3*p1*(4 + 3*p2 - 6*r_rc3)) + (x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rc3)) + log10(pmax(1e-6,r_rc3))) + (x[,"n_20"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(1 - 2*r_rc3 + 2*r_rc3^2)) + (x[,"n_12"] + x[,"n_32"])*log10((8 - 3*p1)*(4 + 3*p2) - (4 + 3*p1)*(-2 + 3*p2)*(-1 + r_rc3)^2 + (4 + 3*p1)*(2 - 3*p2)*r_rc3^2) + (x[,"n_11"] + x[,"n_33"])*log10(3*p1*(-2 + 3*p2 + 6*r_rc3) + 4*(4 + p2*(-6 + 9*r_rc3))) + x[,"n_22"]*log10(8*(4 + 2*r_rc3 - 2*r_rc3^2 + 3*p2*(1 - r_rc3 + r_rc3^2)) + 3*p1*(-4*(1 - r_rc3 + r_rc3^2) + p2*(-3 - 6*r_rc3 + 6*r_rc3^2)))


logL_rc3 <- (-5*x[,"n_10"] - 5*x[,"n_11"] - 5*x[,"n_12"] - 5*x[,"n_13"] - 5*x[,"n_14"] - 5*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 4*x[,"n_23"] - 5*x[,"n_24"] - 5*x[,"n_30"] - 5*x[,"n_31"] - 5*x[,"n_32"] - 5*x[,"n_33"] - 5*x[,"n_34"])*log(2) + 2*(-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(3) + (x[,"n_21"] + x[,"n_23"])*log(16 - 6*p2 + p1*(3 + 9*p2)) + (x[,"n_13"] + x[,"n_31"])*log(4*(4 + p2*(3 - 9*r_rc3)) + 3*p1*(4 + 3*p2 - 6*r_rc3)) + (x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rc3)) + log(pmax(1e-6,r_rc3))) + (x[,"n_20"] + x[,"n_24"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r_rc3 + 2*r_rc3^2)) + (x[,"n_12"] + x[,"n_32"])*log((8 - 3*p1)*(4 + 3*p2) - (4 + 3*p1)*(-2 + 3*p2)*(-1 + r_rc3)^2 + (4 + 3*p1)*(2 - 3*p2)*r_rc3^2) + (x[,"n_11"] + x[,"n_33"])*log(3*p1*(-2 + 3*p2 + 6*r_rc3) + 4*(4 + p2*(-6 + 9*r_rc3))) + x[,"n_22"]*log(8*(4 + 2*r_rc3 - 2*r_rc3^2 + 3*p2*(1 - r_rc3 + r_rc3^2)) + 3*p1*(-4*(1 - r_rc3 + r_rc3^2) + p2*(-3 - 6*r_rc3 + 6*r_rc3^2)))


logL_rc4 <- function(r,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-6*n10 - 6*n11 - 6*n12 - 6*n13 - 6*n14 - 6*n20 - 5*n21 - 5*n22 - 5*n23 - 6*n24 - 6*n30 - 6*n31 - 6*n32 - 6*n33 - 6*n34)*log(2) + 2*(-n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(3) + (n21 + n23)*log(32 + p1*(6 - 9*p2) + 6*p2) + (n11 + n33)*log(4*(8 + p2*(6 - 9*r)) - 3*p1*(4 + 3*p2 - 12*r)) + (n10 + n14 + n30 + n34)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + n22*log((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r)*r) + (n20 + n24)*(log(4 + 3*p1) + log(4 + 3*p2) + log((-1 + r)^2 + r^2)) + (n12 + n32)*log((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r)^2 + (4 + 3*p1)*(4 + 3*p2)*r^2) + (n13 + n31)*log(-3*p1*(-8 + 3*p2 + 12*r) + 4*(8 + p2*(-3 + 9*r)))
return(L)}
interlogL_rc4 <- function(n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rc4,c(0,0.5), n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc4 <- parallel::mcmapply(interlogL_rc4,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rc4 <- -((x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2))) - (x[,"n_20"] + x[,"n_24"])*(-log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_13"] + x[,"n_31"])*log10(4*(8 + (3*p2)/2) - 3*p1*(-2 + 3*p2)) - (x[,"n_11"] + x[,"n_33"])*log10(4*(8 + (3*p2)/2) - 3*p1*(-2 + 3*p2)) - x[,"n_22"]*log10((-8 + 3*p1)*(-8 + 3*p2) + ((4 + 3*p1)*(4 + 3*p2))/2) - (x[,"n_12"] + x[,"n_32"])*log10((-8 + 3*p1)*(-8 + 3*p2) + ((4 + 3*p1)*(4 + 3*p2))/2) + (x[,"n_11"] + x[,"n_33"])*log10(4*(8 + p2*(6 - 9*r_rc4)) - 3*p1*(4 + 3*p2 - 12*r_rc4)) + (x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rc4)) + log10(pmax(1e-6,r_rc4))) + x[,"n_22"]*log10((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_rc4)*r_rc4) + (x[,"n_20"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10((-1 + r_rc4)^2 + r_rc4^2)) + (x[,"n_12"] + x[,"n_32"])*log10((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r_rc4)^2 + (4 + 3*p1)*(4 + 3*p2)*r_rc4^2) + (x[,"n_13"] + x[,"n_31"])*log10(-3*p1*(-8 + 3*p2 + 12*r_rc4) + 4*(8 + p2*(-3 + 9*r_rc4)))


logL_rc4 <- (-6*x[,"n_10"] - 6*x[,"n_11"] - 6*x[,"n_12"] - 6*x[,"n_13"] - 6*x[,"n_14"] - 6*x[,"n_20"] - 5*x[,"n_21"] - 5*x[,"n_22"] - 5*x[,"n_23"] - 6*x[,"n_24"] - 6*x[,"n_30"] - 6*x[,"n_31"] - 6*x[,"n_32"] - 6*x[,"n_33"] - 6*x[,"n_34"])*log(2) + 2*(-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(3) + (x[,"n_21"] + x[,"n_23"])*log(32 + p1*(6 - 9*p2) + 6*p2) + (x[,"n_11"] + x[,"n_33"])*log(4*(8 + p2*(6 - 9*r_rc4)) - 3*p1*(4 + 3*p2 - 12*r_rc4)) + (x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rc4)) + log(pmax(1e-6,r_rc4))) + x[,"n_22"]*log((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_rc4)*r_rc4) + (x[,"n_20"] + x[,"n_24"])*(log(4 + 3*p1) + log(4 + 3*p2) + log((-1 + r_rc4)^2 + r_rc4^2)) + (x[,"n_12"] + x[,"n_32"])*log((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r_rc4)^2 + (4 + 3*p1)*(4 + 3*p2)*r_rc4^2) + (x[,"n_13"] + x[,"n_31"])*log(-3*p1*(-8 + 3*p2 + 12*r_rc4) + 4*(8 + p2*(-3 + 9*r_rc4)))


logL_rm1 <- function(r,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-6*n10 - 5*n11 - 6*n12 - 5*n13 - 6*n14 - 5*n20 - 5*n21 - 5*n22 - 5*n23 - 5*n24 - 6*n30 - 5*n31 - 6*n32 - 5*n33 - 6*n34)*log(2) + 2*(-n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(3) + (n21 + n23)*log(32 + p1*(6 - 9*p2) + 6*p2) + (n13 + n31)*(log(32 + 6*p2 - p1*(-6 + 9*p2)) + log(pmax(1e-6,1 - r))) + (n11 + n33)*(log(32 + p1*(6 - 9*p2) + 6*p2) + log(pmax(1e-6,r))) + (n20 + n24)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n34)*(log(4 + 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r))) + (n12 + n32)*log((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r)*r) + (n14 + n30)*(log(4 + 3*p1) + log(4 + 3*p2) + log(1 - 2*r + r^2)) + n22*log((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r)^2 + (4 + 3*p1)*(4 + 3*p2)*r^2)
return(L)}
interlogL_rm1 <- function(n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rm1,c(0,0.5), n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rm1 <- parallel::mcmapply(interlogL_rm1,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rm1 <- -((x[,"n_20"] + x[,"n_24"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2))) - (x[,"n_14"] + x[,"n_30"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_34"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_11"] + x[,"n_33"])*(-log10(2) + log10(32 + p1*(6 - 9*p2) + 6*p2)) - x[,"n_22"]*log10((-8 + 3*p1)*(-8 + 3*p2) + ((4 + 3*p1)*(4 + 3*p2))/2) - (x[,"n_12"] + x[,"n_32"])*log10((-8 + 3*p1)*(-8 + 3*p2) + ((4 + 3*p1)*(4 + 3*p2))/2) - (x[,"n_13"] + x[,"n_31"])*(-log10(2) + log10(32 + 6*p2 - p1*(-6 + 9*p2))) + (x[,"n_13"] + x[,"n_31"])*(log10(32 + 6*p2 - p1*(-6 + 9*p2)) + log10(pmax(1e-6,1 - r_rm1))) + (x[,"n_11"] + x[,"n_33"])*(log10(32 + p1*(6 - 9*p2) + 6*p2) + log10(pmax(1e-6,r_rm1))) + (x[,"n_20"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rm1)) + log10(pmax(1e-6,r_rm1))) + (x[,"n_10"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_rm1))) + (x[,"n_12"] + x[,"n_32"])*log10((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_rm1)*r_rm1) + (x[,"n_14"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(1 - 2*r_rm1 + r_rm1^2)) + x[,"n_22"]*log10((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r_rm1)^2 + (4 + 3*p1)*(4 + 3*p2)*r_rm1^2)


logL_rm1 <- (-6*x[,"n_10"] - 5*x[,"n_11"] - 6*x[,"n_12"] - 5*x[,"n_13"] - 6*x[,"n_14"] - 5*x[,"n_20"] - 5*x[,"n_21"] - 5*x[,"n_22"] - 5*x[,"n_23"] - 5*x[,"n_24"] - 6*x[,"n_30"] - 5*x[,"n_31"] - 6*x[,"n_32"] - 5*x[,"n_33"] - 6*x[,"n_34"])*log(2) + 2*(-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(3) + (x[,"n_21"] + x[,"n_23"])*log(32 + p1*(6 - 9*p2) + 6*p2) + (x[,"n_13"] + x[,"n_31"])*(log(32 + 6*p2 - p1*(-6 + 9*p2)) + log(pmax(1e-6,1 - r_rm1))) + (x[,"n_11"] + x[,"n_33"])*(log(32 + p1*(6 - 9*p2) + 6*p2) + log(pmax(1e-6,r_rm1))) + (x[,"n_20"] + x[,"n_24"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rm1)) + log(pmax(1e-6,r_rm1))) + (x[,"n_10"] + x[,"n_34"])*(log(4 + 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r_rm1))) + (x[,"n_12"] + x[,"n_32"])*log((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_rm1)*r_rm1) + (x[,"n_14"] + x[,"n_30"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(1 - 2*r_rm1 + r_rm1^2)) + x[,"n_22"]*log((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r_rm1)^2 + (4 + 3*p1)*(4 + 3*p2)*r_rm1^2)


logL_rm2 <- function(r,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-5*n10 - 4*n11 - 5*n12 - 4*n13 - 5*n14 - 4*n20 - 4*n21 - 3*n22 - 4*n23 - 4*n24 - 5*n30 - 4*n31 - 5*n32 - 4*n33 - 5*n34)*log(2) + 2*(-n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(3) + (n21 + n23)*log(16 - 6*p2 + p1*(3 + 9*p2)) + (n13 + n31)*(log(16 - 6*p2 + p1*(3 + 9*p2)) + log(pmax(1e-6,1 - r))) + (n11 + n33)*(log(16 - 6*p2 + p1*(3 + 9*p2)) + log(pmax(1e-6,r))) + (n20 + n24)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n34)*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r))) + (n12 + n32)*log((8 - 3*p1)*(4 + 3*p2) + 2*(4 + 3*p1)*(-2 + 3*p2)*(-1 + r)*r) + (n14 + n30)*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r + r^2)) + n22*log(20 - 8*r + 8*r^2 + p2*(6 + 12*r - 12*r^2) - 3*p1*(1 + 2*r - 2*r^2 + 3*p2*(1 - r + r^2)))
return(L)}
interlogL_rm2 <- function(n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rm2,c(0,0.5), n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rm2 <- parallel::mcmapply(interlogL_rm2,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rm2 <- -((x[,"n_20"] + x[,"n_24"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2))) - (x[,"n_14"] + x[,"n_30"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_34"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - x[,"n_22"]*log10(18 + 9*p2 - 3*p1*(3/2 + (9*p2)/4)) - (x[,"n_12"] + x[,"n_32"])*log10(-((4 + 3*p1)*(-2 + 3*p2))/2 + (8 - 3*p1)*(4 + 3*p2)) - (x[,"n_13"] + x[,"n_31"])*(-log10(2) + log10(16 - 6*p2 + p1*(3 + 9*p2))) - (x[,"n_11"] + x[,"n_33"])*(-log10(2) + log10(16 - 6*p2 + p1*(3 + 9*p2))) + (x[,"n_13"] + x[,"n_31"])*(log10(16 - 6*p2 + p1*(3 + 9*p2)) + log10(pmax(1e-6,1 - r_rm2))) + (x[,"n_11"] + x[,"n_33"])*(log10(16 - 6*p2 + p1*(3 + 9*p2)) + log10(pmax(1e-6,r_rm2))) + (x[,"n_20"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rm2)) + log10(pmax(1e-6,r_rm2))) + (x[,"n_10"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_rm2))) + (x[,"n_12"] + x[,"n_32"])*log10((8 - 3*p1)*(4 + 3*p2) + 2*(4 + 3*p1)*(-2 + 3*p2)*(-1 + r_rm2)*r_rm2) + (x[,"n_14"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(1 - 2*r_rm2 + r_rm2^2)) + x[,"n_22"]*log10(20 - 8*r_rm2 + 8*r_rm2^2 + p2*(6 + 12*r_rm2 - 12*r_rm2^2) - 3*p1*(1 + 2*r_rm2 - 2*r_rm2^2 + 3*p2*(1 - r_rm2 + r_rm2^2)))


logL_rm2 <- (-5*x[,"n_10"] - 4*x[,"n_11"] - 5*x[,"n_12"] - 4*x[,"n_13"] - 5*x[,"n_14"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 3*x[,"n_22"] - 4*x[,"n_23"] - 4*x[,"n_24"] - 5*x[,"n_30"] - 4*x[,"n_31"] - 5*x[,"n_32"] - 4*x[,"n_33"] - 5*x[,"n_34"])*log(2) + 2*(-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(3) + (x[,"n_21"] + x[,"n_23"])*log(16 - 6*p2 + p1*(3 + 9*p2)) + (x[,"n_13"] + x[,"n_31"])*(log(16 - 6*p2 + p1*(3 + 9*p2)) + log(pmax(1e-6,1 - r_rm2))) + (x[,"n_11"] + x[,"n_33"])*(log(16 - 6*p2 + p1*(3 + 9*p2)) + log(pmax(1e-6,r_rm2))) + (x[,"n_20"] + x[,"n_24"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rm2)) + log(pmax(1e-6,r_rm2))) + (x[,"n_10"] + x[,"n_34"])*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r_rm2))) + (x[,"n_12"] + x[,"n_32"])*log((8 - 3*p1)*(4 + 3*p2) + 2*(4 + 3*p1)*(-2 + 3*p2)*(-1 + r_rm2)*r_rm2) + (x[,"n_14"] + x[,"n_30"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r_rm2 + r_rm2^2)) + x[,"n_22"]*log(20 - 8*r_rm2 + 8*r_rm2^2 + p2*(6 + 12*r_rm2 - 12*r_rm2^2) - 3*p1*(1 + 2*r_rm2 - 2*r_rm2^2 + 3*p2*(1 - r_rm2 + r_rm2^2)))


logL_rm3 <- function(r,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-6*n10 - 5*n11 - 6*n12 - 5*n13 - 6*n14 - 5*n20 - 5*n21 - 5*n22 - 5*n23 - 5*n24 - 6*n30 - 5*n31 - 6*n32 - 5*n33 - 6*n34)*log(2) + 2*(-n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(3) + (n21 + n23)*log(32 + p1*(6 - 9*p2) + 6*p2) + (n13 + n31)*(log(32 + 6*p2 - p1*(-6 + 9*p2)) + log(pmax(1e-6,1 - r))) + (n11 + n33)*(log(32 + p1*(6 - 9*p2) + 6*p2) + log(pmax(1e-6,r))) + (n20 + n24)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n34)*(log(4 + 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r))) + (n12 + n32)*log((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r)*r) + (n14 + n30)*(log(4 + 3*p1) + log(4 + 3*p2) + log(1 - 2*r + r^2)) + n22*log((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r)^2 + (4 + 3*p1)*(4 + 3*p2)*r^2)
return(L)}
interlogL_rm3 <- function(n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rm3,c(0,0.5), n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rm3 <- parallel::mcmapply(interlogL_rm3,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rm3 <- -((x[,"n_20"] + x[,"n_24"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2))) - (x[,"n_14"] + x[,"n_30"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_34"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_11"] + x[,"n_33"])*(-log10(2) + log10(32 + p1*(6 - 9*p2) + 6*p2)) - x[,"n_22"]*log10((-8 + 3*p1)*(-8 + 3*p2) + ((4 + 3*p1)*(4 + 3*p2))/2) - (x[,"n_12"] + x[,"n_32"])*log10((-8 + 3*p1)*(-8 + 3*p2) + ((4 + 3*p1)*(4 + 3*p2))/2) - (x[,"n_13"] + x[,"n_31"])*(-log10(2) + log10(32 + 6*p2 - p1*(-6 + 9*p2))) + (x[,"n_13"] + x[,"n_31"])*(log10(32 + 6*p2 - p1*(-6 + 9*p2)) + log10(pmax(1e-6,1 - r_rm3))) + (x[,"n_11"] + x[,"n_33"])*(log10(32 + p1*(6 - 9*p2) + 6*p2) + log10(pmax(1e-6,r_rm3))) + (x[,"n_20"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rm3)) + log10(pmax(1e-6,r_rm3))) + (x[,"n_10"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_rm3))) + (x[,"n_12"] + x[,"n_32"])*log10((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_rm3)*r_rm3) + (x[,"n_14"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(1 - 2*r_rm3 + r_rm3^2)) + x[,"n_22"]*log10((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r_rm3)^2 + (4 + 3*p1)*(4 + 3*p2)*r_rm3^2)


logL_rm3 <- (-6*x[,"n_10"] - 5*x[,"n_11"] - 6*x[,"n_12"] - 5*x[,"n_13"] - 6*x[,"n_14"] - 5*x[,"n_20"] - 5*x[,"n_21"] - 5*x[,"n_22"] - 5*x[,"n_23"] - 5*x[,"n_24"] - 6*x[,"n_30"] - 5*x[,"n_31"] - 6*x[,"n_32"] - 5*x[,"n_33"] - 6*x[,"n_34"])*log(2) + 2*(-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(3) + (x[,"n_21"] + x[,"n_23"])*log(32 + p1*(6 - 9*p2) + 6*p2) + (x[,"n_13"] + x[,"n_31"])*(log(32 + 6*p2 - p1*(-6 + 9*p2)) + log(pmax(1e-6,1 - r_rm3))) + (x[,"n_11"] + x[,"n_33"])*(log(32 + p1*(6 - 9*p2) + 6*p2) + log(pmax(1e-6,r_rm3))) + (x[,"n_20"] + x[,"n_24"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rm3)) + log(pmax(1e-6,r_rm3))) + (x[,"n_10"] + x[,"n_34"])*(log(4 + 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r_rm3))) + (x[,"n_12"] + x[,"n_32"])*log((-8 + 3*p1)*(-8 + 3*p2) - 2*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_rm3)*r_rm3) + (x[,"n_14"] + x[,"n_30"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(1 - 2*r_rm3 + r_rm3^2)) + x[,"n_22"]*log((-8 + 3*p1)*(-8 + 3*p2) + (4 + 3*p1)*(4 + 3*p2)*(-1 + r_rm3)^2 + (4 + 3*p1)*(4 + 3*p2)*r_rm3^2)


logL_rm4 <- function(r,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-5*n10 - 4*n11 - 5*n12 - 4*n13 - 5*n14 - 4*n20 - 4*n21 - 3*n22 - 4*n23 - 4*n24 - 5*n30 - 4*n31 - 5*n32 - 4*n33 - 5*n34)*log(2) + 2*(-n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(3) + (n21 + n23)*log(16 - 6*p2 + p1*(3 + 9*p2)) + (n13 + n31)*(log(16 - 6*p2 + p1*(3 + 9*p2)) + log(pmax(1e-6,1 - r))) + (n11 + n33)*(log(16 - 6*p2 + p1*(3 + 9*p2)) + log(pmax(1e-6,r))) + (n20 + n24)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n34)*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r))) + (n12 + n32)*log((8 - 3*p1)*(4 + 3*p2) + 2*(4 + 3*p1)*(-2 + 3*p2)*(-1 + r)*r) + (n14 + n30)*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r + r^2)) + n22*log(20 - 8*r + 8*r^2 + p2*(6 + 12*r - 12*r^2) - 3*p1*(1 + 2*r - 2*r^2 + 3*p2*(1 - r + r^2)))
return(L)}
interlogL_rm4 <- function(n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rm4,c(0,0.5), n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rm4 <- parallel::mcmapply(interlogL_rm4,x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rm4 <- -((x[,"n_20"] + x[,"n_24"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2))) - (x[,"n_14"] + x[,"n_30"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_34"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - x[,"n_22"]*log10(18 + 9*p2 - 3*p1*(3/2 + (9*p2)/4)) - (x[,"n_12"] + x[,"n_32"])*log10(-((4 + 3*p1)*(-2 + 3*p2))/2 + (8 - 3*p1)*(4 + 3*p2)) - (x[,"n_13"] + x[,"n_31"])*(-log10(2) + log10(16 - 6*p2 + p1*(3 + 9*p2))) - (x[,"n_11"] + x[,"n_33"])*(-log10(2) + log10(16 - 6*p2 + p1*(3 + 9*p2))) + (x[,"n_13"] + x[,"n_31"])*(log10(16 - 6*p2 + p1*(3 + 9*p2)) + log10(pmax(1e-6,1 - r_rm4))) + (x[,"n_11"] + x[,"n_33"])*(log10(16 - 6*p2 + p1*(3 + 9*p2)) + log10(pmax(1e-6,r_rm4))) + (x[,"n_20"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rm4)) + log10(pmax(1e-6,r_rm4))) + (x[,"n_10"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_rm4))) + (x[,"n_12"] + x[,"n_32"])*log10((8 - 3*p1)*(4 + 3*p2) + 2*(4 + 3*p1)*(-2 + 3*p2)*(-1 + r_rm4)*r_rm4) + (x[,"n_14"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(1 - 2*r_rm4 + r_rm4^2)) + x[,"n_22"]*log10(20 - 8*r_rm4 + 8*r_rm4^2 + p2*(6 + 12*r_rm4 - 12*r_rm4^2) - 3*p1*(1 + 2*r_rm4 - 2*r_rm4^2 + 3*p2*(1 - r_rm4 + r_rm4^2)))


logL_rm4 <- (-5*x[,"n_10"] - 4*x[,"n_11"] - 5*x[,"n_12"] - 4*x[,"n_13"] - 5*x[,"n_14"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 3*x[,"n_22"] - 4*x[,"n_23"] - 4*x[,"n_24"] - 5*x[,"n_30"] - 4*x[,"n_31"] - 5*x[,"n_32"] - 4*x[,"n_33"] - 5*x[,"n_34"])*log(2) + 2*(-x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(3) + (x[,"n_21"] + x[,"n_23"])*log(16 - 6*p2 + p1*(3 + 9*p2)) + (x[,"n_13"] + x[,"n_31"])*(log(16 - 6*p2 + p1*(3 + 9*p2)) + log(pmax(1e-6,1 - r_rm4))) + (x[,"n_11"] + x[,"n_33"])*(log(16 - 6*p2 + p1*(3 + 9*p2)) + log(pmax(1e-6,r_rm4))) + (x[,"n_20"] + x[,"n_24"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rm4)) + log(pmax(1e-6,r_rm4))) + (x[,"n_10"] + x[,"n_34"])*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r_rm4))) + (x[,"n_12"] + x[,"n_32"])*log((8 - 3*p1)*(4 + 3*p2) + 2*(4 + 3*p1)*(-2 + 3*p2)*(-1 + r_rm4)*r_rm4) + (x[,"n_14"] + x[,"n_30"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r_rm4 + r_rm4^2)) + x[,"n_22"]*log(20 - 8*r_rm4 + 8*r_rm4^2 + p2*(6 + 12*r_rm4 - 12*r_rm4^2) - 3*p1*(1 + 2*r_rm4 - 2*r_rm4^2 + 3*p2*(1 - r_rm4 + r_rm4^2)))


return(list(
r_mat = cbind(r_cc1,r_cc2,r_cc3,r_cc4,r_cm1,r_cm2,r_cm3,r_cm4,r_rc1,r_rc2,r_rc3,r_rc4,r_rm1,r_rm2,r_rm3,r_rm4,0.499),
LOD_mat = cbind(LOD_cc1,LOD_cc2,LOD_cc3,LOD_cc4,LOD_cm1,LOD_cm2,LOD_cm3,LOD_cm4,LOD_rc1,LOD_rc2,LOD_rc3,LOD_rc4,LOD_rm1,LOD_rm2,LOD_rm3,LOD_rm4,0),
logL_mat = cbind(logL_cc1,logL_cc2,logL_cc3,logL_cc4,logL_cm1,logL_cm2,logL_cm3,logL_cm4,logL_rc1,logL_rc2,logL_rc3,logL_rc4,logL_rm1,logL_rm2,logL_rm3,logL_rm4,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling coupling","coupling coupling","coupling coupling","coupling mixed","coupling mixed","coupling mixed","coupling mixed","repulsion coupling","repulsion coupling","repulsion coupling","repulsion coupling","repulsion mixed","repulsion mixed","repulsion mixed","repulsion mixed","unknown")
)
)
}

#' @rdname p4_functions
p4_2.0_1.1 <- function(x,p1,p2,ncores=1){


r_c1 <- (x[,"n_02"] + x[,"n_20"])/(x[,"n_00"] + x[,"n_02"] + x[,"n_20"] + x[,"n_22"])
logL_c1 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"])*(3*log(2) + log(3)) + (x[,"n_01"] + x[,"n_21"])*log(2 - 3*p1) + (x[,"n_10"] + x[,"n_12"])*log(1/6 + p1/8) + x[,"n_11"]*log(1/3 + p1/4) + (x[,"n_00"] + x[,"n_22"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_c1))) + (x[,"n_02"] + x[,"n_20"])*(log(2 - 3*p1) + log(pmax(1e-6,r_c1)))
LOD_c1 <- -((x[,"n_02"] + x[,"n_20"])*(-log10(2) + log10(2 - 3*p1))) - (x[,"n_00"] + x[,"n_22"])*(-log10(2) + log10(2 - 3*p1)) + (x[,"n_00"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_c1))) + (x[,"n_02"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(pmax(1e-6,r_c1)))


r_c2 <- (x[,"n_02"] + x[,"n_20"])/(x[,"n_00"] + x[,"n_02"] + x[,"n_20"] + x[,"n_22"])
logL_c2 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_12"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"])*(4*log(2) + log(3)) + (x[,"n_10"] + x[,"n_12"])*log(8 - 3*p1) + x[,"n_11"]*log(1/3 - p1/8) + (x[,"n_01"] + x[,"n_21"])*log(4 + 3*p1) + (x[,"n_00"] + x[,"n_22"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_c2))) + (x[,"n_02"] + x[,"n_20"])*(log(4 + 3*p1) + log(pmax(1e-6,r_c2)))
LOD_c2 <- -((x[,"n_02"] + x[,"n_20"])*(-log10(2) + log10(4 + 3*p1))) - (x[,"n_00"] + x[,"n_22"])*(-log10(2) + log10(4 + 3*p1)) + (x[,"n_00"] + x[,"n_22"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_c2))) + (x[,"n_02"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(pmax(1e-6,r_c2)))


r_r1 <- (x[,"n_00"] + x[,"n_22"])/(x[,"n_00"] + x[,"n_02"] + x[,"n_20"] + x[,"n_22"])
logL_r1 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"])*(3*log(2) + log(3)) + (x[,"n_01"] + x[,"n_21"])*log(2 - 3*p1) + (x[,"n_10"] + x[,"n_12"])*log(1/6 + p1/8) + x[,"n_11"]*log(1/3 + p1/4) + (x[,"n_02"] + x[,"n_20"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_r1))) + (x[,"n_00"] + x[,"n_22"])*(log(2 - 3*p1) + log(pmax(1e-6,r_r1)))
LOD_r1 <- -((x[,"n_02"] + x[,"n_20"])*(-log10(2) + log10(2 - 3*p1))) - (x[,"n_00"] + x[,"n_22"])*(-log10(2) + log10(2 - 3*p1)) + (x[,"n_02"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_r1))) + (x[,"n_00"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(pmax(1e-6,r_r1)))


r_r2 <- (x[,"n_02"] + x[,"n_20"])/(x[,"n_00"] + x[,"n_02"] + x[,"n_20"] + x[,"n_22"])
logL_r2 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_12"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"])*(4*log(2) + log(3)) + (x[,"n_10"] + x[,"n_12"])*log(8 - 3*p1) + x[,"n_11"]*log(1/3 - p1/8) + (x[,"n_01"] + x[,"n_21"])*log(4 + 3*p1) + (x[,"n_00"] + x[,"n_22"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_r2))) + (x[,"n_02"] + x[,"n_20"])*(log(4 + 3*p1) + log(pmax(1e-6,r_r2)))
LOD_r2 <- -((x[,"n_02"] + x[,"n_20"])*(-log10(2) + log10(4 + 3*p1))) - (x[,"n_00"] + x[,"n_22"])*(-log10(2) + log10(4 + 3*p1)) + (x[,"n_00"] + x[,"n_22"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_r2))) + (x[,"n_02"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(pmax(1e-6,r_r2)))


return(list(
r_mat = cbind(r_c1,r_c2,r_r1,r_r2,0.499),
LOD_mat = cbind(LOD_c1,LOD_c2,LOD_r1,LOD_r2,0),
logL_mat = NULL,
phasing_strategy = "MINR",
possible_phases = c("coupling","coupling","repulsion","repulsion","unknown")
)
)
}

#' @rdname p4_functions
p4_2.0_1.2 <- function(x,p1,p2,ncores=1){


logL_c1 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*(4*log(2) + 2*log(3)) + (n10 + n13)*(log(4 + 3*p1) + log(2 - 3*p2)) + (n11 + n12)*(log(4 + 3*p1) + log(10 + 3*p2)) + (n00 + n23)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r))) + (n03 + n20)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r))) + (n01 + n22)*(log(2 - 3*p1) + log(8 - 6*r - p2*(-6 + 9*r))) + (n02 + n21)*(log(2 - 3*p1) + log(2 + 6*r + p2*(-3 + 9*r)))
return(L)}
interlogL_c1 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
optimize(logL_c1,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_c1 <- parallel::mcmapply(interlogL_c1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_c1 <- -((x[,"n_03"] + x[,"n_20"])*(-log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_23"])*(-log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_21"])*(log10(2 - 3*p1) + log10(5 + (3*p2)/2)) - (x[,"n_01"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(5 + (3*p2)/2)) + (x[,"n_00"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_c1))) + (x[,"n_03"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_c1))) + (x[,"n_01"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(8 - 6*r_c1 - p2*(-6 + 9*r_c1))) + (x[,"n_02"] + x[,"n_21"])*(log10(2 - 3*p1) + log10(2 + 6*r_c1 + p2*(-3 + 9*r_c1)))


logL_c2 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*(5*log(2) + 2*log(3)) + (n11 + n12)*(log(4 + 3*p1) + log(20 - 3*p2)) + (n10 + n13)*(log(4 + 3*p1) + log(4 + 3*p2)) + (n00 + n23)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r))) + (n03 + n20)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r))) + (n02 + n21)*(log(2 - 3*p1) + log(4 + 3*p2 + 12*r - 9*p2*r)) + (n01 + n22)*(log(2 - 3*p1) + log(16 - 12*r + p2*(-6 + 9*r)))
return(L)}
interlogL_c2 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
optimize(logL_c2,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_c2 <- parallel::mcmapply(interlogL_c2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_c2 <- -((x[,"n_02"] + x[,"n_21"])*(log10(2 - 3*p1) + log10(10 - (3*p2)/2))) - (x[,"n_01"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(10 - (3*p2)/2)) - (x[,"n_03"] + x[,"n_20"])*(-log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_23"])*(-log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) + (x[,"n_00"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_c2))) + (x[,"n_03"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_c2))) + (x[,"n_02"] + x[,"n_21"])*(log10(2 - 3*p1) + log10(4 + 3*p2 + 12*r_c2 - 9*p2*r_c2)) + (x[,"n_01"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(16 - 12*r_c2 + p2*(-6 + 9*r_c2)))


logL_c3 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*(5*log(2) + 2*log(3)) + (n10 + n13)*(log(8 - 3*p1) + log(2 - 3*p2)) + (n11 + n12)*(log(8 - 3*p1) + log(10 + 3*p2)) + (n00 + n23)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r))) + (n03 + n20)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r))) + (n01 + n22)*(log(4 + 3*p1) + log(8 - 6*r - p2*(-6 + 9*r))) + (n02 + n21)*(log(4 + 3*p1) + log(2 + 6*r + p2*(-3 + 9*r)))
return(L)}
interlogL_c3 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
optimize(logL_c3,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_c3 <- parallel::mcmapply(interlogL_c3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_c3 <- -((x[,"n_03"] + x[,"n_20"])*(-log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_23"])*(-log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_21"])*(log10(4 + 3*p1) + log10(5 + (3*p2)/2)) - (x[,"n_01"] + x[,"n_22"])*(log10(4 + 3*p1) + log10(5 + (3*p2)/2)) + (x[,"n_00"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_c3))) + (x[,"n_03"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_c3))) + (x[,"n_01"] + x[,"n_22"])*(log10(4 + 3*p1) + log10(8 - 6*r_c3 - p2*(-6 + 9*r_c3))) + (x[,"n_02"] + x[,"n_21"])*(log10(4 + 3*p1) + log10(2 + 6*r_c3 + p2*(-3 + 9*r_c3)))


logL_c4 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*(6*log(2) + 2*log(3)) + (n11 + n12)*(log(8 - 3*p1) + log(20 - 3*p2)) + (n10 + n13)*(log(8 - 3*p1) + log(4 + 3*p2)) + (n00 + n23)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r))) + (n03 + n20)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r))) + (n02 + n21)*(log(4 + 3*p1) + log(4 + 3*p2 + 12*r - 9*p2*r)) + (n01 + n22)*(log(4 + 3*p1) + log(16 - 12*r + p2*(-6 + 9*r)))
return(L)}
interlogL_c4 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
optimize(logL_c4,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_c4 <- parallel::mcmapply(interlogL_c4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_c4 <- -((x[,"n_02"] + x[,"n_21"])*(log10(4 + 3*p1) + log10(10 - (3*p2)/2))) - (x[,"n_01"] + x[,"n_22"])*(log10(4 + 3*p1) + log10(10 - (3*p2)/2)) - (x[,"n_03"] + x[,"n_20"])*(-log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_23"])*(-log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) + (x[,"n_00"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_c4))) + (x[,"n_03"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_c4))) + (x[,"n_02"] + x[,"n_21"])*(log10(4 + 3*p1) + log10(4 + 3*p2 + 12*r_c4 - 9*p2*r_c4)) + (x[,"n_01"] + x[,"n_22"])*(log10(4 + 3*p1) + log10(16 - 12*r_c4 + p2*(-6 + 9*r_c4)))


logL_r1 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*(4*log(2) + 2*log(3)) + (n10 + n13)*(log(4 + 3*p1) + log(2 - 3*p2)) + (n11 + n12)*(log(4 + 3*p1) + log(10 + 3*p2)) + (n03 + n20)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r))) + (n00 + n23)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r))) + (n02 + n21)*(log(2 - 3*p1) + log(8 - 6*r - p2*(-6 + 9*r))) + (n01 + n22)*(log(2 - 3*p1) + log(2 + 6*r + p2*(-3 + 9*r)))
return(L)}
interlogL_r1 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
optimize(logL_r1,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_r1 <- parallel::mcmapply(interlogL_r1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_r1 <- -((x[,"n_03"] + x[,"n_20"])*(-log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_23"])*(-log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_21"])*(log10(2 - 3*p1) + log10(5 + (3*p2)/2)) - (x[,"n_01"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(5 + (3*p2)/2)) + (x[,"n_03"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_r1))) + (x[,"n_00"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_r1))) + (x[,"n_02"] + x[,"n_21"])*(log10(2 - 3*p1) + log10(8 - 6*r_r1 - p2*(-6 + 9*r_r1))) + (x[,"n_01"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(2 + 6*r_r1 + p2*(-3 + 9*r_r1)))


logL_r2 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*(5*log(2) + 2*log(3)) + (n11 + n12)*(log(4 + 3*p1) + log(20 - 3*p2)) + (n10 + n13)*(log(4 + 3*p1) + log(4 + 3*p2)) + (n03 + n20)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r))) + (n00 + n23)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r))) + (n01 + n22)*(log(2 - 3*p1) + log(4 + 3*p2 + 12*r - 9*p2*r)) + (n02 + n21)*(log(2 - 3*p1) + log(16 - 12*r + p2*(-6 + 9*r)))
return(L)}
interlogL_r2 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
optimize(logL_r2,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_r2 <- parallel::mcmapply(interlogL_r2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_r2 <- -((x[,"n_02"] + x[,"n_21"])*(log10(2 - 3*p1) + log10(10 - (3*p2)/2))) - (x[,"n_01"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(10 - (3*p2)/2)) - (x[,"n_03"] + x[,"n_20"])*(-log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_23"])*(-log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) + (x[,"n_03"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_r2))) + (x[,"n_00"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_r2))) + (x[,"n_01"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(4 + 3*p2 + 12*r_r2 - 9*p2*r_r2)) + (x[,"n_02"] + x[,"n_21"])*(log10(2 - 3*p1) + log10(16 - 12*r_r2 + p2*(-6 + 9*r_r2)))


logL_r3 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*(5*log(2) + 2*log(3)) + (n10 + n13)*(log(8 - 3*p1) + log(2 - 3*p2)) + (n11 + n12)*(log(8 - 3*p1) + log(10 + 3*p2)) + (n03 + n20)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r))) + (n00 + n23)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r))) + (n02 + n21)*(log(4 + 3*p1) + log(8 - 6*r - p2*(-6 + 9*r))) + (n01 + n22)*(log(4 + 3*p1) + log(2 + 6*r + p2*(-3 + 9*r)))
return(L)}
interlogL_r3 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
optimize(logL_r3,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_r3 <- parallel::mcmapply(interlogL_r3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_r3 <- -((x[,"n_03"] + x[,"n_20"])*(-log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_23"])*(-log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_21"])*(log10(4 + 3*p1) + log10(5 + (3*p2)/2)) - (x[,"n_01"] + x[,"n_22"])*(log10(4 + 3*p1) + log10(5 + (3*p2)/2)) + (x[,"n_03"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_r3))) + (x[,"n_00"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_r3))) + (x[,"n_02"] + x[,"n_21"])*(log10(4 + 3*p1) + log10(8 - 6*r_r3 - p2*(-6 + 9*r_r3))) + (x[,"n_01"] + x[,"n_22"])*(log10(4 + 3*p1) + log10(2 + 6*r_r3 + p2*(-3 + 9*r_r3)))


logL_r4 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*(6*log(2) + 2*log(3)) + (n11 + n12)*(log(8 - 3*p1) + log(20 - 3*p2)) + (n10 + n13)*(log(8 - 3*p1) + log(4 + 3*p2)) + (n03 + n20)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r))) + (n00 + n23)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r))) + (n01 + n22)*(log(4 + 3*p1) + log(4 + 3*p2 + 12*r - 9*p2*r)) + (n02 + n21)*(log(4 + 3*p1) + log(16 - 12*r + p2*(-6 + 9*r)))
return(L)}
interlogL_r4 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
optimize(logL_r4,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_r4 <- parallel::mcmapply(interlogL_r4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_r4 <- -((x[,"n_02"] + x[,"n_21"])*(log10(4 + 3*p1) + log10(10 - (3*p2)/2))) - (x[,"n_01"] + x[,"n_22"])*(log10(4 + 3*p1) + log10(10 - (3*p2)/2)) - (x[,"n_03"] + x[,"n_20"])*(-log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_23"])*(-log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) + (x[,"n_03"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_r4))) + (x[,"n_00"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_r4))) + (x[,"n_01"] + x[,"n_22"])*(log10(4 + 3*p1) + log10(4 + 3*p2 + 12*r_r4 - 9*p2*r_r4)) + (x[,"n_02"] + x[,"n_21"])*(log10(4 + 3*p1) + log10(16 - 12*r_r4 + p2*(-6 + 9*r_r4)))


return(list(
r_mat = cbind(r_c1,r_c2,r_c3,r_c4,r_r1,r_r2,r_r3,r_r4,0.499),
LOD_mat = cbind(LOD_c1,LOD_c2,LOD_c3,LOD_c4,LOD_r1,LOD_r2,LOD_r3,LOD_r4,0),
logL_mat = NULL,
phasing_strategy = "MINR",
possible_phases = c("coupling","coupling","coupling","coupling","repulsion","repulsion","repulsion","repulsion","unknown")
)
)
}

#' @rdname p4_functions
p4_2.0_1.3 <- function(x,p1,p2,ncores=1){


r_c1 <- (x[,"n_03"] + x[,"n_21"])/(x[,"n_01"] + x[,"n_03"] + x[,"n_21"] + x[,"n_23"])
logL_c1 <- (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*(3*log(2) + log(3)) + (x[,"n_02"] + x[,"n_22"])*log(2 - 3*p1) + (x[,"n_11"] + x[,"n_13"])*log(1/6 + p1/8) + x[,"n_12"]*log(1/3 + p1/4) + (x[,"n_01"] + x[,"n_23"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_c1))) + (x[,"n_03"] + x[,"n_21"])*(log(2 - 3*p1) + log(pmax(1e-6,r_c1)))
LOD_c1 <- -((x[,"n_03"] + x[,"n_21"])*(-log10(2) + log10(2 - 3*p1))) - (x[,"n_01"] + x[,"n_23"])*(-log10(2) + log10(2 - 3*p1)) + (x[,"n_01"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_c1))) + (x[,"n_03"] + x[,"n_21"])*(log10(2 - 3*p1) + log10(pmax(1e-6,r_c1)))


r_c2 <- (x[,"n_03"] + x[,"n_21"])/(x[,"n_01"] + x[,"n_03"] + x[,"n_21"] + x[,"n_23"])
logL_c2 <- (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_11"] - x[,"n_13"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*(4*log(2) + log(3)) + (x[,"n_11"] + x[,"n_13"])*log(8 - 3*p1) + x[,"n_12"]*log(1/3 - p1/8) + (x[,"n_02"] + x[,"n_22"])*log(4 + 3*p1) + (x[,"n_01"] + x[,"n_23"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_c2))) + (x[,"n_03"] + x[,"n_21"])*(log(4 + 3*p1) + log(pmax(1e-6,r_c2)))
LOD_c2 <- -((x[,"n_03"] + x[,"n_21"])*(-log10(2) + log10(4 + 3*p1))) - (x[,"n_01"] + x[,"n_23"])*(-log10(2) + log10(4 + 3*p1)) + (x[,"n_01"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_c2))) + (x[,"n_03"] + x[,"n_21"])*(log10(4 + 3*p1) + log10(pmax(1e-6,r_c2)))


r_r1 <- (x[,"n_01"] + x[,"n_23"])/(x[,"n_01"] + x[,"n_03"] + x[,"n_21"] + x[,"n_23"])
logL_r1 <- (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*(3*log(2) + log(3)) + (x[,"n_02"] + x[,"n_22"])*log(2 - 3*p1) + (x[,"n_11"] + x[,"n_13"])*log(1/6 + p1/8) + x[,"n_12"]*log(1/3 + p1/4) + (x[,"n_03"] + x[,"n_21"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_r1))) + (x[,"n_01"] + x[,"n_23"])*(log(2 - 3*p1) + log(pmax(1e-6,r_r1)))
LOD_r1 <- -((x[,"n_03"] + x[,"n_21"])*(-log10(2) + log10(2 - 3*p1))) - (x[,"n_01"] + x[,"n_23"])*(-log10(2) + log10(2 - 3*p1)) + (x[,"n_03"] + x[,"n_21"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_r1))) + (x[,"n_01"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(pmax(1e-6,r_r1)))


r_r2 <- (x[,"n_01"] + x[,"n_23"])/(x[,"n_01"] + x[,"n_03"] + x[,"n_21"] + x[,"n_23"])
logL_r2 <- (-x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_11"] - x[,"n_13"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*(4*log(2) + log(3)) + (x[,"n_11"] + x[,"n_13"])*log(8 - 3*p1) + x[,"n_12"]*log(1/3 - p1/8) + (x[,"n_02"] + x[,"n_22"])*log(4 + 3*p1) + (x[,"n_03"] + x[,"n_21"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_r2))) + (x[,"n_01"] + x[,"n_23"])*(log(4 + 3*p1) + log(pmax(1e-6,r_r2)))
LOD_r2 <- -((x[,"n_03"] + x[,"n_21"])*(-log10(2) + log10(4 + 3*p1))) - (x[,"n_01"] + x[,"n_23"])*(-log10(2) + log10(4 + 3*p1)) + (x[,"n_03"] + x[,"n_21"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_r2))) + (x[,"n_01"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(pmax(1e-6,r_r2)))


return(list(
r_mat = cbind(r_c1,r_c2,r_r1,r_r2,0.499),
LOD_mat = cbind(LOD_c1,LOD_c2,LOD_r1,LOD_r2,0),
logL_mat = NULL,
phasing_strategy = "MINR",
possible_phases = c("coupling","coupling","repulsion","repulsion","unknown")
)
)
}

#' @rdname p4_functions
p4_2.0_2.0 <- function(x,p1,p2,ncores=1){


logL_c1 <- function(r,n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
L <- (-2*n00 - n01 - 2*n02 - n10 - n11 - n12 - 2*n20 - n21 - 2*n22)*log(2) + (-n00 - n01 - n02 - n10 - n11 - n12 - n20 - n21 - n22)*log(3) + (n01 + n10 + n12 + n21)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n02 + n20)*(log(2 - 3*p1) + 2*log(pmax(1e-6,r))) + (n00 + n22)*(log(2 - 3*p1) + log(1 - 2*r + r^2)) + n11*log(p1*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2))
return(L)}
interlogL_c1 <- function(n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
optimize(logL_c1,c(0,0.5), n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_c1 <- parallel::mcmapply(interlogL_c1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_20"],x[,"n_21"],x[,"n_22"], p1, p2, mc.cores = ncores)


LOD_c1 <- -((x[,"n_02"] + x[,"n_20"])*(-2*log10(2) + log10(2 - 3*p1))) - (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(-2*log10(2) + log10(2 - 3*p1)) - (x[,"n_00"] + x[,"n_22"])*(-2*log10(2) + log10(2 - 3*p1)) - x[,"n_11"]*log10(3 + (9*p1)/2) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_c1)) + log10(pmax(1e-6,r_c1))) + (x[,"n_02"] + x[,"n_20"])*(log10(2 - 3*p1) + 2*log10(pmax(1e-6,r_c1))) + (x[,"n_00"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(1 - 2*r_c1 + r_c1^2)) + x[,"n_11"]*log10(p1*(3 + 6*r_c1 - 6*r_c1^2) + 4*(1 - r_c1 + r_c1^2))


logL_c1 <- (-2*x[,"n_00"] - x[,"n_01"] - 2*x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - 2*x[,"n_20"] - x[,"n_21"] - 2*x[,"n_22"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"])*log(3) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_c1)) + log(pmax(1e-6,r_c1))) + (x[,"n_02"] + x[,"n_20"])*(log(2 - 3*p1) + 2*log(pmax(1e-6,r_c1))) + (x[,"n_00"] + x[,"n_22"])*(log(2 - 3*p1) + log(1 - 2*r_c1 + r_c1^2)) + x[,"n_11"]*log(p1*(3 + 6*r_c1 - 6*r_c1^2) + 4*(1 - r_c1 + r_c1^2))


logL_c2 <- function(r,n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
L <- (-3*n00 - 2*n01 - 3*n02 - 2*n10 - 2*n11 - 2*n12 - 3*n20 - 2*n21 - 3*n22)*log(2) + (-n00 - n01 - n02 - n10 - n11 - n12 - n20 - n21 - n22)*log(3) + (n01 + n10 + n12 + n21)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n02 + n20)*(log(4 + 3*p1) + 2*log(pmax(1e-6,r))) + (n00 + n22)*(log(4 + 3*p1) + log(1 - 2*r + r^2)) + n11*log(8*(1 - r + r^2) + p1*(-3 - 6*r + 6*r^2))
return(L)}
interlogL_c2 <- function(n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
optimize(logL_c2,c(0,0.5), n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_c2 <- parallel::mcmapply(interlogL_c2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_20"],x[,"n_21"],x[,"n_22"], p1, p2, mc.cores = ncores)


LOD_c2 <- -(x[,"n_11"]*log10(6 - (9*p1)/2)) - (x[,"n_02"] + x[,"n_20"])*(-2*log10(2) + log10(4 + 3*p1)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(-2*log10(2) + log10(4 + 3*p1)) - (x[,"n_00"] + x[,"n_22"])*(-2*log10(2) + log10(4 + 3*p1)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_c2)) + log10(pmax(1e-6,r_c2))) + (x[,"n_02"] + x[,"n_20"])*(log10(4 + 3*p1) + 2*log10(pmax(1e-6,r_c2))) + (x[,"n_00"] + x[,"n_22"])*(log10(4 + 3*p1) + log10(1 - 2*r_c2 + r_c2^2)) + x[,"n_11"]*log10(8*(1 - r_c2 + r_c2^2) + p1*(-3 - 6*r_c2 + 6*r_c2^2))


logL_c2 <- (-3*x[,"n_00"] - 2*x[,"n_01"] - 3*x[,"n_02"] - 2*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 3*x[,"n_20"] - 2*x[,"n_21"] - 3*x[,"n_22"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"])*log(3) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_c2)) + log(pmax(1e-6,r_c2))) + (x[,"n_02"] + x[,"n_20"])*(log(4 + 3*p1) + 2*log(pmax(1e-6,r_c2))) + (x[,"n_00"] + x[,"n_22"])*(log(4 + 3*p1) + log(1 - 2*r_c2 + r_c2^2)) + x[,"n_11"]*log(8*(1 - r_c2 + r_c2^2) + p1*(-3 - 6*r_c2 + 6*r_c2^2))


logL_m1 <- function(r,n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
L <- (-3*n00 - 2*n01 - 3*n02 - 3*n10 - 2*n11 - 3*n12 - 3*n20 - 2*n21 - 3*n22)*log(2) + (-n00 - n01 - n02 - n10 - n11 - n12 - n20 - n21 - n22)*log(3) + (n00 + n02 + n20 + n22)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n01 + n21)*(log(2 - 3*p1) + log(1 - r + r^2)) + (n10 + n12)*log(p1*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2)) + n11*log(4 + 4*r - 4*r^2 + p1*(3 - 6*r + 6*r^2))
return(L)}
interlogL_m1 <- function(n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
optimize(logL_m1,c(0,0.5), n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_m1 <- parallel::mcmapply(interlogL_m1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_20"],x[,"n_21"],x[,"n_22"], p1, p2, mc.cores = ncores)


LOD_m1 <- -((x[,"n_00"] + x[,"n_02"] + x[,"n_20"] + x[,"n_22"])*(-2*log10(2) + log10(2 - 3*p1))) - (x[,"n_01"] + x[,"n_21"])*(-2*log10(2) + log10(3) + log10(2 - 3*p1)) - x[,"n_11"]*log10(5 + (3*p1)/2) - (x[,"n_10"] + x[,"n_12"])*log10(3 + (9*p1)/2) + (x[,"n_00"] + x[,"n_02"] + x[,"n_20"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_m1)) + log10(pmax(1e-6,r_m1))) + (x[,"n_01"] + x[,"n_21"])*(log10(2 - 3*p1) + log10(1 - r_m1 + r_m1^2)) + (x[,"n_10"] + x[,"n_12"])*log10(p1*(3 + 6*r_m1 - 6*r_m1^2) + 4*(1 - r_m1 + r_m1^2)) + x[,"n_11"]*log10(4 + 4*r_m1 - 4*r_m1^2 + p1*(3 - 6*r_m1 + 6*r_m1^2))


logL_m1 <- (-3*x[,"n_00"] - 2*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_10"] - 2*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_20"] - 2*x[,"n_21"] - 3*x[,"n_22"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"])*log(3) + (x[,"n_00"] + x[,"n_02"] + x[,"n_20"] + x[,"n_22"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_m1)) + log(pmax(1e-6,r_m1))) + (x[,"n_01"] + x[,"n_21"])*(log(2 - 3*p1) + log(1 - r_m1 + r_m1^2)) + (x[,"n_10"] + x[,"n_12"])*log(p1*(3 + 6*r_m1 - 6*r_m1^2) + 4*(1 - r_m1 + r_m1^2)) + x[,"n_11"]*log(4 + 4*r_m1 - 4*r_m1^2 + p1*(3 - 6*r_m1 + 6*r_m1^2))


logL_m2 <- function(r,n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
L <- (-3*n00 - 3*n01 - 3*n02 - 2*n10 - 2*n11 - 2*n12 - 3*n20 - 3*n21 - 3*n22)*log(2) + (-n00 - n01 - n02 - n10 - n11 - n12 - n20 - n21 - n22)*log(3) + (n00 + n02 + n20 + n22)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n12)*(log(2 - 3*p1) + log(1 - r + r^2)) + (n01 + n21)*log(p1*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2)) + n11*log(4 + 4*r - 4*r^2 + p1*(3 - 6*r + 6*r^2))
return(L)}
interlogL_m2 <- function(n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
optimize(logL_m2,c(0,0.5), n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_m2 <- parallel::mcmapply(interlogL_m2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_20"],x[,"n_21"],x[,"n_22"], p1, p2, mc.cores = ncores)


LOD_m2 <- -((x[,"n_00"] + x[,"n_02"] + x[,"n_20"] + x[,"n_22"])*(-2*log10(2) + log10(2 - 3*p1))) - (x[,"n_10"] + x[,"n_12"])*(-2*log10(2) + log10(3) + log10(2 - 3*p1)) - x[,"n_11"]*log10(5 + (3*p1)/2) - (x[,"n_01"] + x[,"n_21"])*log10(3 + (9*p1)/2) + (x[,"n_00"] + x[,"n_02"] + x[,"n_20"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_m2)) + log10(pmax(1e-6,r_m2))) + (x[,"n_10"] + x[,"n_12"])*(log10(2 - 3*p1) + log10(1 - r_m2 + r_m2^2)) + (x[,"n_01"] + x[,"n_21"])*log10(p1*(3 + 6*r_m2 - 6*r_m2^2) + 4*(1 - r_m2 + r_m2^2)) + x[,"n_11"]*log10(4 + 4*r_m2 - 4*r_m2^2 + p1*(3 - 6*r_m2 + 6*r_m2^2))


logL_m2 <- (-3*x[,"n_00"] - 3*x[,"n_01"] - 3*x[,"n_02"] - 2*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 3*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"])*log(3) + (x[,"n_00"] + x[,"n_02"] + x[,"n_20"] + x[,"n_22"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_m2)) + log(pmax(1e-6,r_m2))) + (x[,"n_10"] + x[,"n_12"])*(log(2 - 3*p1) + log(1 - r_m2 + r_m2^2)) + (x[,"n_01"] + x[,"n_21"])*log(p1*(3 + 6*r_m2 - 6*r_m2^2) + 4*(1 - r_m2 + r_m2^2)) + x[,"n_11"]*log(4 + 4*r_m2 - 4*r_m2^2 + p1*(3 - 6*r_m2 + 6*r_m2^2))


logL_r1 <- function(r,n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
L <- (-2*n00 - n01 - 2*n02 - n10 - n11 - n12 - 2*n20 - n21 - 2*n22)*log(2) + (-n00 - n01 - n02 - n10 - n11 - n12 - n20 - n21 - n22)*log(3) + (n01 + n10 + n12 + n21)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n22)*(log(2 - 3*p1) + 2*log(pmax(1e-6,r))) + (n02 + n20)*(log(2 - 3*p1) + log(1 - 2*r + r^2)) + n11*log(p1*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2))
return(L)}
interlogL_r1 <- function(n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
optimize(logL_r1,c(0,0.5), n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_r1 <- parallel::mcmapply(interlogL_r1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_20"],x[,"n_21"],x[,"n_22"], p1, p2, mc.cores = ncores)


LOD_r1 <- -((x[,"n_02"] + x[,"n_20"])*(-2*log10(2) + log10(2 - 3*p1))) - (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(-2*log10(2) + log10(2 - 3*p1)) - (x[,"n_00"] + x[,"n_22"])*(-2*log10(2) + log10(2 - 3*p1)) - x[,"n_11"]*log10(3 + (9*p1)/2) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_r1)) + log10(pmax(1e-6,r_r1))) + (x[,"n_00"] + x[,"n_22"])*(log10(2 - 3*p1) + 2*log10(pmax(1e-6,r_r1))) + (x[,"n_02"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(1 - 2*r_r1 + r_r1^2)) + x[,"n_11"]*log10(p1*(3 + 6*r_r1 - 6*r_r1^2) + 4*(1 - r_r1 + r_r1^2))


logL_r1 <- (-2*x[,"n_00"] - x[,"n_01"] - 2*x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - 2*x[,"n_20"] - x[,"n_21"] - 2*x[,"n_22"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"])*log(3) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_r1)) + log(pmax(1e-6,r_r1))) + (x[,"n_00"] + x[,"n_22"])*(log(2 - 3*p1) + 2*log(pmax(1e-6,r_r1))) + (x[,"n_02"] + x[,"n_20"])*(log(2 - 3*p1) + log(1 - 2*r_r1 + r_r1^2)) + x[,"n_11"]*log(p1*(3 + 6*r_r1 - 6*r_r1^2) + 4*(1 - r_r1 + r_r1^2))


logL_r2 <- function(r,n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
L <- (-3*n00 - 2*n01 - 3*n02 - 2*n10 - 2*n11 - 2*n12 - 3*n20 - 2*n21 - 3*n22)*log(2) + (-n00 - n01 - n02 - n10 - n11 - n12 - n20 - n21 - n22)*log(3) + (n01 + n10 + n12 + n21)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n22)*(log(4 + 3*p1) + 2*log(pmax(1e-6,r))) + (n02 + n20)*(log(4 + 3*p1) + log(1 - 2*r + r^2)) + n11*log(8*(1 - r + r^2) + p1*(-3 - 6*r + 6*r^2))
return(L)}
interlogL_r2 <- function(n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2) {
optimize(logL_r2,c(0,0.5), n00,n01,n02,n10,n11,n12,n20,n21,n22,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_r2 <- parallel::mcmapply(interlogL_r2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_20"],x[,"n_21"],x[,"n_22"], p1, p2, mc.cores = ncores)


LOD_r2 <- -(x[,"n_11"]*log10(6 - (9*p1)/2)) - (x[,"n_02"] + x[,"n_20"])*(-2*log10(2) + log10(4 + 3*p1)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(-2*log10(2) + log10(4 + 3*p1)) - (x[,"n_00"] + x[,"n_22"])*(-2*log10(2) + log10(4 + 3*p1)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_r2)) + log10(pmax(1e-6,r_r2))) + (x[,"n_00"] + x[,"n_22"])*(log10(4 + 3*p1) + 2*log10(pmax(1e-6,r_r2))) + (x[,"n_02"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(1 - 2*r_r2 + r_r2^2)) + x[,"n_11"]*log10(8*(1 - r_r2 + r_r2^2) + p1*(-3 - 6*r_r2 + 6*r_r2^2))


logL_r2 <- (-3*x[,"n_00"] - 2*x[,"n_01"] - 3*x[,"n_02"] - 2*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 3*x[,"n_20"] - 2*x[,"n_21"] - 3*x[,"n_22"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"])*log(3) + (x[,"n_01"] + x[,"n_10"] + x[,"n_12"] + x[,"n_21"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_r2)) + log(pmax(1e-6,r_r2))) + (x[,"n_00"] + x[,"n_22"])*(log(4 + 3*p1) + 2*log(pmax(1e-6,r_r2))) + (x[,"n_02"] + x[,"n_20"])*(log(4 + 3*p1) + log(1 - 2*r_r2 + r_r2^2)) + x[,"n_11"]*log(8*(1 - r_r2 + r_r2^2) + p1*(-3 - 6*r_r2 + 6*r_r2^2))


return(list(
r_mat = cbind(r_c1,r_c2,r_m1,r_m2,r_r1,r_r2,0.499),
LOD_mat = cbind(LOD_c1,LOD_c2,LOD_m1,LOD_m2,LOD_r1,LOD_r2,0),
logL_mat = cbind(logL_c1,logL_c2,logL_m1,logL_m2,logL_r1,logL_r2,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","coupling","mixed","mixed","repulsion","repulsion","unknown")
)
)
}

#' @rdname p4_functions
p4_2.0_2.1 <- function(x,p1,p2,ncores=1){


logL_c1 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
L <- (-3*n00 - 3*n01 - 3*n02 - 3*n03 - 2*n10 - 2*n11 - 2*n12 - 2*n13 - 3*n20 - 3*n21 - 3*n22 - 3*n23)*log(2) + (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*log(3) + (n10 + n13)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n02 + n21)*(log(2 - 3*p1) + log(2 - r) + log(pmax(1e-6,r))) + (n03 + n20)*(log(2 - 3*p1) + 2*log(pmax(1e-6,r))) + (n01 + n22)*(log(2 - 3*p1) + log(1 - r^2)) + (n00 + n23)*(log(2 - 3*p1) + log(1 - 2*r + r^2)) + (n11 + n12)*log(p1*(3 + 3*r - 3*r^2) + 2*(2 - r + r^2))
return(L)}
interlogL_c1 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
optimize(logL_c1,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_c1 <- parallel::mcmapply(interlogL_c1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_c1 <- -((x[,"n_10"] + x[,"n_13"])*(-2*log10(2) + log10(2 - 3*p1))) - (x[,"n_03"] + x[,"n_20"])*(-2*log10(2) + log10(2 - 3*p1)) - (x[,"n_00"] + x[,"n_23"])*(-2*log10(2) + log10(2 - 3*p1)) - (x[,"n_02"] + x[,"n_21"])*(-2*log10(2) + log10(3) + log10(2 - 3*p1)) - (x[,"n_01"] + x[,"n_22"])*(-2*log10(2) + log10(3) + log10(2 - 3*p1)) - (x[,"n_11"] + x[,"n_12"])*log10(7/2 + (15*p1)/4) + (x[,"n_10"] + x[,"n_13"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_c1)) + log10(pmax(1e-6,r_c1))) + (x[,"n_02"] + x[,"n_21"])*(log10(2 - 3*p1) + log10(2 - r_c1) + log10(pmax(1e-6,r_c1))) + (x[,"n_03"] + x[,"n_20"])*(log10(2 - 3*p1) + 2*log10(pmax(1e-6,r_c1))) + (x[,"n_01"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(1 - r_c1^2)) + (x[,"n_00"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(1 - 2*r_c1 + r_c1^2)) + (x[,"n_11"] + x[,"n_12"])*log10(p1*(3 + 3*r_c1 - 3*r_c1^2) + 2*(2 - r_c1 + r_c1^2))


logL_c1 <- (-3*x[,"n_00"] - 3*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 2*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 3*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*log(3) + (x[,"n_10"] + x[,"n_13"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_c1)) + log(pmax(1e-6,r_c1))) + (x[,"n_02"] + x[,"n_21"])*(log(2 - 3*p1) + log(2 - r_c1) + log(pmax(1e-6,r_c1))) + (x[,"n_03"] + x[,"n_20"])*(log(2 - 3*p1) + 2*log(pmax(1e-6,r_c1))) + (x[,"n_01"] + x[,"n_22"])*(log(2 - 3*p1) + log(1 - r_c1^2)) + (x[,"n_00"] + x[,"n_23"])*(log(2 - 3*p1) + log(1 - 2*r_c1 + r_c1^2)) + (x[,"n_11"] + x[,"n_12"])*log(p1*(3 + 3*r_c1 - 3*r_c1^2) + 2*(2 - r_c1 + r_c1^2))


logL_c2 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
L <- (-4*n00 - 4*n01 - 4*n02 - 4*n03 - 3*n10 - 3*n11 - 3*n12 - 3*n13 - 4*n20 - 4*n21 - 4*n22 - 4*n23)*log(2) + (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*log(3) + (n10 + n13)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n02 + n21)*(log(4 + 3*p1) + log(2 - r) + log(pmax(1e-6,r))) + (n03 + n20)*(log(4 + 3*p1) + 2*log(pmax(1e-6,r))) + (n01 + n22)*(log(4 + 3*p1) + log(1 - r^2)) + (n00 + n23)*(log(4 + 3*p1) + log(1 - 2*r + r^2)) + (n11 + n12)*log(3*p1*(-1 - r + r^2) + 4*(2 - r + r^2))
return(L)}
interlogL_c2 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
optimize(logL_c2,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_c2 <- parallel::mcmapply(interlogL_c2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_c2 <- -((x[,"n_11"] + x[,"n_12"])*log10(7 - (15*p1)/4)) - (x[,"n_10"] + x[,"n_13"])*(-2*log10(2) + log10(4 + 3*p1)) - (x[,"n_03"] + x[,"n_20"])*(-2*log10(2) + log10(4 + 3*p1)) - (x[,"n_00"] + x[,"n_23"])*(-2*log10(2) + log10(4 + 3*p1)) - (x[,"n_02"] + x[,"n_21"])*(-2*log10(2) + log10(3) + log10(4 + 3*p1)) - (x[,"n_01"] + x[,"n_22"])*(-2*log10(2) + log10(3) + log10(4 + 3*p1)) + (x[,"n_10"] + x[,"n_13"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_c2)) + log10(pmax(1e-6,r_c2))) + (x[,"n_02"] + x[,"n_21"])*(log10(4 + 3*p1) + log10(2 - r_c2) + log10(pmax(1e-6,r_c2))) + (x[,"n_03"] + x[,"n_20"])*(log10(4 + 3*p1) + 2*log10(pmax(1e-6,r_c2))) + (x[,"n_01"] + x[,"n_22"])*(log10(4 + 3*p1) + log10(1 - r_c2^2)) + (x[,"n_00"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(1 - 2*r_c2 + r_c2^2)) + (x[,"n_11"] + x[,"n_12"])*log10(3*p1*(-1 - r_c2 + r_c2^2) + 4*(2 - r_c2 + r_c2^2))


logL_c2 <- (-4*x[,"n_00"] - 4*x[,"n_01"] - 4*x[,"n_02"] - 4*x[,"n_03"] - 3*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 4*x[,"n_23"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*log(3) + (x[,"n_10"] + x[,"n_13"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_c2)) + log(pmax(1e-6,r_c2))) + (x[,"n_02"] + x[,"n_21"])*(log(4 + 3*p1) + log(2 - r_c2) + log(pmax(1e-6,r_c2))) + (x[,"n_03"] + x[,"n_20"])*(log(4 + 3*p1) + 2*log(pmax(1e-6,r_c2))) + (x[,"n_01"] + x[,"n_22"])*(log(4 + 3*p1) + log(1 - r_c2^2)) + (x[,"n_00"] + x[,"n_23"])*(log(4 + 3*p1) + log(1 - 2*r_c2 + r_c2^2)) + (x[,"n_11"] + x[,"n_12"])*log(3*p1*(-1 - r_c2 + r_c2^2) + 4*(2 - r_c2 + r_c2^2))


logL_m1 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*(4*log(2) + log(3)) + (n00 + n03 + n20 + n23)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n01 + n02 + n21 + n22)*(log(2 - 3*p1) + log(2 - r + r^2)) + (n10 + n13)*log(p1*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2)) + (n11 + n12)*log(4*(3 + r - r^2) + p1*(9 - 6*r + 6*r^2))
return(L)}
interlogL_m1 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
optimize(logL_m1,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_m1 <- parallel::mcmapply(interlogL_m1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_m1 <- -((x[,"n_00"] + x[,"n_03"] + x[,"n_20"] + x[,"n_23"])*(-2*log10(2) + log10(2 - 3*p1))) - (x[,"n_01"] + x[,"n_02"] + x[,"n_21"] + x[,"n_22"])*(-2*log10(2) + log10(7) + log10(2 - 3*p1)) - (x[,"n_10"] + x[,"n_13"])*log10(3 + (9*p1)/2) - (x[,"n_11"] + x[,"n_12"])*log10(13 + (15*p1)/2) + (x[,"n_00"] + x[,"n_03"] + x[,"n_20"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_m1)) + log10(pmax(1e-6,r_m1))) + (x[,"n_01"] + x[,"n_02"] + x[,"n_21"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(2 - r_m1 + r_m1^2)) + (x[,"n_10"] + x[,"n_13"])*log10(p1*(3 + 6*r_m1 - 6*r_m1^2) + 4*(1 - r_m1 + r_m1^2)) + (x[,"n_11"] + x[,"n_12"])*log10(4*(3 + r_m1 - r_m1^2) + p1*(9 - 6*r_m1 + 6*r_m1^2))


logL_m1 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*(4*log(2) + log(3)) + (x[,"n_00"] + x[,"n_03"] + x[,"n_20"] + x[,"n_23"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_m1)) + log(pmax(1e-6,r_m1))) + (x[,"n_01"] + x[,"n_02"] + x[,"n_21"] + x[,"n_22"])*(log(2 - 3*p1) + log(2 - r_m1 + r_m1^2)) + (x[,"n_10"] + x[,"n_13"])*log(p1*(3 + 6*r_m1 - 6*r_m1^2) + 4*(1 - r_m1 + r_m1^2)) + (x[,"n_11"] + x[,"n_12"])*log(4*(3 + r_m1 - r_m1^2) + p1*(9 - 6*r_m1 + 6*r_m1^2))


logL_m2 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
L <- (-4*n00 - 4*n01 - 4*n02 - 4*n03 - 3*n10 - 3*n11 - 3*n12 - 3*n13 - 4*n20 - 4*n21 - 4*n22 - 4*n23)*log(2) + (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*log(3) + (n00 + n03 + n20 + n23)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n13)*(log(2 - 3*p1) + log(1 - r + r^2)) + (n11 + n12)*log(6 + (2 - 3*p1)*r + (-2 + 3*p1)*r^2) + (n01 + n02 + n21 + n22)*log(p1*(3 + 3*r - 3*r^2) + 2*(2 - r + r^2))
return(L)}
interlogL_m2 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
optimize(logL_m2,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_m2 <- parallel::mcmapply(interlogL_m2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_m2 <- -((x[,"n_00"] + x[,"n_03"] + x[,"n_20"] + x[,"n_23"])*(-2*log10(2) + log10(2 - 3*p1))) - (x[,"n_10"] + x[,"n_13"])*(-2*log10(2) + log10(3) + log10(2 - 3*p1)) - (x[,"n_01"] + x[,"n_02"] + x[,"n_21"] + x[,"n_22"])*log10(7/2 + (15*p1)/4) - (x[,"n_11"] + x[,"n_12"])*log10(6 + (2 - 3*p1)/2 + (-2 + 3*p1)/4) + (x[,"n_00"] + x[,"n_03"] + x[,"n_20"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_m2)) + log10(pmax(1e-6,r_m2))) + (x[,"n_10"] + x[,"n_13"])*(log10(2 - 3*p1) + log10(1 - r_m2 + r_m2^2)) + (x[,"n_11"] + x[,"n_12"])*log10(6 + (2 - 3*p1)*r_m2 + (-2 + 3*p1)*r_m2^2) + (x[,"n_01"] + x[,"n_02"] + x[,"n_21"] + x[,"n_22"])*log10(p1*(3 + 3*r_m2 - 3*r_m2^2) + 2*(2 - r_m2 + r_m2^2))


logL_m2 <- (-4*x[,"n_00"] - 4*x[,"n_01"] - 4*x[,"n_02"] - 4*x[,"n_03"] - 3*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 4*x[,"n_23"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*log(3) + (x[,"n_00"] + x[,"n_03"] + x[,"n_20"] + x[,"n_23"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_m2)) + log(pmax(1e-6,r_m2))) + (x[,"n_10"] + x[,"n_13"])*(log(2 - 3*p1) + log(1 - r_m2 + r_m2^2)) + (x[,"n_11"] + x[,"n_12"])*log(6 + (2 - 3*p1)*r_m2 + (-2 + 3*p1)*r_m2^2) + (x[,"n_01"] + x[,"n_02"] + x[,"n_21"] + x[,"n_22"])*log(p1*(3 + 3*r_m2 - 3*r_m2^2) + 2*(2 - r_m2 + r_m2^2))


logL_r1 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
L <- (-3*n00 - 3*n01 - 3*n02 - 3*n03 - 2*n10 - 2*n11 - 2*n12 - 2*n13 - 3*n20 - 3*n21 - 3*n22 - 3*n23)*log(2) + (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*log(3) + (n10 + n13)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n01 + n22)*(log(2 - 3*p1) + log(2 - r) + log(pmax(1e-6,r))) + (n00 + n23)*(log(2 - 3*p1) + 2*log(pmax(1e-6,r))) + (n02 + n21)*(log(2 - 3*p1) + log(1 - r^2)) + (n03 + n20)*(log(2 - 3*p1) + log(1 - 2*r + r^2)) + (n11 + n12)*log(p1*(3 + 3*r - 3*r^2) + 2*(2 - r + r^2))
return(L)}
interlogL_r1 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
optimize(logL_r1,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_r1 <- parallel::mcmapply(interlogL_r1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_r1 <- -((x[,"n_10"] + x[,"n_13"])*(-2*log10(2) + log10(2 - 3*p1))) - (x[,"n_03"] + x[,"n_20"])*(-2*log10(2) + log10(2 - 3*p1)) - (x[,"n_00"] + x[,"n_23"])*(-2*log10(2) + log10(2 - 3*p1)) - (x[,"n_02"] + x[,"n_21"])*(-2*log10(2) + log10(3) + log10(2 - 3*p1)) - (x[,"n_01"] + x[,"n_22"])*(-2*log10(2) + log10(3) + log10(2 - 3*p1)) - (x[,"n_11"] + x[,"n_12"])*log10(7/2 + (15*p1)/4) + (x[,"n_10"] + x[,"n_13"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_r1)) + log10(pmax(1e-6,r_r1))) + (x[,"n_01"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(2 - r_r1) + log10(pmax(1e-6,r_r1))) + (x[,"n_00"] + x[,"n_23"])*(log10(2 - 3*p1) + 2*log10(pmax(1e-6,r_r1))) + (x[,"n_02"] + x[,"n_21"])*(log10(2 - 3*p1) + log10(1 - r_r1^2)) + (x[,"n_03"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(1 - 2*r_r1 + r_r1^2)) + (x[,"n_11"] + x[,"n_12"])*log10(p1*(3 + 3*r_r1 - 3*r_r1^2) + 2*(2 - r_r1 + r_r1^2))


logL_r1 <- (-3*x[,"n_00"] - 3*x[,"n_01"] - 3*x[,"n_02"] - 3*x[,"n_03"] - 2*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 3*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*log(3) + (x[,"n_10"] + x[,"n_13"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_r1)) + log(pmax(1e-6,r_r1))) + (x[,"n_01"] + x[,"n_22"])*(log(2 - 3*p1) + log(2 - r_r1) + log(pmax(1e-6,r_r1))) + (x[,"n_00"] + x[,"n_23"])*(log(2 - 3*p1) + 2*log(pmax(1e-6,r_r1))) + (x[,"n_02"] + x[,"n_21"])*(log(2 - 3*p1) + log(1 - r_r1^2)) + (x[,"n_03"] + x[,"n_20"])*(log(2 - 3*p1) + log(1 - 2*r_r1 + r_r1^2)) + (x[,"n_11"] + x[,"n_12"])*log(p1*(3 + 3*r_r1 - 3*r_r1^2) + 2*(2 - r_r1 + r_r1^2))


logL_r2 <- function(r,n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
L <- (-4*n00 - 4*n01 - 4*n02 - 4*n03 - 3*n10 - 3*n11 - 3*n12 - 3*n13 - 4*n20 - 4*n21 - 4*n22 - 4*n23)*log(2) + (-n00 - n01 - n02 - n03 - n10 - n11 - n12 - n13 - n20 - n21 - n22 - n23)*log(3) + (n10 + n13)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n01 + n22)*(log(4 + 3*p1) + log(2 - r) + log(pmax(1e-6,r))) + (n00 + n23)*(log(4 + 3*p1) + 2*log(pmax(1e-6,r))) + (n02 + n21)*(log(4 + 3*p1) + log(1 - r^2)) + (n03 + n20)*(log(4 + 3*p1) + log(1 - 2*r + r^2)) + (n11 + n12)*log(3*p1*(-1 - r + r^2) + 4*(2 - r + r^2))
return(L)}
interlogL_r2 <- function(n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2) {
optimize(logL_r2,c(0,0.5), n00,n01,n02,n03,n10,n11,n12,n13,n20,n21,n22,n23,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_r2 <- parallel::mcmapply(interlogL_r2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"], p1, p2, mc.cores = ncores)


LOD_r2 <- -((x[,"n_11"] + x[,"n_12"])*log10(7 - (15*p1)/4)) - (x[,"n_10"] + x[,"n_13"])*(-2*log10(2) + log10(4 + 3*p1)) - (x[,"n_03"] + x[,"n_20"])*(-2*log10(2) + log10(4 + 3*p1)) - (x[,"n_00"] + x[,"n_23"])*(-2*log10(2) + log10(4 + 3*p1)) - (x[,"n_02"] + x[,"n_21"])*(-2*log10(2) + log10(3) + log10(4 + 3*p1)) - (x[,"n_01"] + x[,"n_22"])*(-2*log10(2) + log10(3) + log10(4 + 3*p1)) + (x[,"n_10"] + x[,"n_13"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_r2)) + log10(pmax(1e-6,r_r2))) + (x[,"n_01"] + x[,"n_22"])*(log10(4 + 3*p1) + log10(2 - r_r2) + log10(pmax(1e-6,r_r2))) + (x[,"n_00"] + x[,"n_23"])*(log10(4 + 3*p1) + 2*log10(pmax(1e-6,r_r2))) + (x[,"n_02"] + x[,"n_21"])*(log10(4 + 3*p1) + log10(1 - r_r2^2)) + (x[,"n_03"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(1 - 2*r_r2 + r_r2^2)) + (x[,"n_11"] + x[,"n_12"])*log10(3*p1*(-1 - r_r2 + r_r2^2) + 4*(2 - r_r2 + r_r2^2))


logL_r2 <- (-4*x[,"n_00"] - 4*x[,"n_01"] - 4*x[,"n_02"] - 4*x[,"n_03"] - 3*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 4*x[,"n_23"])*log(2) + (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"])*log(3) + (x[,"n_10"] + x[,"n_13"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_r2)) + log(pmax(1e-6,r_r2))) + (x[,"n_01"] + x[,"n_22"])*(log(4 + 3*p1) + log(2 - r_r2) + log(pmax(1e-6,r_r2))) + (x[,"n_00"] + x[,"n_23"])*(log(4 + 3*p1) + 2*log(pmax(1e-6,r_r2))) + (x[,"n_02"] + x[,"n_21"])*(log(4 + 3*p1) + log(1 - r_r2^2)) + (x[,"n_03"] + x[,"n_20"])*(log(4 + 3*p1) + log(1 - 2*r_r2 + r_r2^2)) + (x[,"n_11"] + x[,"n_12"])*log(3*p1*(-1 - r_r2 + r_r2^2) + 4*(2 - r_r2 + r_r2^2))


return(list(
r_mat = cbind(r_c1,r_c2,r_m1,r_m2,r_r1,r_r2,0.499),
LOD_mat = cbind(LOD_c1,LOD_c2,LOD_m1,LOD_m2,LOD_r1,LOD_r2,0),
logL_mat = cbind(logL_c1,logL_c2,logL_m1,logL_m2,logL_r1,logL_r2,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","coupling","mixed","mixed","repulsion","repulsion","unknown")
)
)
}

#' @rdname p4_functions
p4_2.0_2.2 <- function(x,p1,p2,ncores=1){


logL_c1 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-4*n00 - 3*n01 - 4*n02 - 3*n03 - 4*n04 - 3*n10 - 2*n11 - 2*n12 - 2*n13 - 3*n14 - 4*n20 - 3*n21 - 4*n22 - 3*n23 - 4*n24)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(3) + (n10 + n14)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n04 + n20)*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r))) + (n01 + n23)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(4 + 3*p2 - 2*r - 6*p2*r)) + (n03 + n21)*(log(2 - 3*p1) + log(pmax(1e-6,r)) + log(2 - 3*p2 + 2*r + 6*p2*r)) + (n00 + n24)*(log(2 - 3*p1) + log(2 - 3*p2) + log(1 - 2*r + r^2)) + (n02 + n22)*(log(2 - 3*p1) + log(2*(1 + 6*r - 6*r^2) - 3*p2*(1 - 6*r + 6*r^2))) + n12*log((2 - 3*p1)*(-2 + 3*p2)*(-1 + r)*r + (4 + 3*p2)*(p1*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2))) + (n11 + n13)*log((-2 + 3*p1)*(4 + 3*p2)*(-1 + r)*r + ((-2 + 3*p2)*(-4*(1 - r + r^2) + p1*(-3 - 6*r + 6*r^2)))/2)
return(L)}
interlogL_c1 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_c1,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_c1 <- parallel::mcmapply(interlogL_c1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_c1 <- -((x[,"n_03"] + x[,"n_21"])*(-log10(2) + log10(3) + log10(2 - 3*p1))) - (x[,"n_01"] + x[,"n_23"])*(-log10(2) + log10(3) + log10(2 - 3*p1)) - (x[,"n_10"] + x[,"n_14"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_04"] + x[,"n_20"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_24"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(5 + (3*p2)/2)) - (x[,"n_11"] + x[,"n_13"])*log10(((-3 - (9*p1)/2)*(-2 + 3*p2))/2 - ((-2 + 3*p1)*(4 + 3*p2))/4) - x[,"n_12"]*log10(-((2 - 3*p1)*(-2 + 3*p2))/4 + (3 + (9*p1)/2)*(4 + 3*p2)) + (x[,"n_10"] + x[,"n_14"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_c1)) + log10(pmax(1e-6,r_c1))) + (x[,"n_04"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_c1))) + (x[,"n_01"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_c1)) + log10(4 + 3*p2 - 2*r_c1 - 6*p2*r_c1)) + (x[,"n_03"] + x[,"n_21"])*(log10(2 - 3*p1) + log10(pmax(1e-6,r_c1)) + log10(2 - 3*p2 + 2*r_c1 + 6*p2*r_c1)) + (x[,"n_00"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(1 - 2*r_c1 + r_c1^2)) + (x[,"n_02"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(2*(1 + 6*r_c1 - 6*r_c1^2) - 3*p2*(1 - 6*r_c1 + 6*r_c1^2))) + x[,"n_12"]*log10((2 - 3*p1)*(-2 + 3*p2)*(-1 + r_c1)*r_c1 + (4 + 3*p2)*(p1*(3 + 6*r_c1 - 6*r_c1^2) + 4*(1 - r_c1 + r_c1^2))) + (x[,"n_11"] + x[,"n_13"])*log10((-2 + 3*p1)*(4 + 3*p2)*(-1 + r_c1)*r_c1 + ((-2 + 3*p2)*(-4*(1 - r_c1 + r_c1^2) + p1*(-3 - 6*r_c1 + 6*r_c1^2)))/2)


logL_c1 <- (-4*x[,"n_00"] - 3*x[,"n_01"] - 4*x[,"n_02"] - 3*x[,"n_03"] - 4*x[,"n_04"] - 3*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 3*x[,"n_14"] - 4*x[,"n_20"] - 3*x[,"n_21"] - 4*x[,"n_22"] - 3*x[,"n_23"] - 4*x[,"n_24"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(3) + (x[,"n_10"] + x[,"n_14"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_c1)) + log(pmax(1e-6,r_c1))) + (x[,"n_04"] + x[,"n_20"])*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r_c1))) + (x[,"n_01"] + x[,"n_23"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_c1)) + log(4 + 3*p2 - 2*r_c1 - 6*p2*r_c1)) + (x[,"n_03"] + x[,"n_21"])*(log(2 - 3*p1) + log(pmax(1e-6,r_c1)) + log(2 - 3*p2 + 2*r_c1 + 6*p2*r_c1)) + (x[,"n_00"] + x[,"n_24"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(1 - 2*r_c1 + r_c1^2)) + (x[,"n_02"] + x[,"n_22"])*(log(2 - 3*p1) + log(2*(1 + 6*r_c1 - 6*r_c1^2) - 3*p2*(1 - 6*r_c1 + 6*r_c1^2))) + x[,"n_12"]*log((2 - 3*p1)*(-2 + 3*p2)*(-1 + r_c1)*r_c1 + (4 + 3*p2)*(p1*(3 + 6*r_c1 - 6*r_c1^2) + 4*(1 - r_c1 + r_c1^2))) + (x[,"n_11"] + x[,"n_13"])*log((-2 + 3*p1)*(4 + 3*p2)*(-1 + r_c1)*r_c1 + ((-2 + 3*p2)*(-4*(1 - r_c1 + r_c1^2) + p1*(-3 - 6*r_c1 + 6*r_c1^2)))/2)


logL_c2 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-5*n00 - 4*n01 - 5*n02 - 4*n03 - 5*n04 - 4*n10 - n11 - n12 - n13 - 4*n14 - 5*n20 - 4*n21 - 5*n22 - 4*n23 - 5*n24)*log(2) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - 2*n10 - n11 - n12 - n13 - 2*n14 - 2*n20 - 2*n21 - 2*n22 - 2*n23 - 2*n24)*log(3) + (n10 + n14)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n04 + n20)*(log(2 - 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r))) + (n00 + n24)*(log(2 - 3*p1) + log(4 + 3*p2) + log(1 - 2*r + r^2)) + (n03 + n21)*(log(2 - 3*p1) + log(pmax(1e-6,r)) + log(4*(1 + r) - p2*(-3 + 6*r))) + (n01 + n23)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(8 - 4*r + p2*(-3 + 6*r))) + (n02 + n22)*(log(2 - 3*p1) + log(4 + 24*r - 24*r^2 + 3*p2*(1 - 6*r + 6*r^2))) + n12*log(((-2 + 3*p1)*(4 + 3*p2)*(-1 + r)*r)/12 + (2/3 - p2/4)*(p1*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2))) + (n11 + n13)*log((-2 + 3*p1)*(2/3 - p2/4)*(-1 + r)*r + (1/6 + p2/8)*(p1*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2)))
return(L)}
interlogL_c2 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_c2,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_c2 <- parallel::mcmapply(interlogL_c2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_c2 <- -((x[,"n_03"] + x[,"n_21"])*(log10(3) + log10(2 - 3*p1))) - (x[,"n_01"] + x[,"n_23"])*(log10(3) + log10(2 - 3*p1)) - (x[,"n_11"] + x[,"n_13"])*log10(-((-2 + 3*p1)*(2/3 - p2/4))/4 + (3 + (9*p1)/2)*(1/6 + p2/8)) - (x[,"n_02"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(10 - (3*p2)/2)) - (x[,"n_10"] + x[,"n_14"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_04"] + x[,"n_20"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_24"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - x[,"n_12"]*log10((3 + (9*p1)/2)*(2/3 - p2/4) - ((-2 + 3*p1)*(4 + 3*p2))/48) + (x[,"n_10"] + x[,"n_14"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_c2)) + log10(pmax(1e-6,r_c2))) + (x[,"n_04"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_c2))) + (x[,"n_00"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(1 - 2*r_c2 + r_c2^2)) + (x[,"n_03"] + x[,"n_21"])*(log10(2 - 3*p1) + log10(pmax(1e-6,r_c2)) + log10(4*(1 + r_c2) - p2*(-3 + 6*r_c2))) + (x[,"n_01"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_c2)) + log10(8 - 4*r_c2 + p2*(-3 + 6*r_c2))) + (x[,"n_02"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(4 + 24*r_c2 - 24*r_c2^2 + 3*p2*(1 - 6*r_c2 + 6*r_c2^2))) + x[,"n_12"]*log10(((-2 + 3*p1)*(4 + 3*p2)*(-1 + r_c2)*r_c2)/12 + (2/3 - p2/4)*(p1*(3 + 6*r_c2 - 6*r_c2^2) + 4*(1 - r_c2 + r_c2^2))) + (x[,"n_11"] + x[,"n_13"])*log10((-2 + 3*p1)*(2/3 - p2/4)*(-1 + r_c2)*r_c2 + (1/6 + p2/8)*(p1*(3 + 6*r_c2 - 6*r_c2^2) + 4*(1 - r_c2 + r_c2^2)))


logL_c2 <- (-5*x[,"n_00"] - 4*x[,"n_01"] - 5*x[,"n_02"] - 4*x[,"n_03"] - 5*x[,"n_04"] - 4*x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - 4*x[,"n_14"] - 5*x[,"n_20"] - 4*x[,"n_21"] - 5*x[,"n_22"] - 4*x[,"n_23"] - 5*x[,"n_24"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"])*log(3) + (x[,"n_10"] + x[,"n_14"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_c2)) + log(pmax(1e-6,r_c2))) + (x[,"n_04"] + x[,"n_20"])*(log(2 - 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r_c2))) + (x[,"n_00"] + x[,"n_24"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(1 - 2*r_c2 + r_c2^2)) + (x[,"n_03"] + x[,"n_21"])*(log(2 - 3*p1) + log(pmax(1e-6,r_c2)) + log(4*(1 + r_c2) - p2*(-3 + 6*r_c2))) + (x[,"n_01"] + x[,"n_23"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_c2)) + log(8 - 4*r_c2 + p2*(-3 + 6*r_c2))) + (x[,"n_02"] + x[,"n_22"])*(log(2 - 3*p1) + log(4 + 24*r_c2 - 24*r_c2^2 + 3*p2*(1 - 6*r_c2 + 6*r_c2^2))) + x[,"n_12"]*log(((-2 + 3*p1)*(4 + 3*p2)*(-1 + r_c2)*r_c2)/12 + (2/3 - p2/4)*(p1*(3 + 6*r_c2 - 6*r_c2^2) + 4*(1 - r_c2 + r_c2^2))) + (x[,"n_11"] + x[,"n_13"])*log((-2 + 3*p1)*(2/3 - p2/4)*(-1 + r_c2)*r_c2 + (1/6 + p2/8)*(p1*(3 + 6*r_c2 - 6*r_c2^2) + 4*(1 - r_c2 + r_c2^2)))


logL_c3 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-5*n00 - 4*n01 - 5*n02 - 4*n03 - 5*n04 - 4*n10 - 3*n11 - 3*n12 - 3*n13 - 4*n14 - 5*n20 - 4*n21 - 5*n22 - 4*n23 - 5*n24)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(3) + (n10 + n14)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n04 + n20)*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r))) + (n01 + n23)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(4 + 3*p2 - 2*r - 6*p2*r)) + (n03 + n21)*(log(4 + 3*p1) + log(pmax(1e-6,r)) + log(2 - 3*p2 + 2*r + 6*p2*r)) + (n00 + n24)*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r + r^2)) + (n02 + n22)*(log(4 + 3*p1) + log(2*(1 + 6*r - 6*r^2) - 3*p2*(1 - 6*r + 6*r^2))) + (n11 + n13)*log((-4 - 3*p1)*(4 + 3*p2)*(-1 + r)*r - ((-2 + 3*p2)*(8*(1 - r + r^2) + p1*(-3 - 6*r + 6*r^2)))/2) + n12*log((4 + 3*p1)*(-2 + 3*p2)*(-1 + r)*r + (4 + 3*p2)*(8*(1 - r + r^2) + p1*(-3 - 6*r + 6*r^2)))
return(L)}
interlogL_c3 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_c3,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_c3 <- parallel::mcmapply(interlogL_c3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_c3 <- -((x[,"n_03"] + x[,"n_21"])*(-log10(2) + log10(3) + log10(4 + 3*p1))) - (x[,"n_01"] + x[,"n_23"])*(-log10(2) + log10(3) + log10(4 + 3*p1)) - (x[,"n_10"] + x[,"n_14"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_04"] + x[,"n_20"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_24"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_22"])*(log10(4 + 3*p1) + log10(5 + (3*p2)/2)) - x[,"n_12"]*log10(-((4 + 3*p1)*(-2 + 3*p2))/4 + (6 - (9*p1)/2)*(4 + 3*p2)) - (x[,"n_11"] + x[,"n_13"])*log10(-((6 - (9*p1)/2)*(-2 + 3*p2))/2 - ((-4 - 3*p1)*(4 + 3*p2))/4) + (x[,"n_10"] + x[,"n_14"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_c3)) + log10(pmax(1e-6,r_c3))) + (x[,"n_04"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_c3))) + (x[,"n_01"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_c3)) + log10(4 + 3*p2 - 2*r_c3 - 6*p2*r_c3)) + (x[,"n_03"] + x[,"n_21"])*(log10(4 + 3*p1) + log10(pmax(1e-6,r_c3)) + log10(2 - 3*p2 + 2*r_c3 + 6*p2*r_c3)) + (x[,"n_00"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(1 - 2*r_c3 + r_c3^2)) + (x[,"n_02"] + x[,"n_22"])*(log10(4 + 3*p1) + log10(2*(1 + 6*r_c3 - 6*r_c3^2) - 3*p2*(1 - 6*r_c3 + 6*r_c3^2))) + (x[,"n_11"] + x[,"n_13"])*log10((-4 - 3*p1)*(4 + 3*p2)*(-1 + r_c3)*r_c3 - ((-2 + 3*p2)*(8*(1 - r_c3 + r_c3^2) + p1*(-3 - 6*r_c3 + 6*r_c3^2)))/2) + x[,"n_12"]*log10((4 + 3*p1)*(-2 + 3*p2)*(-1 + r_c3)*r_c3 + (4 + 3*p2)*(8*(1 - r_c3 + r_c3^2) + p1*(-3 - 6*r_c3 + 6*r_c3^2)))


logL_c3 <- (-5*x[,"n_00"] - 4*x[,"n_01"] - 5*x[,"n_02"] - 4*x[,"n_03"] - 5*x[,"n_04"] - 4*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 4*x[,"n_14"] - 5*x[,"n_20"] - 4*x[,"n_21"] - 5*x[,"n_22"] - 4*x[,"n_23"] - 5*x[,"n_24"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(3) + (x[,"n_10"] + x[,"n_14"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_c3)) + log(pmax(1e-6,r_c3))) + (x[,"n_04"] + x[,"n_20"])*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r_c3))) + (x[,"n_01"] + x[,"n_23"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_c3)) + log(4 + 3*p2 - 2*r_c3 - 6*p2*r_c3)) + (x[,"n_03"] + x[,"n_21"])*(log(4 + 3*p1) + log(pmax(1e-6,r_c3)) + log(2 - 3*p2 + 2*r_c3 + 6*p2*r_c3)) + (x[,"n_00"] + x[,"n_24"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r_c3 + r_c3^2)) + (x[,"n_02"] + x[,"n_22"])*(log(4 + 3*p1) + log(2*(1 + 6*r_c3 - 6*r_c3^2) - 3*p2*(1 - 6*r_c3 + 6*r_c3^2))) + (x[,"n_11"] + x[,"n_13"])*log((-4 - 3*p1)*(4 + 3*p2)*(-1 + r_c3)*r_c3 - ((-2 + 3*p2)*(8*(1 - r_c3 + r_c3^2) + p1*(-3 - 6*r_c3 + 6*r_c3^2)))/2) + x[,"n_12"]*log((4 + 3*p1)*(-2 + 3*p2)*(-1 + r_c3)*r_c3 + (4 + 3*p2)*(8*(1 - r_c3 + r_c3^2) + p1*(-3 - 6*r_c3 + 6*r_c3^2)))


logL_c4 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-6*n00 - 5*n01 - 6*n02 - 5*n03 - 6*n04 - 5*n10 - 5*n11 - 2*n12 - 5*n13 - 5*n14 - 6*n20 - 5*n21 - 6*n22 - 5*n23 - 6*n24)*log(2) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - 2*n10 - 2*n11 - n12 - 2*n13 - 2*n14 - 2*n20 - 2*n21 - 2*n22 - 2*n23 - 2*n24)*log(3) + (n10 + n14)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n04 + n20)*(log(4 + 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r))) + (n00 + n24)*(log(4 + 3*p1) + log(4 + 3*p2) + log(1 - 2*r + r^2)) + (n03 + n21)*(log(4 + 3*p1) + log(pmax(1e-6,r)) + log(4*(1 + r) - p2*(-3 + 6*r))) + (n01 + n23)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(8 - 4*r + p2*(-3 + 6*r))) + (n02 + n22)*(log(4 + 3*p1) + log(4 + 24*r - 24*r^2 + 3*p2*(1 - 6*r + 6*r^2))) + n12*log(-((4 + 3*p1)*(4 + 3*p2)*(-1 + r)*r)/12 + (2/3 - p2/4)*(8*(1 - r + r^2) + p1*(-3 - 6*r + 6*r^2))) + (n11 + n13)*log(3*p1*(-4 + 8*r - 8*r^2 + 3*p2*(-1 - 4*r + 4*r^2)) + 8*(4 + 4*r - 4*r^2 + p2*(3 - 6*r + 6*r^2)))
return(L)}
interlogL_c4 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_c4,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_c4 <- parallel::mcmapply(interlogL_c4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_c4 <- -((x[,"n_03"] + x[,"n_21"])*(log10(3) + log10(4 + 3*p1))) - (x[,"n_01"] + x[,"n_23"])*(log10(3) + log10(4 + 3*p1)) - (x[,"n_02"] + x[,"n_22"])*(log10(4 + 3*p1) + log10(10 - (3*p2)/2)) - (x[,"n_10"] + x[,"n_14"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_04"] + x[,"n_20"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_24"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_11"] + x[,"n_13"])*log10(3*p1*(-2 - 6*p2) + 8*(5 + (3*p2)/2)) - x[,"n_12"]*log10((6 - (9*p1)/2)*(2/3 - p2/4) + ((4 + 3*p1)*(4 + 3*p2))/48) + (x[,"n_10"] + x[,"n_14"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_c4)) + log10(pmax(1e-6,r_c4))) + (x[,"n_04"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_c4))) + (x[,"n_00"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(1 - 2*r_c4 + r_c4^2)) + (x[,"n_03"] + x[,"n_21"])*(log10(4 + 3*p1) + log10(pmax(1e-6,r_c4)) + log10(4*(1 + r_c4) - p2*(-3 + 6*r_c4))) + (x[,"n_01"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_c4)) + log10(8 - 4*r_c4 + p2*(-3 + 6*r_c4))) + (x[,"n_02"] + x[,"n_22"])*(log10(4 + 3*p1) + log10(4 + 24*r_c4 - 24*r_c4^2 + 3*p2*(1 - 6*r_c4 + 6*r_c4^2))) + x[,"n_12"]*log10(-((4 + 3*p1)*(4 + 3*p2)*(-1 + r_c4)*r_c4)/12 + (2/3 - p2/4)*(8*(1 - r_c4 + r_c4^2) + p1*(-3 - 6*r_c4 + 6*r_c4^2))) + (x[,"n_11"] + x[,"n_13"])*log10(3*p1*(-4 + 8*r_c4 - 8*r_c4^2 + 3*p2*(-1 - 4*r_c4 + 4*r_c4^2)) + 8*(4 + 4*r_c4 - 4*r_c4^2 + p2*(3 - 6*r_c4 + 6*r_c4^2)))


logL_c4 <- (-6*x[,"n_00"] - 5*x[,"n_01"] - 6*x[,"n_02"] - 5*x[,"n_03"] - 6*x[,"n_04"] - 5*x[,"n_10"] - 5*x[,"n_11"] - 2*x[,"n_12"] - 5*x[,"n_13"] - 5*x[,"n_14"] - 6*x[,"n_20"] - 5*x[,"n_21"] - 6*x[,"n_22"] - 5*x[,"n_23"] - 6*x[,"n_24"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_10"] - 2*x[,"n_11"] - x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"])*log(3) + (x[,"n_10"] + x[,"n_14"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_c4)) + log(pmax(1e-6,r_c4))) + (x[,"n_04"] + x[,"n_20"])*(log(4 + 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r_c4))) + (x[,"n_00"] + x[,"n_24"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(1 - 2*r_c4 + r_c4^2)) + (x[,"n_03"] + x[,"n_21"])*(log(4 + 3*p1) + log(pmax(1e-6,r_c4)) + log(4*(1 + r_c4) - p2*(-3 + 6*r_c4))) + (x[,"n_01"] + x[,"n_23"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_c4)) + log(8 - 4*r_c4 + p2*(-3 + 6*r_c4))) + (x[,"n_02"] + x[,"n_22"])*(log(4 + 3*p1) + log(4 + 24*r_c4 - 24*r_c4^2 + 3*p2*(1 - 6*r_c4 + 6*r_c4^2))) + x[,"n_12"]*log(-((4 + 3*p1)*(4 + 3*p2)*(-1 + r_c4)*r_c4)/12 + (2/3 - p2/4)*(8*(1 - r_c4 + r_c4^2) + p1*(-3 - 6*r_c4 + 6*r_c4^2))) + (x[,"n_11"] + x[,"n_13"])*log(3*p1*(-4 + 8*r_c4 - 8*r_c4^2 + 3*p2*(-1 - 4*r_c4 + 4*r_c4^2)) + 8*(4 + 4*r_c4 - 4*r_c4^2 + p2*(3 - 6*r_c4 + 6*r_c4^2)))


logL_m1 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-5*n00 - 4*n01 - 4*n02 - 4*n03 - 5*n04 - 5*n10 - 3*n11 - 4*n12 - 3*n13 - 5*n14 - 5*n20 - 4*n21 - 4*n22 - 4*n23 - 5*n24)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(3) + (n00 + n04 + n20 + n24)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n14)*(log(2 - 3*p2) + log(4*(1 - r + r^2) - p1*(-3 - 6*r + 6*r^2))) + (n01 + n03 + n21 + n23)*(log(2 - 3*p1) + log(2*(1 + r - r^2) - p2*(3 - 6*r + 6*r^2))) + (n02 + n22)*(log(2 - 3*p1) + log(8 - 6*r + 6*r^2 + p2*(6 - 9*r + 9*r^2))) + (n11 + n13)*log(p1*(9 + 6*(1 + 3*p2)*r - 6*(1 + 3*p2)*r^2) + 4*(3 - (1 + 3*p2)*r + (1 + 3*p2)*r^2)) + n12*log(4*(10 + 6*r - 6*r^2 + p2*(3 + 9*r - 9*r^2)) + p1*(9*p2*(1 - 6*r + 6*r^2) + 6*(5 - 6*r + 6*r^2)))
return(L)}
interlogL_m1 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_m1,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_m1 <- parallel::mcmapply(interlogL_m1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_m1 <- -((x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2))) - (x[,"n_10"] + x[,"n_14"])*(log10(3 + (9*p1)/2) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_03"] + x[,"n_21"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(5/2 - (3*p2)/2)) - (x[,"n_02"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(13/2 + (15*p2)/4)) - x[,"n_12"]*log10(p1*(21 - (9*p2)/2) + 4*(23/2 + (21*p2)/4)) - (x[,"n_11"] + x[,"n_13"])*log10(4*(3 + (-1 - 3*p2)/2 + (1 + 3*p2)/4) + p1*(9 + (3*(1 + 3*p2))/2)) + (x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_m1)) + log10(pmax(1e-6,r_m1))) + (x[,"n_10"] + x[,"n_14"])*(log10(2 - 3*p2) + log10(4*(1 - r_m1 + r_m1^2) - p1*(-3 - 6*r_m1 + 6*r_m1^2))) + (x[,"n_01"] + x[,"n_03"] + x[,"n_21"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(2*(1 + r_m1 - r_m1^2) - p2*(3 - 6*r_m1 + 6*r_m1^2))) + (x[,"n_02"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(8 - 6*r_m1 + 6*r_m1^2 + p2*(6 - 9*r_m1 + 9*r_m1^2))) + (x[,"n_11"] + x[,"n_13"])*log10(p1*(9 + 6*(1 + 3*p2)*r_m1 - 6*(1 + 3*p2)*r_m1^2) + 4*(3 - (1 + 3*p2)*r_m1 + (1 + 3*p2)*r_m1^2)) + x[,"n_12"]*log10(4*(10 + 6*r_m1 - 6*r_m1^2 + p2*(3 + 9*r_m1 - 9*r_m1^2)) + p1*(9*p2*(1 - 6*r_m1 + 6*r_m1^2) + 6*(5 - 6*r_m1 + 6*r_m1^2)))


logL_m1 <- (-5*x[,"n_00"] - 4*x[,"n_01"] - 4*x[,"n_02"] - 4*x[,"n_03"] - 5*x[,"n_04"] - 5*x[,"n_10"] - 3*x[,"n_11"] - 4*x[,"n_12"] - 3*x[,"n_13"] - 5*x[,"n_14"] - 5*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 4*x[,"n_23"] - 5*x[,"n_24"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(3) + (x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_m1)) + log(pmax(1e-6,r_m1))) + (x[,"n_10"] + x[,"n_14"])*(log(2 - 3*p2) + log(4*(1 - r_m1 + r_m1^2) - p1*(-3 - 6*r_m1 + 6*r_m1^2))) + (x[,"n_01"] + x[,"n_03"] + x[,"n_21"] + x[,"n_23"])*(log(2 - 3*p1) + log(2*(1 + r_m1 - r_m1^2) - p2*(3 - 6*r_m1 + 6*r_m1^2))) + (x[,"n_02"] + x[,"n_22"])*(log(2 - 3*p1) + log(8 - 6*r_m1 + 6*r_m1^2 + p2*(6 - 9*r_m1 + 9*r_m1^2))) + (x[,"n_11"] + x[,"n_13"])*log(p1*(9 + 6*(1 + 3*p2)*r_m1 - 6*(1 + 3*p2)*r_m1^2) + 4*(3 - (1 + 3*p2)*r_m1 + (1 + 3*p2)*r_m1^2)) + x[,"n_12"]*log(4*(10 + 6*r_m1 - 6*r_m1^2 + p2*(3 + 9*r_m1 - 9*r_m1^2)) + p1*(9*p2*(1 - 6*r_m1 + 6*r_m1^2) + 6*(5 - 6*r_m1 + 6*r_m1^2)))


logL_m2 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-5*n00 - 4*n01 - 4*n02 - 4*n03 - 5*n04 - 4*n10 - 4*n11 - 3*n12 - 4*n13 - 4*n14 - 5*n20 - 4*n21 - 4*n22 - 4*n23 - 5*n24)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(3) + (n00 + n04 + n20 + n24)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n14)*(log(2 - 3*p1) + log(2 - 3*p2) + log(1 - r + r^2)) + (n02 + n22)*log((2 - 3*p1)*(-2 + 3*p2)*(-1 + r)*r + (4 + 3*p2)*(p1*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2))) + (n11 + n13)*log(8*(3 - (1 + 3*p2)*r + (1 + 3*p2)*r^2) - 3*p1*(6 - 4*r + 4*r^2 + 3*p2*(3 - 4*r + 4*r^2))) + (n01 + n03 + n21 + n23)*log((-2 + 3*p1)*(4 + 3*p2)*(-1 + r)*r + ((-2 + 3*p2)*(-4*(1 - r + r^2) + p1*(-3 - 6*r + 6*r^2)))/2) + n12*log((-2 + 3*p1)*(-2 + 3*p2)*(1 - r + r^2) + (4 + 3*p2)*(4 + 4*r - 4*r^2 + p1*(3 - 6*r + 6*r^2)))
return(L)}
interlogL_m2 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_m2,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_m2 <- parallel::mcmapply(interlogL_m2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_m2 <- -((x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2))) - (x[,"n_10"] + x[,"n_14"])*(-2*log10(2) + log10(3) + log10(2 - 3*p1) + log10(2 - 3*p2)) - x[,"n_12"]*log10((3*(-2 + 3*p1)*(-2 + 3*p2))/4 + (5 + (3*p1)/2)*(4 + 3*p2)) - (x[,"n_01"] + x[,"n_03"] + x[,"n_21"] + x[,"n_23"])*log10(((-3 - (9*p1)/2)*(-2 + 3*p2))/2 - ((-2 + 3*p1)*(4 + 3*p2))/4) - (x[,"n_02"] + x[,"n_22"])*log10(-((2 - 3*p1)*(-2 + 3*p2))/4 + (3 + (9*p1)/2)*(4 + 3*p2)) - (x[,"n_11"] + x[,"n_13"])*log10(-3*p1*(5 + 6*p2) + 8*(3 + (-1 - 3*p2)/2 + (1 + 3*p2)/4)) + (x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_m2)) + log10(pmax(1e-6,r_m2))) + (x[,"n_10"] + x[,"n_14"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(1 - r_m2 + r_m2^2)) + (x[,"n_02"] + x[,"n_22"])*log10((2 - 3*p1)*(-2 + 3*p2)*(-1 + r_m2)*r_m2 + (4 + 3*p2)*(p1*(3 + 6*r_m2 - 6*r_m2^2) + 4*(1 - r_m2 + r_m2^2))) + (x[,"n_11"] + x[,"n_13"])*log10(8*(3 - (1 + 3*p2)*r_m2 + (1 + 3*p2)*r_m2^2) - 3*p1*(6 - 4*r_m2 + 4*r_m2^2 + 3*p2*(3 - 4*r_m2 + 4*r_m2^2))) + (x[,"n_01"] + x[,"n_03"] + x[,"n_21"] + x[,"n_23"])*log10((-2 + 3*p1)*(4 + 3*p2)*(-1 + r_m2)*r_m2 + ((-2 + 3*p2)*(-4*(1 - r_m2 + r_m2^2) + p1*(-3 - 6*r_m2 + 6*r_m2^2)))/2) + x[,"n_12"]*log10((-2 + 3*p1)*(-2 + 3*p2)*(1 - r_m2 + r_m2^2) + (4 + 3*p2)*(4 + 4*r_m2 - 4*r_m2^2 + p1*(3 - 6*r_m2 + 6*r_m2^2)))


logL_m2 <- (-5*x[,"n_00"] - 4*x[,"n_01"] - 4*x[,"n_02"] - 4*x[,"n_03"] - 5*x[,"n_04"] - 4*x[,"n_10"] - 4*x[,"n_11"] - 3*x[,"n_12"] - 4*x[,"n_13"] - 4*x[,"n_14"] - 5*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 4*x[,"n_23"] - 5*x[,"n_24"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(3) + (x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_m2)) + log(pmax(1e-6,r_m2))) + (x[,"n_10"] + x[,"n_14"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(1 - r_m2 + r_m2^2)) + (x[,"n_02"] + x[,"n_22"])*log((2 - 3*p1)*(-2 + 3*p2)*(-1 + r_m2)*r_m2 + (4 + 3*p2)*(p1*(3 + 6*r_m2 - 6*r_m2^2) + 4*(1 - r_m2 + r_m2^2))) + (x[,"n_11"] + x[,"n_13"])*log(8*(3 - (1 + 3*p2)*r_m2 + (1 + 3*p2)*r_m2^2) - 3*p1*(6 - 4*r_m2 + 4*r_m2^2 + 3*p2*(3 - 4*r_m2 + 4*r_m2^2))) + (x[,"n_01"] + x[,"n_03"] + x[,"n_21"] + x[,"n_23"])*log((-2 + 3*p1)*(4 + 3*p2)*(-1 + r_m2)*r_m2 + ((-2 + 3*p2)*(-4*(1 - r_m2 + r_m2^2) + p1*(-3 - 6*r_m2 + 6*r_m2^2)))/2) + x[,"n_12"]*log((-2 + 3*p1)*(-2 + 3*p2)*(1 - r_m2 + r_m2^2) + (4 + 3*p2)*(4 + 4*r_m2 - 4*r_m2^2 + p1*(3 - 6*r_m2 + 6*r_m2^2)))


logL_m3 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-6*n00 - 5*n01 - 5*n02 - 5*n03 - 6*n04 - 6*n10 - 3*n11 - 5*n12 - 3*n13 - 6*n14 - 6*n20 - 5*n21 - 5*n22 - 5*n23 - 6*n24)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(3) + (n00 + n04 + n20 + n24)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n14)*(log(4 + 3*p2) + log(4*(1 - r + r^2) - p1*(-3 - 6*r + 6*r^2))) + (n01 + n03 + n21 + n23)*(log(2 - 3*p1) + log(4 + 4*r - 4*r^2 + p2*(3 - 6*r + 6*r^2))) + (n02 + n22)*(log(2 - 3*p1) + log(4*(4 - 3*r + 3*r^2) - p2*(6 - 9*r + 9*r^2))) + (n11 + n13)*log(12 + (-4 + 6*p2)*r + (4 - 6*p2)*r^2 + p1*(9 + (6 - 9*p2)*r + (-6 + 9*p2)*r^2)) + n12*log(p1*(-9*p2*(1 - 6*r + 6*r^2) + 12*(5 - 6*r + 6*r^2)) + 4*(4*(5 + 3*r - 3*r^2) + p2*(-3 - 9*r + 9*r^2)))
return(L)}
interlogL_m3 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_m3,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_m3 <- parallel::mcmapply(interlogL_m3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_m3 <- -((x[,"n_02"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(13 - (15*p2)/4))) - (x[,"n_01"] + x[,"n_03"] + x[,"n_21"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(5 + (3*p2)/2)) - (x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_14"])*(log10(3 + (9*p1)/2) + log10(4 + 3*p2)) - x[,"n_12"]*log10(4*(23 - (21*p2)/4) + p1*(42 + (9*p2)/2)) - (x[,"n_11"] + x[,"n_13"])*log10(12 + (4 - 6*p2)/4 + (-4 + 6*p2)/2 + p1*(9 + (6 - 9*p2)/2 + (-6 + 9*p2)/4)) + (x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_m3)) + log10(pmax(1e-6,r_m3))) + (x[,"n_10"] + x[,"n_14"])*(log10(4 + 3*p2) + log10(4*(1 - r_m3 + r_m3^2) - p1*(-3 - 6*r_m3 + 6*r_m3^2))) + (x[,"n_01"] + x[,"n_03"] + x[,"n_21"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(4 + 4*r_m3 - 4*r_m3^2 + p2*(3 - 6*r_m3 + 6*r_m3^2))) + (x[,"n_02"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(4*(4 - 3*r_m3 + 3*r_m3^2) - p2*(6 - 9*r_m3 + 9*r_m3^2))) + (x[,"n_11"] + x[,"n_13"])*log10(12 + (-4 + 6*p2)*r_m3 + (4 - 6*p2)*r_m3^2 + p1*(9 + (6 - 9*p2)*r_m3 + (-6 + 9*p2)*r_m3^2)) + x[,"n_12"]*log10(p1*(-9*p2*(1 - 6*r_m3 + 6*r_m3^2) + 12*(5 - 6*r_m3 + 6*r_m3^2)) + 4*(4*(5 + 3*r_m3 - 3*r_m3^2) + p2*(-3 - 9*r_m3 + 9*r_m3^2)))


logL_m3 <- (-6*x[,"n_00"] - 5*x[,"n_01"] - 5*x[,"n_02"] - 5*x[,"n_03"] - 6*x[,"n_04"] - 6*x[,"n_10"] - 3*x[,"n_11"] - 5*x[,"n_12"] - 3*x[,"n_13"] - 6*x[,"n_14"] - 6*x[,"n_20"] - 5*x[,"n_21"] - 5*x[,"n_22"] - 5*x[,"n_23"] - 6*x[,"n_24"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(3) + (x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_m3)) + log(pmax(1e-6,r_m3))) + (x[,"n_10"] + x[,"n_14"])*(log(4 + 3*p2) + log(4*(1 - r_m3 + r_m3^2) - p1*(-3 - 6*r_m3 + 6*r_m3^2))) + (x[,"n_01"] + x[,"n_03"] + x[,"n_21"] + x[,"n_23"])*(log(2 - 3*p1) + log(4 + 4*r_m3 - 4*r_m3^2 + p2*(3 - 6*r_m3 + 6*r_m3^2))) + (x[,"n_02"] + x[,"n_22"])*(log(2 - 3*p1) + log(4*(4 - 3*r_m3 + 3*r_m3^2) - p2*(6 - 9*r_m3 + 9*r_m3^2))) + (x[,"n_11"] + x[,"n_13"])*log(12 + (-4 + 6*p2)*r_m3 + (4 - 6*p2)*r_m3^2 + p1*(9 + (6 - 9*p2)*r_m3 + (-6 + 9*p2)*r_m3^2)) + x[,"n_12"]*log(p1*(-9*p2*(1 - 6*r_m3 + 6*r_m3^2) + 12*(5 - 6*r_m3 + 6*r_m3^2)) + 4*(4*(5 + 3*r_m3 - 3*r_m3^2) + p2*(-3 - 9*r_m3 + 9*r_m3^2)))


logL_m4 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-6*n00 - 6*n01 - 3*n02 - 6*n03 - 6*n04 - 5*n10 - 5*n11 - 2*n12 - 5*n13 - 5*n14 - 6*n20 - 6*n21 - 3*n22 - 6*n23 - 6*n24)*log(2) + (-2*n00 - 2*n01 - n02 - 2*n03 - 2*n04 - 2*n10 - 2*n11 - n12 - 2*n13 - 2*n14 - 2*n20 - 2*n21 - n22 - 2*n23 - 2*n24)*log(3) + (n00 + n04 + n20 + n24)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n14)*(log(2 - 3*p1) + log(4 + 3*p2) + log(1 - r + r^2)) + (n02 + n22)*log(((-2 + 3*p1)*(4 + 3*p2)*(-1 + r)*r)/12 + (2/3 - p2/4)*(p1*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2))) + (n11 + n13)*log(-8*(-6 + (2 - 3*p2)*r + (-2 + 3*p2)*r^2) + 3*p1*(-12 + 8*r - 8*r^2 + 3*p2*(3 - 4*r + 4*r^2))) + n12*log(-((-2 + 3*p1)*(4 + 3*p2)*(1 - r + r^2))/12 + (2/3 - p2/4)*(4 + 4*r - 4*r^2 + p1*(3 - 6*r + 6*r^2))) + (n01 + n03 + n21 + n23)*log(3*p1*(4 - 8*r + 8*r^2 + p2*(3 + 12*r - 12*r^2)) + 4*(4 + 4*r - 4*r^2 + p2*(3 - 6*r + 6*r^2)))
return(L)}
interlogL_m4 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_m4,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_m4 <- parallel::mcmapply(interlogL_m4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_m4 <- -((x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2))) - (x[,"n_10"] + x[,"n_14"])*(-2*log10(2) + log10(3) + log10(2 - 3*p1) + log10(4 + 3*p2)) - x[,"n_12"]*log10((5 + (3*p1)/2)*(2/3 - p2/4) - ((-2 + 3*p1)*(4 + 3*p2))/16) - (x[,"n_02"] + x[,"n_22"])*log10((3 + (9*p1)/2)*(2/3 - p2/4) - ((-2 + 3*p1)*(4 + 3*p2))/48) - (x[,"n_01"] + x[,"n_03"] + x[,"n_21"] + x[,"n_23"])*log10(4*(5 + (3*p2)/2) + 3*p1*(2 + 6*p2)) - (x[,"n_11"] + x[,"n_13"])*log10(3*p1*(-10 + 6*p2) - 8*(-6 + (2 - 3*p2)/2 + (-2 + 3*p2)/4)) + (x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_m4)) + log10(pmax(1e-6,r_m4))) + (x[,"n_10"] + x[,"n_14"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(1 - r_m4 + r_m4^2)) + (x[,"n_02"] + x[,"n_22"])*log10(((-2 + 3*p1)*(4 + 3*p2)*(-1 + r_m4)*r_m4)/12 + (2/3 - p2/4)*(p1*(3 + 6*r_m4 - 6*r_m4^2) + 4*(1 - r_m4 + r_m4^2))) + (x[,"n_11"] + x[,"n_13"])*log10(-8*(-6 + (2 - 3*p2)*r_m4 + (-2 + 3*p2)*r_m4^2) + 3*p1*(-12 + 8*r_m4 - 8*r_m4^2 + 3*p2*(3 - 4*r_m4 + 4*r_m4^2))) + x[,"n_12"]*log10(-((-2 + 3*p1)*(4 + 3*p2)*(1 - r_m4 + r_m4^2))/12 + (2/3 - p2/4)*(4 + 4*r_m4 - 4*r_m4^2 + p1*(3 - 6*r_m4 + 6*r_m4^2))) + (x[,"n_01"] + x[,"n_03"] + x[,"n_21"] + x[,"n_23"])*log10(3*p1*(4 - 8*r_m4 + 8*r_m4^2 + p2*(3 + 12*r_m4 - 12*r_m4^2)) + 4*(4 + 4*r_m4 - 4*r_m4^2 + p2*(3 - 6*r_m4 + 6*r_m4^2)))


logL_m4 <- (-6*x[,"n_00"] - 6*x[,"n_01"] - 3*x[,"n_02"] - 6*x[,"n_03"] - 6*x[,"n_04"] - 5*x[,"n_10"] - 5*x[,"n_11"] - 2*x[,"n_12"] - 5*x[,"n_13"] - 5*x[,"n_14"] - 6*x[,"n_20"] - 6*x[,"n_21"] - 3*x[,"n_22"] - 6*x[,"n_23"] - 6*x[,"n_24"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_10"] - 2*x[,"n_11"] - x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_20"] - 2*x[,"n_21"] - x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"])*log(3) + (x[,"n_00"] + x[,"n_04"] + x[,"n_20"] + x[,"n_24"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_m4)) + log(pmax(1e-6,r_m4))) + (x[,"n_10"] + x[,"n_14"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(1 - r_m4 + r_m4^2)) + (x[,"n_02"] + x[,"n_22"])*log(((-2 + 3*p1)*(4 + 3*p2)*(-1 + r_m4)*r_m4)/12 + (2/3 - p2/4)*(p1*(3 + 6*r_m4 - 6*r_m4^2) + 4*(1 - r_m4 + r_m4^2))) + (x[,"n_11"] + x[,"n_13"])*log(-8*(-6 + (2 - 3*p2)*r_m4 + (-2 + 3*p2)*r_m4^2) + 3*p1*(-12 + 8*r_m4 - 8*r_m4^2 + 3*p2*(3 - 4*r_m4 + 4*r_m4^2))) + x[,"n_12"]*log(-((-2 + 3*p1)*(4 + 3*p2)*(1 - r_m4 + r_m4^2))/12 + (2/3 - p2/4)*(4 + 4*r_m4 - 4*r_m4^2 + p1*(3 - 6*r_m4 + 6*r_m4^2))) + (x[,"n_01"] + x[,"n_03"] + x[,"n_21"] + x[,"n_23"])*log(3*p1*(4 - 8*r_m4 + 8*r_m4^2 + p2*(3 + 12*r_m4 - 12*r_m4^2)) + 4*(4 + 4*r_m4 - 4*r_m4^2 + p2*(3 - 6*r_m4 + 6*r_m4^2)))


logL_r1 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-4*n00 - 3*n01 - 4*n02 - 3*n03 - 4*n04 - 3*n10 - 2*n11 - 2*n12 - 2*n13 - 3*n14 - 4*n20 - 3*n21 - 4*n22 - 3*n23 - 4*n24)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(3) + (n10 + n14)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n24)*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r))) + (n03 + n21)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(4 + 3*p2 - 2*r - 6*p2*r)) + (n01 + n23)*(log(2 - 3*p1) + log(pmax(1e-6,r)) + log(2 - 3*p2 + 2*r + 6*p2*r)) + (n04 + n20)*(log(2 - 3*p1) + log(2 - 3*p2) + log(1 - 2*r + r^2)) + (n02 + n22)*(log(2 - 3*p1) + log(2*(1 + 6*r - 6*r^2) - 3*p2*(1 - 6*r + 6*r^2))) + n12*log((2 - 3*p1)*(-2 + 3*p2)*(-1 + r)*r + (4 + 3*p2)*(p1*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2))) + (n11 + n13)*log((-2 + 3*p1)*(4 + 3*p2)*(-1 + r)*r + ((-2 + 3*p2)*(-4*(1 - r + r^2) + p1*(-3 - 6*r + 6*r^2)))/2)
return(L)}
interlogL_r1 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_r1,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_r1 <- parallel::mcmapply(interlogL_r1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_r1 <- -((x[,"n_03"] + x[,"n_21"])*(-log10(2) + log10(3) + log10(2 - 3*p1))) - (x[,"n_01"] + x[,"n_23"])*(-log10(2) + log10(3) + log10(2 - 3*p1)) - (x[,"n_10"] + x[,"n_14"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_04"] + x[,"n_20"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_24"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(5 + (3*p2)/2)) - (x[,"n_11"] + x[,"n_13"])*log10(((-3 - (9*p1)/2)*(-2 + 3*p2))/2 - ((-2 + 3*p1)*(4 + 3*p2))/4) - x[,"n_12"]*log10(-((2 - 3*p1)*(-2 + 3*p2))/4 + (3 + (9*p1)/2)*(4 + 3*p2)) + (x[,"n_10"] + x[,"n_14"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_r1)) + log10(pmax(1e-6,r_r1))) + (x[,"n_00"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_r1))) + (x[,"n_03"] + x[,"n_21"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_r1)) + log10(4 + 3*p2 - 2*r_r1 - 6*p2*r_r1)) + (x[,"n_01"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(pmax(1e-6,r_r1)) + log10(2 - 3*p2 + 2*r_r1 + 6*p2*r_r1)) + (x[,"n_04"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(1 - 2*r_r1 + r_r1^2)) + (x[,"n_02"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(2*(1 + 6*r_r1 - 6*r_r1^2) - 3*p2*(1 - 6*r_r1 + 6*r_r1^2))) + x[,"n_12"]*log10((2 - 3*p1)*(-2 + 3*p2)*(-1 + r_r1)*r_r1 + (4 + 3*p2)*(p1*(3 + 6*r_r1 - 6*r_r1^2) + 4*(1 - r_r1 + r_r1^2))) + (x[,"n_11"] + x[,"n_13"])*log10((-2 + 3*p1)*(4 + 3*p2)*(-1 + r_r1)*r_r1 + ((-2 + 3*p2)*(-4*(1 - r_r1 + r_r1^2) + p1*(-3 - 6*r_r1 + 6*r_r1^2)))/2)


logL_r1 <- (-4*x[,"n_00"] - 3*x[,"n_01"] - 4*x[,"n_02"] - 3*x[,"n_03"] - 4*x[,"n_04"] - 3*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 3*x[,"n_14"] - 4*x[,"n_20"] - 3*x[,"n_21"] - 4*x[,"n_22"] - 3*x[,"n_23"] - 4*x[,"n_24"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(3) + (x[,"n_10"] + x[,"n_14"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_r1)) + log(pmax(1e-6,r_r1))) + (x[,"n_00"] + x[,"n_24"])*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r_r1))) + (x[,"n_03"] + x[,"n_21"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_r1)) + log(4 + 3*p2 - 2*r_r1 - 6*p2*r_r1)) + (x[,"n_01"] + x[,"n_23"])*(log(2 - 3*p1) + log(pmax(1e-6,r_r1)) + log(2 - 3*p2 + 2*r_r1 + 6*p2*r_r1)) + (x[,"n_04"] + x[,"n_20"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(1 - 2*r_r1 + r_r1^2)) + (x[,"n_02"] + x[,"n_22"])*(log(2 - 3*p1) + log(2*(1 + 6*r_r1 - 6*r_r1^2) - 3*p2*(1 - 6*r_r1 + 6*r_r1^2))) + x[,"n_12"]*log((2 - 3*p1)*(-2 + 3*p2)*(-1 + r_r1)*r_r1 + (4 + 3*p2)*(p1*(3 + 6*r_r1 - 6*r_r1^2) + 4*(1 - r_r1 + r_r1^2))) + (x[,"n_11"] + x[,"n_13"])*log((-2 + 3*p1)*(4 + 3*p2)*(-1 + r_r1)*r_r1 + ((-2 + 3*p2)*(-4*(1 - r_r1 + r_r1^2) + p1*(-3 - 6*r_r1 + 6*r_r1^2)))/2)


logL_r2 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-5*n00 - 4*n01 - 5*n02 - 4*n03 - 5*n04 - 4*n10 - n11 - n12 - n13 - 4*n14 - 5*n20 - 4*n21 - 5*n22 - 4*n23 - 5*n24)*log(2) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - 2*n10 - n11 - n12 - n13 - 2*n14 - 2*n20 - 2*n21 - 2*n22 - 2*n23 - 2*n24)*log(3) + (n10 + n14)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n24)*(log(2 - 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r))) + (n04 + n20)*(log(2 - 3*p1) + log(4 + 3*p2) + log(1 - 2*r + r^2)) + (n01 + n23)*(log(2 - 3*p1) + log(pmax(1e-6,r)) + log(4*(1 + r) - p2*(-3 + 6*r))) + (n03 + n21)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(8 - 4*r + p2*(-3 + 6*r))) + (n02 + n22)*(log(2 - 3*p1) + log(4 + 24*r - 24*r^2 + 3*p2*(1 - 6*r + 6*r^2))) + n12*log(((-2 + 3*p1)*(4 + 3*p2)*(-1 + r)*r)/12 + (2/3 - p2/4)*(p1*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2))) + (n11 + n13)*log((-2 + 3*p1)*(2/3 - p2/4)*(-1 + r)*r + (1/6 + p2/8)*(p1*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2)))
return(L)}
interlogL_r2 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_r2,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_r2 <- parallel::mcmapply(interlogL_r2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_r2 <- -((x[,"n_03"] + x[,"n_21"])*(log10(3) + log10(2 - 3*p1))) - (x[,"n_01"] + x[,"n_23"])*(log10(3) + log10(2 - 3*p1)) - (x[,"n_11"] + x[,"n_13"])*log10(-((-2 + 3*p1)*(2/3 - p2/4))/4 + (3 + (9*p1)/2)*(1/6 + p2/8)) - (x[,"n_02"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(10 - (3*p2)/2)) - (x[,"n_10"] + x[,"n_14"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_04"] + x[,"n_20"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_24"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - x[,"n_12"]*log10((3 + (9*p1)/2)*(2/3 - p2/4) - ((-2 + 3*p1)*(4 + 3*p2))/48) + (x[,"n_10"] + x[,"n_14"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_r2)) + log10(pmax(1e-6,r_r2))) + (x[,"n_00"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_r2))) + (x[,"n_04"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(1 - 2*r_r2 + r_r2^2)) + (x[,"n_01"] + x[,"n_23"])*(log10(2 - 3*p1) + log10(pmax(1e-6,r_r2)) + log10(4*(1 + r_r2) - p2*(-3 + 6*r_r2))) + (x[,"n_03"] + x[,"n_21"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_r2)) + log10(8 - 4*r_r2 + p2*(-3 + 6*r_r2))) + (x[,"n_02"] + x[,"n_22"])*(log10(2 - 3*p1) + log10(4 + 24*r_r2 - 24*r_r2^2 + 3*p2*(1 - 6*r_r2 + 6*r_r2^2))) + x[,"n_12"]*log10(((-2 + 3*p1)*(4 + 3*p2)*(-1 + r_r2)*r_r2)/12 + (2/3 - p2/4)*(p1*(3 + 6*r_r2 - 6*r_r2^2) + 4*(1 - r_r2 + r_r2^2))) + (x[,"n_11"] + x[,"n_13"])*log10((-2 + 3*p1)*(2/3 - p2/4)*(-1 + r_r2)*r_r2 + (1/6 + p2/8)*(p1*(3 + 6*r_r2 - 6*r_r2^2) + 4*(1 - r_r2 + r_r2^2)))


logL_r2 <- (-5*x[,"n_00"] - 4*x[,"n_01"] - 5*x[,"n_02"] - 4*x[,"n_03"] - 5*x[,"n_04"] - 4*x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - 4*x[,"n_14"] - 5*x[,"n_20"] - 4*x[,"n_21"] - 5*x[,"n_22"] - 4*x[,"n_23"] - 5*x[,"n_24"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"])*log(3) + (x[,"n_10"] + x[,"n_14"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_r2)) + log(pmax(1e-6,r_r2))) + (x[,"n_00"] + x[,"n_24"])*(log(2 - 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r_r2))) + (x[,"n_04"] + x[,"n_20"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(1 - 2*r_r2 + r_r2^2)) + (x[,"n_01"] + x[,"n_23"])*(log(2 - 3*p1) + log(pmax(1e-6,r_r2)) + log(4*(1 + r_r2) - p2*(-3 + 6*r_r2))) + (x[,"n_03"] + x[,"n_21"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_r2)) + log(8 - 4*r_r2 + p2*(-3 + 6*r_r2))) + (x[,"n_02"] + x[,"n_22"])*(log(2 - 3*p1) + log(4 + 24*r_r2 - 24*r_r2^2 + 3*p2*(1 - 6*r_r2 + 6*r_r2^2))) + x[,"n_12"]*log(((-2 + 3*p1)*(4 + 3*p2)*(-1 + r_r2)*r_r2)/12 + (2/3 - p2/4)*(p1*(3 + 6*r_r2 - 6*r_r2^2) + 4*(1 - r_r2 + r_r2^2))) + (x[,"n_11"] + x[,"n_13"])*log((-2 + 3*p1)*(2/3 - p2/4)*(-1 + r_r2)*r_r2 + (1/6 + p2/8)*(p1*(3 + 6*r_r2 - 6*r_r2^2) + 4*(1 - r_r2 + r_r2^2)))


logL_r3 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-5*n00 - 4*n01 - 5*n02 - 4*n03 - 5*n04 - 4*n10 - 3*n11 - 3*n12 - 3*n13 - 4*n14 - 5*n20 - 4*n21 - 5*n22 - 4*n23 - 5*n24)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24)*log(3) + (n10 + n14)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n24)*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r))) + (n03 + n21)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(4 + 3*p2 - 2*r - 6*p2*r)) + (n01 + n23)*(log(4 + 3*p1) + log(pmax(1e-6,r)) + log(2 - 3*p2 + 2*r + 6*p2*r)) + (n04 + n20)*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r + r^2)) + (n02 + n22)*(log(4 + 3*p1) + log(2*(1 + 6*r - 6*r^2) - 3*p2*(1 - 6*r + 6*r^2))) + (n11 + n13)*log((-4 - 3*p1)*(4 + 3*p2)*(-1 + r)*r - ((-2 + 3*p2)*(8*(1 - r + r^2) + p1*(-3 - 6*r + 6*r^2)))/2) + n12*log((4 + 3*p1)*(-2 + 3*p2)*(-1 + r)*r + (4 + 3*p2)*(8*(1 - r + r^2) + p1*(-3 - 6*r + 6*r^2)))
return(L)}
interlogL_r3 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_r3,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_r3 <- parallel::mcmapply(interlogL_r3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_r3 <- -((x[,"n_03"] + x[,"n_21"])*(-log10(2) + log10(3) + log10(4 + 3*p1))) - (x[,"n_01"] + x[,"n_23"])*(-log10(2) + log10(3) + log10(4 + 3*p1)) - (x[,"n_10"] + x[,"n_14"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_04"] + x[,"n_20"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_24"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_22"])*(log10(4 + 3*p1) + log10(5 + (3*p2)/2)) - x[,"n_12"]*log10(-((4 + 3*p1)*(-2 + 3*p2))/4 + (6 - (9*p1)/2)*(4 + 3*p2)) - (x[,"n_11"] + x[,"n_13"])*log10(-((6 - (9*p1)/2)*(-2 + 3*p2))/2 - ((-4 - 3*p1)*(4 + 3*p2))/4) + (x[,"n_10"] + x[,"n_14"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_r3)) + log10(pmax(1e-6,r_r3))) + (x[,"n_00"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_r3))) + (x[,"n_03"] + x[,"n_21"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_r3)) + log10(4 + 3*p2 - 2*r_r3 - 6*p2*r_r3)) + (x[,"n_01"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(pmax(1e-6,r_r3)) + log10(2 - 3*p2 + 2*r_r3 + 6*p2*r_r3)) + (x[,"n_04"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(1 - 2*r_r3 + r_r3^2)) + (x[,"n_02"] + x[,"n_22"])*(log10(4 + 3*p1) + log10(2*(1 + 6*r_r3 - 6*r_r3^2) - 3*p2*(1 - 6*r_r3 + 6*r_r3^2))) + (x[,"n_11"] + x[,"n_13"])*log10((-4 - 3*p1)*(4 + 3*p2)*(-1 + r_r3)*r_r3 - ((-2 + 3*p2)*(8*(1 - r_r3 + r_r3^2) + p1*(-3 - 6*r_r3 + 6*r_r3^2)))/2) + x[,"n_12"]*log10((4 + 3*p1)*(-2 + 3*p2)*(-1 + r_r3)*r_r3 + (4 + 3*p2)*(8*(1 - r_r3 + r_r3^2) + p1*(-3 - 6*r_r3 + 6*r_r3^2)))


logL_r3 <- (-5*x[,"n_00"] - 4*x[,"n_01"] - 5*x[,"n_02"] - 4*x[,"n_03"] - 5*x[,"n_04"] - 4*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 4*x[,"n_14"] - 5*x[,"n_20"] - 4*x[,"n_21"] - 5*x[,"n_22"] - 4*x[,"n_23"] - 5*x[,"n_24"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"])*log(3) + (x[,"n_10"] + x[,"n_14"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_r3)) + log(pmax(1e-6,r_r3))) + (x[,"n_00"] + x[,"n_24"])*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r_r3))) + (x[,"n_03"] + x[,"n_21"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_r3)) + log(4 + 3*p2 - 2*r_r3 - 6*p2*r_r3)) + (x[,"n_01"] + x[,"n_23"])*(log(4 + 3*p1) + log(pmax(1e-6,r_r3)) + log(2 - 3*p2 + 2*r_r3 + 6*p2*r_r3)) + (x[,"n_04"] + x[,"n_20"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 2*r_r3 + r_r3^2)) + (x[,"n_02"] + x[,"n_22"])*(log(4 + 3*p1) + log(2*(1 + 6*r_r3 - 6*r_r3^2) - 3*p2*(1 - 6*r_r3 + 6*r_r3^2))) + (x[,"n_11"] + x[,"n_13"])*log((-4 - 3*p1)*(4 + 3*p2)*(-1 + r_r3)*r_r3 - ((-2 + 3*p2)*(8*(1 - r_r3 + r_r3^2) + p1*(-3 - 6*r_r3 + 6*r_r3^2)))/2) + x[,"n_12"]*log((4 + 3*p1)*(-2 + 3*p2)*(-1 + r_r3)*r_r3 + (4 + 3*p2)*(8*(1 - r_r3 + r_r3^2) + p1*(-3 - 6*r_r3 + 6*r_r3^2)))


logL_r4 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
L <- (-6*n00 - 5*n01 - 6*n02 - 5*n03 - 6*n04 - 5*n10 - 5*n11 - 2*n12 - 5*n13 - 5*n14 - 6*n20 - 5*n21 - 6*n22 - 5*n23 - 6*n24)*log(2) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - 2*n10 - 2*n11 - n12 - 2*n13 - 2*n14 - 2*n20 - 2*n21 - 2*n22 - 2*n23 - 2*n24)*log(3) + (n10 + n14)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n00 + n24)*(log(4 + 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r))) + (n04 + n20)*(log(4 + 3*p1) + log(4 + 3*p2) + log(1 - 2*r + r^2)) + (n01 + n23)*(log(4 + 3*p1) + log(pmax(1e-6,r)) + log(4*(1 + r) - p2*(-3 + 6*r))) + (n03 + n21)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(8 - 4*r + p2*(-3 + 6*r))) + (n02 + n22)*(log(4 + 3*p1) + log(4 + 24*r - 24*r^2 + 3*p2*(1 - 6*r + 6*r^2))) + n12*log(-((4 + 3*p1)*(4 + 3*p2)*(-1 + r)*r)/12 + (2/3 - p2/4)*(8*(1 - r + r^2) + p1*(-3 - 6*r + 6*r^2))) + (n11 + n13)*log(3*p1*(-4 + 8*r - 8*r^2 + 3*p2*(-1 - 4*r + 4*r^2)) + 8*(4 + 4*r - 4*r^2 + p2*(3 - 6*r + 6*r^2)))
return(L)}
interlogL_r4 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2) {
optimize(logL_r4,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_r4 <- parallel::mcmapply(interlogL_r4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"], p1, p2, mc.cores = ncores)


LOD_r4 <- -((x[,"n_03"] + x[,"n_21"])*(log10(3) + log10(4 + 3*p1))) - (x[,"n_01"] + x[,"n_23"])*(log10(3) + log10(4 + 3*p1)) - (x[,"n_02"] + x[,"n_22"])*(log10(4 + 3*p1) + log10(10 - (3*p2)/2)) - (x[,"n_10"] + x[,"n_14"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_04"] + x[,"n_20"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_24"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_11"] + x[,"n_13"])*log10(3*p1*(-2 - 6*p2) + 8*(5 + (3*p2)/2)) - x[,"n_12"]*log10((6 - (9*p1)/2)*(2/3 - p2/4) + ((4 + 3*p1)*(4 + 3*p2))/48) + (x[,"n_10"] + x[,"n_14"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_r4)) + log10(pmax(1e-6,r_r4))) + (x[,"n_00"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_r4))) + (x[,"n_04"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(1 - 2*r_r4 + r_r4^2)) + (x[,"n_01"] + x[,"n_23"])*(log10(4 + 3*p1) + log10(pmax(1e-6,r_r4)) + log10(4*(1 + r_r4) - p2*(-3 + 6*r_r4))) + (x[,"n_03"] + x[,"n_21"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_r4)) + log10(8 - 4*r_r4 + p2*(-3 + 6*r_r4))) + (x[,"n_02"] + x[,"n_22"])*(log10(4 + 3*p1) + log10(4 + 24*r_r4 - 24*r_r4^2 + 3*p2*(1 - 6*r_r4 + 6*r_r4^2))) + x[,"n_12"]*log10(-((4 + 3*p1)*(4 + 3*p2)*(-1 + r_r4)*r_r4)/12 + (2/3 - p2/4)*(8*(1 - r_r4 + r_r4^2) + p1*(-3 - 6*r_r4 + 6*r_r4^2))) + (x[,"n_11"] + x[,"n_13"])*log10(3*p1*(-4 + 8*r_r4 - 8*r_r4^2 + 3*p2*(-1 - 4*r_r4 + 4*r_r4^2)) + 8*(4 + 4*r_r4 - 4*r_r4^2 + p2*(3 - 6*r_r4 + 6*r_r4^2)))


logL_r4 <- (-6*x[,"n_00"] - 5*x[,"n_01"] - 6*x[,"n_02"] - 5*x[,"n_03"] - 6*x[,"n_04"] - 5*x[,"n_10"] - 5*x[,"n_11"] - 2*x[,"n_12"] - 5*x[,"n_13"] - 5*x[,"n_14"] - 6*x[,"n_20"] - 5*x[,"n_21"] - 6*x[,"n_22"] - 5*x[,"n_23"] - 6*x[,"n_24"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_10"] - 2*x[,"n_11"] - x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 2*x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 2*x[,"n_24"])*log(3) + (x[,"n_10"] + x[,"n_14"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_r4)) + log(pmax(1e-6,r_r4))) + (x[,"n_00"] + x[,"n_24"])*(log(4 + 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r_r4))) + (x[,"n_04"] + x[,"n_20"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(1 - 2*r_r4 + r_r4^2)) + (x[,"n_01"] + x[,"n_23"])*(log(4 + 3*p1) + log(pmax(1e-6,r_r4)) + log(4*(1 + r_r4) - p2*(-3 + 6*r_r4))) + (x[,"n_03"] + x[,"n_21"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_r4)) + log(8 - 4*r_r4 + p2*(-3 + 6*r_r4))) + (x[,"n_02"] + x[,"n_22"])*(log(4 + 3*p1) + log(4 + 24*r_r4 - 24*r_r4^2 + 3*p2*(1 - 6*r_r4 + 6*r_r4^2))) + x[,"n_12"]*log(-((4 + 3*p1)*(4 + 3*p2)*(-1 + r_r4)*r_r4)/12 + (2/3 - p2/4)*(8*(1 - r_r4 + r_r4^2) + p1*(-3 - 6*r_r4 + 6*r_r4^2))) + (x[,"n_11"] + x[,"n_13"])*log(3*p1*(-4 + 8*r_r4 - 8*r_r4^2 + 3*p2*(-1 - 4*r_r4 + 4*r_r4^2)) + 8*(4 + 4*r_r4 - 4*r_r4^2 + p2*(3 - 6*r_r4 + 6*r_r4^2)))


return(list(
r_mat = cbind(r_c1,r_c2,r_c3,r_c4,r_m1,r_m2,r_m3,r_m4,r_r1,r_r2,r_r3,r_r4,0.499),
LOD_mat = cbind(LOD_c1,LOD_c2,LOD_c3,LOD_c4,LOD_m1,LOD_m2,LOD_m3,LOD_m4,LOD_r1,LOD_r2,LOD_r3,LOD_r4,0),
logL_mat = cbind(logL_c1,logL_c2,logL_c3,logL_c4,logL_m1,logL_m2,logL_m3,logL_m4,logL_r1,logL_r2,logL_r3,logL_r4,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling","coupling","coupling","coupling","mixed","mixed","mixed","mixed","repulsion","repulsion","repulsion","repulsion","unknown")
)
)
}

#' @rdname p4_functions
p4_2.1_2.2 <- function(x,p1,p2,ncores=1){


logL_cc1 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- 4*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(2) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - n10 - 2*n11 - 2*n12 - 2*n13 - n14 - n20 - 2*n21 - 2*n22 - 2*n23 - n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34)*log(3) + (n02 + n32)*(log(2 - 3*p1) + log(10 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n14 + n20)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n04 + n30)*(log(2 - 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,r))) + (n00 + n34)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + r^2)) + (n10 + n24)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n01 + n33)*(log(2 - 3*p1) + log(1 - 2*r + r^2) + log(4*(1 + r) - p2*(-3 + 6*r))) + (n03 + n31)*(log(2 - 3*p1) + 2*log(pmax(1e-6,r)) + log(8 - 4*r + p2*(-3 + 6*r))) + (n12 + n22)*log(6*p2*(3 - r + r^2) + 4*(9 - 5*r + 5*r^2) + p1*(6*(3 + 5*r - 5*r^2) - 9*p2*(-3 - r + r^2))) + (n13 + n21)*(log(pmax(1e-6,r)) + log(8*(5 - 5*r + 3*r^2) - 6*p2*(4 - 7*r + 6*r^2) + 3*p1*(-4*(2 - 5*r + 3*r^2) + 3*p2*(-2 - 7*r + 6*r^2)))) + (n11 + n23)*(log(pmax(1e-6,1 - r)) + log(4*(6 + (-2 + 3*p1)*r + (6 - 9*p1)*r^2) + 3*p2*(3*p1*(-3 - 5*r + 6*r^2) - 2*(3 - 5*r + 6*r^2))))
return(L)}
interlogL_cc1 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cc1,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc1 <- parallel::mcmapply(interlogL_cc1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cc1 <- -((x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(3) + log10(2 - 3*p1))) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(2 - 3*p1)) - (x[,"n_13"] + x[,"n_21"])*(-log10(2) + log10(26 + 3*p1*(-1 - 12*p2) - 12*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(-2*log10(2) + log10(2 - 3*p1) + log10(10 + 3*p2)) - (x[,"n_11"] + x[,"n_23"])*(-log10(2) + log10(4*(6 + (6 - 9*p1)/4 + (-2 + 3*p1)/2) + 3*(-4 - 12*p1)*p2)) - (x[,"n_12"] + x[,"n_22"])*log10(31 + (33*p2)/2 + p1*(51/2 + (117*p2)/4)) + (x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p1) + log10(10 + 3*p2) + log10(pmax(1e-6,1 - r_cc1)) + log10(pmax(1e-6,r_cc1))) + (x[,"n_14"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cc1)) + 2*log10(pmax(1e-6,r_cc1))) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 3*log10(pmax(1e-6,r_cc1))) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cc1)) + log10(1 - 2*r_cc1 + r_cc1^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_cc1)) + log10(1 - 2*r_cc1 + r_cc1^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(2 - 3*p1) + log10(1 - 2*r_cc1 + r_cc1^2) + log10(4*(1 + r_cc1) - p2*(-3 + 6*r_cc1))) + (x[,"n_03"] + x[,"n_31"])*(log10(2 - 3*p1) + 2*log10(pmax(1e-6,r_cc1)) + log10(8 - 4*r_cc1 + p2*(-3 + 6*r_cc1))) + (x[,"n_12"] + x[,"n_22"])*log10(6*p2*(3 - r_cc1 + r_cc1^2) + 4*(9 - 5*r_cc1 + 5*r_cc1^2) + p1*(6*(3 + 5*r_cc1 - 5*r_cc1^2) - 9*p2*(-3 - r_cc1 + r_cc1^2))) + (x[,"n_13"] + x[,"n_21"])*(log10(pmax(1e-6,r_cc1)) + log10(8*(5 - 5*r_cc1 + 3*r_cc1^2) - 6*p2*(4 - 7*r_cc1 + 6*r_cc1^2) + 3*p1*(-4*(2 - 5*r_cc1 + 3*r_cc1^2) + 3*p2*(-2 - 7*r_cc1 + 6*r_cc1^2)))) + (x[,"n_11"] + x[,"n_23"])*(log10(pmax(1e-6,1 - r_cc1)) + log10(4*(6 + (-2 + 3*p1)*r_cc1 + (6 - 9*p1)*r_cc1^2) + 3*p2*(3*p1*(-3 - 5*r_cc1 + 6*r_cc1^2) - 2*(3 - 5*r_cc1 + 6*r_cc1^2))))


logL_cc1 <- 4*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"])*log(3) + (x[,"n_02"] + x[,"n_32"])*(log(2 - 3*p1) + log(10 + 3*p2) + log(pmax(1e-6,1 - r_cc1)) + log(pmax(1e-6,r_cc1))) + (x[,"n_14"] + x[,"n_20"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cc1)) + 2*log(pmax(1e-6,r_cc1))) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,r_cc1))) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cc1)) + log(1 - 2*r_cc1 + r_cc1^2)) + (x[,"n_10"] + x[,"n_24"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_cc1)) + log(1 - 2*r_cc1 + r_cc1^2)) + (x[,"n_01"] + x[,"n_33"])*(log(2 - 3*p1) + log(1 - 2*r_cc1 + r_cc1^2) + log(4*(1 + r_cc1) - p2*(-3 + 6*r_cc1))) + (x[,"n_03"] + x[,"n_31"])*(log(2 - 3*p1) + 2*log(pmax(1e-6,r_cc1)) + log(8 - 4*r_cc1 + p2*(-3 + 6*r_cc1))) + (x[,"n_12"] + x[,"n_22"])*log(6*p2*(3 - r_cc1 + r_cc1^2) + 4*(9 - 5*r_cc1 + 5*r_cc1^2) + p1*(6*(3 + 5*r_cc1 - 5*r_cc1^2) - 9*p2*(-3 - r_cc1 + r_cc1^2))) + (x[,"n_13"] + x[,"n_21"])*(log(pmax(1e-6,r_cc1)) + log(8*(5 - 5*r_cc1 + 3*r_cc1^2) - 6*p2*(4 - 7*r_cc1 + 6*r_cc1^2) + 3*p1*(-4*(2 - 5*r_cc1 + 3*r_cc1^2) + 3*p2*(-2 - 7*r_cc1 + 6*r_cc1^2)))) + (x[,"n_11"] + x[,"n_23"])*(log(pmax(1e-6,1 - r_cc1)) + log(4*(6 + (-2 + 3*p1)*r_cc1 + (6 - 9*p1)*r_cc1^2) + 3*p2*(3*p1*(-3 - 5*r_cc1 + 6*r_cc1^2) - 2*(3 - 5*r_cc1 + 6*r_cc1^2))))


logL_cc2 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- 5*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(2) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - n10 - 2*n11 - 2*n12 - 2*n13 - n14 - n20 - 2*n21 - 2*n22 - 2*n23 - n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34)*log(3) + (n02 + n32)*(log(2 - 3*p1) + log(20 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n14 + n20)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n04 + n30)*(log(2 - 3*p1) + log(4 + 3*p2) + 3*log(pmax(1e-6,r))) + (n00 + n34)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + r^2)) + (n10 + n24)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n01 + n33)*(log(2 - 3*p1) + log(8 - 3*p2 + 8*r + 6*p2*r) + log(1 - 2*r + r^2)) + (n03 + n31)*(log(2 - 3*p1) + 2*log(pmax(1e-6,r)) + log(8*(2 - r) - p2*(-3 + 6*r))) + (n12 + n22)*log(-6*p2*(3 - r + r^2) + 8*(9 - 5*r + 5*r^2) + 3*p1*(4*(3 + 5*r - 5*r^2) + 3*p2*(-3 - r + r^2))) + (n13 + n21)*(log(pmax(1e-6,r)) + log(16*(5 - 5*r + 3*r^2) + 6*p2*(4 - 7*r + 6*r^2) - 3*p1*(8*(2 - 5*r + 3*r^2) + 3*p2*(-2 - 7*r + 6*r^2)))) + (n11 + n23)*(log(pmax(1e-6,1 - r)) + log(-8*(-6 + (2 - 3*p1)*r + (-6 + 9*p1)*r^2) - 3*p2*(3*p1*(-3 - 5*r + 6*r^2) - 2*(3 - 5*r + 6*r^2))))
return(L)}
interlogL_cc2 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cc2,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc2 <- parallel::mcmapply(interlogL_cc2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cc2 <- -((x[,"n_03"] + x[,"n_31"])*(log10(3) + log10(2 - 3*p1))) - (x[,"n_01"] + x[,"n_33"])*(log10(3) + log10(2 - 3*p1)) - (x[,"n_12"] + x[,"n_22"])*log10(62 + 3*p1*(17 - (39*p2)/4) - (33*p2)/2) - (x[,"n_02"] + x[,"n_32"])*(-2*log10(2) + log10(2 - 3*p1) + log10(20 - 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_13"] + x[,"n_21"])*(-log10(2) + log10(52 - 3*p1*(2 - 12*p2) + 12*p2)) - (x[,"n_11"] + x[,"n_23"])*(-log10(2) + log10(-8*(-6 + (2 - 3*p1)/2 + (-6 + 9*p1)/4) - 3*(-4 - 12*p1)*p2)) + (x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p1) + log10(20 - 3*p2) + log10(pmax(1e-6,1 - r_cc2)) + log10(pmax(1e-6,r_cc2))) + (x[,"n_14"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cc2)) + 2*log10(pmax(1e-6,r_cc2))) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + 3*log10(pmax(1e-6,r_cc2))) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cc2)) + log10(1 - 2*r_cc2 + r_cc2^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_cc2)) + log10(1 - 2*r_cc2 + r_cc2^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(2 - 3*p1) + log10(8 - 3*p2 + 8*r_cc2 + 6*p2*r_cc2) + log10(1 - 2*r_cc2 + r_cc2^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(2 - 3*p1) + 2*log10(pmax(1e-6,r_cc2)) + log10(8*(2 - r_cc2) - p2*(-3 + 6*r_cc2))) + (x[,"n_12"] + x[,"n_22"])*log10(-6*p2*(3 - r_cc2 + r_cc2^2) + 8*(9 - 5*r_cc2 + 5*r_cc2^2) + 3*p1*(4*(3 + 5*r_cc2 - 5*r_cc2^2) + 3*p2*(-3 - r_cc2 + r_cc2^2))) + (x[,"n_13"] + x[,"n_21"])*(log10(pmax(1e-6,r_cc2)) + log10(16*(5 - 5*r_cc2 + 3*r_cc2^2) + 6*p2*(4 - 7*r_cc2 + 6*r_cc2^2) - 3*p1*(8*(2 - 5*r_cc2 + 3*r_cc2^2) + 3*p2*(-2 - 7*r_cc2 + 6*r_cc2^2)))) + (x[,"n_11"] + x[,"n_23"])*(log10(pmax(1e-6,1 - r_cc2)) + log10(-8*(-6 + (2 - 3*p1)*r_cc2 + (-6 + 9*p1)*r_cc2^2) - 3*p2*(3*p1*(-3 - 5*r_cc2 + 6*r_cc2^2) - 2*(3 - 5*r_cc2 + 6*r_cc2^2))))


logL_cc2 <- 5*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"])*log(3) + (x[,"n_02"] + x[,"n_32"])*(log(2 - 3*p1) + log(20 - 3*p2) + log(pmax(1e-6,1 - r_cc2)) + log(pmax(1e-6,r_cc2))) + (x[,"n_14"] + x[,"n_20"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cc2)) + 2*log(pmax(1e-6,r_cc2))) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(4 + 3*p2) + 3*log(pmax(1e-6,r_cc2))) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cc2)) + log(1 - 2*r_cc2 + r_cc2^2)) + (x[,"n_10"] + x[,"n_24"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_cc2)) + log(1 - 2*r_cc2 + r_cc2^2)) + (x[,"n_01"] + x[,"n_33"])*(log(2 - 3*p1) + log(8 - 3*p2 + 8*r_cc2 + 6*p2*r_cc2) + log(1 - 2*r_cc2 + r_cc2^2)) + (x[,"n_03"] + x[,"n_31"])*(log(2 - 3*p1) + 2*log(pmax(1e-6,r_cc2)) + log(8*(2 - r_cc2) - p2*(-3 + 6*r_cc2))) + (x[,"n_12"] + x[,"n_22"])*log(-6*p2*(3 - r_cc2 + r_cc2^2) + 8*(9 - 5*r_cc2 + 5*r_cc2^2) + 3*p1*(4*(3 + 5*r_cc2 - 5*r_cc2^2) + 3*p2*(-3 - r_cc2 + r_cc2^2))) + (x[,"n_13"] + x[,"n_21"])*(log(pmax(1e-6,r_cc2)) + log(16*(5 - 5*r_cc2 + 3*r_cc2^2) + 6*p2*(4 - 7*r_cc2 + 6*r_cc2^2) - 3*p1*(8*(2 - 5*r_cc2 + 3*r_cc2^2) + 3*p2*(-2 - 7*r_cc2 + 6*r_cc2^2)))) + (x[,"n_11"] + x[,"n_23"])*(log(pmax(1e-6,1 - r_cc2)) + log(-8*(-6 + (2 - 3*p1)*r_cc2 + (-6 + 9*p1)*r_cc2^2) - 3*p2*(3*p1*(-3 - 5*r_cc2 + 6*r_cc2^2) - 2*(3 - 5*r_cc2 + 6*r_cc2^2))))


logL_cc3 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- 5*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(2) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - n10 - 2*n11 - 2*n12 - 2*n13 - n14 - n20 - 2*n21 - 2*n22 - 2*n23 - n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34)*log(3) + (n02 + n32)*(log(4 + 3*p1) + log(10 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n14 + n20)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n04 + n30)*(log(4 + 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,r))) + (n00 + n34)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + r^2)) + (n10 + n24)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n01 + n33)*(log(4 + 3*p1) + log(1 - 2*r + r^2) + log(4*(1 + r) - p2*(-3 + 6*r))) + (n03 + n31)*(log(4 + 3*p1) + 2*log(pmax(1e-6,r)) + log(8 - 4*r + p2*(-3 + 6*r))) + (n12 + n22)*log(3*p1*(3*p2*(-3 - r + r^2) + 2*(-3 - 5*r + 5*r^2)) + 4*(3*p2*(3 - r + r^2) + 2*(9 - 5*r + 5*r^2))) + (n13 + n21)*(log(pmax(1e-6,r)) + log(16*(5 - 5*r + 3*r^2) - 12*p2*(4 - 7*r + 6*r^2) + p1*(12*(2 - 5*r + 3*r^2) - 9*p2*(-2 - 7*r + 6*r^2)))) + (n11 + n23)*(log(pmax(1e-6,1 - r)) + log(4*(12 - (4 + 3*p1)*r + 3*(4 + 3*p1)*r^2) - 3*p2*(3*p1*(-3 - 5*r + 6*r^2) + 4*(3 - 5*r + 6*r^2))))
return(L)}
interlogL_cc3 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cc3,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc3 <- parallel::mcmapply(interlogL_cc3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cc3 <- -((x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(3) + log10(4 + 3*p1))) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(4 + 3*p1)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(-2*log10(2) + log10(4 + 3*p1) + log10(10 + 3*p2)) - (x[,"n_11"] + x[,"n_23"])*(-log10(2) + log10(4*(12 + (-4 - 3*p1)/2 + (3*(4 + 3*p1))/4) - 3*(8 - 12*p1)*p2)) - (x[,"n_12"] + x[,"n_22"])*log10(3*p1*(-17/2 - (39*p2)/4) + 4*(31/2 + (33*p2)/4)) - (x[,"n_13"] + x[,"n_21"])*(-log10(2) + log10(52 - 24*p2 + p1*(3 + 36*p2))) + (x[,"n_02"] + x[,"n_32"])*(log10(4 + 3*p1) + log10(10 + 3*p2) + log10(pmax(1e-6,1 - r_cc3)) + log10(pmax(1e-6,r_cc3))) + (x[,"n_14"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cc3)) + 2*log10(pmax(1e-6,r_cc3))) + (x[,"n_04"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + 3*log10(pmax(1e-6,r_cc3))) + (x[,"n_00"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cc3)) + log10(1 - 2*r_cc3 + r_cc3^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_cc3)) + log10(1 - 2*r_cc3 + r_cc3^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(4 + 3*p1) + log10(1 - 2*r_cc3 + r_cc3^2) + log10(4*(1 + r_cc3) - p2*(-3 + 6*r_cc3))) + (x[,"n_03"] + x[,"n_31"])*(log10(4 + 3*p1) + 2*log10(pmax(1e-6,r_cc3)) + log10(8 - 4*r_cc3 + p2*(-3 + 6*r_cc3))) + (x[,"n_12"] + x[,"n_22"])*log10(3*p1*(3*p2*(-3 - r_cc3 + r_cc3^2) + 2*(-3 - 5*r_cc3 + 5*r_cc3^2)) + 4*(3*p2*(3 - r_cc3 + r_cc3^2) + 2*(9 - 5*r_cc3 + 5*r_cc3^2))) + (x[,"n_13"] + x[,"n_21"])*(log10(pmax(1e-6,r_cc3)) + log10(16*(5 - 5*r_cc3 + 3*r_cc3^2) - 12*p2*(4 - 7*r_cc3 + 6*r_cc3^2) + p1*(12*(2 - 5*r_cc3 + 3*r_cc3^2) - 9*p2*(-2 - 7*r_cc3 + 6*r_cc3^2)))) + (x[,"n_11"] + x[,"n_23"])*(log10(pmax(1e-6,1 - r_cc3)) + log10(4*(12 - (4 + 3*p1)*r_cc3 + 3*(4 + 3*p1)*r_cc3^2) - 3*p2*(3*p1*(-3 - 5*r_cc3 + 6*r_cc3^2) + 4*(3 - 5*r_cc3 + 6*r_cc3^2))))


logL_cc3 <- 5*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"])*log(3) + (x[,"n_02"] + x[,"n_32"])*(log(4 + 3*p1) + log(10 + 3*p2) + log(pmax(1e-6,1 - r_cc3)) + log(pmax(1e-6,r_cc3))) + (x[,"n_14"] + x[,"n_20"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cc3)) + 2*log(pmax(1e-6,r_cc3))) + (x[,"n_04"] + x[,"n_30"])*(log(4 + 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,r_cc3))) + (x[,"n_00"] + x[,"n_34"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cc3)) + log(1 - 2*r_cc3 + r_cc3^2)) + (x[,"n_10"] + x[,"n_24"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_cc3)) + log(1 - 2*r_cc3 + r_cc3^2)) + (x[,"n_01"] + x[,"n_33"])*(log(4 + 3*p1) + log(1 - 2*r_cc3 + r_cc3^2) + log(4*(1 + r_cc3) - p2*(-3 + 6*r_cc3))) + (x[,"n_03"] + x[,"n_31"])*(log(4 + 3*p1) + 2*log(pmax(1e-6,r_cc3)) + log(8 - 4*r_cc3 + p2*(-3 + 6*r_cc3))) + (x[,"n_12"] + x[,"n_22"])*log(3*p1*(3*p2*(-3 - r_cc3 + r_cc3^2) + 2*(-3 - 5*r_cc3 + 5*r_cc3^2)) + 4*(3*p2*(3 - r_cc3 + r_cc3^2) + 2*(9 - 5*r_cc3 + 5*r_cc3^2))) + (x[,"n_13"] + x[,"n_21"])*(log(pmax(1e-6,r_cc3)) + log(16*(5 - 5*r_cc3 + 3*r_cc3^2) - 12*p2*(4 - 7*r_cc3 + 6*r_cc3^2) + p1*(12*(2 - 5*r_cc3 + 3*r_cc3^2) - 9*p2*(-2 - 7*r_cc3 + 6*r_cc3^2)))) + (x[,"n_11"] + x[,"n_23"])*(log(pmax(1e-6,1 - r_cc3)) + log(4*(12 - (4 + 3*p1)*r_cc3 + 3*(4 + 3*p1)*r_cc3^2) - 3*p2*(3*p1*(-3 - 5*r_cc3 + 6*r_cc3^2) + 4*(3 - 5*r_cc3 + 6*r_cc3^2))))


logL_cc4 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- 6*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(2) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - n10 - 2*n11 - 2*n12 - 2*n13 - n14 - n20 - 2*n21 - 2*n22 - 2*n23 - n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34)*log(3) + (n02 + n32)*(log(4 + 3*p1) + log(20 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n14 + n20)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n04 + n30)*(log(4 + 3*p1) + log(4 + 3*p2) + 3*log(pmax(1e-6,r))) + (n00 + n34)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + r^2)) + (n10 + n24)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n01 + n33)*(log(4 + 3*p1) + log(8 - 3*p2 + 8*r + 6*p2*r) + log(1 - 2*r + r^2)) + (n03 + n31)*(log(4 + 3*p1) + 2*log(pmax(1e-6,r)) + log(8*(2 - r) - p2*(-3 + 6*r))) + (n12 + n22)*log(p1*(-9*p2*(-3 - r + r^2) + 12*(-3 - 5*r + 5*r^2)) - 4*(3*p2*(3 - r + r^2) - 4*(9 - 5*r + 5*r^2))) + (n13 + n21)*(log(pmax(1e-6,r)) + log(3*p1*(8*(2 - 5*r + 3*r^2) + 3*p2*(-2 - 7*r + 6*r^2)) + 4*(8*(5 - 5*r + 3*r^2) + 3*p2*(4 - 7*r + 6*r^2)))) + (n11 + n23)*(log(pmax(1e-6,1 - r)) + log(8*(12 - (4 + 3*p1)*r + 3*(4 + 3*p1)*r^2) + 3*p2*(3*p1*(-3 - 5*r + 6*r^2) + 4*(3 - 5*r + 6*r^2))))
return(L)}
interlogL_cc4 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cc4,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc4 <- parallel::mcmapply(interlogL_cc4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cc4 <- -((x[,"n_03"] + x[,"n_31"])*(log10(3) + log10(4 + 3*p1))) - (x[,"n_01"] + x[,"n_33"])*(log10(3) + log10(4 + 3*p1)) - (x[,"n_02"] + x[,"n_32"])*(-2*log10(2) + log10(4 + 3*p1) + log10(20 - 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_11"] + x[,"n_23"])*(-log10(2) + log10(8*(12 + (-4 - 3*p1)/2 + (3*(4 + 3*p1))/4) + 3*(8 - 12*p1)*p2)) - (x[,"n_13"] + x[,"n_21"])*(-log10(2) + log10(3*p1*(2 - 12*p2) + 4*(26 + 6*p2))) - (x[,"n_12"] + x[,"n_22"])*log10(-4*(-31 + (33*p2)/4) + p1*(-51 + (117*p2)/4)) + (x[,"n_02"] + x[,"n_32"])*(log10(4 + 3*p1) + log10(20 - 3*p2) + log10(pmax(1e-6,1 - r_cc4)) + log10(pmax(1e-6,r_cc4))) + (x[,"n_14"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cc4)) + 2*log10(pmax(1e-6,r_cc4))) + (x[,"n_04"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + 3*log10(pmax(1e-6,r_cc4))) + (x[,"n_00"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cc4)) + log10(1 - 2*r_cc4 + r_cc4^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_cc4)) + log10(1 - 2*r_cc4 + r_cc4^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(4 + 3*p1) + log10(8 - 3*p2 + 8*r_cc4 + 6*p2*r_cc4) + log10(1 - 2*r_cc4 + r_cc4^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(4 + 3*p1) + 2*log10(pmax(1e-6,r_cc4)) + log10(8*(2 - r_cc4) - p2*(-3 + 6*r_cc4))) + (x[,"n_12"] + x[,"n_22"])*log10(p1*(-9*p2*(-3 - r_cc4 + r_cc4^2) + 12*(-3 - 5*r_cc4 + 5*r_cc4^2)) - 4*(3*p2*(3 - r_cc4 + r_cc4^2) - 4*(9 - 5*r_cc4 + 5*r_cc4^2))) + (x[,"n_13"] + x[,"n_21"])*(log10(pmax(1e-6,r_cc4)) + log10(3*p1*(8*(2 - 5*r_cc4 + 3*r_cc4^2) + 3*p2*(-2 - 7*r_cc4 + 6*r_cc4^2)) + 4*(8*(5 - 5*r_cc4 + 3*r_cc4^2) + 3*p2*(4 - 7*r_cc4 + 6*r_cc4^2)))) + (x[,"n_11"] + x[,"n_23"])*(log10(pmax(1e-6,1 - r_cc4)) + log10(8*(12 - (4 + 3*p1)*r_cc4 + 3*(4 + 3*p1)*r_cc4^2) + 3*p2*(3*p1*(-3 - 5*r_cc4 + 6*r_cc4^2) + 4*(3 - 5*r_cc4 + 6*r_cc4^2))))


logL_cc4 <- 6*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"])*log(3) + (x[,"n_02"] + x[,"n_32"])*(log(4 + 3*p1) + log(20 - 3*p2) + log(pmax(1e-6,1 - r_cc4)) + log(pmax(1e-6,r_cc4))) + (x[,"n_14"] + x[,"n_20"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cc4)) + 2*log(pmax(1e-6,r_cc4))) + (x[,"n_04"] + x[,"n_30"])*(log(4 + 3*p1) + log(4 + 3*p2) + 3*log(pmax(1e-6,r_cc4))) + (x[,"n_00"] + x[,"n_34"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cc4)) + log(1 - 2*r_cc4 + r_cc4^2)) + (x[,"n_10"] + x[,"n_24"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_cc4)) + log(1 - 2*r_cc4 + r_cc4^2)) + (x[,"n_01"] + x[,"n_33"])*(log(4 + 3*p1) + log(8 - 3*p2 + 8*r_cc4 + 6*p2*r_cc4) + log(1 - 2*r_cc4 + r_cc4^2)) + (x[,"n_03"] + x[,"n_31"])*(log(4 + 3*p1) + 2*log(pmax(1e-6,r_cc4)) + log(8*(2 - r_cc4) - p2*(-3 + 6*r_cc4))) + (x[,"n_12"] + x[,"n_22"])*log(p1*(-9*p2*(-3 - r_cc4 + r_cc4^2) + 12*(-3 - 5*r_cc4 + 5*r_cc4^2)) - 4*(3*p2*(3 - r_cc4 + r_cc4^2) - 4*(9 - 5*r_cc4 + 5*r_cc4^2))) + (x[,"n_13"] + x[,"n_21"])*(log(pmax(1e-6,r_cc4)) + log(3*p1*(8*(2 - 5*r_cc4 + 3*r_cc4^2) + 3*p2*(-2 - 7*r_cc4 + 6*r_cc4^2)) + 4*(8*(5 - 5*r_cc4 + 3*r_cc4^2) + 3*p2*(4 - 7*r_cc4 + 6*r_cc4^2)))) + (x[,"n_11"] + x[,"n_23"])*(log(pmax(1e-6,1 - r_cc4)) + log(8*(12 - (4 + 3*p1)*r_cc4 + 3*(4 + 3*p1)*r_cc4^2) + 3*p2*(3*p1*(-3 - 5*r_cc4 + 6*r_cc4^2) + 4*(3 - 5*r_cc4 + 6*r_cc4^2))))


logL_cr1 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(5*log(2) + 2*log(3)) + (n04 + n30)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n00 + n34)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n14 + n20)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n10 + n24)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + 3*r^2)) + (n03 + n31)*(log(2 - 3*p1) + log(pmax(1e-6,r)) + log(8*(1 - r + r^2) + 3*p2*(2 - 5*r + 2*r^2))) + (n01 + n33)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(8*(1 - r + r^2) + 3*p2*(-1 + r + 2*r^2))) + (n02 + n32)*(log(2 - 3*p1) + log(4 + 4*r - 4*r^2 + 3*p2*(1 - 5*r + 5*r^2))) + (n12 + n22)*log(8*(8 - r + r^2) - 6*p2*(4 - 5*r + 5*r^2) + 3*p1*(4*(4 + r - r^2) + 3*p2*(-2 - 5*r + 5*r^2))) + (n13 + n21)*log(16*(2 - 2*r + 4*r^2 - 3*r^3) - 6*p2*(-4 + 10*r - 11*r^2 + 6*r^3) + 3*p1*(8*(1 - r - 4*r^2 + 3*r^3) + 3*p2*(2 + 4*r - 11*r^2 + 6*r^3))) + (n11 + n23)*log(16*(1 + 3*r - 5*r^2 + 3*r^3) + 6*p2*(-1 + 6*r - 7*r^2 + 6*r^3) - 3*p1*(8*(1 - 5*r^2 + 3*r^3) + 3*p2*(-1 - 7*r^2 + 6*r^3)))
return(L)}
interlogL_cr1 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cr1,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr1 <- parallel::mcmapply(interlogL_cr1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cr1 <- -((x[,"n_03"] + x[,"n_31"])*(log10(3) + log10(2 - 3*p1))) - (x[,"n_01"] + x[,"n_33"])*(log10(3) + log10(2 - 3*p1)) - (x[,"n_12"] + x[,"n_22"])*log10(62 + 3*p1*(17 - (39*p2)/4) - (33*p2)/2) - (x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p1) + log10(5 - (3*p2)/4)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(3) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(3) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_11"] + x[,"n_23"])*log10(26 - 3*p1*(1 - 6*p2) + 6*p2) - (x[,"n_13"] + x[,"n_21"])*log10(26 + 6*p2 + 3*p1*(-1 + 6*p2)) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cr1)) + 2*log10(pmax(1e-6,r_cr1))) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_cr1)) + log10(1 - 2*r_cr1 + r_cr1^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_cr1)) + log10(2 - 4*r_cr1 + 3*r_cr1^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cr1)) + log10(1 - 2*r_cr1 + 3*r_cr1^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(2 - 3*p1) + log10(pmax(1e-6,r_cr1)) + log10(8*(1 - r_cr1 + r_cr1^2) + 3*p2*(2 - 5*r_cr1 + 2*r_cr1^2))) + (x[,"n_01"] + x[,"n_33"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_cr1)) + log10(8*(1 - r_cr1 + r_cr1^2) + 3*p2*(-1 + r_cr1 + 2*r_cr1^2))) + (x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p1) + log10(4 + 4*r_cr1 - 4*r_cr1^2 + 3*p2*(1 - 5*r_cr1 + 5*r_cr1^2))) + (x[,"n_12"] + x[,"n_22"])*log10(8*(8 - r_cr1 + r_cr1^2) - 6*p2*(4 - 5*r_cr1 + 5*r_cr1^2) + 3*p1*(4*(4 + r_cr1 - r_cr1^2) + 3*p2*(-2 - 5*r_cr1 + 5*r_cr1^2))) + (x[,"n_13"] + x[,"n_21"])*log10(16*(2 - 2*r_cr1 + 4*r_cr1^2 - 3*r_cr1^3) - 6*p2*(-4 + 10*r_cr1 - 11*r_cr1^2 + 6*r_cr1^3) + 3*p1*(8*(1 - r_cr1 - 4*r_cr1^2 + 3*r_cr1^3) + 3*p2*(2 + 4*r_cr1 - 11*r_cr1^2 + 6*r_cr1^3))) + (x[,"n_11"] + x[,"n_23"])*log10(16*(1 + 3*r_cr1 - 5*r_cr1^2 + 3*r_cr1^3) + 6*p2*(-1 + 6*r_cr1 - 7*r_cr1^2 + 6*r_cr1^3) - 3*p1*(8*(1 - 5*r_cr1^2 + 3*r_cr1^3) + 3*p2*(-1 - 7*r_cr1^2 + 6*r_cr1^3)))


logL_cr1 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(5*log(2) + 2*log(3)) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cr1)) + 2*log(pmax(1e-6,r_cr1))) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_cr1)) + log(1 - 2*r_cr1 + r_cr1^2)) + (x[,"n_14"] + x[,"n_20"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_cr1)) + log(2 - 4*r_cr1 + 3*r_cr1^2)) + (x[,"n_10"] + x[,"n_24"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cr1)) + log(1 - 2*r_cr1 + 3*r_cr1^2)) + (x[,"n_03"] + x[,"n_31"])*(log(2 - 3*p1) + log(pmax(1e-6,r_cr1)) + log(8*(1 - r_cr1 + r_cr1^2) + 3*p2*(2 - 5*r_cr1 + 2*r_cr1^2))) + (x[,"n_01"] + x[,"n_33"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_cr1)) + log(8*(1 - r_cr1 + r_cr1^2) + 3*p2*(-1 + r_cr1 + 2*r_cr1^2))) + (x[,"n_02"] + x[,"n_32"])*(log(2 - 3*p1) + log(4 + 4*r_cr1 - 4*r_cr1^2 + 3*p2*(1 - 5*r_cr1 + 5*r_cr1^2))) + (x[,"n_12"] + x[,"n_22"])*log(8*(8 - r_cr1 + r_cr1^2) - 6*p2*(4 - 5*r_cr1 + 5*r_cr1^2) + 3*p1*(4*(4 + r_cr1 - r_cr1^2) + 3*p2*(-2 - 5*r_cr1 + 5*r_cr1^2))) + (x[,"n_13"] + x[,"n_21"])*log(16*(2 - 2*r_cr1 + 4*r_cr1^2 - 3*r_cr1^3) - 6*p2*(-4 + 10*r_cr1 - 11*r_cr1^2 + 6*r_cr1^3) + 3*p1*(8*(1 - r_cr1 - 4*r_cr1^2 + 3*r_cr1^3) + 3*p2*(2 + 4*r_cr1 - 11*r_cr1^2 + 6*r_cr1^3))) + (x[,"n_11"] + x[,"n_23"])*log(16*(1 + 3*r_cr1 - 5*r_cr1^2 + 3*r_cr1^3) + 6*p2*(-1 + 6*r_cr1 - 7*r_cr1^2 + 6*r_cr1^3) - 3*p1*(8*(1 - 5*r_cr1^2 + 3*r_cr1^3) + 3*p2*(-1 - 7*r_cr1^2 + 6*r_cr1^3)))


logL_cr2 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-4*n00 - 3*n01 - 4*n02 - 3*n03 - 4*n04 - 4*n10 - 4*n11 - 4*n12 - 4*n13 - 4*n14 - 4*n20 - 4*n21 - 4*n22 - 4*n23 - 4*n24 - 4*n30 - 3*n31 - 4*n32 - 3*n33 - 4*n34)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(3) + (n04 + n30)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n03 + n31)*(log(2 - 3*p1) + log(pmax(1e-6,r)) + log((2 - 3*p2)*(-1 + r)^2 + ((4 + 3*p2)*r)/2)) + (n00 + n34)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n14 + n20)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n10 + n24)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + 3*r^2)) + (n01 + n33)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(-((4 + 3*p2)*(-1 + r))/2 + (2 - 3*p2)*r^2)) + (n02 + n32)*(log(2 - 3*p1) + log(2*(1 + r - r^2) - 3*p2*(1 - 5*r + 5*r^2))) + (n12 + n22)*log(4*(8 - r + r^2) + 6*p2*(4 - 5*r + 5*r^2) + p1*(6*(4 + r - r^2) - 9*p2*(-2 - 5*r + 5*r^2))) + (n13 + n21)*log(-3*p1*(4*(-1 + r + 4*r^2 - 3*r^3) + 3*p2*(2 + 4*r - 11*r^2 + 6*r^3)) + 2*(8 - 8*r + 16*r^2 - 12*r^3 + 3*p2*(-4 + 10*r - 11*r^2 + 6*r^3))) + (n11 + n23)*log(8*(1 + 3*r - 5*r^2 + 3*r^3) - 6*p2*(-1 + 6*r - 7*r^2 + 6*r^3) + 3*p1*(-4*(1 - 5*r^2 + 3*r^3) + 3*p2*(-1 - 7*r^2 + 6*r^3)))
return(L)}
interlogL_cr2 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cr2,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr2 <- parallel::mcmapply(interlogL_cr2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cr2 <- -((x[,"n_11"] + x[,"n_23"])*log10(13 + 3*p1*(-1/2 - 6*p2) - 6*p2)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(3) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(3) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p1) + log10(5/2 + (3*p2)/4)) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(2 - 3*p1) + log10((2 - 3*p2)/4 + (4 + 3*p2)/4)) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(2 - 3*p1) + log10((2 - 3*p2)/4 + (4 + 3*p2)/4)) - (x[,"n_13"] + x[,"n_21"])*log10(2*(13/2 - 3*p2) - 3*p1*(1/2 + 6*p2)) - (x[,"n_12"] + x[,"n_22"])*log10(31 + (33*p2)/2 + p1*(51/2 + (117*p2)/4)) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cr2)) + 2*log10(pmax(1e-6,r_cr2))) + (x[,"n_03"] + x[,"n_31"])*(log10(2 - 3*p1) + log10(pmax(1e-6,r_cr2)) + log10((2 - 3*p2)*(-1 + r_cr2)^2 + ((4 + 3*p2)*r_cr2)/2)) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_cr2)) + log10(1 - 2*r_cr2 + r_cr2^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_cr2)) + log10(2 - 4*r_cr2 + 3*r_cr2^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cr2)) + log10(1 - 2*r_cr2 + 3*r_cr2^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_cr2)) + log10(-((4 + 3*p2)*(-1 + r_cr2))/2 + (2 - 3*p2)*r_cr2^2)) + (x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p1) + log10(2*(1 + r_cr2 - r_cr2^2) - 3*p2*(1 - 5*r_cr2 + 5*r_cr2^2))) + (x[,"n_12"] + x[,"n_22"])*log10(4*(8 - r_cr2 + r_cr2^2) + 6*p2*(4 - 5*r_cr2 + 5*r_cr2^2) + p1*(6*(4 + r_cr2 - r_cr2^2) - 9*p2*(-2 - 5*r_cr2 + 5*r_cr2^2))) + (x[,"n_13"] + x[,"n_21"])*log10(-3*p1*(4*(-1 + r_cr2 + 4*r_cr2^2 - 3*r_cr2^3) + 3*p2*(2 + 4*r_cr2 - 11*r_cr2^2 + 6*r_cr2^3)) + 2*(8 - 8*r_cr2 + 16*r_cr2^2 - 12*r_cr2^3 + 3*p2*(-4 + 10*r_cr2 - 11*r_cr2^2 + 6*r_cr2^3))) + (x[,"n_11"] + x[,"n_23"])*log10(8*(1 + 3*r_cr2 - 5*r_cr2^2 + 3*r_cr2^3) - 6*p2*(-1 + 6*r_cr2 - 7*r_cr2^2 + 6*r_cr2^3) + 3*p1*(-4*(1 - 5*r_cr2^2 + 3*r_cr2^3) + 3*p2*(-1 - 7*r_cr2^2 + 6*r_cr2^3)))


logL_cr2 <- (-4*x[,"n_00"] - 3*x[,"n_01"] - 4*x[,"n_02"] - 3*x[,"n_03"] - 4*x[,"n_04"] - 4*x[,"n_10"] - 4*x[,"n_11"] - 4*x[,"n_12"] - 4*x[,"n_13"] - 4*x[,"n_14"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 4*x[,"n_23"] - 4*x[,"n_24"] - 4*x[,"n_30"] - 3*x[,"n_31"] - 4*x[,"n_32"] - 3*x[,"n_33"] - 4*x[,"n_34"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(3) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cr2)) + 2*log(pmax(1e-6,r_cr2))) + (x[,"n_03"] + x[,"n_31"])*(log(2 - 3*p1) + log(pmax(1e-6,r_cr2)) + log((2 - 3*p2)*(-1 + r_cr2)^2 + ((4 + 3*p2)*r_cr2)/2)) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_cr2)) + log(1 - 2*r_cr2 + r_cr2^2)) + (x[,"n_14"] + x[,"n_20"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_cr2)) + log(2 - 4*r_cr2 + 3*r_cr2^2)) + (x[,"n_10"] + x[,"n_24"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cr2)) + log(1 - 2*r_cr2 + 3*r_cr2^2)) + (x[,"n_01"] + x[,"n_33"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_cr2)) + log(-((4 + 3*p2)*(-1 + r_cr2))/2 + (2 - 3*p2)*r_cr2^2)) + (x[,"n_02"] + x[,"n_32"])*(log(2 - 3*p1) + log(2*(1 + r_cr2 - r_cr2^2) - 3*p2*(1 - 5*r_cr2 + 5*r_cr2^2))) + (x[,"n_12"] + x[,"n_22"])*log(4*(8 - r_cr2 + r_cr2^2) + 6*p2*(4 - 5*r_cr2 + 5*r_cr2^2) + p1*(6*(4 + r_cr2 - r_cr2^2) - 9*p2*(-2 - 5*r_cr2 + 5*r_cr2^2))) + (x[,"n_13"] + x[,"n_21"])*log(-3*p1*(4*(-1 + r_cr2 + 4*r_cr2^2 - 3*r_cr2^3) + 3*p2*(2 + 4*r_cr2 - 11*r_cr2^2 + 6*r_cr2^3)) + 2*(8 - 8*r_cr2 + 16*r_cr2^2 - 12*r_cr2^3 + 3*p2*(-4 + 10*r_cr2 - 11*r_cr2^2 + 6*r_cr2^3))) + (x[,"n_11"] + x[,"n_23"])*log(8*(1 + 3*r_cr2 - 5*r_cr2^2 + 3*r_cr2^3) - 6*p2*(-1 + 6*r_cr2 - 7*r_cr2^2 + 6*r_cr2^3) + 3*p1*(-4*(1 - 5*r_cr2^2 + 3*r_cr2^3) + 3*p2*(-1 - 7*r_cr2^2 + 6*r_cr2^3)))


logL_cr3 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(6*log(2) + 2*log(3)) + (n04 + n30)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n00 + n34)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n14 + n20)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n10 + n24)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + 3*r^2)) + (n03 + n31)*(log(4 + 3*p1) + log(pmax(1e-6,r)) + log(8*(1 - r + r^2) + 3*p2*(2 - 5*r + 2*r^2))) + (n01 + n33)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(8*(1 - r + r^2) + 3*p2*(-1 + r + 2*r^2))) + (n02 + n32)*(log(4 + 3*p1) + log(4 + 4*r - 4*r^2 + 3*p2*(1 - 5*r + 5*r^2))) + (n12 + n22)*log(-3*p1*(4*(4 + r - r^2) + 3*p2*(-2 - 5*r + 5*r^2)) + 4*(4*(8 - r + r^2) - 3*p2*(4 - 5*r + 5*r^2))) + (n13 + n21)*log(-3*p1*(8*(1 - r - 4*r^2 + 3*r^3) + 3*p2*(2 + 4*r - 11*r^2 + 6*r^3)) + 4*(-8*(-2 + 2*r - 4*r^2 + 3*r^3) - 3*p2*(-4 + 10*r - 11*r^2 + 6*r^3))) + (n11 + n23)*log(3*p1*(8*(1 - 5*r^2 + 3*r^3) + 3*p2*(-1 - 7*r^2 + 6*r^3)) + 4*(8*(1 + 3*r - 5*r^2 + 3*r^3) + 3*p2*(-1 + 6*r - 7*r^2 + 6*r^3)))
return(L)}
interlogL_cr3 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cr3,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr3 <- parallel::mcmapply(interlogL_cr3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cr3 <- -((x[,"n_03"] + x[,"n_31"])*(log10(3) + log10(4 + 3*p1))) - (x[,"n_01"] + x[,"n_33"])*(log10(3) + log10(4 + 3*p1)) - (x[,"n_12"] + x[,"n_22"])*log10(-3*p1*(17 - (39*p2)/4) + 4*(31 - (33*p2)/4)) - (x[,"n_02"] + x[,"n_32"])*(log10(4 + 3*p1) + log10(5 - (3*p2)/4)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(3) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(3) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_11"] + x[,"n_23"])*log10(3*p1*(1 - 6*p2) + 4*(13 + 3*p2)) - (x[,"n_13"] + x[,"n_21"])*log10(4*(13 + 3*p2) - 3*p1*(-1 + 6*p2)) + (x[,"n_04"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cr3)) + 2*log10(pmax(1e-6,r_cr3))) + (x[,"n_00"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_cr3)) + log10(1 - 2*r_cr3 + r_cr3^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_cr3)) + log10(2 - 4*r_cr3 + 3*r_cr3^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cr3)) + log10(1 - 2*r_cr3 + 3*r_cr3^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(4 + 3*p1) + log10(pmax(1e-6,r_cr3)) + log10(8*(1 - r_cr3 + r_cr3^2) + 3*p2*(2 - 5*r_cr3 + 2*r_cr3^2))) + (x[,"n_01"] + x[,"n_33"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_cr3)) + log10(8*(1 - r_cr3 + r_cr3^2) + 3*p2*(-1 + r_cr3 + 2*r_cr3^2))) + (x[,"n_02"] + x[,"n_32"])*(log10(4 + 3*p1) + log10(4 + 4*r_cr3 - 4*r_cr3^2 + 3*p2*(1 - 5*r_cr3 + 5*r_cr3^2))) + (x[,"n_12"] + x[,"n_22"])*log10(-3*p1*(4*(4 + r_cr3 - r_cr3^2) + 3*p2*(-2 - 5*r_cr3 + 5*r_cr3^2)) + 4*(4*(8 - r_cr3 + r_cr3^2) - 3*p2*(4 - 5*r_cr3 + 5*r_cr3^2))) + (x[,"n_13"] + x[,"n_21"])*log10(-3*p1*(8*(1 - r_cr3 - 4*r_cr3^2 + 3*r_cr3^3) + 3*p2*(2 + 4*r_cr3 - 11*r_cr3^2 + 6*r_cr3^3)) + 4*(-8*(-2 + 2*r_cr3 - 4*r_cr3^2 + 3*r_cr3^3) - 3*p2*(-4 + 10*r_cr3 - 11*r_cr3^2 + 6*r_cr3^3))) + (x[,"n_11"] + x[,"n_23"])*log10(3*p1*(8*(1 - 5*r_cr3^2 + 3*r_cr3^3) + 3*p2*(-1 - 7*r_cr3^2 + 6*r_cr3^3)) + 4*(8*(1 + 3*r_cr3 - 5*r_cr3^2 + 3*r_cr3^3) + 3*p2*(-1 + 6*r_cr3 - 7*r_cr3^2 + 6*r_cr3^3)))


logL_cr3 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(6*log(2) + 2*log(3)) + (x[,"n_04"] + x[,"n_30"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cr3)) + 2*log(pmax(1e-6,r_cr3))) + (x[,"n_00"] + x[,"n_34"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_cr3)) + log(1 - 2*r_cr3 + r_cr3^2)) + (x[,"n_14"] + x[,"n_20"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_cr3)) + log(2 - 4*r_cr3 + 3*r_cr3^2)) + (x[,"n_10"] + x[,"n_24"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cr3)) + log(1 - 2*r_cr3 + 3*r_cr3^2)) + (x[,"n_03"] + x[,"n_31"])*(log(4 + 3*p1) + log(pmax(1e-6,r_cr3)) + log(8*(1 - r_cr3 + r_cr3^2) + 3*p2*(2 - 5*r_cr3 + 2*r_cr3^2))) + (x[,"n_01"] + x[,"n_33"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_cr3)) + log(8*(1 - r_cr3 + r_cr3^2) + 3*p2*(-1 + r_cr3 + 2*r_cr3^2))) + (x[,"n_02"] + x[,"n_32"])*(log(4 + 3*p1) + log(4 + 4*r_cr3 - 4*r_cr3^2 + 3*p2*(1 - 5*r_cr3 + 5*r_cr3^2))) + (x[,"n_12"] + x[,"n_22"])*log(-3*p1*(4*(4 + r_cr3 - r_cr3^2) + 3*p2*(-2 - 5*r_cr3 + 5*r_cr3^2)) + 4*(4*(8 - r_cr3 + r_cr3^2) - 3*p2*(4 - 5*r_cr3 + 5*r_cr3^2))) + (x[,"n_13"] + x[,"n_21"])*log(-3*p1*(8*(1 - r_cr3 - 4*r_cr3^2 + 3*r_cr3^3) + 3*p2*(2 + 4*r_cr3 - 11*r_cr3^2 + 6*r_cr3^3)) + 4*(-8*(-2 + 2*r_cr3 - 4*r_cr3^2 + 3*r_cr3^3) - 3*p2*(-4 + 10*r_cr3 - 11*r_cr3^2 + 6*r_cr3^3))) + (x[,"n_11"] + x[,"n_23"])*log(3*p1*(8*(1 - 5*r_cr3^2 + 3*r_cr3^3) + 3*p2*(-1 - 7*r_cr3^2 + 6*r_cr3^3)) + 4*(8*(1 + 3*r_cr3 - 5*r_cr3^2 + 3*r_cr3^3) + 3*p2*(-1 + 6*r_cr3 - 7*r_cr3^2 + 6*r_cr3^3)))


logL_cr4 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(5*log(2) + 2*log(3)) + (n04 + n30)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n00 + n34)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n14 + n20)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n10 + n24)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + 3*r^2)) + (n03 + n31)*(log(4 + 3*p1) + log(pmax(1e-6,r)) + log(4*(1 - r + r^2) - 3*p2*(2 - 5*r + 2*r^2))) + (n01 + n33)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(4*(1 - r + r^2) - 3*p2*(-1 + r + 2*r^2))) + (n02 + n32)*(log(4 + 3*p1) + log(2*(1 + r - r^2) - 3*p2*(1 - 5*r + 5*r^2))) + (n12 + n22)*log(8*(8 - r + r^2) + 12*p2*(4 - 5*r + 5*r^2) + 3*p1*(2*(-4 - r + r^2) + 3*p2*(-2 - 5*r + 5*r^2))) + (n11 + n23)*log(16*(1 + 3*r - 5*r^2 + 3*r^3) - 12*p2*(-1 + 6*r - 7*r^2 + 6*r^3) + p1*(p2*(9 + 63*r^2 - 54*r^3) + 12*(1 - 5*r^2 + 3*r^3))) + (n13 + n21)*log(3*p1*(4*(-1 + r + 4*r^2 - 3*r^3) + 3*p2*(2 + 4*r - 11*r^2 + 6*r^3)) + 4*(8 - 8*r + 16*r^2 - 12*r^3 + 3*p2*(-4 + 10*r - 11*r^2 + 6*r^3)))
return(L)}
interlogL_cr4 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_cr4,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr4 <- parallel::mcmapply(interlogL_cr4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_cr4 <- -((x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(3) + log10(4 + 3*p1))) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(4 + 3*p1)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(3) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(3) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(log10(4 + 3*p1) + log10(5/2 + (3*p2)/4)) - (x[,"n_12"] + x[,"n_22"])*log10(62 + 3*p1*(-17/2 - (39*p2)/4) + 33*p2) - (x[,"n_13"] + x[,"n_21"])*log10(4*(13/2 - 3*p2) + 3*p1*(1/2 + 6*p2)) - (x[,"n_11"] + x[,"n_23"])*log10(26 - 12*p2 + p1*(3/2 + 18*p2)) + (x[,"n_04"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cr4)) + 2*log10(pmax(1e-6,r_cr4))) + (x[,"n_00"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_cr4)) + log10(1 - 2*r_cr4 + r_cr4^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_cr4)) + log10(2 - 4*r_cr4 + 3*r_cr4^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cr4)) + log10(1 - 2*r_cr4 + 3*r_cr4^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(4 + 3*p1) + log10(pmax(1e-6,r_cr4)) + log10(4*(1 - r_cr4 + r_cr4^2) - 3*p2*(2 - 5*r_cr4 + 2*r_cr4^2))) + (x[,"n_01"] + x[,"n_33"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_cr4)) + log10(4*(1 - r_cr4 + r_cr4^2) - 3*p2*(-1 + r_cr4 + 2*r_cr4^2))) + (x[,"n_02"] + x[,"n_32"])*(log10(4 + 3*p1) + log10(2*(1 + r_cr4 - r_cr4^2) - 3*p2*(1 - 5*r_cr4 + 5*r_cr4^2))) + (x[,"n_12"] + x[,"n_22"])*log10(8*(8 - r_cr4 + r_cr4^2) + 12*p2*(4 - 5*r_cr4 + 5*r_cr4^2) + 3*p1*(2*(-4 - r_cr4 + r_cr4^2) + 3*p2*(-2 - 5*r_cr4 + 5*r_cr4^2))) + (x[,"n_11"] + x[,"n_23"])*log10(16*(1 + 3*r_cr4 - 5*r_cr4^2 + 3*r_cr4^3) - 12*p2*(-1 + 6*r_cr4 - 7*r_cr4^2 + 6*r_cr4^3) + p1*(p2*(9 + 63*r_cr4^2 - 54*r_cr4^3) + 12*(1 - 5*r_cr4^2 + 3*r_cr4^3))) + (x[,"n_13"] + x[,"n_21"])*log10(3*p1*(4*(-1 + r_cr4 + 4*r_cr4^2 - 3*r_cr4^3) + 3*p2*(2 + 4*r_cr4 - 11*r_cr4^2 + 6*r_cr4^3)) + 4*(8 - 8*r_cr4 + 16*r_cr4^2 - 12*r_cr4^3 + 3*p2*(-4 + 10*r_cr4 - 11*r_cr4^2 + 6*r_cr4^3)))


logL_cr4 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(5*log(2) + 2*log(3)) + (x[,"n_04"] + x[,"n_30"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cr4)) + 2*log(pmax(1e-6,r_cr4))) + (x[,"n_00"] + x[,"n_34"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_cr4)) + log(1 - 2*r_cr4 + r_cr4^2)) + (x[,"n_14"] + x[,"n_20"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_cr4)) + log(2 - 4*r_cr4 + 3*r_cr4^2)) + (x[,"n_10"] + x[,"n_24"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cr4)) + log(1 - 2*r_cr4 + 3*r_cr4^2)) + (x[,"n_03"] + x[,"n_31"])*(log(4 + 3*p1) + log(pmax(1e-6,r_cr4)) + log(4*(1 - r_cr4 + r_cr4^2) - 3*p2*(2 - 5*r_cr4 + 2*r_cr4^2))) + (x[,"n_01"] + x[,"n_33"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_cr4)) + log(4*(1 - r_cr4 + r_cr4^2) - 3*p2*(-1 + r_cr4 + 2*r_cr4^2))) + (x[,"n_02"] + x[,"n_32"])*(log(4 + 3*p1) + log(2*(1 + r_cr4 - r_cr4^2) - 3*p2*(1 - 5*r_cr4 + 5*r_cr4^2))) + (x[,"n_12"] + x[,"n_22"])*log(8*(8 - r_cr4 + r_cr4^2) + 12*p2*(4 - 5*r_cr4 + 5*r_cr4^2) + 3*p1*(2*(-4 - r_cr4 + r_cr4^2) + 3*p2*(-2 - 5*r_cr4 + 5*r_cr4^2))) + (x[,"n_11"] + x[,"n_23"])*log(16*(1 + 3*r_cr4 - 5*r_cr4^2 + 3*r_cr4^3) - 12*p2*(-1 + 6*r_cr4 - 7*r_cr4^2 + 6*r_cr4^3) + p1*(p2*(9 + 63*r_cr4^2 - 54*r_cr4^3) + 12*(1 - 5*r_cr4^2 + 3*r_cr4^3))) + (x[,"n_13"] + x[,"n_21"])*log(3*p1*(4*(-1 + r_cr4 + 4*r_cr4^2 - 3*r_cr4^3) + 3*p2*(2 + 4*r_cr4 - 11*r_cr4^2 + 6*r_cr4^3)) + 4*(8 - 8*r_cr4 + 16*r_cr4^2 - 12*r_cr4^3 + 3*p2*(-4 + 10*r_cr4 - 11*r_cr4^2 + 6*r_cr4^3)))


logL_mc1 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(5*log(2) + 2*log(3)) + (n04 + n30)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n00 + n34)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n14 + n20)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(6 - (8 - 12*p1)*r - (-6 + 9*p1)*r^2)) + (n01 + n33)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(4*(1 + r^2) - p2*(6 - 9*r + 6*r^2))) + (n03 + n31)*(log(2 - 3*p1) + log(pmax(1e-6,r)) + log(4*(2 - 2*r + r^2) - p2*(3 - 3*r + 6*r^2))) + (n02 + n32)*(log(2 - 3*p1) + log(8 - 6*r + 6*r^2 + p2*(6 - 9*r + 9*r^2))) + (n10 + n24)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(4 - 4*r + 6*r^2 - p1*(-3 - 6*r + 9*r^2))) + (n12 + n22)*log(2*(28 + 6*r - 6*r^2 + p2*(12 + 9*r - 9*r^2)) + 3*p1*(2 - 6*r + 6*r^2 + p2*(-3 - 9*r + 9*r^2))) + (n11 + n23)*log(-2*(-4*(4 - 4*r^2 + 3*r^3) + 3*p2*(2 + 3*r - 11*r^2 + 6*r^3)) + 3*p1*(4*(2 - 3*r + 4*r^2 - 3*r^3) + 3*p2*(-1 + 9*r - 11*r^2 + 6*r^3))) + (n13 + n21)*log(24 - 2*(4 + 3*p2)*r + (40 - 42*p2)*r^2 + 12*(-2 + 3*p2)*r^3 + p1*(12*r*(4 - 5*r + 3*r^2) - 9*p2*(-3 + 5*r - 7*r^2 + 6*r^3)))
return(L)}
interlogL_mc1 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_mc1,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_mc1 <- parallel::mcmapply(interlogL_mc1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_mc1 <- -((x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-log10(2) + log10(7/2 + (15*p1)/4) + log10(2 - 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-log10(2) + log10(6 + (6 - 9*p1)/4 + (-8 + 12*p1)/2) + log10(2 - 3*p2)) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(2 - 3*p1) + log10(5 - 3*p2)) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(2 - 3*p1) + log10(5 - 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p1) + log10(13/2 + (15*p2)/4)) - (x[,"n_11"] + x[,"n_23"])*log10(-2*(-27/2 + (9*p2)/2) + 3*p1*(9/2 + (9*p2)/2)) - (x[,"n_13"] + x[,"n_21"])*log10(20 + (40 - 42*p2)/4 - 3*p2 + (3*(-2 + 3*p2))/2 + p1*(27/2 + (27*p2)/2)) - (x[,"n_12"] + x[,"n_22"])*log10(3*p1*(1/2 - (21*p2)/4) + 2*(59/2 + (57*p2)/4)) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_mc1)) + 2*log10(pmax(1e-6,r_mc1))) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_mc1)) + log10(1 - 2*r_mc1 + r_mc1^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_mc1)) + log10(6 - (8 - 12*p1)*r_mc1 - (-6 + 9*p1)*r_mc1^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_mc1)) + log10(4*(1 + r_mc1^2) - p2*(6 - 9*r_mc1 + 6*r_mc1^2))) + (x[,"n_03"] + x[,"n_31"])*(log10(2 - 3*p1) + log10(pmax(1e-6,r_mc1)) + log10(4*(2 - 2*r_mc1 + r_mc1^2) - p2*(3 - 3*r_mc1 + 6*r_mc1^2))) + (x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p1) + log10(8 - 6*r_mc1 + 6*r_mc1^2 + p2*(6 - 9*r_mc1 + 9*r_mc1^2))) + (x[,"n_10"] + x[,"n_24"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_mc1)) + log10(4 - 4*r_mc1 + 6*r_mc1^2 - p1*(-3 - 6*r_mc1 + 9*r_mc1^2))) + (x[,"n_12"] + x[,"n_22"])*log10(2*(28 + 6*r_mc1 - 6*r_mc1^2 + p2*(12 + 9*r_mc1 - 9*r_mc1^2)) + 3*p1*(2 - 6*r_mc1 + 6*r_mc1^2 + p2*(-3 - 9*r_mc1 + 9*r_mc1^2))) + (x[,"n_11"] + x[,"n_23"])*log10(-2*(-4*(4 - 4*r_mc1^2 + 3*r_mc1^3) + 3*p2*(2 + 3*r_mc1 - 11*r_mc1^2 + 6*r_mc1^3)) + 3*p1*(4*(2 - 3*r_mc1 + 4*r_mc1^2 - 3*r_mc1^3) + 3*p2*(-1 + 9*r_mc1 - 11*r_mc1^2 + 6*r_mc1^3))) + (x[,"n_13"] + x[,"n_21"])*log10(24 - 2*(4 + 3*p2)*r_mc1 + (40 - 42*p2)*r_mc1^2 + 12*(-2 + 3*p2)*r_mc1^3 + p1*(12*r_mc1*(4 - 5*r_mc1 + 3*r_mc1^2) - 9*p2*(-3 + 5*r_mc1 - 7*r_mc1^2 + 6*r_mc1^3)))


logL_mc1 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(5*log(2) + 2*log(3)) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_mc1)) + 2*log(pmax(1e-6,r_mc1))) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_mc1)) + log(1 - 2*r_mc1 + r_mc1^2)) + (x[,"n_14"] + x[,"n_20"])*(log(2 - 3*p2) + log(pmax(1e-6,r_mc1)) + log(6 - (8 - 12*p1)*r_mc1 - (-6 + 9*p1)*r_mc1^2)) + (x[,"n_01"] + x[,"n_33"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_mc1)) + log(4*(1 + r_mc1^2) - p2*(6 - 9*r_mc1 + 6*r_mc1^2))) + (x[,"n_03"] + x[,"n_31"])*(log(2 - 3*p1) + log(pmax(1e-6,r_mc1)) + log(4*(2 - 2*r_mc1 + r_mc1^2) - p2*(3 - 3*r_mc1 + 6*r_mc1^2))) + (x[,"n_02"] + x[,"n_32"])*(log(2 - 3*p1) + log(8 - 6*r_mc1 + 6*r_mc1^2 + p2*(6 - 9*r_mc1 + 9*r_mc1^2))) + (x[,"n_10"] + x[,"n_24"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_mc1)) + log(4 - 4*r_mc1 + 6*r_mc1^2 - p1*(-3 - 6*r_mc1 + 9*r_mc1^2))) + (x[,"n_12"] + x[,"n_22"])*log(2*(28 + 6*r_mc1 - 6*r_mc1^2 + p2*(12 + 9*r_mc1 - 9*r_mc1^2)) + 3*p1*(2 - 6*r_mc1 + 6*r_mc1^2 + p2*(-3 - 9*r_mc1 + 9*r_mc1^2))) + (x[,"n_11"] + x[,"n_23"])*log(-2*(-4*(4 - 4*r_mc1^2 + 3*r_mc1^3) + 3*p2*(2 + 3*r_mc1 - 11*r_mc1^2 + 6*r_mc1^3)) + 3*p1*(4*(2 - 3*r_mc1 + 4*r_mc1^2 - 3*r_mc1^3) + 3*p2*(-1 + 9*r_mc1 - 11*r_mc1^2 + 6*r_mc1^3))) + (x[,"n_13"] + x[,"n_21"])*log(24 - 2*(4 + 3*p2)*r_mc1 + (40 - 42*p2)*r_mc1^2 + 12*(-2 + 3*p2)*r_mc1^3 + p1*(12*r_mc1*(4 - 5*r_mc1 + 3*r_mc1^2) - 9*p2*(-3 + 5*r_mc1 - 7*r_mc1^2 + 6*r_mc1^3)))


logL_mc2 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(6*log(2) + 2*log(3)) + (n04 + n30)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n00 + n34)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n14 + n20)*(log(4 + 3*p2) + log(pmax(1e-6,r)) + log(6 - (8 - 12*p1)*r - (-6 + 9*p1)*r^2)) + (n01 + n33)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(8*(1 + r^2) + p2*(6 - 9*r + 6*r^2))) + (n03 + n31)*(log(2 - 3*p1) + log(pmax(1e-6,r)) + log(8*(2 - 2*r + r^2) + p2*(3 - 3*r + 6*r^2))) + (n02 + n32)*(log(2 - 3*p1) + log(4*(4 - 3*r + 3*r^2) - p2*(6 - 9*r + 9*r^2))) + (n10 + n24)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(4 - 4*r + 6*r^2 - p1*(-3 - 6*r + 9*r^2))) + (n12 + n22)*log(2*(56 + 12*r - 12*r^2 + 3*p2*(-4 - 3*r + 3*r^2)) + 3*p1*(p2*(3 + 9*r - 9*r^2) + 4*(1 - 3*r + 3*r^2))) + (n11 + n23)*log(2*(8*(4 - 4*r^2 + 3*r^3) + 3*p2*(2 + 3*r - 11*r^2 + 6*r^3)) + p1*(24*(2 - 3*r + 4*r^2 - 3*r^3) - 9*p2*(-1 + 9*r - 11*r^2 + 6*r^3))) + (n13 + n21)*log(48 + 2*(-8 + 3*p2)*r + (80 + 42*p2)*r^2 - 12*(4 + 3*p2)*r^3 + 3*p1*(8*r*(4 - 5*r + 3*r^2) + 3*p2*(-3 + 5*r - 7*r^2 + 6*r^3)))
return(L)}
interlogL_mc2 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_mc2,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_mc2 <- parallel::mcmapply(interlogL_mc2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_mc2 <- -((x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p1) + log10(13 - (15*p2)/4))) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-log10(2) + log10(7/2 + (15*p1)/4) + log10(4 + 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-log10(2) + log10(6 + (6 - 9*p1)/4 + (-8 + 12*p1)/2) + log10(4 + 3*p2)) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(2 - 3*p1) + log10(10 + 3*p2)) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(2 - 3*p1) + log10(10 + 3*p2)) - (x[,"n_11"] + x[,"n_23"])*log10(p1*(27 - (27*p2)/2) + 2*(27 + (9*p2)/2)) - (x[,"n_12"] + x[,"n_22"])*log10(2*(59 - (57*p2)/4) + 3*p1*(1 + (21*p2)/4)) - (x[,"n_13"] + x[,"n_21"])*log10(40 + 3*p1*(9 - (9*p2)/2) + 3*p2 - (3*(4 + 3*p2))/2 + (80 + 42*p2)/4) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_mc2)) + 2*log10(pmax(1e-6,r_mc2))) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_mc2)) + log10(1 - 2*r_mc2 + r_mc2^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(4 + 3*p2) + log10(pmax(1e-6,r_mc2)) + log10(6 - (8 - 12*p1)*r_mc2 - (-6 + 9*p1)*r_mc2^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_mc2)) + log10(8*(1 + r_mc2^2) + p2*(6 - 9*r_mc2 + 6*r_mc2^2))) + (x[,"n_03"] + x[,"n_31"])*(log10(2 - 3*p1) + log10(pmax(1e-6,r_mc2)) + log10(8*(2 - 2*r_mc2 + r_mc2^2) + p2*(3 - 3*r_mc2 + 6*r_mc2^2))) + (x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p1) + log10(4*(4 - 3*r_mc2 + 3*r_mc2^2) - p2*(6 - 9*r_mc2 + 9*r_mc2^2))) + (x[,"n_10"] + x[,"n_24"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_mc2)) + log10(4 - 4*r_mc2 + 6*r_mc2^2 - p1*(-3 - 6*r_mc2 + 9*r_mc2^2))) + (x[,"n_12"] + x[,"n_22"])*log10(2*(56 + 12*r_mc2 - 12*r_mc2^2 + 3*p2*(-4 - 3*r_mc2 + 3*r_mc2^2)) + 3*p1*(p2*(3 + 9*r_mc2 - 9*r_mc2^2) + 4*(1 - 3*r_mc2 + 3*r_mc2^2))) + (x[,"n_11"] + x[,"n_23"])*log10(2*(8*(4 - 4*r_mc2^2 + 3*r_mc2^3) + 3*p2*(2 + 3*r_mc2 - 11*r_mc2^2 + 6*r_mc2^3)) + p1*(24*(2 - 3*r_mc2 + 4*r_mc2^2 - 3*r_mc2^3) - 9*p2*(-1 + 9*r_mc2 - 11*r_mc2^2 + 6*r_mc2^3))) + (x[,"n_13"] + x[,"n_21"])*log10(48 + 2*(-8 + 3*p2)*r_mc2 + (80 + 42*p2)*r_mc2^2 - 12*(4 + 3*p2)*r_mc2^3 + 3*p1*(8*r_mc2*(4 - 5*r_mc2 + 3*r_mc2^2) + 3*p2*(-3 + 5*r_mc2 - 7*r_mc2^2 + 6*r_mc2^3)))


logL_mc2 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(6*log(2) + 2*log(3)) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_mc2)) + 2*log(pmax(1e-6,r_mc2))) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_mc2)) + log(1 - 2*r_mc2 + r_mc2^2)) + (x[,"n_14"] + x[,"n_20"])*(log(4 + 3*p2) + log(pmax(1e-6,r_mc2)) + log(6 - (8 - 12*p1)*r_mc2 - (-6 + 9*p1)*r_mc2^2)) + (x[,"n_01"] + x[,"n_33"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_mc2)) + log(8*(1 + r_mc2^2) + p2*(6 - 9*r_mc2 + 6*r_mc2^2))) + (x[,"n_03"] + x[,"n_31"])*(log(2 - 3*p1) + log(pmax(1e-6,r_mc2)) + log(8*(2 - 2*r_mc2 + r_mc2^2) + p2*(3 - 3*r_mc2 + 6*r_mc2^2))) + (x[,"n_02"] + x[,"n_32"])*(log(2 - 3*p1) + log(4*(4 - 3*r_mc2 + 3*r_mc2^2) - p2*(6 - 9*r_mc2 + 9*r_mc2^2))) + (x[,"n_10"] + x[,"n_24"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_mc2)) + log(4 - 4*r_mc2 + 6*r_mc2^2 - p1*(-3 - 6*r_mc2 + 9*r_mc2^2))) + (x[,"n_12"] + x[,"n_22"])*log(2*(56 + 12*r_mc2 - 12*r_mc2^2 + 3*p2*(-4 - 3*r_mc2 + 3*r_mc2^2)) + 3*p1*(p2*(3 + 9*r_mc2 - 9*r_mc2^2) + 4*(1 - 3*r_mc2 + 3*r_mc2^2))) + (x[,"n_11"] + x[,"n_23"])*log(2*(8*(4 - 4*r_mc2^2 + 3*r_mc2^3) + 3*p2*(2 + 3*r_mc2 - 11*r_mc2^2 + 6*r_mc2^3)) + p1*(24*(2 - 3*r_mc2 + 4*r_mc2^2 - 3*r_mc2^3) - 9*p2*(-1 + 9*r_mc2 - 11*r_mc2^2 + 6*r_mc2^3))) + (x[,"n_13"] + x[,"n_21"])*log(48 + 2*(-8 + 3*p2)*r_mc2 + (80 + 42*p2)*r_mc2^2 - 12*(4 + 3*p2)*r_mc2^3 + 3*p1*(8*r_mc2*(4 - 5*r_mc2 + 3*r_mc2^2) + 3*p2*(-3 + 5*r_mc2 - 7*r_mc2^2 + 6*r_mc2^3)))


logL_mc3 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(5*log(2) + 2*log(3)) + (n04 + n30)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n00 + n34)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n14 + n20)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(3 - 4*r + 3*r^2)) + (n10 + n24)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(2 - 2*r + 3*r^2)) + (n03 + n31)*(log(pmax(1e-6,r)) + log((-2 + 3*p1)*(4 + 3*p2)*(-1 + r) + (2 - 3*p2)*(p1*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2)))) + (n12 + n22)*log(2*(28 + 6*r - 6*r^2 + p2*(12 + 9*r - 9*r^2)) + 3*p1*(8 - 6*r + 6*r^2 + 3*p2*(5 - 3*r + 3*r^2))) + (n01 + n33)*(log(pmax(1e-6,1 - r)) + log(2*(p2*(-6 + 9*r - 6*r^2) + 4*(1 + r^2)) + 3*p1*(2 - 4*r^2 + p2*(-3 - 9*r + 6*r^2)))) + (n02 + n32)*log(-3*p1*(-4 - 6*r + 6*r^2 + p2*(-3 - 9*r + 9*r^2)) + 2*(8 - 6*r + 6*r^2 + p2*(6 - 9*r + 9*r^2))) + (n11 + n23)*log(-2*(-4*(4 - 4*r^2 + 3*r^3) + 3*p2*(2 + 3*r - 11*r^2 + 6*r^3)) + 3*p1*(3*p2*(-4 + 6*r - 11*r^2 + 6*r^3) - 2*(2 + 3*r - 8*r^2 + 6*r^3))) + (n13 + n21)*log(24 - 2*(4 + 3*p2)*r + (40 - 42*p2)*r^2 + 12*(-2 + 3*p2)*r^3 - 3*p1*(6 - 10*r + 20*r^2 - 12*r^3 + 3*p2*(3 + 2*r - 7*r^2 + 6*r^3)))
return(L)}
interlogL_mc3 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_mc3,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_mc3 <- parallel::mcmapply(interlogL_mc3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_mc3 <- -((x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(3*p1*(1 - 6*p2) + 2*(5 - 3*p2)))) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(7) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(7) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10((3 + (9*p1)/2)*(2 - 3*p2) - ((-2 + 3*p1)*(4 + 3*p2))/2)) - (x[,"n_02"] + x[,"n_32"])*log10(-3*p1*(-11/2 - (21*p2)/4) + 2*(13/2 + (15*p2)/4)) - (x[,"n_11"] + x[,"n_23"])*log10(3*p1*(-9/2 - 9*p2) - 2*(-27/2 + (9*p2)/2)) - (x[,"n_13"] + x[,"n_21"])*log10(20 + (40 - 42*p2)/4 - 3*p2 + (3*(-2 + 3*p2))/2 - 3*p1*(9/2 + 9*p2)) - (x[,"n_12"] + x[,"n_22"])*log10(3*p1*(13/2 + (51*p2)/4) + 2*(59/2 + (57*p2)/4)) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_mc3)) + 2*log10(pmax(1e-6,r_mc3))) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_mc3)) + log10(1 - 2*r_mc3 + r_mc3^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_mc3)) + log10(3 - 4*r_mc3 + 3*r_mc3^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_mc3)) + log10(2 - 2*r_mc3 + 3*r_mc3^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(pmax(1e-6,r_mc3)) + log10((-2 + 3*p1)*(4 + 3*p2)*(-1 + r_mc3) + (2 - 3*p2)*(p1*(3 + 6*r_mc3 - 6*r_mc3^2) + 4*(1 - r_mc3 + r_mc3^2)))) + (x[,"n_12"] + x[,"n_22"])*log10(2*(28 + 6*r_mc3 - 6*r_mc3^2 + p2*(12 + 9*r_mc3 - 9*r_mc3^2)) + 3*p1*(8 - 6*r_mc3 + 6*r_mc3^2 + 3*p2*(5 - 3*r_mc3 + 3*r_mc3^2))) + (x[,"n_01"] + x[,"n_33"])*(log10(pmax(1e-6,1 - r_mc3)) + log10(2*(p2*(-6 + 9*r_mc3 - 6*r_mc3^2) + 4*(1 + r_mc3^2)) + 3*p1*(2 - 4*r_mc3^2 + p2*(-3 - 9*r_mc3 + 6*r_mc3^2)))) + (x[,"n_02"] + x[,"n_32"])*log10(-3*p1*(-4 - 6*r_mc3 + 6*r_mc3^2 + p2*(-3 - 9*r_mc3 + 9*r_mc3^2)) + 2*(8 - 6*r_mc3 + 6*r_mc3^2 + p2*(6 - 9*r_mc3 + 9*r_mc3^2))) + (x[,"n_11"] + x[,"n_23"])*log10(-2*(-4*(4 - 4*r_mc3^2 + 3*r_mc3^3) + 3*p2*(2 + 3*r_mc3 - 11*r_mc3^2 + 6*r_mc3^3)) + 3*p1*(3*p2*(-4 + 6*r_mc3 - 11*r_mc3^2 + 6*r_mc3^3) - 2*(2 + 3*r_mc3 - 8*r_mc3^2 + 6*r_mc3^3))) + (x[,"n_13"] + x[,"n_21"])*log10(24 - 2*(4 + 3*p2)*r_mc3 + (40 - 42*p2)*r_mc3^2 + 12*(-2 + 3*p2)*r_mc3^3 - 3*p1*(6 - 10*r_mc3 + 20*r_mc3^2 - 12*r_mc3^3 + 3*p2*(3 + 2*r_mc3 - 7*r_mc3^2 + 6*r_mc3^3)))


logL_mc3 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(5*log(2) + 2*log(3)) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_mc3)) + 2*log(pmax(1e-6,r_mc3))) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_mc3)) + log(1 - 2*r_mc3 + r_mc3^2)) + (x[,"n_14"] + x[,"n_20"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_mc3)) + log(3 - 4*r_mc3 + 3*r_mc3^2)) + (x[,"n_10"] + x[,"n_24"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_mc3)) + log(2 - 2*r_mc3 + 3*r_mc3^2)) + (x[,"n_03"] + x[,"n_31"])*(log(pmax(1e-6,r_mc3)) + log((-2 + 3*p1)*(4 + 3*p2)*(-1 + r_mc3) + (2 - 3*p2)*(p1*(3 + 6*r_mc3 - 6*r_mc3^2) + 4*(1 - r_mc3 + r_mc3^2)))) + (x[,"n_12"] + x[,"n_22"])*log(2*(28 + 6*r_mc3 - 6*r_mc3^2 + p2*(12 + 9*r_mc3 - 9*r_mc3^2)) + 3*p1*(8 - 6*r_mc3 + 6*r_mc3^2 + 3*p2*(5 - 3*r_mc3 + 3*r_mc3^2))) + (x[,"n_01"] + x[,"n_33"])*(log(pmax(1e-6,1 - r_mc3)) + log(2*(p2*(-6 + 9*r_mc3 - 6*r_mc3^2) + 4*(1 + r_mc3^2)) + 3*p1*(2 - 4*r_mc3^2 + p2*(-3 - 9*r_mc3 + 6*r_mc3^2)))) + (x[,"n_02"] + x[,"n_32"])*log(-3*p1*(-4 - 6*r_mc3 + 6*r_mc3^2 + p2*(-3 - 9*r_mc3 + 9*r_mc3^2)) + 2*(8 - 6*r_mc3 + 6*r_mc3^2 + p2*(6 - 9*r_mc3 + 9*r_mc3^2))) + (x[,"n_11"] + x[,"n_23"])*log(-2*(-4*(4 - 4*r_mc3^2 + 3*r_mc3^3) + 3*p2*(2 + 3*r_mc3 - 11*r_mc3^2 + 6*r_mc3^3)) + 3*p1*(3*p2*(-4 + 6*r_mc3 - 11*r_mc3^2 + 6*r_mc3^3) - 2*(2 + 3*r_mc3 - 8*r_mc3^2 + 6*r_mc3^3))) + (x[,"n_13"] + x[,"n_21"])*log(24 - 2*(4 + 3*p2)*r_mc3 + (40 - 42*p2)*r_mc3^2 + 12*(-2 + 3*p2)*r_mc3^3 - 3*p1*(6 - 10*r_mc3 + 20*r_mc3^2 - 12*r_mc3^3 + 3*p2*(3 + 2*r_mc3 - 7*r_mc3^2 + 6*r_mc3^3)))


logL_mc4 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(6*log(2) + 2*log(3)) + (n04 + n30)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n00 + n34)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n14 + n20)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(3 - 4*r + 3*r^2)) + (n10 + n24)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(2 - 2*r + 3*r^2)) + (n03 + n31)*(log(pmax(1e-6,r)) + log((2 - 3*p1)*(-8 + 3*p2)*(-1 + r) + (4 + 3*p2)*(p1*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2)))) + (n12 + n22)*log(2*(56 + 12*r - 12*r^2 + 3*p2*(-4 - 3*r + 3*r^2)) - 3*p1*(-4*(4 - 3*r + 3*r^2) + 3*p2*(5 - 3*r + 3*r^2))) + (n01 + n33)*(log(pmax(1e-6,1 - r)) + log(-3*p1*(-4 + 8*r^2 + p2*(-3 - 9*r + 6*r^2)) + 2*(8*(1 + r^2) + p2*(6 - 9*r + 6*r^2)))) + (n02 + n32)*log(2*(p2*(-6 + 9*r - 9*r^2) + 4*(4 - 3*r + 3*r^2)) + 3*p1*(8 + 12*r - 12*r^2 + p2*(-3 - 9*r + 9*r^2))) + (n11 + n23)*log(2*(8*(4 - 4*r^2 + 3*r^3) + 3*p2*(2 + 3*r - 11*r^2 + 6*r^3)) - 3*p1*(3*p2*(-4 + 6*r - 11*r^2 + 6*r^3) + 4*(2 + 3*r - 8*r^2 + 6*r^3))) + (n13 + n21)*log(48 + 2*(-8 + 3*p2)*r + (80 + 42*p2)*r^2 - 12*(4 + 3*p2)*r^3 + 3*p1*(4*(-3 + 5*r - 10*r^2 + 6*r^3) + 3*p2*(3 + 2*r - 7*r^2 + 6*r^3)))
return(L)}
interlogL_mc4 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_mc4,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_mc4 <- parallel::mcmapply(interlogL_mc4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_mc4 <- -((x[,"n_02"] + x[,"n_32"])*log10(3*p1*(11 - (21*p2)/4) + 2*(13 - (15*p2)/4))) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(7) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(7) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(-((2 - 3*p1)*(-8 + 3*p2))/2 + (3 + (9*p1)/2)*(4 + 3*p2))) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(-3*p1*(-2 - 6*p2) + 2*(10 + 3*p2))) - (x[,"n_11"] + x[,"n_23"])*log10(-3*p1*(9 - 9*p2) + 2*(27 + (9*p2)/2)) - (x[,"n_12"] + x[,"n_22"])*log10(2*(59 - (57*p2)/4) - 3*p1*(-13 + (51*p2)/4)) - (x[,"n_13"] + x[,"n_21"])*log10(40 + 3*p2 - (3*(4 + 3*p2))/2 + 3*p1*(-9 + 9*p2) + (80 + 42*p2)/4) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_mc4)) + 2*log10(pmax(1e-6,r_mc4))) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_mc4)) + log10(1 - 2*r_mc4 + r_mc4^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_mc4)) + log10(3 - 4*r_mc4 + 3*r_mc4^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_mc4)) + log10(2 - 2*r_mc4 + 3*r_mc4^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(pmax(1e-6,r_mc4)) + log10((2 - 3*p1)*(-8 + 3*p2)*(-1 + r_mc4) + (4 + 3*p2)*(p1*(3 + 6*r_mc4 - 6*r_mc4^2) + 4*(1 - r_mc4 + r_mc4^2)))) + (x[,"n_12"] + x[,"n_22"])*log10(2*(56 + 12*r_mc4 - 12*r_mc4^2 + 3*p2*(-4 - 3*r_mc4 + 3*r_mc4^2)) - 3*p1*(-4*(4 - 3*r_mc4 + 3*r_mc4^2) + 3*p2*(5 - 3*r_mc4 + 3*r_mc4^2))) + (x[,"n_01"] + x[,"n_33"])*(log10(pmax(1e-6,1 - r_mc4)) + log10(-3*p1*(-4 + 8*r_mc4^2 + p2*(-3 - 9*r_mc4 + 6*r_mc4^2)) + 2*(8*(1 + r_mc4^2) + p2*(6 - 9*r_mc4 + 6*r_mc4^2)))) + (x[,"n_02"] + x[,"n_32"])*log10(2*(p2*(-6 + 9*r_mc4 - 9*r_mc4^2) + 4*(4 - 3*r_mc4 + 3*r_mc4^2)) + 3*p1*(8 + 12*r_mc4 - 12*r_mc4^2 + p2*(-3 - 9*r_mc4 + 9*r_mc4^2))) + (x[,"n_11"] + x[,"n_23"])*log10(2*(8*(4 - 4*r_mc4^2 + 3*r_mc4^3) + 3*p2*(2 + 3*r_mc4 - 11*r_mc4^2 + 6*r_mc4^3)) - 3*p1*(3*p2*(-4 + 6*r_mc4 - 11*r_mc4^2 + 6*r_mc4^3) + 4*(2 + 3*r_mc4 - 8*r_mc4^2 + 6*r_mc4^3))) + (x[,"n_13"] + x[,"n_21"])*log10(48 + 2*(-8 + 3*p2)*r_mc4 + (80 + 42*p2)*r_mc4^2 - 12*(4 + 3*p2)*r_mc4^3 + 3*p1*(4*(-3 + 5*r_mc4 - 10*r_mc4^2 + 6*r_mc4^3) + 3*p2*(3 + 2*r_mc4 - 7*r_mc4^2 + 6*r_mc4^3)))


logL_mc4 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(6*log(2) + 2*log(3)) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_mc4)) + 2*log(pmax(1e-6,r_mc4))) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_mc4)) + log(1 - 2*r_mc4 + r_mc4^2)) + (x[,"n_14"] + x[,"n_20"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_mc4)) + log(3 - 4*r_mc4 + 3*r_mc4^2)) + (x[,"n_10"] + x[,"n_24"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_mc4)) + log(2 - 2*r_mc4 + 3*r_mc4^2)) + (x[,"n_03"] + x[,"n_31"])*(log(pmax(1e-6,r_mc4)) + log((2 - 3*p1)*(-8 + 3*p2)*(-1 + r_mc4) + (4 + 3*p2)*(p1*(3 + 6*r_mc4 - 6*r_mc4^2) + 4*(1 - r_mc4 + r_mc4^2)))) + (x[,"n_12"] + x[,"n_22"])*log(2*(56 + 12*r_mc4 - 12*r_mc4^2 + 3*p2*(-4 - 3*r_mc4 + 3*r_mc4^2)) - 3*p1*(-4*(4 - 3*r_mc4 + 3*r_mc4^2) + 3*p2*(5 - 3*r_mc4 + 3*r_mc4^2))) + (x[,"n_01"] + x[,"n_33"])*(log(pmax(1e-6,1 - r_mc4)) + log(-3*p1*(-4 + 8*r_mc4^2 + p2*(-3 - 9*r_mc4 + 6*r_mc4^2)) + 2*(8*(1 + r_mc4^2) + p2*(6 - 9*r_mc4 + 6*r_mc4^2)))) + (x[,"n_02"] + x[,"n_32"])*log(2*(p2*(-6 + 9*r_mc4 - 9*r_mc4^2) + 4*(4 - 3*r_mc4 + 3*r_mc4^2)) + 3*p1*(8 + 12*r_mc4 - 12*r_mc4^2 + p2*(-3 - 9*r_mc4 + 9*r_mc4^2))) + (x[,"n_11"] + x[,"n_23"])*log(2*(8*(4 - 4*r_mc4^2 + 3*r_mc4^3) + 3*p2*(2 + 3*r_mc4 - 11*r_mc4^2 + 6*r_mc4^3)) - 3*p1*(3*p2*(-4 + 6*r_mc4 - 11*r_mc4^2 + 6*r_mc4^3) + 4*(2 + 3*r_mc4 - 8*r_mc4^2 + 6*r_mc4^3))) + (x[,"n_13"] + x[,"n_21"])*log(48 + 2*(-8 + 3*p2)*r_mc4 + (80 + 42*p2)*r_mc4^2 - 12*(4 + 3*p2)*r_mc4^3 + 3*p1*(4*(-3 + 5*r_mc4 - 10*r_mc4^2 + 6*r_mc4^3) + 3*p2*(3 + 2*r_mc4 - 7*r_mc4^2 + 6*r_mc4^3)))


logL_mr1 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(6*log(2) + 2*log(3)) + (n00 + n34)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n04 + n30)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n10 + n24)*(log(4 + 3*p2) + log(pmax(1e-6,r)) + log(6 - (8 - 12*p1)*r - (-6 + 9*p1)*r^2)) + (n03 + n31)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(8*(1 + r^2) + p2*(6 - 9*r + 6*r^2))) + (n01 + n33)*(log(2 - 3*p1) + log(pmax(1e-6,r)) + log(8*(2 - 2*r + r^2) + p2*(3 - 3*r + 6*r^2))) + (n02 + n32)*(log(2 - 3*p1) + log(4*(4 - 3*r + 3*r^2) - p2*(6 - 9*r + 9*r^2))) + (n14 + n20)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(4 - 4*r + 6*r^2 - p1*(-3 - 6*r + 9*r^2))) + (n12 + n22)*log(2*(56 + 12*r - 12*r^2 + 3*p2*(-4 - 3*r + 3*r^2)) + 3*p1*(p2*(3 + 9*r - 9*r^2) + 4*(1 - 3*r + 3*r^2))) + (n13 + n21)*log(2*(8*(4 - 4*r^2 + 3*r^3) + 3*p2*(2 + 3*r - 11*r^2 + 6*r^3)) + p1*(24*(2 - 3*r + 4*r^2 - 3*r^3) - 9*p2*(-1 + 9*r - 11*r^2 + 6*r^3))) + (n11 + n23)*log(48 + 2*(-8 + 3*p2)*r + (80 + 42*p2)*r^2 - 12*(4 + 3*p2)*r^3 + 3*p1*(8*r*(4 - 5*r + 3*r^2) + 3*p2*(-3 + 5*r - 7*r^2 + 6*r^3)))
return(L)}
interlogL_mr1 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_mr1,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_mr1 <- parallel::mcmapply(interlogL_mr1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_mr1 <- -((x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p1) + log10(13 - (15*p2)/4))) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-log10(2) + log10(7/2 + (15*p1)/4) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-log10(2) + log10(6 + (6 - 9*p1)/4 + (-8 + 12*p1)/2) + log10(4 + 3*p2)) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(2 - 3*p1) + log10(10 + 3*p2)) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(2 - 3*p1) + log10(10 + 3*p2)) - (x[,"n_13"] + x[,"n_21"])*log10(p1*(27 - (27*p2)/2) + 2*(27 + (9*p2)/2)) - (x[,"n_12"] + x[,"n_22"])*log10(2*(59 - (57*p2)/4) + 3*p1*(1 + (21*p2)/4)) - (x[,"n_11"] + x[,"n_23"])*log10(40 + 3*p1*(9 - (9*p2)/2) + 3*p2 - (3*(4 + 3*p2))/2 + (80 + 42*p2)/4) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_mr1)) + 2*log10(pmax(1e-6,r_mr1))) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_mr1)) + log10(1 - 2*r_mr1 + r_mr1^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(4 + 3*p2) + log10(pmax(1e-6,r_mr1)) + log10(6 - (8 - 12*p1)*r_mr1 - (-6 + 9*p1)*r_mr1^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_mr1)) + log10(8*(1 + r_mr1^2) + p2*(6 - 9*r_mr1 + 6*r_mr1^2))) + (x[,"n_01"] + x[,"n_33"])*(log10(2 - 3*p1) + log10(pmax(1e-6,r_mr1)) + log10(8*(2 - 2*r_mr1 + r_mr1^2) + p2*(3 - 3*r_mr1 + 6*r_mr1^2))) + (x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p1) + log10(4*(4 - 3*r_mr1 + 3*r_mr1^2) - p2*(6 - 9*r_mr1 + 9*r_mr1^2))) + (x[,"n_14"] + x[,"n_20"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_mr1)) + log10(4 - 4*r_mr1 + 6*r_mr1^2 - p1*(-3 - 6*r_mr1 + 9*r_mr1^2))) + (x[,"n_12"] + x[,"n_22"])*log10(2*(56 + 12*r_mr1 - 12*r_mr1^2 + 3*p2*(-4 - 3*r_mr1 + 3*r_mr1^2)) + 3*p1*(p2*(3 + 9*r_mr1 - 9*r_mr1^2) + 4*(1 - 3*r_mr1 + 3*r_mr1^2))) + (x[,"n_13"] + x[,"n_21"])*log10(2*(8*(4 - 4*r_mr1^2 + 3*r_mr1^3) + 3*p2*(2 + 3*r_mr1 - 11*r_mr1^2 + 6*r_mr1^3)) + p1*(24*(2 - 3*r_mr1 + 4*r_mr1^2 - 3*r_mr1^3) - 9*p2*(-1 + 9*r_mr1 - 11*r_mr1^2 + 6*r_mr1^3))) + (x[,"n_11"] + x[,"n_23"])*log10(48 + 2*(-8 + 3*p2)*r_mr1 + (80 + 42*p2)*r_mr1^2 - 12*(4 + 3*p2)*r_mr1^3 + 3*p1*(8*r_mr1*(4 - 5*r_mr1 + 3*r_mr1^2) + 3*p2*(-3 + 5*r_mr1 - 7*r_mr1^2 + 6*r_mr1^3)))


logL_mr1 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(6*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_mr1)) + 2*log(pmax(1e-6,r_mr1))) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_mr1)) + log(1 - 2*r_mr1 + r_mr1^2)) + (x[,"n_10"] + x[,"n_24"])*(log(4 + 3*p2) + log(pmax(1e-6,r_mr1)) + log(6 - (8 - 12*p1)*r_mr1 - (-6 + 9*p1)*r_mr1^2)) + (x[,"n_03"] + x[,"n_31"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_mr1)) + log(8*(1 + r_mr1^2) + p2*(6 - 9*r_mr1 + 6*r_mr1^2))) + (x[,"n_01"] + x[,"n_33"])*(log(2 - 3*p1) + log(pmax(1e-6,r_mr1)) + log(8*(2 - 2*r_mr1 + r_mr1^2) + p2*(3 - 3*r_mr1 + 6*r_mr1^2))) + (x[,"n_02"] + x[,"n_32"])*(log(2 - 3*p1) + log(4*(4 - 3*r_mr1 + 3*r_mr1^2) - p2*(6 - 9*r_mr1 + 9*r_mr1^2))) + (x[,"n_14"] + x[,"n_20"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_mr1)) + log(4 - 4*r_mr1 + 6*r_mr1^2 - p1*(-3 - 6*r_mr1 + 9*r_mr1^2))) + (x[,"n_12"] + x[,"n_22"])*log(2*(56 + 12*r_mr1 - 12*r_mr1^2 + 3*p2*(-4 - 3*r_mr1 + 3*r_mr1^2)) + 3*p1*(p2*(3 + 9*r_mr1 - 9*r_mr1^2) + 4*(1 - 3*r_mr1 + 3*r_mr1^2))) + (x[,"n_13"] + x[,"n_21"])*log(2*(8*(4 - 4*r_mr1^2 + 3*r_mr1^3) + 3*p2*(2 + 3*r_mr1 - 11*r_mr1^2 + 6*r_mr1^3)) + p1*(24*(2 - 3*r_mr1 + 4*r_mr1^2 - 3*r_mr1^3) - 9*p2*(-1 + 9*r_mr1 - 11*r_mr1^2 + 6*r_mr1^3))) + (x[,"n_11"] + x[,"n_23"])*log(48 + 2*(-8 + 3*p2)*r_mr1 + (80 + 42*p2)*r_mr1^2 - 12*(4 + 3*p2)*r_mr1^3 + 3*p1*(8*r_mr1*(4 - 5*r_mr1 + 3*r_mr1^2) + 3*p2*(-3 + 5*r_mr1 - 7*r_mr1^2 + 6*r_mr1^3)))


logL_mr2 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(5*log(2) + 2*log(3)) + (n00 + n34)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n04 + n30)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n10 + n24)*(log(2 - 3*p2) + log(pmax(1e-6,r)) + log(6 - (8 - 12*p1)*r - (-6 + 9*p1)*r^2)) + (n03 + n31)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(4*(1 + r^2) - p2*(6 - 9*r + 6*r^2))) + (n01 + n33)*(log(2 - 3*p1) + log(pmax(1e-6,r)) + log(4*(2 - 2*r + r^2) - p2*(3 - 3*r + 6*r^2))) + (n02 + n32)*(log(2 - 3*p1) + log(8 - 6*r + 6*r^2 + p2*(6 - 9*r + 9*r^2))) + (n14 + n20)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(4 - 4*r + 6*r^2 - p1*(-3 - 6*r + 9*r^2))) + (n12 + n22)*log(2*(28 + 6*r - 6*r^2 + p2*(12 + 9*r - 9*r^2)) + 3*p1*(2 - 6*r + 6*r^2 + p2*(-3 - 9*r + 9*r^2))) + (n13 + n21)*log(-2*(-4*(4 - 4*r^2 + 3*r^3) + 3*p2*(2 + 3*r - 11*r^2 + 6*r^3)) + 3*p1*(4*(2 - 3*r + 4*r^2 - 3*r^3) + 3*p2*(-1 + 9*r - 11*r^2 + 6*r^3))) + (n11 + n23)*log(24 - 2*(4 + 3*p2)*r + (40 - 42*p2)*r^2 + 12*(-2 + 3*p2)*r^3 + p1*(12*r*(4 - 5*r + 3*r^2) - 9*p2*(-3 + 5*r - 7*r^2 + 6*r^3)))
return(L)}
interlogL_mr2 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_mr2,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_mr2 <- parallel::mcmapply(interlogL_mr2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_mr2 <- -((x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-log10(2) + log10(7/2 + (15*p1)/4) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-log10(2) + log10(6 + (6 - 9*p1)/4 + (-8 + 12*p1)/2) + log10(2 - 3*p2)) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(2 - 3*p1) + log10(5 - 3*p2)) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(2 - 3*p1) + log10(5 - 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p1) + log10(13/2 + (15*p2)/4)) - (x[,"n_13"] + x[,"n_21"])*log10(-2*(-27/2 + (9*p2)/2) + 3*p1*(9/2 + (9*p2)/2)) - (x[,"n_11"] + x[,"n_23"])*log10(20 + (40 - 42*p2)/4 - 3*p2 + (3*(-2 + 3*p2))/2 + p1*(27/2 + (27*p2)/2)) - (x[,"n_12"] + x[,"n_22"])*log10(3*p1*(1/2 - (21*p2)/4) + 2*(59/2 + (57*p2)/4)) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_mr2)) + 2*log10(pmax(1e-6,r_mr2))) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_mr2)) + log10(1 - 2*r_mr2 + r_mr2^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(2 - 3*p2) + log10(pmax(1e-6,r_mr2)) + log10(6 - (8 - 12*p1)*r_mr2 - (-6 + 9*p1)*r_mr2^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_mr2)) + log10(4*(1 + r_mr2^2) - p2*(6 - 9*r_mr2 + 6*r_mr2^2))) + (x[,"n_01"] + x[,"n_33"])*(log10(2 - 3*p1) + log10(pmax(1e-6,r_mr2)) + log10(4*(2 - 2*r_mr2 + r_mr2^2) - p2*(3 - 3*r_mr2 + 6*r_mr2^2))) + (x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p1) + log10(8 - 6*r_mr2 + 6*r_mr2^2 + p2*(6 - 9*r_mr2 + 9*r_mr2^2))) + (x[,"n_14"] + x[,"n_20"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_mr2)) + log10(4 - 4*r_mr2 + 6*r_mr2^2 - p1*(-3 - 6*r_mr2 + 9*r_mr2^2))) + (x[,"n_12"] + x[,"n_22"])*log10(2*(28 + 6*r_mr2 - 6*r_mr2^2 + p2*(12 + 9*r_mr2 - 9*r_mr2^2)) + 3*p1*(2 - 6*r_mr2 + 6*r_mr2^2 + p2*(-3 - 9*r_mr2 + 9*r_mr2^2))) + (x[,"n_13"] + x[,"n_21"])*log10(-2*(-4*(4 - 4*r_mr2^2 + 3*r_mr2^3) + 3*p2*(2 + 3*r_mr2 - 11*r_mr2^2 + 6*r_mr2^3)) + 3*p1*(4*(2 - 3*r_mr2 + 4*r_mr2^2 - 3*r_mr2^3) + 3*p2*(-1 + 9*r_mr2 - 11*r_mr2^2 + 6*r_mr2^3))) + (x[,"n_11"] + x[,"n_23"])*log10(24 - 2*(4 + 3*p2)*r_mr2 + (40 - 42*p2)*r_mr2^2 + 12*(-2 + 3*p2)*r_mr2^3 + p1*(12*r_mr2*(4 - 5*r_mr2 + 3*r_mr2^2) - 9*p2*(-3 + 5*r_mr2 - 7*r_mr2^2 + 6*r_mr2^3)))


logL_mr2 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(5*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_mr2)) + 2*log(pmax(1e-6,r_mr2))) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_mr2)) + log(1 - 2*r_mr2 + r_mr2^2)) + (x[,"n_10"] + x[,"n_24"])*(log(2 - 3*p2) + log(pmax(1e-6,r_mr2)) + log(6 - (8 - 12*p1)*r_mr2 - (-6 + 9*p1)*r_mr2^2)) + (x[,"n_03"] + x[,"n_31"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_mr2)) + log(4*(1 + r_mr2^2) - p2*(6 - 9*r_mr2 + 6*r_mr2^2))) + (x[,"n_01"] + x[,"n_33"])*(log(2 - 3*p1) + log(pmax(1e-6,r_mr2)) + log(4*(2 - 2*r_mr2 + r_mr2^2) - p2*(3 - 3*r_mr2 + 6*r_mr2^2))) + (x[,"n_02"] + x[,"n_32"])*(log(2 - 3*p1) + log(8 - 6*r_mr2 + 6*r_mr2^2 + p2*(6 - 9*r_mr2 + 9*r_mr2^2))) + (x[,"n_14"] + x[,"n_20"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_mr2)) + log(4 - 4*r_mr2 + 6*r_mr2^2 - p1*(-3 - 6*r_mr2 + 9*r_mr2^2))) + (x[,"n_12"] + x[,"n_22"])*log(2*(28 + 6*r_mr2 - 6*r_mr2^2 + p2*(12 + 9*r_mr2 - 9*r_mr2^2)) + 3*p1*(2 - 6*r_mr2 + 6*r_mr2^2 + p2*(-3 - 9*r_mr2 + 9*r_mr2^2))) + (x[,"n_13"] + x[,"n_21"])*log(-2*(-4*(4 - 4*r_mr2^2 + 3*r_mr2^3) + 3*p2*(2 + 3*r_mr2 - 11*r_mr2^2 + 6*r_mr2^3)) + 3*p1*(4*(2 - 3*r_mr2 + 4*r_mr2^2 - 3*r_mr2^3) + 3*p2*(-1 + 9*r_mr2 - 11*r_mr2^2 + 6*r_mr2^3))) + (x[,"n_11"] + x[,"n_23"])*log(24 - 2*(4 + 3*p2)*r_mr2 + (40 - 42*p2)*r_mr2^2 + 12*(-2 + 3*p2)*r_mr2^3 + p1*(12*r_mr2*(4 - 5*r_mr2 + 3*r_mr2^2) - 9*p2*(-3 + 5*r_mr2 - 7*r_mr2^2 + 6*r_mr2^3)))


logL_mr3 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(6*log(2) + 2*log(3)) + (n00 + n34)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n04 + n30)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n10 + n24)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(3 - 4*r + 3*r^2)) + (n14 + n20)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(2 - 2*r + 3*r^2)) + (n01 + n33)*(log(pmax(1e-6,r)) + log((2 - 3*p1)*(-8 + 3*p2)*(-1 + r) + (4 + 3*p2)*(p1*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2)))) + (n12 + n22)*log(2*(56 + 12*r - 12*r^2 + 3*p2*(-4 - 3*r + 3*r^2)) - 3*p1*(-4*(4 - 3*r + 3*r^2) + 3*p2*(5 - 3*r + 3*r^2))) + (n03 + n31)*(log(pmax(1e-6,1 - r)) + log(-3*p1*(-4 + 8*r^2 + p2*(-3 - 9*r + 6*r^2)) + 2*(8*(1 + r^2) + p2*(6 - 9*r + 6*r^2)))) + (n02 + n32)*log(2*(p2*(-6 + 9*r - 9*r^2) + 4*(4 - 3*r + 3*r^2)) + 3*p1*(8 + 12*r - 12*r^2 + p2*(-3 - 9*r + 9*r^2))) + (n13 + n21)*log(2*(8*(4 - 4*r^2 + 3*r^3) + 3*p2*(2 + 3*r - 11*r^2 + 6*r^3)) - 3*p1*(3*p2*(-4 + 6*r - 11*r^2 + 6*r^3) + 4*(2 + 3*r - 8*r^2 + 6*r^3))) + (n11 + n23)*log(48 + 2*(-8 + 3*p2)*r + (80 + 42*p2)*r^2 - 12*(4 + 3*p2)*r^3 + 3*p1*(4*(-3 + 5*r - 10*r^2 + 6*r^3) + 3*p2*(3 + 2*r - 7*r^2 + 6*r^3)))
return(L)}
interlogL_mr3 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_mr3,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_mr3 <- parallel::mcmapply(interlogL_mr3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_mr3 <- -((x[,"n_02"] + x[,"n_32"])*log10(3*p1*(11 - (21*p2)/4) + 2*(13 - (15*p2)/4))) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(7) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(7) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(-((2 - 3*p1)*(-8 + 3*p2))/2 + (3 + (9*p1)/2)*(4 + 3*p2))) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(-3*p1*(-2 - 6*p2) + 2*(10 + 3*p2))) - (x[,"n_13"] + x[,"n_21"])*log10(-3*p1*(9 - 9*p2) + 2*(27 + (9*p2)/2)) - (x[,"n_12"] + x[,"n_22"])*log10(2*(59 - (57*p2)/4) - 3*p1*(-13 + (51*p2)/4)) - (x[,"n_11"] + x[,"n_23"])*log10(40 + 3*p2 - (3*(4 + 3*p2))/2 + 3*p1*(-9 + 9*p2) + (80 + 42*p2)/4) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_mr3)) + 2*log10(pmax(1e-6,r_mr3))) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_mr3)) + log10(1 - 2*r_mr3 + r_mr3^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_mr3)) + log10(3 - 4*r_mr3 + 3*r_mr3^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_mr3)) + log10(2 - 2*r_mr3 + 3*r_mr3^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(pmax(1e-6,r_mr3)) + log10((2 - 3*p1)*(-8 + 3*p2)*(-1 + r_mr3) + (4 + 3*p2)*(p1*(3 + 6*r_mr3 - 6*r_mr3^2) + 4*(1 - r_mr3 + r_mr3^2)))) + (x[,"n_12"] + x[,"n_22"])*log10(2*(56 + 12*r_mr3 - 12*r_mr3^2 + 3*p2*(-4 - 3*r_mr3 + 3*r_mr3^2)) - 3*p1*(-4*(4 - 3*r_mr3 + 3*r_mr3^2) + 3*p2*(5 - 3*r_mr3 + 3*r_mr3^2))) + (x[,"n_03"] + x[,"n_31"])*(log10(pmax(1e-6,1 - r_mr3)) + log10(-3*p1*(-4 + 8*r_mr3^2 + p2*(-3 - 9*r_mr3 + 6*r_mr3^2)) + 2*(8*(1 + r_mr3^2) + p2*(6 - 9*r_mr3 + 6*r_mr3^2)))) + (x[,"n_02"] + x[,"n_32"])*log10(2*(p2*(-6 + 9*r_mr3 - 9*r_mr3^2) + 4*(4 - 3*r_mr3 + 3*r_mr3^2)) + 3*p1*(8 + 12*r_mr3 - 12*r_mr3^2 + p2*(-3 - 9*r_mr3 + 9*r_mr3^2))) + (x[,"n_13"] + x[,"n_21"])*log10(2*(8*(4 - 4*r_mr3^2 + 3*r_mr3^3) + 3*p2*(2 + 3*r_mr3 - 11*r_mr3^2 + 6*r_mr3^3)) - 3*p1*(3*p2*(-4 + 6*r_mr3 - 11*r_mr3^2 + 6*r_mr3^3) + 4*(2 + 3*r_mr3 - 8*r_mr3^2 + 6*r_mr3^3))) + (x[,"n_11"] + x[,"n_23"])*log10(48 + 2*(-8 + 3*p2)*r_mr3 + (80 + 42*p2)*r_mr3^2 - 12*(4 + 3*p2)*r_mr3^3 + 3*p1*(4*(-3 + 5*r_mr3 - 10*r_mr3^2 + 6*r_mr3^3) + 3*p2*(3 + 2*r_mr3 - 7*r_mr3^2 + 6*r_mr3^3)))


logL_mr3 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(6*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_mr3)) + 2*log(pmax(1e-6,r_mr3))) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_mr3)) + log(1 - 2*r_mr3 + r_mr3^2)) + (x[,"n_10"] + x[,"n_24"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_mr3)) + log(3 - 4*r_mr3 + 3*r_mr3^2)) + (x[,"n_14"] + x[,"n_20"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_mr3)) + log(2 - 2*r_mr3 + 3*r_mr3^2)) + (x[,"n_01"] + x[,"n_33"])*(log(pmax(1e-6,r_mr3)) + log((2 - 3*p1)*(-8 + 3*p2)*(-1 + r_mr3) + (4 + 3*p2)*(p1*(3 + 6*r_mr3 - 6*r_mr3^2) + 4*(1 - r_mr3 + r_mr3^2)))) + (x[,"n_12"] + x[,"n_22"])*log(2*(56 + 12*r_mr3 - 12*r_mr3^2 + 3*p2*(-4 - 3*r_mr3 + 3*r_mr3^2)) - 3*p1*(-4*(4 - 3*r_mr3 + 3*r_mr3^2) + 3*p2*(5 - 3*r_mr3 + 3*r_mr3^2))) + (x[,"n_03"] + x[,"n_31"])*(log(pmax(1e-6,1 - r_mr3)) + log(-3*p1*(-4 + 8*r_mr3^2 + p2*(-3 - 9*r_mr3 + 6*r_mr3^2)) + 2*(8*(1 + r_mr3^2) + p2*(6 - 9*r_mr3 + 6*r_mr3^2)))) + (x[,"n_02"] + x[,"n_32"])*log(2*(p2*(-6 + 9*r_mr3 - 9*r_mr3^2) + 4*(4 - 3*r_mr3 + 3*r_mr3^2)) + 3*p1*(8 + 12*r_mr3 - 12*r_mr3^2 + p2*(-3 - 9*r_mr3 + 9*r_mr3^2))) + (x[,"n_13"] + x[,"n_21"])*log(2*(8*(4 - 4*r_mr3^2 + 3*r_mr3^3) + 3*p2*(2 + 3*r_mr3 - 11*r_mr3^2 + 6*r_mr3^3)) - 3*p1*(3*p2*(-4 + 6*r_mr3 - 11*r_mr3^2 + 6*r_mr3^3) + 4*(2 + 3*r_mr3 - 8*r_mr3^2 + 6*r_mr3^3))) + (x[,"n_11"] + x[,"n_23"])*log(48 + 2*(-8 + 3*p2)*r_mr3 + (80 + 42*p2)*r_mr3^2 - 12*(4 + 3*p2)*r_mr3^3 + 3*p1*(4*(-3 + 5*r_mr3 - 10*r_mr3^2 + 6*r_mr3^3) + 3*p2*(3 + 2*r_mr3 - 7*r_mr3^2 + 6*r_mr3^3)))


logL_mr4 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(5*log(2) + 2*log(3)) + (n00 + n34)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n04 + n30)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n10 + n24)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(3 - 4*r + 3*r^2)) + (n14 + n20)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(2 - 2*r + 3*r^2)) + (n01 + n33)*(log(pmax(1e-6,r)) + log((-2 + 3*p1)*(4 + 3*p2)*(-1 + r) + (2 - 3*p2)*(p1*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2)))) + (n12 + n22)*log(2*(28 + 6*r - 6*r^2 + p2*(12 + 9*r - 9*r^2)) + 3*p1*(8 - 6*r + 6*r^2 + 3*p2*(5 - 3*r + 3*r^2))) + (n03 + n31)*(log(pmax(1e-6,1 - r)) + log(2*(p2*(-6 + 9*r - 6*r^2) + 4*(1 + r^2)) + 3*p1*(2 - 4*r^2 + p2*(-3 - 9*r + 6*r^2)))) + (n02 + n32)*log(-3*p1*(-4 - 6*r + 6*r^2 + p2*(-3 - 9*r + 9*r^2)) + 2*(8 - 6*r + 6*r^2 + p2*(6 - 9*r + 9*r^2))) + (n13 + n21)*log(-2*(-4*(4 - 4*r^2 + 3*r^3) + 3*p2*(2 + 3*r - 11*r^2 + 6*r^3)) + 3*p1*(3*p2*(-4 + 6*r - 11*r^2 + 6*r^3) - 2*(2 + 3*r - 8*r^2 + 6*r^3))) + (n11 + n23)*log(24 - 2*(4 + 3*p2)*r + (40 - 42*p2)*r^2 + 12*(-2 + 3*p2)*r^3 - 3*p1*(6 - 10*r + 20*r^2 - 12*r^3 + 3*p2*(3 + 2*r - 7*r^2 + 6*r^3)))
return(L)}
interlogL_mr4 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_mr4,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_mr4 <- parallel::mcmapply(interlogL_mr4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_mr4 <- -((x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(3*p1*(1 - 6*p2) + 2*(5 - 3*p2)))) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(7) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(7) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10((3 + (9*p1)/2)*(2 - 3*p2) - ((-2 + 3*p1)*(4 + 3*p2))/2)) - (x[,"n_02"] + x[,"n_32"])*log10(-3*p1*(-11/2 - (21*p2)/4) + 2*(13/2 + (15*p2)/4)) - (x[,"n_13"] + x[,"n_21"])*log10(3*p1*(-9/2 - 9*p2) - 2*(-27/2 + (9*p2)/2)) - (x[,"n_11"] + x[,"n_23"])*log10(20 + (40 - 42*p2)/4 - 3*p2 + (3*(-2 + 3*p2))/2 - 3*p1*(9/2 + 9*p2)) - (x[,"n_12"] + x[,"n_22"])*log10(3*p1*(13/2 + (51*p2)/4) + 2*(59/2 + (57*p2)/4)) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_mr4)) + 2*log10(pmax(1e-6,r_mr4))) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_mr4)) + log10(1 - 2*r_mr4 + r_mr4^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_mr4)) + log10(3 - 4*r_mr4 + 3*r_mr4^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_mr4)) + log10(2 - 2*r_mr4 + 3*r_mr4^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(pmax(1e-6,r_mr4)) + log10((-2 + 3*p1)*(4 + 3*p2)*(-1 + r_mr4) + (2 - 3*p2)*(p1*(3 + 6*r_mr4 - 6*r_mr4^2) + 4*(1 - r_mr4 + r_mr4^2)))) + (x[,"n_12"] + x[,"n_22"])*log10(2*(28 + 6*r_mr4 - 6*r_mr4^2 + p2*(12 + 9*r_mr4 - 9*r_mr4^2)) + 3*p1*(8 - 6*r_mr4 + 6*r_mr4^2 + 3*p2*(5 - 3*r_mr4 + 3*r_mr4^2))) + (x[,"n_03"] + x[,"n_31"])*(log10(pmax(1e-6,1 - r_mr4)) + log10(2*(p2*(-6 + 9*r_mr4 - 6*r_mr4^2) + 4*(1 + r_mr4^2)) + 3*p1*(2 - 4*r_mr4^2 + p2*(-3 - 9*r_mr4 + 6*r_mr4^2)))) + (x[,"n_02"] + x[,"n_32"])*log10(-3*p1*(-4 - 6*r_mr4 + 6*r_mr4^2 + p2*(-3 - 9*r_mr4 + 9*r_mr4^2)) + 2*(8 - 6*r_mr4 + 6*r_mr4^2 + p2*(6 - 9*r_mr4 + 9*r_mr4^2))) + (x[,"n_13"] + x[,"n_21"])*log10(-2*(-4*(4 - 4*r_mr4^2 + 3*r_mr4^3) + 3*p2*(2 + 3*r_mr4 - 11*r_mr4^2 + 6*r_mr4^3)) + 3*p1*(3*p2*(-4 + 6*r_mr4 - 11*r_mr4^2 + 6*r_mr4^3) - 2*(2 + 3*r_mr4 - 8*r_mr4^2 + 6*r_mr4^3))) + (x[,"n_11"] + x[,"n_23"])*log10(24 - 2*(4 + 3*p2)*r_mr4 + (40 - 42*p2)*r_mr4^2 + 12*(-2 + 3*p2)*r_mr4^3 - 3*p1*(6 - 10*r_mr4 + 20*r_mr4^2 - 12*r_mr4^3 + 3*p2*(3 + 2*r_mr4 - 7*r_mr4^2 + 6*r_mr4^3)))


logL_mr4 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(5*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_mr4)) + 2*log(pmax(1e-6,r_mr4))) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_mr4)) + log(1 - 2*r_mr4 + r_mr4^2)) + (x[,"n_10"] + x[,"n_24"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_mr4)) + log(3 - 4*r_mr4 + 3*r_mr4^2)) + (x[,"n_14"] + x[,"n_20"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_mr4)) + log(2 - 2*r_mr4 + 3*r_mr4^2)) + (x[,"n_01"] + x[,"n_33"])*(log(pmax(1e-6,r_mr4)) + log((-2 + 3*p1)*(4 + 3*p2)*(-1 + r_mr4) + (2 - 3*p2)*(p1*(3 + 6*r_mr4 - 6*r_mr4^2) + 4*(1 - r_mr4 + r_mr4^2)))) + (x[,"n_12"] + x[,"n_22"])*log(2*(28 + 6*r_mr4 - 6*r_mr4^2 + p2*(12 + 9*r_mr4 - 9*r_mr4^2)) + 3*p1*(8 - 6*r_mr4 + 6*r_mr4^2 + 3*p2*(5 - 3*r_mr4 + 3*r_mr4^2))) + (x[,"n_03"] + x[,"n_31"])*(log(pmax(1e-6,1 - r_mr4)) + log(2*(p2*(-6 + 9*r_mr4 - 6*r_mr4^2) + 4*(1 + r_mr4^2)) + 3*p1*(2 - 4*r_mr4^2 + p2*(-3 - 9*r_mr4 + 6*r_mr4^2)))) + (x[,"n_02"] + x[,"n_32"])*log(-3*p1*(-4 - 6*r_mr4 + 6*r_mr4^2 + p2*(-3 - 9*r_mr4 + 9*r_mr4^2)) + 2*(8 - 6*r_mr4 + 6*r_mr4^2 + p2*(6 - 9*r_mr4 + 9*r_mr4^2))) + (x[,"n_13"] + x[,"n_21"])*log(-2*(-4*(4 - 4*r_mr4^2 + 3*r_mr4^3) + 3*p2*(2 + 3*r_mr4 - 11*r_mr4^2 + 6*r_mr4^3)) + 3*p1*(3*p2*(-4 + 6*r_mr4 - 11*r_mr4^2 + 6*r_mr4^3) - 2*(2 + 3*r_mr4 - 8*r_mr4^2 + 6*r_mr4^3))) + (x[,"n_11"] + x[,"n_23"])*log(24 - 2*(4 + 3*p2)*r_mr4 + (40 - 42*p2)*r_mr4^2 + 12*(-2 + 3*p2)*r_mr4^3 - 3*p1*(6 - 10*r_mr4 + 20*r_mr4^2 - 12*r_mr4^3 + 3*p2*(3 + 2*r_mr4 - 7*r_mr4^2 + 6*r_mr4^3)))


logL_rc1 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-4*n00 - 3*n01 - 4*n02 - 3*n03 - 4*n04 - 4*n10 - 4*n11 - 4*n12 - 4*n13 - 4*n14 - 4*n20 - 4*n21 - 4*n22 - 4*n23 - 4*n24 - 4*n30 - 3*n31 - 4*n32 - 3*n33 - 4*n34)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(3) + (n00 + n34)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n01 + n33)*(log(2 - 3*p1) + log(pmax(1e-6,r)) + log((2 - 3*p2)*(-1 + r)^2 + ((4 + 3*p2)*r)/2)) + (n04 + n30)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n10 + n24)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n14 + n20)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + 3*r^2)) + (n03 + n31)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(-((4 + 3*p2)*(-1 + r))/2 + (2 - 3*p2)*r^2)) + (n02 + n32)*(log(2 - 3*p1) + log(2*(1 + r - r^2) - 3*p2*(1 - 5*r + 5*r^2))) + (n12 + n22)*log(4*(8 - r + r^2) + 6*p2*(4 - 5*r + 5*r^2) + p1*(6*(4 + r - r^2) - 9*p2*(-2 - 5*r + 5*r^2))) + (n11 + n23)*log(-3*p1*(4*(-1 + r + 4*r^2 - 3*r^3) + 3*p2*(2 + 4*r - 11*r^2 + 6*r^3)) + 2*(8 - 8*r + 16*r^2 - 12*r^3 + 3*p2*(-4 + 10*r - 11*r^2 + 6*r^3))) + (n13 + n21)*log(8*(1 + 3*r - 5*r^2 + 3*r^3) - 6*p2*(-1 + 6*r - 7*r^2 + 6*r^3) + 3*p1*(-4*(1 - 5*r^2 + 3*r^3) + 3*p2*(-1 - 7*r^2 + 6*r^3)))
return(L)}
interlogL_rc1 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rc1,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc1 <- parallel::mcmapply(interlogL_rc1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rc1 <- -((x[,"n_13"] + x[,"n_21"])*log10(13 + 3*p1*(-1/2 - 6*p2) - 6*p2)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(3) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(3) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p1) + log10(5/2 + (3*p2)/4)) - (x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(2 - 3*p1) + log10((2 - 3*p2)/4 + (4 + 3*p2)/4)) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(2 - 3*p1) + log10((2 - 3*p2)/4 + (4 + 3*p2)/4)) - (x[,"n_11"] + x[,"n_23"])*log10(2*(13/2 - 3*p2) - 3*p1*(1/2 + 6*p2)) - (x[,"n_12"] + x[,"n_22"])*log10(31 + (33*p2)/2 + p1*(51/2 + (117*p2)/4)) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rc1)) + 2*log10(pmax(1e-6,r_rc1))) + (x[,"n_01"] + x[,"n_33"])*(log10(2 - 3*p1) + log10(pmax(1e-6,r_rc1)) + log10((2 - 3*p2)*(-1 + r_rc1)^2 + ((4 + 3*p2)*r_rc1)/2)) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_rc1)) + log10(1 - 2*r_rc1 + r_rc1^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_rc1)) + log10(2 - 4*r_rc1 + 3*r_rc1^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rc1)) + log10(1 - 2*r_rc1 + 3*r_rc1^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_rc1)) + log10(-((4 + 3*p2)*(-1 + r_rc1))/2 + (2 - 3*p2)*r_rc1^2)) + (x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p1) + log10(2*(1 + r_rc1 - r_rc1^2) - 3*p2*(1 - 5*r_rc1 + 5*r_rc1^2))) + (x[,"n_12"] + x[,"n_22"])*log10(4*(8 - r_rc1 + r_rc1^2) + 6*p2*(4 - 5*r_rc1 + 5*r_rc1^2) + p1*(6*(4 + r_rc1 - r_rc1^2) - 9*p2*(-2 - 5*r_rc1 + 5*r_rc1^2))) + (x[,"n_11"] + x[,"n_23"])*log10(-3*p1*(4*(-1 + r_rc1 + 4*r_rc1^2 - 3*r_rc1^3) + 3*p2*(2 + 4*r_rc1 - 11*r_rc1^2 + 6*r_rc1^3)) + 2*(8 - 8*r_rc1 + 16*r_rc1^2 - 12*r_rc1^3 + 3*p2*(-4 + 10*r_rc1 - 11*r_rc1^2 + 6*r_rc1^3))) + (x[,"n_13"] + x[,"n_21"])*log10(8*(1 + 3*r_rc1 - 5*r_rc1^2 + 3*r_rc1^3) - 6*p2*(-1 + 6*r_rc1 - 7*r_rc1^2 + 6*r_rc1^3) + 3*p1*(-4*(1 - 5*r_rc1^2 + 3*r_rc1^3) + 3*p2*(-1 - 7*r_rc1^2 + 6*r_rc1^3)))


logL_rc1 <- (-4*x[,"n_00"] - 3*x[,"n_01"] - 4*x[,"n_02"] - 3*x[,"n_03"] - 4*x[,"n_04"] - 4*x[,"n_10"] - 4*x[,"n_11"] - 4*x[,"n_12"] - 4*x[,"n_13"] - 4*x[,"n_14"] - 4*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 4*x[,"n_23"] - 4*x[,"n_24"] - 4*x[,"n_30"] - 3*x[,"n_31"] - 4*x[,"n_32"] - 3*x[,"n_33"] - 4*x[,"n_34"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(3) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rc1)) + 2*log(pmax(1e-6,r_rc1))) + (x[,"n_01"] + x[,"n_33"])*(log(2 - 3*p1) + log(pmax(1e-6,r_rc1)) + log((2 - 3*p2)*(-1 + r_rc1)^2 + ((4 + 3*p2)*r_rc1)/2)) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_rc1)) + log(1 - 2*r_rc1 + r_rc1^2)) + (x[,"n_10"] + x[,"n_24"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_rc1)) + log(2 - 4*r_rc1 + 3*r_rc1^2)) + (x[,"n_14"] + x[,"n_20"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rc1)) + log(1 - 2*r_rc1 + 3*r_rc1^2)) + (x[,"n_03"] + x[,"n_31"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_rc1)) + log(-((4 + 3*p2)*(-1 + r_rc1))/2 + (2 - 3*p2)*r_rc1^2)) + (x[,"n_02"] + x[,"n_32"])*(log(2 - 3*p1) + log(2*(1 + r_rc1 - r_rc1^2) - 3*p2*(1 - 5*r_rc1 + 5*r_rc1^2))) + (x[,"n_12"] + x[,"n_22"])*log(4*(8 - r_rc1 + r_rc1^2) + 6*p2*(4 - 5*r_rc1 + 5*r_rc1^2) + p1*(6*(4 + r_rc1 - r_rc1^2) - 9*p2*(-2 - 5*r_rc1 + 5*r_rc1^2))) + (x[,"n_11"] + x[,"n_23"])*log(-3*p1*(4*(-1 + r_rc1 + 4*r_rc1^2 - 3*r_rc1^3) + 3*p2*(2 + 4*r_rc1 - 11*r_rc1^2 + 6*r_rc1^3)) + 2*(8 - 8*r_rc1 + 16*r_rc1^2 - 12*r_rc1^3 + 3*p2*(-4 + 10*r_rc1 - 11*r_rc1^2 + 6*r_rc1^3))) + (x[,"n_13"] + x[,"n_21"])*log(8*(1 + 3*r_rc1 - 5*r_rc1^2 + 3*r_rc1^3) - 6*p2*(-1 + 6*r_rc1 - 7*r_rc1^2 + 6*r_rc1^3) + 3*p1*(-4*(1 - 5*r_rc1^2 + 3*r_rc1^3) + 3*p2*(-1 - 7*r_rc1^2 + 6*r_rc1^3)))


logL_rc2 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(5*log(2) + 2*log(3)) + (n00 + n34)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n04 + n30)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n10 + n24)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n14 + n20)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + 3*r^2)) + (n01 + n33)*(log(2 - 3*p1) + log(pmax(1e-6,r)) + log(8*(1 - r + r^2) + 3*p2*(2 - 5*r + 2*r^2))) + (n03 + n31)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(8*(1 - r + r^2) + 3*p2*(-1 + r + 2*r^2))) + (n02 + n32)*(log(2 - 3*p1) + log(4 + 4*r - 4*r^2 + 3*p2*(1 - 5*r + 5*r^2))) + (n12 + n22)*log(8*(8 - r + r^2) - 6*p2*(4 - 5*r + 5*r^2) + 3*p1*(4*(4 + r - r^2) + 3*p2*(-2 - 5*r + 5*r^2))) + (n11 + n23)*log(16*(2 - 2*r + 4*r^2 - 3*r^3) - 6*p2*(-4 + 10*r - 11*r^2 + 6*r^3) + 3*p1*(8*(1 - r - 4*r^2 + 3*r^3) + 3*p2*(2 + 4*r - 11*r^2 + 6*r^3))) + (n13 + n21)*log(16*(1 + 3*r - 5*r^2 + 3*r^3) + 6*p2*(-1 + 6*r - 7*r^2 + 6*r^3) - 3*p1*(8*(1 - 5*r^2 + 3*r^3) + 3*p2*(-1 - 7*r^2 + 6*r^3)))
return(L)}
interlogL_rc2 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rc2,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc2 <- parallel::mcmapply(interlogL_rc2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rc2 <- -((x[,"n_03"] + x[,"n_31"])*(log10(3) + log10(2 - 3*p1))) - (x[,"n_01"] + x[,"n_33"])*(log10(3) + log10(2 - 3*p1)) - (x[,"n_12"] + x[,"n_22"])*log10(62 + 3*p1*(17 - (39*p2)/4) - (33*p2)/2) - (x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p1) + log10(5 - (3*p2)/4)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(3) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(3) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_13"] + x[,"n_21"])*log10(26 - 3*p1*(1 - 6*p2) + 6*p2) - (x[,"n_11"] + x[,"n_23"])*log10(26 + 6*p2 + 3*p1*(-1 + 6*p2)) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rc2)) + 2*log10(pmax(1e-6,r_rc2))) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_rc2)) + log10(1 - 2*r_rc2 + r_rc2^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_rc2)) + log10(2 - 4*r_rc2 + 3*r_rc2^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rc2)) + log10(1 - 2*r_rc2 + 3*r_rc2^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(2 - 3*p1) + log10(pmax(1e-6,r_rc2)) + log10(8*(1 - r_rc2 + r_rc2^2) + 3*p2*(2 - 5*r_rc2 + 2*r_rc2^2))) + (x[,"n_03"] + x[,"n_31"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_rc2)) + log10(8*(1 - r_rc2 + r_rc2^2) + 3*p2*(-1 + r_rc2 + 2*r_rc2^2))) + (x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p1) + log10(4 + 4*r_rc2 - 4*r_rc2^2 + 3*p2*(1 - 5*r_rc2 + 5*r_rc2^2))) + (x[,"n_12"] + x[,"n_22"])*log10(8*(8 - r_rc2 + r_rc2^2) - 6*p2*(4 - 5*r_rc2 + 5*r_rc2^2) + 3*p1*(4*(4 + r_rc2 - r_rc2^2) + 3*p2*(-2 - 5*r_rc2 + 5*r_rc2^2))) + (x[,"n_11"] + x[,"n_23"])*log10(16*(2 - 2*r_rc2 + 4*r_rc2^2 - 3*r_rc2^3) - 6*p2*(-4 + 10*r_rc2 - 11*r_rc2^2 + 6*r_rc2^3) + 3*p1*(8*(1 - r_rc2 - 4*r_rc2^2 + 3*r_rc2^3) + 3*p2*(2 + 4*r_rc2 - 11*r_rc2^2 + 6*r_rc2^3))) + (x[,"n_13"] + x[,"n_21"])*log10(16*(1 + 3*r_rc2 - 5*r_rc2^2 + 3*r_rc2^3) + 6*p2*(-1 + 6*r_rc2 - 7*r_rc2^2 + 6*r_rc2^3) - 3*p1*(8*(1 - 5*r_rc2^2 + 3*r_rc2^3) + 3*p2*(-1 - 7*r_rc2^2 + 6*r_rc2^3)))


logL_rc2 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(5*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rc2)) + 2*log(pmax(1e-6,r_rc2))) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_rc2)) + log(1 - 2*r_rc2 + r_rc2^2)) + (x[,"n_10"] + x[,"n_24"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_rc2)) + log(2 - 4*r_rc2 + 3*r_rc2^2)) + (x[,"n_14"] + x[,"n_20"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rc2)) + log(1 - 2*r_rc2 + 3*r_rc2^2)) + (x[,"n_01"] + x[,"n_33"])*(log(2 - 3*p1) + log(pmax(1e-6,r_rc2)) + log(8*(1 - r_rc2 + r_rc2^2) + 3*p2*(2 - 5*r_rc2 + 2*r_rc2^2))) + (x[,"n_03"] + x[,"n_31"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_rc2)) + log(8*(1 - r_rc2 + r_rc2^2) + 3*p2*(-1 + r_rc2 + 2*r_rc2^2))) + (x[,"n_02"] + x[,"n_32"])*(log(2 - 3*p1) + log(4 + 4*r_rc2 - 4*r_rc2^2 + 3*p2*(1 - 5*r_rc2 + 5*r_rc2^2))) + (x[,"n_12"] + x[,"n_22"])*log(8*(8 - r_rc2 + r_rc2^2) - 6*p2*(4 - 5*r_rc2 + 5*r_rc2^2) + 3*p1*(4*(4 + r_rc2 - r_rc2^2) + 3*p2*(-2 - 5*r_rc2 + 5*r_rc2^2))) + (x[,"n_11"] + x[,"n_23"])*log(16*(2 - 2*r_rc2 + 4*r_rc2^2 - 3*r_rc2^3) - 6*p2*(-4 + 10*r_rc2 - 11*r_rc2^2 + 6*r_rc2^3) + 3*p1*(8*(1 - r_rc2 - 4*r_rc2^2 + 3*r_rc2^3) + 3*p2*(2 + 4*r_rc2 - 11*r_rc2^2 + 6*r_rc2^3))) + (x[,"n_13"] + x[,"n_21"])*log(16*(1 + 3*r_rc2 - 5*r_rc2^2 + 3*r_rc2^3) + 6*p2*(-1 + 6*r_rc2 - 7*r_rc2^2 + 6*r_rc2^3) - 3*p1*(8*(1 - 5*r_rc2^2 + 3*r_rc2^3) + 3*p2*(-1 - 7*r_rc2^2 + 6*r_rc2^3)))


logL_rc3 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(5*log(2) + 2*log(3)) + (n00 + n34)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n04 + n30)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n10 + n24)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n14 + n20)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + 3*r^2)) + (n01 + n33)*(log(4 + 3*p1) + log(pmax(1e-6,r)) + log(4*(1 - r + r^2) - 3*p2*(2 - 5*r + 2*r^2))) + (n03 + n31)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(4*(1 - r + r^2) - 3*p2*(-1 + r + 2*r^2))) + (n02 + n32)*(log(4 + 3*p1) + log(2*(1 + r - r^2) - 3*p2*(1 - 5*r + 5*r^2))) + (n12 + n22)*log(8*(8 - r + r^2) + 12*p2*(4 - 5*r + 5*r^2) + 3*p1*(2*(-4 - r + r^2) + 3*p2*(-2 - 5*r + 5*r^2))) + (n13 + n21)*log(16*(1 + 3*r - 5*r^2 + 3*r^3) - 12*p2*(-1 + 6*r - 7*r^2 + 6*r^3) + p1*(p2*(9 + 63*r^2 - 54*r^3) + 12*(1 - 5*r^2 + 3*r^3))) + (n11 + n23)*log(3*p1*(4*(-1 + r + 4*r^2 - 3*r^3) + 3*p2*(2 + 4*r - 11*r^2 + 6*r^3)) + 4*(8 - 8*r + 16*r^2 - 12*r^3 + 3*p2*(-4 + 10*r - 11*r^2 + 6*r^3)))
return(L)}
interlogL_rc3 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rc3,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc3 <- parallel::mcmapply(interlogL_rc3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rc3 <- -((x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(3) + log10(4 + 3*p1))) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(4 + 3*p1)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(3) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(3) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(log10(4 + 3*p1) + log10(5/2 + (3*p2)/4)) - (x[,"n_12"] + x[,"n_22"])*log10(62 + 3*p1*(-17/2 - (39*p2)/4) + 33*p2) - (x[,"n_11"] + x[,"n_23"])*log10(4*(13/2 - 3*p2) + 3*p1*(1/2 + 6*p2)) - (x[,"n_13"] + x[,"n_21"])*log10(26 - 12*p2 + p1*(3/2 + 18*p2)) + (x[,"n_00"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rc3)) + 2*log10(pmax(1e-6,r_rc3))) + (x[,"n_04"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_rc3)) + log10(1 - 2*r_rc3 + r_rc3^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_rc3)) + log10(2 - 4*r_rc3 + 3*r_rc3^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rc3)) + log10(1 - 2*r_rc3 + 3*r_rc3^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(4 + 3*p1) + log10(pmax(1e-6,r_rc3)) + log10(4*(1 - r_rc3 + r_rc3^2) - 3*p2*(2 - 5*r_rc3 + 2*r_rc3^2))) + (x[,"n_03"] + x[,"n_31"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_rc3)) + log10(4*(1 - r_rc3 + r_rc3^2) - 3*p2*(-1 + r_rc3 + 2*r_rc3^2))) + (x[,"n_02"] + x[,"n_32"])*(log10(4 + 3*p1) + log10(2*(1 + r_rc3 - r_rc3^2) - 3*p2*(1 - 5*r_rc3 + 5*r_rc3^2))) + (x[,"n_12"] + x[,"n_22"])*log10(8*(8 - r_rc3 + r_rc3^2) + 12*p2*(4 - 5*r_rc3 + 5*r_rc3^2) + 3*p1*(2*(-4 - r_rc3 + r_rc3^2) + 3*p2*(-2 - 5*r_rc3 + 5*r_rc3^2))) + (x[,"n_13"] + x[,"n_21"])*log10(16*(1 + 3*r_rc3 - 5*r_rc3^2 + 3*r_rc3^3) - 12*p2*(-1 + 6*r_rc3 - 7*r_rc3^2 + 6*r_rc3^3) + p1*(p2*(9 + 63*r_rc3^2 - 54*r_rc3^3) + 12*(1 - 5*r_rc3^2 + 3*r_rc3^3))) + (x[,"n_11"] + x[,"n_23"])*log10(3*p1*(4*(-1 + r_rc3 + 4*r_rc3^2 - 3*r_rc3^3) + 3*p2*(2 + 4*r_rc3 - 11*r_rc3^2 + 6*r_rc3^3)) + 4*(8 - 8*r_rc3 + 16*r_rc3^2 - 12*r_rc3^3 + 3*p2*(-4 + 10*r_rc3 - 11*r_rc3^2 + 6*r_rc3^3)))


logL_rc3 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(5*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_34"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rc3)) + 2*log(pmax(1e-6,r_rc3))) + (x[,"n_04"] + x[,"n_30"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_rc3)) + log(1 - 2*r_rc3 + r_rc3^2)) + (x[,"n_10"] + x[,"n_24"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_rc3)) + log(2 - 4*r_rc3 + 3*r_rc3^2)) + (x[,"n_14"] + x[,"n_20"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rc3)) + log(1 - 2*r_rc3 + 3*r_rc3^2)) + (x[,"n_01"] + x[,"n_33"])*(log(4 + 3*p1) + log(pmax(1e-6,r_rc3)) + log(4*(1 - r_rc3 + r_rc3^2) - 3*p2*(2 - 5*r_rc3 + 2*r_rc3^2))) + (x[,"n_03"] + x[,"n_31"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_rc3)) + log(4*(1 - r_rc3 + r_rc3^2) - 3*p2*(-1 + r_rc3 + 2*r_rc3^2))) + (x[,"n_02"] + x[,"n_32"])*(log(4 + 3*p1) + log(2*(1 + r_rc3 - r_rc3^2) - 3*p2*(1 - 5*r_rc3 + 5*r_rc3^2))) + (x[,"n_12"] + x[,"n_22"])*log(8*(8 - r_rc3 + r_rc3^2) + 12*p2*(4 - 5*r_rc3 + 5*r_rc3^2) + 3*p1*(2*(-4 - r_rc3 + r_rc3^2) + 3*p2*(-2 - 5*r_rc3 + 5*r_rc3^2))) + (x[,"n_13"] + x[,"n_21"])*log(16*(1 + 3*r_rc3 - 5*r_rc3^2 + 3*r_rc3^3) - 12*p2*(-1 + 6*r_rc3 - 7*r_rc3^2 + 6*r_rc3^3) + p1*(p2*(9 + 63*r_rc3^2 - 54*r_rc3^3) + 12*(1 - 5*r_rc3^2 + 3*r_rc3^3))) + (x[,"n_11"] + x[,"n_23"])*log(3*p1*(4*(-1 + r_rc3 + 4*r_rc3^2 - 3*r_rc3^3) + 3*p2*(2 + 4*r_rc3 - 11*r_rc3^2 + 6*r_rc3^3)) + 4*(8 - 8*r_rc3 + 16*r_rc3^2 - 12*r_rc3^3 + 3*p2*(-4 + 10*r_rc3 - 11*r_rc3^2 + 6*r_rc3^3)))


logL_rc4 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- (-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*(6*log(2) + 2*log(3)) + (n00 + n34)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n04 + n30)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n10 + n24)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(2 - 4*r + 3*r^2)) + (n14 + n20)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + 3*r^2)) + (n01 + n33)*(log(4 + 3*p1) + log(pmax(1e-6,r)) + log(8*(1 - r + r^2) + 3*p2*(2 - 5*r + 2*r^2))) + (n03 + n31)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(8*(1 - r + r^2) + 3*p2*(-1 + r + 2*r^2))) + (n02 + n32)*(log(4 + 3*p1) + log(4 + 4*r - 4*r^2 + 3*p2*(1 - 5*r + 5*r^2))) + (n12 + n22)*log(-3*p1*(4*(4 + r - r^2) + 3*p2*(-2 - 5*r + 5*r^2)) + 4*(4*(8 - r + r^2) - 3*p2*(4 - 5*r + 5*r^2))) + (n11 + n23)*log(-3*p1*(8*(1 - r - 4*r^2 + 3*r^3) + 3*p2*(2 + 4*r - 11*r^2 + 6*r^3)) + 4*(-8*(-2 + 2*r - 4*r^2 + 3*r^3) - 3*p2*(-4 + 10*r - 11*r^2 + 6*r^3))) + (n13 + n21)*log(3*p1*(8*(1 - 5*r^2 + 3*r^3) + 3*p2*(-1 - 7*r^2 + 6*r^3)) + 4*(8*(1 + 3*r - 5*r^2 + 3*r^3) + 3*p2*(-1 + 6*r - 7*r^2 + 6*r^3)))
return(L)}
interlogL_rc4 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rc4,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rc4 <- parallel::mcmapply(interlogL_rc4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rc4 <- -((x[,"n_03"] + x[,"n_31"])*(log10(3) + log10(4 + 3*p1))) - (x[,"n_01"] + x[,"n_33"])*(log10(3) + log10(4 + 3*p1)) - (x[,"n_12"] + x[,"n_22"])*log10(-3*p1*(17 - (39*p2)/4) + 4*(31 - (33*p2)/4)) - (x[,"n_02"] + x[,"n_32"])*(log10(4 + 3*p1) + log10(5 - (3*p2)/4)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(3) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(3) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_13"] + x[,"n_21"])*log10(3*p1*(1 - 6*p2) + 4*(13 + 3*p2)) - (x[,"n_11"] + x[,"n_23"])*log10(4*(13 + 3*p2) - 3*p1*(-1 + 6*p2)) + (x[,"n_00"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rc4)) + 2*log10(pmax(1e-6,r_rc4))) + (x[,"n_04"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_rc4)) + log10(1 - 2*r_rc4 + r_rc4^2)) + (x[,"n_10"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_rc4)) + log10(2 - 4*r_rc4 + 3*r_rc4^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rc4)) + log10(1 - 2*r_rc4 + 3*r_rc4^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(4 + 3*p1) + log10(pmax(1e-6,r_rc4)) + log10(8*(1 - r_rc4 + r_rc4^2) + 3*p2*(2 - 5*r_rc4 + 2*r_rc4^2))) + (x[,"n_03"] + x[,"n_31"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_rc4)) + log10(8*(1 - r_rc4 + r_rc4^2) + 3*p2*(-1 + r_rc4 + 2*r_rc4^2))) + (x[,"n_02"] + x[,"n_32"])*(log10(4 + 3*p1) + log10(4 + 4*r_rc4 - 4*r_rc4^2 + 3*p2*(1 - 5*r_rc4 + 5*r_rc4^2))) + (x[,"n_12"] + x[,"n_22"])*log10(-3*p1*(4*(4 + r_rc4 - r_rc4^2) + 3*p2*(-2 - 5*r_rc4 + 5*r_rc4^2)) + 4*(4*(8 - r_rc4 + r_rc4^2) - 3*p2*(4 - 5*r_rc4 + 5*r_rc4^2))) + (x[,"n_11"] + x[,"n_23"])*log10(-3*p1*(8*(1 - r_rc4 - 4*r_rc4^2 + 3*r_rc4^3) + 3*p2*(2 + 4*r_rc4 - 11*r_rc4^2 + 6*r_rc4^3)) + 4*(-8*(-2 + 2*r_rc4 - 4*r_rc4^2 + 3*r_rc4^3) - 3*p2*(-4 + 10*r_rc4 - 11*r_rc4^2 + 6*r_rc4^3))) + (x[,"n_13"] + x[,"n_21"])*log10(3*p1*(8*(1 - 5*r_rc4^2 + 3*r_rc4^3) + 3*p2*(-1 - 7*r_rc4^2 + 6*r_rc4^3)) + 4*(8*(1 + 3*r_rc4 - 5*r_rc4^2 + 3*r_rc4^3) + 3*p2*(-1 + 6*r_rc4 - 7*r_rc4^2 + 6*r_rc4^3)))


logL_rc4 <- (-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*(6*log(2) + 2*log(3)) + (x[,"n_00"] + x[,"n_34"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rc4)) + 2*log(pmax(1e-6,r_rc4))) + (x[,"n_04"] + x[,"n_30"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_rc4)) + log(1 - 2*r_rc4 + r_rc4^2)) + (x[,"n_10"] + x[,"n_24"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_rc4)) + log(2 - 4*r_rc4 + 3*r_rc4^2)) + (x[,"n_14"] + x[,"n_20"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rc4)) + log(1 - 2*r_rc4 + 3*r_rc4^2)) + (x[,"n_01"] + x[,"n_33"])*(log(4 + 3*p1) + log(pmax(1e-6,r_rc4)) + log(8*(1 - r_rc4 + r_rc4^2) + 3*p2*(2 - 5*r_rc4 + 2*r_rc4^2))) + (x[,"n_03"] + x[,"n_31"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_rc4)) + log(8*(1 - r_rc4 + r_rc4^2) + 3*p2*(-1 + r_rc4 + 2*r_rc4^2))) + (x[,"n_02"] + x[,"n_32"])*(log(4 + 3*p1) + log(4 + 4*r_rc4 - 4*r_rc4^2 + 3*p2*(1 - 5*r_rc4 + 5*r_rc4^2))) + (x[,"n_12"] + x[,"n_22"])*log(-3*p1*(4*(4 + r_rc4 - r_rc4^2) + 3*p2*(-2 - 5*r_rc4 + 5*r_rc4^2)) + 4*(4*(8 - r_rc4 + r_rc4^2) - 3*p2*(4 - 5*r_rc4 + 5*r_rc4^2))) + (x[,"n_11"] + x[,"n_23"])*log(-3*p1*(8*(1 - r_rc4 - 4*r_rc4^2 + 3*r_rc4^3) + 3*p2*(2 + 4*r_rc4 - 11*r_rc4^2 + 6*r_rc4^3)) + 4*(-8*(-2 + 2*r_rc4 - 4*r_rc4^2 + 3*r_rc4^3) - 3*p2*(-4 + 10*r_rc4 - 11*r_rc4^2 + 6*r_rc4^3))) + (x[,"n_13"] + x[,"n_21"])*log(3*p1*(8*(1 - 5*r_rc4^2 + 3*r_rc4^3) + 3*p2*(-1 - 7*r_rc4^2 + 6*r_rc4^3)) + 4*(8*(1 + 3*r_rc4 - 5*r_rc4^2 + 3*r_rc4^3) + 3*p2*(-1 + 6*r_rc4 - 7*r_rc4^2 + 6*r_rc4^3)))


logL_rr1 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- 5*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(2) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - n10 - 2*n11 - 2*n12 - 2*n13 - n14 - n20 - 2*n21 - 2*n22 - 2*n23 - n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34)*log(3) + (n02 + n32)*(log(2 - 3*p1) + log(20 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n24)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n00 + n34)*(log(2 - 3*p1) + log(4 + 3*p2) + 3*log(pmax(1e-6,r))) + (n04 + n30)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + r^2)) + (n14 + n20)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n03 + n31)*(log(2 - 3*p1) + log(8 - 3*p2 + 8*r + 6*p2*r) + log(1 - 2*r + r^2)) + (n01 + n33)*(log(2 - 3*p1) + 2*log(pmax(1e-6,r)) + log(8*(2 - r) - p2*(-3 + 6*r))) + (n12 + n22)*log(-6*p2*(3 - r + r^2) + 8*(9 - 5*r + 5*r^2) + 3*p1*(4*(3 + 5*r - 5*r^2) + 3*p2*(-3 - r + r^2))) + (n11 + n23)*(log(pmax(1e-6,r)) + log(16*(5 - 5*r + 3*r^2) + 6*p2*(4 - 7*r + 6*r^2) - 3*p1*(8*(2 - 5*r + 3*r^2) + 3*p2*(-2 - 7*r + 6*r^2)))) + (n13 + n21)*(log(pmax(1e-6,1 - r)) + log(-8*(-6 + (2 - 3*p1)*r + (-6 + 9*p1)*r^2) - 3*p2*(3*p1*(-3 - 5*r + 6*r^2) - 2*(3 - 5*r + 6*r^2))))
return(L)}
interlogL_rr1 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rr1,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr1 <- parallel::mcmapply(interlogL_rr1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rr1 <- -((x[,"n_03"] + x[,"n_31"])*(log10(3) + log10(2 - 3*p1))) - (x[,"n_01"] + x[,"n_33"])*(log10(3) + log10(2 - 3*p1)) - (x[,"n_12"] + x[,"n_22"])*log10(62 + 3*p1*(17 - (39*p2)/4) - (33*p2)/2) - (x[,"n_02"] + x[,"n_32"])*(-2*log10(2) + log10(2 - 3*p1) + log10(20 - 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_11"] + x[,"n_23"])*(-log10(2) + log10(52 - 3*p1*(2 - 12*p2) + 12*p2)) - (x[,"n_13"] + x[,"n_21"])*(-log10(2) + log10(-8*(-6 + (2 - 3*p1)/2 + (-6 + 9*p1)/4) - 3*(-4 - 12*p1)*p2)) + (x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p1) + log10(20 - 3*p2) + log10(pmax(1e-6,1 - r_rr1)) + log10(pmax(1e-6,r_rr1))) + (x[,"n_10"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rr1)) + 2*log10(pmax(1e-6,r_rr1))) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + 3*log10(pmax(1e-6,r_rr1))) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rr1)) + log10(1 - 2*r_rr1 + r_rr1^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_rr1)) + log10(1 - 2*r_rr1 + r_rr1^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(2 - 3*p1) + log10(8 - 3*p2 + 8*r_rr1 + 6*p2*r_rr1) + log10(1 - 2*r_rr1 + r_rr1^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(2 - 3*p1) + 2*log10(pmax(1e-6,r_rr1)) + log10(8*(2 - r_rr1) - p2*(-3 + 6*r_rr1))) + (x[,"n_12"] + x[,"n_22"])*log10(-6*p2*(3 - r_rr1 + r_rr1^2) + 8*(9 - 5*r_rr1 + 5*r_rr1^2) + 3*p1*(4*(3 + 5*r_rr1 - 5*r_rr1^2) + 3*p2*(-3 - r_rr1 + r_rr1^2))) + (x[,"n_11"] + x[,"n_23"])*(log10(pmax(1e-6,r_rr1)) + log10(16*(5 - 5*r_rr1 + 3*r_rr1^2) + 6*p2*(4 - 7*r_rr1 + 6*r_rr1^2) - 3*p1*(8*(2 - 5*r_rr1 + 3*r_rr1^2) + 3*p2*(-2 - 7*r_rr1 + 6*r_rr1^2)))) + (x[,"n_13"] + x[,"n_21"])*(log10(pmax(1e-6,1 - r_rr1)) + log10(-8*(-6 + (2 - 3*p1)*r_rr1 + (-6 + 9*p1)*r_rr1^2) - 3*p2*(3*p1*(-3 - 5*r_rr1 + 6*r_rr1^2) - 2*(3 - 5*r_rr1 + 6*r_rr1^2))))


logL_rr1 <- 5*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"])*log(3) + (x[,"n_02"] + x[,"n_32"])*(log(2 - 3*p1) + log(20 - 3*p2) + log(pmax(1e-6,1 - r_rr1)) + log(pmax(1e-6,r_rr1))) + (x[,"n_10"] + x[,"n_24"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rr1)) + 2*log(pmax(1e-6,r_rr1))) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(4 + 3*p2) + 3*log(pmax(1e-6,r_rr1))) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rr1)) + log(1 - 2*r_rr1 + r_rr1^2)) + (x[,"n_14"] + x[,"n_20"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_rr1)) + log(1 - 2*r_rr1 + r_rr1^2)) + (x[,"n_03"] + x[,"n_31"])*(log(2 - 3*p1) + log(8 - 3*p2 + 8*r_rr1 + 6*p2*r_rr1) + log(1 - 2*r_rr1 + r_rr1^2)) + (x[,"n_01"] + x[,"n_33"])*(log(2 - 3*p1) + 2*log(pmax(1e-6,r_rr1)) + log(8*(2 - r_rr1) - p2*(-3 + 6*r_rr1))) + (x[,"n_12"] + x[,"n_22"])*log(-6*p2*(3 - r_rr1 + r_rr1^2) + 8*(9 - 5*r_rr1 + 5*r_rr1^2) + 3*p1*(4*(3 + 5*r_rr1 - 5*r_rr1^2) + 3*p2*(-3 - r_rr1 + r_rr1^2))) + (x[,"n_11"] + x[,"n_23"])*(log(pmax(1e-6,r_rr1)) + log(16*(5 - 5*r_rr1 + 3*r_rr1^2) + 6*p2*(4 - 7*r_rr1 + 6*r_rr1^2) - 3*p1*(8*(2 - 5*r_rr1 + 3*r_rr1^2) + 3*p2*(-2 - 7*r_rr1 + 6*r_rr1^2)))) + (x[,"n_13"] + x[,"n_21"])*(log(pmax(1e-6,1 - r_rr1)) + log(-8*(-6 + (2 - 3*p1)*r_rr1 + (-6 + 9*p1)*r_rr1^2) - 3*p2*(3*p1*(-3 - 5*r_rr1 + 6*r_rr1^2) - 2*(3 - 5*r_rr1 + 6*r_rr1^2))))


logL_rr2 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- 4*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(2) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - n10 - 2*n11 - 2*n12 - 2*n13 - n14 - n20 - 2*n21 - 2*n22 - 2*n23 - n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34)*log(3) + (n02 + n32)*(log(2 - 3*p1) + log(10 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n24)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n00 + n34)*(log(2 - 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,r))) + (n04 + n30)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + r^2)) + (n14 + n20)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n03 + n31)*(log(2 - 3*p1) + log(1 - 2*r + r^2) + log(4*(1 + r) - p2*(-3 + 6*r))) + (n01 + n33)*(log(2 - 3*p1) + 2*log(pmax(1e-6,r)) + log(8 - 4*r + p2*(-3 + 6*r))) + (n12 + n22)*log(6*p2*(3 - r + r^2) + 4*(9 - 5*r + 5*r^2) + p1*(6*(3 + 5*r - 5*r^2) - 9*p2*(-3 - r + r^2))) + (n11 + n23)*(log(pmax(1e-6,r)) + log(8*(5 - 5*r + 3*r^2) - 6*p2*(4 - 7*r + 6*r^2) + 3*p1*(-4*(2 - 5*r + 3*r^2) + 3*p2*(-2 - 7*r + 6*r^2)))) + (n13 + n21)*(log(pmax(1e-6,1 - r)) + log(4*(6 + (-2 + 3*p1)*r + (6 - 9*p1)*r^2) + 3*p2*(3*p1*(-3 - 5*r + 6*r^2) - 2*(3 - 5*r + 6*r^2))))
return(L)}
interlogL_rr2 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rr2,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr2 <- parallel::mcmapply(interlogL_rr2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rr2 <- -((x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(3) + log10(2 - 3*p1))) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(2 - 3*p1)) - (x[,"n_11"] + x[,"n_23"])*(-log10(2) + log10(26 + 3*p1*(-1 - 12*p2) - 12*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(-2*log10(2) + log10(2 - 3*p1) + log10(10 + 3*p2)) - (x[,"n_13"] + x[,"n_21"])*(-log10(2) + log10(4*(6 + (6 - 9*p1)/4 + (-2 + 3*p1)/2) + 3*(-4 - 12*p1)*p2)) - (x[,"n_12"] + x[,"n_22"])*log10(31 + (33*p2)/2 + p1*(51/2 + (117*p2)/4)) + (x[,"n_02"] + x[,"n_32"])*(log10(2 - 3*p1) + log10(10 + 3*p2) + log10(pmax(1e-6,1 - r_rr2)) + log10(pmax(1e-6,r_rr2))) + (x[,"n_10"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rr2)) + 2*log10(pmax(1e-6,r_rr2))) + (x[,"n_00"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 3*log10(pmax(1e-6,r_rr2))) + (x[,"n_04"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rr2)) + log10(1 - 2*r_rr2 + r_rr2^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_rr2)) + log10(1 - 2*r_rr2 + r_rr2^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(2 - 3*p1) + log10(1 - 2*r_rr2 + r_rr2^2) + log10(4*(1 + r_rr2) - p2*(-3 + 6*r_rr2))) + (x[,"n_01"] + x[,"n_33"])*(log10(2 - 3*p1) + 2*log10(pmax(1e-6,r_rr2)) + log10(8 - 4*r_rr2 + p2*(-3 + 6*r_rr2))) + (x[,"n_12"] + x[,"n_22"])*log10(6*p2*(3 - r_rr2 + r_rr2^2) + 4*(9 - 5*r_rr2 + 5*r_rr2^2) + p1*(6*(3 + 5*r_rr2 - 5*r_rr2^2) - 9*p2*(-3 - r_rr2 + r_rr2^2))) + (x[,"n_11"] + x[,"n_23"])*(log10(pmax(1e-6,r_rr2)) + log10(8*(5 - 5*r_rr2 + 3*r_rr2^2) - 6*p2*(4 - 7*r_rr2 + 6*r_rr2^2) + 3*p1*(-4*(2 - 5*r_rr2 + 3*r_rr2^2) + 3*p2*(-2 - 7*r_rr2 + 6*r_rr2^2)))) + (x[,"n_13"] + x[,"n_21"])*(log10(pmax(1e-6,1 - r_rr2)) + log10(4*(6 + (-2 + 3*p1)*r_rr2 + (6 - 9*p1)*r_rr2^2) + 3*p2*(3*p1*(-3 - 5*r_rr2 + 6*r_rr2^2) - 2*(3 - 5*r_rr2 + 6*r_rr2^2))))


logL_rr2 <- 4*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"])*log(3) + (x[,"n_02"] + x[,"n_32"])*(log(2 - 3*p1) + log(10 + 3*p2) + log(pmax(1e-6,1 - r_rr2)) + log(pmax(1e-6,r_rr2))) + (x[,"n_10"] + x[,"n_24"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rr2)) + 2*log(pmax(1e-6,r_rr2))) + (x[,"n_00"] + x[,"n_34"])*(log(2 - 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,r_rr2))) + (x[,"n_04"] + x[,"n_30"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rr2)) + log(1 - 2*r_rr2 + r_rr2^2)) + (x[,"n_14"] + x[,"n_20"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_rr2)) + log(1 - 2*r_rr2 + r_rr2^2)) + (x[,"n_03"] + x[,"n_31"])*(log(2 - 3*p1) + log(1 - 2*r_rr2 + r_rr2^2) + log(4*(1 + r_rr2) - p2*(-3 + 6*r_rr2))) + (x[,"n_01"] + x[,"n_33"])*(log(2 - 3*p1) + 2*log(pmax(1e-6,r_rr2)) + log(8 - 4*r_rr2 + p2*(-3 + 6*r_rr2))) + (x[,"n_12"] + x[,"n_22"])*log(6*p2*(3 - r_rr2 + r_rr2^2) + 4*(9 - 5*r_rr2 + 5*r_rr2^2) + p1*(6*(3 + 5*r_rr2 - 5*r_rr2^2) - 9*p2*(-3 - r_rr2 + r_rr2^2))) + (x[,"n_11"] + x[,"n_23"])*(log(pmax(1e-6,r_rr2)) + log(8*(5 - 5*r_rr2 + 3*r_rr2^2) - 6*p2*(4 - 7*r_rr2 + 6*r_rr2^2) + 3*p1*(-4*(2 - 5*r_rr2 + 3*r_rr2^2) + 3*p2*(-2 - 7*r_rr2 + 6*r_rr2^2)))) + (x[,"n_13"] + x[,"n_21"])*(log(pmax(1e-6,1 - r_rr2)) + log(4*(6 + (-2 + 3*p1)*r_rr2 + (6 - 9*p1)*r_rr2^2) + 3*p2*(3*p1*(-3 - 5*r_rr2 + 6*r_rr2^2) - 2*(3 - 5*r_rr2 + 6*r_rr2^2))))


logL_rr3 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- 6*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(2) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - n10 - 2*n11 - 2*n12 - 2*n13 - n14 - n20 - 2*n21 - 2*n22 - 2*n23 - n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34)*log(3) + (n02 + n32)*(log(4 + 3*p1) + log(20 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n24)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n00 + n34)*(log(4 + 3*p1) + log(4 + 3*p2) + 3*log(pmax(1e-6,r))) + (n04 + n30)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + r^2)) + (n14 + n20)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n03 + n31)*(log(4 + 3*p1) + log(8 - 3*p2 + 8*r + 6*p2*r) + log(1 - 2*r + r^2)) + (n01 + n33)*(log(4 + 3*p1) + 2*log(pmax(1e-6,r)) + log(8*(2 - r) - p2*(-3 + 6*r))) + (n12 + n22)*log(p1*(-9*p2*(-3 - r + r^2) + 12*(-3 - 5*r + 5*r^2)) - 4*(3*p2*(3 - r + r^2) - 4*(9 - 5*r + 5*r^2))) + (n11 + n23)*(log(pmax(1e-6,r)) + log(3*p1*(8*(2 - 5*r + 3*r^2) + 3*p2*(-2 - 7*r + 6*r^2)) + 4*(8*(5 - 5*r + 3*r^2) + 3*p2*(4 - 7*r + 6*r^2)))) + (n13 + n21)*(log(pmax(1e-6,1 - r)) + log(8*(12 - (4 + 3*p1)*r + 3*(4 + 3*p1)*r^2) + 3*p2*(3*p1*(-3 - 5*r + 6*r^2) + 4*(3 - 5*r + 6*r^2))))
return(L)}
interlogL_rr3 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rr3,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr3 <- parallel::mcmapply(interlogL_rr3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rr3 <- -((x[,"n_03"] + x[,"n_31"])*(log10(3) + log10(4 + 3*p1))) - (x[,"n_01"] + x[,"n_33"])*(log10(3) + log10(4 + 3*p1)) - (x[,"n_02"] + x[,"n_32"])*(-2*log10(2) + log10(4 + 3*p1) + log10(20 - 3*p2)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_13"] + x[,"n_21"])*(-log10(2) + log10(8*(12 + (-4 - 3*p1)/2 + (3*(4 + 3*p1))/4) + 3*(8 - 12*p1)*p2)) - (x[,"n_11"] + x[,"n_23"])*(-log10(2) + log10(3*p1*(2 - 12*p2) + 4*(26 + 6*p2))) - (x[,"n_12"] + x[,"n_22"])*log10(-4*(-31 + (33*p2)/4) + p1*(-51 + (117*p2)/4)) + (x[,"n_02"] + x[,"n_32"])*(log10(4 + 3*p1) + log10(20 - 3*p2) + log10(pmax(1e-6,1 - r_rr3)) + log10(pmax(1e-6,r_rr3))) + (x[,"n_10"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rr3)) + 2*log10(pmax(1e-6,r_rr3))) + (x[,"n_00"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + 3*log10(pmax(1e-6,r_rr3))) + (x[,"n_04"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_rr3)) + log10(1 - 2*r_rr3 + r_rr3^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,r_rr3)) + log10(1 - 2*r_rr3 + r_rr3^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(4 + 3*p1) + log10(8 - 3*p2 + 8*r_rr3 + 6*p2*r_rr3) + log10(1 - 2*r_rr3 + r_rr3^2)) + (x[,"n_01"] + x[,"n_33"])*(log10(4 + 3*p1) + 2*log10(pmax(1e-6,r_rr3)) + log10(8*(2 - r_rr3) - p2*(-3 + 6*r_rr3))) + (x[,"n_12"] + x[,"n_22"])*log10(p1*(-9*p2*(-3 - r_rr3 + r_rr3^2) + 12*(-3 - 5*r_rr3 + 5*r_rr3^2)) - 4*(3*p2*(3 - r_rr3 + r_rr3^2) - 4*(9 - 5*r_rr3 + 5*r_rr3^2))) + (x[,"n_11"] + x[,"n_23"])*(log10(pmax(1e-6,r_rr3)) + log10(3*p1*(8*(2 - 5*r_rr3 + 3*r_rr3^2) + 3*p2*(-2 - 7*r_rr3 + 6*r_rr3^2)) + 4*(8*(5 - 5*r_rr3 + 3*r_rr3^2) + 3*p2*(4 - 7*r_rr3 + 6*r_rr3^2)))) + (x[,"n_13"] + x[,"n_21"])*(log10(pmax(1e-6,1 - r_rr3)) + log10(8*(12 - (4 + 3*p1)*r_rr3 + 3*(4 + 3*p1)*r_rr3^2) + 3*p2*(3*p1*(-3 - 5*r_rr3 + 6*r_rr3^2) + 4*(3 - 5*r_rr3 + 6*r_rr3^2))))


logL_rr3 <- 6*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"])*log(3) + (x[,"n_02"] + x[,"n_32"])*(log(4 + 3*p1) + log(20 - 3*p2) + log(pmax(1e-6,1 - r_rr3)) + log(pmax(1e-6,r_rr3))) + (x[,"n_10"] + x[,"n_24"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rr3)) + 2*log(pmax(1e-6,r_rr3))) + (x[,"n_00"] + x[,"n_34"])*(log(4 + 3*p1) + log(4 + 3*p2) + 3*log(pmax(1e-6,r_rr3))) + (x[,"n_04"] + x[,"n_30"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_rr3)) + log(1 - 2*r_rr3 + r_rr3^2)) + (x[,"n_14"] + x[,"n_20"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,r_rr3)) + log(1 - 2*r_rr3 + r_rr3^2)) + (x[,"n_03"] + x[,"n_31"])*(log(4 + 3*p1) + log(8 - 3*p2 + 8*r_rr3 + 6*p2*r_rr3) + log(1 - 2*r_rr3 + r_rr3^2)) + (x[,"n_01"] + x[,"n_33"])*(log(4 + 3*p1) + 2*log(pmax(1e-6,r_rr3)) + log(8*(2 - r_rr3) - p2*(-3 + 6*r_rr3))) + (x[,"n_12"] + x[,"n_22"])*log(p1*(-9*p2*(-3 - r_rr3 + r_rr3^2) + 12*(-3 - 5*r_rr3 + 5*r_rr3^2)) - 4*(3*p2*(3 - r_rr3 + r_rr3^2) - 4*(9 - 5*r_rr3 + 5*r_rr3^2))) + (x[,"n_11"] + x[,"n_23"])*(log(pmax(1e-6,r_rr3)) + log(3*p1*(8*(2 - 5*r_rr3 + 3*r_rr3^2) + 3*p2*(-2 - 7*r_rr3 + 6*r_rr3^2)) + 4*(8*(5 - 5*r_rr3 + 3*r_rr3^2) + 3*p2*(4 - 7*r_rr3 + 6*r_rr3^2)))) + (x[,"n_13"] + x[,"n_21"])*(log(pmax(1e-6,1 - r_rr3)) + log(8*(12 - (4 + 3*p1)*r_rr3 + 3*(4 + 3*p1)*r_rr3^2) + 3*p2*(3*p1*(-3 - 5*r_rr3 + 6*r_rr3^2) + 4*(3 - 5*r_rr3 + 6*r_rr3^2))))


logL_rr4 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
L <- 5*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34)*log(2) + (-2*n00 - 2*n01 - 2*n02 - 2*n03 - 2*n04 - n10 - 2*n11 - 2*n12 - 2*n13 - n14 - n20 - 2*n21 - 2*n22 - 2*n23 - n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34)*log(3) + (n02 + n32)*(log(4 + 3*p1) + log(10 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n10 + n24)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n00 + n34)*(log(4 + 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,r))) + (n04 + n30)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(1 - 2*r + r^2)) + (n14 + n20)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r)) + log(1 - 2*r + r^2)) + (n03 + n31)*(log(4 + 3*p1) + log(1 - 2*r + r^2) + log(4*(1 + r) - p2*(-3 + 6*r))) + (n01 + n33)*(log(4 + 3*p1) + 2*log(pmax(1e-6,r)) + log(8 - 4*r + p2*(-3 + 6*r))) + (n12 + n22)*log(3*p1*(3*p2*(-3 - r + r^2) + 2*(-3 - 5*r + 5*r^2)) + 4*(3*p2*(3 - r + r^2) + 2*(9 - 5*r + 5*r^2))) + (n11 + n23)*(log(pmax(1e-6,r)) + log(16*(5 - 5*r + 3*r^2) - 12*p2*(4 - 7*r + 6*r^2) + p1*(12*(2 - 5*r + 3*r^2) - 9*p2*(-2 - 7*r + 6*r^2)))) + (n13 + n21)*(log(pmax(1e-6,1 - r)) + log(4*(12 - (4 + 3*p1)*r + 3*(4 + 3*p1)*r^2) - 3*p2*(3*p1*(-3 - 5*r + 6*r^2) + 4*(3 - 5*r + 6*r^2))))
return(L)}
interlogL_rr4 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2) {
optimize(logL_rr4,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_rr4 <- parallel::mcmapply(interlogL_rr4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"], p1, p2, mc.cores = ncores)


LOD_rr4 <- -((x[,"n_03"] + x[,"n_31"])*(-log10(2) + log10(3) + log10(4 + 3*p1))) - (x[,"n_01"] + x[,"n_33"])*(-log10(2) + log10(3) + log10(4 + 3*p1)) - (x[,"n_14"] + x[,"n_20"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_24"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_04"] + x[,"n_30"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_34"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_32"])*(-2*log10(2) + log10(4 + 3*p1) + log10(10 + 3*p2)) - (x[,"n_13"] + x[,"n_21"])*(-log10(2) + log10(4*(12 + (-4 - 3*p1)/2 + (3*(4 + 3*p1))/4) - 3*(8 - 12*p1)*p2)) - (x[,"n_12"] + x[,"n_22"])*log10(3*p1*(-17/2 - (39*p2)/4) + 4*(31/2 + (33*p2)/4)) - (x[,"n_11"] + x[,"n_23"])*(-log10(2) + log10(52 - 24*p2 + p1*(3 + 36*p2))) + (x[,"n_02"] + x[,"n_32"])*(log10(4 + 3*p1) + log10(10 + 3*p2) + log10(pmax(1e-6,1 - r_rr4)) + log10(pmax(1e-6,r_rr4))) + (x[,"n_10"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rr4)) + 2*log10(pmax(1e-6,r_rr4))) + (x[,"n_00"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + 3*log10(pmax(1e-6,r_rr4))) + (x[,"n_04"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_rr4)) + log10(1 - 2*r_rr4 + r_rr4^2)) + (x[,"n_14"] + x[,"n_20"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,r_rr4)) + log10(1 - 2*r_rr4 + r_rr4^2)) + (x[,"n_03"] + x[,"n_31"])*(log10(4 + 3*p1) + log10(1 - 2*r_rr4 + r_rr4^2) + log10(4*(1 + r_rr4) - p2*(-3 + 6*r_rr4))) + (x[,"n_01"] + x[,"n_33"])*(log10(4 + 3*p1) + 2*log10(pmax(1e-6,r_rr4)) + log10(8 - 4*r_rr4 + p2*(-3 + 6*r_rr4))) + (x[,"n_12"] + x[,"n_22"])*log10(3*p1*(3*p2*(-3 - r_rr4 + r_rr4^2) + 2*(-3 - 5*r_rr4 + 5*r_rr4^2)) + 4*(3*p2*(3 - r_rr4 + r_rr4^2) + 2*(9 - 5*r_rr4 + 5*r_rr4^2))) + (x[,"n_11"] + x[,"n_23"])*(log10(pmax(1e-6,r_rr4)) + log10(16*(5 - 5*r_rr4 + 3*r_rr4^2) - 12*p2*(4 - 7*r_rr4 + 6*r_rr4^2) + p1*(12*(2 - 5*r_rr4 + 3*r_rr4^2) - 9*p2*(-2 - 7*r_rr4 + 6*r_rr4^2)))) + (x[,"n_13"] + x[,"n_21"])*(log10(pmax(1e-6,1 - r_rr4)) + log10(4*(12 - (4 + 3*p1)*r_rr4 + 3*(4 + 3*p1)*r_rr4^2) - 3*p2*(3*p1*(-3 - 5*r_rr4 + 6*r_rr4^2) + 4*(3 - 5*r_rr4 + 6*r_rr4^2))))


logL_rr4 <- 5*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - 2*x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"])*log(3) + (x[,"n_02"] + x[,"n_32"])*(log(4 + 3*p1) + log(10 + 3*p2) + log(pmax(1e-6,1 - r_rr4)) + log(pmax(1e-6,r_rr4))) + (x[,"n_10"] + x[,"n_24"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rr4)) + 2*log(pmax(1e-6,r_rr4))) + (x[,"n_00"] + x[,"n_34"])*(log(4 + 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,r_rr4))) + (x[,"n_04"] + x[,"n_30"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_rr4)) + log(1 - 2*r_rr4 + r_rr4^2)) + (x[,"n_14"] + x[,"n_20"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,r_rr4)) + log(1 - 2*r_rr4 + r_rr4^2)) + (x[,"n_03"] + x[,"n_31"])*(log(4 + 3*p1) + log(1 - 2*r_rr4 + r_rr4^2) + log(4*(1 + r_rr4) - p2*(-3 + 6*r_rr4))) + (x[,"n_01"] + x[,"n_33"])*(log(4 + 3*p1) + 2*log(pmax(1e-6,r_rr4)) + log(8 - 4*r_rr4 + p2*(-3 + 6*r_rr4))) + (x[,"n_12"] + x[,"n_22"])*log(3*p1*(3*p2*(-3 - r_rr4 + r_rr4^2) + 2*(-3 - 5*r_rr4 + 5*r_rr4^2)) + 4*(3*p2*(3 - r_rr4 + r_rr4^2) + 2*(9 - 5*r_rr4 + 5*r_rr4^2))) + (x[,"n_11"] + x[,"n_23"])*(log(pmax(1e-6,r_rr4)) + log(16*(5 - 5*r_rr4 + 3*r_rr4^2) - 12*p2*(4 - 7*r_rr4 + 6*r_rr4^2) + p1*(12*(2 - 5*r_rr4 + 3*r_rr4^2) - 9*p2*(-2 - 7*r_rr4 + 6*r_rr4^2)))) + (x[,"n_13"] + x[,"n_21"])*(log(pmax(1e-6,1 - r_rr4)) + log(4*(12 - (4 + 3*p1)*r_rr4 + 3*(4 + 3*p1)*r_rr4^2) - 3*p2*(3*p1*(-3 - 5*r_rr4 + 6*r_rr4^2) + 4*(3 - 5*r_rr4 + 6*r_rr4^2))))


return(list(
r_mat = cbind(r_cc1,r_cc2,r_cc3,r_cc4,r_cr1,r_cr2,r_cr3,r_cr4,r_mc1,r_mc2,r_mc3,r_mc4,r_mr1,r_mr2,r_mr3,r_mr4,r_rc1,r_rc2,r_rc3,r_rc4,r_rr1,r_rr2,r_rr3,r_rr4,0.499),
LOD_mat = cbind(LOD_cc1,LOD_cc2,LOD_cc3,LOD_cc4,LOD_cr1,LOD_cr2,LOD_cr3,LOD_cr4,LOD_mc1,LOD_mc2,LOD_mc3,LOD_mc4,LOD_mr1,LOD_mr2,LOD_mr3,LOD_mr4,LOD_rc1,LOD_rc2,LOD_rc3,LOD_rc4,LOD_rr1,LOD_rr2,LOD_rr3,LOD_rr4,0),
logL_mat = cbind(logL_cc1,logL_cc2,logL_cc3,logL_cc4,logL_cr1,logL_cr2,logL_cr3,logL_cr4,logL_mc1,logL_mc2,logL_mc3,logL_mc4,logL_mr1,logL_mr2,logL_mr3,logL_mr4,logL_rc1,logL_rc2,logL_rc3,logL_rc4,logL_rr1,logL_rr2,logL_rr3,logL_rr4,-1e6),
phasing_strategy = "MLL",
possible_phases = c("coupling coupling","coupling coupling","coupling coupling","coupling coupling","coupling repulsion","coupling repulsion","coupling repulsion","coupling repulsion","mixed coupling","mixed coupling","mixed coupling","mixed coupling","mixed repulsion","mixed repulsion","mixed repulsion","mixed repulsion","repulsion coupling","repulsion coupling","repulsion coupling","repulsion coupling","repulsion repulsion","repulsion repulsion","repulsion repulsion","repulsion repulsion","unknown")
)
)
}

#' @rdname p4_functions
p4_2.2_2.2 <- function(x,p1,p2,ncores=1){


logL_cc1 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
L <- (-4*n00 - 2*n01 - 3*n02 - 2*n03 - 4*n04 - 2*n10 - 2*n11 - 2*n12 - 2*n13 - 2*n14 - 3*n20 - 2*n21 - 3*n22 - 2*n23 - 3*n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34 - 4*n40 - 2*n41 - 3*n42 - 2*n43 - 4*n44)*log(2) + (-2*n00 - 2*n01 - n02 - 2*n03 - 2*n04 - 2*n10 - 2*n11 - 2*n12 - 2*n13 - 2*n14 - n20 - 2*n21 - 2*n22 - 2*n23 - n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34 - 2*n40 - 2*n41 - n42 - 2*n43 - 2*n44)*log(3) + (n01 + n10 + n34 + n43)*(log(2 - 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n02 + n20 + n24 + n42)*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n03 + n14 + n30 + n41)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 3*log(pmax(1e-6,r))) + (n04 + n40)*(log(2 - 3*p1) + log(2 - 3*p2) + 4*log(pmax(1e-6,r))) + (n00 + n44)*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(1 - 2*r + r^2)) + n22*log((-2 + 3*p1)*(-2 + 3*p2)*(-1 + r)^4 + 8*(-2 + 3*p1)*(-2 + 3*p2)*(-1 + r)^2*r^2 + (-2 + 3*p1)*(-2 + 3*p2)*r^4 + 2*(p1*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2))*(p2*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2))) + (n13 + n31)*(2*log(pmax(1e-6,r)) + log(8*(2 - 3*r + 2*r^2) - 3*p2*(5 - 12*r + 8*r^2) + 3*p1*(-5 + 12*r - 8*r^2 + 3*p2*(1 - 6*r + 4*r^2)))) + (n11 + n33)*(2*log(pmax(1e-6,1 - r)) + log(8 - 8*r + 16*r^2 - 3*p2*(1 - 4*r + 8*r^2) + 3*p1*(-1 + 4*r - 8*r^2 + 3*p2*(-1 - 2*r + 4*r^2)))) + (n12 + n21 + n23 + n32)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(3*p1*(-4*(1 - 3*r + 3*r^2) + 3*p2*(-1 - 6*r + 6*r^2)) - 4*(-5 + 6*r - 6*r^2 + p2*(3 - 9*r + 9*r^2))))
return(L)}
interlogL_cc1 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
optimize(logL_cc1,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc1 <- parallel::mcmapply(interlogL_cc1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"], p1, p2, mc.cores = ncores)


LOD_cc1 <- -((x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(-2*log10(2) + log10(3*p1*(-1 - (15*p2)/2) - 4*(-7/2 + (3*p2)/4)))) - (x[,"n_04"] + x[,"n_40"])*(-4*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(-4*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(-4*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(-4*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_44"])*(-4*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_13"] + x[,"n_31"])*(-2*log10(2) + log10(8 + 3*p1*(-1 - 3*p2) - 3*p2)) - (x[,"n_11"] + x[,"n_33"])*(-2*log10(2) + log10(8 + 3*p1*(-1 - 3*p2) - 3*p2)) - x[,"n_22"]*log10((5*(-2 + 3*p1)*(-2 + 3*p2))/8 + 2*(3 + (9*p1)/2)*(3 + (9*p2)/2)) + (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 3*log10(pmax(1e-6,1 - r_cc1)) + log10(pmax(1e-6,r_cc1))) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,1 - r_cc1)) + 2*log10(pmax(1e-6,r_cc1))) + (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cc1)) + 3*log10(pmax(1e-6,r_cc1))) + (x[,"n_04"] + x[,"n_40"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 4*log10(pmax(1e-6,r_cc1))) + (x[,"n_00"] + x[,"n_44"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 2*log10(1 - 2*r_cc1 + r_cc1^2)) + x[,"n_22"]*log10((-2 + 3*p1)*(-2 + 3*p2)*(-1 + r_cc1)^4 + 8*(-2 + 3*p1)*(-2 + 3*p2)*(-1 + r_cc1)^2*r_cc1^2 + (-2 + 3*p1)*(-2 + 3*p2)*r_cc1^4 + 2*(p1*(3 + 6*r_cc1 - 6*r_cc1^2) + 4*(1 - r_cc1 + r_cc1^2))*(p2*(3 + 6*r_cc1 - 6*r_cc1^2) + 4*(1 - r_cc1 + r_cc1^2))) + (x[,"n_13"] + x[,"n_31"])*(2*log10(pmax(1e-6,r_cc1)) + log10(8*(2 - 3*r_cc1 + 2*r_cc1^2) - 3*p2*(5 - 12*r_cc1 + 8*r_cc1^2) + 3*p1*(-5 + 12*r_cc1 - 8*r_cc1^2 + 3*p2*(1 - 6*r_cc1 + 4*r_cc1^2)))) + (x[,"n_11"] + x[,"n_33"])*(2*log10(pmax(1e-6,1 - r_cc1)) + log10(8 - 8*r_cc1 + 16*r_cc1^2 - 3*p2*(1 - 4*r_cc1 + 8*r_cc1^2) + 3*p1*(-1 + 4*r_cc1 - 8*r_cc1^2 + 3*p2*(-1 - 2*r_cc1 + 4*r_cc1^2)))) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(log10(pmax(1e-6,1 - r_cc1)) + log10(pmax(1e-6,r_cc1)) + log10(3*p1*(-4*(1 - 3*r_cc1 + 3*r_cc1^2) + 3*p2*(-1 - 6*r_cc1 + 6*r_cc1^2)) - 4*(-5 + 6*r_cc1 - 6*r_cc1^2 + p2*(3 - 9*r_cc1 + 9*r_cc1^2))))


logL_cc1 <- (-4*x[,"n_00"] - 2*x[,"n_01"] - 3*x[,"n_02"] - 2*x[,"n_03"] - 4*x[,"n_04"] - 2*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - 3*x[,"n_20"] - 2*x[,"n_21"] - 3*x[,"n_22"] - 2*x[,"n_23"] - 3*x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - 4*x[,"n_40"] - 2*x[,"n_41"] - 3*x[,"n_42"] - 2*x[,"n_43"] - 4*x[,"n_44"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_40"] - 2*x[,"n_41"] - x[,"n_42"] - 2*x[,"n_43"] - 2*x[,"n_44"])*log(3) + (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(log(2 - 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,1 - r_cc1)) + log(pmax(1e-6,r_cc1))) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r_cc1)) + 2*log(pmax(1e-6,r_cc1))) + (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cc1)) + 3*log(pmax(1e-6,r_cc1))) + (x[,"n_04"] + x[,"n_40"])*(log(2 - 3*p1) + log(2 - 3*p2) + 4*log(pmax(1e-6,r_cc1))) + (x[,"n_00"] + x[,"n_44"])*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(1 - 2*r_cc1 + r_cc1^2)) + x[,"n_22"]*log((-2 + 3*p1)*(-2 + 3*p2)*(-1 + r_cc1)^4 + 8*(-2 + 3*p1)*(-2 + 3*p2)*(-1 + r_cc1)^2*r_cc1^2 + (-2 + 3*p1)*(-2 + 3*p2)*r_cc1^4 + 2*(p1*(3 + 6*r_cc1 - 6*r_cc1^2) + 4*(1 - r_cc1 + r_cc1^2))*(p2*(3 + 6*r_cc1 - 6*r_cc1^2) + 4*(1 - r_cc1 + r_cc1^2))) + (x[,"n_13"] + x[,"n_31"])*(2*log(pmax(1e-6,r_cc1)) + log(8*(2 - 3*r_cc1 + 2*r_cc1^2) - 3*p2*(5 - 12*r_cc1 + 8*r_cc1^2) + 3*p1*(-5 + 12*r_cc1 - 8*r_cc1^2 + 3*p2*(1 - 6*r_cc1 + 4*r_cc1^2)))) + (x[,"n_11"] + x[,"n_33"])*(2*log(pmax(1e-6,1 - r_cc1)) + log(8 - 8*r_cc1 + 16*r_cc1^2 - 3*p2*(1 - 4*r_cc1 + 8*r_cc1^2) + 3*p1*(-1 + 4*r_cc1 - 8*r_cc1^2 + 3*p2*(-1 - 2*r_cc1 + 4*r_cc1^2)))) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(log(pmax(1e-6,1 - r_cc1)) + log(pmax(1e-6,r_cc1)) + log(3*p1*(-4*(1 - 3*r_cc1 + 3*r_cc1^2) + 3*p2*(-1 - 6*r_cc1 + 6*r_cc1^2)) - 4*(-5 + 6*r_cc1 - 6*r_cc1^2 + p2*(3 - 9*r_cc1 + 9*r_cc1^2))))


logL_cc2 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
L <- (-5*n00 - 3*n01 - 4*n02 - 3*n03 - 5*n04 - 3*n10 - 3*n11 - 3*n12 - 3*n13 - 3*n14 - 4*n20 - 3*n21 - 4*n22 - 3*n23 - 4*n24 - 3*n30 - 3*n31 - 3*n32 - 3*n33 - 3*n34 - 5*n40 - 3*n41 - 4*n42 - 3*n43 - 5*n44)*log(2) + (-2*n00 - 2*n01 - n02 - 2*n03 - 2*n04 - 2*n10 - 2*n11 - 2*n12 - 2*n13 - 2*n14 - n20 - 2*n21 - 2*n22 - 2*n23 - n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34 - 2*n40 - 2*n41 - n42 - 2*n43 - 2*n44)*log(3) + (n00 + n44)*(log(2 - 3*p1) + log(4 + 3*p2) + 4*log(pmax(1e-6,1 - r))) + (n01 + n10 + n34 + n43)*(log(2 - 3*p1) + log(4 + 3*p2) + 3*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n02 + n20 + n24 + n42)*(log(2 - 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n03 + n14 + n30 + n41)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + 3*log(pmax(1e-6,r))) + (n04 + n40)*(log(2 - 3*p1) + log(4 + 3*p2) + 4*log(pmax(1e-6,r))) + (n11 + n33)*(2*log(pmax(1e-6,1 - r)) + log(16*(1 - r + 2*r^2) + 3*p2*(1 - 4*r + 8*r^2) - 3*p1*(2 - 8*r + 16*r^2 + 3*p2*(-1 - 2*r + 4*r^2)))) + n22*log((2 - 3*p1)*(4 + 3*p2)*(-1 + r)^4 - 8*(-2 + 3*p1)*(4 + 3*p2)*(-1 + r)^2*r^2 - (-2 + 3*p1)*(4 + 3*p2)*r^4 + 2*(p1*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2))*(8*(1 - r + r^2) + p2*(-3 - 6*r + 6*r^2))) + (n13 + n31)*(2*log(pmax(1e-6,r)) + log(16*(2 - 3*r + 2*r^2) + 3*p2*(5 - 12*r + 8*r^2) - 3*p1*(3*p2*(1 - 6*r + 4*r^2) + 2*(5 - 12*r + 8*r^2)))) + (n12 + n21 + n23 + n32)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(-3*p1*(8*(1 - 3*r + 3*r^2) + 3*p2*(-1 - 6*r + 6*r^2)) + 4*(2*(5 - 6*r + 6*r^2) + p2*(3 - 9*r + 9*r^2))))
return(L)}
interlogL_cc2 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
optimize(logL_cc2,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc2 <- parallel::mcmapply(interlogL_cc2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"], p1, p2, mc.cores = ncores)


LOD_cc2 <- -((x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(-2*log10(2) + log10(-3*p1*(2 - (15*p2)/2) + 4*(7 + (3*p2)/4)))) - (x[,"n_04"] + x[,"n_40"])*(-4*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(-4*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(-4*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(-4*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_44"])*(-4*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_13"] + x[,"n_31"])*(-2*log10(2) + log10(16 - 3*p1*(2 - 3*p2) + 3*p2)) - (x[,"n_11"] + x[,"n_33"])*(-2*log10(2) + log10(16 - 3*p1*(2 - 3*p2) + 3*p2)) - x[,"n_22"]*log10(2*(3 + (9*p1)/2)*(6 - (9*p2)/2) + ((2 - 3*p1)*(4 + 3*p2))/16 - (9*(-2 + 3*p1)*(4 + 3*p2))/16) + (x[,"n_00"] + x[,"n_44"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + 4*log10(pmax(1e-6,1 - r_cc2))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + 3*log10(pmax(1e-6,1 - r_cc2)) + log10(pmax(1e-6,r_cc2))) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + 2*log10(pmax(1e-6,1 - r_cc2)) + 2*log10(pmax(1e-6,r_cc2))) + (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cc2)) + 3*log10(pmax(1e-6,r_cc2))) + (x[,"n_04"] + x[,"n_40"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + 4*log10(pmax(1e-6,r_cc2))) + (x[,"n_11"] + x[,"n_33"])*(2*log10(pmax(1e-6,1 - r_cc2)) + log10(16*(1 - r_cc2 + 2*r_cc2^2) + 3*p2*(1 - 4*r_cc2 + 8*r_cc2^2) - 3*p1*(2 - 8*r_cc2 + 16*r_cc2^2 + 3*p2*(-1 - 2*r_cc2 + 4*r_cc2^2)))) + x[,"n_22"]*log10((2 - 3*p1)*(4 + 3*p2)*(-1 + r_cc2)^4 - 8*(-2 + 3*p1)*(4 + 3*p2)*(-1 + r_cc2)^2*r_cc2^2 - (-2 + 3*p1)*(4 + 3*p2)*r_cc2^4 + 2*(p1*(3 + 6*r_cc2 - 6*r_cc2^2) + 4*(1 - r_cc2 + r_cc2^2))*(8*(1 - r_cc2 + r_cc2^2) + p2*(-3 - 6*r_cc2 + 6*r_cc2^2))) + (x[,"n_13"] + x[,"n_31"])*(2*log10(pmax(1e-6,r_cc2)) + log10(16*(2 - 3*r_cc2 + 2*r_cc2^2) + 3*p2*(5 - 12*r_cc2 + 8*r_cc2^2) - 3*p1*(3*p2*(1 - 6*r_cc2 + 4*r_cc2^2) + 2*(5 - 12*r_cc2 + 8*r_cc2^2)))) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(log10(pmax(1e-6,1 - r_cc2)) + log10(pmax(1e-6,r_cc2)) + log10(-3*p1*(8*(1 - 3*r_cc2 + 3*r_cc2^2) + 3*p2*(-1 - 6*r_cc2 + 6*r_cc2^2)) + 4*(2*(5 - 6*r_cc2 + 6*r_cc2^2) + p2*(3 - 9*r_cc2 + 9*r_cc2^2))))


logL_cc2 <- (-5*x[,"n_00"] - 3*x[,"n_01"] - 4*x[,"n_02"] - 3*x[,"n_03"] - 5*x[,"n_04"] - 3*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 4*x[,"n_20"] - 3*x[,"n_21"] - 4*x[,"n_22"] - 3*x[,"n_23"] - 4*x[,"n_24"] - 3*x[,"n_30"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - 5*x[,"n_40"] - 3*x[,"n_41"] - 4*x[,"n_42"] - 3*x[,"n_43"] - 5*x[,"n_44"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_40"] - 2*x[,"n_41"] - x[,"n_42"] - 2*x[,"n_43"] - 2*x[,"n_44"])*log(3) + (x[,"n_00"] + x[,"n_44"])*(log(2 - 3*p1) + log(4 + 3*p2) + 4*log(pmax(1e-6,1 - r_cc2))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(log(2 - 3*p1) + log(4 + 3*p2) + 3*log(pmax(1e-6,1 - r_cc2)) + log(pmax(1e-6,r_cc2))) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(log(2 - 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,1 - r_cc2)) + 2*log(pmax(1e-6,r_cc2))) + (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cc2)) + 3*log(pmax(1e-6,r_cc2))) + (x[,"n_04"] + x[,"n_40"])*(log(2 - 3*p1) + log(4 + 3*p2) + 4*log(pmax(1e-6,r_cc2))) + (x[,"n_11"] + x[,"n_33"])*(2*log(pmax(1e-6,1 - r_cc2)) + log(16*(1 - r_cc2 + 2*r_cc2^2) + 3*p2*(1 - 4*r_cc2 + 8*r_cc2^2) - 3*p1*(2 - 8*r_cc2 + 16*r_cc2^2 + 3*p2*(-1 - 2*r_cc2 + 4*r_cc2^2)))) + x[,"n_22"]*log((2 - 3*p1)*(4 + 3*p2)*(-1 + r_cc2)^4 - 8*(-2 + 3*p1)*(4 + 3*p2)*(-1 + r_cc2)^2*r_cc2^2 - (-2 + 3*p1)*(4 + 3*p2)*r_cc2^4 + 2*(p1*(3 + 6*r_cc2 - 6*r_cc2^2) + 4*(1 - r_cc2 + r_cc2^2))*(8*(1 - r_cc2 + r_cc2^2) + p2*(-3 - 6*r_cc2 + 6*r_cc2^2))) + (x[,"n_13"] + x[,"n_31"])*(2*log(pmax(1e-6,r_cc2)) + log(16*(2 - 3*r_cc2 + 2*r_cc2^2) + 3*p2*(5 - 12*r_cc2 + 8*r_cc2^2) - 3*p1*(3*p2*(1 - 6*r_cc2 + 4*r_cc2^2) + 2*(5 - 12*r_cc2 + 8*r_cc2^2)))) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(log(pmax(1e-6,1 - r_cc2)) + log(pmax(1e-6,r_cc2)) + log(-3*p1*(8*(1 - 3*r_cc2 + 3*r_cc2^2) + 3*p2*(-1 - 6*r_cc2 + 6*r_cc2^2)) + 4*(2*(5 - 6*r_cc2 + 6*r_cc2^2) + p2*(3 - 9*r_cc2 + 9*r_cc2^2))))


logL_cc3 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
L <- (-5*n00 - 3*n01 - 4*n02 - 3*n03 - 5*n04 - 3*n10 - 3*n11 - 3*n12 - 3*n13 - 3*n14 - 4*n20 - 3*n21 - 4*n22 - 3*n23 - 4*n24 - 3*n30 - 3*n31 - 3*n32 - 3*n33 - 3*n34 - 5*n40 - 3*n41 - 4*n42 - 3*n43 - 5*n44)*log(2) + (-2*n00 - 2*n01 - n02 - 2*n03 - 2*n04 - 2*n10 - 2*n11 - 2*n12 - 2*n13 - 2*n14 - n20 - 2*n21 - 2*n22 - 2*n23 - n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34 - 2*n40 - 2*n41 - n42 - 2*n43 - 2*n44)*log(3) + (n00 + n44)*(log(4 + 3*p1) + log(2 - 3*p2) + 4*log(pmax(1e-6,1 - r))) + (n01 + n10 + n34 + n43)*(log(4 + 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n02 + n20 + n24 + n42)*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n03 + n14 + n30 + n41)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 3*log(pmax(1e-6,r))) + (n04 + n40)*(log(4 + 3*p1) + log(2 - 3*p2) + 4*log(pmax(1e-6,r))) + (n13 + n31)*(2*log(pmax(1e-6,r)) + log(16*(2 - 3*r + 2*r^2) - 6*p2*(5 - 12*r + 8*r^2) - 3*p1*(-5 + 12*r - 8*r^2 + 3*p2*(1 - 6*r + 4*r^2)))) + n22*log((-4 - 3*p1)*(-2 + 3*p2)*(-1 + r)^4 - 8*(4 + 3*p1)*(-2 + 3*p2)*(-1 + r)^2*r^2 - (4 + 3*p1)*(-2 + 3*p2)*r^4 + 2*(p2*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2))*(8*(1 - r + r^2) + p1*(-3 - 6*r + 6*r^2))) + (n11 + n33)*(2*log(pmax(1e-6,1 - r)) + log(-3*p1*(-1 + 4*r - 8*r^2 + 3*p2*(-1 - 2*r + 4*r^2)) - 2*(8*(-1 + r - 2*r^2) + 3*p2*(1 - 4*r + 8*r^2)))) + (n12 + n21 + n23 + n32)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(-3*p1*(-4*(1 - 3*r + 3*r^2) + 3*p2*(-1 - 6*r + 6*r^2)) - 8*(-5 + 6*r - 6*r^2 + p2*(3 - 9*r + 9*r^2))))
return(L)}
interlogL_cc3 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
optimize(logL_cc3,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc3 <- parallel::mcmapply(interlogL_cc3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"], p1, p2, mc.cores = ncores)


LOD_cc3 <- -((x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(-2*log10(2) + log10(-3*p1*(-1 - (15*p2)/2) - 8*(-7/2 + (3*p2)/4)))) - (x[,"n_13"] + x[,"n_31"])*(-2*log10(2) + log10(16 - 3*p1*(-1 - 3*p2) - 6*p2)) - (x[,"n_04"] + x[,"n_40"])*(-4*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(-4*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(-4*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(-4*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_44"])*(-4*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_11"] + x[,"n_33"])*(-2*log10(2) + log10(-3*p1*(-1 - 3*p2) - 2*(-8 + 3*p2))) - x[,"n_22"]*log10(((-4 - 3*p1)*(-2 + 3*p2))/16 - (9*(4 + 3*p1)*(-2 + 3*p2))/16 + 2*(6 - (9*p1)/2)*(3 + (9*p2)/2)) + (x[,"n_00"] + x[,"n_44"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + 4*log10(pmax(1e-6,1 - r_cc3))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + 3*log10(pmax(1e-6,1 - r_cc3)) + log10(pmax(1e-6,r_cc3))) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,1 - r_cc3)) + 2*log10(pmax(1e-6,r_cc3))) + (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cc3)) + 3*log10(pmax(1e-6,r_cc3))) + (x[,"n_04"] + x[,"n_40"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + 4*log10(pmax(1e-6,r_cc3))) + (x[,"n_13"] + x[,"n_31"])*(2*log10(pmax(1e-6,r_cc3)) + log10(16*(2 - 3*r_cc3 + 2*r_cc3^2) - 6*p2*(5 - 12*r_cc3 + 8*r_cc3^2) - 3*p1*(-5 + 12*r_cc3 - 8*r_cc3^2 + 3*p2*(1 - 6*r_cc3 + 4*r_cc3^2)))) + x[,"n_22"]*log10((-4 - 3*p1)*(-2 + 3*p2)*(-1 + r_cc3)^4 - 8*(4 + 3*p1)*(-2 + 3*p2)*(-1 + r_cc3)^2*r_cc3^2 - (4 + 3*p1)*(-2 + 3*p2)*r_cc3^4 + 2*(p2*(3 + 6*r_cc3 - 6*r_cc3^2) + 4*(1 - r_cc3 + r_cc3^2))*(8*(1 - r_cc3 + r_cc3^2) + p1*(-3 - 6*r_cc3 + 6*r_cc3^2))) + (x[,"n_11"] + x[,"n_33"])*(2*log10(pmax(1e-6,1 - r_cc3)) + log10(-3*p1*(-1 + 4*r_cc3 - 8*r_cc3^2 + 3*p2*(-1 - 2*r_cc3 + 4*r_cc3^2)) - 2*(8*(-1 + r_cc3 - 2*r_cc3^2) + 3*p2*(1 - 4*r_cc3 + 8*r_cc3^2)))) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(log10(pmax(1e-6,1 - r_cc3)) + log10(pmax(1e-6,r_cc3)) + log10(-3*p1*(-4*(1 - 3*r_cc3 + 3*r_cc3^2) + 3*p2*(-1 - 6*r_cc3 + 6*r_cc3^2)) - 8*(-5 + 6*r_cc3 - 6*r_cc3^2 + p2*(3 - 9*r_cc3 + 9*r_cc3^2))))


logL_cc3 <- (-5*x[,"n_00"] - 3*x[,"n_01"] - 4*x[,"n_02"] - 3*x[,"n_03"] - 5*x[,"n_04"] - 3*x[,"n_10"] - 3*x[,"n_11"] - 3*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 4*x[,"n_20"] - 3*x[,"n_21"] - 4*x[,"n_22"] - 3*x[,"n_23"] - 4*x[,"n_24"] - 3*x[,"n_30"] - 3*x[,"n_31"] - 3*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - 5*x[,"n_40"] - 3*x[,"n_41"] - 4*x[,"n_42"] - 3*x[,"n_43"] - 5*x[,"n_44"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_40"] - 2*x[,"n_41"] - x[,"n_42"] - 2*x[,"n_43"] - 2*x[,"n_44"])*log(3) + (x[,"n_00"] + x[,"n_44"])*(log(4 + 3*p1) + log(2 - 3*p2) + 4*log(pmax(1e-6,1 - r_cc3))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(log(4 + 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,1 - r_cc3)) + log(pmax(1e-6,r_cc3))) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r_cc3)) + 2*log(pmax(1e-6,r_cc3))) + (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cc3)) + 3*log(pmax(1e-6,r_cc3))) + (x[,"n_04"] + x[,"n_40"])*(log(4 + 3*p1) + log(2 - 3*p2) + 4*log(pmax(1e-6,r_cc3))) + (x[,"n_13"] + x[,"n_31"])*(2*log(pmax(1e-6,r_cc3)) + log(16*(2 - 3*r_cc3 + 2*r_cc3^2) - 6*p2*(5 - 12*r_cc3 + 8*r_cc3^2) - 3*p1*(-5 + 12*r_cc3 - 8*r_cc3^2 + 3*p2*(1 - 6*r_cc3 + 4*r_cc3^2)))) + x[,"n_22"]*log((-4 - 3*p1)*(-2 + 3*p2)*(-1 + r_cc3)^4 - 8*(4 + 3*p1)*(-2 + 3*p2)*(-1 + r_cc3)^2*r_cc3^2 - (4 + 3*p1)*(-2 + 3*p2)*r_cc3^4 + 2*(p2*(3 + 6*r_cc3 - 6*r_cc3^2) + 4*(1 - r_cc3 + r_cc3^2))*(8*(1 - r_cc3 + r_cc3^2) + p1*(-3 - 6*r_cc3 + 6*r_cc3^2))) + (x[,"n_11"] + x[,"n_33"])*(2*log(pmax(1e-6,1 - r_cc3)) + log(-3*p1*(-1 + 4*r_cc3 - 8*r_cc3^2 + 3*p2*(-1 - 2*r_cc3 + 4*r_cc3^2)) - 2*(8*(-1 + r_cc3 - 2*r_cc3^2) + 3*p2*(1 - 4*r_cc3 + 8*r_cc3^2)))) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(log(pmax(1e-6,1 - r_cc3)) + log(pmax(1e-6,r_cc3)) + log(-3*p1*(-4*(1 - 3*r_cc3 + 3*r_cc3^2) + 3*p2*(-1 - 6*r_cc3 + 6*r_cc3^2)) - 8*(-5 + 6*r_cc3 - 6*r_cc3^2 + p2*(3 - 9*r_cc3 + 9*r_cc3^2))))


logL_cc4 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
L <- (-6*n00 - 4*n01 - 5*n02 - 4*n03 - 6*n04 - 4*n10 - 4*n11 - 4*n12 - 4*n13 - 4*n14 - 5*n20 - 4*n21 - 5*n22 - 4*n23 - 5*n24 - 4*n30 - 4*n31 - 4*n32 - 4*n33 - 4*n34 - 6*n40 - 4*n41 - 5*n42 - 4*n43 - 6*n44)*log(2) + (-2*n00 - 2*n01 - n02 - 2*n03 - 2*n04 - 2*n10 - 2*n11 - 2*n12 - 2*n13 - 2*n14 - n20 - 2*n21 - 2*n22 - 2*n23 - n24 - 2*n30 - 2*n31 - 2*n32 - 2*n33 - 2*n34 - 2*n40 - 2*n41 - n42 - 2*n43 - 2*n44)*log(3) + (n00 + n44)*(log(4 + 3*p1) + log(4 + 3*p2) + 4*log(pmax(1e-6,1 - r))) + (n01 + n10 + n34 + n43)*(log(4 + 3*p1) + log(4 + 3*p2) + 3*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n02 + n20 + n24 + n42)*(log(4 + 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n03 + n14 + n30 + n41)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + 3*log(pmax(1e-6,r))) + (n04 + n40)*(log(4 + 3*p1) + log(4 + 3*p2) + 4*log(pmax(1e-6,r))) + (n11 + n33)*(2*log(pmax(1e-6,1 - r)) + log(32*(1 - r + 2*r^2) + 6*p2*(1 - 4*r + 8*r^2) + 3*p1*(2 - 8*r + 16*r^2 + 3*p2*(-1 - 2*r + 4*r^2)))) + n22*log((4 + 3*p1)*(4 + 3*p2)*(-1 + r)^4 + 8*(4 + 3*p1)*(4 + 3*p2)*(-1 + r)^2*r^2 + (4 + 3*p1)*(4 + 3*p2)*r^4 + 2*(8*(1 - r + r^2) + p1*(-3 - 6*r + 6*r^2))*(8*(1 - r + r^2) + p2*(-3 - 6*r + 6*r^2))) + (n13 + n31)*(2*log(pmax(1e-6,r)) + log(32*(2 - 3*r + 2*r^2) + 6*p2*(5 - 12*r + 8*r^2) + p1*(9*p2*(1 - 6*r + 4*r^2) + 6*(5 - 12*r + 8*r^2)))) + (n12 + n21 + n23 + n32)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(3*p1*(8*(1 - 3*r + 3*r^2) + 3*p2*(-1 - 6*r + 6*r^2)) + 8*(2*(5 - 6*r + 6*r^2) + p2*(3 - 9*r + 9*r^2))))
return(L)}
interlogL_cc4 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
optimize(logL_cc4,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cc4 <- parallel::mcmapply(interlogL_cc4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"], p1, p2, mc.cores = ncores)


LOD_cc4 <- -((x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(-2*log10(2) + log10(3*p1*(2 - (15*p2)/2) + 8*(7 + (3*p2)/4)))) - (x[,"n_04"] + x[,"n_40"])*(-4*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(-4*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(-4*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(-4*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_44"])*(-4*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_13"] + x[,"n_31"])*(-2*log10(2) + log10(32 + p1*(6 - 9*p2) + 6*p2)) - (x[,"n_11"] + x[,"n_33"])*(-2*log10(2) + log10(32 + 3*p1*(2 - 3*p2) + 6*p2)) - x[,"n_22"]*log10(2*(6 - (9*p1)/2)*(6 - (9*p2)/2) + (5*(4 + 3*p1)*(4 + 3*p2))/8) + (x[,"n_00"] + x[,"n_44"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + 4*log10(pmax(1e-6,1 - r_cc4))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + 3*log10(pmax(1e-6,1 - r_cc4)) + log10(pmax(1e-6,r_cc4))) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + 2*log10(pmax(1e-6,1 - r_cc4)) + 2*log10(pmax(1e-6,r_cc4))) + (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cc4)) + 3*log10(pmax(1e-6,r_cc4))) + (x[,"n_04"] + x[,"n_40"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + 4*log10(pmax(1e-6,r_cc4))) + (x[,"n_11"] + x[,"n_33"])*(2*log10(pmax(1e-6,1 - r_cc4)) + log10(32*(1 - r_cc4 + 2*r_cc4^2) + 6*p2*(1 - 4*r_cc4 + 8*r_cc4^2) + 3*p1*(2 - 8*r_cc4 + 16*r_cc4^2 + 3*p2*(-1 - 2*r_cc4 + 4*r_cc4^2)))) + x[,"n_22"]*log10((4 + 3*p1)*(4 + 3*p2)*(-1 + r_cc4)^4 + 8*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_cc4)^2*r_cc4^2 + (4 + 3*p1)*(4 + 3*p2)*r_cc4^4 + 2*(8*(1 - r_cc4 + r_cc4^2) + p1*(-3 - 6*r_cc4 + 6*r_cc4^2))*(8*(1 - r_cc4 + r_cc4^2) + p2*(-3 - 6*r_cc4 + 6*r_cc4^2))) + (x[,"n_13"] + x[,"n_31"])*(2*log10(pmax(1e-6,r_cc4)) + log10(32*(2 - 3*r_cc4 + 2*r_cc4^2) + 6*p2*(5 - 12*r_cc4 + 8*r_cc4^2) + p1*(9*p2*(1 - 6*r_cc4 + 4*r_cc4^2) + 6*(5 - 12*r_cc4 + 8*r_cc4^2)))) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(log10(pmax(1e-6,1 - r_cc4)) + log10(pmax(1e-6,r_cc4)) + log10(3*p1*(8*(1 - 3*r_cc4 + 3*r_cc4^2) + 3*p2*(-1 - 6*r_cc4 + 6*r_cc4^2)) + 8*(2*(5 - 6*r_cc4 + 6*r_cc4^2) + p2*(3 - 9*r_cc4 + 9*r_cc4^2))))


logL_cc4 <- (-6*x[,"n_00"] - 4*x[,"n_01"] - 5*x[,"n_02"] - 4*x[,"n_03"] - 6*x[,"n_04"] - 4*x[,"n_10"] - 4*x[,"n_11"] - 4*x[,"n_12"] - 4*x[,"n_13"] - 4*x[,"n_14"] - 5*x[,"n_20"] - 4*x[,"n_21"] - 5*x[,"n_22"] - 4*x[,"n_23"] - 5*x[,"n_24"] - 4*x[,"n_30"] - 4*x[,"n_31"] - 4*x[,"n_32"] - 4*x[,"n_33"] - 4*x[,"n_34"] - 6*x[,"n_40"] - 4*x[,"n_41"] - 5*x[,"n_42"] - 4*x[,"n_43"] - 6*x[,"n_44"])*log(2) + (-2*x[,"n_00"] - 2*x[,"n_01"] - x[,"n_02"] - 2*x[,"n_03"] - 2*x[,"n_04"] - 2*x[,"n_10"] - 2*x[,"n_11"] - 2*x[,"n_12"] - 2*x[,"n_13"] - 2*x[,"n_14"] - x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - x[,"n_24"] - 2*x[,"n_30"] - 2*x[,"n_31"] - 2*x[,"n_32"] - 2*x[,"n_33"] - 2*x[,"n_34"] - 2*x[,"n_40"] - 2*x[,"n_41"] - x[,"n_42"] - 2*x[,"n_43"] - 2*x[,"n_44"])*log(3) + (x[,"n_00"] + x[,"n_44"])*(log(4 + 3*p1) + log(4 + 3*p2) + 4*log(pmax(1e-6,1 - r_cc4))) + (x[,"n_01"] + x[,"n_10"] + x[,"n_34"] + x[,"n_43"])*(log(4 + 3*p1) + log(4 + 3*p2) + 3*log(pmax(1e-6,1 - r_cc4)) + log(pmax(1e-6,r_cc4))) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(log(4 + 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,1 - r_cc4)) + 2*log(pmax(1e-6,r_cc4))) + (x[,"n_03"] + x[,"n_14"] + x[,"n_30"] + x[,"n_41"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cc4)) + 3*log(pmax(1e-6,r_cc4))) + (x[,"n_04"] + x[,"n_40"])*(log(4 + 3*p1) + log(4 + 3*p2) + 4*log(pmax(1e-6,r_cc4))) + (x[,"n_11"] + x[,"n_33"])*(2*log(pmax(1e-6,1 - r_cc4)) + log(32*(1 - r_cc4 + 2*r_cc4^2) + 6*p2*(1 - 4*r_cc4 + 8*r_cc4^2) + 3*p1*(2 - 8*r_cc4 + 16*r_cc4^2 + 3*p2*(-1 - 2*r_cc4 + 4*r_cc4^2)))) + x[,"n_22"]*log((4 + 3*p1)*(4 + 3*p2)*(-1 + r_cc4)^4 + 8*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_cc4)^2*r_cc4^2 + (4 + 3*p1)*(4 + 3*p2)*r_cc4^4 + 2*(8*(1 - r_cc4 + r_cc4^2) + p1*(-3 - 6*r_cc4 + 6*r_cc4^2))*(8*(1 - r_cc4 + r_cc4^2) + p2*(-3 - 6*r_cc4 + 6*r_cc4^2))) + (x[,"n_13"] + x[,"n_31"])*(2*log(pmax(1e-6,r_cc4)) + log(32*(2 - 3*r_cc4 + 2*r_cc4^2) + 6*p2*(5 - 12*r_cc4 + 8*r_cc4^2) + p1*(9*p2*(1 - 6*r_cc4 + 4*r_cc4^2) + 6*(5 - 12*r_cc4 + 8*r_cc4^2)))) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(log(pmax(1e-6,1 - r_cc4)) + log(pmax(1e-6,r_cc4)) + log(3*p1*(8*(1 - 3*r_cc4 + 3*r_cc4^2) + 3*p2*(-1 - 6*r_cc4 + 6*r_cc4^2)) + 8*(2*(5 - 6*r_cc4 + 6*r_cc4^2) + p2*(3 - 9*r_cc4 + 9*r_cc4^2))))


logL_cm1 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
L <- (-5*n00 - 4*n01 - 5*n02 - 4*n03 - 5*n04 - 5*n10 - 4*n11 - 5*n12 - 4*n13 - 5*n14 - 5*n20 - 4*n21 - 4*n22 - 4*n23 - 5*n24 - 5*n30 - 4*n31 - 5*n32 - 4*n33 - 5*n34 - 5*n40 - 4*n41 - 5*n42 - 4*n43 - 5*n44)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(3) + (n00 + n44)*(log(2 - 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n04 + n40)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 3*log(pmax(1e-6,r))) + (n03 + n41)*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r)) + log(2 - 3*r + 2*r^2)) + (n01 + n43)*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r)) + log(1 - r + 2*r^2)) + (n02 + n42)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(5 - 6*r + 6*r^2)) + (n14 + n30)*(log(2 - 3*p1) + 2*log(pmax(1e-6,r)) + log(8 - 12*r + 8*r^2 - 3*p2*(1 - 6*r + 4*r^2))) + (n10 + n34)*(log(2 - 3*p1) + 2*log(pmax(1e-6,1 - r)) + log(4 - 4*r + 8*r^2 - 3*p2*(-1 - 2*r + 4*r^2))) + (n20 + n24)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(3*p2*(1 + 6*r - 6*r^2) + 2*(5 - 6*r + 6*r^2))) + (n13 + n31)*(log(pmax(1e-6,r)) + log(-8*(-3 + 5*r - 7*r^2 + 4*r^3) + 6*p2*(-3 + 10*r - 14*r^2 + 8*r^3) - 3*p1*(6 + 7*(-2 + 3*p2)*r + (28 - 42*p2)*r^2 + 8*(-2 + 3*p2)*r^3))) + (n11 + n33)*(log(pmax(1e-6,1 - r)) + log(8*(1 + 3*r - 5*r^2 + 4*r^3) - 6*p2*(-1 + 6*r - 10*r^2 + 8*r^3) + 3*p1*(3*p2*(-1 + 3*r - 10*r^2 + 8*r^3) - 2*(2 + 3*r - 10*r^2 + 8*r^3)))) + n22*log(-3*p1*(-2*(4 + 3*r + 15*r^2 - 36*r^3 + 18*r^4) + 3*p2*(-2 - 3*r + 21*r^2 - 36*r^3 + 18*r^4)) + 2*(16 + 18*r - 54*r^2 + 72*r^3 - 36*r^4 + 3*p2*(4 - 15*r + 33*r^2 - 36*r^3 + 18*r^4))) + (n12 + n32)*log(3*p1*(4 + 4*r - 52*r^2 + 96*r^3 - 48*r^4 + 3*p2*(-5 - 8*r + 32*r^2 - 48*r^3 + 24*r^4)) - 2*(-4*(5 - 7*r + 19*r^2 - 24*r^3 + 12*r^4) + 3*p2*(7 - 20*r + 44*r^2 - 48*r^3 + 24*r^4))) + (n21 + n23)*log(-6*p2*(-1 - 7*r + 19*r^2 - 24*r^3 + 12*r^4) + 4*(5 - 7*r + 19*r^2 - 24*r^3 + 12*r^4) + 3*p1*(2 + 2*r - 26*r^2 + 48*r^3 - 24*r^4 + p2*(6 - 3*r + 39*r^2 - 72*r^3 + 36*r^4)))
return(L)}
interlogL_cm1 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
optimize(logL_cm1,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cm1 <- parallel::mcmapply(interlogL_cm1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"], p1, p2, mc.cores = ncores)


LOD_cm1 <- -((x[,"n_04"] + x[,"n_40"])*(-4*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_44"])*(-4*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_03"] + x[,"n_41"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_43"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_42"])*(-3*log10(2) + log10(7) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_11"] + x[,"n_33"])*(-log10(2) + log10(14 + 3*p1*(-4 - 3*p2) - 3*p2)) - (x[,"n_14"] + x[,"n_30"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_34"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_20"] + x[,"n_24"])*(-2*log10(2) + log10(2 - 3*p1) + log10(7 + (15*p2)/2)) - x[,"n_22"]*log10(-3*p1*(-47/4 - (39*p2)/8) + 2*(73/4 + (33*p2)/8)) - (x[,"n_21"] + x[,"n_23"])*log10(16 + 12*p2 + 3*p1*(1 + (15*p2)/2)) - (x[,"n_12"] + x[,"n_32"])*log10(3*p1*(2 - (33*p2)/2) - 2*(-16 + (21*p2)/2)) - (x[,"n_13"] + x[,"n_31"])*(-log10(2) + log10(14 - 3*p2 - 3*p1*(4 + (28 - 42*p2)/4 + 3*p2 + (7*(-2 + 3*p2))/2))) + (x[,"n_00"] + x[,"n_44"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 3*log10(pmax(1e-6,1 - r_cm1)) + log10(pmax(1e-6,r_cm1))) + (x[,"n_04"] + x[,"n_40"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cm1)) + 3*log10(pmax(1e-6,r_cm1))) + (x[,"n_03"] + x[,"n_41"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_cm1)) + log10(2 - 3*r_cm1 + 2*r_cm1^2)) + (x[,"n_01"] + x[,"n_43"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,1 - r_cm1)) + log10(1 - r_cm1 + 2*r_cm1^2)) + (x[,"n_02"] + x[,"n_42"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cm1)) + log10(pmax(1e-6,r_cm1)) + log10(5 - 6*r_cm1 + 6*r_cm1^2)) + (x[,"n_14"] + x[,"n_30"])*(log10(2 - 3*p1) + 2*log10(pmax(1e-6,r_cm1)) + log10(8 - 12*r_cm1 + 8*r_cm1^2 - 3*p2*(1 - 6*r_cm1 + 4*r_cm1^2))) + (x[,"n_10"] + x[,"n_34"])*(log10(2 - 3*p1) + 2*log10(pmax(1e-6,1 - r_cm1)) + log10(4 - 4*r_cm1 + 8*r_cm1^2 - 3*p2*(-1 - 2*r_cm1 + 4*r_cm1^2))) + (x[,"n_20"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_cm1)) + log10(pmax(1e-6,r_cm1)) + log10(3*p2*(1 + 6*r_cm1 - 6*r_cm1^2) + 2*(5 - 6*r_cm1 + 6*r_cm1^2))) + (x[,"n_13"] + x[,"n_31"])*(log10(pmax(1e-6,r_cm1)) + log10(-8*(-3 + 5*r_cm1 - 7*r_cm1^2 + 4*r_cm1^3) + 6*p2*(-3 + 10*r_cm1 - 14*r_cm1^2 + 8*r_cm1^3) - 3*p1*(6 + 7*(-2 + 3*p2)*r_cm1 + (28 - 42*p2)*r_cm1^2 + 8*(-2 + 3*p2)*r_cm1^3))) + (x[,"n_11"] + x[,"n_33"])*(log10(pmax(1e-6,1 - r_cm1)) + log10(8*(1 + 3*r_cm1 - 5*r_cm1^2 + 4*r_cm1^3) - 6*p2*(-1 + 6*r_cm1 - 10*r_cm1^2 + 8*r_cm1^3) + 3*p1*(3*p2*(-1 + 3*r_cm1 - 10*r_cm1^2 + 8*r_cm1^3) - 2*(2 + 3*r_cm1 - 10*r_cm1^2 + 8*r_cm1^3)))) + x[,"n_22"]*log10(-3*p1*(-2*(4 + 3*r_cm1 + 15*r_cm1^2 - 36*r_cm1^3 + 18*r_cm1^4) + 3*p2*(-2 - 3*r_cm1 + 21*r_cm1^2 - 36*r_cm1^3 + 18*r_cm1^4)) + 2*(16 + 18*r_cm1 - 54*r_cm1^2 + 72*r_cm1^3 - 36*r_cm1^4 + 3*p2*(4 - 15*r_cm1 + 33*r_cm1^2 - 36*r_cm1^3 + 18*r_cm1^4))) + (x[,"n_12"] + x[,"n_32"])*log10(3*p1*(4 + 4*r_cm1 - 52*r_cm1^2 + 96*r_cm1^3 - 48*r_cm1^4 + 3*p2*(-5 - 8*r_cm1 + 32*r_cm1^2 - 48*r_cm1^3 + 24*r_cm1^4)) - 2*(-4*(5 - 7*r_cm1 + 19*r_cm1^2 - 24*r_cm1^3 + 12*r_cm1^4) + 3*p2*(7 - 20*r_cm1 + 44*r_cm1^2 - 48*r_cm1^3 + 24*r_cm1^4))) + (x[,"n_21"] + x[,"n_23"])*log10(-6*p2*(-1 - 7*r_cm1 + 19*r_cm1^2 - 24*r_cm1^3 + 12*r_cm1^4) + 4*(5 - 7*r_cm1 + 19*r_cm1^2 - 24*r_cm1^3 + 12*r_cm1^4) + 3*p1*(2 + 2*r_cm1 - 26*r_cm1^2 + 48*r_cm1^3 - 24*r_cm1^4 + p2*(6 - 3*r_cm1 + 39*r_cm1^2 - 72*r_cm1^3 + 36*r_cm1^4)))


logL_cm1 <- (-5*x[,"n_00"] - 4*x[,"n_01"] - 5*x[,"n_02"] - 4*x[,"n_03"] - 5*x[,"n_04"] - 5*x[,"n_10"] - 4*x[,"n_11"] - 5*x[,"n_12"] - 4*x[,"n_13"] - 5*x[,"n_14"] - 5*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 4*x[,"n_23"] - 5*x[,"n_24"] - 5*x[,"n_30"] - 4*x[,"n_31"] - 5*x[,"n_32"] - 4*x[,"n_33"] - 5*x[,"n_34"] - 5*x[,"n_40"] - 4*x[,"n_41"] - 5*x[,"n_42"] - 4*x[,"n_43"] - 5*x[,"n_44"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(3) + (x[,"n_00"] + x[,"n_44"])*(log(2 - 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,1 - r_cm1)) + log(pmax(1e-6,r_cm1))) + (x[,"n_04"] + x[,"n_40"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cm1)) + 3*log(pmax(1e-6,r_cm1))) + (x[,"n_03"] + x[,"n_41"])*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r_cm1)) + log(2 - 3*r_cm1 + 2*r_cm1^2)) + (x[,"n_01"] + x[,"n_43"])*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r_cm1)) + log(1 - r_cm1 + 2*r_cm1^2)) + (x[,"n_02"] + x[,"n_42"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cm1)) + log(pmax(1e-6,r_cm1)) + log(5 - 6*r_cm1 + 6*r_cm1^2)) + (x[,"n_14"] + x[,"n_30"])*(log(2 - 3*p1) + 2*log(pmax(1e-6,r_cm1)) + log(8 - 12*r_cm1 + 8*r_cm1^2 - 3*p2*(1 - 6*r_cm1 + 4*r_cm1^2))) + (x[,"n_10"] + x[,"n_34"])*(log(2 - 3*p1) + 2*log(pmax(1e-6,1 - r_cm1)) + log(4 - 4*r_cm1 + 8*r_cm1^2 - 3*p2*(-1 - 2*r_cm1 + 4*r_cm1^2))) + (x[,"n_20"] + x[,"n_24"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_cm1)) + log(pmax(1e-6,r_cm1)) + log(3*p2*(1 + 6*r_cm1 - 6*r_cm1^2) + 2*(5 - 6*r_cm1 + 6*r_cm1^2))) + (x[,"n_13"] + x[,"n_31"])*(log(pmax(1e-6,r_cm1)) + log(-8*(-3 + 5*r_cm1 - 7*r_cm1^2 + 4*r_cm1^3) + 6*p2*(-3 + 10*r_cm1 - 14*r_cm1^2 + 8*r_cm1^3) - 3*p1*(6 + 7*(-2 + 3*p2)*r_cm1 + (28 - 42*p2)*r_cm1^2 + 8*(-2 + 3*p2)*r_cm1^3))) + (x[,"n_11"] + x[,"n_33"])*(log(pmax(1e-6,1 - r_cm1)) + log(8*(1 + 3*r_cm1 - 5*r_cm1^2 + 4*r_cm1^3) - 6*p2*(-1 + 6*r_cm1 - 10*r_cm1^2 + 8*r_cm1^3) + 3*p1*(3*p2*(-1 + 3*r_cm1 - 10*r_cm1^2 + 8*r_cm1^3) - 2*(2 + 3*r_cm1 - 10*r_cm1^2 + 8*r_cm1^3)))) + x[,"n_22"]*log(-3*p1*(-2*(4 + 3*r_cm1 + 15*r_cm1^2 - 36*r_cm1^3 + 18*r_cm1^4) + 3*p2*(-2 - 3*r_cm1 + 21*r_cm1^2 - 36*r_cm1^3 + 18*r_cm1^4)) + 2*(16 + 18*r_cm1 - 54*r_cm1^2 + 72*r_cm1^3 - 36*r_cm1^4 + 3*p2*(4 - 15*r_cm1 + 33*r_cm1^2 - 36*r_cm1^3 + 18*r_cm1^4))) + (x[,"n_12"] + x[,"n_32"])*log(3*p1*(4 + 4*r_cm1 - 52*r_cm1^2 + 96*r_cm1^3 - 48*r_cm1^4 + 3*p2*(-5 - 8*r_cm1 + 32*r_cm1^2 - 48*r_cm1^3 + 24*r_cm1^4)) - 2*(-4*(5 - 7*r_cm1 + 19*r_cm1^2 - 24*r_cm1^3 + 12*r_cm1^4) + 3*p2*(7 - 20*r_cm1 + 44*r_cm1^2 - 48*r_cm1^3 + 24*r_cm1^4))) + (x[,"n_21"] + x[,"n_23"])*log(-6*p2*(-1 - 7*r_cm1 + 19*r_cm1^2 - 24*r_cm1^3 + 12*r_cm1^4) + 4*(5 - 7*r_cm1 + 19*r_cm1^2 - 24*r_cm1^3 + 12*r_cm1^4) + 3*p1*(2 + 2*r_cm1 - 26*r_cm1^2 + 48*r_cm1^3 - 24*r_cm1^4 + p2*(6 - 3*r_cm1 + 39*r_cm1^2 - 72*r_cm1^3 + 36*r_cm1^4)))


logL_cm2 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
L <- (-5*n00 - 5*n01 - 5*n02 - 5*n03 - 5*n04 - 4*n10 - 4*n11 - 4*n12 - 4*n13 - 4*n14 - 5*n20 - 5*n21 - 4*n22 - 5*n23 - 5*n24 - 4*n30 - 4*n31 - 4*n32 - 4*n33 - 4*n34 - 5*n40 - 5*n41 - 5*n42 - 5*n43 - 5*n44)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(3) + (n00 + n44)*(log(2 - 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n04 + n40)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 3*log(pmax(1e-6,r))) + (n14 + n30)*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r)) + log(2 - 3*r + 2*r^2)) + (n10 + n34)*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r)) + log(1 - r + 2*r^2)) + (n20 + n24)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(5 - 6*r + 6*r^2)) + (n03 + n41)*(log(2 - 3*p1) + 2*log(pmax(1e-6,r)) + log(8 - 12*r + 8*r^2 - 3*p2*(1 - 6*r + 4*r^2))) + (n01 + n43)*(log(2 - 3*p1) + 2*log(pmax(1e-6,1 - r)) + log(4 - 4*r + 8*r^2 - 3*p2*(-1 - 2*r + 4*r^2))) + (n02 + n42)*(log(2 - 3*p1) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(3*p2*(1 + 6*r - 6*r^2) + 2*(5 - 6*r + 6*r^2))) + (n13 + n31)*(log(pmax(1e-6,r)) + log(-8*(-3 + 5*r - 7*r^2 + 4*r^3) + 6*p2*(-3 + 10*r - 14*r^2 + 8*r^3) - 3*p1*(6 + 7*(-2 + 3*p2)*r + (28 - 42*p2)*r^2 + 8*(-2 + 3*p2)*r^3))) + (n11 + n33)*(log(pmax(1e-6,1 - r)) + log(8*(1 + 3*r - 5*r^2 + 4*r^3) - 6*p2*(-1 + 6*r - 10*r^2 + 8*r^3) + 3*p1*(3*p2*(-1 + 3*r - 10*r^2 + 8*r^3) - 2*(2 + 3*r - 10*r^2 + 8*r^3)))) + n22*log(-3*p1*(-2*(4 + 3*r + 15*r^2 - 36*r^3 + 18*r^4) + 3*p2*(-2 - 3*r + 21*r^2 - 36*r^3 + 18*r^4)) + 2*(16 + 18*r - 54*r^2 + 72*r^3 - 36*r^4 + 3*p2*(4 - 15*r + 33*r^2 - 36*r^3 + 18*r^4))) + (n21 + n23)*log(3*p1*(4 + 4*r - 52*r^2 + 96*r^3 - 48*r^4 + 3*p2*(-5 - 8*r + 32*r^2 - 48*r^3 + 24*r^4)) - 2*(-4*(5 - 7*r + 19*r^2 - 24*r^3 + 12*r^4) + 3*p2*(7 - 20*r + 44*r^2 - 48*r^3 + 24*r^4))) + (n12 + n32)*log(-6*p2*(-1 - 7*r + 19*r^2 - 24*r^3 + 12*r^4) + 4*(5 - 7*r + 19*r^2 - 24*r^3 + 12*r^4) + 3*p1*(2 + 2*r - 26*r^2 + 48*r^3 - 24*r^4 + p2*(6 - 3*r + 39*r^2 - 72*r^3 + 36*r^4)))
return(L)}
interlogL_cm2 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
optimize(logL_cm2,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cm2 <- parallel::mcmapply(interlogL_cm2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"], p1, p2, mc.cores = ncores)


LOD_cm2 <- -((x[,"n_04"] + x[,"n_40"])*(-4*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_44"])*(-4*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_14"] + x[,"n_30"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_34"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_20"] + x[,"n_24"])*(-3*log10(2) + log10(7) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_11"] + x[,"n_33"])*(-log10(2) + log10(14 + 3*p1*(-4 - 3*p2) - 3*p2)) - (x[,"n_03"] + x[,"n_41"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_43"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_42"])*(-2*log10(2) + log10(2 - 3*p1) + log10(7 + (15*p2)/2)) - x[,"n_22"]*log10(-3*p1*(-47/4 - (39*p2)/8) + 2*(73/4 + (33*p2)/8)) - (x[,"n_12"] + x[,"n_32"])*log10(16 + 12*p2 + 3*p1*(1 + (15*p2)/2)) - (x[,"n_21"] + x[,"n_23"])*log10(3*p1*(2 - (33*p2)/2) - 2*(-16 + (21*p2)/2)) - (x[,"n_13"] + x[,"n_31"])*(-log10(2) + log10(14 - 3*p2 - 3*p1*(4 + (28 - 42*p2)/4 + 3*p2 + (7*(-2 + 3*p2))/2))) + (x[,"n_00"] + x[,"n_44"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 3*log10(pmax(1e-6,1 - r_cm2)) + log10(pmax(1e-6,r_cm2))) + (x[,"n_04"] + x[,"n_40"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cm2)) + 3*log10(pmax(1e-6,r_cm2))) + (x[,"n_14"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_cm2)) + log10(2 - 3*r_cm2 + 2*r_cm2^2)) + (x[,"n_10"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,1 - r_cm2)) + log10(1 - r_cm2 + 2*r_cm2^2)) + (x[,"n_20"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cm2)) + log10(pmax(1e-6,r_cm2)) + log10(5 - 6*r_cm2 + 6*r_cm2^2)) + (x[,"n_03"] + x[,"n_41"])*(log10(2 - 3*p1) + 2*log10(pmax(1e-6,r_cm2)) + log10(8 - 12*r_cm2 + 8*r_cm2^2 - 3*p2*(1 - 6*r_cm2 + 4*r_cm2^2))) + (x[,"n_01"] + x[,"n_43"])*(log10(2 - 3*p1) + 2*log10(pmax(1e-6,1 - r_cm2)) + log10(4 - 4*r_cm2 + 8*r_cm2^2 - 3*p2*(-1 - 2*r_cm2 + 4*r_cm2^2))) + (x[,"n_02"] + x[,"n_42"])*(log10(2 - 3*p1) + log10(pmax(1e-6,1 - r_cm2)) + log10(pmax(1e-6,r_cm2)) + log10(3*p2*(1 + 6*r_cm2 - 6*r_cm2^2) + 2*(5 - 6*r_cm2 + 6*r_cm2^2))) + (x[,"n_13"] + x[,"n_31"])*(log10(pmax(1e-6,r_cm2)) + log10(-8*(-3 + 5*r_cm2 - 7*r_cm2^2 + 4*r_cm2^3) + 6*p2*(-3 + 10*r_cm2 - 14*r_cm2^2 + 8*r_cm2^3) - 3*p1*(6 + 7*(-2 + 3*p2)*r_cm2 + (28 - 42*p2)*r_cm2^2 + 8*(-2 + 3*p2)*r_cm2^3))) + (x[,"n_11"] + x[,"n_33"])*(log10(pmax(1e-6,1 - r_cm2)) + log10(8*(1 + 3*r_cm2 - 5*r_cm2^2 + 4*r_cm2^3) - 6*p2*(-1 + 6*r_cm2 - 10*r_cm2^2 + 8*r_cm2^3) + 3*p1*(3*p2*(-1 + 3*r_cm2 - 10*r_cm2^2 + 8*r_cm2^3) - 2*(2 + 3*r_cm2 - 10*r_cm2^2 + 8*r_cm2^3)))) + x[,"n_22"]*log10(-3*p1*(-2*(4 + 3*r_cm2 + 15*r_cm2^2 - 36*r_cm2^3 + 18*r_cm2^4) + 3*p2*(-2 - 3*r_cm2 + 21*r_cm2^2 - 36*r_cm2^3 + 18*r_cm2^4)) + 2*(16 + 18*r_cm2 - 54*r_cm2^2 + 72*r_cm2^3 - 36*r_cm2^4 + 3*p2*(4 - 15*r_cm2 + 33*r_cm2^2 - 36*r_cm2^3 + 18*r_cm2^4))) + (x[,"n_21"] + x[,"n_23"])*log10(3*p1*(4 + 4*r_cm2 - 52*r_cm2^2 + 96*r_cm2^3 - 48*r_cm2^4 + 3*p2*(-5 - 8*r_cm2 + 32*r_cm2^2 - 48*r_cm2^3 + 24*r_cm2^4)) - 2*(-4*(5 - 7*r_cm2 + 19*r_cm2^2 - 24*r_cm2^3 + 12*r_cm2^4) + 3*p2*(7 - 20*r_cm2 + 44*r_cm2^2 - 48*r_cm2^3 + 24*r_cm2^4))) + (x[,"n_12"] + x[,"n_32"])*log10(-6*p2*(-1 - 7*r_cm2 + 19*r_cm2^2 - 24*r_cm2^3 + 12*r_cm2^4) + 4*(5 - 7*r_cm2 + 19*r_cm2^2 - 24*r_cm2^3 + 12*r_cm2^4) + 3*p1*(2 + 2*r_cm2 - 26*r_cm2^2 + 48*r_cm2^3 - 24*r_cm2^4 + p2*(6 - 3*r_cm2 + 39*r_cm2^2 - 72*r_cm2^3 + 36*r_cm2^4)))


logL_cm2 <- (-5*x[,"n_00"] - 5*x[,"n_01"] - 5*x[,"n_02"] - 5*x[,"n_03"] - 5*x[,"n_04"] - 4*x[,"n_10"] - 4*x[,"n_11"] - 4*x[,"n_12"] - 4*x[,"n_13"] - 4*x[,"n_14"] - 5*x[,"n_20"] - 5*x[,"n_21"] - 4*x[,"n_22"] - 5*x[,"n_23"] - 5*x[,"n_24"] - 4*x[,"n_30"] - 4*x[,"n_31"] - 4*x[,"n_32"] - 4*x[,"n_33"] - 4*x[,"n_34"] - 5*x[,"n_40"] - 5*x[,"n_41"] - 5*x[,"n_42"] - 5*x[,"n_43"] - 5*x[,"n_44"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(3) + (x[,"n_00"] + x[,"n_44"])*(log(2 - 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,1 - r_cm2)) + log(pmax(1e-6,r_cm2))) + (x[,"n_04"] + x[,"n_40"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cm2)) + 3*log(pmax(1e-6,r_cm2))) + (x[,"n_14"] + x[,"n_30"])*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r_cm2)) + log(2 - 3*r_cm2 + 2*r_cm2^2)) + (x[,"n_10"] + x[,"n_34"])*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r_cm2)) + log(1 - r_cm2 + 2*r_cm2^2)) + (x[,"n_20"] + x[,"n_24"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cm2)) + log(pmax(1e-6,r_cm2)) + log(5 - 6*r_cm2 + 6*r_cm2^2)) + (x[,"n_03"] + x[,"n_41"])*(log(2 - 3*p1) + 2*log(pmax(1e-6,r_cm2)) + log(8 - 12*r_cm2 + 8*r_cm2^2 - 3*p2*(1 - 6*r_cm2 + 4*r_cm2^2))) + (x[,"n_01"] + x[,"n_43"])*(log(2 - 3*p1) + 2*log(pmax(1e-6,1 - r_cm2)) + log(4 - 4*r_cm2 + 8*r_cm2^2 - 3*p2*(-1 - 2*r_cm2 + 4*r_cm2^2))) + (x[,"n_02"] + x[,"n_42"])*(log(2 - 3*p1) + log(pmax(1e-6,1 - r_cm2)) + log(pmax(1e-6,r_cm2)) + log(3*p2*(1 + 6*r_cm2 - 6*r_cm2^2) + 2*(5 - 6*r_cm2 + 6*r_cm2^2))) + (x[,"n_13"] + x[,"n_31"])*(log(pmax(1e-6,r_cm2)) + log(-8*(-3 + 5*r_cm2 - 7*r_cm2^2 + 4*r_cm2^3) + 6*p2*(-3 + 10*r_cm2 - 14*r_cm2^2 + 8*r_cm2^3) - 3*p1*(6 + 7*(-2 + 3*p2)*r_cm2 + (28 - 42*p2)*r_cm2^2 + 8*(-2 + 3*p2)*r_cm2^3))) + (x[,"n_11"] + x[,"n_33"])*(log(pmax(1e-6,1 - r_cm2)) + log(8*(1 + 3*r_cm2 - 5*r_cm2^2 + 4*r_cm2^3) - 6*p2*(-1 + 6*r_cm2 - 10*r_cm2^2 + 8*r_cm2^3) + 3*p1*(3*p2*(-1 + 3*r_cm2 - 10*r_cm2^2 + 8*r_cm2^3) - 2*(2 + 3*r_cm2 - 10*r_cm2^2 + 8*r_cm2^3)))) + x[,"n_22"]*log(-3*p1*(-2*(4 + 3*r_cm2 + 15*r_cm2^2 - 36*r_cm2^3 + 18*r_cm2^4) + 3*p2*(-2 - 3*r_cm2 + 21*r_cm2^2 - 36*r_cm2^3 + 18*r_cm2^4)) + 2*(16 + 18*r_cm2 - 54*r_cm2^2 + 72*r_cm2^3 - 36*r_cm2^4 + 3*p2*(4 - 15*r_cm2 + 33*r_cm2^2 - 36*r_cm2^3 + 18*r_cm2^4))) + (x[,"n_21"] + x[,"n_23"])*log(3*p1*(4 + 4*r_cm2 - 52*r_cm2^2 + 96*r_cm2^3 - 48*r_cm2^4 + 3*p2*(-5 - 8*r_cm2 + 32*r_cm2^2 - 48*r_cm2^3 + 24*r_cm2^4)) - 2*(-4*(5 - 7*r_cm2 + 19*r_cm2^2 - 24*r_cm2^3 + 12*r_cm2^4) + 3*p2*(7 - 20*r_cm2 + 44*r_cm2^2 - 48*r_cm2^3 + 24*r_cm2^4))) + (x[,"n_12"] + x[,"n_32"])*log(-6*p2*(-1 - 7*r_cm2 + 19*r_cm2^2 - 24*r_cm2^3 + 12*r_cm2^4) + 4*(5 - 7*r_cm2 + 19*r_cm2^2 - 24*r_cm2^3 + 12*r_cm2^4) + 3*p1*(2 + 2*r_cm2 - 26*r_cm2^2 + 48*r_cm2^3 - 24*r_cm2^4 + p2*(6 - 3*r_cm2 + 39*r_cm2^2 - 72*r_cm2^3 + 36*r_cm2^4)))


logL_cm3 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
L <- (-6*n00 - 5*n01 - 6*n02 - 5*n03 - 6*n04 - 6*n10 - 5*n11 - 6*n12 - 5*n13 - 6*n14 - 6*n20 - 5*n21 - 5*n22 - 5*n23 - 6*n24 - 6*n30 - 5*n31 - 6*n32 - 5*n33 - 6*n34 - 6*n40 - 5*n41 - 6*n42 - 5*n43 - 6*n44)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(3) + (n00 + n44)*(log(4 + 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n04 + n40)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 3*log(pmax(1e-6,r))) + (n03 + n41)*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r)) + log(2 - 3*r + 2*r^2)) + (n01 + n43)*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r)) + log(1 - r + 2*r^2)) + (n02 + n42)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(5 - 6*r + 6*r^2)) + (n14 + n30)*(log(4 + 3*p1) + 2*log(pmax(1e-6,r)) + log(8 - 12*r + 8*r^2 - 3*p2*(1 - 6*r + 4*r^2))) + (n10 + n34)*(log(4 + 3*p1) + 2*log(pmax(1e-6,1 - r)) + log(4 - 4*r + 8*r^2 - 3*p2*(-1 - 2*r + 4*r^2))) + (n20 + n24)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(3*p2*(1 + 6*r - 6*r^2) + 2*(5 - 6*r + 6*r^2))) + (n13 + n31)*(log(pmax(1e-6,r)) + log(3*p1*(6 + 7*(-2 + 3*p2)*r + (28 - 42*p2)*r^2 + 8*(-2 + 3*p2)*r^3) + 4*(-4*(-3 + 5*r - 7*r^2 + 4*r^3) + 3*p2*(-3 + 10*r - 14*r^2 + 8*r^3)))) + (n11 + n33)*(log(pmax(1e-6,1 - r)) + log(-3*p1*(3*p2*(-1 + 3*r - 10*r^2 + 8*r^3) - 2*(2 + 3*r - 10*r^2 + 8*r^3)) - 4*(-4*(1 + 3*r - 5*r^2 + 4*r^3) + 3*p2*(-1 + 6*r - 10*r^2 + 8*r^3)))) + n22*log(3*p1*(-2*(4 + 3*r + 15*r^2 - 36*r^3 + 18*r^4) + 3*p2*(-2 - 3*r + 21*r^2 - 36*r^3 + 18*r^4)) + 4*(16 + 18*r - 54*r^2 + 72*r^3 - 36*r^4 + 3*p2*(4 - 15*r + 33*r^2 - 36*r^3 + 18*r^4))) + (n12 + n32)*log(-3*p1*(4 + 4*r - 52*r^2 + 96*r^3 - 48*r^4 + 3*p2*(-5 - 8*r + 32*r^2 - 48*r^3 + 24*r^4)) - 4*(-4*(5 - 7*r + 19*r^2 - 24*r^3 + 12*r^4) + 3*p2*(7 - 20*r + 44*r^2 - 48*r^3 + 24*r^4))) + (n21 + n23)*log(-4*(3*p2*(-1 - 7*r + 19*r^2 - 24*r^3 + 12*r^4) - 2*(5 - 7*r + 19*r^2 - 24*r^3 + 12*r^4)) - 3*p1*(2 + 2*r - 26*r^2 + 48*r^3 - 24*r^4 + p2*(6 - 3*r + 39*r^2 - 72*r^3 + 36*r^4)))
return(L)}
interlogL_cm3 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
optimize(logL_cm3,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cm3 <- parallel::mcmapply(interlogL_cm3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"], p1, p2, mc.cores = ncores)


LOD_cm3 <- -((x[,"n_04"] + x[,"n_40"])*(-4*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_44"])*(-4*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_03"] + x[,"n_41"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_43"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_42"])*(-3*log10(2) + log10(7) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_14"] + x[,"n_30"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_34"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_20"] + x[,"n_24"])*(-2*log10(2) + log10(4 + 3*p1) + log10(7 + (15*p2)/2)) - (x[,"n_11"] + x[,"n_33"])*(-log10(2) + log10(-3*p1*(-4 - 3*p2) - 4*(-7 + (3*p2)/2))) - x[,"n_22"]*log10(3*p1*(-47/4 - (39*p2)/8) + 4*(73/4 + (33*p2)/8)) - (x[,"n_21"] + x[,"n_23"])*log10(-4*(-8 - 6*p2) - 3*p1*(1 + (15*p2)/2)) - (x[,"n_12"] + x[,"n_32"])*log10(-3*p1*(2 - (33*p2)/2) - 4*(-16 + (21*p2)/2)) - (x[,"n_13"] + x[,"n_31"])*(-log10(2) + log10(4*(7 - (3*p2)/2) + 3*p1*(4 + (28 - 42*p2)/4 + 3*p2 + (7*(-2 + 3*p2))/2))) + (x[,"n_00"] + x[,"n_44"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + 3*log10(pmax(1e-6,1 - r_cm3)) + log10(pmax(1e-6,r_cm3))) + (x[,"n_04"] + x[,"n_40"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cm3)) + 3*log10(pmax(1e-6,r_cm3))) + (x[,"n_03"] + x[,"n_41"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_cm3)) + log10(2 - 3*r_cm3 + 2*r_cm3^2)) + (x[,"n_01"] + x[,"n_43"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,1 - r_cm3)) + log10(1 - r_cm3 + 2*r_cm3^2)) + (x[,"n_02"] + x[,"n_42"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cm3)) + log10(pmax(1e-6,r_cm3)) + log10(5 - 6*r_cm3 + 6*r_cm3^2)) + (x[,"n_14"] + x[,"n_30"])*(log10(4 + 3*p1) + 2*log10(pmax(1e-6,r_cm3)) + log10(8 - 12*r_cm3 + 8*r_cm3^2 - 3*p2*(1 - 6*r_cm3 + 4*r_cm3^2))) + (x[,"n_10"] + x[,"n_34"])*(log10(4 + 3*p1) + 2*log10(pmax(1e-6,1 - r_cm3)) + log10(4 - 4*r_cm3 + 8*r_cm3^2 - 3*p2*(-1 - 2*r_cm3 + 4*r_cm3^2))) + (x[,"n_20"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_cm3)) + log10(pmax(1e-6,r_cm3)) + log10(3*p2*(1 + 6*r_cm3 - 6*r_cm3^2) + 2*(5 - 6*r_cm3 + 6*r_cm3^2))) + (x[,"n_13"] + x[,"n_31"])*(log10(pmax(1e-6,r_cm3)) + log10(3*p1*(6 + 7*(-2 + 3*p2)*r_cm3 + (28 - 42*p2)*r_cm3^2 + 8*(-2 + 3*p2)*r_cm3^3) + 4*(-4*(-3 + 5*r_cm3 - 7*r_cm3^2 + 4*r_cm3^3) + 3*p2*(-3 + 10*r_cm3 - 14*r_cm3^2 + 8*r_cm3^3)))) + (x[,"n_11"] + x[,"n_33"])*(log10(pmax(1e-6,1 - r_cm3)) + log10(-3*p1*(3*p2*(-1 + 3*r_cm3 - 10*r_cm3^2 + 8*r_cm3^3) - 2*(2 + 3*r_cm3 - 10*r_cm3^2 + 8*r_cm3^3)) - 4*(-4*(1 + 3*r_cm3 - 5*r_cm3^2 + 4*r_cm3^3) + 3*p2*(-1 + 6*r_cm3 - 10*r_cm3^2 + 8*r_cm3^3)))) + x[,"n_22"]*log10(3*p1*(-2*(4 + 3*r_cm3 + 15*r_cm3^2 - 36*r_cm3^3 + 18*r_cm3^4) + 3*p2*(-2 - 3*r_cm3 + 21*r_cm3^2 - 36*r_cm3^3 + 18*r_cm3^4)) + 4*(16 + 18*r_cm3 - 54*r_cm3^2 + 72*r_cm3^3 - 36*r_cm3^4 + 3*p2*(4 - 15*r_cm3 + 33*r_cm3^2 - 36*r_cm3^3 + 18*r_cm3^4))) + (x[,"n_12"] + x[,"n_32"])*log10(-3*p1*(4 + 4*r_cm3 - 52*r_cm3^2 + 96*r_cm3^3 - 48*r_cm3^4 + 3*p2*(-5 - 8*r_cm3 + 32*r_cm3^2 - 48*r_cm3^3 + 24*r_cm3^4)) - 4*(-4*(5 - 7*r_cm3 + 19*r_cm3^2 - 24*r_cm3^3 + 12*r_cm3^4) + 3*p2*(7 - 20*r_cm3 + 44*r_cm3^2 - 48*r_cm3^3 + 24*r_cm3^4))) + (x[,"n_21"] + x[,"n_23"])*log10(-4*(3*p2*(-1 - 7*r_cm3 + 19*r_cm3^2 - 24*r_cm3^3 + 12*r_cm3^4) - 2*(5 - 7*r_cm3 + 19*r_cm3^2 - 24*r_cm3^3 + 12*r_cm3^4)) - 3*p1*(2 + 2*r_cm3 - 26*r_cm3^2 + 48*r_cm3^3 - 24*r_cm3^4 + p2*(6 - 3*r_cm3 + 39*r_cm3^2 - 72*r_cm3^3 + 36*r_cm3^4)))


logL_cm3 <- (-6*x[,"n_00"] - 5*x[,"n_01"] - 6*x[,"n_02"] - 5*x[,"n_03"] - 6*x[,"n_04"] - 6*x[,"n_10"] - 5*x[,"n_11"] - 6*x[,"n_12"] - 5*x[,"n_13"] - 6*x[,"n_14"] - 6*x[,"n_20"] - 5*x[,"n_21"] - 5*x[,"n_22"] - 5*x[,"n_23"] - 6*x[,"n_24"] - 6*x[,"n_30"] - 5*x[,"n_31"] - 6*x[,"n_32"] - 5*x[,"n_33"] - 6*x[,"n_34"] - 6*x[,"n_40"] - 5*x[,"n_41"] - 6*x[,"n_42"] - 5*x[,"n_43"] - 6*x[,"n_44"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(3) + (x[,"n_00"] + x[,"n_44"])*(log(4 + 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,1 - r_cm3)) + log(pmax(1e-6,r_cm3))) + (x[,"n_04"] + x[,"n_40"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cm3)) + 3*log(pmax(1e-6,r_cm3))) + (x[,"n_03"] + x[,"n_41"])*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r_cm3)) + log(2 - 3*r_cm3 + 2*r_cm3^2)) + (x[,"n_01"] + x[,"n_43"])*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r_cm3)) + log(1 - r_cm3 + 2*r_cm3^2)) + (x[,"n_02"] + x[,"n_42"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cm3)) + log(pmax(1e-6,r_cm3)) + log(5 - 6*r_cm3 + 6*r_cm3^2)) + (x[,"n_14"] + x[,"n_30"])*(log(4 + 3*p1) + 2*log(pmax(1e-6,r_cm3)) + log(8 - 12*r_cm3 + 8*r_cm3^2 - 3*p2*(1 - 6*r_cm3 + 4*r_cm3^2))) + (x[,"n_10"] + x[,"n_34"])*(log(4 + 3*p1) + 2*log(pmax(1e-6,1 - r_cm3)) + log(4 - 4*r_cm3 + 8*r_cm3^2 - 3*p2*(-1 - 2*r_cm3 + 4*r_cm3^2))) + (x[,"n_20"] + x[,"n_24"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_cm3)) + log(pmax(1e-6,r_cm3)) + log(3*p2*(1 + 6*r_cm3 - 6*r_cm3^2) + 2*(5 - 6*r_cm3 + 6*r_cm3^2))) + (x[,"n_13"] + x[,"n_31"])*(log(pmax(1e-6,r_cm3)) + log(3*p1*(6 + 7*(-2 + 3*p2)*r_cm3 + (28 - 42*p2)*r_cm3^2 + 8*(-2 + 3*p2)*r_cm3^3) + 4*(-4*(-3 + 5*r_cm3 - 7*r_cm3^2 + 4*r_cm3^3) + 3*p2*(-3 + 10*r_cm3 - 14*r_cm3^2 + 8*r_cm3^3)))) + (x[,"n_11"] + x[,"n_33"])*(log(pmax(1e-6,1 - r_cm3)) + log(-3*p1*(3*p2*(-1 + 3*r_cm3 - 10*r_cm3^2 + 8*r_cm3^3) - 2*(2 + 3*r_cm3 - 10*r_cm3^2 + 8*r_cm3^3)) - 4*(-4*(1 + 3*r_cm3 - 5*r_cm3^2 + 4*r_cm3^3) + 3*p2*(-1 + 6*r_cm3 - 10*r_cm3^2 + 8*r_cm3^3)))) + x[,"n_22"]*log(3*p1*(-2*(4 + 3*r_cm3 + 15*r_cm3^2 - 36*r_cm3^3 + 18*r_cm3^4) + 3*p2*(-2 - 3*r_cm3 + 21*r_cm3^2 - 36*r_cm3^3 + 18*r_cm3^4)) + 4*(16 + 18*r_cm3 - 54*r_cm3^2 + 72*r_cm3^3 - 36*r_cm3^4 + 3*p2*(4 - 15*r_cm3 + 33*r_cm3^2 - 36*r_cm3^3 + 18*r_cm3^4))) + (x[,"n_12"] + x[,"n_32"])*log(-3*p1*(4 + 4*r_cm3 - 52*r_cm3^2 + 96*r_cm3^3 - 48*r_cm3^4 + 3*p2*(-5 - 8*r_cm3 + 32*r_cm3^2 - 48*r_cm3^3 + 24*r_cm3^4)) - 4*(-4*(5 - 7*r_cm3 + 19*r_cm3^2 - 24*r_cm3^3 + 12*r_cm3^4) + 3*p2*(7 - 20*r_cm3 + 44*r_cm3^2 - 48*r_cm3^3 + 24*r_cm3^4))) + (x[,"n_21"] + x[,"n_23"])*log(-4*(3*p2*(-1 - 7*r_cm3 + 19*r_cm3^2 - 24*r_cm3^3 + 12*r_cm3^4) - 2*(5 - 7*r_cm3 + 19*r_cm3^2 - 24*r_cm3^3 + 12*r_cm3^4)) - 3*p1*(2 + 2*r_cm3 - 26*r_cm3^2 + 48*r_cm3^3 - 24*r_cm3^4 + p2*(6 - 3*r_cm3 + 39*r_cm3^2 - 72*r_cm3^3 + 36*r_cm3^4)))


logL_cm4 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
L <- (-6*n00 - 6*n01 - 6*n02 - 6*n03 - 6*n04 - 5*n10 - 5*n11 - 5*n12 - 5*n13 - 5*n14 - 6*n20 - 6*n21 - 5*n22 - 6*n23 - 6*n24 - 5*n30 - 5*n31 - 5*n32 - 5*n33 - 5*n34 - 6*n40 - 6*n41 - 6*n42 - 6*n43 - 6*n44)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(3) + (n00 + n44)*(log(4 + 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n04 + n40)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 3*log(pmax(1e-6,r))) + (n14 + n30)*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r)) + log(2 - 3*r + 2*r^2)) + (n10 + n34)*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r)) + log(1 - r + 2*r^2)) + (n20 + n24)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(5 - 6*r + 6*r^2)) + (n03 + n41)*(log(4 + 3*p1) + 2*log(pmax(1e-6,r)) + log(8 - 12*r + 8*r^2 - 3*p2*(1 - 6*r + 4*r^2))) + (n01 + n43)*(log(4 + 3*p1) + 2*log(pmax(1e-6,1 - r)) + log(4 - 4*r + 8*r^2 - 3*p2*(-1 - 2*r + 4*r^2))) + (n02 + n42)*(log(4 + 3*p1) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(3*p2*(1 + 6*r - 6*r^2) + 2*(5 - 6*r + 6*r^2))) + (n13 + n31)*(log(pmax(1e-6,r)) + log(3*p1*(6 + 7*(-2 + 3*p2)*r + (28 - 42*p2)*r^2 + 8*(-2 + 3*p2)*r^3) + 4*(-4*(-3 + 5*r - 7*r^2 + 4*r^3) + 3*p2*(-3 + 10*r - 14*r^2 + 8*r^3)))) + (n11 + n33)*(log(pmax(1e-6,1 - r)) + log(-3*p1*(3*p2*(-1 + 3*r - 10*r^2 + 8*r^3) - 2*(2 + 3*r - 10*r^2 + 8*r^3)) - 4*(-4*(1 + 3*r - 5*r^2 + 4*r^3) + 3*p2*(-1 + 6*r - 10*r^2 + 8*r^3)))) + n22*log(3*p1*(-2*(4 + 3*r + 15*r^2 - 36*r^3 + 18*r^4) + 3*p2*(-2 - 3*r + 21*r^2 - 36*r^3 + 18*r^4)) + 4*(16 + 18*r - 54*r^2 + 72*r^3 - 36*r^4 + 3*p2*(4 - 15*r + 33*r^2 - 36*r^3 + 18*r^4))) + (n21 + n23)*log(-3*p1*(4 + 4*r - 52*r^2 + 96*r^3 - 48*r^4 + 3*p2*(-5 - 8*r + 32*r^2 - 48*r^3 + 24*r^4)) - 4*(-4*(5 - 7*r + 19*r^2 - 24*r^3 + 12*r^4) + 3*p2*(7 - 20*r + 44*r^2 - 48*r^3 + 24*r^4))) + (n12 + n32)*log(-4*(3*p2*(-1 - 7*r + 19*r^2 - 24*r^3 + 12*r^4) - 2*(5 - 7*r + 19*r^2 - 24*r^3 + 12*r^4)) - 3*p1*(2 + 2*r - 26*r^2 + 48*r^3 - 24*r^4 + p2*(6 - 3*r + 39*r^2 - 72*r^3 + 36*r^4)))
return(L)}
interlogL_cm4 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
optimize(logL_cm4,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cm4 <- parallel::mcmapply(interlogL_cm4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"], p1, p2, mc.cores = ncores)


LOD_cm4 <- -((x[,"n_04"] + x[,"n_40"])*(-4*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2))) - (x[,"n_00"] + x[,"n_44"])*(-4*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_14"] + x[,"n_30"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_34"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_20"] + x[,"n_24"])*(-3*log10(2) + log10(7) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_03"] + x[,"n_41"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_43"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_42"])*(-2*log10(2) + log10(4 + 3*p1) + log10(7 + (15*p2)/2)) - (x[,"n_11"] + x[,"n_33"])*(-log10(2) + log10(-3*p1*(-4 - 3*p2) - 4*(-7 + (3*p2)/2))) - x[,"n_22"]*log10(3*p1*(-47/4 - (39*p2)/8) + 4*(73/4 + (33*p2)/8)) - (x[,"n_12"] + x[,"n_32"])*log10(-4*(-8 - 6*p2) - 3*p1*(1 + (15*p2)/2)) - (x[,"n_21"] + x[,"n_23"])*log10(-3*p1*(2 - (33*p2)/2) - 4*(-16 + (21*p2)/2)) - (x[,"n_13"] + x[,"n_31"])*(-log10(2) + log10(4*(7 - (3*p2)/2) + 3*p1*(4 + (28 - 42*p2)/4 + 3*p2 + (7*(-2 + 3*p2))/2))) + (x[,"n_00"] + x[,"n_44"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + 3*log10(pmax(1e-6,1 - r_cm4)) + log10(pmax(1e-6,r_cm4))) + (x[,"n_04"] + x[,"n_40"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cm4)) + 3*log10(pmax(1e-6,r_cm4))) + (x[,"n_14"] + x[,"n_30"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_cm4)) + log10(2 - 3*r_cm4 + 2*r_cm4^2)) + (x[,"n_10"] + x[,"n_34"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,1 - r_cm4)) + log10(1 - r_cm4 + 2*r_cm4^2)) + (x[,"n_20"] + x[,"n_24"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cm4)) + log10(pmax(1e-6,r_cm4)) + log10(5 - 6*r_cm4 + 6*r_cm4^2)) + (x[,"n_03"] + x[,"n_41"])*(log10(4 + 3*p1) + 2*log10(pmax(1e-6,r_cm4)) + log10(8 - 12*r_cm4 + 8*r_cm4^2 - 3*p2*(1 - 6*r_cm4 + 4*r_cm4^2))) + (x[,"n_01"] + x[,"n_43"])*(log10(4 + 3*p1) + 2*log10(pmax(1e-6,1 - r_cm4)) + log10(4 - 4*r_cm4 + 8*r_cm4^2 - 3*p2*(-1 - 2*r_cm4 + 4*r_cm4^2))) + (x[,"n_02"] + x[,"n_42"])*(log10(4 + 3*p1) + log10(pmax(1e-6,1 - r_cm4)) + log10(pmax(1e-6,r_cm4)) + log10(3*p2*(1 + 6*r_cm4 - 6*r_cm4^2) + 2*(5 - 6*r_cm4 + 6*r_cm4^2))) + (x[,"n_13"] + x[,"n_31"])*(log10(pmax(1e-6,r_cm4)) + log10(3*p1*(6 + 7*(-2 + 3*p2)*r_cm4 + (28 - 42*p2)*r_cm4^2 + 8*(-2 + 3*p2)*r_cm4^3) + 4*(-4*(-3 + 5*r_cm4 - 7*r_cm4^2 + 4*r_cm4^3) + 3*p2*(-3 + 10*r_cm4 - 14*r_cm4^2 + 8*r_cm4^3)))) + (x[,"n_11"] + x[,"n_33"])*(log10(pmax(1e-6,1 - r_cm4)) + log10(-3*p1*(3*p2*(-1 + 3*r_cm4 - 10*r_cm4^2 + 8*r_cm4^3) - 2*(2 + 3*r_cm4 - 10*r_cm4^2 + 8*r_cm4^3)) - 4*(-4*(1 + 3*r_cm4 - 5*r_cm4^2 + 4*r_cm4^3) + 3*p2*(-1 + 6*r_cm4 - 10*r_cm4^2 + 8*r_cm4^3)))) + x[,"n_22"]*log10(3*p1*(-2*(4 + 3*r_cm4 + 15*r_cm4^2 - 36*r_cm4^3 + 18*r_cm4^4) + 3*p2*(-2 - 3*r_cm4 + 21*r_cm4^2 - 36*r_cm4^3 + 18*r_cm4^4)) + 4*(16 + 18*r_cm4 - 54*r_cm4^2 + 72*r_cm4^3 - 36*r_cm4^4 + 3*p2*(4 - 15*r_cm4 + 33*r_cm4^2 - 36*r_cm4^3 + 18*r_cm4^4))) + (x[,"n_21"] + x[,"n_23"])*log10(-3*p1*(4 + 4*r_cm4 - 52*r_cm4^2 + 96*r_cm4^3 - 48*r_cm4^4 + 3*p2*(-5 - 8*r_cm4 + 32*r_cm4^2 - 48*r_cm4^3 + 24*r_cm4^4)) - 4*(-4*(5 - 7*r_cm4 + 19*r_cm4^2 - 24*r_cm4^3 + 12*r_cm4^4) + 3*p2*(7 - 20*r_cm4 + 44*r_cm4^2 - 48*r_cm4^3 + 24*r_cm4^4))) + (x[,"n_12"] + x[,"n_32"])*log10(-4*(3*p2*(-1 - 7*r_cm4 + 19*r_cm4^2 - 24*r_cm4^3 + 12*r_cm4^4) - 2*(5 - 7*r_cm4 + 19*r_cm4^2 - 24*r_cm4^3 + 12*r_cm4^4)) - 3*p1*(2 + 2*r_cm4 - 26*r_cm4^2 + 48*r_cm4^3 - 24*r_cm4^4 + p2*(6 - 3*r_cm4 + 39*r_cm4^2 - 72*r_cm4^3 + 36*r_cm4^4)))


logL_cm4 <- (-6*x[,"n_00"] - 6*x[,"n_01"] - 6*x[,"n_02"] - 6*x[,"n_03"] - 6*x[,"n_04"] - 5*x[,"n_10"] - 5*x[,"n_11"] - 5*x[,"n_12"] - 5*x[,"n_13"] - 5*x[,"n_14"] - 6*x[,"n_20"] - 6*x[,"n_21"] - 5*x[,"n_22"] - 6*x[,"n_23"] - 6*x[,"n_24"] - 5*x[,"n_30"] - 5*x[,"n_31"] - 5*x[,"n_32"] - 5*x[,"n_33"] - 5*x[,"n_34"] - 6*x[,"n_40"] - 6*x[,"n_41"] - 6*x[,"n_42"] - 6*x[,"n_43"] - 6*x[,"n_44"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(3) + (x[,"n_00"] + x[,"n_44"])*(log(4 + 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,1 - r_cm4)) + log(pmax(1e-6,r_cm4))) + (x[,"n_04"] + x[,"n_40"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cm4)) + 3*log(pmax(1e-6,r_cm4))) + (x[,"n_14"] + x[,"n_30"])*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r_cm4)) + log(2 - 3*r_cm4 + 2*r_cm4^2)) + (x[,"n_10"] + x[,"n_34"])*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r_cm4)) + log(1 - r_cm4 + 2*r_cm4^2)) + (x[,"n_20"] + x[,"n_24"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cm4)) + log(pmax(1e-6,r_cm4)) + log(5 - 6*r_cm4 + 6*r_cm4^2)) + (x[,"n_03"] + x[,"n_41"])*(log(4 + 3*p1) + 2*log(pmax(1e-6,r_cm4)) + log(8 - 12*r_cm4 + 8*r_cm4^2 - 3*p2*(1 - 6*r_cm4 + 4*r_cm4^2))) + (x[,"n_01"] + x[,"n_43"])*(log(4 + 3*p1) + 2*log(pmax(1e-6,1 - r_cm4)) + log(4 - 4*r_cm4 + 8*r_cm4^2 - 3*p2*(-1 - 2*r_cm4 + 4*r_cm4^2))) + (x[,"n_02"] + x[,"n_42"])*(log(4 + 3*p1) + log(pmax(1e-6,1 - r_cm4)) + log(pmax(1e-6,r_cm4)) + log(3*p2*(1 + 6*r_cm4 - 6*r_cm4^2) + 2*(5 - 6*r_cm4 + 6*r_cm4^2))) + (x[,"n_13"] + x[,"n_31"])*(log(pmax(1e-6,r_cm4)) + log(3*p1*(6 + 7*(-2 + 3*p2)*r_cm4 + (28 - 42*p2)*r_cm4^2 + 8*(-2 + 3*p2)*r_cm4^3) + 4*(-4*(-3 + 5*r_cm4 - 7*r_cm4^2 + 4*r_cm4^3) + 3*p2*(-3 + 10*r_cm4 - 14*r_cm4^2 + 8*r_cm4^3)))) + (x[,"n_11"] + x[,"n_33"])*(log(pmax(1e-6,1 - r_cm4)) + log(-3*p1*(3*p2*(-1 + 3*r_cm4 - 10*r_cm4^2 + 8*r_cm4^3) - 2*(2 + 3*r_cm4 - 10*r_cm4^2 + 8*r_cm4^3)) - 4*(-4*(1 + 3*r_cm4 - 5*r_cm4^2 + 4*r_cm4^3) + 3*p2*(-1 + 6*r_cm4 - 10*r_cm4^2 + 8*r_cm4^3)))) + x[,"n_22"]*log(3*p1*(-2*(4 + 3*r_cm4 + 15*r_cm4^2 - 36*r_cm4^3 + 18*r_cm4^4) + 3*p2*(-2 - 3*r_cm4 + 21*r_cm4^2 - 36*r_cm4^3 + 18*r_cm4^4)) + 4*(16 + 18*r_cm4 - 54*r_cm4^2 + 72*r_cm4^3 - 36*r_cm4^4 + 3*p2*(4 - 15*r_cm4 + 33*r_cm4^2 - 36*r_cm4^3 + 18*r_cm4^4))) + (x[,"n_21"] + x[,"n_23"])*log(-3*p1*(4 + 4*r_cm4 - 52*r_cm4^2 + 96*r_cm4^3 - 48*r_cm4^4 + 3*p2*(-5 - 8*r_cm4 + 32*r_cm4^2 - 48*r_cm4^3 + 24*r_cm4^4)) - 4*(-4*(5 - 7*r_cm4 + 19*r_cm4^2 - 24*r_cm4^3 + 12*r_cm4^4) + 3*p2*(7 - 20*r_cm4 + 44*r_cm4^2 - 48*r_cm4^3 + 24*r_cm4^4))) + (x[,"n_12"] + x[,"n_32"])*log(-4*(3*p2*(-1 - 7*r_cm4 + 19*r_cm4^2 - 24*r_cm4^3 + 12*r_cm4^4) - 2*(5 - 7*r_cm4 + 19*r_cm4^2 - 24*r_cm4^3 + 12*r_cm4^4)) - 3*p1*(2 + 2*r_cm4 - 26*r_cm4^2 + 48*r_cm4^3 - 24*r_cm4^4 + p2*(6 - 3*r_cm4 + 39*r_cm4^2 - 72*r_cm4^3 + 36*r_cm4^4)))


logL_cr1 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
L <- (-4*n00 - 3*n01 - 4*n02 - 3*n03 - 4*n04 - 3*n10 - 3*n11 - 2*n12 - 3*n13 - 3*n14 - 4*n20 - 2*n21 - 2*n22 - 2*n23 - 4*n24 - 3*n30 - 3*n31 - 2*n32 - 3*n33 - 3*n34 - 4*n40 - 3*n41 - 4*n42 - 3*n43 - 4*n44)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(3) + (n00 + n04 + n40 + n44)*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n01 + n03 + n10 + n14 + n30 + n34 + n41 + n43)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(1 - 2*r + 2*r^2)) + (n02 + n20 + n24 + n42)*(log(2 - 3*p1) + log(2 - 3*p2) + log((-1 + r)^4 + 4*(-1 + r)^2*r^2 + r^4)) + n22*log(5*(-2 + 3*p1)*(-2 + 3*p2)*(-1 + r)^2*r^2 + (p1*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2))*(p2*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2))) + (n11 + n33)*log(4*(-2 + 3*p1)*(-2 + 3*p2)*(-1 + r)^2*r^2 + (-2 + 3*p2)*r^2*(-4*(1 - r + r^2) + p1*(-3 - 6*r + 6*r^2)) + (-2 + 3*p1)*(-1 + r)^2*(-4*(1 - r + r^2) + p2*(-3 - 6*r + 6*r^2))) + (n12 + n21 + n23 + n32)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(3*p1*(-4*(1 - 3*r + 3*r^2) + 3*p2*(-1 - 6*r + 6*r^2)) - 4*(-5 + 6*r - 6*r^2 + p2*(3 - 9*r + 9*r^2)))) + (n13 + n31)*log(8*(1 - 3*r + 7*r^2 - 8*r^3 + 4*r^4) - 6*p2*(2 - 6*r + 11*r^2 - 16*r^3 + 8*r^4) + 3*p1*(2 - 22*r^2 + 32*r^3 - 16*r^4 + 3*p2*(-1 + 8*r^2 - 16*r^3 + 8*r^4)))
return(L)}
interlogL_cr1 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
optimize(logL_cr1,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr1 <- parallel::mcmapply(interlogL_cr1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"], p1, p2, mc.cores = ncores)


LOD_cr1 <- -((x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(-2*log10(2) + log10(3*p1*(-1 - (15*p2)/2) - 4*(-7/2 + (3*p2)/4)))) - (x[,"n_00"] + x[,"n_04"] + x[,"n_40"] + x[,"n_44"])*(-4*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_03"] + x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"] + x[,"n_41"] + x[,"n_43"])*(-3*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(-3*log10(2) + log10(3) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_13"] + x[,"n_31"])*log10(4 + 3*p1*(-1/2 - (3*p2)/2) - (3*p2)/2) - (x[,"n_11"] + x[,"n_33"])*log10(((-2 + 3*p1)*(-3 - (9*p2)/2))/4 + ((-3 - (9*p1)/2)*(-2 + 3*p2))/4 + ((-2 + 3*p1)*(-2 + 3*p2))/4) - x[,"n_22"]*log10((5*(-2 + 3*p1)*(-2 + 3*p2))/16 + (3 + (9*p1)/2)*(3 + (9*p2)/2)) + (x[,"n_00"] + x[,"n_04"] + x[,"n_40"] + x[,"n_44"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,1 - r_cr1)) + 2*log10(pmax(1e-6,r_cr1))) + (x[,"n_01"] + x[,"n_03"] + x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"] + x[,"n_41"] + x[,"n_43"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cr1)) + log10(pmax(1e-6,r_cr1)) + log10(1 - 2*r_cr1 + 2*r_cr1^2)) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10((-1 + r_cr1)^4 + 4*(-1 + r_cr1)^2*r_cr1^2 + r_cr1^4)) + x[,"n_22"]*log10(5*(-2 + 3*p1)*(-2 + 3*p2)*(-1 + r_cr1)^2*r_cr1^2 + (p1*(3 + 6*r_cr1 - 6*r_cr1^2) + 4*(1 - r_cr1 + r_cr1^2))*(p2*(3 + 6*r_cr1 - 6*r_cr1^2) + 4*(1 - r_cr1 + r_cr1^2))) + (x[,"n_11"] + x[,"n_33"])*log10(4*(-2 + 3*p1)*(-2 + 3*p2)*(-1 + r_cr1)^2*r_cr1^2 + (-2 + 3*p2)*r_cr1^2*(-4*(1 - r_cr1 + r_cr1^2) + p1*(-3 - 6*r_cr1 + 6*r_cr1^2)) + (-2 + 3*p1)*(-1 + r_cr1)^2*(-4*(1 - r_cr1 + r_cr1^2) + p2*(-3 - 6*r_cr1 + 6*r_cr1^2))) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(log10(pmax(1e-6,1 - r_cr1)) + log10(pmax(1e-6,r_cr1)) + log10(3*p1*(-4*(1 - 3*r_cr1 + 3*r_cr1^2) + 3*p2*(-1 - 6*r_cr1 + 6*r_cr1^2)) - 4*(-5 + 6*r_cr1 - 6*r_cr1^2 + p2*(3 - 9*r_cr1 + 9*r_cr1^2)))) + (x[,"n_13"] + x[,"n_31"])*log10(8*(1 - 3*r_cr1 + 7*r_cr1^2 - 8*r_cr1^3 + 4*r_cr1^4) - 6*p2*(2 - 6*r_cr1 + 11*r_cr1^2 - 16*r_cr1^3 + 8*r_cr1^4) + 3*p1*(2 - 22*r_cr1^2 + 32*r_cr1^3 - 16*r_cr1^4 + 3*p2*(-1 + 8*r_cr1^2 - 16*r_cr1^3 + 8*r_cr1^4)))


logL_cr1 <- (-4*x[,"n_00"] - 3*x[,"n_01"] - 4*x[,"n_02"] - 3*x[,"n_03"] - 4*x[,"n_04"] - 3*x[,"n_10"] - 3*x[,"n_11"] - 2*x[,"n_12"] - 3*x[,"n_13"] - 3*x[,"n_14"] - 4*x[,"n_20"] - 2*x[,"n_21"] - 2*x[,"n_22"] - 2*x[,"n_23"] - 4*x[,"n_24"] - 3*x[,"n_30"] - 3*x[,"n_31"] - 2*x[,"n_32"] - 3*x[,"n_33"] - 3*x[,"n_34"] - 4*x[,"n_40"] - 3*x[,"n_41"] - 4*x[,"n_42"] - 3*x[,"n_43"] - 4*x[,"n_44"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(3) + (x[,"n_00"] + x[,"n_04"] + x[,"n_40"] + x[,"n_44"])*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r_cr1)) + 2*log(pmax(1e-6,r_cr1))) + (x[,"n_01"] + x[,"n_03"] + x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"] + x[,"n_41"] + x[,"n_43"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cr1)) + log(pmax(1e-6,r_cr1)) + log(1 - 2*r_cr1 + 2*r_cr1^2)) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(log(2 - 3*p1) + log(2 - 3*p2) + log((-1 + r_cr1)^4 + 4*(-1 + r_cr1)^2*r_cr1^2 + r_cr1^4)) + x[,"n_22"]*log(5*(-2 + 3*p1)*(-2 + 3*p2)*(-1 + r_cr1)^2*r_cr1^2 + (p1*(3 + 6*r_cr1 - 6*r_cr1^2) + 4*(1 - r_cr1 + r_cr1^2))*(p2*(3 + 6*r_cr1 - 6*r_cr1^2) + 4*(1 - r_cr1 + r_cr1^2))) + (x[,"n_11"] + x[,"n_33"])*log(4*(-2 + 3*p1)*(-2 + 3*p2)*(-1 + r_cr1)^2*r_cr1^2 + (-2 + 3*p2)*r_cr1^2*(-4*(1 - r_cr1 + r_cr1^2) + p1*(-3 - 6*r_cr1 + 6*r_cr1^2)) + (-2 + 3*p1)*(-1 + r_cr1)^2*(-4*(1 - r_cr1 + r_cr1^2) + p2*(-3 - 6*r_cr1 + 6*r_cr1^2))) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(log(pmax(1e-6,1 - r_cr1)) + log(pmax(1e-6,r_cr1)) + log(3*p1*(-4*(1 - 3*r_cr1 + 3*r_cr1^2) + 3*p2*(-1 - 6*r_cr1 + 6*r_cr1^2)) - 4*(-5 + 6*r_cr1 - 6*r_cr1^2 + p2*(3 - 9*r_cr1 + 9*r_cr1^2)))) + (x[,"n_13"] + x[,"n_31"])*log(8*(1 - 3*r_cr1 + 7*r_cr1^2 - 8*r_cr1^3 + 4*r_cr1^4) - 6*p2*(2 - 6*r_cr1 + 11*r_cr1^2 - 16*r_cr1^3 + 8*r_cr1^4) + 3*p1*(2 - 22*r_cr1^2 + 32*r_cr1^3 - 16*r_cr1^4 + 3*p2*(-1 + 8*r_cr1^2 - 16*r_cr1^3 + 8*r_cr1^4)))


logL_cr2 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
L <- (-5*n00 - 4*n01 - 5*n02 - 4*n03 - 5*n04 - 4*n10 - 4*n11 - 3*n12 - 4*n13 - 4*n14 - 5*n20 - 3*n21 - 3*n22 - 3*n23 - 5*n24 - 4*n30 - 4*n31 - 3*n32 - 4*n33 - 4*n34 - 5*n40 - 4*n41 - 5*n42 - 4*n43 - 5*n44)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(3) + (n00 + n04 + n40 + n44)*(log(2 - 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n01 + n03 + n10 + n14 + n30 + n34 + n41 + n43)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log((-1 + r)^2 + r^2)) + (n02 + n20 + n24 + n42)*(log(2 - 3*p1) + log(4 + 3*p2) + log(1 - 4*r + 10*r^2 - 12*r^3 + 6*r^4)) + (n11 + n33)*log(-4*(-2 + 3*p1)*(4 + 3*p2)*(-1 + r)^2*r^2 + (4 + 3*p2)*r^2*(p1*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2)) - (-2 + 3*p1)*(-1 + r)^2*(8*(1 - r + r^2) + p2*(-3 - 6*r + 6*r^2))) + (n13 + n31)*log(-4*(-2 + 3*p1)*(4 + 3*p2)*(-1 + r)^2*r^2 + (4 + 3*p2)*(-1 + r)^2*(p1*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2)) - (-2 + 3*p1)*r^2*(8*(1 - r + r^2) + p2*(-3 - 6*r + 6*r^2))) + n22*log(-5*(-2 + 3*p1)*(4 + 3*p2)*(-1 + r)^2*r^2 + (p1*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2))*(8*(1 - r + r^2) + p2*(-3 - 6*r + 6*r^2))) + (n12 + n21 + n23 + n32)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(-3*p1*(8*(1 - 3*r + 3*r^2) + 3*p2*(-1 - 6*r + 6*r^2)) + 4*(2*(5 - 6*r + 6*r^2) + p2*(3 - 9*r + 9*r^2))))
return(L)}
interlogL_cr2 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
optimize(logL_cr2,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr2 <- parallel::mcmapply(interlogL_cr2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"], p1, p2, mc.cores = ncores)


LOD_cr2 <- -((x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(-2*log10(2) + log10(-3*p1*(2 - (15*p2)/2) + 4*(7 + (3*p2)/4)))) - (x[,"n_00"] + x[,"n_04"] + x[,"n_40"] + x[,"n_44"])*(-4*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_03"] + x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"] + x[,"n_41"] + x[,"n_43"])*(-3*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(-3*log10(2) + log10(3) + log10(2 - 3*p1) + log10(4 + 3*p2)) - x[,"n_22"]*log10((3 + (9*p1)/2)*(6 - (9*p2)/2) - (5*(-2 + 3*p1)*(4 + 3*p2))/16) - (x[,"n_13"] + x[,"n_31"])*log10(-((-2 + 3*p1)*(6 - (9*p2)/2))/4 - ((-2 + 3*p1)*(4 + 3*p2))/4 + ((3 + (9*p1)/2)*(4 + 3*p2))/4) - (x[,"n_11"] + x[,"n_33"])*log10(-((-2 + 3*p1)*(6 - (9*p2)/2))/4 - ((-2 + 3*p1)*(4 + 3*p2))/4 + ((3 + (9*p1)/2)*(4 + 3*p2))/4) + (x[,"n_00"] + x[,"n_04"] + x[,"n_40"] + x[,"n_44"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + 2*log10(pmax(1e-6,1 - r_cr2)) + 2*log10(pmax(1e-6,r_cr2))) + (x[,"n_01"] + x[,"n_03"] + x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"] + x[,"n_41"] + x[,"n_43"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cr2)) + log10(pmax(1e-6,r_cr2)) + log10((-1 + r_cr2)^2 + r_cr2^2)) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(1 - 4*r_cr2 + 10*r_cr2^2 - 12*r_cr2^3 + 6*r_cr2^4)) + (x[,"n_11"] + x[,"n_33"])*log10(-4*(-2 + 3*p1)*(4 + 3*p2)*(-1 + r_cr2)^2*r_cr2^2 + (4 + 3*p2)*r_cr2^2*(p1*(3 + 6*r_cr2 - 6*r_cr2^2) + 4*(1 - r_cr2 + r_cr2^2)) - (-2 + 3*p1)*(-1 + r_cr2)^2*(8*(1 - r_cr2 + r_cr2^2) + p2*(-3 - 6*r_cr2 + 6*r_cr2^2))) + (x[,"n_13"] + x[,"n_31"])*log10(-4*(-2 + 3*p1)*(4 + 3*p2)*(-1 + r_cr2)^2*r_cr2^2 + (4 + 3*p2)*(-1 + r_cr2)^2*(p1*(3 + 6*r_cr2 - 6*r_cr2^2) + 4*(1 - r_cr2 + r_cr2^2)) - (-2 + 3*p1)*r_cr2^2*(8*(1 - r_cr2 + r_cr2^2) + p2*(-3 - 6*r_cr2 + 6*r_cr2^2))) + x[,"n_22"]*log10(-5*(-2 + 3*p1)*(4 + 3*p2)*(-1 + r_cr2)^2*r_cr2^2 + (p1*(3 + 6*r_cr2 - 6*r_cr2^2) + 4*(1 - r_cr2 + r_cr2^2))*(8*(1 - r_cr2 + r_cr2^2) + p2*(-3 - 6*r_cr2 + 6*r_cr2^2))) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(log10(pmax(1e-6,1 - r_cr2)) + log10(pmax(1e-6,r_cr2)) + log10(-3*p1*(8*(1 - 3*r_cr2 + 3*r_cr2^2) + 3*p2*(-1 - 6*r_cr2 + 6*r_cr2^2)) + 4*(2*(5 - 6*r_cr2 + 6*r_cr2^2) + p2*(3 - 9*r_cr2 + 9*r_cr2^2))))


logL_cr2 <- (-5*x[,"n_00"] - 4*x[,"n_01"] - 5*x[,"n_02"] - 4*x[,"n_03"] - 5*x[,"n_04"] - 4*x[,"n_10"] - 4*x[,"n_11"] - 3*x[,"n_12"] - 4*x[,"n_13"] - 4*x[,"n_14"] - 5*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 5*x[,"n_24"] - 4*x[,"n_30"] - 4*x[,"n_31"] - 3*x[,"n_32"] - 4*x[,"n_33"] - 4*x[,"n_34"] - 5*x[,"n_40"] - 4*x[,"n_41"] - 5*x[,"n_42"] - 4*x[,"n_43"] - 5*x[,"n_44"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(3) + (x[,"n_00"] + x[,"n_04"] + x[,"n_40"] + x[,"n_44"])*(log(2 - 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,1 - r_cr2)) + 2*log(pmax(1e-6,r_cr2))) + (x[,"n_01"] + x[,"n_03"] + x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"] + x[,"n_41"] + x[,"n_43"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cr2)) + log(pmax(1e-6,r_cr2)) + log((-1 + r_cr2)^2 + r_cr2^2)) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(1 - 4*r_cr2 + 10*r_cr2^2 - 12*r_cr2^3 + 6*r_cr2^4)) + (x[,"n_11"] + x[,"n_33"])*log(-4*(-2 + 3*p1)*(4 + 3*p2)*(-1 + r_cr2)^2*r_cr2^2 + (4 + 3*p2)*r_cr2^2*(p1*(3 + 6*r_cr2 - 6*r_cr2^2) + 4*(1 - r_cr2 + r_cr2^2)) - (-2 + 3*p1)*(-1 + r_cr2)^2*(8*(1 - r_cr2 + r_cr2^2) + p2*(-3 - 6*r_cr2 + 6*r_cr2^2))) + (x[,"n_13"] + x[,"n_31"])*log(-4*(-2 + 3*p1)*(4 + 3*p2)*(-1 + r_cr2)^2*r_cr2^2 + (4 + 3*p2)*(-1 + r_cr2)^2*(p1*(3 + 6*r_cr2 - 6*r_cr2^2) + 4*(1 - r_cr2 + r_cr2^2)) - (-2 + 3*p1)*r_cr2^2*(8*(1 - r_cr2 + r_cr2^2) + p2*(-3 - 6*r_cr2 + 6*r_cr2^2))) + x[,"n_22"]*log(-5*(-2 + 3*p1)*(4 + 3*p2)*(-1 + r_cr2)^2*r_cr2^2 + (p1*(3 + 6*r_cr2 - 6*r_cr2^2) + 4*(1 - r_cr2 + r_cr2^2))*(8*(1 - r_cr2 + r_cr2^2) + p2*(-3 - 6*r_cr2 + 6*r_cr2^2))) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(log(pmax(1e-6,1 - r_cr2)) + log(pmax(1e-6,r_cr2)) + log(-3*p1*(8*(1 - 3*r_cr2 + 3*r_cr2^2) + 3*p2*(-1 - 6*r_cr2 + 6*r_cr2^2)) + 4*(2*(5 - 6*r_cr2 + 6*r_cr2^2) + p2*(3 - 9*r_cr2 + 9*r_cr2^2))))


logL_cr3 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
L <- (-5*n00 - 4*n01 - 5*n02 - 4*n03 - 5*n04 - 4*n10 - 4*n11 - 3*n12 - 4*n13 - 4*n14 - 5*n20 - 3*n21 - 3*n22 - 3*n23 - 5*n24 - 4*n30 - 4*n31 - 3*n32 - 4*n33 - 4*n34 - 5*n40 - 4*n41 - 5*n42 - 4*n43 - 5*n44)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(3) + (n00 + n04 + n40 + n44)*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n01 + n03 + n10 + n14 + n30 + n34 + n41 + n43)*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log((-1 + r)^2 + r^2)) + (n02 + n20 + n24 + n42)*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 4*r + 10*r^2 - 12*r^3 + 6*r^4)) + (n13 + n31)*log(-4*(4 + 3*p1)*(-2 + 3*p2)*(-1 + r)^2*r^2 + (4 + 3*p1)*r^2*(p2*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2)) - (-2 + 3*p2)*(-1 + r)^2*(8*(1 - r + r^2) + p1*(-3 - 6*r + 6*r^2))) + (n11 + n33)*log(-4*(4 + 3*p1)*(-2 + 3*p2)*(-1 + r)^2*r^2 + (4 + 3*p1)*(-1 + r)^2*(p2*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2)) - (-2 + 3*p2)*r^2*(8*(1 - r + r^2) + p1*(-3 - 6*r + 6*r^2))) + n22*log(-5*(4 + 3*p1)*(-2 + 3*p2)*(-1 + r)^2*r^2 + (p2*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2))*(8*(1 - r + r^2) + p1*(-3 - 6*r + 6*r^2))) + (n12 + n21 + n23 + n32)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(-3*p1*(-4*(1 - 3*r + 3*r^2) + 3*p2*(-1 - 6*r + 6*r^2)) - 8*(-5 + 6*r - 6*r^2 + p2*(3 - 9*r + 9*r^2))))
return(L)}
interlogL_cr3 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
optimize(logL_cr3,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr3 <- parallel::mcmapply(interlogL_cr3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"], p1, p2, mc.cores = ncores)


LOD_cr3 <- -((x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(-2*log10(2) + log10(-3*p1*(-1 - (15*p2)/2) - 8*(-7/2 + (3*p2)/4)))) - (x[,"n_00"] + x[,"n_04"] + x[,"n_40"] + x[,"n_44"])*(-4*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_03"] + x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"] + x[,"n_41"] + x[,"n_43"])*(-3*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(-3*log10(2) + log10(3) + log10(4 + 3*p1) + log10(2 - 3*p2)) - x[,"n_22"]*log10((-5*(4 + 3*p1)*(-2 + 3*p2))/16 + (6 - (9*p1)/2)*(3 + (9*p2)/2)) - (x[,"n_13"] + x[,"n_31"])*log10(-((6 - (9*p1)/2)*(-2 + 3*p2))/4 - ((4 + 3*p1)*(-2 + 3*p2))/4 + ((4 + 3*p1)*(3 + (9*p2)/2))/4) - (x[,"n_11"] + x[,"n_33"])*log10(-((6 - (9*p1)/2)*(-2 + 3*p2))/4 - ((4 + 3*p1)*(-2 + 3*p2))/4 + ((4 + 3*p1)*(3 + (9*p2)/2))/4) + (x[,"n_00"] + x[,"n_04"] + x[,"n_40"] + x[,"n_44"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,1 - r_cr3)) + 2*log10(pmax(1e-6,r_cr3))) + (x[,"n_01"] + x[,"n_03"] + x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"] + x[,"n_41"] + x[,"n_43"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_cr3)) + log10(pmax(1e-6,r_cr3)) + log10((-1 + r_cr3)^2 + r_cr3^2)) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(log10(4 + 3*p1) + log10(2 - 3*p2) + log10(1 - 4*r_cr3 + 10*r_cr3^2 - 12*r_cr3^3 + 6*r_cr3^4)) + (x[,"n_13"] + x[,"n_31"])*log10(-4*(4 + 3*p1)*(-2 + 3*p2)*(-1 + r_cr3)^2*r_cr3^2 + (4 + 3*p1)*r_cr3^2*(p2*(3 + 6*r_cr3 - 6*r_cr3^2) + 4*(1 - r_cr3 + r_cr3^2)) - (-2 + 3*p2)*(-1 + r_cr3)^2*(8*(1 - r_cr3 + r_cr3^2) + p1*(-3 - 6*r_cr3 + 6*r_cr3^2))) + (x[,"n_11"] + x[,"n_33"])*log10(-4*(4 + 3*p1)*(-2 + 3*p2)*(-1 + r_cr3)^2*r_cr3^2 + (4 + 3*p1)*(-1 + r_cr3)^2*(p2*(3 + 6*r_cr3 - 6*r_cr3^2) + 4*(1 - r_cr3 + r_cr3^2)) - (-2 + 3*p2)*r_cr3^2*(8*(1 - r_cr3 + r_cr3^2) + p1*(-3 - 6*r_cr3 + 6*r_cr3^2))) + x[,"n_22"]*log10(-5*(4 + 3*p1)*(-2 + 3*p2)*(-1 + r_cr3)^2*r_cr3^2 + (p2*(3 + 6*r_cr3 - 6*r_cr3^2) + 4*(1 - r_cr3 + r_cr3^2))*(8*(1 - r_cr3 + r_cr3^2) + p1*(-3 - 6*r_cr3 + 6*r_cr3^2))) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(log10(pmax(1e-6,1 - r_cr3)) + log10(pmax(1e-6,r_cr3)) + log10(-3*p1*(-4*(1 - 3*r_cr3 + 3*r_cr3^2) + 3*p2*(-1 - 6*r_cr3 + 6*r_cr3^2)) - 8*(-5 + 6*r_cr3 - 6*r_cr3^2 + p2*(3 - 9*r_cr3 + 9*r_cr3^2))))


logL_cr3 <- (-5*x[,"n_00"] - 4*x[,"n_01"] - 5*x[,"n_02"] - 4*x[,"n_03"] - 5*x[,"n_04"] - 4*x[,"n_10"] - 4*x[,"n_11"] - 3*x[,"n_12"] - 4*x[,"n_13"] - 4*x[,"n_14"] - 5*x[,"n_20"] - 3*x[,"n_21"] - 3*x[,"n_22"] - 3*x[,"n_23"] - 5*x[,"n_24"] - 4*x[,"n_30"] - 4*x[,"n_31"] - 3*x[,"n_32"] - 4*x[,"n_33"] - 4*x[,"n_34"] - 5*x[,"n_40"] - 4*x[,"n_41"] - 5*x[,"n_42"] - 4*x[,"n_43"] - 5*x[,"n_44"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(3) + (x[,"n_00"] + x[,"n_04"] + x[,"n_40"] + x[,"n_44"])*(log(4 + 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r_cr3)) + 2*log(pmax(1e-6,r_cr3))) + (x[,"n_01"] + x[,"n_03"] + x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"] + x[,"n_41"] + x[,"n_43"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_cr3)) + log(pmax(1e-6,r_cr3)) + log((-1 + r_cr3)^2 + r_cr3^2)) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(log(4 + 3*p1) + log(2 - 3*p2) + log(1 - 4*r_cr3 + 10*r_cr3^2 - 12*r_cr3^3 + 6*r_cr3^4)) + (x[,"n_13"] + x[,"n_31"])*log(-4*(4 + 3*p1)*(-2 + 3*p2)*(-1 + r_cr3)^2*r_cr3^2 + (4 + 3*p1)*r_cr3^2*(p2*(3 + 6*r_cr3 - 6*r_cr3^2) + 4*(1 - r_cr3 + r_cr3^2)) - (-2 + 3*p2)*(-1 + r_cr3)^2*(8*(1 - r_cr3 + r_cr3^2) + p1*(-3 - 6*r_cr3 + 6*r_cr3^2))) + (x[,"n_11"] + x[,"n_33"])*log(-4*(4 + 3*p1)*(-2 + 3*p2)*(-1 + r_cr3)^2*r_cr3^2 + (4 + 3*p1)*(-1 + r_cr3)^2*(p2*(3 + 6*r_cr3 - 6*r_cr3^2) + 4*(1 - r_cr3 + r_cr3^2)) - (-2 + 3*p2)*r_cr3^2*(8*(1 - r_cr3 + r_cr3^2) + p1*(-3 - 6*r_cr3 + 6*r_cr3^2))) + x[,"n_22"]*log(-5*(4 + 3*p1)*(-2 + 3*p2)*(-1 + r_cr3)^2*r_cr3^2 + (p2*(3 + 6*r_cr3 - 6*r_cr3^2) + 4*(1 - r_cr3 + r_cr3^2))*(8*(1 - r_cr3 + r_cr3^2) + p1*(-3 - 6*r_cr3 + 6*r_cr3^2))) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(log(pmax(1e-6,1 - r_cr3)) + log(pmax(1e-6,r_cr3)) + log(-3*p1*(-4*(1 - 3*r_cr3 + 3*r_cr3^2) + 3*p2*(-1 - 6*r_cr3 + 6*r_cr3^2)) - 8*(-5 + 6*r_cr3 - 6*r_cr3^2 + p2*(3 - 9*r_cr3 + 9*r_cr3^2))))


logL_cr4 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
L <- (-6*n00 - 5*n01 - 6*n02 - 5*n03 - 6*n04 - 5*n10 - 5*n11 - 4*n12 - 5*n13 - 5*n14 - 6*n20 - 4*n21 - 4*n22 - 4*n23 - 6*n24 - 5*n30 - 5*n31 - 4*n32 - 5*n33 - 5*n34 - 6*n40 - 5*n41 - 6*n42 - 5*n43 - 6*n44)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(3) + (n00 + n04 + n40 + n44)*(log(4 + 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n01 + n03 + n10 + n14 + n30 + n34 + n41 + n43)*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(1 - 2*r + 2*r^2)) + (n02 + n20 + n24 + n42)*(log(4 + 3*p1) + log(4 + 3*p2) + log((-1 + r)^4 + 4*(-1 + r)^2*r^2 + r^4)) + (n11 + n33)*log(4*(4 + 3*p1)*(4 + 3*p2)*(-1 + r)^2*r^2 + (4 + 3*p2)*r^2*(8*(1 - r + r^2) + p1*(-3 - 6*r + 6*r^2)) + (4 + 3*p1)*(-1 + r)^2*(8*(1 - r + r^2) + p2*(-3 - 6*r + 6*r^2))) + (n13 + n31)*log(4*(4 + 3*p1)*(4 + 3*p2)*(-1 + r)^2*r^2 + (4 + 3*p2)*(-1 + r)^2*(8*(1 - r + r^2) + p1*(-3 - 6*r + 6*r^2)) + (4 + 3*p1)*r^2*(8*(1 - r + r^2) + p2*(-3 - 6*r + 6*r^2))) + n22*log(5*(4 + 3*p1)*(4 + 3*p2)*(-1 + r)^2*r^2 + (8*(1 - r + r^2) + p1*(-3 - 6*r + 6*r^2))*(8*(1 - r + r^2) + p2*(-3 - 6*r + 6*r^2))) + (n12 + n21 + n23 + n32)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(3*p1*(8*(1 - 3*r + 3*r^2) + 3*p2*(-1 - 6*r + 6*r^2)) + 8*(2*(5 - 6*r + 6*r^2) + p2*(3 - 9*r + 9*r^2))))
return(L)}
interlogL_cr4 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
optimize(logL_cr4,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_cr4 <- parallel::mcmapply(interlogL_cr4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"], p1, p2, mc.cores = ncores)


LOD_cr4 <- -((x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(-2*log10(2) + log10(3*p1*(2 - (15*p2)/2) + 8*(7 + (3*p2)/4)))) - (x[,"n_00"] + x[,"n_04"] + x[,"n_40"] + x[,"n_44"])*(-4*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_03"] + x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"] + x[,"n_41"] + x[,"n_43"])*(-3*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(-3*log10(2) + log10(3) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_13"] + x[,"n_31"])*log10(((4 + 3*p1)*(6 - (9*p2)/2))/4 + ((6 - (9*p1)/2)*(4 + 3*p2))/4 + ((4 + 3*p1)*(4 + 3*p2))/4) - (x[,"n_11"] + x[,"n_33"])*log10(((4 + 3*p1)*(6 - (9*p2)/2))/4 + ((6 - (9*p1)/2)*(4 + 3*p2))/4 + ((4 + 3*p1)*(4 + 3*p2))/4) - x[,"n_22"]*log10((6 - (9*p1)/2)*(6 - (9*p2)/2) + (5*(4 + 3*p1)*(4 + 3*p2))/16) + (x[,"n_00"] + x[,"n_04"] + x[,"n_40"] + x[,"n_44"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + 2*log10(pmax(1e-6,1 - r_cr4)) + 2*log10(pmax(1e-6,r_cr4))) + (x[,"n_01"] + x[,"n_03"] + x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"] + x[,"n_41"] + x[,"n_43"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_cr4)) + log10(pmax(1e-6,r_cr4)) + log10(1 - 2*r_cr4 + 2*r_cr4^2)) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(log10(4 + 3*p1) + log10(4 + 3*p2) + log10((-1 + r_cr4)^4 + 4*(-1 + r_cr4)^2*r_cr4^2 + r_cr4^4)) + (x[,"n_11"] + x[,"n_33"])*log10(4*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_cr4)^2*r_cr4^2 + (4 + 3*p2)*r_cr4^2*(8*(1 - r_cr4 + r_cr4^2) + p1*(-3 - 6*r_cr4 + 6*r_cr4^2)) + (4 + 3*p1)*(-1 + r_cr4)^2*(8*(1 - r_cr4 + r_cr4^2) + p2*(-3 - 6*r_cr4 + 6*r_cr4^2))) + (x[,"n_13"] + x[,"n_31"])*log10(4*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_cr4)^2*r_cr4^2 + (4 + 3*p2)*(-1 + r_cr4)^2*(8*(1 - r_cr4 + r_cr4^2) + p1*(-3 - 6*r_cr4 + 6*r_cr4^2)) + (4 + 3*p1)*r_cr4^2*(8*(1 - r_cr4 + r_cr4^2) + p2*(-3 - 6*r_cr4 + 6*r_cr4^2))) + x[,"n_22"]*log10(5*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_cr4)^2*r_cr4^2 + (8*(1 - r_cr4 + r_cr4^2) + p1*(-3 - 6*r_cr4 + 6*r_cr4^2))*(8*(1 - r_cr4 + r_cr4^2) + p2*(-3 - 6*r_cr4 + 6*r_cr4^2))) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(log10(pmax(1e-6,1 - r_cr4)) + log10(pmax(1e-6,r_cr4)) + log10(3*p1*(8*(1 - 3*r_cr4 + 3*r_cr4^2) + 3*p2*(-1 - 6*r_cr4 + 6*r_cr4^2)) + 8*(2*(5 - 6*r_cr4 + 6*r_cr4^2) + p2*(3 - 9*r_cr4 + 9*r_cr4^2))))


logL_cr4 <- (-6*x[,"n_00"] - 5*x[,"n_01"] - 6*x[,"n_02"] - 5*x[,"n_03"] - 6*x[,"n_04"] - 5*x[,"n_10"] - 5*x[,"n_11"] - 4*x[,"n_12"] - 5*x[,"n_13"] - 5*x[,"n_14"] - 6*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 4*x[,"n_23"] - 6*x[,"n_24"] - 5*x[,"n_30"] - 5*x[,"n_31"] - 4*x[,"n_32"] - 5*x[,"n_33"] - 5*x[,"n_34"] - 6*x[,"n_40"] - 5*x[,"n_41"] - 6*x[,"n_42"] - 5*x[,"n_43"] - 6*x[,"n_44"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(3) + (x[,"n_00"] + x[,"n_04"] + x[,"n_40"] + x[,"n_44"])*(log(4 + 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,1 - r_cr4)) + 2*log(pmax(1e-6,r_cr4))) + (x[,"n_01"] + x[,"n_03"] + x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"] + x[,"n_41"] + x[,"n_43"])*(log(4 + 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_cr4)) + log(pmax(1e-6,r_cr4)) + log(1 - 2*r_cr4 + 2*r_cr4^2)) + (x[,"n_02"] + x[,"n_20"] + x[,"n_24"] + x[,"n_42"])*(log(4 + 3*p1) + log(4 + 3*p2) + log((-1 + r_cr4)^4 + 4*(-1 + r_cr4)^2*r_cr4^2 + r_cr4^4)) + (x[,"n_11"] + x[,"n_33"])*log(4*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_cr4)^2*r_cr4^2 + (4 + 3*p2)*r_cr4^2*(8*(1 - r_cr4 + r_cr4^2) + p1*(-3 - 6*r_cr4 + 6*r_cr4^2)) + (4 + 3*p1)*(-1 + r_cr4)^2*(8*(1 - r_cr4 + r_cr4^2) + p2*(-3 - 6*r_cr4 + 6*r_cr4^2))) + (x[,"n_13"] + x[,"n_31"])*log(4*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_cr4)^2*r_cr4^2 + (4 + 3*p2)*(-1 + r_cr4)^2*(8*(1 - r_cr4 + r_cr4^2) + p1*(-3 - 6*r_cr4 + 6*r_cr4^2)) + (4 + 3*p1)*r_cr4^2*(8*(1 - r_cr4 + r_cr4^2) + p2*(-3 - 6*r_cr4 + 6*r_cr4^2))) + x[,"n_22"]*log(5*(4 + 3*p1)*(4 + 3*p2)*(-1 + r_cr4)^2*r_cr4^2 + (8*(1 - r_cr4 + r_cr4^2) + p1*(-3 - 6*r_cr4 + 6*r_cr4^2))*(8*(1 - r_cr4 + r_cr4^2) + p2*(-3 - 6*r_cr4 + 6*r_cr4^2))) + (x[,"n_12"] + x[,"n_21"] + x[,"n_23"] + x[,"n_32"])*(log(pmax(1e-6,1 - r_cr4)) + log(pmax(1e-6,r_cr4)) + log(3*p1*(8*(1 - 3*r_cr4 + 3*r_cr4^2) + 3*p2*(-1 - 6*r_cr4 + 6*r_cr4^2)) + 8*(2*(5 - 6*r_cr4 + 6*r_cr4^2) + p2*(3 - 9*r_cr4 + 9*r_cr4^2))))


logL_mc1 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
L <- (-5*n00 - 4*n01 - 5*n02 - 4*n03 - 5*n04 - 5*n10 - 4*n11 - 5*n12 - 4*n13 - 5*n14 - 5*n20 - 4*n21 - 4*n22 - 4*n23 - 5*n24 - 5*n30 - 4*n31 - 5*n32 - 4*n33 - 5*n34 - 5*n40 - 4*n41 - 5*n42 - 4*n43 - 5*n44)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(3) + (n00 + n44)*(log(2 - 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n04 + n40)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 3*log(pmax(1e-6,r))) + (n03 + n41)*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r)) + log(2 - 3*r + 2*r^2)) + (n01 + n43)*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r)) + log(1 - r + 2*r^2)) + (n02 + n42)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(5 - 6*r + 6*r^2)) + (n14 + n30)*(log(2 - 3*p2) + 2*log(pmax(1e-6,r)) + log(8 - 12*r + 8*r^2 - 3*p1*(1 - 6*r + 4*r^2))) + (n10 + n34)*(log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r)) + log(4 - 4*r + 8*r^2 - 3*p1*(-1 - 2*r + 4*r^2))) + (n20 + n24)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(3*p1*(1 + 6*r - 6*r^2) + 2*(5 - 6*r + 6*r^2))) + (n13 + n31)*(log(pmax(1e-6,r)) + log(-8*(-3 + 5*r - 7*r^2 + 4*r^3) + 6*p2*(-3 + 7*r - 14*r^2 + 8*r^3) - 3*p1*(6 + (-20 + 21*p2)*r + (28 - 42*p2)*r^2 + 8*(-2 + 3*p2)*r^3))) + (n11 + n33)*(log(pmax(1e-6,1 - r)) + log(8*(1 + 3*r - 5*r^2 + 4*r^3) - 6*p2*(2 + 3*r - 10*r^2 + 8*r^3) + 3*p1*(2 - 12*r + 20*r^2 - 16*r^3 + 3*p2*(-1 + 3*r - 10*r^2 + 8*r^3)))) + n22*log(2*(16 + 18*r - 54*r^2 + 72*r^3 - 36*r^4 + 3*p2*(4 + 3*r + 15*r^2 - 36*r^3 + 18*r^4)) - 3*p1*(-8 + 30*r - 66*r^2 + 72*r^3 - 36*r^4 + 3*p2*(-2 - 3*r + 21*r^2 - 36*r^3 + 18*r^4))) + (n12 + n32)*log(3*p1*(-14 + 40*r - 88*r^2 + 96*r^3 - 48*r^4 + 3*p2*(-5 - 8*r + 32*r^2 - 48*r^3 + 24*r^4)) - 4*(-2*(5 - 7*r + 19*r^2 - 24*r^3 + 12*r^4) + p2*(-3 - 3*r + 39*r^2 - 72*r^3 + 36*r^4))) + (n21 + n23)*log(6*p2*(1 + r - 13*r^2 + 24*r^3 - 12*r^4) + 4*(5 - 7*r + 19*r^2 - 24*r^3 + 12*r^4) + 3*p1*(2 + 14*r - 38*r^2 + 48*r^3 - 24*r^4 + p2*(6 - 3*r + 39*r^2 - 72*r^3 + 36*r^4)))
return(L)}
interlogL_mc1 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
optimize(logL_mc1,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_mc1 <- parallel::mcmapply(interlogL_mc1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"], p1, p2, mc.cores = ncores)


LOD_mc1 <- -((x[,"n_12"] + x[,"n_32"])*log10(3*p1*(-7 - (33*p2)/2) - 4*(-8 - (3*p2)/2))) - (x[,"n_11"] + x[,"n_33"])*(-log10(2) + log10(14 + 3*p1*(-1 - 3*p2) - 12*p2)) - (x[,"n_04"] + x[,"n_40"])*(-4*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_44"])*(-4*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_03"] + x[,"n_41"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_43"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_42"])*(-3*log10(2) + log10(7) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_14"] + x[,"n_30"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_34"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_20"] + x[,"n_24"])*(-2*log10(2) + log10(7 + (15*p1)/2) + log10(2 - 3*p2)) - (x[,"n_21"] + x[,"n_23"])*log10(16 + 3*p2 + 3*p1*(4 + (15*p2)/2)) - x[,"n_22"]*log10(-3*p1*(-11/4 - (39*p2)/8) + 2*(73/4 + (141*p2)/8)) - (x[,"n_13"] + x[,"n_31"])*(-log10(2) + log10(14 - 12*p2 - 3*p1*(4 + (28 - 42*p2)/4 + 3*p2 + (-20 + 21*p2)/2))) + (x[,"n_00"] + x[,"n_44"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 3*log10(pmax(1e-6,1 - r_mc1)) + log10(pmax(1e-6,r_mc1))) + (x[,"n_04"] + x[,"n_40"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_mc1)) + 3*log10(pmax(1e-6,r_mc1))) + (x[,"n_03"] + x[,"n_41"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_mc1)) + log10(2 - 3*r_mc1 + 2*r_mc1^2)) + (x[,"n_01"] + x[,"n_43"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,1 - r_mc1)) + log10(1 - r_mc1 + 2*r_mc1^2)) + (x[,"n_02"] + x[,"n_42"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_mc1)) + log10(pmax(1e-6,r_mc1)) + log10(5 - 6*r_mc1 + 6*r_mc1^2)) + (x[,"n_14"] + x[,"n_30"])*(log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_mc1)) + log10(8 - 12*r_mc1 + 8*r_mc1^2 - 3*p1*(1 - 6*r_mc1 + 4*r_mc1^2))) + (x[,"n_10"] + x[,"n_34"])*(log10(2 - 3*p2) + 2*log10(pmax(1e-6,1 - r_mc1)) + log10(4 - 4*r_mc1 + 8*r_mc1^2 - 3*p1*(-1 - 2*r_mc1 + 4*r_mc1^2))) + (x[,"n_20"] + x[,"n_24"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_mc1)) + log10(pmax(1e-6,r_mc1)) + log10(3*p1*(1 + 6*r_mc1 - 6*r_mc1^2) + 2*(5 - 6*r_mc1 + 6*r_mc1^2))) + (x[,"n_13"] + x[,"n_31"])*(log10(pmax(1e-6,r_mc1)) + log10(-8*(-3 + 5*r_mc1 - 7*r_mc1^2 + 4*r_mc1^3) + 6*p2*(-3 + 7*r_mc1 - 14*r_mc1^2 + 8*r_mc1^3) - 3*p1*(6 + (-20 + 21*p2)*r_mc1 + (28 - 42*p2)*r_mc1^2 + 8*(-2 + 3*p2)*r_mc1^3))) + (x[,"n_11"] + x[,"n_33"])*(log10(pmax(1e-6,1 - r_mc1)) + log10(8*(1 + 3*r_mc1 - 5*r_mc1^2 + 4*r_mc1^3) - 6*p2*(2 + 3*r_mc1 - 10*r_mc1^2 + 8*r_mc1^3) + 3*p1*(2 - 12*r_mc1 + 20*r_mc1^2 - 16*r_mc1^3 + 3*p2*(-1 + 3*r_mc1 - 10*r_mc1^2 + 8*r_mc1^3)))) + x[,"n_22"]*log10(2*(16 + 18*r_mc1 - 54*r_mc1^2 + 72*r_mc1^3 - 36*r_mc1^4 + 3*p2*(4 + 3*r_mc1 + 15*r_mc1^2 - 36*r_mc1^3 + 18*r_mc1^4)) - 3*p1*(-8 + 30*r_mc1 - 66*r_mc1^2 + 72*r_mc1^3 - 36*r_mc1^4 + 3*p2*(-2 - 3*r_mc1 + 21*r_mc1^2 - 36*r_mc1^3 + 18*r_mc1^4))) + (x[,"n_12"] + x[,"n_32"])*log10(3*p1*(-14 + 40*r_mc1 - 88*r_mc1^2 + 96*r_mc1^3 - 48*r_mc1^4 + 3*p2*(-5 - 8*r_mc1 + 32*r_mc1^2 - 48*r_mc1^3 + 24*r_mc1^4)) - 4*(-2*(5 - 7*r_mc1 + 19*r_mc1^2 - 24*r_mc1^3 + 12*r_mc1^4) + p2*(-3 - 3*r_mc1 + 39*r_mc1^2 - 72*r_mc1^3 + 36*r_mc1^4))) + (x[,"n_21"] + x[,"n_23"])*log10(6*p2*(1 + r_mc1 - 13*r_mc1^2 + 24*r_mc1^3 - 12*r_mc1^4) + 4*(5 - 7*r_mc1 + 19*r_mc1^2 - 24*r_mc1^3 + 12*r_mc1^4) + 3*p1*(2 + 14*r_mc1 - 38*r_mc1^2 + 48*r_mc1^3 - 24*r_mc1^4 + p2*(6 - 3*r_mc1 + 39*r_mc1^2 - 72*r_mc1^3 + 36*r_mc1^4)))


logL_mc1 <- (-5*x[,"n_00"] - 4*x[,"n_01"] - 5*x[,"n_02"] - 4*x[,"n_03"] - 5*x[,"n_04"] - 5*x[,"n_10"] - 4*x[,"n_11"] - 5*x[,"n_12"] - 4*x[,"n_13"] - 5*x[,"n_14"] - 5*x[,"n_20"] - 4*x[,"n_21"] - 4*x[,"n_22"] - 4*x[,"n_23"] - 5*x[,"n_24"] - 5*x[,"n_30"] - 4*x[,"n_31"] - 5*x[,"n_32"] - 4*x[,"n_33"] - 5*x[,"n_34"] - 5*x[,"n_40"] - 4*x[,"n_41"] - 5*x[,"n_42"] - 4*x[,"n_43"] - 5*x[,"n_44"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(3) + (x[,"n_00"] + x[,"n_44"])*(log(2 - 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,1 - r_mc1)) + log(pmax(1e-6,r_mc1))) + (x[,"n_04"] + x[,"n_40"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_mc1)) + 3*log(pmax(1e-6,r_mc1))) + (x[,"n_03"] + x[,"n_41"])*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r_mc1)) + log(2 - 3*r_mc1 + 2*r_mc1^2)) + (x[,"n_01"] + x[,"n_43"])*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r_mc1)) + log(1 - r_mc1 + 2*r_mc1^2)) + (x[,"n_02"] + x[,"n_42"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_mc1)) + log(pmax(1e-6,r_mc1)) + log(5 - 6*r_mc1 + 6*r_mc1^2)) + (x[,"n_14"] + x[,"n_30"])*(log(2 - 3*p2) + 2*log(pmax(1e-6,r_mc1)) + log(8 - 12*r_mc1 + 8*r_mc1^2 - 3*p1*(1 - 6*r_mc1 + 4*r_mc1^2))) + (x[,"n_10"] + x[,"n_34"])*(log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r_mc1)) + log(4 - 4*r_mc1 + 8*r_mc1^2 - 3*p1*(-1 - 2*r_mc1 + 4*r_mc1^2))) + (x[,"n_20"] + x[,"n_24"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_mc1)) + log(pmax(1e-6,r_mc1)) + log(3*p1*(1 + 6*r_mc1 - 6*r_mc1^2) + 2*(5 - 6*r_mc1 + 6*r_mc1^2))) + (x[,"n_13"] + x[,"n_31"])*(log(pmax(1e-6,r_mc1)) + log(-8*(-3 + 5*r_mc1 - 7*r_mc1^2 + 4*r_mc1^3) + 6*p2*(-3 + 7*r_mc1 - 14*r_mc1^2 + 8*r_mc1^3) - 3*p1*(6 + (-20 + 21*p2)*r_mc1 + (28 - 42*p2)*r_mc1^2 + 8*(-2 + 3*p2)*r_mc1^3))) + (x[,"n_11"] + x[,"n_33"])*(log(pmax(1e-6,1 - r_mc1)) + log(8*(1 + 3*r_mc1 - 5*r_mc1^2 + 4*r_mc1^3) - 6*p2*(2 + 3*r_mc1 - 10*r_mc1^2 + 8*r_mc1^3) + 3*p1*(2 - 12*r_mc1 + 20*r_mc1^2 - 16*r_mc1^3 + 3*p2*(-1 + 3*r_mc1 - 10*r_mc1^2 + 8*r_mc1^3)))) + x[,"n_22"]*log(2*(16 + 18*r_mc1 - 54*r_mc1^2 + 72*r_mc1^3 - 36*r_mc1^4 + 3*p2*(4 + 3*r_mc1 + 15*r_mc1^2 - 36*r_mc1^3 + 18*r_mc1^4)) - 3*p1*(-8 + 30*r_mc1 - 66*r_mc1^2 + 72*r_mc1^3 - 36*r_mc1^4 + 3*p2*(-2 - 3*r_mc1 + 21*r_mc1^2 - 36*r_mc1^3 + 18*r_mc1^4))) + (x[,"n_12"] + x[,"n_32"])*log(3*p1*(-14 + 40*r_mc1 - 88*r_mc1^2 + 96*r_mc1^3 - 48*r_mc1^4 + 3*p2*(-5 - 8*r_mc1 + 32*r_mc1^2 - 48*r_mc1^3 + 24*r_mc1^4)) - 4*(-2*(5 - 7*r_mc1 + 19*r_mc1^2 - 24*r_mc1^3 + 12*r_mc1^4) + p2*(-3 - 3*r_mc1 + 39*r_mc1^2 - 72*r_mc1^3 + 36*r_mc1^4))) + (x[,"n_21"] + x[,"n_23"])*log(6*p2*(1 + r_mc1 - 13*r_mc1^2 + 24*r_mc1^3 - 12*r_mc1^4) + 4*(5 - 7*r_mc1 + 19*r_mc1^2 - 24*r_mc1^3 + 12*r_mc1^4) + 3*p1*(2 + 14*r_mc1 - 38*r_mc1^2 + 48*r_mc1^3 - 24*r_mc1^4 + p2*(6 - 3*r_mc1 + 39*r_mc1^2 - 72*r_mc1^3 + 36*r_mc1^4)))


logL_mc2 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
L <- (-6*n00 - 5*n01 - 6*n02 - 5*n03 - 6*n04 - 6*n10 - 5*n11 - 6*n12 - 5*n13 - 6*n14 - 6*n20 - 5*n21 - 5*n22 - 5*n23 - 6*n24 - 6*n30 - 5*n31 - 6*n32 - 5*n33 - 6*n34 - 6*n40 - 5*n41 - 6*n42 - 5*n43 - 6*n44)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(3) + (n00 + n44)*(log(2 - 3*p1) + log(4 + 3*p2) + 3*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n04 + n40)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + 3*log(pmax(1e-6,r))) + (n03 + n41)*(log(2 - 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r)) + log(2 - 3*r + 2*r^2)) + (n01 + n43)*(log(2 - 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,1 - r)) + log(1 - r + 2*r^2)) + (n02 + n42)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(5 - 6*r + 6*r^2)) + (n14 + n30)*(log(4 + 3*p2) + 2*log(pmax(1e-6,r)) + log(8 - 12*r + 8*r^2 - 3*p1*(1 - 6*r + 4*r^2))) + (n10 + n34)*(log(4 + 3*p2) + 2*log(pmax(1e-6,1 - r)) + log(4 - 4*r + 8*r^2 - 3*p1*(-1 - 2*r + 4*r^2))) + (n20 + n24)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(3*p1*(1 + 6*r - 6*r^2) + 2*(5 - 6*r + 6*r^2))) + (n13 + n31)*(log(pmax(1e-6,r)) + log(-16*(-3 + 5*r - 7*r^2 + 4*r^3) - 6*p2*(-3 + 7*r - 14*r^2 + 8*r^3) + 3*p1*(-12 + (40 + 21*p2)*r - 14*(4 + 3*p2)*r^2 + 8*(4 + 3*p2)*r^3))) + (n11 + n33)*(log(pmax(1e-6,1 - r)) + log(2*(8*(1 + 3*r - 5*r^2 + 4*r^3) + 3*p2*(2 + 3*r - 10*r^2 + 8*r^3)) - 3*p1*(3*p2*(-1 + 3*r - 10*r^2 + 8*r^3) + 4*(-1 + 6*r - 10*r^2 + 8*r^3)))) + n22*log(-2*(3*p2*(4 + 3*r + 15*r^2 - 36*r^3 + 18*r^4) + 4*(-8 - 9*r + 27*r^2 - 36*r^3 + 18*r^4)) + 3*p1*(3*p2*(-2 - 3*r + 21*r^2 - 36*r^3 + 18*r^4) + 4*(4 - 15*r + 33*r^2 - 36*r^3 + 18*r^4))) + (n12 + n32)*log(-3*p1*(3*p2*(-5 - 8*r + 32*r^2 - 48*r^3 + 24*r^4) + 4*(7 - 20*r + 44*r^2 - 48*r^3 + 24*r^4)) + 4*(4*(5 - 7*r + 19*r^2 - 24*r^3 + 12*r^4) + p2*(-3 - 3*r + 39*r^2 - 72*r^3 + 36*r^4))) + (n21 + n23)*log(2*(4*(5 - 7*r + 19*r^2 - 24*r^3 + 12*r^4) + p2*(-3 - 3*r + 39*r^2 - 72*r^3 + 36*r^4)) - 3*p1*(-4 - 28*r + 76*r^2 - 96*r^3 + 48*r^4 + p2*(6 - 3*r + 39*r^2 - 72*r^3 + 36*r^4)))
return(L)}
interlogL_mc2 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
optimize(logL_mc2,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_mc2 <- parallel::mcmapply(interlogL_mc2,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"], p1, p2, mc.cores = ncores)


LOD_mc2 <- -((x[,"n_12"] + x[,"n_32"])*log10(-3*p1*(14 - (33*p2)/2) + 4*(16 - (3*p2)/2))) - (x[,"n_04"] + x[,"n_40"])*(-4*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_44"])*(-4*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_03"] + x[,"n_41"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_43"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_42"])*(-3*log10(2) + log10(7) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_14"] + x[,"n_30"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_34"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_20"] + x[,"n_24"])*(-2*log10(2) + log10(7 + (15*p1)/2) + log10(4 + 3*p2)) - (x[,"n_11"] + x[,"n_33"])*(-log10(2) + log10(-3*p1*(2 - 3*p2) + 2*(14 + 6*p2))) - (x[,"n_21"] + x[,"n_23"])*log10(2*(16 - (3*p2)/2) - 3*p1*(-8 + (15*p2)/2)) - x[,"n_22"]*log10(3*p1*(11/2 - (39*p2)/8) - 2*(-73/2 + (141*p2)/8)) - (x[,"n_13"] + x[,"n_31"])*(-log10(2) + log10(28 + 12*p2 + 3*p1*(-8 + 3*p2 - (7*(4 + 3*p2))/2 + (40 + 21*p2)/2))) + (x[,"n_00"] + x[,"n_44"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + 3*log10(pmax(1e-6,1 - r_mc2)) + log10(pmax(1e-6,r_mc2))) + (x[,"n_04"] + x[,"n_40"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_mc2)) + 3*log10(pmax(1e-6,r_mc2))) + (x[,"n_03"] + x[,"n_41"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_mc2)) + log10(2 - 3*r_mc2 + 2*r_mc2^2)) + (x[,"n_01"] + x[,"n_43"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + 2*log10(pmax(1e-6,1 - r_mc2)) + log10(1 - r_mc2 + 2*r_mc2^2)) + (x[,"n_02"] + x[,"n_42"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_mc2)) + log10(pmax(1e-6,r_mc2)) + log10(5 - 6*r_mc2 + 6*r_mc2^2)) + (x[,"n_14"] + x[,"n_30"])*(log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_mc2)) + log10(8 - 12*r_mc2 + 8*r_mc2^2 - 3*p1*(1 - 6*r_mc2 + 4*r_mc2^2))) + (x[,"n_10"] + x[,"n_34"])*(log10(4 + 3*p2) + 2*log10(pmax(1e-6,1 - r_mc2)) + log10(4 - 4*r_mc2 + 8*r_mc2^2 - 3*p1*(-1 - 2*r_mc2 + 4*r_mc2^2))) + (x[,"n_20"] + x[,"n_24"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_mc2)) + log10(pmax(1e-6,r_mc2)) + log10(3*p1*(1 + 6*r_mc2 - 6*r_mc2^2) + 2*(5 - 6*r_mc2 + 6*r_mc2^2))) + (x[,"n_13"] + x[,"n_31"])*(log10(pmax(1e-6,r_mc2)) + log10(-16*(-3 + 5*r_mc2 - 7*r_mc2^2 + 4*r_mc2^3) - 6*p2*(-3 + 7*r_mc2 - 14*r_mc2^2 + 8*r_mc2^3) + 3*p1*(-12 + (40 + 21*p2)*r_mc2 - 14*(4 + 3*p2)*r_mc2^2 + 8*(4 + 3*p2)*r_mc2^3))) + (x[,"n_11"] + x[,"n_33"])*(log10(pmax(1e-6,1 - r_mc2)) + log10(2*(8*(1 + 3*r_mc2 - 5*r_mc2^2 + 4*r_mc2^3) + 3*p2*(2 + 3*r_mc2 - 10*r_mc2^2 + 8*r_mc2^3)) - 3*p1*(3*p2*(-1 + 3*r_mc2 - 10*r_mc2^2 + 8*r_mc2^3) + 4*(-1 + 6*r_mc2 - 10*r_mc2^2 + 8*r_mc2^3)))) + x[,"n_22"]*log10(-2*(3*p2*(4 + 3*r_mc2 + 15*r_mc2^2 - 36*r_mc2^3 + 18*r_mc2^4) + 4*(-8 - 9*r_mc2 + 27*r_mc2^2 - 36*r_mc2^3 + 18*r_mc2^4)) + 3*p1*(3*p2*(-2 - 3*r_mc2 + 21*r_mc2^2 - 36*r_mc2^3 + 18*r_mc2^4) + 4*(4 - 15*r_mc2 + 33*r_mc2^2 - 36*r_mc2^3 + 18*r_mc2^4))) + (x[,"n_12"] + x[,"n_32"])*log10(-3*p1*(3*p2*(-5 - 8*r_mc2 + 32*r_mc2^2 - 48*r_mc2^3 + 24*r_mc2^4) + 4*(7 - 20*r_mc2 + 44*r_mc2^2 - 48*r_mc2^3 + 24*r_mc2^4)) + 4*(4*(5 - 7*r_mc2 + 19*r_mc2^2 - 24*r_mc2^3 + 12*r_mc2^4) + p2*(-3 - 3*r_mc2 + 39*r_mc2^2 - 72*r_mc2^3 + 36*r_mc2^4))) + (x[,"n_21"] + x[,"n_23"])*log10(2*(4*(5 - 7*r_mc2 + 19*r_mc2^2 - 24*r_mc2^3 + 12*r_mc2^4) + p2*(-3 - 3*r_mc2 + 39*r_mc2^2 - 72*r_mc2^3 + 36*r_mc2^4)) - 3*p1*(-4 - 28*r_mc2 + 76*r_mc2^2 - 96*r_mc2^3 + 48*r_mc2^4 + p2*(6 - 3*r_mc2 + 39*r_mc2^2 - 72*r_mc2^3 + 36*r_mc2^4)))


logL_mc2 <- (-6*x[,"n_00"] - 5*x[,"n_01"] - 6*x[,"n_02"] - 5*x[,"n_03"] - 6*x[,"n_04"] - 6*x[,"n_10"] - 5*x[,"n_11"] - 6*x[,"n_12"] - 5*x[,"n_13"] - 6*x[,"n_14"] - 6*x[,"n_20"] - 5*x[,"n_21"] - 5*x[,"n_22"] - 5*x[,"n_23"] - 6*x[,"n_24"] - 6*x[,"n_30"] - 5*x[,"n_31"] - 6*x[,"n_32"] - 5*x[,"n_33"] - 6*x[,"n_34"] - 6*x[,"n_40"] - 5*x[,"n_41"] - 6*x[,"n_42"] - 5*x[,"n_43"] - 6*x[,"n_44"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(3) + (x[,"n_00"] + x[,"n_44"])*(log(2 - 3*p1) + log(4 + 3*p2) + 3*log(pmax(1e-6,1 - r_mc2)) + log(pmax(1e-6,r_mc2))) + (x[,"n_04"] + x[,"n_40"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_mc2)) + 3*log(pmax(1e-6,r_mc2))) + (x[,"n_03"] + x[,"n_41"])*(log(2 - 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r_mc2)) + log(2 - 3*r_mc2 + 2*r_mc2^2)) + (x[,"n_01"] + x[,"n_43"])*(log(2 - 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,1 - r_mc2)) + log(1 - r_mc2 + 2*r_mc2^2)) + (x[,"n_02"] + x[,"n_42"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_mc2)) + log(pmax(1e-6,r_mc2)) + log(5 - 6*r_mc2 + 6*r_mc2^2)) + (x[,"n_14"] + x[,"n_30"])*(log(4 + 3*p2) + 2*log(pmax(1e-6,r_mc2)) + log(8 - 12*r_mc2 + 8*r_mc2^2 - 3*p1*(1 - 6*r_mc2 + 4*r_mc2^2))) + (x[,"n_10"] + x[,"n_34"])*(log(4 + 3*p2) + 2*log(pmax(1e-6,1 - r_mc2)) + log(4 - 4*r_mc2 + 8*r_mc2^2 - 3*p1*(-1 - 2*r_mc2 + 4*r_mc2^2))) + (x[,"n_20"] + x[,"n_24"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_mc2)) + log(pmax(1e-6,r_mc2)) + log(3*p1*(1 + 6*r_mc2 - 6*r_mc2^2) + 2*(5 - 6*r_mc2 + 6*r_mc2^2))) + (x[,"n_13"] + x[,"n_31"])*(log(pmax(1e-6,r_mc2)) + log(-16*(-3 + 5*r_mc2 - 7*r_mc2^2 + 4*r_mc2^3) - 6*p2*(-3 + 7*r_mc2 - 14*r_mc2^2 + 8*r_mc2^3) + 3*p1*(-12 + (40 + 21*p2)*r_mc2 - 14*(4 + 3*p2)*r_mc2^2 + 8*(4 + 3*p2)*r_mc2^3))) + (x[,"n_11"] + x[,"n_33"])*(log(pmax(1e-6,1 - r_mc2)) + log(2*(8*(1 + 3*r_mc2 - 5*r_mc2^2 + 4*r_mc2^3) + 3*p2*(2 + 3*r_mc2 - 10*r_mc2^2 + 8*r_mc2^3)) - 3*p1*(3*p2*(-1 + 3*r_mc2 - 10*r_mc2^2 + 8*r_mc2^3) + 4*(-1 + 6*r_mc2 - 10*r_mc2^2 + 8*r_mc2^3)))) + x[,"n_22"]*log(-2*(3*p2*(4 + 3*r_mc2 + 15*r_mc2^2 - 36*r_mc2^3 + 18*r_mc2^4) + 4*(-8 - 9*r_mc2 + 27*r_mc2^2 - 36*r_mc2^3 + 18*r_mc2^4)) + 3*p1*(3*p2*(-2 - 3*r_mc2 + 21*r_mc2^2 - 36*r_mc2^3 + 18*r_mc2^4) + 4*(4 - 15*r_mc2 + 33*r_mc2^2 - 36*r_mc2^3 + 18*r_mc2^4))) + (x[,"n_12"] + x[,"n_32"])*log(-3*p1*(3*p2*(-5 - 8*r_mc2 + 32*r_mc2^2 - 48*r_mc2^3 + 24*r_mc2^4) + 4*(7 - 20*r_mc2 + 44*r_mc2^2 - 48*r_mc2^3 + 24*r_mc2^4)) + 4*(4*(5 - 7*r_mc2 + 19*r_mc2^2 - 24*r_mc2^3 + 12*r_mc2^4) + p2*(-3 - 3*r_mc2 + 39*r_mc2^2 - 72*r_mc2^3 + 36*r_mc2^4))) + (x[,"n_21"] + x[,"n_23"])*log(2*(4*(5 - 7*r_mc2 + 19*r_mc2^2 - 24*r_mc2^3 + 12*r_mc2^4) + p2*(-3 - 3*r_mc2 + 39*r_mc2^2 - 72*r_mc2^3 + 36*r_mc2^4)) - 3*p1*(-4 - 28*r_mc2 + 76*r_mc2^2 - 96*r_mc2^3 + 48*r_mc2^4 + p2*(6 - 3*r_mc2 + 39*r_mc2^2 - 72*r_mc2^3 + 36*r_mc2^4)))


logL_mc3 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
L <- (-5*n00 - 5*n01 - 5*n02 - 5*n03 - 5*n04 - 4*n10 - 4*n11 - 4*n12 - 4*n13 - 4*n14 - 5*n20 - 5*n21 - 4*n22 - 5*n23 - 5*n24 - 4*n30 - 4*n31 - 4*n32 - 4*n33 - 4*n34 - 5*n40 - 5*n41 - 5*n42 - 5*n43 - 5*n44)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(3) + (n00 + n44)*(log(2 - 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n04 + n40)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + 3*log(pmax(1e-6,r))) + (n14 + n30)*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r)) + log(2 - 3*r + 2*r^2)) + (n10 + n34)*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r)) + log(1 - r + 2*r^2)) + (n20 + n24)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(5 - 6*r + 6*r^2)) + (n03 + n41)*(log(2 - 3*p2) + 2*log(pmax(1e-6,r)) + log(8 - 12*r + 8*r^2 - 3*p1*(1 - 6*r + 4*r^2))) + (n01 + n43)*(log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r)) + log(4 - 4*r + 8*r^2 - 3*p1*(-1 - 2*r + 4*r^2))) + (n02 + n42)*(log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(3*p1*(1 + 6*r - 6*r^2) + 2*(5 - 6*r + 6*r^2))) + (n13 + n31)*(log(pmax(1e-6,r)) + log(-8*(-3 + 5*r - 7*r^2 + 4*r^3) + 6*p2*(-3 + 7*r - 14*r^2 + 8*r^3) - 3*p1*(6 + (-20 + 21*p2)*r + (28 - 42*p2)*r^2 + 8*(-2 + 3*p2)*r^3))) + (n11 + n33)*(log(pmax(1e-6,1 - r)) + log(8*(1 + 3*r - 5*r^2 + 4*r^3) - 6*p2*(2 + 3*r - 10*r^2 + 8*r^3) + 3*p1*(2 - 12*r + 20*r^2 - 16*r^3 + 3*p2*(-1 + 3*r - 10*r^2 + 8*r^3)))) + n22*log(2*(16 + 18*r - 54*r^2 + 72*r^3 - 36*r^4 + 3*p2*(4 + 3*r + 15*r^2 - 36*r^3 + 18*r^4)) - 3*p1*(-8 + 30*r - 66*r^2 + 72*r^3 - 36*r^4 + 3*p2*(-2 - 3*r + 21*r^2 - 36*r^3 + 18*r^4))) + (n21 + n23)*log(3*p1*(-14 + 40*r - 88*r^2 + 96*r^3 - 48*r^4 + 3*p2*(-5 - 8*r + 32*r^2 - 48*r^3 + 24*r^4)) - 4*(-2*(5 - 7*r + 19*r^2 - 24*r^3 + 12*r^4) + p2*(-3 - 3*r + 39*r^2 - 72*r^3 + 36*r^4))) + (n12 + n32)*log(6*p2*(1 + r - 13*r^2 + 24*r^3 - 12*r^4) + 4*(5 - 7*r + 19*r^2 - 24*r^3 + 12*r^4) + 3*p1*(2 + 14*r - 38*r^2 + 48*r^3 - 24*r^4 + p2*(6 - 3*r + 39*r^2 - 72*r^3 + 36*r^4)))
return(L)}
interlogL_mc3 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
optimize(logL_mc3,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_mc3 <- parallel::mcmapply(interlogL_mc3,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"], p1, p2, mc.cores = ncores)


LOD_mc3 <- -((x[,"n_21"] + x[,"n_23"])*log10(3*p1*(-7 - (33*p2)/2) - 4*(-8 - (3*p2)/2))) - (x[,"n_11"] + x[,"n_33"])*(-log10(2) + log10(14 + 3*p1*(-1 - 3*p2) - 12*p2)) - (x[,"n_04"] + x[,"n_40"])*(-4*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_00"] + x[,"n_44"])*(-4*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_14"] + x[,"n_30"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_10"] + x[,"n_34"])*(-2*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_20"] + x[,"n_24"])*(-3*log10(2) + log10(7) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_03"] + x[,"n_41"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_01"] + x[,"n_43"])*(-2*log10(2) + log10(4 + 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_42"])*(-2*log10(2) + log10(7 + (15*p1)/2) + log10(2 - 3*p2)) - (x[,"n_12"] + x[,"n_32"])*log10(16 + 3*p2 + 3*p1*(4 + (15*p2)/2)) - x[,"n_22"]*log10(-3*p1*(-11/4 - (39*p2)/8) + 2*(73/4 + (141*p2)/8)) - (x[,"n_13"] + x[,"n_31"])*(-log10(2) + log10(14 - 12*p2 - 3*p1*(4 + (28 - 42*p2)/4 + 3*p2 + (-20 + 21*p2)/2))) + (x[,"n_00"] + x[,"n_44"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 3*log10(pmax(1e-6,1 - r_mc3)) + log10(pmax(1e-6,r_mc3))) + (x[,"n_04"] + x[,"n_40"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_mc3)) + 3*log10(pmax(1e-6,r_mc3))) + (x[,"n_14"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_mc3)) + log10(2 - 3*r_mc3 + 2*r_mc3^2)) + (x[,"n_10"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,1 - r_mc3)) + log10(1 - r_mc3 + 2*r_mc3^2)) + (x[,"n_20"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_mc3)) + log10(pmax(1e-6,r_mc3)) + log10(5 - 6*r_mc3 + 6*r_mc3^2)) + (x[,"n_03"] + x[,"n_41"])*(log10(2 - 3*p2) + 2*log10(pmax(1e-6,r_mc3)) + log10(8 - 12*r_mc3 + 8*r_mc3^2 - 3*p1*(1 - 6*r_mc3 + 4*r_mc3^2))) + (x[,"n_01"] + x[,"n_43"])*(log10(2 - 3*p2) + 2*log10(pmax(1e-6,1 - r_mc3)) + log10(4 - 4*r_mc3 + 8*r_mc3^2 - 3*p1*(-1 - 2*r_mc3 + 4*r_mc3^2))) + (x[,"n_02"] + x[,"n_42"])*(log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_mc3)) + log10(pmax(1e-6,r_mc3)) + log10(3*p1*(1 + 6*r_mc3 - 6*r_mc3^2) + 2*(5 - 6*r_mc3 + 6*r_mc3^2))) + (x[,"n_13"] + x[,"n_31"])*(log10(pmax(1e-6,r_mc3)) + log10(-8*(-3 + 5*r_mc3 - 7*r_mc3^2 + 4*r_mc3^3) + 6*p2*(-3 + 7*r_mc3 - 14*r_mc3^2 + 8*r_mc3^3) - 3*p1*(6 + (-20 + 21*p2)*r_mc3 + (28 - 42*p2)*r_mc3^2 + 8*(-2 + 3*p2)*r_mc3^3))) + (x[,"n_11"] + x[,"n_33"])*(log10(pmax(1e-6,1 - r_mc3)) + log10(8*(1 + 3*r_mc3 - 5*r_mc3^2 + 4*r_mc3^3) - 6*p2*(2 + 3*r_mc3 - 10*r_mc3^2 + 8*r_mc3^3) + 3*p1*(2 - 12*r_mc3 + 20*r_mc3^2 - 16*r_mc3^3 + 3*p2*(-1 + 3*r_mc3 - 10*r_mc3^2 + 8*r_mc3^3)))) + x[,"n_22"]*log10(2*(16 + 18*r_mc3 - 54*r_mc3^2 + 72*r_mc3^3 - 36*r_mc3^4 + 3*p2*(4 + 3*r_mc3 + 15*r_mc3^2 - 36*r_mc3^3 + 18*r_mc3^4)) - 3*p1*(-8 + 30*r_mc3 - 66*r_mc3^2 + 72*r_mc3^3 - 36*r_mc3^4 + 3*p2*(-2 - 3*r_mc3 + 21*r_mc3^2 - 36*r_mc3^3 + 18*r_mc3^4))) + (x[,"n_21"] + x[,"n_23"])*log10(3*p1*(-14 + 40*r_mc3 - 88*r_mc3^2 + 96*r_mc3^3 - 48*r_mc3^4 + 3*p2*(-5 - 8*r_mc3 + 32*r_mc3^2 - 48*r_mc3^3 + 24*r_mc3^4)) - 4*(-2*(5 - 7*r_mc3 + 19*r_mc3^2 - 24*r_mc3^3 + 12*r_mc3^4) + p2*(-3 - 3*r_mc3 + 39*r_mc3^2 - 72*r_mc3^3 + 36*r_mc3^4))) + (x[,"n_12"] + x[,"n_32"])*log10(6*p2*(1 + r_mc3 - 13*r_mc3^2 + 24*r_mc3^3 - 12*r_mc3^4) + 4*(5 - 7*r_mc3 + 19*r_mc3^2 - 24*r_mc3^3 + 12*r_mc3^4) + 3*p1*(2 + 14*r_mc3 - 38*r_mc3^2 + 48*r_mc3^3 - 24*r_mc3^4 + p2*(6 - 3*r_mc3 + 39*r_mc3^2 - 72*r_mc3^3 + 36*r_mc3^4)))


logL_mc3 <- (-5*x[,"n_00"] - 5*x[,"n_01"] - 5*x[,"n_02"] - 5*x[,"n_03"] - 5*x[,"n_04"] - 4*x[,"n_10"] - 4*x[,"n_11"] - 4*x[,"n_12"] - 4*x[,"n_13"] - 4*x[,"n_14"] - 5*x[,"n_20"] - 5*x[,"n_21"] - 4*x[,"n_22"] - 5*x[,"n_23"] - 5*x[,"n_24"] - 4*x[,"n_30"] - 4*x[,"n_31"] - 4*x[,"n_32"] - 4*x[,"n_33"] - 4*x[,"n_34"] - 5*x[,"n_40"] - 5*x[,"n_41"] - 5*x[,"n_42"] - 5*x[,"n_43"] - 5*x[,"n_44"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(3) + (x[,"n_00"] + x[,"n_44"])*(log(2 - 3*p1) + log(2 - 3*p2) + 3*log(pmax(1e-6,1 - r_mc3)) + log(pmax(1e-6,r_mc3))) + (x[,"n_04"] + x[,"n_40"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_mc3)) + 3*log(pmax(1e-6,r_mc3))) + (x[,"n_14"] + x[,"n_30"])*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,r_mc3)) + log(2 - 3*r_mc3 + 2*r_mc3^2)) + (x[,"n_10"] + x[,"n_34"])*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r_mc3)) + log(1 - r_mc3 + 2*r_mc3^2)) + (x[,"n_20"] + x[,"n_24"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_mc3)) + log(pmax(1e-6,r_mc3)) + log(5 - 6*r_mc3 + 6*r_mc3^2)) + (x[,"n_03"] + x[,"n_41"])*(log(2 - 3*p2) + 2*log(pmax(1e-6,r_mc3)) + log(8 - 12*r_mc3 + 8*r_mc3^2 - 3*p1*(1 - 6*r_mc3 + 4*r_mc3^2))) + (x[,"n_01"] + x[,"n_43"])*(log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r_mc3)) + log(4 - 4*r_mc3 + 8*r_mc3^2 - 3*p1*(-1 - 2*r_mc3 + 4*r_mc3^2))) + (x[,"n_02"] + x[,"n_42"])*(log(2 - 3*p2) + log(pmax(1e-6,1 - r_mc3)) + log(pmax(1e-6,r_mc3)) + log(3*p1*(1 + 6*r_mc3 - 6*r_mc3^2) + 2*(5 - 6*r_mc3 + 6*r_mc3^2))) + (x[,"n_13"] + x[,"n_31"])*(log(pmax(1e-6,r_mc3)) + log(-8*(-3 + 5*r_mc3 - 7*r_mc3^2 + 4*r_mc3^3) + 6*p2*(-3 + 7*r_mc3 - 14*r_mc3^2 + 8*r_mc3^3) - 3*p1*(6 + (-20 + 21*p2)*r_mc3 + (28 - 42*p2)*r_mc3^2 + 8*(-2 + 3*p2)*r_mc3^3))) + (x[,"n_11"] + x[,"n_33"])*(log(pmax(1e-6,1 - r_mc3)) + log(8*(1 + 3*r_mc3 - 5*r_mc3^2 + 4*r_mc3^3) - 6*p2*(2 + 3*r_mc3 - 10*r_mc3^2 + 8*r_mc3^3) + 3*p1*(2 - 12*r_mc3 + 20*r_mc3^2 - 16*r_mc3^3 + 3*p2*(-1 + 3*r_mc3 - 10*r_mc3^2 + 8*r_mc3^3)))) + x[,"n_22"]*log(2*(16 + 18*r_mc3 - 54*r_mc3^2 + 72*r_mc3^3 - 36*r_mc3^4 + 3*p2*(4 + 3*r_mc3 + 15*r_mc3^2 - 36*r_mc3^3 + 18*r_mc3^4)) - 3*p1*(-8 + 30*r_mc3 - 66*r_mc3^2 + 72*r_mc3^3 - 36*r_mc3^4 + 3*p2*(-2 - 3*r_mc3 + 21*r_mc3^2 - 36*r_mc3^3 + 18*r_mc3^4))) + (x[,"n_21"] + x[,"n_23"])*log(3*p1*(-14 + 40*r_mc3 - 88*r_mc3^2 + 96*r_mc3^3 - 48*r_mc3^4 + 3*p2*(-5 - 8*r_mc3 + 32*r_mc3^2 - 48*r_mc3^3 + 24*r_mc3^4)) - 4*(-2*(5 - 7*r_mc3 + 19*r_mc3^2 - 24*r_mc3^3 + 12*r_mc3^4) + p2*(-3 - 3*r_mc3 + 39*r_mc3^2 - 72*r_mc3^3 + 36*r_mc3^4))) + (x[,"n_12"] + x[,"n_32"])*log(6*p2*(1 + r_mc3 - 13*r_mc3^2 + 24*r_mc3^3 - 12*r_mc3^4) + 4*(5 - 7*r_mc3 + 19*r_mc3^2 - 24*r_mc3^3 + 12*r_mc3^4) + 3*p1*(2 + 14*r_mc3 - 38*r_mc3^2 + 48*r_mc3^3 - 24*r_mc3^4 + p2*(6 - 3*r_mc3 + 39*r_mc3^2 - 72*r_mc3^3 + 36*r_mc3^4)))


logL_mc4 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
L <- (-6*n00 - 6*n01 - 6*n02 - 6*n03 - 6*n04 - 5*n10 - 5*n11 - 5*n12 - 5*n13 - 5*n14 - 6*n20 - 6*n21 - 5*n22 - 6*n23 - 6*n24 - 5*n30 - 5*n31 - 5*n32 - 5*n33 - 5*n34 - 6*n40 - 6*n41 - 6*n42 - 6*n43 - 6*n44)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(3) + (n00 + n44)*(log(2 - 3*p1) + log(4 + 3*p2) + 3*log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r))) + (n04 + n40)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + 3*log(pmax(1e-6,r))) + (n14 + n30)*(log(2 - 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r)) + log(2 - 3*r + 2*r^2)) + (n10 + n34)*(log(2 - 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,1 - r)) + log(1 - r + 2*r^2)) + (n20 + n24)*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(5 - 6*r + 6*r^2)) + (n03 + n41)*(log(4 + 3*p2) + 2*log(pmax(1e-6,r)) + log(8 - 12*r + 8*r^2 - 3*p1*(1 - 6*r + 4*r^2))) + (n01 + n43)*(log(4 + 3*p2) + 2*log(pmax(1e-6,1 - r)) + log(4 - 4*r + 8*r^2 - 3*p1*(-1 - 2*r + 4*r^2))) + (n02 + n42)*(log(4 + 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(3*p1*(1 + 6*r - 6*r^2) + 2*(5 - 6*r + 6*r^2))) + (n13 + n31)*(log(pmax(1e-6,r)) + log(-16*(-3 + 5*r - 7*r^2 + 4*r^3) - 6*p2*(-3 + 7*r - 14*r^2 + 8*r^3) + 3*p1*(-12 + (40 + 21*p2)*r - 14*(4 + 3*p2)*r^2 + 8*(4 + 3*p2)*r^3))) + (n11 + n33)*(log(pmax(1e-6,1 - r)) + log(2*(8*(1 + 3*r - 5*r^2 + 4*r^3) + 3*p2*(2 + 3*r - 10*r^2 + 8*r^3)) - 3*p1*(3*p2*(-1 + 3*r - 10*r^2 + 8*r^3) + 4*(-1 + 6*r - 10*r^2 + 8*r^3)))) + n22*log(-2*(3*p2*(4 + 3*r + 15*r^2 - 36*r^3 + 18*r^4) + 4*(-8 - 9*r + 27*r^2 - 36*r^3 + 18*r^4)) + 3*p1*(3*p2*(-2 - 3*r + 21*r^2 - 36*r^3 + 18*r^4) + 4*(4 - 15*r + 33*r^2 - 36*r^3 + 18*r^4))) + (n21 + n23)*log(-3*p1*(3*p2*(-5 - 8*r + 32*r^2 - 48*r^3 + 24*r^4) + 4*(7 - 20*r + 44*r^2 - 48*r^3 + 24*r^4)) + 4*(4*(5 - 7*r + 19*r^2 - 24*r^3 + 12*r^4) + p2*(-3 - 3*r + 39*r^2 - 72*r^3 + 36*r^4))) + (n12 + n32)*log(2*(4*(5 - 7*r + 19*r^2 - 24*r^3 + 12*r^4) + p2*(-3 - 3*r + 39*r^2 - 72*r^3 + 36*r^4)) - 3*p1*(-4 - 28*r + 76*r^2 - 96*r^3 + 48*r^4 + p2*(6 - 3*r + 39*r^2 - 72*r^3 + 36*r^4)))
return(L)}
interlogL_mc4 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
optimize(logL_mc4,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_mc4 <- parallel::mcmapply(interlogL_mc4,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"], p1, p2, mc.cores = ncores)


LOD_mc4 <- -((x[,"n_21"] + x[,"n_23"])*log10(-3*p1*(14 - (33*p2)/2) + 4*(16 - (3*p2)/2))) - (x[,"n_04"] + x[,"n_40"])*(-4*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_00"] + x[,"n_44"])*(-4*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_14"] + x[,"n_30"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_10"] + x[,"n_34"])*(-2*log10(2) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_20"] + x[,"n_24"])*(-3*log10(2) + log10(7) + log10(2 - 3*p1) + log10(4 + 3*p2)) - (x[,"n_03"] + x[,"n_41"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_01"] + x[,"n_43"])*(-2*log10(2) + log10(4 + 3*p1) + log10(4 + 3*p2)) - (x[,"n_02"] + x[,"n_42"])*(-2*log10(2) + log10(7 + (15*p1)/2) + log10(4 + 3*p2)) - (x[,"n_11"] + x[,"n_33"])*(-log10(2) + log10(-3*p1*(2 - 3*p2) + 2*(14 + 6*p2))) - (x[,"n_12"] + x[,"n_32"])*log10(2*(16 - (3*p2)/2) - 3*p1*(-8 + (15*p2)/2)) - x[,"n_22"]*log10(3*p1*(11/2 - (39*p2)/8) - 2*(-73/2 + (141*p2)/8)) - (x[,"n_13"] + x[,"n_31"])*(-log10(2) + log10(28 + 12*p2 + 3*p1*(-8 + 3*p2 - (7*(4 + 3*p2))/2 + (40 + 21*p2)/2))) + (x[,"n_00"] + x[,"n_44"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + 3*log10(pmax(1e-6,1 - r_mc4)) + log10(pmax(1e-6,r_mc4))) + (x[,"n_04"] + x[,"n_40"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_mc4)) + 3*log10(pmax(1e-6,r_mc4))) + (x[,"n_14"] + x[,"n_30"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_mc4)) + log10(2 - 3*r_mc4 + 2*r_mc4^2)) + (x[,"n_10"] + x[,"n_34"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + 2*log10(pmax(1e-6,1 - r_mc4)) + log10(1 - r_mc4 + 2*r_mc4^2)) + (x[,"n_20"] + x[,"n_24"])*(log10(2 - 3*p1) + log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_mc4)) + log10(pmax(1e-6,r_mc4)) + log10(5 - 6*r_mc4 + 6*r_mc4^2)) + (x[,"n_03"] + x[,"n_41"])*(log10(4 + 3*p2) + 2*log10(pmax(1e-6,r_mc4)) + log10(8 - 12*r_mc4 + 8*r_mc4^2 - 3*p1*(1 - 6*r_mc4 + 4*r_mc4^2))) + (x[,"n_01"] + x[,"n_43"])*(log10(4 + 3*p2) + 2*log10(pmax(1e-6,1 - r_mc4)) + log10(4 - 4*r_mc4 + 8*r_mc4^2 - 3*p1*(-1 - 2*r_mc4 + 4*r_mc4^2))) + (x[,"n_02"] + x[,"n_42"])*(log10(4 + 3*p2) + log10(pmax(1e-6,1 - r_mc4)) + log10(pmax(1e-6,r_mc4)) + log10(3*p1*(1 + 6*r_mc4 - 6*r_mc4^2) + 2*(5 - 6*r_mc4 + 6*r_mc4^2))) + (x[,"n_13"] + x[,"n_31"])*(log10(pmax(1e-6,r_mc4)) + log10(-16*(-3 + 5*r_mc4 - 7*r_mc4^2 + 4*r_mc4^3) - 6*p2*(-3 + 7*r_mc4 - 14*r_mc4^2 + 8*r_mc4^3) + 3*p1*(-12 + (40 + 21*p2)*r_mc4 - 14*(4 + 3*p2)*r_mc4^2 + 8*(4 + 3*p2)*r_mc4^3))) + (x[,"n_11"] + x[,"n_33"])*(log10(pmax(1e-6,1 - r_mc4)) + log10(2*(8*(1 + 3*r_mc4 - 5*r_mc4^2 + 4*r_mc4^3) + 3*p2*(2 + 3*r_mc4 - 10*r_mc4^2 + 8*r_mc4^3)) - 3*p1*(3*p2*(-1 + 3*r_mc4 - 10*r_mc4^2 + 8*r_mc4^3) + 4*(-1 + 6*r_mc4 - 10*r_mc4^2 + 8*r_mc4^3)))) + x[,"n_22"]*log10(-2*(3*p2*(4 + 3*r_mc4 + 15*r_mc4^2 - 36*r_mc4^3 + 18*r_mc4^4) + 4*(-8 - 9*r_mc4 + 27*r_mc4^2 - 36*r_mc4^3 + 18*r_mc4^4)) + 3*p1*(3*p2*(-2 - 3*r_mc4 + 21*r_mc4^2 - 36*r_mc4^3 + 18*r_mc4^4) + 4*(4 - 15*r_mc4 + 33*r_mc4^2 - 36*r_mc4^3 + 18*r_mc4^4))) + (x[,"n_21"] + x[,"n_23"])*log10(-3*p1*(3*p2*(-5 - 8*r_mc4 + 32*r_mc4^2 - 48*r_mc4^3 + 24*r_mc4^4) + 4*(7 - 20*r_mc4 + 44*r_mc4^2 - 48*r_mc4^3 + 24*r_mc4^4)) + 4*(4*(5 - 7*r_mc4 + 19*r_mc4^2 - 24*r_mc4^3 + 12*r_mc4^4) + p2*(-3 - 3*r_mc4 + 39*r_mc4^2 - 72*r_mc4^3 + 36*r_mc4^4))) + (x[,"n_12"] + x[,"n_32"])*log10(2*(4*(5 - 7*r_mc4 + 19*r_mc4^2 - 24*r_mc4^3 + 12*r_mc4^4) + p2*(-3 - 3*r_mc4 + 39*r_mc4^2 - 72*r_mc4^3 + 36*r_mc4^4)) - 3*p1*(-4 - 28*r_mc4 + 76*r_mc4^2 - 96*r_mc4^3 + 48*r_mc4^4 + p2*(6 - 3*r_mc4 + 39*r_mc4^2 - 72*r_mc4^3 + 36*r_mc4^4)))


logL_mc4 <- (-6*x[,"n_00"] - 6*x[,"n_01"] - 6*x[,"n_02"] - 6*x[,"n_03"] - 6*x[,"n_04"] - 5*x[,"n_10"] - 5*x[,"n_11"] - 5*x[,"n_12"] - 5*x[,"n_13"] - 5*x[,"n_14"] - 6*x[,"n_20"] - 6*x[,"n_21"] - 5*x[,"n_22"] - 6*x[,"n_23"] - 6*x[,"n_24"] - 5*x[,"n_30"] - 5*x[,"n_31"] - 5*x[,"n_32"] - 5*x[,"n_33"] - 5*x[,"n_34"] - 6*x[,"n_40"] - 6*x[,"n_41"] - 6*x[,"n_42"] - 6*x[,"n_43"] - 6*x[,"n_44"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(3) + (x[,"n_00"] + x[,"n_44"])*(log(2 - 3*p1) + log(4 + 3*p2) + 3*log(pmax(1e-6,1 - r_mc4)) + log(pmax(1e-6,r_mc4))) + (x[,"n_04"] + x[,"n_40"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_mc4)) + 3*log(pmax(1e-6,r_mc4))) + (x[,"n_14"] + x[,"n_30"])*(log(2 - 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,r_mc4)) + log(2 - 3*r_mc4 + 2*r_mc4^2)) + (x[,"n_10"] + x[,"n_34"])*(log(2 - 3*p1) + log(4 + 3*p2) + 2*log(pmax(1e-6,1 - r_mc4)) + log(1 - r_mc4 + 2*r_mc4^2)) + (x[,"n_20"] + x[,"n_24"])*(log(2 - 3*p1) + log(4 + 3*p2) + log(pmax(1e-6,1 - r_mc4)) + log(pmax(1e-6,r_mc4)) + log(5 - 6*r_mc4 + 6*r_mc4^2)) + (x[,"n_03"] + x[,"n_41"])*(log(4 + 3*p2) + 2*log(pmax(1e-6,r_mc4)) + log(8 - 12*r_mc4 + 8*r_mc4^2 - 3*p1*(1 - 6*r_mc4 + 4*r_mc4^2))) + (x[,"n_01"] + x[,"n_43"])*(log(4 + 3*p2) + 2*log(pmax(1e-6,1 - r_mc4)) + log(4 - 4*r_mc4 + 8*r_mc4^2 - 3*p1*(-1 - 2*r_mc4 + 4*r_mc4^2))) + (x[,"n_02"] + x[,"n_42"])*(log(4 + 3*p2) + log(pmax(1e-6,1 - r_mc4)) + log(pmax(1e-6,r_mc4)) + log(3*p1*(1 + 6*r_mc4 - 6*r_mc4^2) + 2*(5 - 6*r_mc4 + 6*r_mc4^2))) + (x[,"n_13"] + x[,"n_31"])*(log(pmax(1e-6,r_mc4)) + log(-16*(-3 + 5*r_mc4 - 7*r_mc4^2 + 4*r_mc4^3) - 6*p2*(-3 + 7*r_mc4 - 14*r_mc4^2 + 8*r_mc4^3) + 3*p1*(-12 + (40 + 21*p2)*r_mc4 - 14*(4 + 3*p2)*r_mc4^2 + 8*(4 + 3*p2)*r_mc4^3))) + (x[,"n_11"] + x[,"n_33"])*(log(pmax(1e-6,1 - r_mc4)) + log(2*(8*(1 + 3*r_mc4 - 5*r_mc4^2 + 4*r_mc4^3) + 3*p2*(2 + 3*r_mc4 - 10*r_mc4^2 + 8*r_mc4^3)) - 3*p1*(3*p2*(-1 + 3*r_mc4 - 10*r_mc4^2 + 8*r_mc4^3) + 4*(-1 + 6*r_mc4 - 10*r_mc4^2 + 8*r_mc4^3)))) + x[,"n_22"]*log(-2*(3*p2*(4 + 3*r_mc4 + 15*r_mc4^2 - 36*r_mc4^3 + 18*r_mc4^4) + 4*(-8 - 9*r_mc4 + 27*r_mc4^2 - 36*r_mc4^3 + 18*r_mc4^4)) + 3*p1*(3*p2*(-2 - 3*r_mc4 + 21*r_mc4^2 - 36*r_mc4^3 + 18*r_mc4^4) + 4*(4 - 15*r_mc4 + 33*r_mc4^2 - 36*r_mc4^3 + 18*r_mc4^4))) + (x[,"n_21"] + x[,"n_23"])*log(-3*p1*(3*p2*(-5 - 8*r_mc4 + 32*r_mc4^2 - 48*r_mc4^3 + 24*r_mc4^4) + 4*(7 - 20*r_mc4 + 44*r_mc4^2 - 48*r_mc4^3 + 24*r_mc4^4)) + 4*(4*(5 - 7*r_mc4 + 19*r_mc4^2 - 24*r_mc4^3 + 12*r_mc4^4) + p2*(-3 - 3*r_mc4 + 39*r_mc4^2 - 72*r_mc4^3 + 36*r_mc4^4))) + (x[,"n_12"] + x[,"n_32"])*log(2*(4*(5 - 7*r_mc4 + 19*r_mc4^2 - 24*r_mc4^3 + 12*r_mc4^4) + p2*(-3 - 3*r_mc4 + 39*r_mc4^2 - 72*r_mc4^3 + 36*r_mc4^4)) - 3*p1*(-4 - 28*r_mc4 + 76*r_mc4^2 - 96*r_mc4^3 + 48*r_mc4^4 + p2*(6 - 3*r_mc4 + 39*r_mc4^2 - 72*r_mc4^3 + 36*r_mc4^4)))


logL_mm1 <- function(r,n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
L <- (-6*n00 - 4*n01 - 5*n02 - 4*n03 - 6*n04 - 5*n10 - 4*n11 - 4*n12 - 4*n13 - 5*n14 - 6*n20 - 4*n21 - 5*n22 - 4*n23 - 6*n24 - 5*n30 - 4*n31 - 4*n32 - 4*n33 - 5*n34 - 6*n40 - 4*n41 - 5*n42 - 4*n43 - 6*n44)*log(2) + 2*(-n00 - n01 - n02 - n03 - n04 - n10 - n11 - n12 - n13 - n14 - n20 - n21 - n22 - n23 - n24 - n30 - n31 - n32 - n33 - n34 - n40 - n41 - n42 - n43 - n44)*log(3) + (n00 + n04 + n40 + n44)*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r)) + 2*log(pmax(1e-6,r))) + (n01 + n03 + n41 + n43)*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(1 - r + r^2)) + (n02 + n42)*(log(2 - 3*p1) + log(2 - 3*p2) + log(2 - 4*r + 7*r^2 - 6*r^3 + 3*r^4)) + (n20 + n24)*log(2*(-2 + 3*p1)*(-2 + 3*p2)*(-1 + r)^2*r^2 + (p1*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2))*(p2*(3 + 6*r - 6*r^2) + 4*(1 - r + r^2))) + (n10 + n14 + n30 + n34)*(log(pmax(1e-6,1 - r)) + log(pmax(1e-6,r)) + log(-3*p2*(1 - 2*r)^2 + 8*(1 - r + r^2) + 3*p1*(-(1 - 2*r)^2 + p2*(-3 - 6*r + 6*r^2)))) + (n11 + n13 + n31 + n33)*log(8*(1 - r + 3*r^2 - 4*r^3 + 2*r^4) - 3*p2*(1 - 4*r + 12*r^2 - 16*r^3 + 8*r^4) + 3*p1*(-1 + 4*r - 12*r^2 + 16*r^3 - 8*r^4 + 3*p2*(-1 - 2*r + 6*r^2 - 8*r^3 + 4*r^4))) + (n21 + n23)*log(4*(2*(2 + r - 4*r^2 + 6*r^3 - 3*r^4) + 3*p2*(1 - r + 4*r^2 - 6*r^3 + 3*r^4)) - 3*p1*(-4*(1 - r + 4*r^2 - 6*r^3 + 3*r^4) + 3*p2*(-1 - 2*r + 8*r^2 - 12*r^3 + 6*r^4))) + (n12 + n32)*log(8*(2 + r - 4*r^2 + 6*r^3 - 3*r^4) + 3*p2*(-2 - 7*r + 19*r^2 - 24*r^3 + 12*r^4) - 3*p1*(2 + 7*r - 19*r^2 + 24*r^3 - 12*r^4 + 3*p2*(2 - 5*r + 11*r^2 - 12*r^3 + 6*r^4))) + n22*log(4*(p2*(3 + 9*r - 36*r^2 + 54*r^3 - 27*r^4) + 2*(8 + 9*r^2 - 18*r^3 + 9*r^4)) + 3*p1*(4 + 12*r - 48*r^2 + 72*r^3 - 36*r^4 + 3*p2*(7 - 12*r + 30*r^2 - 36*r^3 + 18*r^4)))
return(L)}
interlogL_mm1 <- function(n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2) {
optimize(logL_mm1,c(0,0.5), n00,n01,n02,n03,n04,n10,n11,n12,n13,n14,n20,n21,n22,n23,n24,n30,n31,n32,n33,n34,n40,n41,n42,n43,n44,p1,p2, maximum=TRUE, lower=0, upper=0.5)$maximum}


r_mm1 <- parallel::mcmapply(interlogL_mm1,x[,"n_00"],x[,"n_01"],x[,"n_02"],x[,"n_03"],x[,"n_04"],x[,"n_10"],x[,"n_11"],x[,"n_12"],x[,"n_13"],x[,"n_14"],x[,"n_20"],x[,"n_21"],x[,"n_22"],x[,"n_23"],x[,"n_24"],x[,"n_30"],x[,"n_31"],x[,"n_32"],x[,"n_33"],x[,"n_34"],x[,"n_40"],x[,"n_41"],x[,"n_42"],x[,"n_43"],x[,"n_44"], p1, p2, mc.cores = ncores)


LOD_mm1 <- -((x[,"n_00"] + x[,"n_04"] + x[,"n_40"] + x[,"n_44"])*(-4*log10(2) + log10(2 - 3*p1) + log10(2 - 3*p2))) - (x[,"n_01"] + x[,"n_03"] + x[,"n_41"] + x[,"n_43"])*(-4*log10(2) + log10(3) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_02"] + x[,"n_42"])*(-4*log10(2) + log10(19) + log10(2 - 3*p1) + log10(2 - 3*p2)) - (x[,"n_11"] + x[,"n_13"] + x[,"n_31"] + x[,"n_33"])*log10(7 + 3*p1*(-1/2 - (15*p2)/4) - (3*p2)/2) - (x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"])*(-2*log10(2) + log10(6 - (27*p1*p2)/2)) - (x[,"n_21"] + x[,"n_23"])*log10(-3*p1*(-15/4 - (27*p2)/8) + 4*(33/8 + (45*p2)/16)) - (x[,"n_12"] + x[,"n_32"])*log10(33/2 - 9*p2 - 3*p1*(3 + (27*p2)/8)) - (x[,"n_20"] + x[,"n_24"])*log10(((-2 + 3*p1)*(-2 + 3*p2))/8 + (3 + (9*p1)/2)*(3 + (9*p2)/2)) - x[,"n_22"]*log10(4*(137/8 + (57*p2)/16) + 3*p1*(19/4 + (123*p2)/8)) + (x[,"n_00"] + x[,"n_04"] + x[,"n_40"] + x[,"n_44"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + 2*log10(pmax(1e-6,1 - r_mm1)) + 2*log10(pmax(1e-6,r_mm1))) + (x[,"n_01"] + x[,"n_03"] + x[,"n_41"] + x[,"n_43"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(pmax(1e-6,1 - r_mm1)) + log10(pmax(1e-6,r_mm1)) + log10(1 - r_mm1 + r_mm1^2)) + (x[,"n_02"] + x[,"n_42"])*(log10(2 - 3*p1) + log10(2 - 3*p2) + log10(2 - 4*r_mm1 + 7*r_mm1^2 - 6*r_mm1^3 + 3*r_mm1^4)) + (x[,"n_20"] + x[,"n_24"])*log10(2*(-2 + 3*p1)*(-2 + 3*p2)*(-1 + r_mm1)^2*r_mm1^2 + (p1*(3 + 6*r_mm1 - 6*r_mm1^2) + 4*(1 - r_mm1 + r_mm1^2))*(p2*(3 + 6*r_mm1 - 6*r_mm1^2) + 4*(1 - r_mm1 + r_mm1^2))) + (x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"])*(log10(pmax(1e-6,1 - r_mm1)) + log10(pmax(1e-6,r_mm1)) + log10(-3*p2*(1 - 2*r_mm1)^2 + 8*(1 - r_mm1 + r_mm1^2) + 3*p1*(-(1 - 2*r_mm1)^2 + p2*(-3 - 6*r_mm1 + 6*r_mm1^2)))) + (x[,"n_11"] + x[,"n_13"] + x[,"n_31"] + x[,"n_33"])*log10(8*(1 - r_mm1 + 3*r_mm1^2 - 4*r_mm1^3 + 2*r_mm1^4) - 3*p2*(1 - 4*r_mm1 + 12*r_mm1^2 - 16*r_mm1^3 + 8*r_mm1^4) + 3*p1*(-1 + 4*r_mm1 - 12*r_mm1^2 + 16*r_mm1^3 - 8*r_mm1^4 + 3*p2*(-1 - 2*r_mm1 + 6*r_mm1^2 - 8*r_mm1^3 + 4*r_mm1^4))) + (x[,"n_21"] + x[,"n_23"])*log10(4*(2*(2 + r_mm1 - 4*r_mm1^2 + 6*r_mm1^3 - 3*r_mm1^4) + 3*p2*(1 - r_mm1 + 4*r_mm1^2 - 6*r_mm1^3 + 3*r_mm1^4)) - 3*p1*(-4*(1 - r_mm1 + 4*r_mm1^2 - 6*r_mm1^3 + 3*r_mm1^4) + 3*p2*(-1 - 2*r_mm1 + 8*r_mm1^2 - 12*r_mm1^3 + 6*r_mm1^4))) + (x[,"n_12"] + x[,"n_32"])*log10(8*(2 + r_mm1 - 4*r_mm1^2 + 6*r_mm1^3 - 3*r_mm1^4) + 3*p2*(-2 - 7*r_mm1 + 19*r_mm1^2 - 24*r_mm1^3 + 12*r_mm1^4) - 3*p1*(2 + 7*r_mm1 - 19*r_mm1^2 + 24*r_mm1^3 - 12*r_mm1^4 + 3*p2*(2 - 5*r_mm1 + 11*r_mm1^2 - 12*r_mm1^3 + 6*r_mm1^4))) + x[,"n_22"]*log10(4*(p2*(3 + 9*r_mm1 - 36*r_mm1^2 + 54*r_mm1^3 - 27*r_mm1^4) + 2*(8 + 9*r_mm1^2 - 18*r_mm1^3 + 9*r_mm1^4)) + 3*p1*(4 + 12*r_mm1 - 48*r_mm1^2 + 72*r_mm1^3 - 36*r_mm1^4 + 3*p2*(7 - 12*r_mm1 + 30*r_mm1^2 - 36*r_mm1^3 + 18*r_mm1^4)))


logL_mm1 <- (-6*x[,"n_00"] - 4*x[,"n_01"] - 5*x[,"n_02"] - 4*x[,"n_03"] - 6*x[,"n_04"] - 5*x[,"n_10"] - 4*x[,"n_11"] - 4*x[,"n_12"] - 4*x[,"n_13"] - 5*x[,"n_14"] - 6*x[,"n_20"] - 4*x[,"n_21"] - 5*x[,"n_22"] - 4*x[,"n_23"] - 6*x[,"n_24"] - 5*x[,"n_30"] - 4*x[,"n_31"] - 4*x[,"n_32"] - 4*x[,"n_33"] - 5*x[,"n_34"] - 6*x[,"n_40"] - 4*x[,"n_41"] - 5*x[,"n_42"] - 4*x[,"n_43"] - 6*x[,"n_44"])*log(2) + 2*(-x[,"n_00"] - x[,"n_01"] - x[,"n_02"] - x[,"n_03"] - x[,"n_04"] - x[,"n_10"] - x[,"n_11"] - x[,"n_12"] - x[,"n_13"] - x[,"n_14"] - x[,"n_20"] - x[,"n_21"] - x[,"n_22"] - x[,"n_23"] - x[,"n_24"] - x[,"n_30"] - x[,"n_31"] - x[,"n_32"] - x[,"n_33"] - x[,"n_34"] - x[,"n_40"] - x[,"n_41"] - x[,"n_42"] - x[,"n_43"] - x[,"n_44"])*log(3) + (x[,"n_00"] + x[,"n_04"] + x[,"n_40"] + x[,"n_44"])*(log(2 - 3*p1) + log(2 - 3*p2) + 2*log(pmax(1e-6,1 - r_mm1)) + 2*log(pmax(1e-6,r_mm1))) + (x[,"n_01"] + x[,"n_03"] + x[,"n_41"] + x[,"n_43"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(pmax(1e-6,1 - r_mm1)) + log(pmax(1e-6,r_mm1)) + log(1 - r_mm1 + r_mm1^2)) + (x[,"n_02"] + x[,"n_42"])*(log(2 - 3*p1) + log(2 - 3*p2) + log(2 - 4*r_mm1 + 7*r_mm1^2 - 6*r_mm1^3 + 3*r_mm1^4)) + (x[,"n_20"] + x[,"n_24"])*log(2*(-2 + 3*p1)*(-2 + 3*p2)*(-1 + r_mm1)^2*r_mm1^2 + (p1*(3 + 6*r_mm1 - 6*r_mm1^2) + 4*(1 - r_mm1 + r_mm1^2))*(p2*(3 + 6*r_mm1 - 6*r_mm1^2) + 4*(1 - r_mm1 + r_mm1^2))) + (x[,"n_10"] + x[,"n_14"] + x[,"n_30"] + x[,"n_34"])*(log(pmax(1e-6,1 - r_mm1)) + log(pmax(1e-6,r_mm1)) + log(-3*p2*(1 - 2*r_mm1)^2 + 8*(1 - r_mm1 + r_mm1^2) + 3*p1*(-(1 - 2*r_mm1)^2 + p2*(-3 - 6*r_mm1 + 6*r_mm1^2)))) + (x[,"n_11"] + x[,"n_13"] + x[,"n_31"] + x[,"n_33"])*log(8*(1 - r_mm1 + 3*r_mm1^2 - 4*r_mm1^3 + 2*r_mm1^4) - 3*p2*(1 - 4*r_mm1 + 12*r_mm1^2 - 16*r_mm1^3 + 8*r_mm1^4) + 3*p1*(-1 + 4*r_mm1 - 12*r_mm1^2 + 16*r_mm1^3 - 8*r_mm1^4 + 3*p2*(-1 - 2*r_mm1 + 6*r_mm1^2 - 8*r_mm1^3 + 4*r_mm1^4))) + (x[,"n_21"] + x[,"n_23"])*log(4*(2*(2 + r_mm1 - 4*r_mm1^2 + 6*r_mm1^3 - 3*r_mm1^4) + 3*p2*(1 - r_mm1 + 4*r_mm1^2 - 6*r_mm1^3 + 3*r_mm1^4)) - 3*p1*(-4*(1 - r_mm1 + 4*r_mm1^2 - 6*r_mm1^3 + 3*r_mm1^4) + 3*p2*(-1 - 2*r_mm1 + 8*r_mm1^2 - 12*r_mm1^3 + 6*r_mm1^4))) + (x[,"n_12"] + x[,"n_32"])*log(8*(2 + r_mm1 - 4*r_mm1^2 + 6*r_mm1^3 - 3*r_mm1^4) + 3*p2*(-2 - 7*r_mm1 + 19*r_mm1^2 - 24*r_mm1^3 + 12*r_mm1^4) - 3*p1*(2 + 7*r_mm1 - 19*r_mm1^2 + 24*r_mm1^3 - 12*r_mm1^4 + 3*p2*(2 - 5*r_mm1 + 11*r_mm1^2 - 12*r_mm1^3 + 6*r_mm1^4))) + x[,"n_22"]*log(4*(p2*(3 + 9*r_mm1 - 36*