% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot_belief_space.R
\name{plot_belief_space}
\alias{plot_belief_space}
\title{Plot a 2D or 3D Projection of the Belief Space}
\usage{
plot_belief_space(
  model,
  projection = NULL,
  epoch = 1,
  sample = "regular",
  n = 100,
  what = c("action", "pg_node", "reward"),
  legend = TRUE,
  pch = 20,
  col = NULL,
  jitter = 0,
  oneD = TRUE,
  ...
)
}
\arguments{
\item{model}{a solved \link{POMDP}.}

\item{projection}{Sample in a projected belief space. See \code{\link[=projection]{projection()}} for details.}

\item{epoch}{display this epoch.}

\item{sample}{a matrix with belief points as rows or a character string
specifying the \code{method} used for \code{\link[=sample_belief_space]{sample_belief_space()}}.}

\item{n}{number of points sampled.}

\item{what}{what to plot.}

\item{legend}{logical; add a legend? If the legend is covered by the plot then you
need to increase the plotting region of the plotting device.}

\item{pch}{plotting symbols.}

\item{col}{plotting colors.}

\item{jitter}{jitter amount for 2D belief spaces (good values are between 0 and 1, while using \code{ylim = c(0,1)}).}

\item{oneD}{plot projections on two states in one dimension.}

\item{...}{additional arguments are passed on to \code{plot} for 2D or
\code{TerneryPlot} for 3D.}
}
\value{
Returns invisibly the sampled points.
}
\description{
Plots the optimal action, the node in the policy graph or the reward for a
given set of belief points on a line (2D) or on a ternary plot (3D). If no
points are given, points are sampled using a regular arrangement or randomly
from the (projected) belief space.
}
\examples{
# two-state POMDP
data("Tiger")
sol <- solve_POMDP(Tiger)

plot_belief_space(sol)
plot_belief_space(sol, oneD = FALSE)
plot_belief_space(sol, n = 10)
plot_belief_space(sol, n = 100, sample = "random")

# plot the belief points used by the grid-based solver
plot_belief_space(sol, sample = sol $solution$belief_points_solver)

# plot different measures
plot_belief_space(sol, what = "pg_node")
plot_belief_space(sol, what = "reward")

# three-state POMDP
# Note: If the plotting region is too small then the legend might run into the plot
data("Three_doors")
sol <- solve_POMDP(Three_doors)
sol

# plotting needs the suggested package Ternary
if ("Ternary" \%in\% installed.packages()) {
plot_belief_space(sol)
plot_belief_space(sol, n = 10000)
plot_belief_space(sol, what = "reward", sample = "random", n = 1000)
plot_belief_space(sol, what = "pg_node", n = 10000)

# holding tiger-left constant at .5 follows this line in the ternary plot 
Ternary::TernaryLines(list(c(.5, 0, .5), c(.5, .5, 0)), col = "black", lty = 2)
# we can plot the projection for this line 
plot_belief_space(sol, what = "pg_node", n = 1000, projection = c("tiger-left" = .5))

# plot the belief points used by the grid-based solver
plot_belief_space(sol, sample = sol$solution$belief_points_solver, what = "pg_node")

# plot the belief points obtained using simulated trajectories with an epsilon-greedy policy.
# Note that we only use n = 50 to save time.
plot_belief_space(sol, 
  sample = simulate_POMDP(sol, n = 50, horizon = 100,
    epsilon = 0.1, return_beliefs = TRUE)$belief_states)
}

# plot a 3-state belief space using ggtern (ggplot2)
\dontrun{
library(ggtern)
samp <- sample_belief_space(sol, n = 1000)
df <- cbind(as.data.frame(samp), reward_node_action(sol, belief = samp))
df$pg_node <- factor(df$pg_node)

ggtern(df, aes(x = `tiger-left`, y = `tiger-center`, z = `tiger-right`)) +
  geom_point(aes(color = pg_node), size = 2)

ggtern(df, aes(x = `tiger-left`, y = `tiger-center`, z = `tiger-right`)) +
  geom_point(aes(color = action), size = 2)

ggtern(df, aes(x = `tiger-left`, y = `tiger-center`, z = `tiger-right`)) +
  geom_point(aes(color = reward), size = 2)
}
}
\seealso{
Other policy: 
\code{\link{estimate_belief_for_nodes}()},
\code{\link{optimal_action}()},
\code{\link{plot_policy_graph}()},
\code{\link{policy_graph}()},
\code{\link{policy}()},
\code{\link{projection}()},
\code{\link{reward}()},
\code{\link{solve_POMDP}()},
\code{\link{solve_SARSOP}()},
\code{\link{value_function}()}

Other POMDP: 
\code{\link{POMDP_accessors}},
\code{\link{POMDP}()},
\code{\link{projection}()},
\code{\link{regret}()},
\code{\link{sample_belief_space}()},
\code{\link{simulate_POMDP}()},
\code{\link{solve_POMDP}()},
\code{\link{solve_SARSOP}()},
\code{\link{transition_graph}()},
\code{\link{update_belief}()},
\code{\link{value_function}()},
\code{\link{write_POMDP}()}
}
\author{
Michael Hahsler
}
\concept{POMDP}
\concept{policy}
\keyword{hplot}
