#' Visualize Total Costs for Pooling Design as a Function of Pool Size
#'
#' Useful for determining whether pooling is a good idea, what pool size
#' minimizes costs, and how many assays are needed for a target power.
#'
#'
#' @param g Numeric vector of pool sizes to include.
#' @param d Numeric value specifying true difference in group means.
#' @param sigsq Numeric value specifying the variance of observations.
#' @param sigsq_p Numeric value specifying the variance of processing errors.
#' @param sigsq_m Numeric value specifying the variance of measurement errors.
#' @param multiplicative Logical value for whether to assume multiplicative
#' rather than additive errors.
#' @param mu Numeric value specifying the larger of the two suspected means.
#' Only used if \code{multiplicative = TRUE}.
#' @param alpha Numeric value specifying significance level.
#' @param beta Numeric value specifying beta = 1 - power.
#' @param type Character string specifying type of t-test. Choices are
#' \code{"two.sample"}, \code{"one.sample"}, and \code{"paired"}.
#' @param assay_cost Numeric value specifying cost of each assay.
#' @param other_costs Numeric value specifying other per-subject costs.
#' @param labels Logical value.
#' @param ylim Numeric vector.
#'
#'
#' @return Plot of total costs vs. pool size generated by
#' \code{\link[ggplot2]{ggplot}}.
#'
#'
#' @examples
#' # Plot study costs vs. pool size with default settings
#' poolcost_t()
#'
#' # Add processing error and other per-subject costs
#' poolcost_t(sigsq_p = 0.2, other_costs = 0.1)
#'
#'
#'@export
poolcost_t <- function(g = 1: 10,
                       d = 0.2,
                       sigsq = 1,
                       sigsq_p = 0,
                       sigsq_m = 0,
                       multiplicative = FALSE,
                       mu = 1,
                       alpha = 0.05,
                       beta = 0.2,
                       type = "two.sample",
                       assay_cost = 100,
                       other_costs = 0,
                       labels = TRUE,
                       ylim = NULL) {

  # Prepare data for ggplot
  z <- qnorm(p = 1 - beta) + qnorm(p = 1 - alpha / 2)
  if (multiplicative) {
    sigsq_pm <- sigsq_p * ifelse(g > 1, 1, 0) + sigsq_m +
      sigsq_p * ifelse(g > 1, 1, 0) * sigsq_m
    n <- ceiling(2 * z^2 / d^2 * (sigsq_pm * (mu^2 + sigsq / g) + sigsq / g))
  } else {
    n <- ceiling(2 * z^2 / d^2 * (sigsq / g + sigsq_p * ifelse(g > 1, 1, 0) + sigsq_m))
  }
  costs <-  2 * n * (assay_cost + g * other_costs)
  df <- data.frame(g = g, n = n, costs = costs)
  df$lab <- 0
  df$labeltext <- paste(n, " assays", sep = "")
  locs <- unique(c(1, which.min(df$costs)))
  df$labeltext[-locs] <- ""
  df$lab[locs] <- 1

  # Default ylim
  if (is.null(ylim)) {
    ylim <- c(0, max(costs) * 1.06)
  }

  # Create plot
  p <- ggplot(df, aes_string(x = "g", y = "costs", label = "labeltext")) +
    geom_col() +
    labs(title = "Total Study Costs vs. Pool Size",
         y = "Total costs ($)",
         x = "Pool size") +
    ylim(ylim) +
    scale_x_continuous(breaks = g) +
    theme_bw()

  # Label min
  if (labels) {
    p <- p + geom_label_repel(point.padding = 0.3,
                              box.padding = 1.5,
                              direction = "y",
                              nudge_y = max(costs) / 25,
                              label.padding = 0.4)
  }
  p

}
