
#' Analyze aov and anova objects
#'
#' Analyze aov and anova objects.
#'
#' @param x aov object.
#' @param effsize_rules Grid for effect size interpretation. See \link[=interpret_omega_sq]{interpret_omega_sq}.
#' @param ... Arguments passed to or from other methods.
#'
#' @return output
#'
#' @examples
#' \dontrun{
#' library(psycho)
#'
#' df <- psycho::affective
#'
#' x <- aov(df$Tolerating ~ df$Salary)
#' x <- aov(df$Tolerating ~ df$Salary * df$Sex)
#'
#' x <- anova(lm(df$Tolerating ~ df$Salary * df$Sex))
#'
#'
#' summary(analyze(x))
#' print(analyze(x))
#'
#' df <- psycho::emotion %>%
#'   mutate(Recall = ifelse(Recall == TRUE, 1, 0)) %>%
#'   group_by(Participant_ID, Emotion_Condition) %>%
#'   summarise(Recall = sum(Recall) / n())
#'
#' x <- aov(Recall ~ Emotion_Condition + Error(Participant_ID), data = df)
#' x <- anova(lmerTest::lmer(Recall ~ Emotion_Condition + (1 | Participant_ID), data = df))
#' analyze(x)
#' summary(x)
#' }
#'
#' @references
#' \itemize{
#'  \item{Levine, T. R., & Hullett, C. R. (2002). Eta squared, partial eta squared, and misreporting of effect size in communication research. Human Communication Research, 28(4), 612-625.}
#'  \item{Pierce, C. A., Block, R. A., & Aguinis, H. (2004). Cautionary note on reporting eta-squared values from multifactor ANOVA designs. Educational and psychological measurement, 64(6), 916-924.}
#' }
#'
#' @seealso http://imaging.mrc-cbu.cam.ac.uk/statswiki/FAQ/os2
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @import broom
#'
#' @export
analyze.aov <- function(x, effsize_rules = "field2013", ...) {
  if (!"aov" %in% class(x)) {
    if (!"Residuals" %in% row.names(x)) {
      if (!is.null(x$Within)) {
        x <- x$Within
        message("(Repeated measures ANOVAs are bad, you should use mixed-models...)")
      } else {
        return(.analyze.anova_lmer(x))
      }
    }
  } else {
    if (!is.null(x$Within)) {
      x <- x$Within
      message("(Repeated measures ANOVAs are bad, you should use mixed-models...)")
    }
  }




  # Processing
  # -------------


  # Effect Size
  omega <- tryCatch({
    omega_sq(x, partial = TRUE)
  }, warning = function(w) {
    stop("I believe there are within and between subjects variables that caused the error. You should REALLY use mixed-models.")
  })




  all_values <- x %>%
    broom::tidy() %>%
    dplyr::full_join(data.frame("Omega" = omega) %>%
      tibble::rownames_to_column("term"), by = "term") %>%
    mutate_("Effect_Size" = "interpret_omega_sq(Omega, rules = 'field2013')") %>%
    rename_(
      "Effect" = "term",
      "Sum_Squares" = "sumsq",
      "Mean_Square" = "meansq",
      "F" = "statistic",
      "p" = "p.value"
    )

  varnames <- all_values$Effect
  df_residuals <- all_values[all_values$Effect == "Residuals", ]$df

  values <- list()
  for (var in varnames) {
    values[[var]] <- list()
    current_values <- dplyr::filter_(all_values, "Effect == var")
    values[[var]]$df <- current_values$df
    values[[var]]$Sum_Squares <- current_values$Sum_Squares
    values[[var]]$Mean_Square <- current_values$Mean_Square
    values[[var]]$F <- current_values$F
    values[[var]]$p <- current_values$p
    values[[var]]$Omega <- current_values$Omega
    values[[var]]$Effect_Size <- current_values$Effect_Size

    if (var != "Residuals") {
      if (current_values$p < .05) {
        significance <- "significant"
      } else {
        significance <- "not significant"
      }

      if (grepl(":", var)) {
        effect <- "interaction between"
        varname <- stringr::str_replace_all(var, ":", " and ")
      } else {
        varname <- var
        effect <- "effect of"
      }

      values[[var]]$text <- paste0(
        "The ",
        effect,
        " ",
        varname,
        " is ",
        significance,
        " (F(",
        current_values$df,
        ", ",
        df_residuals,
        ") = ",
        format_digit(current_values$F),
        ", p ",
        format_p(current_values$p, stars = FALSE),
        ") and can be considered as ",
        current_values$Effect_Size,
        " (Partial Omega-squared = ",
        format_digit(current_values$Omega),
        ")."
      )
    }
  }

  # Summary
  # -------------
  summary <- all_values

  # Text
  # -------------
  text <- c()
  for (var in varnames[varnames != "Residuals"]) {
    text <- c(text, paste("   -", values[[var]]$text))
  }


  # Plot
  # -------------
  plot <- "Not available yet"

  output <- list(text = text, plot = plot, summary = summary, values = values)

  class(output) <- c("psychobject", "list")
  return(output)
}










#' @export
analyze.anova <- analyze.aov

#' @export
analyze.aovlist <- analyze.aov



#' @keywords internal
.analyze.anova_lmer <- function(x) {
  if (!"NumDF" %in% colnames(x)) {
    stop("Cannot analyze the anova from lme4. Please refit the model using lmerTest.")
  }

  summary <- x %>%
    as.data.frame() %>%
    tibble::rownames_to_column("term") %>%
    rename_(
      "Effect" = "term",
      "df" = "NumDF",
      "df_Residuals" = "DenDF",
      "Sum_Squares" = "`Sum Sq`",
      "Mean_Square" = "`Mean Sq`",
      "F" = "`F value`",
      "p" = "`Pr(>F)`"
    ) %>%
    select_("Effect", "df", "df_Residuals", "Sum_Squares", "Mean_Square", "F", "p")

  varnames <- summary$Effect

  values <- list()
  for (var in varnames) {
    values[[var]] <- list()
    current_values <- dplyr::filter_(summary, "Effect == var")
    values[[var]]$df <- current_values$df
    values[[var]]$df_Residuals <- current_values$df_Residuals
    values[[var]]$Sum_Squares <- current_values$Sum_Squares
    values[[var]]$Mean_Square <- current_values$Mean_Square
    values[[var]]$F <- current_values$F
    values[[var]]$p <- current_values$p
    # values[[var]]$Omega <- current_values$Omega
    # values[[var]]$Effect_Size <- current_values$Effect_Size

    if (current_values$p < .05) {
      significance <- "significant"
    } else {
      significance <- "not significant"
    }

    if (grepl(":", var)) {
      effect <- "interaction between"
      varname <- stringr::str_replace_all(var, ":", " and ")
    } else {
      varname <- var
      effect <- "effect of"
    }

    values[[var]]$text <- paste0(
      "The ",
      effect,
      " ",
      varname,
      " is ",
      significance,
      " (F(",
      current_values$df,
      ", ",
      format_digit(current_values$df_Residuals, 0),
      ") = ",
      format_digit(current_values$F),
      ", p ",
      format_p(current_values$p, stars = FALSE),
      ")."
    )
  }


  # Text
  # -------------
  text <- c()
  for (var in varnames[varnames != "Residuals"]) {
    text <- c(text, paste("   -", values[[var]]$text))
  }

  # Plot
  # -------------
  plot <- "Not available yet"

  output <- list(text = text, plot = plot, summary = summary, values = values)

  class(output) <- c("psychobject", "list")
  return(output)
}





#' Partial Omega Squared.
#'
#' Partial Omega Squared.
#'
#' @param x aov object.
#' @param partial Return partial omega squared.
#'
#' @return output
#'
#' @examples
#' library(psycho)
#'
#' df <- psycho::affective
#'
#' x <- aov(df$Tolerating ~ df$Salary)
#' x <- aov(df$Tolerating ~ df$Salary * df$Sex)
#'
#' omega_sq(x)
#' @seealso http://stats.stackexchange.com/a/126520
#'
#' @author Arnoud Plantinga
#' @importFrom stringr str_trim
#' @export
omega_sq <- function(x, partial = TRUE) {
  if ("aov" %in% class(x)) {
    summary_aov <- summary(x)[[1]]
  } else {
    summary_aov <- x
  }
  residRow <- nrow(summary_aov)
  dfError <- summary_aov[residRow, 1]
  msError <- summary_aov[residRow, 3]
  nTotal <- sum(summary_aov$Df)
  dfEffects <- summary_aov[1:{
    residRow - 1
  }, 1]
  ssEffects <- summary_aov[1:{
    residRow - 1
  }, 2]
  msEffects <- summary_aov[1:{
    residRow - 1
  }, 3]
  ssTotal <- rep(sum(summary_aov[1:residRow, 2]), 3)
  Omegas <- abs((ssEffects - dfEffects * msError) / (ssTotal + msError))
  names(Omegas) <- stringr::str_trim(rownames(summary_aov)[1:{
    residRow - 1
  }])

  partOmegas <- abs((dfEffects * (msEffects - msError)) /
    (ssEffects + (nTotal - dfEffects) * msError))
  names(partOmegas) <- stringr::str_trim(rownames(summary_aov)[1:{
    residRow - 1
  }])

  if (partial == TRUE) {
    return(partOmegas)
  } else {
    return(Omegas)
  }
}









#' Remove empty columns.
#'
#' Removes all columns containing ony NaNs.
#'
#' @param df Dataframe.
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
remove_empty_cols <- function(df) {
  df <- df[, colSums(is.na(df)) < nrow(df)]
  return(df)
}




#' Creates or tests for objects of mode "psychobject".
#'
#' @param x an arbitrary R object.
#'
#' @export
is.psychobject <- function(x) inherits(x, "psychobject")








#' Create a reference grid.
#'
#' Create a reference grid.
#'
#' @param df The dataframe.
#' @param target String or list of strings to indicate target columns. Can be "all".
#' @param length.out Length of numeric target variables.
#' @param factors Type of summary for factors. Can be "combination" or "reference".
#' @param numerics Type of summary for numerics Can be "combination", any function ("mean", "median", ...) or a value.
#' @param na.rm Remove NaNs.
#'
#' @examples
#' library(psycho)
#'
#' df <- psycho::affective
#' newdata <- refdata(df, target = "Sex")
#' newdata <- refdata(df, target = "Sex", factors = "combinations")
#' newdata <- refdata(df, target = c("Sex", "Salary", "Tolerating"), length.out = 3)
#' newdata <- refdata(df, target = c("Sex", "Salary", "Tolerating"), numerics = 0)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @importFrom purrr keep
#' @import tidyr
#' @export
refdata <- function(df, target = "all", length.out = 10, factors = "reference", numerics = "mean", na.rm = TRUE) {

  # Target
  if (all(target == "all") | ncol(df) == 1) {
    return(.refdata_target(target = df[c(names(df))], length.out = length.out))
  }

  target_df <- .refdata_target(target = df[c(target)], length.out = length.out)

  # Rest
  df_rest <- df[!names(df) %in% c(target)]
  var_order <- names(df_rest)

  facs <- purrr::discard(df_rest, is.numeric)
  facs <- mutate_all(facs, as.factor)
  nums <- purrr::keep(df_rest, is.numeric)


  smart_summary <- function(x, numerics) {
    if (na.rm == TRUE) x <- na.omit(x)

    if (is.numeric(x)) {
      fun <- paste0(numerics, "(x)")
      out <- eval(parse(text = fun))
    } else if (is.factor(x)) {
      out <- levels(x)[1]
    } else if (is.character(x)) {
      out <- unique(x)[1]
    } else if (is.logical(x)) {
      out <- unique(x)[1]
    } else {
      warning("Argument is not numeric nor factor: returning NA.")
      out <- NA
    }
    return(out)
  }


  if (factors == "reference") {
    facs <- dplyr::summarise_all(facs, smart_summary)
  } else {
    facs <- tidyr::expand_(facs, names(facs))
  }

  if (is.numeric(numerics)) {
    nums[1, ] <- numerics
    nums <- nums[1, ]
  } else if (numerics == "combination") {
    nums <- tidyr::expand_(nums, names(nums))
  } else {
    nums <- dplyr::summarise_all(nums, smart_summary, numerics)
  }


  if (nrow(facs) == 0 | ncol(facs) == 0) {
    refrest <- nums
  } else if (nrow(nums) == 0 | ncol(nums) == 0) {
    refrest <- facs
  } else {
    refrest <- merge(facs, nums)
  }

  refrest <- refrest[var_order]
  refdata <- merge(target_df, refrest)

  return(refdata)
}










#' @keywords internal
.refdata_target <- function(target, length.out = 10) {
  at_vars <- names(target)
  at_df <- data.frame()
  for (var in at_vars) {
    ref_var <- .refdata_var(x = target[[var]], length.out = length.out, varname = var)
    if (nrow(at_df) == 0) {
      at_df <- ref_var
    } else {
      at_df <- merge(at_df, ref_var)
    }
  }
  return(at_df)
}


















#' @keywords internal
.refdata_var <- function(x, length.out = 10, varname = NULL) {
  if (is.numeric(x)) {
    out <- data.frame(seq(min(x, na.rm = TRUE),
      max(x, na.rm = TRUE),
      length.out = length.out
    ))
  } else if (is.factor(x)) {
    out <- data.frame(levels(x))
  } else if (is.character(x)) {
    x <- as.factor(x)
    out <- data.frame(levels(x))
  } else {
    warning("Argument is not numeric nor factor: returning NA.")
    out <- NA
    return()
  }

  if (is.null(varname)) {
    names(out) <- "x"
  } else {
    names(out) <- varname
  }
  return(out)
}





#' Remove outliers.
#'
#' Removes outliers (with the z-score method only for now).
#'
#' @param df Dataframe.
#' @param target String or list of strings of variables
#' @param threshold The z-score value (deviation of SD) by which to consider outliers.
#' @param direction Can be "both", "upper" or "lower".
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
remove_outliers <- function(df, target, threshold = qnorm(0.95), direction = "both") {
  for (var in c(target)) {
    df <- .remove_outliers(df, var, threshold, direction)
  }
  return(df)
}






#' @keywords internal
.remove_outliers <- function(df, target, threshold = qnorm(0.95), direction = "both") {
  df <- df %>%
    mutate_("outlier_criterion" = target) %>%
    standardize(subset = "outlier_criterion")
  if (direction %in% c("both", "upper")) {
    df <- df %>%
      filter_("outlier_criterion <= threshold")
  }
  if (direction %in% c("both", "lower")) {
    df <- df %>%
      filter_("outlier_criterion >= -threshold")
  }

  df <- df %>%
    select_("-outlier_criterion")

  return(df)
}





#' Perfect Normal Distribution.
#'
#' Generates a sample of size n with a near-perfect normal distribution.
#'
#' @param n number of observations. If length(n) > 1, the length is taken to be the number required.
#' @param mean vector of means.
#' @param sd vector of standard deviations.
#' @param method "qnorm" or "average".
#' @param iter number of iterations (precision).
#'
#' @examples
#' library(psycho)
#' x <- rnorm_perfect(10)
#' plot(density(x))
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @importFrom stats rnorm
#' @export
rnorm_perfect <- function(n, mean = 0, sd = 1, method = "qnorm", iter = 10000) {
  if (method == "average") {
    x <- rowMeans(replicate(iter, sort(rnorm(n, mean, sd))))
  } else {
    x <- qnorm(seq(1 / n, 1 - 1 / n, length.out = n), mean, sd)
  }
  return(x)
}






#' Region of Practical Equivalence (ROPE)
#'
#' Compute the proportion of a posterior distribution that lies within a region of practical equivalence.
#'
#' @param posterior Posterior Distribution.
#' @param bounds Rope lower and higher bounds.
#' @param CI The credible interval to use.
#' @param overlap Compute rope overlap (EXPERIMENTAL).
#'
#'
#' @return list containing rope indices
#'
#' @examples
#' library(psycho)
#'
#' posterior <- rnorm(1000, 0, 0.01)
#' results <- rope(posterior)
#' results$decision
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
rope <- function(posterior, bounds = c(-0.1, 0.1), CI = 95, overlap = FALSE) {


  # Basic rope --------------------------------------------------------------


  HDI_area <- HDI(posterior, CI / 100)
  HDI_area <- posterior[dplyr::between(
    posterior,
    HDI_area$values$HDImin,
    HDI_area$values$HDImax
  )]

  area_within <- HDI_area[dplyr::between(HDI_area, bounds[1], bounds[2])]
  area_outside <- HDI_area[!dplyr::between(HDI_area, bounds[1], bounds[2])]

  p_within <- length(area_within) / length(posterior)
  p_outside <- length(area_outside) / length(posterior)

  rope_decision <- ifelse(p_within == 0, "Accept",
    ifelse(p_outside == 0, "Reject", "Undecided")
  )



  # Rope Overlap ------------------------------------------------------------
  if (overlap == TRUE) {
    sd <- abs(bounds[1] - bounds[2]) / 2
    sd <- sd / 3
    norm <- rnorm_perfect(length(posterior), 0, sd)
    rope_overlap <- overlap(posterior, norm) * 100
    output <- list(rope_decision = rope_decision, rope_probability = p_within, rope_overlap = rope_overlap)
  } else {
    output <- list(rope_decision = rope_decision, rope_probability = p_within)
  }



  return(output)
}









#' Simulates data for single or multiple regression.
#'
#' Simulates data for single or multiple regression.
#'
#' @param coefs Desired theorethical coefs. Can be a single value or a list.
#' @param sample Desired sample size.
#' @param error The error (standard deviation of gaussian noise).
#'
#' @examples
#' library(psycho)
#'
#' data <- simulate_data_regression(coefs = c(0.1, 0.8), sample = 50, error = 0)
#' fit <- lm(y ~ ., data = data)
#' coef(fit)
#' analyze(fit)
#' @details See https://stats.stackexchange.com/questions/59062/multiple-linear-regression-simulation
#'
#' @author TPArrow
#'
#' @export
simulate_data_regression <- function(coefs = 0.5, sample = 100, error = 0) {

  # Prevent error
  coefs[coefs == 0] <- 0.01

  y <- rnorm(sample, 0, 1)

  n_var <- length(coefs)
  X <- scale(matrix(rnorm(sample * (n_var), 0, 1), ncol = n_var))
  X <- cbind(y, X)

  # find the current correlation matrix
  cor_0 <- var(X)

  # cholesky decomposition to get independence
  chol_0 <- solve(chol(cor_0))

  X <- X %*% chol_0

  # create new correlation structure (zeros can be replaced with other r vals)
  coefs_structure <- diag(x = 1, nrow = n_var + 1, ncol = n_var + 1)
  coefs_structure[-1, 1] <- coefs
  coefs_structure[1, -1] <- coefs

  X <- X %*% chol(coefs_structure) * sd(y) + mean(y)
  X <- X[, -1]

  # Add noise
  y <- y + rnorm(sample, 0, error)

  data <- data.frame(X)
  names(data) <- paste0("V", 1:n_var)
  data$y <- as.vector(y)

  return(data)
}






#' Standardize.
#'
#' Standardize objects. See the documentation for your object's class:
#' \itemize{
#' \item{\link[=standardize.numeric]{standardize.numeric}}
#' \item{\link[=standardize.data.frame]{standardize.data.frame}}
#' \item{\link[=standardize.stanreg]{standardize.stanreg}}
#' \item{\link[=standardize.lm]{standardize.lm}}
#' \item{\link[=standardize.glm]{standardize.glm}}
#'  }
#'
#' @param x Object.
#' @param ... Arguments passed to or from other methods.
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
standardize <- function(x, ...) {
  UseMethod("standardize")
}
























#' Standardize (scale and reduce) numeric variables.
#'
#' Standardize (Z-score, "normalize") a vector.
#'
#' @param x Numeric vector.
#' @param normalize Will perform a normalization instead of a standardization. This scales all numeric variables in the range 0 - 1.
#' @param ... Arguments passed to or from other methods.
#'
#' @examples
#' standardize(x = c(1, 4, 6, 2))
#' standardize(x = c(1, 4, 6, 2), normalize = TRUE)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#'
#' @export
standardize.numeric <- function(x, normalize = FALSE, ...) {
  if (all(is.na(x)) | length(unique(x)) == 2) {
    return(x)
  }

  if (normalize == FALSE) {
    return(as.vector(scale(x, ...)))
  } else {
    return(as.vector((x - min(x, na.rm = TRUE)) / diff(range(x, na.rm = TRUE), na.rm = TRUE)))
  }
}


















#' Standardize (scale and reduce) Dataframe.
#'
#' Selects numeric variables and standardize (Z-score, "normalize") them.
#'
#' @param x Dataframe.
#' @param subset Character or list of characters of column names to be
#' standardized.
#' @param except Character or list of characters of column names to be excluded
#' from standardization.
#' @param normalize Will perform a normalization instead of a standardization. This scales all numeric variables in the range 0 - 1.
#' @param ... Arguments passed to or from other methods.
#'
#' @return Dataframe.
#'
#' @examples
#' \dontrun{
#' df <- data.frame(
#'   Participant = as.factor(rep(1:25, each = 4)),
#'   Condition = base::rep_len(c("A", "B", "C", "D"), 100),
#'   V1 = rnorm(100, 30, .2),
#'   V2 = runif(100, 3, 5),
#'   V3 = rnorm(100, 100, 10)
#' )
#'
#' dfZ <- standardize(df)
#' dfZ <- standardize(df, except = "V3")
#' dfZ <- standardize(df, except = c("V1", "V2"))
#' dfZ <- standardize(df, subset = "V3")
#' dfZ <- standardize(df, subset = c("V1", "V2"))
#' dfZ <- standardize(df, normalize = TRUE)
#'
#' # Respects grouping
#' dfZ <- df %>%
#'   dplyr::group_by(Participant) %>%
#'   standardize(df)
#' }
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#'
#' @importFrom purrr keep discard
#' @import dplyr
#' @export
standardize.data.frame <- function(x, subset = NULL, except = NULL, normalize = FALSE, ...) {
  if (inherits(x, "grouped_df")) {
    dfZ <- x %>% dplyr::do_(".standardize_df(., subset=subset, except=except, normalize=normalize, ...)")
  } else {
    dfZ <- .standardize_df(x, subset = subset, except = except, normalize = normalize, ...)
  }

  return(dfZ)
}

















#' @keywords internal
.standardize_df <- function(x, subset = NULL, except = NULL, normalize = FALSE, ...) {
  df <- x

  # Variable order
  var_order <- names(df)

  # Keep subset
  if (!is.null(subset) && subset %in% names(df)) {
    to_keep <- as.data.frame(df[!names(df) %in% c(subset)])
    df <- df[names(df) %in% c(subset)]
  } else {
    to_keep <- NULL
  }

  # Remove exceptions
  if (!is.null(except) && except %in% names(df)) {
    if (is.null(to_keep)) {
      to_keep <- as.data.frame(df[except])
    } else {
      to_keep <- cbind(to_keep, as.data.frame(df[except]))
    }

    df <- df[!names(df) %in% c(except)]
  }

  # Remove non-numerics
  dfother <- purrr::discard(df, is.numeric)
  dfnum <- purrr::keep(df, is.numeric)

  # Scale
  dfnum <- as.data.frame(sapply(dfnum, standardize, normalize = normalize))

  # Add non-numerics
  if (is.null(ncol(dfother))) {
    df <- dfnum
  } else {
    df <- dplyr::bind_cols(dfother, dfnum)
  }

  # Add exceptions
  if (!is.null(subset) | !is.null(except) && exists("to_keep")) {
    df <- dplyr::bind_cols(df, to_keep)
  }

  # Reorder
  df <- df[var_order]

  return(df)
}













#' Standardize Posteriors.
#'
#' Compute standardized posteriors from which to get standardized coefficients.
#'
#' @param x A stanreg model.
#' @param method "refit" (default) will entirely refit the model based on standardized data. Can take a long time. Other post-hoc methods are "posterior" (based on estimated SD) or "sample" (based on the sample SD).
#' @param ... Arguments passed to or from other methods.
#'
#' @examples
#' \dontrun{
#' library(psycho)
#' library(rstanarm)
#'
#' fit <- rstanarm::stan_glm(Sepal.Length ~ Sepal.Width * Species, data = iris)
#' fit <- rstanarm::stan_glm(Sepal.Length ~ Sepal.Width * Species, data = standardize(iris))
#' posteriors <- standardize(fit)
#' posteriors <- standardize(fit, method = "posterior")
#' }
#'
#' @author \href{https://github.com/jgabry}{Jonah Gabry}, \href{https://github.com/bgoodri}{bgoodri}
#'
#' @seealso https://github.com/stan-dev/rstanarm/issues/298
#'
#' @importFrom utils capture.output
#' @export
standardize.stanreg <- function(x, method = "refit", ...) {
  fit <- x

  predictors <- get_info(fit)$predictors
  predictors <- c("(Intercept)", predictors)

  if (method == "sample") {
    # By jgabry
    predictors <- all.vars(as.formula(fit$formula))
    outcome <- predictors[[1]]
    X <- as.matrix(model.matrix(fit)[, -1]) # -1 to drop column of 1s for intercept
    sd_X_over_sd_y <- apply(X, 2, sd) / sd(fit$data[[outcome]])
    beta <- as.matrix(fit, pars = colnames(X)) # posterior distribution of regression coefficients
    posteriors_std <- sweep(beta, 2, sd_X_over_sd_y, "*") # multiply each row of b by sd_X_over_sd_y
  } else if (method == "posterior") {
    # By bgoordi
    X <- model.matrix(fit)
    # if(preserve_factors == TRUE){
    #   X <- as.data.frame(X)
    #   X[!names(as.data.frame(X)) %in% predictors] <- scale(X[!names(as.data.frame(X)) %in% predictors])
    #   X <- as.matrix(X)
    # }
    sd_X <- apply(X, MARGIN = 2, FUN = sd)[-1]
    sd_Y <- apply(rstanarm::posterior_predict(fit), MARGIN = 1, FUN = sd)
    beta <- as.matrix(fit)[, 2:ncol(X), drop = FALSE]
    posteriors_std <- sweep(
      sweep(beta, MARGIN = 2, STATS = sd_X, FUN = `*`),
      MARGIN = 1, STATS = sd_Y, FUN = `/`
    )
  } else {
    useless_output <- capture.output(fit_std <- update(fit, data = standardize(fit$data)))
    posteriors_std <- as.data.frame(fit_std)
  }

  return(posteriors_std)
}







#' Standardize Coefficients.
#'
#' Compute standardized coefficients.
#'
#' @param x A linear model.
#' @param method The standardization method. Can be "refit" (will entirely refit the model based on standardized data. Can take some time) or "agresti".
#' @param ... Arguments passed to or from other methods.
#'
#' @examples
#' \dontrun{
#' library(psycho)
#' fit <- glm(Sex ~ Adjusting, data = psycho::affective, family = "binomial")
#' fit <- lme4::glmer(Sex ~ Adjusting + (1 | Sex), data = psycho::affective, family = "binomial")
#'
#' standardize(fit)
#' }
#'
#' @author Kamil Barton
#' @importFrom stats model.frame model.response model.matrix
#'
#' @seealso https://think-lab.github.io/d/205/
#'
#' @export
standardize.glm <- function(x, method = "refit", ...) {
  fit <- x

  if (method == "agresti") {
    coefs <- MuMIn::coefTable(fit)[, 1:2]
    X <- as.matrix(model.matrix(fit)[, -1]) # -1 to drop column of 1s for intercept
    sd_X <- sd(X, na.rm = TRUE)
    coefs <- coefs * sd_X
  } else {
    # refit method
    data <- get_data(fit)
    fit_std <- update(fit, data = standardize(data))


    coefs <- MuMIn::coefTable(fit_std)[, 1:2]
  }

  coefs <- as.data.frame(coefs)
  names(coefs) <- c("Coef_std", "SE_std")
  return(coefs)
}

#' @export
standardize.glmerMod <- standardize.glm



#' Standardize Coefficients.
#'
#' Compute standardized coefficients.
#'
#' @param x A linear model.
#' @param method The standardization method. Can be "refit" (will entirely refit the model based on standardized data. Can take some time) or "posthoc".
#' @param partial_sd Logical, if set to TRUE, model coefficients are multiplied by partial SD, otherwise they are multiplied by the ratio of the standard deviations of the independent variable and dependent variable.
#' @param preserve_factors Standardize factors-related coefs only by the dependent variable (i.e., do not standardize the dummies generated by factors).
#' @param ... Arguments passed to or from other methods.
#'
#' @examples
#' \dontrun{
#' library(psycho)
#'
#' df <- mtcars %>%
#'   mutate(cyl = as.factor(cyl))
#'
#' fit <- lm(wt ~ mpg * cyl, data = df)
#' fit <- lmerTest::lmer(wt ~ mpg * cyl + (1 | gear), data = df)
#'
#' summary(fit)
#' standardize(fit)
#' }
#'
#' @author Kamil Barton
#' @importFrom stats model.frame model.response model.matrix
#'
#' @export
standardize.lm <- function(x, method = "refit", partial_sd = FALSE, preserve_factors = TRUE, ...) {
  fit <- x

  if (method == "posthoc") {
    coefs <- .standardize_coefs(fit, partial_sd = partial_sd, preserve_factors = preserve_factors)
  } else {
    data <- get_data(fit)
    fit_std <- update(fit, data = standardize(data))
    coefs <- MuMIn::coefTable(fit_std)[, 1:2]
  }

  coefs <- as.data.frame(coefs)
  names(coefs) <- c("Coef_std", "SE_std")
  return(coefs)
}


#' @export
standardize.lmerMod <- standardize.lm

















#' @keywords internal
.partialsd <-
  function(x, sd, vif, n, p = length(x) - 1) {
    sd * sqrt(1 / vif) * sqrt((n - 1) / (n - p))
  }


#' @importFrom stats vcov
#' @keywords internal
.vif <-
  function(x) {
    v <- vcov(x)
    nam <- dimnames(v)[[1L]]
    if (dim(v)[1L] < 2L) {
      return(structure(rep_len(1, dim(v)[1L]),
        names = dimnames(v)[[1L]]
      ))
    }
    if ((ndef <- sum(is.na(MuMIn::coeffs(x)))) > 0L) {
      stop(sprintf(ngettext(
        ndef, "one coefficient is not defined",
        "%d coefficients are not defined"
      ), ndef))
    }
    o <- attr(model.matrix(x), "assign")
    if (any(int <- (o == 0))) {
      v <- v[!int, !int, drop = FALSE]
    } else {
      warning("no intercept: VIFs may not be sensible")
    }
    d <- sqrt(diag(v))
    rval <- numeric(length(nam))
    names(rval) <- nam
    rval[!int] <- diag(solve(v / (d %o% d)))
    rval[int] <- 1
    rval
  }



#' @importFrom stats nobs vcov
#' @keywords internal
.standardize_coefs <- function(fit, partial_sd = FALSE, preserve_factors = TRUE, ...) {
  # coefs <- MuMIn::coefTable(fit, ...)
  coefs <- as.data.frame(MuMIn::coefTable(fit))
  model_matrix <- model.matrix(fit)

  predictors <- get_info(fit)$predictors
  predictors <- c("(Intercept)", predictors)

  if (preserve_factors == TRUE) {
    response_sd <- sd(model.response(model.frame(fit)))
    factors <- as.data.frame(model_matrix)[!names(as.data.frame(model_matrix)) %in% predictors]
    bx_factors <- rep(1 / response_sd, length(names(factors)))
    bx_factors <- data.frame(t(bx_factors))
    names(bx_factors) <- names(factors)
    coefs_factors <- coefs[names(factors), ]
    model_matrix_factors <- as.matrix(factors)

    coefs <- coefs[!rownames(coefs) %in% names(factors), ]
    model_matrix <- as.matrix(as.data.frame(model_matrix)[names(as.data.frame(model_matrix)) %in% predictors])
  }

  if (partial_sd == TRUE) {
    bx <- .partialsd(
      coefs[, 1L],
      apply(model_matrix, 2L, sd),
      .vif(fit),
      nobs(fit),
      sum(attr(model_matrix, "assign") != 0)
    )
  } else {
    response_sd <- sd(model.response(model.frame(fit)))
    bx <- apply(model_matrix, 2L, sd) / response_sd
  }
  bx <- as.data.frame(t(bx))
  names(bx) <- row.names(coefs)

  if (preserve_factors == TRUE) {
    bx <- cbind(bx, bx_factors)
  }


  # coefs <- MuMIn::coefTable(fit, ...)
  coefs <- as.data.frame(MuMIn::coefTable(fit))
  multiplier <- as.numeric(bx[row.names(coefs)])

  coefs[, 1L:2L] <- coefs[, 1L:2L] * multiplier
  colnames(coefs)[1L:2L] <- c("Coef.std", "SE.std")
  return(coefs)
}







#' Print the results.
#'
#' Print the results.
#'
#' @param object A psychobject class object.
#' @param round Round the ouput.
#' @param ... Further arguments passed to or from other methods.
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @method summary psychobject
#' @export
summary.psychobject <- function(object, round = NULL, ...) {
  summary <- object$summary

  if (!is.null(round)) {
    nums <- dplyr::select_if(summary, is.numeric)
    nums <- round(nums, round)
    fact <- dplyr::select_if(summary, is.character)
    fact <- cbind(fact, dplyr::select_if(summary, is.factor))
    summary <- cbind(fact, nums)
  }

  return(summary)
}







#' Extract values as list.
#'
#' @param x A psychobject class object.
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
values <- function(x) {
  values <- x$values
  return(values)
}




#' Analyze blavaan (SEM or CFA) objects.
#'
#' Analyze blavaan (SEM or CFA) objects.
#'
#' @param x lavaan object.
#' @param CI Credible interval level.
#' @param standardize Compute standardized coefs.
#' @param ... Arguments passed to or from other methods.
#'
#' @return output
#'
#' @examples
#' library(psycho)
#' library(lavaan)
#'
#' model <- " visual  =~ x1 + x2 + x3\ntextual =~ x4 + x5 + x6\nspeed   =~ x7 + x8 + x9 "
#' x <- lavaan::cfa(model, data = HolzingerSwineford1939)
#'
#' rez <- analyze(x)
#' print(rez)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @seealso
#' https://www.researchgate.net/post/Whats_the_standard_of_fit_indices_in_SEM
#'
#' @importFrom lavaan parameterEstimates fitmeasures
#' @importFrom blavaan standardizedposterior
#'
#' @export
analyze.blavaan <- function(x, CI = 90, standardize = FALSE, ...) {
  fit <- x


  # Processing
  # -------------
  values <- list()
  values$CI <- CI

  # Fit measures
  values$Fit_Measures <- interpret_lavaan(fit)


  # Text
  # -------------
  computations <- .get_info_computations(fit)
  fitmeasures <- values$Fit_Measures$text
  text <- paste0(
    "A Bayesian model was fitted (",
    computations,
    "). The fit indices are as following: ",
    fitmeasures
  )

  # Summary
  # -------------
  summary <- .summary_blavaan(fit, CI = CI, standardize = standardize)

  # Plot
  # -------------
  plot <- "Use `get_graph` in association with ggraph."

  output <- list(text = values$Fit_Measures$text, plot = plot, summary = summary, values = values)

  class(output) <- c("psychobject", "list")
  return(output)
}






#' @keywords internal
.get_info_computations <- function(fit) {
  chains <- blavaan::blavInspect(fit, "n.chains")
  sample <- fit@external$sample
  warmup <- fit@external$burnin
  text <- paste0(
    chains,
    " chains, each with iter = ",
    sample,
    "; warmup = ",
    warmup
  )
  return(text)
}




#' @keywords internal
.process_blavaan <- function(fit, standardize = FALSE, CI = 90) {
  # Get relevant rows
  PE <- parameterEstimates(fit,
    se = FALSE, ci = FALSE, remove.eq = FALSE, remove.system.eq = TRUE,
    remove.ineq = FALSE, remove.def = FALSE,
    add.attributes = TRUE
  )
  if (!("group" %in% names(PE))) PE$group <- 1
  newpt <- fit@ParTable
  pte2 <- which(newpt$free > 0)
  relevant_rows <- match(
    with(newpt, paste(lhs[pte2], op[pte2], rhs[pte2], group[pte2], sep = "")),
    paste(PE$lhs, PE$op, PE$rhs, PE$group, sep = "")
  )

  # Priors
  priors <- rep(NA, nrow(PE))
  priors[relevant_rows] <- newpt$prior[pte2]
  priors[is.na(PE$prior)] <- ""




  # Posterior
  if (standardize == FALSE) {
    posteriors <- blavaan::blavInspect(fit, "draws") %>%
      as.matrix() %>%
      as.data.frame()
    names(posteriors) <- names(lavaan::coef(fit))
  } else {
    posteriors <- blavaan::standardizedposterior(fit) %>%
      as.data.frame()
  }



  # Effects
  MPE <- c()
  Median <- c()
  MAD <- c()
  Effect <- c()
  CI_lower <- c()
  CI_higher <- c()
  for (effect in names(posteriors)) {
    posterior <- posteriors[[effect]]
    Effect <- c(Effect, effect)
    MPE <- c(MPE, mpe(posterior)$MPE)
    Median <- c(Median, median(posterior))
    MAD <- c(MAD, mad(posterior))

    CI_values <- HDI(posterior, prob = CI / 100)
    CI_lower <- c(CI_lower, CI_values$values$HDImin)
    CI_higher <- c(CI_higher, CI_values$values$HDImax)
  }

  if (standardize == FALSE) {
    Effects <- rep(NA, nrow(PE))
    Effects[relevant_rows] <- Effect
    MPEs <- rep(NA, nrow(PE))
    MPEs[relevant_rows] <- MPE
    Medians <- rep(NA, nrow(PE))
    Medians[relevant_rows] <- Median
    MADs <- rep(NA, nrow(PE))
    MADs[relevant_rows] <- MAD
    CI_lowers <- rep(NA, nrow(PE))
    CI_lowers[relevant_rows] <- CI_lower
    CI_highers <- rep(NA, nrow(PE))
    CI_highers[relevant_rows] <- CI_higher
  } else {
    Effects <- Effect
    MPEs <- MPE
    Medians <- Median
    MADs <- MAD
    CI_lowers <- CI_lower
    CI_highers <- CI_higher
  }

  data <- data.frame(
    "Effect" = Effects,
    "Median" = Medians,
    "MAD" = MADs,
    "MPE" = MPEs,
    "CI_lower" = CI_lowers,
    "CI_higher" = CI_highers,
    "Prior" = priors
  )

  return(data)
}



#' @keywords internal
.summary_blavaan <- function(fit, CI = 90, standardize = FALSE) {
  solution <- lavaan::parameterEstimates(fit, se = TRUE, ci = TRUE, standardized = FALSE, level = CI / 100)

  solution <- solution %>%
    rename(
      "From" = "rhs",
      "To" = "lhs",
      "Operator" = "op",
      "Coef" = "est",
      "SE" = "se",
      "CI_lower" = "ci.lower",
      "CI_higher" = "ci.upper"
    ) %>%
    mutate(Type = dplyr::case_when(
      Operator == "=~" ~ "Loading",
      Operator == "~" ~ "Regression",
      Operator == "~~" ~ "Correlation",
      TRUE ~ NA_character_
    )) %>%
    select(one_of(c("To", "Operator", "From", "Type"))) %>%
    mutate_("Effect" = "as.character(paste0(To, Operator, From))") %>%
    full_join(.process_blavaan(fit, CI = CI, standardize = standardize) %>%
      mutate_("Effect" = "as.character(Effect)"), by = "Effect") %>%
    select_("-Effect") %>%
    mutate_(
      "Median" = "replace_na(Median, 1)",
      "MAD" = "replace_na(MAD, 0)",
      "MPE" = "replace_na(MPE, 100)"
    ) %>%
    select(one_of(c("From", "Operator", "To", "Median", "MAD", "CI_lower", "CI_higher", "MPE", "Prior", "Type"))) %>%
    dplyr::filter_("Operator != '~1'")


  return(solution)
}










#' Analyze fa objects.
#'
#' Analyze fa objects.
#'
#' @param x An psych object.
#' @param labels Supply a additional column with e.g. item labels.
#' @param treshold 'max' or numeric. The treshold over which to associate an item with its component.
#' @param ... Arguments passed to or from other methods.
#'
#' @return output
#'
#' @examples
#' library(psycho)
#' library(psych)
#'
#' x <- psych::fa(psych::Thurstone.33, 2)
#'
#' results <- analyze(x)
#' print(results)
#' summary(results)
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
analyze.fa <- function(x, labels = NULL, treshold = "max", ...) {
  loadings <- format_loadings(x, labels)

  values <- list()
  values$variance <- x$Vaccounted
  values$loadings <- loadings$loadings
  values$loadings_max <- loadings$max
  values$cfa_model <- get_cfa_model(loadings$loadings, treshold = treshold)

  text <- .fa_variance_text(values$variance)
  text <- paste0(text, "\n\n", format(values$cfa_model))
  summary <- values$loadings
  plot <- plot_loadings(values$loadings)

  output <- list(text = text, plot = plot, summary = summary, values = values)

  class(output) <- c("psychobject", "list")
  return(output)
}









#' @export
.fa_variance_text <- function(variance) {
  variance <- as.data.frame(variance)
  n_factors <- ncol(variance)

  if (ncol(variance) == 1) {
    t <- as.data.frame(t(variance))
    tot_var <- t$`Proportion Var`
    text <- paste0(
      "The unique component accounted for ",
      format_digit(tot_var * 100),
      "% of the total variance."
    )
  } else {
    t <- as.data.frame(t(variance))
    tot_var <- max(t$`Cumulative Var`)

    factors <- names(variance)
    var <- variance["Proportion Var", ]
    text_var <- paste0(factors,
      " = ",
      format_digit(var * 100),
      "%",
      collapse = ", "
    )

    text <- paste0(
      "The ",
      n_factors,
      " components accounted for ",
      format_digit(tot_var * 100),
      "% of the total variance ("
    )
    text <- paste0(text, text_var, ").")
  }

  return(text)
}







#' Format the loadings of a factor analysis.
#'
#' Format the loadings of a factor analysis.
#'
#' @param x An psych object.
#' @param labels Supply a additional column with e.g. item labels.
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @examples
#' \dontrun{
#' library(psycho)
#'
#' x <- psych::fa(psych::Thurstone.33, 2)
#' format_loadings(x)
#' }
#'
#' @import dplyr
#' @export
format_loadings <- function(x, labels = NULL) {


  # Check loadings and remove those inferior to a treshold
  loadings <- x$loadings %>%
    unclass() %>%
    as.data.frame()

  # Save n factors
  n_factors <- length(loadings)

  # Add item labels
  loadings$Item <- rownames(loadings)
  if (length(labels) == nrow(loadings)) {
    loadings$Label <- labels
  } else {
    loadings$Label <- 1:nrow(loadings)
  }

  # Keep Order
  loadings$N <- 1:nrow(loadings)


  # Select the max loading for each item
  max <- get_loadings_max(loadings)


  # Reorder the loading matrix accordingly
  loadings <- loadings[max$N, ] %>%
    select_("N", "Item", "Label", "everything()")

  return(list(loadings = loadings, max = max))
}



#' Get loadings max.
#'
#' Get loadings max.
#'
#' @param loadings Formatted loadings.
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @examples
#' \dontrun{
#' library(psycho)
#'
#' x <- psych::fa(psych::Thurstone.33, 2)
#' get_loadings_max(format_loadings(x)$loadings)
#' }
#'
#' @import dplyr
#' @export
get_loadings_max <- function(loadings) {
  max <- loadings %>%
    tidyr::gather_("Component", "Loading", names(loadings)[!names(loadings) %in% c("Item", "N", "Label")]) %>%
    dplyr::group_by_("Item") %>%
    dplyr::slice_("which.max(abs(Loading))") %>%
    dplyr::arrange_("Component", "desc(Loading)")
  return(max)
}



#' Get CFA model.
#'
#' Get CFA model.
#'
#' @param loadings Formatted loadings.
#' @param treshold 'max' or numeric. The treshold over which to associate an item with its component.
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @examples
#' \dontrun{
#' library(psycho)
#'
#' x <- psych::fa(psych::Thurstone.33, 2)
#' loadings <- format_loadings(x)$loadings
#' get_cfa_model(loadings, treshold = "max")
#' get_cfa_model(loadings, treshold = 0.1)
#' }
#'
#' @import dplyr
#' @export
get_cfa_model <- function(loadings, treshold = "max") {
  if (treshold == "max") {
    filtered_loadings <- get_loadings_max(loadings)
  } else {
    filtered_loadings <- loadings %>%
      tidyr::gather_("Component", "Loading", names(loadings)[!names(loadings) %in% c("Item", "N", "Label")]) %>%
      filter_("Loading > treshold")
  }

  cfa_model <- filtered_loadings %>%
    select_("Item", "Component") %>%
    group_by_("Component") %>%
    summarise_("Observed" = 'paste(Item, collapse=" + ")') %>%
    transmute_("Latent_Variable" = 'paste(Component, Observed, sep=" =~ ")') %>%
    pull()

  cfa_model <- c("#Latent variables", cfa_model) %>%
    paste(collapse = "\n")

  return(cfa_model)
}




#' Plot loadings.
#'
#' Plot loadings.
#'
#' @param loadings Loadings by variable.
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @examples
#' \dontrun{
#' library(psycho)
#'
#' x <- psych::fa(psych::Thurstone.33, 2)
#' plot_loadings(format_loadings(x)$loadings)
#' }
#'
#' @import dplyr
#' @export
plot_loadings <- function(loadings) {
  if (all(loadings$Label != loadings$N)) {
    loadings$Item <- paste0(loadings$Label, " (", loadings$Item, ")")
  }

  p <- loadings %>%
    gather("Component", "Loading", matches("\\d$")) %>%
    mutate_("Loading" = "abs(Loading)") %>%
    mutate_("Item" = "factor(Item, levels=rev(get_loadings_max(loadings)$Item))") %>%
    ggplot(aes_string(y = "Loading", x = "Item", fill = "Component")) +
    geom_bar(stat = "identity") +
    coord_flip() +
    ylab("\nLoading Strength") +
    xlab("Item\n")

  return(p)
}







#' Analyze glm objects.
#'
#' Analyze glm objects.
#'
#' @param x glm object.
#' @param CI Confidence interval bounds. Set to NULL turn off their computation.
#' @param effsize_rules Grid for effect size interpretation. See \link[=interpret_odds]{interpret_odds}.
#' @param ... Arguments passed to or from other methods.
#'
#' @return output
#'
#' @examples
#' library(psycho)
#' fit <- glm(Sex ~ Adjusting, data = psycho::affective, family = "binomial")
#'
#' results <- analyze(fit)
#' summary(results)
#' print(results)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @references Nakagawa, S., & Schielzeth, H. (2013). A general and simple method for obtaining R2 from generalized linear mixed-effects models. Methods in Ecology and Evolution, 4(2), 133-142.
#'
#' @seealso \link[=get_R2.glm]{"get_R2.glm"}
#'
#' @import dplyr
#' @importFrom stats formula
#' @importFrom stringr str_squish
#' @export
analyze.glm <- function(x, CI = 95, effsize_rules = "cohen1988", ...) {


  # Processing
  # -------------
  fit <- x

  if (fit$family$family != "binomial") {
    stop(paste("Models of family", fit$family$family, "not supported yet."))
  }

  info <- get_info(fit)
  outcome <- info$outcome
  predictors <- info$predictors

  # R2 <- tjur_D(fit)
  R2 <- get_R2(fit, method = "nakagawa")

  # Summary
  # -------------
  summary <- data.frame(summary(fit)$coefficients)

  summary$Variable <- rownames(summary)
  summary$Coef <- summary$Estimate
  summary$SE <- summary$`Std..Error`
  summary$z <- summary$`z.value`
  summary$p <- summary$`Pr...z..`

  # standardized coefficients
  standardized <- tibble::rownames_to_column(standardize(fit, method = "refit"), "Variable")
  summary <- merge(summary, standardized, by = "Variable", all.x = TRUE, sort = FALSE)
  summary$Effect_Size <- c(NA, interpret_odds(tail(summary$Coef_std, -1), log = TRUE, rules = effsize_rules))

  summary <- dplyr::select_(
    summary, "Variable", "Coef", "SE", "z", "Coef_std", "SE_std",
    "p", "Effect_Size"
  )

  if (!is.null(CI)) {
    CI_values <- suppressMessages(confint(fit, level = CI / 100))
    CI_values <- tail(CI_values, n = length(rownames(summary)))
    summary$CI_lower <- CI_values[, 1]
    summary$CI_higher <- CI_values[, 2]
  }


  # Varnames
  varnames <- summary$Variable
  row.names(summary) <- varnames



  # Values
  # -------------
  # Initialize empty values
  values <- list(model = list(), effects = list())

  # Loop over all variables
  for (varname in varnames) {
    if (summary[varname, "p"] < .1) {
      significance <- " "
    } else {
      significance <- " not "
    }

    if (!is.null(CI)) {
      CI_text <- paste0(
        ", ",
        CI, "% CI [",
        format_digit(summary[varname, "CI_lower"]),
        ", ",
        format_digit(summary[varname, "CI_higher"]),
        "]"
      )
    } else {
      CI_text <- ""
    }



    text <- paste0(
      "The effect of ",
      varname,
      " is",
      significance,
      "significant (beta = ",
      format_digit(summary[varname, "Coef"], 2), ", SE = ",
      format_digit(summary[varname, "SE"], 2),
      CI_text,
      ", z = ",
      format_digit(summary[varname, "z"], 2), ", p ",
      format_p(summary[varname, "p"], stars = FALSE),
      ") and can be considered as ",
      tolower(summary[varname, "Effect_Size"]),
      " (std. beta = ",
      format_digit(summary[varname, "Coef_std"], 2),
      ", std. SE = ",
      format_digit(summary[varname, "SE_std"], 2), ")."
    )

    if (varname == "(Intercept)") {
      text <- paste0(
        "The model's intercept is at ",
        format_digit(summary[varname, "Coef"], 2),
        " (SE = ",
        format_digit(summary[varname, "SE"], 2),
        CI_text,
        "). Within this model:"
      )
    }

    values$effects[[varname]] <- list(
      Coef = summary[varname, "Coef"],
      SE = summary[varname, "SE"],
      CI_lower = summary[varname, "CI_lower"],
      CI_higher = summary[varname, "CI_higher"],
      z = summary[varname, "z"],
      Coef_std = summary[varname, "Coef_std"],
      SE_std = summary[varname, "SE_std"],
      p = summary[varname, "p"],
      Effect_Size = summary[varname, "Effect_Size"],
      Text = text
    )
  }



  # Text
  # -------------
  text <- c(paste0(
    "The overall model predicting ",
    outcome,
    " (formula = ",
    stringr::str_squish(paste0(format(stats::formula(fit)), collapse = "")),
    ") has an explanatory power of ",
    format_digit(R2 * 100, 2),
    "%. ",
    values$effects[["(Intercept)"]]$Text
  ))

  for (varname in varnames) {
    if (varname != "(Intercept)") {
      text <- c(text, paste("   -", values$effects[[varname]]$Text))
    }
  }



  # Plot
  # -------------
  plot <- "Not available yet"

  output <- list(text = text, plot = plot, summary = summary, values = values)

  class(output) <- c("psychobject", "list")
  return(output)
}





#' Analyze glmerMod objects.
#'
#' Analyze glmerMod objects.
#'
#' @param x merModLmerTest object.
#' @param CI Bootsrapped confidence interval bounds (slow). Set to NULL turn off their computation.
#' @param effsize_rules Grid for effect size interpretation. See \link[=interpret_odds]{interpret_odds}.
#' @param ... Arguments passed to or from other methods.
#'
#' @return output
#'
#' @examples
#' \dontrun{
#' library(psycho)
#' library(lme4)
#'
#' fit <- lme4::glmer(vs ~ wt + (1 | gear), data = mtcars, family = "binomial")
#'
#' results <- analyze(fit)
#' summary(results)
#' print(results)
#' }
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @references Nakagawa, S., & Schielzeth, H. (2013). A general and simple method for obtaining R2 from generalized linear mixed-effects models. Methods in Ecology and Evolution, 4(2), 133-142.
#'
#' @importFrom MuMIn r.squaredGLMM
#' @importFrom MuMIn std.coef
#' @importFrom stringr str_squish
#' @import lmerTest
#' @import dplyr
#' @export
analyze.glmerMod <- function(x, CI = 95, effsize_rules = "cohen1988", ...) {


  # Processing
  # -------------
  fit <- x

  info <- get_info(fit)
  R2 <- tryCatch({
    get_R2(fit)
  }, error = function(e) {
    warning("Couldn't compute R2. Might be caused by the presence of missing data.")
    R2 <- list(R2m = NA, R2c = NA)
    return(R2)
  })







  # Summary
  # -------------
  summary <- data.frame(summary(fit)$coefficients)

  summary$Variable <- rownames(summary)
  summary$Coef <- summary$Estimate
  summary$SE <- summary$`Std..Error`
  summary$z <- summary$`z.value`
  summary$p <- summary$`Pr...z..`

  # standardized coefficients
  standardized <- tibble::rownames_to_column(standardize(fit, method = "refit"), "Variable")
  summary <- merge(summary, standardized, by = "Variable", all.x = TRUE, sort = FALSE)
  summary$Effect_Size <- c(NA, interpret_odds(tail(summary$Coef_std, -1), log = TRUE, rules = effsize_rules))


  # Summary
  summary <- dplyr::select_(summary, "Variable", "Coef", "SE", "z", "p", "Coef_std", "SE_std", "Effect_Size")

  # CI computation
  if (!is.null(CI)) {
    CI_values <- tryCatch({
      suppressMessages(confint(fit, level = CI / 100))
    }, error = function(e) {
      warning("Couldn't compute CI. Skipping.")
      CI_values <- NA
      return(CI_values)
    })
    if (!all(is.na(CI_values))) {
      CI_values <- tail(CI_values, n = length(rownames(summary)))
      summary$CI_lower <- CI_values[, 1]
      summary$CI_higher <- CI_values[, 2]
    } else {
      CI <- NULL
    }
  }


  # Varnames
  varnames <- summary$Variable
  row.names(summary) <- varnames


  # Values
  # -------------
  # Initialize empty values
  values <- list(model = list(), effects = list())
  values$model$R2m <- R2$R2m
  values$model$R2c <- R2$R2c

  # Loop over all variables
  for (varname in varnames) {
    if (summary[varname, "p"] < .1) {
      significance <- " "
    } else {
      significance <- " not "
    }

    if (!is.null(CI)) {
      CI_text <- paste0(
        ", ",
        CI, "% CI [",
        format_digit(summary[varname, "CI_lower"]),
        ", ",
        format_digit(summary[varname, "CI_higher"]),
        "]"
      )
    } else {
      CI_text <- ""
    }



    if (varname == "(Intercept)") {
      text <- paste0(
        "The model's intercept is at ",
        format_digit(summary[varname, "Coef"], 2),
        " (SE = ",
        format_digit(summary[varname, "SE"], 2),
        CI_text,
        "). Within this model:"
      )
    } else {
      text <- paste0(
        "The effect of ",
        varname,
        " is",
        significance,
        "significant (beta = ",
        format_digit(summary[varname, "Coef"], 2),
        ", SE = ",
        format_digit(summary[varname, "SE"], 2),
        CI_text,
        ", z = ",
        format_digit(summary[varname, "z"], 2),
        ", p ",
        format_p(summary[varname, "p"], stars = FALSE),
        ") and can be considered as ",
        tolower(summary[varname, "Effect_Size"]),
        " (std. beta = ",
        format_digit(summary[varname, "Coef_std"], 2),
        ", std. SE = ",
        format_digit(summary[varname, "SE_std"], 2),
        ")."
      )
    }

    values$effects[[varname]] <- list(
      Coef = summary[varname, "Coef"],
      SE = summary[varname, "SE"],
      z = summary[varname, "z"],
      p = summary[varname, "p"],
      Effect_Size = summary[varname, "Effect_Size"],
      Text = text
    )
  }



  # Text
  # -------------
  text <- c(paste0(
    "The overall model predicting ",
    info$outcome,
    " (formula = ",
    format(info$formula),
    ") has an explanatory power (conditional R2) of ",
    format_digit(R2$R2c * 100, 2),
    "%, in which the fixed effects' part is ",
    format_digit(R2$R2m * 100, 2), "% (marginal R2). ",
    values$effects[["(Intercept)"]]$Text
  ))

  for (varname in varnames) {
    if (varname != "(Intercept)") {
      text <- c(text, paste("   -", values$effects[[varname]]$Text))
    }
  }



  # Plot
  # -------------
  plot <- "Not available yet"

  output <- list(text = text, plot = plot, summary = summary, values = values)

  class(output) <- c("psychobject", "list")
  return(output)
}









#' Analyze htest (correlation, t-test...) objects.
#'
#' Analyze htest (correlation, t-test...) objects.
#'
#' @param x htest object.
#' @param effsize_rules Grid for effect size interpretation. See \link[=interpret_r]{interpret_r}.
#' @param ... Arguments passed to or from other methods.
#'
#' @return output
#'
#' @examples
#' library(psycho)
#'
#' df <- psycho::affective
#'
#' x <- t.test(df$Tolerating, df$Adjusting)
#' x <- t.test(df$Tolerating ~ df$Sex)
#' x <- t.test(df$Tolerating, mu = 2)
#' x <- cor.test(df$Tolerating, df$Adjusting)
#'
#' results <- analyze(x)
#' summary(results)
#' print(results)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @import dplyr
#'
#' @export
analyze.htest <- function(x, effsize_rules = "cohen1988", ...) {


  # Processing
  # -------------
  values <- list()
  values$method <- x$method
  values$names <- x$data.name
  values$statistic <- x$statistic
  values$effect <- x$estimate
  values$p <- x$p.value
  values$df <- x$parameter
  values$CI <- x$conf.int
  values$signif <- ifelse(values$p < .05, "significant", "not significant")
  values$CI_level <- attr(values$CI, "conf.level") * 100
  values$CI_format <- paste0(values$CI_level, "% CI [", format_digit(values$CI[1]), ", ", format_digit(values$CI[2]), "]")



  # Text
  # -------------

  # CORRELATION
  if (grepl("correlation", values$method)) {
    text <- paste0(
      "The ",
      values$method,
      " between ",
      values$names,
      " is ",
      values$signif,
      ", ",
      interpret_r(values$effect, rules = effsize_rules),
      " (r(",
      format_digit(values$df),
      ") = ",
      format_digit(values$effect),
      ", ",
      values$CI_format,
      ", p ",
      format_p(values$p, stars = FALSE),
      ")."
    )

    # T-TEST
  } else if (grepl("t-test", values$method)) {
    if (names(x$null.value) == "mean") {
      means <- paste0(
        " (mean = ",
        format_digit(values$effect),
        ")"
      )
      vars <- paste0(values$names, means, " and mu = ", x$null.value)
    } else {
      means <- paste0(
        c(
          paste0(
            names(values$effect), " = ",
            format_digit(values$effect)
          ),
          paste0(
            "difference = ",
            format_digit(values$effect[1] - values$effect[2])
          )
        ),
        collapse = ", "
      )
      vars <- paste0(values$names, " (", means, ")")
    }

    values$effect <- values$effect[1] - values$effect[2]

    text <- paste0(
      "The ",
      values$method,
      " suggests that the difference ",
      ifelse(grepl(" by ", values$names), "of ", "between "),
      vars,
      " is ",
      values$signif,
      " (t(",
      format_digit(values$df),
      ") = ",
      format_digit(values$statistic),
      ", ",
      values$CI_format,
      ", p ",
      format_p(values$p, stars = FALSE),
      ")."
    )
    # OTHER
  } else {
    stop(paste0("The ", values$method, " is not implemented yet."))
  }


  # Summary
  # -------------
  summary <- data.frame(
    effect = values$effect,
    statistic = values$statistic,
    df = values$df,
    p = values$p,
    CI_lower = values$CI[1],
    CI_higher = values$CI[2]
  )
  rownames(summary) <- NULL

  # Plot
  # -------------
  plot <- "Not available yet"

  output <- list(text = text, plot = plot, summary = summary, values = values)

  class(output) <- c("psychobject", "list")
  return(output)
}










#' Analyze lavaan SEM or CFA) objects.
#'
#' Analyze lavaan (SEM or CFA) objects.
#'
#' @param x lavaan object.
#' @param CI Confidence interval level.
#' @param standardize Compute standardized coefs.
#' @param ... Arguments passed to or from other methods.
#'
#' @return output
#'
#' @examples
#' library(psycho)
#' library(lavaan)
#'
#' model <- " visual  =~ x1 + x2 + x3\ntextual =~ x4 + x5 + x6\nspeed   =~ x7 + x8 + x9 "
#' x <- lavaan::cfa(model, data = HolzingerSwineford1939)
#'
#' rez <- analyze(x)
#' print(rez)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @seealso
#' https://www.researchgate.net/post/Whats_the_standard_of_fit_indices_in_SEM
#'
#'
#' @importFrom lavaan parameterEstimates fitmeasures
#'
#' @export
analyze.lavaan <- function(x, CI = 95, standardize = FALSE, ...) {
  fit <- x


  # Processing
  # -------------
  values <- list()
  values$CI <- CI

  # Fit measures
  values$Fit_Measures <- interpret_lavaan(fit)




  # Summary
  # -------------
  summary <- .summary_lavaan(fit, CI = CI, standardize = standardize)

  # Plot
  # -------------
  plot <- "Use `get_graph` in association with ggraph."

  output <- list(text = values$Fit_Measures$text, plot = plot, summary = summary, values = values)

  class(output) <- c("psychobject", "list")
  return(output)
}














#' @keywords internal
.summary_lavaan <- function(fit, CI = 95, standardize = FALSE) {
  if (standardize == FALSE) {
    solution <- lavaan::parameterEstimates(fit, se = TRUE, standardized = standardize, level = CI / 100)
  } else {
    solution <- lavaan::standardizedsolution(fit, se = TRUE, level = CI / 100) %>%
      rename_("est" = "est.std")
  }

  solution <- solution %>%
    rename(
      "From" = "rhs",
      "To" = "lhs",
      "Operator" = "op",
      "Coef" = "est",
      "SE" = "se",
      "p" = "pvalue",
      "CI_lower" = "ci.lower",
      "CI_higher" = "ci.upper"
    ) %>%
    mutate(Type = dplyr::case_when(
      Operator == "=~" ~ "Loading",
      Operator == "~" ~ "Regression",
      Operator == "~~" ~ "Correlation",
      TRUE ~ NA_character_
    )) %>%
    mutate_("p" = "replace_na(p, 0)")

  if ("group" %in% names(solution)) {
    solution <- solution %>%
      rename("Group" = "group") %>%
      select(one_of(c("Group", "From", "Operator", "To", "Coef", "SE", "CI_lower", "CI_higher", "p", "Type")))
  } else {
    solution <- select(solution, one_of(c("From", "Operator", "To", "Coef", "SE", "CI_lower", "CI_higher", "p", "Type")))
  }

  return(solution)
}






#' Analyze lm objects.
#'
#' Analyze lm objects.
#'
#' @param x lm object.
#' @param CI Confidence interval bounds. Set to NULL turn off their computation.
#' @param effsize_rules Grid for effect size interpretation. See \link[=interpret_d]{interpret_d}.
#' @param ... Arguments passed to or from other methods.
#'
#' @return output
#'
#' @examples
#' library(psycho)
#' fit <- lm(Sepal.Length ~ Sepal.Width, data = iris)
#' fit <- lm(Sepal.Length ~ Sepal.Width * Species, data = iris)
#'
#' results <- analyze(fit)
#' summary(results)
#' print(results)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @import dplyr
#' @importFrom stats formula
#' @importFrom stringr str_squish
#' @export
analyze.lm <- function(x, CI = 95, effsize_rules = "cohen1988", ...) {


  # Processing
  # -------------
  fit <- x

  info <- get_info(fit)
  outcome <- info$outcome
  predictors <- info$predictors

  R2 <- get_R2(fit)
  R2adj <- R2$R2.adj
  R2 <- R2$R2

  # Summary
  # -------------
  summary <- data.frame(summary(fit)$coefficients)

  summary$Variable <- rownames(summary)
  summary$Coef <- summary$Estimate
  summary$SE <- summary$`Std..Error`
  summary$t <- summary$`t.value`
  summary$p <- summary$`Pr...t..`

  # standardized coefficients
  standardized <- tibble::rownames_to_column(standardize(fit, method = "refit", data = data), "Variable")
  summary <- merge(summary, standardized, by = "Variable", all.x = TRUE, sort = FALSE)
  summary$Effect_Size <- c(NA, interpret_d(tail(summary$Coef_std, -1), rules = effsize_rules))

  summary <- dplyr::select_(
    summary, "Variable", "Coef", "SE", "t", "Coef_std", "SE_std",
    "p", "Effect_Size"
  )

  if (!is.null(CI)) {
    CI_values <- confint(fit, level = CI / 100)
    CI_values <- tail(CI_values, n = length(rownames(summary)))
    summary$CI_lower <- CI_values[, 1]
    summary$CI_higher <- CI_values[, 2]
  }


  # Varnames
  varnames <- summary$Variable
  row.names(summary) <- varnames



  # Values
  # -------------
  # Initialize empty values
  values <- list(model = list(), effects = list())
  values$model$R2 <- R2
  values$model$R2adj <- R2adj


  # Loop over all variables
  for (varname in varnames) {
    if (summary[varname, "p"] < .1) {
      significance <- " "
    } else {
      significance <- " not "
    }

    if (!is.null(CI)) {
      CI_text <- paste0(
        ", ",
        CI, "% CI [",
        format_digit(summary[varname, "CI_lower"]),
        ", ",
        format_digit(summary[varname, "CI_higher"]),
        "]"
      )
    } else {
      CI_text <- ""
    }



    text <- paste0(
      "The effect of ",
      varname,
      " is",
      significance,
      "significant (beta = ",
      format_digit(summary[varname, "Coef"], 2), ", SE = ",
      format_digit(summary[varname, "SE"], 2),
      CI_text,
      ", t = ",
      format_digit(summary[varname, "t"], 2), ", p ",
      format_p(summary[varname, "p"], stars = FALSE),
      ") and can be considered as ",
      tolower(summary[varname, "Effect_Size"]),
      " (std. beta = ",
      format_digit(summary[varname, "Coef_std"], 2),
      ", std. SE = ",
      format_digit(summary[varname, "SE_std"], 2), ")."
    )

    if (varname == "(Intercept)") {
      text <- paste0(
        "The model's intercept is at ",
        format_digit(summary[varname, "Coef"], 2),
        " (SE = ",
        format_digit(summary[varname, "SE"], 2),
        CI_text,
        "). Within this model:"
      )
    }

    values$effects[[varname]] <- list(
      Coef = summary[varname, "Coef"],
      SE = summary[varname, "SE"],
      CI_lower = summary[varname, "CI_lower"],
      CI_higher = summary[varname, "CI_higher"],
      t = summary[varname, "t"],
      Coef_std = summary[varname, "Coef_std"],
      SE_std = summary[varname, "SE_std"],
      p = summary[varname, "p"],
      Effect_Size = summary[varname, "Effect_Size"],
      Text = text
    )
  }



  # Text
  # -------------
  text <- c(paste0(
    "The overall model predicting ",
    outcome,
    " (formula = ",
    stringr::str_squish(paste0(format(stats::formula(fit)), collapse = "")),
    ") explains ",
    format_digit(R2 * 100, 2),
    "% of the variance of the endogen (adj. R2 = ",
    format_digit(R2adj * 100, 2),
    "). ",
    values$effects[["(Intercept)"]]$Text
  ))

  for (varname in varnames) {
    if (varname != "(Intercept)") {
      text <- c(text, paste("   -", values$effects[[varname]]$Text))
    }
  }



  # Plot
  # -------------
  plot <- "Not available yet"

  output <- list(text = text, plot = plot, summary = summary, values = values)

  class(output) <- c("psychobject", "list")
  return(output)
}












#' Analyze lmerModLmerTest objects.
#'
#' Analyze lmerModLmerTest objects.
#'
#' @param x lmerModLmerTest object.
#' @param CI Bootsrapped confidence interval bounds (slow). Set to NULL turn off their computation.
#' @param effsize_rules Grid for effect size interpretation. See \link[=interpret_d]{interpret_d}.
#' @param ... Arguments passed to or from other methods.
#'
#' @return output
#'
#' @examples
#' library(psycho)
#' library(lmerTest)
#' fit <- lmerTest::lmer(Sepal.Length ~ Sepal.Width + (1 | Species), data = iris)
#'
#' results <- analyze(fit)
#' summary(results)
#' print(results)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @references Nakagawa, S., & Schielzeth, H. (2013). A general and simple method for obtaining R2 from generalized linear mixed-effects models. Methods in Ecology and Evolution, 4(2), 133-142.
#'
#' @importFrom MuMIn r.squaredGLMM
#' @importFrom MuMIn std.coef
#' @importFrom stringr str_squish
#' @import dplyr
#' @export
analyze.lmerModLmerTest <- function(x, CI = 95, effsize_rules = "cohen1988", ...) {


  # Processing
  # -------------
  fit <- x

  info <- get_info(fit)
  R2 <- get_R2(fit)



  # TODO: Bootstrapped p values
  # nsim determines p-value decimal places
  # boot.out = lme4::bootMer(fit, lme4::fixef, nsim=1000)
  # p = rbind(
  #   (1-apply(boot.out$t<0, 2, mean))*2,
  #   (1-apply(boot.out$t>0, 2, mean))*2)
  # p = apply(p, 2, min)



  # Summary
  # -------------
  summary <- data.frame(summary(fit)$coefficients)

  summary$Variable <- rownames(summary)
  summary$Coef <- summary$Estimate
  summary$SE <- summary$`Std..Error`
  summary$df <- as.numeric(summary$df)
  summary$t <- summary$`t.value`
  summary$p <- summary$`Pr...t..`

  # standardized coefficients
  standardized <- tibble::rownames_to_column(standardize(fit, method = "refit"), "Variable")
  summary <- merge(summary, standardized, by = "Variable", all.x = TRUE, sort = FALSE)
  summary$Effect_Size <- c(NA, interpret_d(tail(summary$Coef_std, -1), rules = effsize_rules))

  summary <- dplyr::select_(
    summary, "Variable", "Coef", "SE", "t", "df", "p", "Coef_std", "SE_std", "Effect_Size"
  )

  # CI computation
  if (!is.null(CI)) {
    CI_values <- tryCatch({
      suppressMessages(confint(fit, level = CI / 100))
    }, error = function(e) {
      warning("Couldn't compute CI. Skipping.")
      CI_values <- NA
      return(CI_values)
    })
    if (!all(is.na(CI_values))) {
      CI_values <- tail(CI_values, n = length(rownames(summary)))
      summary$CI_lower <- CI_values[, 1]
      summary$CI_higher <- CI_values[, 2]
    } else {
      CI <- NULL
    }
  }


  # Varnames
  varnames <- summary$Variable
  row.names(summary) <- varnames


  # Values
  # -------------
  # Initialize empty values
  values <- list(model = list(), effects = list())
  values$model$R2m <- R2$R2m
  values$model$R2c <- R2$R2c


  # Loop over all variables
  for (varname in varnames) {
    if (summary[varname, "p"] < .1) {
      significance <- " "
    } else {
      significance <- " not "
    }

    if (!is.null(CI)) {
      CI_text <- paste0(
        ", ",
        CI, "% CI [",
        format_digit(summary[varname, "CI_lower"]),
        ", ",
        format_digit(summary[varname, "CI_higher"]),
        "]"
      )
    } else {
      CI_text <- ""
    }




    if (varname == "(Intercept)") {
      text <- paste0(
        "The model's intercept is at ",
        format_digit(summary[varname, "Coef"], 2),
        " (SE = ",
        format_digit(summary[varname, "SE"], 2),
        CI_text,
        "). Within this model:"
      )
    } else {
      text <- paste0(
        "The effect of ",
        varname,
        " is",
        significance,
        "significant (beta = ",
        format_digit(summary[varname, "Coef"], 2),
        ", SE = ",
        format_digit(summary[varname, "SE"], 2),
        CI_text,
        ", t(",
        format_digit(summary[varname, "df"], 0),
        ") = ",
        format_digit(summary[varname, "t"], 2),
        ", p ",
        format_p(summary[varname, "p"], stars = FALSE),
        ") and can be considered as ",
        tolower(summary[varname, "Effect_Size"]),
        " (std. beta = ",
        format_digit(summary[varname, "Coef_std"], 2),
        ", std. SE = ",
        format_digit(summary[varname, "SE_std"], 2),
        ")."
      )
    }

    values$effects[[varname]] <- list(
      Coef = summary[varname, "Coef"],
      SE = summary[varname, "SE"],
      CI_lower = summary[varname, "CI_lower"],
      CI_higher = summary[varname, "CI_higher"],
      t = summary[varname, "t"],
      df = summary[varname, "df"],
      Coef_std = summary[varname, "Coef_std"],
      SE_std = summary[varname, "SE_std"],
      p = summary[varname, "p"],
      Effect_Size = summary[varname, "Effect_Size"],
      Text = text
    )
  }



  # Text
  # -------------
  text <- c(paste0(
    "The overall model predicting ",
    info$outcome,
    " (formula = ",
    format(info$formula),
    ") has an total explanatory power (conditional R2) of ",
    format_digit(R2$R2c * 100, 2),
    "%, in which the fixed effects explain ",
    format_digit(R2$R2m * 100, 2), "% of the variance (marginal R2). ",
    values$effects[["(Intercept)"]]$Text
  ))

  for (varname in varnames) {
    if (varname != "(Intercept)") {
      text <- c(text, paste("   -", values$effects[[varname]]$Text))
    }
  }



  # Plot
  # -------------
  plot <- "Not available yet"

  output <- list(text = text, plot = plot, summary = summary, values = values)

  class(output) <- c("psychobject", "list")
  return(output)
}









#' Analyze fa objects.
#'
#' Analyze fa objects.
#'
#' @param x An psych object.
#' @param labels Supply a additional column with e.g. item labels.
#' @param treshold 'max' or numeric. The treshold over which to associate an item with its component.
#' @param ... Arguments passed to or from other methods.
#'
#' @return output
#'
#' @examples
#' library(psycho)
#' library(psych)
#'
#' x <- psych::pca(psych::Thurstone.33, 2)
#'
#' results <- analyze(x)
#' print(results)
#' summary(results)
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
analyze.principal <- function(x, labels = NULL, treshold = "max", ...) {
  loadings <- format_loadings(x, labels)

  values <- list()
  values$variance <- x$Vaccounted
  values$loadings <- loadings$loadings
  values$loadings_max <- loadings$max
  values$cfa_model <- get_cfa_model(loadings$loadings, treshold = treshold)

  text <- .fa_variance_text(values$variance)
  text <- paste0(text, "\n\n", format(values$cfa_model))
  summary <- values$loadings
  plot <- plot_loadings(values$loadings)

  output <- list(text = text, plot = plot, summary = summary, values = values)

  class(output) <- c("psychobject", "list")
  return(output)
}






#' Analyze objects.
#'
#' Analyze objects. See the documentation for your object's class:
#' \itemize{
#'  \item{\link[=analyze.stanreg]{analyze.stanreg}}
#'  \item{\link[=analyze.lmerModLmerTest]{analyze.merModLmerTest}}
#'  \item{\link[=analyze.glmerMod]{analyze.glmerMod}}
#'  \item{\link[=analyze.lm]{analyze.lm}}
#'  \item{\link[=analyze.glm]{analyze.glm}}
#'  }
#'  \itemize{
#'  \item{\link[=analyze.htest]{analyze.htest}}
#'  \item{\link[=analyze.aov]{analyze.aov}}
#'  }
#' \itemize{
#'  \item{\link[=analyze.fa]{analyze.fa}}
#'  \item{\link[=analyze.principal]{analyze.principal}}
#'  \item{\link[=analyze.lavaan]{analyze.lavaan}}
#'  \item{\link[=analyze.blavaan]{analyze.blavaan}}
#'  }
#'
#' @param x object to analyze.
#' @param ... Arguments passed to or from other methods.
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
analyze <- function(x, ...) {
  UseMethod("analyze")
}




#' Analyze stanreg objects.
#'
#' Analyze stanreg objects.
#'
#' @param x A stanreg model.
#' @param CI Credible interval bounds.
#' @param index Index of effect existence to report. Can be 'overlap' or 'ROPE'.
#' @param ROPE_bounds Bounds of the ROPE. If NULL and effsize is TRUE, than the ROPE.
#' will have default values c(-0.1, 0.1) and computed on the standardized posteriors.
#' @param effsize Compute Effect Sizes according to Cohen (1988). For linear models only.
#' @param effsize_rules Grid for effect size interpretation. See \link[=interpret_d]{interpret_d}.
#' @param ... Arguments passed to or from other methods.
#'
#' @return Contains the following indices:
#' \itemize{
#'  \item{the Median of the posterior distribution of the parameter (can be used as a point estimate, similar to the beta of frequentist models).}
#'  \item{the Median Absolute Deviation (MAD), a robust measure of dispertion (could be seen as a robust version of SD).}
#'  \item{the Credible Interval (CI) (by default, the 90\% CI; see Kruschke, 2018), representing a range of possible parameter.}
#'  \item{the Maximum Probability of Effect (MPE), the probability that the effect is positive or negative (depending on the medianâ€™s direction).}
#'  \item{the Overlap (O), the percentage of overlap between the posterior distribution and a normal distribution of mean 0 and same SD than the posterior. Can be interpreted as the probability that a value from the posterior distribution comes from a null distribution.}
#'  \item{the ROPE, the proportion of the 95\% CI of the posterior distribution that lies within the region of practical equivalence.}
#'  }
#'
#' @examples
#' \dontrun{
#' library(psycho)
#' library(rstanarm)
#'
#' data <- attitude
#' fit <- rstanarm::stan_glm(rating ~ advance + privileges, data = data)
#'
#' results <- analyze(fit, effsize = TRUE)
#' summary(results)
#' print(results)
#' plot(results)
#'
#'
#' fit <- rstanarm::stan_lmer(Sepal.Length ~ Sepal.Width + (1 | Species), data = iris)
#' results <- analyze(fit)
#' summary(results)
#'
#' fit <- rstanarm::stan_glm(Sex ~ Adjusting,
#'   data = psycho::affective, family = "binomial"
#' )
#' results <- analyze(fit)
#' summary(results)
#'
#' fit <- rstanarm::stan_glmer(Sex ~ Adjusting + (1 | Salary),
#'   data = psycho::affective, family = "binomial"
#' )
#' results <- analyze(fit)
#' summary(results)
#' }
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @seealso
#' \link[=get_R2.stanreg]{"get_R2.stanreg"}
#' \link[=bayes_R2.stanreg]{"bayes_R2.stanreg"}
#'
#' @import loo
#' @import tidyr
#' @import dplyr
#' @import ggplot2
#' @importFrom stats quantile as.formula
#' @importFrom utils head tail capture.output
#' @importFrom broom tidy
#' @importFrom stringr str_squish str_replace
#' @export
analyze.stanreg <- function(x, CI = 90, index = "overlap", ROPE_bounds = NULL, effsize = FALSE, effsize_rules = "cohen1988", ...) {
  fit <- x

  # Info --------------------------------------------------------------------

  # Algorithm
  if (fit$algorithm == "optimizing") {
    stop("Can't analyze models fitted with 'optimizing' algorithm.")
  }
  computations <- capture.output(fit$stanfit)
  computations <- paste0(computations[2], computations[3], collapse = "")
  computations <- stringr::str_remove_all(computations, ", total post-warmup draws.*")
  computations <- stringr::str_remove_all(computations, " draws per chain")
  computations <- stringr::str_replace_all(computations, "=", " = ")

  # Extract posterior distributions
  posteriors <- as.data.frame(fit)


  # Varnames
  info <- get_info(fit)
  outcome <- info$outcome
  predictors <- info$predictors

  varnames <- names(fit$coefficients)
  varnames <- varnames[grepl("b\\[", varnames) == FALSE]

  # Initialize empty values
  values <- list(model = list(), effects = list())

  values$model$formula <- fit$formula
  values$model$outcome <- outcome
  values$model$predictors <- predictors

  # Priors
  info_priors <- rstanarm::prior_summary(fit)
  values$priors <- info_priors

  # R2 ----------------------------------------------------------------------

  R2 <- get_R2(fit, silent = TRUE)
  if (is.list(R2)) {
    posteriors$R2 <- R2$R2_posterior
    R2.adj <- R2$R2.adj
    if (!"R2" %in% varnames) {
      varnames <- c("R2", varnames)
    }
    R2 <- TRUE
  } else {
    R2 <- FALSE
  }

  # Random effect info --------------------------------------------
  if (is.mixed(fit)) {
    random_info <- broom::tidy(fit, parameters = "varying") %>%
      dplyr::rename_(
        "Median" = "estimate",
        "MAD" = "std.error"
      )
    values$random <- random_info
  }

  # Standardized posteriors --------------------------------------------
  if (effsize == TRUE) {
    posteriors_std <- standardize(fit, method = "refit")
    # Avoir some problems
    if (length(setdiff(names(posteriors_std), varnames[varnames != "R2"])) != 0) {
      names(posteriors_std) <- varnames[varnames != "R2"]
    }
  } else {
    posteriors_std <- as.data.frame(fit)
  }

  # Get indices of each variable --------------------------------------------

  # Loop over all variables
  for (varname in varnames) {
    if (varname == "R2") {
      values$effects[[varname]] <- .process_R2(varname,
        posteriors,
        info_priors,
        R2.adj = R2.adj,
        CI = CI,
        effsize = effsize
      )
    } else if (varname == "(Intercept)") {
      values$effects[[varname]] <- .process_intercept(varname,
        posteriors,
        info_priors,
        predictors,
        CI = CI,
        effsize = effsize
      )
    } else {
      values$effects[[varname]] <- .process_effect(varname,
        posteriors,
        posteriors_std = posteriors_std,
        info_priors,
        predictors,
        CI = CI,
        effsize = effsize,
        effsize_rules = effsize_rules,
        fit = fit,
        index = index,
        ROPE_bounds = ROPE_bounds
      )
    }
  }


  # Summary --------------------------------------------------------------------
  summary <- data.frame()
  for (varname in varnames) {
    summary <- rbind(
      summary,
      data.frame(
        Variable = varname,
        Median = values$effects[[varname]]$median,
        MAD = values$effects[[varname]]$mad,
        CI_lower = values$effects[[varname]]$CI_values[1],
        CI_higher = values$effects[[varname]]$CI_values[2],
        Median_std = values$effects[[varname]]$std_median,
        MAD_std = values$effects[[varname]]$std_mad,
        MPE = values$effects[[varname]]$MPE,
        ROPE = values$effects[[varname]]$ROPE,
        Overlap = values$effects[[varname]]$overlap
      )
    )
  }

  if (effsize == FALSE) {
    summary <- select_(summary, "-Median_std", "-MAD_std")
  }

  if (index == "ROPE") {
    summary <- select_(summary, "-Overlap")
  } else {
    summary <- select_(summary, "-ROPE")
  }

  # Text --------------------------------------------------------------------
  # -------------------------------------------------------------------------
  # Model
  info <- paste0(
    "We fitted a ",
    ifelse(fit$algorithm == "sampling", "Markov Chain Monte Carlo", fit$algorithm),
    " ",
    fit$family$family,
    " (link = ",
    fit$family$link,
    ") model (",
    computations,
    ") to predict ",
    outcome,
    " (formula = ", stringr::str_squish(paste0(format(fit$formula), collapse = "")),
    "). The model's priors were set as follows: "
  )

  # Priors
  text_priors <- rstanarm::prior_summary(fit)
  if ("adjusted_scale" %in% names(text_priors$prior) & !is.null(text_priors$prior$adjusted_scale)) {
    scale <- paste0(
      "), scale = (",
      paste(sapply(text_priors$prior$adjusted_scale, format_digit), collapse = ", ")
    )
  } else {
    scale <- paste0(
      "), scale = (",
      paste(sapply(text_priors$prior$scale, format_digit), collapse = ", ")
    )
  }

  info_priors_text <- paste0(
    "  ~ ",
    text_priors$prior$dist,
    " (location = (",
    paste(text_priors$prior$location, collapse = ", "),
    scale,
    "))"
  )

  # Coefs
  coefs_text <- c()
  for (varname in varnames) {
    effect_text <- values$effects[[varname]]$text
    if (effsize == TRUE) {
      if (!varname %in% c("(Intercept)", "R2")) {
        effsize_text <- stringr::str_replace(
          values$effects[[varname]]$EffSize_text,
          "The effect's size",
          "It"
        )[1]
        effect_text <- paste(effect_text, effsize_text)
      }
    }
    coefs_text <- c(coefs_text, effect_text)
  }

  # Text
  if ("R2" %in% varnames) {
    text <- c(
      info,
      "",
      info_priors_text,
      "",
      "",
      paste0(
        coefs_text[1],
        coefs_text[2]
      ),
      "",
      tail(coefs_text, -2)
    )
  } else {
    text <- c(
      info,
      "",
      info_priors_text,
      "",
      "",
      coefs_text[1],
      "",
      tail(coefs_text, -1)
    )
  }




  # Plot --------------------------------------------------------------------
  # -------------------------------------------------------------------------

  plot <- posteriors[varnames] %>%
    # select(-`(Intercept)`) %>%
    gather() %>%
    rename_(Variable = "key", Coefficient = "value") %>%
    ggplot(aes_string(x = "Variable", y = "Coefficient", fill = "Variable")) +
    geom_violin() +
    geom_boxplot(fill = "grey", alpha = 0.3, outlier.shape = NA) +
    stat_summary(
      fun.y = "mean", geom = "errorbar",
      aes_string(ymax = "..y..", ymin = "..y.."),
      width = .75, linetype = "dashed", colour = "red"
    ) +
    geom_hline(aes(yintercept = 0)) +
    theme_classic() +
    coord_flip() +
    scale_fill_brewer(palette = "Set1") +
    scale_colour_brewer(palette = "Set1")



  output <- list(text = text, plot = plot, summary = summary, values = values)

  class(output) <- c("psychobject", "list")
  return(output)
}




















#' @keywords internal
.get_info_priors <- function(varname, info_priors, predictors = NULL) {
  # Prior
  # TBD: this doesn't work with categorical predictors :(
  values <- list()

  if (varname == "(Intercept)") {
    values["prior_distribution"] <- info_priors$prior_intercept$dist
    values["prior_location"] <- info_priors$prior_intercept$location
    values["prior_scale"] <- info_priors$prior_intercept$scale
    values["prior_adjusted_scale"] <- info_priors$prior_intercept$adjusted_scale
  } else {
    if (varname %in% predictors) {
      predictor_index <- which(predictors == varname)
      if (length(info_priors$prior$dist) == 1) {
        info_priors$prior$dist <- rep(
          info_priors$prior$dist,
          length(info_priors$prior$location)
        )
      }
      values["prior_distribution"] <- info_priors$prior$dist[predictor_index]
      values["prior_location"] <- info_priors$prior$location[predictor_index]
      values["prior_scale"] <- info_priors$prior$scale[predictor_index]
      values["prior_adjusted_scale"] <- info_priors$prior$adjusted_scale[predictor_index]
    }
  }
  return(values)
}








#' @keywords internal
.process_R2 <- function(varname, posteriors, info_priors, R2.adj = NULL, CI = 90, effsize = FALSE) {
  values <- .get_info_priors(varname, info_priors)
  posterior <- posteriors[, varname]

  # Find basic posterior indices
  values$posterior <- posterior
  values$median <- median(posterior)
  values$mad <- mad(posterior)
  values$mean <- mean(posterior)
  values$sd <- sd(posterior)
  values$CI_values <- HDI(posterior, prob = CI / 100)
  values$CI_values <- c(values$CI_values$values$HDImin, values$CI_values$values$HDImax)
  values$MPE <- NA
  values$MPE_values <- NA
  values$overlap <- NA
  values$ROPE <- NA
  values$adjusted_r_squared <- R2.adj

  # Text
  values$text <- paste0(
    "The model has an explanatory power (R2) of about ",
    format_digit(values$median * 100),
    "% (MAD = ",
    format_digit(values$mad),
    ", ",
    CI,
    "% CI [",
    format_digit(values$CI_values[1]),
    ", ",
    format_digit(values$CI_values[2]),
    "]"
  )

  if (is.null(R2.adj) | is.na(R2.adj)) {
    values$text <- paste0(
      values$text,
      ")."
    )
  } else {
    values$text <- paste0(
      values$text,
      ", adj. R2 = ",
      format_digit(R2.adj),
      ")."
    )
  }


  # Effize
  if (effsize == TRUE) {
    values$std_posterior <- NA
    values$std_median <- NA
    values$std_mad <- NA
    values$std_mean <- NA
    values$std_sd <- NA
    values$std_CI_values <- NA
    values$std_CI_values <- NA

    values$EffSize <- NA
    values$EffSize_text <- NA
    values$EffSize_VeryLarge <- NA
    values$EffSize_Large <- NA
    values$EffSize_Moderate <- NA
    values$EffSize_Small <- NA
    values$EffSize_VerySmall <- NA
    values$EffSize_Opposite <- NA
  } else {
    values$std_median <- NA
    values$std_mad <- NA
  }

  return(values)
}




#' @keywords internal
.process_intercept <- function(varname, posteriors, info_priors, predictors, CI = 90, effsize = FALSE) {
  values <- .get_info_priors(varname, info_priors, predictors)
  posterior <- posteriors[, varname]

  # Find basic posterior indices
  values$posterior <- posterior
  values$median <- median(posterior)
  values$mad <- mad(posterior)
  values$mean <- mean(posterior)
  values$sd <- sd(posterior)
  values$CI_values <- HDI(posterior, prob = CI / 100)
  values$CI_values <- c(values$CI_values$values$HDImin, values$CI_values$values$HDImax)
  values$MPE <- NA
  values$MPE_values <- NA
  values$overlap <- NA
  values$ROPE <- NA



  # Text
  values$text <- paste0(
    " The intercept is at ",
    format_digit(values$median),
    " (MAD = ",
    format_digit(values$mad),
    ", ",
    CI,
    "% CI [",
    format_digit(values$CI_values[1]),
    ", ",
    format_digit(values$CI_values[2]),
    "]). Within this model:"
  )

  # Effize
  if (effsize == TRUE) {
    values$std_posterior <- NA
    values$std_median <- NA
    values$std_mad <- NA
    values$std_mean <- NA
    values$std_sd <- NA
    values$std_CI_values <- NA
    values$std_CI_values <- NA

    values$EffSize <- NA
    values$EffSize_text <- NA
    values$EffSize_VeryLarge <- NA
    values$EffSize_Large <- NA
    values$EffSize_Moderate <- NA
    values$EffSize_Small <- NA
    values$EffSize_VerySmall <- NA
    values$EffSize_Opposite <- NA
  } else {
    values$std_median <- NA
    values$std_mad <- NA
  }

  return(values)
}




#' @keywords internal
.process_effect <- function(varname,
                            posteriors,
                            posteriors_std,
                            info_priors,
                            predictors,
                            CI = 90,
                            effsize = FALSE,
                            effsize_rules = FALSE,
                            fit,
                            index = "overlap",
                            ROPE_bounds = NULL) {
  values <- .get_info_priors(varname, info_priors, predictors)
  posterior <- posteriors[, varname]


  # Find basic posterior indices
  values$posterior <- posterior
  values$median <- median(posterior)
  values$mad <- mad(posterior)
  values$mean <- mean(posterior)
  values$sd <- sd(posterior)
  values$CI_values <- HDI(posterior, prob = CI / 100)
  values$CI_values <- c(values$CI_values$values$HDImin, values$CI_values$values$HDImax)
  values$MPE <- mpe(posterior)$MPE
  values$MPE_values <- mpe(posterior)$values

  # Index
  values$overlap <- 100 * overlap(
    posterior,
    rnorm_perfect(
      length(posterior),
      0,
      sd(posterior)
    )
  )

  if (!is.null(ROPE_bounds)) {
    rope <- rope(posterior, bounds = ROPE_bounds)
    values$ROPE_decision <- rope$rope_decision
    values$ROPE <- rope$rope_probability
  } else {
    values$ROPE <- NA
    values$ROPE_decision <- NA
  }

  if (index == "overlap") {
    index <- paste0(
      "Overlap = ",
      format_digit(values$overlap),
      "%)."
    )
  } else if (index == "ROPE") {
    if (!is.null(ROPE_bounds)) {
      index <- paste0(
        "ROPE = ",
        format_digit(values$ROPE),
        ")."
      )
    } else {
      if (effsize == TRUE) {
        rope <- rope(posteriors_std[, varname], bounds = c(-0.1, 0.1))
        values$ROPE_decision <- rope$rope_decision
        values$ROPE <- rope$rope_probability
        index <- paste0(
          "ROPE = ",
          format_digit(values$ROPE),
          ")."
        )
      } else {
        warning("you need to specify ROPE_bounds (e.g. 'c(-0.1, 0.1)'). Computing overlap instead.")
        index <- paste0(
          "Overlap = ",
          format_digit(values$overlap),
          "%)."
        )
      }
    }
  } else {
    warning("Parameter 'index' should be 'overlap' or 'ROPE'. Computing overlap.")
    index <- paste0(
      "Overlap = ",
      format_digit(values$overlap),
      "%)."
    )
  }





  # Text
  if (grepl(":", varname)) {
    splitted <- strsplit(varname, ":")[[1]]
    if (length(splitted) == 2) {
      name <- paste0(
        "interaction between ",
        splitted[1], " and ", splitted[2]
      )
    } else {
      name <- varname
    }
  } else {
    name <- paste0("effect of ", varname)
  }

  direction <- ifelse(values$median > 0, "positive", "negative")

  values$text <- paste0(
    "  - The ",
    name,
    " has a probability of ",
    format_digit(values$MPE),
    "% of being ",
    direction,
    " (Median = ",
    format_digit(values$median),
    ", MAD = ",
    format_digit(values$mad),
    ", ",
    CI,
    "% CI [",
    format_digit(values$CI_values[1]), ", ",
    format_digit(values$CI_values[2]), "], ",
    index
  )



  # Effize
  if (effsize == TRUE) {
    posterior_std <- posteriors_std[, varname]
    values$std_posterior <- posterior_std
    values$std_median <- median(posterior_std)
    values$std_mad <- mad(posterior_std)
    values$std_mean <- mean(posterior_std)
    values$std_sd <- sd(posterior_std)
    values$std_CI_values <- HDI(posterior_std, prob = CI / 100)
    values$std_CI_values <- c(values$std_CI_values$values$HDImin, values$std_CI_values$values$HDImax)

    if (fit$family$family == "binomial" & fit$family$link == "logit") {
      EffSize <- interpret_odds_posterior(posterior_std, log = TRUE, rules = effsize_rules)
    } else {
      EffSize <- interpret_d_posterior(posterior_std, rules = effsize_rules)
    }

    values$EffSize <- EffSize$summary
    values$EffSize$Variable <- varname
    values$EffSize_text <- EffSize$text
  } else {
    values$std_median <- NA
    values$std_mad <- NA
  }

  return(values)
}






#' Coerce to a Data Frame.
#'
#' Functions to check if an object is a data frame, or coerce it if possible.
#'
#' @param x any R object.
#' @param ... additional arguments to be passed to or from methods.
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @method as.data.frame density
#' @export
as.data.frame.density <- function(x, ...) {
  df <- data.frame(x = x$x, y = x$y)

  return(df)
}



#' Compare a patient's score to a control group
#'
#' Compare a patient's score to a control group.
#'
#' @param patient Single value (patient's score).
#' @param controls Vector of values (control's scores).
#' @param mean Mean of the control sample.
#' @param sd SD of the control sample.
#' @param n Size of the control sample.
#' @param CI Credible interval bounds.
#' @param treshold Significance treshold.
#' @param iter Number of iterations.
#' @param color_controls Color of the controls distribution.
#' @param color_CI Color of CI distribution.
#' @param color_score Color of the line representing the patient's score.
#' @param color_size Size of the line representing the patient's score.
#' @param alpha_controls Alpha of the CI distribution.
#' @param alpha_CI lpha of the controls distribution.
#' @param verbose Print possible warnings.
#'
#' @return output
#'
#' @examples
#' result <- assess(patient = 124, mean = 100, sd = 15, n = 100)
#' print(result)
#' plot(result)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @details Until relatively recently the standard way of testing for a difference between a case and controls was to convert the case’s score to a z score using the control sample mean and standard deviation (SD). If z was less than -1.645 (i.e., below 95% of the controls) then it was concluded that the case was significantly lower than controls. However, this method has serious disadvantages (Crawford and Garthwaite, 2012).
#'
#' @importFrom stats ecdf
#' @import ggplot2
#' @import dplyr
#' @export
assess <- function(patient,
                   mean = 0,
                   sd = 1,
                   n = NULL,
                   controls = NULL,
                   CI = 95,
                   treshold = 0.05,
                   iter = 10000,
                   color_controls = "#2196F3",
                   color_CI = "#E91E63",
                   color_score = "black",
                   color_size = 2,
                   alpha_controls = 1,
                   alpha_CI = 0.8,
                   verbose = TRUE) {
  if (is.null(controls)) {
    if (is.null(n)) {
      if (verbose == TRUE) {
        warning("Sample size (n) not provided, thus set to 1000.")
      }
      n <- 1000
    }
  }




  # If score is list
  if (length(patient) > 1) {
    if (verbose == TRUE) {
      warning("Multiple scores were provided. Returning a list of results.")
    }
    results <- list()
    for (i in seq_len(length(patient))) {
      results[[i]] <- crawford.test(
        patient[i],
        controls,
        mean,
        sd,
        n,
        CI,
        treshold,
        iter,
        color_controls,
        color_CI,
        color_score,
        color_size,
        alpha_controls,
        alpha_CI
      )
      return(results)
    }
  } else {
    result <- crawford.test(
      patient,
      controls,
      mean,
      sd,
      n,
      CI,
      treshold,
      iter,
      color_controls,
      color_CI,
      color_score,
      color_size,
      alpha_controls,
      alpha_CI
    )
    return(result)
  }
}








#' Performs a Bayesian correlation.
#'
#' Performs a Bayesian correlation.
#'
#' @param x First continuous variable.
#' @param y Second continuous variable.
#' @param CI Credible interval bounds.
#' @param iterations The number of iterations to sample.
#' @param effsize_rules_r Grid for effect size interpretation. See \link[=interpret_r]{interpret_r}.
#' @param effsize_rules_bf Grid for effect size interpretation. See \link[=interpret_bf]{interpret_bf}.
#'
#' @return A psychobject.
#'
#' @examples
#' \dontrun{
#' library(psycho)
#' x <- psycho::affective$Concealing
#' y <- psycho::affective$Tolerating
#'
#' bayes_cor.test(x, y)
#' summary(bayes_cor.test(x, y))
#' }
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @importFrom BayesFactor correlationBF posterior
#' @importFrom stats complete.cases cor.test
#' @import dplyr
#' @export
bayes_cor.test <- function(x, y, CI = 90, iterations = 10000, effsize_rules_r = "cohen1988", effsize_rules_bf = "jeffreys1961") {


  # Varnames ----------------------------------------------------------------


  if (is.null(names(x))) {
    var1 <- deparse(substitute(x))
  } else {
    var1 <- names(x)
    x <- pull(x)
  }

  if (is.null(names(y))) {
    var2 <- deparse(substitute(y))
  } else {
    var2 <- names(y)
    y <- pull(y)
  }

  # Remove missing
  var_x <- x[complete.cases(x, y)]
  var_y <- y[complete.cases(x, y)]

  # Correlation -------------------------------------------------------------

  # Stop if same variable
  if (cor.test(var_x, var_y)$estimate > 0.999) {
    return(1)
  }


  cor <- BayesFactor::correlationBF(var_x, var_y)
  posterior <- as.vector(suppressMessages(BayesFactor::posterior(cor, iterations = iterations, progress = FALSE)))

  values <- list()
  values$posterior <- posterior
  values$bf <- as.vector(cor)[1]
  values$median <- median(posterior)
  values$mad <- mad(posterior)
  values$mean <- mean(posterior)
  values$sd <- sd(posterior)
  values$CI <- HDI(posterior, prob = CI / 100)$text
  values$CI_values <- HDI(posterior, prob = CI / 100)
  values$CI_values <- c(values$CI_values$values$HDImin, values$CI_values$values$HDImax)
  values$MPE <- mpe(posterior)$MPE
  values$MPE_values <- mpe(posterior)$values

  norm <- rnorm_perfect(length(posterior), 0, sd(posterior))
  values$overlap <- overlap(posterior, norm) * 100

  rope_indices <- rope(posterior, bounds = c(-0.1, 0.1), CI = 95, overlap = TRUE)
  values$rope_decision <- rope_indices$rope_decision
  values$rope_probability <- rope_indices$rope_probability
  values$rope_overlap <- rope_indices$rope_overlap


  summary <- data.frame(
    Median = values$median,
    MAD = values$mad,
    CI_lower = values$CI_values[1],
    CI_higher = values$CI_values[2],
    MPE = values$MPE,
    BF = values$bf,
    Overlap = values$overlap,
    Rope = values$rope_decision
  )
  rownames(summary) <- paste0(var1, " / ", var2)

  values$effect_size <- interpret_r_posterior(posterior, rules = effsize_rules_r)
  interpretation_r <- interpret_r(values$median, strength = FALSE, rules = effsize_rules_r)
  interpretation_bf <- interpret_bf(values$bf, direction = FALSE, rules = effsize_rules_bf)
  if (values$bf < 1) {
    interpretation_bf <- paste(interpretation_bf, "in favour of an absence of a ")
  } else {
    interpretation_bf <- paste(interpretation_bf, "in favour of the existence of a ")
  }

  text <- paste0(
    "Results of the Bayesian correlation indicate ",
    interpretation_bf,
    interpretation_r,
    " association between ",
    var1,
    " and ",
    var2,
    " (r = ",
    format_digit(values$median),
    ", MAD = ",
    format_digit(values$mad),
    ", ",
    CI,
    "% CI [",
    format_digit(values$CI_values[1]),
    ", ",
    format_digit(values$CI_values[2]),
    "]). ",
    values$effect_size$text
  )

  plot <- "Not available."

  output <- list(text = text, plot = plot, summary = summary, values = values)
  class(output) <- c("psychobject", "list")

  return(output)
}





#' Bayesian Correlation Matrix.
#'
#' Bayesian Correlation Matrix.
#'
#' @param df The dataframe.
#' @param df2 Optional dataframe to correlate with the first one.
#' @param reorder Reorder matrix by correlation strength. Only for square matrices.
#'
#' @return A list of dataframes
#'
#' @examples
#' \dontrun{
#' library(psycho)
#'
#' df <- psycho::affective
#' cor <- bayes_cor(df)
#' summary(cor)
#' print(cor)
#' plot(cor)
#'
#' df <- select(psycho::affective, Adjusting, Tolerating)
#' df2 <- select(psycho::affective, -Adjusting, -Tolerating)
#' cor <- bayes_cor(df, df2)
#' summary(cor)
#' print(cor)
#' plot(cor)
#' }
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#' @export
bayes_cor <- function(df, df2 = NULL, reorder = TRUE) {
  df <- purrr::keep(df, is.numeric)

  if (!is.null(df2)) {
    df2 <- purrr::keep(df2, is.numeric)
    combinations <- expand.grid(names(df), names(df2))
    df <- cbind(df, df2)
  } else {
    combinations <- expand.grid(names(df), names(df))
  }

  size_row <- length(unique(combinations$Var1))
  size_col <- length(unique(combinations$Var2))
  dimnames <- list(
    unique(combinations$Var1),
    unique(combinations$Var2)
  )

  r <- matrix(0, nrow = size_row, ncol = size_col, dimnames = dimnames)
  mpe <- matrix(0, nrow = size_row, ncol = size_col, dimnames = dimnames)
  bf <- matrix(0, nrow = size_row, ncol = size_col, dimnames = dimnames)
  ci <- matrix(0, nrow = size_row, ncol = size_col, dimnames = dimnames)
  text <- matrix("", nrow = size_row, ncol = size_col, dimnames = dimnames)

  counter <- 0
  for (j in seq_len(size_col)) {
    for (i in seq_len(size_row)) {
      counter <- counter + 1

      x <- df[[as.character(combinations$Var1[counter])]]
      y <- df[[as.character(combinations$Var2[counter])]]
      result <- bayes_cor.test(x, y)

      if (!is.psychobject(result)) {
        text[i, j] <- ""
        r[i, j] <- 1
        mpe[i, j] <- 100
        bf[i, j] <- Inf
        ci[i, j] <- "100% CI [1, 1]"
      } else {
        text[i, j] <- paste0(
          "   - ",
          names(df)[j],
          " / ",
          names(df)[i],
          ":   ",
          result$text
        )
        text[i, j] <- stringr::str_remove(text[i, j], "between x and y ")
        r[i, j] <- result$values$median
        mpe[i, j] <- result$values$MPE
        bf[i, j] <- result$values$bf
        ci[i, j] <- result$values$CI
      }
    }
  }


  # Reorder
  if (is.null(df2) & reorder == TRUE) {
    r <- reorder_matrix(r, r)
    mpe <- reorder_matrix(mpe, r)
    bf <- reorder_matrix(bf, r)
    ci <- reorder_matrix(ci, r)
    text <- reorder_matrix(text, r)
  }


  stars <- ifelse(bf > 30, "***",
    ifelse(bf > 10, "**",
      ifelse(bf > 3, "*", "")
    )
  )



  summary <- round(r, 2)
  summary <- matrix(paste(summary, stars, sep = ""), ncol = ncol(r), dimnames = dimnames(r))

  if (is.null(df2)) {
    summary[upper.tri(summary, diag = TRUE)] <- "" # remove upper triangle
    summary <- summary[-1, -ncol(summary)] # Remove first row and last column

    text[upper.tri(text, diag = TRUE)] <- "" # remove upper triangle
    text <- text[-1, -ncol(text)] # Remove first row and last column
  }

  summary <- as.data.frame(summary)
  text <- as.vector(text)
  text <- text[!text == ""]


  # Values
  values <- list(
    r = r,
    mpe = mpe,
    bf = bf,
    ci = ci,
    stars = stars
  )

  # Plot
  plot <- round(r, 2) %>%
    as.data.frame() %>%
    tibble::rownames_to_column("Var1") %>%
    gather_("Var2", "Correlation", as.character(unique(combinations$Var2))) %>%
    ggplot(aes_string(x = "Var2", y = "Var1", fill = "Correlation", label = "Correlation")) +
    geom_tile(color = "white") +
    scale_fill_gradient2(
      low = "#2196F3", high = "#E91E63", mid = "white",
      midpoint = 0, limit = c(-1, 1)
    ) +
    theme_minimal() +
    theme(
      axis.title = element_blank(),
      axis.text.x = element_text(
        angle = 45,
        vjust = 1,
        hjust = 1
      ),
      legend.position = "none"
    ) +
    coord_fixed() +
    geom_text(color = "black")


  # Output
  # -------------
  output <- list(text = text, plot = plot, summary = summary, values = values)

  class(output) <- c("psychobject", "list")
  return(output)
}



#' Reorder square matrix.
#'
#' Reorder square matrix.
#'
#' @param mat A square matrix.
#' @param dmat A square matrix with values to use as distance.
#'
#' @examples
#' library(psycho)
#'
#' r <- correlation(iris)
#' r <- r$values$r
#' r <- reorder_matrix(r)
#' @importFrom stats as.dist hclust
#' @export
reorder_matrix <- function(mat, dmat = NULL) {
  if (is.null(dmat)) {
    dmat <- mat
  }

  if (ncol(mat) != nrow(mat) | ncol(dmat) != nrow(dmat)) {
    warning("Matrix must be squared.")
    return(mat)
  }

  dmat <- as.dist((1 - dmat) / 2, diag = TRUE, upper = TRUE)
  hc <- hclust(dmat)
  mat <- mat[hc$order, hc$order]
  return(mat)
}








#' Citations of loaded packages.
#'
#' Get the citations of loaded packages.
#'
#' @param session A `devtools::sessionInfo()` object.
#'
#' @examples
#' \dontrun{
#' library(psycho)
#' cite_packages(sessionInfo())
#' }
#'
#' @author \href{https://github.com/DominiqueMakowski}{Dominique Makowski}
#'
#' @export
cite_packages <- function(session) {
  pkgs <- session$otherPkgs
  citations <- c()
  for (pkg_name in names(pkgs)) {
    pkg <- pkgs[[pkg_name]]

    citation <- format(citation(pkg_name))[[2]] %>%
      stringr::str_split("\n") %>%
      flatten() %>%
      paste(collapse = "SPLIT") %>%
      stringr::str_split("SPLITSPLIT")

    i <- 1
    while (stringr::str_detect(citation[[1]][i], "To cite ")) {
      i <- i + 1
    }


    citation <- citation[[1]][i] %>%
      stringr::str_remove_all("SPLIT") %>%
      stringr::str_trim() %>%
      stringr::str_squish()

    citations <- c(citations, citation)
  }
  return(data.frame("Packages" = citations))
}








#' Multiple Correlations.
#'
#' Compute different kinds of correlation matrices.
#'
#' @param df The dataframe.
#' @param df2 Optional dataframe to correlate with the first one.
#' @param type A character string indicating which correlation type is to be
#'   computed. One of "full" (default), "partial" (partial correlations),
#'   "semi" (semi-partial correlations), "glasso"
#'   (Graphical lasso- estimation of Gaussian graphical models) or "cor_auto"
#'   (will use the qgraph::cor_auto function to return pychoric or polyserial
#'   correlations if needed).
#' @param method A character string indicating which correlation coefficient is
#'   to be computed. One of "pearson" (default), "kendall", or "spearman" can be
#'   abbreviated.
#' @param adjust What adjustment for multiple tests should be used? ("holm",
#'   "hochberg", "hommel", "bonferroni", "BH", "BY", "fdr", "none"). See
#'   \link[stats]{p.adjust} for details about why to use "holm" rather than
#'   "bonferroni").
#' @param i_am_cheating Set to TRUE to run many uncorrected correlations.
#'
#' @return output
#'
#' @examples
#' df <- attitude
#'
#' # Normal correlations
#' results <- psycho::correlation(df)
#' print(results)
#' plot(results)
#'
#' # Partial correlations with correction
#' results <- psycho::correlation(df,
#'   type = "partial",
#'   method = "spearman",
#'   adjust = "holm"
#' )
#' print(results)
#' plot(results)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @importFrom stats na.omit p.adjust cor runif
#' @importFrom psych corr.test
#' @importFrom ggplot2 theme element_text
#' @importFrom stringr str_to_title
#' @import ggcorrplot
#' @import ppcor
#' @import dplyr
#' @export
correlation <- function(df,
                        df2 = NULL,
                        type = "full",
                        method = "pearson",
                        adjust = "holm",
                        i_am_cheating = FALSE) {

  # Processing
  # -------------------
  if (method == "bayes" | method == "bayesian") {
    return(bayes_cor(df, df2, reorder = TRUE))
  }


  # N samples
  n <- nrow(df)

  # Remove non numeric
  df <- purrr::keep(df, is.numeric)
  if (is.null(df2) == FALSE) {
    df2 <- purrr::keep(df2, is.numeric)
  }

  # P-fishing prevention
  if (ncol(df) > 10 && adjust == "none" && i_am_cheating == FALSE) {
    warning("We've detected that you are running a lot (> 10) of correlation tests without adjusting the p values. To help you in your p-fishing, we've added some interesting variables: You never know, you might find something significant!\nTo deactivate this, change the 'i_am_cheating' argument to TRUE.")
    df_complete <- dplyr::mutate_all(df, dplyr::funs_("replace(., is.na(.), 0)"))
    df$Local_Air_Density <- svd(df_complete)$u[, 1]
    df$Reincarnation_Cycle <- runif(nrow(df), max = 100)
    df$Communism_Level <- -1 * svd(df_complete)$u[, 2]
    df$Alien_Mothership_Distance <- rnorm(nrow(df), mean = 50000, sd = 5000)
    df$Schopenhauers_Optimism <- svd(df_complete)$u[, 3]
    df$Hulks_Power <- runif(nrow(df), max = 10)
  }



  # Compute r coefficients
  if (type == "full") {
    corr <- psych::corr.test(df, y = df2, use = "pairwise", method = method, adjust = "none")
    r <- corr$r
    p <- corr$p
    t <- corr$t
    ci <- corr$ci
    ci.adj <- corr$ci.adj
  } else {
    if (is.null(df2) == FALSE) {
      df <- cbind(df, df2)
    }

    df <- stats::na.omit(df) # enable imputation
    if (type == "semi") {
      corr <- ppcor::spcor(df, method = method)
      r <- corr$estimate
      p <- corr$p.value
      t <- corr$statistic
      ci <- "Not available for partial and semipartial correlations."
      ci.adj <- "Not available for partial and semipartial correlations."
    }
    else if (type == "partial") {
      corr <- ppcor::pcor(df, method = method)
      r <- corr$estimate
      p <- corr$p.value
      t <- corr$statistic
      ci <- "Not available for partial and semipartial correlations."
      ci.adj <- "Not available for partial and semipartial correlations."
    }
    else if (type == "glasso") {
      corr <- qgraph::EBICglasso(cor(df), n, gamma = 0.5)
      r <- corr
      p <- NULL
      t <- NULL
      ci <- "Not available for glasso estimation."
      ci.adj <- "Not available for glasso estimation."
    }
    else if (type == "cor_auto") {
      corr <- qgraph::cor_auto(df, forcePD = FALSE)
      r <- corr
      p <- NULL
      t <- NULL
      ci <- "Not available for cor_auto estimation."
      ci.adj <- "Not available for cor_auto estimation."
    }
    else {
      warning("type parameter must be 'full', 'semi', 'partial', 'glasso' or 'cor_auto'")
      return()
    }
  }



  # Adjust P values
  if (is.null(p) == FALSE) {
    if (adjust != "none") {
      if ((type == "full" & is.null(df2) == FALSE) | (type == "semi")) {
        p[, ] <- p.adjust(p, method = adjust)
      } else {
        p[lower.tri(p)] <- p.adjust(p[lower.tri(p)], method = adjust, n = choose(nrow(p), 2))
        p[upper.tri(p)] <- p.adjust(p[upper.tri(p)], method = adjust, n = choose(nrow(p), 2))
      }
    }
  }




  # Values
  # -------------
  values <- list(r = r, p = p, t = t, ci = ci, ci.adj = ci.adj, n = n)





  # Summary
  # -------------

  # Define notions for significance levels; spacing is important.
  if (is.null(p) == FALSE) {
    stars <- ifelse(p < .001, "***",
      ifelse(p < .01, "** ",
        ifelse(p < .05, "* ", " ")
      )
    )
  } else {
    stars <- ""
  }


  # build a new correlation matrix with significance stars
  table <- matrix(paste0(round(r, 2), stars), ncol = ncol(r))


  # Format
  rownames(table) <- colnames(df)
  if (isSymmetric(r)) {
    diag(table) <- paste0(diag(round(r, 2)), " ")
    colnames(table) <- colnames(df)
    table[upper.tri(table, diag = TRUE)] <- "" # remove upper triangle
    table <- as.data.frame(table)
    # remove last column and return the matrix (which is now a data frame)
    summary <- cbind(table[seq_len(length(table) - 1)])
  } else {
    if (is.null(df2)) {
      colnames(table) <- colnames(df)
    } else {
      if (type == "semi") {
        colnames(table) <- colnames(df)
      } else {
        colnames(table) <- colnames(df2)
      }
    }
    table <- as.data.frame(table)
    summary <- table
  }




  # Text
  # -------------
  sentences <- c()
  for (row in seq_len(nrow(r))) {
    for (col in seq_len(ncol(r))) {
      if (as.matrix(table)[row, col] == "") next # skip iteration and go to next iteration

      val_r <- as.matrix(r)[row, col]
      val_t <- tryCatch({
        as.matrix(t)[row, col]
      }, error = function(e) {
        "NA"
      })
      val_p <- tryCatch({
        as.matrix(p)[row, col]
      }, error = function(e) {
        "NA"
      })
      var1 <- colnames(r)[col]
      var2 <- row.names(r)[row]

      if (is.numeric(val_p) & val_p <= .05) {
        significance <- "significant "
      } else if (is.numeric(val_p) & val_p > .05) {
        significance <- "non significant "
      } else {
        significance <- ""
      }


      sentence <- paste0(
        "   - ",
        var1,
        " / ",
        var2,
        ":   ",
        "Results of the ",
        stringr::str_to_title(method),
        " correlation showed a ",
        significance,
        interpret_r(val_r),
        " association between ",
        var1,
        " and ",
        var2,
        " (r(",
        n - 2,
        ") = ",
        psycho::format_digit(val_r),
        ", p ",
        psycho::format_p(val_p),
        ")."
      )

      sentences <- c(sentences, sentence)
    }
  }

  sentences <- c(paste0(
    stringr::str_to_title(method),
    " ",
    stringr::str_to_title(type),
    " correlation (p value correction: ",
    adjust,
    "):\n"
  ), sentences)

  text <- sentences




  # Plot
  # -------------
  if (is.null(df2) == FALSE & type == "full") {
    corr <- psych::corr.test(cbind(df, df2), use = "pairwise", method = method, adjust = "none")
    r <- corr$r
    p <- corr$p
    p[lower.tri(p)] <- p.adjust(p[lower.tri(p)], method = adjust, n = choose(nrow(p), 2))
    p[upper.tri(p)] <- p.adjust(p[upper.tri(p)], method = adjust, n = choose(nrow(p), 2))
    # warning("Due to the presence of two dataframes, the plot might be incorrect. Consider with caution.")
  }

  if (type == "semi") {
    plot <- ggcorrplot::ggcorrplot(
      r,
      title = paste("A ", type, "'s correlation matrix (correction: ", adjust, ")\n", sep = ""),
      method = "circle",
      type = "full",
      colors = c("#E91E63", "white", "#03A9F4"),
      hc.order = TRUE,
      p.mat = p,
      insig = "pch",
      legend.title = "",
      lab = FALSE
    ) +
      ggplot2::theme(plot.title = ggplot2::element_text(hjust = 0.7))
  } else {
    plot <- ggcorrplot::ggcorrplot(
      r,
      title = paste("A ", type, "'s correlation matrix (correction: ", adjust, ")\n", sep = ""),
      method = "circle",
      type = "lower",
      colors = c("#E91E63", "white", "#03A9F4"),
      hc.order = TRUE,
      p.mat = p,
      insig = "pch",
      legend.title = "",
      lab = FALSE
    ) +
      ggplot2::theme(plot.title = ggplot2::element_text(hjust = 0.7))
  }



  # Output
  # -------------
  output <- list(text = text, plot = plot, summary = summary, values = values)

  class(output) <- c("psychobject", "psychobject_correlation", "list")
  return(output)
}











#'  Crawford-Garthwaite (2007) Bayesian test for single-case analysis.
#'
#' Neuropsychologists often need to compare a single case to a small control group. However, the standard two-sample t-test does not work because the case is only one observation. Crawford and Garthwaite (2007) demonstrate that the Bayesian test is a better approach than other commonly-used alternatives.
#' .
#'
#' @param patient Single value (patient's score).
#' @param controls Vector of values (control's scores).
#' @param mean Mean of the control sample.
#' @param sd SD of the control sample.
#' @param n Size of the control sample.
#' @param CI Credible interval bounds.
#' @param treshold Significance treshold.
#' @param iter Number of iterations.
#' @param color_controls Color of the controls distribution.
#' @param color_CI Color of CI distribution.
#' @param color_score Color of the line representing the patient's score.
#' @param color_size Size of the line representing the patient's score.
#' @param alpha_controls Alpha of the CI distribution.
#' @param alpha_CI lpha of the controls distribution.
#'
#'
#' @details The p value obtained when this test is used to test significance also simultaneously provides a point estimate of the abnormality of the patient’s score; for example if the one-tailed probability is .013 then we know that the patient’s score is significantly (p < .05) below the control mean and that it is estimated that 1.3% of the control population would obtain a score lower than the patient’s. As for the credible interval interpretation, we could say that there is a 95% probability that the true level of abnormality of the patient’s score lies within the stated limits, or that There is 95% confidence that the percentage of people who have a score lower than the patient’s is between 0.01% and 6.66%.
#'
#' @examples
#' library(psycho)
#'
#' crawford.test(patient = 125, mean = 100, sd = 15, n = 100)
#' plot(crawford.test(patient = 80, mean = 100, sd = 15, n = 100))
#'
#' crawford.test(patient = 10, controls = c(0, -2, 5, 2, 1, 3, -4, -2))
#' test <- crawford.test(patient = 7, controls = c(0, -2, 5, -6, 0, 3, -4, -2))
#' plot(test)
#' @author Dominique Makowski
#'
#' @importFrom stats pnorm var approx rchisq
#' @importFrom scales rescale
#' @import ggplot2
#' @export
crawford.test <- function(patient,
                          controls = NULL,
                          mean = NULL,
                          sd = NULL,
                          n = NULL,
                          CI = 95,
                          treshold = 0.1,
                          iter = 10000,
                          color_controls = "#2196F3",
                          color_CI = "#E91E63",
                          color_score = "black",
                          color_size = 2,
                          alpha_controls = 1,
                          alpha_CI = 0.8) {
  if (is.null(controls)) {
    # Check if a parameter is null
    if (length(c(mean, sd, n)) != 3) {
      stop("Please provide either controls or mean, sd and n.")
    }
    sample_mean <- mean
    sample_sd <- sd
    sample_var <- sd^2
  } else {
    sample_mean <- mean(controls)
    sample_var <- var(controls)
    sample_sd <- sd(controls)
    n <- length(controls)
  }
  degfree <- n - 1


  # Computation -------------------------------------------------------------


  pvalues <- c()
  for (i in 1:iter) {
    # step 1
    psi <- rchisq(1, df = degfree, ncp = 0)
    o <- (n - 1) * sample_var / psi

    # step 2
    z <- rnorm(1, 0, 1)
    u <- sample_mean + z * sqrt((o / n))

    # step 3
    z_patient <- (patient - u) / sqrt(o)
    p <- 2 * (1 - pnorm(abs(z_patient), lower.tail = TRUE)) # One-tailed p-value
    pvalues <- c(pvalues, p)
  }


  # Point estimates ---------------------------------------------------------

  z_score <- (patient - sample_mean) / sample_sd
  perc <- percentile(z_score)

  pvalues <- pvalues / 2
  p <- mean(pvalues)
  CI <- HDI(pvalues, prob = CI / 100)
  # CI_1 <- sort(pvalues)[iter * (100 - CI) / 100]


  # Text --------------------------------------------------------------------

  p_interpretation <- ifelse(p < treshold, " significantly ", " not significantly ")
  direction <- ifelse(patient - sample_mean < 0, " lower than ", " higher than ")


  text <- paste0(
    "The Bayesian test for single case assessment (Crawford, Garthwaite, 2007) suggests that the patient's score (Raw = ",
    format_digit(patient),
    ", Z = ",
    format_digit(z_score),
    ", percentile = ",
    format_digit(perc),
    ") is",
    p_interpretation,
    "different from the controls (M = ",
    format_digit(sample_mean),
    ", SD = ",
    format_digit(sample_sd),
    ", p ",
    format_p(p),
    ").",
    " The patient's score is",
    direction,
    format_digit((1 - p) * 100),
    "% (95% CI [",
    paste(format_digit(sort(c((1 - CI$values$HDImin) * 100, (1 - CI$values$HDImax) * 100))), collapse = ", "),
    "]) of the control population."
  )



  # Store values ------------------------------------------------------------

  values <- list(
    patient_raw = patient,
    patient_z = z_score,
    patient_percentile = perc,
    controls_mean = sample_mean,
    controls_sd = sample_sd,
    controls_var = sample_var,
    controls_sd = sample_sd,
    controls_n = n,
    text = text,
    p = p,
    CI_lower = CI$values$HDImin,
    CI_higher = CI$values$HDImax
  )

  summary <- data.frame(
    controls_mean = sample_mean,
    controls_sd = sample_sd,
    controls_n = n,
    p = p,
    CI_lower = CI$values$HDImin,
    CI_higher = CI$values$HDImax
  )

  if (is.null(controls)) {
    controls <- rnorm_perfect(n, sample_mean, sample_sd)
  }


  # Plot --------------------------------------------------------------------
  if (patient - sample_mean < 0) {
    uncertainty <- percentile_to_z(pvalues * 100)
  } else {
    uncertainty <- percentile_to_z((1 - pvalues) * 100)
  }




  plot <- rnorm_perfect(length(uncertainty), 0, 1) %>%
    density() %>%
    as.data.frame() %>%
    mutate_(y = "y/max(y)") %>%
    mutate(distribution = "Control") %>%
    rbind(uncertainty %>%
      density() %>%
      as.data.frame() %>%
      mutate_(y = "y/max(y)") %>%
      mutate(distribution = "Uncertainty")) %>%
    mutate_(x = "scales::rescale(x, from=c(0, 1), to = c(sample_mean, sample_mean+sample_sd))") %>%
    ggplot(aes_string(x = "x", ymin = 0, ymax = "y")) +
    geom_ribbon(aes_string(fill = "distribution", alpha = "distribution")) +
    geom_vline(xintercept = patient, colour = color_score, size = color_size) +
    scale_fill_manual(values = c(color_controls, color_CI)) +
    scale_alpha_manual(values = c(alpha_controls, alpha_CI)) +
    xlab("\nScore") +
    ylab("") +
    theme_minimal() +
    theme(
      legend.position = "none",
      axis.ticks.y = element_blank(),
      axis.text.y = element_blank()
    )



  output <- list(text = text, plot = plot, summary = summary, values = values)
  class(output) <- c("psychobject", "list")
  return(output)
}










#' Crawford-Howell (1998) frequentist t-test for single-case analysis.
#'
#' Neuropsychologists often need to compare a single case to a small control group. However, the standard two-sample t-test does not work because the case is only one observation. Crawford and Garthwaite (2012) demonstrate that the Crawford-Howell (1998) t-test is a better approach (in terms of controlling Type I error rate) than other commonly-used alternatives.
#' .
#'
#' @param patient Single value (patient's score).
#' @param controls Vector of values (control's scores).
#'
#' @return Returns a data frame containing the t-value, degrees of freedom, and p-value. If significant, the patient is different from the control group.
#'
#' @examples
#' library(psycho)
#'
#' crawford.test.freq(patient = 10, controls = c(0, -2, 5, 2, 1, 3, -4, -2))
#' crawford.test.freq(patient = 7, controls = c(0, -2, 5, 2, 1, 3, -4, -2))
#' @author Dan Mirman, Dominique Makowski
#'
#' @importFrom stats pt sd
#' @export
crawford.test.freq <- function(patient, controls) {
  tval <- (patient - mean(controls)) / (sd(controls) * sqrt((length(controls) + 1) / length(controls)))

  degfree <- length(controls) - 1

  pval <- 2 * (1 - pt(abs(tval), df = degfree)) # One-tailed p-value

  # One-tailed p value
  if (pval > .05 & pval / 2 < .05) {
    one_tailed <- paste0(
      " However, the null hypothesis of no difference can be rejected at a one-tailed 5% significance level (one-tailed p ",
      format_p(pval / 2),
      ")."
    )
  } else {
    one_tailed <- ""
  }


  p_interpretation <- ifelse(pval < 0.05, " significantly ", " not significantly ")
  t_interpretation <- ifelse(tval < 0, " lower than ", " higher than ")

  text <- paste0(
    "The Crawford-Howell (1998) t-test suggests that the patient's score (",
    format_digit(patient),
    ") is",
    p_interpretation,
    "different from the controls (M = ",
    format_digit(mean(controls)),
    ", SD = ",
    format_digit(sd(controls)),
    ", t(",
    degfree,
    ") = ",
    format_digit(tval),
    ", p ",
    format_p(pval),
    ").",
    one_tailed,
    " The patient's score is",
    t_interpretation,
    format_digit((1 - pval) * 100),
    "% of the control population."
  )

  values <- list(
    text = text,
    p = pval,
    df = degfree,
    t = tval
  )
  summary <- data.frame(t = tval, df = degfree, p = pval)
  plot <- "Not available yet"


  output <- list(text = text, plot = plot, summary = summary, values = values)
  class(output) <- c("psychobject", "list")
  return(output)
}








#' Crawford-Howell (1998) modified t-test for testing difference between a patientâ€™s performance on two tasks.
#'
#' Assessing dissociation between processes is a fundamental part of clinical neuropsychology. However, while the detection of suspected impairments is a fundamental feature of single-case studies, evidence of an impairment on a given task usually becomes of theoretical interest only if it is observed in the context of less impaired or normal performance on other tasks. Crawford and Garthwaite (2012) demonstrate that the Crawford-Howell (1998) t-test for dissociation is a better approach (in terms of controlling Type I error rate) than other commonly-used alternatives.
#' .
#'
#' @param case_X Single value (patient's score on test X).
#' @param case_Y Single value (patient's score on test Y).
#' @param controls_X Vector of values (control's scores of X).
#' @param controls_Y Vector of values (control's scores of Y).
#' @param verbose True or False. Prints the interpretation text.
#'
#' @return Returns a data frame containing the t-value, degrees of freedom, and p-value. If significant, the dissociation between test X and test Y is significant.
#'
#' @examples
#' library(psycho)
#'
#' case_X <- 142
#' case_Y <- 7
#' controls_X <- c(100, 125, 89, 105, 109, 99)
#' controls_Y <- c(7, 8, 9, 6, 7, 10)
#'
#' crawford_dissociation.test(case_X, case_Y, controls_X, controls_Y)
#' @author Dominique Makowski
#'
#' @importFrom stats sd pt
#' @export
crawford_dissociation.test <- function(case_X, case_Y, controls_X, controls_Y, verbose = TRUE) {
  X_mean <- mean(controls_X)
  X_sd <- sd(controls_X)
  Y_mean <- mean(controls_Y)
  Y_sd <- sd(controls_Y)
  r <- cor(controls_X, controls_Y)
  n <- length(controls_X)
  degfree <- n - 1

  case_X_Z <- (case_X - X_mean) / X_sd
  case_Y_Z <- (case_Y - Y_mean) / Y_sd

  tval <- (case_X_Z - case_Y_Z) / sqrt((2 - 2 * r) * ((n + 1) / n))

  pval <- 2 * (1 - pt(abs(tval), df = degfree)) # two-tailed p-value





  p_interpretation <- ifelse(pval < 0.05, " a significant ", " no ")
  p_interpretation2 <- ifelse(pval < 0.05, " ", " not ")
  z_interpretation <- ifelse(tval < 0, " below ", " above ")
  pop_interpretation <- ifelse(tval < 0, " above ", " below ")

  if (abs(case_X_Z) > abs(case_Y_Z)) {
    var_interpretation1 <- "test X"
    var_interpretation2 <- "test Y"
  } else {
    var_interpretation1 <- "test Y"
    var_interpretation2 <- "test X"
  }

  text <- paste0(
    "The Crawford-Howell (1998) t-test suggests",
    p_interpretation,
    "dissociation between test X and test Y (t(",
    degfree,
    ") = ",
    format_digit(tval),
    ", p ",
    format_p(pval),
    "). The patient's score on ",
    var_interpretation1,
    " is",
    p_interpretation2,
    "significantly altered compared to its score on ",
    var_interpretation2,
    "."
  )


  result <- data.frame(t = tval, df = degfree, p = pval)

  if (verbose == TRUE) {
    cat(paste0(text, "\n\n"))
  }

  return(result)
}







#' Overlap of Two Empirical Distributions.
#'
#' A method to calculate the overlap coefficient of two kernel density estimates (a measure of similarity between two samples).
#'
#' @param x A vector of numerics.
#' @param n Number of intervals to create, OR
#' @param length Length of each interval.
#' @param equal_range Makes n groups with with equal range (TRUE) or (approximately) equal numbers of observations (FALSE).
#' @param labels Can be a custom list, "NULL", "FALSE" or "median".
#' @param dig.lab Integer which is used when labels are not given. It determines the number of digits used in formatting the break numbers.
#'
#' @examples
#' library(psycho)
#'
#' x <- rnorm(100, 0, 1)
#'
#' create_intervals(x, n = 4)
#' create_intervals(x, n = 4, equal_range = FALSE)
#' create_intervals(x, length = 1)
#'
#' create_intervals(x, n = 4, labels = "median")
#' create_intervals(x, n = 4, labels = FALSE)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @importFrom ggplot2 cut_interval cut_number
#' @export
create_intervals <- function(x, n = NULL, length = NULL, equal_range = TRUE, labels = NULL, dig.lab = 3) {
  if (equal_range) {
    if (is.character(labels) && labels == "median") {
      cuts <- ggplot2::cut_interval(x, n = n, length = length, labels = FALSE)
    } else {
      cuts <- ggplot2::cut_interval(x, n = n, length = length, labels = labels, dig.lab = dig.lab)
    }
  } else {
    if (is.character(labels) && labels == "median") {
      cuts <- ggplot2::cut_number(x, n = n, labels = FALSE)
    } else {
      cuts <- ggplot2::cut_number(x, n = n, labels = labels, dig.lab = dig.lab)
    }
  }


  if (is.character(labels) && labels == "median") {
    cuts <- cuts %>%
      data.frame(x) %>%
      group_by_(".") %>%
      mutate_("cuts" = "median(x)") %>%
      ungroup() %>%
      select_("cuts") %>%
      pull()
  }

  return(cuts)
}











#' Dprime and Other Signal Detection Theory indices.
#'
#' Computes Signal Detection Theory indices (d', beta, A', B''D, c).
#'
#' @param n_hit Number of hits.
#' @param n_fa Number of false alarms.
#' @param n_miss Number of misses.
#' @param n_cr Number of correct rejections.
#' @param n_targets Number of targets (n_hit + n_miss).
#' @param n_distractors Number of distractors (n_fa + n_cr).
#' @param adjusted Should it use the Hautus (1995) adjustments for extreme values.
#'
#' @return Calculates the d', the beta, the A' and the B''D based on the signal detection theory (SRT). See Pallier (2002) for the algorithms.
#'
#' Returns a list containing the following indices:
#' \itemize{
#'  \item{\strong{dprime (d')}: }{The sensitivity. Reflects the distance between the two distributions: signal, and signal+noise and corresponds to the Z value of the hit-rate minus that of the false-alarm rate.}
#'  \item{\strong{beta}: }{The bias (criterion). The value for beta is the ratio of the normal density functions at the criterion of the Z values used in the computation of d'. This reflects an observer's bias to say 'yes' or 'no' with the unbiased observer having a value around 1.0. As the bias to say 'yes' increases (liberal), resulting in a higher hit-rate and false-alarm-rate, beta approaches 0.0. As the bias to say 'no' increases (conservative), resulting in a lower hit-rate and false-alarm rate, beta increases over 1.0 on an open-ended scale.}
#'  \item{\strong{c}: }{Another index of bias. the number of standard deviations from the midpoint between these two distributions, i.e., a measure on a continuum from "conservative" to "liberal".}
#'  \item{\strong{aprime (A')}: }{Non-parametric estimate of discriminability. An A' near 1.0 indicates good discriminability, while a value near 0.5 means chance performance.}
#'  \item{\strong{bppd (B''D)}: }{Non-parametric estimate of bias. A B''D equal to 0.0 indicates no bias, positive numbers represent conservative bias (i.e., a tendency to answer 'no'), negative numbers represent liberal bias (i.e. a tendency to answer 'yes'). The maximum absolute value is 1.0.}
#'  }
#'
#'
#' Note that for d' and beta, adjustement for extreme values are made following the recommandations of Hautus (1995).


#' @examples
#' library(psycho)
#'
#' n_hit <- 9
#' n_fa <- 2
#' n_miss <- 1
#' n_cr <- 7
#'
#' indices <- psycho::dprime(n_hit, n_fa, n_miss, n_cr)
#'
#'
#' df <- data.frame(
#'   Participant = c("A", "B", "C"),
#'   n_hit = c(1, 2, 5),
#'   n_fa = c(6, 8, 1)
#' )
#'
#' indices <- psycho::dprime(
#'   n_hit = df$n_hit,
#'   n_fa = df$n_fa,
#'   n_targets = 10,
#'   n_distractors = 10,
#'   adjusted = FALSE
#' )
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @importFrom stats qnorm
#' @export
dprime <- function(n_hit, n_fa, n_miss = NULL, n_cr = NULL, n_targets = NULL, n_distractors = NULL, adjusted = TRUE) {
  if (is.null(n_targets)) {
    n_targets <- n_hit + n_miss
  }

  if (is.null(n_distractors)) {
    n_distractors <- n_fa + n_cr
  }



  # Parametric Indices ------------------------------------------------------


  if (adjusted == TRUE) {
    if (is.null(n_miss) | is.null(n_cr)) {
      warning("Please provide n_miss and n_cr in order to compute adjusted ratios. Computing indices anyway with non-adjusted ratios...")

      # Non-Adjusted ratios
      hit_rate_adjusted <- n_hit / n_targets
      fa_rate_adjusted <- n_fa / n_distractors
    } else {
      # Adjusted ratios
      hit_rate_adjusted <- (n_hit + 0.5) / ((n_hit + 0.5) + n_miss + 1)
      fa_rate_adjusted <- (n_fa + 0.5) / ((n_fa + 0.5) + n_cr + 1)
    }

    # dprime
    dprime <- qnorm(hit_rate_adjusted) - qnorm(fa_rate_adjusted)

    # beta
    zhr <- qnorm(hit_rate_adjusted)
    zfar <- qnorm(fa_rate_adjusted)
    beta <- exp(-zhr * zhr / 2 + zfar * zfar / 2)

    # c
    c <- -(qnorm(hit_rate_adjusted) + qnorm(fa_rate_adjusted)) / 2
  } else {
    # Ratios
    hit_rate <- n_hit / n_targets
    fa_rate <- n_fa / n_distractors

    # dprime
    dprime <- qnorm(hit_rate) - qnorm(fa_rate)

    # beta
    zhr <- qnorm(hit_rate)
    zfar <- qnorm(fa_rate)
    beta <- exp(-zhr * zhr / 2 + zfar * zfar / 2)

    # c
    c <- -(qnorm(hit_rate) + qnorm(fa_rate)) / 2
  }

  # Non-Parametric Indices ------------------------------------------------------

  if(any(n_distractors == 0)){
    warning("Several rows have 0 distractors. Please remove these rows to compute non-parametric indices.")
    return(list(dprime = dprime, beta = beta, aprime = NA, bppd = NA, c = c))
  }

  # Ratios
  hit_rate <- n_hit / n_targets
  fa_rate <- n_fa / n_distractors

  # aprime
  a <- 1 / 2 + ((hit_rate - fa_rate) * (1 + hit_rate - fa_rate) / (4 * hit_rate * (1 - fa_rate)))
  b <- 1 / 2 - ((fa_rate - hit_rate) * (1 + fa_rate - hit_rate) / (4 * fa_rate * (1 - hit_rate)))

  a[fa_rate > hit_rate] <- b[fa_rate > hit_rate]
  a[fa_rate == hit_rate] <- .5
  aprime <- a

  # bppd
  bppd <- (hit_rate * (1 - hit_rate) - fa_rate * (1 - fa_rate)) / (hit_rate * (1 - hit_rate) + fa_rate * (1 - fa_rate))
  bppd_b <- (fa_rate * (1 - fa_rate) - hit_rate * (1 - hit_rate)) / (fa_rate * (1 - fa_rate) + hit_rate * (1 - hit_rate))
  bppd[fa_rate > hit_rate] <- bppd_b[fa_rate > hit_rate]



  return(list(dprime = dprime, beta = beta, aprime = aprime, bppd = bppd, c = c))
}








#' Returns all combinations of lavaan models with their indices of fit.
#'
#' Returns all combinations of lavaan models with their indices of fit.
#'
#' @param fit A lavaan object.
#' @param latent Copy/paste the part related to latent variables loadings.
#' @param samples Number of random draws.
#' @param verbose Show progress.
#' @param ... Arguments passed to or from other methods.
#'
#' @return list containing all combinations.
#'
#' @examples
#' library(psycho)
#' library(lavaan)
#'
#' model <- " visual  =~ x1 + x2 + x3
#' textual =~ x4 + x5 + x6
#' speed   =~ x7 + x8 + x9
#' visual ~ textual
#' textual ~ speed"
#' fit <- lavaan::sem(model, data = HolzingerSwineford1939)
#'
#' models <- find_best_model(fit, latent = "visual  =~ x1 + x2 + x3
#' textual =~ x4 + x5 + x6
#' speed   =~ x7 + x8 + x9")
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @import dplyr
#'
#' @method find_best_model lavaan
#' @export
find_best_model.lavaan <- function(fit, latent = "", samples = 1000, verbose = FALSE, ...) {
  update_model <- function(fit, latent, model) {
    newfit <- update(fit, paste0(latent, "\n", model))

    indices <- data.frame(Value = lavaan::fitMeasures(newfit)) %>%
      tibble::rownames_to_column("Index") %>%
      tidyr::spread_("Index", "Value") %>%
      cbind(data.frame(
        model = model,
        n_links = nrow(lavaan::lavInspect(fit, "est")$beta)
      ))
    return(indices)
  }

  vars <- row.names(lavaan::lavInspect(fit, "est")$beta)
  # info <- fit@Model

  data <- data.frame()
  for (outcome in vars) {
    remaning_vars <- vars[!stringr::str_detect(vars, outcome)]
    combinations <- c()
    for (y in 1:length(remaning_vars)) {
      combinations <- c(combinations, combn(remaning_vars, y, simplify = FALSE))
    }
    combinations <- sapply(combinations, paste0, collapse = "+")
    combinations <- paste0(outcome, "~", combinations)
    x <- data.frame(A = combinations)
    names(x) <- c(outcome)
    if (nrow(data) == 0) {
      data <- x
    } else {
      data <- cbind(data, x)
    }
  }

  data <- rbind(data, head(data[NA, ], 1))
  data[] <- lapply(data, as.character)
  data[is.na(data)] <- ""
  rownames(data) <- NULL

  out <- data.frame()
  for (i in 1:samples) {
    if (verbose == TRUE) {
      cat(".")
    }
    model <- ""
    for (var in names(data)) {
      model <- paste0(model, sample(data[[var]], 1), "\n")
    }

    if (!model %in% out$model) {
      out <- tryCatch(
        rbind(out, update_model(fit, latent, model)),
        error = function(e) out,
        warning = function(w) out
      )
    }
  }
  return(out)
}








#' Returns the best combination of predictors for lmerTest objects.
#'
#' Returns the best combination of predictors for lmerTest objects.
#'
#' @param fit A merModLmerTest object.
#' @param interaction Include interaction term.
#' @param fixed Additional formula part to add at the beginning of
#' each formula
#' @param ... Arguments passed to or from other methods.
#'
#' @return list containing all combinations.
#'
#' @examples
#' \dontrun{
#' library(psycho)
#' library(lmerTest)
#'
#' data <- standardize(iris)
#' fit <- lmerTest::lmer(Sepal.Length ~ Sepal.Width + Petal.Length + (1 | Species), data = data)
#'
#' best <- find_best_model(fit)
#' best_formula <- best$formula
#' best$table
#' }
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @importFrom stats update
#' @import dplyr
#'
#' @method find_best_model lmerModLmerTest
#' @export
find_best_model.lmerModLmerTest <- function(fit, interaction = TRUE, fixed = NULL, ...) {

  # Extract infos
  combinations <- find_combinations(as.formula(get_formula(fit)), interaction = interaction, fixed = fixed)


  # Recreating the dataset without NA
  dataComplete <- fit@frame[complete.cases(fit@frame), ]


  # fit models
  models <- c()
  for (formula in combinations) {
    newfit <- update(fit, formula, data = dataComplete)
    models <- c(models, newfit)
  }


  # No warning messages for this part
  options(warn = -1)

  # Model comparison
  comparison <- as.data.frame(do.call("anova", models))

  # Re-displaying warning messages
  options(warn = 0)

  # Creating row names to the combinations array equivalent to the comparison data frame
  combinations <- as.data.frame(combinations, row.names = paste0("MODEL", seq(1, length(combinations))))

  # Reordering the rows in the same way for both combinations and comparison before implementing the formulas
  comparison <- comparison[ order(row.names(comparison)), ]
  comparison$formula <- combinations[order(row.names(combinations)), ]

  # Sorting the data frame by the AIC then BIC
  comparison <- comparison[order(comparison$AIC, comparison$BIC), ]



  # Best model by criterion
  best_aic <- dplyr::arrange_(comparison, "AIC") %>%
    dplyr::select_("formula") %>%
    head(1)
  best_aic <- as.character(best_aic[[1]])

  best_bic <- dplyr::arrange_(comparison, "BIC") %>%
    dplyr::select_("formula") %>%
    head(1)
  best_bic <- as.character(best_bic[[1]])

  by_criterion <- data.frame(formula = c(best_aic, best_bic), criterion = c("AIC", "BIC"))

  # Best formula
  best <- table(by_criterion$formula)
  best <- names(best[which.max(best)])

  best <- list(formula = best, by_criterion = by_criterion, table = comparison)
  return(best)
}








#' Returns the best model.
#'
#' Returns the best model. See the
#' documentation for your model's class:
#' \itemize{
#'  \item{\link[=find_best_model.stanreg]{find_best_model.stanreg}}
#'  \item{\link[=find_best_model.lmerModLmerTest]{find_best_model.lmerModLmerTest}}
#'  }
#'
#' @param fit Model
#' @param ... Arguments passed to or from other methods.
#'
#' @seealso \code{\link{find_best_model.stanreg}}
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
find_best_model <- function(fit, ...) {
  UseMethod("find_best_model")
}









#' Returns the best combination of predictors based on LOO cross-validation indices.
#'
#' Returns the best combination of predictors based on LOO cross-validation indices.
#'
#' @param fit A stanreg object.
#' @param interaction Include interaction term.
#' @param fixed Additional formula part to add at the beginning of
#' each formula
#' @param K For kfold, the number of subsets of equal (if possible) size into
#' which the data will be randomly partitioned for performing K-fold
#' cross-validation. The model is refit K times, each time leaving out one of
#' the K subsets. If K is equal to the total number of observations in the data
#' then K-fold cross-validation is equivalent to exact leave-one-out
#' cross-validation.
#' @param k_treshold Threshold for flagging estimates of the Pareto shape
#' parameters k estimated by loo.
#' @param ... Arguments passed to or from other methods.
#'
#' @return list containing all combinations.
#'
#' @examples
#' \dontrun{
#' library(psycho)
#' library(rstanarm)
#'
#' data <- standardize(attitude)
#' fit <- rstanarm::stan_glm(rating ~ advance + privileges, data = data)
#'
#' best <- find_best_model(fit)
#' best_formula <- best$formula
#' best$table
#'
#' # To deactivate Kfold evaluation
#' best <- find_best_model(fit, K = 0)
#' }
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @importFrom rstanarm bayes_R2
#' @importFrom loo loo kfold
#' @importFrom stats update median
#' @import dplyr
#'
#' @method find_best_model stanreg
#' @export
find_best_model.stanreg <- function(fit, interaction = TRUE, fixed = NULL, K = 10, k_treshold = NULL, ...) {

  # Extract infos
  combinations <- find_combinations(fit$formula, interaction = interaction, fixed = fixed)

  # Compute fitting indices
  loos <- list()
  kfolds <- list()
  complexities <- list()
  R2s <- list()
  for (i in seq_len(length(combinations))) {
    print(paste0(i, "/", length(combinations)))

    formula <- combinations[i]
    newfit <- update(fit, formula = formula, verbose = FALSE)
    R2s[[formula]] <- median(rstanarm::bayes_R2(newfit))


    if (!is.null(k_treshold)) {
      loo <- loo::loo(newfit, k_treshold = k_treshold)
    } else {
      loo <- loo::loo(newfit)
    }


    complexities[[formula]] <- length(newfit$coefficients)
    loos[[formula]] <- loo
    if (K > 1) {
      kfold <- loo::kfold(newfit, K = K)
    } else {
      kfold <- list(elpd_kfold = 0, se_elpd_kfold = 0)
    }
    kfolds[[formula]] <- kfold
  }

  # Model comparison
  comparison <- data.frame()
  for (formula in names(loos)) {
    loo <- loos[[formula]]
    kfold <- kfolds[[formula]]
    complexity <- complexities[[formula]]
    Estimates <- loo[["estimates"]]
    model <- data.frame(
      formula = formula,
      complexity = complexity - 1,
      R2 = R2s[[formula]],
      looic = Estimates["looic", "Estimate"],
      looic_se = Estimates["looic", "SE"],
      elpd_loo = Estimates["elpd_loo", "Estimate"],
      elpd_loo_se = Estimates["elpd_loo", "SE"],
      p_loo = Estimates["p_loo", "Estimate"],
      p_loo_se = Estimates["p_loo", "SE"],
      elpd_kfold = Estimates["p_loo", "Estimate"],
      elpd_kfold_se = Estimates["p_loo", "SE"]
    )
    comparison <- rbind(comparison, model)
  }

  # Format
  comparison <- comparison %>%
    dplyr::mutate_(
      "looic_d" = "looic - min(looic)",
      "elpd_loo_d" = "elpd_loo - max(elpd_loo)",
      "elpd_kfold_d" = "elpd_kfold - max(elpd_kfold)"
    )

  # Best model by criterion
  best_looic <- dplyr::arrange_(comparison, "looic") %>%
    dplyr::select_("formula") %>%
    head(1)
  best_looic <- as.character(best_looic[[1]])

  best_elpd_loo <- dplyr::arrange_(comparison, "desc(elpd_loo)") %>%
    dplyr::select_("formula") %>%
    head(1)
  best_elpd_loo <- as.character(best_elpd_loo[[1]])

  if (K > 1) {
    best_elpd_kfold <- dplyr::arrange_(comparison, "desc(elpd_kfold)") %>%
      dplyr::select_("formula") %>%
      head(1)
    best_elpd_kfold <- as.character(best_elpd_kfold[[1]])
  } else {
    best_elpd_kfold <- NA
  }

  by_criterion <- data.frame(formula = c(best_looic, best_elpd_loo, best_elpd_kfold), criterion = c("looic", "elpd_loo", "elpd_kfold"))

  # Best formula
  best <- table(by_criterion$formula)
  best <- names(best[which.max(best)])

  best <- list(formula = best, by_criterion = by_criterion, table = comparison)
  return(best)
}







#' Generate all combinations.
#'
#' Generate all combinations.
#'
#' @param object Object
#' @param ... Arguments passed to or from other methods.
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
find_combinations <- function(object, ...) {
  UseMethod("find_combinations")
}














#' Generate all combinations of predictors of a formula.
#'
#' Generate all combinations of predictors of a formula.
#'
#' @param object Formula.
#' @param interaction Include interaction term.
#' @param fixed Additional formula part to add at the beginning of
#' each combination.
#' @param ... Arguments passed to or from other methods.
#'
#' @return list containing all combinations.
#'
#' @examples
#' library(psycho)
#'
#' f <- as.formula("Y ~ A + B + C + D")
#' f <- as.formula("Y ~ A + B + C + D + (1|E)")
#' f <- as.formula("Y ~ A + B + C + D + (1|E) + (1|F)")
#'
#' find_combinations(f)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @method find_combinations formula
#' @importFrom utils combn
#' @importFrom stats terms
#' @export
find_combinations.formula <- function(object, interaction = TRUE, fixed = NULL, ...) {

  # Extract infos
  formula <- object
  vars <- attributes(terms(formula))$term.labels
  outcome <- all.vars(formula)[1]
  pred <- vars[!grepl("\\|", vars)]
  if (length(vars[grepl("\\|", vars)]) > 0) {
    random <- paste0(" + (", vars[grepl("\\|", vars)], ")")
  } else {
    random <- ""
  }

  if (is.null(fixed)) {
    fixed <- ""
  } else {
    fixed <- fixed
  }

  # Generate combinations
  n <- length(pred)

  id <- unlist(
    lapply(
      1:n,
      function(i) combn(1:n, i, simplify = FALSE)
    ),
    recursive = FALSE
  )

  combinations <- sapply(id, function(i)
    paste(paste(pred[i], collapse = " + ")))


  # Generate interactions
  if (interaction == TRUE) {
    for (comb in combinations) {
      n_signs <- stringr::str_count(comb, "\\+")
      if (n_signs > 0) {
        new_formula <- comb
        for (i in 1:n_signs) {
          new_formula <- stringr::str_replace(new_formula, "\\+", "*")
          combinations <- c(combinations, new_formula)
        }
      }
    }
  }

  combinations <- paste0(outcome, " ~ ", fixed, combinations, paste0(random, collapse = ""))
  return(combinations)
}








#' Find the distance of a point with its kmean cluster.
#'
#' Find the distance of a point with its kmean cluster.
#'
#' @param df Data
#' @param km kmean object.
#'
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
find_distance_cluster <- function(df, km) {
  myDist <- function(p1, p2) sqrt((p1[, 1] - p2[, 1])^2 + (p1[, 2] - p2[, 2])^2)

  data <- df %>%
    as.data.frame() %>%
    select(one_of(colnames(km$centers)))

  n_clusters <- nrow(km$centers)

  data$Distance <- NA
  for (clust in 1:n_clusters) {
    data$Distance[km$cluster == clust] <- myDist(data[km$cluster == clust, ], km$centers[clust, , drop = FALSE])
  }

  return(data$Distance)
}







#' Find the Highest Density Point.
#'
#' Returns the Highest Density Point.
#'
#' @param x Vector.
#' @param precision Number of points in density.
#'
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
find_highest_density_point <- function(x, precision = 1e+03) {
  d <- x %>%
    density(n = precision) %>%
    as.data.frame()
  y <- d$x[which.max(d$y)]
  return(y)
}






#' Fuzzy string matching.
#'
#' @param x Strings.
#' @param y List of strings to be matched.
#' @param value Return value or the index of the closest string.
#' @param step Step by which decrease the distance.
#' @param ignore.case if FALSE, the pattern matching is case sensitive and if TRUE, case is ignored during matching.
#'
#' @examples
#' library(psycho)
#' find_matching_string("Hwo rea ouy", c("How are you", "Not this word", "Nice to meet you"))
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
find_matching_string <- function(x, y, value = TRUE, step = 0.1, ignore.case = TRUE) {
  z <- c()
  for (i in seq_len(length(x))) {
    s <- x[i]
    distance <- 0.99
    closest <- agrep(s, y, max.distance = distance, value = value, ignore.case = ignore.case)

    while (length(closest) != 1) {
      closest <- agrep(s, closest, max.distance = distance, value = value, ignore.case = ignore.case)
      distance <- distance - step
      if (distance < 0) {
        warning(paste0("Couldn't find matching string for '", s, "'. Try lowering the step parameter."))
        closest <- s
      }
    }
    z <- c(z, closest)
  }
  return(z)
}










#' Find random effects in formula.
#'
#' @param formula Formula

#' @examples
#' library(psycho)
#' find_random_effects("Y ~ X + (1|Group)")
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @importFrom stringr str_remove_all
#' @importFrom lme4 findbars
#' @export
find_random_effects <- function(formula) {
  random <- lme4::findbars(as.formula(formula))
  random <- paste0("(", random, ")")
  random <- stringr::str_remove_all(random, " ")
  random <- paste(random, collapse = " + ")
  return(random)
}






#' Find season of dates.
#'
#' Returns the season of an array of dates.
#'
#' @param dates Array of dates.
#' @param winter month-day of winter solstice.
#' @param spring month-day of spring equinox.
#' @param summer month-day of summer solstice.
#' @param fall month-day of fall equinox.
#'
#' @return season
#'
#' @examples
#' library(psycho)
#'
#' dates <- c("2012-02-15", "2017-05-15", "2009-08-15", "1912-11-15")
#' find_season(dates)
#' @author Josh O'Brien
#'
#' @seealso
#' https://stackoverflow.com/questions/9500114/find-which-season-a-particular-date-belongs-to
#'
#' @export
find_season <- function(dates, winter = "12-21", spring = "3-20", summer = "6-21", fall = "9-22") {
  WS <- as.Date(paste0("2012-", winter), format = "%Y-%m-%d") # Winter Solstice
  SE <- as.Date(paste0("2012-", spring), format = "%Y-%m-%d") # Spring Equinox
  SS <- as.Date(paste0("2012-", summer), format = "%Y-%m-%d") # Summer Solstice
  FE <- as.Date(paste0("2012-", fall), format = "%Y-%m-%d") # Fall Equinox

  # Convert dates from any year to 2012 dates
  d <- as.Date(strftime(as.character(dates), format = "2012-%m-%d"))

  season <- ifelse(d >= WS | d < SE, "Winter",
    ifelse(d >= SE & d < SS, "Spring",
      ifelse(d >= SS & d < FE, "Summer", "Fall")
    )
  )
  return(season)
}










#' Tidyverse-friendly sprintf.
#'
#' @param x Values.
#' @param fmt A character vector of format strings, each of up to 8192 bytes.
#' @param ... values to be passed into fmt. Only logical, integer, real and
#' character vectors are supported, but some coercion will be done: see the ‘Details’ section. Up to 100.
#'
#' @export
format_string <- function(x, fmt, ...) {
  x <- sprintf(fmt, x, ...)
  return(x)
}






#' Format p values.
#'
#' @param pvalues p values (scalar or vector).
#' @param stars Add significance stars.
#' @param stars_only Return only significance stars.
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @importFrom stringr str_remove_all
#' @export
format_p <- function(pvalues, stars = TRUE, stars_only=FALSE) {
  p <- ifelse(pvalues < 0.001, "< .001***",
    ifelse(pvalues < 0.01, "< .01**",
      ifelse(pvalues < 0.05, "< .05*",
        ifelse(pvalues < 0.1, paste0("= ", round(pvalues, 2), "\xB0"),
          "> .1"
        )
      )
    )
  )

  if (stars_only == TRUE) {
    p <- stringr::str_remove_all(p, "[^\\*]")
  } else {
    if (stars == FALSE) {
      p <- stringr::str_remove_all(p, "\\*")
    }
  }

  return(p)
}








#' Clean and format formula.
#'
#' Clean and format formula.
#'
#' @param formula formula
#' @param ... Arguments passed to or from other methods.
#'
#'
#' @examples
#' library(psycho)
#' library(lme4)
#'
#' fit <- lme4::glmer(vs ~ wt + (1 | gear), data = mtcars, family = "binomial")
#' fit <- lm(hp ~ wt, data = mtcars)
#'
#' format_formula(get_formula(fit))
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
format_formula <- function(formula) {
  formula <- tryCatch({
    stringr::str_squish(paste(format(eval(formula)), collapse = ""))
  }, error = function(e) {
    formula <- stringr::str_squish(paste(format(formula), collapse = ""))
  })

  return(formula)
}









#' Compute estimated contrasts from models.
#'
#' Compute estimated contrasts between factor levels based on a fitted model.
#' See the documentation for your model's class:
#' \itemize{
#'  \item{\link[=get_contrasts.glm]{get_contrasts.glm}}
#'  \item{\link[=get_contrasts.lmerModLmerTest]{get_contrasts.merModLmerTest}}
#'  \item{\link[=get_contrasts.glmerMod]{get_contrasts.glmerMod}}
#'  \item{\link[=get_contrasts.stanreg]{get_contrasts.stanreg}}
#'  }
#'
#'
#' @param fit A model.
#' @param ... Arguments passed to or from other methods.
#'
#' @return Estimated contrasts.
#'
#'
#' @examples
#' \dontrun{
#' library(psycho)
#' require(lmerTest)
#' require(rstanarm)
#'
#' fit <- lm(Adjusting ~ Birth_Season * Salary, data = affective)
#' get_contrasts(fit)
#'
#' fit <- lm(Adjusting ~ Birth_Season * Salary, data = affective)
#' get_contrasts(fit, adjust = "bonf")
#'
#' fit <- lmerTest::lmer(Adjusting ~ Birth_Season * Salary + (1 | Salary), data = affective)
#' get_contrasts(fit, formula = "Birth_Season")
#'
#' fit <- rstanarm::stan_glm(Adjusting ~ Birth_Season, data = affective)
#' get_contrasts(fit, formula = "Birth_Season", ROPE_bounds = c(-0.1, 0.1))
#' }
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#'
#' @export
get_contrasts <- function(fit, ...) {
  UseMethod("get_contrasts")
}


#' Compute estimated contrasts from models.
#'
#' Compute estimated contrasts from models.
#'
#' @param fit A Bayesian model.
#' @param formula A character vector (formula like format, i.e., including
#' interactions or nesting terms) specifying the names of the predictors over which EMMs are desired.
#' @param CI Determine the confidence or credible interval bounds.
#' @param ROPE_bounds Optional bounds of the ROPE for Bayesian models.
#' @param overlap Set to TRUE to add Overlap index (for Bayesian models).
#' @param ... Arguments passed to or from other methods.
#' @method get_contrasts stanreg
#' @export
get_contrasts.stanreg <- function(fit, formula = NULL, CI = 90, ROPE_bounds = NULL, overlap = FALSE, ...) {
  .get_contrasts_bayes(fit, formula, CI, ROPE_bounds, overlap, ...)
}


#' Compute estimated contrasts from models.
#'
#' Compute estimated contrasts from models.
#'
#' @param fit A frequentist model.
#' @param formula A character vector (formula like format, i.e., including
#' interactions or nesting terms) specifying the names of the predictors over which EMMs are desired.
#' @param CI Determine the confidence or credible interval bounds.
#' @param adjust P value adjustment method for frequentist models. Default is "tukey". Can be "holm",
#' "hochberg", "hommel", "bonferroni", "BH", "BY", "fdr" or "none".
#' @param ... Arguments passed to or from other methods.
#' @method get_contrasts lm
#' @export
get_contrasts.lm <- function(fit, formula = NULL, CI = 95, adjust = "tukey", ...) {
  .get_contrasts_freq(fit, formula, CI, adjust, ...)
}

#' Compute estimated contrasts from models.
#'
#' Compute estimated contrasts from models.
#'
#' @inheritParams get_contrasts.lm
#' @method get_contrasts glm
#' @export
get_contrasts.glm <- function(fit, formula = NULL, CI = 95, adjust = "tukey", ...) {
  .get_contrasts_freq(fit, formula, CI, adjust, ...)
}

#' Compute estimated contrasts from models.
#'
#' Compute estimated contrasts from models.
#'
#' @inheritParams get_contrasts.lm
#' @method get_contrasts lmerModLmerTest
#' @export
get_contrasts.lmerModLmerTest <- function(fit, formula = NULL, CI = 95, adjust = "tukey", ...) {
  .get_contrasts_freq(fit, formula, CI, adjust, ...)
}

#' Compute estimated contrasts from models.
#'
#' Compute estimated contrasts from models.
#'
#' @inheritParams get_contrasts.lm
#' @method get_contrasts glmerMod
#' @export
get_contrasts.glmerMod <- function(fit, formula = NULL, CI = 95, adjust = "tukey", ...) {
  .get_contrasts_freq(fit, formula, CI, adjust, ...)
}

#' Compute estimated contrasts from models.
#'
#' Compute estimated contrasts from models.
#'
#' @inheritParams get_contrasts.lm
#' @method get_contrasts lmerMod
#' @export
get_contrasts.lmerMod <- function(fit, formula = NULL, CI = 95, adjust = "tukey", ...) {
  .get_contrasts_freq(fit, formula, CI, adjust, ...)
}




#' @import dplyr
#' @importFrom emmeans emmeans
#' @importFrom graphics pairs
#' @importFrom stats confint mad
#' @keywords internal
.get_contrasts_bayes <- function(fit, formula = NULL, CI = 90, ROPE_bounds = NULL, overlap = FALSE, ...) {
  if (is.null(formula)) {
    formula <- paste(get_info(fit)$predictors, collapse = " * ")
  }

  if (is.character(formula)) {
    formula <- as.formula(paste0("~ ", formula))
  }

  # Contrasts ---------------------------------------------------------------
  contrasts_posterior <- fit %>%
    emmeans::emmeans(formula) %>%
    graphics::pairs() %>%
    emmeans::as.mcmc.emmGrid() %>%
    as.matrix() %>%
    as.data.frame()

  contrasts <- data.frame()


  for (name in names(contrasts_posterior)) {
    posterior <- contrasts_posterior[[name]]

    CI_values <- HDI(posterior, prob = CI / 100)
    CI_values <- c(CI_values$values$HDImin, CI_values$values$HDImax)

    var <- data.frame(
      Contrast = stringr::str_remove(name, "contrast "),
      Median = median(posterior),
      MAD = mad(posterior),
      CI_lower = CI_values[seq(1, length(CI_values), 2)],
      CI_higher = CI_values[seq(2, length(CI_values), 2)],
      MPE = mpe(posterior)$MPE
    )

    if (overlap == TRUE) {
      var$Overlap <- 100 * overlap(
        posterior,
        rnorm_perfect(
          length(posterior),
          0,
          sd(posterior)
        )
      )
    }

    if (!is.null(ROPE_bounds)) {
      var$ROPE <- rope(posterior, ROPE_bounds, CI = 95)$rope_probability
    }

    contrasts <- rbind(contrasts, var)
  }


  return(contrasts)
}




#' @import dplyr
#' @importFrom emmeans emmeans
#' @importFrom graphics pairs
#' @importFrom stats confint
#' @keywords internal
.get_contrasts_freq <- function(fit, formula = NULL, CI = 95, adjust = "tukey", ...) {
  if (is.null(formula)) {
    formula <- paste(get_info(fit)$predictors, collapse = " * ")
  }

  if (is.character(formula)) {
    formula <- as.formula(paste0("~ ", formula))
  }

  # Contrasts ---------------------------------------------------------------
  contrasts <- fit %>%
    emmeans::emmeans(formula) %>%
    graphics::pairs(adjust = adjust)

  # Confint
  CI <- contrasts %>%
    confint(CI / 100) %>%
    select(contains("CL"))


  contrasts <- contrasts %>%
    as.data.frame() %>%
    cbind(CI) %>%
    dplyr::rename_(
      "Contrast" = "contrast",
      "Difference" = "estimate",
      "p" = "p.value"
    )
  names(contrasts) <- stringr::str_replace(names(contrasts), "lower.CL", "CI_lower")
  names(contrasts) <- stringr::str_replace(names(contrasts), "upper.CL", "CI_higher")
  names(contrasts) <- stringr::str_replace(names(contrasts), "asymp.LCL", "CI_lower")
  names(contrasts) <- stringr::str_replace(names(contrasts), "asymp.UCL", "CI_higher")
  names(contrasts) <- stringr::str_replace(names(contrasts), "t.ratio", "t")
  names(contrasts) <- stringr::str_replace(names(contrasts), "z.ratio", "z")

  return(contrasts)
}








#' Extract the dataframe used in a model.
#'
#' Extract the dataframe used in a model.
#'
#' @param fit A model.
#' @param ... Arguments passed to or from other methods.
#'
#' @examples
#' \dontrun{
#' library(tidyverse)
#' library(psycho)
#'
#' df <- mtcars %>%
#'   mutate(
#'     cyl = as.factor(cyl),
#'     gear = as.factor(gear)
#'   )
#'
#' fit <- lm(wt ~ mpg, data = df)
#' fit <- lm(wt ~ cyl, data = df)
#' fit <- lm(wt ~ mpg * cyl, data = df)
#' fit <- lm(wt ~ cyl * gear, data = df)
#' fit <- lmerTest::lmer(wt ~ mpg * gear + (1 | cyl), data = df)
#' fit <- rstanarm::stan_lmer(wt ~ mpg * gear + (1 | cyl), data = df)
#'
#' get_data(fit)
#' }
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#' @export
get_data <- function(fit, ...) {
  UseMethod("get_data")
}


#' @importFrom stats getCall
#' @importFrom utils data
#' @export
get_data.lm <- function(fit, ...) {
  tryCatch({
    data <- eval(getCall(fit)$data, environment(formula(fit)))
    return(data)
  })

  info <- get_info(fit)

  outcome <- info$outcome
  predictors <- info$predictors

  data <- as.data.frame(model.frame(fit))


  effects <- names(MuMIn::coeffs(fit))
  effects <- unique(unlist(stringr::str_split(effects, ":")))
  numerics <- predictors[predictors %in% effects]

  numerics <- numerics[!is.na(numerics)]
  if (length(unique(model.response(model.frame(fit)))) > 2) {
    numerics <- c(outcome, numerics)
  }


  data[!names(data) %in% numerics] <- lapply(data[!names(data) %in% numerics], as.factor)
  data[names(data) %in% numerics] <- lapply(data[names(data) %in% numerics], as.numeric)

  return(as.data.frame(data))
}

#' @export
get_data.merMod <- get_data.lm





#' @export
get_data.stanreg <- function(fit, ...) {
  data <- fit$data
  return(data)
}










#' Get formula of models.
#'
#' Get formula of models. Implemented for:
#' \itemize{
#'  \item{analyze.merModLmerTest}
#'  \item{analyze.glmerMod}
#'  \item{analyze.lm}
#'  \item{analyze.glm}
#'  \item{analyze.stanreg}
#'  }
#'
#' @param x Object.
#' @param ... Arguments passed to or from other methods.
#'
#'
#' @examples
#' library(psycho)
#' library(lme4)
#'
#' fit <- lme4::glmer(vs ~ wt + (1 | gear), data = mtcars, family = "binomial")
#' fit <- lm(hp ~ wt, data = mtcars)
#'
#' get_formula(fit)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
get_formula <- function(x, ...) {
  UseMethod("get_formula")
}


#' @export
get_formula.lmerModLmerTest <- function(x, ...) {
  return(x@call$formula)
}
#' @export
get_formula.glmerMod <- get_formula.lmerModLmerTest
#' @export
get_formula.lmerMod <- get_formula.lmerModLmerTest


#' @export
get_formula.lm <- function(x, ...) {
  return(stats::formula(x))
}
#' @export
get_formula.glm <- get_formula.lm



#' @export
get_formula.stanreg <- function(x, ...) {
  return(x$formula)
}










#' Get graph data.
#'
#' To be used with tidygraph::tbl_graph. See the documentation for your object's class:
#' \itemize{
#'  \item{\link[=get_graph.lavaan]{get_graph.lavaan}}
#'  \item{\link[=get_graph.fa]{get_graph.fa}}
#'  \item{\link[=get_graph.psychobject_correlation]{get_graph.psychobject_correlation}}
#'  }
#'
#' @param fit Object from which to extract the graph data.
#' @param ... Arguments passed to or from other methods.
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
get_graph <- function(fit, ...) {
  UseMethod("get_graph")
}


















#' Get graph data from lavaan or blavaan objects.
#'
#' Get graph data from lavaan or blavaan objects.
#'
#' @param fit lavaan object.
#' @param links Which links to include? A list including at least one of "Regression", "Loading" or "Correlation".
#' @param standardize Use standardized coefs.
#' @param threshold_Coef Omit all links with a Coefs below this value.
#' @param threshold_p Omit all links with a p value above this value.
#' @param threshold_MPE In case of a blavaan model, omit all links with a MPE value below this value.
#' @param digits Edges' labels rounding.
#' @param CI CI level.
#' @param labels_CI Add the CI in the edge label.
#' @param ... Arguments passed to or from other methods.
#'
#' @return A list containing nodes and edges data to be used by `tidygraph::tbl_graph()`.
#'
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#'
#' @export
get_graph.lavaan <- function(fit, links = c("Regression", "Correlation", "Loading"), standardize = FALSE, threshold_Coef = NULL, threshold_p = NULL, threshold_MPE = NULL, digits = 2, CI = "default", labels_CI = TRUE, ...) {
  # https://www.r-bloggers.com/ggplot2-sem-models-with-tidygraph-and-ggraph/


  if (labels_CI == TRUE) {
    if (CI != "default") {
      results <- analyze(fit, CI = CI, standardize = standardize)
    } else {
      results <- analyze(fit, standardize = standardize)
    }
  } else {
    results <- analyze(fit, standardize = standardize)
  }

  summary <- summary(results)
  CI <- results$values$CI

  # Check what type of model
  if (class(fit) %in% c("blavaan")) {
    summary$Coef <- summary$Median
    if (is.null(threshold_MPE)) {
      threshold_MPE <- -1
    }
    summary <- summary %>%
      filter_("MPE >= threshold_MPE")
  } else if (class(fit) %in% c("lavaan")) {
    if (is.null(threshold_p)) {
      threshold_p <- 1.1
    }
    summary <- summary %>%
      filter_("p <= threshold_p")
  } else {
    stop(paste("Error in UseMethod('plot_lavaan') : no applicable method for 'plot_lavaan' applied to an object of class", class(fit)))
  }

  # Deal with thresholds
  if (is.null(threshold_Coef)) {
    threshold_Coef <- min(abs(summary$Coef)) - 1
  }

  # Edge properties
  edges <- summary %>%
    mutate_("abs_coef" = "abs(Coef)") %>%
    filter_(
      "Type %in% c(links)",
      "From != To",
      "abs_coef >= threshold_Coef"
    ) %>%
    select(-one_of("abs_coef")) %>%
    rename_(
      "to" = "To",
      "from" = "From"
    )

  # Labels
  if (labels_CI == TRUE) {
    edges <- edges %>%
      mutate_("Label" = 'paste0(format_digit(Coef, digits),
              ", ", CI, "% CI [", format_digit(CI_lower, digits),
              ", ", format_digit(CI_higher, digits), "]")')
  } else {
    edges <- edges %>%
      mutate_("Label" = "format_digit(Coef, digits)")
  }
  edges <- edges %>%
    mutate_(
      "Label_Regression" = "ifelse(Type=='Regression', Label, '')",
      "Label_Correlation" = "ifelse(Type=='Correlation', Label, '')",
      "Label_Loading" = "ifelse(Type=='Loading', Label, '')"
    )
  edges <- edges[colSums(!is.na(edges)) > 0]

  # Identify latent variables for nodes
  latent_nodes <- edges %>%
    filter_('Type == "Loading"') %>%
    distinct_("to") %>%
    transmute_("Name" = "to", "Latent" = TRUE)

  nodes_list <- unique(c(edges$from, edges$to))

  # Node properties
  nodes <- summary %>%
    filter_(
      "From == To",
      "From %in% nodes_list"
    ) %>%
    mutate_("Name" = "From") %>%
    left_join(latent_nodes, by = "Name") %>%
    mutate_("Latent" = "if_else(is.na(Latent), FALSE, Latent)") %>%
    select(one_of(c("Name", "Latent")))

  return(list(nodes = nodes, edges = edges))
}





#' Get graph data from factor analysis.
#'
#' Get graph data from fa objects.
#'
#' @param fit psych::fa object.
#' @param threshold_Coef Omit all links with a Coefs below this value.
#' @param digits Edges' labels rounding.
#' @param ... Arguments passed to or from other methods.
#'
#' @return A list containing nodes and edges data to be used by `tidygraph::tbl_graph()`.
#'
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#'
#' @export
get_graph.fa <- function(fit, threshold_Coef = NULL, digits = 2, ...) {
  edges <- summary(analyze(fit)) %>%
    gather("To", "Coef", -one_of("N", "Item", "Label")) %>%
    rename_("From" = "Item") %>%
    mutate_("Label" = "format_digit(Coef, digits)") %>%
    select(one_of("From", "To", "Coef", "Label"), everything()) %>%
    filter()

  # Deal with thresholds
  if (is.null(threshold_Coef)) {
    threshold_Coef <- min(abs(edges$Coef)) - 1
  }

  edges <- edges %>%
    filter_("Coef > threshold_Coef")

  nodes <- data.frame("Name" = c(edges$From, edges$To)) %>%
    distinct_("Name")

  return(list(nodes = nodes, edges = edges))
}




#' Get graph data from correlation.
#'
#' Get graph data from correlation.
#'
#' @param fit Object from psycho::correlation.
#' @param ... Arguments passed to or from other methods.
#'
#' @return A list containing nodes and edges data to be used by `igraph::graph_from_data_frame()`.
#'
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#'
#' @export
get_graph.psychobject_correlation <- function(fit, ...) {
  vars <- row.names(fit$values$r)

  r <- fit$values$r %>%
    as.data.frame() %>%
    tibble::rownames_to_column("from") %>%
    tidyr::gather("to", "r", vars)

  if ("p" %in% names(fit$values)) {
    r <- r %>%
      full_join(
        fit$values$p %>%
          as.data.frame() %>%
          tibble::rownames_to_column("from") %>%
          tidyr::gather("to", "p", vars),
        by = c("from", "to")
      )
  }

  r <- filter_(r, "!from == to")
  return(r)
}













#' Get information about objects.
#'
#' Get information about models.
#'
#'
#' @param x object.
#' @param ... Arguments passed to or from other methods.
#'
#' @return output
#'
#' @examples
#' library(psycho)
#' library(lme4)
#'
#' fit <- lme4::glmer(vs ~ wt + (1 | gear), data = mtcars, family = "binomial")
#'
#' info <- get_info(fit)
#' info
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
get_info <- function(x, ...) {
  UseMethod("get_info")
}










#' Get information about models.
#'
#' Get information about models.
#'
#' @param x object.
#' @param ... Arguments passed to or from other methods.
#'
#' @return output
#'
#' @examples
#' library(psycho)
#' library(lme4)
#'
#' fit <- lme4::glmer(vs ~ wt + (1 | gear), data = mtcars, family = "binomial")
#'
#' info <- get_info(fit)
#' info
#'
#' #
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
get_info.lmerModLmerTest <- function(x, ...) {
  fit <- x

  info <- tryCatch({

    # Get formula
    formula <- get_formula(fit)
    # Get variables
    predictors <- all.vars(formula)
    outcome <- predictors[[1]]
    predictors <- tail(predictors, -1)
    random <- names(lme4::ranef(fit))[names(lme4::ranef(fit)) %in% predictors]
    predictors <- predictors[!predictors %in% random]

    return(list(
      formula = formula,
      predictors = predictors,
      outcome = outcome,
      random = random
    ))
  }, error = function(e) {

    # Get formula
    formula <- get_formula(fit)
    # Get variables
    predictors <- NA
    outcome <- "Y"
    random <- NA

    return(list(
      formula = formula,
      predictors = predictors,
      outcome = outcome,
      random = random
    ))
  })

  return(info)
}
#' @export
get_info.glmerMod <- get_info.lmerModLmerTest
#' @export
get_info.lmerMod <- get_info.lmerModLmerTest



#' Get information about models.
#'
#' Get information about models.
#'
#' @param x object.
#' @param ... Arguments passed to or from other methods.
#'
#' @return output
#'
#' @examples
#' library(psycho)
#' library(lme4)
#'
#' fit <- lm(vs ~ wt, data = mtcars, family = "binomial")
#'
#' info <- get_info(fit)
#' info
#'
#' #
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
get_info.lm <- function(x, ...) {
  fit <- x

  info <- tryCatch({

    # Get formula
    formula <- get_formula(fit)
    # Get variables
    predictors <- all.vars(formula)
    outcome <- predictors[[1]]
    predictors <- tail(predictors, -1)

    return(list(
      formula = formula,
      predictors = predictors,
      outcome = outcome
    ))
  }, error = function(e) {

    # Get formula
    formula <- get_formula(fit)
    # Get variables
    predictors <- NA
    outcome <- "Y"
    random <- NA

    return(list(
      formula = formula,
      predictors = predictors,
      outcome = outcome
    ))
  })

  return(info)
}

#' @export
get_info.stanreg <- get_info.lm
#' @export
get_info.lm <- get_info.lm
#' @export
get_info.glm <- get_info.lm












#' Compute estimated means from models.
#'
#' Compute estimated means of factor levels based on a fitted model.
#'
#' @param fit A model (lm, lme4 or rstanarm).
#' @param formula A character vector (formula like format, i.e., including
#' interactions or nesting terms) specifying the names of the predictors over which EMMs are desired.
#' @param CI Determine the confidence or credible interval bounds.
#' @param ... Arguments passed to or from other methods. For instance, transform="response".
#'
#'
#' @return Estimated means (or median of means for Bayesian models)
#'
#'
#' @examples
#' \dontrun{
#' library(psycho)
#' require(lmerTest)
#' require(rstanarm)
#'
#'
#' fit <- glm(Sex ~ Birth_Season, data = affective, family = "binomial")
#' get_means(fit)
#'
#' fit <- lmerTest::lmer(Adjusting ~ Birth_Season * Salary + (1 | Salary), data = affective)
#' get_means(fit, formula = "Birth_Season")
#'
#' fit <- rstanarm::stan_glm(Adjusting ~ Birth_Season, data = affective)
#' get_means(fit, formula = "Birth_Season")
#' }
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
get_means <- function(fit, formula = NULL, CI = 90, ...) {
  UseMethod("get_means")
}


#' @method get_means stanreg
#' @export
get_means.stanreg <- function(fit, formula = NULL, CI = 90, ...) {
  .get_means_bayes(fit, formula, CI, ...)
}

#' @method get_means lm
#' @export
get_means.lm <- function(fit, formula = NULL, CI = 95, ...) {
  .get_means_freq(fit, formula, CI, ...)
}

#' @method get_means glm
#' @export
get_means.glm <- function(fit, formula = NULL, CI = 95, ...) {
  .get_means_freq(fit, formula, CI, ...)
}

#' @method get_means lmerModLmerTest
#' @export
get_means.lmerModLmerTest <- function(fit, formula = NULL, CI = 95, ...) {
  .get_means_freq(fit, formula, CI, ...)
}

#' @method get_means glmerMod
#' @export
get_means.glmerMod <- function(fit, formula = NULL, CI = 95, ...) {
  .get_means_freq(fit, formula, CI, ...)
}

#' @method get_means lmerMod
#' @export
get_means.lmerMod <- function(fit, formula = NULL, CI = 95, ...) {
  .get_means_freq(fit, formula, CI, ...)
}




#' @import dplyr
#' @importFrom emmeans emmeans
#' @importFrom stats confint mad
#' @keywords internal
.get_means_bayes <- function(fit, formula = NULL, CI = 90, ...) {
  if (is.null(formula)) {
    formula <- paste(get_info(fit)$predictors, collapse = " * ")
  }

  if (is.character(formula)) {
    formula <- as.formula(paste0("~ ", formula))
  }

  # Means ---------------------------------------------------------------
  means_posterior <- fit %>%
    emmeans::emmeans(formula) %>%
    emmeans::as.mcmc.emmGrid() %>%
    as.matrix() %>%
    as.data.frame()

  means <- data.frame()

  for (name in names(means_posterior)) {
    var <- means_posterior[[name]]

    CI_values <- HDI(var, prob = CI / 100)
    CI_values <- c(CI_values$values$HDImin, CI_values$values$HDImax)

    var <- data.frame(
      Level = name,
      Median = median(var),
      MAD = mad(var),
      CI_lower = CI_values[seq(1, length(CI_values), 2)],
      CI_higher = CI_values[seq(2, length(CI_values), 2)]
    )

    means <- rbind(means, var)
  }

  return(means)
}




#' @import dplyr
#' @importFrom emmeans emmeans
#' @importFrom stats confint
#' @keywords internal
.get_means_freq <- function(fit, formula = NULL, CI = 95, ...) {
  if (is.null(formula)) {
    formula <- paste(get_info(fit)$predictors, collapse = " * ")
  }

  if (is.character(formula)) {
    formula <- as.formula(paste0("~ ", formula))
  }

  # Means ---------------------------------------------------------------
  means <- fit %>%
    emmeans::emmeans(formula, ...) %>%
    confint(CI / 100) %>%
    as.data.frame()

  names(means) <- stringr::str_replace(names(means), "emmean", "Mean")
  names(means) <- stringr::str_replace(names(means), "lower.CL", "CI_lower")
  names(means) <- stringr::str_replace(names(means), "upper.CL", "CI_higher")
  names(means) <- stringr::str_replace(names(means), "asymp.LCL", "CI_lower")
  names(means) <- stringr::str_replace(names(means), "asymp.UCL", "CI_higher")

  return(means)
}









#' Compute predicted values of lm models.
#'
#' Compute predicted from a lm model.
#'
#' @param fit An lm model.
#' @param newdata A data frame in which to look for variables with which to predict. If omitted, the model matrix is used. If "model", the model's data is used.
#' @param prob Probability of confidence intervals (0.9 (default) will compute 2.5-97.5\% CI). Can also be a list of probs (e.g., c(0.90, 0.95)).
#' @param odds_to_probs Transform log odds ratios in logistic models to probabilies.
#' @param ... Arguments passed to or from other methods.
#'
#'
#' @return dataframe with predicted values.
#'
#'
#' @examples
#' \dontrun{
#' library(psycho)
#' library(ggplot2)
#'
#' fit <- glm(Sex ~ Adjusting, data = affective, family = "binomial")
#'
#' refgrid <- psycho::refdata(affective, "Adjusting")
#' predicted <- get_predicted(fit, newdata = refgrid)
#'
#' ggplot(predicted, aes(x = Adjusting, y = Sex_Predicted)) +
#'   geom_line() +
#'   geom_ribbon(aes(
#'     ymin = Sex_CI_2.5,
#'     ymax = Sex_CI_97.5
#'   ),
#'   alpha = 0.1
#'   )
#' }
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @importFrom dplyr bind_cols
#' @importFrom tibble rownames_to_column
#' @export
get_predicted.glm <- function(fit, newdata = "model", prob = 0.95, odds_to_probs = TRUE, ...) {


  # Extract names
  info <- get_info(fit)
  outcome <- info$outcome
  predictors <- info$predictors

  # Set newdata if refgrid
  if ("emmGrid" %in% class(newdata)) {
    newdata <- newdata@grid
    newdata[".wgt."] <- NULL
  }

  # Set newdata to actual data
  original_data <- FALSE
  if (!is.null(newdata)) {
    if (is.character(newdata)) {
      if (newdata == "model") {
        original_data <- TRUE
        newdata <- fit$data[predictors]
        newdata <- na.omit(fit$data[predictors])
      }
    }
  }


  # Compute ----------------------------------------------------------

  # Predicted Y
  prediction <- as.data.frame(predict(fit, newdata = newdata, type = "link", se.fit = TRUE))
  SE <- as.data.frame(prediction$se.fit)
  pred_y <- as.data.frame(prediction$fit)
  names(pred_y) <- paste0(outcome, "_Predicted")

  # Credible Interval
  for (CI in c(prob)) {
    pred_y_interval <- data.frame(
      lwr = prediction$fit - (qnorm(CI) * SE),
      upr = prediction$fit + (qnorm(CI) * SE)
    )
    names(pred_y_interval) <- paste(outcome, "CI", c((1 - CI) / 2 * 100, 100 - ((1 - CI) / 2 * 100)), sep = "_")
    pred_y <- cbind(pred_y, pred_y_interval)
  }


  # Transform odds to probs ----------------------------------------------------------

  if (family(fit)$family == "binomial" & family(fit)$link == "logit") {
    if (odds_to_probs == TRUE) {
      pred_y <- odds_to_probs(pred_y)
    }
  }


  # Add predictors ----------------------------------------------------------


  if (!is.null(newdata)) {
    if (original_data) {
      predicted <- newdata %>%
        tibble::rownames_to_column() %>%
        dplyr::bind_cols(pred_y) %>%
        dplyr::right_join(fit$data[!names(fit$data) %in% predictors] %>%
          tibble::rownames_to_column(),
        by = "rowname"
        ) %>%
        select_("-rowname")
    } else {
      predicted <- dplyr::bind_cols(newdata, pred_y)
    }
  } else {
    predicted <- dplyr::bind_cols(as.data.frame(model.matrix(fit)), pred_y)
  }


  return(predicted)
}









#' Compute predicted values of lm models.
#'
#' Compute predicted from a lm model.
#'
#' @param fit An lm model.
#' @param newdata A data frame in which to look for variables with which to predict. If omitted, the model matrix is used. If "model", the model's data is used.
#' @param prob Probability of confidence intervals (0.95 (default) will compute 2.5-97.5\% CI). Can also be a list of probs (e.g., c(0.90, 0.95)).
#' @param ... Arguments passed to or from other methods.
#'
#'
#' @return dataframe with predicted values.
#'
#'
#' @examples
#' \dontrun{
#' library(psycho)
#' library(ggplot2)
#'
#' fit <- lm(Tolerating ~ Adjusting, data = affective)
#'
#' refgrid <- psycho::refdata(affective, "Adjusting")
#' predicted <- get_predicted(fit, newdata = refgrid)
#'
#' ggplot(predicted, aes(x = Adjusting, y = Tolerating_Predicted)) +
#'   geom_line() +
#'   geom_ribbon(aes(
#'     ymin = Tolerating_CI_2.5,
#'     ymax = Tolerating_CI_97.5
#'   ),
#'   alpha = 0.1
#'   )
#' }
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @importFrom dplyr bind_cols
#' @importFrom tibble rownames_to_column
#' @export
get_predicted.lm <- function(fit, newdata = "model", prob = 0.95, ...) {


  # Extract names
  info <- get_info(fit)
  outcome <- info$outcome
  predictors <- info$predictors

  # Set newdata if refgrid
  if ("emmGrid" %in% class(newdata)) {
    newdata <- newdata@grid
    newdata[".wgt."] <- NULL
  }

  # Set newdata to actual data
  original_data <- FALSE
  if (!is.null(newdata)) {
    if (is.character(newdata)) {
      if (newdata == "model") {
        original_data <- TRUE
        newdata <- as.data.frame(fit$model[predictors])
        newdata <- na.omit(fit$model[predictors])
      }
    }
  }


  # Compute ----------------------------------------------------------

  # Predicted Y
  pred_y <- as.data.frame(predict(fit, newdata))
  names(pred_y) <- paste0(outcome, "_Predicted")

  # Credible Interval
  for (CI in c(prob)) {
    pred_y_interval <- as.data.frame(predict(fit, newdata, interval = "confidence", level = CI)[, -1])
    names(pred_y_interval) <- paste(outcome, "CI", c((1 - CI) / 2 * 100, 100 - ((1 - CI) / 2 * 100)), sep = "_")
    pred_y <- cbind(pred_y, pred_y_interval)
  }



  # Add predictors ----------------------------------------------------------
  if (!is.null(newdata)) {
    if (original_data) {
      predicted <- newdata %>%
        tibble::rownames_to_column() %>%
        dplyr::bind_cols(pred_y) %>%
        dplyr::right_join(fit$model[!names(fit$model) %in% predictors] %>%
          tibble::rownames_to_column(),
        by = "rowname"
        ) %>%
        select_("-rowname")
    } else {
      predicted <- dplyr::bind_cols(newdata, pred_y)
    }
  } else {
    predicted <- dplyr::bind_cols(as.data.frame(model.matrix(fit)), pred_y)
  }


  return(predicted)
}









#' Compute predicted values of lm models.
#'
#' Compute predicted from a lm model.
#'
#' @param fit An lm model.
#' @param newdata A data frame in which to look for variables with which to predict. If omitted, the model matrix is used. If "model", the model's data is used.
#' @param prob Probability of confidence intervals (0.95 will compute 2.5-97.5\% CI). Can also be a list of probs (e.g., c(0.90, 0.95)). Default to NULL as it takes a very long time to compute (see \link[lme4]{bootMer}).
#' @param odds_to_probs Transform log odds ratios in logistic models to probabilies.
#' @param iter An integer indicating the number of iterations for bootstrapping (when prob is not null).
#' @param seed An optional seed to use.
#' @param re.form Formula for random effects to condition on. If NULL, include all random effects; if NA or ~0, include no random effects (see \link[lme4]{predict.merMod}). If "default", then will ne NULL if the random are present in the data, and NA if not.
#' @param use.u logical, indicating whether the spherical random effects should be simulated / bootstrapped as well. If TRUE, they are not changed, and all inference is conditional on these values. If FALSE, new normal deviates are drawn (see\link[lme4]{bootMer}).
#' @param ... Arguments passed to or from other methods.
#'
#'
#' @return dataframe with predicted values.
#'
#'
#' @examples
#' \dontrun{
#' library(psycho)
#' library(ggplot2)
#'
#' fit <- lmerTest::lmer(Tolerating ~ Adjusting + (1 | Salary), data = affective)
#'
#' refgrid <- psycho::refdata(affective, "Adjusting")
#' predicted <- get_predicted(fit, newdata = refgrid)
#'
#' ggplot(predicted, aes(x = Adjusting, y = Tolerating_Predicted)) +
#'   geom_line()
#'
#' predicted <- get_predicted(fit, newdata = refgrid, prob = 0.95, iter = 100) # Takes a long time
#'
#' ggplot(predicted, aes(x = Adjusting, y = Tolerating_Predicted)) +
#'   geom_line() +
#'   geom_ribbon(aes(
#'     ymin = Tolerating_CI_2.5,
#'     ymax = Tolerating_CI_97.5
#'   ),
#'   alpha = 0.1
#'   )
#'
#'
#'
#' fit <- lme4::glmer(Sex ~ Adjusting + (1 | Salary), data = affective, family = "binomial")
#'
#' refgrid <- psycho::refdata(affective, "Adjusting")
#' predicted <- get_predicted(fit, newdata = refgrid)
#'
#' ggplot(predicted, aes(x = Adjusting, y = Sex_Predicted)) +
#'   geom_line()
#'
#' predicted <- get_predicted(fit, newdata = refgrid, prob = 0.95, iter = 100) # Takes a long time
#'
#' ggplot(predicted, aes(x = Adjusting, y = Sex_Predicted)) +
#'   geom_line() +
#'   geom_ribbon(aes(
#'     ymin = Sex_CI_2.5,
#'     ymax = Sex_CI_97.5
#'   ),
#'   alpha = 0.1
#'   )
#' }
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @importFrom dplyr bind_cols
#' @importFrom tibble rownames_to_column
#' @export
get_predicted.merMod <- function(fit, newdata = "model", prob = NULL, odds_to_probs = TRUE, iter = 100, seed = NULL, re.form = "default", use.u = FALSE, ...) {


  # Extract names
  info <- get_info(fit)
  outcome <- info$outcome
  predictors <- info$predictors

  # Set newdata if refgrid
  if ("emmGrid" %in% class(newdata)) {
    newdata <- newdata@grid
    newdata[".wgt."] <- NULL
  }

  # Set newdata to actual data
  original_data <- FALSE
  if (!is.null(newdata)) {
    if (is.character(newdata)) {
      if (newdata == "model") {
        original_data <- TRUE
        newdata <- na.omit(fit@frame)
      }
    }
  }


  # Deal with random
  if (!is.na(re.form)) {
    if (re.form == "default") {
      # Check if all predictors are in variables
      if (all(get_info(fit)$predictors %in% names(newdata))) {
        re.form <- NULL
      } else {
        re.form <- NA
      }
    }
  }



  # Compute ----------------------------------------------------------

  pred_y <- as.data.frame(predict(fit, newdata = newdata, re.form = re.form))
  names(pred_y) <- paste0(outcome, "_Predicted")

  if (!is.null(prob)) {
    predFun <- function(fit) {
      predict(fit, newdata, newdata = newdata, re.form = re.form)
    }
    predMat <- lme4::bootMer(fit, nsim = iter, FUN = predFun, use.u = use.u, seed = seed)$t

    for (CI in c(prob)) {
      pred_y_interval <- as.data.frame(t(apply(predMat, 2, quantile, c((1 - CI) / 2, CI + (1 - CI) / 2), na.rm = TRUE)))
      names(pred_y_interval) <- paste(outcome, "CI", c((1 - CI) / 2 * 100, 100 - ((1 - CI) / 2 * 100)), sep = "_")
      pred_y <- cbind(pred_y, pred_y_interval)
    }
  }


  # Transform odds to probs ----------------------------------------------------------

  if (family(fit)$family == "binomial" & family(fit)$link == "logit") {
    if (odds_to_probs == TRUE) {
      pred_y <- odds_to_probs(pred_y)
    }
  }


  # Add predictors ----------------------------------------------------------


  if (!is.null(newdata)) {
    if (original_data) {
      predicted <- newdata %>%
        tibble::rownames_to_column() %>%
        dplyr::bind_cols(pred_y) %>%
        dplyr::right_join(fit@frame[!names(fit@frame) %in% names(newdata)] %>%
          tibble::rownames_to_column(),
        by = "rowname"
        ) %>%
        select_("-rowname")
    } else {
      predicted <- dplyr::bind_cols(newdata, pred_y)
    }
  } else {
    predicted <- dplyr::bind_cols(as.data.frame(model.matrix(fit)), pred_y)
  }


  return(predicted)
}






#' Compute predicted values from models.
#'
#' Compute predicted values from models. See the
#' documentation for your model's class:
#' \itemize{
#'  \item{\link[=get_predicted.stanreg]{get_predicted.stanreg}}
#'  \item{\link[=get_predicted.merMod]{get_predicted.merMod}}
#'  \item{\link[=get_predicted.lm]{get_predicted.lm}}
#'  \item{\link[=get_predicted.glm]{get_predicted.glm}}
#'  }
#'
#' @param fit Model.
#' @param ... Arguments passed to or from other methods.
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
get_predicted <- function(fit, ...) {
  UseMethod("get_predicted")
}








#' Compute predicted values of stanreg models.
#'
#' Compute predicted from a stanreg model.
#'
#' @param fit A stanreg model.
#' @param newdata A data frame in which to look for variables with which to predict. If omitted, the model matrix is used. If "model", the model's data is used.
#' @param prob Probability of credible intervals (0.9 (default) will compute 5-95\% CI). Can also be a list of probs (e.g., c(0.90, 0.95)).
#' @param odds_to_probs Transform log odds ratios in logistic models to probabilies.
#' @param keep_iterations Keep all prediction iterations.
#' @param draws An integer indicating the number of draws to return. The default and maximum number of draws is the size of the posterior sample.
#' @param posterior_predict Posterior draws of the outcome instead of the link function (i.e., the regression "line").
#' @param seed An optional seed to use.
#' @param transform If posterior_predict is False, should the linear predictor be transformed using the inverse-link function? The default is FALSE, in which case the untransformed linear predictor is returned.
#' @param re.form If object contains group-level parameters, a formula indicating which group-level parameters to condition on when making predictions. re.form is specified in the same form as for predict.merMod. NULL indicates that all estimated group-level parameters are conditioned on. To refrain from conditioning on any group-level parameters, specify NA or ~0. The newdata argument may include new levels of the grouping factors that were specified when the model was estimated, in which case the resulting posterior predictions marginalize over the relevant variables (see \link[rstanarm]{posterior_predict.stanreg}). If "default", then will ne NULL if the random are present in the data, and NA if not.
#' @param ... Arguments passed to or from other methods.
#'
#'
#' @return dataframe with predicted values.
#'
#'
#' @examples
#' \dontrun{
#' library(psycho)
#' library(ggplot2)
#' require(rstanarm)
#'
#' fit <- rstanarm::stan_glm(Tolerating ~ Adjusting, data = affective)
#'
#' refgrid <- psycho::refdata(affective, "Adjusting")
#' predicted <- get_predicted(fit, newdata = refgrid)
#'
#' ggplot(predicted, aes(x = Adjusting, y = Tolerating_Median)) +
#'   geom_line() +
#'   geom_ribbon(aes(
#'     ymin = Tolerating_CI_5,
#'     ymax = Tolerating_CI_95
#'   ),
#'   alpha = 0.1
#'   )
#'
#' fit <- rstanarm::stan_glm(Sex ~ Adjusting, data = affective, family = "binomial")
#'
#' refgrid <- psycho::refdata(affective, "Adjusting")
#' predicted <- get_predicted(fit, newdata = refgrid)
#'
#' ggplot(predicted, aes(x = Adjusting, y = Sex_Median)) +
#'   geom_line() +
#'   geom_ribbon(aes(
#'     ymin = Sex_CI_5,
#'     ymax = Sex_CI_95
#'   ),
#'   alpha = 0.1
#'   )
#' }
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @importFrom stats median family model.matrix
#' @importFrom dplyr bind_cols
#' @importFrom tibble rownames_to_column
#' @export
get_predicted.stanreg <- function(fit, newdata = "model", prob = 0.9, odds_to_probs = TRUE, keep_iterations = FALSE, draws = NULL, posterior_predict = FALSE, seed = NULL, transform = FALSE, re.form = "default", ...) {

  # Extract names
  predictors <- all.vars(as.formula(fit$formula))
  outcome <- predictors[[1]]
  predictors <- tail(predictors, -1)

  # Set newdata if refgrid
  if ("emmGrid" %in% class(newdata)) {
    newdata <- newdata@grid
    newdata[".wgt."] <- NULL
  }

  # Set newdata to actual data
  original_data <- FALSE
  if (!is.null(newdata)) {
    if (is.character(newdata)) {
      if (newdata == "model") {
        original_data <- TRUE
        newdata <- fit$data[predictors]
        newdata <- na.omit(fit$data[predictors])
      }
    }
  }

  # Deal with potential random
  if (!is.na(re.form)) {
    if (re.form == "default") {
      if (is.mixed(fit)) {
        # Check if all predictors are in variables
        if (all(get_info(fit)$predictors %in% names(newdata))) {
          re.form <- NULL
        } else {
          re.form <- NA
        }
      }
    }
  }

  # Generate draws -------------------------------------------------------
  if (posterior_predict == FALSE) {
    posterior <- rstanarm::posterior_linpred(fit, newdata = newdata, re.form = re.form, seed = seed, draws = draws, transform = transform)
  } else {
    posterior <- rstanarm::posterior_predict(fit, newdata = newdata, re.form = re.form, seed = seed, draws = draws)
  }

  # Format -------------------------------------------------------

  # Predicted Y
  pred_y <- as.data.frame(apply(posterior, 2, median))
  names(pred_y) <- paste0(outcome, "_Median")

  # Credible Interval
  for (CI in c(prob)) {
    pred_y_interval <- HDI(posterior, prob = CI)
    names(pred_y_interval) <- paste(outcome, "CI", c((1 - CI) / 2 * 100, 100 - ((1 - CI) / 2 * 100)), sep = "_")
    pred_y <- cbind(pred_y, pred_y_interval)
  }


  # Keep iterations ---------------------------------------------------------

  if (keep_iterations == TRUE) {
    iterations <- as.data.frame(t(posterior))
    names(iterations) <- paste0("iter_", seq_len(length(names(iterations))))
    pred_y <- cbind(pred_y, iterations)
  }

  # Transform odds to probs ----------------------------------------------------------

  if (family(fit)$family == "binomial" & family(fit)$link == "logit") {
    if (odds_to_probs == TRUE) {
      pred_y <- odds_to_probs(pred_y)
    }
  }


  # Add predictors ----------------------------------------------------------


  if (!is.null(newdata)) {
    if (original_data) {
      predicted <- newdata %>%
        tibble::rownames_to_column() %>%
        dplyr::bind_cols(pred_y) %>%
        dplyr::right_join(fit$data[!names(fit$data) %in% predictors] %>%
          tibble::rownames_to_column(),
        by = "rowname"
        ) %>%
        select_("-rowname")
    } else {
      predicted <- dplyr::bind_cols(newdata, pred_y)
    }
  } else {
    predicted <- dplyr::bind_cols(as.data.frame(model.matrix(fit)), pred_y)
  }


  return(predicted)
}








#' Get Indices of Explanatory Power.
#'
#' See the documentation for your object's class:
#' \itemize{
#' \item{\link[=get_R2.lm]{get_R2.lm}}
#' \item{\link[=get_R2.glm]{get_R2.glm}}
#' \item{\link[=get_R2.stanreg]{get_R2.stanreg}}
#'  }
#'
#' @param fit Object.
#' @param ... Arguments passed to or from other methods.
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
get_R2 <- function(fit, ...) {
  UseMethod("get_R2")
}


#' R2 and adjusted R2 for Linear Models.
#'
#' R2 and adjusted R2 for Linear Models.
#'
#' @param fit A linear model.
#' @param ... Arguments passed to or from other methods.
#'
#' @examples
#' \dontrun{
#' library(psycho)
#'
#' fit <- lm(Tolerating ~ Adjusting, data = psycho::affective)
#'
#' get_R2(fit)
#' }
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#' @export
get_R2.lm <- function(fit, ...) {
  R2 <- summary(fit)$r.squared
  R2.adj <- summary(fit)$adj.r.squared

  out <- list(R2 = R2, R2.adj = R2.adj)
  return(out)
}



#' Pseudo-R-squared for Logistic Models.
#'
#' Pseudo-R-squared for Logistic Models.
#'
#' @param fit A logistic model.
#' @param method Can be \link[=R2_nakagawa]{"nakagawa"} or \link[=R2_tjur]{"tjur"}.
#' @param ... Arguments passed to or from other methods.
#'
#' @examples
#' \dontrun{
#' library(psycho)
#'
#' fit <- glm(vs ~ wt, data = mtcars, family = "binomial")
#' fit <- glm(Sex ~ Adjusting, data = psycho::affective, family = "binomial")
#'
#' get_R2(fit)
#' }
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
get_R2.glm <- function(fit, method = "nakagawa", ...) {
  if (method == "nakagawa") {
    R2 <- as.numeric(R2_nakagawa(fit)$R2m)
  } else if (method == "tjur") {
    R2 <- R2_tjur(fit)
  } else {
    stop("Method must be 'nakagawa' or 'tjur'.")
  }
  return(R2)
}




#' R2 or Bayesian Models.
#'
#' Computes R2 and \link[=R2_LOO_Adjusted]{LOO-adjusted R2}.
#'
#' @param fit A stanreg model.
#' @param silent If R2 not available, throw warning.
#' @param ... Arguments passed to or from other methods.
#'
#' @examples
#' \dontrun{
#' library(psycho)
#' library(rstanarm)
#'
#' fit <- rstanarm::stan_glm(Adjusting ~ Tolerating, data = psycho::affective)
#'
#' get_R2(fit)
#' }
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @seealso \link[=bayes_R2.stanreg]{"bayes_R2.stanreg"}
#'
#' @export
get_R2.stanreg <- function(fit, silent = FALSE, ...) {
  tryCatch({
    R2 <- rstanarm::bayes_R2(fit)
  }, error = function(e) {
    R2 <- "NA"
  })

  if (!is.numeric(R2)) {
    if (silent) {
      return(R2)
    } else {
      stop("Couldn't compute R2 for this model.")
    }
  }

  out <- list(
    R2_median = median(R2),
    R2_MAD = mad(R2),
    R2_posterior = R2
  )

  if (fit$family$family == "gaussian") {
    out$R2.adj <- R2_LOO_Adjusted(fit)
  } else {
    out$R2.adj <- NA
  }

  return(out)
}



#' R2 and adjusted R2 for GLMMs.
#'
#' R2 and adjusted R2 for GLMMs.
#'
#' @param fit A GLMM.
#' @param ... Arguments passed to or from other methods.
#'
#' @examples
#' \dontrun{
#' library(psycho)
#'
#' fit <- lmerTest::lmer(Tolerating ~ Adjusting + (1 | Sex),
#'   data = psycho::affective
#' )
#' fit <- lme4::glmer(Sex ~ Adjusting + (1 | Salary),
#'   data = na.omit(psycho::affective), family = "binomial"
#' )
#'
#' get_R2(fit)
#' }
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#' @export
get_R2.merMod <- function(fit, ...) {
  out <- suppressMessages(R2_nakagawa(fit))
  return(out)
}





#' Pseudo-R-squared for Generalized Mixed-Effect models.
#'
#' For mixed-effects models, R² can be categorized into two types. Marginal R_GLMM² represents the variance explained by fixed factors, and Conditional R_GLMM² is interpreted as variance explained by both fixed and random factors (i.e. the entire model). IMPORTANT: Looking for help to reimplement this method.
#'
#' @param fit A mixed model.
#'
#' @examples
#' \dontrun{
#' library(psycho)
#'
#' fit <- lmerTest::lmer(Sepal.Length ~ Sepal.Width + (1 | Species), data = iris)
#'
#' R2_nakagawa(fit)
#' }
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @references
#' Nakagawa, S., Johnson, P. C., & Schielzeth, H. (2017). The coefficient of determination R2 and intra-class correlation coefficient from generalized linear mixed-effects models revisited and expanded. Journal of the Royal Society Interface, 14(134), 20170213.
#' Nakagawa, S., & Schielzeth, H. (2013). A general and simple method for obtaining R2 from generalized linear mixed-effects models. Methods in Ecology and Evolution, 4(2), 133-142.
#'
#' @export
R2_nakagawa <- function(fit) {
  out <- MuMIn::r.squaredGLMM(fit)
  out <- list(
    R2m = as.numeric(out[1]),
    R2c = as.numeric(out[2])
  )
  return(out)
}



#' Compute LOO-adjusted R2.
#'
#' Compute LOO-adjusted R2.
#'
#' @param fit A stanreg model.
#'
#' @examples
#' \dontrun{
#' library(psycho)
#' library(rstanarm)
#'
#' data <- attitude
#' fit <- rstanarm::stan_glm(rating ~ advance + privileges, data = data)
#'
#' R2_LOO_Adjusted(fit)
#' }
#'
#' @author \href{https://github.com/strengejacke}{Daniel Luedecke}
#'
#' @import rstantools
#'
#' @export
R2_LOO_Adjusted <- function(fit) {
  predictors <- all.vars(as.formula(fit$formula))
  y <- fit$data[[predictors[[1]]]]
  ypred <- rstantools::posterior_linpred(fit)
  ll <- rstantools::log_lik(fit)

  nsamples <- 0
  nchains <- length(fit$stanfit@stan_args)
  for (chain in fit$stanfit@stan_args) {
    nsamples <- nsamples + (chain$iter - chain$warmup)
  }


  r_eff <- loo::relative_eff(exp(ll),
    chain_id = rep(1:nchains, each = nsamples / nchains)
  )

  psis_object <- loo::psis(log_ratios = -ll, r_eff = r_eff)
  ypredloo <- loo::E_loo(ypred, psis_object, log_ratios = -ll)$value
  if (length(ypredloo) != length(y)) {
    warning("Something went wrong in the Loo-adjusted R2 computation.")
    return(NA)
  }
  eloo <- ypredloo - y

  adj_r_squared <- 1 - stats::var(eloo) / stats::var(y)
  return(adj_r_squared)
}




#' Tjur's (2009) coefficient of determination.
#'
#' Computes Tjur's (2009) coefficient of determination.
#'
#' @param fit Logistic Model.
#'
#' @examples
#' library(psycho)
#' library(lme4)
#'
#' fit <- lme4::glmer(vs ~ wt + (1 | gear), data = mtcars, family = "binomial")
#' R2_tjur(fit)
#' @author \href{https://github.com/strengejacke}{Daniel Lüdecke}
#'
#' @import dplyr
#' @importFrom stats predict residuals
#' @importFrom lme4 getME
#'
#' @references Tjur, T. (2009). Coefficients of determination in logistic regression models—A new proposal: The coefficient of discrimination. The American Statistician, 63(4), 366-372.
#'
#' @export
R2_tjur <- function(fit) {
  # check for valid object class
  if (!inherits(fit, c("glmerMod", "glm"))) {
    stop("`x` must be an object of class `glm` or `glmerMod`.", call. = F)
  }

  # mixed models (lme4)
  if (inherits(fit, "glmerMod")) {
    # check for package availability
    y <- lme4::getME(fit, "y")
    pred <- stats::predict(fit, type = "response", re.form = NULL)
  } else {
    y <- fit$y
    pred <- stats::predict.glm(fit, type = "response")
  }

  # delete pred for cases with missing residuals
  if (anyNA(stats::residuals(fit))) pred <- pred[!is.na(stats::residuals(fit))]

  categories <- unique(y)
  m1 <- mean(pred[which(y == categories[1])], na.rm = T)
  m2 <- mean(pred[which(y == categories[2])], na.rm = T)

  D <- abs(m2 - m1)
  names(D) <- "Tjur's D"

  return(D)
}








#' Golden Ratio.
#'
#' Returns the golden ratio (1.618034...).
#'
#' @param x A number to be multiplied by the golden ratio. The default (x=1) returns the value of the golden ratio.
#'
#' @examples
#' library(psycho)
#'
#' golden()
#' golden(8)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
golden <- function(x = 1) {
  return(x * (1 + sqrt(5)) / 2)
}









#' Highest Density Intervals (HDI).
#'
#' Compute the Highest Density Intervals (HDI) of a distribution.
#'
#' @param x A vector of values from a probability distribution (e.g., posterior probabilities from MCMC sampling).
#' @param prob Scalar between 0 and 1, indicating the mass within the credible interval that is to be estimated.
#'
#' @examples
#' library(psycho)
#'
#' distribution <- rnorm(1000, 0, 1)
#' HDI_values <- HDI(distribution)
#' print(HDI_values)
#' plot(HDI_values)
#' summary(HDI_values)
#'
#' x <- matrix(rexp(200), 100)
#' HDI_values <- HDI(x)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
HDI <- function(x, prob = .95) {

  # From CI to prob if necessary
  if (prob > 1 & prob <= 100) {
    prob <- prob / 100
  }

  # If x is a matrix
  if (ncol(as.matrix(x)) > 1) {
    HDImin <- c()
    HDImax <- c()
    for (col in seq_len(ncol(x))) {
      HDI <- .HDI(x[, col], prob = prob)
      HDImin <- c(HDImin, HDI[1])
      HDImax <- c(HDImax, HDI[2])
    }
    return(data.frame(HDImin = HDImin, HDImax = HDImax))


    # If x is a vector
  } else {
    # Process
    # -------------
    HDI <- .HDI(x, prob = prob)
    HDImin <- HDI[1]
    HDImax <- HDI[2]

    # Store results
    # -------------
    values <- list(HDImin = HDImin, HDImax = HDImax, prob = prob)
    text <- paste(
      prob * 100,
      "% CI [",
      format_string(HDImin, "%.2f"),
      ", ",
      format_string(HDImax, "%.2f"),
      "]",
      sep = ""
    )
    summary <- data.frame(Probability = prob, HDImin = HDImin, HDImax = HDImax)


    # Plot
    # -------------
    data <- as.data.frame(x = x)
    plot <- ggplot(data = data, aes(x)) +
      geom_density(fill = "#2196F3") +
      geom_vline(
        data = data, aes(xintercept = HDImin),
        linetype = "dashed", color = "#E91E63", size = 1
      ) +
      geom_vline(
        data = data, aes(xintercept = HDImax),
        linetype = "dashed", color = "#E91E63", size = 1
      ) +
      theme_minimal()


    # Output
    # -------------
    output <- list(text = text, plot = plot, summary = summary, values = values)

    class(output) <- c("psychobject", "list")
    return(output)
  }
}




#' Highest Density Intervals (HDI)
#'
#' See \link[=HDI]{HDI}
#'
#' @param x A vector of values from a probability distribution (e.g., posterior probabilities from MCMC sampling).
#' @param prob Scalar between 0 and 1, indicating the mass within the credible interval that is to be estimated.
#'
#' @export
HDImin <- function(x, prob = .95) {
  HDImin <- HDI(x, prob = prob)$values$HDImin
  return(HDImin)
}

#' Highest Density Intervals (HDI)
#'
#' See \link[=HDI]{HDI}
#'
#' @param x A vector of values from a probability distribution (e.g., posterior probabilities from MCMC sampling).
#' @param prob Scalar between 0 and 1, indicating the mass within the credible interval that is to be estimated.
#'
#' @export
HDImax <- function(x, prob = .95) {
  HDImax <- HDI(x, prob = prob)$values$HDImax
  return(HDImax)
}






#' @keywords internal
.HDI <- function(x, prob) {
  x <- sort(x)
  ci.index <- ceiling(prob * length(x))
  nCIs <- length(x) - ci.index
  ci.width <- purrr::map_dbl(1:nCIs, ~ x[.x + ci.index] - x[.x])
  HDImin <- x[which.min(ci.width)]
  HDImax <- x[which.min(ci.width) + ci.index]
  return(c(HDImin, HDImax))
}








#' Bayes Factor Interpretation
#'
#' Return the interpretation of a Bayes Factor.
#'
#' @param x Bayes Factor.
#' @param direction Include direction (against / in favour).
#' @param bf Include Bayes Factor.
#' @param rules Can be "jeffreys1961" (default), "raftery1995", or a custom list.
#'
#'
#' @examples
#' library(psycho)
#' interpret_bf(x = 10)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @references
#' \itemize{
#'  \item{Jeffreys, H. (1961), Theory of Probability, 3rd ed., Oxford University Press, Oxford.}
#'  \item{Jarosz, A. F., & Wiley, J. (2014). What are the odds? A practical guide to computing and reporting Bayes factors. The Journal of Problem Solving, 7(1), 2.}
#'  }
#' @export
interpret_bf <- function(x, direction = TRUE, bf = TRUE, rules = "jeffreys1961") {
  interpretation <- sapply(x, .interpret_bf, direction = direction, bf = bf, rules = rules, return_rules = FALSE)
  return(interpretation)
}





#' Bayes factor formatting
#'
#' Bayes factor formatting
#'
#' @param bf Bayes Factor.
#' @param max Treshold for maximum.
#'
#' @export
format_bf <- function(bf, max = 100) {
  if (bf > max) {
    bf <- paste0("BF > ", max)
  } else {
    bf <- paste0("BF = ", format_digit(bf))
  }
  return(bf)
}










#' @keywords internal
.interpret_bf <- function(x, direction = TRUE, bf = TRUE, rules = "jeffreys1961", return_rules = TRUE) {
  if (x < 1) {
    x <- 1 / abs(x)
    dir <- "against"
  } else {
    dir <- "in favour of"
  }


  if (!is.list(rules)) {
    if (rules == "jeffreys1961") {
      rules <- list(
        "no" = 0,
        "anecdotal" = 1,
        "moderate" = 3,
        "strong" = 10,
        "very strong" = 30,
        "extreme" = 100
      )
    } else if (rules == "raftery1995") {
      rules <- list(
        "no" = 0,
        "weak" = 1,
        "positive" = 3,
        "strong" = 20,
        "very strong" = 150
      )
    } else {
      stop("rules must be either a list or 'jeffreys1961' or 'raftery1995'.")
    }
  }



  s <- (abs(x) - unlist(rules))
  s <- names(which.min(s[s >= 0]))
  if (is.null(s)) {
    s <- NA
  } else {
    s <- paste(s, "evidence")
  }




  if (bf == TRUE) {
    bf <- paste0("(", format_bf(x), ")")
    s <- paste(s, bf)
  }
  if (direction == TRUE) {
    s <- paste(s, dir)
  }

  return(s)
}








#' Standardized difference (Cohen's d) interpreation.
#'
#' Interpret d with a set of rules.
#'
#' @param x Standardized difference.
#' @param direction Return direction.
#' @param rules Can be "cohen1988" (default), "sawilowsky2009", or a custom list.
#'
#' @examples
#' library(psycho)
#' interpret_d(-0.42)
#' interpret_d(-0.62)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
interpret_d <- function(x, direction = FALSE, rules = "cohen1988") {
  interpretation <- sapply(x, .interpret_d, direction = direction, rules = rules, return_rules = FALSE)
  return(interpretation)
}







#' Standardized difference (Cohen's d) interpreation for a posterior distribution.
#'
#' Interpret d with a set of rules.
#'
#' @param posterior Posterior distribution of standardized differences.
#' @param rules Can be "cohen1988" (default), "sawilowsky2009", or a custom list.
#'
#' @examples
#' library(psycho)
#' posterior <- rnorm(1000, 0.6, 0.05)
#' interpret_d_posterior(posterior)
#' interpret_d_posterior(rnorm(1000, 0.1, 1))
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
interpret_d_posterior <- function(posterior, rules = "cohen1988") {
  interpretation <- sapply(posterior, .interpret_d, rules = rules, direction = TRUE, return_rules = TRUE)
  rules <- unlist(interpretation[, 1]$rules)
  interpretation <- as.data.frame(unlist(interpretation[1, ]))
  interpretation <- na.omit(interpretation)
  names(interpretation) <- "Interpretation"

  summary <- interpretation %>%
    group_by_("Interpretation") %>%
    summarise_("Probability" = "n() / length(posterior)") %>%
    tidyr::separate("Interpretation",
      c("Size", "Direction"),
      " and ",
      remove = FALSE
    ) %>%
    mutate_(
      "Median" = 'ifelse(median(posterior) > 0, "positive", "negative")',
      "Opposite" = "ifelse(Median == Direction, FALSE, TRUE)",
      "Size" = "factor(Size)"
    ) %>%
    arrange_("Size")

  values <- list()
  for (size in names(sort(rules, decreasing = TRUE))) {
    if (size %in% summary$Size) {
      if (nrow(summary[summary$Size == size & summary$Opposite == FALSE, ]) == 0) {
        values[size] <- 0
      } else {
        values[size] <- summary[summary$Size == size & summary$Opposite == FALSE, ]$Probability
      }
    } else {
      values[size] <- 0
    }
  }
  values$opposite <- sum(summary[summary$Opposite == TRUE, ]$Probability)


  # Text
  if (length(summary[summary$Opposite == FALSE, ]$Size) > 1) {
    text_sizes <- paste0(paste0(head(summary[summary$Opposite == FALSE, ]$Size, -1), collapse = ", "), " or ", tail(summary[summary$Opposite == FALSE, ]$Size, 1))
    text_effects <- paste0(
      paste0(paste0(format_digit(head(summary[summary$Opposite == FALSE, ]$Probability * 100, -1)), "%"), collapse = ", "),
      " and ",
      paste0(format_digit(tail(summary[summary$Opposite == FALSE, ]$Probability, 1) * 100), "%")
    )

    text <- paste0(
      "The effect's size can be considered as ",
      text_sizes,
      " with respective probabilities of ",
      text_effects,
      "."
    )
  } else {
    text_sizes <- summary[summary$Opposite == FALSE, ]$Size
    text_effects <- paste0(format_digit(summary[summary$Opposite == FALSE, ]$Probability * 100), "%")

    text <- paste0(
      "The effect's size can be considered as ",
      text_sizes,
      " with a probability of ",
      text_effects,
      "."
    )
  }



  plot <- "Not available."

  output <- list(text = text, plot = plot, summary = summary, values = values)
  class(output) <- c("psychobject", "list")

  return(output)
}






#' @keywords internal
.interpret_d <- function(x, direction = FALSE, rules = "cohen1988", return_rules = TRUE) {
  if (!is.list(rules)) {
    if (rules == "cohen1988") {
      rules <- list(
        "very small" = 0,
        "small" = 0.2,
        "medium" = 0.5,
        "large" = 0.8
      )
    } else if (rules == "sawilowsky2009") {
      rules <- list(
        "tiny" = 0,
        "very small" = 0.1,
        "small" = 0.2,
        "medium" = 0.5,
        "large" = 0.8,
        "very large" = 1.2,
        "huge" = 2.0
      )
    } else {
      stop("rules must be either a list or 'cohen1988' or 'sawilowsky2009'.")
    }
  }


  if (x > 0) {
    d <- "positive"
  } else {
    d <- "negative"
  }

  x <- (abs(x) - unlist(rules))
  s <- names(which.min(x[x >= 0]))
  if (is.null(s)) {
    s <- NA
  }


  if (direction) {
    interpretation <- paste(s, "and", d)
  } else {
    interpretation <- s
  }


  if (return_rules) {
    return(list(interpretation = interpretation, rules = rules))
  } else {
    return(interpretation)
  }
}







#' Interpret fit measures of lavaan or blavaan objects
#'
#' Interpret fit measures of lavaan or blavaan objects
#'
#' @param fit lavaan or blavaan object.
#' @param ... Arguments passed to or from other methods.
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
interpret_lavaan <- function(fit, ...) {
  UseMethod("interpret_lavaan")
}






#' Interpret fit measures of lavaan objects
#'
#' Interpret fit measures of lavaan objects
#'
#' @param fit lavaan or blavaan object.
#' @param ... Arguments passed to or from other methods.
#'
#' @importFrom lavaan fitmeasures
#' @export
interpret_lavaan.lavaan <- function(fit, ...) {
  values <- list()

  indices <- lavaan::fitmeasures(fit)


  for (index in names(indices)) {
    values[index] <- indices[index]
  }

  # awang2012
  # https://www.researchgate.net/post/Whats_the_standard_of_fit_indices_in_SEM
  if (values$cfi >= 0.9) {
    cfi <- "satisfactory"
  } else {
    cfi <- "poor"
  }
  if (values$rmsea <= 0.08) {
    rmsea <- "satisfactory"
  } else {
    rmsea <- "poor"
  }
  if (values$gfi >= 0.9) {
    gfi <- "satisfactory"
  } else {
    gfi <- "poor"
  }
  if (values$tli >= 0.9) {
    tli <- "satisfactory"
  } else {
    tli <- "poor"
  }
  if (values$nfi >= 0.9) {
    nfi <- "satisfactory"
  } else {
    nfi <- "poor"
  }

  # Summary
  summary <- data.frame(
    Index = c("RMSEA", "CFI", "GFI", "TLI", "NFI", "Chisq"),
    Value = c(values$rmsea, values$cfi, values$gfi, values$tli, values$nfi, values$chisq),
    Interpretation = c(rmsea, cfi, gfi, tli, nfi, NA),
    Treshold = c("< .08", "> .90", "> 0.90", "> 0.90", "> 0.90", NA)
  )

  # Text
  if ("satisfactory" %in% summary$Interpretation) {
    satisfactory <- summary %>%
      filter_("Interpretation == 'satisfactory'") %>%
      mutate_("Index" = "paste0(Index, ' (', format_digit(Value), ' ', Treshold, ')')") %>%
      select_("Index") %>%
      pull() %>%
      paste0(collapse = ", ")
    satisfactory <- paste0("The ", satisfactory, " show satisfactory indices of fit.")
  } else {
    satisfactory <- ""
  }
  if ("poor" %in% summary$Interpretation) {
    poor <- summary %>%
      filter_("Interpretation == 'poor'") %>%
      mutate_(
        "Treshold" = 'stringr::str_replace(Treshold, "<", "SUP")',
        "Treshold" = 'stringr::str_replace(Treshold, ">", "INF")',
        "Treshold" = 'stringr::str_replace(Treshold, "SUP", ">")',
        "Treshold" = 'stringr::str_replace(Treshold, "INF", "<")'
      ) %>%
      mutate_("Index" = "paste0(Index, ' (', format_digit(Value), ' ', Treshold, ')')") %>%
      select_("Index") %>%
      pull() %>%
      paste0(collapse = ", ")
    poor <- paste0("The ", poor, " show poor indices of fit.")
  } else {
    poor <- ""
  }
  text <- paste(satisfactory, poor)

  output <- list(text = text, summary = summary, values = values, plot = "Not available yet")
  class(output) <- c("psychobject", "list")
  return(output)
}






#' Interpret fit measures of blavaan objects
#'
#' Interpret fit measures of blavaan objects
#'
#' @param indices Vector of strings indicating which indices to report. Only works for bayesian objects for now.
#' @inheritParams interpret_lavaan
#' @export
interpret_lavaan.blavaan <- function(fit, indices = c("BIC", "DIC", "WAIC", "LOOIC"), ...) {
  values <- list()

  indices <- lavaan::fitmeasures(fit)


  for (index in names(indices)) {
    values[index] <- indices[index]
  }

  # Summary
  summary <- as.data.frame(indices) %>%
    rownames_to_column("Index") %>%
    rename_("Value" = "indices") %>%
    mutate_("Index" = "str_to_upper(Index)")

  # Text
  relevant_indices <- summary[summary$Index %in% c("BIC", "DIC", "WAIC", "LOOIC"), ]
  text <- paste0(relevant_indices$Index, " = ", format_digit(relevant_indices$Value), collapse = ", ")

  output <- list(text = text, summary = summary, values = values, plot = "Not available yet")
  class(output) <- c("psychobject", "list")
  return(output)
}








#' Odds ratio interpreation for a posterior distribution.
#'
#' Interpret odds with a set of rules.
#'
#' @param x Odds ratio.
#' @param log Are these log odds ratio?
#' @param direction Return direction.
#' @param rules Can be "chen2010" (default), "cohen1988" (through \link[=odds_to_d]{log odds to Cohen's d transformation}) or a custom list.
#'
#' @examples
#' library(psycho)
#' interpret_odds(x = 2)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @seealso http://imaging.mrc-cbu.cam.ac.uk/statswiki/FAQ/effectSize
#'
#' @references
#' \itemize{
#'  \item{Chen, H., Cohen, P., & Chen, S. (2010). How big is a big odds ratio? Interpreting the magnitudes of odds ratios in epidemiological studies. Communications in Statistics—Simulation and Computation, 39(4), 860-864.}
#'  }
#' @export
interpret_odds <- function(x, log = FALSE, direction = FALSE, rules = "chen2010") {
  if (rules %in% c("cohen1988", "sawilowsky2009")) {
    interpretation <- sapply(odds_to_d(x, log = log), .interpret_d, direction = direction, rules = rules, return_rules = FALSE)
  } else {
    interpretation <- sapply(x, .interpret_odds, log = log, direction = direction, rules = rules, return_rules = FALSE)
  }
  return(interpretation)
}










#' Odds ratio interpreation for a posterior distribution.
#'
#' Interpret odds with a set of rules.
#'
#' @param posterior Posterior distribution of odds ratio.
#' @param log Are these log odds ratio?
#' @param rules Can be "chen2010" (default), "cohen1988" (through \link[=odds_to_d]{log odds to Cohen's d transformation}) or a custom list.
#'
#' @examples
#' library(psycho)
#' posterior <- rnorm(1000, 0.6, 0.05)
#' interpret_odds_posterior(posterior)
#' interpret_odds_posterior(rnorm(1000, 0.1, 1))
#' interpret_odds_posterior(rnorm(1000, 3, 1.5))
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
interpret_odds_posterior <- function(posterior, log = FALSE, rules = "chen2010") {
  if (rules %in% c("cohen1988", "sawilowsky2009")) {
    posterior <- odds_to_d(posterior, log = log)
    interpretation <- sapply(posterior, .interpret_d, direction = TRUE, rules = rules, return_rules = TRUE)
  } else {
    interpretation <- sapply(posterior, .interpret_odds, log = log, direction = TRUE, rules = rules, return_rules = TRUE)
  }
  rules <- unlist(interpretation[, 1]$rules)
  interpretation <- as.data.frame(unlist(interpretation[1, ]))
  interpretation <- na.omit(interpretation)
  names(interpretation) <- "Interpretation"

  summary <- interpretation %>%
    group_by_("Interpretation") %>%
    summarise_("Probability" = "n() / length(posterior)") %>%
    tidyr::separate("Interpretation",
      c("Size", "Direction"),
      " and ",
      remove = FALSE
    ) %>%
    mutate_(
      "Median" = 'ifelse(median(posterior) > 0, "positive", "negative")',
      "Opposite" = "ifelse(Median == Direction, FALSE, TRUE)",
      "Size" = "factor(Size)"
    ) %>%
    arrange_("Size")

  values <- list()
  for (size in names(sort(rules, decreasing = TRUE))) {
    if (size %in% summary$Size) {
      if (nrow(summary[summary$Size == size & summary$Opposite == FALSE, ]) == 0) {
        values[size] <- 0
      } else {
        values[size] <- summary[summary$Size == size & summary$Opposite == FALSE, ]$Probability
      }
    } else {
      values[size] <- 0
    }
  }
  values$opposite <- sum(summary[summary$Opposite == TRUE, ]$Probability)


  # Text
  if (length(summary[summary$Opposite == FALSE, ]$Size) > 1) {
    text_sizes <- paste0(paste0(head(summary[summary$Opposite == FALSE, ]$Size, -1), collapse = ", "), " or ", tail(summary[summary$Opposite == FALSE, ]$Size, 1))
    text_effects <- paste0(
      paste0(paste0(format_digit(head(summary[summary$Opposite == FALSE, ]$Probability * 100, -1)), "%"), collapse = ", "),
      " and ",
      paste0(format_digit(tail(summary[summary$Opposite == FALSE, ]$Probability, 1) * 100), "%")
    )

    text <- paste0(
      "The effect's size can be considered as ",
      text_sizes,
      " with respective probabilities of ",
      text_effects,
      "."
    )
  } else {
    text_sizes <- summary[summary$Opposite == FALSE, ]$Size
    text_effects <- paste0(format_digit(summary[summary$Opposite == FALSE, ]$Probability * 100), "%")

    text <- paste0(
      "The effect's size can be considered as ",
      text_sizes,
      " with a probability of ",
      text_effects,
      "."
    )
  }



  plot <- "Not available."

  output <- list(text = text, plot = plot, summary = summary, values = values)
  class(output) <- c("psychobject", "list")

  return(output)
}








#' @keywords internal
.interpret_odds <- function(x, log = FALSE, direction = FALSE, rules = "chen2010", return_rules = TRUE) {
  if (x > 0) {
    d <- "positive"
  } else {
    d <- "negative"
  }

  if (log == TRUE) {
    x <- exp(abs(x))
  }

  if (!is.list(rules)) {
    if (rules == "chen2010") {
      rules <- list(
        "very small" = 0,
        "small" = 1.68,
        "medium" = 3.47,
        "large" = 6.71
      )
    } else {
      stop("rules must be either a list or 'chen2010'.")
    }
  }


  s <- (abs(x) - unlist(rules))
  s <- names(which.min(s[s >= 0]))
  if (is.null(s)) {
    s <- NA
  }

  if (direction) {
    interpretation <- paste(s, "and", d)
  } else {
    interpretation <- s
  }

  if (return_rules) {
    return(list(interpretation = interpretation, rules = rules))
  } else {
    return(interpretation)
  }
}

















#' (Log) odds ratio to Cohen's d
#'
#' (Log) odds ratio to Cohen's d.
#'
#' @param x Odds ratio.
#' @param log Are these log odds ratio?
#'
#' @examples
#' library(psycho)
#' odds_to_d(x = 2)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @seealso https://www.meta-analysis.com/downloads/Meta-analysis%20Converting%20among%20effect%20sizes.pdf
#'
#' @references
#' \itemize{
#'  \item{Sánchez-Meca, J., Marín-Martínez, F., & Chacón-Moscoso, S. (2003). Effect-size indices for dichotomized outcomes in meta-analysis. Psychological methods, 8(4), 448.}
#'  }
#' @export
odds_to_d <- function(x, log = TRUE) {
  if (log == FALSE) {
    x <- log(x)
  }
  d <- x * (sqrt(3) / pi)
  return(d)
}










#' Omega Squared Interpretation
#'
#' Return the interpretation of Omegas Squared.
#'
#' @param x Omega Squared.
#' @param rules Can be "field2013" (default), or a custom list.
#'
#' @examples
#' library(psycho)
#' interpret_omega_sq(x = 0.05)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @seealso http://imaging.mrc-cbu.cam.ac.uk/statswiki/FAQ/effectSize
#'
#' @references
#' \itemize{
#'  \item{Field, A (2013) Discovering statistics using IBM SPSS Statistics. Fourth Edition. Sage:London.}
#'  }
#' @export
interpret_omega_sq <- function(x, rules = "field2013") {
  interpretation <- sapply(x, .interpret_omega_sq, rules = rules, return_rules = FALSE)
  return(interpretation)
}






#' @keywords internal
.interpret_omega_sq <- function(x, rules = "field2013", return_rules = TRUE) {
  if (!is.list(rules)) {
    if (rules == "field2013") {
      rules <- list(
        "very small" = 0,
        "small" = 0.01,
        "medium" = 0.06,
        "large" = 0.14
      )
    } else {
      stop("rules must be either a list or 'field2013'.")
    }
  }



  interpretation <- (abs(x) - unlist(rules))
  interpretation <- names(which.min(interpretation[interpretation >= 0]))
  if (is.null(interpretation)) {
    interpretation <- NA
  }

  return(interpretation)
}









#' Correlation coefficient r interpreation.
#'
#' Interpret r with a set of rules.
#'
#' @param x Correlation coefficient.
#' @param direction Return direction.
#' @param strength Return strength.
#' @param rules Can be "cohen1988" (default), "evans1996", or a custom list.
#'
#'
#' @examples
#' library(psycho)
#' interpret_r(-0.42)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @seealso Page 88 of APA's 6th Edition
#'
#' @export
interpret_r <- function(x, direction = TRUE, strength = TRUE, rules = "cohen1988") {
  interpretation <- sapply(x, .interpret_r, direction = direction, strength = strength, rules = rules, return_rules = FALSE)
  return(interpretation)
}









#' Correlation coefficient r interpreation for a posterior distribution.
#'
#' Interpret r with a set of rules.
#'
#' @param posterior Posterior distribution of correlation coefficient.
#' @param rules Can be "cohen1988" (default) or "evans1996", or a custom list.
#'
#' @examples
#' library(psycho)
#' posterior <- rnorm(1000, 0.5, 0.5)
#' interpret_r_posterior(posterior)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @seealso Page 88 of APA's 6th Edition
#'
#' @export
interpret_r_posterior <- function(posterior, rules = "cohen1988") {
  interpretation <- sapply(posterior, .interpret_r, rules = rules)
  rules <- unlist(interpretation[, 1]$rules)
  interpretation <- as.data.frame(unlist(interpretation[1, ]))
  interpretation <- na.omit(interpretation)
  names(interpretation) <- "Interpretation"

  summary <- interpretation %>%
    group_by_("Interpretation") %>%
    summarise_("Probability" = "n() / length(posterior)") %>%
    separate("Interpretation",
      c("Strength", "Direction"),
      ", and ",
      remove = FALSE
    ) %>%
    mutate_(
      "Median" = 'ifelse(median(posterior) > 0, "positive", "negative")',
      "Opposite" = "ifelse(Median == Direction, FALSE, TRUE)",
      "Strength" = "factor(Strength)"
    ) %>%
    arrange_("Strength")

  values <- list()
  for (strength in names(sort(rules, decreasing = TRUE))) {
    if (strength %in% summary$Strength) {
      values[strength] <- summary[summary$Strength == strength & summary$Opposite == FALSE, ]$Probability
    } else {
      values[strength] <- 0
    }
  }
  values$opposite <- sum(summary[summary$Opposite == TRUE, ]$Probability)

  # Text
  if (length(summary[summary$Opposite == FALSE, ]$Strength) > 1) {
    text_strength <- paste0(paste0(head(summary[summary$Opposite == FALSE, ]$Strength, -1), collapse = ", "), " or ", tail(summary[summary$Opposite == FALSE, ]$Strength, 1))
    text_effects <- paste0(
      paste0(paste0(format_digit(head(summary[summary$Opposite == FALSE, ]$Probability * 100, -1)), "%"), collapse = ", "),
      " and ",
      paste0(format_digit(tail(summary[summary$Opposite == FALSE, ]$Probability, 1) * 100), "%")
    )

    text <- paste0(
      "The correlation can be considered as ",
      text_strength,
      " with respective probabilities of ",
      text_effects,
      "."
    )
  } else {
    text_sizes <- summary[summary$Opposite == FALSE, ]$Strength
    text_effects <- paste0(format_digit(summary[summary$Opposite == FALSE, ]$Probability * 100), "%")

    text <- paste0(
      "The correlation can be considered as ",
      text_sizes,
      " with a probability of ",
      text_effects,
      "."
    )
  }


  plot <- "Not available."

  output <- list(text = text, plot = plot, summary = summary, values = values)
  class(output) <- c("psychobject", "list")

  return(output)
}
















#' @keywords internal
.interpret_r <- function(x, direction = TRUE, strength = TRUE, rules = "cohen1988", return_rules = TRUE) {
  if (!is.list(rules)) {
    if (rules == "evans1996") {
      rules <- list(
        "very weak" = 0,
        "weak" = 0.20,
        "moderate" = 0.40,
        "strong" = 0.60,
        "very strong" = 0.80
      )
    } else if (rules == "cohen1988") {
      rules <- list(
        "very small" = 0,
        "small" = 0.10,
        "moderate" = 0.30,
        "large" = 0.50
      )
    } else {
      stop("rules must be either a list or 'cohen1988' or 'evans1996'.")
    }
  }


  if (x > 0) {
    d <- "positive"
  } else {
    d <- "negative"
  }

  x <- (abs(x) - unlist(rules))
  s <- names(which.min(x[x >= 0]))
  if (is.null(s)) {
    s <- NA
  }



  if (strength & direction) {
    interpretation <- paste0(s, ", and ", d)
  } else if (strength & direction == FALSE) {
    interpretation <- s
  } else {
    interpretation <- d
  }



  if (return_rules) {
    return(list(interpretation = interpretation, rules = rules))
  } else {
    return(interpretation)
  }
}








#' R2 interpreation.
#'
#' Interpret R2 with a set of rules.
#'
#' @param x Value.
#' @param rules Can be "cohen1988" (default), "chin1998" or "hair2013", or a custom list.
#'
#' @examples
#' library(psycho)
#' interpret_R2(x = 0.42)
#' interpret_R2(x = c(0.42, 0.2, 0.9, 0))
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
interpret_R2 <- function(x, rules = "cohen1988") {
  interpretation <- sapply(x, .interpret_R2, rules = rules, return_rules = FALSE)
  return(interpretation)
}





#' R2 interpreation for a posterior distribution.
#'
#' Interpret R2 with a set of rules.
#'
#' @param posterior Distribution of R2.
#' @param rules Can be "cohen1988" (default), "chin1998" or "hair2013", or a custom list.
#'
#' @examples
#' library(psycho)
#' posterior <- rnorm(1000, 0.4, 0.1)
#' interpret_R2_posterior(posterior)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
interpret_R2_posterior <- function(posterior, rules = "cohen1988") {
  interpretation <- sapply(posterior, .interpret_R2, rules = rules)
  rules <- unlist(interpretation[, 1]$rules)
  interpretation <- as.data.frame(unlist(interpretation[1, ]))
  interpretation <- na.omit(interpretation)
  names(interpretation) <- "Interpretation"

  summary <- interpretation %>%
    group_by_("Interpretation") %>%
    summarise_("Probability" = "n() / length(posterior)")

  values <- list()
  for (value in names(sort(rules, decreasing = TRUE))) {
    if (value %in% summary$Interpretation) {
      values[value] <- summary[summary$Interpretation == value, ]$Probability
    } else {
      values[value] <- 0
    }
  }

  # Text
  if (length(summary$Interpretation) > 1) {
    text_strength <- paste0(paste0(head(summary$Interpretation, -1), collapse = ", "), " or ", tail(summary$Interpretation, 1))
    text_effects <- paste0(
      paste0(paste0(format_digit(head(summary$Probability * 100, -1)), "%"), collapse = ", "),
      " and ",
      paste0(format_digit(tail(summary$Probability, 1) * 100), "%")
    )

    text <- paste0(
      "The R2 can be considered as ",
      text_strength,
      " with respective probabilities of ",
      text_effects,
      "."
    )
  } else {
    text_sizes <- summary$Interpretation
    text_effects <- paste0(format_digit(summary$Probability * 100), "%")

    text <- paste0(
      "The R2 can be considered as ",
      text_sizes,
      " with a probability of ",
      text_effects,
      "."
    )
  }


  plot <- "Not available."

  output <- list(text = text, plot = plot, summary = summary, values = values)
  class(output) <- c("psychobject", "list")

  return(output)
}






#' @keywords internal
.interpret_R2 <- function(x, rules = "cohen1988", return_rules = TRUE) {
  if (!is.list(rules)) {
    if (rules == "cohen1988") {
      rules <- list(
        "very small" = 0,
        "small" = 0.02,
        "medium" = 0.13,
        "large" = 0.26
      )
    } else if (rules == "chin1998") {
      rules <- list(
        "very small" = 0,
        "small" = 0.19,
        "medium" = 0.33,
        "large" = 0.67
      )
    } else if (rules == "hair2013") {
      rules <- list(
        "very small" = 0,
        "small" = 0.25,
        "medium" = 0.50,
        "large" = 0.75
      )
    } else {
      stop("rules must be either a list or 'cohen1988', 'chin1998' or 'hair2013'.")
    }
  }

  x <- (x - unlist(rules))
  interpretation <- names(which.min(x[x >= 0]))
  if (is.null(interpretation)) {
    interpretation <- NA
  }

  if (return_rules) {
    return(list(interpretation = interpretation, rules = rules))
  } else {
    return(interpretation)
  }
}






#' RMSEA interpreation.
#'
#' Interpret RMSEA with a set of rules.
#'
#' @param x RMSEA.
#' @param rules Can be "awang2012", or a custom list.
#'
#' @examples
#' library(psycho)
#' interpret_RMSEA(0.04)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
interpret_RMSEA <- function(x, rules = "awang2012") {
  interpretation <- sapply(x, .interpret_RMSEA, rules = rules, return_rules = FALSE)
  return(interpretation)
}





#' @keywords internal
.interpret_RMSEA <- function(x, rules = "awang2012", return_rules = TRUE) {
  if (!is.list(rules)) {
    if (rules == "awang2012") {
      rules <- list(
        "good" = 0,
        "acceptable" = 0.05,
        "poor" = 0.08
      )
    } else {
      stop("rules must be either a list or 'awang2012'.")
    }
  }

  x <- (abs(x) - unlist(rules))
  s <- names(which.min(x[x >= 0]))
  if (is.null(s)) {
    s <- NA
  }

  if (return_rules) {
    return(list(interpretation = s, rules = rules))
  } else {
    return(s)
  }
}





#' Check if model includes random effects.
#'
#' Check if model is mixed. See the
#' documentation for your model's class:
#' \itemize{
#'  \item{\link[=is.mixed.stanreg]{is.mixed.stanreg}}
#'  }
#'
#' @param fit Model.
#' @param ... Arguments passed to or from other methods.
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
is.mixed <- function(fit, ...) {
  UseMethod("is.mixed")
}













#' Check if model includes random effects.
#'
#' Check if model is mixed.
#'
#' @param fit Model.
#' @param ... Arguments passed to or from other methods.
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
is.mixed.stanreg <- function(fit, ...) {
  mixed <- tryCatch({
    broom::tidy(fit, parameters = "varying")
    TRUE
  }, error = function(e) {
    FALSE
  })
  return(mixed)
}









#' Check if a dataframe is standardized.
#'
#' Check if a dataframe is standardized.
#'
#' @param df A dataframe.
#' @param tol The error treshold.
#'
#' @examples
#' library(psycho)
#'
#' df <- psycho::affective
#' is.standardized(df)
#'
#' dfZ <- psycho::standardize(df)
#' is.standardized(dfZ)
#' @return bool.
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @import purrr
#' @export
is.standardized <- function(df, tol = 0.1) {
  dfZ <- standardize(df)
  dfZnum <- purrr::keep(dfZ, is.numeric)

  dfnum <- purrr::keep(df, is.numeric)

  error <- as.matrix(dfnum) - as.matrix(dfZnum)
  error <- as.data.frame(error)
  names(error) <- names(dfnum)

  error_mean <- error %>%
    summarise_all(mean)

  if (TRUE %in% as.character(error_mean[1, ] > tol)) {
    standardized <- FALSE
  } else {
    standardized <- TRUE
  }
  return(standardized)
}







#' Mellenbergh & van den Brink (1998) test for pre-post comparison.
#'
#' Test for comparing post-test to baseline for a single participant.
#'
#' @param t0 Single value (pretest or baseline score).
#' @param t1 Single value (posttest score).
#' @param controls Vector of scores of the control group OR single value corresponding to the control SD of the score.
#'
#' @return Returns a data frame containing the z-value and p-value. If significant, the difference between pre and post tests is significant.
#'
#' @examples
#' library(psycho)
#'
#' mellenbergh.test(t0 = 4, t1 = 12, controls = c(0, -2, 5, 2, 1, 3, -4, -2))
#' mellenbergh.test(t0 = 8, t1 = 2, controls = 2.6)
#' @author Dominique Makowski
#'
#' @importFrom stats pnorm sd
#' @export
mellenbergh.test <- function(t0, t1, controls) {
  if (length(controls) > 1) {
    sd <- sd(controls) * sqrt(2)
  } else {
    sd <- controls * sqrt(2)
  }

  diff <- t1 - t0

  diff_CI_bottom <- diff - 1.65 * sd
  diff_CI_top <- diff + 1.65 * sd

  z <- diff / sd
  pval <- 2 * pnorm(-abs(z))

  # One-tailed p value
  if (pval > .05 & pval / 2 < .05) {
    one_tailed <- paste0(
      " However, the null hypothesis of no change can be rejected at a one-tailed 5% significance level (one-tailed p ",
      format_p(pval / 2),
      ")."
    )
  } else {
    one_tailed <- ""
  }



  p_interpretation <- ifelse(pval < 0.05, " ", " not ")
  text <- paste0(
    "The Mellenbergh & van den Brink (1998) test suggests that the change is",
    p_interpretation,
    "significant (d = ",
    format_digit(diff),
    ", 90% CI [",
    format_digit(diff_CI_bottom),
    ", ",
    format_digit(diff_CI_top),
    "], z = ",
    format_digit(z),
    ", p ",
    format_p(pval),
    ").",
    one_tailed
  )


  values <- list(
    text = text,
    diff = diff,
    diff_90_CI_lower = diff_CI_bottom,
    diff_90_CI_higher = diff_CI_top,
    z = z,
    p = pval
  )
  summary <- data.frame(diff = diff, diff_90_CI_lower = diff_CI_bottom, diff_90_CI_higher = diff_CI_top, z = z, p = pval)
  plot <- "Not available yet"


  output <- list(text = text, plot = plot, summary = summary, values = values)
  class(output) <- c("psychobject", "list")
  return(output)
  #   return("The method for no-controls is not implemented yet.")
}










#' Model to Prior.
#'
#' Convert a Bayesian model's results to priors.
#'
#' @param fit A stanreg model.
#' @param autoscale Set autoscale.
#' @examples
#' \dontrun{
#' library(rstanarm)
#' library(psycho)
#'
#' fit <- stan_glm(Sepal.Length ~ Petal.Width, data = iris)
#' priors <- model_to_priors(fit)
#' update(fit, prior = priors$prior)
#'
#' fit <- stan_glmer(Subjective_Valence ~ Emotion_Condition + (1 | Participant_ID),
#'   data = psycho::emotion
#' )
#' priors <- model_to_priors(fit)
#'
#' fit1 <- stan_glm(Subjective_Valence ~ Emotion_Condition,
#'   data = filter(psycho::emotion, Participant_ID == "1S")
#' )
#'
#' fit2 <- stan_glm(Subjective_Valence ~ Emotion_Condition,
#'   data = filter(psycho::emotion, Participant_ID == "1S"),
#'   prior = priors$prior, prior_intercept = priors$prior_intercept
#' )
#' }
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @import dplyr
#' @importFrom stats update
#' @importFrom rstanarm normal
#' @export
model_to_priors <- function(fit, autoscale = FALSE) {
  posteriors <- as.data.frame(fit)

  # Varnames
  varnames <- names(posteriors)
  varnames <- varnames[grepl("b\\[", varnames) == FALSE]

  fixed_effects <- names(fit$coefficients)
  fixed_effects <- fixed_effects[grepl("b\\[", fixed_effects) == FALSE]
  fixed_effects <- fixed_effects[fixed_effects != "(Intercept)"]

  # Get priors
  prior_intercept <- list()
  priors <- list()
  prior_aux <- list()
  for (prior in varnames) {
    if (prior == "(Intercept)") {
      prior_intercept$mean <- mean(posteriors[[prior]])
      prior_intercept$sd <- sd(posteriors[[prior]])
    } else if (prior %in% fixed_effects) {
      priors[[prior]] <- list()
      priors[[prior]]$mean <- mean(posteriors[[prior]])
      priors[[prior]]$sd <- sd(posteriors[[prior]])
    } else {
      prior_aux[[prior]] <- list()
      prior_aux[[prior]]$mean <- mean(posteriors[[prior]])
      prior_aux[[prior]]$sd <- sd(posteriors[[prior]])
    }
  }


  prior_intercept <- rstanarm::normal(
    prior_intercept$mean,
    prior_intercept$sd,
    autoscale = autoscale
  )
  prior <- .format_priors(priors, autoscale = autoscale)
  prior_aux <- .format_priors(prior_aux, autoscale = autoscale)

  return(list(prior_intercept = prior_intercept, prior = prior, priox_aux = prior_aux))
}


#' @keywords internal
.format_priors <- function(priors, autoscale = FALSE) {
  prior_mean <- data.frame(priors) %>%
    select(contains("mean")) %>%
    gather() %>%
    select_("value") %>%
    pull()

  prior_sd <- data.frame(priors) %>%
    select(contains("sd")) %>%
    gather() %>%
    select_("value") %>%
    pull()

  prior <- rstanarm::normal(
    prior_mean,
    prior_sd,
    autoscale = autoscale
  )
}







#' Compute Maximum Probability of Effect (MPE).
#'
#' Compute the Maximum Probability of Effect (MPE), i.e., the proportion of posterior distribution that is of the same sign as the median. In other words, it corresponds to the maximum probability that the effect is different from 0 in the medianâ€™s direction.
#'
#' @param posterior Posterior Distribution.
#'
#' @return list containing the MPE and its values.
#'
#' @examples
#' library(psycho)
#' library(rstanarm)
#'
#' fit <- rstanarm::stan_glm(rating ~ advance, data = attitude)
#' posterior <- psycho::analyze(fit)$values$effects$advance$posterior
#' mpe <- psycho::mpe(posterior)
#' print(mpe$MPE)
#' print(mpe$values)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
mpe <- function(posterior) {
  median <- median(posterior)
  if (median >= 0) {
    MPE <- length(posterior[posterior >= 0]) / length(posterior) * 100
    if (MPE == 100) {
      MPE_values <- c(min(posterior), max(posterior))
    } else {
      MPE_values <- c(0, max(posterior))
    }
  } else {
    MPE <- length(posterior[posterior < 0]) / length(posterior) * 100
    if (MPE == 100) {
      MPE_values <- c(min(posterior), max(posterior))
    } else {
      MPE_values <- c(min(posterior), 0)
    }
  }

  MPE <- list(MPE = MPE, values = MPE_values)
  return(MPE)
}







#' Find Optimal Factor Number.
#'
#' Find optimal components number using maximum method aggreement.
#'
#' @param df A dataframe or correlation matrix
#' @param rotate What rotation to use c("none", "varimax", "oblimin","promax")
#' @param fm Factoring method: "pa" for Principal Axis Factor Analysis,
#' "minres" (default) for minimum residual (OLS) factoring, "mle" for
#' Maximum Likelihood FA and "pc" for Principal Components
#' @param n If correlation matrix is passed, the sample size.
#'
#' @return output
#'
#' @examples
#' df <- dplyr::select_if(attitude, is.numeric)
#' results <- psycho::n_factors(df)
#'
#' summary(results)
#' plot(results)
#'
#' # See details on methods
#' psycho::values(results)$methods
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @importFrom qgraph cor_auto
#' @importFrom psych VSS
#' @importFrom MASS mvrnorm
#' @importFrom MASS ginv
#' @importFrom nFactors moreStats
#' @importFrom nFactors nScree
#' @importFrom stats cov
#' @importFrom stats dnorm
#' @importFrom stats qnorm
#' @export
n_factors <- function(df, rotate = "varimax", fm = "minres", n = NULL) {

  # Copy the parallel function from nFactors to correct the use of mvrnorm
  parallel <- function(subject = 100, var = 10, rep = 100, cent = 0.05,
                         quantile = cent, model = "components",
                         sd = diag(1, var), ...) {
    r <- subject
    c <- var
    y <- matrix(c(1:r * c), nrow = r, ncol = c)
    evpea <- NULL
    for (k in c(1:rep)) {
      y <- MASS::mvrnorm(n = r, mu = rep(0, var), Sigma = sd, empirical = FALSE)
      corY <- cov(y, ...)
      if (model == "components") {
        diag(corY) <- diag(sd)
      }
      if (model == "factors") {
        corY <- corY - MASS::ginv(diag(diag(MASS::ginv(corY))))
      }
      evpea <- rbind(evpea, eigen(corY)[[1]])
    }
    SEcentile <- function(sd, n = 100, p = 0.95) {
      return(sd / sqrt(n) * sqrt(p * (1 - p)) / dnorm(qnorm(p)))
    }
    mevpea <- sapply(as.data.frame(evpea), mean)
    sevpea <- sapply(as.data.frame(evpea), sd)
    qevpea <- nFactors::moreStats(evpea, quantile = quantile)[3, ]
    sqevpea <- sevpea
    sqevpea <- sapply(
      as.data.frame(sqevpea), SEcentile,
      n = rep,
      p = cent
    )
    result <- list(
      eigen = data.frame(
        mevpea, sevpea, qevpea,
        sqevpea
      ),
      subject = r,
      variables = c,
      centile = cent
    )
    class(result) <- "parallel"
    return(result)
  }

  # Detect if df us a correlation matrix
  if (length(setdiff(names(df), rownames(df))) != 0) {
    cor <- qgraph::cor_auto(df, forcePD = FALSE)
    n <- nrow(df)
  } else {
    if (is.null(n)) {
      stop("A correlation matrix was passed. You must provided the sample size (n).")
    }
    cor <- df
  }


  ap <- parallel(subject = n, var = ncol(cor))
  nS <- nFactors::nScree(x = eigen(cor)$values, aparallel = ap$eigen$qevpea)

  # Eigeinvalues data
  eigenvalues <- nS$Analysis %>%
    dplyr::select_(
      "Eigenvalues",
      "Exp.Variance" = "Prop",
      "Cum.Variance" = "Cumu"
    ) %>%
    mutate_("n.Factors" = ~ seq_len(nrow(nS$Analysis)))





  # Processing
  # -------------------
  results <- data.frame(
    Method = c(
      "Optimal Coordinates",
      "Acceleration Factor",
      "Parallel Analysis",
      "Eigenvalues (Kaiser Criterion)"
    ),
    n_optimal = as.numeric(nS$Components[1, ])
  )

  # EGA Method
  # Doesn't really work for now :(
  # ega <- EGA::EGA(cor, plot.EGA = F, matrix=TRUE, n = n)
  # ega <- EGA::bootEGA(df, n = 1000)

  # VSS
  vss <- psych::VSS(
    cor,
    n.obs = n,
    rotate = rotate,
    fm = fm, plot = F
  ) # fm can be "pa", "pc", "minres", "mle"
  stats <- vss$vss.stats
  stats$map <- vss$map
  stats$n_factors <- seq_len(nrow(stats))

  # map
  if (length(stats$map[!is.na(stats$map)]) > 0) {
    min <- min(stats$map[!is.na(stats$map)])
    opt <- stats[stats$map == min, ]$n_factors[!is.na(stats[stats$map == min, ]$n_factors)]
    results <- rbind(
      results,
      data.frame(
        Method = c("Velicer MAP"),
        n_optimal = c(opt)
      )
    )
  }
  # bic
  if (length(stats$BIC[!is.na(stats$BIC)]) > 0) {
    min <- min(stats$BIC[!is.na(stats$BIC)])
    opt <- stats[stats$BIC == min, ]$n_factors[!is.na(stats[stats$BIC == min, ]$n_factors)]
    results <- rbind(
      results,
      data.frame(
        Method = c("BIC"),
        n_optimal = c(opt)
      )
    )
  }
  # sabic
  if (length(stats$SABIC[!is.na(stats$SABIC)]) > 0) {
    min <- min(stats$SABIC[!is.na(stats$SABIC)])
    opt <- stats[stats$SABIC == min, ]$n_factors[!is.na(stats[stats$SABIC == min, ]$n_factors)]
    results <- rbind(
      results,
      data.frame(
        Method = c("Sample Size Adjusted BIC"),
        n_optimal = c(opt)
      )
    )
  }


  cfits <- vss[grep("cfit", names(vss))]
  for (name in names(cfits)) {
    cfit <- cfits[[name]]

    cfit <- data.frame(cfit = cfit, n_factors = seq_len(length(cfit)))

    result3 <- data.frame(
      Method = c(gsub("cfit.", "VSS Complexity ", name)),
      n_optimal = c(na.omit(cfit[cfit$cfit == max(cfit$cfit, na.rm = TRUE), ])$n_factors)
    )

    results <- rbind(results, result3)
  }


  eigenvalues <- results %>%
    group_by_("n_optimal") %>%
    summarise_("n_method" = ~ n()) %>%
    mutate_("n_optimal" = ~ factor(n_optimal, levels = seq_len(nrow(eigenvalues)))) %>%
    complete_("n_optimal", fill = list(n_method = 0)) %>%
    arrange_("n_optimal") %>%
    rename_(
      "n.Factors" = "n_optimal",
      "n.Methods" = "n_method"
    ) %>%
    mutate_("n.Factors" = ~ as.integer(n.Factors)) %>%
    left_join(eigenvalues, by = "n.Factors") %>%
    select_("-Exp.Variance")


  # Summary
  # -------------
  summary <- eigenvalues

  # Values
  # -------------

  best_n_df <- filter_(summary, "n.Methods == max(n.Methods)")
  best_n <- best_n_df$n.Factors

  best_n_methods <- list()
  for (i in as.list(best_n)) {
    methods_list <- results[results$n_optimal %in% as.list(i), ]
    methods_list <- as.character(methods_list$Method)
    best_n_methods[[paste0("n_", i)]] <- paste(methods_list, collapse = ", ")
  }



  values <- list(summary = summary, methods = results, best_n_df = best_n)



  # Text
  # -------------
  # Deal with equality
  if (length(best_n) > 1) {
    best_n <- head(best_n, length(best_n) - 1) %>%
      paste(collapse = ", ") %>%
      paste(best_n[length(best_n)], sep = " and ")
    factor_text <- " factors "
    n_methods <- unique(best_n_df$n.Methods)
    best_n_methods <- paste0(paste(best_n_methods, collapse = "; "), "; respectively")
  } else {
    n_methods <- best_n_df$n.Methods
    # Plural
    if (best_n == 1) {
      factor_text <- " factor "
    } else {
      factor_text <- " factors "
    }
  }



  text <- paste0(
    "The choice of ",
    best_n,
    factor_text,
    "is supported by ",
    n_methods,
    " (out of ",
    round(nrow(results)),
    "; ",
    round(n_methods / nrow(results) * 100, 2),
    "%) methods (",
    best_n_methods,
    ")."
  )


  # Plot
  # -------------
  plot_data <- summary
  plot_data$n.Methods.Ratio <- plot_data$n.Methods / sum(plot_data$n.Methods)
  plot_data$n.Methods.Ratio <- plot_data$n.Methods.Ratio * (1 / max(plot_data$n.Methods.Ratio))
  plot_data$area <- plot_data$n.Methods.Ratio / (max(plot_data$n.Methods.Ratio) / max(plot_data$Eigenvalues))
  plot_data$var <- plot_data$Cum.Variance / (max(plot_data$Cum.Variance) / max(plot_data$Eigenvalues))

  plot <- plot_data %>%
    ggplot(aes_string(x = "n.Factors", y = "Eigenvalues")) +
    geom_area(
      aes_string(y = "area"),
      fill = "#FFC107",
      alpha = 0.5
    ) +
    geom_line(
      colour = "#E91E63",
      size = 1
    ) +
    geom_hline(yintercept = 1, linetype = "dashed", colour = "#607D8B") +
    geom_line(
      aes_string(y = "var"),
      colour = "#2196F3",
      size = 1
    ) +
    scale_y_continuous(sec.axis = sec_axis(
      trans = ~ . * (max(plot_data$Cum.Variance) / max(plot_data$Eigenvalues)),
      name = "Cumulative Variance\n"
    )) +
    ylab("Eigenvalues\n") +
    xlab("\nNumber of Factors") +
    theme_minimal()

  # Output
  # -------------
  output <- list(text = text, plot = plot, summary = summary, values = values)

  class(output) <- c("psychobject", "list")
  return(output)
}









#' Convert (log)odds to probabilies.
#'
#' @param odds Odds values in vector or dataframe.
#' @param subset Character or list of characters of column names to be
#' transformed.
#' @param except Character or list of characters of column names to be excluded
#' from transformation.
#' @param log Are these Log odds (such as in logistic models)?
#'
#' @examples
#' library(psycho)
#' odds_to_probs(-1.45)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @importFrom purrr keep discard
#' @export
odds_to_probs <- function(odds, subset = NULL, except = NULL, log = TRUE) {

  # If vector
  if (ncol(as.matrix(odds)) == 1) {
    return(.odds_to_probs(odds, log = log))
  } else {
    df <- odds
  }

  # Variable order
  var_order <- names(df)

  # Keep subset
  if (!is.null(subset) && subset %in% names(df)) {
    to_keep <- as.data.frame(df[!names(df) %in% c(subset)])
    df <- df[names(df) %in% c(subset)]
  } else {
    to_keep <- NULL
  }

  # Remove exceptions
  if (!is.null(except) && except %in% names(df)) {
    if (is.null(to_keep)) {
      to_keep <- as.data.frame(df[except])
    } else {
      to_keep <- cbind(to_keep, as.data.frame(df[except]))
    }

    df <- df[!names(df) %in% c(except)]
  }

  # Remove non-numerics
  dfother <- purrr::discard(df, is.numeric)
  dfnum <- purrr::keep(df, is.numeric)

  # Tranform
  dfnum <- .odds_to_probs(dfnum, log = log)

  # Add non-numerics
  if (is.null(ncol(dfother))) {
    df <- dfnum
  } else {
    df <- dplyr::bind_cols(dfother, dfnum)
  }

  # Add exceptions
  if (!is.null(subset) | !is.null(except) && exists("to_keep")) {
    df <- dplyr::bind_cols(df, to_keep)
  }

  # Reorder
  df <- df[var_order]

  return(df)
}


#' @keywords internal
.odds_to_probs <- function(odds, log = TRUE) {
  if (log == TRUE) {
    odds <- exp(odds)
  }
  probs <- odds / (1 + odds)
  return(probs)
}








#' Overlap of Two Empirical Distributions.
#'
#' A method to calculate the overlap coefficient of two kernel density estimates (a measure of similarity between two samples).
#'
#' @param x A vector of values from a probability distribution (e.g., posterior probabilities from MCMC sampling).
#' @param y Scalar between 0 and 1, indicating the mass within the credible interval that is to be estimated.
#' @param method Method of AUC computation. Can be "trapezoid" (default), "step" or "spline".
#'
#' @examples
#' library(psycho)
#'
#' x <- rnorm(100, 1, 0.5)
#' y <- rnorm(100, 0, 1)
#' overlap(x, y)
#' @author S. Venne
#'
#' @importFrom stats density
#' @importFrom DescTools AUC
#' @export
overlap <- function(x, y, method = "trapezoid") {
  # define limits of a common grid, adding a buffer so that tails aren't cut off
  lower <- min(c(x, y)) - 1
  upper <- max(c(x, y)) + 1

  # generate kernel densities
  da <- stats::density(x, from = lower, to = upper)
  db <- stats::density(y, from = lower, to = upper)
  d <- data.frame(x = da$x, a = da$y, b = db$y)

  # calculate intersection densities
  d$w <- pmin(d$a, d$b)

  # integrate areas under curves
  total <- DescTools::AUC(d$x, d$a, method = method) + DescTools::AUC(d$x, d$b, method = method)
  intersection <- DescTools::AUC(d$x, d$w, method = method)

  # compute overlap coefficient
  overlap <- 2 * intersection / total
  return(overlap)
}











#' Transform z score to percentile.
#'
#' @param z_score Z score.
#'
#' @examples
#' library(psycho)
#' percentile(-1.96)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @importFrom stats pnorm
#' @export
percentile <- function(z_score) {
  perc <- pnorm(z_score) * 100
  return(perc)
}



#' Transform a percentile to a z score.
#'
#' @param percentile Percentile
#'
#' @examples
#' library(psycho)
#' percentile_to_z(95)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @importFrom stats pnorm
#' @export
percentile_to_z <- function(percentile) {
  z <- qnorm(percentile / 100)
  return(z)
}











#' Plot the results.
#'
#' @param x A psychobject class object.
#' @param ... Arguments passed to or from other methods.
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
plot.psychobject <- function(x, ...) {
  plot <- x$plot
  return(plot)
}











#' Power analysis for fitted models.
#'
#' Compute the n models based on n sampling of data.
#'
#' @param fit A lm or stanreg model.
#' @param n_max Max sample size.
#' @param n_min Min sample size. If null, take current nrow.
#' @param step Increment of the sequence.
#' @param n_batch Number of iterations at each sample size.
#' @param groups Grouping variable name (string) to preserve proportions. Can be a list of strings.
#' @param verbose Print progress.
#' @param CI Argument for \link[=analyze]{analyze}.
#' @param effsize Argument for \link[=analyze]{analyze}.
#' @param effsize_rules Argument for \link[=analyze]{analyze}.
#' @param bayes_factor Argument for \link[=analyze]{analyze}.
#' @param overlap rgument for \link[=analyze]{analyze}.
#'
#' @return A dataframe containing the summary of all models for all iterations.
#'
#' @examples
#' \dontrun{
#' library(dplyr)
#' library(psycho)
#'
#' fit <- lm(Sepal.Length ~ Sepal.Width, data = iris)
#'
#' results <- power_analysis(fit, n_max = 300, n_min = 100, step = 5, n_batch = 20)
#'
#' results %>%
#'   filter(Variable == "Sepal.Width") %>%
#'   select(n, p) %>%
#'   group_by(n) %>%
#'   summarise(
#'     p_median = median(p),
#'     p_mad = mad(p)
#'   )
#' }
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @importFrom stats model.frame
#' @import dplyr
#' @export
power_analysis <- function(fit, n_max, n_min = NULL, step = 1, n_batch = 1, groups = NULL, verbose = TRUE, CI = 90, effsize = FALSE, effsize_rules = "cohen1988", bayes_factor = FALSE, overlap = FALSE) {

  # Parameters
  df <- model.frame(fit)

  if (is.null(n_min)) {
    n_min <- nrow(df)
  }


  results <- data.frame()
  for (n in seq(n_min, n_max, step)) {
    for (batch in 1:n_batch) {

      # Progress
      if (verbose == TRUE) {
        cat(".")
      }


      # Sample data.frame
      if (!is.null(groups)) {
        newdf <- df %>%
          group_by_(groups) %>%
          dplyr::sample_frac(n / nrow(df), replace = TRUE)
      } else {
        newdf <- dplyr::sample_frac(df, n / nrow(df), replace = TRUE)
      }

      # Fit new model
      newfit <- update(fit, data = newdf)
      newfit <- analyze(newfit, CI = CI, effsize = effsize, bayes_factor = bayes_factor, overlap = overlap, effsize_rules = effsize_rules)

      # Store results
      newresults <- summary(newfit)
      newresults$n <- n
      newresults$batch <- batch
      results <- rbind(results, newresults)
    }
    # Progress
    if (verbose == TRUE) {
      cat(paste0(format_digit(round((n - n_min) / (n_max - n_min) * 100)), "%\n"))
    }
  }
  return(results)
}











#' Print the results.
#'
#' @param x A psychobject class object.
#' @param ... Further arguments passed to or from other methods.
#'
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
print.psychobject <- function(x, ...) {
  text <- x$text
  cat(text, sep = "\n")
  invisible(text)
}









#' Convert probabilities to (log)odds.
#'
#' @param probs Probabilities values in vector or dataframe.
#' @param log Compute log odds (such as in logistic models)?
#'
#' @examples
#' library(psycho)
#' probs_to_odds(0.75)
#' @author \href{https://dominiquemakowski.github.io/}{Dominique Makowski}
#'
#' @export
probs_to_odds <- function(probs, log = FALSE) {

  # If vector
  if (ncol(as.matrix(probs)) == 1) {
    return(.probs_to_odds(probs, log = log))
  } else {
    warning("Provide single value or vector.")
  }
}


#' @keywords internal
.probs_to_odds <- function(probs, log = FALSE) {
  odds <- probs / (1 - probs)
  if (log == TRUE) {
    odds <- log(odds)
  }
  return(odds)
}






