% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_watershed.R
\name{get_watershed}
\alias{get_watershed}
\title{Delineate watershed for a survey site}
\usage{
get_watershed(
  sites,
  i,
  flow_dir,
  out,
  write_file = FALSE,
  overwrite = FALSE,
  lessmem = FALSE
)
}
\arguments{
\item{sites}{A file path to a shapefile of points or a \code{SpatialPoints*} object.}

\item{i}{An integer which indexes one row of the sites' attribute table.}

\item{flow_dir}{The name of a flow direction grid in the current GRASS mapset.}

\item{out}{The name of the output raster. A raster with the name \code{basename(out)} will be imported into the GRASS mapset. If \code{write_file} is true, then a file with the name \code{out} will be written into the user's current working directory.}

\item{write_file}{A logical indicating whether the output file should be stored as a file in the user's current working directory. Defaults to \code{FALSE}.}

\item{overwrite}{A logical indicating whether the output should be allowed to overwrite existing files. Defaults to \code{FALSE}.}

\item{lessmem}{A logical indicating whether to use the less memory modified watershed module. Defaults to \code{FALSE}.}
}
\value{
Nothing. A raster file with the name \code{out} may be written to file if you have set the \code{write_file} argument accordingly. A raster with the name \code{basename(out)} will be imported into the current GRASS mapset.
}
\description{
This function delineates a watershed around the \code{i}th site from a set of survey sites.
}
\examples{
# Will only run if GRASS is running
if(check_running()){
# Load data set
dem <- system.file("extdata", "dem.tif", package = "rdwplus")
landuse <- system.file("extdata", "landuse.tif", package = "rdwplus")
sites <- system.file("extdata", "site.shp", package = "rdwplus")
stream_shp <- system.file("extdata", "streams.shp", package = "rdwplus")

# Set environment parameters and import data to GRASS
set_envir(dem)
raster_to_mapset(rasters = c(dem, landuse), as_integer = c(FALSE, TRUE))
vector_to_mapset(vectors = c(sites, stream_shp))

# Create binary stream
rasterise_stream("streams", "streams_rast.tif", overwrite = TRUE)
reclassify_streams("streams_rast.tif", "streams_binary.tif", 
out_type = "binary", overwrite = TRUE)

# Burn dem 
burn_in(dem = "dem.tif", stream = "streams_binary.tif", 
out = "dem_burn.tif", burn = 10, overwrite = TRUE)

# Fill sinks
fill_sinks(dem = "dem_burn.tif", out = "dem_fill.tif", 
size = 1, overwrite = TRUE)

# Derive flow accumulation and direction grids
derive_flow(dem = "dem_fill.tif", flow_dir = "fdir.tif", 
flow_acc = "facc.tif", overwrite = TRUE)

# Snap sites to pour points (based on flow accumulation)
snap_sites(sites = "site", flow_acc = "facc.tif", max_move = 2, 
out = "snapsite.shp", overwrite = TRUE)
point_to_raster(outlets = "site", out = "sites_rast.tif", overwrite = TRUE)

# Compute current site's watershed
rowID <- 1
current_watershed <- paste0("watershed_", rowID, ".tif")
get_watershed(sites = "snapsite.shp",
i =  rowID, 
flow_dir = "fdir.tif", 
out = current_watershed, 
write_file = FALSE, 
overwrite = TRUE)

# Plot 
plot_GRASS(current_watershed, col = topo.colors(2))
plot_GRASS("streams_rast.tif", col = "white", add = TRUE)
plot_GRASS("sites_rast.tif", col = "red", add = TRUE)
}
}
