#' @name exportMappings
#' @aliases exportMappings.redcapApiConnection
#' @aliases exportMappings.redcapDbConection
#' @export exportMappings
#' @importFrom httr POST
#' 
#' @title Exports the Event-Form Mappings for a Project
#' @description Retrieve a data frame giving the events-form mapping for a project.
#' 
#' @param rcon A REDCap connection object as generated by \code{redcapConnection}.
#' @param arms A vector of arm numbers that you wish to pull events for (by default,
#'   all events are pulled) 
#' @param ... Arguments to be passed to other methods
#' 
#' @details The data frame that is returned shows the arm number, unique 
#' event name, and forms mapped in a project.
#' 
#' When this function is called for a classic project, a character string is
#'  returned giving the API error message, '400: You cannot export form-event 
#'  mappings for classic projects' but without casting an error in R. This is 
#'  by design and allows more flexible error checks in certain functions.
#'  
#' @author Benjamin Nutter
#' 
#' @references 
#' Please refer to your institution's API documentation.
#' 
#' Additional details on API parameters are found on the package wiki at
#' \url{https://github.com/nutterb/redcapAPI/wiki/REDCap-API-Parameters}
#' 
#' @examples
#' \dontrun{
#' > #*** Note: I cannot provide working examples without
#' > #*** compromising security.  Instead, I will try to 
#' > #*** offer up sample code with the matching results
#' > 
#' > 
#' > #*** Create the connection object
#' > rcon <- redcapConnection(url=[YOUR_REDCAP_URL], token=[API_TOKEN])
#' >
#' > exportMappings(rcon)
#' arm_num        unique_event_name                 form_name
#' 1        1            event_1_arm_1              demographics
#' 2        1            event_1_arm_1 all_the_different_options
#' 3        1            event_1_arm_1        uploading_decimals
#' 4        1            event_1_arm_1              calculations
#' 5        1        follow_up_1_arm_1              calculations
#' 6        1        follow_up_2_arm_1              calculations
#' 7        2 experimental_inter_arm_2              demographics
#' 8        2 experimental_inter_arm_2        uploading_decimals
#' 9        2 experimental_follo_arm_2 all_the_different_options
#' 10       2 experimental_follo_arm_2        uploading_decimals
#' 11       2 experimental_follo_arm_2              calculations
#' 12      10          baseline_arm_10              demographics
#' 13      10          baseline_arm_10        uploading_decimals
#' > 
#' > exportMappings(rcon, 1:2)
#' arm_num        unique_event_name                 form_name
#' 1        1            event_1_arm_1              demographics
#' 2        1            event_1_arm_1 all_the_different_options
#' 3        1            event_1_arm_1        uploading_decimals
#' 4        1            event_1_arm_1              calculations
#' 5        1        follow_up_1_arm_1              calculations
#' 6        1        follow_up_2_arm_1              calculations
#' 7        2 experimental_inter_arm_2              demographics
#' 8        2 experimental_inter_arm_2        uploading_decimals
#' 9        2 experimental_follo_arm_2 all_the_different_options
#' 10       2 experimental_follo_arm_2        uploading_decimals
#' 11       2 experimental_follo_arm_2              calculations
#' }


#' 


exportMappings <- function(rcon, arms, ...) UseMethod("exportMappings")

#' @rdname exportMappings
#' @export

exportMappings.redcapDbConnection <- function(rcon, arms, ...){
  message("Please accept my apologies.  The exportMappings method for redcapDbConnection objects\n",
          "has not yet been written.  Please consider using the API.")
}

#' @rdname exportMappings
#' @export

exportMappings.redcapApiConnection <- function(rcon, arms, ...){
  .params <- list(token=rcon$token, content='formEventMapping', format='csv')
  if (!missing(arms)) .params[['arms']] <- paste(arms, collapse=',')
  x <- httr::POST(url=rcon$url, body=.params, config=rcon$config)
  
  if (x$status_code == "200")
    return(read.csv(textConnection(as.character(x)), stringsAsFactors=FALSE))
  #*** For classic projects, we want to avoid throwing a disruptive error. Instead, we 
  #*** return the message that indicates this is a classic project.
  else if (x$status_code == "400" & as.character(x) %in% 
             c("You cannot export form/event mappings for classic projects",
               "ERROR: You cannot export form/event mappings for classic projects")) 
    paste0(x$status_code, ": ", as.character(x))
  else (stop(paste0(x$status_code, ": ", as.character(x))))
}
