#' @name exportMappings 
#' @title Exports the Event-Form Mappings for a Project
#' @description Retrieve a data frame giving the events-form mapping for a project.
#' 
#' @param rcon A REDCap connection object as generated by \code{redcapConnection}.
#' @param arms A vector of arm numbers that you wish to pull events for (by default,
#'   all events are pulled) 
#' @param ... Arguments to be passed to other methods
#' @param error_handling An option for how to handle errors returned by the API.
#'   see \code{\link{redcap_error}}
#' 
#' @details The data frame that is returned shows the arm number, unique 
#' event name, and forms mapped in a project.
#'  
#' If the project information reports that the project is not longitudinal, 
#' a data frame with 0 rows is returned without calling the API. 
#' 
#' @section REDCap API Documentation:
#' This function allows you to export the instrument-event mappings for a project 
#' (i.e., how the data collection instruments are designated for certain events in a 
#' longitudinal project).
#' 
#' NOTE: this only works for longitudinal projects
#' 
#' @section REDCap Version:
#' 5.8.2+ (and earlier, but we don't know how much earlier)
#' 
#' @section Known REDCap Limitations: 
#' None
#'  
#' @author Benjamin Nutter
#' 
#' @references 
#' Please refer to your institution's API documentation.
#' 
#' Additional details on API parameters are found on the package wiki at
#' \url{https://github.com/vubiostat/redcapAPI/wiki/REDCap-API-Parameters}
#' 
#' @export 

exportMappings <- function(rcon, 
                           arms, 
                           ...){
  UseMethod("exportMappings")
}

#' @rdname exportMappings
#' @export

exportMappings.redcapApiConnection <- function(rcon, 
                                               arms = NULL, 
                                               ...,
                                               error_handling = getOption("redcap_error_handling")){
  coll <- checkmate::makeAssertCollection()
  
  checkmate::assert_class(x = rcon,
                          classes = "redcapApiConnection",
                          add = coll)
  
  checkmate::assert_character(x = arms,
                              null.ok = TRUE,
                              add = coll)
  
  error_handling <- checkmate::matchArg(x = error_handling,
                                        choices = c("null", "error"),
                                        add = coll)
  
  checkmate::reportAssertions(coll)
  
  if (rcon$projectInformation()$is_longitudinal == 0){
    return(data.frame(arm_num = numeric(0), 
                      unique_event_name = character(0), 
                      form = character(0)))
  }
  
  body <- list(token = rcon$token, 
               content = 'formEventMapping', 
               format = 'csv')
  
  if (!is.null(arms)) body[['arms']] <- paste(arms, collapse=',')
  
  x <- httr::POST(url = rcon$url, 
                  body = body, 
                  config = rcon$config)
  
  if (x$status_code != 200) return(redcap_error(x, error_handling))
  
  utils::read.csv(text = as.character(x), 
                  stringsAsFactors = FALSE, 
                  na.strings = "")
}
