#' @name exportEvents
#' @title Export the Events for a Project
#' 
#' @description Retrieve a data frame giving the users, expiration dates,
#' and data access privileges for each user.
#'
#' @param rcon A REDCap connection object as generated by \code{redcapConnection}.
#' @param arms A numeric vector or arm numbers to retrieve. Will also accept
#'   a character vector, but will coerce to numeric before processing.
#' @param ... Arguments to be passed to other methods.
#' @param error_handling An option for how to handle errors returned by the API.
#'   see \code{\link{redcap_error}} 
#' @param config \code{list} Additional configuration parameters to pass to 
#'   \code{\link[httr]{POST}}. These are appended to any parameters in 
#'   \code{rcon$config}.
#' @param api_param \code{list} Additional API parameters to pass into the
#'   body of the API call. This provides users to execute calls with options
#'   that may not otherwise be supported by \code{redcapAPI}.
#' 
#' @details
#' It is not sufficient to make the project a longitudinal project. The
#' project must satisfy one of two conditions: 1) have at least two arms and
#' one event defined; or 2) have one arm and at least two events defined. If 
#' neither of these conditions are satisfied, the API will return a message
#' such as \code{ERROR: You cannot export arms for classic projects}, an 
#' error message that isn't as descriptive of the nature of the problem as 
#' we might like.
#' 
#' If the project information reports that the project is not longitudinal, 
#' a data frame with 0 rows is returned without calling the API. 
#' 
#' @section REDCap API Documentation:
#' This function allows you to export the events for a project
#' 
#' NOTE: this only works for longitudinal projects.
#' 
#' @section REDCap Version:
#' 5.8.2+ 
#' 
#' @section Known REDCap Limitations:
#' None 
#' 
#' @return Returns a data frame with six columns
#' \itemize{
#'   \item{\code{event_name} }{The desciptive name of the event.}
#'   \item{\code{arm_num} }{The arm number in which the event occurs.}
#'   \item{\code{day_offset} }{The days offset from the first event.}
#'   \item{\code{offset_min} }{The minimum offset value.}
#'   \item{\code{offset_max} }{The maximium offset value.}
#'   \item{\code{unique_event_name} }{A unique event identifying name.}
#' }
#'
#' @author Benjamin Nutter
#'
#' @references
#' Please refer to your institution's API documentation.
#'
#' Additional details on API parameters are found on the package wiki at
#' \url{https://github.com/vubiostat/redcapAPI/wiki/REDCap-API-Parameters}
#' 
#' @export

exportEvents <- function(rcon, 
                         ...){
  UseMethod("exportEvents")
}

#' @rdname exportEvents
#' @export

exportEvents.redcapApiConnection <- function(rcon, 
                                             arms           = NULL, 
                                             ...,
                                             error_handling = getOption("redcap_error_handling"), 
                                             config         = list(), 
                                             api_param      = list()){
  
  if (is.character(arms)) arms <- as.numeric(arms)
  
   ##################################################################
  # Argument Validation 
  coll <- checkmate::makeAssertCollection()
  
  checkmate::assert_class(x = rcon,
                          classes = "redcapApiConnection",
                          add = coll)
  
  checkmate::assert_integerish(x = arms,
                               null.ok = TRUE,
                               add = coll)
  
  error_handling <- checkmate::matchArg(x = error_handling, 
                                        choices = c("null", "error"),
                                        .var.name = "error_handling",
                                        add = coll)
  
  checkmate::assert_list(x = config, 
                         names = "named", 
                         add = coll)
  
  checkmate::assert_list(x = api_param, 
                         names = "named", 
                         add = coll)
  
  checkmate::reportAssertions(coll)
  
  Arms <- rcon$arms()
  
  checkmate::assert_subset(x = arms, 
                           choices = Arms$arm_num, 
                           add = coll)
  
  checkmate::reportAssertions(coll)
  
   ##################################################################
  # Return for Classical projects
  
  if (rcon$projectInformation()$is_longitudinal == 0){
    return(REDCAP_EVENT_STRUCTURE) # Defined in redcapDataStructure.R
  }
  
   ##################################################################
  # Make the Body List
  
  body <- list(token = rcon$token, 
               content = 'event', 
               format = 'csv', 
               returnFormat = 'csv')
  body <- c(body, 
            vectorToApiBodyList(arms, "arms"))
  
  body <- body[lengths(body) > 0]
  
   ##################################################################
  # Call the API
  
  response <- makeApiCall(rcon, 
                          body = c(body, api_param), 
                          config = config)
  
  if (response$status_code != 200) return(redcap_error(response, error_handling))

  if (trimws(as.character(response)) == ""){
    REDCAP_EVENT_STRUCTURE
  } else {
    utils::read.csv(text = as.character(response),
                    stringsAsFactors = FALSE,
                    na.strings = "")
  }
}
