% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pffr-ffpc.R
\name{ffpc}
\alias{ffpc}
\title{Construct a PC-based function-on-function regression term}
\usage{
ffpc(
  X,
  yind = NULL,
  xind = seq(0, 1, length = ncol(X)),
  splinepars = list(bs = "ps", m = c(2, 1), k = 8),
  decomppars = list(pve = 0.99, useSymm = TRUE),
  npc.max = 15
)
}
\arguments{
\item{X}{an n by \code{ncol(xind)} matrix of function evaluations \eqn{X_i(s_{i1}),\dots, X_i(s_{iS})}; \eqn{i=1,\dots,n}.}

\item{yind}{\emph{DEPRECATED} used to supply matrix (or vector) of indices of evaluations of \eqn{Y_i(t)}, no longer used.}

\item{xind}{matrix (or vector) of indices of evaluations of \eqn{X_i(t)}, defaults to \code{seq(0, 1, length=ncol(X))}.}

\item{splinepars}{optional arguments supplied to the \code{basistype}-term. Defaults to a cubic
B-spline with first difference penalties and 8 basis functions for each \eqn{\tilde \beta_k(t)}.}

\item{decomppars}{parameters for the FPCA performed with \code{\link{fpca.sc}}.}

\item{npc.max}{maximal number \eqn{K} of FPCs to use, regardless of \code{decomppars}; defaults to 15}
}
\value{
A list containing the necessary information to construct a term to be included in a \code{mgcv::gam}-formula.
}
\description{
Defines a term \eqn{\int X_i(s)\beta(t,s)ds}
for inclusion in an \code{mgcv::gam}-formula (or \code{bam} or \code{gamm} or \code{gamm4:::gamm4}) as constructed
by \code{\link{pffr}}.
}
\details{
In contrast to \code{\link{ff}}, \code{ffpc}
does an FPCA decomposition \eqn{X(s) \approx \sum^K_{k=1} \xi_{ik} \Phi_k(s)} using \code{\link{fpca.sc}} and
represents \eqn{\beta(t,s)} in the function space spanned by these \eqn{\Phi_k(s)}.
That is, since
\deqn{\int X_i(s)\beta(t,s)ds = \sum^K_{k=1} \xi_{ik} \int \Phi_k(s) \beta(s,t) ds = \sum^K_{k=1} \xi_{ik} \tilde \beta_k(t),}
the function-on-function term can be represented as a sum of \eqn{K} univariate functions \eqn{\tilde \beta_k(t)} in \eqn{t} each multiplied by the FPC
scores \eqn{\xi_{ik}}. The truncation parameter \eqn{K} is chosen as described in \code{\link{fpca.sc}}.
Using this instead of \code{ff()} can be beneficial if the covariance operator of the \eqn{X_i(s)}
has low effective rank (i.e., if \eqn{K} is small). If the covariance operator of the \eqn{X_i(s)}
is of (very) high rank, i.e., if \eqn{K} is large, \code{ffpc()} will not be very efficient.

To reduce model complexity, the \eqn{\tilde \beta_k(t)} all have a single joint smoothing parameter
(in \code{mgcv}, they get the same \code{id}, see \code{\link[mgcv]{s}}).\cr

Please see \code{\link[refund]{pffr}} for details on model specification and
implementation.
}
\examples{
\dontrun{
set.seed(1122)
n <- 55
S <- 60
T <- 50
s <- seq(0,1, l=S)
t <- seq(0,1, l=T)

#generate X from a polynomial FPC-basis:
rankX <- 5
Phi <- cbind(1/sqrt(S), poly(s, degree=rankX-1))
lambda <- rankX:1
Xi <- sapply(lambda, function(l)
            scale(rnorm(n, sd=sqrt(l)), scale=FALSE))
X <- Xi \%*\% t(Phi)

beta.st <- outer(s, t, function(s, t) cos(2 * pi * s * t))

y <- (1/S*X) \%*\% beta.st + 0.1 * matrix(rnorm(n * T), nrow=n, ncol=T)

data <- list(y=y, X=X)
# set number of FPCs to true rank of process for this example:
m.pc <- pffr(y ~ c(1) + 0 + ffpc(X, yind=t, decomppars=list(npc=rankX)),
        data=data, yind=t)
summary(m.pc)
m.ff <- pffr(y ~ c(1) + 0 + ff(X, yind=t), data=data, yind=t)
summary(m.ff)

# fits are very similar:
all.equal(fitted(m.pc), fitted(m.ff))

# plot implied coefficient surfaces:
layout(t(1:3))
persp(t, s, t(beta.st), theta=50, phi=40, main="Truth",
    ticktype="detailed")
plot(m.ff, select=1, pers=TRUE, zlim=range(beta.st), theta=50, phi=40,
    ticktype="detailed")
title(main="ff()")
ffpcplot(m.pc, type="surf", auto.layout=FALSE, theta = 50, phi = 40)
title(main="ffpc()")

# show default ffpcplot:
ffpcplot(m.pc)
}
}
\author{
Fabian Scheipl
}
