% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/remify.R
\name{remify}
\alias{remify}
\title{Process a Relational Event History}
\usage{
remify(
  edgelist,
  directed = TRUE,
  ordinal = FALSE,
  model = c("tie", "actor"),
  actors = NULL,
  types = NULL,
  riskset = c("full", "active", "manual"),
  origin = NULL,
  omit_dyad = NULL,
  ncores = 1L
)
}
\arguments{
\item{edgelist}{the relational event history. An object of class \code{\link[base]{data.frame}} with first three columns corresponding to time, and actors forming the dyad. The first three columns will be re-named "time", "actor1", "actor2" (where, for directed networks, "actor1" corresponds to the sender and "actor2" to the receiver of the relational event). Optional columns that can be supplied are: `type` and `weight`. If one or both exist in \code{edgelist}, they have to be named accordingly.}

\item{directed}{logical value indicating whether events are directed (\code{TRUE}) or undirected (\code{FALSE}). (default value is \code{TRUE})}

\item{ordinal}{logical value indicating whether only the order of events matters in the model (\code{TRUE}) or also the waiting time must be considered in the model (\code{FALSE}). (default value is \code{FALSE})}

\item{model}{can be "tie" or "actor" oriented modeling. This argument plays a fundamental role when \code{omit_dyad} is supplied. Indeed, when actor-oriented modeling, the dynamic risk set will consist of two risk sets objects (senders' and dyads' risk sets). In the tie-oriented model the function will return a dynamic risk set referred at a dyad-level.}

\item{actors}{[\emph{optional}] character vector of actors' names that may be observed interacting in the network. If \code{NULL} (default), actors' names will be taken from the input edgelist.}

\item{types}{[\emph{optional}] character vector of event types that may occur in the network. If \code{NULL} (default), types' names will be taken from the input edgelist.}

\item{riskset}{[\emph{optional}] character value indicating the type of risk set to process: \code{riskset = "full"} (default) consists of all the possible dyadic events given the number of actors (and the number of event types) and it mantains the same structure over time. \code{riskset = "active"} considers at risk only the observed dyads and it mantains the same structure over time. \code{riskset = "manual"}, allows the risk set to have a structure that is user-defined, and it is based on the instructions supplied via the argument \code{omit_dyad}. This type of risk set allows for time-varying risk set, in which, for instance, subset of actors can interact only at specific time windows, or events of a specific type (sentiment) can't be observed within time intervals that are defined by the user.}

\item{origin}{[\emph{optional}] starting time point of the observaton period (default is \code{NULL}). If it is supplied, it must have the same class of the `time` column in the input \code{edgelist}.}

\item{omit_dyad}{[\emph{optional}] list of lists. Each list refers to one risk set modification and must have two objects: a first object named `time`, that is a vector of two values defining the first and last time point of the time window where to apply the change to the risk set and a second object, named `dyad`, which is a \code{\link[base]{data.frame}} where dyads to be removed are supplied in the format \code{actor1,actor2,type} (by row). The \code{NA} value can be used to remove multiple objects from the risk set at once with one risk set modification list (see Details).}

\item{ncores}{[\emph{optional}] number of cores used in the parallelization of the processing functions. (default is \code{1}).}
}
\value{
'remify' S3 object, list of: number of events (`M`), number of actors (`N`), number of event types (if present, `C`), number of dyads (`D`, and also `activeD` if `riskset="active"`), vector of inter-event times (waiting times between two subsequent events), processed input edgelist as `data.frame`, processed `omit_dyad` object as `list`. The function returns also several attributes that make efficient the processing of the data for future analysis. For more details about the function, input arguments, output, attributes and methods, please read \code{vignette(package="remify",topic="remify")}.
}
\description{
A function that processes raw relational event history data and returns a S3 object of class 'remify' which is used as input in other functions inside 'remverse'.
}
\details{
In \code{omit_dyad}, the \code{NA} value can be used to remove multiple objects from the risk set at once with one risk set modification list. For example, to remove all events with sender equal to actor “A” add a list with two objects \code{time = c(NA, NA)} and \code{dyad = data.frame(actor1 = A, actor2 = NA, type = NA)} to the \code{omit_dyad} list. For more details about
}
\examples{

# load package and random network 'randomREH'
library(remify)
data(randomREH)

# first events in the sequence
head(randomREH$edgelist)

# actor's names
randomREH$actors

# event type's names
randomREH$types

# start time of the study (origin)
randomREH$origin

# list of changes of the risk set: each one is a list of:
# 'time' (indicating the time window where to apply the risk set reduction)
# 'dyad' (a data.frame describing the dyads to remove from the risk set 
# during the time window specified in 'time')
str(randomREH$omit_dyad)

# -------------------------------------- #
#  processing for tie-oriented modeling  #
# -------------------------------------- #

tie_randomREH <- remify(edgelist = randomREH$edgelist,
       directed = TRUE,
       ordinal = FALSE,
       model = "tie",
       actors = randomREH$actors,
       types = randomREH$types,
       riskset = "manual",
       origin = randomREH$origin,
       omit_dyad = randomREH$omit_dyad)

# summary
summary(tie_randomREH)

# dimensions of the processed network
dim(tie_randomREH)

# Which ID is assigned to the actors with names "Francesca" and "Kayla"?
getActorID(x = tie_randomREH, actorName = c("Francesca","Kayla"))

# Which ID is assigned to the event type "conflict"?
getTypeID(x = tie_randomREH, typeName = "conflict")

# Find dyad composition (names of actor1, actor2 and type) from the dyad ID: c(1,380,760,1140)
getDyad(x = tie_randomREH, dyadID = c(1,380,760,1140))

# visualize descriptive measures of relational event data
# plot(x = tie_randomREH)

# -------------------------------------- #
# processing for actor-oriented modeling #
# -------------------------------------- #

# loading network 'randomREHsmall'
data(randomREHsmall)

# processing small random network
actor_randomREH <- remify(edgelist = randomREHsmall$edgelist,
       directed = TRUE,
       ordinal = FALSE,
       model = "actor",
       actors = randomREHsmall$actors,
       origin = randomREHsmall$origin)
       
# summary
summary(actor_randomREH)

# dimensions of the processed network
dim(actor_randomREH)

# ------------------------------------ #
# for more information about remify()  #
# check: vignette(package="remify")    #
# ------------------------------------ #
 
}
