#' Get a cleaned version of GHCND data from a single weather site
#'
#' This function uses ftp to access the Global Historical Climatology Network
#' daily weather data from NOAA's FTP server for a single weather monitor site.
#' It requires the site identification number for that site and will pull the
#' entire weather dataset for the site. It will then clean this data to convert
#' it to a tidier format and will also, if requested, filter it to a certain
#' date range and to certain weather variables.
#'
#' @export
#' @inheritParams ghcnd
#' @param date_min A character string giving the earliest
#'    date of the daily weather time series that the user would
#'    like in the final output. This character string should be formatted as
#'    "yyyy-mm-dd". If not specified, the default is to keep all daily data for
#'    the queried weather site from the earliest available date.
#' @param date_max A character string giving the latest
#'    date of the daily weather time series that the user would
#'    like in the final output. This character string should be formatted as
#'    "yyyy-mm-dd". If not specified, the default is to keep all daily data for
#'    the queried weather site through the most current available date.
#' @param var A character vector specifying either `"all"` (pull all
#'    available weather parameters for the site) or the weather parameters to
#'    keep in the final data (e.g., `c("TMAX", "TMIN")` to only keep
#'    maximum and minimum temperature). Example choices for this argument
#'    include:
#'
#'    - `PRCP`: Precipitation, in tenths of millimeters
#'    - `TAVG`: Average temperature, in tenths of degrees Celsius
#'    - `TMAX`: Maximum temperature, in tenths of degrees Celsius
#'    - `TMIN`: Minimum temperature, in tenths of degrees Celsius
#'
#'    A full list of possible weather variables is available in NOAA's README
#'    file for the GHCND data
#'    (<https://www1.ncdc.noaa.gov/pub/data/ghcn/daily/readme.txt>).
#'    Most weather stations will only have a small subset of all the possible
#'    weather variables, so the data generated by this function may not include
#'    all of the variables the user specifies through this argument.
#'
#' @return A list object with slots for each of the available specified
#'    weather variables. Each element in the list is a separate time series
#'    dataframe with daily observations, as well as flag values, for one of
#'    the weather variables. The flag values give information on the quality
#'    and source of each observation; see the NOAA README file linked above
#'    for more information. Each data.frame is sorted by date, with the
#'    earliest date first.
#'
#' @author Scott Chamberlain \email{myrmecocystus@@gmail.com},
#' Adam Erickson \email{adam.erickson@@ubc.ca}
#'
#' @note This function calls [ghcnd()], which will download and save
#'    data from all available dates and weather variables for the queried
#'    weather station. The step of limiting the dataset to only certain dates
#'    and / or weather variables, using the `date_min`, `date_max`,
#'    and `var` arguments, does not occur until after the full data has
#'    been pulled.
#'
#' @details
#' Messages are printed to the console about file path, file last modified time
#' which you can suppress with `suppressMessages()`
#'
#' @seealso [meteo_pull_monitors()], [meteo_tidy_ghcnd()]
#'
#' @examples \dontrun{
#' # Search based on variable and/or date
#' ghcnd_search("AGE00147704", var = "PRCP")
#' ghcnd_search("AGE00147704", var = "PRCP", date_min = "1920-01-01")
#' ghcnd_search("AGE00147704", var = "PRCP", date_max = "1915-01-01")
#' ghcnd_search("AGE00147704", var = "PRCP", date_min = "1920-01-01",
#'              date_max = "1925-01-01")
#' ghcnd_search("AGE00147704", date_min = "1920-01-01", date_max = "1925-01-01")
#' ghcnd_search("AGE00147704", var = c("PRCP","TMIN"))
#' ghcnd_search("AGE00147704", var = c("PRCP","TMIN"), date_min = "1920-01-01")
#' ghcnd_search("AGE00147704", var = "adfdf")
#'
#' # refresh the cached file
#' ghcnd_search("AGE00147704", var = "PRCP", refresh = TRUE)
#' }
ghcnd_search <- function(stationid, date_min = NULL, date_max = NULL,
                         var = "all", refresh = FALSE, ...) {
  out <- ghcnd(stationid, refresh = refresh, ...)
  dat <- ghcnd_splitvars(out)

  # date check
  message("file min/max dates: ", min(dat[[1]]$date), " / ", max(dat[[1]]$date))

  possvars <- paste0(names(dat), collapse = ", ")
  if (any(var != "all")) {
    vars_null <- sort(tolower(var))[!sort(tolower(var)) %in% sort(names(dat))]
    dat <- dat[tolower(var)]
  }
  if (any(sapply(dat, is.null))) {
    dat <- noaa_compact(dat)
    warning(
      sprintf("%s not in the dataset\nAvailable variables: %s",
              paste0(vars_null, collapse = ", "), possvars), call. = FALSE)
  }
  if (!is.null(date_min)) {
    dat <- lapply(dat, function(z) dplyr::filter(z, date >= date_min))
  }
  if (!is.null(date_max)) {
    dat <- lapply(dat, function(z) dplyr::filter(z, date <= date_max))
  }
  # arrange by "day", not "month"
  dat <- lapply(dat, function(z) dplyr::arrange(z, date))
  return(dat)
}

#' Get all GHCND data from a single weather site
#'
#' This function uses ftp to access the Global Historical Climatology Network
#' daily weather data from NOAA's FTP server for a single weather site. It
#' requires the site identification number for that site and will pull the
#' entire weather dataset for the site.
#'
#' @export
#' @param stationid (character) A character string giving the identification of
#' the weather station for which the user would like to pull data. To get a full
#' and current list of stations, the user can use the \code{\link{ghcnd_stations}}
#' function. To identify stations within a certain radius of a location, the
#' user can use the \code{\link{meteo_nearby_stations}} function.
#' @param path (character) a path to a file with a \code{.dly} extension - already
#' downloaded on your computer
#' @param refresh (logical) If \code{TRUE} force re-download of data.
#' Default: \code{FALSE}
#' @param ... In the case of \code{ghcnd} additional curl options to pass
#' through to \code{\link[crul]{HttpClient}}. In the case of \code{ghcnd_read}
#' further options passed on to \code{read.csv}
#'
#' @return A tibble (data.frame) which contains data pulled from NOAA's FTP
#' server for the queried weather site. A README file with more information
#' about the format of this file is available from NOAA
#' (\url{http://www1.ncdc.noaa.gov/pub/data/ghcn/daily/readme.txt}).
#' This file is formatted so each line of the file gives the daily weather
#' observations for a single weather variable for all days of one month of
#' one year. In addition to measurements, columns are included for certain
#' flags, which add information on observation sources and quality and are
#' further explained in NOAA's README file for the data.
#'
#' @details  This function saves the full set of weather data for the queried
#' site locally in the directory specified by the \code{path} argument.
#'
#' You can access the path for the cached file via \code{attr(x, "source")}
#'
#' You can access the last modified time for the cached file via
#' \code{attr(x, "file_modified")}
#'
#' Messages are printed to the console about file path and file last modified time
#' which you can suppress with \code{suppressMessages()}
#'
#' @author Scott Chamberlain \email{myrmecocystus@@gmail.com},
#' Adam Erickson \email{adam.erickson@@ubc.ca}
#'
#' @seealso To generate a weather dataset for a single weather site that has
#' been cleaned to a tidier weather format, the user should use the
#' \code{\link{ghcnd_search}} function, which calls \code{\link{ghcnd}} and then
#' processes the output, or \code{\link{meteo_tidy_ghcnd}}, which wraps the
#' \code{\link{ghcnd_search}} function to output a tidy dataframe. To pull
#' GHCND data from multiple monitors, see \code{\link{meteo_pull_monitors}}.
#' @note See [ghcnd_cache] for managing cached files
#' @examples \dontrun{
#' # Get data
#' ghcnd(stationid = "AGE00147704")
#'
#' stations <- ghcnd_stations()
#' ghcnd(stations$id[40])
#'
#' library("dplyr")
#' ghcnd(stations$id[80300]) %>% select(id, element) %>% slice(1:3)
#'
#' # manipulate data
#' ## using built in fxns
#' dat <- ghcnd(stationid = "AGE00147704")
#' (alldat <- ghcnd_splitvars(dat))
#'
#' ## using dplyr
#' library("dplyr")
#' dat <- ghcnd(stationid = "AGE00147704")
#' filter(dat, element == "PRCP", year == 1909)
#'
#' # refresh the cached file
#' ghcnd(stationid = "AGE00147704", refresh = TRUE)
#'
#' # Read in a .dly file you've already downloaded
#' path <- system.file("examples/AGE00147704.dly", package = "rnoaa")
#' ghcnd_read(path)
#' }
ghcnd <- function(stationid, refresh = FALSE, ...) {
  csvpath <- ghcnd_local(stationid)
  if (!is_ghcnd(x = csvpath) || refresh) {
    res <- ghcnd_GET(stationid, ...)
  } else {
    cache_mssg(csvpath)
    res <- read.csv(csvpath, stringsAsFactors = FALSE,
      colClasses = ghcnd_col_classes)
  }
  fi <- file.info(csvpath)
  res <- remove_na_row(res) # remove trailing row of NA's
  res <- tibble::as_tibble(res)
  attr(res, 'source') <- csvpath
  attr(res, 'file_modified') <- fi[['mtime']]
  return(res)
}

remove_na_row <- function(x) {
  if (!any(x[,1] == "NA") && !any(is.na(x[,1]))) return(x)
  return(x[!as.character(x[,1]) %in% c("NA", NA_character_), ])
}

#' @export
#' @rdname ghcnd
ghcnd_read <- function(path, ...) {
  if (!file.exists(path)) stop("file does not exist")
  if (!grepl("\\.dly", path)) {
    warning(".dly extension not detected; file may not be read correctly")
  }
  res <- read.csv(path, stringsAsFactors = FALSE,
                  colClasses = ghcnd_col_classes, ...)
  res <- tibble::as_tibble(res)
  attr(res, 'source') <- path
  return(res)
}


## helpers -------
ghcnd_col_classes <- c(
  "character", "integer", "integer", "character",
  rep(c("integer", "character", "character", "character"), times = 31)
)

fm <- function(n) {
  gsub("\\s", "0", n)
}

#' Get information on the GHCND weather stations
#'
#' This function returns an object with a dataframe with meta-information about
#' all available GHCND weather stations.
#'
#' @export
#' @inheritParams ghcnd
#'
#' @return This function returns a tibble (dataframe) with a weather
#' station on each row with the following columns:
#' \itemize{
#'    \item \code{id}: The weather station's ID number. The first two letters
#'    denote the country (using FIPS country codes).
#'    \item \code{latitude}: The station's latitude, in decimal degrees.
#'    Southern latitudes will be negative.
#'    \item \code{longitude}: The station's longitude, in decimal degrees.
#'    Western longitudes will be negative.
#'    \item \code{elevation}: The station's elevation, in meters.
#'    \item \code{name}: The station's name.
#'    \item \code{gsn_flag}: "GSN" if the monitor belongs to the GCOS Surface
#'    Network (GSN). Otherwise either blank or missing.
#'    \item \code{wmo_id}: If the station has a WMO number, this column gives
#'    that number. Otherwise either blank or missing.
#'    \item \code{element}: A weather variable recorded at some point during
#'    that station's history. See the link below in "References" for definitions
#'    of the abbreviations used for this variable.
#'    \item \code{first_year}: The first year of data available at that station
#'    for that weather element.
#'    \item \code{last_year}: The last year of data available at that station
#'    for that weather element.
#' }
#'
#' If a weather station has data on more than one weather variable, it will
#' be represented in multiple rows of this output dataframe.
#'
#' @note Since this function is pulling a large dataset by ftp, it may take
#' a while to run.
#'
#' @references
#'
#' For more documentation on the returned dataset, see
#' \url{http://www1.ncdc.noaa.gov/pub/data/ghcn/daily/readme.txt}.
#'
#' @examples \dontrun{
#' # Get stations, ghcnd-stations and ghcnd-inventory merged
#' (stations <- ghcnd_stations())
#'
#' library(dplyr)
#' # filter by state
#' stations %>% filter(state == "IL")
#' stations %>% filter(state == "OR")
#' # those without state values
#' stations %>% filter(state == "")
#' # filter by element
#' stations %>% filter(element == "PRCP")
#' # filter by id prefix
#' stations %>% filter(grepl("^AF", id))
#' stations %>% filter(grepl("^AFM", id))
#' # filter by station long name
#' stations %>% filter(name == "CALLATHARRA")
#' }
ghcnd_stations <- function(refresh = FALSE, ...) {
  assert(refresh, "logical")
  stopifnot(length(refresh) == 1)
  sta <- get_stations(refresh, ...)
  inv <- get_inventory(refresh, ...)
  df <- merge(sta, inv[, -c(2, 3)], by = "id")
  tibble::as_tibble(df[stats::complete.cases(df), ])
}

get_stations <- function(refresh = FALSE, ...) {
  ff <- file.path(ghcnd_cache$cache_path_get(), "ghcnd-stations.txt")
  ffrds <- file.path(ghcnd_cache$cache_path_get(), "ghcnd-stations.rds")
  if (file.exists(ffrds) && !refresh) {
    cache_mssg(ffrds)
    return(readRDS(ffrds))
  } else {
    if (file.exists(ff)) unlink(ff)
    if (file.exists(ffrds)) unlink(ffrds)
    res <- GET_retry(
      "ftp://ftp.ncdc.noaa.gov/pub/data/ghcn/daily/ghcnd-stations.txt",
      disk = ff, ...)
    df <- read.fwf(as_tc_p(res),
      widths = c(11, 9, 11, 7, 2, 31, 5, 10),
      header = FALSE, strip.white = TRUE, comment.char = "",
      stringsAsFactors = FALSE)
    nms <- c("id","latitude", "longitude", "elevation",
             "state", "name", "gsn_flag", "wmo_id")
    df <- stats::setNames(df, nms)
    saveRDS(df, file = ffrds)
    unlink(ff)
    return(df)
  }
}

get_inventory <- function(refresh = FALSE, ...) {
  gg <- file.path(ghcnd_cache$cache_path_get(), "ghcnd-inventory.txt")
  ggrds <- file.path(ghcnd_cache$cache_path_get(), "ghcnd-inventory.rds")
  if (file.exists(ggrds) && !refresh) {
    cache_mssg(ggrds)
    return(readRDS(ggrds))
  } else {
    if (file.exists(gg)) unlink(gg)
    if (file.exists(ggrds)) unlink(ggrds)
    res <- GET_retry(
      "ftp://ftp.ncdc.noaa.gov/pub/data/ghcn/daily/ghcnd-inventory.txt",
      disk = gg, ...)
    df <- read.fwf(as_tc_p(res),
      widths = c(11, 9, 10, 5, 5, 5),
      header = FALSE, strip.white = TRUE, comment.char = "",
      stringsAsFactors = FALSE)
    nms <- c("id","latitude", "longitude",
             "element", "first_year", "last_year")
    df <- stats::setNames(df, nms)
    saveRDS(df, file = ggrds)
    unlink(gg)
    return(df)
  }
}

strex <- function(x) str_extract_(x, "[0-9]+")

#' Split variables in data returned from \code{ghcnd}
#'
#' This function is a helper function for \code{\link{ghcnd_search}}. It helps
#' with cleaning up the data returned from \code{\link{ghcnd}}, to get it in a
#' format that is easier to work with.
#'
#' @param x An object returned from \code{\link{ghcnd}}.
#'
#' @author Scott Chamberlain \email{myrmecocystus@@gmail.com},
#' Adam Erickson \email{adam.erickson@@ubc.ca}
#'
#' @export
ghcnd_splitvars <- function(x){
  if (!inherits(x, "data.frame")) stop("input must be a data.frame", call. = FALSE)
  if (!"id" %in% names(x)) stop("input not of correct format", call. = FALSE)
  x <- x[!is.na(x$id), ]
  out <- lapply(as.character(unique(x$element)), function(y){
    ydat <- x[ x$element == y, ]

    dd <- ydat %>%
      dplyr::select(-dplyr::contains("FLAG")) %>%
      tidyr::gather(var, value, -id, -year, -month, -element) %>%
      dplyr::mutate(
        day = strex(var),
        date = as.Date(sprintf("%s-%s-%s", year, month, day), "%Y-%m-%d")) %>%
      dplyr::filter(!is.na(date)) %>%
      dplyr::select(
        -element,
        -var,
        -year,
        -month,
        -day)
    dd <- stats::setNames(dd, c("id", tolower(y), "date"))

    mflag <- ydat %>%
      dplyr::select(-dplyr::contains("VALUE"), -dplyr::contains("QFLAG"),
                    -dplyr::contains("SFLAG")) %>%
      tidyr::gather(var, value, -id, -year, -month, -element) %>%
      dplyr::mutate(
        day = strex(var),
        date = as.Date(sprintf("%s-%s-%s", year, month, day), "%Y-%m-%d")) %>%
      dplyr::filter(!is.na(date)) %>%
      dplyr::select(value) %>%
      dplyr::rename(mflag = value)

    qflag <- ydat %>%
      dplyr::select(-dplyr::contains("VALUE"), -dplyr::contains("MFLAG"),
                    -dplyr::contains("SFLAG")) %>%
      tidyr::gather(var, value, -id, -year, -month, -element) %>%
      dplyr::mutate(
        day = strex(var),
        date = as.Date(sprintf("%s-%s-%s", year, month, day), "%Y-%m-%d")) %>%
      dplyr::filter(!is.na(date)) %>%
      dplyr::select(value) %>%
      dplyr::rename(qflag = value)

    sflag <- ydat %>%
      dplyr::select(-dplyr::contains("VALUE"), -dplyr::contains("QFLAG"),
                    -dplyr::contains("MFLAG")) %>%
      tidyr::gather(var, value, -id, -year, -month, -element) %>%
      dplyr::mutate(
        day = strex(var),
        date = as.Date(sprintf("%s-%s-%s", year, month, day), "%Y-%m-%d")) %>%
      dplyr::filter(!is.na(date)) %>%
      dplyr::select(value) %>%
      dplyr::rename(sflag = value)

    dplyr::tbl_df(cbind(dd, mflag, qflag, sflag))
  })
  stats::setNames(out, tolower(unique(x$element)))
}

# ghcnd_mergevars <- function(x){
#   merge(x[[2]], x[[3]] %>% select(-id), by='date')
# }

#' Get meta-data on the GHCND daily data
#'
#' These function allow you to pull the current versions of certain meta-datasets
#' for the GHCND, including lists of country and state abbreviations used in some
#' of the weather station IDs and information about the current version of the
#' data.
#'
#' @inheritParams ghcnd
#'
#' @author Scott Chamberlain \email{myrmecocystus@@gmail.com},
#' Adam Erickson \email{adam.erickson@@ubc.ca}
#'
#' @details Functions:
#' \itemize{
#'  \item \code{ghcnd_version}: Get current version of GHCND data
#'  \item \code{ghcnd_states}: Get US/Canada state names and 2-letter codes
#'  \item \code{ghcnd_countries}: Get country names and 2-letter codes
#' }
#'
#' @examples \dontrun{
#' # Get metadata
#' ghcnd_states()
#' ghcnd_countries()
#' ghcnd_version()
#' }
#'
#' @export
ghcnd_states <- function(...){
  res <- GET_retry(
    "ftp://ftp.ncdc.noaa.gov/pub/data/ghcn/daily/ghcnd-states.txt", ...)
  df <- read.fwf(as_tc_p(res), widths = c(2, 27),
                 header = FALSE, strip.white = TRUE, comment.char = "",
                 stringsAsFactors = FALSE, col.names = c("code","name"))
  df[ -NROW(df) ,]
}

#' @export
#' @rdname ghcnd_states
ghcnd_countries <- function(...){
  res <- GET_retry(
    "ftp://ftp.ncdc.noaa.gov/pub/data/ghcn/daily/ghcnd-countries.txt", ...)
  df <- read.fwf(as_tc_p(res), widths = c(2, 47),
                 header = FALSE, strip.white = TRUE, comment.char = "",
                 stringsAsFactors = FALSE, col.names = c("code","name"))
  df[ -NROW(df) ,]
}

#' @export
#' @rdname ghcnd_states
ghcnd_version <- function(...){
  res <- GET_retry(
    "ftp://ftp.ncdc.noaa.gov/pub/data/ghcn/daily/ghcnd-version.txt", ...)
  rawToChar(res$content)
}

as_tc <- function(x) textConnection(enc2utf8(rawToChar(x)))
as_tc_p <- function(x) textConnection(x$parse("latin1"))

GET_retry <- function(url, ..., times = 3) {
  cliret <- crul::HttpClient$new(url)
  res <- suppressWarnings(cliret$get(...))
  if (res$status_code > 226) {
    message("Request failed - Retrying")
    stat <- 500
    i <- 0
    while (stat > 226 && i <= times) {
      i <- i + 1
      res <- suppressWarnings(cliret$get(...))
      stat <- res$status_code
    }
    if (res$status_code > 226) stop("Request failed, try again", call. = FALSE)
  }
  return(res)
}

ghcnd_zip <- function(x){
  "adf"
}

ghcnd_GET <- function(stationid, ...){
  ghcnd_cache$mkdir()
  fp <- ghcnd_local(stationid)
  cli <- crul::HttpClient$new(ghcnd_remote(stationid), opts = list(...))
  res <- suppressWarnings(cli$get())
  tt <- res$parse("UTF-8")
  vars <- c("id","year","month","element",
            do.call("c",
      lapply(1:31, function(x) paste0(c("VALUE","MFLAG","QFLAG","SFLAG"), x))))
  df <- read.fwf(textConnection(tt), c(11,4,2,4,rep(c(5,1,1,1), 31)),
                 na.strings = "-9999")
  df[] <- Map(function(a, b) {
    if (b == "integer") {
      a <- as.character(a)
    }
    suppressWarnings(eval(parse(text = paste0("as.", b)))(a))
  }, df, ghcnd_col_classes)
  dat <- stats::setNames(df, vars)
  write.csv(dat, fp, row.names = FALSE)
  return(dat)
}

ghcndbase <- function() "ftp://ftp.ncdc.noaa.gov/pub/data/ghcn/daily/all"
ghcnd_remote <- function(stationid) {
  file.path(ghcndbase(), paste0(stationid, ".dly"))
}
ghcnd_local <- function(stationid) {
  file.path(ghcnd_cache$cache_path_get(), paste0(stationid, ".dly"))
}
is_ghcnd <- function(x) if (file.exists(x)) TRUE else FALSE
str_extract_ <- function(string, pattern) {
  regmatches(string, regexpr(pattern, string))
}
str_extract_all_ <- function(string, pattern) {
  regmatches(string, gregexpr(pattern, string))
}
