% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plainMethod_exportDF.r
\name{exportDF}
\alias{exportDF}
\title{Export a Data Frame as HTML/TEX Code}
\usage{
exportDF(x, tex = FALSE, colnames = NULL, width = NULL, align = NULL,
  funHead = NULL, funCell = NULL, lines = TRUE, indent = 2)
}
\arguments{
\item{x}{The data frame being exported.}

\item{tex}{Logical. Allows to switch between generation of TEX code and HTML.}

\item{colnames}{Displayed column names. If \code{NULL}, the original names
of \code{x} are used. Otherwise it must be a named vector with element
names corresponding to column names in \code{x}. It is OK to supply
alternative names for selected columns only.}

\item{width}{Either \code{NULL} (all columns get equal width) or a named
vector with element names corresponding to column names in \code{x}. If
\code{tex == TRUE}, values (between 0 and 1) are needed for columns with
align code 'p' only. They are interpreted as a multiplier for '\\textwidth'.
If \code{tex == FALSE}, values (between 0 and 100) should be
supplied for all columns of \code{x}.}

\item{align}{Either \code{NULL} (to use automatic alignment) or a named
vector with element names corresponding to column names in \code{x}.
If \code{tex == FALSE} valid alignment codes are 'left', 'right', 'center'.
If \code{tex == TRUE} valid alignment codes are 'l', 'r', 'c', and 'p'. For
columns with code 'p' a corresponding value of \code{width} should be set.
It is OK to supply alignment codes for selected columns only.}

\item{funHead}{Either \code{NULL} or a list of functions whose names
correspond to column names of \code{x}. The functions should have a single
formal argument; the respective column names of \code{x} are used as the
actual arguments. It is OK to supply functions for selected columns only
(an empty function is applied to the remaining columns). See below for some
typical examples.}

\item{funCell}{Like \code{funHead} but these functions are applied to the
cells in columns rather that to the column names.}

\item{lines}{Logical. Switches table borders on/off.}

\item{indent}{Integer. Number of blanks used to indent the generated code.}
}
\value{
A character string (usually needs to be exported to a file).
}
\description{
Generates code to include tabular data in a tex document or web site.
}
\note{
The functions \code{funHead} and \code{funCell} are useful to apply
  formatting or character replacement. For example, one could use

  \code{function(x) {paste0("\\\\bold{",toupper(x),"}")}}

  to generate bold, uppercase column names in a TEX table.
}
\examples{
# Create example table
df <- data.frame(stringsAsFactors=FALSE, name= c("growth", "dead"),
  unit= c("1/d","1/d"), expression= c("r * N * (1 - N/K)"," d * N"))

# Export as TEX: header in bold, 1st colum in italics, last column as math
tex <- exportDF(df, tex=TRUE,
  colnames=c(expression="process rate expression"),
  width=c(expression=0.5),
  align=c(expression="p"),
  funHead=setNames(replicate(ncol(df),
    function(x){paste0("\\\\textbf{",x,"}")}),names(df)),
  funCell=c(name=function(x){paste0("\\\\textit{",x,"}")},
    expression=function(x){paste0("$",x,"$")})
)
cat(tex,"\\n")

# Export as HTML: non-standard colors are used for all columns
tf <- tempfile(fileext=".html")
write(x= exportDF(df, tex=FALSE,
  funHead=setNames(replicate(ncol(df),
    function(x){paste0("<font color='red'>",x,"</font>")}),names(df)),
  funCell=setNames(replicate(ncol(df),
    function(x){paste0("<font color='blue'>",x,"</font>")}),names(df))
), file=tf)
\dontrun{
  browseURL(tf)
  file.remove(tf)
}
}
\seealso{
The \code{xtable} packages provides similar functionality with
  more sophisticated options. Consider the 'pandoc' software do convert
  documents from one markup language to another one. Finally, consider the
  latex package 'datatools' for direct inclusion of delimited text files
  (e.g. produced by \code{write.table}) in tex documents.
}
\author{
David Kneis \email{david.kneis@tu-dresden.de}
}
