% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predictResources.R
\name{predictResources}
\alias{predictResources}
\title{predictResources}
\usage{
predictResources(presence.data = presence.data, absence.data = absence.data,
  sample.label = NULL, env.data = NULL)
}
\arguments{
\item{presence.data}{Object of class \emph{data.frame} with environmental variables for presence samples.}

\item{absence.data}{Object of class \emph{data.frame} with environmental variables for background samples.}

\item{sample.label}{Numeric or character vector with sample region labels. If missing, "presence.data" is assumed as one region.}

\item{env.data}{Object of class \emph{RasterStack} or \emph{RasterBrick} with environmental variables in \emph{presence.data} and \emph{absence.data}.}
}
\value{
A \emph{list}.
}
\description{
Spatially stratified predictive modeling of resource suitability based on presence/absence samples.
}
\details{
{Modeling of resource suitability using animal movement data following a recent paper (Remelgado et al, 2017). For each
unique label in \emph{sample.label}, the function keeps it for validation and uses the remaining samples for training. Then, the
function evaluates the performance of this model reporting (internally) on the number of true positives, false positives and the
number of validation and predicted cases for both presences and absences. Once all sample regions are used for validation, the
reported values are summed and used to derive the F1-measure. The F1-measure is estimated as \emph{2 * (P * R) / (P + R)} where
\emph{P} is the Precision (ratio of true positives within the number of predicted values) and \emph{R} is the Recall (ratio of
true positives within the number of validation samples). As a consequence, rather than reporting on an average performance, the
final performance assessment reported by \emph{predictResources} depicts an objective picture on how the model performed among the
different sets sample regions. This metric is provided for presences (\emph{presence.data}) and absences ({\emph{absence.data}})
separately offering an overview on the stability of the model. This analysis is performed using a Random Forest model as provided
within the \code{\link[caret]{train}} function of the caret package. The final predictive model is then derived with all samples.
The output of \emph{predictResources} is a list object consisting of:
\itemize{
 \item{\emph{f1} - \emph{data.frame} with final F1-measure for presences and absences.}
 \item{validation - \emph{data.frame} with region identifiers and validation sample count at each iteration.}
 \item{\emph{iteration.models} - List of models estimated at each iteration.}
 \item{\emph{final.model} - Final predictive model based on all samples.}
 \item{\emph{probabilities} - Predicted probability image. Given if \emph{env.data} is set.}}}
}
\examples{
\dontrun{

 require(rgdal)
 require(raster)
 require(sp)

 # read remote sensing data
 file <- list.files(system.file('extdata', '', package="rsMove"), 'ndvi.tif', full.names=TRUE)
 r.stk <- stack(file)

 # read movement data
 data(shortMove)

 # observation time
 obs.time <- strptime(paste0(shortMove@data$date, ' ', shortMove@data$time),
 format="\%Y/\%m/\%d \%H:\%M:\%S")

 # remove redundant samples
 shortMove <- moveReduce(xy=shortMove, obs.time=obs.time, img=rsStk)$points

 # retrieve remote sensing data for samples
 rsQuery <- extract(rsStk, shortMove)

 # identify unique sample regions
 label <- labelSample(xy=shortMove, agg.radius=90, pixel.res=rsStk)

 # select background samples
 ind <- which(!is.na(label)) # selected samples
 bSamples <- backSample(xy=shortMove[ind,], region.id=label[ind],
 img=rsStk, sampling.method='pca')

 # derive model predictions
 out <- predictResources(presence.data=rsQuery,
 absence.data=bSamples@data, sample.label=label, env.data=rsStk)

}
}
\references{
\href{10.1002/rse2.70}{Remelgado, R., Leutner, B., Safi, K., Sonnenschein, R., Kuebert, C. and Wegmann, M. (2017), Linking animal movement and remote sensing - mapping resource suitability from a remote sensing perspective. Remote Sens Ecol Conserv.}
}
\seealso{
\code{\link{sampleMove}} \code{\link{labelSample}} \code{\link{backSample}} \code{\link[caret]{train}}
}
