% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Composite.R
\name{Composite}
\alias{Composite}
\title{Compute composites}
\usage{
Composite(
  data,
  occ,
  time_dim = "time",
  space_dim = c("lon", "lat"),
  lag = 0,
  eno = FALSE,
  K = NULL,
  fileout = NULL,
  ncores = NULL
)
}
\arguments{
\item{data}{A numeric array containing two spatial and one temporal 
dimensions.}

\item{occ}{A vector of the occurrence time series of mode(s)/cluster(s).
The length should be the same as the temporal dimension in 'data'.
(*1) When one wants to composite all modes, e.g., all K = 3 clusters then 
  for example occurrences could look like: 1 1 2 3 2 3 1 3 3 2 3 2 2 3 2.
(*2) Otherwise for compositing only the 2nd mode or cluster of the above 
  example occurrences should look like 0 0 1 0 1 0 0 0 0 1 0 1 1 0 1.}

\item{time_dim}{A character string indicating the name of the temporal 
dimension in 'data'. The default value is 'time'.}

\item{space_dim}{A character vector indicating the names of the spatial 
dimensions in 'data'. The default value is c('lon', 'lat').}

\item{lag}{An integer indicating the lag time step. E.g., for lag = 2, 
+2 occurrences will be used (i.e., shifted 2 time steps forward). 
The default value is 0.}

\item{eno}{A logical value indicating whether to use the effective sample 
size (TRUE) or the total sample size (FALSE) for the number of degrees of 
freedom. The default value is FALSE.}

\item{K}{A numeric value indicating the maximum number of composites. The 
default value is NULL, which means the maximum value provided in 'occ' is 
used.}

\item{fileout}{A character string indicating the name of the .sav output file
The default value is NULL, which means not to save the output.}

\item{ncores}{An integer indicating the number of cores to use for parallel 
computation. The default value is NULL.}
}
\value{
A list containing:
\item{$composite}{ 
 A numeric array of the spatial dimensions and new dimension 'K' first, 
 followed by the same dimensions as parameter 'data'. The length of 
 dimension 'K' is parameter 'K'.
}
\item{$p.val}{
 A numeric array with the same dimension as $composite. It is the p-value of
 the composites obtained through a t-test that accounts for the serial
 dependence of the data.
}
}
\description{
Composite a multi-dimensional array which contains two spatial and one 
temporal dimensions, e.g., (lon, lat, time), according to the indices of 
mode/cluster occurrences in time. The p-value by t-test is also computed.
}
\examples{
blank <- array(0, dim = c(20, 10, 30))
x1 <- blank
t1 <- blank
f1 <- blank

for (i in 1:20) {
 x1[i, , ] <- i
}

for (i in 1:30) {
 t1[, , i] <- i
}

# This is 2D propagating sin wave example, where we use f1(lon, lat, time) 
# wave field. Compositing (like using stroboscopicc light) at different
# time steps can lead to modification or cancelation of wave pattern.

for (i in 1:20) {
 for (j in 1:30) {
   f1[i, , j] <- 3 * sin(2 * pi * x1[i, , j] / 5. - 2 * pi * t1[i, , j] / 6.)
 }
}
names(dim(f1)) <- c('lon', 'lat', 'time')
occ <- rep(0, 30)
occ[c(2, 5, 8, 11, 14, 17, 20, 23)] <- 1
res <- Composite(data = f1, occ = occ)
filled.contour(res$composite[, , 1])

occ <- rep(0, 30)
occ[c(3, 9, 15, 21)] <- 1
res <- Composite(data = f1, occ = occ)
filled.contour(res$composite[, , 1])

# Example with one missing composite in occ:
data <- 1:(4 * 5 * 6)
dim(data) <- c(lon = 4, lat = 5, case = 6)
occ <- c(1, 1, 2, 2, 3, 3) 
res <- Composite(data, occ, time_dim = 'case',  K = 4)

}
