% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sar_average.R
\name{sar_average}
\alias{sar_average}
\title{Fit a multimodel averaged SAR curve}
\usage{
sar_average(obj = c("power",
  "powerR","epm1","epm2","p1","p2","loga","koba",
  "mmf","monod","negexpo","chapman","weibull3","asymp",
  "ratio","gompertz","weibull4","betap","heleg", "linear"), data = NULL, crit
  = "Info", normaTest = "lillie", homoTest = "cor.fitted", neg_check = FALSE,
  alpha_normtest = 0.05, alpha_homotest = 0.05, grid_start = FALSE,
  grid_n = NULL, confInt = FALSE, ciN = 100, verb = TRUE)
}
\arguments{
\item{obj}{Either a vector of model names or a fit_collection object created
using \code{\link{sar_multi}}. If a vector of names is provided,
\code{sar_average} first calls \code{sar_multi} before generating the
averaged multimodel curve.}

\item{data}{A dataset in the form of a dataframe with two columns: the first
with island/site areas, and the second with the species richness of each
island/site. If \code{obj} is a fit_collection object, \code{data} should
be NULL.}

\item{crit}{The criterion used to compare models and compute the model
weights. The default \code{crit = "Info"} switches to AIC or AICc depending
on the number of data points in the dataset. AIC (\code{crit = "AIC"}) or
AICc (\code{crit = "AICc"}) can be chosen regardless of the sample size. For
BIC, use \code{crit ="Bayes"}.}

\item{normaTest}{The test used to test the normality of the residuals of each
model. Can be any of "lillie" (Lilliefors Kolmogorov-Smirnov test; the
default), "shapiro" (Shapiro-Wilk test of normality), "kolmo"
(Kolmogorov-Smirnov test), or "none" (no residuals normality test is
undertaken).}

\item{homoTest}{The test used to check for homogeneity of the residuals of
each model. Can be any of "cor.fitted" (a correlation of the residuals with
the model fitted values; the default), "cor.area" (a correlation of the
residuals with the area values), or "none" (no residuals homogeneity test
is undertaken).}

\item{neg_check}{Whether or not a check should be undertaken to flag any
models that predict negative richness values.}

\item{alpha_normtest}{The alpha value used in the residual normality test
(default = 0.05, i.e. any test with a P value < 0.05 is flagged as failing
the test).}

\item{alpha_homotest}{The alpha value used in the residual homogeneity test
(default = 0.05, i.e. any test with a P value < 0.05 is flagged as failing
the test).}

\item{grid_start}{Logical argument specifying whether a grid search procedure
should be implemented to test multiple starting parameter values. The
default is set to FALSE, but we would recommend using it to ensure the fits
of certain models (e.g. Gompertz, Chapman Richards) are optimal.}

\item{grid_n}{If \code{grid_start = TRUE}, the number of points sampled in
the model parameter space.}

\item{confInt}{A logical argument specifying whether confidence intervals
should be calculated for the multimodel curve using bootstrapping.}

\item{ciN}{The number of bootstrap samples to be drawn to calculate the
confidence intervals (if \code{confInt == TRUE}).}

\item{verb}{verbose (default: \code{verb == TRUE}).}
}
\value{
A list of class "multi" and class "sars" with two elements. The first
  element ('mmi') contains the fitted values of the multimodel sar curve. The
  second element ('details') is a list with the following components:
  \itemize{ \item{mod_names} { Names of the models that were successfully
  fitted and passed any model check} \item{fits} { A fit_collection object
  containing the successful model fits} \item{ic} { The information criterion
  selected} \item{norm_test} { The residual normality test selected}
  \item{homo_test} { The residual homogeneity test selected}
  \item{alpha_norm_test} { The alpha value used in the residual normality
  test} \item{alpha_homo_test} { The alpha value used in the residual
  homogeneity test} \item{ics} { The information criterion values (e.g. AIC
  values) of the model fits} \item{delta_ics} { The delta information
  criterion values} \item{weights_ics} { The information criterion weights of
  each model fit} \item{n_points} {  Number of data points} \item{n_mods} {
  The number of successfully fitted models} \item{no_fit} { Names of the
  models which could not be fitted or did not pass model checks} }

  The \code{\link{summary.sars}} function returns a more useful summary of
  the model fit results, and the \code{\link{plot.multi}} plots the
  multimodel curve.
}
\description{
Construct a multimodel averaged species-area relationship curve
  using information criterion weights and up to twenty SAR models.
}
\details{
The multimodel SAR curve is constructed using information criterion
  weights (see Burnham & Anderson, 2002; Guilhaumon et al. 2010). If
  \code{obj} is a vector of n model names the function fits the n models to
  the dataset provided using the \code{sar_multi} function. A dataset must
  have four or more datapoints to fit the multimodel curve. If any models
  cannot be fitted they are removed from the multimodel SAR. If \code{obj} is
  a fit_collection object (created using the \code{sar_multi} function), any
  model fits in the collection which are NA are removed. In addition, if any
  other model checks have been selected (i.e. residual normality and
  heterogeneity tests, and checks for negative predicted richness values),
  these are undertaken and any model that fails the selected test(s) is
  removed from the multimodel SAR. The order of the additional checks inside
  the function is: normality of residuals, homogeneity of residuals, and a
  check for negative fitted values. Once a model fails one test it is removed
  and thus is not available for further tests. Thus, a model may fail
  multiple tests but the returned warning will only provide information on a
  single test.

  The resultant models are then used to construct the multimodel SAR curve.
  For each model in turn, the model fitted values are multiplied by the
  information criterion weight of that model, and the resultant values are
  summed across all models (Burnham & Anderson, 2002). Confidence intervals
  can be calculated (using \code{confInt}) around the multimodel averaged
  curve using the bootstrap procedure outlined in Guilhaumon et al (2010).The
  procedure transforms the residuals from the individual model fits and
  occasionally NAs / Inf values can be produced - in these cases, the model
  is removed from the confidence interval calculation (but not the multimodel
  curve itself). When several SAR models are used and the number of
  bootstraps (\code{ciN}) is large, generating the confidence intervals can
  take a long time.

  The \code{sar_models()} function can be used to bring up a list of the 20
  model names. \code{display_sars_models()} generates a table of the 20
  models with model information.
}
\note{
Occasionally a model fit will converge and pass the model fitting
  checks (e.g. residual normality) but the resulting fit is nonsensical (e.g.
  a horizontal line with intercept at zero). Thus, it can be useful to plot
  the resultant 'multi' object to check the individual model fits. To re-run
  the \code{sar_multi} function without a particular model, simply remove it
  from the \code{obj} argument.
  
  Choosing starting parameter values for non-linear regression optimisation
  algorithms is not always straight forward, depending on the data at hand.
  In the package, we use various approaches to choose default starting
  parameters. However, if any of the resultant model fits does not converge,
  returns a singular gradient at parameter estimates, or the plot of the
  model fit does not look optimum, try using the \code{grid_start} argument
  to undertake a more extensive selection of starting values, or provide your
  own starting values (\code{start}). While using \code{grid_start} is more
  time consuming, it will often provide (much) better fits for certain
  models, and so we recommend its use where possible. Note, that
  \code{grid_start} has been disabled for a small number of models (e.g.
  Weibull 3 par.). See the vignette for more information.

  The generation of confidence intervals around the multimodel curve (using
  \code{confInt == TRUE}), may throw up errors that we have yet to come
  across. Please report any issues to the package maintainer.
  
  There are different formulas for calculating the various information
  criteria (IC) used for model comparison (e.g. AIC, BIC). For example, some
  formulas use the residual sum of squares (rss) and others the
  log-likelihood (ll). Both are valid approaches and will give the same
  parameter estimates, but it is important to only compare IC values that
  have been calculated using the same approach. For example, the 'sars'
  package used to use formulas based on the rss, while the \link[stats]{nls}
  function function in the stats package uses formulas based on the ll. To
  increase the compatability between nls and sars, we have changed our
  formulas such that now our IC formulas are the same as those used in the
  \link[stats]{nls} function. See the "On the calculation of information
  criteria" section in the package vignette for more information.
}
\examples{
data(galap)
#attempt to construct a multimodel SAR curve using all twenty sar models
fit <- sar_average(data = galap)
summary(fit)
plot(fit)

# construct a multimodel SAR curve using a fit_collection object
ff <- sar_multi(galap, obj = c("power", "loga", "monod", "weibull3"))
fit2 <- sar_average(obj = ff, data = NULL)
summary(fit2)

}
\references{
Burnham, K. P., & Anderson, D. R. (2002). Model selection and
  multi-model inference: a practical information-theoretic approach (2nd
  ed.). New-York: Springer.

  Guilhaumon, F., Mouillot, D., & Gimenez, O. (2010). mmSAR: an R-package for
  multimodel species-area relationship inference. Ecography, 33, 420-424.
}
