\name{scaleboot}
\alias{scaleboot}
\alias{countw.assmax}
\alias{countw.shtest}
\alias{countw.shtestass}
\title{Multiscale Bootstrap Resampling}
\description{
  Performing multiscale bootstrap resampling for a specified statistic.
}
\usage{

scaleboot(dat,nb,sa,fun,parm=NULL,count=TRUE,weight=TRUE,
          cluster=NULL,onlyboot=FALSE,seed=NULL,...)

countw.assmax(x,w,ass)

countw.shtest(x,w,obs)

countw.shtestass(x,w,assobs)

}
\arguments{
  \item{dat}{data matrix or data-frame. Row vectors are to be resampled.}
  \item{nb}{vector of numbers of bootstrap replicates.}
  \item{sa}{vector of scales in sigma squared (\eqn{\sigma^2}).}
  \item{fun}{function for a statistic.}
  \item{parm}{parameter to be passed to \code{fun} above.}
  \item{count}{logical. Should only the accumulative counts be returned?
    Otherwise, raw statistic vectors are returned.}
  \item{weight}{logical. In \code{fun} above, resampling is specified by
    weight vector. Otherwise, resampling is specified by a vector of indices.}
  \item{cluster}{\pkg{snow} cluster object which may be generated by
    function \code{makeCluster}.}
  \item{onlyboot}{logical. Should only bootstrap resampling be
    performed? Otherwise, \code{\link{sbfit}} or \code{\link{sbconf}}
    is called internally.}
  \item{seed}{If non NULL, random seed is set. Specifying a seed is
    particularly important when \code{cluster} is non NULL, in which
    case \code{seed + seq(along=cluster)} are set to cluster nodes.}
  \item{...}{further arguments passed to or from other methods.}  

  \item{x}{data matrix or data-frame passed from \code{\link{scaleboot}}.}
  \item{w}{weight vector for resampling.}
  \item{ass}{a list of association vectors. An example of \code{parm} above.}
  \item{obs}{a vector of observed test statistics. An example of \code{parm} above.}
  \item{assobs}{a list of ass and obs above. An example of \code{parm} above.}
}
\details{
  These functions are used internally in implementing \code{\link{relltest}}. 

  \code{scaleboot} performs multiscale bootstrap resampling for a
  statistic defined by \code{fun}, which should be one of the two
  possible forms \code{fun(x,w,parm)} and \code{fun(x,i,parm)}. The form
  \code{fun(x,w,parm)} is used when \code{weight=TRUE}, and the weight
  vector \code{w} is generated by a multinomial distribution. The form
  \code{fun(x,i,parm)} is used when \code{weight=FALSE}, and the index
  vector \code{i} is generated by resampling \eqn{n'} elements from
  \eqn{\{1,...,n\}}. When \code{count=TRUE}, \code{fun} should return
  logical, or a vector of logical.

  Examples of \code{fun(x,w,parm)} are \code{countw.assmax} for AU p-values,
  \code{countw.shtest} for SH-test of trees, and \code{countw.shtestass}
  for SH-test of both trees and edges. The definitions are given below.
  
  \preformatted{
countw.assmax <- function(x,w,ass) {
  y <- maxdif(wsumrow(x,w)) <= 0 # countw.max
  if(is.null(ass)) y
  else {
    z <- vector("logical",length(ass))
    for(i in seq(along=ass)) z[i] <- any(y[ass[[i]]])
    z
  }
}

countw.shtest <- function(x,w,obs)  maxdif(wsumrow(x,w)) >= obs

countw.shtestass <- function(x,w,assobs)
  unlist(assmaxdif(wsumrow(x,w),assobs$ass)) >= assobs$obs
    
### weighted sum of row vectors
##
## x = matrix (array of row vectors)
## w = weight vector (for rows)
##
wsumrow <- function(x,w) {
  apply(w*x,2,sum)*nrow(x)/sum(w)
}

### calc max diff
##
## y[i] := max_{j neq i} x[j] - x[i]
##
maxdif <- function(x) {
  i1 <- which.max(x)  # the largest element
  x <- -x + x[i1]
  x[i1] <- -min(x[-i1])  # the second largest value
  x
}

### calc assmaxdif
##
## y[[i]][j] := max_{k neq ass[[i]]} x[k] - x[ass[[i]][j]]
##
assmaxdif <-  function(x,a) {
  y <- vector("list",length(a))
  names(y) <- names(a)
  for(i in seq(along=a))  y[[i]] <- max(x[-a[[i]]]) - x[a[[i]]]
  y
}
}

When \code{count=TRUE}, the summation of outputs from \code{fun} is
  calculated. This gives the frequencies for hypotheses how many times
  they are supported in bootstrap replicates.

}
\value{
  If \code{onlyboot=TRUE}, a list of raw results from multiscale bootstrap
  resampling is returned. The components are "stat" for list vectors of
  outputs from \code{fun} (only when \code{count=FALSE}), "bps" for a
  matrix of multiscale bootstrap probabilities (only when
  \code{count=FALSE}), "nb" for the number of bootstrap replicates used,
  and "sa" for the scales used. Note that scales are redefined by
  \code{sa <- nsize/round(nsize/sa)}, where \code{nsize} is the sample size.

  
  If \code{onlyboot=FALSE}, a result of a call to 
  \code{\link{sbfit}} is returned if  \code{count=TRUE} and
  that of \code{\link{sbconf}} is returned if  \code{count=FALSE}.
}
\author{Hidetoshi Shimodaira}
\seealso{\code{\link{sbfit}}, \code{\link{relltest}}.}
\examples{

\dontrun{
## a line from the definition of relltest
scaleboot(dat,nb,sa,countw.assmax,ass,cluster=cluster,
                 names.hp=na,nofit=nofit,models=models,seed=seed)

## two lines from rell.shtest (internal function)
scaleboot(z,nb,1,countw.shtest,tobs,cluster=cluster,
                 onlyboot=TRUE,seed=seed)
scaleboot(z,nb,1,countw.shtestass,pa,cluster=cluster,
                 onlyboot=TRUE,seed=seed)
}
}
\keyword{ nonparametric }

