% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vault.R
\name{create_package_vault}
\alias{create_package_vault}
\alias{create_vault}
\title{Create a vault, as a folder or in an R package.}
\usage{
create_package_vault(path = ".")

create_vault(path)
}
\arguments{
\item{path}{Path to the R package. A file or directory within the
package is fine, too. If the vault directory already exists, a message
is given, and the function does nothing.}
}
\value{
The directory of the vault, invisibly.
}
\description{
A vault is a folder that contains information about users and the secrets
they share. You can create a vault as either a standalone folder, or
as part of a package.
}
\details{
A vault is a folder with a specific structure, containing two
directories: \code{users} and \code{secrets}.

In \code{users}, each file contains a public key in PEM format. The name of
the file is the identifier of the key, an arbitrary name. We suggest
that you use email addresses to identify public keys. See also \code{\link[=add_user]{add_user()}}.

In \code{secrets}, each secret is stored in its own directory.
The directory of a secret contains
\enumerate{
\item the secret, encrypted with its own AES key, and
\item the AES key, encrypted with the public keys of all users that
have access to the secret, each in its own file.
}

To add a secret, see \code{\link[=add_secret]{add_secret()}}
}
\section{Creating a package folder}{


When you create a vault in a package, this vault is stored in the
\code{inst/vault} directory of the package during development. At package
install time, this folder is copied to the \code{vault} folder.
}

\examples{

# The `secret` package contains some user keys for demonstration purposes.
# In this example, Alice shares a secret with Bob using a vault.

keys <- function(x){
  file.path(system.file("user_keys", package = "secret"), x)
}
alice_public  <- keys("alice.pub")
alice_private <- keys("alice.pem")
bob_public  <- keys("bob.pub")
bob_private <- keys("bob.pem")
carl_private <- keys("carl.pem")

# Create vault

vault <- file.path(tempdir(), ".vault")
if (dir.exists(vault)) unlink(vault) # ensure vault is empty
create_vault(vault)

# Add users with their public keys

add_user("alice", public_key = alice_public, vault = vault)
add_user("bob", public_key = bob_public, vault = vault)
list_users(vault = vault)

# Share a secret

secret <- list(username = "user123", password = "Secret123!")

add_secret("secret", value = secret, users = c("alice", "bob"),
           vault = vault)
list_secrets(vault = vault)

# Alice and Bob can decrypt the secret with their private keys
# Note that you would not normally have access to the private key
# of any of your collaborators!

get_secret("secret", key = alice_private, vault = vault)
get_secret("secret", key = bob_private, vault = vault)

# But Carl can't decrypt the secret

try(
  get_secret("secret", key = carl_private, vault = vault)
)

# Unshare the secret

unshare_secret("secret", users = "bob", vault = vault)
try(
  get_secret("secret", key = bob_private, vault = vault)
)


# Delete the secret

delete_secret("secret", vault = vault)
list_secrets(vault)

# Delete the users

delete_user("alice", vault = vault)
delete_user("bob", vault = vault)
list_users(vault)
}
\seealso{
\code{\link[=add_user]{add_user()}}, \code{\link[=add_secret]{add_secret()}}
}
