% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/secure.R
\name{secure.path}
\alias{secure}
\alias{secure.path}
\title{Sequential Co-Sparse Factor Regression}
\usage{
secure.path(Y, X = NULL, nrank = 3, nlambda = 100, U0 = NULL,
  V0 = NULL, D0 = NULL, orthXU = FALSE, orthV = FALSE,
  keepPath = TRUE, control = list(), ic = c("GIC", "BICP", "AIC")[1])
}
\arguments{
\item{Y}{response matrix}

\item{X}{covariate matrix; when X = NULL, the fucntion performs unsupervised learning}

\item{nrank}{an integer specifying the desired rank/number of factors}

\item{nlambda}{number of lambda values to be used along each path}

\item{U0}{initial value of U}

\item{V0}{initial value of V}

\item{D0}{initial value of D}

\item{orthXU}{if TRUE, orthogonality of XU is required}

\item{orthV}{if TRUE, orthogonality of V is required}

\item{keepPath}{if TRUE, th solution paths of U, V, D are reported}

\item{control}{a list of internal parameters controlling the model fitting}

\item{ic}{character specifying which information criterion to use for selecting the tuning parameter: "GIC"(default), "BICP", and "AIC"}
}
\value{
\item{C.est}{estimated coefficient matrix; based on modified BIC}
  \item{U}{estimated U matrix (factor weights)}
  \item{D}{estimated singular values}
  \item{V}{estimated V matrix (factor loadings)}
  \item{ortX}{if TRUE, X is treated as an orthogonal matrix in the computation}
  \item{lam}{selected lambda values based on the chosen information criterion}
  \item{lampath}{sequences of lambda values used in model fitting. In each sequential unit-rank estimation step,
  a sequence of length nlambda is first generated between (lamMax*lamMaxFac, lamMax*lamMaxFac*lamMinFac) equally 
  spaced on the log scale, in which lamMax is estimated and the other parameters are specified in secure.control.
  The model fitting starts from the largest lambda and stops when the maximum proportion of nonzero elements is reached in 
  either u or v, as specified by spU and spV in secure.control.}
  \item{IC}{values of information criteria}
  \item{Upath}{solution path of U}
  \item{Dpath}{solution path of D}
  \item{Vpath}{solution path of D}
}
\description{
Sequential factor extraction via co-sparse unit-rank estimation (SeCURE)
}
\examples{
#require(secure)

# Simulate data from a sparse factor regression model
p <- 100; q <- 100; n <- 200
xrho <- 0.5; nlambda <- 100 
nrank <- 3 

U <- matrix(0,ncol=nrank ,nrow=p);  V <- matrix(0,ncol=nrank ,nrow=q)
U[,1]<-c(sample(c(1,-1),8,replace=TRUE),rep(0,p-8))
U[,2]<-c(rep(0,5),sample(c(1,-1),9,replace=TRUE),rep(0,p-14))
U[,3]<-c(rep(0,11),sample(c(1,-1),9,replace=TRUE),rep(0,p-20))
V[,1]<-c(sample(c(1,-1),5,replace=TRUE)*runif(5,0.3,1),rep(0,q-5))
V[,2]<-c(rep(0,5),sample(c(1,-1),5,replace=TRUE)*runif(5,0.3,1),rep(0,q-10))
V[,3]<-c(rep(0,10),sample(c(1,-1),5,replace=TRUE)*runif(5,0.3,1),rep(0,q-15))
U[,1:3]<- apply(U[,1:3],2,function(x)x/sqrt(sum(x^2)))
V[,1:3]<- apply(V[,1:3],2,function(x)x/sqrt(sum(x^2)))
D <- diag(c(20,15,10)) 
C <- U\%*\%D\%*\%t(V)

Xsigma <- xrho^abs(outer(1:p, 1:p,FUN="-"))
sim.sample <- secure.sim(U,D,V,n,snr = 0.25,Xsigma,rho=0.3)
Y <- sim.sample$Y; 
X <- sim.sample$X



# Fitting secure. Set maximum rank to be 4.
rank.ini <- 4

# Set largest model to about 25\% sparsity
# See secure.control for setting other parameters
control <- secure.control(spU=0.25, spV=0.25)

# Complete data case. 
# Fit secure without orthogonality
fit.orthF <- secure.path(Y,X,nrank=rank.ini,nlambda = nlambda,
                        control=control)
# check orthogonality
crossprod(X\%*\%fit.orthF$U)/n
# check solution
# fit.orthF$U
# fit.orthF$V
# fit.orthF$D

# Fit secure with orthogonality if desired. It takes longer time.
# fit.orthT <- secure.path(Y,X,nrank=rank.ini,nlambda = nlambda,
#                                   orthXU=TRUE,orthV=TRUE,control=control)
# check orthogonality
# crossprod(X\%*\%fit.orthT$U)/n

  
# 15\% missing case
miss <- 0.15
t.ind <- sample.int(n*q, size = miss*n*q)
y <- as.vector(Y); y[t.ind] <- NA;  Ym <- matrix(y,n,q)

fit.orthF.miss <- secure.path(Ym, X, nrank = rank.ini, nlambda = nlambda, 
                            control = control) 
# fit.orthT.miss <- secure.path(Ym, X, nrank = rank.ini, nlambda = nlambda,
#                           orthXU=TRUE,orthV=TRUE, control = control)
}
\references{
Mishra, A., Chen, K., & Dey, D. (2017) \emph{ Sequential Co-Sparse Factor Regression, To appear in Journal of Computational and Graphical Statistics (JCGS)}
}

