% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/clade_physig.R
\name{clade_physig}
\alias{clade_physig}
\title{Influential clade detection - Phylogenetic signal}
\usage{
clade_physig(trait.col, data, phy, clade.col, n.species = 5, n.sim = 100,
  method = "K", track = TRUE, ...)
}
\arguments{
\item{trait.col}{The name of a column in the provided data frame with trait 
to be analyzed  (e.g. "Body_mass").}

\item{data}{Data frame containing species traits with row names matching tips
in \code{phy}.}

\item{phy}{A phylogeny (class 'phylo') matching \code{data}.}

\item{clade.col}{The column in the provided data frame which specifies the
clades (a character vector with clade names).}

\item{n.species}{Minimum number of species in a clade for the clade to be
included in the leave-one-out deletion analysis. Default is \code{5}.}

\item{n.sim}{Number of simulations for the randomization test.}

\item{method}{Method to compute signal: can be "K" or "lambda".}

\item{track}{Print a report tracking function progress (default = TRUE)}

\item{...}{Further arguments to be passed to \code{\link[phytools]{phylosig}}}
}
\value{
The function \code{clade_physig} returns a list with the following
components:

\code{trait.col}: Column name of the trait analysed

\code{full.data.estimates}: Phylogenetic signal estimate (K or lambda)
and the P value (for the full data).

\code{sensi.estimates}: A data frame with all simulation
estimates. Each row represents a deleted clade. Columns report the calculated
phylogenetic signal (K or lambda) (\code{estimate}), difference between simulation
signal and full data signal (\code{DF}), the percentage of change
in signal compared to the full data estimate (\code{perc}) and 
p-value of the phylogenetic signal with the reduced data (\code{pval}).

\code{null.dist}: A data frame with estimates for the null distribution
of phylogenetic signal for all clades analysed.

\code{data}: Original full dataset.
}
\description{
Estimate the influence of clade removal on phylogenetic signal estimates
}
\details{
This function sequentially removes one clade at a time, estimates phylogenetic
signal (K or lambda) using \code{\link[phytools]{phylosig}} and stores the
results. The impact of a specific clade on signal estimates is calculated by the
comparison between the full data (with all species) and reduced data estimates
(without the species belonging to a clade).

To account for the influence of the number of species on each 
clade (clade sample size), this function also estimate a null distribution of signal estimates
expected by the removal of the same number of species as in a given clade. This is done by estimating
phylogenetic signal without the same number of species in the given clade. 
 The number of simulations to be performed is set by 'n.sim'. To test if the 
 clade influence differs from the null expectation for a clade of that size, 
 a randomization test can be performed using 'summary(x)'. 

Output can be visualised using \code{sensi_plot}.
}
\note{
The argument "se" from \code{\link[phytools]{phylosig}} is not available in this function. Use the 
argument "V" instead with \code{\link{intra_physig}} to indicate the name of the column containing the standard 
deviation or the standard error of the trait variable instead.
}
\examples{
data(alien)
# Logtransform data
alien.data$logMass <- log(alien.data$adultMass) 
# Run sensitivity analysis:
clade <- clade_physig(trait.col = "logMass", data = alien.data, n.sim = 20,
                 phy = alien.phy[[1]], clade.col = "family", method = "K")
summary(clade)
sensi_plot(clade, "Bovidae")
sensi_plot(clade, "Sciuridae")
}
\references{
Blomberg, S. P., T. Garland Jr., A. R. Ives (2003) 
Testing for phylogenetic signal in comparative data: 
Behavioral traits are more labile. Evolution, 57, 717-745.

Pagel, M. (1999) Inferring the historical patterns of biological evolution. 
Nature, 401, 877-884.

Kamilar, J. M., & Cooper, N. (2013). Phylogenetic signal in primate behaviour,
 ecology and life history. Philosophical Transactions of the Royal Society 
 B: Biological Sciences, 368: 20120341.
}
\seealso{
\code{\link[phytools]{phylosig}}, 
\code{\link{clade_phylm}},\code{\link{sensi_plot}}
}
\author{
Gustavo Paterno
}
