% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tree_influ_phylm.R
\name{tree_influ_phylm}
\alias{tree_influ_phylm}
\title{Interaction between phylogenetic uncertainty and influential species detection - Phylogenetic Linear Regression}
\usage{
tree_influ_phylm(
  formula,
  data,
  phy,
  n.tree = 2,
  cutoff = 2,
  model = "lambda",
  track = TRUE,
  ...
)
}
\arguments{
\item{formula}{The model formula}

\item{data}{Data frame containing species traits with row names matching tips
in \code{phy}.}

\item{phy}{A phylogeny (class 'phylo') matching \code{data}.}

\item{n.tree}{Number of times to repeat the analysis with n different trees picked 
randomly in the multiPhylo file.}

\item{cutoff}{The cutoff value used to identify for influential species
(see Details)}

\item{model}{The phylogenetic model to use (see Details). Default is \code{lambda}.}

\item{track}{Print a report tracking function progress (default = TRUE)}

\item{...}{Further arguments to be passed to \code{phylolm}}
}
\value{
The function \code{influ_phylm} returns a list with the following
components:

\code{cutoff}: The value selected for \code{cutoff}

\code{formula}: The formula

\code{full.model.estimates}: Coefficients, aic and the optimised
value of the phylogenetic parameter (e.g. \code{lambda}) for the full model
without deleted species.

\code{influential_species}: List of influential species, both
based on standardised difference in intercept and in the slope of the
regression. Species are ordered from most influential to less influential and
only include species with a standardised difference > \code{cutoff}.

\code{sensi.estimates}: A data frame with all simulation
estimates. Each row represents a deleted clade for a given random tree. Columns report the calculated
regression intercept (\code{intercept}), difference between simulation
intercept and full model intercept (\code{DIFintercept}), the standardised
difference (\code{sDIFintercept}), the percentage of change in intercept compared
to the full model (\code{intercept.perc}) and intercept p-value
(\code{pval.intercept}). All these parameters are also reported for the regression
slope (\code{DIFestimate} etc.). Additionally, model aic value (\code{AIC}) and
the optimised value (\code{optpar}) of the phylogenetic parameter
(e.g. \code{kappa} or \code{lambda}, depending on the phylogenetic model used) are
reported.

\code{data}: Original full dataset.

\code{errors}: Species where deletion resulted in errors.
}
\description{
Performs leave-one-out deletion analysis for phylogenetic linear regression,
and detects influential species while evaluating uncertainty in trees topology.
}
\details{
This function sequentially removes one species at a time, fits a phylogenetic
linear regression model using \code{\link[phylolm]{phylolm}}, stores the
results and detects influential species. It repeats this operation using n trees, 
randomly picked in a multiPhylo file.

All phylogenetic models from \code{phylolm} can be used, i.e. \code{BM},
\code{OUfixedRoot}, \code{OUrandomRoot}, \code{lambda}, \code{kappa},
\code{delta}, \code{EB} and \code{trend}. See ?\code{phylolm} for details.

\code{influ_phylm} detects influential species based on the standardised
difference in intercept and/or slope when removing a given species compared
to the full model including all species. Species with a standardised difference
above the value of \code{cutoff} are identified as influential. The default
value for the cutoff is 2 standardised differences change.

Currently, this function can only implement simple linear models (i.e. \eqn{trait~
predictor}). In the future we will implement more complex models.

Output can be visualised using \code{sensi_plot}.
}
\examples{
\dontrun{
# Load data:
data(alien)
# run analysis:
tree_influ <- tree_influ_phylm(log(gestaLen) ~ log(adultMass), phy = alien$phy, 
data = alien$data, n.tree = 5)
# To check summary results:
summary(tree_influ)
# Visual diagnostics
sensi_plot(tree_influ)
sensi_plot(tree_influ, graphs = 1)
sensi_plot(tree_influ, graphs = 2)

data(alien)
tree_influ <- tree_influ_phylm(log(gestaLen) ~ log(adultMass), phy = alien$phy, 
data = alien$data[1:25, ], n.tree = 2)
summary(tree_influ)
}
}
\references{
Paterno, G. B., Penone, C. Werner, G. D. A. 
\href{http://doi.wiley.com/10.1111/2041-210X.12990}{sensiPhy: 
An r-package for sensitivity analysis in phylogenetic 
comparative methods.} Methods in Ecology and Evolution 
2018, 9(6):1461-1467

Ho, L. S. T. and Ane, C. 2014. "A linear-time algorithm for 
Gaussian and non-Gaussian trait evolution models". Systematic Biology 63(3):397-408.
}
\seealso{
\code{\link[phylolm]{phylolm}}, \code{\link{tree_phylm}}, 
\code{\link{influ_phylm}}, \code{\link{tree_influ_phyglm}}, \code{\link{sensi_plot}}
}
\author{
Gustavo Paterno, Caterina Penone & Gijsbert D.A. Werner
}
