% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simStateSpace-sim-ssm-var.R
\name{SimSSMVAR}
\alias{SimSSMVAR}
\title{Simulate Data from the Vector Autoregressive Model
using a State Space Model Parameterization (n = 1)}
\usage{
SimSSMVAR(
  mu0,
  sigma0,
  alpha,
  beta,
  psi,
  gamma_eta = NULL,
  x = NULL,
  time = 0,
  burn_in = 0
)
}
\arguments{
\item{mu0}{Numeric vector.
Mean of initial latent variable values
(\eqn{\boldsymbol{\mu}_{\boldsymbol{\eta} \mid 0}}).}

\item{sigma0}{Numeric matrix.
The covariance matrix
of initial latent variable values
(\eqn{\boldsymbol{\Sigma}_{\boldsymbol{\eta} \mid 0}}).}

\item{alpha}{Numeric vector.
Vector of intercepts for the dynamic model
(\eqn{\boldsymbol{\alpha}}).}

\item{beta}{Numeric matrix.
Transition matrix relating the values of the latent variables
at time \code{t - 1} to those at time \code{t}
(\eqn{\boldsymbol{\beta}}).}

\item{psi}{Numeric matrix.
The process noise covariance matrix
(\eqn{\boldsymbol{\Psi}}).}

\item{gamma_eta}{Numeric matrix.
Matrix relating the values of the covariate matrix
at time \code{t} to the latent variables at time \code{t}
(\eqn{\boldsymbol{\Gamma}_{\boldsymbol{\eta}}}).}

\item{x}{Numeric matrix.
The matrix of observed covariates in \code{type = 1} or \code{type = 2}.
The number of rows should be equal to \code{time + burn_in}.}

\item{time}{Positive integer.
Number of time points to simulate.}

\item{burn_in}{Positive integer.
Number of burn-in points to exclude before returning the results.}
}
\value{
Returns an object of class \code{simstatespace}
which is a list with the following elements:
\itemize{
\item \code{call}: Function call.
\item \code{args}: Function arguments.
\item \code{data}: Generated data which is a list of length \code{n}.
\code{data} is a list with the following elements:
\itemize{
\item \code{id}: A vector of ones of length \code{t}.
\item \code{time}: A vector of time points of length \code{t}.
\item \code{y}: A \code{t} by \code{k} matrix of values for the manifest variables.
\item \code{eta}: A \code{t} by \code{p} matrix of values for the latent variables.
\item \code{x}: A \code{t} by \code{j} matrix of values for the covariates.
}
\item \code{fun}: Function used.
}
}
\description{
This function simulates data from the vector autoregressive model
using a state space model parameterization.
}
\details{
The measurement model is given by
\deqn{
    \mathbf{y}_{t}
    =
    \boldsymbol{\eta}_{t} .
  }

The dynamic structure is given by
\deqn{
    \boldsymbol{\eta}_{t}
    =
    \boldsymbol{\alpha}
    +
    \boldsymbol{\beta}
    \boldsymbol{\eta}_{t - 1}
    +
    \boldsymbol{\zeta}_{t}
    \quad
    \mathrm{with}
    \quad
    \boldsymbol{\zeta}_{t}
    \sim
    \mathcal{N}
    \left(
    \mathbf{0},
    \boldsymbol{\Psi}
    \right)
  }
where
\eqn{\boldsymbol{\eta}_{t}},
\eqn{\boldsymbol{\eta}_{t - 1}},
and
\eqn{\boldsymbol{\zeta}_{t}}
are random variables,
and
\eqn{\boldsymbol{\alpha}},
\eqn{\boldsymbol{\beta}},
and
\eqn{\boldsymbol{\Psi}}
are model parameters.
\eqn{\boldsymbol{\eta}_{t}}
is a vector of latent variables
at time \eqn{t},
\eqn{\boldsymbol{\eta}_{t - 1}}
is a vector of latent variables
at time \eqn{t - 1},
and
\eqn{\boldsymbol{\zeta}_{t}}
is a vector of dynamic noise
at time \eqn{t}.
\eqn{\boldsymbol{\alpha}}
is a vector of intercepts,
\eqn{\boldsymbol{\beta}}
is a matrix of autoregression
and cross regression coefficients,
and
\eqn{\boldsymbol{\Psi}}
is the covariance matrix of
\eqn{\boldsymbol{\zeta}_{t}}.

Note that when \code{gamma_eta} and \code{x} are not \code{NULL},
the dynamic structure is given by
\deqn{
    \boldsymbol{\eta}_{t}
    =
    \boldsymbol{\alpha}
    +
    \boldsymbol{\beta}
    \boldsymbol{\eta}_{t - 1}
    +
    \boldsymbol{\Gamma}_{\boldsymbol{\eta}}
    \mathbf{x}_{t}
    +
    \boldsymbol{\zeta}_{t}
    \quad
    \mathrm{with}
    \quad
    \boldsymbol{\zeta}_{t}
    \sim
    \mathcal{N}
    \left(
    \mathbf{0},
    \boldsymbol{\Psi}
    \right)
  }
where
\eqn{\mathbf{x}_{t}} is a vector of covariates at time \eqn{t},
and \eqn{\boldsymbol{\Gamma}_{\boldsymbol{\eta}}} is the coefficient matrix
linking the covariates to the latent variables.
}
\examples{
# prepare parameters
set.seed(42)
k <- 3
iden <- diag(k)
null_vec <- rep(x = 0, times = k)
mu0 <- null_vec
sigma0 <- iden
alpha <- null_vec
beta <- diag(x = 0.5, nrow = k)
psi <- iden
time <- 1000
burn_in <- 0
gamma_eta <- 0.10 * diag(k)
x <- matrix(
  data = rnorm(n = k * (time + burn_in)),
  ncol = k
)

# No covariates
ssm <- SimSSMVAR(
  mu0 = mu0,
  sigma0 = sigma0,
  alpha = alpha,
  beta = beta,
  psi = psi,
  time = time,
  burn_in = burn_in
)

plot(ssm)

# With covariates
ssm <- SimSSMVAR(
  mu0 = mu0,
  sigma0 = sigma0,
  alpha = alpha,
  beta = beta,
  psi = psi,
  gamma_eta = gamma_eta,
  x = x,
  time = time,
  burn_in = burn_in
)

plot(ssm)

}
\references{
Chow, S.-M., Ho, M. R., Hamaker, E. L., & Dolan, C. V. (2010).
Equivalence and differences between structural equation modeling
and state-space modeling techniques.
\emph{Structural Equation Modeling: A Multidisciplinary Journal},
17(2), 303–332.
\doi{10.1080/10705511003661553}
}
\seealso{
Other Simulation of State Space Models Data Functions: 
\code{\link{OU2SSM}()},
\code{\link{SimSSMFixed}()},
\code{\link{SimSSMIVary}()},
\code{\link{SimSSMLinGrowthIVary}()},
\code{\link{SimSSMLinGrowth}()},
\code{\link{SimSSMOUFixed}()},
\code{\link{SimSSMOUIVary}()},
\code{\link{SimSSMOU}()},
\code{\link{SimSSMVARFixed}()},
\code{\link{SimSSMVARIVary}()},
\code{\link{SimSSM}()}
}
\concept{Simulation of State Space Models Data Functions}
\keyword{sim}
\keyword{simStateSpace}
\keyword{var}
