% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sim_pw_surv.R
\name{sim_pw_surv}
\alias{sim_pw_surv}
\title{Simulate a stratified time-to-event outcome randomized trial}
\usage{
sim_pw_surv(
  n = 100,
  stratum = data.frame(stratum = "All", p = 1),
  block = c(rep("control", 2), rep("experimental", 2)),
  enroll_rate = data.frame(rate = 9, duration = 1),
  fail_rate = data.frame(stratum = rep("All", 4), period = rep(1:2, 2), treatment =
    c(rep("control", 2), rep("experimental", 2)), duration = rep(c(3, 1), 2), rate =
    log(2)/c(9, 9, 9, 18)),
  dropout_rate = data.frame(stratum = rep("All", 2), period = rep(1, 2), treatment =
    c("control", "experimental"), duration = rep(100, 2), rate = rep(0.001, 2))
)
}
\arguments{
\item{n}{Number of observations.
If length(n) > 1, the length is taken to be the number required.}

\item{stratum}{A data frame with stratum specified in \code{stratum},
probability (incidence) of each stratum in \code{p}.}

\item{block}{Vector of treatments to be included in each block.}

\item{enroll_rate}{Enrollment rates; see details and examples.}

\item{fail_rate}{Failure rates; see details and examples;
note that treatments need to be the same as input in block.}

\item{dropout_rate}{Dropout rates; see details and examples;
note that treatments need to be the same as input in block.}
}
\value{
A data frame with the following variables for each observation:
\itemize{
\item \code{stratum}.
\item \code{enroll_time}: Enrollment time for the observation.
\item \code{Treatment}: Treatment group; this will be one of the values
in the input \code{block}.
\item \code{fail_time}: Failure time generated using \code{\link[=rpwexp]{rpwexp()}}.
\item \code{dropout_time}: Dropout time generated using \code{\link[=rpwexp]{rpwexp()}}.
\item \code{cte}: Calendar time of enrollment plot the minimum of
failure time and dropout time.
\item \code{fail}: Indicator that \code{cte} was set using failure time;
i.e., 1 is a failure, 0 is a dropout.
}
}
\description{
\code{sim_pw_surv()} enables simulation of a clinical trial with
essentially arbitrary patterns of enrollment, failure rates and censoring.
The piecewise exponential distribution allows a simple method to specify
a distribution and enrollment pattern where the enrollment, failure,
and dropout rate changes over time.
While the main purpose may be to generate a trial that can be analyzed
at a single point in time or using group sequential methods,
the routine can also be used to simulate an adaptive trial design.
Enrollment, failure, and dropout rates are specified by treatment group,
stratum and time period.
Fixed block randomization is used; blocks must include treatments provided
in failure and dropout specification.
Default arguments are set up to allow very simple implementation of
a non-proportional hazards assumption for an unstratified design.
}
\examples{
\dontshow{if (requireNamespace("dplyr", quietly = TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
library(dplyr)

# Example 1
sim_pw_surv(n = 20)

# Example 2
# 3:1 randomization
sim_pw_surv(
  n = 20,
  block = c(rep("experimental", 3), "control")
)

# Example 3
# Simulate 2 stratum; will use defaults for blocking and enrollRates
sim_pw_surv(
  n = 20,
  # 2 stratum,30\% and 70\% prevalence
  stratum = data.frame(stratum = c("Low", "High"), p = c(.3, .7)),
  fail_rate = data.frame(
    stratum = c(rep("Low", 4), rep("High", 4)),
    period = rep(1:2, 4),
    treatment = rep(c(
      rep("control", 2),
      rep("experimental", 2)
    ), 2),
    duration = rep(c(3, 1), 4),
    rate = c(.03, .05, .03, .03, .05, .08, .07, .04)
  ),
  dropout_rate = data.frame(
    stratum = c(rep("Low", 2), rep("High", 2)),
    period = rep(1, 4),
    treatment = rep(c("control", "experimental"), 2),
    duration = rep(1, 4),
    rate = rep(.001, 4)
  )
)
# Example 4
# If you want a more rectangular entry for a data.frame
fail_rate <- bind_rows(
  data.frame(stratum = "Low", period = 1, treatment = "control", duration = 3, rate = .03),
  data.frame(stratum = "Low", period = 1, treatment = "experimental", duration = 3, rate = .03),
  data.frame(stratum = "Low", period = 2, treatment = "experimental", duration = 3, rate = .02),
  data.frame(stratum = "High", period = 1, treatment = "control", duration = 3, rate = .05),
  data.frame(stratum = "High", period = 1, treatment = "experimental", duration = 3, rate = .06),
  data.frame(stratum = "High", period = 2, treatment = "experimental", duration = 3, rate = .03)
)

dropout_rate <- bind_rows(
  data.frame(stratum = "Low", period = 1, treatment = "control", duration = 3, rate = .001),
  data.frame(stratum = "Low", period = 1, treatment = "experimental", duration = 3, rate = .001),
  data.frame(stratum = "High", period = 1, treatment = "control", duration = 3, rate = .001),
  data.frame(stratum = "High", period = 1, treatment = "experimental", duration = 3, rate = .001)
)

sim_pw_surv(
  n = 12,
  stratum = data.frame(stratum = c("Low", "High"), p = c(.3, .7)),
  fail_rate = fail_rate,
  dropout_rate = dropout_rate
)
\dontshow{\}) # examplesIf}
}
