% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sits_classify.R
\name{sits_classify}
\alias{sits_classify}
\alias{sits_classify.sits}
\alias{sits_classify.raster_cube}
\alias{sits_classify.derived_cube}
\alias{sits_classify.tbl_df}
\alias{sits_classify.segs_cube}
\alias{sits_classify.default}
\title{Classify time series or data cubes}
\usage{
sits_classify(
  data,
  ml_model,
  ...,
  filter_fn = NULL,
  multicores = 2L,
  progress = TRUE
)

\method{sits_classify}{sits}(
  data,
  ml_model,
  ...,
  filter_fn = NULL,
  multicores = 2L,
  gpu_memory = 16,
  progress = TRUE
)

\method{sits_classify}{raster_cube}(
  data,
  ml_model,
  ...,
  roi = NULL,
  filter_fn = NULL,
  start_date = NULL,
  end_date = NULL,
  memsize = 8L,
  multicores = 2L,
  gpu_memory = 16,
  output_dir,
  version = "v1",
  verbose = FALSE,
  progress = TRUE
)

\method{sits_classify}{derived_cube}(data, ml_model, ...)

\method{sits_classify}{tbl_df}(data, ml_model, ...)

\method{sits_classify}{segs_cube}(
  data,
  ml_model,
  ...,
  filter_fn = NULL,
  start_date = NULL,
  end_date = NULL,
  memsize = 8L,
  multicores = 2L,
  gpu_memory = 16,
  output_dir,
  version = "v1",
  n_sam_pol = 40,
  verbose = FALSE,
  progress = TRUE
)

\method{sits_classify}{default}(data, ml_model, ...)
}
\arguments{
\item{data}{Data cube (tibble of class "raster_cube")}

\item{ml_model}{R model trained by \code{\link[sits]{sits_train}}
(closure of class "sits_model")}

\item{...}{Other parameters for specific functions.}

\item{filter_fn}{Smoothing filter to be applied - optional
(clousure containing object of class "function").}

\item{multicores}{Number of cores to be used for classification
(integer, min = 1, max = 2048).}

\item{progress}{Logical: Show progress bar?}

\item{gpu_memory}{Memory available in GPU in GB (default = 16)}

\item{roi}{Region of interest (either an sf object, shapefile,
or a numeric vector with named XY values
("xmin", "xmax", "ymin", "ymax") or
named lat/long values
("lon_min", "lat_min", "lon_max", "lat_max").}

\item{start_date}{Start date for the classification
(Date in YYYY-MM-DD format).}

\item{end_date}{End date for the classification
(Date im YYYY-MM-DD format).}

\item{memsize}{Memory available for classification in GB
(integer, min = 1, max = 16384).}

\item{output_dir}{Valid directory for output file.
(character vector of length 1).}

\item{version}{Version of the output
(character vector of length 1).}

\item{verbose}{Logical: print information about processing time?}

\item{n_sam_pol}{Number of time series per segment to be classified
(integer, min = 10, max = 50).}
}
\value{
Time series with predicted labels for
                          each point (tibble of class "sits")
                          or a data cube with probabilities for each class
                          (tibble of class "probs_cube").
}
\description{
This function classifies a set of time series or data cube given
a trained model prediction model created by \code{\link[sits]{sits_train}}.

SITS supports the following models:
(a) support vector machines:  \code{\link[sits]{sits_svm}};
(b) random forests:  \code{\link[sits]{sits_rfor}};
(c) extreme gradient boosting: \code{\link[sits]{sits_xgboost}};
(d) multi-layer perceptrons: \code{\link[sits]{sits_mlp}};
(e) 1D CNN: \code{\link[sits]{sits_tempcnn}};
(f) deep residual networks: \code{\link[sits]{sits_resnet}};
(g) self-attention encoders: \code{\link[sits]{sits_lighttae}}.
}
\note{
The \code{roi} parameter defines a region of interest. It can be
   an sf_object, a shapefile, or a bounding box vector with
   named XY values (\code{xmin}, \code{xmax}, \code{ymin}, \code{ymax}) or
   named lat/long values (\code{lon_min}, \code{lon_max},
   \code{lat_min}, \code{lat_max})

   Parameter \code{filter_fn} parameter specifies a smoothing filter
   to be applied to each time series for reducing noise. Currently, options
   are Savitzky-Golay (see \code{\link[sits]{sits_sgolay}}) and Whittaker
   (see \code{\link[sits]{sits_whittaker}}) filters.

   Parameter \code{memsize} controls the amount of memory available
   for classification, while \code{multicores}  defines the number of cores
   used for processing. We recommend using as much memory as possible.

   When using a GPU for deep learning, \code{gpu_memory} indicates the
   memory of available in the graphics card.

   For classifying vector data cubes created by
   \code{\link[sits]{sits_segment}},
   \code{n_sam_pol} controls is the number of time series to be
   classified per segment.

   Please refer to the sits documentation available in
   <https://e-sensing.github.io/sitsbook/> for detailed examples.
}
\examples{
if (sits_run_examples()) {
    # Example of classification of a time series
    # Retrieve the samples for Mato Grosso
    # train a random forest model
    rf_model <- sits_train(samples_modis_ndvi, ml_method = sits_rfor)

    # classify the point
    point_ndvi <- sits_select(point_mt_6bands, bands = c("NDVI"))
    point_class <- sits_classify(
        data = point_ndvi, ml_model = rf_model
    )
    plot(point_class)

    # Example of classification of a data cube
    # create a data cube from local files
    data_dir <- system.file("extdata/raster/mod13q1", package = "sits")
    cube <- sits_cube(
        source = "BDC",
        collection = "MOD13Q1-6",
        data_dir = data_dir
    )
    # classify a data cube
    probs_cube <- sits_classify(
        data = cube,
        ml_model = rf_model,
        output_dir = tempdir(),
        version = "ex_classify"
    )
    # label the probability cube
    label_cube <- sits_label_classification(
        probs_cube,
        output_dir = tempdir(),
        version = "ex_classify"
    )
    # plot the classified image
    plot(label_cube)
    # segmentation
    # segment the image
    segments <- sits_segment(
        cube = cube,
        seg_fn = sits_slic(step = 5,
                       compactness = 1,
                       dist_fun = "euclidean",
                       avg_fun = "median",
                       iter = 50,
                       minarea = 10,
                       verbose = FALSE
                       ),
        output_dir = tempdir()
    )
    # Create a classified vector cube
    probs_segs <- sits_classify(
        data = segments,
        ml_model = rf_model,
        output_dir = tempdir(),
        n_sam_pol = 20,
        multicores = 4
    )
    # Create a labelled vector cube
    class_segs <- sits_label_classification(
        cube = probs_segs,
        output_dir = tempdir(),
        multicores = 2,
        memsize = 4
    )
    # plot class_segs
    plot(class_segs)
}

}
\author{
Rolf Simoes, \email{rolf.simoes@inpe.br}

Gilberto Camara, \email{gilberto.camara@inpe.br}
}
