% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/smcfcs.r
\name{smcfcs}
\alias{smcfcs}
\title{Substantive model compatible fully conditional specification imputation of covariates.}
\usage{
smcfcs(originaldata, smtype, smformula, method, predictorMatrix = NULL,
  m = 5, numit = 10, rjlimit = 1000, noisy = FALSE)
}
\arguments{
\item{originaldata}{The original data frame with missing values.}

\item{smtype}{A string specifying the type of substantive model. Possible
values are \code{"lm"}, \code{"logistic"}, \code{"coxph"} and \code{"compet"}.}

\item{smformula}{The formula of the substantive model. For \code{"coxph"} substantive
models the left hand side should be of the form \code{"Surv(t,d)"}. For \code{"compet"}
substantive models, a list should be passed consisting of the Cox models
for each cause of failure (see example).}

\item{method}{A required vector of strings specifying for each variable either
that it does not need to be imputed (""), the type of regression model to be
be used to impute. Possible values are \code{"norm"} (normal linear regression),
\code{"logreg"} (logistic regression), \code{"poisson"} (Poisson regression),
\code{"podds"} (proportional odds regression for ordered categorical variables),
\code{"mlogit"} (multinomial logistic regression for unordered categorical variables),
or a custom expression which defines a passively imputed variable, e.g.
\code{"x^2"} or \code{"x1*x2"}.}

\item{predictorMatrix}{An optional predictor matrix. If specified, the matrix defines which
covariates will be used as predictors in the imputation models
(the outcome must not be included). The i'th row of the matrix should consist of
0s and 1s, with a 1 in the j'th column indicating the j'th variable be used
as a covariate when imputing the i'th variable. If not specified, when
imputing a given variable, the imputation model covariates are the other
covariates of the substantive model which are partially observed
(but which are not passively imputed) and any fully observed covariates (if present)
in the substantive model. Note that the outcome variable is implicitly conditioned
on by the rejection sampling scheme used by smcfcs, and should not be specified as a predictor
in the predictor matrix.}

\item{m}{The number of imputed datasets to generate. The default is 5.}

\item{numit}{The number of iterations to run when generating each imputation.
In a (limited) range of simulations good performance was obtained with the
default of 10 iterations. However, particularly when the proportion of missingness
is large, more iterations may be required for convergence to stationarity.}

\item{rjlimit}{Specifies the maximum number of attempts which should be made
when using rejection sampling to draw from imputation models. If the limit is reached
when running a warning will be issued. In this case it is probably advisable to
increase the \code{rjlimit} until the warning does not appear.}

\item{noisy}{logical value (default FALSE) indicating whether output should be noisy, which can
be useful for debugging or checking that models being used are as desired.}
}
\value{
A list containing:

\code{impDatasets} a list containing the imputed datasets

\code{smCoefIter} a three dimension matrix containing the substantive model parameter
values. The matrix is indexed by [imputation,parameter number,iteration]
}
\description{
Multiple imputes missing covariate values using substantive model compatible
fully conditional specification.
}
\details{
smcfcs imputes missing values of covariates using the Substantive Model Compatible
Fully Conditional Specification multiple imputation approach proposed by
Bartlett \emph{et al} 2014 (see references).

Currently imputation is supported for linear regression ("lm"), logistic
regression ("logistic"), Cox regression for time to event
data ("coxph"), and Cox models for competing risks data. For the latter, a Cox
model is assumed for each cause of failure, and the event indicator should be integer
coded with 0 corresponding to censoring, 1 corresponding to failure from the first
cause etc.

The function returns a list. The first element \code{impDataset} of the list is a list of the imputed
datasets. Models (e.g. the substantive model) can be fitted to each and results
combined using Rubin's rules using the mitools package, as illustrated in the examples.

The second element \code{smCoefIter} is a three dimensional array containing the values
of the substantive model parameters obtained at the end of each iteration of the algorithm.
The array is indexed by: imputation number, parameter number, iteration.

If the substantive model is linear or logistic regression, `smcfcs` will automatically impute missing
outcomes, if present, using the specified substantive model. However, even in this case, the
user should specify "" in the element of method corresponding to the outcome variable.

The development of this package was supported by a UK Medical Research Council
Fellowship (MR/K02180X/1). Part of its development took place while the author was
kindly hosted by the University of Michigan's Department of Biostatistics & Institute for
Social Research.

The structure of many of the arguments to \code{smcfcs} are based on those of
the excellent \code{mice} package.
}
\examples{
#set random number seed to make results reproducible
set.seed(123)

#linear substantive model with quadratic covariate effect
imps <- smcfcs(ex_linquad, smtype="lm", smformula="y~z+x+xsq",
               method=c("","","norm","x^2",""))

#if mitools is installed, fit substantive model to imputed datasets
#and combine results using Rubin's rules
if (requireNamespace("mitools", quietly = TRUE)) {
  library(mitools)
  impobj <- imputationList(imps$impDatasets)
  models <- with(impobj, lm(y~z+x+xsq))
  summary(MIcombine(models))
}

#the following examples are not run when the package is compiled on CRAN
#(to keep computation time down), but they can be run by package users
\dontrun{
  #examining convergence, using 100 iterations, setting m=1
  imps <- smcfcs(ex_linquad, smtype="lm", smformula="y~z+x+xsq",
                 method=c("","","norm","x^2",""),m=1,numit=100)
  #convergence plot from first imputation for third coefficient of substantive model
  plot(imps$smCoefIter[1,3,])

  #include auxiliary variable assuming it is conditionally independent of Y (which it is here)
  predMatrix <- array(0, dim=c(ncol(ex_linquad),ncol(ex_linquad)))
  predMatrix[3,] <- c(0,1,0,0,1)
  imps <- smcfcs(ex_linquad, smtype="lm", smformula="y~z+x+xsq",
                 method=c("","","norm","x^2",""),predictorMatrix=predMatrix)

  #impute missing x1 and x2, where they interact in substantive model
  imps <- smcfcs(ex_lininter, smtype="lm", smformula="y~x1+x2+x1*x2",
                 method=c("","norm","logreg"))

  #logistic regression substantive model, with quadratic covariate effects
  imps <- smcfcs(ex_logisticquad, smtype="logistic", smformula="y~z+x+xsq",
                 method=c("","","norm","x^2",""))

  #Cox regression substantive model, with only main covariate effects
  if (requireNamespace("survival", quietly = TRUE)) {
    imps <- smcfcs(ex_coxquad, smtype="coxph", smformula="Surv(t,d)~z+x+xsq",
                   method=c("","","","norm","x^2",""))

    #competing risks substantive model, with only main covariate effects
    imps <- smcfcs(ex_compet, smtype="compet",
                   smformula=c("Surv(t,d==1)~x1+x2", "Surv(t,d==2)~x1+x2"),
                   method=c("","","logreg","norm"))
  }

  #if mitools is installed, fit model for first competing risk
  if (requireNamespace("mitools", quietly = TRUE)) {
    library(mitools)
    impobj <- imputationList(imps$impDatasets)
    models <- with(impobj, coxph(Surv(t,d==1)~x1+x2))
    summary(MIcombine(models))
  }

}
}
\author{
Jonathan Bartlett \email{jonathan.bartlett@lshtm.ac.uk} \url{http://www.missingdata.org.uk}
\url{http://thestatsgeek.com}
}
\references{
Bartlett JW, Seaman SR, White IR, Carpenter JR. Multiple imputation of covariates
by fully conditional specification: accommodating the substantive model. Statistical Methods
in Medical Research 2015; 24(4): 462-487. \url{http://doi.org/10.1177/0962280214521348}
}

