\name{smnet}
\alias{smnet}
%- Also NEED an '\alias' for EACH other topic documented here.
\title{Additive Modelling for Stream Networks}

\description{Fits (Gaussian) additive models to river network data based on the flexible modelling framework described in O'Donnell et al. (2014).  Data must be in the form of a \code{SpatialStreamNetwork} object as used by the \code{SSN} package (Ver Hoef et al., 2012).   Smoothness of covariate effects is represented via a penalised B-spline basis (P-splines) and parameter estimates are obtained using penalised least-squares.  Optimal smoothness is achieved by optimization of \code{AIC}, \code{GCV} or \code{AICC}.  

The formula interpreter used for penalised additive components is modelled on the code found in the package \code{mgcv}. 
}

\usage{
smnet(formula, data.object, control = NULL, method = "AICC", netID = 1)
}
%- maybe also 'usage' for other objects documented here.
\arguments{
\item{formula}{A formula statement similar to those used by \code{lm} and \code{mgcv:::gam}.  Smooth functions are represented with \code{m(..., k = 20)} function, up to 2d smooth interactions are currently supported.  Spatial network components are specified using \code{network(...)} function.  Further details can be found in \link{m} and \link{network} and the examples below.}

\item{data.object}{An object of class "SpatialStreamNetwork"}

\item{control}{A list of options passed to the optimiser for smoothness selection. 
\code{maxit}, default = 500, limits the number of iterations made by the optimiser.  \code{approx}, positive integer specifying the number samples to collect using a Monte-Carlo method to approximate the performance criterion surface when direct evaluation is too slow - this takes advantage of matrix sparsity and may be much faster if the network has a large number of segments or the data is large - \code{approx  = 100} often works well in these cases (defaults to \code{NULL}).  \code{checks}, logical, default = \code{TRUE}, specifies whether additivity checks should be performed on the set of input weights for network modelling; useful when the networks are very large and checks can be expensive.}

\item{method}{Character string determining the performance criterion for choosing optimal smoothness, options are \code{"AICC"}, \code{"AIC"} or \code{"GCV"}. 
}
\item{netID}{Integer indicating the particular stream network to model, generally only user-specified when multiple networks are contained within \code{data.object}, default is 1. 
}
}

\value{Object of class \code{smnet} with components
\item{fitted.values}{vector of fitted mean values}
\item{residuals}{vector of residuals, response minus the fitted values}
\item{coefficients}{vector of estimated coefficients}
\item{R2}{the R^2 statistic}
\item{R2.adj}{the adjusted R^2 statistic}
\item{df.residual}{the residual degrees of freedom}
\item{ssn.object}{the unchanged SSN data object}
\item{internals}{various model objects for internal use by other functions}
}

\references{
Ver Hoef, J.M.,  Peterson, E.E., Clifford, D., Shah, R.  (2012)   SSN: An R Package for Spatial Statistical Modeling on Stream Networks 

O' Donnell, D., Rushworth, A.M., Bowman, A.W., Scott, E.M., Hallard, M.  (2014) Flexible regression models over river networks.  Journal of the Royal Statistical Society: Series C (Applied Statistics). 63(1) 47--63.

Reinhard Furrer, Stephan R. Sain (2010). spam: A Sparse Matrix R
Package with Emphasis on MCMC Methods for Gaussian Markov Random
Fields. Journal of Statistical Software, 36(10), 1-25. URL: http://www.jstatsoft.org/v36/i10/
}
\author{Alastair Rushworth}


\seealso{
\code{\link[=smnet]{get_adjacency}}, \code{\link{plot.smnet}}
}

\examples{

# Set up an SSN object - this part taken 
# from the SSN:::SimulateOnSSN help file
example_network<- createSSN(
                     n          = 50,
                     obsDesign  = binomialDesign(200), 
                     predDesign = binomialDesign(50),
                     importToR  = TRUE, 
                     path = paste(tempdir(),"/example_network",sep = ""),
                     treeFunction = iterativeTreeLayout
                     )

# plot the simulated network structure with prediction locations
# plot(example_network, bty = "n", xlab = "x-coord", ylab = "y-coord")

## create distance matrices, including between predicted and observed
createDistMat(example_network, "preds", o.write=TRUE, amongpred = TRUE)

## extract the observed and predicted data frames
observed_data            <- getSSNdata.frame(example_network, "Obs")
prediction_data          <- getSSNdata.frame(example_network, "preds")

## associate continuous covariates with the observation locations
#  data generated from a normal distribution
obs                      <- rnorm(200)
observed_data[,"X"]      <- obs
observed_data[,"X2"]     <- obs^2

## associate continuous covariates with the prediction locations
#  data generated from a normal distribution
pred                     <- rnorm(50) 
prediction_data[,"X"]    <- pred
prediction_data[,"X2"]   <- pred^2

## simulate some Gaussian data that follows a 'tail-up' spatial process
sims <- SimulateOnSSN(
               ssn.object      = example_network, 
               ObsSimDF        = observed_data, 
               PredSimDF       = prediction_data,  
               PredID          = "preds",  
               formula         = ~ 1 + X,
               coefficients    = c(1, 10),
               CorModels       = c("Exponential.tailup"), 
               use.nugget      = TRUE,
               CorParms        = c(10, 5, 0.1),
               addfunccol      = "addfunccol")$ssn.object


## extract the observed and predicted data frames, now with simulated values
sim1DFpred         <- getSSNdata.frame(sims, "preds")
sim1preds          <- sim1DFpred[,"Sim_Values"]
sim1DFpred[,"Sim_Values"] <- NA
sims               <- putSSNdata.frame(sim1DFpred, sims, "preds")

# create the adjacency matrix for use with smnet
adjacency    <- get_adjacency(
                  paste(tempdir(), "/example_network", sep = ""), 
                  net = 1
                  )

# not run - plot the adjacency matrix
# display(adjacency[[1]])

# sometimes it is useful to see which varaibles are valid network weights 
# in the data contained within the SSN object
show_weights(sims, adjacency)

# fit a penalised spatial model to the stream network data
# Sim_Values are quadratic in the X covariate.  To highlight 
# the fitting of smooth terms, this is treated as non-linear 
# and unknown using m().
mod_smn       <- smnet(formula = Sim_Values ~ m(X) + m(X2) + 
network(adjacency = adjacency, weight = "shreve"), 
                       data.object = sims, netID = 1)

# not run - plot different summaries of the model
plot(mod_smn, type = "covariates")
plot(mod_smn, type = "nodes")
plot(mod_smn, type = "segments", weight = 4, shadow = 2)
plot(mod_smn, type = "full", weight = 4, shadow = 2)

# obtain predictions at the prediction locations and plot 
# against true values
preds <- predict(mod_smn, newdata = getSSNdata.frame(sims, "preds"))
plot(preds$predictions, sim1preds)

# obtain summary of the fitted model
summary(mod_smn)
}





% Add one or more standard keywords, see file 'KEYWORDS' in the
% R documentation directory.
\keyword{network}
\keyword{sparse}
\keyword{P-spline}