% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ml_regression_generalized_linear_regression.R
\name{ml_generalized_linear_regression}
\alias{ml_generalized_linear_regression}
\title{Spark ML -- Generalized Linear Regression}
\usage{
ml_generalized_linear_regression(
  x,
  formula = NULL,
  family = "gaussian",
  link = NULL,
  fit_intercept = TRUE,
  offset_col = NULL,
  link_power = NULL,
  link_prediction_col = NULL,
  reg_param = 0,
  max_iter = 25,
  weight_col = NULL,
  solver = "irls",
  tol = 1e-06,
  variance_power = 0,
  features_col = "features",
  label_col = "label",
  prediction_col = "prediction",
  uid = random_string("generalized_linear_regression_"),
  ...
)
}
\arguments{
\item{x}{A \code{spark_connection}, \code{ml_pipeline}, or a \code{tbl_spark}.}

\item{formula}{Used when \code{x} is a \code{tbl_spark}. R formula as a character string or a formula. This is used to transform the input dataframe before fitting, see \link{ft_r_formula} for details.}

\item{family}{Name of family which is a description of the error distribution to be used in the model. Supported options: "gaussian", "binomial", "poisson", "gamma" and "tweedie". Default is "gaussian".}

\item{link}{Name of link function which provides the relationship between the linear predictor and the mean of the distribution function. See for supported link functions.}

\item{fit_intercept}{Boolean; should the model be fit with an intercept term?}

\item{offset_col}{Offset column name. If this is not set, we treat all instance offsets as 0.0. The feature specified as offset has a constant coefficient of 1.0.}

\item{link_power}{Index in the power link function. Only applicable to the Tweedie family. Note that link power 0, 1, -1 or 0.5 corresponds to the Log, Identity, Inverse or Sqrt link, respectively. When not set, this value defaults to 1 - variancePower, which matches the R "statmod" package.}

\item{link_prediction_col}{Link prediction (linear predictor) column name. Default is not set, which means we do not output link prediction.}

\item{reg_param}{Regularization parameter (aka lambda)}

\item{max_iter}{The maximum number of iterations to use.}

\item{weight_col}{The name of the column to use as weights for the model fit.}

\item{solver}{Solver algorithm for optimization.}

\item{tol}{Param for the convergence tolerance for iterative algorithms.}

\item{variance_power}{Power in the variance function of the Tweedie distribution which provides the relationship between the variance and mean of the distribution. Only applicable to the Tweedie family. (see \href{https://en.wikipedia.org/wiki/Tweedie_distribution}{Tweedie Distribution (Wikipedia)}) Supported values: 0 and [1, Inf). Note that variance power 0, 1, or 2 corresponds to the Gaussian, Poisson or Gamma family, respectively.}

\item{features_col}{Features column name, as a length-one character vector. The column should be single vector column of numeric values. Usually this column is output by \code{\link{ft_r_formula}}.}

\item{label_col}{Label column name. The column should be a numeric column. Usually this column is output by \code{\link{ft_r_formula}}.}

\item{prediction_col}{Prediction column name.}

\item{uid}{A character string used to uniquely identify the ML estimator.}

\item{...}{Optional arguments; see Details.}
}
\value{
The object returned depends on the class of \code{x}. If it is a
\code{spark_connection}, the function returns a \code{ml_estimator} object. If
it is a \code{ml_pipeline}, it will return a pipeline with the predictor
appended to it. If a \code{tbl_spark}, it will return a \code{tbl_spark} with
the predictions added to it.
}
\description{
Perform regression using Generalized Linear Model (GLM).
}
\details{
Valid link functions for each family is listed below. The first link function of each family is the default one.
  \itemize{
    \item gaussian: "identity", "log", "inverse"
    \item binomial: "logit", "probit", "loglog"
    \item poisson: "log", "identity", "sqrt"
    \item gamma: "inverse", "identity", "log"
    \item tweedie: power link function specified through \code{link_power}. The default link power in the tweedie family is \code{1 - variance_power}.
    }
}
\examples{
\dontrun{
library(sparklyr)

sc <- spark_connect(master = "local")
mtcars_tbl <- sdf_copy_to(sc, mtcars, name = "mtcars_tbl", overwrite = TRUE)

partitions <- mtcars_tbl \%>\%
  sdf_random_split(training = 0.7, test = 0.3, seed = 1111)

mtcars_training <- partitions$training
mtcars_test <- partitions$test

# Specify the grid
family <- c("gaussian", "gamma", "poisson")
link <- c("identity", "log")
family_link <- expand.grid(family = family, link = link, stringsAsFactors = FALSE)
family_link <- data.frame(family_link, rmse = 0)

# Train the models
for (i in seq_len(nrow(family_link))) {
  glm_model <- mtcars_training \%>\%
    ml_generalized_linear_regression(mpg ~ .,
      family = family_link[i, 1],
      link = family_link[i, 2]
    )

  pred <- ml_predict(glm_model, mtcars_test)
  family_link[i, 3] <- ml_regression_evaluator(pred, label_col = "mpg")
}

family_link
}

}
\seealso{
Other ml algorithms: 
\code{\link{ml_aft_survival_regression}()},
\code{\link{ml_decision_tree_classifier}()},
\code{\link{ml_gbt_classifier}()},
\code{\link{ml_isotonic_regression}()},
\code{\link{ml_linear_regression}()},
\code{\link{ml_linear_svc}()},
\code{\link{ml_logistic_regression}()},
\code{\link{ml_multilayer_perceptron_classifier}()},
\code{\link{ml_naive_bayes}()},
\code{\link{ml_one_vs_rest}()},
\code{\link{ml_random_forest_classifier}()}
}
\concept{ml algorithms}
