% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/learnGraphTopology.R
\name{learn_bipartite_graph}
\alias{learn_bipartite_graph}
\title{Learn a bipartite graph

Learns a bipartite graph on the basis of an observed data matrix}
\usage{
learn_bipartite_graph(S, is_data_matrix = FALSE, z = 0, nu = 10000,
  alpha = 0, w0 = "naive", m = 7, maxiter = 10000,
  abstol = 1e-06, reltol = 1e-04, record_weights = FALSE,
  verbose = TRUE)
}
\arguments{
\item{S}{either a pxp sample covariance/correlation matrix, or a pxn data
matrix, where p is the number of nodes and n is the number of
features (or data points per node)}

\item{is_data_matrix}{whether the matrix S should be treated as data matrix
or sample covariance matrix}

\item{z}{the number of zero eigenvalues for the Adjancecy matrix}

\item{nu}{regularization hyperparameter for the term ||A(w) - V Psi V'||^2_F}

\item{alpha}{L1 regularization hyperparameter}

\item{w0}{initial estimate for the weight vector the graph or a string
selecting an appropriate method. Available methods are: "qp": finds w0 that minimizes
||ginv(S) - L(w0)||_F, w0 >= 0; "naive": takes w0 as the negative of the
off-diagonal elements of the pseudo inverse, setting to 0 any elements s.t.
w0 < 0}

\item{m}{in case is_data_matrix = TRUE, then we build an affinity matrix based
on Nie et. al. 2017, where m is the maximum number of possible connections
for a given node}

\item{maxiter}{the maximum number of iterations}

\item{abstol}{absolute tolerance on the weight vector w}

\item{reltol}{relative tolerance on the weight vector w}

\item{record_weights}{whether to record the edge values at each iteration}

\item{verbose}{whether to output a progress bar showing the evolution of the
iterations}
}
\value{
A list containing possibly the following elements:
\item{\code{Laplacian}}{the estimated Laplacian Matrix}
\item{\code{Adjacency}}{the estimated Adjacency Matrix}
\item{\code{w}}{the estimated weight vector}
\item{\code{psi}}{optimization variable accounting for the eigenvalues of the Adjacency matrix}
\item{\code{V}}{eigenvectors of the estimated Adjacency matrix}
\item{\code{elapsed_time}}{elapsed time recorded at every iteration}
\item{\code{convergence}}{boolean flag to indicate whether or not the optimization converged}
\item{\code{obj_fun}}{values of the objective function at every iteration in case record_objective = TRUE}
\item{\code{loglike}}{values of the negative loglikelihood at every iteration in case record_objective = TRUE}
\item{\code{w_seq}}{sequence of weight vectors at every iteration in case record_weights = TRUE}
}
\description{
Learn a bipartite graph

Learns a bipartite graph on the basis of an observed data matrix
}
\examples{
library(spectralGraphTopology)
library(igraph)
library(viridis)
library(corrplot)
set.seed(42)
n1 <- 10
n2 <- 6
n <- n1 + n2
pc <- .9
bipartite <- sample_bipartite(n1, n2, type="Gnp", p = pc, directed=FALSE)
# randomly assign edge weights to connected nodes
E(bipartite)$weight <- runif(gsize(bipartite), min = 0, max = 1)
# get true Laplacian and Adjacency
Ltrue <- as.matrix(laplacian_matrix(bipartite))
Atrue <- diag(diag(Ltrue)) - Ltrue
# get samples
Y <- MASS::mvrnorm(100 * n, rep(0, n), Sigma = MASS::ginv(Ltrue))
# compute sample covariance matrix
S <- cov(Y)
# estimate Adjacency matrix
graph <- learn_bipartite_graph(S, z = 4, verbose = FALSE)
graph$Adjacency[graph$Adjacency < 1e-3] <- 0
# Plot Adjacency matrices: true, noisy, and estimated
corrplot(Atrue / max(Atrue), is.corr = FALSE, method = "square",
         addgrid.col = NA, tl.pos = "n", cl.cex = 1.25)
corrplot(graph$Adjacency / max(graph$Adjacency), is.corr = FALSE,
         method = "square", addgrid.col = NA, tl.pos = "n", cl.cex = 1.25)
# build networks
estimated_bipartite <- graph_from_adjacency_matrix(graph$Adjacency,
                                                   mode = "undirected",
                                                   weighted = TRUE)
V(estimated_bipartite)$type <- c(rep(0, 10), rep(1, 6))
la = layout_as_bipartite(estimated_bipartite)
colors <- viridis(20, begin = 0, end = 1, direction = -1)
c_scale <- colorRamp(colors)
E(estimated_bipartite)$color = apply(
  c_scale(E(estimated_bipartite)$weight / max(E(estimated_bipartite)$weight)), 1,
                          function(x) rgb(x[1]/255, x[2]/255, x[3]/255))
E(bipartite)$color = apply(c_scale(E(bipartite)$weight / max(E(bipartite)$weight)), 1,
                      function(x) rgb(x[1]/255, x[2]/255, x[3]/255))
la = la[, c(2, 1)]
# Plot networks: true and estimated
plot(bipartite, layout = la, vertex.color=c("red","black")[V(bipartite)$type + 1],
     vertex.shape = c("square", "circle")[V(bipartite)$type + 1],
     vertex.label = NA, vertex.size = 5)
plot(estimated_bipartite, layout = la,
     vertex.color=c("red","black")[V(estimated_bipartite)$type + 1],
     vertex.shape = c("square", "circle")[V(estimated_bipartite)$type + 1],
     vertex.label = NA, vertex.size = 5)
}
\references{
S. Kumar, J. Ying, J. V. de Miranda Cardoso, D. P. Palomar. A unified
            framework for structured graph learning via spectral constraints (2019).
            https://arxiv.org/pdf/1904.09792.pdf
}
\author{
Ze Vinicius and Daniel Palomar
}
