#' Graph spectral entropy
#'
#' 'graph.entropy' returns the spectral entropy of a given undirected graph.
#'
#' @param A the adjacency matrix of the graph. For an unweighted graph it 
#' contains only 0s and 1s. For a weighted graph, it may contain nonnegative 
#' real values that correspond to the weights of the edges.
#'
#' @param bandwidth string indicating which criterion will be used to choose 
#' the bandwidth during the spectral density estimation. The available criteria 
#' are "Silverman" (default) and "Sturges".
#'
#' @param eigenvalues optional parameter. It contains the eigenvalues of matrix
#' A. Then, if the eigenvalues of matrix A have already been computed, this 
#' parameter can be used instead of A. If no value is passed, then the 
#' eigenvalues of A will be computed by 'graph.entropy'.
#'
#' @return a real number corresponding to the graph spectral entropy.  
#'
#' @keywords spectral_entropy
#' 
#' @references
#' Takahashi, D. Y., Sato, J. R., Ferreira, C. E. and Fujita A. (2012) 
#' Discriminating Different Classes of Biological Networks by Analyzing the 
#' Graph Spectra  Distribution. _PLoS ONE_, *7*, e49949. 
#' doi:10.1371/journal.pone.0049949.
#' 
#' Silverman, B. W. (1986) _Density Estimation_.  London: Chapman and Hall.
#'
#' Sturges, H. A. The Choice of a Class Interval. _J. Am. Statist. Assoc._, 
#' *21*, 65-66.
#'
#' @examples
#' require(igraph)
#' G <- erdos.renyi.game(100, p=0.5)
#' A <- as.matrix(get.adjacency(G))
#' entropy <- graph.entropy(A)
#' entropy
#'
#' @export
graph.entropy <- function(A=NULL, bandwidth="Silverman", eigenvalues=NULL) {
    if (is.null(eigenvalues))
        f <- spectralDensity(A, bandwidth=bandwidth)
    else
        f <- gaussianDensity(eigenvalues, bandwidth=bandwidth)    
    if (sum(is.na(f)) > 0)
        return(NA)
    y <- f$y
    n <- length(y)
    i <- which(y != 0)
    y[i] <- y[i]*log(y[i])
    return(-trapezoidSum(f$x, y))
}

#' Graph Information Criterion (GIC)
#'
#' 'GIC' returns the Kullback-Leibler divergence between an undirected graph and 
#' a given graph model. 
#'
#' @param A the adjacency matrix of the graph. For an unweighted graph it 
#' contains only 0s and 1s. For a weighted graph, it may contain nonnegative 
#' real values that correspond to the weights of the edges.
#'
#' @param model either a list, a string, a function or a matrix describing a 
#' graph model: 
#' 
#' A list that represents the spectral density of a model. It contains the 
#' components "x" and "y", which are numeric vectors of the same size. The "x" 
#' component contains the points at which the density was computed and the "y" 
#' component contains the observed density.
#'
#' A string that indicates one of the following models: "ER" (Erdos-Renyi random 
#' graph), "GRG" (geometric random graph), "KR" (k regular random graph), "WS" 
#' (Watts-Strogatz model), and "BA" (Barabási-Albert model). When the argument 
#' 'model' is a string, the user must also provide the model parameter by using 
#' the argument 'p'.  
#'
#' A function that returns a graph (represented by its adjacency matrix) 
#' generated by a graph model. It must contain two arguments: the first one
#' correponds to the graph size and the second to the parameter of the model. 
#' The model parameter will be provided by the argument 'p' of the 'GIC' 
#' function.
#'
#' A matrix containing the spectrum of the model. Each column contains the 
#' eigenvalues of a graph generated by the model. To estimate the spectral 
#' density of the model, the method will estimate the density of the values of 
#' each column, and then will take the average density.
#'
#' @param p the model parameter. The user must provide a valid parameter if the
#' argument 'model' is a string or a function. 
#' For the predefined models ("ER", "GRG", "KR", "WS", and "BA"), the parameter 
# 'p' corresponds to:
#
#' the probability to connect a pair of vertices, for the "ER" model 
#' (Erdos-Renyi random graph);
#' 
#' the radius used to contruct the geometric graph in a unit square, for the 
#' "GRG" model (geometric random graph);
#'
#' the degree 'k' of a regular graph, for the "KR" model (k regular random 
#' graph);
#' 
#' the probability to reconnect a vertex, for the "WS" model (Watts-Strogatz 
#' model);
#'
#' and the scaling exponent, for the "BA" model (Barabási-Albert model).
#'
#' @param bandwidth string indicating which criterion will be used to choose 
#' the bandwidth for the spectral density estimation. The available criteria are 
#' "Silverman" (default) and "Sturges".
#'
#' @param eigenvalues optional parameter. It contains the eigenvalues of matrix
#' A. Then, it can be used when the eigenvalues of A were previously computed. 
#' If no value is passed, then the eigenvalues of A will be computed by 'GIC'.
#'
#' @return A real number corresponding to the Kullback-Leibler divergence 
#' between the observed graph and the graph model.
#'
#' @keywords graph_information_criterion
#'
#' @references
#' Takahashi, D. Y., Sato, J. R., Ferreira, C. E. and Fujita A. (2012) 
#' Discriminating Different Classes of Biological Networks by Analyzing the 
#' Graph Spectra  Distribution. _PLoS ONE_, *7*, e49949. 
#' doi:10.1371/journal.pone.0049949.
#'
#' Silverman, B. W. (1986) _Density Estimation_.  London: Chapman and Hall.
#'
#' Sturges, H. A. The Choice of a Class Interval. _J. Am. Statist. Assoc._, 
#' *21*, 65-66.
#' 
#' @examples
#' require(igraph)
#' A <- as.matrix(get.adjacency(erdos.renyi.game(100, p=0.5)))
#' # Using a string to indicate the graph model
#' result1 <- GIC(A, "ER", 0.5)
#' result1
#'
#' # Using a function to describe the graph model
#' # Erdos-Renyi graph
#' model <- function(n, p) {
#'    return(as.matrix(get.adjacency(erdos.renyi.game(n, p))))
#' }
#' result2 <- GIC(A, model, 0.5)
#' result2
#' @export
GIC <- function(A, model, p=NULL, bandwidth="Silverman", eigenvalues=NULL) {
    if (is.null(eigenvalues))
        eigenvalues <- (as.numeric(eigen(A, only.values = TRUE, 
                                         symmetric=TRUE)$values)/sqrt(nrow(A)))
    if (class(model) == "list") {
        f2 <- model
        f1 <- gaussianDensity(eigenvalues, from=min(f2$x), to=max(f2$x), 
                              bandwidth=bandwidth, npoints=1024)
    }
    else if (class(model) == "matrix") {
        f2 <- nDensities(model, from=min(eigenvalues), 
                                to=max(eigenvalues), bandwidth=bandwidth,
                                npoints=1024)
        if (sum(is.na(f2)) > 0)
            return(Inf)
        f2 <- list("x"=f2$x, "y"=rowMeans(f2$densities))
        f1 <- gaussianDensity(eigenvalues, from=min(f2$x), to=max(f2$x), 
                              bandwidth=bandwidth, npoints=1024)
    }
    else {
        fun <- model
        if (class(model) == "character") {
            if (model == "WS")
                fun <- WSfun(as.integer(sum(A)/(2*ncol(A))))
            else fun <- matchFunction(model)
        }
        f2 <- modelSpectralDensity(fun, ncol(A), p, from=min(eigenvalues), 
                                   to=max(eigenvalues), bandwidth=bandwidth,
                                   npoints=1024)
        if (sum(is.na(f2)) > 0)
            return(Inf)
        f1 <- gaussianDensity(eigenvalues, from=min(f2$x), to=max(f2$x), 
                              bandwidth=bandwidth, npoints=1024)
    }
    if (sum(is.na(f1)) > 0)
        return(Inf)
    if (sum(is.na(f2)) > 0)
        return(Inf)
    return(KL(f1, f2))
}

#' Graph parameter estimator
#'
#' 'graph.param.estimator' estimates the parameter that best approximates the 
#' model to the observed graph according to the Graph Information Criterion 
#' (GIC).
#'
#' @param A the adjacency matrix of the graph. For an unweighted graph it 
#' contains only 0s and 1s. For a weighted graph, it may contain nonnegative 
#' real values that correspond to the weights of the edges.
#'
#' @param model either a string, a function or an array describing a graph 
#' model: 
#'
#' A string that indicates one of the following models: "ER" (Erdos-Renyi random 
#' graph), "GRG" (geometric random graph), "KR" (k regular random graph), "WS" 
#' (Watts-Strogatz model), and "BA" (Barabási-Albert model). 
#'
#' A function that returns a graph (represented by its adjacency matrix) 
#' generated by a graph model. It must contain two arguments: the first one
#' correponds to the graph size and the second to the parameter of the model.
#'
#' A three-dimensional array containing the spectrum of the model. For each 
#' parameter p that will be considered, the array contains the eigenvalues of N 
#' graphs that were generated by the model with parameter p. The position 
#' (i,j,k) contains the j-th eigenvalue of the k-th graph that were generated 
#' with the i-th parameter. The attribute 'rownames' of the array corresponds to 
#' the values of the parameters converted to string. To obtain the spectral 
#' density for each parameter p, the method will estimate the density of the 
#' eigenvalues of each graph that were generated with parameter p, and then will 
#' take the average density. 
#'
#' @param parameters numeric vector containing the values that that will be 
#' considerated for the parameter estimation. The 'graph.param.estimator' will
#' return the element of 'parameter' that minimizes the Graph Information 
#' Criterion (GIC). 
#' If the user does not provide the argument 'parameters', and 'model' is an 
#' array, then the values considered for the parameter estimation are the 
#' rownames converted to a numeric vector. If 'model' is a string, then
#' default values are used for the predefined models ("ER", "GRG", "KR", "WS", 
#' and "BA"). The default 'parameter' argument corresponds to a sequence from 
#' 
#' 0 to 1 with step 'eps' for the "ER" model (Erdos-Renyi random graph), in 
#' which the parameter corresponds to the probability to connect a pair of 
#' vertices;
#' 
#' 0 to sqrt(2) with step 'eps' for the "GRG" model (geometric random graph), in 
#' which the parameter corresponds to the radius used to contruct the geometric 
#' graph in a unit square;
#'
#' 0 to 'n' with step 'n*eps' for the "KR" model (k regular random graph), in 
#' which the parameter of the model corresponds to the degree 'k' of a regular 
#' graph;
#' 
#' 0 to 1 with step 'eps' for the "WS" model (Watts-Strogatz model), in which 
#' the parameter corresponds to the probability to reconnect a vertex;
#'
#' and 0 to 3 with step 'eps' for the "BA" model (Barabási-Albert model), in  
#' which the parameter corresponds to the scaling exponent.
#'
#' @param eps precision of the grid (default is 0.01).
#'
#' @param bandwidth string indicating which criterion will be used to choose the 
#' bandwidth for the spectral density estimation. The available criteria are 
#' "Silverman" (default) and "Sturges".
#'
#' @param eigenvalues optional parameter. It contains the eigenvalues of matrix
#' A. Then, it can be used when the eigenvalues of A were previously computed. 
#' If no value is passed, then the eigenvalues of A will be computed by 
#' 'graph.param.estimator'.
#'
#' @return A list containing:
#' \item{p}{the parameter estimate. For the "ER", "GRG", "KR", "WS", and "BA"
#' models, the parameter corresponds to the probability to connect a pair of 
#' vertices, the radius used to contruct the geometric graph in a unit square,
#' the degree k of a regular graph, the probability to reconnect a vertex, and 
#' the scaling exponent, respectively.}  
#' \item{KL}{the Graph Information Criterion (GIC), i. e. the Kullback-Leibler 
#' divergence between the observed graph and the graph model with the estimated
#' parameter.}
#'
#' @keywords parameter_estimation
#'
#' @references
#' Takahashi, D. Y., Sato, J. R., Ferreira, C. E. and Fujita A. (2012) 
#' Discriminating Different Classes of Biological Networks by Analyzing the 
#' Graph Spectra  Distribution. _PLoS ONE_, *7*, e49949. 
#' doi:10.1371/journal.pone.0049949.
#' 
#' Silverman, B. W. (1986) _Density Estimation_.  London: Chapman and Hall.
#'
#' Sturges, H. A. The Choice of a Class Interval. _J. Am. Statist. Assoc._, 
#' *21*, 65-66.
#' 
#' @examples
#' require(igraph)
#' A <- as.matrix(get.adjacency(erdos.renyi.game(50, p=0.5)))
#'
#' # Using a string to indicate the graph model
#' result1 <- graph.param.estimator(A, "ER", eps=0.25)
#' result1
#'
#' # Using a function to describe the graph model
#' # Erdos-Renyi graph
#' model <- function(n, p) {
#'    return(as.matrix(get.adjacency(erdos.renyi.game(n, p))))
#' }
#' result2 <- graph.param.estimator(A, model,  seq(0.2, 0.8, 0.1))
#' result2
#'
#' @export
graph.param.estimator <- function(A, model, parameters=NULL, eps=0.01, 
                                  bandwidth="Silverman", eigenvalues=NULL) {
    n <- ncol(A)
    if (class(model) == "function" && is.null(parameters)) {
        stop("It is necessary to enter the parameters that will be evaluated.")
    }
    if (class(model) == "array") {
        if (is.null(parameters))
            parameters <- as.numeric(rownames(model))
    }
    else if (is.null(parameters)) {
            parameters <- seq(0, 1, eps)
            if (model == "GRG")
                parameters <- seq(0, sqrt(2), eps)
            if (model == "BA")
                parameters <- seq(0, 3, eps)
            if (model == "KR")
                parameters <- as.integer(seq(0, 1, eps)*n)
    }
    if (is.null(eigenvalues))
        eigenvalues <- (as.numeric(eigen(A, only.values = TRUE)$values)/
                            sqrt(nrow(A)))
    pmin <- -1
    klmin <- Inf
    for (p in parameters) {
       if (class(model) == "array")
            kl <- GIC(A, model[as.character(p),,], p, bandwidth, 
                      eigenvalues=eigenvalues) 
       else 
            kl <- GIC(A, model, p, bandwidth, eigenvalues=eigenvalues) 
       if (kl < klmin) {
           klmin <- kl
           pmin <- p
       }
    }
    return(list("p"=pmin, "GIC"=klmin))
}

#' Graph model selection
#'
#' 'graph.model.selection' selects the graph model that best approximates the 
#' observed graph according to the Graph Information Criterion (GIC).
#'
#' @param A the adjacency matrix of the graph. For an unweighted graph it 
#' contains only 0s and 1s. For a weighted graph, it may contain nonnegative 
#' real values that correspond to the weights of the edges.
#'
#' @param models either a vector of strings, a list of functions or a list of 
#' arrays describing graph models: 
#' 
#' A vector of strings cotaining some of the following models: "ER" (Erdos-Renyi 
#' random graph), "GRG" (geometric random graph), "KR" (k regular random graph), 
#' "WS" (Watts-Strogatz model), and "BA" (Barabási-Albert model).
#'
#' A list of functions. Each function returns a graph (represented by its 
#' adjacency matrix) generated by a graph model and has two arguments: the graph 
#' size and the model parameter, in this order.
#' 
#' A list of arrays. Each elememt of the list is a three-dimensional array 
#' containing the spectrum of the corresponding model. Let M be a graph model.
#' For each parameter p considered for M, the array of model M contains the 
#' eigenvalues of N graphs that were generated by M with parameter p. The 
#' position (i,j,k) of the array contains the j-th eigenvalue of the k-th graph 
#' that were generated by M with the i-th parameter. The attribute 'rownames' of 
#' the array corresponds to the values of the parameter converted to string. To 
#' obtain the spectral density for each parameter p, the method will estimate 
#' the density of the eigenvalues of each graph that were generated by M with 
#' parameter p, and then will take the average density. 
#'
#' If the argument "models" is NULL, then the "ER", "WS", and "BA" models will 
#' be considered for the model selection.
#'
#' @param parameters list of numeric vectors. Each vector contains the values 
#' that will be considerated for the parameter estimation of the corresponding 
#' model. 
#' If the user does not provide the argument 'parameters', then default values 
#' are used for the predefined models ("ER", "GRG", "KR", "WS", and "BA"). 
#' The default vector corresponds to a sequence from 
#' 
#' 0 to 1 with step 'eps' for the "ER" model (Erdos-Renyi random graph), in 
#' which the parameter corresponds to the probability to connect a pair of 
#' vertices;
#' 
#' 0 to sqrt(2) with step 'eps' for the "GRG" model (geometric random graph), in 
#' which the parameter corresponds to the radius used to contruct the geometric 
#' graph in a unit square;
#'
#' 0 to 'n' with step 'n*eps' for the "KR" model (k regular random graph), in 
#' which the parameter of the model corresponds to the degree 'k' of a regular 
#' graph;
#' 
#' 0 to 1 with step 'eps' for the "WS" model (Watts-Strogatz model), in which 
#' the parameter corresponds to the probability to reconnect a vertex;
#'
#' and 0 to 3 with step 'eps' for the "BA" model (Barabási-Albert model), in  
#' which the parameter corresponds to the scaling exponent.
#'
#' @param eps precision of the grid (default is 0.01).
#'
#' @param bandwidth string indicating which criterion will be used to choose 
#' the bandwidth for the spectral density estimation. The available criteria are 
#' "Silverman" (default) and "Sturges".
#'
#' @param eigenvalues optional parameter. It contains the eigenvalues of matrix
#' A. Then, it can be used when the eigenvalues of A were previously computed. 
#' If no value is passed, then the eigenvalues of A will be computed by 
#' 'graph.model.selection'.
#'
#' @return A list containing:
#' \item{model}{the indice(s) or name(s) of the selected model(s), i. e. the
#' model(s) that minimize(s) the Graph Information Criterion (GIC).}  
#' \item{estimates}{a matrix in which each row corresponds to a model, the 
#' column "p" corresponds to the parameter estimate, and the column "GIC" 
#' corresponds to the Graph Information Criterion (GIC), i. e. the 
#' Kullback-Leibler divergence between the observed graph and the model.}
#'
#' @keywords model_selection
#'
#' @references
#' Takahashi, D. Y., Sato, J. R., Ferreira, C. E. and Fujita A. (2012) 
#' Discriminating Different Classes of Biological Networks by Analyzing the 
#' Graph Spectra  Distribution. _PLoS ONE_, *7*, e49949. 
#' doi:10.1371/journal.pone.0049949.
#' 
#' Silverman, B. W. (1986) _Density Estimation_.  London: Chapman and Hall.
#'
#' Sturges, H. A. The Choice of a Class Interval. _J. Am. Statist. Assoc._, 
#' *21*, 65-66.
#' 
#' 
#' @examples
#' 
#' require(igraph)
#' A <- as.matrix(get.adjacency(erdos.renyi.game(30, p=0.5)))
#' # Using strings to indicate the graph models
#' result1 <- graph.model.selection(A, models=c("ER", "WS"), eps=0.5)
#' result1
#' # Using functions to describe the graph models
#' # Erdos-Renyi graph
#' model1 <- function(n, p) {
#'    return(as.matrix(get.adjacency(erdos.renyi.game(n, p))))
#' }
#' # Watts-Strougatz graph
#' model2 <- function(n, pr, K=8) {
#'     return(as.matrix(get.adjacency(watts.strogatz.game(1, n, K, pr))))
#' }
#' parameters <- list(seq(0, 1, 0.5), seq(0, 1, 0.5))
#' result2 <- graph.model.selection(A, list(model1, model2), parameters)
#' result2
#' @export
graph.model.selection <- function(A, models=NULL, parameters=NULL, eps=0.01,
                                  bandwidth="Silverman", eigenvalues=NULL) {
    n <- ncol(A)
    if (class(models) == "list" && is.null(parameters)) {
        stop("It is necessary to enter the parameters that will be evaluated.")
    }
    if (is.null(models)) {
        models <- c("ER", "WS", "BA")
    }
    results <- matrix(NA, length(models), 2)
    colnames(results) <- c("p", "GIC")
    if (class(models) == "character") {
        rownames(results) <- models
    }
    if (class(models) == "list") {
        if (!is.null(names(parameters)))
            rownames(results) <- names(parameters)
    }
    p <- NULL
    if (is.null(eigenvalues))
        eigenvalues <- (as.numeric(eigen(A, only.values=TRUE, 
                                         symmetric=TRUE)$values)/sqrt(nrow(A)))
    for (i in 1:length(models)) {
        if (!is.null(parameters))
            p <- parameters[[i]]
        r <- graph.param.estimator(A, models[[i]], p, eps, bandwidth, 
                                   eigenvalues=eigenvalues) 
        results[i, "p"] <- r$p
        results[i, "GIC"] <- r$GIC
    }
    m <- which(results[, "GIC"] == min(results[, "GIC"]))
    if (!is.null(rownames(results)))
        m <- rownames(results)[m]
    return(list("model"=m, "estimates"=results))
}

#' Test for the Jensen-Shannon divergence between graphs
#'
#' 'graph.test' tests whether two sets of graphs were generated by the same 
#' random graph model.
#' This bootstrap test is based on the Jensen-Shannon (JS) divergence between 
#' graphs.
#'
#' Given two lists of graphs, 'x' and 'y', 'graph.test' tests H0: "JS divergence 
#' between 'x' and 'y' is 0" against H1: "JS divergence between 'x' and 'y' is 
#' larger than 0".
#'
#' @param x a list of adjacency matrices. For unweighted graphs, each matrix 
#' contains only 0s and 1s. For weighted graphs, each matrix may contain 
#' nonnegative real values that correspond to the weights of the edges.
#'
#' @param y a list of adjacency matrices. For unweighted graphs, each matrix 
#' contains only 0s and 1s. For weighted graphs, each matrix may contain 
#' nonnegative real values that correspond to the weights of the edges.
#'
#' @param numBoot integer indicating the number of bootstrap resamplings.
#'
#' @param bandwidth string indicating which criterion will be used to choose 
#' the bandwidth for the spectral density estimation. The available criteria are 
#' "Silverman" (default) and "Sturges".
#'
#' @return A list containing:
#' \item{JS}{the Jensen-Shannon divergence between 'x' and 'y'.}  
#' \item{p.value}{the p-value of the test.}
#'
#' @keywords graph_comparison
#'
#' @references
#' Takahashi, D. Y., Sato, J. R., Ferreira, C. E. and Fujita A. (2012) 
#' Discriminating Different Classes of Biological Networks by Analyzing the 
#' Graph Spectra  Distribution. _PLoS ONE_, *7*, e49949. 
#' doi:10.1371/journal.pone.0049949.
#' 
#' Silverman, B. W. (1986) _Density Estimation_.  London: Chapman and Hall.
#'
#' Sturges, H. A. The Choice of a Class Interval. _J. Am. Statist. Assoc._, 
#' *21*, 65-66.
#' 
#' @examples
#' library(igraph)
#' x <- y <- list()
#' for (i in 1:20)
#'    x[[i]] <- as.matrix(get.adjacency(erdos.renyi.game(50, p=0.5)))
#' for (i in 1:20)
#'    y[[i]] <- as.matrix(get.adjacency(erdos.renyi.game(50, p=0.51)))
#' 
#' result <- graph.test(x, y, numBoot=100)
#' result
#'
#' @export
graph.test <- function(x, y, numBoot=1000, bandwidth="Silverman") {
    adjacencyMatrices <- append(x, y)
    labels <- c(rep(0, length(x)), rep(1, length(y)))
    f <- nSpectralDensities(adjacencyMatrices, bandwidth=bandwidth)
    densities <- f$densities
    x <- f$x
    y1 <- rowMeans(densities[, labels==0])
    y2 <- rowMeans(densities[, labels==1])
    n1 <- length(which(labels==0))
    n2 <- length(which(labels==1))
    results <- vector(length=numBoot)
    ngraphs <- length(adjacencyMatrices)
    result <- JS(list("x"=x, "y"=y1), list("x"=x, "y"=y2))
    for (i in 1:numBoot) {
        b1 <- sample(1:ngraphs, n1, replace=T)
        b2 <- sample(1:ngraphs, n2, replace=T)
        y1 <- rowMeans(densities[, b1])
        y2 <- rowMeans(densities[, b2])
        results[i] <- JS(list("x"=x, "y"=y1), list("x"=x, "y"=y2))
    }
    pvalue <- (sum(results >= result))/numBoot
    return(list("JS"=result, "p.value"=pvalue))
}

#' ANOGVA Analysis Of Graph Variability
#'
#' 'anogva' statistically tests whether two or more sets of graphs are generated 
#' by the same random graph model. It is a generalization of the 'graph.test' 
#' function.
#'
#' @param graphs a list of adjacency matrices. For unweighted graphs, each 
#' matrix contains only 0s and 1s. For weighted graphs, each matrix may contain 
#' nonnegative real values that correspond to the weights of the edges.
#'
#' @param labels an array of integers indicating the labels of each graph.
#'
#' @param numBoot integer indicating the number of bootstrap resamplings.
#'
#' @param bandwidth string indicating which criterion will be used to choose 
#' the bandwidth for the spectral density estimation. The available criteria are 
#' "Silverman" (default) and "Sturges".
#'
#' @return A list containing:
#' \item{statistic}{the statistic of the test.}  
#' \item{p.value}{the p-value of the test.}
#'
#' @keywords analysis_of_graph_variability
#'
#' @examples
#' 
#' require(igraph)
#' g1 <- g2 <- g3 <- list()
#' for (i in 1:20) {
#'    G1 <- erdos.renyi.game(50, 0.50)
#'    g1[[i]] <- get.adjacency(G1)
#'    G2 <- erdos.renyi.game(50, 0.50)
#'    g2[[i]] <- get.adjacency(G2)
#'    G3 <- erdos.renyi.game(50, 0.52)
#'    g3[[i]] <- get.adjacency(G3)
#' }
#' g <- c(g1, g2, g3)
#' label <- c(rep(1,20),rep(2,20),rep(3,20))
#' result <- anogva(g, label, numBoot=50)
#' result
#' 
#' @references
#'
#' Fujita, A., Vidal, M. C. and Takahashi, D. Y. (2017) A Statistical Method to 
#' Distinguish Functional Brain Networks. _Front. Neurosci._, *11*, 66.
#' doi:10.3389/fnins.2017.00066.
#'
#' Takahashi, D. Y., Sato, J. R., Ferreira, C. E. and Fujita A. (2012) 
#' Discriminating Different Classes of Biological Networks by Analyzing the 
#' Graph Spectra  Distribution. _PLoS ONE_, *7*, e49949. 
#' doi:10.1371/journal.pone.0049949.
#' 
#' Silverman, B. W. (1986) _Density Estimation_.  London: Chapman and Hall.
#'
#' Sturges, H. A. The Choice of a Class Interval. _J. Am. Statist. Assoc._, 
#' *21*, 65-66.
#' @export
anogva <- function(graphs, labels, numBoot=1000, bandwidth="Silverman") {
    f <- nSpectralDensities(graphs, bandwidth=bandwidth)
    densidade <- f$densities
    x_axis <- f$x
    band <- length(x_axis)
        
    media <- matrix(0, max(labels), band)
    mediaAll <- list()
    mediaAll$y <- array(0,band)
    for (j in 1:band) {
        for (i in 1:max(labels)) {
            media[i,j] <- mean(densidade[j, which(labels==i)])
        }
        mediaAll$y[j] <- mean(media[,j])
    }
    mediaAll$x <- x_axis

    distOrig <- 0
    meanGroup <- list()
    meanGroup$x <- x_axis

    for(i in 1:max(labels)) {
        meanGroup$y <- media[i,]
        distOrig <- distOrig + KL(meanGroup, mediaAll)
    }
    distOrig <- distOrig / max(labels)


    ## Permutation test
    distBoot <- array(0, numBoot)

    for (boot in 1:numBoot) {
        labelsB <- sample(labels, length(labels), replace=FALSE)

        mediaB <- matrix(0, max(labels), band)

        for (j in 1:band) {
            for (i in 1:max(labels)) {
                mediaB[i,j] <- mean(densidade[j, which(labelsB==i)])
            }
        }

        meanBoot <- list()
        meanBoot$x <- x_axis
        for(i in 1:max(labelsB)) {
            meanBoot$y <- mediaB[i,]
            distBoot[boot] <- distBoot[boot] + KL(meanBoot, mediaAll)
        }
        distBoot[boot] <- distBoot[boot] / max(labels)
    }

    pvalue <- length(which(distBoot >= distOrig)) / (numBoot+1)

    return(list("statistic"=distOrig, "p.value"=pvalue))
}

#' Test for Association / Correlation Between Paired Samples of Graphs
#'
#' 'graph.cor.test' tests for association between paired samples of graphs, 
#' using Spearman's rho correlation coefficient.
#'    
#' @param x a list of adjacency matrices. For unweighted graphs, each matrix 
#' contains only 0s and 1s. For weighted graphs, each matrix may contain 
#' nonnegative real values that correspond to the weights of the edges.
#'
#' @param y a list with the same length of 'x'. It is list of adjacency 
#' matrices. For unweighted graphs, each matrix contains only 0s and 1s. For 
#' weighted graphs, each matrix may contain nonnegative real values that 
#' correspond to the weights of the edges.
#'
#' @return 
#' \item{statistic}{ the value of the test statistic.}  
#' \item{p.value}{the p-value of the test.}
#' \item{estimate}{the estimated measure of association 'rho'.}
#'
#' @keywords correlation_coefficient
#' 
#' @references   
#' Fujita, A., Takahashi, D. Y., Balardin, J. B., Vidal, M. C. and Sato, J. R. 
#' (2017) Correlation between graphs with an application to brain network 
#' analysis. _Computational Statistics & Data Analysis_ *109*, 76–92.
#'
#' @examples
#' require(igraph)
#' x <- list()
#' y <- list()
#'
#' p <- MASS::mvrnorm(50, mu=c(0,0), Sigma=matrix(c(1, 0.5, 0.5, 1), 2, 2))
#'
#' ma <- max(p)
#' mi <- min(p)
#' p[,1] <- (p[,1] - mi)/(ma - mi)
#' p[,2] <- (p[,2] - mi)/(ma - mi)
#'
#' for (i in 1:50) {
#'     x[[i]] <- get.adjacency(erdos.renyi.game(50, p[i,1]))
#'     y[[i]] <- get.adjacency(erdos.renyi.game(50, p[i,2]))
#' }
#'
#' graph.cor.test(x, y)
#'
#' @import stats
#' @import MASS
#' @export
graph.cor.test <- function(x, y) {

    x.radius <- array(0, length(x))
    y.radius <- array(0, length(y))

    for (i in 1:length(x)) {
        x.radius[i] <- eigen(x[[i]], only.values=TRUE, symmetric=TRUE)$values[1]
        y.radius[i] <- eigen(y[[i]], only.values=TRUE, symmetric=TRUE)$values[1]
    }

    return(cor.test(x.radius, y.radius, method="spearman"))
}

#' Auto Correlation Function Estimation for Graphs
#'
#' The function 'graph.acf' computes estimates of the autocorrelation function
#' for graphs. 
#'    
#' @param x a list of adjacency matrices. For unweighted graphs, each matrix 
#' contains only 0s and 1s. For weighted graphs, each matrix may contain 
#' nonnegative real values that correspond to the weights of the edges.
#'
#' @param plot logical. If TRUE (default) the graph.acf is plotted.
#'
#' @return An object of class acf.
#'
#' @keywords autocorrelation
#'    
#' @references   
#' Fujita, A., Takahashi, D. Y., Balardin, J. B., Vidal, M. C. and Sato, J. R. 
#' (2017) Correlation between graphs with an application to brain network 
#' analysis. _Computational Statistics & Data Analysis_ *109*, 76–92.
#'  
#' @examples
#' require(igraph)
#' x <- list()
#' p <- array(0, 100)
#' p[1:3] <- rnorm(3)  
#' for (t in 4:100) {
#'     p[t] <- 0.5*p[t-3] + rnorm(1)
#' }
#' ma <- max(p)
#' mi <- min(p)
#' p <- (p - mi)/(ma-mi)
#' for (t in 1:100) {
#'     x[[t]] <- get.adjacency(erdos.renyi.game(100, p[t]))
#' }
#' graph.acf(x, plot=TRUE)
#'
#' @import stats
#' @export
graph.acf <- function(x, plot=TRUE) {
    x.radius <- array(0, length(x))
    for (t in 1:length(x)) {
        x.radius[t] <- eigen(x[[t]], only.values=TRUE, symmetric=TRUE)$values[1]
    }
    res <- acf(x.radius, plot=plot)
    return(res)
}

#' Hierarchical cluster analysis on a list of graphs.
#'
#' Given a list of graphs, 'graph.cluster' builds a hierarchy of clusters 
#' according to the Jensen-Shannon divergence between graphs.
#'
#' @param x a list of adjacency matrices. For unweighted graphs, each matrix 
#' contains only 0s and 1s. For weighted graphs, each matrix may contain 
#' nonnegative real values that correspond to the weights of the edges.
#' 
#' @param k the number of clusters.
#'
#' @param method the agglomeration method to be used. This should be (an 
#' unambiguous abbreviation of) one of ‘"ward.D"’, ‘"ward.D2"’, ‘"single"’, 
#' ‘"complete"’, ‘"average"’ (= UPGMA), ‘"mcquitty"’ (= WPGMA), ‘"median"’ 
#' (= WPGMC) or ‘"centroid"’ (= UPGMC).
#'
#' @param bandwidth string indicating which criterion will be used to choose 
#' the bandwidth for the spectral density estimation. The available criteria are 
#' "Silverman" (default) and "Sturges".
#'
#' @return A list containing:
#' \item{hclust}{an object of class *hclust* which describes the tree produced
#' by the clustering process.}
#' \item{cluster}{the clustering labels for each graph.}
#'
#' @keywords clustering
#'
#' @references
#' Takahashi, D. Y., Sato, J. R., Ferreira, C. E. and Fujita A. (2012) 
#' Discriminating Different Classes of Biological Networks by Analyzing the 
#' Graph Spectra  Distribution. _PLoS ONE_, *7*, e49949. 
#' doi:10.1371/journal.pone.0049949.
#'
#' Silverman, B. W. (1986) _Density Estimation_.  London: Chapman and Hall.
#'
#' Sturges, H. A. The Choice of a Class Interval. _J. Am. Statist. Assoc._, 
#' *21*, 65-66.
#'
#' @examples
#' require(igraph)
#' g <- list()
#' for (i in 1:5) {
#'     g[[i]] <- as.matrix(get.adjacency(
#'                         erdos.renyi.game(50, 0.5, type="gnp",
#'                                          directed = FALSE)))
#' }
#' for (i in 6:10) {
#'     g[[i]] <- as.matrix(get.adjacency(
#'                         watts.strogatz.game(1, 50, 8, 0.2)))
#' }
#' for (i in 11:15) {
#'     g[[i]] <- as.matrix(get.adjacency(
#'                         barabasi.game(50, power = 1,
#'                                       directed = FALSE)))
#' }
#' graph.cluster(g, 3)
#' 
#' @import stats
#'
#' @export
graph.cluster <- function(x, k, method="complete", bandwidth="Silverman") {

    f <- nSpectralDensities(x, bandwidth=bandwidth)

    d <- matrix(0, length(x), length(x))
    for (i in 1:(length(x)-1)) {
        f1 <- list("x"=f$x, "y"=f$densities[,i])
        for (j in (i+1) : length(x)) {
            f2 <- list("x"=f$x, "y"=f$densities[,j])
            d[i,j] <- d[j,i] <- sqrt(JS(f1, f2))
        }
    }

    tmp <- hclust(as.dist(d), method=method)

    res <- list()
    res$hclust <- tmp
    res$cluster <- cutree(tmp, k)

    return(res)    
}

#' Multidimensional scaling of graphs
#'
#' 'graph.mult.scaling' performs multidimensional scaling of graphs. It takes 
#' the Jensen-Shannon divergence between graphs (JS) and uses the 'cmdscale' 
#' function from the 'stats' package to obtain a set of points such that the 
#' distances between the points are similar to JS. 
#'
#' @param x a list of adjacency matrices. For unweighted graphs, each matrix 
#' contains only 0s and 1s. For weighted graphs, each matrix may contain 
#' nonnegative real values that correspond to the weights of the edges.
#'
#' @param plot logical. If TRUE (default) the points chosen to represent the 
#' Jensen-Shannon divergence between graphs are plotted.
#'
#' @param bandwidth character string indicating which criterion will be used
#' to choose the bandwidth for the spectral density estimation. The available 
#' criteria are "Silverman" (default) and "Sturges".
#'
#' @param type what type of plot should be drawn. The defaut value is '"n"', 
#' which indicates that the points will not be plotted (i. e. only the labels
#' of the graphs will be plotted).                            
#' 
#' @param main title of the plot (default value is "").
#'
#' @param ... additional plotting parameters. See 'plot' function from the 
#' 'graphics' package for the complete list.
#'
#' @return A matrix in which each column corresponds to a coordinate and each 
#' row corresponds to a graph (point). Then, each row gives the coordinates of  
#' the points chosen to represent the Jensen-Shannon divergence between graphs.
#'
#' @keywords multidimensional_scaling
#' 
#' @references
#' Takahashi, D. Y., Sato, J. R., Ferreira, C. E. and Fujita A. (2012) 
#' Discriminating Different Classes of Biological Networks by Analyzing the 
#' Graph Spectra  Distribution. _PLoS ONE_, *7*, e49949. 
#' doi:10.1371/journal.pone.0049949.
#' 
#' Silverman, B. W. (1986) _Density Estimation_.  London: Chapman and Hall.
#'
#' Sturges, H. A. The Choice of a Class Interval. _J. Am. Statist. Assoc._, 
#' *21*, 65-66.
#'
#' @examples
#' require(igraph)
#' g <- list()
#' for (i in 1:5) {
#'     g[[i]] <- as.matrix(get.adjacency(
#'                         erdos.renyi.game(50, 0.5, type="gnp",
#'                                          directed = FALSE)))
#' }
#' for (i in 6:10) {
#'     g[[i]] <- as.matrix(get.adjacency(
#'                         watts.strogatz.game(1, 50, 8, 0.2)))
#' }
#' for (i in 11:15) {
#'     g[[i]] <- as.matrix(get.adjacency(
#'                         barabasi.game(50, power = 1,
#'                                       directed = FALSE)))
#' }
#' graph.mult.scaling(g)
#' 
#' @import graphics
#' @export
graph.mult.scaling <- function(x, plot=TRUE, bandwidth="Silverman", type="n", 
                               main="", ...) {

    f <- nSpectralDensities(x, bandwidth=bandwidth)

    d <- matrix(0, length(x), length(x))
    for (i in 1:(length(x)-1)) {
        f1 <- list("x"=f$x, "y"=f$densities[,i])
        for (j in (i+1) : length(x)) {
            f2 <- list("x"=f$x, "y"=f$densities[,j])
            d[i,j] <- d[j,i] <- sqrt(JS(f1, f2))
        }
    }
    
    if (is.null(names(x)))
        names <- as.character(1:length(x))
    else
        names <- names(x)
    colnames(d) <- rownames(d) <- names
    fit <- cmdscale(as.dist(d), k=2)

    x <- fit[,1]
    y <- fit[,2]
    names(x) <- names
    names(y) <- names
    if (plot) {
        plot(x, y, xlab="x", ylab="y", main=main, type=type, ...)
        text(x, y, labels=names, cex=1) 
    }
    return(fit)
}

################################################################################
## Auxiliar functions
################################################################################

# Returns the Jensen-Shannon divergence between two densities
JS <- function(f1, f2) {
    fm <- f1
    fm$y <- (f1$y + f2$y)/2
    return((KL(f1, fm) + KL(f2, fm))/2)
}

# Returns the Kullback-Leibler divergence between two densities
KL <- function(f1, f2) {
    y <- f1$y
    n <- length(y)
    for (i in 1:n) {
        if (y[i] != 0 && f2$y[i] == 0)
            return(Inf)
        if (y[i] != 0)
            y[i] <- y[i]*log(y[i]/f2$y[i])
    }
    return(trapezoidSum(f1$x, y))
}

# Given a partition x[1]...x[n] and y[i] = f(x[i]), returns the trapezoid sum
# approximation for int_{x[1]}^{x[n]}{f(x)dx}
trapezoidSum <- function (x, y) {
    n <- length(x)
    delta <- (x[2] - x[1])
    area <- sum(y[2:(n-1)])
    area <- (area + (y[1] + y[n])/2)*delta
    return(area)
}

# Returns the kernel bandwidth for a given sample x
kernelBandwidth <- function(x) {
    n <- length(x)
    # Sturges' criterion
    nbins <- ceiling(log2(n) + 1)
    return(abs(max(x) - min(x))/nbins)
}

# Returns the density function for a given sample x at n points in the interval
# [form, to]
gaussianDensity <- function(x, from=NULL, to=NULL, bandwidth="Silverman", 
                            npoints=1024) {
    if (bandwidth == "Sturges")
        bw <- kernelBandwidth(x)
    else if (bandwidth == "Silverman")
        bw <- bw.nrd0(x)
    if (bw == 0)
       return(NA)
    if (is.null(from) || is.null(to))
        f <- density(x, bw=bw, n=npoints)
        #f <- density(x, n=npoints)

    else
        f <- density(x, bw=bw, from=from, to=to, n=npoints)
        #f <- density(x, from=from, to=to, n=npoints)

    area <- trapezoidSum(f$x, f$y)
    return(list("x"=f$x, "y"=f$y/area))
}

# Returns the spectral density for a given adjacency matrix A
spectralDensity <- function(A, from=NULL, to=NULL, bandwidth="Silverman", 
                            npoints=1024) {
    eigenvalues <- (as.numeric(eigen(A, only.values=TRUE, 
                                     symmetric=TRUE)$values)/sqrt(nrow(A)))
    return(gaussianDensity(eigenvalues, from, to, bandwidth=bandwidth, 
                           npoints=npoints))
}

nDensities <- function (spectra, from=NULL, to=NULL, 
                                bandwidth="Silverman", npoints=1024) {
  ngraphs <- ncol(spectra)
  densities <- matrix(NA, npoints, ngraphs)
  minimum <- min(spectra)
  maximum <- max(spectra)
  if (!is.null(from) && !is.null(to)) {
    minimum <- min(minimum, from)
    maximum <- max(maximum, to)
  }
  for (i in 1:ngraphs) {
    f <- gaussianDensity(spectra[,i], bandwidth=bandwidth,
                         from=minimum, to=maximum,
                         npoints=npoints)
    if (sum(is.na(f)) > 0) {
        return(NA)
    }
    else {
        densities[,i] <- f$y
        x <- f$x
    }
  }
  return(list("x"=x, "densities"=densities))
}


# Returns the spectral densities for given adjacency matrices A1 and A2 at the
# same points
spectralDensities <- function(A1, A2, bandwidth="Silverman",
                              npoints=1024) {
    n1 <- nrow(A1)
    n2 <- nrow(A2)
    e1 <- (as.numeric(eigen(A1, only.values = TRUE, 
                            symmetric=TRUE)$values)/sqrt(n1))
    e2 <- (as.numeric(eigen(A2, only.values = TRUE, 
                            symmetric=TRUE)$values)/sqrt(n2))
    #b1 <- kernelBandwidth(e1)
    #b2 <- kernelBandwidth(e2)
    #from <- min(min(e1) - 3*b1, min(e2) - 3*b2)
    #to <- max(max(e1) + 3*b1, max(e2) + 3*b2)
    from <- min(e1, e2)
    to <- max(e1, e2)
    f1 <- gaussianDensity(e1, from=from, to=to, bandwidth=bandwidth, 
                          npoints=npoints)
    f2 <- gaussianDensity(e2, from=from, to=to, bandwidth=bandwidth, 
                          npoints=npoints)
    if (sum(is.na(f1)) > 0 || sum(is.na(f2)) > 0)
        return(NA)
    return(list("f1"=f1, "f2"=f2))
}

# Returns the spectral densities for a list of adjacency matrices at the
# same points
nSpectralDensities <- function (adjacencyMatrices, from=NULL, to=NULL, 
                                bandwidth="Silverman") {
    npoints <- 1024
    ngraphs <- length(adjacencyMatrices)
    ns <- unlist(lapply(adjacencyMatrices, ncol))
    #n <- ncol(adjacencyMatrices[[1]])
    spectra <- matrix(NA, max(ns), ngraphs)
    for (i in 1:ngraphs) {
        A <- adjacencyMatrices[[i]]
        n <- ncol(A)
        eigenvalues <- (as.numeric(eigen(A, only.values = TRUE, 
                                         symmetric=TRUE)$values)/sqrt(n))
        spectra[1:n,i] <- eigenvalues
    }
    densities <- matrix(NA, npoints, ngraphs)
    minimum <- min(spectra, na.rm=T)
    maximum <- max(spectra, na.rm=T)
    if (!is.null(from) && !is.null(to)) {
        minimum <- min(minimum, from)
        maximum <- max(maximum, to)
    }
    for (i in 1:ngraphs) {
        n <- ns[i]
        f <- gaussianDensity(spectra[1:n,i], bandwidth=bandwidth,
                             from=minimum, to=maximum,
                             npoints=npoints)

        densities[,i] <- f$y
        x <- f$x
    }
    return(list("x"=x, "densities"=densities))
}

# Estimates the spectral density of a graph model
modelSpectralDensity <- function(fun, n, p, ngraphs=50, from=NULL, to=NULL, 
                                 bandwidth="Silverman", npoints=1024) {
    spectra <- matrix(NA, n, ngraphs)
    for (i in 1:ngraphs) {
        A <- fun(n, p)
        eigenvalues <- (as.numeric(eigen(A, only.values = TRUE, 
                                         symmetric=TRUE)$values)/sqrt(nrow(A)))
        spectra[,i] <- eigenvalues
    }
    densities <- matrix(NA, npoints, ngraphs)
    minimum <- min(spectra)
    maximum <- max(spectra)
    if (!is.null(from) && !is.null(to)) {
        minimum <- min(minimum, from)
        maximum <- max(maximum, to)
    }
    for (i in 1:ngraphs) {
        f <- gaussianDensity(spectra[,i], from=minimum, to=maximum, 
                             bandwidth=bandwidth, npoints=npoints)

        densities[,i] <- f$y
        x <- f$x
    }
    return(list("x" = x, "y" = rowMeans(densities)))
}

modelSpectra <- function(model, n, p, ngraphs=50) {
    fun <- model
    if (is.character(model)) {
        if (model == "WS")
            fun <- WSfun(2)
        else fun <- matchFunction(model)
    }
    spectra <- matrix(NA, n, ngraphs)
    for (i in 1:ngraphs) {
        A <- fun(n, p)
        eigenvalues <- (as.numeric(eigen(A, only.values = TRUE, 
                                         symmetric=TRUE)$values)/sqrt(nrow(A)))
        spectra[,i] <- eigenvalues
    }
   return(spectra)
}

matchFunction <- function(name) {
    return(match.fun(name))
}

# Watts-Strougatz graph
WSfun <- function(K){
    f <- function(n, pr) {
        WS(n, pr, K=K)
    }
    return(f)
}

# Graph models  ----------------------------------------------------------------

# Erdos-Renyi graph 
ER <- function(n, p) {
    return(as.matrix(igraph::get.adjacency(igraph::erdos.renyi.game(n, p, 
                                                                   type="gnp",
                                                            directed = FALSE))))
}

# Geometric graph
GRG <- function(n, r) {
    return(as.matrix(igraph::get.adjacency(igraph::grg.game(n, r))))
}

# Barabasi-Albert graph
BA <- function(n, ps) {
    return(as.matrix(igraph::get.adjacency(igraph::barabasi.game(n, power = ps,
                                                 directed = FALSE))))
}

# Watts-Strogatz graph
WS <- function(n, pr, K=8) {
    return(as.matrix(igraph::get.adjacency(igraph::watts.strogatz.game(1, n, K, 
                                                                       pr))))
}

# K-regular game
KR <- function(n, k) {
    return(as.matrix(igraph::get.adjacency(igraph::k.regular.game(n, k))))
}
