#' Generate Kaplan-Meier curves with prediction intervals from parametric bootstrap simulation
#'
#' @export
#' @param sim A `survparamsim` class object generated by \code{\link{surv_param_sim}} function.
#' @param trt An optional string to specify which column define treatment status.
#' You will have survival curves with different colors in \code{\link{plot_km_pi}} function.
#' @param group Optional string(s) to specify grouping variable(s).
#' You will have faceted survival curves for these variables in \code{\link{plot_km_pi}} function.
#' @param pi.range Prediction interval for simulated survival curves.
#' @param calc.obs A logical to specify whether KM estimates will be performed
#' for the observed data. Need be set as FALSE if survival information in the `newdata`` is dummy.
#' @param simtimelast An optional numeric to specify last simulation time for survival curve.
#' If NULL (default), the last observation time in the `newdata`` will be used.
#'
#'
calc_km_pi <- function(sim, trt=NULL, group=NULL, pi.range = 0.95,
                       calc.obs = TRUE, simtimelast = NULL){

  # Replace with packageVersion("tidyr") == '1.0.0' if nest issue is resolved in the next version
  # See https://github.com/tidyverse/tidyr/issues/751
  nest2 <- ifelse(utils::packageVersion("tidyr") >= '1.0.0', tidyr::nest_legacy, tidyr::nest)
  unnest2 <- ifelse(utils::packageVersion("tidyr") >= '1.0.0', tidyr::unnest_legacy, tidyr::unnest)

  ###### Need to throw an error if grouping variable is not present in newdata
# browser()

  if(length(trt) > 1) stop("`trt` can only take one string")

  trt.syms   <- rlang::syms(trt)
  group.syms <- rlang::syms(group)

  # Calc time for output
  formula <-
    paste(attributes(formula(sim$survreg))$variables,"~1")[2] %>%
    stats::as.formula()

  ## Last observed time
  ## Calculate last time from original dataset just in case newdata's survival data is dummy
  t.last.newdata  <- survival::survfit(formula, data = sim$newdata) %>% .$time
  t.last.origdata <- as.numeric(sim$survreg$y[,1])

  t.last <-max(c(t.last.newdata, t.last.origdata))

  ## time for output
  if(is.null(simtimelast)){
    t.out <- seq(0, t.last, length.out = 100)
  } else {
    t.out <- seq(0, simtimelast, length.out = round(100 * max(simtimelast/t.last, 1)))
  }


  # Define function to approximate or extract KM curves from KM fit object
  approx_km <- function(x){
    surv <- stats::approx(c(0,x$time), c(1,x$surv), xout=t.out, method="constant", rule=2)$y
    data.frame(time = t.out,
               surv = surv)
  }
  extract_km_obs <- function(x){
    data.frame(time = c(0, x$time),
               surv = c(1, x$surv),
               cnsr = c(0, x$n.censor))
  }


  if(calc.obs){
    # Fit K-M curve to observed data
    obs.nested <-
      sim$newdata.nona.obs %>%
      dplyr::group_by(!!!trt.syms, !!!group.syms) %>%
      nest2()

    ## Define formula
    formula <-
      paste(attributes(formula(sim$survreg))$variables,"~1")[2] %>%
      stats::as.formula()


    ## Calc median and KM curve
    obs.km.nested <-
      obs.nested %>%
      dplyr::mutate(kmfit = purrr::map(data, function(x) survival::survfit(formula, data=x))) %>%
      dplyr::mutate(median = purrr::map_dbl(kmfit, function(x) summary(x)$table["median"]),
                    n      = purrr::map_dbl(kmfit, function(x) summary(x)$table["records"]),
                    km = purrr::map(kmfit, extract_km_obs))

    obs.km <-
      obs.km.nested %>%
      unnest2(km) %>%
      dplyr::filter(!is.na(surv)) %>%
      dplyr::select(-median)

    obs.median.time <-
      obs.km.nested %>%
      dplyr::select(!!!trt.syms, !!!group.syms, median, n)

  } else {
    obs.km <- NULL
    obs.median.time <- NULL
  }


  # Calculate percentiles for simulated data

  ## First nest data - KM fit will done for each nested data

  newdata.group <-
    sim$newdata.nona.sim %>%
    dplyr::select(subj.sim, !!!trt.syms, !!!group.syms)

  sim.grouped <-
    sim$sim %>%
    dplyr::left_join(newdata.group, by = "subj.sim") %>%
    dplyr::group_by(rep, !!!trt.syms, !!!group.syms)

  sim.nested <- nest2(sim.grouped)

  sim.km <-
    sim.nested %>%
    # Fit each nested data to KM
    dplyr::mutate(kmfit =
                    purrr::map(data, function(x) survival::survfit(Surv(time, event)~1, data=x))) %>%
    # Calc median and KM curve
    dplyr::mutate(median = purrr::map_dbl(kmfit, function(x) summary(x)$table["median"]),
                  n      = purrr::map_dbl(kmfit, function(x) summary(x)$table["records"]),
                  km = purrr::map(kmfit, approx_km))


  ## Calc quantile for survival curves
  sim.km.quantile <-
    sim.km %>%
    unnest2(km) %>%
    dplyr::group_by(!!!trt.syms, !!!group.syms, n, time) %>%
    nest2() %>%
    dplyr::mutate(quantiles = purrr::map(data, function(x)
      dplyr::summarize(x,
                       pi_low = stats::quantile(surv, probs = 0.5 - pi.range/2),
                       pi_med = stats::quantile(surv, probs = 0.5),
                       pi_high= stats::quantile(surv, probs = 0.5 + pi.range/2)))) %>%
    unnest2(quantiles) %>%
    dplyr::select(-data)


  ## Calc quantiles for median survival time
  sim.median.time <-
    sim.km %>%
    dplyr::select(rep, !!!trt.syms, !!!group.syms, median, n)

  quantiles <-
    tibble::tibble(description = c("pi_low", "pi_med", "pi_high"),
                   quantile = c(0.5 - pi.range/2, 0.5, 0.5 + pi.range/2))

  sim.median.pi <-
    sim.median.time %>%
    dplyr::group_by(!!!trt.syms, !!!group.syms, n) %>%
    dplyr::summarize(pi_low = as.numeric(stats::quantile(median, probs = 0.5 - pi.range/2, na.rm = TRUE)),
                     pi_med = as.numeric(stats::quantile(median, probs = 0.5, na.rm = TRUE)),
                     pi_high= as.numeric(stats::quantile(median, probs = 0.5 + pi.range/2, na.rm = TRUE))) %>%
    dplyr::ungroup() %>%
    tidyr::gather(description, median, pi_low:pi_high) %>%
    dplyr::left_join(quantiles, by = "description")

  if(calc.obs){
    obs.median <-
      obs.median.time %>%
      dplyr::mutate(description = "obs")

    median.pi <-
      dplyr::bind_rows(sim.median.pi, obs.median) %>%
      dplyr::arrange(!!!trt.syms, !!!group.syms)

  } else {
    median.pi <-
      sim.median.pi %>%
      dplyr::arrange(!!!trt.syms, !!!group.syms)
  }


  # Output
  out <- list()

  out$calc.obs <- calc.obs
  out$pi.range   <- pi.range

  out$group <- group
  out$trt   <- trt

  out$simtimelast <- simtimelast
  out$t.last <- t.last
  out$censor.dur <- sim$censor.dur

  out$obs.km <- obs.km
  out$obs.median.time <- obs.median.time

  out$sim.km <- sim.km
  out$sim.km.quantile <- sim.km.quantile
  out$sim.median.time <- sim.median.time

  out$median.pi <- median.pi

  structure(out, class = c("survparamsim.kmpi"))
}


#' Plot Kaplan-Meier curves with prediction intervals from parametric bootstrap simulation
#'
#' Need to think about how to apply this for subgroups
#'
#' @export
#' @param km.pi an output from \code{\link{calc_km_pi}} function.
#' @param show.obs A logical specifying whether to show observed K-M curve on the plot.
#'   This will have no effect if `calc.obs` was set to `FALSE` in \code{\link{calc_km_pi}}.
#' @param trunc.sim.censor A logical specifying whether to truncate the simulated
#' curve at the last time of `censor.dur`` specified in \code{\link{surv_param_sim}}.
#'
plot_km_pi <- function(km.pi, show.obs = TRUE, trunc.sim.censor = TRUE){

  group <- km.pi$group
  trt   <- km.pi$trt

  color.lab <- trt

  obs.km <- km.pi$obs.km
  sim.km.quantile.plot <- extract_km_pi(km.pi, trunc.sim.censor = trunc.sim.censor)


  group.syms <- rlang::syms(group)
  trt.syms   <- rlang::syms(trt)



  # Plot
  ## Simulations 1
  if(length(trt.syms) == 0) {
    g <-
      ggplot2::ggplot(sim.km.quantile.plot,
                      ggplot2::aes(time))
  } else {
    g <-
      ggplot2::ggplot(sim.km.quantile.plot,
                      ggplot2::aes(time, color = factor(!!!trt.syms),
                                   fill = factor(!!!trt.syms)))
  }

  ## Observed
  if(km.pi$calc.obs & show.obs) {
    g <-
      g +
      ggplot2::geom_step(data = obs.km,
                         ggplot2::aes(y = surv), size = 1) +
      ggplot2::geom_point(data = dplyr::filter(obs.km, cnsr > 0),
                          ggplot2::aes(y = surv), shape = "|", size = 3)
  }

  ## Simulations 2
  if(length(trt.syms) == 0) {
    g <-
      g + ggplot2::geom_ribbon(ggplot2::aes(ymin = pi_low, ymax = pi_high),
                               alpha = 0.4)
  } else {
    g <-
      g + ggplot2::geom_ribbon(ggplot2::aes(ymin = pi_low, ymax = pi_high),
                               alpha = 0.4) +
      ggplot2::labs(color = color.lab, fill = color.lab)
  }


  # Facet fig based on group
  if(length(group.syms) == 1 || length(group.syms) >= 3 ) {
    g <- g + ggplot2::facet_wrap(ggplot2::vars(!!!group.syms),
                                 labeller = ggplot2::label_both)
  } else if (length(group.syms) == 2) {
    g <- g + ggplot2::facet_grid(ggplot2::vars(!!group.syms[[1]]),
                                 ggplot2::vars(!!group.syms[[2]]),
                                 labeller = ggplot2::label_both)
  }

  return(g)

}





#' @rdname survparamsim-methods
#' @export
print.survparamsim.kmpi <- function(x, ...){
  cat("---- Simulated and observed (if calculated) survival curves ----\n")
  cat("* Use `extract_median_surv()` to extract median survival times\n")
  cat("* Use `extract_km_pi()` to extract prediction intervals of K-M curves\n")
  cat("* Use `plot_km_pi()` to draw survival curves\n\n")
  cat("* Settings:\n")
  cat("    trt:", ifelse(is.null(x$trt), "(NULL)", x$trt), "\n", sep=" ")
  cat("    group:", ifelse(is.null(x$group), "(NULL)", x$group), "\n", sep=" ")
  cat("    pi.range:", x$pi.range, "\n", sep=" ")
  cat("    calc.obs:", x$calc.obs, "\n", sep=" ")


}


#' @rdname survparamsim-methods
#' @export
summary.survparamsim.kmpi <- function(object, ...) {

  return(extract_median_surv(object))
}


# print.summary.survparamsim.kmpi <- function(x, ...){
#   cat("Predicted and observed (if calculated) median event time: \n")
#   print(tibble::as_tibble(x))
# }



