% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/h_survival_duration_subgroups.R
\name{h_survival_duration_subgroups}
\alias{h_survival_duration_subgroups}
\alias{h_survtime_df}
\alias{h_survtime_subgroups_df}
\alias{h_coxph_df}
\alias{h_coxph_subgroups_df}
\title{Helper Functions for Tabulating Survival Duration by Subgroup}
\usage{
h_survtime_df(tte, is_event, arm)

h_survtime_subgroups_df(
  variables,
  data,
  groups_lists = list(),
  label_all = "All Patients"
)

h_coxph_df(tte, is_event, arm, strata_data = NULL, control = control_coxph())

h_coxph_subgroups_df(
  variables,
  data,
  groups_lists = list(),
  control = control_coxph(),
  label_all = "All Patients"
)
}
\arguments{
\item{tte}{(\code{numeric})\cr contains time-to-event duration values.}

\item{is_event}{(\code{logical})\cr \code{TRUE} if event, \code{FALSE} if time to event is censored.}

\item{arm}{(\code{factor})\cr the treatment group variable.}

\item{variables}{(named \code{list} of \code{string})\cr list of additional analysis variables.}

\item{data}{(\code{data.frame})\cr the dataset containing the variables to summarize.}

\item{groups_lists}{(named \code{list} of \code{list})\cr optionally contains for each \code{subgroups} variable a list, which
specifies the new group levels via the names and the levels that belong to it in the character vectors that are
elements of the list.}

\item{label_all}{(\code{string})\cr label for the total population analysis.}

\item{strata_data}{(\code{factor}, \code{data.frame} or \code{NULL})\cr required if stratified analysis is performed.}

\item{control}{(\code{list})\cr parameters for comparison details, specified by using the helper function
\code{\link[=control_coxph]{control_coxph()}}. Some possible parameter options are:
\itemize{
\item \code{pval_method} (\code{string})\cr p-value method for testing hazard ratio = 1. Default method is "log-rank" which
comes from \code{\link[survival:survdiff]{survival::survdiff()}}, can also be set to "wald" or "likelihood" (from \code{\link[survival:coxph]{survival::coxph()}}).
\item \code{ties} (\code{string})\cr specifying the method for tie handling. Default is "efron",
can also be set to "breslow" or "exact". See more in \code{\link[survival:coxph]{survival::coxph()}}
\item \code{conf_level} (\code{proportion})\cr confidence level of the interval for HR.
}}
}
\value{
\itemize{
\item \code{h_survtime_df()} returns a \code{data.frame} with columns \code{arm}, \code{n}, \code{n_events}, and \code{median}.
}

\itemize{
\item \code{h_survtime_subgroups_df()} returns a \code{data.frame} with columns \code{arm}, \code{n}, \code{n_events}, \code{median}, \code{subgroup},
\code{var}, \code{var_label}, and \code{row_type}.
}

\itemize{
\item \code{h_coxph_df()} returns a \code{data.frame} with columns \code{arm}, \code{n_tot}, \code{n_tot_events}, \code{hr}, \code{lcl}, \code{ucl},
\code{conf_level}, \code{pval} and \code{pval_label}.
}

\itemize{
\item \code{h_coxph_subgroups_df()} returns a \code{data.frame} with columns \code{arm}, \code{n_tot}, \code{n_tot_events}, \code{hr},
\code{lcl}, \code{ucl}, \code{conf_level}, \code{pval}, \code{pval_label}, \code{subgroup}, \code{var}, \code{var_label}, and \code{row_type}.
}
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}

Helper functions that tabulate in a data frame statistics such as median survival
time and hazard ratio for population subgroups.
}
\details{
Main functionality is to prepare data for use in a layout-creating function.
}
\section{Functions}{
\itemize{
\item \code{h_survtime_df()}: helper to prepare a data frame of median survival times by arm.

\item \code{h_survtime_subgroups_df()}: summarizes median survival times by arm and across subgroups
in a data frame. \code{variables} corresponds to the names of variables found in \code{data}, passed as a named list and
requires elements \code{tte}, \code{is_event}, \code{arm} and optionally \code{subgroups}. \code{groups_lists} optionally specifies
groupings for \code{subgroups} variables.

\item \code{h_coxph_df()}: helper to prepare a data frame with estimates of
treatment hazard ratio.

\item \code{h_coxph_subgroups_df()}: summarizes estimates of the treatment hazard ratio
across subgroups in a data frame. \code{variables} corresponds to the names of variables found in
\code{data}, passed as a named list and requires elements \code{tte}, \code{is_event}, \code{arm} and
optionally \code{subgroups} and \code{strat}. \code{groups_lists} optionally specifies
groupings for \code{subgroups} variables.

}}
\examples{
library(dplyr)
library(forcats)

adtte <- tern_ex_adtte

# Save variable labels before data processing steps.
adtte_labels <- formatters::var_labels(adtte)

adtte_f <- adtte \%>\%
  filter(
    PARAMCD == "OS",
    ARM \%in\% c("B: Placebo", "A: Drug X"),
    SEX \%in\% c("M", "F")
  ) \%>\%
  mutate(
    # Reorder levels of ARM to display reference arm before treatment arm.
    ARM = droplevels(fct_relevel(ARM, "B: Placebo")),
    SEX = droplevels(SEX),
    is_event = CNSR == 0
  )
labels <- c("ARM" = adtte_labels[["ARM"]], "SEX" = adtte_labels[["SEX"]], "is_event" = "Event Flag")
formatters::var_labels(adtte_f)[names(labels)] <- labels

# Extract median survival time for one group.
h_survtime_df(
  tte = adtte_f$AVAL,
  is_event = adtte_f$is_event,
  arm = adtte_f$ARM
)

# Extract median survival time for multiple groups.
h_survtime_subgroups_df(
  variables = list(
    tte = "AVAL",
    is_event = "is_event",
    arm = "ARM",
    subgroups = c("SEX", "BMRKR2")
  ),
  data = adtte_f
)

# Define groupings for BMRKR2 levels.
h_survtime_subgroups_df(
  variables = list(
    tte = "AVAL",
    is_event = "is_event",
    arm = "ARM",
    subgroups = c("SEX", "BMRKR2")
  ),
  data = adtte_f,
  groups_lists = list(
    BMRKR2 = list(
      "low" = "LOW",
      "low/medium" = c("LOW", "MEDIUM"),
      "low/medium/high" = c("LOW", "MEDIUM", "HIGH")
    )
  )
)

# Extract hazard ratio for one group.
h_coxph_df(adtte_f$AVAL, adtte_f$is_event, adtte_f$ARM)

# Extract hazard ratio for one group with stratification factor.
h_coxph_df(adtte_f$AVAL, adtte_f$is_event, adtte_f$ARM, strata_data = adtte_f$STRATA1)

# Extract hazard ratio for multiple groups.
h_coxph_subgroups_df(
  variables = list(
    tte = "AVAL",
    is_event = "is_event",
    arm = "ARM",
    subgroups = c("SEX", "BMRKR2")
  ),
  data = adtte_f
)

# Define groupings of BMRKR2 levels.
h_coxph_subgroups_df(
  variables = list(
    tte = "AVAL",
    is_event = "is_event",
    arm = "ARM",
    subgroups = c("SEX", "BMRKR2")
  ),
  data = adtte_f,
  groups_lists = list(
    BMRKR2 = list(
      "low" = "LOW",
      "low/medium" = c("LOW", "MEDIUM"),
      "low/medium/high" = c("LOW", "MEDIUM", "HIGH")
    )
  )
)

# Extract hazard ratio for multiple groups with stratification factors.
h_coxph_subgroups_df(
  variables = list(
    tte = "AVAL",
    is_event = "is_event",
    arm = "ARM",
    subgroups = c("SEX", "BMRKR2"),
    strat = c("STRATA1", "STRATA2")
  ),
  data = adtte_f
)

}
