% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scatterplot.R
\name{scatterplot3js}
\alias{scatterplot3js}
\title{Interactive 3D Scatterplots}
\usage{
scatterplot3js(x, y, z, height = NULL, width = NULL, axis = TRUE,
  num.ticks = c(6, 6, 6), x.ticklabs = NULL, y.ticklabs = NULL,
  z.ticklabs = NULL, color = "steelblue", size = 1, labels = NULL,
  label.margin = "10px", stroke = "black", flip.y = TRUE, grid = TRUE,
  renderer = c("auto", "canvas", "webgl"), signif = 8, bg = "#ffffff",
  xlim, ylim, zlim, pch, ...)
}
\arguments{
\item{x}{Either a vector of x-coordinate values or a  three-column
data matrix with columns corresponding to the x,y,z
coordinate axes. Column labels, if present, are used as axis labels.}

\item{y}{(Optional) vector of y-coordinate values, not required if
\code{x} is a matrix.}

\item{z}{(Optional) vector of z-coordinate values, not required if
\code{x} is a matrix.}

\item{height}{The container div height.}

\item{width}{The container div width.}

\item{axis}{A logical value that when \code{TRUE} indicates that the
axes will be displayed.}

\item{num.ticks}{A three-element vector with the suggested number of
ticks to display per axis. Set to NULL to not display ticks. The number
of ticks may be adjusted by the program.}

\item{x.ticklabs}{A vector of tick labels of length \code{num.ticks[1]}, or
\code{NULL} to show numeric labels.}

\item{y.ticklabs}{A vector of tick labels of length \code{num.ticks[2]}, or
\code{NULL} to show numeric labels.}

\item{z.ticklabs}{A vector of tick labels of length \code{num.ticks[3]}, or
\code{NULL} to show numeric labels.}

\item{color}{Either a single hex or named color name (all points same color),
or a vector of #' hex or named color names as long as the number of data
points to plot.}

\item{size}{The plot point radius, either as a single number or a
vector of sizes of length \code{nrow(x)}. A vector of sizes is only
supported by the \code{canvas} renderer. The \code{webgl} renderer accepts
a single size value for all points.}

\item{labels}{Either NULL (no labels), or a vector of labels as long as the
number of data points displayed when the mouse hovers over each point.}

\item{label.margin}{A CSS-style margin string used to display the point
labels.}

\item{stroke}{A single color stroke value (surrounding each point). Set to
null to omit stroke (only available in the canvas renderer).}

\item{flip.y}{Reverse the direction of the y-axis (the default value of
TRUE produces plots similar to those rendered by the R
\code{scatterplot3d} package).}

\item{grid}{Set FALSE to disable display of a grid.}

\item{renderer}{Select from available plot rendering techniques of
'auto', 'canvas', or 'webgl'.}

\item{signif}{Number of significant digits used to represent point
coordinates. Larger numbers increase accuracy but slow plot generation
down.}

\item{bg}{The color to be used for the background of the device region.}

\item{xlim}{Optional two-element vector of x-axis limits. Default auto-scales to data.}

\item{ylim}{Optional two-element vector of y-axis limits. Default auto-scales to data.}

\item{zlim}{Optional two-element vector of z-axis limits. Default auto-scales to data.}

\item{pch}{Not yet used but one day will support changing the point glyph.}

\item{...}{Additional options (see note).}
}
\value{
An htmlwidget object that is displayed using the object's show or print method.
(If you don't see your widget plot, try printing it with the \code{print}) function. The
returned object includes a special \code{points3d} function for adding points to the
plot similar to \code{scatterplot3d}. See the note below and examples for details.
}
\description{
A 3D scatterplot widget using three.js. Many options
follow the \code{scatterplot3d} function from the eponymous package.
}
\note{
Points with missing values are omitted.

Use the \code{renderer} option to manually select from the available
rendering options.
The \code{canvas} renderer is the fallback rendering option when \code{webgl}
is not available. Select \code{auto} to automatically choose between
the two. The two renderers produce slightly different-looking output
and have different available options (see above). Use the \code{webgl}
renderer for plotting large numbers of points (if available). Use the
\code{canvas} renderer to excercise finer control of plotting of smaller
numbers of points. See the examples.

Use the optional \code{...} argument to explicitly supply \code{axisLabels}
as a three-element character vector, see the examples below.

The returned object includes a \code{points3d} function that can add points
to a plot, returning a new htmlwidget plot object. The function signature
is a subset of the full \code{scatterplot3js} function:

 \code{points3d (x, y, z, color="steelblue", size=1, labels="")}

It allows you to add points to a plot using the same syntax as \code{scatterplot3js}
with optionally specified color, size, and labels. New points are plotted in
the same scale as the existing plot. See the examples section below for an
example.
}
\examples{
# Gumball machine
N <- 100
i <- sample(3, N, replace=TRUE)
x <- matrix(rnorm(N*3),ncol=3)
lab <- c("small", "bigger", "biggest")
scatterplot3js(x, color=rainbow(N), labels=lab[i], size=i, renderer="canvas")

# Example 1 from the scatterplot3d package (cf.)
z <- seq(-10, 10, 0.1)
x <- cos(z)
y <- sin(z)
scatterplot3js(x,y,z, color=rainbow(length(z)),
      labels=sprintf("x=\%.2f, y=\%.2f, z=\%.2f", x, y, z))

# Same example with explicit axis labels
scatterplot3js(x,y,z, color=rainbow(length(z)), axisLabels=c("a","b","c"))

# Pretty point cloud example, should run this with WebGL!
N     <- 20000
theta <- runif(N)*2*pi
phi   <- runif(N)*2*pi
R     <- 1.5
r     <- 1.0
x <- (R + r*cos(theta))*cos(phi)
y <- (R + r*cos(theta))*sin(phi)
z <- r*sin(theta)
d <- 6
h <- 6
t <- 2*runif(N) - 1
w <- t^2*sqrt(1-t^2)
x1 <- d*cos(theta)*sin(phi)*w
y1 <- d*sin(theta)*sin(phi)*w
i <- order(phi)
j <- order(t)
col <- c( rainbow(length(phi))[order(i)],
         rainbow(length(t),start=0, end=2/6)[order(j)])
M <- cbind(x=c(x,x1),y=c(y,y1),z=c(z,h*t))
scatterplot3js(M,size=0.25,color=col,bg="black")

# Adding points to a plot with points3d
set.seed(1)
lim <- c(-3,3)
x <- scatterplot3js(rnorm(5),rnorm(5),rnorm(5), xlim=lim, ylim=lim, zlim=lim)
a <- x$points3d(rnorm(3),rnorm(3),rnorm(3)/2, color="red", labels="NEW")
\dontrun{
  # A shiny example
  shiny::runApp(system.file("examples/scatterplot",package="threejs"))
}

}
\references{
The three.js project \url{http://threejs.org}.
}
\seealso{
scatterplot3d, rgl
}

