% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/time_nest.R
\name{time_nest}
\alias{time_nest}
\title{Nest a \code{tbl_time} to a specified time period}
\usage{
time_nest(data, period = "yearly", ..., .key = "data",
  keep_inner_dates = TRUE)
}
\arguments{
\item{data}{A \code{tbl_time} object.}

\item{period}{A formula or character specification used for time-based grouping.

If a formula, e.g. \code{1~year}, the formula is split and parsed to form the
grouping period. The \code{period} argument accepts a formula of the form
\code{multiple ~ period} allowing for flexible period grouping.
The following are examples:
\itemize{
\item 1 Year: \code{1~y}
\item 3 Months: \code{3~m}
\item 90 Days: \code{90~d}
}

Note that while shorthand is used above, an attempt is made to recognize
more explicit period names such as:
\itemize{
\item 2 Year: \code{2~year} / \code{2~years} / \code{2~yearly}
}

The \code{period} argument also accepts characters that are converted to their
corresponding periods. The following are accepted:
\itemize{
\item \code{"yearly"} or \code{"y"}
\item \code{"quarterly"} or \code{"q"}
\item \code{"monthly"} or \code{"m"}
\item \code{"weekly"} or \code{"w"}
\item \code{"daily"} or \code{"d"}
\item \code{"hour"} or \code{"h"}
\item \code{"minute"} or \code{"M"}
\item \code{"second"} or \code{"s"}
}}

\item{...}{Used to specify columns you do not want in the nest. Specified
as \code{-col_name}.}

\item{.key}{The name of the new column, as a string or symbol.

This argument is passed by expression and supports
\link[rlang:quasiquotation]{quasiquotation} (you can unquote strings
and symbols). The name is captured from the expression with
\code{\link[rlang:quo_name]{rlang::quo_name()}} (note that this kind of interface where
symbols do not represent actual objects is now discouraged in the
tidyverse; we support it here for backward compatibility).}

\item{keep_inner_dates}{Whether to add dates to each nested object as the
column, \code{.date}.}
}
\description{
\code{time_nest} allows for quick nesting of \code{tbl_time} objects by period so that
manipulation can be performed on each period's data set individually. See
\code{\link[tidyr:nest]{tidyr::nest()}} for information on nesting generally.
}
\details{
As an example, nesting by a \code{"yearly"} period will return a \code{tbl_time} object
with a date column containing the dates at the end of each year, and a
list-column of smaller \code{tbl_time} objects containing all of the data for
that year.

Because it is likely the case that the original dates will be required
inside the nested \code{tbl_time} objects for further manipulation,
the original dates are kept as a \code{.date} column in each \code{tbl_time} object in
the list-column.

This function respects \code{\link[dplyr:group_by]{dplyr::group_by()}} groups.
}
\examples{

# Basic functionality -------------------------------------------------------

data(FB)
FB <- as_tbl_time(FB, date)

# Nest yearly
time_nest(FB, "yearly")

# Nest yearly, but don't drag symbol into the nest
time_nest(FB, "yearly", -symbol)

# Nest quarterly
time_nest(FB, "quarterly")

# Grouped functionality -----------------------------------------------------

data(FANG)
FANG <- as_tbl_time(FANG, date) \%>\%
  dplyr::group_by(symbol)

# Nest yearly, but by group
FANG \%>\%
  time_nest("yearly")

}
