% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bind-rows-SpatVector.R
\name{bind_rows.SpatVector}
\alias{bind_rows.SpatVector}
\alias{bind_spat_rows}
\alias{bind.Spat}
\title{Bind multiple SpatVectors and data frames by row}
\usage{
bind_spat_rows(..., .id = NULL)
}
\arguments{
\item{...}{SpatVector to combine. The first argument should be a SpatVector
and each of the subsequent arguments can either be a SpatVector, a sf object
or a data frame. Columns are matched by name, and any missing columns will
be filled with \code{NA}.}

\item{.id}{The name of an optional identifier column. Provide a string to
create an output column that identifies each input. The column will use
names if available, otherwise it will use positions.}
}
\value{
A SpatVector of the same type as the first element of \code{...}.
}
\description{
Bind any number of SpatVector, data frames and sf object by row, making a
longer result. This is similar to \code{do.call(rbind, dfs)}, but the output
will contain all columns that appear in any of the inputs.
}
\section{terra equivalent}{


\code{rbind()} method
}

\section{Methods}{


Implementation of the \code{\link[dplyr:bind_rows]{dplyr::bind_rows()}} function for
SpatVectors.

The first element of \code{...} should be a SpatVector. Subsequent elements may
be SpatVector, sf/sfc objects or data frames:
\itemize{
\item If subsequent SpatVector/sf/sfc present a different CRS than the first
element, those elements would be reprojected to the CRS of the first
element with a message.
\item If any element of \code{...} is a tibble/data frame the rows would be
\code{cbind}ed with empty geometries with a message.
}
}

\examples{

library(terra)
v <- vect(system.file("extdata/cyl.gpkg", package = "tidyterra"))

v1 <- v[1, "cpro"]
v2 <- v[3:5, c("name", "iso2")]

# You can supply individual SpatVector as arguments:
bind_spat_rows(v1, v2)

# When you supply a column name with the `.id` argument, a new
# column is created to link each row to its original data frame
bind_spat_rows(v1, v2, .id = "id")

# Use with sf
sfobj <- sf::st_as_sf(v2[1, ])

sfobj

bind_spat_rows(v1, sfobj)

# Would reproject with a message on different CRS
sfobj_3857 <- as_spatvector(sfobj) \%>\% project("EPSG:3857")

bind_spat_rows(v1, sfobj_3857)

# And with data frames with a message
data("mtcars")
bind_spat_rows(v1, sfobj, mtcars, .id = "id2")


# Use lists
bind_spat_rows(list(v1[1, ], sfobj[1:2, ]))

# Or named list combined with .id
bind_spat_rows(list(
  SpatVector = v1[1, ], sf = sfobj[1, ],
  mtcars = mtcars[1, ]
), .id = "source")
}
\seealso{
\code{\link[dplyr:bind_rows]{dplyr::bind_rows()}}

Other dplyr verbs that operate on pairs Spat*/data.frame:
\code{\link{bind_cols.SpatVector}},
\code{\link{filter-joins.SpatVector}},
\code{\link{mutate-joins.SpatVector}}

Other dplyr methods:
\code{\link{arrange.SpatVector}()},
\code{\link{bind_cols.SpatVector}},
\code{\link{count.SpatVector}()},
\code{\link{distinct.SpatVector}()},
\code{\link{filter-joins.SpatVector}},
\code{\link{filter.Spat}},
\code{\link{glimpse.Spat}},
\code{\link{group-by.SpatVector}},
\code{\link{mutate-joins.SpatVector}},
\code{\link{mutate.Spat}},
\code{\link{pull.Spat}},
\code{\link{relocate.Spat}},
\code{\link{rename.Spat}},
\code{\link{rowwise.SpatVector}()},
\code{\link{select.Spat}},
\code{\link{slice.Spat}},
\code{\link{summarise.SpatVector}()}
}
\concept{dplyr.methods}
\concept{dplyr.pairs}
