% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/travel_times.R
\name{travel_times}
\alias{travel_times}
\title{Calculate shortest travel times from a stop to all reachable stops}
\usage{
travel_times(
  filtered_stop_times,
  stop_name,
  time_range = 3600,
  arrival = FALSE,
  max_transfers = NULL,
  max_departure_time = NULL,
  return_coords = FALSE,
  return_DT = FALSE,
  stop_dist_check = 300
)
}
\arguments{
\item{filtered_stop_times}{stop_times data.table (with transfers and stops tables as
attributes) created with \code{\link[=filter_stop_times]{filter_stop_times()}} where the
departure or arrival time has been set. Alternatively,
a filtered feed created by \code{\link[=filter_feed_by_date]{filter_feed_by_date()}} can be
used.}

\item{stop_name}{Stop name for which travel times should be calculated. A vector with
multiple names is accepted.}

\item{time_range}{All departures within this range in seconds after the first departure
of \code{filtered_stop_times} are considered for journeys. If arrival is
TRUE, all journeys arriving within time range before the latest arrival
of \code{filtered_stop_times} are considered.}

\item{arrival}{If FALSE (default), all journeys \emph{start} from \code{stop_name}. If
TRUE, all journeys \emph{end} at \code{stop_name}.}

\item{max_transfers}{The maximimum number of transfers}

\item{max_departure_time}{Either set this parameter or \code{time_range}. Only departures
before \code{max_departure_time} are used. Accepts "HH:MM:SS" or
seconds as a numerical value. Unused if \code{arrival} is TRUE.}

\item{return_coords}{Returns stop coordinates as columns. Default is FALSE.}

\item{return_DT}{travel_times() returns a data.table if TRUE. Default is FALSE which
returns a tibble/tbl_df.}

\item{stop_dist_check}{stop_names are not structured identifiers like
stop_ids or parent_stations, so it's possible that
stops with the same name are far apart. travel_times()
errors if the distance among stop_ids with the same name is
above this threshold (in meters).
Use FALSE to turn check off. However, it is recommended to
either use \code{\link[=raptor]{raptor()}} or fix the feed (see \code{\link[=cluster_stops]{cluster_stops()}}).}
}
\value{
A table with travel times to/from all stops reachable by \code{stop_name} and their
corresponding journey departure and arrival times.
}
\description{
Function to calculate the shortest travel times from a stop (given by \code{stop_name})
to all other stop_names of a feed. \code{filtered_stop_times} needs to be created before with
\code{\link[=filter_stop_times]{filter_stop_times()}} or \code{\link[=filter_feed_by_date]{filter_feed_by_date()}}.
}
\details{
This function allows easier access to \code{\link[=raptor]{raptor()}} by using stop names instead of ids and
returning shortest travel times by default.

Note however that stop_name might not be a suitable identifier for a feed. It is possible
that multiple stops have the same name while not being related or geographically close to
each other. \code{\link[=stop_group_distances]{stop_group_distances()}} and \code{\link[=cluster_stops]{cluster_stops()}} can help identify and fix
issues with stop_names.
}
\examples{
\donttest{
nyc_path <- system.file("extdata", "google_transit_nyc_subway.zip", package = "tidytransit")
nyc <- read_gtfs(nyc_path)

# stop_names in this feed are not restricted to an area, create clusters of stops to fix
nyc <- cluster_stops(nyc, group_col = "stop_name", cluster_colname = "stop_name")

# Use journeys departing after 7 AM with arrival time before 9 AM on 26th June
stop_times <- filter_stop_times(nyc, "2018-06-26", 7*3600, 9*3600)

tts <- travel_times(stop_times, "34 St - Herald Sq", return_coords = TRUE)
library(dplyr)
tts <- tts \%>\% filter(travel_time <= 3600)

# travel time to Queensboro Plaza is 810 seconds, 13:30 minutes
tts \%>\% filter(to_stop_name == "Queensboro Plaza") \%>\% pull(travel_time) \%>\% hms::hms()

# plot a simple map showing travel times to all reachable stops
# this can be expanded to isochron maps
library(ggplot2)
ggplot(tts) + geom_point(aes(x=to_stop_lon, y=to_stop_lat, color = travel_time))
}
}
