% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/shift_geometry.R
\name{shift_geometry}
\alias{shift_geometry}
\title{Shift and rescale Alaska, Hawaii, and Puerto Rico in a US-wide sf object}
\usage{
shift_geometry(
  input_sf,
  geoid_column = NULL,
  preserve_area = FALSE,
  position = c("below", "outside")
)
}
\arguments{
\item{input_sf}{The input sf dataset}

\item{geoid_column}{The GEOID column of the dataset that contains a state ID.  If used, will speed up
processing and avoid spatial overlay to infer locations.  Defaults to \code{NULL}.}

\item{preserve_area}{If TRUE, the areas of Alaska/Hawaii/Puerto Rico relative to the continental US
will be preserved.  Defaults to FALSE where Alaska is proportionally smaller
and Hawaii/Puerto Rico are proportionally larger.}

\item{position}{One of \code{"below"} (the default) or \code{"outside"}.  If \code{"below"}, Alaska, Hawaii,
and Puerto Rico will be placed below the continental United States.  If \code{"outside"},
features will be moved outside the continental US; Alaska will be northwest of Washington,
Hawaii southwest of California, and Puerto Rico southeast of Florida.}
}
\value{
The input sf object with transformed geometry
}
\description{
This function will shift and optionally rescale a US dataset for thematic mapping of Alaska,
Hawaii, and Puerto Rico with respect to the continental United States.  Features in the continental
United States will have their CRS transformed to USA Contiguous Albers Equal Area Conic ('ESRI:102003').
Alaska, Hawaii, and Puerto Rico features are transformed to appropriate coordinate systems for those areas,
then shifted and optionally re-scaled before being assigned the 'ESRI:102003' CRS.  Options for users
include \code{preserve_area} which allows for preservation of the area of AK/HI/PR relative to the
continental US if desired, and two possible arrangements which are specified with \code{position = "below"}
or \code{position = "outside"}
}
\details{
\code{shift_geometry()}, while designed for use with objects from the tigris package, will work with any US
dataset. If aligning datasets from multiple sources, you must take care to ensure that your options
specified in \code{preserve_area} and \code{position} are identical across layers.  Otherwise your layers
will not align correctly.

The function is also designed to work exclusively with features in the continental United States,
Alaska, Hawaii, and Puerto Rico.  If your dataset includes features outside these areas (e.g. other
US territories or other countries), you may get unworkable results.  It is advisable to filter out
those features before using \code{shift_geometry()}.

Work on this function is inspired by and adapts some code from Claus Wilke's book \emph{Fundamentals of
Data Visualization} (\url{https://clauswilke.com/dataviz/geospatial-data.html}); Bob Rudis's
albersusa R package (\url{https://github.com/hrbrmstr/albersusa}); and the ggcart R package
(\url{https://uncoast-unconf.github.io/ggcart/}).
}
\examples{
\dontrun{

# Shift and rescale AK/HI/PR for thematic mapping
library(tigris)
library(tidycensus)
library(tidyverse)

us_states <- states(cb = TRUE, resolution = "20m") \%>\%
  shift_geometry()

# Shift but preserve area
us_states_eqarea <- states(cb = TRUE, resolution = "20m") \%>\%
  shift_geometry(preserve_area = TRUE)

# Shift and rescale but position AK/HI/PR outside the continental US rather than below
us_states_outside <- states(cb = TRUE, resolution = "20m") \%>\%
  shift_geometry(position = "outside")

# Shift a dataset obtained outside tigris and make a map
income_by_metro <- get_acs(
  geography = "cbsa",
  variables = "B01002_001",
  geometry = TRUE
) \%>\%
  shift_geometry()

ggplot() +
  geom_sf(data = income_by_metro, aes(fill = estimate), color = NA) +
  geom_sf(data = us_states, fill = NA, color = "black", size = 0.1) +
  scale_fill_viridis_c() +
  theme_void(base_size = 16) +
  labs(title = "Median age by CBSA, 2015-2019 ACS",
       fill = "ACS estimate  ",
       caption = "Note: Alaska, Hawaii, and Puerto Rico are shifted and not to scale.") +
  theme(plot.title = element_text(hjust = 0.5))


}
}
