% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/time_count.R
\name{time_count}
\alias{time_count}
\title{Fast count, fill implicit missing gaps, and (time)
aggregate a time series based data frame.}
\usage{
time_count(
  data,
  time = NULL,
  ...,
  time_by = NULL,
  from = NULL,
  to = NULL,
  complete = TRUE,
  wt = NULL,
  name = NULL,
  sort = FALSE,
  .by = NULL,
  time_floor = FALSE,
  week_start = getOption("lubridate.week.start", 1),
  time_type = c("auto", "duration", "period"),
  roll_month = "preday",
  roll_dst = "pre",
  include_interval = FALSE
)
}
\arguments{
\item{data}{A data frame.}

\item{time}{Time variable.}

\item{...}{Additional variables to include.}

\item{time_by}{Time unit. \cr
Must be one of the three:
\itemize{
\item string, specifying either the unit or the number and unit, e.g
\code{time_by = "days"} or \code{time_by = "2 weeks"}
\item named list of length one, the unit being the name, and
the number the value of the list, e.g. \code{list("days" = 7)}.
For the vectorized time functions, you can supply multiple values,
e.g. \code{list("days" = 1:10)}.
\item Numeric vector. If time_by is a numeric vector and x is not a date/datetime,
then arithmetic is used, e.g \code{time_by = 1}.
}}

\item{from}{Time series start date. If \code{NULL} then min time is used.}

\item{to}{Time series end date. If \code{NULL} then max time is used.}

\item{complete}{Logical. If \code{TRUE} implicit gaps in time are filled
after counting and time aggregation (through \code{time_by}).
The default is \code{TRUE}.}

\item{wt}{Frequency weights.
\code{dplyr} "data-masking" is used for variable selection.
Can be \code{NULL} or a variable:
\itemize{
\item If \code{NULL} (the default), counts the number of rows in each group.
\item If a variable, computes \code{sum(wt)} for each group.
}}

\item{name}{Character vector of length 1, specifying the name of
the new column in the output.}

\item{sort}{If \code{TRUE} the groups with largest counts will be sorted first.
If \code{FALSE} the result is sorted by groups + time + ... groups.}

\item{.by}{(Optional). A selection of columns to group by for this operation.
Columns are specified using tidy-select.}

\item{time_floor}{Should \code{from} be floored to the nearest unit
specified through the \code{time_by}
argument? This is particularly useful for starting
sequences at the beginning of a week
or month for example.}

\item{week_start}{day on which week starts following ISO conventions - 1
means Monday (default), 7 means Sunday.
This is only used when \code{time_floor = TRUE}.}

\item{time_type}{If "auto", \code{periods} are used for
the time expansion when days, weeks, months or
years are specified, and \code{durations}
are used otherwise.}

\item{roll_month}{Control how impossible dates are handled when
month or year arithmetic is involved.
Options are "preday", "boundary", "postday", "full" and "NA".
See \code{?timechange::time_add} for more details.}

\item{roll_dst}{See \code{?timechange::time_add} for the full list of details.}

\item{include_interval}{Logical. If \code{TRUE} then
a column "interval" of the form \verb{time_min <= x < time_max} is added
showing the time interval in which the respective counts belong to.
The rightmost interval will always be closed.}
}
\value{
An object of class \code{data.frame}
containing the input time variable
which is expanded for each supplied group.
}
\description{
This function operates like \code{dplyr::count()}
but with emphasis on
a specified time variable.
}
\details{
\code{time_count} Creates an
aggregated frequency time series where time
can be aggregated to
both lower and higher time units.

An important note is that when the data are grouped, time ranges are expanded
on a group-by-group basis. This works like dplyr where you can supply either a
grouped_df or specify the groupings through \code{.by}.
When groups are supplied through
\code{...}, the time range of the entire data is used for
the expansion of each group.
Depending on the analysis, this may or may not be what you want
and can sometimes result in huge expansions if dealing with time variables
with large span sizes.

If \code{complete = TRUE} then the gaps in the time are expanded.

To avoid filling in missing gaps in time, as well as possible expansion,
simply set \code{complete = FALSE}
which will simply perform a count
across your aggregated time variable and specified groups.
}
\examples{
library(timeplyr)
library(dplyr)
library(lubridate)
library(nycflights13)
\dontshow{
.n_dt_threads <- data.table::getDTthreads()
.n_collapse_threads <- collapse::get_collapse()$nthreads
data.table::setDTthreads(threads = 2L)
collapse::set_collapse(nthreads = 1L)
}
df <- flights \%>\%
  mutate(date = as_date(time_hour),
         date_num = as.integer(date)) \%>\%
  select(year, month, day, origin, dest, date, time_hour, date_num)

# By default time_count() guesses the time granularity
df \%>\%
  time_count(time_hour)
# Aggregated to week level
df \%>\%
  time_count(time = date, time_by = "2 weeks")
df \%>\%
  time_count(time = date, time_by = list("months" = 3),
             from = dmy("15-01-2013"),
             time_floor = TRUE,
             include_interval = TRUE)
\dontshow{
data.table::setDTthreads(threads = .n_dt_threads)
collapse::set_collapse(nthreads = .n_collapse_threads)
}
}
