#' Specify the shape object
#' 
#' Creates a \code{\link{tmap-element}} that specifies a spatial data object, which we refer to as shape. Also the projection and covered area (bounding box) can be set. It is possible to use multiple shape objects within one plot (see \code{\link{tmap-element}}).
#'  
#' @param shp shape object, which is an object from a class defined by the \code{\link[sf:sf]{sf}}, \code{\link[sp:sp]{sp}}, or \code{\link[raster:raster-package]{raster}} package. For instance, an \code{\link[sf:sf]{sf}} object, an \code{\link[sp:SpatialPolygonsDataFrame]{SpatialPolygons(DataFrame)}}, or a \code{\link[raster:Raster-class]{RasterBrick}}.
#' @param name name of the shape object (character) as it appears in the legend in \code{"view"} mode. Default value is the name of \code{shp}.
#' @param is.master logical that determines whether this \code{tm_shape} is the master shape element. The bounding box, projection settings, and the unit specifications of the resulting thematic map are taken from the \code{tm_shape} element of the master shape object. By default, the first master shape element with a raster shape is the master, and if there are no raster shapes used, then the first \code{tm_shape} is the master shape element.
#' @param projection Either a \code{\link[sf:st_crs]{crs}} object or a character value. If it is a character, it can either be a \code{PROJ.4} character string or a shortcut. See \code{\link[tmaptools:get_proj4]{get_proj4}} for a list of shortcut values. By default, the projection is used that is defined in the \code{shp} object itself, which can be obtained with \code{\link[tmaptools:get_projection]{get_projection}}.
#' @param bbox bounding box. One of the following:
#' \itemize{
#' \item A bounding box (an \code{\link[sf:sf]{sf}} bbox object, see \code{\link[sf:st_bbox]{st_bbox}}, a 2 by 2 matrix (used by the \code{sp} package), or an \code{\link[raster:Extent]{Extent}} object used by the \code{raster} package). 
#' \item Open Street Map search query. The bounding is automatically generated by querying \code{q} from Open Street Map Nominatim. See \url{http://wiki.openstreetmap.org/wiki/Nominatim}.
#' \item Another shape object, from which the bounding box is extracted.}
#' If unspecified, the current bounding box of \code{shp} is taken. The bounding box is feed to \code{\link[tmaptools:bb]{bb}} (as argument \code{x}. The other arguments of \code{\link[tmaptools:bb]{bb}} can be specified directly as well (see \code{..}). 
#' @param unit desired units of the map. One of \code{"metric"} (default), \code{"imperial"}, \code{"km"}, \code{"m"}, \code{"mi"} and \code{"ft"}. Used to specify the scale bar (see \code{\link{tm_scale_bar}}) and to calculate densities for choropleths (see argument \code{convert2density} in \code{\link{tm_fill}}).
#' @param simplify simplification factor for spatial polygons and spatial lines. A number between 0 and 1 that indicates how many coordinates are kept. See the underlying function \code{\link[tmaptools:simplify_shape]{simplify_shape}}, from which the arguments \code{keep.units} and \code{keep.subunits} can be passed on (see \code{...}). This requires the suggested package \code{rmapshaper}.
#' @param point.per specification of how points or text labels are plotted when the geometry is a multi line or a multi polygon. One of \code{"feature"}, \code{"segment"} or \code{"largest"}. The first generates a point/label for every feature, the second for every segment (i.e. subfeature), the third only for the largest segment (subfeature). Note that the last two options can be significant slower. By default, it is set to \code{"segment"} if the geometry of shp is a (multi)points geometry or a geometrycollection, and \code{"feature"} otherwise.
#' @param line.center specification of where points are placed for (multi)line geometries. Either \code{"midpoint"} or \code{"centroid"}. The former places a point at the middle of the line, the latter at the controid.
#' @param filter logical vector which indicated per feature whether it should be included. Features for which filter is \code{FALSE} will be colored light gray (see the \code{colorNULL} argument in the layer functions)
#' @param ... Arguments passed on to \code{\link[tmaptools:bb]{bb}} (e.g. \code{ext} can be used to enlarge or shrinke a bounding box), and \code{\link[tmaptools:simplify_shape]{simplify_shape}} (the arguments \code{keep.units} and \code{keep.subunits})
#' @export
#' @seealso \code{\link[tmaptools:read_shape]{read_shape}} to read ESRI shape files, \code{\link[tmaptools:set_projection]{set_projection}}, \href{../doc/tmap-getstarted.html}{\code{vignette("tmap-getstarted")}} 
#' @references Tennekes, M., 2018, {tmap}: Thematic Maps in {R}, Journal of Statistical Software, 84(6), 1-39, \href{https://doi.org/10.18637/jss.v084.i06}{DOI}
#' @example ./examples/tm_shape.R
#' @return \code{\link{tmap-element}}
tm_shape <- function(shp, 
					 name = NULL,
					 is.master = NA,
					 projection=NULL,
					 bbox = NULL,
					 unit = NULL,
					 simplify = 1,
					 point.per = NA,
					 line.center = "midpoint",
					 filter = NULL,
					 ...) {
	shp_name <- ifelse(is.null(name) == TRUE, deparse(substitute(shp))[1], name)
	g <- list(tm_shape=c(as.list(environment()), list(...), check_shape = TRUE))
	class(g) <- "tmap"
	g
}
