% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/build_run_modify.R
\name{umxCP}
\alias{umxCP}
\title{umxCP: Build and run a Common pathway twin model}
\usage{
umxCP(
  name = "CP",
  selDVs,
  dzData = NULL,
  mzData = NULL,
  sep = NULL,
  nFac = 1,
  type = c("Auto", "FIML", "cov", "cor", "WLS", "DWLS", "ULS"),
  data = NULL,
  zyg = "zygosity",
  allContinuousMethod = c("cumulants", "marginals"),
  correlatedA = FALSE,
  dzAr = 0.5,
  dzCr = 1,
  autoRun = getOption("umx_auto_run"),
  tryHard = c("no", "yes", "mxTryHard", "mxTryHardOrdinal", "mxTryHardWideSearch"),
  optimizer = NULL,
  equateMeans = TRUE,
  weightVar = NULL,
  bVector = FALSE,
  boundDiag = 0,
  addStd = TRUE,
  addCI = TRUE,
  numObsDZ = NULL,
  numObsMZ = NULL,
  freeLowerA = FALSE,
  freeLowerC = FALSE,
  freeLowerE = FALSE
)
}
\arguments{
\item{name}{The name of the model (defaults to "CP").}

\item{selDVs}{The variables to include.
omit sep in selDVs, i.e., just "dep" not c("dep_T1", "dep_T2").}

\item{dzData}{The DZ dataframe.}

\item{mzData}{The MZ dataframe.}

\item{sep}{(required) The suffix for twin 1 and twin 2, often "_T".}

\item{nFac}{How many common factors (default = 1)}

\item{type}{One of "Auto", "FIML", "cov", "cor", "WLS", "DWLS", "ULS"}

\item{data}{If provided, dzData and mzData are treated as valid levels of zyg to select() data sets (default = NULL)}

\item{zyg}{If data provided, this column is used to select rows by zygosity (Default = "zygosity")}

\item{allContinuousMethod}{"cumulants" or "marginals". Used in all-continuous WLS data to determine if a means model needed.}

\item{correlatedA}{?? (default = FALSE).}

\item{dzAr}{The DZ genetic correlation (defaults to .5, vary to examine assortative mating).}

\item{dzCr}{The DZ "C" correlation (defaults to 1: set to .25 to make an ADE model).}

\item{autoRun}{Whether to run the model (default), or just to create it and return without running.}

\item{tryHard}{Default ('no') uses normal mxRun. "yes" uses mxTryHard. Other options: "mxTryHardOrdinal", "mxTryHardWideSearch"}

\item{optimizer}{optionally set the optimizer (default NULL does nothing).}

\item{equateMeans}{Whether to equate the means across twins (defaults to TRUE).}

\item{weightVar}{If provided, a vector objective will be used to weight the data. (default = NULL).}

\item{bVector}{Whether to compute row-wise likelihoods (defaults to FALSE).}

\item{boundDiag}{= Numeric lbound for diagonal of the a_cp, c_cp, & e_cp matrices. Set = NULL to ignore.}

\item{addStd}{Whether to add the algebras to compute a std model (defaults to TRUE).}

\item{addCI}{Whether to add the interval requests for CIs (defaults to TRUE).}

\item{numObsDZ}{= not yet implemented: Ordinal Number of DZ twins: Set this if you input covariance data.}

\item{numObsMZ}{= not yet implemented: Ordinal Number of MZ twins: Set this if you input covariance data.}

\item{freeLowerA}{(ignore): Whether to leave the lower triangle of A free (default = FALSE).}

\item{freeLowerC}{(ignore): Whether to leave the lower triangle of C free (default = FALSE).}

\item{freeLowerE}{(ignore): Whether to leave the lower triangle of E free (default = FALSE).}
}
\value{
\itemize{
\item \code{\link[=mxModel]{mxModel()}}
}
}
\description{
Make a 2-group Common Pathway twin model (Common-factor common-pathway multivariate model).
}
\details{
The common-pathway model provides a powerful tool for theory-based decomposition of genetic
and environmental differences.

umxCP supports this with pairs of mono-zygotic (MZ) and di-zygotic (DZ) twins reared together
to model the genetic and environmental structure of multiple phenotypes
(measured behaviors).

Common-pathway path diagram:

\if{html}{\figure{CP.png}{options: width="50\%" alt="Figure: CP.png"}}
\if{latex}{\figure{CP.pdf}{options: width=7cm}}

As can be seen, each phenotype also by default has A, C, and E influences specific to that phenotype.

Features include the ability to include more than one common pathway, to use ordinal data.

\strong{note}: The function \code{umx_set_mvn_optimization_options}() allow users to see and set \code{mvnRelEps} and \code{mvnMaxPointsA}
It defaults to .001. You might find that '0.01' works better for ordinal models.

Like the \code{\link[=umxACE]{umxACE()}} model, the CP model decomposes phenotypic variance
into Additive genetic, unique environmental (E) and, optionally, either
common or shared-environment (C) or
non-additive genetic effects (D).

Unlike the Cholesky, these factors do not act directly on the phenotype. Instead latent A,
C, and E influences impact on one or more latent factors which in turn account for variance in the phenotypes (see Figure).

\strong{Data Input}
Currently, the umxCP function accepts only raw data. This may change in future versions.

\strong{Ordinal Data}

In an important capability, the model transparently handles ordinal (binary or multi-level
ordered factor data) inputs, and can handle mixtures of continuous, binary, and ordinal
data in any combination.

\strong{Additional features}

The umxCP function supports varying the DZ genetic association (defaulting to .5)
to allow exploring assortative mating effects, as well as varying the DZ \dQuote{C} factor
from 1 (the default for modeling family-level effects shared 100\% by twins in a pair),
to .25 to model dominance effects.

\strong{Matrices and Labels in CP model}

A good way to see which matrices are used in umxCP is to run an example model and plot it.

All the shared matrices are in the model "top".

Matrices \code{top$as}, \code{top$cs}, and \code{top$es} contain the path loadings specific to each variable on their diagonals.

So, to see the 'as' values, labels, or free states, you can say:

\code{m1$top$as$values}

\code{m1$top$as$free}

\code{m1$top$as$labels}

Labels relevant to modifying the specific loadings take the form "as_r1c1", "as_r2c2" etc.

The common-pathway loadings on the factors are in matrices \code{top$a_cp}, \code{top$c_cp}, \code{top$e_cp}.

The common factors themselves are in the matrix \code{top$cp_loadings} (an nVar * 1 matrix)

Less commonly-modified matrices are the mean matrix \code{expMean}. This has 1 row, and the columns are laid out for each variable for twin 1, followed by each variable for twin 2.
So, in a model where the means for twin 1 and twin 2 had been equated (set = to T1), you could make them independent again with this line:

\code{m1$top$expMean$labels[1,4:6] = c("expMean_r1c4", "expMean_r1c5", "expMean_r1c6")}
}
\examples{
\dontrun{
# ========================================================
# = Run a 3-factor Common pathway twin model of 6 traits =
# ========================================================
require(umx)
data(GFF)
mzData = subset(GFF, zyg_2grp == "MZ")
dzData = subset(GFF, zyg_2grp == "DZ")
selDVs = c("gff", "fc", "qol", "hap", "sat", "AD") 
m1 = umxCP("new", selDVs = selDVs, sep = "_T", nFac = 3, optimizer = "SLSQP",
		dzData = dzData, mzData = mzData, tryHard = "mxTryHardOrdinal")

# Shortcut using "data ="
selDVs = c("gff", "fc", "qol", "hap", "sat", "AD") 
m1 = umxCP(selDVs = selDVs, nFac = 3, data=GFF, zyg="zyg_2grp")

# ===================
# = Do it using WLS =
# ===================
m2 = umxCP("new", selDVs = selDVs, sep = "_T", nFac = 3, optimizer = "SLSQP",
		dzData = dzData, mzData = mzData,
		tryHard = "mxTryHardOrdinal", type= "DWLS", allContinuousMethod='marginals'
)

# =================================================
# = Find and test dropping of shared environment  =
# =================================================
# Show all labels for C parameters 
umxParameters(m1, patt = "^c")
# Test dropping the 9 specific and common-factor C paths
m2 = umxModify(m1, regex = "(cs_.*$)|(c_cp_)", name = "dropC", comp = TRUE)
umxSummaryCP(m2, comparison = m1, file = NA)
umxCompare(m1, m2)

# =======================================
# = Mixed continuous and binary example =
# =======================================
data(GFF)
# Cut to form umxFactor 20\% depressed  DEP
cutPoints = quantile(GFF[, "AD_T1"], probs = .2, na.rm = TRUE)
ADLevels  = c('normal', 'depressed')
GFF$DEP_T1 = cut(GFF$AD_T1, breaks = c(-Inf, cutPoints, Inf), labels = ADLevels) 
GFF$DEP_T2 = cut(GFF$AD_T2, breaks = c(-Inf, cutPoints, Inf), labels = ADLevels) 
ordDVs = c("DEP_T1", "DEP_T2")
GFF[, ordDVs] = umxFactor(GFF[, ordDVs])

selDVs = c("gff","fc","qol","hap","sat","DEP") 
mzData = subset(GFF, zyg_2grp == "MZ")
dzData = subset(GFF, zyg_2grp == "DZ")

# umx_set_optimizer("NPSOL")
# umx_set_mvn_optimization_options("mvnRelEps", .01)
m1 = umxCP(selDVs = selDVs, sep = "_T", nFac = 3, dzData = dzData, mzData = mzData)
m2 = umxModify(m1, regex = "(cs_r[3-5]|c_cp_r[12])", name = "dropC", comp= TRUE)

# Do it using WLS
m3 = umxCP(selDVs = selDVs, sep = "_T", nFac = 3, dzData = dzData, mzData = mzData,
		tryHard = "mxTryHardOrdinal", type= "DWLS")
# TODO umxCPL fix WLS here
# label at row 1 and column 1 of matrix 'top.binLabels'' in model 'CP3fac' : object 'Vtot'

# Correlated factors example
data(GFF)
mzData = subset(GFF, zyg_2grp == "MZ")
dzData = subset(GFF, zyg_2grp == "DZ")
selDVs = c("gff", "fc", "qol", "hap", "sat", "AD")
m1 = umxCP("new", selDVs = selDVs, sep = "_T", dzData = dzData, mzData = mzData, 
	nFac = 3, correlatedA = TRUE, tryHard = "mxTryHard")
}

}
\references{
\itemize{
\item \url{https://www.github.com/tbates/umx}
}
}
\seealso{
\itemize{
\item \code{\link[=umxSummaryCP]{umxSummaryCP()}}, \code{\link[=umxPlotCP]{umxPlotCP()}}. See \code{\link[=umxACE]{umxACE()}} for more examples of twin modeling.
\code{\link[=plot]{plot()}} and \code{\link[=umxSummary]{umxSummary()}} work for all twin models, e.g., \code{\link[=umxIP]{umxIP()}}, \code{\link[=umxCP]{umxCP()}}, \code{\link[=umxGxE]{umxGxE()}}, and \code{\link[=umxACE]{umxACE()}}. For a deep-dive, see \code{\link[=xmu_make_top_twin]{xmu_make_top_twin()}}
}

Other Twin Modeling Functions: \code{\link{power.ACE.test}()},
  \code{\link{umxACE_cov_fixed}()},
  \code{\link{umxACEcov}()}, \code{\link{umxACEv}()},
  \code{\link{umxACE}()}, \code{\link{umxGxE_window}()},
  \code{\link{umxGxEbiv}()}, \code{\link{umxGxE}()},
  \code{\link{umxIP}()},
  \code{\link{umxRotate.MxModelCP}()},
  \code{\link{umxSexLim}()}, \code{\link{umxSimplex}()},
  \code{\link{umx}}
}
\concept{Twin Modeling Functions}
