% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/matrixFunctions.r
\name{bcdmat}
\alias{bcdmat}
\alias{classicmat}
\alias{kmatMarg}
\alias{kmatFull}
\alias{gudmat}
\title{Transition Probability Matrices for Up-and-Down Designs}
\usage{
bcdmat(cdf, target)

classicmat(cdf)

kmatMarg(cdf, k, lowTarget)

kmatFull(cdf, k, lowTarget, fluffup = FALSE)

gudmat(cdf, cohort, lower, upper)
}
\arguments{
\item{cdf}{monotone increasing vector with positive-response probabilities. The number of dose levels \eqn{M} is deduced from vector's length.}

\item{target}{the design's target response rate (\code{bcdmat()} only).}

\item{k}{the number of consecutive identical responses required for dose transitions (k-in-a-row functions only).}

\item{lowTarget}{logical k-in-a-row functions only: is the design targeting below-median percentiles, with \eqn{k} repeated negative responses needed to level up and only one to level down - or vice versa? Default \code{FALSE}. See "Details" for more information.}

\item{fluffup}{logical (\code{kmatFull} only): in the full k-in-a-row internal-state representation, should we \emph{"fluff"} the matrix up so that it has \eqn{Mk} rows and columns (\code{TRUE}), or exclude \eqn{k-1} "phantom" states near the less-likely-to-be-visited boundary (\code{FALSE}, default)?}

\item{cohort}{\code{gudmat} only: the cohort (group) size}

\item{lower, upper}{(\code{gudmat} only) how many positive responses are allowed for a move upward, and how many are required for a move downward, respectively. For example \verb{cohort=3, lower=0, upper=2} evaluates groups of 3 observations at a time, moves up if none are positive, down if \eqn{>=2} are positive, and repeats the same dose with 1 positive.}
}
\value{
An \eqn{M\times M} transition probability matrix, except for \code{kmatFull()} with \eqn{k>1} which returns a larger square matrix.
}
\description{
Transition Probability Matrices for Common Up-and-Down Designs
}
\details{
Up-and-Down designs (UDDs) generate random walk behavior, whose theoretical properties can be summarized via a transition probability matrix (TPM). Given the number of doses \eqn{M}, and the value of the cdf \eqn{F} at each dose (i.e., the positive-response probabilities), the specific UDD rules uniquely determine the TPM.

The utilities described here calculate the TPMs of the most common and simplest UDDs:
\itemize{
\item The k-in-a-row or \strong{fixed staircase} design common in sensory studies: \verb{kmatMarg(), kmatFull()} (Gezmu, 1996; Oron and Hoff, 2009; see Note). Design parameters are k, a natural number, and whether k negative responses are required for dose transition, or k positive responses. The former is for targets below the median and vice versa.
\item The Durham-Flournoy Biased Coin Design: \code{bcdmat()}. This design can target any percentile via the \code{target} argument (Durham and Flournoy, 1994).
\item The original \emph{"classical"} median-targeting UDD: \code{classicmat()} (Dixon and Mood, 1948). This is simply a wrapper for \code{bcdmat()} with \code{target} set to 0.5.
\item Cohort or group UDD: \code{gudmat()}, with three design parameters for the group size and the up/down rule thresholds  (Gezmu and Flournoy, 2006).
}
}
\note{
As Gezmu (1996) discovered and Oron and Hoff (2009) further extended, k-in-a-row UDDs with \eqn{k>1} generate a random walk \emph{with internal states}. Their full TPM is therefore larger than \eqn{M\times M.} However, in terms of random-walk behavior, most salient properties are better represented via an \eqn{M\times M} matrix analogous to those of the other designs, with transition probabilities marginalized over internal states using their asymptotic frequencies. This matrix is provided by \code{kmatMarg()}, while \code{kmatFull()} returns the full matrix including internal states.

Also, in \code{kmatFull()} there are two matrix-size options. Near one of the boundaries (upper boundary with \code{lowTarget = TRUE}, and vice versa), the most extreme \eqn{k} internal states are practically indistinguishable, so in some sense only one of them really exists. Using the \code{fluffup} argument, users can choose between having a more aesthetically symmetric (but a bit misleading) full \eqn{Mk\times Mk} matrix, or reducing it to its effectivelly true size by removing \eqn{k-1} rows and columns.
}
\examples{
#  Let's use an 8-dose design, and  a somewhat asymmetric CDF

exampleF = pweibull(1:8, shape = 2, scale = 4)
# You can plot if you want: plot(exampleF)

# Here's how the transition matrix looks for the median-finding classic up-and-down

round(classicmat(exampleF), 2)
# Note how the only nonzero diagonals are at the opposite corners. That's how 
#   odd-n and even-n distributions communicate (see examples for vector functions).
# Also note how "up" probabilities (the 1st upper off-diagnoal) are decreasing, 
#   while "down" probabilities (1st lower off-diagonal) are increasing, and 
#   start exceeding "up" moves at row 4.

# Now, let's use the same F to target the 90th percentile, which is often
#    the goal of anesthesiology dose-finding studies.
#    We use the biased-coin design (BCD) presented by Durham and Flournoy (1994):

round(bcdmat(exampleF, target = 0.9), 2)

# Note that now there's plenty of probability mass on the diagonal (i.e., repeating same dose).

# Another option, actually with somewhat better operational characteristics, 
#   is "k-in-a-row". Let's see what k to use:

ktargOptions(.9, tolerance = 0.05)

# Even though nominally k=7's target is closest to 0.9, it's generally preferable
#    to choose a somewhat smaller k. So let's go with k=6.
# We must also specify whether this is a low (<0.5) or high target.

round(kmatMarg(exampleF, k = 6, lowTarget = FALSE), 2)

# Compare and contrast with the BCD matrix above! At what dose do the "up" and "down"
#   probabilities flip? 

# Lastly, if you want to see a 43 x 43 matrix - the full state matrix for k-in-a-row, 
#      run the following line:


\donttest{
  round(kmatFull(exampleF, k = 6, lowTarget = FALSE), 2)
}



}
\references{
\itemize{
\item Dixon WJ, Mood AM. A method for obtaining and analyzing sensitivity data. \emph{J Am Stat Assoc.} 1948;43:109-126.
\item Durham SD, Flournoy N. Random walks for quantile estimation. In: \emph{Statistical Decision Theory and Related Topics V} (West Lafayette, IN, 1992). Springer; 1994:467-476.
\item Gezmu M. The Geometric Up-and-Down Design for Allocating Dosage Levels. PhD Thesis. American University; 1996.
\item Gezmu M, Flournoy N. Group up-and-down designs for dose-finding. \emph{J Stat Plan Inference.} 2006;136(6):1749-1764.
\item Oron AP, Hoff PD. The k-in-a-row up-and-down design, revisited. \emph{Stat Med.} 2009;28:1805-1820.
\item Oron AP, Souter MJ, Flournoy N. Understanding Research Methods: Up-and-down Designs for Dose-finding. \emph{Anesthesiology} 2022; 137:137–50.
}
}
\seealso{
\itemize{
\item \code{\link{k2targ}}, \code{\link{ktargOptions}} to find the k-in-a-row target-response rate for specific k and vice versa.
\item \code{\link{g2targ}}, , \code{\link{gtargOptions}} likewise for group up-and-down.
\item \code{\link{pivec}}, \code{\link{currentvec}}, \code{\link{cumulvec}}, which provide probability vectors of dose-allocation distributions using Up-and-Down TPMs.
}
}
\author{
Assaf P. Oron \code{<assaf.oron.at.gmail.com>}
}
