\name{mra}
\alias{mra}
\title{Multiresolution Analysis of Time Series}
\description{
  This function performs a level \eqn{J} additive decomposition of the
  input vector or time series using the pyramid algorithm (Mallat 1989).
}
\usage{mra(x, wf = "la8", J = 4, method = "modwt", boundary = "periodic")
}
\arguments{
  \item{x}{A vector or time series containing the data be to
    decomposed.  This must be a dyadic length vector (power of 2) for
    \code{method="dwt"}.}
  \item{wf}{Name of the wavelet filter to use in the decomposition.  By
    default this is set to \code{"la8"}, the Daubechies orthonormal
    compactly supported wavelet of length \eqn{L=8} least asymmetric
    family.}
  \item{J}{Specifies the depth of the decomposition.  This must be a
    number less than or equal to
    \eqn{\log(\mbox{length}(x),2)}{log(length(x),2)}.}
  \item{method}{Either \code{"dwt"} or \code{"modwt"}.}
  \item{boundary}{Character string specifying the boundary condition.
    If \code{boundary=="periodic"} the default, then the vector you
    decompose is assumed to be periodic on its defined interval,\cr
    if \code{boundary=="reflection"}, the vector beyond its boundaries
    is assumed to be a symmetric reflection of itself.}
}
\value{
  Basically, a list with the following components
  \item{D?}{Wavelet detail vectors.}
  \item{S?}{Wavelet smooth  vector.}
  \item{wavelet}{Name of the wavelet filter used.}
  \item{boundary}{How the boundaries were handled.}

}
\details{
  This code implements a one-dimensional multiresolution analysis
  introduced by Mallat (1989).  Either the DWT or MODWT may be used to
  compute the multiresolution analysis, which is an additive
  decomposition of the original time series.
}
\references{
  Gencay, R., F. Selcuk and B. Whitcher (2001)
  \emph{An Introduction to Wavelets and Other Filtering Methods in
    Finance and Economics},
  Academic Press.
  
  Mallat, S. G. (1989)
  A theory for multiresolution signal decomposition: the wavelet
  representation,
  \emph{IEEE Transactions on Pattern Analysis and Machine Intelligence},
  \bold{11}, No. 7, 674-693.

  Percival, D. B. and A. T. Walden (2000)
  \emph{Wavelet Methods for Time Series Analysis},
  Cambridge University Press.
}
\seealso{
  \code{\link{dwt}}, \code{\link{modwt}}.
}
\examples{
## Easy check to see if it works...
x <- rnorm(32)
x.mra <- mra(x)
sum(x - apply(matrix(unlist(x.mra), nrow=32), 1, sum))^2

## Figure 4.19 in Gencay, Selcuk and Whitcher (2001)
data(ibm)     
ibm.returns <- diff(log(ibm))
ibm.volatility <- abs(ibm.returns)
## Haar
ibmv.haar <- mra(ibm.volatility, "haar", 4, "dwt")
names(ibmv.haar) <- c("d1", "d2", "d3", "d4", "s4")
## LA(8)
ibmv.la8 <- mra(ibm.volatility, "la8", 4, "dwt")
names(ibmv.la8) <- c("d1", "d2", "d3", "d4", "s4")
## plot multiresolution analysis of IBM data
par(mfcol=c(6,1), pty="m", mar=c(5-2,4,4-2,2))
plot.ts(ibm.volatility, axes=FALSE, ylab="", main="(a)")
for(i in 1:5)
  plot.ts(ibmv.haar[[i]], axes=FALSE, ylab=names(ibmv.haar)[i])
axis(side=1, at=seq(0,368,by=23), 
  labels=c(0,"",46,"",92,"",138,"",184,"",230,"",276,"",322,"",368))
par(mfcol=c(6,1), pty="m", mar=c(5-2,4,4-2,2))
plot.ts(ibm.volatility, axes=FALSE, ylab="", main="(b)")
for(i in 1:5)
  plot.ts(ibmv.la8[[i]], axes=FALSE, ylab=names(ibmv.la8)[i])
axis(side=1, at=seq(0,368,by=23), 
  labels=c(0,"",46,"",92,"",138,"",184,"",230,"",276,"",322,"",368))
}
\author{B. Whitcher}
\keyword{ts}
