#' @title Buffer raster
#'
#' @description Maps a distance-based buffer around each non-background (non-zero/non-nodata) grid cell in an input image.
#'
#' @param input Input raster file.
#' @param output Output raster file.
#' @param size Buffer size.
#' @param gridcells Optional flag to indicate that the 'size' threshold should be measured in grid cells instead of the default map units.
#' @param wd Changes the working directory.
#' @param verbose_mode Sets verbose mode. If verbose mode is `FALSE`, tools will not print output messages.
#' @param compress_rasters Sets the flag used by 'WhiteboxTools' to determine whether to use compression for output rasters.
#' @param command_only Return command that would be executed by `system()` rather than running tool.
#'
#' @keywords GISAnalysisDistanceTools
#'
#' @return Returns the tool text outputs.
#' @export
wbt_buffer_raster <- function(input, output, size, gridcells=FALSE, wd=NULL, verbose_mode=FALSE, compress_rasters=FALSE, command_only=FALSE) {
  wbt_init()
  args <- ""
  args <- paste(args, paste0("--input=", wbt_file_path(input)))
  args <- paste(args, paste0("--output=", wbt_file_path(output)))
  args <- paste(args, paste0("--size=", wbt_file_path(size)))
  if (gridcells) {
    args <- paste(args, "--gridcells")
  }
  if (!missing(wd)) {
    args <- paste(args, paste0("--wd=", wbt_file_path(wd)))
  }
  if (!missing(compress_rasters)) {
    args <- paste(args, paste0("--compress_rasters=", compress_rasters))
  }
  tool_name <- "buffer_raster"
  wbt_run_tool(tool_name, args, verbose_mode, command_only)
}


#' @title Cost allocation
#'
#' @description Identifies the source cell to which each grid cell is connected by a least-cost pathway in a cost-distance analysis.
#'
#' @param source Input source raster file.
#' @param backlink Input backlink raster file generated by the cost-distance tool.
#' @param output Output raster file.
#' @param wd Changes the working directory.
#' @param verbose_mode Sets verbose mode. If verbose mode is `FALSE`, tools will not print output messages.
#' @param compress_rasters Sets the flag used by 'WhiteboxTools' to determine whether to use compression for output rasters.
#' @param command_only Return command that would be executed by `system()` rather than running tool.
#'
#' @keywords GISAnalysisDistanceTools
#'
#' @return Returns the tool text outputs.
#' @export
wbt_cost_allocation <- function(source, backlink, output, wd=NULL, verbose_mode=FALSE, compress_rasters=FALSE, command_only=FALSE) {
  wbt_init()
  args <- ""
  args <- paste(args, paste0("--source=", wbt_file_path(source)))
  args <- paste(args, paste0("--backlink=", wbt_file_path(backlink)))
  args <- paste(args, paste0("--output=", wbt_file_path(output)))
  if (!missing(wd)) {
    args <- paste(args, paste0("--wd=", wbt_file_path(wd)))
  }
  if (!missing(compress_rasters)) {
    args <- paste(args, paste0("--compress_rasters=", compress_rasters))
  }
  tool_name <- "cost_allocation"
  wbt_run_tool(tool_name, args, verbose_mode, command_only)
}


#' @title Cost distance
#'
#' @description Performs cost-distance accumulation on a cost surface and a group of source cells.
#'
#' @param source Input source raster file.
#' @param cost Input cost (friction) raster file.
#' @param out_accum Output cost accumulation raster file.
#' @param out_backlink Output backlink raster file.
#' @param wd Changes the working directory.
#' @param verbose_mode Sets verbose mode. If verbose mode is `FALSE`, tools will not print output messages.
#' @param compress_rasters Sets the flag used by 'WhiteboxTools' to determine whether to use compression for output rasters.
#' @param command_only Return command that would be executed by `system()` rather than running tool.
#'
#' @keywords GISAnalysisDistanceTools
#'
#' @return Returns the tool text outputs.
#' @export
wbt_cost_distance <- function(source, cost, out_accum, out_backlink, wd=NULL, verbose_mode=FALSE, compress_rasters=FALSE, command_only=FALSE) {
  wbt_init()
  args <- ""
  args <- paste(args, paste0("--source=", wbt_file_path(source)))
  args <- paste(args, paste0("--cost=", wbt_file_path(cost)))
  args <- paste(args, paste0("--out_accum=", wbt_file_path(out_accum)))
  args <- paste(args, paste0("--out_backlink=", wbt_file_path(out_backlink)))
  if (!missing(wd)) {
    args <- paste(args, paste0("--wd=", wbt_file_path(wd)))
  }
  if (!missing(compress_rasters)) {
    args <- paste(args, paste0("--compress_rasters=", compress_rasters))
  }
  tool_name <- "cost_distance"
  wbt_run_tool(tool_name, args, verbose_mode, command_only)
}


#' @title Cost pathway
#'
#' @description Performs cost-distance pathway analysis using a series of destination grid cells.
#'
#' @param destination Input destination raster file.
#' @param backlink Input backlink raster file generated by the cost-distance tool.
#' @param output Output cost pathway raster file.
#' @param zero_background Flag indicating whether zero values should be treated as a background.
#' @param wd Changes the working directory.
#' @param verbose_mode Sets verbose mode. If verbose mode is `FALSE`, tools will not print output messages.
#' @param compress_rasters Sets the flag used by 'WhiteboxTools' to determine whether to use compression for output rasters.
#' @param command_only Return command that would be executed by `system()` rather than running tool.
#'
#' @keywords GISAnalysisDistanceTools
#'
#' @return Returns the tool text outputs.
#' @export
wbt_cost_pathway <- function(destination, backlink, output, zero_background=FALSE, wd=NULL, verbose_mode=FALSE, compress_rasters=FALSE, command_only=FALSE) {
  wbt_init()
  args <- ""
  args <- paste(args, paste0("--destination=", wbt_file_path(destination)))
  args <- paste(args, paste0("--backlink=", wbt_file_path(backlink)))
  args <- paste(args, paste0("--output=", wbt_file_path(output)))
  if (zero_background) {
    args <- paste(args, "--zero_background")
  }
  if (!missing(wd)) {
    args <- paste(args, paste0("--wd=", wbt_file_path(wd)))
  }
  if (!missing(compress_rasters)) {
    args <- paste(args, paste0("--compress_rasters=", compress_rasters))
  }
  tool_name <- "cost_pathway"
  wbt_run_tool(tool_name, args, verbose_mode, command_only)
}


#' @title Euclidean allocation
#'
#' @description Assigns grid cells in the output raster the value of the nearest target cell in the input image, measured by the Shih and Wu (2004) Euclidean distance transform.
#'
#' @param input Input raster file.
#' @param output Output raster file.
#' @param wd Changes the working directory.
#' @param verbose_mode Sets verbose mode. If verbose mode is `FALSE`, tools will not print output messages.
#' @param compress_rasters Sets the flag used by 'WhiteboxTools' to determine whether to use compression for output rasters.
#' @param command_only Return command that would be executed by `system()` rather than running tool.
#'
#' @keywords GISAnalysisDistanceTools
#'
#' @return Returns the tool text outputs.
#' @export
wbt_euclidean_allocation <- function(input, output, wd=NULL, verbose_mode=FALSE, compress_rasters=FALSE, command_only=FALSE) {
  wbt_init()
  args <- ""
  args <- paste(args, paste0("--input=", wbt_file_path(input)))
  args <- paste(args, paste0("--output=", wbt_file_path(output)))
  if (!missing(wd)) {
    args <- paste(args, paste0("--wd=", wbt_file_path(wd)))
  }
  if (!missing(compress_rasters)) {
    args <- paste(args, paste0("--compress_rasters=", compress_rasters))
  }
  tool_name <- "euclidean_allocation"
  wbt_run_tool(tool_name, args, verbose_mode, command_only)
}


#' @title Euclidean distance
#'
#' @description Calculates the Shih and Wu (2004) Euclidean distance transform.
#'
#' @param input Input raster file.
#' @param output Output raster file.
#' @param wd Changes the working directory.
#' @param verbose_mode Sets verbose mode. If verbose mode is `FALSE`, tools will not print output messages.
#' @param compress_rasters Sets the flag used by 'WhiteboxTools' to determine whether to use compression for output rasters.
#' @param command_only Return command that would be executed by `system()` rather than running tool.
#'
#' @keywords GISAnalysisDistanceTools
#'
#' @return Returns the tool text outputs.
#' @export
wbt_euclidean_distance <- function(input, output, wd=NULL, verbose_mode=FALSE, compress_rasters=FALSE, command_only=FALSE) {
  wbt_init()
  args <- ""
  args <- paste(args, paste0("--input=", wbt_file_path(input)))
  args <- paste(args, paste0("--output=", wbt_file_path(output)))
  if (!missing(wd)) {
    args <- paste(args, paste0("--wd=", wbt_file_path(wd)))
  }
  if (!missing(compress_rasters)) {
    args <- paste(args, paste0("--compress_rasters=", compress_rasters))
  }
  tool_name <- "euclidean_distance"
  wbt_run_tool(tool_name, args, verbose_mode, command_only)
}


